/* aweio.c -- primitives for the Algol W "Input/Output System"  -*-C-*-

This file requires "scanner.inc", which is the switch branches of a 
state machine for scanning input. "scanner.inc" is generated by "scanner.py".

--

This file is part of Awe. Copyright 2012 Glyn Webster.

This file is free software: you can redistribute it and/or modify it
under the terms of the GNU Limited General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Awe is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU Limited General Public
License along with Awe.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "awe.h"
#include "aweio.h"

#include <assert.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdbool.h>
#include <limits.h>
#include <complex.h>
#include <fenv.h>


/* READING  -------------------------------------------------------------------------------- */


_awe_Scanner _awe_stdin_scanner;
_awe_Scanner *_awe_active_scanner;


typedef enum {Integer, Real, Imaginary, Complex, Bits, Logical, String, Error, Eof} Scanner_result;


void 
_awe_Scanner_initialize (_awe_Scanner *scanner, FILE *file, char *input_name)
{
  scanner->input = file;
  scanner->input_name = input_name;
  scanner->eof = false;
  scanner->state = 0;
  scanner->line = 1;
  scanner->column = 0;
  scanner->buflen = 0;
}


static
void 
Scanner_start (_awe_Scanner *scanner)
{
  scanner->start_line = scanner->line;
  scanner->buflen = 0;
}


static
int
Scanner_error (_awe_Scanner *scanner, _awe_loc loc, const char *message)
{
  _awe_error( loc, "%s on line %d of %s.", message, scanner->start_line, 
             scanner->input_name ? scanner->input_name : "input" );
}


static
const char *
Scanner_result_string (Scanner_result result)
{
  switch (result) {
  case Integer:   return "an integer"; break;
  case Real:      return "a real"; break;
  case Imaginary: return "an imaginary"; break;
  case Complex:   return "a complex"; break;
  case Bits:      return "bits"; break;
  case Logical:   return "a logical"; break;
  case String:    return "a string"; break;
  case Error:     return "a syntax error"; break;
  case Eof:       return "the end of the input"; break;
  }
}


/* This is the only function that reads characters for a scanner. 
   '\r' characters are silently ignored to handle handle Windows '\r\n' linebreaks. */

static
int
Scanner_fgetc (_awe_Scanner *scanner)
{
  int c;
  do { c = fgetc(scanner->input); } while (c == '\r');
  switch (c) {
  case EOF:  ++scanner->line; scanner->column = 0; scanner->eof = true; break;
  case '\n': ++scanner->line; scanner->column = 0; break;
  default:   ++scanner->column; break;
  }
  return c;
}


static
void 
Scanner_new_card (_awe_Scanner *scanner, _awe_loc loc)
{
  if (scanner->eof) return;
  while (scanner->column >= 1) {
    (void)Scanner_fgetc(scanner);
    if (scanner->eof) return;
  }
}


static
void 
Scanner_addchar(_awe_Scanner *scanner, char c)
{
  assert(scanner->buflen < Scanner_BUFSIZE - 1);
  scanner->buffer[scanner->buflen++] = c;
}


static
void 
Scanner_close_buffer(_awe_Scanner *scanner)
{
  assert(scanner->buflen < Scanner_BUFSIZE - 1);
  scanner->buffer[scanner->buflen] = '\0';
}


static
void 
Scanner_addstring (_awe_Scanner *scanner, char *s)
{
  char *p;
  for (p = s; *p; ++p)
    Scanner_addchar(scanner, *p);
}


static
Scanner_result
Scanner_scan (_awe_Scanner *scanner)
{
  char c;

  while (true) {
    c = Scanner_fgetc(scanner);
    switch (scanner->state) {
#include "scanner.inc"
    }
  }
}



static
void
Scanner_type_error (_awe_Scanner *scanner, _awe_loc loc, Scanner_result expected, Scanner_result found)
{
    _awe_error( loc, "Expected to read %s on line %d of %s; found %s.",
               Scanner_result_string(expected), 
               scanner->line, 
               scanner->input_name ? scanner->input_name : "input",
               Scanner_result_string(found) );
}


static
int
Scanner_exception (_awe_Scanner *scanner, _awe_loc loc, Scanner_result expected, Scanner_result result)
{
  if (result == Eof)
    _awe_process_exception(loc, endfile);
  else
    Scanner_type_error (scanner, loc, expected, result);
}


static
int
Scanner_scan_for (_awe_Scanner *scanner, _awe_loc loc, Scanner_result expected)
{
  Scanner_result result;

  result = Scanner_scan(scanner);
  if (result == expected)
    return 1;
  else {
    Scanner_exception(scanner, loc, expected, result);
    return 0;  /* An exceptional condition was processed, the caller must supply a default value. */
  }
}


void
_awe_readcard (_awe_loc loc, _awe_str recipient, int length)
{
    int c;
    int i;
    
    _awe_str_cpy(recipient, length, " ", 1); /* empty string */
    Scanner_new_card(_awe_active_scanner, loc);
    for (i = 0; i < length; ++i) {
        c = Scanner_fgetc(_awe_active_scanner);
        if (c == '\n') 
            break;
        else if (c == EOF) {
            _awe_process_exception(loc, endfile);
            break;
        }
        else
            recipient[i] = c;
    }
    Scanner_new_card(_awe_active_scanner, loc);
}


void
_awe_readcard_char (_awe_loc loc, unsigned char *recipient)
{
    int c;

    Scanner_new_card(_awe_active_scanner, loc);
    c = Scanner_fgetc(_awe_active_scanner);
    if (c == '\n') 
        *recipient =  ' ';
    else if (c == EOF) {
        _awe_process_exception(loc, endfile);
        *recipient = ' ';
    }
    else
        *recipient = c;
    Scanner_new_card(_awe_active_scanner, loc);
}


void
_awe_read_integer (_awe_loc loc, int *recipient)
{
  int i;
  char *tailptr;

  if (!Scanner_scan_for(_awe_active_scanner, loc, Integer)) {
    *recipient = 0;
    return;
  };
  i = strtol((char *)_awe_active_scanner->buffer, &tailptr, 10);
  if (tailptr == (char *)_awe_active_scanner->buffer && _awe_active_scanner->buffer[0] == '-')
    Scanner_error(_awe_active_scanner, loc, "Integer too low");
  else if (tailptr == (char *)_awe_active_scanner->buffer)
    Scanner_error(_awe_active_scanner, loc, "Integer too high");
  else
    *recipient = i;
}


void
_awe_read_bits (_awe_loc loc, unsigned int *recipient)
{
  unsigned int i;
  char *tailptr;

  if (!Scanner_scan_for(_awe_active_scanner, loc, Bits)) {
    *recipient = 0;
    return;
  };
  i = strtoul((char *)_awe_active_scanner->buffer, &tailptr, 16);
  if (tailptr == (char *)_awe_active_scanner->buffer)
    Scanner_error(_awe_active_scanner, loc, "Bits constant too high");
  else
    *recipient = i;
}


void
_awe_read_string (_awe_loc loc, _awe_str recipient, int length)
{
  if (!Scanner_scan_for(_awe_active_scanner, loc, String)) {
      _awe_str_cpy(recipient, length, " ", 1); /* empty string */
    return;
  };
  if (_awe_active_scanner->buflen > length)
    Scanner_error(_awe_active_scanner, loc, "String too long");
  else
    _awe_str_cpy(recipient, length, _awe_active_scanner->buffer, _awe_active_scanner->buflen);
}


void
_awe_read_char (_awe_loc loc, unsigned char *recipient)
{
  if (!Scanner_scan_for(_awe_active_scanner, loc, String)) {
      *recipient = ' ';
      return;
  };
  if (_awe_active_scanner->buflen > 1)
      Scanner_error(_awe_active_scanner, loc, "String too long");
  else if (_awe_active_scanner->buflen == 0)
      *recipient = ' ';
  else
      *recipient = _awe_active_scanner->buffer[0];
}


void
_awe_read_logical (_awe_loc loc, int *recipient)
{
  if (!Scanner_scan_for(_awe_active_scanner, loc, Logical)) {
    *recipient = 0;
    return;
  };
  *recipient = (_awe_active_scanner->buffer[0] == 'T' || _awe_active_scanner->buffer[0] == 't');
}


static
void
test_real (_awe_loc loc)
{
  if (fetestexcept(FE_UNDERFLOW)) {
    feclearexcept(FE_UNDERFLOW);
    Scanner_error(_awe_active_scanner, loc, "Real number out of range");
  }
  else if (fetestexcept(FE_OVERFLOW)) {
    feclearexcept(FE_OVERFLOW);
    Scanner_error(_awe_active_scanner, loc, "Real number out of range");
  }
}


void _awe_read_real (_awe_loc loc, double *recipient) 
{
  double r;
  Scanner_result result;

  result = Scanner_scan(_awe_active_scanner);
  switch (result) {
  case Real:
  case Integer:
    r = strtod (_awe_active_scanner->buffer, NULL);
    test_real(loc);
    *recipient = r;
    break;
  default:
    Scanner_exception(_awe_active_scanner, loc, Real, result);
    *recipient = 0.0;
  }
}


void _awe_read_complex (_awe_loc loc, _Complex double *recipient) 
{
  Scanner_result result;
  char *tailptr;
  double r, i;

  result = Scanner_scan(_awe_active_scanner);
  Scanner_close_buffer(_awe_active_scanner);
  switch (result) {
  case Real:
  case Integer:
    r = strtod (_awe_active_scanner->buffer, NULL);
    test_real(loc);
    *recipient = r;
    break;
  case Imaginary:
    r = strtod (_awe_active_scanner->buffer, NULL);
    test_real(loc);
    *recipient = r * I;
    break;
  case Complex:
    r = strtod (_awe_active_scanner->buffer, &tailptr);
    test_real(loc);
    i = strtod (tailptr, NULL);
    test_real(loc);
    *recipient = r + i * I;
    break;
  default:
    Scanner_exception(_awe_active_scanner, loc, Real, result);
    *recipient = 0.0;
  }
}


/* WRITE  -------------------------------------------------------------------------------- */



/* The Editing Variables. See section 7.9.3. Note that these have special scoping rules.*/

int  i_w;
int  s_w;
int  r_w;
int  r_d;
unsigned char r_format;


void 
_awe_Editing_save (_awe_Editing_t* state) 
{
  state->i_w = i_w;
  state->s_w = s_w;
  state->r_w = r_w;
  state->r_d = r_d;
  state->r_format = r_format;
}


void 
_awe_Editing_restore (_awe_Editing_t* state)
{
  i_w = state->i_w;
  s_w = state->s_w;
  r_w = state->r_w;
  r_d = state->r_d;
  r_format = state->r_format;
}


#define LIMIT_WIDTH(w) ((w) > 132 ? 132 : (w))   /* Limit a width editing variable to 132 or lower. */


_awe_Printer _awe_stdout_printer;   /* Printer writing to the standard output */
_awe_Printer _awe_stderr_printer;   /* Printer writing to the standard error stream */
_awe_Printer *_awe_active_printer; /* The printer in use. */


void 
_awe_Printer_initialize (_awe_Printer *printer, FILE *file)
{
  printer->output = file;
  printer->line = 1;
  printer->column = 1;
  printer->true_column = 1;
  printer->page = 1;
  printer->page_estimate      = _awe_env_int(NULL,  "AWE_PAGE_ESTIMATE",      INT_MAX,  0, INT_MAX);
  printer->page_width         = _awe_env_int(NULL,  "AWE_PAGE_WIDTH",         132,      1, INT_MAX);
  printer->page_height        = _awe_env_int(NULL,  "AWE_PAGE_HEIGHT",        60,       1, INT_MAX);
  printer->hard_page_breaks   = _awe_env_bool(NULL, "AWE_HARD_PAGE_BREAKS",   false);
  printer->pretty_page_breaks = _awe_env_bool(NULL, "AWE_PRETTY_PAGE_BREAKS", false);
  printer->strict_line_breaks = _awe_env_bool(NULL, "AWE_STRICT_LINE_BREAKS", false);
  printer->trim_lines         = _awe_env_bool(NULL, "AWE_TRIM_LINES",         true);
  printer->eject_last_page    = _awe_env_bool(NULL, "AWE_EJECT_LAST_PAGE",    false);
}


/* Line and page breaking. 
   WRITE statements print to an imaginary printer full of fan-fold paper. */

static void Printer_start_field (_awe_Printer *printer, _awe_loc loc, int true_field_width);
static void Printer_end_field (_awe_Printer *printer, _awe_loc loc, int true_field_width, int field_width);
static void Printer_line_break (_awe_Printer *printer, _awe_loc loc);
static void Printer_page_break (_awe_Printer *printer, _awe_loc loc);


static
void
Printer_break_field (_awe_Printer *printer, _awe_loc loc, int field_width)
{
  int i;

  /* printf("[%d - %d:%d]", field_width, printer->column, printer->true_column); */

  if (printer->page_estimate == 0) 
    _awe_error( loc, "The page estimate is 0 pages, nothing should be written.");

  if (printer->strict_line_breaks && field_width > printer->page_width)
    _awe_error( loc, "A WRITE field was too wide for the page here.\nThe page width is %d but the field width was %d.", printer->page_width, field_width );

  if (printer->column + field_width - 1 > printer->page_width) /* won't fit on line */
    Printer_line_break(_awe_active_printer, loc); 
}


static
void
Printer_tab_field (_awe_Printer *printer, _awe_loc loc)
{
  int i;

  for (i = printer->true_column; i < printer->column; ++i)
    fputc(' ', printer->output);
  printer->true_column = printer->column;
}


static
void
Printer_start_field (_awe_Printer *printer, _awe_loc loc, int field_width)
{
  Printer_break_field (printer, loc, field_width);
  Printer_tab_field (printer, loc);
}


static
void
Printer_end_field (_awe_Printer *printer, _awe_loc loc, int field_width, int full_field_width)
{
  assert(field_width <= full_field_width);
  field_width = abs(field_width);
  printer->column += full_field_width;
  printer->true_column += field_width;
  /* printf("{%d:%d - %d:%d}", field_width, full_field_width, printer->column, printer->true_column); */
  if (printer->column - 1 > printer->page_width)  /* no more room on line */   /* XXX should this happen here? */
      { /* printf("[*]"); */
    Printer_line_break(_awe_active_printer, loc); }
}


/* Starts a new line by printing a line or page break. */

static
void
Printer_really_line_break (_awe_Printer *printer, _awe_loc loc)
{
  assert(printer->column >= 1);

  if (printer->line == printer->page_height) 
    Printer_page_break(printer, loc);
  else {
    /* start a new line */
    fputc('\n', printer->output);
    printer->column = 1;
    printer->true_column = 1;
    printer->line++;
  }
}


/* Ensures that the printhead is at the start of a line.  Does nothing if the printhead 
   is in the right place. (This is how Algol W line breaking works.) */

static
void
Printer_line_break (_awe_Printer *printer, _awe_loc loc)
{
  assert(printer->column >= 1);
  assert(printer->true_column >= 1);

  if (printer->column == 1) 
    return;
  else
    Printer_really_line_break(printer, loc);
}


static
void
Printer_page_break (_awe_Printer *printer, _awe_loc loc)
{
  /* If the page height has been reached, and hard_page_breaks is on, then 
     replace the last line feed of the page with a form feed. */
  assert(printer->line >= 1 && printer->line <= printer->page_height);
  while (printer->line < printer->page_height) {
        fputc('\n', printer->output);
      ++printer->line;
   }
    if (printer->hard_page_breaks || printer->pretty_page_breaks)
        if (printer->pretty_page_breaks)
            { 
                int i;
                fputc('\n', printer->output);
                for (i = 0; i < printer->page_width; ++i)
                    fputc('~', printer->output);
                fputc('\n', printer->output);
            }
        else
            fputc('\f', printer->output);
    else 
        fputc('\n', printer->output);
    printer->line = 1;
    printer->column = 1;
    printer->true_column = 1;
    if (printer->page == printer->page_estimate) 
        _awe_error( loc, "The page estimate, %d pages, has been reached.", printer->page_estimate);
    else
        ++printer->page;
}


void 
_awe_write_integer(_awe_loc loc, int i)
{
  int w = LIMIT_WIDTH(i_w);

  Printer_start_field(_awe_active_printer, loc, abs(w));
  fprintf(_awe_active_printer->output, "%*d", w, i);
  Printer_end_field(_awe_active_printer, loc, abs(w), abs(w) + LIMIT_WIDTH(s_w));

}


void 
_awe_write_logical(_awe_loc loc, int b)
{
  Printer_start_field(_awe_active_printer, loc, 6);
  fprintf(_awe_active_printer->output, "%6s", (b ? "TRUE" : "FALSE"));
  Printer_end_field(_awe_active_printer, loc, 6, 6 + LIMIT_WIDTH(s_w));
}


void 
_awe_write_bits(_awe_loc loc, unsigned int x)
{
  Printer_start_field(_awe_active_printer, loc, 14);
  fprintf(_awe_active_printer->output, "%14X", x);
  Printer_end_field(_awe_active_printer, loc, 14, 14 + LIMIT_WIDTH(s_w));
}


void 
_awe_write_string (_awe_loc loc, _awe_str s, int length)
{
  int i, n;

  Printer_break_field(_awe_active_printer, loc, length);
  n = _awe_active_printer->trim_lines ? _awe_str_unpadded_length(s, length) : length;
  if (n > 0) {
    Printer_tab_field(_awe_active_printer, loc);
    for (i = 0; i < n; ++i)
      fputc(s[i], _awe_active_printer->output);
  }
  Printer_end_field(_awe_active_printer, loc, n, length);
}


void 
_awe_write_char (_awe_loc loc, unsigned char c)
{
  Printer_break_field(_awe_active_printer, loc, 1);
  if (c == ' ' && _awe_active_printer->trim_lines)
    Printer_end_field(_awe_active_printer, loc, 0, 1);
  else {
    Printer_tab_field(_awe_active_printer, loc);
    fputc(c, _awe_active_printer->output);
    Printer_end_field(_awe_active_printer, loc, 1, 1);
  }
}



static char real_buffer[256];

/* Replace the "e" in C exponent notation with Algol W's "'" */
static
void 
replace_e (void) {
    int i;
    for (i = 0; i < 256 && real_buffer[i] != '\0'; ++i)
        if (real_buffer[i] == 'e') {
            real_buffer[i] = '\'';
            break;
        }
}

/* Writes a REAL number using the format specified in 'r_format'.
   Note that this prints "e" signs instead of "'" signs.
   Returns the length of the string actually printed. It will be longer than specified when necessary. */
static
int 
_awe_write_real_any(_awe_loc loc, double r)
{
  int w = LIMIT_WIDTH(r_w);
  int d = LIMIT_WIDTH(r_d);

  switch (r_format) {
  case 'A': case 'a':
    sprintf(real_buffer, "%*.*f", w, d, r);
    break;
  case 'S': case 's':
    if (r == 0.0) 
      /* Is is done to be consistent with page 42 of the June 1972 Reference Manual */
      sprintf(real_buffer, "%*s", w, "0    ");
    else
      sprintf(real_buffer, "%*.*e", w, (w - 8), r); 
    break;
  case 'F': case 'f':
    sprintf(real_buffer, "%*.*g", w, (w - 7), r);
    break;
  default:
    _awe_error(loc, "R_FORMAT = \"%c\", this is not a valid format code.", r_format);
  }
  replace_e();
  fprintf(_awe_active_printer->output, "%*s", w, real_buffer);
  return strlen(real_buffer);
}


void 
_awe_write_real(_awe_loc loc, double r)
{
  int w = LIMIT_WIDTH(r_w);

  Printer_start_field(_awe_active_printer, loc, abs(w));
  w = _awe_write_real_any(loc,r);
  Printer_end_field(_awe_active_printer, loc, abs(w), abs(w) + LIMIT_WIDTH(s_w));
}


void 
_awe_write_long_real(_awe_loc loc, double r)
{
  _awe_write_real(loc, r);
}


void 
_awe_write_complex(_awe_loc loc, _Complex double x)
{
  int w = LIMIT_WIDTH(r_w);
  int w2;

  Printer_start_field(_awe_active_printer, loc, w * 2);
  w2 = _awe_write_real_any(loc, creal(x));
  fputc(' ', _awe_active_printer->output);
  w2 += _awe_write_real_any(loc, cimag(x));
  fputc('I', _awe_active_printer->output);
  Printer_end_field(_awe_active_printer, loc, abs(w2), abs(w2) + LIMIT_WIDTH(s_w));
}


void 
_awe_write_long_complex(_awe_loc loc, _Complex double x)
{
  _awe_write_complex(loc, x);
}


void
_awe_write_reference(_awe_loc loc, void *ref)
{
    static char buffer [128];
    int w = LIMIT_WIDTH(i_w);
    
    Printer_start_field(_awe_active_printer, loc, w);
    if (ref == NULL)
        fprintf(_awe_active_printer->output, "%*s", w, "null");
    else if (ref == _awe_uninitialized_reference)
        fprintf(_awe_active_printer->output, "%*s", w, "UNINITIALIZED");
    else {
        sprintf(buffer, "%s.%i", _awe_class(ref), _awe_record_number(ref));
        assert(strlen(buffer) < 127);
        fprintf(_awe_active_printer->output, "%*s", w, buffer);
    }
    Printer_end_field(_awe_active_printer, loc, w, w + LIMIT_WIDTH(s_w));
}


/* IOCONTROL  -------------------------------------------------------------------------------- */


void 
_awe_iocontrol (_awe_loc loc, int code)
{
  int parameter;
  _awe_Printer *printer = _awe_active_printer;

  parameter = code % 10000;

  switch(code / 10000) {

  /* standard control codes */
  case 0:
    switch (code) {
    case 1: Scanner_new_card(_awe_active_scanner, loc); break;
    case 2: Printer_line_break(printer, loc); break;
    case 3: Printer_page_break(printer, loc); break;
    case 4: printer->hard_page_breaks = false; break;
    case 5: printer->hard_page_breaks = true; break;
    default:
      _awe_error( loc, "IOCONTROL code %d is undefined.", code);
      break;
    }
    break;

  /* Awe extended control codes. See awe.txt, INPUT/OUTPUT SYSTEM.  */
  case 1: printer->page_width    = (parameter == 9999) ? 2147483647 : parameter; break;
  case 2: printer->page_height   = (parameter == 9999) ?          1 : parameter; break;
  case 3: printer->page_estimate = (parameter == 9999) ? 2147483647 : parameter; break;
  case 4:
    switch (parameter) {
    case 1: printer->page = 1; printer->line = 1; break;
    case 2: Printer_really_line_break(printer, loc); break;
    case 4: printer->pretty_page_breaks = false; break;
    case 5: printer->pretty_page_breaks = true; break;
    case 6: printer->strict_line_breaks = false; break;
    case 7: printer->strict_line_breaks = true; break;
    case 8: printer->trim_lines = false; break;
    case 9: printer->trim_lines = true; break;
    case 10: printer->eject_last_page = false; break;
    case 11: printer->eject_last_page = true; break;
    default:
        _awe_error( loc, "IOCONTROL code %d is undefined.", code);
        break;
    }
    break;
    case 5:
        switch (parameter) {
        case 0: _awe_active_printer = &_awe_stdout_printer; break;
        case 1: _awe_active_printer = &_awe_stderr_printer; break;
        default:
            _awe_error( loc, "IOCONTROL code %d is undefined.", code);
            break;
        }
        break;

  default:
    _awe_error( loc, "IOCONTROL code %d is undefined.", code);
    break;
  }
}


/* init & exit  ---------------------------------------------------------------------------- */


void
_awe_init_aweio (_awe_loc loc)
{
  _awe_Scanner_initialize(&_awe_stdin_scanner, stdin, "the standard input");
  _awe_active_scanner = &_awe_stdin_scanner;

  _awe_Printer_initialize(&_awe_stdout_printer, stdout);
  _awe_Printer_initialize(&_awe_stderr_printer, stderr);
  _awe_active_printer = &_awe_stdout_printer;

  i_w = 14;
  s_w = 2;
  r_w = 14;
  r_d = 0;
  r_format = 'F'; 
}


void
_awe_Printer_finalize (_awe_loc loc, _awe_Printer *printer)
{
    if (printer->eject_last_page)
        Printer_page_break(printer, loc);  
    else
        Printer_line_break(printer, loc);  
}


void
_awe_exit_aweio (_awe_loc loc)
{
    _awe_Printer_finalize(loc, &_awe_stdout_printer);
    _awe_Printer_finalize(loc, &_awe_stderr_printer);
}


/* end */
