
SECTION "C#MAIN"  // Last modified 83-10-20

/* This is a CHEF utility program that requests the two-character
terminal identifier, as used in the UNIX /etc/termcap file, reads
this  terminal's  capabilitites  from that file and then stores a
predigested tabular form, suitable for the use of  CHEF,  in  the
file  'captab'.  It should be loaded with EF10.  This predigested
form is a sequence of entries c:l,a,b,c ended by 0: , where c  is
the internal capability number, l is the length of the string and
a,b,c are the string elements.  This is followed by a recognizing
data  tree  consisting  of triples of integers followed by 0,0,0.
This pair of tables, which appear in the  file  'captab',  should
then  be  inserted  in the message file and the directory of that
file updated.
    This code uses the convention that 'cap' refers to  a  number
greater  than 256 and that 'cp = cap - 256'.  Also 'acap' is that
same capability in two character form stored  in  a  cell,  while
'scap' is a two character string representation.  */

GET "CHEF_EF0"

MANIFEST
{ end_state  = #B01   //  End of a capability string
  end_choice = #B10   //  No alternative character
  escape_ch  = #O033
  ctl_mask   = #O037  //  Mask for making control characters
  bs_ch      = #O010  //  Back space
  cr_ch      = #O015  //  Carriage return
  ff_ch      = #O014  //  Form feed
  lf_ch      = #O012  //  Line feed
  rb_ch      = #O177  //  Rubout or delete
  ascii_mask = #O177  //  May be different on the Prime!
  // Values for modevec entries
  mode_undefined  = 0
  mode_speak      = 1
  mode_hear       = 2
  mode_error      = 3
  mode_number     = 4
  mode_boolean    = 5
  mode_preset     = 6    //  Set by default
  cap_tc          = 256 + 0 }

STATIC {
  tcfile      = ?
  pvec        = ?
  svec        = ?
  svec_p      = ?
  gvec_t      = ?
  gvec_csz    = ?
  nm_tab      = ?
  modevec     = ? }

LET start() BE
{1 LET v1 = VEC cap_sz; nm_tab := v1
 { LET v2 = VEC caps_vec_csz; caps_vec := v2
 { LET v3 = VEC cap_sz; modevec := v3
 { LET s = VEC 255/cell_bsz
 { LET inf = sys_msdos -> "b:termcap",
             sys_emas -> "GO_TERMCAP",
             "TERMCAP"
 { LET ouf = sys_msdos -> "b:captab", "CAPTAB"
   LET outfile = ?
   pvec := caps_vec; svec := caps_vec + pvec_csz
   svec_p := 1
   gvec_t := 0
   console_out_stream := output()
   FOR i = 1 TO cap_sz DO modevec!i := mode_undefined
   mk_acaps()
   {r writef("Enter two-character terminal identifier*N")
      IF sys_emas \/ sys_unix32 \/ sys_cpm \/ sys_vms THEN
        prompt("? ")
      { LET len = 0
        { LET ch = rdch()
          IF ch = '*N' THEN { s%0 := len; BREAK }
          len := len + 1
          s%len := ch
        } REPEAT
      }
   }r REPEATUNTIL s%0 = 2     // Insist !
   tcfile := findinput(inf)
   UNLESS valid_stream(tcfile) THEN err(" no TERMCAP file")
   selectinput(tcfile)
   outfile := findoutput(ouf)
   UNLESS valid_stream(outfile) THEN err(" no output file")
   selectoutput(outfile)
   UNLESS seek_name(s) THEN err(" %S not found", s)
   do_defaults()
   wr_heading(s)
   rd_termcaps()
   wr_pvecs()
   mk_gvecs()
   wr_gvecs()
   endwrite()
   selectoutput(console_out_stream)
   writes("Done*N") }1

AND do_defaults() BE
/* These capabilities are assumed by default:  automatic  margins
(off),   carriage-return,   cursor-down,   initialisation  string
(none), rubout (delete).  */
{1 LET scap_am = VEC 0
   AND scap_cr = VEC 0
   AND scap_do = VEC 0
   AND scap_is = VEC 0
   AND scap_rb = VEC 0
   scap_am%0, scap_am%1 := 1, 0   // initially off
   scap_cr%0, scap_cr%1 := 1, cr_ch
   scap_do%0, scap_do%1 := 1, lf_ch
   scap_is%0 := 0
   scap_rb%0, scap_rb%1 := 1, rb_ch
   preset(scap_am, cap_am)
   preset(scap_cr, cap_cr)
   preset(scap_do, cap_do)
   preset(scap_is, cap_is)
   preset(scap_rb, cap_rb) }1

AND enter(cp, sp, k, gp) BE
/* Enter into the 'gvec's table at 'gp' the tail of the string in
'svec'  whose  head is at 'sp' and tail starts at 'p+k'.  This is
the string for the  capability  number  'cp'.   This  routine  is
recursive.   The  table  entries  are  {g1,g2,g3} (implemented as
three vectors of bytes) where:
   g1 - is the current character
   g2 - is 'end_choice \/ end_state'
   g3 - is either the internal code for a capability (if
         'end_state' is true) or a pointer to the next triple. */
{1 // report("enter: cp=%N sp=%N k=%N gp=%N", cp, sp, k, gp)
 { LET len = svec%sp          // length of the string
   LET c = svec%(sp+k)        // Current character
   LET choice = end_choice    // The current choice
   IF len = 0 RETURN
   IF gp < gvec_t THEN        // It may be there already.
   {2 LET g1, g2, g3 = g1vec%gp, g2vec%gp, g3vec%gp
      TEST c = g1 THEN        // Is it the character?
        TEST k >= len THEN    // Yes, but are we done?
        { report(" Ambiguous capability %S", mk_scap(cp)); RETURN }
        ELSE                  // No, so go to next state
        { enter(cp, sp, k+1, g3); RETURN }
      ELSE                    // not equal
        TEST (g2 /\ end_choice) = end_choice THEN
        { mk_gap(gp); choice := 0 } // Insertion
        ELSE                  // Try next choice
        { enter(cp, sp, k, gp+1); RETURN } }2
  IF gp > gvec_t THEN err("enter")
  g1vec%gp := c               // Put the character away
  IF gp = gvec_t THEN gvec_t := gvec_t + 1
  TEST k >= len THEN          // Are we done?
    g1vec%gp, g2vec%gp, g3vec%gp :=
      c, choice + end_state, cp // Finish it off
  ELSE                        // Keep entering
  { g2vec%gp, g3vec%gp := choice, gvec_t
    enter(cp, sp, k+1, gvec_t) } }1

AND err(f, a, b, c) BE { report(f, a, b, c); FINISH }

AND expel(cp) = VALOF
/*  Outputs  the  capability  'cp'  as  a speaking string because
another string is to be used for listening and this must  now  be
remembered.  The parameter 'k' keeps track of the output position
on the line.  */
{1 LET p = pvec%cp
   LET mode = modevec!cp
   SWITCHON mode INTO
   {S DEFAULT: RETURN
      CASE mode_speak: CASE mode_number: CASE mode_boolean:
      CASE mode_preset:
      {2 LET len = svec%p AND k = 0
         writef(" %N:%N", cp, len); k := k + 2
         UNLESS len = 0 THEN wrch(',')
         FOR j = 1 TO len DO
         {3 k := k + 1; IF k > 15 THEN { writes("*N  "); k := 0 }
            writen(svec%(p+j))
            UNLESS j = len DO wrch(',') }3
         }2 }S newline() }1

AND get_nxt_acap() = VALOF
/*  Get the next capability.  The last character was a ':' but we
have to watch for '\' followed by a new line which indicates that
there are more.  */
{1 // writef("get_nxt_acap:*N")
   LET c = rdch() AND acap = 0
   SWITCHON c INTO
   {S CASE '\':
        c := rdch()
        WHILE c = '*S' \/ c = '*T' DO c := rdch()
        TEST c = '*N' THEN
        { c := rdch();                  ENDCASE }
        ELSE err("nxt acap")
      DEFAULT: acap := (c<<8) + rdch(); BREAK
      CASE '*N':                        BREAK
      CASE '*S': CASE ':':
      CASE '*T': c := rdch() }S
   REPEAT
   RESULTIS acap }1

AND mk_acap(s, cap) BE
/* Take the two character capability in the string 's' and store it
in the appropriate cell of the vector 'nm_tab'.  These are the ones
that CHEF is looking for.   The others will be ignored.  */
{1 LET cp = cap - 256
   nm_tab!cp := (s%1<<8) + s%2 }1

AND mk_acaps() BE
/*  This  makes  up the vector 'nm_tab' which is used for storing
the two character capabilities that we are looking for.  */
{ mk_acap("tc", cap_tc)     // cap_tc = 0, for MCAP only
  mk_acap("al", cap_al)
  mk_acap("am", cap_am)
  mk_acap("ce", cap_ce)
  mk_acap("cl", cap_cl)
  mk_acap("cm", cap_cm)
  mk_acap("co", cap_co)
  mk_acap("cr", cap_cr)
  mk_acap("dc", cap_dc)
  mk_acap("dl", cap_dl)
  mk_acap("do", cap_do)
  mk_acap("ei", cap_ei)
  mk_acap("ho", cap_ho)
  mk_acap("ic", cap_ic)
  mk_acap("im", cap_im)
  mk_acap("is", cap_is)
  mk_acap("k0", cap_k0)
  mk_acap("k1", cap_k1)
  mk_acap("k2", cap_k2)
  mk_acap("k3", cap_k3)
  mk_acap("k4", cap_k4)
  mk_acap("k5", cap_k5)
  mk_acap("kb", cap_kb)
  mk_acap("kd", cap_kd)
  mk_acap("ke", cap_ke)
  mk_acap("kl", cap_kl)
  mk_acap("kr", cap_kr)
  mk_acap("ks", cap_ks)
  mk_acap("ku", cap_ku)
  mk_acap("li", cap_li)
  mk_acap("lg", cap_lg)
  mk_acap("rb", cap_rb)
  mk_acap("se", cap_se)
  mk_acap("so", cap_so)
  mk_acap("wl", cap_wl)
  mk_acap("wr", cap_wr) }

AND mk_gap(gp) BE
/*  Move  the triples, representing the listening tree, one place
to the right at the position 'gp' to make a gap.  */
{1 // First shift pointers
   FOR i = 0 TO gvec_t DO
   {2 UNLESS (g2vec%i /\ end_state) = end_state THEN
      { LET g3 = g3vec%i
        IF g3 > gp THEN g3vec%i := g3 + 1 } }2
   // Now shift triples
   FOR i = gvec_t TO gp BY -1 DO
   {2 LET g1, g2, g3 = g1vec%i, g2vec%i, g3vec%i
      g1vec%(i+1), g2vec%(i+1), g3vec%(i+1) := g1, g2, g3 }2
   gvec_t := gvec_t + 1
   IF gvec_t > gvec_csz*cell_bsz THEN err("gvecs %N", gvec_t) }1

AND mk_gvecs() BE
/*  Make  up  the listening tree.  Note that several capabilities
need not be listened  for.   The  work  is  done  mostly  by  the
recursive routine 'enter'.  */
{1 LET used_csz = pvec_csz + (svec_p + cell_bsz)/cell_bsz
   gvec_csz := (caps_vec_csz - used_csz)/3
   g1vec := caps_vec + used_csz
   g2vec, g3vec := g1vec + gvec_csz, g2vec + gvec_csz
   FOR cp = 1 TO cap_sz DO
   {2 LET mode = modevec!cp
      SWITCHON 256+cp INTO
      {3 CASE cap_cl: CASE cap_cm: CASE cap_co:
         CASE cap_ho: CASE cap_li: CASE cap_se:
         CASE cap_so: CASE cap_am: CASE cap_ks:
         CASE cap_is: CASE cap_lg: CASE cap_ke:   ENDCASE
         CASE cap_ce: IF sys_msdos ENDCASE
         DEFAULT:
         SWITCHON mode INTO
         {S CASE mode_speak: CASE mode_hear: CASE mode_preset:
              enter(cp, pvec%cp, 1, 0); LOOP
            DEFAULT: ENDCASE }S }3
      report(" cap %S will not be heard, mode is %S",
        mk_scap(cp), mode_s(mode)) }2
   gvec_t := gvec_t - 1 }1

AND mk_scap(cp) = VALOF
/* Yield a string of two characters.    We rely on the fact that
tables are static.  */
{1 LET s = (TABLE 0, 0)
   s%0 := 2
   s%1, s%2 := (nm_tab!cp)>>8, nm_tab!cp
   RESULTIS s }1

AND mode_s(n) = VALOF
/* A printable string representing the mode 'n'.  */
  SWITCHON n INTO
  {1 CASE mode_undefined:  RESULTIS "undefined"
     CASE mode_boolean:    RESULTIS "boolean"
     CASE mode_speak:      RESULTIS "speak"
     CASE mode_hear:       RESULTIS "heard"
     CASE mode_error:      RESULTIS "error"
     CASE mode_number:     RESULTIS "number"
     CASE mode_preset:     RESULTIS "preset"
     DEFAULT:              RESULTIS "mode unknown" }1

AND preset(s, cap) BE
/* Used for presetting the standard capabilities.  */
{1 // report("preset: cap=%S", mk_scap(cap-256))
 { LET cp = cap - 256
   pvec%cp := svec_p
   modevec!cp := mode_preset
   FOR i = 0 TO s%0 DO put_c(s%i) }1

AND put_c(c) BE
/*  Put  the character 'c' into 'svec' and increment the pointer.
*/
{1 svec%svec_p := c; svec_p := svec_p + 1 }1

AND report(f,a,b,c,d) BE
/* For console error messages and reports.  */
{1 LET oo = output()
   selectoutput(console_out_stream)
   writef(f, a, b, c, d)
   newline()
   selectoutput(oo) }1

AND rd_termcaps() BE
/* Read the terminal capabilities.  We have just read a ':'.
Most of the work is done by 'rd_cap'.   */
{1 // writef("rd_termcap:*N")
   {R LET acap = get_nxt_acap()
      IF acap = 0 THEN BREAK
      FOR cp = 0 TO cap_sz DO
      { IF acap = nm_tab!cp THEN
        { TEST cp + 256 = cap_tc THEN
          { LET s = VEC 1
            rdch()  // lose '='
            s%0 := 2; s%1 := rdch(); s%2 := rdch()
            report(" using caps for screen %S", s)
            make_indexed(tcfile)
            set_file_position(tcfile, 0)  // rewind
            make_sequential(tcfile)
            UNLESS seek_name(s) DO err(" %S not found", s) }
          ELSE rd_cap(cp)
          GOTO found } }
      report(" cap %C%C not used", acap>>8, acap)
      UNTIL rdch() = ':' DO LOOP
      found: LOOP }R
   REPEAT
   svec%0 := svec_p-1 }1

AND rd_cap(cp) BE
/*  Now  read one capability.  It is 'cp'.  Note that '\' is used
as an escape and that '^' indicates a control character.  Not all
possibilities that may occur in TERMCAP are handled here.  We may
need to add more code later.  */
{1 // writef("rd_cap: cap=%N*N", cp)
 { STATIC { old_cp = 0 }
   LET c = rdch() AND yield = 0 AND n = 0 AND mode = mode_speak
   LET cap = 256 + cp
   LET orig = svec_p
   SWITCHON c INTO
   {2 // The numbers
      CASE '#': n := readn()
        SWITCHON cap INTO
        { CASE cap_co: screen_cols := n; ENDCASE
          CASE cap_li: screen_rows := n; ENDCASE
          DEFAULT:
              report(" cap %S #=%N not used",mk_scap(cp),n) }
        put_c(1); put_c(n); yield := orig
        mode := mode_number;                    ENDCASE
      // The strings
      CASE '=':
      {3 svec_p := svec_p + 1 // Allow for length of string
         {4 c := rdch()
         next:
            SWITCHON c INTO
            {5 CASE '\': c := rdch()
                 SWITCHON c INTO
                 {6 CASE 'E': put_c(escape_ch); ENDCASE
                    CASE 'n': put_c(lf_ch);     ENDCASE
                    CASE 't': put_c('*T');      ENDCASE
                    CASE 'r': put_c(cr_ch);     ENDCASE
                    CASE 'b': put_c(bs_ch);     ENDCASE
                    CASE 'f': put_c(ff_ch);     ENDCASE
                    CASE '*N':c := rdch()
                               REPEATWHILE c = '*S' \/ c = '*T'
                              GOTO next
                    DEFAULT:
                      TEST '0' <= c <= '7' THEN
                      { c := (c-'0') * 8 + rdch() - '0'
                        c := c * 8 + rdch() - '0'
                        put_c(c /\ ascii_mask) }
                      ELSE put_c(c) }6          ENDCASE
               CASE '#': c := readn(); put_c(c);
                 c := terminator; GOTO next
               CASE '^': c := rdch()
                 put_c(c /\ ctl_mask);          ENDCASE
               CASE ':': svec%orig := svec_p - orig - 1
                 yield := orig; mode := mode_speak
                                                BREAK
               DEFAULT: put_c(c);               ENDCASE }5 }4
         REPEAT }3;                 ENDCASE
      // Cancel previous capability (for use with cap_tc)
      CASE '@': report(" cap %S cancelled", mk_scap(cp))
                modevec!cp := mode_undefined;   ENDCASE
      // The booleans
      CASE ':': put_c(1); put_c(1)
        yield := orig; mode := mode_boolean;   ENDCASE
      DEFAULT: writef("[\^#: expected] c=%C*N", c)
        yield := 0; mode := mode_error }2
   SWITCHON modevec!cp INTO   // Look at old mode
   {S CASE mode_speak:
        IF cp = old_cp THEN
        { expel(cp); mode := mode_hear;        ENDCASE }
      CASE mode_hear: CASE mode_boolean: CASE mode_error:
      CASE mode_preset:
        report(" Preset capability %S is replaced", mk_scap(cp))
        mode := mode_speak;                    ENDCASE
      CASE mode_number:
        TEST mode = mode_number THEN
          report(" cap %S redefined", mk_scap(cp))
        ELSE
          report(" Ambiguous cap %S mode %S", mk_scap(cp),
          mode_s(mode))
          mode := mode_error;                  ENDCASE
      DEFAULT:
        report("Status error %S %S", mk_scap(cp), mode_s(mode))
        mode := mode_error;                    ENDCASE
      CASE mode_undefined:                     ENDCASE }S
   old_cp := cp
   IF mode = mode_error THEN RETURN
   modevec!cp := mode
   pvec%cp := yield }1

AND seek_name(snm) = VALOF
/* Get the name of the terminal 'snm' in TERMCAP.   We  look  for
two characters that begin a line and are followed by '|'.  */
{1 // writef("seek_name:*N")
 { LET c0, c1, c2, c3 = '*S', '*S', '*S', '*N'
   {r {s c0 := c1; c1 := c2; c2 := c3; c3 := rdch()
         IF c3 = endstreamch THEN RESULTIS FALSE }s
      REPEATUNTIL c3 = '|' }r
   REPEATUNTIL ((snm%1 = c1) /\ (snm%2 = c2)) /\ (c0 = '*N')
   UNTIL c3 = ':' DO c3 := rdch()
   RESULTIS TRUE }1

AND wr_gvecs() BE
/* Write out the listening tree as a sequence of triples
                            c, n, p;
followed  by  0,0,0; where 'p' is the character to be looked for,
'n' is a combination of 'end_choice' and 'end_state' and 'p' is a
pointer  to  the  next triple (if 'end_state' is false) or is the
internal coding of the capability (if 'end_state' is  true).   If
'end_choice'  is  true,  then  we  know  that  there  are no more
characters to choose from, so if there was no match, the  default
capability is implied.  */
{1 FOR i = 0 TO gvec_t DO
   { writef(" %N,%N,%N", g1vec%i, g2vec%i, g3vec%i)
     IF i REM 8 = 7 THEN newline() }
   writef(" %N,%N,%N*N",0,0,0) }1

AND wr_heading(s) BE
  writef(" %S|*N", s)

AND wr_pvecs() BE
/* Write out the capability strings in the form
                          c:l,a,b,c
followed by 0: , where 'c' is the capability  code,  'l'  is  the
length  of the string and that many characters follow it.  We use
decimal numbers because escape and control  characters  may  give
problems in transporting them.  */
{1 FOR cp = 1 TO cap_sz DO
     UNLESS modevec!cp = mode_hear THEN expel(cp)
   writes(" 0:*N") }1
.

