; cdag.lsp                  Gordon S. Novak Jr.            ; 12 Jan 09

; Copyright (c) 2009 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Computations represented as Directed Acyclic Graphs (DAGs)

; 18 Dec 02; 19 Dec 02; 20 Dec 02; 23 Dec 02; 24 Dec 02; 26 Dec 02; 27 Dec 02
; 30 Dec 02; 31 Dec 02; 16 Jan 03; 21 Jan 03; 16 Jan 03; 23 Jan 03; 09 Nov 06
; 06 Nov 07; 08 Nov 07; 13 Nov 07; 15 Nov 07; 24 Dec 07; 27 Dec 07; 23 Dec 08

; See also cdagtest.lsp

; There are two kinds of DAG: a DAG spec (cf. program source code)
; and DAG instance (cf. runtime stack).  The DAG instance contains
; data values and has pointers to the corresponding DAG spec.

(glispobjects

; data spec
(cdatas (list (name  symbol)
	      (type  gltype)
	      (value anything)
	      (kind  symbol)) )              ; var, file, url, const

; computation spec
(ccomps (list (name symbol)
	      (cprog cprogs)
              ; parm is in name space of the cprog
              ; source/dest is in name space of the cdag
	      (inputs  (listof (list (parm symbol) (source symbol))))
	      (outputs (listof (list (parm symbol) (dest symbol)))) ) )

; program spec
(cprogs (symbol (proplist
	      (storedkind symbol)                  ; lisp, unix, dag
	      (storedfnname symbol)                ; function name
	      (storedinputs  (listof glnametype))
	      (storedoutputs (listof glnametype)) ))
  prop ((kind    ((or (storedkind self) *gp-language*)) result symbol)
        (fnname  ((or (storedfnname self) self)) result symbol)
        (inputs  ((or (storedinputs self)
                      (glargs (fnname self)) ))
                ;;;;;      '((in anything) (in2 anything))))
                     result (listof glnametype))
        (outputs ((or (storedoutputs self)
                      '((out anything))))
                     result (listof glnametype)) ) )

; directed acyclic graph spec
(cdags (symbol (proplist
	 (inputs  (listof glnametype))
	 (outputs (listof glnametype))
	 (datas   (listof cdatas))
	 (comps   (listof ccomps)) ))
  msg  ((dspec  (glambda (self nm) (that (datas self) with name == nm)))
	(cspec  (glambda (self nm) (that (comps self) with name == nm))) ) )

; data instance
(cdatai (list (name symbol)
	      (value anything)
	      (ready boolean)) )

; dag instance
(cdagi (list (spec cdags)
	     (datas (listof cdatai)) )
  msg  ((dspec  (glambda (self nm) (dspec (spec self) nm)))
	(cspec  (glambda (self nm) (cspec (spec self) nm))) ) )

(binding (cons (name symbol) (value anything)))

) ; glispobjects

(setf (symbol-function '^) (symbol-function 'expt))

; 19 Dec 02; 13 Nov 07
; Define a Lisp function for use with the DAG system
(gldefun deflispfn ((fn cprogs) (nm symbol) (ins (listof glnametype))
		    (outs (listof glnametype)))
  ((storedkind fn) = 'lisp)
  ((storedfnname fn) = nm)
  ((storedinputs fn) = ins)
  ((storedoutputs fn) = outs) )

; 20 Dec 02; 06 Nov 07
; Define a DAG
(gldefun defcdag ((spec cdags) (ins (listof glnametype))
                  (outs (listof glnametype))
		  (ds (listof cdatas)) (cs (listof ccomps)))
  ((inputs  spec) = ins)
  ((outputs spec) = outs)
  ((datas   spec) = ds)
  ((comps   spec) = cs) )

; 18 Dec 02; 19 Dec 02; 24 Dec 02
; Make a DAG instance from a DAG spec
; (setq sumsqi (daginst 'sumsq))
(gldefun daginst ((d cdags)) (result cdagi)
  (a cdagi with spec = d
     datas = (for x in (datas d) collect
		  (a cdatai with name = (name x)
		                 value = (value x)
				 ready = ((kind x) == 'const) ))))

; 19 Dec 02; 20 Dec 02; 27 Dec 02
; Evaluate a DAG given a set of input values
; (dageval sumsqi '((x . 3) (y . 4)))
(gldefun dageval ((d cdagi) (vals (listof binding)))
  (let (outv)
    (for val in vals do (dagdataval d (name val) (value val)))
    (outv = (that (datas d) with
		  name == (name (first (outputs (spec d))))))
    (and (ready outv) (value outv)) ))

; 26 Dec 02; 27 Dec 02
; dagevals is an alternate interface to dageval, with a list of arg values.
; (dagevals 'sumsq '(3 4))
(gldefun dagevals ((d cdags) (vals (listof anything)))
  (dageval (daginst d) (mapcar #'(lambda (nt val) (cons (name nt) val))
			       (inputs d) vals) ) )

; 19 Dec 02; 20 Dec 02; 09 Nov 06
; Define a data value in a DAG
(gldefun dagdataval ((d cdagi) (nm symbol) (val anything))
  (let (di flag args tmp res)
    (di = (that (datas d) with name == nm))
    ((ready di) = *gltrue*)
    ((value di) = val)
    (for c in (comps (spec d)) do
      (if (that (inputs c) with source == nm)
	  (progn (flag = *gltrue*)     ; flag = all inputs available
		 (args = (for i in (inputs c) collect
			      (a binding with name = (parm i)
				 value = (if (ready
					      (tmp = (that (datas d) with
						       name == (source i))))
					     (value tmp)
					     (flag = *glfalse*)))))
		 (if flag
		     (progn (res = (dagcompevalb c args))
                            ; could modify for multiple outputs
			    (dagdataval d (dest (first (outputs c)))
					res) )) ))) ))

; 19 Dec 02; 24 Dec 02; 26 Dec 02
; Evaluate a DAG program given a binding list of argument values
(gldefun dagcompevalb ((c ccomps) (inargs (listof binding)))
  (let (arglist outfile command args)
    (if (or ((kind (cprog c)) == 'lisp) ((kind (cprog c)) == 'unix))
	(args = (for nm in (inputs (cprog c)) collect
		     (value (that inargs with name == (name nm))))))
    (case (kind (cprog c))
      (lisp (apply (fnname (cprog c)) args) )
      (unix (outfile = (gensym "glfile"))
	    (command = (string-downcase (symbol-name (fnname (cprog c)))))
	    (for arg in args do (command = (cat2 command arg)))
	    (system (cat2 command
			 (concatenate 'string ">" (symbol-name outfile))))
	    (symbol-name outfile))
      (dag  (dageval (daginst (fnname (cprog c))) inargs) ) ) ))

; 19 Dec 02
; Reset a DAG instance
(gldefun dagreset ((d cdagi))
  (for di in (datas d) do ((ready di) = *glfalse*) ((value di) = *glnull*))
  nil)

; 20 Dec 02; 09 Nov 06; 06 Nov 07; 15 Nov 07
; Given a DAG, return data values that are used more than once
(gldefun dagusedv ((spec cdags))
  (let ((once (listof symbol)) (twice (listof symbol)))
    (for c in (comps spec)
      (for i in (inputs c)
           when (not (eq (kind (dspec spec (source i))) 'const))
	(if (member (source i) once)
	    (twice +_ (source i))
	    (once +_ (source i)) ) ) )
    twice))

; 20 Dec 02; 23 Dec 02; 27 Dec 02; 09 Nov 06; 14 Nov 07; 24 Dec 07; 27 Dec 07
; Given a DAG, return Lisp code
; e.g. (dag2lisp 'sumsq) = (LAMBDA (X Y) (+ (* X X) (* Y Y)))
; If glispflag, includes agr types
(gldefun dag2lisp ((spec cdags) &optional glispflag)
  (let ((twice (dagusedv spec)) (dagi (daginst spec)) code vars body)
    (code = (list nil))
    (for i in (inputs spec) do 
	 (dagsdataval dagi code twice (name i) (name i)))
    (for j in (datas spec) when (eq (kind j) 'const)
         (dagsdataval dagi code nil (name j) (value j)))
    (vars = (set-difference twice (for i in (inputs spec) collect (name i))))
    (if (member (name (first (outputs spec))) twice)
	(push (name (first (outputs spec))) (cdr code))
        (push (value (that (datas dagi) with
                           name == (name (first (outputs spec)))))
              (cdr code)))
    (body = (reverse (cdr code)))
    (if vars
	(body = (list (cons 'let (cons vars body)))))
    (cons 'lambda (cons (if glispflag
                            (inputs spec)
                            (mapcar #'car (inputs spec)))
                        (if (and glispflag
                                 (outputs spec)
                                 (null (cdr (outputs spec)))
                                 (type (first (outputs spec))))
                            (cons (list 'result
                                        (type (first (outputs spec))))
                                  body)
                            body))) ))


; 06 Nov 07; 08 Nov 07; 14 Nov 07; 24 Dec 07
; make a dag into a Lisp function
; (dag2lispfn 'sumsq 'sumsqfn)
(defun dag2lispfn (spec &optional name glispflag)
  (let ((code (dag2lisp spec glispflag)))
    (if glispflag
        (gldefun-expr (cons (or name spec) (cdr code)))
        (progn (setf (gloriginalexpr (or name spec)) code)
               (eval (cons 'defun (cons (or name spec)
                                        (cdr code)))) ) ) ))

; 20 Dec 02; 09 Nov 06; 23 Dec 08
; Define a symbolic data value in a DAG
; code has a nil sentinel on the front so we can do a push onto it
; twice = list of vars that are used more than once
(gldefun dagsdataval ((d cdagi) code twice (nm symbol) (val anything))
  (let (di flag args tmp res)
    (or (di = (that (datas d) with name == nm))
        (error "dagsdataval nm = ~A~%" nm))
    ((ready di) = *gltrue*)
    (if (member nm twice)
	(progn (if (not (eq nm val))
		   (push (list 'setq nm val) (cdr code)) )
	       ((value di) = nm))
        ((value di) = val) )
    (for c in (comps (spec d)) do
      (if (that (inputs c) with source == nm)
	  (progn (flag = *gltrue*)     ; flag = all inputs available
		 (args = (for i in (inputs c) collect
			      (a binding with name = (parm i)
				 value = (if (ready
					      (tmp = (that (datas d) with
						       name == (source i))))
					     (value tmp)
					     (flag = *glfalse*)))))
		 (if flag
		     (progn (res = (dagscompevalb c args))
                            ; could modify for multiple outputs
			    (dagsdataval d code twice
					 (dest (first (outputs c)))
					 res) )) )))
    d))

; 20 Dec 02; 26 Dec 02; 08 Nov 07
; Evaluate a DAG program given a binding list of argument values
(gldefun dagscompevalb ((c ccomps) (args (listof binding)))
  (let (arglist)
    (args = (for nm in (daginputs c) collect
		 (value (that args with name == (name nm)))))
    (case (kind (cprog c))
      (lisp (cons (fnname (cprog c)) args))
      ((unix dag) (list (kind (cprog c)) (cons (fnname (cprog c)) args))) ) ))

; 08 Nov 07
; get or default input names
(gldefun daginputs ((c ccomps))   (result (listof glnametype))
  (or (inputs (cprog c))
      (for inp in (inputs c)
        collect (a glnametype with name = (parm inp) type = nil) ) ) )

; 23 Dec 02; 26 Dec 02; 27 Dec 02; 09 Nov 06; 13 Nov 07
; convert Lisp code to a DAG
; (lisp2dag 'sumsqb '(lambda (x y) (+ (* x x) (* y y))))
; (lisp2dag 'test2
;           '(lambda (x y) (unix (paste "" (cut "-c1-3" x) (cut "-c1-3" y)))))
(gldefun lisp2dag ((name cdags) (lambdaexp anything)
		   &optional (restype gltype))
	 (result cdags)
  (let ((formals (cadr lambdaexp)) code vars cmps outvar)
    (code = (if (cdddr lambdaexp)
		(cons 'progn (cddr lambdaexp))
                (caddr lambdaexp)))
    (vars = (list nil))                ; new vars added to end
    (for formal in formals do
	 (if (consp formal)
	     (dagvar (car formal) (cadr formal) vars)
	     (dagvar formal nil vars)))
    (cmps = (list nil))                ; new comps added to end
    (outvar = (code2dag nil code vars cmps 'lisp))
    ((inputs name) = (for formal in formals collect
			  (if (consp formal)
			      formal
			      (a glnametype with name = formal
				    type = 'anything))))
    ((outputs name) = (list (a glnametype with name = outvar
			       type = (or restype 'anything))))
    ((datas name) = (cdr vars))
    (if restype ((type (that (datas name)
			     with (name (that cdatas))== outvar))
		  = restype))
    ((comps name) = (cdr cmps))
    name ))

; 23 Dec 02; 24 Dec 02; 31 Dec 02
(defun code2dag (outvar code vars cmps system)
  (let ()
    (if (constantp code)
	(dagconst outvar code vars)
        (if (symbolp code)
	    (if outvar
		(iddag outvar code vars cmps system)
	        code)
	    (if (member (car code) '(setq setf))
		(code2dag (cadr code) (caddr code) vars cmps system)
	        (if (member (car code) '(unix dag lisp))
		    (code2dag outvar (cadr code) vars cmps (car code))
		    (c2dag outvar code vars cmps system)))) ) ))

; 23 Dec 02; 24 Dec 02
; Make an identity computation
(defun iddag (outvar code vars cmps system)
  (c2dag outvar (list (case system (lisp 'identity) (unix 'cat)) code)
	 vars cmps system))

; 23 Dec 02; 24 Dec 02; 26 Dec 02; 27 Dec 02
; Produce DAG for a Lisp function call.
; Result is the name of a data node holding the result of the call.
; vars and cmps are side-effected by adding things to the ends.
(gldefun c2dag ((outvar symbol) (code anything) (vars (listof cdatas))
		(cmps (listof ccomps)) (system symbol))
  (let (args cmp (fn cprogs) (dag cdags) outv)
    (fn = (car code))
    (or (symbolp fn) (error "Bad code form ~A" code))
    (setq args (mapcar #'(lambda (x) (code2dag nil x vars cmps system))
		       (cdr code)))
    (outv = (dagvar outvar (dagouttype fn) vars))
    (if (not (eq (kind fn) system)) (initcprogs fn args system))
    (cmp = (a ccomps with name = (glmkatom 'c)
	      cprog = fn
	      inputs = (dagzipargs (inputs fn) args)
	      outputs = (dagzipargs (outputs fn) (list outv))))
    (cmps _+ cmp)
    outv ))

; 23 Dec 02; 24 Dec 02; 26 Dec 02; 06 Nov 07; 13 Nov 07
; initialize a Lisp function as a cprogs, given an arg list
(gldefun initcprogs ((fn cprogs) (args (listof anything)) (system symbol))
  (let (inps inames)
    (inames = (mapcar #'car (or (get fn 'inputs) (inputs fn))))
    (inps =
      (for arg in args collect
	   (a glnametype with name = (pop inames)
		              type = (or (and (glconstantp arg)
					      (glconstanttype arg))
					 'anything))))
    ((storedkind fn) = system)
    ((storedfnname fn) = fn)
    ((storedinputs fn) = inps)
    ((storedoutputs fn) = (list (a glnametype with name = 'out
                                   type = 'anything)))
    fn))

; 23 Dec 02
; zip together lists of formals (name type) and sources
(gldefun dagzipargs ((ins (listof glnametype)) (ds (listof symbol)))
  (for nt in ins collect (list (name nt) (pop ds))) )

; 23 Dec 02
; Create a cdatas for a variable, if needed.
; if outvar is non-NIL, it is used as the name.
(gldefun dagvar ((outvar symbol) (type gltype) (vars (listof cdatas)))
  (let (newvar)
    (or outvar (outvar = (glmkatom 'd)))
    (or (that vars with name == outvar)
	(progn (newvar = (a cdatas with name = outvar
			    type = type
			    kind = 'var))
	       (vars _+ newvar)
	       (name newvar) ) ) ))

; 24 Dec 02; 31 Dec 02; 09 Nov 06
; Create a cdatas for a constant
(gldefun dagconst ((outvar symbol) (const anything) (vars (listof cdatas)))
  (let (newvar)
    (or outvar (outvar = (glmkatom 'd)))
    (newvar = (a cdatas with name = outvar
		             type = (if (quotep const)
					(glconstanttype (cadr const))
				        (glconstanttype const))
			     value = (if (quotep const)
					 (cadr const)
				         const)
			     kind = 'const))
    (vars _+ newvar)
    (name newvar) ))

; 23 Dec 02
; Determine output type for a function
(gldefun dagouttype (fn) (glfnresulttype fn))

(gldefun cat2 ((x string) (y string)) (result string)
  (concatenate 'string x " " y))

; 27 Dec 02; 30 Dec 02
; Write an object to a file, tab-delimited, in depth-first order
(defun writer (value type &optional filename)
  (let ()
    (or filename (setq filename (symbol-name (gensym "glfile"))))
    (with-open-file (ofile filename :direction :output :if-exists :supersede)
      (writeitem value type ofile)
      (terpri ofile) )
    filename) )

; 27 Dec 02; 30 Dec 02; 31 Dec 02
; Write an object to a line, tab-delimited, in depth-first order
(defun writeitem (value type stream)
  (let ()
    (if (symbolp type)
	(if (member type *glbasictypes*)
	    (progn (write value :stream stream)
		   (princ #\Tab stream))
	    (if (gltypep type)
		(writeitem value (car (glstr type)) stream)
	        (error "bad type ~A ~A~%" value type)))
        (if (consp type)
	    (if (or (member (car type) *gltypenames*)
		    (member (car type) '(linked-list)))
		(case (car type)
		  (cons (writeitem (car value) (cadr type) stream)
			(writeitem (cdr value) (caddr type) stream))
		  (list (mapc #'(lambda (val typ) (writeitem val typ stream))
			      value (cdr type)))
		  (crecord
		    (dolist (typ (cddr type))
		      (writeitem (cdr (assoc (car typ) (cddr value)))
				 (cadr typ) stream)))
		  ((listof arrayof linked-list) (writeseq value type stream))
		  (t (error "bad type ~A ~A~%" value type)))
	        (if (not (member (car type) '(^)))
		    (writeitem value (cadr type) stream)))
	    (error "bad type ~A ~A~%" value type))) ))

; 30 Dec 02; 31 Dec 02
; Write a list of objects to a file, delimited by blank line at the end
; type should be (listof <item>), (arrayof <item>), (linked-list <item>)
(defun writeseq (value type stream)
  (let (ptrtype)
    (case (car type)
      (listof (dolist (item value)
		(writeitem item (cadr type) stream)
		(terpri stream) ) )
      (arrayof (dotimes (i (array-dimension value 0))
		 (writeitem (aref value i) (cadr type) stream)
		 (terpri stream) ) )
      (linked-list
        (setq ptrtype (glfindview (cadr type) (car type)))
	(while (not (glsendd value ptrtype 'null))
	  (writeitem value (cadr type) stream)
	  (terpri stream)
	  (setq value (glsendd value ptrtype 'rest)) ) )
      (t (error "bad type ~A ~A~%" value type)))
    (terpri stream) ))

; 30 Dec 02; 31 Dec 02
; Read an object from a file, tab-delimited, in depth-first order
(defun reader (filename type)
  (let (value)
    (with-open-file (ifile filename :direction :input)
      (setq value (readitem nil 0 type ifile)) )
    value))

; 30 Dec 02
; Read a list of objects from a file, delimited by blank line at the end
; type should be (listof <item>), (arrayof <item>), (linked-list <item>)
(defun readseq (ifile type)
  (let (lst item line done n arr i last ptrtype rectype)
    (while (not done)
      (setq line (read-line ifile nil 'zzeof))
      (if (or (eq line 'zzeof) (= (length line) 0))
	  (setq done t)
	  (progn (setq item (readitem line 0 (cadr type) ifile))
		 (push item lst) ) ) )
    (case (car type)
      (listof (nreverse lst))
      (arrayof (setq n (length lst))
	       (setq arr (make-array n))
	       (setq i 1)
	       (dolist (item lst) (setf (aref arr (- n i)) item) (incf i))
	       arr)
      (linked-list (setq ptrtype (glfindview (cadr type) 'linked-list))
		   (setq rectype (glclusterrole (glcluster ptrtype) 'record))
		   (setq last (glsendd nil ptrtype 'null-value))
		   (dolist (item lst)
		     (glsendd item rectype 'link\: last)
		     (setq last item))
		   last) ) ))

; 30 Dec 02; 31 Dec 02; 23 Jan 03
; read an object from a line, tab-delimited, in depth-first order
; returns a value and a column
(defun readitem (line col type ifile)
  (let (val valb lst)
    (if (symbolp type)
	(if (member type *glbasictypes*)
	    (progn (unless line
		     (setq line (read-line ifile nil 'zzeof))
		     (setq col 0))	       
		   (read-from-string line nil nil :start col))
	    (if (gltypep type)
		(readitem line col (car (glstr type)) ifile)
	        (error "bad type ~A ~A~%" value type)))
        (if (consp type)
	    (if (or (member (car type) *gltypenames*)
		    (member (car type) '(linked-list)))
		(case (car type)
		  (cons (multiple-value-setq
			  (val col) (readitem line col (cadr type) ifile))
			(multiple-value-setq
			  (valb col) (readitem line col (caddr type) ifile))
			(values (cons val valb) col))
		  (list (mapc #'(lambda (typ)
				  (multiple-value-setq
				    (val col) (readitem line col (cadr type)
							ifile))
				  (push val lst)
				  (if (<= col 0)
				      (setq line (read-line ifile nil 'zzeof))))
			      (cdr type))
			(values (nreverse lst) col))
		  (crecord
		    (dolist (typ (cddr type))
		      (multiple-value-setq (val col)
					   (readitem line col typ ifile))
		      (push (cons (car typ) val) lst)
		      (if (<= col 0)
			  (setq line (read-line ifile nil 'zzeof))))
		    (values (cons 'crecord (cons (cadr type) (nreverse lst)))
			    col))
		  ((listof arrayof linked-list) (readseq ifile type))
		  (t (error "bad type ~A ~A~%" value type)))
	        (if (not (member (car type) '(^)))
		    (readitem line col (cadr type) ifile)))
	    (error "bad type ~A ~A~%" value type))) ))

; 21 Jan 03
; unix defined as a macro for Lisp.
; The macro returns a file name (the output file) and unix return value
(defmacro unix (form)
  (let ((filename (gensym)))
    `(let ((,filename (symbol-name (gensym "glfile"))))
       (values ,filename
	       (system (concatenate 'string
			 ,(string-downcase (symbol-name (car form))) " "
			 ,@(mapcan #'(lambda (x) (list x " "))
				   (cdr form))
			 ">" ,filename)))) ))

