;  glhead.lsp       Common Lisp Header file for GLISP       ; 21 Sep 16

; Copyright (c) 2016 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

;  This is the header portion of the Common Lisp version of the GLISP compiler.

; 11 Jan 96; 02 Jan 97; 13 Mar 97; 03 Oct 97; 26 May 98; 03 Jan 00; 15 Feb 00
; 22 Feb 02; 01 Oct 02; 08 Oct 02; 22 Oct 02; 25 Oct 02; 14 Mar 05; 08 Sep 06
; 12 Sep 06; 27 Sep 06; 19 Oct 06; 30 Jan 07; 09 Jan 09; 21 Sep 12

(proclaim '(special *glfn* *glbreakonerror* *glexprstack*
		    *gllastfncompiled* *glspecfncompiled*
		    *glspecfnscompiled* *glglsendflg*
                    *glsilenceerrors* *glfounderror* ))

(defmacro concat (&rest args) `(concatenate 'string ,@args))

(defmacro getd (fn) `(and (fboundp ,fn) (symbol-function ,fn)))

(defmacro nconc1 (lst x) `(setf ,lst (nconc ,lst (cons ,x nil))))


;
;  Translate a file containing GLISP functions into a Common Lisp file
;  which can be compiled.
;
;  To translate GLISP files for subsequent Lisp compilation, the general
;  procedure is to load the GLISP compiler and all files first (so that
;  structure declarations and generic functions will be present).
;
;  The result is a file in plain Common Lisp which can be compiled and
;  (possibly) used by itself without the GLISP compiler being present.
;
;  A combined function which does the above is:
;
;  (glcompfiles <directory> <auxiliary file list> <file list> <output file>)
;     where the args are file name strings or lists of them.

; 15 Aug 91; 05 Oct 94; 06 Oct 94; 06 Oct 95; 08 Sep 06
; Read a file, translate it, and append to the output file.
(defun gltransread (infile outfile except nodecls)
  (prog (expr res)
lp  (setq expr (read infile nil 'gltransreadeofvalue))
    (cond ((eq expr 'gltransreadeofvalue) (return))
          ((listp expr)
             (eval expr)
             (terpri outfile)
	     (setq res (gltransexpr expr except nodecls))
	     (if (and (consp res) (eq (first res) 'progn))
		 (dolist (x (rest res)) (prin1 x outfile) (terpri outfile))
		 (if res (prin1 res outfile)))
	     (terpri outfile)
	     (dolist (fn (nreverse *glspecfnscompiled*))
	       (prin1 (cons 'defun (cons fn (cdr (symbol-function fn))))
		      outfile)
	       (terpri outfile))
	     (setq *glspecfnscompiled* nil)))
    (go lp)))

; 15 Aug 91; 09 Sep 92; 02 Oct 92; 06 Oct 94; 11 Jan 96; 14 Mar 05; 08 Sep 06
; translate an expression into plain lisp.
(defun gltransexpr (expr except nodecls)
  (let (res trans restype)
    (setq res
	  (cond ((eq (car expr) 'gldefun)
		  (if (member (cadr expr) except)
		      (setq trans expr)
		      (progn
			(glcc (cadr expr))
			(setq trans (cons 'defun
				      (cons (cadr expr)
					    (cdr (glcompiled (cadr expr))))))
			(setq restype (glfnresulttype (cadr expr)))))
		  (if (and restype (not (member 'glfnresulttype except))
			   (not nodecls))
		      (list 'progn trans
			    `(setf (glarguments ',(cadr expr))
				   ',(glarguments (cadr expr)))
			    `(setf (glfnresulttype ',(cadr expr)) ',restype))
		      trans))
		((and (consp expr)
		      (eq (first expr) 'setf)
		      (member (caadr expr) except))
		  nil)
		((and (consp expr)
		      (eq (first expr) 'defun)
		      (member (cadr expr) except))
		  nil)
		((and (consp expr)
		      (eq (first expr) 'setf)
		      (glmacrop (second expr))
		      (not (member 'glmacro except)))
		  (cons (first expr)
			(cons (macroexpand (second expr))
			      (cddr expr))))
		((and (consp expr)
		      (eq (first expr) 'glispobjects))
		  (if (not (member 'glispobjects except))
		      (cons 'progn
			    (mapcar #'(lambda (exp)
					(gltransexpr
				         `(setf (glstructure ',(first exp))
						',(rest exp))
					 except nodecls))
				    (rest expr)))))
		((and (consp expr)
		      (glmacrop expr)
		      (not (member (first expr)
				   '(defvar proclaim setf defun defmacro))))
		  (if (not (member (first expr) except))
		      (macroexpand expr)))
		(t expr) ))
    (if (and (consp res)
	     (eq (first res) 'progn))
	(cons 'progn (mapcar #'(lambda (exp) (gltransexpr exp except nodecls))
			     (rest res)))
	res) ))

; 03 Oct 91; 09 Sep 92; 13 Jul 93; 17 Dec 93; 19 Dec 93; 05 Oct 94; 06 Oct 95
; 08 Sep 06
(defun glcompfiles (dir auxfilelst filelst outfilename
                    &optional headerfile except nodecls)
  (let (prettysave lengthsave levelsave line)
    (dolist (file auxfilelst) (load (file-name file dir "")))
    (dolist (file filelst)    (load (file-name file dir "")))
    (with-open-file (outfile (file-name outfilename dir "")
			     :direction :output
			     :if-exists :supersede)
         (setq prettysave *print-pretty*)
	 (setq lengthsave *print-length*)
	 (setq levelsave *print-level*)
	 (setq *print-pretty* t)
	 (setq *print-length* nil)         ; unlimited
	 (setq *print-level* nil)          ; unlimited
	 (princ "; " outfile)
	 (princ (get-time-string) outfile)
	 (terpri outfile)
	 (if headerfile
	     (with-open-file (infile (file-name headerfile dir "")
				     :direction :input)
	       (while (not (eq (setq line (read-line infile nil 'gleofvalue))
			       'gleofvalue))
		 (princ line outfile)
		 (terpri outfile)) ) )
	 (setq *glspecfnscompiled* nil)
	 (dolist (infilename filelst)
		 (with-open-file (infile (file-name infilename dir "")
					 :direction :input)
			 (gltransread infile outfile except nodecls) ) )
	 (setq *print-pretty* prettysave)
	 (setq *print-length* lengthsave)
	 (setq *print-level*  levelsave))
    outfilename ))

(defun glmacrop (x)
  (and (consp x) (symbolp (car x)) (macro-function (car x))) )

; Functions for Common Lisp for Interlisp compatibility
; Gordon S. Novak Jr.     25 MAY 82 - 03 Sept. 86


(defun kwote (x) (if (constantp x) x (list 'quote x) ) )

; Print n spaces.
(defun spaces (n) (dotimes (i n) (princ " ")) )

; 12 Oct 90
(defun glmatchsubatom (atm start length atmb)
  (let ((atmstr (symbol-name atm)) (atmstrb (symbol-name atmb)))
    (if (>= (length atmstr) (1- (+ start length)))
      (dotimes (i length atmb)
        (unless (char= (char atmstr (1- (+ start i)))
		       (char atmstrb i))
	        (return nil)) )) ))

; 12 Oct 90
; cf. interlisp nthchar, but it returns characters, not character symbols.
; Returns blank for n out of range.
(defun nthchar (x n)
  (let ((xname (symbol-name x)))
    (setq n (if (minusp n) (+ (length xname) n) (1- n)))
    (if (< n (length xname))
	(char xname n)
	#\ ) ))

; subatom as in interlisp
(defun subatom (atm n m)
  (let ((atmname (symbol-name atm)) substr flg l ll)
    (setq l (length atmname))
    (setq n (if (minusp n) (+ n l) (1- n)))
    (setq m (if (minusp m) (1+ (+ m l)) m))
    (if (> m l) (setq m l))
    (setq substr (subseq atmname n m))
    (setq ll (- m n))
    (dotimes (i ll) (if (char= (char substr i) #\:)
			(setq flg t)))
    (if flg (if (and (= ll 1) (char= (char substr 0) #\:))
		'\:
	        (if (and (= ll 2) (char= (char substr 0) #\:)
		                (char= (char substr 0) #\=))
		    '\:=
		    (intern substr)))
	      (read-from-string substr))
  ))

; 24 Dec 93; 27 Oct 94
; find the string position in atom atm where a character in the
; bit table bittbl occurs, starting with character n.
(defun strposl (bittbl atm n)
 (prog (nc atmname)
   (if (stringp atm)
       (setq atmname atm)
       (if (symbolp atm)
	   (setq atmname (symbol-name atm))
	   (return nil)))
  (cond ((null n)(setq n 1)))
  (setq nc (length atmname))
a (cond ((> n nc)(return nil))
        ((aref bittbl (char-code (char atmname (1- n))))
	  (return n)))
  (setq n (1+ n))
  (go a) ))

; make a bit table from a list of characters.
(defun makebittable (l)
 (let (bittbl)
  (setq bittbl (make-array char-code-limit :initial-element nil))
  (mapc #'(lambda (x)
            (setf (aref bittbl
			(cond ((numberp x) x)
                              ((characterp x) (char-code x))
	                      ((symbolp x)
			        (char-code (character
				             (symbol-name x))))))
	          t))
        l)
  bittbl ))


; 28 Dec 89; 27 Sep 06; 19 Oct 06
; gldefun sets up a macro definition so a function will be compiled
; the first time it is called.
(defmacro gldefun (&rest l) `(gldefun-expr (quote ,l)))
(defun gldefun-expr (x)
  (let ((orig (gloriginalexpr (car x))) res)
    (when (and orig (not (equal (cdr orig) (cdr x))))
	  (mapc #'gluncompile (glinstancefns (car x)))
	  (setf (glinstancefns (car x)) nil) )
    (setf (gloriginalexpr (car x)) (cons 'lambda (cdr x)) )
    (glputhook (car x))
    (setq res (glsimpleresulttype (car x)))
    (if (and res (symbolp res) (not (glbasictypep res)))
        (pushnew (car x) (glresultof res)))  ))

; Put a fn def to cause auto compilation.
(defun glputhook (fn)
  (setf (glcompiled fn) nil)
  (eval `(defun ,fn (&rest l)
	   (glcompileme ',fn l))))

; 05 Jan 94; 30 Jan 07
; glcompileme calls the compiler the first time a function is called to
; automatically compile it.
(defun glcompileme (fn args)
  (glcc fn)
  (if (get fn 'glcompiled)
      (apply fn args)) )

; 19 Oct 06
; Get simple result type declaration
(defun glsimpleresulttype (fn)
  (let ((fndef (glgetd fn)) done res)
    (setq fndef (cddr fndef))
    (while (and (not done) (consp fndef))
      (if (and (consp (car fndef))
               (eq (caar fndef) 'result)
	       (glokstr? (cadar fndef)))
	  (progn (setq done t)
                 (setq res (cadar fndef)))
          (pop fndef)))
    res))

; get function definition for the glisp compiler.
(defun glgetd  (fn) (gloriginalexpr fn))

(defvar lambdaword (find-symbol "LAMBDA-BLOCK" :si))

; 03 Jan 95; 15 Feb 00; 08 Oct 02; 09 Jan 09
; Uses *glfn* freely
(defun glambdatran (glexpr)
 (let (newexpr)
  (setf (gloriginalexpr *glfn*) glexpr)
  (cond ((setq newexpr (glcomp *glfn* glexpr nil))
	   (setf (symbol-function *glfn*) (glambdafix *glfn* newexpr))
           (setf (glcompiled *glfn*) newexpr)))
  newexpr ))

; 09 Jan 09; 21 Sep 16
; Change a LAMBDA form at top of a function defintion to LAMBDA-BLOCK
; This is specific to GCL
(defun glambdafix (fn expr)
  (if (and fn (consp expr) (eq (car expr) 'lambda))
      (cons lambdaword (cons fn (cdr expr)))
      expr))

; 27 Mar 89; 06 Jun 90; 20 May 93; 03 Jan 95; 21 Sep 12
; Rewritten for new calling sequence.
(defun glerror (fn msgstr &rest args)
  (if (and (boundp '*glsilenceerrors*) *glsilenceerrors*)
      (setq *glfounderror* t)
      (progn
        (format t "glisp error detected by ~A in function ~A~%" fn *glfn*)
        (apply #'format (cons t (cons msgstr args)))
        (terpri)
        (when (boundp '*glexprstack*)
          (format t "in expression: ~s~%"     (car *glexprstack*))
          (format t "within expression: ~s~%" (cadr *glexprstack*))
          (cond (*glbreakonerror* (error "break within glisp compilation")))
          (list (list 'glerr (list 'quote (car *glexprstack*))) nil) )) ))

; 17 Mar 95
(defun gevstringify (x) (stringify x))
(setf (glfnresulttype 'gevstringify) 'string)

; 17 Mar 95
; stringify commented out because it is part of dwtrans.lsp
; and because this version causes problems when used with Sun.
; use (copy-seq (symbol-name x)) on Sun
#|
(defun stringify (x)
  (cond ((stringp x) x)
        ((symbolp x) (symbol-name x))
	(t (princ-to-string x))))
|#
(setf (glfnresulttype 'stringify) 'string)

; edited:  4-May-83 16:32 
(defun glapply (fn args)
  (if (and (symbolp fn)
	   (not (fboundp fn))
	   (gloriginalexpr fn)
	   (not (glcompiled fn)))
      (glcc fn))
  (apply fn args))

(defun nleft (lst n)
  (prog ((ll lst))
    (if (<= n 0) (return))
    (while (and lst (> n 0)) (pop lst) (decf n))
    (unless (or lst (= n 0)) (return))
    (while lst (pop lst) (pop ll))
    (return ll) ))

; 02 Nov 92; 24 Dec 93; 27 Oct 94
(defun glglobalvarp (symbol)
  (and symbol (symbolp symbol)
       (or (get symbol 'glispglobalvar)
	   (let ((pn (symbol-name symbol)))
	     (and (char= (char pn 0) #\*)
		  (char= (char pn (1- (length pn))) #\*) ) ))))

(defun lastchar (symbol)
  (let ((pn (symbol-name symbol)))
    (char pn (1- (length pn))) ))

(defun symbol<  (x y)
  (and (symbolp x) (symbolp y) (string<  (symbol-name x) (symbol-name y))))
(defun symbol>  (x y)
  (and (symbolp x) (symbolp y) (string>  (symbol-name x) (symbol-name y))))
(defun symbol<= (x y)
  (and (symbolp x) (symbolp y) (string<= (symbol-name x) (symbol-name y))))
(defun symbol>= (x y)
  (and (symbolp x) (symbolp y) (string>= (symbol-name x) (symbol-name y))))

(defun true  (&rest x) (declare (ignore x)) t)
(defun false (&rest x) (declare (ignore x)) nil)
(setf (glfnresulttype 'true) 'boolean)
(setf (glfnresulttype 'false) 'boolean)

; set flag since "send" is a Lucid function.
(setq *glglsendflg* t)
