; gp.lsp             Gordon S. Novak Jr.         ; 21 Apr 14

; Copyright (c) 2011 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; 30 Oct 07; 01 Nov 07; 08 Apr 08; 17 Apr 08; 22 Apr 08; 24 Apr 08; 06 May 08
; 12 May 08; 13 May 08; 14 May 08; 15 May 08; 23 May 08; 30 May 08; 24 Nov 08
; 24 Dec 08; 31 Dec 08; 14 Apr 09; 22 May 09; 31 Aug 09; 28 Jan 10; 02 Feb 10
; 04 Feb 10; 11 Feb 10; 12 Feb 10; 17 Mar 10; 23 Mar 10; 25 Mar 10; 01 Nov 10
; 10 Feb 11; 27 Sep 11; 05 Oct 11; 07 Oct 11; 14 Oct 11

; GP: Graphical Programming interface

(glispobjects

(gp-group (list (name         symbol)
		(language     symbol)
                (startflag    boolean)
                (sources      (listof glnametype))
		(cigroup      cigroup)
	        (conns        menu-conns) )

  prop ((window        ((window (menu-set (conns self)))))
	(menu-set      ((menu-set (conns self))))
	(command-menu  ((menu (named-item (menu-set (conns self))
					  'command))) )
	(message-area-x (10))
	(message-area-y ((max 0 (- (height (window self)) 140)))) )
  msg  ((sym           gp-group-sym)
        (redraw        gp-group-redraw)
	(set-sym       gp-group-set-sym) ) )

  ) ; glispobjects

(defvar *gp-window-width*    600)
(defvar *gp-window-height*   700)
(defvar *gp-programs*)
(setq   *gp-programs*        '(iter-do iter-acc heurs convh find dijkstra plot
                               plotb))
(defvar *gp-trace*           nil)
(defvar *gp-types*           nil)
(defvar *gp-group*           nil)
(defvar *gp-language*        'lisp)
(defvar *gp-automaticity*    5)
(defvar *gp-taskpq*          (make-array '(10) :initial-element nil))

; for testing
(setq *gp-types* '(assembly assemblyc string file-of-words mycvh))

(glispglobals
 (*gp-trace*      boolean)
 (*gp-programs*   (listof symbol))
 (*gp-types*      (listof gltype))
 (*gp-group*      gp-group)
 (*gp-language*   symbol)
    )  ; glispglobals

(defun gp-trace ()
  (setq *gp-trace* t)
  (setq *vac-trace* t)
  (setq *vac-alltrace* t) )

(defun gp-untrace ()
  (setq *gp-trace* nil)
  (setq *vac-trace* nil)
  (setq *vac-alltrace* nil) )

(defun gp (&optional sources) (gp-server sources))      ; for convenience
(gldefun gpg () (cigroup *gp-group*))

; 23 Dec 03; 31 Dec 03; 19 Jan 04; 16 Jun 04; 18 Jun 04; 20 Jun 04; 23 Jun 04
; 25 Jun 04; 03 Aug 04; 04 Nov 04; 03 Feb 05; 10 Feb 05; 17 Apr 08; 22 Apr 08
; 24 Apr 08; 12 May 08; 13 May 08; 30 May 08; 24 Nov 08; 18 Mar 10
; Graphical Programming server: make programs from generics
(gldefun gp-server ((sources (listof glnametype)))
  (let (w group fn)
    (w = (window-create *gp-window-width* *gp-window-height*
                        "Graphical Programming Server"))
    (setq *wio-window-width* *gp-window-width*)
    (setq *wio-window-height* *gp-window-height*)
    (setq *vac-newtypes* nil)
    (group = (gp-init-window w))
    ((startflag group) = t)
    ((sources group) = sources)
    (setq *gp-group* group)             ; save for debugging
    (fn = (gp-interaction group))
    (close w)
    (destroy w)
    ((window *gp-group*) = nil)
    fn ))

; 17 Mar 10
; get rid of gp window
(gldefun gp-close ()
  (let (w)
    (when (and *gp-group* (w = (window *gp-group*)))
      (close w)
      (destroy w)
      ((window *gp-group*) = nil) ) ))

; 25 Jun 04; 30 Jul 04; 13 Aug 04; 18 Aug 04; 26 Oct 04; 28 Oct 04; 11 Oct 05
; 13 Apr 06; 08 Apr 08; 17 Apr 08; 12 May 08; 30 May 08; 21 May 09; 28 Jan 10
(gldefun gp-interaction ((group gp-group))
  (let ((redraw t) done sel sents datadef geom undef res)
    (while (not done)
      (geom = (window-geometry (window group)))
      (if (or (*wio-window-width* != (third geom))
	      (*wio-window-height* != (fourth geom)))
	  (progn (*wio-window-width* = (third geom))
		 (*wio-window-height* = (fourth geom))
		 (redraw = t)))
      (if redraw (progn (redraw group) (redraw = nil)))
      (sel = (menu-set-select *wio-menu-set* redraw))
      (case (menu-name sel)
	(command
	  (case (port sel)
	    (quit          (done = t))
	    (done          (undef = (cgalldef (cigroup group)))
			   (if undef
			       (progn
				 (gp-msg (window group)
			           (format nil "*** Undefined: ~A" undef))
				 (redraw = t))
			       (progn (done = t)
                                      (res = (gp-finish group)))) )
	    (redraw        (redraw = t))
	    (move          (move (conns group)))
	    (delete    )
	    (make-view     (ap-server-make-view (window group)))
	    (program       (gp-select-program group)
			   (setq *glspecfnscompiled* nil)
			   (redraw = t))
	    (variable      (gp-variable group))
;           "Grok Data" handled directly by dag-grok-data
	    (documentation (ap-server-documentation (window group)))
	    (language      (gp-language group))
            (t     (if (fboundp (port sel))
                       (funcall (port sel) group)
                       (error "Bad selection ~A in gp-interaction"
                              (port sel)) ) ) ))
	(newm1 (setq sents (port sel))
	       (datadef = (ap-read-from-strings (window group) sents))
	       (if datadef (gp-define-data group datadef)))
	else (if ((menu-name sel) != 'background)
		 (progn (redraw = t)
			(gp-slot-click group sel) ) )))
    res ))

; 24 Dec 03; 31 Dec 03; 16 Jun 04; 18 Jun 04; 20 Jun 04; 21 Jun 04
; 23 Jun 04; 25 Jun 04; 03 Aug 04; 13 Aug 04; 18 Aug 04; 11 Oct 05; 13 Apr 06
; 08 Apr 08
(gldefun gp-init-window ((w window))     (result gp-group)
  (let (conns language)
    (open w)
    (clear w)
    (set-font w *wio-font*)
    (wio-init-menus w '(("Quit"          . quit)
			("Done"          . done)
			("Redraw"        . redraw)
			("Move"          . move)
;			("Delete"        . delete)
			("Make a View"   . make-view)
			("Program"       . program)
			("Variable"      . variable)
                        ("Grok Data"     . dag-grok-data)
			("Documentation" . documentation)
			("Language"      . language)) )
    (add-item *wio-menu-set* 'newm1 nil
	      (editmenu-create (- *wio-window-width* 140) 48 nil w
		     10 (- *wio-window-height* 100) t t *wio-font* t))
    (conns = (menu-conns-create *wio-menu-set*))
    (a gp-group with name = (glmkatom 'gp-group)
                     conns = conns
		     language = *gp-language*) ))

; 24 Dec 03; 23 Jun 04; 25 Jun 04; 03 Aug 04; 04 Oct 05; 17 Apr 08
(gldefun gp-group-redraw ((group gp-group))
  (let ()
    (clear (window group))
    (draw (conns group))
    (if (startflag group)
	(progn (window-print-lines (window group)
		 '("Select Language, then click Program.")
		 6 (- *wio-window-height* 14))
	       ((startflag group) = nil)))
    (for item in (menu-items (menu-set (conns group)))
	 when (and (sym item) (get (sym item) 'spec))
	 (gp-show-box group (menu-name item)) )
    (force-output (window group)) ))

; 13 Aug 04
(gldefun gp-language ((group gp-group))
  (let (language)
    (language = (or (menu '(("Lisp" . lisp) ("C" . c) ("C++" . c++)
			    ("Java" . java) ("Pascal" . pascal))
			  "Output Language:")
		    'lisp))
    (*gp-language* = language)
    ((language group) = language) ))

; 24 Dec 03; 26 Dec 03; 25 Jun 04; 12 May 08
; a crude hack to use the existing code
(gldefun gp-define-data ((group gp-group) datadef)
  (let (oldtypes newtypes)
    (setq *ap-server-language* (language group))
    (setq oldtypes (gp-input-types group))
    (setq *ap-server-types* oldtypes)
    (ap-server-define-data (window group) datadef nil
			   #'gp-erase-bottom (gp-echo-y))
    (if (setq newtypes (set-difference *ap-server-types* oldtypes))
        (setq *gp-types (append newtypes oldtypes)) ) ))

; 24 Dec 03; 16 Jun 04; 21 Jun 04; 25 Jun 04; 28 Jul 04; 30 Jul 04; 02 Aug 04
; 03 Aug 04; 04 Aug 04; 05 Aug 04; 03 Nov 05; 12 Dec 06; 16 May 07; 06 May 08
; 13 May 08; 14 May 08; 15 May 08; 31 Aug 09; 01 Sep 09; 01 Nov 10; 07 Oct 11
; 21 Apr 14
(gldefun gp-select-program ((group gp-group))
  (let ((pgm cgspec) inputs grp boxnm boxt inps inp frmls formal)
    (gp-erase-top (window group))
    (gp-msg (window group) "Select program")
    (pgm = (gp-menu *gp-programs*))
    (when pgm
      (boxnm = (gp-make-menu group pgm nil))
      (redraw group)
      (inputs = (gp-get-inputs group pgm))
      ((cigroup group) = (instcigroup pgm inputs))
      (set-sym group boxnm (params (first (insts (cigroup group)))))
      (inps = inputs)
      (frmls = (formals pgm))
      (while (inp = (pop inps))
        (when (formal = (pop frmls))
         ; 2nd arg below was (type (first (args (first (comps pgm)))))
          (boxt = (gp-make-menu group 'input 'out (name inp)
                              (item-position (menu-set (conns group))
                                             (a menu-port
                                                port (port (to formal))
                                                menu-name boxnm))))
          (new-conn (conns group) boxt 'out boxnm (port (to formal)) ) ) ) )
         ; last arg above was (type (first (args (first (comps pgm)))))
    (gp-erase-message (window group))
  ))

; 13 Apr 06; 22 Nov 06
(gldefun gp-variable ((group gp-group))
  (let (varname box)
    (varname = (gp-get-input group "Enter variable name:"))
    (box = (gp-make-menu group varname 'both))
    (gp-erase-message (window group))
    (redraw group)
  ))

; 16 Jun 04; 18 Jun 04; 02 Aug 04; 04 Aug 04; 05 Aug 04; 16 Aug 04; 01 Feb 07
; 22 May 09; 11 Feb 10
; process click on a box               ; (portname boxname button)
(gldefun gp-slot-click ((group gp-group) (sel menu-selection))
  (let ((boxname (menu-name sel)) (portname (port sel)) int res
	(inst cinst) (newinst cinst) box choices choice cmd nms nm)
    (inst = (sym (named-item (conns group) boxname)))
    (if portname
        (if (eql (button sel) 1)      ; left click
            (progn
              (int = (that (interfaces (spec inst)) with
                           (and (name == portname)
                                (direction == 'uses))))
              (if (and int       ; test for interface, vs. fnspec
                       (or (null (findconns (cigroup group) inst portname))
                           (gp-arity-multi (spec inst) portname)))
                  (newinst = (ciselect inst portname))
                  (if (eq portname 'in)  ; ??? is this ever true?
                      (progn (choices = (for x in (fnspecs (spec inst))
                                          when (and (consp (howspec x))
                                                    (member (car (howspec x))
                                                            '(prop default)))
                                          collect (name x)))
                             (choice = (gp-menu choices))
                             (if choice
                                 (ciredoprop inst choice)))
                      (if (choice = (fnspec (spec inst) portname))
                          (ciredoprop inst portname)) ))
              (if newinst
                  (progn (box = (gp-make-menu group (spec newinst) 'in))
                         (new-conn (conns group) boxname portname box 'in)
                         (set-sym group box newinst)
                         (redraw group) )) )
            (progn      ; other than left click
              ))
        (progn (cmd = (menu '(show edit redo)))
               (when cmd
                 (nms = (cifnspecnames inst))
                 (nm = (menu (if (cmd == 'show)
                                 (cons "All" (append (citypenames inst) nms))
                                 nms)))
                 (case cmd
                   (show  (gp-show group inst nm))
                   (edit  (gp-edit group inst nm))
                   (redo  (ciredoprop inst nm))
                   (t nil))) ) ) ))

; 22 May 09
; process show command
(gldefun gp-show ((group gp-group) (inst cinst) (nm symbol))
  (let (res)
    (if (and (stringp nm) (string= nm "All"))
        (progn (pprint (cons inst (symbol-plist inst))) (terpri))
        (if (res = (arg inst nm))
            (progn (prin1 res) (terpri)) ) ) ))

; 22 May 09; 11 Feb 10; 12 Feb 10
; process edit command
(gldefun gp-edit ((group gp-group) (inst cinst) (nm symbol))
  (let (orig res new editwindow)
    (orig = (arg inst nm))
    (when (and (consp orig) (eq (car orig) nm))
      (editwindow = (window-create 600 250 "Edit Window"))
      (res = (window-edit editwindow 20 20 560 210
                          (gp-edit-strings orig) ) )
      (if (and (consp res) (stringp (first res)))
          (progn (setq new (read-from-string
                             (apply #'concatenate (cons 'string res))))
                 (if (and (consp new) (eq (car new) nm))
                     (setf (rest orig) (rest new)) ) ) )
      (window-destroy editwindow)  ) ))

; 11 Aug 10
(defun gp-edit-strings (x)
  (let (str res (last 0))
    (setq str (prin1-to-string x))
    (dotimes (i (length str))
      (if (or (char= (char str i) #\Newline)
              (char= (char str i) #\Return))
          (progn (push (subseq str last i) res)
                 (setq last (1+ i)) ) ) )
    (if (> (length str) last)
        (push (subseq str last (length str)) res) )
    (nreverse res) ) )

; 25 Jun 04; 02 Aug 04; 03 Aug 04; 05 Aug 04; 28 Oct 04; 04 Nov 04; 16 Nov 04
; 18 Nov 04; 17 Apr 08; 24 Apr 08; 30 May 08; 24 Dec 08; 02 Sep 09; 02 Feb 10
; 04 Feb 10
(gldefun gp-finish ((group gp-group) &optional fnname stream)
  (let (fndef args)
    (or stream (setq stream t))
    (princ "<PRE>" stream) (terpri stream)  ; html preformatted text command
    (or fnname
        (fnname = (glmkatom (or (fn (first (comps (spec (cigroup group)))))
                                'gpfn))))
    (cgspecialize (cigroup group) fnname)
    ((synopsis (cigroup group)) = (cigroup-synopsis (cigroup group)) )
    (if (fboundp fnname)
	(progn 
          (args = (mapcar #'car (sources group)))
          (fndef = (symbol-function fnname))
          (if (and (consp fndef)               ; put args in right order
                   (eq (first fndef) 'lambda)
                   (not (equal (second fndef) args))
                   (set-equal (second fndef) args))
              (setf (second fndef) args))
          (gp-outputtypes group stream)
;          (dolist (tp *vac-newtypes*)
;		 (lcrecordtrans tp (language group)) (terpri stream))
          (dolist (fn *glspecfnscompiled*)
            (ap-server-print-fn fn stream (language group))
            (terpri stream) )
          (if (eq *gp-language* 'java)
              (for fnname in *glspecfnscompiled*
                   when (and (setq specfn
                                   (caar (get fnname 'glspecialization)))
                             (get specfn 'java-static))
                   (setf (get fnname 'java-static) t)))
          (ap-server-print-fn fnname stream (language group)) ) )
      ; close html preformatted text
    (terpri stream) (princ "</PRE>" stream) (terpri stream) (terpri stream)
    fnname ))

; 02 Feb 10
; output types needed by generated programs, in order so that
; needed types are defined first, and including types that are
; referenced by others.
(gldefun gp-outputtypes ((group gp-group) stream)
  (let (needed done (progress t) tp extras need diff)
    (needed = *vac-newtypes*)
    (while progress
      (progress = nil)
      (extras = nil)
      (tp = (some #'(lambda (x)
                      (need = (gp-depends x done))
                      (if (null need)
                          x
                          (progn (setq extras (union extras need))
                                 nil)))
                  needed))
      (diff = (set-difference extras needed))
      (when diff
        (needed = (union needed diff))
        (progress = t))
      (when tp
        (lcrecordtrans tp (language group))
        (terpri stream)
        (push tp done)
        (needed = (remove tp needed))
        (progress = t)) )
    (if needed (format t "Vars still not output: ~A~%" needed)) ))

; 02 Feb 10; 03 Feb 10; 04 Feb 10
; find type dependencies, other than basic and members of except
(gldefun gp-depends (type except)
  (gp-dependsb (glxtrtypeb type) (cons type except)))

(gldefun gp-dependsb (type except)
  (let (new)
    (if (symbolp type)
        (new = (list type))
        (if (consp type)
            (new = (if (member (car type) *gltypenames*)
                       (if (eq (car type) 'crecord)
                           (cons (cadr type)
                                 (mapcan #'(lambda (x)
                                             (gp-dependsb x except))
                                         (cddr type)))
                           (mapcan #'(lambda (x) (gp-dependsb x except))
                                   (cdr type)))
                       (gp-dependsb (cadr type) except)))
            (error "Bad type ~A~%" type)))
    (subset #'(lambda (x) (not (or (glbasictypep (glxtrtypeb x))
                                   (glpointerp x)
                                   (member x except))))
            new) ))

; 24 Dec 03
; Menu select from list, or select automatically if only one choice
(defun gp-menu (lst)
  (if (consp lst)
      (if (null (cdr lst))
	  (car lst)
	  (menu lst) ) ) )

; 24 Dec 03; 03 Aug 04
(gldefun gp-msg ((w window) (str string))
  (let ()
    (gp-erase-message w)
    (wio-echo w str (gp-echo-y)) ))

; 26 Dec 03; 18 Aug 04
(defun gp-echo-y () (- *wio-window-height* 120))

; 03 Aug 04
(gldefun gp-erase-message ((w window))
  (erase-area-xy w 2 (- (gp-echo-y) 2)
		     (- *wio-window-width* 136) (+ (gp-echo-y) 10)) )

; 03 Aug 04
(gldefun gp-erase-top ((w window))
  (erase-area-xy w 6 (- *wio-window-height* 20)
		     (- *wio-window-width* 136) (- *wio-window-height* 2)) )

; 26 Dec 03
(gldefun gp-erase-bottom ((w window))
  (erase-area-xy w 2 (- *wio-window-height* 110)
		     (- *wio-window-width* 136) 28) )

; 26 Dec 03; 19 Jan 04; 28 Jul 04; 02 Aug 04; 03 Aug 04; 20 Aug 04; 22 Nov 06
; 01 Nov 07; 24 Apr 08; 14 May 08; 23 May 08; 31 Aug 09
; cf. conn-menu
; Make a menu and add it to the group.  Returns box name
; kind = kind of box; some boxes may have their own picmenu and drawing
; where = in, out, both, nil for kind of arg buttons a box should have
(gldefun gp-make-menu ((group gp-group) (kind symbol) (where symbol)
                       &optional (nm symbol) (target vector))
  (result menu)
  (let ((w (window group)) pmspec pm fnname boxwidth nargs boxheight
	boxname label buttons)
    (boxname = (conn-uniquename (or nm kind)))
    (label = (or nm kind))
    (if (pmspec = (picmenu-spec kind))
	(pm = (picmenu-create-from-spec
	        (copy-tree pmspec)
		(unless (get kind 'picmenu-nobox)
		        (gp-menu-title kind))
		w 0 0 t t (not (get kind 'picmenu-nobox))))
        (progn
	  (fnname = (intern (concatenate 'string "GP-DRAW-"
					 (symbol-name label))))
	  (if (gp-accumulator? kind)
	      (progn (boxwidth = (+ 105 (string-width w (stringify label))))
		     (nargs = (or (cinargs kind) 1))
		     (boxheight = 48))
	      (progn (boxwidth = (+ 25 (string-width w (stringify label))))
		     (nargs = (or (cinargs kind) 1))
		     (boxheight = (+ 24 (* 12 (max 0 (- nargs 2))))) ) )
	  (if (and (gp-storage? kind) (eq where 'in))
	      (where = 'top))
	  (unless (and (fboundp fnname) (symbol-function fnname))
	    (setf (symbol-function fnname)
		  `(lambda (w x y)
		    (gp-draw-op-box w x y ,(- boxwidth 4) ,boxheight ',label))))
          (buttons = (case where
			 (in (list (list 'in (list 2 (truncate boxheight 2)))))
			 (top (list (list 'in (list (truncate boxwidth 2)
						    (- boxheight 2)))))
			 (out (list (list 'out
					  (list (- boxwidth 2)
						(truncate boxheight 2)))))
                         (both (list (list 'in (list (truncate boxwidth 2)
                                                     (- boxheight 2)))
                                     (list 'out (list (truncate boxwidth 2) 2)) ))
			 (t (append (conn-input-pos nargs)
				    (list (list 'out
						(list (- boxwidth 2)
						      (truncate boxheight 2))))))))
	  (pm = (picmenu-create-from-spec
		 (list 'picmenu-spec boxwidth boxheight buttons
    ;   (gp-picmenu kind where boxwidth boxheight)
		       t fnname *wio-font*)
		 nil w 0 0 t t nil)) ))
    (add-item (conns group) boxname kind pm)
    (if (and target buttons)
        (menu-reposition-line pm (cadar buttons) target)
        (menu-reposition pm))
    boxname))

; 26 Dec 03
(gldefun gp-menu-title ((kind symbol)) (symbol-name kind))

; 28 Jul 04
(gldefun gp-group-sym ((group gp-group) (name symbol))
  (sym (named-item (conns group) name)) )

; 28 Jul 04
(gldefun gp-group-set-sym ((group gp-group) (name symbol) val)
  ((sym (named-item (conns group) name)) = val) )

; 02 Aug 04
; Test if op is the name of an accumulator box
(defun gp-accumulator? (op) (member op (cioffers 'accumulator)))

; 20 Aug 04
; Test if op is the name of a storage box
(defun gp-storage? (op) (member op (cioffers 'storage)))

; 02 Aug 04; 03 Aug 04
(defun gp-draw-op-box (w x y boxwidth boxheight op)
  (window-draw-box-xy w (+ x 2) (+ y 0) boxwidth boxheight)
  (window-printat-xy w op (- (+ x boxwidth)
			     (+ 12 (window-string-width w (stringify op))))
		          (+ y (- (truncate boxheight 2) 5)))
  (window-force-output w) )

; 02 Aug 04; 03 Aug 04; 05 Aug 04; 26 Aug 04; 29 Jan 07
; Show summand and test for an accumulator box
; Format of a property spec is: (name offset suffix)
(gldefun gp-show-box ((group gp-group) (boxnm symbol))
  (let ((item (named-item (conns group) boxnm)) boxwidth boxheight
	sprop (inst cinst) specs offs dx dy)
    (inst = (sym item))
    (boxwidth = (picture-width (menu item)))
    (boxheight = (picture-height (menu item)))
    (specs = (or (get (gp-kind (sym item)) 'gpdisplayspecs)
		 '((summand (4 4)) (test (0 -12) " ?"))))
    (dolist (spec specs)
      (sprop = (arg inst (car spec)))
      (offs = (cadr spec))
      (dx = (if (> (car offs) 0)
		(car offs)
	        (if (= (car offs) 0)
		    (- (truncate boxwidth 2) 32)  ; center
		    (+ boxwidth (car offs)))))
      (dy = (if (> (cadr offs) 0)
		(cadr offs)
	        (if (= (cadr offs) 0)
		    (- (truncate boxheight 2) 32)  ; center
		    (+ boxheight (cadr offs)))))
      (if (and (choice sprop)
	       (not (eq (source sprop) 'default)))
	  (window-printat-xy (window group)
			     (gp-fix-strb (string-limit 
					    (gp-fix-str (choice sprop)) 8)
					  (caddr spec))
			     (+ (parent-offset-x (menu item)) dx)
			     (+ (parent-offset-y (menu item)) dy)) ) ) ))

; 03 Aug 04
(gldefun gp-kind ((inst cinst)) (spec inst))

; 04 Aug 04
; Check whether a port has multiple arity
(gldefun gp-arity-multi ((spec cspec) (portname symbol))
  (let (intfc itm typspec)
    (and (intfc = (interfaced spec portname 'uses))
	 (itm = (that (items intfc)
		      with name == portname and direction == 'in))
	 (typspec = (type spec (typename itm)))
	 (eq (arity typspec) '*)) ))

; 05 Aug 04
(defun gp-fix-str (s)
  (if (consp s)
      (if (consp (car s))
	  (gp-fix-str (car s))
	  s)
      s))

; 26 Aug 04
(defun gp-fix-strb (s suffix)
  (if (stringp suffix)
      (concatenate 'string (stringify s) suffix)
      (stringify s)))

; 05 Aug 04; 10 Feb 05; 03 Nov 05; 17 Apr 08; 12 May 08; 13 May 08; 23 May 08
; 14 Apr 09; 11 Feb 10
; Get inputs for a program to be constructed.
; Sources are used if available
(gldefun gp-get-inputs ((group gp-group) (gspec cgspec))
         (result (listof glnametype))
  (let (fargs srcs nm src args srcnt tp tps pair acttp res)
    (if (sources group)
        (progn
          (fargs = (formals gspec))
          (srcs = (sources group))
          (dolist (farg fargs)
            (nm = (caar farg))
      ; it would be good to filter the possible sources here,
      ; but we don't have that info in good form for iterate-accumulate.
            (if srcs
                (if (cdr srcs)
                    (progn (src = (menu (mapcar #'car srcs)
                                        (format nil "Specify arg for ~A" nm)))
                           (if src
                               (progn (srcnt = (assoc src srcs))
                                      (args = (append args (list srcnt)))
                                      (srcs = (remove srcnt srcs)))))
                    (args = (append args (list (pop srcs)))))))
          (append args srcs))
        (progn (for formal in (formals gspec)
                    (tp = (port (to formal)))
                    (or (pair = (assoc tp tps))
                        (progn (acttp = (gp-get-input-type group
                                                      (stringify tp)))
                               (pair = (list tp acttp))
                               (push pair tps)) )
                    (push (list (glgensym (port (from formal)))
                                (second pair)) res))
               (reverse res) ) ) ))

; 05 Aug 04; 10 Feb 05; 03 Nov 05; 17 Apr 08; 12 May 08; 14 Apr 09; 10 Feb 11
(gldefun gp-get-input-type ((group gp-group) (typename string))
  (let (type)
    (type = (menu (cons 'type-in
                        (gp-language-filter (gp-input-types group)
                                            *gp-language*))
                  (or typename "Input Type")))
    (if (or (null type) (eq type 'type-in))
	(progn
	  (setq type (gp-get-input group
                       (concatenate 'string "Enter "
                                    (or typename "type") ": " )))
	  (pushnew type *gp-types* :test #'equal)))
    type))

; 12 May 08
(gldefun gp-input-types ((group gp-group))
   (union (for source in (sources group) collect (type source))
          *gp-types*)) )

; 03 Nov 05; 30 Jan 07
(gldefun gp-get-input ((group gp-group) (msg string))
  (let (mx my mw str)
    (mx = (message-area-x group))
    (my = (message-area-y group))
    (mw = (string-width (window group) msg))
    (erase-area-xy (window group) (+ mx mw) (- my 4) 100 14)
    (printat-xy (window group) msg mx my)
    (str = (input-string (window group) "" (mx + mw + 10) (my - 4)
			 (- (width (window group)) 400)))
    (ap-read-from-strings (window group) (list str)) ))

; 03 Nov 05; 10 Feb 11
(defun gp-language-filter (lst language)
  (let (str res)
    (setq res (subset #'(lambda (x)
                          (xor (eq language 'lisp)
			       (and (consp (setq str (car (glstr x))))
				    (eq (car str) 'crecord))))
                      lst) )
    (mapcar #'(lambda (x)
                (if (consp x)
                    (cons (stringify x) x)
                    x))
             res) ))

; 30 Oct 07
; Compute positions for buttons
; n     = number of buttons
; width = total width of box
; other = other coordinate
; horiz = t for horizontal (else vertical)
; vars  = variable names
(defun gp-button-pos (n width other horiz vars)
  (let (delta offset res)
    (setq delta (truncate width n))
    (setq offset (truncate delta 2))
    (dotimes (i n)
      (push (list (pop vars)
                  (if horiz (list offset other) (list other offset)))
            res)
      (incf offset delta))
    (reverse res) ))

; 01 Nov 07
; Make button list for a picmenu
; spec  = symbol that has a cspec definition
; where = in, top, out, both
(defun gp-picmenu (spec where boxwidth boxheight)
  (let (inargs outargs)
    (setq inargs (or (ciinterfacenames spec 'in) '(in)))
    (setq outargs (or (ciinterfacenames spec 'out) '(out)))
    (or where (setq where 'both))
    (case where
      (in (gp-button-pos (length inargs) boxheight 2 nil inargs))
      (top (gp-button-pos (length inargs) boxwidth (- boxheight 2) t inargs))
      (out (gp-button-pos (length outargs) boxheight (- boxwidth 2) nil outargs))
      (both (append (gp-picmenu spec 'in boxwidth boxheight)
                    (gp-picmenu spec 'out boxwidth boxheight))) ) ))

; 27 Sep 11
; add a task to the ready queue
; priority = 0..9, 0 is highest
(defun gp-addtask (task priority)
  (setf (aref *gp-taskpq* priority)
        (nconc (aref *gp-taskpq* priority) (list task)) ) )

; 27 Sep 11
; remove next task from the ready queue
(defun gp-nexttask (&optional (priority 0))
  (let (task)
    (while (and (null task) (< priority 10))
      (if (aref *gp-taskpq* priority)
          (setq task (pop (aref *gp-taskpq* priority)))
          (incf priority) ) )
    task))

; 30 Sep 11
(defun gp-executetask (&optional (priority 0))
  (let (task)
    (setq task (gp-nexttask priority))
    (if task
        (apply (first task) (rest task)) ) ))

; 30 Sep 11
; see if there is any task up to priority; returns priority
(defun gp-anytask (priority)
  (let (res)
    (dotimes (i (1+ (min 9 priority)) res)
      (if (aref *gp-taskpq* i) (setq res i)) ) ))
