; lisptoc.lsp             Gordon S. Novak Jr.        ; 28 Apr 11

; Copyright (c) 2011 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; 09 Jan 97; 23 Jan 97; 28 Jan 97; 29 Jan 97; 04 Feb 97; 03 Mar 97; 17 Apr 97
; 29 Jul 97; 20 Mar 98; 03 Jan 00; 18 Sep 01; 28 Feb 02; 22 Oct 02; 21 Nov 02
; 10 Dec 02; 23 Oct 03; 09 Dec 03; 25 Mar 04; 02 Sep 04; 14 Oct 04; 19 Oct 04
; 26 Oct 04; 02 Nov 04; 04 Nov 04; 11 Nov 04; 12 Nov 04; 15 Nov 04; 16 Nov 04
; 17 Nov 04; 19 Nov 04; 20 Sep 06; 18 Feb 08; 08 Dec 09; 09 Dec 09; 06 Jan 10
; 08 Jan 10; 11 Jan 10; 26 Jan 10; 21 Apr 10; 22 Apr 10

; Conversion from Lisp to other languages -- obviously the wrong direction!

; These programs only convert some Lisp forms; it may be necessary
; to make the Lisp source be in a form that can be translated.

; Known problems:
; 1. returning a value from a let, as in (print (pop x)), (f (let ...)).
;    A LET inside AND/OR is even more problematic since AND and OR
;    can quit early.
; 2. (get x 'lcprintname) causes problems when switching between Java and C.
;    The name (capitalized wrong if language is switched) is still stored.
; 3. Nested if: (if (if ...) ...)
; Types of some let glvar's are not being saved.

; (ltoc-file <filename>) converts file and prints to standard output.
; (gltoc 'fn)            compiles, converts, prints the function fn.

(proclaim '(special *spaces*))

(defvar *lcindent* 2)      ; spaces to indent for nested statements
(defvar *lisptoc-output-stream* t)
(defvar *lclanguage* 'c)   ; target language
(defvar *lcfnname*)        ; name of the current function
(defvar *lcvartypes*)      ; alist of var types
(defvar *lcbracket*)       ; t after printing a bracketed group { ... }
(defvar *lccolumn* 0)      ; current printing column, starting with 1
(defvar *lcmaxwidth* 76)   ; max printing width
(defvar *lcifexpr* nil)    ; t to expand (if p x y)  as  p ? x : y
(defvar *lcnewvars* nil)   ; new local variables created in translation

(defmacro lcpointer (x) `(get ,x 'lcpointer))  ; name to use in target language

; 07 Apr 92; 10 Apr 92; 01 Oct 92; 16 Apr 93; 22 Apr 93; 17 Nov 93; 27 Apr 95
; 17 Dec 96; 06 Jan 97; 07 Jan 97
; Translate a glisp function to other languages
(defun gltolang (fn language &optional (stream t))
  (let (expr)
    (setq *lisptoc-output-stream* stream)
    (setq *lclanguage* (or language 'c))
    (setq expr (or (glcompiled fn)
		   (and (gloriginalexpr fn) (glcc fn) (glcompiled fn))
		   (and (fboundp fn)
			(consp (symbol-function fn))
			(cons 'lambda (cddr (symbol-function fn))))))
    (unless (and (consp (caddr expr))
		 (member (caaddr expr) '(let prog)))
      (setq expr (list (car expr) (cadr expr)
		       (cons 'let (cons '() (cddr expr))))))
    (fntolang fn (glfnresulttype fn) expr (gltypesdefined fn)) ))

; 08 Apr 92; 16 Apr 93; 20 Apr 93; 22 Apr 93; 27 Apr 93; 29 Apr 93; 19 Mar 93
; 05 Nov 93; 09 Nov 93; 27 Apr 95; 18 Dec 96; 19 Dec 96; 20 Dec 96; 02 Jan 97
; 06 Jan 97; 07 Jan 97; 09 Jan 97; 28 Jan 97; 29 Jan 97; 30 Jan 97; 28 Jul 97
; 02 Nov 04; 28 Apr 11
; Translate a function definition.  Expr is (xxxx (args) code...)
; Note: if fntype is nil, assumes no result is returned.
(defun fntolang (fnname fntype expr vartypes)
  (let (code tcode (firstarg t) specfn static)
    (setq specfn (caar (get fnname 'glspecialization)))
    (if (and specfn (setq static (get specfn 'java-static)))
	(setf (get fnname 'java-static) t))
    (setq *lcfnname*
	  (lcfixname (or (and static
			      (eq *lclanguage* 'java)
			      (get specfn 'lcshortname))
			 fnname)
		     'fn))
    (setf (get fnname 'lcprintname) *lcfnname*)
    (setq *lcvartypes* vartypes)
    (setq *lccolumn* 1)
; print the function/procedure header
    (case *lclanguage*                               ; print function name
      ((c c++) (lcprintypeb fntype *lcfnname* 'axolotl nil nil t)
               (lcspaces 1)
	       (lcprinc "("))
      (java (lcprintypeb fntype *lcfnname* 'axolotl t static)
	    (lcspaces 1)
	    (lcprinc "("))
      (pascal (lcprinc (if fntype "function " "procedure "))
	      (lcprinc *lcfnname*)
	      (if (cadr expr) (lcprinc "(")) ) )
    (vars-to-lang (case *lclanguage*
		 ((c c++ pascal) (second expr))
		 (java (if static
			   (cadr expr)
			   (cdadr expr))))  ; first arg is implicit for method
	       *lcindent* t)                ; print args and types
    (case *lclanguage*
      ((c c++ java) (lcprinc ")"))
      (pascal (if (cadr expr) (lcprinc ")"))
	      (when fntype (lcprinc " : ")
		(lcprintype (glxtrtypeb fntype)))
	      (lcprinc ";") ) )
    (lcterpri)
    (setq code (if (and (consp (caddr expr))
			(member (caaddr expr) '(let prog do))
			(null (cdddr expr)))
		   (caddr expr)
		   (cons 'let (cons '() (cddr expr)))))
    (if fntype (setq code (lcpushcode '(return) code)))
    ; for Java, make first arg a let var and set it to 'this'
    (if (and (eq *lclanguage* 'java) (not static) (consp (cadr expr)))
	(if (gloccurs 'this code)
	    (setq code (cons (car code)
			     (cons (cons (caadr expr) (cadr code))
				   (cons (list 'setq (caadr expr) 'this)
					 (cddr code)))))
	    (setq code (subst 'this (caadr expr) code)) ) )
    (setq *lcnewvars* nil)
    (setq tcode (lctransformtop code))      ; transform code for target language
    (setq tcode (cons (car tcode)
		      (cons (append (cadr tcode) *lcnewvars*) (cddr tcode))))
    (setq tcode (lcptmatch tcode 'lisptoc)) ; transform with patterns
    (setq tcode (cons (car tcode)           ; get rid of nested progn's
		      (cons (cadr tcode)
			    (lcexpandprogntop (cddr tcode) fntype
					      (car tcode)))))
    (case *lclanguage*
      ((c c++ java) (progn-to-lang (list tcode) nil t *lcindent*)
                    (lcspaces 3) (lcshortcomment *lcfnname*))
      (pascal
        (when (cadr tcode)
	  (lcspaces *lcindent*)
	  (lcprinc "var ")
	  (vars-to-lang (cadr tcode) (+ *lcindent* 3) nil)
	  (lcprinc ";")
	  (lcterpri))
	(progn-to-lang (cddr tcode) nil t *lcindent*)
	(lcprinc ";")
	(lcspaces 3)
	(lcshortcomment *lcfnname*)
	(lcterpri) ) ) ))

; 07 Apr 92; 08 Apr 92; 17 Mar 93; 16 Apr 93; 22 Apr 93; 27 Apr 93; 21 May 93
; 18 Dec 96; 19 Dec 96; 02 Jan 97; 07 Jan 97; 17 Apr 97; 29 Jul 97
; Translate a list of vars or (var value) to declaration.
; fnparams = T if this is a parameter list of a function or procedure
(defun vars-to-lang (vars spaces fnparams)
  (let (lasttype type line tp var (origvars vars))
    (setq *spaces* spaces)
    (while vars
      (setq var (car vars))
      (setq type (lcvartype var))
      (if (glunittypep type)
	  (setq type (cadr type)))
      (setq tp (or type 'integer))
      ; crude approx to keep line from being too long
      (if (> (+ *lccolumn* 8 (lcflatsize (stringify var) 0)
		(if (or (eq *lclanguage* 'java)
			(not (equal tp lasttype)))
		    10 0))
	     *lcmaxwidth*)
	  (lcterprisp spaces))
      (case *lclanguage*
	((c c++)
	  (if fnparams
	      (if (not (eq vars origvars)) (lcprinc ", "))
	      (if (and lasttype (not (equal tp lasttype)))
		  (lcprinc "; ")))
	  (lcprintypeb tp (lcfixname var 'var) (if fnparams 'axolotl lasttype))
	  (setq lasttype tp)
	  (setq line t))
	(java
	  (if (and fnparams (not (eq vars origvars)))
	      (lcprinc ", ")
	      (if (and lasttype (not (equal tp lasttype)))
		  (lcprinc "; ")))
	  (lcprintypeb tp (lcfixname var 'var)
		       (if fnparams 'axolotl lasttype))
	  (setq lasttype tp)
	  (setq line t))
	(pascal (if (not (eq vars origvars)) (lcspaces 1))
		(lcprintypeb tp (lcfixname var 'var) lasttype)
		(if (cdr vars) (lcsemicolon))) )
      (setq vars (cdr vars)) ) ))

; 27 Apr 93; 09 Nov 93
; Find the type of a variable.  var is a variable or (v value).
(defun lcvartype (var)
  (let (v tmp)
    (setq v (if (consp var) (car var) var))
    (if (setq tmp (assoc v *lcvartypes*))
	(cdr tmp)
	(if (glispglobalvar v)
	    (glispglobalvartype v))) ))

; 27 Apr 93; 24 May 93; 15 May 94
; Find the type of an expression
(defun lcexprtype (expr)
  (let (type argtypes)
    (if (atom expr)
	(if (numberp expr)
	    (if (floatp expr) 'real 'integer)
	    (if (and expr (symbolp expr))
		(progn (setq type (lcvartype expr))
		       (if (atom type) type
			   (if (eq (car type) 'units)
			       (cadr type)
			       type)))))
	(progn (setq argtypes (mapcar #'lcexprtype (cdr expr)))
	       (case (car expr)
		 ((and or not > >= = eql <= < /= char=) 'boolean)
		 ((+ - * / truncate)
		   (if (member 'real argtypes) 'real 'integer))
		 ((lsquare lsqrt lfpow llpow) 'integer)
		 ((sqrt sin cos tan asin acos atan square flpow) 'real)
		 (expt (first argtypes))
		 (glcfield (and (car argtypes)
				(consp (caddr expr))
				(eq (caaddr expr) 'quote)
				(cadr (assoc (cadr (caddr expr))
					     (gldatanames (car argtypes))))))
		 )) ) ))

; 07 Apr 92; 14 Apr 92; 20 Apr 93; 29 Apr 93; 19 May 93; 18 Dec 96; 19 Dec 96
; 10 Dec 02; 28 Apr 11
; translate a list of statements to target language
(defun progn-to-lang (lst progflg brackets spaces)
  (let ()
    (setq *spaces* spaces)
    (if (and lst (null (cdr lst)) (consp (car lst))
	     (member (caar lst) '(let prog do)))
	(setq brackets nil))
    (lcterprib)
    (when brackets
      (lcspaces spaces)
      (case *lclanguage*
	((c c++ java) (lcprinc "{"))
	(pascal (lcprinc "begin")))
      (lcterpri))
    (while lst
      (setq st (car lst))
      (if st   ; ignore any nil's as statements
	  (statement-to-lang st progflg (+ spaces (if brackets *lcindent* 0))))
      (if (and (eq *lclanguage* 'pascal)
	       (cdr lst))
	  (lcprinc ";"))
      (setq lst (cdr lst)) )
    (when brackets
      (lcterprib)
      (lcspaces spaces)
      (case *lclanguage*
	((c c++ java) (lcprinc "}"))
	(pascal (lcprinc "end")))
      (setq *lcbracket* t)) ))

; 08 Apr 92; 10 Apr 92; 20 Apr 93; 21 Apr 93; 22 Apr 93; 27 Apr 93; 19 May 93
; 09 Nov 93; 11 Nov 93; 18 Dec 96; 19 Dec 96; 20 Dec 96; 26 Dec 96; 27 Dec 96
; 02 Jan 97; 03 Jan 97; 07 Jan 97; 23 Jan 97; 28 Jan 97; 29 Jan 97; 30 Jan 97
; 17 Apr 97; 20 Mar 98; 11 Nov 04; 08 Jan 10; 28 Apr 11
; Translate a statement to target language
(defun statement-to-lang (st progflg spaces)
  (let (recname)
    (setq *spaces* spaces)
    (unless (and (consp st) (member (first st) '(progn malloc-ptr)))
      (if (> *lccolumn* (1+ spaces)) (lcterpri))
      (unless (and progflg (atom st)) (lcspaces (- (1+ spaces) *lccolumn*))))
    (if (atom st)
	(if progflg
	    (progn (lcprinlc st) (lcprinc ":") (lcterpri))   ; label
	    (progn (lcprinc (atom-to-lang st))))
	(case (first st)
	  ((let prog)
	    (case *lclanguage*
	      ((c c++ java) (lcprinc "{"))
	      (pascal (lcprinc "begin")))
	    (lcterpri)
	    (if (second st)
		(case *lclanguage*
		  ((c c++ java)
		    (lcspaces (+ spaces *lcindent*))
		    (vars-to-lang (second st) (+ spaces *lcindent*) nil)
		    (lcsemicolon)
		    (lcterpri))
		  (pascal (lcspaces (+ spaces *lcindent*))
			  (vars-to-lang (second st) (+ spaces *lcindent*) nil)
			  (lcterpri))))
	    (progn-to-lang (cddr st) (eq (first st) 'prog) nil
			(+ spaces *lcindent*))
	    (lcterprib)
	    (lcspaces spaces)
	    (case *lclanguage*
	      ((c c++ java) (lcprinc "}"))
	      (pascal (lcprinc "end")))
	    (setq *lcbracket* t))
	  ((setq setf return)
	    (if (and (eq *lclanguage* 'pascal)
		     (member (first st) '(setq setf))
		     (symbolp (second st))
		     (consp (third st))
		     (eq (first (third st)) 'malloc-ptr))
		(progn (lcprinc "new(")
		       (lcprinc (lcfixname (cadr st) 'type))
		       (lcprinc ")"))
	        (progn
		  (if (and (eq (first st) 'setq)
			   (symbolp (second st))
			   (consp (third st))
			   (eq (first (third st)) 'pop)
			   (eq (glxtrtypec (lcvartype (second st))) 'integer))
		      (setq st (list (first st) (second st)
				     (list 'lccastto 'long (third st)))))
		  (expr-to-lang st 0)
		  (case *lclanguage* ((c c++ java) (lcsemicolon))) ) ) )
	  (if (lcprinc "if ")
	      (paren-expr (second st))
	      (lcterpri)
	      (case *lclanguage*
		(pascal (lcspaces spaces) (lcprinc "then ") (lcterpri)))
	      (statement-to-lang (third st) nil (+ spaces *lcindent*))
	      (when (cdddr st)
		(if (> *lccolumn* (1+ spaces)) (lcterpri))
		(lcspaces spaces)
		(lcprinc "else ")
		(lcterpri)
		(statement-to-lang (fourth st) nil (+ spaces *lcindent*))))
	  (malloc-ptr
	    (case *lclanguage*
	      ((c c++)
	        (lcprinc "(")
                (setq recname (lccpointer (cadr st)))  ; record name  08 Jan 10
                (lcprinc recname)
		(lcprinc " *) malloc(sizeof(")
		(lcprinc recname)
		(lcprinc "))"))
	      ((java)
	        (lcprinc "new ")
		(lcprinc (lcfixname (cadr st) 'type))
		(lcprinc "()") ) ) )
	  (progn (progn-to-lang (cdr st) nil t (+ spaces *lcindent*)))
	  (t (setq *lcbracket* nil)       ; print the lisp or translation.
	     (funcall-to-lang st)
	     (case *lclanguage*
	       ((c c++ java) (unless *lcbracket* (lcsemicolon))) )
	     ) ) ) ))

; 28 Apr 11
; translate a (do ...) to (let () (while ...))
(defun dotowhile (expr)
  (let (vars inits updates)
    (dolist (varlist (cadr expr))
      (push (car varlist) vars)
      (if (cdr varlist)
          (push (list 'setq (car varlist) (cadr varlist))
                inits))
      (if (and (cdr varlist) (cddr varlist))
          (push (list 'setq (car varlist) (caddr varlist))
                updates) ) )
    (cons 'let
          (cons (reverse vars)
                (append (reverse inits)
                        (list (cons 'while
                                    (cons (list 'not (caaddr expr))
                                          (append (cdddr expr)
                                                  (reverse updates)))))
                        (cdr (caddr expr))))) ))

(defvar *exprops*)
(setq *exprops* '((setq = 1)   (or \|\| 2)  (and \&\& 3)
                  (logior \| 4) (logxor \^ 5) (logand \& 6)
		  (= == 7)     (eql == 7) (/= != 7) (char= == 7) (equal == 7)
		  (<= <= 8)    (< < 8)      (>= >= 8)	  (> > 8)      
		  (leftshift << 9) (rightshift >> 9)
                  (+ + 16)      (- - 16)
		  (* * 20)      (/ / 20)    (mod % 20)  (truncate / 20)
		  (lognot \~ 22)
		  ))

(defvar *pascalexprops*)
(setq *pascalexprops*
      '((setq \:= 1)
	(= = 8)    (eql = 8)  (/= <> 8)        (char= = 8) (equal = 8)
	(<= <= 8)  (< < 8)    (>= >= 8)        (> > 8)      
	(+ + 16)   (- - 16)   (or or 16)
	(* * 20)   (/ / 20)   (truncate / 20)  (and and 20) ))

; 13 Oct 92; 30 Dec 92; 31 Dec 92; 16 Apr 93; 20 Apr 93; 29 Apr 93; 01 Oct 93
; 09 Nov 93; 16 Nov 93; 18 Dec 96; 19 Dec 96; 26 Dec 96; 22 Oct 02; 09 Dec 03
; 14 Oct 04; 04 Nov 04
; Translate expression to C.  prec is the precedence outside.
(defun expr-to-lang (expr prec)
  (let (tmp op-prec c-op crflg col)
    (if (atom expr)
	(lcprinc (atom-to-lang expr))
	(if (and (symbolp (first expr))
		 (gloktoeval (first expr))
		 (every #'constantp (rest expr)))
	    (expr-to-lang (eval expr) prec)
	    (if (and (cddr expr)
		     (setq tmp (assoc (first expr)
				      (case  *lclanguage*
					((c c++ java) *exprops*)
					(pascal *pascalexprops*)))))
		(progn (setq c-op (second tmp))
		       (setq op-prec (third tmp))
		       (if (> (+ (lcflatsize expr prec) *lccolumn*)
			      *lcmaxwidth*)
			   (progn (setq crflg t)
				  (setq col *lccolumn*)))
		       (if (< op-prec prec) (lcprinc "("))
		       (expr-to-lang (second expr) op-prec)
		       (dolist (subexpr (cddr expr))
			 (if crflg
			     (progn (lcterpri) (lcspaces (1- col)))
			     (lcspaces 1))
			 (lcprinc c-op)
			 (lcspaces 1)
			 (expr-to-lang subexpr
				    (if (member (first expr) '(- / truncate))
					(1+ op-prec)
					op-prec)))
		       (if (< op-prec prec) (lcprinc ")")) )
		(if (and (null (cddr expr))
			 (setq tmp (assoc (first expr)
					  '((- -) (not !) (lognot \~)))))
		    (progn (lcprinlc (second tmp))
			   (lcprinc "(")
			   (expr-to-lang (second expr) 0)
			   (lcprinc ")") )
		    (if (member (first expr)
			   '(setq if when unless let prog progn malloc-ptr))
			(statement-to-lang expr nil *spaces*)
			(if (eq (first expr) 'glcstring)
			    (case *lclanguage*
			      ((c c++ java) (lcprins (cadr expr)))
			      (pascal (lcprinc (concatenate 'string "'"
							    (cadr expr) "'")))) 
			    (if (and (eq (first expr) 'aref)
				     (symbolp (second expr))
				     (setq tmp (lcvartype (second expr)))
				     (consp tmp)
				     (eq (car tmp) 'arrayof)
				     (lcrecordp (cadr tmp)))
				(progn (lcprinc "(&")
				       (funcall-to-lang expr)
				       (lcprinc ")"))
				(funcall-to-lang expr)))))) ) ) ))


; 07 Apr 92; 27 Apr 92; 20 May 93; 10 Nov 93; 08 Dec 93; 22 Mar 94; 18 Dec 96
; 20 Dec 96; 02 Jan 97; 09 Jan 97; 23 Jan 97; 28 Jan 97; 30 Jan 97; 04 Feb 97
; 18 Sep 01
; general function call
(defun funcall-to-lang (expr)
  (let (fn tmp patres fntrans)
    (setq fn (first expr))
    (if (assoc fn (case *lclanguage*
		    ((c c++ java) *exprops*)
		    (pascal *pascalexprops*)))
	(expr-to-lang expr 0)
	(progn
	  (if (eq fn 'quote)   ; should not happen, but just in case...
	      (setq expr (list fn (princ-to-string (cadr expr)))) )
	  (if (eq fn 'expt)
	      (if (eq (lcexprtype (second expr)) 'integer)
		  (if (and (numberp (third expr))
			   (integerp (third expr))
			   (> (third expr) 0))
		      (if (= (third expr) 2)
			  (setq expr (list 'lsquare (second expr)))
			  (setq expr (cons 'llpow (rest expr))))
		      (if (eq (lcexprtype (third expr)) 'integer)
			  (setq expr (cons 'llpow (rest expr)))
			  (setq expr (cons 'lfpow (rest expr)))))
		  (if (and (eq (lcexprtype (third expr)) 'integer)
			   (eql (third expr) 2)
			   (member *lclanguage* '(pascal)))
		      (setq expr (list 'square (second expr)))
		      (setq expr (cons 'pow (rest expr))) ) ) )
	  (if (and (eq fn 'sqrt)
		   (member *lclanguage* '(c c++))
		   (eq (lcexprtype (second expr)) 'integer))
	      (setq expr (list 'lsqrt (second expr))))
	  (if (and (eq fn 'abs)
		   (eq (lcexprtype (second expr)) 'real))
	      (setq expr (list 'fabs (second expr))))
	  (setq fn (first expr))
	  (if (setq fntrans (assoc fn (case *lclanguage* (java *javafns*))))
	      (progn (setq fn (cadr fntrans))
		     (setq expr (cons fn (cdr expr)))))
	  (if (and (eq fn 'return)
		   (eq *lclanguage* 'pascal)
		   *lcfnname*)
	      (statement-to-lang (list 'setq *lcfnname* (cadr expr))
				 nil *spaces*)
	      (if (and (eq fn 'glcfield)
		       (consp (second expr))
		       (eq (caadr expr) 'aref)
		       (symbolp (cadadr expr))
		       (setq tmp (lcvartype (cadadr expr)))
		       (consp tmp)
		       (eq (car tmp) 'arrayof)
		       (lcrecordp (cadr tmp))
		       (consp (caddr expr))
		       (eq (caaddr expr) 'quote))
		  (pat-to-lang (list (list (cons 'r (cons 'arefrec (cdadr expr)))
					(cons 'f (cadr (caddr expr))))
				  '(r "." f)))
		  (if (setq patres (glptmatchc expr
					       (case *lclanguage*
						 (c      'lisptocout)
						 (c++    'lisptoc++out)
						 (java   'lisptojavaout)
						 (pascal 'lisptopasout))))
		      (pat-to-lang patres)
		      (case *lclanguage* 
			(java
			 (if (symbolp (car expr))
			     (progn
			       (if (get (car expr) 'java-static)
				   (lcprintype
				     (glxtrtypeb
				       (cadar (glarguments (car expr)))))
				   (expr-to-lang (cadr expr) 99)) ; force ()
			       (lcprinc ".")
			       (lcprinlc (lcfixname (first expr) 'fn)))
			     (lcprinc (car expr)))
			 (lcprinc "(")
			 (args-to-lang
			   (if (or (and (symbolp (car expr))
					(get (car expr) 'java-static))
				   (stringp (car expr)))
			       (cdr expr)
			     (cddr expr)))
			 (lcprinc ")"))
			((c c++) (lcprinlc (lcfixname (first expr) 'fn))
				 (lcprinc "(")
				 (args-to-lang (cdr expr))
				 (lcprinc ")"))
			(pascal (lcprinlc (lcfixname (first expr) 'fn))
				(if (cdr expr)
				    (progn (lcprinc "(")
					   (args-to-lang (cdr expr))
					   (lcprinc ")")))) ) ) ) ) ) ) ))

; array subscripts
(defun args-to-lang (args)
  (mapc #'(lambda (arg)
	    (unless (eq arg (first args))
	      (lcprinc ", "))
	    (expr-to-lang arg 0) )
	args) )

; 09 Apr 92; 12 Oct 92; 20 Dec 96; 26 Dec 96; 27 Dec 96; 02 Jan 97
(defun atom-to-lang (atm)
  (let (tmp)
    (if (or (numberp atm)
	    (stringp atm))
	atm
	(if (symbolp atm)
	    (if (setq tmp
		      (assoc atm
		       (case *lclanguage*
			 ((c c++) '((*gltrue* 1) (t 1) (*glfalse* 0)
				    (nil "NULL") (*glnull* "NULL")))
			 (java    '((*gltrue* "true") (t "true")
				    (*glfalse* "false") (nil "null")
				    (*glnull* "null")))
			 (pascal  '((*gltrue* "true") (t "true")
				    (*glfalse* "false") (nil "nil")
				    (*glnull* "nil"))) ) ))
		(cadr tmp)
	        (if (and (eq atm 'pi)
			 (member *lclanguage* '(c c++ pascal)) )
		    pi
		    (lcfixname atm 'var)))
	    (if (characterp atm)
		(if (setq tmp (assoc atm '((#\' "\\'") (#\space " ")
					   (#\newline "\\n") (#\tab "\\t"))))
		    (progn (lcprinc "'") (lcprinc (second tmp)) (lcprinc "'"))
		    (progn (lcprinc "'") (lcprinc atm) (lcprinc "'")))
		(lcprinc atm)))) ))

; Make a negated expression in Lisp
(defun not-to-lang (expr)
  (let (tmp)
   (if (eq (first expr) 'not)
       (second expr)
       (if (setq tmp (assoc (first expr)
			    '((<= >) (>= <) (> <=) (< >=) (eql /=) (= /=)
			      (char= /=) (/= =))))
	   (cons (second tmp) (rest expr))
	   (list 'not expr))) ))

; Parenthesized expression
(defun paren-expr (expr)
  (lcprinc "(")
  (expr-to-lang expr 0)
  (lcprinc ")") )

; 22 Apr 93; 23 Apr 93; 18 May 93; 19 May 93; 21 May 93; 01 Oct 93; 08 Aug 95
; 08 Jan 97; 19 Oct 04; 21 Apr 10; 28 Apr 11
; ******** need to make let vars unique **********
; Transform Lisp code so that it can be printed as C code.
(defun lctransformtop (expr) (lctransform expr t (list nil)))

; expr    = expression to be translated
; stmt    = t if the expression occurs at "statement level"
; varlist = (cons nil vars) where vars are from the containing let.
(defun lctransform (expr stmt varlist &optional restcode)
  (let ()
    (if (consp expr)
	(case (first expr)
	  ((setq setf)
	    (if (and (consp (caddr expr))
		     (eq (caaddr expr) 'glmakecrecord))
		(lctransform (lcrecord (caddr expr) (cadr expr) restcode)
			     stmt varlist)
		(lcpushcode (list (first expr) (second expr))
			    (lctransform (third expr) nil varlist))))
	  (return
	     (lcpushcode (list (car expr))
			 (if (and (consp (cadr expr))
				  (eq (caadr expr) 'glmakecrecord))
			     (lcrecord (cadr expr) nil nil)
			     (cadr expr))))
	  (progn (cons (car expr) (lctransforml (cdr expr) t varlist)))
	  (when (lctransform (list 'if (cadr expr) (lcprognify (cddr expr)))
			     stmt varlist))
	  (unless (lctransform (list 'if (list 'not (cadr expr))
				         (lcprognify (cddr expr)))
			       stmt varlist))
	  ((let prog) (lctransformlet expr stmt varlist))
	  (cond (lctransformcond expr stmt varlist))
	  (glcstring expr)
          (glmakecrecord (lcrecord expr nil nil))
          (do (lctransform (dotowhile expr) stmt varlist restcode))
	  (t (cons (car expr) (lctransforml (cdr expr) nil varlist))) )
	(if (stringp expr)
	    (list 'glcstring expr)
	    expr)) ))

; 18 May 93; 08 Jan 97; 21 Apr 10
; Transform a list l of items
(defun lctransforml (l stmt varlist)
  (if (consp l)
      (maplist #'(lambda (x) (lctransform (car x) stmt varlist (cdr x)))
               l)
      l))

; 08 Aug 95
(defun lctransformcond (expr stmt varlist)
  (lctransform
    (if (rest expr)
	(if (consp (cadr expr))
	    (if (eq (caadr expr) t)
		(lcprognify (cdadr expr))
	        (list 'if (caadr expr) (lcprognify (cdadr expr))
		      (cons 'cond (cddr expr))))))
    stmt varlist) )

; 18 May 93; 19 May 93; 21 May 93; 28 May 93
; Transform a let
; varlist is (nil vars ...) so it can be added to.
(defun lctransformlet (expr stmt varlist)
  (let (vars varsets code newvarlist subs newv)
    (setq code (cddr expr))
    (dolist (v (second expr))
      (if (consp v)
	  (if (and (glvarp (car v))                 ; if a local var is just a
		   (consp (cadr v))                     ; record reference
		   (eq (caadr v) 'glcfield))
	      (setq code (subst (cadr v) (car v) code)) ; substitute for it.
	      (if (and (glvarp (car v))
		       (consp (cadr v))
		       (eq (caadr v) 'glmakecrecord))
		  (progn (push (car v) vars)
			 (push (lctransform (lcrecord (cadr v) (car v) code)
					    t varlist)
			       varsets))
		  (progn (push (car v) vars)
			 (push (lctransform (cons 'setq v) t varlist)
			       varsets))))
	  (push v vars)))
    (if stmt
	(setq newvarlist (cons nil (nreverse vars)))
	(progn (dolist (v vars)
		 (when (member v varlist)
		   (setq newv (lcuniquevar v varlist))
		   (push (cons v newv) subs)
		   (setq v newv))
		 (setf (cdr varlist)
		       (nconc (cdr varlist) (list v))))
	       (setq newvarlist varlist)))
    (when subs
      (setq code (sublis subs code))
      (dolist (sub subs)                 ; ***** but what if type differs?
	(push (cons (cdr sub) (cdr (assoc (car sub) *lcvartypes*)))
	      *lcvartypes*)))
    (setq code (nconc (nreverse varsets) (lctransforml code t newvarlist)))
    (if stmt
	(cons (car expr) (cons (cdr newvarlist) code))
	(lcprognify code)) ))

; 20 Apr 93; 21 Apr 93; 10 May 95; 03 Jan 97; 08 Jan 97; 30 Jan 97; 21 Nov 02
; 23 Oct 03; 19 Oct 04
; Make a C record and fill in initializations of fields.
; e.g. expr = (GLMAKECRECORD 'CVECTOR (LIST (CONS 'X U) '(Y . 7.0))))
(defun lcrecord (expr var restcode)
  (let (recname newvar code inits (initcodeflg t) initcode tmp)
    (setq recname (cadadr expr))
    (setq newvar (or var (glmkatom (string-downcase (symbol-name recname)))))
    (setq inits (mapcan #'(lambda (st)               ; find fields whose inits
			    (if (and initcodeflg     ; should be eliminated
				     (consp st)
				     (eq (car st) 'setf)
				     (consp (cadr st))
				     (eq (caadr st) 'glcfield)
				     (eq (cadadr st) newvar) )
				(list (cadr (caddr (cadr st))))
			        (setq initcodeflg nil)) )
			restcode))
    (setq initcode (caddr expr))
    (if initcode
	(setq code
	      (case (car initcode)
		(list
		  (mapcan
		    #'(lambda (field)
			(case (car field)
			  (quote (lcinitfield (caadr field) (cdadr field)
					      newvar inits) )
			  (copy-tree
			    (if (and (consp (cadr field))
				     (eq (caadr field) 'quote))
				(progn (setq tmp (cadadr field))
				       (lcinitfield (car tmp) (cdr tmp)
						    newvar inits))))
			  (cons (if (and (consp (cadr field))
					 (eq (caadr field) 'quote))
				    (lcinitfield (cadadr field) (caddr field)
						 newvar inits)) )
			  (t (error "lcrecord: unk form in glmakerecord"))))
		    (cdr initcode)) )
		((copy-list cons copy-tree)
		 (mapcan #'(lambda (field)
			     (lcinitfield (car field) (cdr field)
					  newvar inits))
			 (if (member (car initcode) '(copy-list copy-tree))
			     (cadadr initcode)
			     (list (cadadr initcode)))) )
		(t (error "lcrecord: unknown form in glmakerecord")) ) ) )
    (if (not (eq var newvar))
	(setq code (nconc code (list newvar))))
    (push (list 'setq newvar (list 'malloc-ptr recname)) code)
    (if (not (eq var newvar))
	(progn (setq *lcnewvars* (nconc *lcnewvars* (list newvar)))
	       (push (cons newvar (list '^ recname)) *lcvartypes*)))
    (lcprognify code) ))

; 08 Jan 97; 02 Sep 04
; Generate code to initialize a field of a newly created record.
; Generates no code if the field is initialized to a default value
; and it is followed by an assignment to the same field (list inits).
; Returns a list of code for use with mapcan.
(defun lcinitfield (fieldname initval var inits)
  (if (and (member fieldname inits)
	   (member initval '(0 0.0 "" t nil *glnull* *gltrue* *glfalse*)
		   :test #'equal))
      nil
      (list (list 'setf (list 'glcfield var (kwote fieldname))
	(if (stringp initval)
	    (list 'glcstring initval)
	    initval)) ) ) )

; 20 Apr 93; 21 Apr 93; 22 Apr 93; 21 May 93; 28 May 93; 08 Aug 95; 08 Jan 97
; 09 Jan 97; 28 Apr 11
; a lot of this is no longer used ...
; In order to handle Lisp code that returns a value from a nested progn
; computation, push the code that uses the value inside.
; code = code that uses the nested computation as its last arg, e.g. (setq v)
; expr = the rest of the expression, e.g. (progn ... foo)
; returns (progn ... (setq v foo))
(defun lcpushcode (code expr)
  (let (rec rest)
    (if (consp expr)
	(case (car expr)
	  (let    ; test for a setq of a let containing a glmakecrecord
	    (if (and (eq (car code) 'setq)
		     (cadr expr)
		     (consp (caadr expr))
		     (consp (cadr (caadr expr)))
		     (eq (caadr (caadr expr)) 'glmakecrecord)
		     (eq (caaadr expr) (car (last expr))))
		(progn
		  (setq expr (subst (cadr code) (caaadr expr) expr))
		  (setq rec (lcrecord (cadr (caadr expr)) (cadr code) nil))
		  (setq rest
			(mapcan   ; eliminate redundant field inits
			 #'(lambda (s)
			     (if (and (consp s)
				      (eq (car s) 'setf)
				      (consp (cadr s))
				      (eq (caadr s) 'glcfield)
				      (every
				       #'(lambda (x)
					   (and (consp x)
						(eq (car x) 'setf)
						(consp (cadr x))
						(eq (caadr x) 'glcfield)
						(not (equal (caddr (cadr s))
						           (caddr (cadr x))))))
				       (cddr expr)))
				 (list s)))
			 (cddr rec)))
		  (cons (first rec)
			(cons (second rec)
			      (nconc rest (butlast (cddr expr))))) )
		(maplist #'(lambda (l)
			     (if (null (cdr l))
				 (lcpushcode code (car l))
				 (car l)))
			 expr)))
	  ((progn when unless)
	    (maplist #'(lambda (l)
			 (if (null (cdr l))
			     (lcpushcode code (car l))
			     (car l)))
		     expr))
	  (if (if (and (member *lclanguage* '(c c++ java))
		       *lcifexpr*
		       (cdddr expr)
		       (glpurecode (caddr expr))
		       (glpurecode (cadddr expr)))
		  (append code (list expr))
		  (cons 'if
		    (cons (cadr expr)
			  (cons (lcpushcode code (caddr expr))
				(if (cdddr expr)
				    (list (lcpushcode code
						      (cadddr expr)))))))))
	  (cond (cons 'cond
		      (mapcar #'(lambda (x)
				  (maplist #'(lambda (l)
					       (if (null (cdr l))
						   (lcpushcode code (car l))
						 (car l)))
					   x))
			      (cdr expr))))
	  ((setq setf) (list 'progn expr (append code (list (cadr expr)))))
          (do (cons (car expr)
                    (cons (cadr expr)
                          (cons (list (car (caddr expr))
                                      (lcpushcode code (cadr (caddr expr))))
                                (cdddr expr)))) )
	  (t (append code (list expr))))
	(append code (list expr))) ))

; 08 Jan 97
; Test whether code is "pure", i.e. computational
(defun glpurecode (x)
  (or (atom x)
      (and (symbolp (car x))
	   (glpure (car x))
	   (every #'glpurecode (cdr x)) ) ) )

; 22 Apr 93; 19 May 93
; derived from glexpandprogn
; expand a list of code within a progn to eliminate nested progn's.
(defun lcexpandprogntop (lst busy progwd) (lcexpandprogn lst busy progwd))
(defun lcexpandprogn (lst busy progwd)
  (mapcon #'(lambda (x)
	      (let ((busyflg (and busy (case progwd
					 (prog1 (eq x lst))
					 ((progn let) (null (cdr x)))
					 (prog2 (eq x (cdr lst)))
					 (prog nil)
					 (t (error "NOMSG"))))))
		(lcexpandprognitem (car x) busyflg progwd)))
	  lst))

; 22 Apr 93; 19 May 93
; Expand a PROGN item by splicing its contents into the top-level list 
;   when appropriate. Returns a list of the result. 
; derived from glexpandprognitem
(defun lcexpandprognitem (item busy progwd)
  (if (atom item)                       ; eliminate non-busy atomic items. 
      (if (or busy (and item (eq progwd 'prog)))
	  (list item))
      (case (car item)
	(progn                          ; expand contained progns in-line. 
	  (lcexpandprogn (cdr item) busy 'progn))
	((prog let)                     ; expand contained simple progs. 
	  (if (and (null (cadr item))
		   (or (eq (car item) 'let)
		       (and (every #'(lambda (y) (consp y)) (cddr item))
			    (not (gloccurs 'return (cddr item))))))
	      (lcexpandprogn (cddr item) (and busy (eq (car item) 'let))
			     'progn)
	      (list (cons (car item)
			  (cons (cadr item)
				(lcexpandprogn (cddr item) 
					       (and busy (eq (car item) 'let))
					       'progn))))))
	((if while)
	 (list (cons (car item)
		     (mapcar #'(lambda (x) (lcexpanditem x t))
			     (cdr item)))))			  
	(t (list item)) ) ) )

; 19 May 93
; Expand progn's contained within an item if appropriate
(defun lcexpanditem (expr busy)
  (let (res)
    (if (atom expr) expr
	(case (car expr)
	  (let (list (car expr) (cadr expr)
		     (lcexpandprogn (cddr expr) busy 'progn)))
	  (progn
	    (setq res (lcexpandprogn (cdr expr) busy 'progn))
	    (if (and (consp (car res))
		     (eq (caar res) 'let))
		(nconc (car res) (cdr res))
		(lcprognify res)))
	  (t (mapcar #'(lambda (x) (lcexpanditem x t)) expr))))))
	  
; 18 May 93
; Make a progn from statmements if needed.
(defun lcprognify (exprlist)
  (if (cdr exprlist) (cons 'progn exprlist) (car exprlist)))

; 18 May 93
; Test if a variable name is a compiler variable GLVARnnn
(defun glvarp (var)
  (let ((name (symbol-name var)))
    (and (> (length name) 5)
	 (string= (subseq name 0 5) "GLVAR"))))

; 28 May 93
(defun lcuniquevar (v vars)
  (let (newv (n 0))
    (while (or (null newv)
	       (member newv vars))
      (incf n)
      (setq newv (intern (concatenate 'string (symbol-name v)
				      (princ-to-string n)))) )
    newv))

; 07 Apr 92; 08 Apr 92; 20 Apr 93; 18 May 93; 02 Nov 04
; Convert output based on a pattern.  The input is a pair of bindings and
; a pattern.  Strings in the pattern are output directly; variables are
; output from their bindings.
(defun pat-to-lang (pair)
  (let (tmp)
    (dolist (x (second pair))
      (if (stringp x)
	  (lcprinc x)
	  (if (symbolp x)
	      (if (or (setq tmp (assoc x (first pair)))
		      (setq tmp (assoc x '((*glnull* . *glnull*)))))
		  (expr-to-lang (cdr tmp) 0)
		  (error "Unbound var ~A in pattern ~A~%" x pair))
	      (if (consp x)
		  (expr-to-lang (sublis (first pair) x) 0)
		  (if (numberp x)
		      (lcprinc x)
		      (if (eq x #\Return)
			  (lcterpri)
			  (if (eq x #\Tab)
			      (incf *spaces* *lcindent*)
			      (error "Bad pattern element ~A in pattern ~A~%"
				     x pair))))))) )  ))

; 06 Jan 97
; Use a special print name for a symbol if it is defined
(defun lcprintname (sym)
  (or (and (symbolp sym)
	   (get sym 'lcprintname))
      sym))

; 07 Apr 92
; Print to output stream
(defun lcsemicolon ()
  (princ ";" *lisptoc-output-stream*)
  (incf *lccolumn*))

(defun lcprinc (x &optional col)
  (let (l)
    (setq x (stringify x))
    (setq l (length x))
    (if (and col (> (+ *lccolumn* l) *lcmaxwidth*))
	(progn (lcterpri) (lcspaces col)))
    (princ x *lisptoc-output-stream*)
    (incf *lccolumn* l)))

(defun lcterpri ()
  (terpri *lisptoc-output-stream*)
  (setq *lccolumn* 1))

; terpri if not already at column 1
(defun lcterprib () (if (> *lccolumn* 1) (lcterpri)))

(defun lcterprisp (spaces) (lcterpri) (lcspaces spaces))

(defun lcspaces (n)
  (incf *lccolumn* n)
  (while (>= n 8) (princ "        " *lisptoc-output-stream*) (decf n 8))
  (dotimes (i n) (princ " " *lisptoc-output-stream*)))

(defun lcprinlc (x)
  (lcprinc (if (symbolp x)
	       (string-downcase (symbol-name x))
	       x)) )

; Print a string that is supposed to go out with string quotes
(defun lcprins (s)
  (lcprinc "\"")
  (princ s *lisptoc-output-stream*)
  (lcprinc "\"")
  (incf *lccolumn* (+ 2 (length s))) )

; 09 Nov 93; 19 Dec 96; 20 Dec 96; 23 Dec 96; 26 Dec 96; 30 Dec 96; 02 Jan 97
; 06 Jan 97; 09 Jan 97; 24 Jan 97; 04 Mar 97; 17 Apr 97; 16 Nov 04; 07 Jan 10
; Modify a name to fit conventions of target language
; kind = kind of name: fn, type, var
(defun lcfixname (x kind)
  (let (xs l pos)
    (or (and (symbolp x) (get x 'lcprintname))
	(progn
	  (setq xs (string-downcase (stringify x)))
	  (setq l (length xs))
	  (if (and (char= (char xs 0) #\*) (char= (char xs (1- l)) #\*))
	      (setq xs (subseq xs 1 (1- l))))
	  (case *lclanguage*
	    ((c c++) (setq xs (substitute #\_ #\- xs)))
	    (java  ; remove - and numbers at the end
;	     (if (eq kind 'fn)
;		 (while (or (digit-char-p (char xs (1- (length xs))))
;			(char= (char xs (1- (length xs))) #\-))
;		   (setq xs (subseq xs 0 (1- (length xs)))) ))
	     (while (find #\- xs)
	       (setq pos (position #\- xs))
	       (setq xs
		     (concatenate 'string (subseq xs 0 pos)
			          (string-capitalize (subseq xs (1+ pos)))))))
	    (pascal (while (find #\- xs)
		      (setq pos (position #\- xs))
		      (setq xs (concatenate 'string (subseq xs 0 pos)
					    (subseq xs (1+ pos)))) ) ) )
	  (case *lclanguage*
	    ((java c c++)
              (if (eq kind 'type) (setq xs (string-capitalize xs)))))
	  xs) ) ))

; 07 Jan 10
; Fix a record name for use with C
(defun lccpointer (recname)
  (or (lcpointer recname)
      (setf (lcpointer recname) (lcfixname recname 'type)) ) )

; 03 Jan 97; 06 Jan 97
; Make an abbreviation of a name for pointer definition
(defun lcabbrev (sym)
  (let ((xs (stringify sym)) (abbrev "") pos)
    (while (> (length xs) 0)
      (if (digit-char-p (char xs 0))
	  (progn (setq abbrev (concatenate 'string abbrev xs))
		 (setq xs ""))
	  (progn (setq abbrev
		       (concatenate 'string abbrev
				    (if (eq *lclanguage* 'pascal)
					(string-downcase (subseq xs 0 1))
				        (string-upcase (subseq xs 0 1)))))
		 (if (find #\- xs)
		     (setq xs (subseq xs (1+ (position #\- xs))))
		     (setq xs "")) ) ) )
    abbrev))

; 04 Aug 93; 19 Dec 96; 23 Dec 96; 28 Jul 97
(defun lcprincomment (str &rest args)
  (lcterprib)
  (lcprinc (case *lclanguage*
	     (c "/* ")
	     ((c++ java) "// ")
	     (pascal "{ ")))
  (apply #'format (cons *lisptoc-output-stream* (cons str args)))
  (case *lclanguage*
    (c (lcprinc " */"))
    (pascal (lcprinc " }")) )
  (lcterpri) )

; 08 Jan 10
(defun lcinlinecomment (str &rest args)
  (lcprinc (case *lclanguage*
	     (c "/* ")
	     ((c++ java) "// ")
	     (pascal "{ ")))
  (apply #'format (cons *lisptoc-output-stream* (cons str args)))
  (case *lclanguage*
    (c (lcprinc " */"))
    (pascal (lcprinc " }")) ) )

; 19 Dec 96
(defun lcshortcomment (str)
  (case *lclanguage*
    ((c c++ java) (lcprinc "/* ") (lcprinc (stringify str)) (lcprinc " */") )
    (pascal (lcprinc "{ ") (lcprinc (stringify str)) (lcprinc " }") ) ))

; 09 Apr 91; 20 May 91; 29 May 92; 20 Aug 92; 13 Oct 92; 20 Apr 93; 22 Apr 93
; 04 Aug 93; 09 Nov 93; 27 Sep 94; 17 Dec 96; 23 Dec 96; 02 Jan 97; 03 Jan 97
; 06 Jan 96; 09 Dec 09; 08 Jan 10
; Print a type, translated for target language
(defun lcprintype (tp)
  (let (tmp)
    (cond ((and (symbolp tp) (setq tmp (get tp 'lcprintname)))
             (lcprinc tmp))
          ((setq tmp (lctypetrans tp)) (lcprinc tmp))
	  ((member tp '(char int float double)) (lcprinlc tp))
	  ((and (setq tmp (glbasetype tp))
		(not (equal tmp tp))
                (not (and (consp tmp)
                          (member (car tmp) *gltypenames*))))
	    (lcprintype tmp))
	  ((and (consp tp) (member (car tp) '(record crecord ^)))
	    (case *lclanguage*                            ; record name
              (pascal  (lcprinc (lccpointer (cadr tp))))
	      ((c c++) (lcprinc (lccpointer (cadr tp))) (lcprinc " * "))
	      (java    (lcprinc (lcfixname (cadr tp) 'type))) ) )
	  ((and (consp tp)
		(eq (car tp) 'listof))
	    (lcprinc "CONS"))
	  ((and (consp tp)
		(eq (car tp) 'units))
	    (lcprintype (cadr tp)))
	  ((symbolp tp) (lcprinc (lccpointer tp)))
	  (t (lcprinc (lcfixname tp 'type)))) ))

; 17 Dec 96; 26 Dec 96; 30 Dec 96; 03 Jan 97; 17 Apr 97; 26 Oct 04; 15 Nov 04
; 08 Dec 09
; Translate type for target language if it is a standard type
(defun lctypetrans (type)
  (let (pair)
    (if (setq pair
	      (assoc type
		     (case *lclanguage*
		       (c '((integer "int") (real "double")
			    (boolean "int") (string "char *")
			    (nil "void") (character "char")))
		       (c++ '((integer "int") (real "double")
			      (boolean "int") (string "char *")
			      (nil "void") (character "char")))
		       (java '((integer "int") (real "double")
			       (string "String") (boolean "boolean")
			       (nil "void") (character "char")))
		       (pascal '((integer "integer") (real "real")
				 (string "alfa") (boolean "boolean")
				 (character "char"))) )))
	(cadr pair) ) ))

; 20 Apr 93; 22 Apr 93; 17 May 93; 28 May 93; 01 Oct 93; 09 Nov 93; 15 Nov 93
; 23 Nov 93; 27 Sep 94; 18 Dec 96; 19 Dec 96; 20 Dec 96; 23 Dec 96; 27 Dec 96
; 30 Dec 96; 02 Jan 97; 03 Jan 97; 06 Jan 97; 09 Jan 97; 28 Jul 97; 02 Nov 04
; 04 Nov 04; 11 Nov 04; 19 Nov 04; 09 Dec 09; 10 Dec 09; 08 Jan 10; 11 Jan 10
; 26 Jan 10
; Print a variable name and type as translated for target language
(defun lcprintypeb (type name lasttype &optional public static fnp structp)
  (let (arrtype str typ tp nm tmp)
    (setq nm (if (stringp name)
		 name
	         (lcfixname (if (atom name) name (first name)) 'var)))
    (setq typ (glxtrtypeb type))
    (if (and (consp typ)
	     (eq (car typ) 'arrayof))
	(setq arrtype (cadr typ))
        (if (and (symbolp typ)
		 (setq tmp (glstructure typ))
		 (consp (car tmp))
		 (eq (caar tmp) 'arrayof))
	    (setq arrtype (cadar tmp))))
    (setq tp (or arrtype
                 (if (or (consp type)
                         (and (consp typ)
                              (member (car typ) '(record crecord ^)))
                         (lcwrapperp type))   ; any cases left???
                     typ type)))
                 ; 06 Jan 10    was  (if (and (consp type) (eq (car type) '^))
    (setq str (if (symbolp tp)
		  (car (glstructure tp))
		  tp))
    (case *lclanguage*
      ((c c++)
        (if (and (equal type lasttype) (not structp))
	    (progn (lcprinc ", ")
		   (if (and (consp str)	           ; C record name
			    (member (car str) '(record crecord ^)))
		       (lcprinc "*"))
		   (lcprinc nm))
	    (progn
	      (if (and (consp str)	         ; C record name
		       (member (car str) '(record crecord ^)))
                  (progn (if structp (lcprinc "struct _"))
                         (lcprinc (lccpointer (cadr str))))
                  (lcprintype tp))
	      (lcspaces 1)
	      (if (or arrtype
		      (and (consp str)	          ; C record name
			   (member (car str) '(record crecord ^))))
		  (lcprinc "* "))
	      (lcprinc nm)
	      (if (and (consp typ) (eq (car typ) 'units))
		  (progn
                    (lcspaces 4)
                    (lcinlinecomment "Unit of ~A is ~A" nm (caddr type))
                    (lcspaces 1))) ) ) )
      (java
        (if (equal type lasttype)
	    (progn (lcprinc ", ") (lcprinc nm))
	    (progn
	      (if (and (consp typ) (eq (car typ) 'units))
		  (lcprincomment "Unit of ~A is ~A" name (caddr type)))
	      (if public (lcprinc "public "))
	      (if static (lcprinc "static "))
	      (lcprintype tp)
	      (lcspaces 1)
	      (if arrtype (lcprinc "[] "))
	      (lcprinc nm))))
      (pascal (lcprinc nm)
	      (lcprinc " : ")
	      (if arrtype (lcprinc " array [] of "))
	      (lcprintype tp)) ) ))

; 09 Nov 93; 18 Dec 96
; Test whether type is a C record type
(defun lcrecordp (type)
  (let (tmp)
    (setq tmp (if (symbolp type)
		  (car (glstructure type))
		  type))
    (and (consp tmp) (member (car tmp) '(record crecord))) ))

; 26 Jan 10
; test whether type is a wrapper of a basic type
(defun lcwrapperp (type)
  (let (typ)
    (setq typ (glxtrtypeb type))
    (and (symbolp type)
         (glbasictypep typ)
         typ) ))

; 08 Apr 92; 20 Apr 93; 17 Nov 93
; Restructure code according to patterns until a fixpoint is reached.
(defun lcptmatch (expr patwd)
  (let (args exprprime)
    (if (atom expr) expr
	(progn (setq args
		     (if (consp (rest expr))
			 (lcptmatchl (rest expr) patwd)
			 (rest expr)))
	       (setq exprprime
		     (glptmatch (if (eq args (rest expr))
				    expr
				    (cons (first expr) args))
				patwd))
	       (if (glconstfnp exprprime)
		   (setq exprprime (glconstfnval exprprime)))
	       (if (eq exprprime expr)
		   expr
		   (lcptmatch exprprime patwd)) ) ) ))

; 08 Apr 92; 16 Apr 93
; Restructure a list of things using patterns.  Returns new structure if
; there is a change, else the old structure.
(defun lcptmatchl (l patwd)
  (let (lprime lrest)
    (if (atom l) nil
	(progn (setq lprime (lcptmatch (first l) patwd))
	       (setq lrest (lcptmatchl (rest l) patwd))
	       (if (and (eq lprime (first l))
			(eq lrest (rest l)))
		   l
		   (cons lprime lrest)))) ))

; Translate the defuns in a file
(defun ltoc-file (filename)
  (let (defn)
    (with-open-file (instream filename :direction :input)
      (loop (if (not (eq (setq defn (read instream nil '*eof*))
			 '*eof*))
		(if (and (consp defn)
			 (eq (first defn) 'defun))
		    (defun-to-lang defn))
		(return)) ) ) ))

; 28 Apr 93; 29 Apr 93; 01 Oct 93; 15 Mar 94; 25 Mar 04
; Compute length of an expression if printed flat on a single line.
(defun lcflatsize (expr prec)
  (let (sz tmp c-op op-prec)
    (if (atom expr)
	(length (stringify expr))
	(if (eq (first expr) 'glcfield)
	    (+ 2 (lcflatsize (cadr expr) 0)
	         (lcflatsize (cadr (caddr expr)) 0))
	    (if (eq (first expr) 'glcstring)
		(+ 2 (length (cadr expr)))
		(progn
		  (setq tmp (assoc (first expr) *exprops*))
		  (setq c-op (second tmp))
		  (setq op-prec (third tmp))
		  (setq sz (+ (if (and op-prec (< op-prec prec)) 4 2)
			      (lcflatsize (or c-op (first expr)) 0)))
		  (dolist (arg (rest expr)) (incf sz (lcflatsize arg 0)))
		  (unless c-op
		    (incf sz (* 2 (1- 
				   (if (consp (rest expr))
				       (length (rest expr))
				       2)))))
		  sz))) ) ))

(defun leftshift (x n) (ash x n))
(defun rightshift (x n) (ash x (- n)))

; 03 Jan 97; 06 Jan 97; 04 Nov 04; 08 Jan 10; 11 Jan 10
; Translate a record structure to other languages
(defun lcrecordtrans (str language)
  (let (strname)
    (if (symbolp str)
	(setq str (glxtrtypeb str)))
    (when (and (consp str)
	       (member (car str) '(record crecord)))
      (setq *lclanguage* language)
      (setq *lisptoc-output-stream* t)
      (setq strname (lcfixname (cadr str) 'type))
      (setf (get (cadr str) 'lcprintname) strname)
      (when (member language '(c c++ pascal))
	(case language
	  ((c c++) (princ "typedef ") )     ; 06 Jan 10
	  (pascal (princ "type ")
		  (princ (lcpointertrans (cadr str)))
		  (princ " = ^")
		  (princ strname)
		  (princ ";")
                  (terpri)) ) )
      (case language
	((c c++) (princ "struct _")     ; 06 Jan 10
	         (princ strname)
		 (princ " {"))
	(java (princ "class ")
	      (princ strname)
	      (princ " {"))
	(pascal (princ "     ")
		(princ strname)
		(princ " = record ")) )
      (terpri)
      (dolist (item (cddr str))
	(case language
	  ((c c++) (princ "    ")
	           (lcprintypeb (cadr item) (car item) nil nil nil nil t)
		   (princ ";"))
	  (java (princ "   public ")
		(lcprintypeb (cadr item) (car item) nil)
		(princ ";"))
	  (pascal (princ "        ")
	          (lcprintypeb (cadr item) (car item) nil)
		  (unless (eq item (car (last str))) (princ ";"))) )
	(terpri))
      (case language
	((c c++) (princ "    } ") (princ strname) (princ " ;"))     ; 06 Jan 10
	(java )
	(pascal (princ "     end;")) )
      (terpri) ) ))

; 06 Jan 97; 08 Jan 97
; Make an abbreviation for a pointer to a record
(defun lcpointertrans (record)
  (or (get record 'lcpointer)
      (let (ptr str (count 0))
	(setq str (lcabbrev record))
	(if (< (length str) 3)
	    (setq str (concatenate 'string str "ptr")))
	(setq ptr str)
	(while (find-symbol ptr)    ; make sure the pointer name is unique
	  (incf count)
	  (setq ptr (concatenate 'string str (stringify count))) )
	(intern ptr)
	(setf (get record 'lcpointer) ptr) )))


(setf (lcpointer 'string) "String")

; 08 Aug 95; 02 Jan 97; 14 Oct 04; 26 Oct 04
; Define some useful transformation patterns
(gldefpatterns 'lisptoc
  '( 
     ((setq ?x (+ ?x 1))      (incf ?x))
     ((setq ?x (+ 1 ?x))      (incf ?x))
     ((setq ?x (- ?x 1))      (decf ?x))
     ((setf ?x (+ ?x 1))      (incf ?x))
     ((setf ?x (+ 1 ?x))      (incf ?x))
     ((setf ?x (- ?x 1))      (decf ?x))
     ((setq ?x (+ ?x ?y))      (incf ?x ?y))
     ((setq ?x (+ ?y ?x))      (incf ?x ?y))
     ((setq ?x (* ?x ?y))      (prodf ?x ?y))
     ((setq ?x (* ?y ?x))      (prodf ?x ?y))
     ((setq ?x (- ?x ?y))      (decf ?x ?y))
; following was commented out -- uncommented on 05 Nov 93; and re- on 09 Jan 97
;     ((if ?v ?x ?y)            (cases ((symbolp (quote ?v)) (if (/= ?v 0) ?x ?y))))
;     ((if ?v ?x)              (cases ((symbolp (quote ?v)) (if (/= ?v 0) ?x))))
     ((1+ ?x)                (+ ?x 1))
     ((1- ?x)                (- ?x 1))
     ((minusp ?x)            (< ?x 0))
     ((plusp ?x)             (> ?x 0))
     ((zerop ?x)             (= ?x 0))
     ((not (> ?x ?y))         (<= ?x ?y))
     ((not (< ?x ?y))         (>= ?x ?y))
     ((not (>= ?x ?y))        (< ?x ?y))
     ((not (<= ?x ?y))        (> ?x ?y))
     ((not (/= ?x ?y))        (= ?x ?y))
     ((not (= ?x ?y))         (/= ?x ?y))
     ((not (eql ?x ?y))       (/= ?x ?y))
     ((not (eq ?x ?y))        (/= ?x ?y))
     ((when ?p . ?s)          (if ?p (progn . ?s)))
     ((unless ?p . ?s)        (if (not ?p) (progn . ?s)))
     ((not (not ?p))         ?p)
     ((tagbody ?label (when ?p ?q (go ?label)))
                            (while ?p ?q))
     ((tagbody ?label (if ?p (progn ?q (go ?label))))
                            (while ?p ?q))
     ((tagbody ?label (if ?p (progn ?q ?r (go ?label))))
                            (while ?p ?q ?r))
     ((tagbody ?label (if ?p (progn ?q ?r ?s (go ?label))))
                            (while ?p ?q ?r ?s))
     ((tagbody ?label (if ?p (progn ?q ?r ?s ?u (go ?label))))
                            (while ?p ?q ?r ?s ?u))
     ((tagbody ?label (if ?p (progn ?q ?r ?s ?u ?v (go ?label))))
                            (while ?p ?q ?r ?s ?u ?v))
     ((tagbody ?label (if ?p (progn ?q ?r ?s ?u ?v ?w (go ?label))))
                            (while ?p ?q ?r ?s ?u ?v ?w))
     ((tagbody ?label (if ?p (progn ?q ?r ?s ?u ?v ?w ?x (go ?label))))
                            (while ?p ?q ?r ?s ?u ?v ?w ?x))
     ((tagbody ?label (if ?p (progn ?q ?r ?s ?u ?v ?w ?x ?y (go ?label))))
                            (while ?p ?q ?r ?s ?u ?v ?w ?x ?y))
     ((tagbody ?label (if ?p (progn ?q ?r ?s ?u ?v ?w ?x ?y ?z (go ?label))))
                            (while ?p ?q ?r ?s ?u ?v ?w ?x ?y ?z))
     ((tagbody ?label ?q (if ?p (go ?label))) (dowhile (statement ?q) ?p))
     ((tagbody ?label ?q ?r (if ?p (go ?label))) (dowhile (progn ?q ?r) ?p))
     ((tagbody ?label ?q ?r ?s (if ?p (go ?label))) (dowhile (progn ?q ?r ?s) ?p))
     ((tagbody ?label ?q ?r ?s ?u (if ?p (go ?label)))
                            (dowhile (progn ?q ?r ?s ?u) ?p))
     ((tagbody ?label ?q ?r ?s ?u ?v (if ?p (go ?label)))
                            (dowhile (progn ?q ?r ?s ?u ?v) ?p))
     ((tagbody ?label ?q ?r ?s ?u ?v ?w (if ?p (go ?label)))
                            (dowhile (progn ?q ?r ?s ?u ?v ?w) ?p))
     ((tagbody ?label ?q ?r ?s ?u ?v ?w ?x (if ?p (go ?label)))
                            (dowhile (progn ?q ?r ?s ?u ?v ?w ?x) ?p))
     ((tagbody ?label ?q ?r ?s ?u ?v ?w ?x ?y (if ?p (go ?label)))
                            (dowhile (progn ?q ?r ?s ?u ?v ?w ?x ?y) ?p))
     ((tagbody ?label ?q ?r ?s ?u ?v ?w ?x ?y ?z (if ?p (go ?label)))
                            (dowhile (progn ?q ?r ?s ?u ?v ?w ?x ?y ?z) ?p))
     ((progn ?s)             ?s)
     ((setq ?x (if ?p ?q))     (if ?p (setq ?x ?q) (setq ?x nil)))
     ((setf ?x (if ?p ?q))     (if ?p (setf ?x ?q) (setf ?x nil)))
     ((setq ?x (if ?p ?q ?r))   (if ?p (setq ?x ?q) (setq ?x ?r)))
     ((setf ?x (if ?p ?q ?r))   (if ?p (setf ?x ?q) (setf ?x ?r)))
     ((return (setq ?x ?z))   (progn (setq ?x ?z) (return ?x)))
     ((return (if ?p ?q ?r))   (if ?p (return ?q) (return ?r)))
     ((return (if ?p ?q))     (if ?p (return ?q) (return nil)))
     ((+ ?x (if ?p ?q ?r))      (if ?p (+ ?x ?q) (+ ?x ?r)))
 ; NB the above pattern should be (fn ?x (if ?p ?q ?r))
 ; should change pattern matcher to allow a variable function.
     ((dotimes (?i ?j) . ?s)  (let (?i) (forloop ?i ?j . ?s)))    ; 11 Nov 04
     ((string=  ?x ?y)        (=  (strcmp ?x ?y) 0))
     ((string/= ?x ?y)        (/= (strcmp ?x ?y) 0))
     ((string>= ?x ?y)        (>= (strcmp ?x ?y) 0))
     ((string<= ?x ?y)        (<= (strcmp ?x ?y) 0))
     ((string>  ?x ?y)        (>  (strcmp ?x ?y) 0))
     ((string<  ?x ?y)        (<  (strcmp ?x ?y) 0))
     ((ash ?x ?n)             ?x  (and (numberp ?n) (= ?n 0)))
     ((ash ?x ?n)             (leftshift ?x ?n) (and (numberp ?n) (> ?n 0)))
     ((ash ?x ?n)             (rightshift ?x (- ?n))
                                             (and (numberp ?n) (< ?n 0)) )
     ((ldb (byte ?n ?m) ?x)    (logand (ash ?x (- ?m)) (glbitmask ?n)))
     ((dpb ?y (byte ?n ?m) ?x)  (logior (logand x
					    (lognot (ash (glbitmask ?n) ?m)))
				    (ash (logand ?y (glbitmask ?n)) ?m)))
     ((leftshift ?x 0)       ?x)
     ((rightshift ?x 0)      ?x)
     ((logior ?x 0)          ?x)
     ((logior 0 ?x)          ?x)
     ((null ?x)              (= ?x *glnull*))
     ((truncate ?x)          (lccastto int ?x))
     ((truncate (truncate ?x))    (truncate ?x))
     ((truncate (truncate ?x ?y))  (truncate ?x ?y))
     ((truncate (/ ?x ?y))    (truncate ?x ?y))
     ((string-length ?s)     (strlen ?s))
     ((char=  ?x ?y)        (=  ?x ?y))
     ((char/= ?x ?y)        (/= ?x ?y))
     ((char>= ?x ?y)        (>= ?x ?y))
     ((char<= ?x ?y)        (<= ?x ?y))
     ((char>  ?x ?y)        (>  ?x ?y))
     ((char<  ?x ?y)        (<  ?x ?y))
     ((atan ?x ?y)          (atan2 ?x ?y))
     ((let () (let . ?rest)) (let . ?rest))
 ))

; 02 Nov 04
; Output transformation patterns to convert to C
(gldefpatterns 'lisptocout
  '( ((aref ?x ?y)             ("" ?x "[" ?y "]"))
     ((char ?x ?y)             ("" ?x "[" ?y "]"))
     ((aref ?x ?y ?z)           ("" ?x "[" ?y "][" ?z "]"))
     ((aref ?x ?y ?z ?w)         ("" ?x "[" ?y "][" ?z "][" ?w "]"))
     ((arefrec ?x ?y)          ("" ?x "[" ?y "]"))
     ((arefrec ?x ?y ?z)        ("" ?x "[" ?y "][" ?z "]"))
     ((arefrec ?x ?y ?z ?w)      ("" ?x "[" ?y "][" ?z "][" ?w "]"))
     ((glcfield ?r (quote ?f)) ("" ?r "->" ?f))
     ((incf ?x)               ("++" ?x))
     ((incf ?x ?y)             ("" ?x " += " ?y))
     ((prodf ?x ?y)            ("" ?x " *= " ?y))
     ((eql ?x ?y)              ("(" ?x " == " ?y ")"))
     ((decf ?x)               ("--" ?x))
     ((decf ?x ?y)             ("" ?x " -= " ?y))
     ((go ?x)                 ("goto " ?x))
     ((setq ?x t)             ("" ?x " = " 1))
     ((setf ?x t)             ("" ?x " = " 1))
     ((setq ?x nil)           ("" ?x " = " 0))
     ((setf ?x nil)           ("" ?x " = " 0))
     ((setq ?x ?y)             ("" ?x " = " ?y))
     ((setf ?x ?y)             ("" ?x " = " ?y))
  ;   ((if ?p ?y ?z)             ("" ?p " ? " ?y " : " ?z))
     ((if ?p ?x)               ("if ( " ?p " )" #\Tab #\Return ?x))
     ((if ?p ?x ?y)             ("if ( " ?p " )" #\Tab #\Return x
                                      #\Return "else " ?y))
     ((forloop ?i ?j . ?s)      ("for ( " ?i " = 0; " ?i " < " ?j "; "
				    "++ " ?i " )" #\Tab #\Return
                                    (progn . ?s)))
     ((declare-int ?i)        ("int " ?i))
     ((while ?p . ?s)          ("while ( " ?p " ) " #\Tab
                                     #\Return (progn . ?s)))
     ((dowhile ?s ?p)          ("do " ?s " while ( " ?p " )"))
     ((statement ?s)          ("" ?s ";"))
     ((return ?x)             ("return " ?x))
     ((pop ?s)                ("pop(&" ?s ")"))
     ((push ?x ?s)             ("push(" ?x ", &" ?s ")"))
     ((lccastto ?x ?y)         ("(" ?x ") " ?y))
     ((princ ?x)              ("printf(" ?x ")"))
     ((terpri)               ("printf(\"\\n\"" ")"))
     ((make-array ?n :initial-element 0)
        ("(int *) " (calloc ?n (sizeoftype int))) )
     ((make-array ?n :initial-element 0.0)
        ("(double *) " (calloc ?n (sizeoftype double))) )
     ((make-array 0 . ?rest)  (*glnull*))
     ((make-array ?n :element-type (quote (crecord ?tp . ?rest)))
        ("(struct " ?tp " * *) " (malloc (* ?n (sizeofstruct* ?tp)) )))
     ((sizeofstruct ?tp)      ("sizeof(struct " ?tp ")") )
     ((sizeofstruct* ?tp)     ("sizeof(struct " ?tp " *)") )
     ((sizeof* ?tp)     ("sizeof(" ?tp " *)") )
     ((make-array ?n :element-type (quote integer))
        ("(int *) " (malloc (* ?n (sizeoftype "int")) )))
     ((make-array ?n :element-type (quote real))
        ("(double *) " (malloc (* ?n (sizeoftype "double")) )))
     ((make-array ?n)
        ("(void *) " (malloc (* ?n (sizeof* "void")) )))
     ((sizeoftype ?tp)        ("sizeof(" ?tp ")") )
 ))

; Output transformation patterns to convert to C++
(gldefpatterns 'lisptoc++out
  '( ((aref ?x ?y)             ("" ?x "[" ?y "]"))
     ((aref ?x ?y ?z)           ("" ?x "[" ?y "][" ?z "]"))
     ((aref ?x ?y ?z ?w)         ("" ?x "[" ?y "][" ?z "][" ?w "]"))
     ((arefrec ?x ?y)          ("" ?x "[" ?y "]"))
     ((arefrec ?x ?y ?z)        ("" ?x "[" ?y "][" ?z "]"))
     ((arefrec ?x ?y ?z ?w)      ("" ?x "[" ?y "][" ?z "][" ?w "]"))
     ((glcfield ?r (quote ?f)) ("" ?r "->" ?f))
     ((incf ?x)               ("++" ?x))
     ((incf ?x ?y)             ("" ?x " += " ?y))
     ((prodf ?x ?y)            ("" ?x " *= " ?y))
     ((eql ?x ?y)              ("(" ?x " == " ?y ")"))
     ((decf ?x)               ("--" ?x))
     ((decf ?x ?y)             ("" ?x " -= " ?y))
     ((go ?x)                 ("goto " ?x))
     ((setq ?x t)             ("" ?x " = " 1))
     ((setf ?x t)             ("" ?x " = " 1))
     ((setq ?x nil)           ("" ?x " = " 0))
     ((setf ?x nil)           ("" ?x " = " 0))
     ((setq ?x ?y)             ("" ?x " = " ?y))
     ((setf ?x ?y)             ("" ?x " = " ?y))
     ((if ?p ?x)               ("if ( " ?p " )" #\Tab #\Return ?x))
     ((if ?p ?x ?y)             ("if ( " ?p " )" #\Tab #\Return x
                                      #\Return "else " ?y))
     ((forloop ?i ?j . ?s)      ("for ( " ?i " = 0; " ?i " < " ?j "; "
				    "++ " ?i " )" #\Tab #\Return
                                    (progn . ?s)))
     ((declare-int ?i)        ("int " ?i))
     ((while ?p . ?s)          ("while ( " ?p " ) " #\Tab
                                     #\Return (progn . ?s)))
     ((dowhile ?s ?p)          ("do " ?s " while ( " ?p " )"))
     ((statement ?s)          ("" ?s ";"))
     ((return ?x)             ("return " ?x))
     ((pop ?s)                ("pop(&" ?s ")"))
     ((push ?x ?s)             ("push(" ?x ", &" ?s ")"))
     ((lccastto ?x ?y)         ("(" ?x ") " ?y))
     ((princ ?x)              ("cout << " ?x))
     ((terpri)               ("cout << \"\\n\""))
     ((make-array ?n :initial-element 0)
        ("(int *) " (calloc ?n (sizeoftype int))) )
     ((make-array ?n :initial-element 0.0)
        ("(double *) " (calloc ?n (sizeoftype double))) )
     ((make-array 0 . ?rest)  (*glnull*))
     ((make-array ?n :element-type (quote (crecord ?tp . ?rest)))
        ("(struct " ?tp " * *) " (malloc (* ?n (sizeofstruct* ?tp)) )))
     ((sizeofstruct ?tp)      ("sizeof(struct " ?tp ")") )
     ((sizeofstruct* ?tp)     ("sizeof(struct " ?tp " *)") )
     ((sizeof* ?tp)     ("sizeof(" ?tp " *)") )
     ((make-array ?n :element-type (quote integer))
        ("(int *) " (malloc (* ?n (sizeoftype "int")) )))
     ((make-array ?n :element-type (quote real))
        ("(double *) " (malloc (* ?n (sizeoftype "double")) )))
     ((make-array ?n)
        ("(void *) " (malloc (* ?n (sizeof* "void")) )))
     ((sizeoftype ?tp)        ("sizeof(" ?tp ")") )
 ))

; Output transformation patterns to convert to Java
(gldefpatterns 'lisptojavaout
  '( ((aref ?x ?y)             ("" ?x "[" ?y "]"))
     ((aref ?x ?y ?z)           ("" ?x "[" ?y "][" ?z "]"))
     ((aref ?x ?y ?z ?w)         ("" ?x "[" ?y "][" ?z "][" ?w "]"))
     ((arefrec ?x ?y)          ("" ?x "[" ?y "]"))
     ((arefrec ?x ?y ?z)        ("" ?x "[" ?y "][" ?z "]"))
     ((arefrec ?x ?y ?z ?w)      ("" ?x "[" ?y "][" ?z "][" ?w "]"))
     ((glcfield ?r (quote ?f)) ("" ?r "." ?f))
     ((incf ?x)               ("++" ?x))
     ((incf ?x ?y)             ("" ?x " += " ?y))
     ((prodf ?x ?y)            ("" ?x " *= " ?y))
     ((eql ?x ?y)              ("(" ?x " == " ?y ")"))
     ((decf ?x)               ("--" ?x))
     ((decf ?x ?y)             ("" ?x " -= " ?y))
     ((go ?x)                 ("goto " ?x))
     ((setq ?x t)             ("" ?x " = " 1))
     ((setf ?x t)             ("" ?x " = " 1))
     ((setq ?x nil)           ("" ?x " = " 0))
     ((setf ?x nil)           ("" ?x " = " 0))
     ((setq ?x ?y)             ("" ?x " = " ?y))
     ((setf ?x ?y)             ("" ?x " = " ?y))
     ((if ?p ?x)               ("if ( " ?p " )" #\Tab #\Return ?x))
     ((if ?p ?x ?y)             ("if ( " ?p " )" #\Tab #\Return x
                                      #\Return "else " ?y))
     ((forloop ?i ?j . ?s)      ("for ( " ?i " = 0; " ?i " < " ?j "; "
				    "++ " ?i " )" #\Tab #\Return
                                    (progn . ?s)))
     ((declare-int ?i)        ("int " ?i))
     ((while ?p . ?s)          ("while ( " ?p " ) " #\Tab
                                     #\Return (progn . ?s)))
     ((dowhile ?s ?p)          ("do " ?s " while ( " ?p " )"))
     ((statement ?s)          ("" ?s ";"))
     ((return ?x)             ("return " ?x))
     ((pop ?s)                ("pop(&" ?s ")"))
     ((push ?x ?s)             ("push(" ?x ", &" ?s ")"))
     ((lccastto ?x ?y)         ("(" ?x ") " ?y))
     ((princ ?x)              ("System.out.print(" ?x ")"))
     ((terpri)               ("System.out.println()"))
     ((strcmp ?x ?y)           ("" ?x ".compareTo(" ?y ")"))
     ((make-array ?n :initial-element 0)   ("new int [" ?n "]"))
     ((make-array ?n :initial-element 0.0)   ("new double [" ?n "]"))
     ((make-array 0 . ?rest)  (*glnull*))
     ((make-array ?n :element-type (quote (crecord ?tp . ?rest)))
        ("new " ?tp " [" ?n "]"))
     ((make-array ?n :element-type (quote integer))   ("new int [" ?n "]"))
     ((make-array ?n :element-type (quote real))   ("new double [" ?n "]"))
     ((make-array ?n)   ("new Object [" ?n "]"))
 ))

; Output transformation patterns to convert to Pascal
(gldefpatterns 'lisptopasout
  '( ((aref ?x ?y)             ("" ?x "[" ?y "]"))
     ((aref ?x ?y ?z)           ("" ?x "[" ?y "][" ?z "]"))
     ((aref ?x ?y ?z ?w)         ("" ?x "[" ?y "][" ?z "][" ?w "]"))
     ((arefrec ?x ?y)          ("" ?x "[" ?y "]"))
     ((arefrec ?x ?y ?z)        ("" ?x "[" ?y "][" ?z "]"))
     ((arefrec ?x ?y ?z ?w)      ("" ?x "[" ?y "][" ?z "][" ?w "]"))
     ((glcfield ?r (quote ?f)) ("" ?r "^." ?f))
     ((incf ?x)               ("" ?x " := " ?x " + 1"))
     ((incf ?x ?y)             ("" ?x " := " ?x " + " ?y))
     ((eql ?x ?y)              ("(" ?x " = " ?y ")"))
     ((decf ?x)               ("" ?x " := " ?x " - 1"))
     ((decf ?x ?y)             ("" ?x " := " ?x " - " ?y))
     ((go ?x)                 ("goto " ?x))
     ((setq ?x t)             ("" ?x " := true"))
     ((setf ?x t)             ("" ?x " := true"))
     ((setq ?x nil)           ("" ?x " := false"))
     ((setf ?x nil)           ("" ?x " := false"))
     ((setq ?x ?y)             ("" ?x " := " ?y))
     ((setf ?x ?y)             ("" ?x " := " ?y))
     ((if ?p ?x)               ("if ( " ?p " )" #\Tab #\Return "then " ?x))
     ((if ?p ?x ?y)             ("if ( " ?p " )" #\Tab #\Return "then " x
                                      #\Return "else " ?y))
     ((forloop ?i ?j . ?s)      ("for  " ?i " := 0 " " to " ?j " - 1 do "
				   #\Tab #\Return
                                    (progn . ?s)))
     ((while ?p . ?s)          ("while ( " ?p " ) do" #\Tab
                                     #\Return (progn . ?s)))
     ((dowhile ?s ?p)          ("repeat " ?s " until " (not ?p) ))
     ((statement ?s)          ("" ?s ))
     ((pop ?s)                ("pop(&" ?s ")"))
     ((princ ?x)              ("write(" ?x ")"))
     ((terpri)               ("writeln"))
 ))

(defvar *javafns*)
(setq *javafns*
      '((sin "Math.sin") (cos "Math.cos") (tan "Math.tan") (atan "Math.atan")
	(sqrt "Math.sqrt") (exp "Math.exp") (log "Math.log")
	(round "Math.round") (pow  "Math.pow")
	(princ "System.out.print") (prin1 "System.out.print")
	(print "System.out.println")
	) )
