; makev.lsp               Gordon S. Novak Jr.               ; 30 Jan 08

; Functions to make viewers by graphical connections

; Copyright (c) 2008 Gordon S. Novak Jr. and The University of Texas at Austin
; All rights reserved.

; Example:     (makev 'vector 'person)
;              (mkv 'cone 'xmas-tree)

; 14 May 96; 11 Mar 99; 15 Mar 99; 19 Mar 99; 23 Mar 99; 28 Feb 02; 28 May 02
; 02 Jan 03; 03 Jan 03; 09 Jan 03; 21 Apr 03; 23 Apr 03; 05 Jan 04; 07 Jul 06
; 07 Nov 06; 17 Jan 08

; Files needed: dwindow.lsp vector.lsp menu-set.lsp conn.lsp
; For examples, see mkvtest.lsp

(defvar *makev-group*        nil)

(defvar *makev-eqns-set*     nil)

(defvar *makev-goal-types*   nil)
(defvar *makev-source-names* nil)   ; used to require first sel to be a goal
(defvar *makev-sources*      nil)
(defvar *makev-undef-basis-vars* nil)
(glispglobals
 (*makev-group*        conn-group)
 (*makev-goal-types*   (listof gltype))
 (*makev-source-names* (listof symbol))
    )  ; glispglobals


; 24 Jul 91; 30 Jul 91; 25 Sep 92; 28 Sep 92; 06 Oct 92; 21 Oct 92; 26 Oct 92
; 28 Oct 92; 02 Nov 92; 24 Nov 92; 04 Aug 93; 28 Jan 94; 01 Feb 94; 04 Feb 94
; 23 Mar 99; 28 Feb 02; 02 Jan 03; 05 Jan 04
; Make a viewer between two types
(defun makev (goals sources) (makevfn (listify goals) (listify sources)) )

(gldefun makevfn ((goaltypes (listof gltype)) (sourcetypes (listof gltype)))
  (let ((goals (listof glnametype)) (sources (listof glnametype))
	group box res resb done)
    (*makev-goal-types* = goaltypes)
    (goals   = (mapcar #'(lambda (x) (list (glmkatom x) x)) goaltypes))
    (sources = (mapcar #'(lambda (x) (list (glmkatom x) x)) sourcetypes))
    (group = (conn-init sources goals (vip-window) 'props t
			  nil #'makev-connp))
    (*makev-group* = group)
    (while ~ done do
      (for goal in goals do
	   (if (get (type goal) 'equations)
	       (progn (box = (assoc (name goal) (boxes group)))
		      ((equations box) = (gleqns-init-equations
					   (get (type goal) 'equations))))))
      (res = (conn-edit-conns group))
      (if (res == 'quit)
	  (done = t)
	  (if (res != 'redo)
	      (progn (vip-propagate-inputs group nil)
		     (resb = (list 'viewer goals sources (makev-conns group)))
		     (done = t))) ))
    (unless *vip-perm-window* (close *vip-window*))
    (or resb res) ))

; 28 Jan 94; 01 Feb 94; 16 Feb 94; 05 Jan 04
; (menu-port (list (port symbol) (menu-name symbol)) )
; (menu-set-conn (list (from menu-port) (to   menu-port)))
; This function is called when a connection is made
(gldefun makev-connp ((group conn-group) (conn menu-set-conn))
  (let (goal-port box m vars)
    (if (assoc (menu-name (from conn)) (goals group))
	(goal-port = (from conn))
        (goal-port = (to conn)))
    (box = (conn-box group (menu-name goal-port)))
    (vars = (gleqns-var-defined (equations box) (port goal-port)))
    (m = (named-menu (mconns group) (menu-name goal-port)))
    (if m is a picmenu
	(progn (for var in vars do (delete-named-button m var))
	       (for var in (makev-tuple-vars (equations box))
		    do (delete-named-button m var)) ))
    t ))

; 16 Feb 94
; Identify any TUPLE variables that should be deleted from the menu
; because they cannot be specified independently.
(gldefun makev-tuple-vars ((eqns equation-set))
  (let (tuples neweqns deletes tuplevars tvar done vars)
    (when (tuples = (subset #'(lambda (x) (and (consp (caddr x))
						 (eq (caaddr x) 'tuple)))
			      (equations eqns)))
      (dolist (tupleeq tuples)
	(tuplevars = (cdaddr tupleeq))
	(done = nil)
	(while (and (not done) (cdr tuplevars))
	  (tvar = (pop tuplevars))
	  (neweqns = (an equation-set with
			 equations        = (copy-tree (equations eqns))
			 solved-vars      = (copy-tree (solved-vars eqns))
			 solved-equations = (copy-tree (solved-equations eqns))
			 defined-vars     = (copy-tree (defined-vars eqns))
			 deleted-tuples   = (copy-tree (deleted-tuples eqns))
			 all-equations    = (all-equations eqns) ) )
	  (vars = (gleqns-var-defined neweqns (second tvar)))
	  (if (some #'(lambda (v) (find v tuplevars :key #'cadr)) vars)
	      (done = t)) )
	(if done (push (second tupleeq) deletes)) )
      deletes) ))

; 04 Feb 94; 05 Jan 04; 07 Jul 06
; Convert connections from diagram into proper form for viewer.
(gldefun makev-conns ((group conn-group))
  (let (port newconns)
    (for c in (connections (mconns group)) do
      (if (port = (if (assoc (menu-name (from c)) (goals group))
		      (from c)
		      (if (assoc (menu-name (to c)) (goals group))
			  (to c))))
	  (push (list (list (port port) (menu-name port))
		      (portcode (assoc (port port)
				       (portvals (assoc (menu-name port)
							(boxes group))))))
		newconns)) )
    newconns))


; Following functions derived from getexpr.lsp

; 25 Sep 89; 30 Sep 92
; Materialize a viewer as a GLISP type with the view as a PROP
(defun glmatv (viewer)
  (prog (sym sourcetype goaltype viewprop)
    (unless viewer (return))
    (or (and (consp viewer) (eq (car viewer) 'viewer)
	     (null (cdadr viewer)) (null (cdaddr viewer)))
	(error "Improper arg ~A" viewer))
    (setq goaltype (cadar (cadr viewer)))
    (setq sourcetype (cadar (caddr viewer)))
    (setq sym (intern
	       (symbol-name (gensym (concatenate 'string
				      (if (symbolp sourcetype)
					  (symbol-name sourcetype)
					  "TUPLE")
				      "-AS-" (if (symbolp goaltype)
						 (symbol-name goaltype)
					       "TUPLE"))))))
    (setq viewprop
	  (list goaltype
		(if (and (null (cdr (fourth viewer)))
			 (eq (caaar (fourth viewer)) 'self))
		    (cadar (fourth viewer))
		    (list
		      (list
		        'view 'self
			(kwote (cons goaltype
				     (mapcar
				      #'(lambda (l)
					  (list (caar l) (list (cadr l))))
				      (cadddr viewer)))))))) )

    (eval (list 'glispobjects
		(list sym
		      (caaddr viewer)
		      'prop
		      (list viewprop))))
    (return sym)))

; 29 Sep 89; 30 Sep 92
; Materialize a viewer as a GLISP type that implements the view as PROPs
(defun glmatvb (viewer)
  (prog (sym sourcetype goaltype viewprops)
    (unless viewer (return))
    (or (and (consp viewer) (eq (car viewer) 'viewer)
	     (null (cdadr viewer)) (null (cdaddr viewer)))
	(error "Improper arg ~A" viewer))
    (setq goaltype (cadar (cadr viewer)))
    (setq sourcetype (cadar (caddr viewer)))
    (setq sym (intern
	       (symbol-name (gensym (concatenate 'string
				      (if (symbolp sourcetype)
					  (symbol-name sourcetype)
					  "TUPLE")
				      "-AS-" (if (symbolp goaltype)
						 (symbol-name goaltype)
					       "TUPLE"))))))
    (setq viewprops
	  (mapcar #'(lambda (l) (list (caar l) (list (cadr l))))
		  (cadddr viewer)))
    (eval (list 'glispobjects
		(list sym     (caaddr viewer)
		      'prop   viewprops
		      'supers (list goaltype))))
    (return sym)))

; Make a viewer and materialize it
(gldefun matv ((goals (listof gltype)) (sources (listof gltype)))
  (glmatv (makev goals sources)))

(gldefun matvb ((goal gltype) (source gltype))
  (glmatvb (makev (list goal) (list source))))

; 30 Sep 92; 06 Oct 92; 16 Oct 92; 21 Oct 92; 26 Oct 92; 02 Nov 92; 13 Nov 92
; 16 Nov 92; 19 Mar 93; 01 Feb 94; 10 Feb 94; 19 Oct 94; 28 Feb 02; 05 Jan 04
; Make a view and add it to the source type
(gldefun mkv ((goal gltype) (source gltype)
			   &optional (redo boolean) (viewname symbol))
  (let (vw box goal-name)
    (or viewname (viewname = goal))
    (if ~ redo and (makev-get-view-choices source viewname)
        (remakeview source viewname)
	(progn (makev-delete-view source goal)
	       (vw = (makev goal source))
	       (if (and vw (vw != 'quit))
		   (progn (goal-name = (name (first (goals *makev-group*))))
			  (box = (conn-box *makev-group* goal-name))
			  (mkvb goal source vw viewname
				(eqns-solved-equations (equations box))
				(eqns-defined-vars (equations box))))) )) ))

; 21 Oct 92; 02 Nov 92; 13 Nov 92; 19 Mar 93; 19 Oct 94; 15 Mar 99; 05 Jan 04
; Remake a view using previous sets of choices
(gldefun remakeview ((source gltype) (viewname symbol))
  (let (goal choice)
    (choice = (makev-get-view-choices source viewname))
    (if choice
      (progn
	(goal = (second choice))
	(setq *makev-eqns-set* (gleqns-init-equations (get goal 'equations)))
	(setq *makev-group* nil) ; used by mkvb
	(for z in (fifth choice) do (gleqns-var-defined *makev-eqns-set*
							(caar z)) )
	(mkvb goal source (cons 'viewer (cddr choice)) (or viewname goal)
	      (eqns-solved-equations *makev-eqns-set*)
	      (eqns-defined-vars *makev-eqns-set*) ) ) )))

; 21 Oct 92; 23 Oct 92; 02 Nov 92; 06 Nov 92; 07 Nov 92; 08 Nov 92; 09 Nov 92
; 05 Mar 93; 16 Mar 93; 25 Mar 93; 21 Dec 93; 06 Jun 95; 11 Mar 99; 15 Mar 99
; 16 Mar 99; 19 Mar 99; 28 May 02; 02 Jan 03; 03 Jan 03; 09 Jan 03; 21 Feb 03
; 23 Apr 03; 05 Jan 04; 30 Jan 08
; This function does the work for mkv and remakeview.
(gldefun mkvb ((goal gltype) (source gltype) vw (viewname symbol) props varsdef)
  (let (viewtype str stores tmp basisvars eqns undef bfv sfv)
    (or viewname (viewname = goal))
    (viewtype = (intern (symbol-name (gensym
                    (concatenate 'string (symbol-name source)
				 "-AS-"  (symbol-name goal) "-")))))
    (when (glstr source)
      (setf (glget source 'views)
	    (nconc (delete-if #'(lambda (x) (eq (car x) viewname))
			      (glget source 'views))
		   (list (list viewname goal viewtype))) )
      (if (symbolp source) (glstrchanged source))
      (setf (glget source 'view-choices)
	    (nconc (delete-if #'(lambda (x) (eq (car x) viewname))
			      (glget source 'view-choices))
		   (list (cons viewname (cons goal (cdr vw)))))) )
    (setq basisvars (gleqns-basis goal))
    (or (eqns = (gleqns-basis-eqns goal))
	(if *makev-group*                                       ; 21 Apr 03
	    (progn (eqns = (makev-equations *makev-group*))
		   (varsdef = (makev-varsdef *makev-group*)) )
	    (eqns = (gleqns-init-equations nil)) ) )
    (setq str
	  (list (caaddr vw)
		'prop
		(nconc (for x in (fourth vw) collect (mkvprop x goal source))
		       (cons (list 'gltransfernames
				   (list (list 'quote varsdef)))
			     (cons (list 'glbasisvars
					 (list (list 'quote basisvars)))
				   (mapcar #'(lambda (eqn)
					       (list (cadr eqn)
						     (list (caddr eqn))))
					   props))) ) ))
    (setf (glstructure viewtype) str)
    (if (undef = (for x in basisvars
			when (not (assoc x (third str))) collect x))
	(progn
	 (glusermsg
	  (format nil
		  "MKV Warning: Basis variables ~A are not defined.~%" undef))
	 (setf (third str)
	       (nconc (third str)
		      (for var in undef collect
			   (list var
				 (list (list 'error
			(format nil "Undefined basis var ~A" var)))))))))
    (setq *makev-undef-basis-vars* undef)
; ******** the following only works if goal has a set of equations.
    (when eqns
	(for var in basisvars
	     (if (tmp = (gleqns-store-var goal viewtype var eqns undef))
		 (push (list (intern (concatenate 'string (symbol-name var)
						  ":"))
			     tmp)
		       stores)))
        (if (setq bfv (gleqns-build-from-view goal viewtype eqns))
            (push (list 'glbuildfromview bfv 'result viewtype)
                  stores))
        (if (setq sfv (gleqns-store-from-view goal viewtype eqns undef))
            (push (list 'glstorefromview sfv 'result viewtype)
                  stores))
	(push (list 'materialize
		    (gleqns-materialize-view goal viewtype undef)
		    'result goal)
	      stores))
    (nconc str (list 'msg (nreverse stores)))
    (nconc str (list 'supers (list goal)))
    viewtype ))

; 15 Mar 99
; Make an equation set for a composite diagram
(gldefun makev-equations ((group conn-group))
  (let (tmp)
    (and group
	 (tmp = (vip-make-eqb (gentemp) group))
         (gleqns-basis-eqns tmp)) ))

; 16 Mar 99; 28 Feb 02; 05 Jan 04
; Compute the transfer vars for a composite diagram, = fields of input vars
(gldefun makev-varsdef ((group conn-group))
  (let (portval xvar names fields varsdef)
    (when group
      (for box in (boxes group)
	   when (and ((boxtype box) == 'variable)
		     ((direction box) == 'input))
	   (portval = (first (portvals box)))
	   (xvar = (portname portval))
	   (if (names = (gldatanames (type box)))
	       (fields = (mapcar #'car names))
	       (fields = (list xvar)) )
	   (varsdef = (append fields varsdef)) )
      varsdef) ))
  
; 10 Feb 94
; Delete the view viewname from the type source.
(defun makev-delete-view (source viewname)
  (setf (glget source 'views)
	(delete-if #'(lambda (x) (eq (car x) viewname))
		   (glget source 'views))) )

; 19 Oct 94; 20 Oct 94
(defun makev-get-view-choices (source viewname)
  (let (specs)
    (or (assoc viewname (glget source 'view-choices))
	(find viewname (glget source 'view-choices) :key #'cadr)
	(and (setq specs (glgetviewspecs source viewname nil))
	     (makev-expand-viewspecs source specs))) ))

; 18 Oct 94; 19 Oct 94
; Expand an abbreviated viewspec form into view-choices form
; lst is (name goaltype (prop choices) ...)
(defun makev-expand-viewspecs (sourcetype lst)
  (let (goaltype sourcename goalname mappings code vcform)
    (setq goaltype (or (cadr lst) (car lst)))
    (setq sourcename (gentemp (symbol-name sourcetype)))
    (setq goalname (gentemp (symbol-name goaltype)))
    (dolist (l (cddr lst))
      (if (and (consp (cadr l)) (eq (caadr l) 'quote))
	  (setq code (cadr l))
	  (progn (setq code sourcename)
		 (dolist (nm (reverse (rest l)))
		   (setq code (list nm code)))))
      (push (list (list (first l) goalname) code)
	    mappings))
    (setq vcform (list (first lst) (second lst)
		       (list (list goalname goaltype))
		       (list (list sourcename sourcetype))
		       mappings))
    (push vcform (glget sourcetype 'view-choices))
    vcform))

; 09 Jan 03; 23 Apr 03
; Make a prop entry for mkvb.
; x is of the form ((goalprop goalvar) (sourceprop sourcevar))
(defun mkvprop (x goal source)
  (let (goaltype)
    (setq goaltype (cadr (assoc (caar x) (gldatanames goal))))
    (if (and (or (member goaltype '(integer real))
		 (glunittypep goaltype))
	     (not (equal goaltype (cadr (assoc (caadr x)
					       (glallnames source))))))
	(list (caar x) (list (list 'coercetype (cadr x) (kwote goaltype))))
        (list (caar x) (list (cadr x)))) ))

; 05 Mar 93; 13 Jul 93; 01 Feb 94; 14 May 96
; Compile makev.lsp into a plain Lisp file
(defun compile-makev ()
  (glcompfiles *directory*
	       '("glisp/vector.lsp"          ; auxiliary files
                 "X/dwindow.lsp"
	         "glisp/menu-set.lsp"
	         "glisp/equations.lsp"
	         "glisp/conn.lsp")
	       '("glisp/makev.lsp")          ; translated files
	       "glisp/makevtrans.lsp"        ; output file
	       "glisp/makev-header.lsp")     ; header file
  (cf makevtrans)
  )
