; vip.lsp                   Gordon S. Novak Jr.               ; 09 Mar 10 10

; Copyright (c) 2010 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; View Interactive Programming by graphical connections

; 02 Jan 97; 05 Feb 98; 18 Mar 98; 19 Mar 98; 20 Mar 98; 23 Apr 98; 26 Feb 99
; 02 Mar 99; 11 Mar 99; 16 Mar 99; 23 Mar 99; 22 Apr 99; 28 Feb 02; 24 May 02
; 30 May 02; 03 Jan 03; 05 Jan 04; 01 Jun 04; 30 Jun 06; 14 Jul 06; 26 Sep 06
; 17 Jan 08; 16 Feb 10; 18 Feb 10; 23 Feb 10

; initially derived from makev.lsp
; Example:     (vip radar)

; Files needed: dwindow.lsp vector.lsp conn.lsp

; VIP could check dimensionality of two quantities that are connected
; by the user, use a red line to indicate mismatch.

; VIP allows input to a tuple variable such as P2 (vector), but does
; not handle output of a tuple variable, e.g. connect P2 to OUTPUT.

; VIP does not handle simultaneous equations.
; should handle linked equation sets independently of VIP.
; input-output direction matters: xout = xin + yin, yout = xin - yin
; is easy in one direction, simultaneous equations the other direction.

(defvar *vip-window*          nil)
(defvar *vip-window-width*    700)
(defvar *vip-window-height*   600)
(defvar *vip-perm-window*     nil)  ; t to leave window displayed permanently
(defvar *vip-group*           nil)
(defvar *vip-save-group*      nil)
(defvar *vip-programs*        nil)
(defvar *vip-code*            nil)
(defvar *vip-fnname*          nil)
(defvar *vip-fntype*          nil)
(defvar *vip-trace*           nil)
(defvar *vip-let-allowed*     nil)  ; t if allowed to introduce let vars
(defvar *vip-let-size*        2)    ; size of expression to make let var

(glispglobals
 (*vip-window*         window)
 (*vip-group*          conn-group)
 (*vip-save-group*     conn-group)
 (*vip-perm-window*    boolean)
    )  ; glispglobals


(setq *vip-programs*
      '(z (drawing (line draw-line)
		   (beam draw-beam)
		   (circle draw-circle)
		   ("Type-in" type-in))
	  (program ("Type-in" type-in)) ) )

; 25 Sep 92; 30 May 02
; Make a window to draw in.
(setf (glfnresulttype 'vip-window) 'window)
(defun vip-window ()
  (or *vip-window*
      (setq *vip-window*
	    (window-create *vip-window-width* *vip-window-height*
			   "Vip window")))
  (if (boundp '*wio-font*) (window-set-font *vip-window* *wio-font*))
  *vip-window* )

; 21 Nov 92; 24 Nov 92; 25 Nov 92; 27 Nov 92; 01 Dec 92; 02 Dec 92; 07 Dec 92
; 19 Mar 93; 30 Apr 93; 04 Aug 93; 06 Jan 94; 20 Jan 94; 17 Feb 94; 05 Feb 98
; 02 Mar 99
; Make a program by graphical connections
; sources = ((name type) ...)
; outputs = ((name type) ...)
; fnname  = name of function to be created.  If nil, just returns code.
; globals = vars to be treated as globals rather than args
(defun vip (sources &optional outputs fnname globals)
  (vipfn (vip-listify sources)
	 (or (vip-listify outputs) (list (list 'output nil)))
	 fnname globals) )

; 17 Feb 94
; Make arguments into (listof glnametype)
(defun vip-listify (x)
  (if (consp x)
      (if (consp (car x)) x (list x))
      (if x (list (list x nil))) ) )

; 20 Jan 94; 27 Jan 94; 17 Feb 94; 01 Dec 94; 10 Jan 96; 02 Mar 99; 23 Mar 99
; 22 Apr 99; 28 Feb 02; 05 Jan 04
(gldefun vipfn ((sources (listof glnametype)) (outputs (listof glnametype))
			 fnname (globals (listof symbol)))
  (let (res resb)
    (*vip-fnname* = fnname)
    (*vip-group* = (conn-init sources
				(or outputs
				    (and (null sources)
					 (list (list 'output nil))))
				(vip-window) 'props nil nil nil))
    (*conn-programs* = *vip-programs*)
    (res = (conn-edit-conns *vip-group*))
    (if (res != 'quit)
	(setq *vip-save-group* (copy-tree *vip-group*)) )
    (if (res == 'done)
	(progn (resb = (vip-make-program *vip-group* fnname globals))
	       (if fnname
		   (progn (eval (cons 'gldefun (cons fnname (cdr resb))))
			  (res = fnname)))))
    (unless *vip-perm-window* (close *vip-window*))
    (or fnname resb res) ))

; 01 Dec 92; 30 Apr 93; 20 Jan 94; 25 Jan 94; 10 Jan 96
(defun vip-remake ()
  (vip-make-program (setq *vip-group* (copy-tree *vip-save-group*))
		    *vip-fnname*) )

; ---------------------------------------------------------------------------
; Code to make a program from the diagram follows.
; ---------------------------------------------------------------------------

; 25 Jan 94; 10 Jan 96
(defun vip-trace ()
  (trace vip-make-program vip-propagate-value vip-update-port
	 vip-propagate-law vip-law-update vip-propagate-var vip-update-var
	 vip-propagate-fn)
  (setq *vip-trace* t))

; 25 Jan 94; 10 Jan 96
(defun vip-untrace ()
  (untrace vip-make-program vip-propagate-value vip-update-port
	 vip-propagate-law vip-law-update vip-propagate-var vip-update-var
	 vip-propagate-fn)
  (setq *vip-trace* nil))

; 08 Apr 93; 20 Jan 94; 25 Jan 94
; Find the box with a given name
(gldefun vip-box ((group conn-group) (boxname symbol)) (result conn-box)
  (assoc boxname (boxes group)))

; 27 Nov 92; 29 Nov 92; 30 Nov 92; 01 Dec 92; 10 Dec 92; 22 Mar 93; 07 Apr 93
; 29 Apr 93; 30 Apr 93; 20 Jan 94; 21 Jan 94; 25 Jan 94; 01 Feb 94; 17 Feb 94
; 10 Jan 96; 05 Feb 98; 02 Mar 99; 05 Jan 04; 26 Sep 06
; Make a program from a set of boxes and connections
(gldefun vip-make-program ((group conn-group) (fnname symbol)
					     (globals (listof symbol)))
  (let (outs undefs code inside)
    (setq *vip-code* nil)
    (vip-propagate-inputs group t)
    (for outvar in (goals group)
      (if (filled (first (portvals (conn-var-box group (name outvar)))))
	  (outs _+ (name outvar))
	  (undefs _+ (name outvar)) ) )
    (if undefs
	(format t "Warning: output vars ~A are undefined.~%" undefs))
; see if we can just produce (f inputs) instead of
; (let (output) (setq output (f inputs)) output)
    (if (and (consp *vip-code*) (null (cdr *vip-code*))
	     (null (cdr outs))
	     (consp (car *vip-code*))
	     (eq (caar *vip-code*) 'setq)
	     (eq (cadar *vip-code*) (car outs))
	     (or (null letvars)
		 (and (null (cdr letvars))
		      (eq (caar letvars) (car outs)))))
	(code = (caddar *vip-code*))
	(progn (inside = (reverse (if outs
				      (cons (if (cdr outs)
						(cons 'list outs)
					        (car outs))
					    *vip-code*)
				      *vip-code*)))
	       (code = (if (letvars group)
			   (cons 'let (cons (mapcan #'vip-name-type
						    (letvars group))
					    inside))
			   (vip-prognify inside)))))
    (setq *vip-fntype*
          (if (cdr outs)
              (cons 'list (mapcar #'(lambda (x)
                                      (type (conn-var-box *vip-group* x)))
                                  outs))
              (type (conn-var-box *vip-group* (car outs)))))
    (if fnname
	(cons 'glambda
	      (cons (mapcan #'(lambda (x)
                                (and (not (member (name x) globals))
                                     (vip-name-type x)))
                            (sources group))
                    (if *vip-fntype*
                        (list (list 'result *vip-fntype*) code)
                        (list code))))
	code) ))


; 01 Feb 94; 02 Feb 94; 08 Feb 94; 09 Feb 94; 10 Feb 94; 30 Nov 94; 01 Dec 94
; 28 Dec 94; 26 Sep 95; 11 Jan 96; 15 Mar 96; 18 Mar 98; 19 Mar 98; 20 Mar 98
; 28 Feb 02; 05 Jan 04; 01 Jun 04
; Initialize equations for boxes and propagate inputs (source vars, constants)
; through connections and boxes.
(gldefun vip-propagate-inputs ((group conn-group) (letvarflg boolean))
  (let (name val)
    ((letvars group) = nil)
    (setq *vip-let-allowed* letvarflg)
 ; init equations for laws and ops
    (for box in (boxes group) do
      (if (or (member (boxtype box) '(law op prop))
	      (and (eq (boxtype box) 'variable)
		   (symbolp (type box))
		   (get (type box) 'equations)))
	  ((equations box) =
		 (gleqns-init-equations
		   (case (boxtype box)
		     (law (get (name box) 'equations))
		     (variable (get (type box) 'equations))
		     (op   (gleqns-op-equations (name box)))
		     (prop (gleqns-prop-equations (name box))) ) ) ) ) )
 ; init variable boxes
    (for box in (boxes group) do
      (name = (boxname box))
      (val = (first (portvals box)))
      (if *vip-trace*
	 (format t
           "vip-prop-inp: box name ~A  val ~A~%  boxtype ~A  sourcep ~A~%"
	   name val (boxtype box) (sourcep group (portname val))))
      (if ((boxtype box) == 'constant)
	  (progn (if *vip-trace*
		     (format t "vip: propagating constant ~A~%" val))
		 (vip-propagate-value group name val))
	  (if ((boxtype box) == 'variable)
	      (if (sourcep group (portname val))
		  (progn (if *vip-trace*
			     (format t "vip: propagating input var ~A~%" name))
			 (vip-propagate-props group box))
		  (if letvarflg
		      (vip-add-letvar group (portname val)(type box)) )))) ) ))


; 28 Nov 92; 30 Nov 92; 01 Dec 92; 02 Dec 92; 04 Dec 92; 10 Dec 92; 22 Mar 93
; 08 Apr 93; 27 Apr 93; 30 Apr 93; 07 May 93; 04 Aug 93; 20 Jan 94; 25 Jan 94
; 01 Feb 94; 02 Feb 94; 09 Feb 94; 10 Feb 94; 17 Feb 94; 26 Nov 94; 13 Mar 96
; 20 May 96; 28 Feb 02; 05 Jan 04
; Propagate value from a port across its connections
(gldefun vip-propagate-value ((group conn-group) (boxname symbol)
						(portval conn-port))
  (let (targets var (code anything) target-box dels done newport
		(port conn-port))
    (if *vip-trace*
	(format t "vip-propagate-value boxname ~A   portval ~A~%"
		boxname portval))
; get a list of everything this port is connected to
    (targets = (find-conns (mconns group)
			     boxname (portname portval)))
; eliminate any connections that already have a value
    (for target in targets do
      (target-box = (vip-box group (menu-name target)))
      (if (and (port = (assoc (port target) (portvals target-box)))
	       (filled port))
	  (dels +_ target)))
    (targets = (set-difference targets dels))
; port of a target is the port name, name is the destination box name
    (code = (portcode portval))
    (if (and targets *vip-trace*)
	(format t "vip: propagating value ~A from ~A of ~A~%   to ~A~%"
		(portcode portval) (portname portval) boxname targets))
; if any target is a var, do a greedy assignment to it and change code to var
    (for target in targets do
      (target-box = (vip-box group (menu-name target)))
      (var = (port target))
      (if (and (not done)
	       ((boxtype target-box) == 'variable)
	       ((portname (first (portvals target-box))) == var))
	  (progn (vip-update-var group target-box var code (type portval))
		 (code = var)
		 (done = target) ) ) )
    (if done (targets = (delete done targets)))
; make it a local temporary var if used more than once.
    (if (and *vip-let-allowed*
	     (or (and (cdr targets)
		      (not (atom code)))
		 (and targets (> (glsize code) *vip-let-size*)))
	     (symbolp boxname)
	     ((boxtype (vip-box group boxname)) <> 'constant))
	(code = (vip-make-letvar group code (portname portval)
					(type portval))) )
    (if (and (numberp code)
	     (consp (type portval))
	     (eq (first (type portval)) 'units))
	(setq code (list 'quote (list 'q code (third (type portval))))))
    (for target in targets do
      (target-box = (vip-box group (menu-name target)))
      (if (target <> done)
	  (progn
	    (newport = (vip-update-port target-box (port target)
					code (type portval)))
	    (case (boxtype target-box)
	      ((law op prop) (vip-propagate-law group target-box newport))
	      (fn (vip-propagate-fn group target-box newport))
	      (variable (vip-propagate-var group target-box (port target))))) ) )
    ))

; 17 Feb 94; 23 Aug 94; 01 Dec 94; 26 Sep 95; 13 Mar 96
; Make a LET variable and assign code to it.  Returns the var.
(gldefun vip-make-letvar ((group conn-group) (code anything) (varname symbol)
					    (vartype gltype))
  (result symbol)
  (let (var)
    (var = (glmkatom varname))
    (vip-add-letvar group var vartype)
    (push (list var '= code) *vip-code*)
    (conn-add-box group nil 'variable var vartype nil
		    (list (a conn-port portname = var
			               portcode = code
				       type     = vartype
				       filled   = T )))
    var))

; 20 Mar 98; 05 Jan 04
; Add a let variable to a group if not already present
(gldefun vip-add-letvar ((group conn-group) (varname symbol) (vartype gltype))
  (let (pair)
    (pair = (assoc varname (letvars group)))
    (if pair
	(if (null (type pair))
	    ((type pair) = vartype))
        ((letvars group) _+ (a glnametype with name = varname
				               type = vartype))) ))

; 28 Nov 92; 01 Dec 92; 04 Dec 92; 10 Dec 92; 26 Feb 99; 01 Jun 04; 16 Feb 10
; Propagate value into a box involving an equation (law or operator)
(gldefun vip-propagate-law ((group conn-group) (target-box conn-box)
					      (portval conn-port))
  (let (new)
    (vip-propagate-value group (boxname target-box) portval)
    (new = (delete (portname portval)
		     (gleqns-var-defined (equations target-box)
					 (portname portval))))
    (for var in new when (not (vip-tuple-var var target-box)) do
	 (vip-law-update group target-box var))
    (vip-whole-box group target-box) ))

; 26 Feb 99
; Test to see if a var is defined by a tuple equation
(gldefun vip-tuple-var ((var symbol) (box conn-box))
  (let ((eqn (gleqns-findeq var (all-equations (equations box)))))
    (and eqn (consp (caddr eqn))
	 (eq (caaddr eqn) 'tuple)) ))

; 28 Nov 92; 02 Dec 92; 10 Dec 92; 19 Mar 93; 25 Jan 94; 01 Jun 04
; Update values computed from a box involving an equation (law or op)
(gldefun vip-law-update ((group conn-group) (target-box conn-box)
					   (portname symbol))
  (let (code newport)
    (code = (vip-fixcode target-box
			   (gleqns-def portname (equations target-box))))
    (newport = (vip-update-port target-box portname code nil))
    (vip-propagate-value group (boxname target-box) newport) ))

; 01 Dec 92; 02 Dec 92; 20 Jan 94; 01 Jun 04; 23 Feb 10; 09 Mar 10
; Update value of a port if it is not already defined
(gldefun vip-update-port ((target-box conn-box) (portname symbol) code type)
  (let (port)
    (unless (port = (assoc portname (portvals target-box)))
      (port = (a conn-port with portname = portname  type = nil))
      ((portvals target-box) _+ port) )
    ((portcode port) = code)
    (if type ((type port) = type))
    ((filled port) = t)
    port))

; 02 Dec 92; 10 Dec 92; 27 Oct 93; 25 Jan 94; 03 May 94; 05 Nov 96; 18 Mar 98
; 28 Feb 02; 01 Jun 04; 23 Feb 10
; Update value of a simple variable
(gldefun vip-update-var ((group conn-group) (target-box conn-box)
					   (var symbol) code type)
  (let ((cd code))
    (if (var == code) (error "foo"))
    (if (and (numberp code)
	     (consp type)
	     (eq (car type) 'units))
	(setq cd (kwote (list 'q code (third type)))))
    (push (list 'setq var cd) *vip-code*)             ; was =
    (vip-update-port target-box var code type)
    ((type target-box) = type)
    (vip-propagate-value group (boxname target-box)
			 (a conn-port with portname = var
			                   portcode = var
					   type = type))
    (vip-propagate-props group target-box) ))

; 20 Mar 98; 23 Feb 10
; Propagate properties of a var that has a value
(gldefun vip-propagate-props ((group conn-group) (box conn-box))
  (let (portval)
    (for port in (connected-ports (mconns group) (boxname box))
	 (or (portval = (assoc port (portvals box)))
	     (progn (portval = 
                      (if (eq port (boxname box))           ; whole box?
                          (a conn-port with portname = port
                                    portcode = (name box)
                                    type     = (type box))
                          (a conn-port with
				    portname = port
				    portcode =
				      (list port (portname (first
							    (portvals box))))
				    type     = nil)))   ; *********
		    ((portvals box) _+ portval))) 
	 ((filled portval) = t)
	 (vip-propagate-value group (boxname box)
			      (assoc port (portvals box)) ) ) ))

; 28 Nov 92; 19 Mar 93; 25 Apr 94; 01 Dec 94; 05 Jan 04; 30 Jun 06
; Fix code to incorporate values of input ports
(gldefun vip-fixcode ((box conn-box) form)
  (let (port eqn res)
    (if (or (atom form) (quotep form))
	(if (symbolp form)
	    (if (setq port (assoc form (portvals box)))
		(portcode port)
	        (if (setq eqn (find-if #'(lambda (x)
					   (eq (cadr x) form))
						      (eqns-solved-equations
						       (equations box))))
		    (vip-fixcode box (caddr eqn))
		    form))
	    form)
        (progn
	  (setq res
		(cons (car form)
		      (mapcar #'(lambda (x) (vip-fixcode box x))
			      (cdr form))))
	  (if (glconstantp res)
	      (glconstfnval res)
	      res) ) ) ))

; 28 Nov 92; 29 Nov 92; 30 Nov 92; 01 Dec 92; 02 Dec 92; 10 Dec 92
; 25 Jan 94; 18 Mar 98; 19 Mar 98; 23 Apr 98; 11 Mar 99; 05 Jan 04; 01 Jun 04
; Propagate value into a variable box that represents a record structure
(gldefun vip-propagate-var ((group conn-group) (target-box conn-box)
					      (portname symbol))
  (let (datanames code vartype portv portc)
    (if (not (null (equations target-box)))
	(vip-propagate-law group target-box
			   (assoc portname (portvals target-box))))
    (vartype = (type target-box))
    (if (and (datanames = (gldatanames vartype))
	     (every #'(lambda (pair)
			(and (setq portv (assoc (car pair)
						(portvals target-box)))
			     (setq portc (portcode portv))
			     (not (and (consp portc)
				       (eq (car portc) (car pair))
				       (consp (cdr portc))
				       (eq (cadr portc) (name target-box))))))
		    datanames))
	(progn (for pair in datanames do
		  (push (first pair) code)
		  (push (portcode (assoc (first pair) (portvals target-box)))
			code))
	       (vip-update-var group target-box
			       (portname (first (portvals target-box)))
			       (cons 'a (cons vartype (nreverse code)))
			       nil)) ) ))

; 16 Feb 10; 18 Feb 10
; If a law box also has a data structure and it is used, create it
(gldefun vip-whole-box ((group conn-group) (target-box conn-box))
  (let (datanames code vartype portv portc)
    (vartype = (or (type target-box) (name target-box)))
    (if (and vartype
             (datanames = (gldatanames vartype))
	     (every #'(lambda (pair)
			(and (setq portv (assoc (car pair)
						(portvals target-box)))
			     (setq portc (portcode portv))
			     (not (and (consp portc)
				       (eq (car portc) (car pair))
				       (consp (cdr portc))
				       (eq (cadr portc) (name target-box))))))
		    datanames))
	(progn (for pair in datanames do
		  (push (first pair) code)
		  (push (portcode (assoc (first pair) (portvals target-box)))
			code))
               (vip-add-letvar group (boxname target-box) vartype)
	       (vip-update-var group target-box
			       (boxname target-box)
			       (cons 'a (cons vartype (nreverse code)))
			       vartype)) ) ))

; 29 Nov 92; 30 Nov 92; 14 Jul 06
; Make name-type list into pair for arg list.
; This function is used with mapcan, so its result is returned in a list.
(defun vip-name-type (nametype)
  (list (if (second nametype)
	    nametype
	    (list (first nametype)) ) ) )

; 17 Feb 94; 21 Feb 94; 22 Feb 94; 26 Apr 94; 01 Dec 94; 02 Mar 99; 28 Feb 02
; 05 Jan 04
; Abstract from a VIP diagram an equation set for the composite system
(gldefun vip-make-eq ((name symbol) &optional (group conn-group))
  (let (portval var code vars basis locals eqns dels subs)
    (or group (group = *vip-group*))
    (for box in (boxes group) when ((boxtype box) == 'variable)
      (portval = (first (portvals box)))
      (var = (portname portval))
      (if (name box)
	  (pushnew var vars)
	  (pushnew var locals))
      (if ((direction box) == 'input)
	  (pushnew var basis))
      (code = (portcode portval))
      (if (and code (code <> var))
	  (eqns +_ (list '= var code)) ) )
; If a local is equal to a variable or number, remove it and substitute.
    (for eqn in eqns
      (if (or (symbolp (third eqn))
	      (numberp (third eqn)))
	  (if (and (member (second eqn) vars)
		   (or (member (third eqn) locals)
		       (numberp (third eqn))))
	      (progn (push eqn dels)
		     (push (cons (third eqn) (second eqn)) subs))
	      (if (and (member (third eqn) vars)
		       (member (second eqn) locals))
		  (progn (push eqn dels)
			 (push (cons (second eqn) (third eqn)) subs))))))
    (if dels (progn (eqns = (sublis subs (set-difference eqns dels)))
		    (locals = (set-difference locals (mapcar #'car subs))) ))
    (eqns = (for eqn in eqns collect
		   (list (first eqn) (second eqn)
			 (gleqns-simplify (third eqn)))))
    (setf (get name 'equations) eqns)
    (if locals (setf (get name 'locals) locals))
    (setf (get name 'basis-vars) basis)
    (setf (get name 'variables) vars)
    (pushnew (list name name) *conn-user-laws* :test #'equal)
    name))

; 17 Feb 94; 21 Feb 94; 22 Feb 94; 26 Apr 94; 01 Dec 94; 02 Mar 99; 11 Mar 99
; 12 Mar 99; 15 Mar 99; 16 Mar 99; 28 Feb 02; 03 Jan 03; 05 Jan 04
; Abstract from a VIP diagram an equation set for the composite system
; This is done by renaming variables of each box if necessary,
; substituting the new vars into the box equations, collecting the
; union of the box equations, and adding equations for connections.
(gldefun vip-make-eqb ((name symbol) &optional (group conn-group))
  (let (xvar var vars tuplevars allsubs fields
        basis eqns subs outvar lhs rhs v val)
    (or group (group = *vip-group*))
  ; first process all the vars
    (for box in (boxes group) when ((boxtype box) == 'variable)
      (xvar = (portname (first (portvals box))))  ; name of the var or record
      (if ((direction box) == 'output) (outvar = xvar))
      (fields = (mapcar #'car (gldatanames (type box))))
      (tuplevars = '())
      (subs = '())
      (for v in (or fields (list xvar))
	(var = (gluniquename v vars xvar))
	(push var vars)
	(if (v != var) (push (cons v var) subs))
	(if ((direction box) == 'output) (push var basis))
	(if fields
	    (progn (push (list '= var (list v xvar)) eqns)
		   (push (list v var) tuplevars) ) ))
      (if fields (push (list '= xvar (cons 'tuple tuplevars)) eqns))
      (push (cons xvar (cons (type box) subs)) allsubs) )
  ; process boxes other than vars
    (for box in (boxes group)
      (case (boxtype box)
	(variable) ; already done
	(constant
	  (xvar = (boxname box))
	  (v = (portname (first (portvals box))))
	  (val = (portcode (first (portvals box))))
	  (subs = (list (cons v val)))
	  (push (cons xvar (cons (name box) subs)) allsubs))
	((op law prop)
	  (fields = (glvarsin (all-equations (equations box))))
	  (xvar = (boxname box))
	  (subs = '())
	  (for v in fields
	    (var = (gluniquename v vars xvar))
	    (push var vars)
	    (if (v <> var) (push (cons v var) subs)) )
	  (push (cons xvar (cons (name box) subs)) allsubs) ) )
      (eqns = (append (sublis subs (all-equations (equations box)))
			eqns)) )
  ; make equations for connections
    (for c in (connections (mconns group))
      (lhs = (vipconnvar (from c) allsubs))
      (rhs = (vipconnvar (to c) allsubs))
      (push (list '= lhs rhs) eqns))
    (setf (get name 'equations) eqns)
    (setf (get name 'basis-vars) basis)
    (setf (get name 'variables) vars)
    (setf (get name 'varsubs) allsubs)
    (pushnew (list name name) *conn-user-laws* :test #'equal)
    name))

; 28 May 93; 29 Oct 96; 15 Mar 99
; Make a variable name unique wrt a list or alist of names
(defun gluniquename (name vars &optional prefix)
  (let ((newv name) (n 0) hyph)
    (if (and (boundp name) (constantp name))
	name
      (progn
	(while (if (symbolp (car vars))
		   (member newv vars)
		   (assoc newv vars))
	  (if (or hyph (null prefix))
	      (progn (incf n)
		(setq newv (intern (concatenate 'string (symbol-name newv)
						(princ-to-string n)))) )
	      (progn (setq newv (glhyphenate prefix newv))
		     (setq hyph t)) ) )
	newv)) ))

; 12 Mar 99; 15 Mar 99
; Fine the variable to substitute for a connection terminal, (var box)
(gldefun vipconnvar ((terminal (list (var symbol) (boxname symbol))) allsubs)
  (or (cdr (assoc (var terminal)
		  (cddr (assoc (boxname terminal) allsubs))))
      (var terminal)) )

; 26 Nov 94; 01 Dec 94; 11 Jan 96; 12 Jan 96; 05 Mar 96; 13 Mar 96; 05 Jan 04
; 01 Jun 04; 18 Feb 10; 23 Feb 10
; Propagate value into a box that implements a function call
; Fn properties:
;   required-args = (listof (name type))
;   default-args  = (listof (name value))
;   code-pattern  = code using args
;   outputs       = (listof (name type))
(gldefun vip-propagate-fn ((group conn-group) (target-box conn-box)
					      (portval conn-port))
  (let (fn args port subs code (ready t) l optflg newport)
    (if *vip-trace*
	(format t "vip-propagate-fn box ~A   portval ~A~%"
		target-box portval))
    (vip-propagate-value group (boxname target-box) portval)
    (fn = (name target-box))
    (args = (vip-args fn))
    (l = args)
    (while l
      (if (and (consp (car l))
	       (eq (caar l) '&optional))
	  (optflg = t)
	  (or optflg
	      (and (port = (assoc (caar l) (portvals target-box)))
		   (filled port))
	      (ready = nil)))
      (l = (rest l)))
    (if ready
	(progn
	  (subs = (for port in (portvals target-box) collect
		       (cons (portname port) (portcode port))))
	  (for default in (get fn 'default-args)
	       when (not (assoc (first default) subs))
	       (push (cons (first default) (second default)) subs) )
	  (code = (if (get fn 'code-pattern)
		      (sublis subs (get fn 'code-pattern))
		      (cons fn (vip-make-arglist args subs nil))))
	  (newport = (vip-update-port target-box 
                                      (or (vip-output-port target-box fn) fn)
                                      code (type target-box)))
	  (vip-propagate-value group (boxname target-box) newport) ) )))

; 23 Feb 10
; find the name of the output port of a box
(gldefun vip-output-port ((box conn-box) (fn symbol))
  (some #'(lambda (port) (and (eq (direction port) 'output)
                              (not (filled port))
                              (or (member (portname port) '(out output))
                                  (eq (portname port) fn))
                              (portname port) ) )
        (portvals box) ) )

; 12 Jan 96; 05 Mar 96
; Make an argument list for a function call
; args = list of argument vars or pattern
; subs = substitution alist
; optflg = t if args at this point are optional.
; Quits at first unspecified optional arg.
(defun vip-make-arglist (args subs optflg)
  (let (tmp)
    (if (and (consp args) (consp (car args)))
        (if (eq (caar args) '&optional)
	    (vip-make-arglist (cdr args) subs t)
	    (if (setq tmp (assoc (caar args) subs))
		(cons (cdr tmp) (vip-make-arglist (cdr args) subs optflg))
	        (if optflg
		    nil
		    (cons nil (vip-make-arglist (cdr args) subs optflg)))))) ))

; 10 Jan 96; 11 Jan 96; 12 Jan 96
; Get the arguments and types of a function
(setf (glfnresulttype 'vip-args) '(listof glnametype))
(defun vip-args (fn)
  (let (fndef)
    (and (fboundp fn)
	 (or (glarguments fn)
	     (and (setq fndef (symbol-function fn))
		  (consp fndef)
		  (eq (car fndef) 'lambda-block)
		  (if (and (consp (cadddr fndef))
			   (eq (car (cadddr fndef)) 'glcompileme))
		      (progn (glcc? fn)
			   (glarguments fn))
		      (mapcar #'(lambda (x) (list x 'anything))
			      (caddr fndef))))
	     (and (consp fndef)
		  (eq (car fndef) 'lambda)
		  (mapcar #'(lambda (x) (list x 'anything))
			  (cadr fndef)))
	     (and (gloriginalexpr fn)
		  (glcc fn)
		  (glarguments fn)) ) ) ))

; 11 Jan 96
; Make a list of code into a progn or extract if singleton
(defun vip-prognify (l)   (if (cdr l) (cons 'progn l) (car l)))

; 18 Mar 98
; Get result type of a function, compiling if necessary
(gldefun glfnresulttype! ((fn symbol))
  (result (listof glnametype))
  (or (glfnresulttype fn)
      (and (gloriginalexpr fn)
	   (glcc fn)
	   (glfnresulttype fn) ) ) )

; 26 Sep 06
; Use VIP to make a prop for a type
(defun vipprop (type name)
  (let (fn)
    (setq fn (vip (list (list 'self type))))
    (when fn
      (gladdprop type 'prop
                 (cons name (cons (list fn)
                                  (if *vip-fntype*
                                      (list 'result *vip-fntype*))))) ) ))

; 03 May 94
; Compile the vip.lsp and conn.lsp files into a plain Lisp file
(defun compile-vip ()
  (glcompfiles *directory*
	       '("glisp/vector.lsp"          ; auxiliary files
                 "X/dwindow.lsp"
		 "glisp/menu-set.lsp")
	       '("glisp/conn.lsp"        ; translated files
		 "glisp/vip.lsp")
	       "glisp/viptrans.lsp"         ; output file
	       "glisp/vip-header.lsp")      ; header file
  (cf viptrans) )
