; windowio.lsp         Gordon S. Novak Jr.             ; 08 Mar 13

; Copyright (c) 2013 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Functions to perform I/O via X windows for use in www demos.

; also uses: textmenutrans.lsp, menu-settrans.lsp, glispextras.lsp

; 18 Apr 95; 20 Apr 95; 08 May 95; 02 Jan 97; 06 Jan 97; 21 Jan 97; 23 Jan 97
; 28 Jan 97; 13 Feb 97; 06 Mar 97; 12 Mar 97; 14 Mar 97; 28 Jul 97; 24 Sep 97
; 28 Oct 97; 16 Jul 98; 14 Mar 01; 17 May 02; 28 May 02; 29 May 02; 30 May 02
; 31 May 02; 23 Dec 03; 24 Dec 03; 26 Dec 03; 31 Dec 03; 19 Jan 04; 16 Jun 04
; 18 Jun 04; 20 Jun 04; 23 Jun 04; 25 Jun 04; 28 Jul 04; 30 Jul 04; 02 Aug 04
; 03 Aug 04; 04 Aug 04; 05 Aug 04; 13 Aug 04; 16 Aug 04; 18 Aug 04; 20 Aug 04
; 26 Aug 04; 26 Oct 04; 28 Oct 04; 04 Nov 04; 16 Nov 04; 18 Nov 04; 03 Feb 05
; 10 Feb 05; 04 Oct 05; 11 Oct 05; 03 Nov 05; 13 Apr 06; 07 Nov 06; 22 Nov 06
; 12 Dec 06; 29 Jan 07; 30 Jan 07; 01 Feb 07; 15 Mar 07; 17 Apr 07; 16 May 07
; 25 Oct 07; 14 Jul 08; 09 Mar 10; 08 Oct 10

(defvar *wio-window*           nil)
(defvar *wio-window-width*     600)
(defvar *wio-window-height*    400)
(defvar *wio-menu-set*         nil)
(defvar *wio-subwindow*        nil)
(defvar *wio-subwindow-width*  600)
(defvar *wio-subwindow-height* 400)
(defvar *wio-submenu-set*      nil)
(defvar *wio-font* '8x13)
(defvar *isaac-bad-sentence*   nil)
(glispglobals (*wio-window*           window)
	      (*wio-window-width*     integer)
	      (*wio-window-height*    integer)
	      (*wio-menu-set*         menu-set)
	      (*wio-subwindow*        window)
	      (*wio-subwindow-width*  integer)
	      (*wio-subwindow-height* integer)
	      (*wio-submenu-set*      menu-set)
	      (*isaac-bad-sentence*   anything) )

; 18 Apr 95; 20 Apr 95; 08 May 95; 31 May 02; 08 Mar 13
; Make a window to use.
(setf (glfnresulttype 'wio-window) 'window)
(defun wio-window (&optional title width height (posx 0) (posy 0) font)
  (if (and *wio-window*
           (or (and width (> width *wio-window-width*))
               (and height (> height *wio-window-height*))))
      (progn (window-destroy *wio-window*)
             (setq *wio-window* nil)))
  (if width (setq *wio-window-width* width))
  (if height (setq *wio-window-height* height))
  (or *wio-window*
      (setq *wio-window*
	    (window-create *wio-window-width* *wio-window-height* title
			   nil posx posy font))) )

; 02 May 95
; Make a second window to use.
(setf (glfnresulttype 'wio-window) 'window)
(defun wio-subwindow (&optional title)
  (or *wio-subwindow*
      (setq *wio-subwindow*
	    (window-create *wio-subwindow-width* *wio-subwindow-height*
			   title))) )

; 19 Apr 95; 14 Jul 08
(defun wio-init-menus (w commands)
  (let ()
    (setq *wio-window* w)    ; added to make sure it gets set
    (window-clear w)
    (setq *wio-menu-set* (menu-set-create w nil))
    (menu-set-add-menu *wio-menu-set* 'command nil "Commands"
		       commands (list 0 0))
    (menu-set-adjust *wio-menu-set* 'command 'top nil 2)
    (menu-set-adjust *wio-menu-set* 'command 'right nil 2)
    ))

; 19 Apr 95; 20 Apr 95; 25 Apr 95; 02 May 95; 29 May 02; 09 Mar 10
; Lisp server example
(gldefun lisp-server ()
  (let (w inputm done sel (redraw t) str result sents input)
    (w = (wio-window "Lisp Server" 500 300))
    (open w)
    (clear w)
    (set-font w *wio-font*)
    (wio-init-menus w '(("Quit" . quit)))
    (window-print-lines w
      '("Click mouse in the input box, then enter"
	"a Lisp expression, then click outside the box."
	""
	"Input:   e.g.  (+ 3 4)  or  (sqrt 2)")
      10 (- *wio-window-height* 20))
    (window-printat-xy w "Result:" 10 50)
    (inputm = (editmenu-create 400 100 nil w
			       10 80 t t
                               *wio-font* t))
    (add-item *wio-menu-set* 'input nil inputm)
    (while ~ done do
      (sel = (menu-set-select *wio-menu-set* redraw))
      (redraw = nil)
      (case (menu-name sel)
	(command
	  (case (port sel)
	    (quit    (done = t))
	    ))
	(input (setq sents (port sel))
	       (input = (ap-read-from-strings w sents))
	       (result = (catch 'error
			     (eval input)))
	       (erase-area-xy w 75 40 (- *wio-window-width* 20) 30)
	       (window-print-line w (write-to-string result :pretty t)
				  75 50))
	) )
    (close w)
    ))

; 19 Apr 95; 20 Apr 95; 21 Apr 95; 02 May 95; 12 May 95; 17 Aug 95
; 19 Jan 04
; Unit server: convert or simplify units of measurement
(gldefun unit-server ()
  (let (w done sel (redraw t) sourcem goalm result sourceunit goalunit)
    (w = (wio-window "Unit Server"))
    (open w)
    (clear w)
    (wio-init-menus w '(("Quit" . quit) ("Convert" . convert)
			("Simplify" . simplify)))
    (window-print-lines w
      '("Click mouse in the input boxes, then enter input and"
	"output unit expressions followed by Return."
	"Then click either the Convert or Simplify command.")
      10 (- *wio-window-height* 20))
    (window-printat-xy w
      "Source Unit:   e.g.  m   or  (/ (* atto parsec) (* micro fortnight))"
      10 (- *wio-window-height* 80))
    (window-printat-xy w "Goal Unit:     e.g.  ft  or  (/ inch second)"
		       10 (- *wio-window-height* 150))
    (sourcem = (textmenu-create (- *wio-window-width* 100) 30 nil w
				 20 (- *wio-window-height* 120) t t '9x15 t))
    (goalm = (textmenu-create (- *wio-window-width* 100) 30 nil w
				 20 (- *wio-window-height* 190) t t '9x15 t))
    (add-item *wio-menu-set* 'source nil sourcem)
    (add-item *wio-menu-set* 'goal nil goalm)
    (while ~ done do
      (sel = (menu-set-select *wio-menu-set* redraw))
      (redraw = nil)
      (case (menu-name sel)
	(command
	  (case (port sel)
	    (quit    (done = t))
	    (convert
	      (when (and sourceunit goalunit)
		(erase-area-xy w 20 2 (- *wio-window-width* 20)
			      (- *wio-window-height* 240))
		(if (setq result (glconvertunit sourceunit goalunit))
		    (wio-echo w
		      (format nil
                     "Multiply source quantity by: ~12,6,2,1,'*,' ,'EE" result)
			   (- *wio-window-height* 250))
		    (wio-echo w
		      "Sorry, these units have different dimensions."
			   (- *wio-window-height* 250)) ) ))
	    (simplify
	      (when sourceunit
		(setq result (glsimplifyunit sourceunit))
		(erase-area-xy w 20 2 (- *wio-window-width* 20)
			      (- *wio-window-height* 240))
		(window-printat-xy w "Source unit simplified:"
				   10 (- *wio-window-height* 250))
		(window-print-line w (unit-server-write-to-string result)
				   20 (- *wio-window-height* 270)) ) )
	    ))
	(source (erase-area-xy w 10 (- *wio-window-height* 214)
			       (- *wio-window-width* 20) 20)
	        (sourceunit = (safe-read-from-string (port sel)))
		(unless (glunitp sourceunit)
		  (wio-echo w "Sorry, that is not a legitimate unit."
			    (- *wio-window-height* 210))
		  (sourceunit = nil)) )
	(goal   (erase-area-xy w 10 (- *wio-window-height* 214)
			       (- *wio-window-width* 20) 20)
		(goalunit = (safe-read-from-string (port sel)))
		(unless (glunitp goalunit)
		  (wio-echo w "Sorry, that is not a legitimate unit."
			    (- *wio-window-height* 210))
		  (goalunit = nil)) )
	) )
    (close w)
    ))

; 12 May 95
(defun unit-server-write-to-string (x)
  (with-open-stream (stream (make-string-output-stream))
    (unit-server-write x stream)
    (get-output-stream-string stream) ) )

; 12 May 95
(defun unit-server-write (x stream)
  (if (consp x)
      (progn (princ "(" stream)
	     (mapl #'(lambda (y) (unit-server-write (car y) stream)
		       (if (cdr y) (princ " " stream)))
		   x)
	     (princ ")" stream))
      (if (and (numberp x) (floatp x))
	  (format stream "~12,6,2,1,'*,' ,'EE" x)
	  (prin1 x stream))))

(defun load-isaac ()
(setq graphicsterminal 'xakcl)
(load "/u/novak/isaac/isaacfix.lsp")
(load "/u/novak/isaac/isaac2.lsp")
(load "/u/novak/isaac/diagram.lsp")
(load "/u/novak/isaac/euclid.lsp")
(load "/u/novak/isaac/math.lsp")
(load "/u/novak/isaac/parser.lsp")
(load "/u/novak/isaac/psolver.lsp")
(setq *saveprobs* nil)
)

; 19 Apr 95; 20 Apr 95; 21 Apr 95; 24 Apr 95; 02 May 95; 26 Jul 95; 21 Feb 96
; 30 May 02; 31 May 02; 19 Jan 04
; Isaac server: run Isaac program for physics problems
(gldefun isaac-server ()
  (let (w done sel (redraw t) numberm newm1 newm2 newm3 newm4 newm5
	  snt1 snt2 snt3 snt4 snt5 sents problemn)
    (setq *screenxmax* 350)
    (setq *screenymax* 350)
    (setq *screenxoff* 620)
    (setq *screenyoff* 10)
    (setq *isaac-silent* t)
    (setq *saveprobs* nil)
    (initialize)
    (w = (wio-window "Isaac Server" 600 400 10 10 *wio-font*))
    (open w)
    (clear w)
    (wio-init-menus w '(("Quit" . quit) ("Try New Problem" . trynew)))
    (window-print-lines w
      '("Click mouse in an input box; end input by Enter/Return.")
      10 (- *wio-window-height* 20))
    (window-print-lines w
      '("To try a predefined problem, enter"
	"the problem number (1 - 37) here:")
      10 (- *wio-window-height* 46))
    (window-print-lines w
      '("To try a new problem, enter sentence lines below,"
	"one sentence per line, then click Try New Problem.")
      10 (- *wio-window-height* 92))
    (numberm = (textmenu-create 40 30 nil w
			        310 (- *wio-window-height* 70) t t '9x15 t))
    (newm1 = (textmenu-create (- *wio-window-width* 100) 20 nil w
				 20 (- *wio-window-height* 140) t t '9x15 t))
    (newm2 = (textmenu-create (- *wio-window-width* 100) 20 nil w
				 20 (- *wio-window-height* 160) t t '9x15 t))
    (newm3 = (textmenu-create (- *wio-window-width* 100) 20 nil w
				 20 (- *wio-window-height* 180) t t '9x15 t))
    (newm4 = (textmenu-create (- *wio-window-width* 100) 20 nil w
				 20 (- *wio-window-height* 200) t t '9x15 t))
    (newm5 = (textmenu-create (- *wio-window-width* 100) 20 nil w
				 20 (- *wio-window-height* 220) t t '9x15 t))
    (add-item *wio-menu-set* 'numberm nil numberm)
    (add-item *wio-menu-set* 'newm1 nil newm1)
    (add-item *wio-menu-set* 'newm2 nil newm2)
    (add-item *wio-menu-set* 'newm3 nil newm3)
    (add-item *wio-menu-set* 'newm4 nil newm4)
    (add-item *wio-menu-set* 'newm5 nil newm5)
    (while ~ done do
      (sel = (menu-set-select *wio-menu-set* redraw))
      (redraw = nil)
      (case (menu-name sel)
	(command
	  (case (port sel)
	    (quit    (done = t))
	    (trynew
	      (setq sents (if snt5 (list snt5)))
	      (if snt4 (push snt4 sents))
	      (if snt3 (push snt3 sents))
	      (if snt2 (push snt2 sents))
	      (if snt1 (push snt1 sents))
	      (isaac-echo-problem w sents)
	      (try sents)
	      (isaac-server-output w) )) )
	(numberm
	  (problemn = (safe-read-from-string (port sel)))
	  (if (and (integerp problemn) (> problemn 0)
		   (<= problemn (length lprobs)))
	      (progn
	        (isaac-echo-problem w (car (cddddr (nth (1- problemn) lprobs))))
	        (doprob problemn)
	        (isaac-server-output w))
	      (progn (isaac-erase-problem-area w)
		     (window-printat-xy w
		       "Sorry, that is not a legitimate problem number."
		       10 (- *wio-window-height* 320))
		     (problemn = nil)) ) )
	(newm1 (setq snt1 (isaac-read-from-string
			   (isaac-filter-input (port sel)) w)))
	(newm2 (setq snt2 (isaac-read-from-string
			   (isaac-filter-input (port sel)) w)))
	(newm3 (setq snt3 (isaac-read-from-string
			   (isaac-filter-input (port sel)) w)))
	(newm4 (setq snt4 (isaac-read-from-string
			   (isaac-filter-input (port sel)) w)))
	(newm5 (setq snt5 (isaac-read-from-string
			   (isaac-filter-input (port sel)) w)))
	) )
    (close w)
    ))

; 25 Apr 95; 21 Feb 95
(defun isaac-read-from-string (s w)
  (let (snt bad)
    (when (> (length s) 0)
      (setq snt (safe-read-from-string s))
      (dolist (wd snt)
	(if (not (word? wd))
	    (push wd bad)))
      (if bad
	  (progn (isaac-erase-problem-area w)
		 (window-printat-xy w
		   (format nil "The following words are not known: ~A" bad)
		   10 (- *wio-window-height* 320))
		 nil)
	  snt) ) ))
				

; 24 Apr 95; 25 Apr 95
(defun isaac-echo-problem (w sentences)
  (let (y)
    (isaac-erase-problem-area w)
    (setq y 250)
    (dolist (s (isaac-write-to-string sentences))
      (window-printat-xy w s 10 (- *wio-window-height* y))
      (setq y (+ y 16))) ))

(defun isaac-erase-problem-area (w)
  (window-erase-area-xy w 10 2 (- *wio-window-width* 12)
			       (- *wio-window-height* 240)) )

; 25 Apr 95; 26 Jul 95
(defun isaac-server-output (w)
  (let (y)
    (with-open-stream (stream (make-string-output-stream))
      (if *isaac-bad-sentence*
	  (format stream "Sorry, I had trouble with the sentence ~A~%"
		  *isaac-bad-sentence*)
	  (prtsol stream))
      (setq y 24)
      (dolist (s (isaac-split-string (get-output-stream-string stream)))
	(window-printat-xy w s 10 y)
	(decf y 12) ) ) ))

(defvar *isaac-server-string* (make-string 100))
(defvar *isaac-string-n* 0)

(defun isaac-clear-string ()
  (dotimes (i 100) (setf (char *isaac-server-string* i) #\Space)))

; 24 Apr 95
(defun isaac-write-to-string (sentences)
  (let (strings)
    (dolist (s sentences)
      (setq strings (nconc (isaac-write-to-string-b s) strings)))
    (nreverse strings) ))

; 24 Apr 95
; Print a sentence as a list of strings, limiting to line length
(defun isaac-write-to-string-b (sentence &optional (line-length 72))
  (let (strings str n l)
    (isaac-clear-string)
    (setf (char *isaac-server-string* 0) #\()
    (setq n 1)
    (dolist (wd sentence)
      (setq str (princ-to-string wd))
      (setq l (length str))
      (if (> (+ n l) line-length)
	  (progn (push (subseq *isaac-server-string* 0 n) strings)
		 (dotimes (i 3) (setf (char *isaac-server-string* i) #\Space))
		 (setq n 3)) )
      (dotimes (i l) (setf (char *isaac-server-string* (+ n i))
			   (char str i)))
      (incf n l)
      (setf (char *isaac-server-string* n) #\Space)
      (incf n 1))
    (setf (char *isaac-server-string* n) #\))
    (incf n 1)
    (push (subseq *isaac-server-string* 0 n) strings)
    strings ))

; 25 Apr 95
; Split a string into a list of strings to limit line length
(defun isaac-split-string (str &optional (line-length 72))
  (let ((lng (length str)) n (start 0) strings)
    (while (> (- (1- lng) start) line-length)
      (setq n (1- (+ line-length start)))
      (while (not (char= (char str n) #\Space)) (decf n))
      (push (subseq str start (if (> n start) n (+ start line-length)))
	    strings)
      (setq start (1+ n)) )
    (if (> lng start) (push (subseq str start lng) strings))
    (nreverse strings) ))

; 25 Apr 95
; Filter an input string to make it Lisp-readable for Isaac
(defun isaac-filter-input (str)
  (let ((i 0) (lng (length str)) c)
    (setq *isaac-string-n* 0)
    (while (< i lng)
      (setq c (char str i))
      (incf i)
      (if (or (alpha-char-p c) (digit-char-p c) (char= c #\Space))
	  (isaac-output-char c)
	  (if (char= c #\.)
	      (if (and (> i 1) (< i lng)
		       (digit-char-p (char str (- i 2))))
		  (progn (isaac-output-char c)
			 (if (not (digit-char-p (char str i)))
			     (isaac-output-char #\0)))
		  (if (and (< i lng)
			   (digit-char-p (char str i)))
		      (progn (isaac-output-char #\0)
			     (isaac-output-char c))))
	      (if (char= c #\,)
		  (progn (isaac-output-char #\Space)
			 (isaac-output-char #\\)
			 (isaac-output-char c)
			 (isaac-output-char #\Space))
		  (isaac-output-char #\Space)) ) ) )
    (concatenate 'string "("
		 (subseq *isaac-server-string* 0 *isaac-string-n*) ")") ))

; 25 Apr 95; 27 Apr 95; 02 May 95; 04 May 95; 16 Aug 95; 07 Jan 97; 18 Jul 97
; 30 May 02; 31 May 02; 19 Jan 04
; VIP server: run VIP program that writes programs and solves physics problems
; specified by connections of diagrams.
(gldefun vip-server ()
  (let (w done language sel (redraw t) newm1 newm2 newm3 newm4 newm5
	  snt1 snt2 snt3 snt4 snt5 vars resultstr)
    (w = (wio-window "VIP Server" nil nil 10 210))
    (open w)
    (clear w)
    (set-font w *wio-font*)
    (wio-init-menus w '(("Quit" . quit) ("New Program" . new-program)
			("Output Language" . language)))
    (window-print-lines w
      '("Specify the program inputs (if any): click mouse in an"
        "input box, then describe each input as   (name type)  ,"
        "one per line, followed by Enter/Return, e.g."
        " (x real)   or   (velocity (units real (/ meter second)))"
	"     [Note: 't' cannot be used as a variable name.]"
	"Select the language (default is Lisp), then click New Program.")
      4 (- *wio-window-height* 20))
    (newm1 = (textmenu-create (- *wio-window-width* 100) 20 nil w
				 20 (- *wio-window-height* 140) t t '9x15 t))
    (newm2 = (textmenu-create (- *wio-window-width* 100) 20 nil w
				 20 (- *wio-window-height* 160) t t '9x15 t))
    (newm3 = (textmenu-create (- *wio-window-width* 100) 20 nil w
				 20 (- *wio-window-height* 180) t t '9x15 t))
    (newm4 = (textmenu-create (- *wio-window-width* 100) 20 nil w
				 20 (- *wio-window-height* 200) t t '9x15 t))
    (newm5 = (textmenu-create (- *wio-window-width* 100) 20 nil w
				 20 (- *wio-window-height* 220) t t '9x15 t))
    (add-item *wio-menu-set* 'newm1 nil newm1)
    (add-item *wio-menu-set* 'newm2 nil newm2)
    (add-item *wio-menu-set* 'newm3 nil newm3)
    (add-item *wio-menu-set* 'newm4 nil newm4)
    (add-item *wio-menu-set* 'newm5 nil newm5)
    (while ~ done do
      (sel = (menu-set-select *wio-menu-set* redraw))
      (redraw = nil)
      (case (menu-name sel)
	(command
	  (case (port sel)
	    (quit    (done = t))
	    (new-program
	      (setq vars (if snt5 (list snt5)))
	      (if snt4 (push snt4 vars))
	      (if snt3 (push snt3 vars))
	      (if snt2 (push snt2 vars))
	      (if snt1 (push snt1 vars))
	      (vip vars '((output)) 'vip-server-fn)
	      (glcc 'vip-server-fn)
	      (with-open-stream (stream (make-string-output-stream))
		(if (not (member language '(lisp nil)))
		    (gltolang 'vip-server-fn language stream)
		    (progn (format stream "; Result type: ~A~%"
				   (glfnresulttype 'vip-server-fn))
			   (write (cons 'defun
					(cons 'vip-server-fn
					      (cdr (glcompiled
						    'vip-server-fn))))
				  :stream stream :pretty t)))
		(setq resultstr (get-output-stream-string stream))
		(terpri) (princ resultstr) (terpri)
		(erase-area-xy w 20 2 (- *wio-window-width* 20)
			              (- *wio-window-height* 230))
	        (window-print-line w resultstr
				   20 (- *wio-window-height* 240))) )
	    (language (setq language
			    (menu '(("Lisp" . lisp) ("C" . c)
				    ("C++" . c++) ("Java" . java)
				    ("Pascal" . pascal))))) ))
	(newm1 (setq snt1 (safe-read-from-string (port sel))))
	(newm2 (setq snt2 (safe-read-from-string (port sel))))
	(newm3 (setq snt3 (safe-read-from-string (port sel))))
	(newm4 (setq snt4 (safe-read-from-string (port sel))))
	(newm5 (setq snt5 (safe-read-from-string (port sel))))
	) )
    (close w)
    ))

; 25 Apr 95; 14 Mar 01
(defun safe-read-from-string (str)
  (if (and (stringp str) (> (length str) 0))
        ; following options to read-from-string don't seem to work ...
      (read-from-string str nil 'read-error)))

; 25 Apr 95
; Output one char to the string
(defun isaac-output-char (c)
  (setf (char *isaac-server-string* *isaac-string-n*) c)
  (incf *isaac-string-n*))

; 02 May 95; 19 Jan 04
(gldefun wio-echo ((w window) (str string) (y integer) &optional delta)
  (if delta
      (progn
	(erase-area-xy w 2 2 (- *wio-window-width* 4)
			     (- *wio-window-height* (+ delta 2)))
	(erase-area-xy w 2 (- y 6) (- *wio-window-width* 4) 20)))
  (printat-xy w str 10 y))
		  

(defun compile-windowio ()
  (glcompfiles *directory*
	       '("glisp/vector.lsp")   ; auxiliary files
               '("glisp/windowio.lsp")      ; translated files
	       "glisp/windowiotrans.lsp")       ; output file
  (compile-file (concatenate 'string *directory* "glisp/windowiotrans.lsp")) )


(defvar *ap-server-language* 'lisp)
(defvar *ap-server-types* nil)          ; list of user type names
(defvar *ap-server-views* nil)          ; alist (user-type . views)
(defvar *ap-server-math-views*
  '(vector line-segment region circle cone sphere))
(defvar *ap-server-data-views*
  '(linked-list sorted-linked-list binary-tree avl-tree))
(defvar *ap-server-doc* '(ap-server views data-structures lisp-data-structures
			 other-data-structures))
(defvar *ap-server-class* nil)          ; class name for Java
(defvar *ap-server-quietflg* t)         ; stop (glcomp ...) printouts

; 31 Dec 96
(glispglobals
  (*ap-server-views* (listof (cons (type symbol)
				   (views (listof (list (viewname symbol)
							(viewtype gltype)))))))
  (*glspecfnscompiled* (listof symbol)) ; list of specialized fns
 ) ; glispglobals


; 28 Apr 95; 30 Apr 95; 02 May 95; 03 May 95; 04 May 95; 05 May 95; 08 May 95
; 10 May 95; 01 Jun 95; 12 Aug 96; 15 Aug 96; 19 Dec 96; 30 Dec 96; 02 Jan 97
; 07 Jan 97; 24 Sep 97; 14 Mar 01; 17 May 02; 23 Dec 03; 24 Dec 03; 19 Jan 04
; 05 Aug 04
; Automatic Programming server: make specialized versions of generic programs
(gldefun ap-server ()
  (let (w done sel (redraw t) sents datadef geom)
    (w = (wio-window "Automatic Programming Server" 600 700))
    (open w)
    (setq *ap-server-types* nil)
    (setq *ap-server-views* nil)
    (setq *ap-server-class* nil)
    (ap-server-init-window w)
    (princ "<PRE>") (terpri)  ; html preformatted text command
    (while ~ done do
      (geom = (window-geometry *wio-window*))
      (if (or (*wio-window-width* <> (third geom))
	      (*wio-window-height* <> (fourth geom)))
	  (progn (*wio-window-width* = (third geom))
		 (*wio-window-height* = (fourth geom))
		 (redraw = t)))
      (if redraw
	  (progn (ap-server-draw-window w)
		 (redraw = nil)))
      (sel = (menu-set-select *wio-menu-set* redraw))
      (case (menu-name sel)
	(command
	  (case (port sel)
	    (quit    (done = t))
	    (make-view (ap-server-make-view w))
	    (make-program (ap-server-make-program w))
	    (convert-data (ap-server-convert-data w))
	    (documentation (ap-server-documentation w)) ))
	(newm1 (setq sents (port sel))
	       (datadef = (ap-read-from-strings w sents))
	       (if datadef
		   (ap-server-define-data w datadef t
		       #'ap-server-erase-bottom (ap-server-echo-y))))
	else (redraw = t)) )
    (ap-server-close-class *ap-server-class* *ap-server-language*)
    (close w)
    (terpri) (princ "</PRE>") (terpri) (terpri) ; close html preformatted text
    ))

; 08 May 95; 19 Dec 96; 20 Dec 96; 23 Dec 96; 30 Dec 96; 02 Jan 97; 03 Jan 97
; 06 Jan 97; 18 Sep 01; 28 May 02; 24 Dec 03; 26 Dec 03; 19 Jan 04
; Define a GLISP type based on input data structure from user.
; The input may be:
;    Basic type, e.g. STRING
;    Name and list of fields: (part (name string) ...) for non-Lisp languages
;    Name, structure, prop's etc.
(gldefun ap-server-define-data ((w window) (datastr anything) (redef boolean)
				erasefn (msgy integer))
  (let (newname userstrname)
    (if (and (or (and (symbolp datastr)
		      (or (glstr datastr)                     ; 26 Dec 03
			  (member datastr *glbasictypes*)))
		 (and (consp datastr)
		      (or (symbolp (car datastr))
			  (stringp (car datastr)))
		      (consp (cdr datastr))
		      (or (and (eq *ap-server-language* 'lisp)
			       (consp (cadr datastr))
			       (member (caadr datastr) *gltypenames*))
			  (and (null (cddr datastr))
			       (glbasictypep (cadr datastr)))
			  (setq datastr (ap-server-fix-str datastr)) ) ) )
	     (or (symbolp datastr) (glokstr? (cadr datastr))))
	(progn (userstrname =
	          (if (symbolp datastr)
		      datastr
		      (if (or (not redef)
			      (not (glstr (first datastr)))
			      (member (first datastr) *ap-server-types*))
			  (first datastr)
			  (progn
			    (newname = (glmkatom (first datastr)))
			    (datastr = (subst newname (first datastr)
					      datastr))
			    newname)) ) )
	     (pushnew userstrname *ap-server-types*)
	     (unless (symbolp datastr)
	       (gldefstr (cons userstrname (cdr datastr)) nil))
	     (server-echo w (format nil "~A data structure defined."
				        (lcprintname userstrname))
			  #'ap-server-erase-bottom msgy))
        (server-echo w (format nil "Sorry, ~A is not a legal data spec."
			       datastr)
		     #'ap-server-erase-bottom msgy) ) ))

; 17 May 02
(defun ap-server-echo-y () (- *wio-window-height* 250))

; 26 Dec 03
(gldefun server-echo ((w window) (msg string) erasefn (msgy integer))
  (funcall erasefn w)
  (wio-echo w msg msgy) )

; 06 Jan 97; 18 Sep 01
; Fix a structure description by substituting a symbol if a string
; used as the name of the structure.
; Also adds 'crecord at the front if it is not already present.
(defun ap-server-fix-str (str)
  (let ((name (car str)) (origname (car str)) new)
    (if (stringp name)
	(progn (setq name (gentemp (string-upcase name)))
	       (setf (get name 'lcprintname) origname)) )
    (setq new (subst name origname str :test #'equal))
    (if (and (consp (cadr str)) (eq (caadr str) 'crecord))
	new
        (list name (cons 'crecord new))) ))

; 08 May 95; 10 May 95; 01 Jun 95; 08 Aug 95; 09 Aug 95; 12 Aug 96; 02 Jan 97
; 17 May 02; 24 May 02; 28 May 02
(gldefun ap-server-init-window ((w window))
  (let ()
    (set-font w *wio-font*)
    (wio-init-menus w '(("Quit" . quit)
			("Make a View" . make-view)
			("Make Programs" . make-program)
			("Convert Data" . convert-data)
			("Documentation" . documentation)))
    (add-item *wio-menu-set* 'newm1 nil
	      (editmenu-create (- *wio-window-width* 20) 100 nil w
			       10 (- *wio-window-height* 220) t t *wio-font* t))
    (setq *ap-server-language* nil)
    (while (null *ap-server-language*)
      (setq *ap-server-language*
	    (menu '(("Lisp" . lisp) ("C" . c) ("C++" . c++) ("Java" . java)
		    ("Pascal" . pascal))
		  "Output Language:")))
   ;  (window-change-cursor)
    ))

; 08 May 95; 08 Aug 95; 15 Aug 96; 24 Sep 96; 02 Jan 97; 16 Jan 97; 17 May 02
(gldefun ap-server-draw-window ((w window))
  (let ()
    (window-print-lines w
      '(
      ;  "Follow these steps:"
      ;  "1. Specify your data structures, one at a time: click mouse in the"
      ;  "   input area below, then input data description (Emacs-like editor)."
      ;  "   When done, click the mouse outside the edit area to stop editing."
      ;  "2. Click Make a View to view your data types as abstract data."
      ;  "3. Click Make Programs to select the programs you want."
      ;  ""
        "Click inside the box and specify your record name"
	" and fields, e.g.:"
        " (part (name string) (size integer) (next (^ part)))"
        "For Lisp, specify the actual Lisp data structure:"
	" (part (list (name string) (size integer)"
        "             (next (^ part))))"
        "When done, click outside the box." )
      6 (- *wio-window-height* 14))
    (draw *wio-menu-set*)
    (force-output w) ))

; 10 May 95; 21 Feb 96; 12 Aug 96; 14 Mar 01; 19 Jan 04; 08 Oct 10
(gldefun ap-read-from-strings ((w window) (strings (listof string)))
  (let (s str res)
    (setq s (or (first strings) ""))
    (setq str (cdr strings))
    (while str
      (setq s (concatenate 'string s " " (pop str))) )
    (res = (safe-read-from-string s))
    (if (eq res 'read-error)
        (progn
          (if w
              (ap-server-print-doc w (cons "Read error: check parens."
                                           strings))
              (format t "Read error: check parens.~%~A~%" strings) )
	  nil)
        res) ))

; 02 May 95; 17 May 02
; Menu of choices, but if there is only a single choice, return it directly.
(defun wio-cond-menu (lst w msg &optional delta)
  (if (cdr lst)
      (progn (wio-echo w msg (ap-server-echo-y) delta)
	     (menu lst))
      (if (consp (first lst))
	  (caar lst)
	  (car lst))) )


; 02 May 95; 31 Dec 96; 06 Jan 97; 07 Jan 97; 17 May 02
; Add a view viewname to type
(defun ap-server-add-view (w type viewname viewtype carrierp)
  (let (vl)
    (if (setq vl (assoc type *ap-server-views*))
	(unless (assoc viewname (cdr vl))
	  (nconc vl (list (list viewname viewtype))))
        (push (list type (list viewname viewtype)) *ap-server-views*))
    (wio-echo w (format nil 
			(if carrierp
			    "A carrier record for ~A of ~A has been defined."
			    "~A view of ~A has been defined.")
			viewname (lcprintname type))
	      (ap-server-echo-y) 306) ))

; 02 May 95; 03 May 95; 04 May 95; 06 Jun 95; 31 Dec 96; 06 Jan 97; 07 Jan 97
; 28 Jan 97; 13 Feb 97; 17 May 02; 19 Jan 04
; Make a view of a previously defined type
(gldefun ap-server-make-view (w)
  (let (sourcetype result view-goal ptrs)
    (if (sourcetype = (wio-cond-menu *ap-server-types*
				       w "Specify type to be viewed"))
        (case (menu '(("Data Structure View" . viewas)
		      ("Mathematical View" . mkv)))
	  (mkv (if (and (view-goal = (menu *ap-server-math-views*))
			(result = (mkv view-goal sourcetype)) )
		   (progn (ap-server-add-view w sourcetype view-goal
					      result nil)
			  (ap-server-open-class sourcetype)
			  (if *makev-undef-basis-vars*
			      (wio-echo w
	                (format nil "Warning: Basis vars ~A were not defined."
				*makev-undef-basis-vars*)
			           (ap-server-echo-y))) )))
	  (viewas
	   (if (view-goal = (menu *ap-server-data-views*))
	       (if (and (setq ptrs (glpointers sourcetype sourcetype))
			(>= (length ptrs) (get view-goal 'glnpointers)))
		   (if (result = (glviewas view-goal sourcetype
						   nil nil nil))
		       (progn
			 (ap-server-add-view w sourcetype view-goal
					     (glclmaintype result) nil)
			 (ap-server-open-class sourcetype)))
		       (ap-make-carrier w view-goal sourcetype)) ) ) ) ) ))

; 28 Jan 97; 23 Dec 03
; Print definition for a class
(defun ap-server-open-class (sourcetype)
  (if *ap-server-class*
      (ap-server-close-class *ap-server-class* *ap-server-language*))
  (lcrecordtrans sourcetype *ap-server-language*)
  (setq *ap-server-class* sourcetype))

; 07 Jan 97; 23 Dec 03
; Print closing bracket and comment for Java class
(defun ap-server-close-class (str language)
  (if (and str (symbolp str))
      (case language
	(java (princ "   }  ")
	      (princ (ap-server-make-comment (lcfixname str 'type) t))
	      (terpri) ) ) ) )

; 02 May 95; 13 Feb 97
; Get pointers to a goal type contained in type's record
; e.g. (glpointers 'box 'box) = ((NEXT (^ BOX)))
(defun glpointers (type goaltype)
  (subset #'(lambda (x) (and (consp (cadr x))
			     (eq (caadr x) '^)
			     (gltypematch (cadadr x) goaltype)))
	  (gldatanames type)))

; 04 May 95
(defun ap-server-name-type (name type)
  (intern (concatenate 'string (stringify name) "\:" (stringify type))))

; 05 May 95
(defun ap-find-view-type (view-name)
  (if (member view-name *ap-server-data-views*
	      :test #'(lambda (x y) (eql x (if (consp y) (cdr y) y))))
      (or (glclmaintype view-name) (get view-name 'ap-type) view-name)
      view-name) )

; 05 May 95; 17 May 02
(gldefun ap-server-erase-bottom ((w window))
  (erase-area-xy w 2 2 (- *wio-window-width* 4) (- *wio-window-height* 224)) )

; 05 May 95; 09 May 95; 01 Jun 95; 24 Aug 95; 15 Aug 96
(gldefun ap-find-propnames (view-type)
  (sort
   (remove-duplicates
    (union (ap-find-includes view-type)
      (set-difference
        (union (get view-type 'basis-vars)
	  (union (glpropnames view-type 'prop)
		 (union (glpropnames view-type 'adj)
			(union (glpropnames view-type 'isa)
			       (glpropnames view-type 'msg)))))
	(append '(displayprops) (ap-find-omits view-type)) ) ) )
   #'(lambda (x y) (string< (symbol-name x) (symbol-name y))) ) )

; 09 May 95; 12 Sep 95
(gldefun ap-find-omits ((view-type gltype))
  (let ((omits (listof symbol)))
    (omits = (get view-type 'ap-omit))
    (for super in (glsupers view-type)
	 (omits = (union omits (ap-find-omits super))) )
    omits))

; 01 Jun 95; 12 Sep 95
(gldefun ap-find-includes ((view-type gltype))
  (let ((includes (listof symbol)))
    (includes = (get view-type 'ap-include))
    (for super in (glsupers view-type)
	 (includes = (union includes (ap-find-includes super))) )
    includes))

; 10 May 95; 12 May 95; 30 May 95; 11 Aug 95; 26 Dec 96; 31 Dec 96; 03 Jan 97
; 06 Jan 97; 07 Jan 97; 12 Mar 97; 28 Jul 97; 19 Jan 04
; Make a carrier record to hold a user data type.
(gldefun ap-make-carrier ((w window) (view-goal symbol) (sourcetype symbol))
  (let (str carrier result)
    (carrier =
      (ap-make-carrier-type view-goal
        (list (list 'contents
		    (if (or (glbasictypep sourcetype)
			    (glbasictypep (glxtrtype (car (glstr sourcetype)))))
		        sourcetype
		        (list '^ sourcetype))))
	*ap-server-language*))
    (if (result = (glviewas view-goal carrier nil nil nil))
	(progn (ap-server-add-view w sourcetype view-goal carrier t)
	       (lcrecordtrans (car (glstr carrier)) *ap-server-language*)
	       (setq *ap-server-class* carrier)) )))

; 10 May 95; 12 May 95; 30 May 95; 11 Aug 95; 26 Dec 96; 31 Dec 96; 03 Jan 97
; 06 Jan 97; 07 Jan 97; 06 Mar 97; 12 Mar 97; 14 Mar 97; 28 Jul 97; 30 Sep 97
; 15 Mar 07
; Make a carrier record type to hold a user data type.
; view      = name of view goal, e.g. avl-tree
; includes  = list of included fields, e.g. ((contents string))
;             if the type of a field is (tuple ...) it is spliced in.
; e.g.(ap-make-carrier-type 'avl-tree '((contents (tuple (foo bar) (baz del)))))
; name      = name of the carrier record type
; viewspecs = list of selections for making the view, e.g. ((sort-value name))
(defun ap-make-carrier-type (view includes &optional (language 'lisp)
				  name viewspecs)
  (let (pattern str strb prev val)
    (when (setq pattern (glcarrier view))
      (if (setq prev (some #'(lambda (x)
			       (if name
				   (and (eq name (car x)) x)
				   (and (eq language (cadr x))
					(set-equal includes (caddr x))
					x)))
			   (glcarriers view)))
	  (car prev)          ; use existing one if already defined
	  (progn
	    (or name (setq name (ap-unique-name view)))
	    (setq str (cons (caar pattern)
			    (append
			     (mapcan #'(lambda (pair)
					 (if (setq val
						   (assoc (car pair) includes))
					     (if (and (consp (cadr val))
						      (eq (caadr val) 'tuple))
						 (copy-tree (cdadr val))
					         (list val))
					     (list pair)))
				     (subst (list '^ name) 'pointer
					    (cdar pattern)))
			     (subset #'(lambda (pair)
					 (not (assoc (car pair)
						     (cdar pattern))))
				     includes))))
	    (setq strb (case language
			 (lisp str)
			 ((c c++ java pascal)
			  (cons 'crecord (cons name (cdr str))))))
	    (eval (list 'glispobjects
			(list name strb 'viewspecs
			      (list (cons view
					  (cons view
						(append viewspecs
							(cadr pattern))))))))
	    (push (list name language includes) (glcarriers view))
	    name ) ) ) ))

; 31 Dec 96
; Make sure a name is unique
(defun ap-unique-name (name)
  (if (symbol-plist name)
      (gentemp (symbol-name name))
      name))

(defvar *ap-server-make-program-break* nil)

; 04 May 95; 05 May 95; 09 May 95; 12 May 95; 29 May 95; 30 May 95; 31 May 95
; 10 Aug 95; 18 Aug 95; 31 Dec 96; 07 Jan 97; 21 Jan 97; 23 Jan 97; 24 Sep 97
; 16 Jul 98; 17 May 02; 19 Jan 04; 28 Oct 04
; Make a specialized version of a generic program.
(gldefun ap-server-make-program ((w window))
  (let (user-type view-name view-type propnames selector fnname genericfn
	(args (listof glnametype)) (callargs (listof anything)) specfn nm
        printfns resultstr propent
	(arglist (listof (list (name symbol) (value anything) (type gltype))))
	poss arg-type arg-view-name (prevs (listof anything)) tmp quiet)
    (if (and 
	  (user-type = (wio-cond-menu (mapcar #'car *ap-server-views*) w
			   "Specify your type" 306))
	  (view-name = (wio-cond-menu
			  (mapcar #'car
				  (cdr (assoc user-type *ap-server-views*)))
			  w "Specify view to use" 306))
;	  (view-type = (cadr (assoc view-name (cdr (assoc user-type
    ;                                                      *ap-server-views*)))))
   (view-type = (ap-find-view-type view-name))  ; 24 Apr 17
	  (propnames = (ap-find-propnames view-type))
	  (selector = (wio-cond-menu propnames w
				       "Select the function you want" 306)) )
	(progn (*glspecfnscompiled* = nil)
	     (quiet = *glquietflg*)
	     (*glquietflg* = *ap-server-quietflg*)
	     (fnname = (glmkatom selector))
; args = arguments and types for gldefun
	     (arg-type = (cadr (assoc view-name
				      (cdr (assoc user-type
						  *ap-server-views*)))))
	     (if (eq (glxtrtypeb arg-type) (glxtrtypeb user-type))
		 (arg-type = user-type))
	     (args = (list (a glnametype name (glmkatom arg-type)
				         type arg-type)))
; callargs = view names and args for parameters in call
	     (callargs = (list (list view-name (name (first args)))))
	     (if (and (setq propent (glpropent view-type selector))
		      (symbolp (cadr propent))
		      (genericfn = (cadr propent))
		      (arglist =
			       (glarglist (second (gloriginalexpr genericfn)))))
		 (progn
	          (prevs = (list (list (caar arglist) arg-type view-name)))
		  (for pair in (cdr arglist) do
		    (nm = (glmkatom (name pair)))
		    (if (and (consp (type pair))
			     (eq (car (type pair)) 'typeof))
		        (if (symbolp (cadr (type pair)))
			    (progn (tmp = (assoc (cadr (type pair)) prevs))
				   (arg-type = (second tmp))
				   (arg-view-name = (third tmp)))
			    (progn (arg-type =
					 (ap-server-eval-type prevs
						    (cadr (type pair))))
				      (arg-view-name = nil)))
			(progn
			  (poss = (ap-server-arg-possibilities (type pair)))
			  (arg-type = (wio-cond-menu poss w
				        (format nil "Type for arg ~A" nm) 306))
			  (arg-view-name =
			    (wio-cond-menu
			      (or (subset #'(lambda (x)
				    (or (eq x (type pair))
					(gldescendantp
					  (cadr (assoc x
						  (getf (cdr (glstr arg-type))
							'views)))
					  (type pair))))
					  (mapcar #'car
						  (cdr (assoc arg-type
						      *ap-server-views*))))
				  (mapcar #'car
					  (cdr (assoc arg-type
						      *ap-server-views*))))
				w "Specify view to use" 306)) ))
		    (args _+ (a glnametype name nm
				           type (or arg-type (type pair))))
		    (callargs _+ (if arg-view-name
				    (list arg-view-name nm)
				    nm))
		    (prevs +_ (list (name pair) arg-type arg-view-name)) ) ))
	     (eval (list 'gldefun fnname
			 (mapcar #'(lambda (pair)
				     (ap-server-name-type
				       (first pair) (second pair)))
				 args)
			 (cons selector callargs)))
	     (glcc fnname)
	     (printfns = (or (nreverse *glspecfnscompiled*)
			       (list fnname)))
	     (for fnname in printfns
		  when (and (setq specfn (caar (get fnname 'glspecialization)))
			    (get specfn 'java-static))
		  (setf (get fnname 'java-static) t))
	     (with-open-stream (stream (make-string-output-stream))
	       (ap-server-fn-header stream)
	       (for comment in (or (glinfo genericfn) (glinfo selector))
		 (princ (ap-server-make-comment comment) stream)
		 (terpri stream))
	       (for printfn in printfns
		    (ap-server-print-fn printfn stream *ap-server-language*)
		    (terpri stream))
	       (setq resultstr (get-output-stream-string stream))
	       (terpri) (princ resultstr) (terpri)
	       (ap-server-erase-bottom w)
	       (window-print-line w resultstr
				  20 (ap-server-echo-y))) ))
    (*glquietflg* = quiet)
    (if *ap-server-make-program-break* (error "ap-server-make-program")) ))

; 29 May 95; 30 May 95; 31 May 96; 03 Jan 97
; Get user types that are possibilities to match an arg type
(defun ap-server-arg-possibilities (argtype)
  (if (not (glbasictypep argtype))
      (mapcan #'(lambda (lst)
		  (if (or (assoc argtype (cdr lst))
			  (some #'(lambda (x)
				    (gldescendantp (glclmaintype (cadr x))
						   argtype))
				(cdr lst)))
		      (list (car lst))))
	      *ap-server-views*)))

; 10 Aug 95
; Try to determine argument type by evaluating a type expression by compiling it
; prevs = ((var type view-name)*)    e.g. expr = (sort-value (^. var))
(defun ap-server-eval-type (prevs expr)
  (let (args)
    (setq args (mapcar #'(lambda (l)
			   (intern (concatenate 'string (symbol-name (first l))
						":" (symbol-name (second l))) ))
		       prevs))
    (dolist (arg prevs)
      (setq expr (subst (list (third arg) (first arg)) (first arg) expr)) )
    (eval (list 'gldefun 'ap-server-type-fn args expr))
    (glcc 'ap-server-type-fn)
    (glfnresulttype 'ap-server-type-fn) ))

; 31 May 95
; Get property entry from type definition.
(defun glpropent (type selector)
  (or (glstrprop type 'prop selector nil)
      (glstrprop type 'msg selector nil)
      (glstrprop type 'adj selector nil)
      (glstrprop type 'isa selector nil)))

; 10 May 95; 01 Jun 95; 10 Aug 95; 12 Jan 96; 15 Mar 96; 03 Jan 97; 07 Jan 97
(gldefun ap-server-fn-header (stream)
  (format stream "~A~%"
    (ap-server-make-comment
      "This code was produced by the Automatic Programming Server,"))
  (format stream "~A~%"
    (ap-server-make-comment
      (concatenate 'string "University of Texas at Austin, "
		   (get-time-string) ".") ))
  (format stream "~A~%"
    (ap-server-make-comment
      "http://www.cs.utexas.edu/users/novak")) )

; 10 May 95; 31 May 95; 10 Jan 96; 19 Dec 96; 20 Dec 96; 06 Jan 97; 07 Jan 97
; 28 Oct 97; 28 Oct 04
(gldefun ap-server-print-fn (printfn stream language)
  (case language
    (lisp
       (princ "; Arg types:   " stream)
       (mapc #'(lambda (vartype) (format stream "~A:~A "
					 (car vartype) (cadr vartype)))
	     (glarguments printfn))
       (terpri stream)
       (format stream "; Result type: ~A~%" (glfnresulttype printfn))
	  (write (cons 'defun (cons printfn
				    (sublis '((*glfalse* . nil)
					      (*glnull* . nil))
					    (cdr (glcompiled printfn)))))
		 :stream stream :pretty t)
	  (terpri stream) (terpri stream))
    ((c c++ java pascal)
       (gltolang printfn language stream)
       (terpri stream)) ) )

; 10 May 95; 12 May 95; 31 Dec 96; 17 May 02; 19 Jan 04; 28 Oct 04
; Make a program to convert one type of data to another
(gldefun ap-server-convert-data ((w window))
  (let (source-type possible-goals goal-type fncode fn view-name resultstr)
    (if (and 
	  (source-type = (wio-cond-menu (mapcar #'car *ap-server-views*) w
			   "Specify source type" 306))
	  (view-name = (wio-cond-menu
			   (mapcar #'car
				   (cdr (assoc source-type *ap-server-views*)))
			  w "Specify view to use" 306))
	  (possible-goals = (for x in *ap-server-views*
				   when (and (not (eq (type x) source-type))
					     (assoc view-name (views x)))
				   collect (type x)))
	  (goal-type = (wio-cond-menu possible-goals w
					"Specify goal type" 306))
	  (fncode = (gleqns-transfer-by-view goal-type source-type
					       view-name view-name)))
       (progn (fn = (intern (concatenate 'string (symbol-name source-type)
					  "-TO-" (symbol-name goal-type))))
	     (eval (cons 'gldefun (cons fn (cdr fncode))))
	     (glcc fn)
	     (with-open-stream (stream (make-string-output-stream))
	       (ap-server-fn-header stream)
	       (ap-server-print-fn fn stream *ap-server-language*)
	       (resultstr = (get-output-stream-string stream))
	       (terpri) (princ resultstr) (terpri)
	       (ap-server-erase-bottom w)
	       (window-print-line w resultstr
				  20 (ap-server-echo-y)) ) ) )))

; 01 Jun 95; 26 Jul 95; 08 Aug 95; 09 Aug 95; 10 Aug 95; 07 Jan 97; 17 May 02
; 19 Jan 04
(gldefun ap-server-documentation ((w window))
  (let (view-name view-type propnames selector propent resultstr
	context (arglist (listof glnametype)) typ restype str fnname)
    (if (view-name = (menu (append *ap-server-doc* *ap-server-data-views*)))
        (if (member view-name *ap-server-doc*)
	    (ap-server-print-doc w (glinfo view-name))
	 (if (and 
	  (view-type = (ap-find-view-type view-name))
	  (propnames = (ap-find-propnames view-type))
	  (selector = (wio-cond-menu (if (glinfo view-name)
					 (cons (cons view-name 'itself)
					       propnames)
					 propnames)
		            w "Select which to document" 306))
	  (if (eq selector 'itself)
	      (ap-server-print-doc w (glinfo view-name))
	      t)
	  (propent = (glpropent view-type selector)) )
       (progn (if (consp (cadr propent))
		  (if (member (caadr propent) '(glambda lambda))
		      (arglist = (cadadr propent))
		      (arglist = (list (list 'self view-name))) )
		  (if (symbolp (cadr propent))
			  (progn (fnname = (cadr propent))
				 (arglist = (glarglist
					       (cadr (gloriginalexpr
						       fnname))))) ))
	     (context = (list nil))
	     (for pair in arglist
	       (typ = (glevalstr (type pair) context))
	       (push (list 'type (name pair) type)
		     (car context)) )
	     (restype = (glevalstr
			    (or (getf (cddr propent) 'result)
				(and (symbolp (cadr propent))
				     (glresulttype (cadr propent) nil))
				(glfnresulttype selector))
			    context)))
        (with-open-stream (stream (make-string-output-stream))
	  (format stream "Documentation of ~A for ~A~%" selector view-name)
	  (format stream "Arguments:       ")
	  (for item in (nreverse (car context))
	    (format stream "~A : ~A  " (second item) (third item)) )
	  (terpri stream)
	  (if restype (format stream "Result type:     ~A~%" restype))
	  (terpri stream)
	  (if (and (or fnname selector)
		   (setq str (or (get fnname 'glinfo) (get selector 'glinfo))))
	      (if (stringp (car str))
		  (dolist (s str) (princ s stream) (terpri stream))
	          (progn (princ str stream) (terpri stream))))
	  (setq resultstr (get-output-stream-string stream))
	  (terpri) (princ resultstr) (terpri)
	  (ap-server-erase-bottom w)
	  (window-print-line w resultstr 20 (ap-server-echo-y)) ) )))))

; 08 Aug 95
(gldefun ap-server-print-doc ((w window) (lines (listof string)))
  (let (resultstr)
    (with-open-stream (stream (make-string-output-stream))
      (for line in lines (princ line stream) (terpri stream))
      (setq resultstr (get-output-stream-string stream))
      (terpri) (princ resultstr) (terpri)
      (ap-server-erase-bottom w)
      (window-print-line w resultstr 20 (- *wio-window-height* 320)) ) ))

; 10 Aug 95; 20 Dec 96; 23 Dec 96
; Make a string into a comment for the output language
(defun ap-server-make-comment (str &optional short)
  (case *ap-server-language*
    (lisp (concatenate 'string "; " str))
    (c (concatenate 'string "/* " str " */"))
    ((c++ java)
      (if short (concatenate 'string "/* " str " */")
	        (concatenate 'string "// " str)))
    (pascal (concatenate 'string "{ " str " }"))  ))
