/* picoIO.c
 * 10jul91abu
 */

#include "pico.h"
#include <fcntl.h>

Afile *streamSave;

/* Prototypes */
static void cmprWhite(int*);
static void doComment(void);
static int fileRead(Afile*);
static void fileWrite(char,Afile*);
static int fillNext(void);
static void picoFlush(Afile*);
static Afile *getStream(pico);
static Afile *picoOpen(pico,int);
static void prLine(pico,bool);
static pico rdList(void);
static char skipWhite(void);
static bool symChar(char);
static void testEsc(int*);


Afile *getStream(x)
register pico x;
{
   if (isNil(x))
      return (Afile*)NULL;
   NEEDNUM(x);
   return (Afile*)unBox(x);
}

void setStream(f)
Afile *f;
{
   if (stream)
      stream->nextChar = nextChar;
   else
      nextSave = nextChar;
   if (stream = f)
      nextChar = stream->nextChar;
   else
      nextChar = nextSave;
}

pico Align(x)
register pico x;
{
   register number n;

   push(EVAL1(x));
   NEEDSTRING(tos);
   x = cdr(x);
   x = EVAL1(x);
   NEEDNUM(x);
   n = unBox(x) - strLength(tos);
   if (n > 0) {
      if (isNum(x = car(tos))  &&  unBox(x) < 0)
         tos = newCell(boxNum(unBox(x) - n), cdr(tos));
      else
         tos = newCell(boxNum(-n), tos);
   }
   return pop();
}

pico Pipe(x)
register pico x;
{
   int c;
   Afile *src,*dst;

   src = picoOpen(EVAL1(x),O_RDONLY);
   x = cdr(x);
   dst = picoOpen(EVAL1(x),O_CREAT|O_TRUNC|O_WRONLY);
   if (!src || !dst)
      err("Can't pipe");
   x = cdr(x);
   push(EVAL1(x));
   while ((c = fileRead(src)) >= 0) {
      x = apply1(tos,boxNum(c));
      if (isNum(x))
         fileWrite(unBox(x),dst);
      else {
         while (isCell(x)) {
            fileWrite(unBox(car(x)),dst);
            x = cdr(x);
         }
      }
   }
   drop();
   closeFile(src);
   closeFile(dst);
   return tSym;
}

Afile *picoOpen(x,flg)
register pico x;
{
   char fName[FILENAME];
   int fd;
   Afile *f;

   bufString(x, fName, FILENAME);
   if ((fd = open(fName, flg, 0666)) < 0)
      return (Afile*)NULL;
   f = (Afile*)malloc(sizeof(Afile));
   f->fd = fd;
   f->cnt = 0;
   f->max = 0;
   f->dirty = NO;
   f->pos = 0;
   f->nextChar = 0;
   val(fileSym) = newCell(newCell(boxNum(f), x), val(fileSym));
   return f;
}

void closeFile(f)
Afile *f;
{
   register pico x,y;

   x = boxNum(f);
   y = val(fileSym);
   if (x == car(car(y)))
      val(fileSym) = cdr(y);
   else  {
      while (isCell(cdr(y))) {
         if (x == car(car(cdr(y)))) {
            cdr(y) = cdr(cdr(y));
            goto doClose;
         }
         y = cdr(y);
      }
      errObj(x, "Can't close");
   }
doClose:
   picoFlush(f);
   if (close(f->fd) < 0)
      errObj(x, "File close error");
   free((char *)f);
}

void picoFlush(f)
register Afile *f;
{
   if (f->dirty) {
      if (lseek(f->fd, f->pos, 0) < 0  ||
               write(f->fd, f->buf, f->max) != f->max )
         err("Write error");
      f->dirty = NO;
   }
}

int fileRead(f)
register Afile *f;
{
   long count;

   if (f->cnt == f->max) {
      picoFlush(f);
      if ((f->pos = lseek(f->fd,0,1)) < 0 ||
            (count = read(f->fd, f->buf, BUFSIZE)) < 0)
         err("File read error");
      if (!count)
         return -1;
      f->cnt = 0;
      f->max = count;
      f->dirty = NO;
   }
   return f->buf[f->cnt++] & 0xFF;
}

void fileWrite(c,f)
char c;
register Afile *f;
{
   if (f->cnt == BUFSIZE) {
      picoFlush(f);
      if ((f->pos = lseek(f->fd,0,1)) < 0)
         err("Write seek error");
      f->cnt = 0;
      f->max = 0;
   }
   f->buf[f->cnt++] = c;
   f->dirty = YES;
   if (f->max < f->cnt)
      f->max = f->cnt;
}

pico Load(x)
register pico x;
{
   Afile *str, *streamSave;
   pico fName, f2Name;

   fName  =  isCell(x)?  EVAL1(x) : car(val(editSym));
   if (isNum(fName) || isSym(fName)) {
      if (isNil(f2Name = EVAL1(cdr(x))))
         x = get(fName,srcFlg);
      else
         x = get(f2Name,fName);
      if (isNil(x))
         errObj(fName, "Can't load");
      fName = cdr(x);
   }
   NEEDSTRING(fName);
   if (!(str = picoOpen(fName,O_RDONLY)))
      errStrObj(fName, "File open error");
   streamSave = stream;
   setStream(str);
   val(loadSym) = newCell(newCell(ONE, fName), val(loadSym));
   revalo(nilSym);
   val(loadSym) = cdr(val(loadSym));
   closeFile(stream);
   setStream(streamSave);
   return tSym;
}

pico Open(x)
pico x;
{
   Afile *f;

   if (!(f = picoOpen(EVAL1(x),O_RDWR)))
      return nilSym;
   return boxNum(f);
}

pico Creat(x)
register pico x;
{
   Afile *f;

   if (!(f = picoOpen(EVAL1(x), O_CREAT|O_TRUNC|O_RDWR)))
      return nilSym;
   return boxNum(f);
}

pico Close(x)
register pico x;
{
   x = EVAL1(x);
   NEEDNUM(x);
   closeFile((Afile*)unBox(x));
   return x;
}

pico Erase(x)
pico x;
{
   char buf[FILENAME];

   bufString(EVAL1(x), buf, FILENAME);
   return boxNum(unlink(buf));
}

pico fRename(x)
register pico x;
{
   char old[FILENAME];
   char new[FILENAME];

   bufString(EVAL1(x), old, FILENAME);
   bufString(EVAL1(cdr(x)), new, FILENAME);
   return boxBool(link(old,new) >= 0  &&  unlink(old) >= 0);
}

pico Where(x)
pico x;
{
   register Afile *f;

   f = getStream(EVAL1(x));
   return boxNum(f->pos + f->cnt);
}

pico Seek(x)
register pico x;
{
   register pico y;
   register Afile *f;

   y = EVAL1(x);
   NEEDNUM(y);
   x = cdr(x);
   f = getStream(EVAL1(x));
   picoFlush(f);
   if (lseek(f->fd, f->pos = unBox(y), 0) < 0)
      return nilSym;
   f->cnt = f->max = 0;
   f->nextChar = 0;
   return y;
}

pico FSize(x)
register pico x;
{
   register Afile *f;

   f = getStream(EVAL1(x));
   picoFlush(f);
   x = boxNum(lseek(f->fd, 0, 2));
   if (lseek(f->fd, f->pos, 0) < 0)
      return nilSym;
   return x;
}

pico Read(x)
register pico x;
{
   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   x = read0(YES);
   setStream(streamSave);
   return x;
}

pico Raw(x)
register pico x;
{
   Afile *f;
   long buffer;

   f = (Afile*)EVAL1(x);
   NEEDNUM(f);
   f = (Afile*)unBox(f);
   if (isCell(x = cdr(x))) {
      x = EVAL1(x);
      buffer = unBox(x);
      return (write(f->fd, &(char*)buffer, 4L) == 4)? x : nilSym;
   }
   return (read(f->fd, &(char*)buffer, 4L) == 4)? boxNum(buffer) : nilSym;
}

pico ReadBlock(x)
pico x;
{
   Afile *f;
   long count;
   char *buffer;

   f = (Afile*)nextNum(&x);
   buffer = (char*)nextNum(&x);
   count = nextNum(&x);
   if ((count = read(f->fd, buffer, count)) < 0)
      return nilSym;
   return boxNum(count);
}

pico WriteBlock(x)
pico x;
{
   Afile *f;
   long count;
   char *buffer;

   f = (Afile*)nextNum(&x);
   buffer = (char*)nextNum(&x);
   count = nextNum(&x);
   if ((count = write(f->fd, buffer, count)) < 0)
      return nilSym;
   return boxNum(count);
}

void prLine(x,suppress)
register pico x;
bool suppress;
{
   register int c, dummy;

   NEEDSTRING(x);
   while (isCell(x)) {
      keyBreak();
      c = unBox(car(x));
      x = cdr(x);
      if (c >= 0)
         chrOut(c);
      else if (suppress && !isCell(x))
         return;
      else do {
         keyBreak();
         space();
      } while (++c);
   }
}

pico Prin2(x)
register pico x;
{
   push(EVAL1(x));
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   x = pop();
   if (isCell(x) || isNil(x))
      prLine(x,NO);
   else
      prin0(x);
   setStream(streamSave);
   return x;
}

pico PrLine(x)
register pico x;
{
   push(EVAL1(x));
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   prLine(x = xpop(),YES);
   crlf();
   setStream(streamSave);
   return x;
}

pico Prin1(x)
register pico x;
{
   push(EVAL1(x));
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   prin0(x = pop());
   setStream(streamSave);
   return x;
}

pico Print(x)
register pico x;
{
   push(EVAL1(x));
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   prin0(x = pop());
   crlf();
   setStream(streamSave);
   return x;
}

pico PrHex(x)
register pico x;
{
   push(EVAL1(x));
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   x = pop();
   NEEDNUM(x);
   prHexNum((unsigned long)(unBox(x)));
   setStream(streamSave);
   return x;
}

pico CutPr(x)
register pico x;
{
   push(EVAL1(x));
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   cutPr(x = pop());
   setStream(streamSave);
   return x;
}

pico Comment(x)
register pico x;
{
   push(EVAL1(x));
   x = cdr(x);
   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   chrOut('[');
   prLine(tos,NO);
   chrOut(']');
   space();
   setStream(streamSave);
   return pop();
}

pico Terpri(x)
pico x;
{
   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   crlf();
   setStream(streamSave);
   return nilSym;
}

pico Space(x)
pico x;
{
   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   space();
   setStream(streamSave);
   return nilSym;
}

pico Tab(x)
pico x;
{
   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   tab();
   setStream(streamSave);
   return nilSym;
}

pico Bell(x)
pico x;
{
   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   chrOut(7);
   setStream(streamSave);
   return nilSym;
}

pico Backsp(x)
pico x;
{
   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   chrOut(8);
   setStream(streamSave);
   return nilSym;
}

pico Putc(x)
register pico x;
{
   push(EVAL1(x));
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   if (isNum(x = pop()))
      chrOut(unBox(x));
   setStream(streamSave);
   return x;
}

pico Getc(x)
register pico x;
{
   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   if (!nextChar)
      fillNext();
   if (nextChar >= 0)
      x = boxNum(nextChar);
   else
      x =  nilSym;
   fillNext();
   setStream(streamSave);
   return x;
}

void cmprWhite(p)
register int *p;
{
   if (*p == ' ')
      *p = -1;
   else if (*p == 9)
      *p = -TABLEN;
}

pico unBufString(cnt,s)
register int cnt;
unsigned char *s;
{
   int c;
   register pico x;

   if (!cnt)
      return nilSym;
   c = *s++;
   cmprWhite(&c);
   push(x = newCell(boxNum(c), nilSym));
   while (--cnt > 0) {
      c = *s++;
      cmprWhite(&c);
      if (c < 0  &&  num(car(x)) < 0)
         car(x) = boxNum(c + unBox(car(x)));
      else {
         cdr(x) = newCell(boxNum(c), nilSym);
         x = cdr(x);
      }
   }
   return pop();
}

pico unBufCString(s)
char *s;
{
   int c;
   register pico x;

   if (!*s)
      return nilSym;
   c = *s++;
   cmprWhite(&c);
   push(x = newCell(boxNum(c), nilSym));
   while (*s) {
      c = *s++;
      cmprWhite(&c);
      if (c < 0  &&  num(car(x)) < 0)
         car(x) = boxNum(c + unBox(car(x)));
      else {
         cdr(x) = newCell(boxNum(c), nilSym);
         x = cdr(x);
      }
   }
   return pop();
}

pico GetLine(x)
register pico x;
{
   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   if (!nextChar)
      fillNext();
   if (nextChar < 0)
      x = nilSym;
   else if (!nextChar || nextChar==EOL) {
      x = nilSym;
      if (stream)
         x = newCell(boxNum(-1),x);
   }
   else {
      cmprWhite(&nextChar);
      push(x = newCell(boxNum(nextChar), nilSym));
      while ((fillNext(), nextChar > 0) && nextChar != EOL) {
         cmprWhite(&nextChar);
         if (nextChar < 0  &&  num(car(x)) < 0)
            car(x) = boxNum(nextChar + unBox(car(x)));
         else {
            cdr(x) = newCell(boxNum(nextChar), nilSym);
            x = cdr(x);
         }
      }
      x = shareList(pop());
   }
   if (stream)
      fillNext();
   setStream(streamSave);
   return x;
}

/* Character I/O */
char upc(c)
register char c;
{
   return (c<'a' || c>'z')? c : (c-32);
}

void chrOut(c)
char c;
{
   if (stream)
      fileWrite(c,stream);
   else {
      if (c == EOL) {
         ttyOut(CR);
         ttyOut(NL);
      }
      else
         ttyOut(c);
   }
}

void space()
{
   chrOut(' ');
}

void crlf()
{
   chrOut(EOL);
}

void tab()
{
   prString("   ");
}

void prString(s)
register char *s;
{
   while (*s)
      chrOut(*s++);
}

void prNumber(n)
register number n;
{
   char buf[16];
   register int i = 0;

   if (n < 0){
      chrOut('-');
      n = -n;
   }
   do {
      buf[i++] = n % 10 + '0';
   } while (n /= 10);
   while (--i >= 0)
      chrOut(buf[i]);
}

void hexChar(c)
register char c;
{
   if ((c += '0') > '9')
      c += 7;
   chrOut(c);
}

void prHexNum(n)
register unsigned long n;
{
   char buf[16];
   register int i = 0;

   keyBreak();
   chrOut('0');
   do {
      buf[i++] = n % 16;
   } while (n /= 16);
   while (--i >= 0)
      hexChar(buf[i]);
}

int fillNext()
{
   register int c;
   Afile *sSave;

   c = nextChar;
   if (!stream) {
      if (!c) {
         if (revaLevel)
            chrOut(revaLevel + '0');
         chrOut(':');
         lbp = getLine(lBuff, lBuff+LBSIZE-1);
         crlf();
         c = EOL;
      }
      nextChar = *lbp++;
   }
   else {
      if (!isNil(val(echoFlg)) && c) {
         sSave = stream;
         stream = NULL;
         chrOut(c);
         stream = sSave;
      }
      if ((nextChar = fileRead(stream)) >= 0) {
         if (nextChar == EOL  &&  isCell(car(val(loadSym))))
            car(car(val(loadSym))) = (pico)(num(car(car(val(loadSym)))) + 4);
      }
   }
   return c;
}

char chrIn()
{
   int c, dummy;

   if ((c = fillNext()) < 0)
      err("Read past End of file");
   return c;
}

/* Skip nested Comments */
void doComment()
{
   chrIn();
   if (nextChar == '#') {
      register pico x,y;

      chrIn();
      y = read0(NO);
      if ((x = EVAL(y)) == voidSym)
         errObj(y,"Undefined Conditional");
      if (!isNil(x))
         return;
   }
   while (nextChar != ']')
      if (nextChar == '[')
         doComment();
      else
         chrIn();
   chrIn();
}

/* Skip White Space */
char skipWhite()
{
   loop {
      while (nextChar <= ' ' || nextChar == ']')
         chrIn();
      if (nextChar != '[')
         return nextChar;
      doComment();
   }
}

void testEsc(p)
register int *p;
{
   if (*p == '\\')
      *p = chrIn();
   else if (*p == '^')
      *p = chrIn() & 0x1F;
   cmprWhite(p);
}

bool symChar(c)
register char c;
{
   return (c>='A' && c<='Z' || c=='1' || c=='2' || c=='$' || c=='*' || c=='-');
}

/* Read one Expression */
pico read0(top)
bool top;
{
   register pico x,y;
   register int i;
   number n;
   int c, sign;
   bool first, dummy;

   do
      skipWhite();
   while ((c = chrIn()) == ')'  ||  c == '>');
   if (top) {
      x = car(val(loadSym));
      loadPos = car(x);
      loadName = cdr(x);
   }
   if (c == '(') {
      x =  rdList();
      if (top && nextChar == '>')
         chrIn();
   }
   else if (c == '<') {
      x = rdList();
      if (chrIn() != '>')
         err("Super parentheses mismatch");
   }
   else if (c == '\'')
      x = share(quoteSym, share(read0(NO), nilSym));
   else if (c == '\\')
      x = boxNum(chrIn());
   else if (c == '^')
      x = boxNum(chrIn() & 0x1F);
   else if (c == '#') {
      y = read0(NO);
      if ((x = EVAL(y)) == voidSym)
         errObj(y,"Undefined Read Macro");
   }
   else if (c == ';')
      x = nilSym;
   else if (c == '"') {
      if ((c = chrIn()) == '"')
         x = nilSym;
      else {
         testEsc(&c);
         push(x = newCell(boxNum(c), nilSym));
         while ((c = chrIn()) != '"') {
            testEsc(&c);
            if (c < 0  &&  num(car(x)) < 0)
               car(x) = boxNum(c + unBox(car(x)));
            else {
               cdr(x) = newCell(boxNum(c), nilSym);
               x = cdr(x);
            }
         }
         x = shareList(pop());
      }
   }
   else if (c>='0' && c<='9' || c=='+' ||
                c=='-' && nextChar>='0' && nextChar<='9') {
      sign = 0;
      if (c=='+' || c=='-' && ++sign)
         c = chrIn();
      if (c < '0' || c > '9')
         err("Reading bad number");
      if (n = c - '0')
         while ((c = nextChar) >= '0' && c <= '9') {
            n  =  n * 10 + c - '0';
            chrIn();
         }
      else
         while ((c = upc(nextChar))>='0' && c<='9' || c>='A' && c<='F') {
            if ((c -= '0') > 9)
               c -= 7;
            n  =  (n << 4) + c;
            chrIn();
         }
      x = boxNum(sign? -n : n);
   }
   else {
      i = 5;
      first = YES;
      if (!symChar(c = upc(c)))
         err("Bad character");
      accumulate(c,&n);
      while (symChar(c=upc(nextChar))) {
         if (--i < 0) {
            if (first) {
               first = NO;
               push(x = newCell(boxNum(n),nilSym));
            }
            else {
               cdr(x) = newCell(boxNum(n),nilSym);
               x = cdr(x);
            }
            i = 5;
         }
         accumulate(c,&n);
         chrIn();
      }
      while (--i >= 0)
         accumulate(0,&n);
      if (first)
         y = boxNum(n);
      else {
         cdr(x) = boxNum(n);
         y = pop();
      }
      if (!(x = find(y)))
         x = intern(newSym(y,voidSym));
   }
   return x;
}

/* Read a List */
pico rdList()
{
   register pico x;
   register int c;

   if ((c = skipWhite()) == ')') {
      chrIn();
      return nilSym;
   }
   if (c == '>')
      return nilSym;
   if (c == '.') {
      chrIn();
      x = read0(NO);
      if (skipWhite() != '>'  &&  chrIn() != ')')
         err("Reading bad dotted pair");
      return x;
   }
   push(read0(NO));
   x = rdList();
   return share(xpop(), x);
}

/* Print one expression */
void prin0(x)
register pico x;
{
   register int strFlg;

   keyBreak();
   if (isNum(x))
      prNumber(unBox(x));
   else if (isSym(x))
      prName(getPname(x));
   else {
      /* List or dotted pair */
      chrOut('(');
      loop {
         prin0(car(x));
         if (isNil(x = cdr(x)))
            break;
         if (!isCell(x)) {
            prString(" . ");
            prin0(x);
            break;
         }
         space();
      }
      chrOut(')');
   }
}

void cutPr(x)
register pico x;
{
   if (val(printFlg) != nilSym  ||  isNum(x)  ||  isSym(x))
      prin0(x);
   else {
      chrOut('(');
      cutPr(car(x));
      if (!isNil(x = cdr(x))) {
         if (!isCell(x)) {
            prString(" . ");
            prin0(x);
         }
         else {
            space();
            cutPr(car(x));
            if (!isNil(cdr(x)))
               prString(" ..");
         }
      }
      chrOut(')');
   }
}
