/* picoSyst.c
 * 16feb90abu
 */

#include "pico.h"
#include <sys/types.h>
#include <malloc.h>
#include <memory.h>
#include <fcntl.h>
#include <sys/ndir.h>
#include <sgtty.h>
#include <time.h>
#include <sys/times.h>


/* Prototypes */
static void crash(long);
static void doBs(void);
static void echoChar(char);
static bool fKey(char);
static bool keyHit(void);
static char readKey(void);
static void waitTTY(void);


char charQueue[1];
bool qFull = NO;
struct sgttyb ttyOld,ttyNew,auxOld,auxNew;


/* Init serial driver */
void initSerial()
{
   if ((ttyDev = open(lName, O_RDWR)) == -1)
      giveup("Can't open serial device");

   gtty(ttyDev,&ttyOld);
   gtty(ttyDev,&ttyNew);
   ttyNew.sg_flags |= RAW;
   ttyNew.sg_flags &= ~(ECHO | CRMOD | ODDP);
   stty(ttyDev,&ttyNew);
}

void exitPico(flag)
int flag;
{
   stopApplication();
   stty(ttyDev,&ttyOld);
   if (auxDev)
      stty(auxDev,&auxOld);
   exit(flag);
}

pico System(x)
register pico x;
{
   char buf[1024];

   bufString(EVAL1(x),buf,1024);
   stty(ttyDev,&ttyOld);
   x = boxNum(system(buf));
   stty(ttyDev,&ttyNew);
   return x;
}

static pico crashFun;

void crash(n)
long n;
{
   apply1(crashFun,boxNum(n));
}

pico Crash(x)
register pico x;
{
   register number n;

   n = num(EVAL1(x)); /* Signal number */
   NEEDNUM(n);
   x = cdr(x);
   if (isNil(x = EVAL1(x)))
      signal(unBox(n), SIG_DFL);
   else {
      crashFun = x;
      signal(unBox(n), crash);
   }
   return tSym;
}

pico Check()
{
   keyBreak();
   return tSym;
}

void initSysSyms()
{
}

void initSysVars()
{
}

pico Path()
{
   return nilSym;
}

pico Dir(x)
register pico x;
{
   char fName[FILENAME];
   int fd;
   DIR *dp;
   struct direct *d;

   bufString(EVAL1(x), fName, FILENAME);
   if (!(dp = opendir(fName)))
      return nilSym;
   push(nilSym);
   while (d = readdir(dp))
      tos = newCell(unBufCString(d->d_name),tos);
   return pop();
}

pico pDate()
{
   struct tm *p;
   long clock;

   time(&clock);
   p = localtime(&clock);
   return packNum(p->tm_year, p->tm_mon+1, p->tm_mday);
}

pico pTime()
{
   struct tm *p;
   long clock;

   time(&clock);
   p = localtime(&clock);
   return packNum(p->tm_hour, p->tm_min, p->tm_sec);
}

pico pBench(x)
register pico x;
{
   register long n;
   struct tms tim1,tim2;
   Afile *sSave;

   times(&tim1);
   x = evalBody(x);
   times(&tim2);
   n = tim2.tms_utime - tim1.tms_utime;
   sSave = stream;
   setStream(NULL);
   prNumber(n), prString(" ticks [");
   n /= 6;
   prNumber(n/10), chrOut('.'), prNumber(n%10), prString(" sec]\n");
   stream = sSave;
   return x;
}

pico DelayTicks(x)
register pico x;
{
   register long n;
   struct tms tim;

   x = EVAL1(x);
   NEEDNUM(x);
   times(&tim);
   n = unBox(x) + tim.tms_utime + tim.tms_stime;
   while (times(&tim), (tim.tms_utime + tim.tms_stime) < n)
      keyBreak();
   return tSym;
}

pico AccString(x)
register pico x;
{
   unsigned char *p;
   register int c;
   pico y;

   p = (unsigned char*)EVAL1(x);
   NEEDNUM(p);
   p = (unsigned char*)unBox(p);
   if (isNil(x = EVAL1(cdr(x))))
      return unBufCString(p);
   y = x;
   while (isCell(x)) {
      c = unBox(car(x));
      if (c >= 0)
         *p++ = c;
      else do {
         *p++ = ' ';
      } while (++c);
      x = cdr(x);
   }
   *p = 0;
   return y;
}

pico Alloc(x)
pico x;
{
   register char c, *p, *q;
   register number size;

   size = nextNum(&x);
   q = malloc((unsigned)size);
   if (isCell(x)) {
      c = (char)nextNum(&x);
      p = q;
      while (--size >= 0)
         *p++ = c;
   }
   return boxPtr(q);
}

pico Free(x)
pico x;
{
   free((char *)nextNum(&x));
   return tSym;
}

pico Stuff(x)
register pico x;
{
   register unsigned char *p;
   register pico y;
   register number cnt;

   p = (unsigned char*)EVAL1(x);
   x = cdr(x);
   NEEDNUM(p);
   p = (unsigned char*)unBox(p);
   y = EVAL1(x);
   if (isNum(y)) {
      if (!(cnt = unBox(y)))
         return nilSym;
      --cnt;
      push(x = newCell(boxNum(*p++), nilSym));
      while (--cnt >= 0) {
         cdr(x) = newCell(boxNum(*p++), nilSym);
         x = cdr(x);
      }
      return pop();
   }
   loop {
      NEEDLIST(y);
      while (isCell(y)) {
         NEEDNUM(car(y));
         *p++ = (char)unBox(car(y));
         y = cdr(y);
      }
      if (!isCell(x = cdr(x)))
         return boxNum(p);
      y = EVAL1(x);
   }
}

pico Block(x)
register pico x;
{
   register pico src,dst;

   src = EVAL1(x);
   NEEDNUM(src);
   x = cdr(x);
   dst = EVAL1(x);
   NEEDNUM(dst);
   x = EVAL1(cdr(x));
   NEEDNUM(x);
   memcpy((char*)unBox(dst),(char*)unBox(src),unBox(x));
   return src;
}

pico Tty(x)
pico x;
{
   char buff[64];

   nextString(&x,buff,64);
   if (auxDev) {
      stty(auxDev,&auxOld);
      close(auxDev);
   }
   if ((auxDev = open(buff, O_RDWR)) == -1)
      return nilSym;
   gtty(auxDev,&auxOld);
   gtty(auxDev,&auxNew);
   auxNew.sg_flags |= RAW;
   auxNew.sg_flags &= ~(ECHO | CRMOD | ODDP);
   stty(auxDev,&auxNew);
   return boxNum(auxDev);
}

bool serialStat()
{
   return  auxDev && rdchk(auxDev) > 0;
}

int serialIn()
{
   char buff[4];

   if (!auxDev || read(auxDev, buff, 1) < 0)
      err("Serial input error");
   return buff[0];
}

void serialOut(c)
int c;
{
   char buff[4];

   buff[0] = c;
   if (!auxDev || write(auxDev, buff, 1) < 0)
      err("Serial output error");
}

pico Serial(x)
pico x;
{
   long count;

   if (isNil(x = EVAL1(x))) {
      if (!serialStat())
         return nilSym;
      return boxNum(serialIn());
   }
   NEEDNUM(x);
   serialOut(unBox(x));
   return x;
}

static void waitTTY()
{
   if (read(ttyDev, charQueue, 1) <  0)
      err("TTY input error");
}

static bool keyHit()
{
   long count;

   if (qFull)
      return YES;
   if (rdchk(ttyDev)) {
      waitTTY();
      return (qFull = YES);
   }
   return NO;
}

static char readKey()
{
   if (qFull)
      qFull = NO;
   else
      waitTTY();
   if (!isNil(val(logFlg)))
      val(logFlg) = newCell(boxNum(charQueue[0]),val(logFlg));
   return charQueue[0];
}

pico Key()
{
   if (!keyHit())
      return nilSym;
   return boxNum(readKey());
}

pico HitKey()
{
   register pico c;

   if (isCell(val(macFlg))) {
      c = car(val(macFlg));
      val(macFlg) = cdr(val(macFlg));
   }
   else
      c = boxNum(readKey());
   return c;
}

void keyBreak()
{
   if (keyHit() && charQueue[0] == BREAK) {
      qFull = NO;
      cBreak();
   }
}

void ttyOut(c)
char c;
{
   char buf[1];

   buf[0] = c;
   if (qFull && charQueue[0] == ctrl('S')) {
      read(ttyDev, charQueue, 1);
      qFull = NO;
   }
   if (write(ttyDev, buf, 1) < 0)
      err("TTY output error");
}

static void echoChar(c)
char c;
{
   ttyOut(c < ' ' ?  '_' : c);
}

static void doBs()
{
   ttyOut(8);
   ttyOut(' ');
   ttyOut(8);
}

static bool fKey(c)
char c;
{
   register pico x = val(fkeySym);

   while (isCell(x)) {
      if (unBox(car(car(x))) == c) {
         evalBody(cdr(car(x)));
         return YES;
      }
      x = cdr(x);
   }
   return NO;
}

char *getLine(first,last)
char *first,*last;
{
   register pico x;
   register char c, *s, *t;
   register int i, again;
   bool overwrite;

   s = first;
   again = -1;
   overwrite = YES;
   loop {
      if (isCell(val(macFlg))) {
         c = unBox(car(val(macFlg)));
         val(macFlg) = cdr(val(macFlg));
      }
      else {
         loadPos = NULL;
         do
            c = readKey();
         while (c == ESCAPE  &&  fKey(c = readKey()));
      }
      switch (c) {
         case EOL:
         case '\r':
            while (s <= last)
               *s++ = 0;
            return first;
         case BREAK:
            break;
         case ctrl('H'):
            if (s > first) {
               --s;
               doBs();
            }
            break;
         case ctrl('X'):
            while (s > first) {
               --s;
               doBs();
            }
            break;
         case ctrl('I'):
            for (i=0; i<TABLEN && s<last; ++i) {
               if (!overwrite) {
                  t = last;
                  while (t > s) {
                     *t = *(t-1);
                     --t;
                  }
               }
               ttyOut(*s++ = ' ');
            }
            break;
         case ctrl('A'):
            while (s > first) {
               --s;
               doBs();
            }
            x = val(againSym);
            ++again;
            for (i=0; isCell(x) && i<again; ++i)
               x = cdr(x);
            if (isCell(x)) {
               bufString(car(x), first, LBSIZE);
               while (*s)
                  echoChar(*s++);
            }
            break;
         case ctrl('Z'):
            while (s > first) {
               --s;
               doBs();
            }
            x = val(againSym);
            if (again)
               --again;
            for (i=0; isCell(x) && i<again; ++i)
               x = cdr(x);
            if (isCell(x)) {
               bufString(car(x), first, LBSIZE);
               while (*s)
                  echoChar(*s++);
            }
            break;
         case ctrl('R'):
            while (*s)
               echoChar(*s++);
            break;
         case ctrl('D'):
            t = s;
            while (*t) {
               *t = *(t+1);
               ++t;
            }
            break;
         case ctrl('O'):
            overwrite = !overwrite;
            break;
         case ctrl('F'):
            if (*s)
               echoChar(*s++);
            break;
         case ctrl('C'):
            reset();
            prString("^C\n");
            unwind();
            closeAll();
            longjmp(errRst,-1);
         default:
            if (s < last) {
               if (!overwrite) {
                  t = last;
                  while (t > s) {
                  *t = *(t-1);
                     --t;
                  }
               }
               echoChar(*s++ = c);
            }
            break;
      }
   }
}
