/* picoGC.c
 * 24jan90abu
 */

#include "pico.h"
#include "stack.h"

static pico adjLimit;

/* Prototypes */
static void adjust(pico*);
static void allFree(void);
static void compact(void);
static void deLocate(pico*,pico*);
static pico doShare(pico,pico);
static void doZap(pico);
static void freezErr(void);
static pico gc(void);
#if ASM68K
static void mark(...);
#else
static void mark(pico);
#endif
static void pushGc(pico);
static void reLocate(pico*,pico*);
static pico share1(pico);
static pico share1List(pico);

static long gcStat;	/* Incremental garbage collector status */
static pico collect, curr, avMark, avStk;
static pico *gcPtr;
static pico shareX,shareY;
static number zaps;

void resetGC()
{
	gcStat = 0;
	avStk = NULL;
	gcPtr = NULL;
	collect = heap;
}

void pushGc(x)
pico x;
{
	register pico p;

	if (!(p = avail))
		resetGC();
	else {
		avail = p->link;
		if (p == avMark)
			avMark = p->link;
		p->link = avStk;
		avStk = p;
		p->data = x;
	}
}

pico setCar(x,y)
register pico x,y;
{
	if (!gcStat)
		x->data = (pico)(num(y) | 1);
	else {
		if (!isNum(x->link) && isFree(cellPtr(x->link)))
			pushGc(x->link);
		if (!isNum(y) && isFree(cellPtr(y)))
			pushGc(y);
		x->data = clr(y); /* clr() for boxFun-number */
	}
	return y;
}

pico setCdr(x,y)
register pico x,y;
{
	if (gcStat && !isFree(x) && !isNum(y) && isFree(cellPtr(y)))
		pushGc(y);
	return x->link = y;
}

pico setTail(x,y)
register pico x,y;
{
	if (!gcStat)
		(x-1)->link = (pico)(num(y) | 1);
	else {
		if (!isNum(x->data) && isFree(cellPtr(x->data)))
			pushGc(x->data);
		if (!isNum(y) && isFree(cellPtr(y)))
			pushGc(y);
		(x-1)->link = y;
	}
	return y;
}

pico setVal(x,y)
register pico x,y;
{
	if (gcStat && !isFree(x) && !isNum(y) && isFree(cellPtr(y)))
		pushGc(y);
	return x->data = y;
}

pico dynamo(foo,x)
register pico foo,x;
{
	NEEDFUN(foo);
	push(newSym(DYNAMO, newCell(foo,x)));
	if (!avail)
		gc();
	x = avail;
	avail = x->link;
	if (x == avMark)
		avMark = x->link;
	x->data = tos;
	x->link = dynamos;
	dynamos = x;
	return pop();
}

pico Dynamo(x)
register pico x;
{
	push(EVAL1(x));		/* DisposeFun */
	x = EVAL1(cdr(x));	/* Object */
	return dynamo(pop(),x);
}

pico newCell(x,y)
register pico x,y;
{
	register pico p;

	if (!avail) {
		push(x), push(y);
		gc();
		y = pop(), x = pop();
	}
	p = avail;
	avail = p->link;
	if (p == avMark)
		avMark = p->link;
	setCar(p,x);
	setCdr(p,y);
	return p;
}

pico newCell2(a,b,c,d)
pico a,b,c,d;
{
	push(newCell(a,b));
	a = newCell(c,d);
	return newCell(pop(), a);
}

/* Share common subexpressions */
pico share1(x)
register pico x;
{
	pico p;

	while (isCell(x)) {
		if (cdr(x)==shareY && car(x)==shareX)
			return x;
		if (p = share1(car(x)))
			return p;
		x = cdr(x);
	}
	return NULL;
}

pico doShare(x,y)
pico x,y;
{
	register pico z,p,*ptr;
	register int i;

	shareX = x;
	shareY = y;
	ptr = stkPtr;					/* Search stack */
	while (ptr) {
		if (p = share1(*(ptr+1)))
			return p;
		ptr = (pico*)*ptr;
	}
	for (i=0; i<=THREADS; ++i) {	/* Search oblist */
		z = env.threads[i];
		while (isCell(z)) {
			if (p = share1(mkCell(car(z))))
				return p;
			z = cdr(z);
		}
	}
	return newCell(x,y);
}

pico share(x,y)
pico x,y;
{
	if (!isNil(val(shareFlg)))
		return doShare(x,y);
	return newCell(x,y);
}

pico share1List(x)
register pico x;
{
	if (!isCell(x))
		return x;
	return doShare(share1List(car(x)),share1List(cdr(x)));
}

pico shareList(x)
register pico x;
{
	if (!isNil(val(shareFlg)))
		return share1List(x);
	return x;
}

#if ASM68K

void mark()
{
	asm {
		move.l 4(a7),d0		;Object
	@1:
		btst #1,d0			;Number?
		bne.s @2			;Yes
		andi.l #-8,d0		;Force to cell
		move.l d0,a0
		bclr #0,3(a0)		;Mark it
		beq.s @2			;Branch if it was already marked
		move.l 4(a0),-(a7)	;Save cdr
		move.l (a0),d0		;Recurse on car
		bsr.s @1
		move.l (a7)+,d0		;Get cdr
		bra.s @1			;Tail-recurse
	@2:
	}
}

#else /* ! ASM68K */

void mark(x)
register pico x;
{
	while (!isNum(x) && isFree(x = cellPtr(x))) {
		*markPtr(x) &= ~1;
		mark(x->data);
		x = x->link;
	}
}

#endif


void allFree()
{
	register char *p;
	register long n;

	p = markPtr(heap);
	n = heapEnd - heap;
	do {
		*p |= 1;
		p += 8;
	} while (--n);
}

pico gc()
{
	register pico *ptr,p,q;
	register number i;
	file *sSave;

	sSave = stream;
	stream = NULL;
	prString("Garbage collector: ");
	allFree();
	i = THREADS;					/* Mark oblist */
	do
		mark(env.threads[i]);
	while (--i >= 0);
	mark(env.run);					/* Mark initial runtime expression */
	ptr = stkPtr;					/* Mark stack */
	while (ptr) {
		mark(*(ptr+1));
		ptr = (pico*)*ptr;
	}
	ptr = &dynamos;						/* Dispose of dynamic memory objects */
	while (p = *ptr) {
		if (isFree(cellPtr(q = car(p)))) {
			q = val(q);
			apply1(car(q),cdr(q));
			*ptr = p->link;
		}
		else {
			*markPtr(p) &= ~1;
			ptr = &p->link;
		}
	}
	ptr = NULL;						/* Collect free cells */
	i = 0;
	p = heapEnd - 1;
	do {
		if (isFree(p)) {
			p->link = (pico)ptr;
			ptr = (pico*)p;
			++i;
		}
	} while (--p > dolSym);
	*markPtr(mkCell(nilSym)+1) &= ~1;
	avail = (pico)ptr;
	if (!i)
		err("Out of cell space");
	prNumber(i);
	prString(" cells free\r");
	stream = sSave;
	resetGC();
}

pico Gc(x)
register pico x;
{
	register number n;
	static pico *dyn;

	if (!isCell(x)) {
		gc();
		return tSym;
	}
	x = EVAL1(x);
	NEEDNUM(x);
	n = unBox(x);
	while (--n >= 0 || gcPtr) {
		if (!gcStat) {
			if (collect != mkCell(nilSym)+1)
				*markPtr(collect) |= 1;
			if (++collect == heapEnd) {
				avMark = avail;
				curr = env.threads[0];
				dyn = &dynamos;
				gcStat = 1;
			}
		}
		else if (isNum(curr) || (curr=cellPtr(curr), !isFree(curr))) {
			if (avStk) {
				x = avStk;
				avStk = x->link;
				x->link = avail;
				avail = x;
				curr = x->data;
			}
			else if (gcStat <= THREADS)
				curr = env.threads[gcStat++];
			else if (gcStat == THREADS+1) {
				curr = env.run;
				++gcStat;
			}
			else if (gcStat == THREADS+2) {
				gcPtr = stkPtr;
				++gcStat;
			}
			else if (gcPtr) {
				curr = *(gcPtr+1);
				gcPtr = (pico*)*gcPtr;
			}
			else if (x = *dyn) {
				if (isFree(cellPtr(car(x)))) {
					*markPtr(x) |= 1;
					*dyn = x->link;
					x = val(car(x));
					apply1(car(x),cdr(x));
				}
				else {
					*markPtr(x) &= ~1;
					dyn = &x->link;
				}
			}
			else if (avMark) {
				*markPtr(avMark) &= ~1;
				avMark = avMark->link;
			}
			else if (--collect > dolSym) {
				if (isFree(collect)) {
					collect->link = avail;
					avail = collect;
				}
			}
			else {
				resetGC();
				return tSym;
			}
		}
		else {
			*markPtr(curr) &= ~1;
			if (!isNum(car(curr)) && isFree(cellPtr(car(curr)))) {
				if (!(x = avail)) {
					gc();
					return tSym;
				}
				avail = x->link;
				if (x == avMark)
					avMark = x->link;
				x->link = avStk;
				avStk = x;
				x->data = car(curr);
			}
			curr = cdr(curr);
		}
	}
	return nilSym;
}

/* Collect and compact cell Space */
void adjust(p)
register pico *p;
{
	register pico x = clr(*p);

	if (!isNum(x)  &&  x >= adjLimit)
		*p = isSym(x)? mkSym(tail(x)) : car(x);
}

void compact()
{
	register pico p, q;
	register long n;

	allFree();
	n = THREADS;
	do
		mark(env.threads[n]);
	while (--n >= 0);
	mark(env.run);
	p = mkCell(dolSym+1);
	q = heapEnd;
	do {
		while (!isFree(p))
			if (++p == q)
				goto adjPtr;
		do {
			if (--q == p)
				goto adjPtr;
		} while (isFree(q));
		*p = *q;
		q->data = p;
	} while (++p != q);
adjPtr:								/* Adjust pointers */
	adjLimit = p;
	n = THREADS;					/* Adjust threads and heap */
	do
		adjust(env.threads+n);
	while (--n >= 0);
	adjust(&env.run);
	do {
		--q;
		adjust(&q->link);
		adjust(&q->data);
	} while (q > heap);
	*markPtr(mkCell(nilSym)+1) &= ~1;
	avail = NULL;					/* Build new avail list */
	q = heapEnd;
	while (--q >= p) {
		q->link = avail;
		avail = q;
	}
	resetGC();
}

void deLocate(adr1,adr2)
register pico *adr1, *adr2;
{
	do {
		if (isNum(*adr1)) {
			if (num(*adr1) & 1)
				*adr1 = boxFun(unBox(*adr1) - num(CurrentA5));
		}
		else
			*adr1 = (pico)((num(*adr1) & ~1L) - num(heap));
	} while (++adr1 < adr2);
}

void reLocate(adr1,adr2)
register pico *adr1, *adr2;
{
	do {
		if (isNum(*adr1)) {
			if (num(*adr1) & 1)
				*adr1 = boxFun(unBox(*adr1) + num(CurrentA5));
		}
		else
			*adr1 = (pico)((num(*adr1) & ~1L) + num(heap));
	} while (++adr1 < adr2);
}

void freezErr()
{
	err("FREEZE error");
}

/* Generate freeze file */
pico Freeze(x)
register pico x;
{
	char fName[FILENAME];
	integer refNum;
	long count;
	long buffer[2];
	OSErr e;

	CtoPstr(bufString(EVAL1(x), fName, FILENAME));
	closeAll();
	setVal(loadSym, nilSym);
	picoErase((StringPtr)fName);
	if (Create(fName, 0, CREATOR, 'FREZ') || FSOpen(fName, 0, &refNum))
		err("Can't create FREEZE");
	env.run = cdr(x);
	reset();
	unwind();
	compact();
	/* Write header */
	buffer[0] = VERSION;
	buffer[1] = (char*)adjLimit - (char*)heap;
	count = 8;
	if (FSWrite(refNum, &count, (char*)buffer))
		freezErr();
	/* Write environment */
	deLocate(env.threads, &env.run+1);
	count = sizeof(env);
	e = FSWrite(refNum, &count, (char*)&env);
	reLocate(env.threads, &env.run+1);
	if (e)
		freezErr();
	/* Write heap */
	deLocate((pico*)heap, (pico*)adjLimit);
	e = FSWrite(refNum, &buffer[1], (char*)heap);
	reLocate((pico*)heap, (pico*)adjLimit);
	if (e)
		freezErr();
	FSClose(refNum), FlushVol((StringPtr)NULL,0);
	longjmp(errRst,-1);
}

bool unFreeze(fName,vRefNum)
StringPtr fName;
integer vRefNum;
{
	register pico p,q;
	long count;
	long buffer[2];
	integer refNum, dummy;

	if (FSOpen(fName, vRefNum, &refNum))
		return NO;
	/* Read header */
	count = 8;
	if (FSRead(refNum, &count, (char*)buffer) || buffer[0] != VERSION) {
		FSClose(refNum);
		return NO;
	}
	p = heap + buffer[1]/sizeof(cell);
	/* Read environment */
	count = sizeof(env);
	if (FSRead(refNum, &count, (char*)&env)) {
		FSClose(refNum);
		return NO;
	}
	reLocate(env.threads, &env.run+1);
	/* Read heap */
	if (FSRead(refNum, &buffer[1], (char*)heap)) {
		FSClose(refNum);
		return NO;
	}
	reLocate((pico*)heap, (pico*)p);
	FSClose(refNum);
	avail = NULL;					/* Build avail list */
	q = heapEnd;
	while (--q >= p) {
		q->link = avail;
		avail = q;
	}
	return YES;
}

/* Cut long symbol names to 6 chars */
void doZap(x)
register pico x;
{
	register pico y;

	if (!isNum(x)) {
		if (isCell(x)) {
			if (isFree(x)) {
				*markPtr(x) &= ~1;
				doZap(car(x));
				doZap(cdr(x));
			}
		}
		else {
			x = mkCell(x);
			if (isFree(x)) {
				if (!isNum(y = car(x))) {
					if (isNum(car(y))) {
						++zaps;
						setCar(x, (pico)(num(car(y)) | 0x7C));
					}
					else {
						while (!isNum(cdr(y)) && !isNum(car(cdr(y))))
							y = cdr(y);
						if (!isNum(cdr(y))) {
							++zaps;
							setCdr(y, (pico)(num(car(cdr(y))) | 0x7C));
						}
					}
				}
				*markPtr(x) &= ~1;
				doZap(car(x));
				doZap(cdr(x));
			}
		}
	}
}

pico Zap()
{
	register long n;

	allFree();
	zaps = 0;
	n = THREADS;					/* Mark oblist */
	do
		doZap(env.threads[n]);
	while (--n >= 0);
	*markPtr(mkCell(nilSym)+1) &= ~1;
	resetGC();
	return boxNum(zaps);
}

#if 0
/* Share common subexpressions */
void rplacs(x,l)
register pico x,l;
{
	while (isCell(l)) {
		if (equal(x,car(l)))
			setCar(l,x);
		else
			rplacs(x,car(l));
		if (equal(x,cdr(l))) {
			setCdr(l,x);
			return;
		}
		l = cdr(l);
	}
}

void cs1expr(x)
register pico x;
{
	register pico l;

	while (isCell(x)) {
		l = syms;
		while (isCell(l)) {
			rplacs(x,mkCell(car(l)));
			l = cdr(l);
		}
		cs1expr(car(x));
		x = cdr(x);
	}
}

pico Comsex(x)
pico x;
{
	register number n;
	file *sSave;

	syms = EVAL1(x);
	sSave = stream;
	stream = NULL;
	n = length(syms);
	while (isCell(syms)) {
		prNumber(n), crlf();
		keyBreak();
		cs1expr(mkCell(car(syms)));
		syms = cdr(syms);
	}
	stream = sSave;
	return tSym;
}
#endif

pico Avail()
{
	register pico p;
	register number n;

	p = avail;
	n = 0;
	while (p) {
		if (p < heap || p >= heapEnd)
			err("Bad avail list");
		if (++n > MAXLIST)
			circError();
		p = p->link;
	}
	return boxNum(n);
}
