/*
	(C) Copyright 1986. THINK Technologies, Inc.  All rights reserved.


	The qsort and qksort functions provided with the LightspeedC compiler
	are contained in this module.  

*/

#include "MemoryMgr.h"

/* Prototypes */
static qksort1(int,int);
static qksort(int, int (*)(), int (*)());
static int qxcompare(int,int);
static int qxswap(int,int);
qsort(char*, int, int, int (*)());


/*	Based on the algorithm given in Knuth volume 3, p. 116 


 	qsort(base, nelms, size, compare)
 
 		where:	base		pointer to start of data
 				nelms		number of elements (int)
 				size		size of each array element in bytes
 				compare		pointer to procedure described below
 
  	compare(p1, p2)		p1 & p2 are pointer to two elements in the array
 							compare returns <0 if *p1 < *p2
 											>0 if *p1 > *p2
 											0  if *p1 == *p2

***************************************************************************
NOTE:	qsort assumes that elements are aligned on word boundaries for
		sizes of 2 & 4 to provide an efficient implementation.
***************************************************************************
		
 
Alternate calling sequence:
 
 	qksort(nelms, compare, swap)
 
 	where:	nelms			the number of elements to be sorted
 			compare 		pointer to procedure described below
 			swap			pointer to procedure described below
 			
 	
 	compare(i, j)		i & j are short int args which are indices into
 						the array that compareproc knows about.
 						compare returns <0 if item[i] < item[j]
 											>0 if item[i] > item[j]
 											0  if item[i] == item[j]

 	swap(i, j)			i & j are short int args which are indices into
 						the array that swapproc knows about.
 						swapproc swaps array items i & j.

*/

 
static int (*qkCompare)(), (*qkSwap)();

/*  sort elements "first" through "last"-1 */
static qksort1(first, last)
{
	static short i;		/*  "static" to save stack space  */
	register short j;
 
	while (last - first > 1) {
		i = first;
		j = last;
		for (;;) {
			while (++i < last && (*qkCompare)(i, first) < 0)
				;
			while (--j > first && (*qkCompare)(j, first) > 0)
				;
			if (i >= j)
 				break;
			(*qkSwap)(i, j);
		}
		(*qkSwap)(first, j);
		if (j - first < last - (j + 1)) {
			qksort1(first, j);
			first = j + 1;	/*  qsort1(j + 1, last);  */
		}
		else {
			qksort1(j + 1, last);
			last = j;		/*  qsort1(first, j);  */
		}
	}
}

/*  sort "nelems" elements, using user's "compare" and "swap" routines */
qksort(nelems, compare, swap)
int (*compare)(), (*swap)();
{
	qkCompare = compare;
	qkSwap = swap;
	qksort1(0, nelems);
}


/* qsort call support variables */

static int (*qCompare)();
static char *qbase, *qtemp;
static unsigned long qsize;

qxcompare(i, j)
int i, j;
{
	(*qCompare) (qbase + i * qsize, qbase + j * qsize);
}

qxswap(i, j)
register int i, j;
{
register char c;	/* save stack space and cut execution overhead */
register int k;
register long l;
register char *p1, *p2;

	switch (qsize)
	{
		case 1:	c = qbase[i];
				qbase[i] = qbase[j];
				qbase[j] = c;
				break;

		case 2:	k = ((int*)(qbase))[i];
				((int*)(qbase))[i] = ((int*)(qbase))[j];
				((int*)(qbase))[j] = k;
				break;

		case 4:	l = ((long*)(qbase))[i];
				((long*)(qbase))[i] = ((long*)(qbase))[j];
				((long*)(qbase))[j] = l;
				break;


		default:	p1 = qbase + i * qsize;
					p2 = qbase + j * qsize;
					BlockMove(p1, qtemp, qsize);
					BlockMove(p2, p1, qsize);
					BlockMove(qtemp, p2, qsize);
					break;
	}
}


qsort(base, nelms, size, compare)
char *base;
int nelms;
register int size;
int (*compare)();
{
		if (size <= 0) return (-1);

		/* allocate a buffer unless swapping char, int, or long */

		qtemp = 0;

		if (size==3 || size>4)
		{
			qtemp=NewPtr(size);
			if (MemErr) return (-1);
		}
		
		qCompare = compare;
		qbase = base;
		qsize = size;
		
		qksort(nelms, &qxcompare, &qxswap);
		
		if (qtemp) DisposPtr(qtemp);
		
		return (0);
}
