/* rtPrim.c
 * 31dec92abu
 */

#pragma segment rtPrim

#include "retch.h"

#define RGBRange 255

/* Prototypes */
static pico Gamma(pico);
static pico SameShape(pico);
static pico ViewBlot(pico);
static pico HintBlot(pico);
static pico InterBlot(pico);
static pico BlotBt(pico);
static pico BtBlot(pico);
static pico colEdge(pico);
static pico PixPtr(pico);
static pico PxCopy(pico);
static pico PxFill(pico);
static pico PxSave(pico);
static pico PxUndo(pico);
static pico ColSet(pico);
static pico ChgCol(pico);
static uchar pxLo(uchar*);
static uchar pxHi(uchar*);
static pico PixHi(pico);
static pico PixLo(pico);
static pico WixHi(pico);
static pico WixLo(pico);
static pico CmykRange(pico);
static pico pRGB(pico);
static pico CmykBits(pico);

/* Globals */
Handle rgbTab = NULL;
static uchar colorTables[4][256];
static number dotMin, dotMax;
static unsigned long width2;

static number gammaTab[] = {
      0,  495,  975, 1437, 1884, 2316, 2732, 3133, 3520, 3893,
   4252, 4597, 4930, 5250, 5557, 5853, 6136, 6407, 6666, 6915,
   7152, 7379, 7595, 7801, 7996, 8181, 8356, 8522, 8678, 8825,
   8962, 9090, 9209, 9319, 9421, 9514, 9599, 9675, 9743, 9803,
   9854, 9898, 9933, 9961, 9981, 9994,10000, 9997, 9986, 9969,
   9944, 9913, 9874, 9827, 9774, 9714, 9646, 9572, 9491, 9402,
   9307, 9206, 9097, 8982, 8860, 8730, 8595, 8452, 8304, 8148,
   7986, 7817, 7641, 7458, 7270, 7075, 6873, 6664, 6448, 6226,
   5998, 5762, 5520, 5271, 5016, 4754, 4485, 4209, 3927, 3638,
   3341, 3039, 2729, 2412, 2089, 1758, 1421, 1076,  724,  366,
      0
};

symInit rtPrimSyms[] = {
   {"gamma",      Gamma},
	{"sameShape",	SameShape},
	{"viewBlot",	ViewBlot},
	{"hintBlot",	HintBlot},
	{"interBlot",	InterBlot},
	{"blotBt",		BlotBt},
	{"btBlot",		BtBlot},
	{"colEdge",		colEdge},
	{"pixPtr",		PixPtr},
	{"pxCopy",		PxCopy},
	{"pxFill",		PxFill},
	{"pxSave",		PxSave},
	{"pxUndo",		PxUndo},
	{"colSet",		ColSet},
	{"chgCol",		ChgCol},
	{"pixHi",		PixHi},
	{"pixLo",		PixLo},
	{"wixHi",		WixHi},
	{"wixLo",		WixLo},
	{"cmykRange",	CmykRange},
	{"rgb",			pRGB},
	{"cmykBits",	CmykBits},
	NULL
};

/* Gamma function, inverted-mapped from 1.00 .. 2.00 to 0..100 */
pico Gamma(x)
register pico x;
{
   x = EVAL1(x);
   NEEDNUM(x);
   return boxNum(gammaTab[unBox(x)]);
}

/* Check if two masks have the same shape */
pico SameShape(x)
pico x;
{
	register pico z1,z2,g1,g2;
	register number h1,v1,h2,v2;
	cell c1;

	push(g1 = EVAL1(x), c1); /* First graf */
	g2 = EVAL1(cdr(x)); /* Second graf */
	drop(c1);
	h1 = unBox(car(car(car(g1))));
	v1 = unBox(cdr(car(car(g1))));
	h2 = unBox(car(car(car(g2))));
	v2 = unBox(cdr(car(car(g2))));
	while (isCell(g1) && isCell(g2)) {
		z1 = car(g1),  g1 = cdr(g1);
		z2 = car(g2),  g2 = cdr(g2);
		while (isCell(z1) && isCell(z2)) {
			if (absNumber(unBox(car(car(z1))) - h1)  !=
						absNumber(unBox(car(car(z2))) - h2))
				return nilSym;
			if (absNumber(unBox(cdr(car(z1))) - v1)  !=
						absNumber(unBox(cdr(car(z2))) - v2))
				return nilSym;
			z1 = cdr(z1);
			z2 = cdr(z2);
		}
		if (isCell(z1) || isCell(z2))
			return nilSym;
	}
	return boxBool(!isCell(g1) && !isCell(g2));
}

/* ThreeD  view of Blot */
static number posH,posV,dx,dz,fcl,sinP,cosP;

pico ViewBlot(w)
pico w;
{
	register int i,j;
	uchar *p;
	number dist;
	number x,y,z,py,pz;
	integer h[32][32], v[32][32], val[32][32];

	p = (uchar*)nextNum(&w);
	posH = nextNum(&w);
	posV = nextNum(&w);
	dx = nextNum(&w);
	dz = -nextNum(&w);
	fcl = nextNum(&w);

	dist = pythag(dx,dz);
	sinP = muldiv(dz,10000,dist);
	cosP = muldiv(dx,10000,dist);

	for (i = 0, x = dx+16*16; i < 32; ++i, x -= 16)
		for (j = 0, y = -16*16; j < 32; ++j, y += 16) {
			z = dz + (val[i][j] = *p++);
			py = muldiv(x,sinP,10000) + muldiv(z,-cosP,10000);
			pz = muldiv(x,cosP,10000) + muldiv(z,sinP,10000);
			h[i][j] = posH + muldiv(fcl,y,pz);
			v[i][j] = posV + muldiv(fcl,py,pz);
		}
	for (i = 0; i < 32; ++i) {
		MoveTo(h[i][0], v[i][0]);
		for (j = 1; j < 32; ++j)
			if (val[i][j-1] || val[i][j])
				LineTo(h[i][j], v[i][j]);
			else
				MoveTo(h[i][j], v[i][j]);
	}
	for (j = 0; j < 32; ++j) {
		MoveTo(h[0][j], v[0][j]);
		for (i = 1; i < 32; ++i)
			if (val[i-1][j]  ||  val[i][j])
				LineTo(h[i][j], v[i][j]);
			else
				MoveTo(h[i][j], v[i][j]);
	}
	return tSym;
}

pico HintBlot(w)
pico w;
{
	number x,y,z,py,pz;
	integer h,v;

	y = 16*nextNum(&w) - 16*16; /* h */
	x = dx - 16*nextNum(&w) + 16*16; /* v */
	z = dz + nextNum(&w);
	py = muldiv(x,sinP,10000) + muldiv(z,-cosP,10000);
	pz = muldiv(x,cosP,10000) + muldiv(z,sinP,10000);
	h = posH + muldiv(fcl,y,pz);
	v = posV + muldiv(fcl,py,pz);
	PenMode(patXor), PenSize(4,4);
	MoveTo(h,v), Line(0,0);
	PenNormal();
	return tSym;
}

/* Interpolate Blot data */
pico InterBlot(x)
pico x;
{
	register uchar *b,*p;
	register int h,v,n,max1,max2;
	uchar res[32*32];

	b = (uchar*)nextNum(&x);
	p = res;
	for (v = 0; v < 32; ++v) {
		for (h = 0; h < 32; ++h) {
			n = 0;
			if (h > 0)
				n += b[h-1+v*32];
			if (h < 31)
				n += b[h+1+v*32];
			if (v > 0)
				n += b[h+(v-1)*32];
			if (v < 31)
				n += b[h+(v+1)*32];
			*p++ = n / 4;
		}
	}
	max1 = max2 = 0;
	for (h = 0; h < 32*32; ++h) {
		if (b[h] > max1)
			max1 = b[h];
		if (res[h] > max2)
			max2 = res[h];
	}
	if (max2)
		for (h = 0; h < 32*32; ++h)
			b[h] = muldiv(res[h], max1, max2);
	return tSym;
}

/* Move data between Blot edit map and bitmap */
pico BlotBt(x)
pico x;
{
	register uchar *p;
	register word *q;
	register number l,n,i,j;

	l = nextNum(&x);
	p = (uchar*)nextNum(&x);
	q = (word*)nextNum(&x);
	j = 64;
	do {
		n = 0;
		i = 16;
		do {
			n <<= 1;
			if (*p++ >= l)
				++n;
		} while (--i);
		*q++ = n;
	} while (--j);
	return tSym;
}

pico BtBlot(x)
pico x;
{
	register word *p;
	register uchar *q;
	register number l,n,i,j;

	l = nextNum(&x);
	p = (word*)nextNum(&x);
	q = (uchar*)nextNum(&x);
	j = 64;
	do {
		n = *p++;
		i = 16;
		do {
			if (n & 0x8000) {
				if (*q < l)
					*q = l;
			}
			else {
				if (*q >= l)
					*q = 0;
			}
			++q;
			n <<= 1;
		} while (--i);
	} while (--j);
	return tSym;
}

/* Return measure for the color gradient */
pico colEdge(x)
pico x;
{
	register uchar *p;
	integer h1,v1,h2,v2;
	register number w,sh,sv,e,i,dh,dv,temp;
	bool xchg;

	p = *(uchar**)nextNum(&x); /* base */
	h1 = (integer)nextNum(&x);
	v1 = (integer)nextNum(&x);
	h2 = (integer)nextNum(&x);
	v2 = (integer)nextNum(&x);
	p += (h1 + v1 * WINSIZE) * sizeof(pixel);
	dh = h2 - h1;
	dv = v2 - v1;
	sh = 0;
	if (dh > 0)
		sh = sizeof(pixel);
	else if (dh < 0) {
		dh = -dh;
		sh = -sizeof(pixel);
	}
	sv = 0;
	if (dv > 0)
		sv = WINSIZE*sizeof(pixel);
	else if (dv < 0) {
		dv = -dv;
		sv = -WINSIZE*sizeof(pixel);
	}
	xchg = NO;
	if (dv > dh) {
		xchg = YES;
		temp = dh,  dh = dv,  dv = temp;
	}
	e = 2*dv - dh;
	w = 0;
	if (xchg)
		for (i = 0; i < dh; ++i) {
			w +=
				absNumber(p[-12] + 2*p[-8] + 3*p[-4] - 3*p[0] - 2*p[4] - p[8]) +
				absNumber(p[-11] + 2*p[-7] + 3*p[-3] - 3*p[1] - 2*p[5] - p[9]) +
				absNumber(p[-10] + 2*p[-6] + 3*p[-2] - 3*p[2] - 2*p[6] - p[10]) +
				absNumber(p[-9] + 2*p[-5] + 3*p[-1] - 3*p[3] - 2*p[7] - p[11]);
			if (e >= 0) {
				p += sh;
				e -= 2*dh;
			}
			p += sv;
			e += 2*dv;
		}
	else
		for (i = 0; i < dh; ++i) {
			w +=
				absNumber(p[-12*WINSIZE] + 2*p[-8*WINSIZE] + 3*p[-4*WINSIZE] -
						3*p[0] - 2*p[4*WINSIZE] - p[8*WINSIZE]) +
				absNumber(p[-11*WINSIZE] + 2*p[-7*WINSIZE] + 3*p[-3*WINSIZE] -
						3*p[WINSIZE] - 2*p[5*WINSIZE] - p[9*WINSIZE]) +
				absNumber(p[-10*WINSIZE] + 2*p[-6*WINSIZE] + 3*p[-2*WINSIZE] -
						3*p[2*WINSIZE] - 2*p[6*WINSIZE] - p[10*WINSIZE]) +
				absNumber(p[-9*WINSIZE] + 2*p[-5*WINSIZE] + 3*p[-1*WINSIZE] -
						3*p[3*WINSIZE] - 2*p[7*WINSIZE] - p[11*WINSIZE]);
			if (e >= 0) {
				p += sv;
				e -= 2*dh;
			}
			p += sh;
			e += 2*dv;
		}
	return  dh?  boxNum(w/dh) : boxNum(0);
}

/* Return pointer to pixel */
pico PixPtr(x)
pico x;
{
	register pixel *p;
	long h,v;

	p = *(pixel**)nextNum(&x);
	if ((h = nextNum(&x)) < 0 || h >= WINSIZE)
		return nilSym;
	if ((v = nextNum(&x)) < 0 || v >= WINSIZE)
		return nilSym;
	return boxNum(p + v * WINSIZE + h);
}

/* Copy one blot */
pico PxCopy(x)
pico x;
{
	register pico l;
	register uchar *p, *q;
	register number i,j,lines,w,size,ww,www;
	pico blot;
	uchar *srcBase,*dstBase;
	number h1,v1,h2,v2,srcH,srcV,dstH,dstV;
	integer cnt[2*MAXBLOT];
	uchar weight[2*MAXBLOT][2*MAXBLOT];
	uchar *ptr[2*MAXBLOT];
	number h[2*MAXBLOT];
	number v[2*MAXBLOT];
	number hinc, vinc;

	srcBase = *(uchar**)nextNum(&x);
	srcH = nextNum(&x);
	srcV = nextNum(&x);
	dstBase = *(uchar**)nextNum(&x);
	dstH = nextNum(&x);
	dstV = nextNum(&x);
	ww = nextNum(&x) - dotMin;
	www = dotMax - dotMin;
	blot = EVAL1(x);
	
	size = i = 0;
	while (isCell(blot)) {
		l = car(blot),  blot = cdr(blot);
		h[i] = unBox(car(l)),  l = cdr(l);
		v[i] = unBox(car(l)),  l = cdr(l);
		for (cnt[i] = 0, p = weight[i]; isCell(l); l = cdr(l)) {
			++cnt[i];
			*p++ = (uchar)unBox(car(l));
			++size;
		}
		ptr[i] = p-1;
		++i;
	}
	lines = i;
	if (dstH > srcH)
		hinc = -1;
	else {
		hinc = 1;
		for (j = 0; j < lines; ++j)
			ptr[j] = weight[j];
	}
	if (dstV > srcV) {
		vinc = -1;
		i = lines-1;
	}
	else {
		vinc = 1;
		i = 0;
	}
	while (--lines >= 0) {
		h1 = srcH + h[i];
		v1 = srcV + v[i];
		h2 = dstH + h[i];
		v2 = dstV + v[i];
		if (inCmyk(v1) && inCmyk(v2)) {
			j = cnt[i];
			p = srcBase + (h1 + v1*WINSIZE) * sizeof(pixel);
			q = dstBase + (h2 + v2*WINSIZE) * sizeof(pixel);
			if (hinc < 0) {
				p += (j-1) * sizeof(pixel);
				q += (j-1) * sizeof(pixel);
				h1 += j-1;
				h2 += j-1;
			}
			while (--j >= 0) {
				if (inCmyk(h1) && inCmyk(h2)) {
					w = num(*ptr[i]) * ww / www;
					*q = blend(*p,*q,w);
					*(q+1) = blend(*(p+1), *(q+1), w);
					*(q+2) = blend(*(p+2), *(q+2), w);
					*(q+3) = blend(*(p+3), *(q+3), w);
				}
				p += hinc * sizeof(pixel);
				q += hinc * sizeof(pixel);
				h1 += hinc; /* For region check */
				h2 += hinc;
				ptr[i] += hinc;
			}
		}
		i += vinc;
	}
	return tSym;
}

/* Fill one blot */
pico PxFill(x)
pico x;
{
	register pico l;
	register uchar *p;
	register number i,j,lines,w,size,ww,www;
	pico blot;
	uchar *dstBase;
	number c,m,y,k,h2,v2,dstH,dstV;
	integer cnt[2*MAXBLOT];
	uchar weight[2*MAXBLOT][2*MAXBLOT];
	uchar *ptr[2*MAXBLOT];
	number h[2*MAXBLOT];
	number v[2*MAXBLOT];

	c = nextNum(&x);
	m = nextNum(&x);
	y = nextNum(&x);
	k = nextNum(&x);
	dstBase = *(uchar**)nextNum(&x);
	dstH = nextNum(&x);
	dstV = nextNum(&x);
	ww = nextNum(&x) - dotMin;
	www = dotMax - dotMin;
	blot = EVAL1(x);
	
	size = i = 0;
	while (isCell(blot)) {
		l = car(blot),  blot = cdr(blot);
		h[i] = unBox(car(l)),  l = cdr(l);
		v[i] = unBox(car(l)),  l = cdr(l);
		for (cnt[i] = 0, p = weight[i]; isCell(l); l = cdr(l)) {
			++cnt[i];
			*p++ = (uchar)unBox(car(l));
			++size;
		}
		ptr[i] = p-1;
		++i;
	}
	lines = i;
	i = lines-1;
	while (--lines >= 0) {
		h2 = dstH + h[i];
		v2 = dstV + v[i];
		if (inCmyk(v2)) {
			j = cnt[i];
			p = dstBase +
					(h2 + v2*WINSIZE) * sizeof(pixel) +
										(j-1) * sizeof(pixel);
			h2 += j-1;
			while (--j >= 0) {
				if (inCmyk(h2)) {
					w = num(*ptr[i]) * ww / www;
					*p = blend(k,*p,w);
					*(p+1) = blend(c, *(p+1), w);
					*(p+2) = blend(m, *(p+2), w);
					*(p+3) = blend(y, *(p+3), w);
				}
				p -= sizeof(pixel);
				--h2; /* For region check */
				--ptr[i];
			}
		}
		--i;
	}
	return tSym;
}

/* Backup one blot */
pico PxSave(x)
pico x;
{
	register pico l;
	register pixel *p, *q;
	register number i,j,lines,size;
	pico blot;
	pixel *base;
	number dstH,dstV;
	integer cnt[2*MAXBLOT];
	number h[2*MAXBLOT];
	number v[2*MAXBLOT];
	Handle hndl;

	base = *(pixel**)nextNum(&x);
	dstH = nextNum(&x);
	dstV = nextNum(&x);
	blot = EVAL1(x);
	
	size = i = 0;
	while (isCell(blot)) {
		l = car(blot),  blot = cdr(blot);
		h[i] = unBox(car(l)),  l = cdr(l);
		v[i] = unBox(car(l)),  l = cdr(l);
		for (cnt[i] = 0; isCell(l); l = cdr(l)) {
			++cnt[i];
			++size;
		}
		++i;
	}
	lines = i;
	if (!(hndl = NewHandle(size*sizeof(pixel))))
		return nilSym;
	q = *(pixel**)hndl;
	for (i = 0; i < lines; ++i) {
		p = base + dstH + h[i] + (dstV + v[i])*WINSIZE;
		j = cnt[i];
		while (--j >= 0)
			*q++ = *p++;
	}
	return dynHandle(hndl);
}

/* Restore blot from backup */
pico PxUndo(x)
pico x;
{
	register pixel *p;
	register pico l, blot;
	register number h1, v1, h2, v2;
	pixel *base;

	base = *(pixel**)nextNum(&x);
	h1 = nextNum(&x);
	v1 = nextNum(&x);
	p = (pixel*)nextDynamo(&x); /* First keep the Handle */
	blot = EVAL1(x);
	p = *(pixel**)p; /* Then deref it */
	while (isCell(blot)) {
		l = car(blot),  blot = cdr(blot);
		h2 = h1 + unBox(car(l)), l = cdr(l);
		v2 = v1 + unBox(car(l)), l = cdr(l);
		if (inCmyk(v2)) {
			while (isCell(l)) {
				if (inCmyk(h2))
					base[h2 + v2*WINSIZE] = *p++;
				++h2;
				l = cdr(l);
			}
		}
	}
	return tSym;
}

/* Set the color tables */
pico ColSet(x)
register pico x;
{
	register pico y;
	register int i,j;

	y = EVAL1(x);
	if (!isCell(y))
		return nilSym;
	for (i = 0; i < 4; ++i) {
		x = car(y);
		for (j = 0; j < 256; ++j) {
			NEEDNUM(car(x));
			colorTables[i][j] = (uchar)unBox(car(x));
			x = cdr(x);
		}
		y = cdr(y);
	}
	return boxNum(colorTables);
}

pico ChgCol(x)
pico x;
{
	register uchar *p;
	register number i, n;

	p = (uchar*)nextNum(&x);
	if ((n = nextNum(&x)) <= 0)
		return nilSym;
	for (i = 0; i < n; ++i) {
		*p = colorTables[3][*p]; /* K */
		*(p+1) = colorTables[2][*(p+1)]; /* C */
		*(p+2) = colorTables[1][*(p+2)]; /* M */
		*(p+3) = colorTables[0][*(p+3)]; /* Y */
		p += sizeof(pixel);
	}
	return tSym;
}

uchar pxLo(p)
register uchar *p;
{
	register number n;

	n = ((*p * 10 + *(p-4) + *(p+4) + *(p-4*WINSIZE) + *(p+4*WINSIZE)) * 2
		+ *(p-4*WINSIZE-4) + *(p-4*WINSIZE+4)
		+ *(p+4*WINSIZE-4) + *(p+4*WINSIZE+4) ) / 32;
	return  n>dotMax? dotMax:n;
}

uchar pxHi(p)
register uchar *p;
{
	register number n;

	n = ((*p * (-22) + *(p-4) + *(p+4) + *(p-4*WINSIZE) + *(p+4*WINSIZE)) * 2
			+ *(p-4*WINSIZE-4) + *(p-4*WINSIZE+4)
			+ *(p+4*WINSIZE-4) + *(p+4*WINSIZE+4) ) / -32;
	return n<dotMin? dotMin : n>dotMax? dotMax:n;
}

pico PixHi(x)
pico x;
{
	register uchar *p, *q;
	register number len;

	p = (uchar*)nextNum(&x);
	q = (uchar*)nextNum(&x);
	len = nextNum(&x);
	while (--len >= 0) {
		*q++ = pxHi(p++);
		*q++ = pxHi(p++);
		*q++ = pxHi(p++);
		*q++ = pxHi(p++);
	}
	return boxNum(q);
}

pico PixLo(x)
pico x;
{
	register uchar *p, *q;
	register number len;

	p = (uchar*)nextNum(&x);
	q = (uchar*)nextNum(&x);
	len = nextNum(&x);
	while (--len >= 0) {
		*q++ = pxLo(p++);
		*q++ = pxLo(p++);
		*q++ = pxLo(p++);
		*q++ = pxLo(p++);
	}
	return boxNum(q);
}

pico WixHi(x)
pico x;
{
	register uchar *p, *q;
	register pico y;
	Handle h;
	register number w;
	long size;
	cell c1;

	p = (uchar*)nextNum(&x);
	h = (Handle)nextNum(&x);
	push(EVAL1(x),c1);
	x = cdr(x);
	w = nextNum(&x);
	y = pop(c1);
	size = GetHandleSize(h);
	SetHandleSize(h, 4*length(y) + size);
	q = (uchar*)(*h) + size;
	do {
		if (unBox(car(y)) >= w) {
			*q++ = pxHi(p++);
			*q++ = pxHi(p++);
			*q++ = pxHi(p++);
			*q++ = pxHi(p++);
		}
		else {
			*q++ = *p++;
			*q++ = *p++;
			*q++ = *p++;
			*q++ = *p++;
		}
	} while (isCell(y = cdr(y)));
	return tSym;
}

pico WixLo(x)
pico x;
{
	register uchar *p, *q;
	register pico y;
	Handle h;
	register number w;
	long size;
	cell c1;

	p = (uchar*)nextNum(&x);
	h = (Handle)nextNum(&x);
	push(EVAL1(x),c1);
	x = cdr(x);
	w = nextNum(&x);
	y = pop(c1);
	size = GetHandleSize(h);
	SetHandleSize(h, 4*length(y) + size);
	q = (uchar*)(*h) + size;
	do {
		if (unBox(car(y)) >= w) {
			*q++ = pxLo(p++);
			*q++ = pxLo(p++);
			*q++ = pxLo(p++);
			*q++ = pxLo(p++);
		}
		else {
			*q++ = *p++;
			*q++ = *p++;
			*q++ = *p++;
			*q++ = *p++;
		}
	} while (isCell(y = cdr(y)));
	return tSym;
}

/* RGB Screen Display */
pico CmykRange(x)
pico x;
{
	bool f;
	unsigned long mulVal, addVal;
	uchar *p;
	int k,c,key,col;

	dotMin = nextNum(&x);
	dotMax = nextNum(&x);
	f = nextBool(&x); /* Suppress Key display */
	if ((width2 = dotMax - dotMin) <= 0)
		return nilSym;
	width2 *= width2;
	if (rgbTab || (rgbTab = NewHandle((RGBRange+1) * (RGBRange+1)))) {
		p = *rgbTab;
		for (key = 0; key <= RGBRange; ++key) {
			k  =  f||key<dotMin? dotMin  :  key>dotMax? dotMax : key;
			mulVal = (RGBRange * (dotMax - k) << 16) / width2;
			addVal = mulVal * dotMax;
			for (col = 0; col <= RGBRange; ++col) {
				c  =  col<dotMin? dotMin  :  col>dotMax? dotMax : col;
				*p++ = (uchar)(addVal - mulVal * c >> 16);
			}
		}
	}
	return boxPtr(rgbTab);
}

pico pRGB(x)
pico x;
{
	number c,m,y,k;
	unsigned long mulVal, addVal;

	c = nextNum(&x);
	m = nextNum(&x);
	y = nextNum(&x);
	k = nextNum(&x);
	if (k < dotMin)
		k = dotMin;
	else if (k > dotMax)
		k = dotMax;
	mulVal = (RGBRange * (dotMax - k) << 16) / width2;
	addVal = mulVal * dotMax;
	if (c < dotMin)
		c = dotMin;
	else if (c > dotMax)
		c = dotMax;
	if (m < dotMin)
		m = dotMin;
	else if (m > dotMax)
		m = dotMax;
	if (y < dotMin)
		y = dotMin;
	else if (y > dotMax)
		y = dotMax;
	return boxColor(
		addVal - mulVal * c >> 8,
		addVal - mulVal * m >> 8,
		addVal - mulVal * y >> 8 );
}

pico CmykBits(x)
pico x;
{
	Handle base;
	number cols,zoom;
	RgnHandle mask, rgn;
	CGrafPtr port;
	PixMapHandle pm;
	Point org;
	RgnPtr rPtr;
	register uchar *p;
	uchar *orgPtr;
	unsigned long bytes;
	register integer v, i;
	Byte mode;

	base = (Handle)nextNum(&x);
	cols = nextNum(&x);
	zoom = nextNum(&x);
	mask = (RgnHandle)nextPtr(&x);

	pm  =  (port = (CGrafPtr)qd.thePort)->portPixMap;
	if (port->portVersion >= 0  ||  (*pm)->pixelSize != 32)
		return nilSym;
	bytes = (*pm)->rowBytes & 0x7FFF;   /* Not 0x1FFF */
	org.h = 0;
	org.v = 0;
	LocalToGlobal(&org);
	rgn = NewRgn();
	RectRgn(rgn, &(*pm)->bounds);
	SectRgn(rgn, port->visRgn, rgn);
	SectRgn(rgn, port->clipRgn, rgn);
	if (mask)
		SectRgn(rgn,mask,rgn);
	if (!EmptyRgn(rgn)) {
		HideCursor();
		HLock((Handle)rgn);
		rPtr = (RgnPtr)StripAddress((Ptr)*rgn);
		orgPtr = (*pm)->baseAddr + org.v * bytes + org.h * sizeof(pixel);
		if (rPtr->rgnSize == 10) {
			mode = true32b, SwapMMUMode(&mode);
			v = rPtr->rgnBBox.top;
			p = orgPtr + v * bytes + rPtr->rgnBBox.left * sizeof(pixel);
			while (v < rPtr->rgnBBox.bottom) {
				cmykRGB(*base + v / zoom * cols * sizeof(pixel),
						p, rPtr->rgnBBox.left, rPtr->rgnBBox.right, zoom );
				p += bytes;
				++v;
			}
			SwapMMUMode(&mode);
		}
		else {
			register integer c, temp, *rp, *edge, *edges;
			int cnt, v2, rd;

			rd = rPtr->rgnBBox.right - rPtr->rgnBBox.left + 1;
			if (edges = (integer*)NewPtr(rd*sizeof(integer))) {
				cnt = 0;
				rp = (integer*)((char*)rPtr + sizeof(Region));
				v = *rp++;
				mode = true32b, SwapMMUMode(&mode);
				loop {
					while ((c = *rp++) != 0x7FFF) {
						i = 0;
						edge = edges;
						while (i < cnt  &&  c > *edge)
							++i, ++edge;
						if (i < cnt  &&  c == *edge) {
							 while (++i < cnt) {
								*edge = *(edge+1);
								++edge;
							}
							--cnt;
						}
						else {
							while (i++ < cnt)
								temp = *edge,  *edge++ = c,  c = temp;
							*edge = c;
							++cnt;
						}
					}
					if ((v2 = *rp++) == 0x7FFF)
						break;
					while (v < v2) {
						for (i = 0; i < cnt; i += 2) {
							cmykRGB(*base + v / zoom * cols * sizeof(pixel),
								orgPtr + v * bytes + edges[i] * sizeof(pixel),
								edges[i], edges[i+1], zoom );
						}
						++v;
					}
				}
				SwapMMUMode(&mode);
				DisposPtr((Ptr)edges);
			}
		}
		ShowCursor();
	}
	DisposeRgn(rgn);
	return tSym;
}
