/* view.c
 * 12jul92abu
 */

#pragma segment view

#include "rcSim.h"

#define EMAX			16
#define FRAMECOLOR	0
#define CLIP			2000000000.0

typedef struct {
	short cnt;
	short h[EMAX];
	unsigned long z[EMAX];
} edge;

/* Prototypes */
pixel unBoxPixel(pico);
pixel nextPixel(pico*);
static void clipLine(double,double,double,double);
static void viewLine(double,double,double,double,double,double);
static void mkEdge(edge*,double,double,double,double,double,double);
static void viewEdge(edge*,double,double,double,double,double,double);
static void zLine(pixel,long,long,unsigned long,long,unsigned long);
static pico Camera(pico);
static pico w_init(pico);
static pico w_done(void);
static pico z_init(pico);
static pico Wire(pico);
static pico Paint(pico);

symInit viewSyms[] = {
	{"camera",		Camera},
	{"w-init"		w_init},
	{"w-done"		w_done},
	{"z-init"		z_init},
	{"wire",			Wire},
	{"paint",		Paint},
	NULL
};

/* Camera parameters */
static vector camLoc;
static vector spot;
static double focLen;

static long sizeH, sizeV, orgH, orgV;
double left,top,right,bottom;
BitMap saveBits;
static pixel *fBuffer;
static unsigned long *zBuffer;

/* Transformation coefficients */
static double sinP,cosP, sinY,cosY;
static double coeff1,coeff2,coeff4,coeff5,coeff6,coeff7,coeff8,coeff9;


pixel unBoxPixel(x)
pico x;
{
	register unsigned long n;

	NEEDNUM(x);
	n = (unsigned long)x;
	return (pixel)
		((n & 0xFF000000) >> 8  |  (n & 0x3FC000) >> 6  |  (n & 0xFF0) >> 4);
}

pixel nextPixel(p)
register pico *p;
{
	pixel c;

	c = unBoxPixel(EVAL1(*p));
	*p = cdr(*p);
	return c;
}

/* Set Camera parameters */
pico Camera(x)
pico x;
{
	double dx,dy,dz, dist2,dist3;

	focLen = (double)nextNum(&x);				/* Focal length */
	nextVector(&x, &camLoc, CENTIMETER);	/* Camera location */
	nextVector(&x, &spot, CENTIMETER);		/* View spot */

	/* Window size should be a multiple of 8 */
	sizeH = (qd.thePort->portRect.left + qd.thePort->portRect.right) & ~7;
	sizeV = (qd.thePort->portRect.top + qd.thePort->portRect.bottom) & ~7;
	orgH = sizeH / 2;
	orgV = sizeV / 2;

	dx = spot.x - camLoc.x;
	dy = spot.y - camLoc.y;
	dz = spot.z - camLoc.z;
	dist2 = sqrt(dx*dx + dy*dy);
	dist3 = sqrt(dx*dx + dy*dy + dz*dz);
	if (dist2 == 0.0) {
		sinY = 0.0;
		cosY = 1.0;
	}
	else {
		sinY = dy / dist2;
		cosY = dx / dist2;
	}
	sinP = dz / dist3;
	cosP = dist2 / dist3;

	coeff1 = -sinY;
	coeff2 = cosY;
	coeff4 = cosY*sinP;
	coeff5 = sinY*sinP;
	coeff6 = -cosP;
	coeff7 = cosY*cosP;
	coeff8 = sinY*cosP;
	coeff9 = sinP;
	return tSym;
}

pico w_init(x)
pico x;
{
	register integer n;

	saveBits = qd.thePort->portBits;
	SetPortBits((BitMap*)nextNum(&x));
	EraseRect(&qd.thePort->portRect);
	n = orgV + (integer)(sinP*focLen / cosP);
	MoveTo(qd.thePort->portRect.left, n);
	LineTo(qd.thePort->portRect.right, n);
	left   = (double)(qd.thePort->portRect.left - orgH);
	top    = (double)(qd.thePort->portRect.top - orgV);
	right  = (double)(qd.thePort->portRect.right - orgH);
	bottom = (double)(qd.thePort->portRect.bottom - orgV);
	return tSym;
}

pico w_done()
{
	SetPortBits(&saveBits);
	return tSym;
}

pico z_init(x)
pico x;
{
	register pixel *p, c1, c2;
	register number i,n;

	fBuffer = (pixel*)nextNum(&x);			/* Buffers */
	zBuffer = (unsigned long*)nextNum(&x);
	c1 = nextPixel(&x);							/* Ground color */
	c2 = nextPixel(&x);							/* Sky color */
	/* Init frame buffer */
	n = maxNumber(0, minNumber(sizeV, orgV + sinP*focLen / cosP));
	p = fBuffer;
	i = sizeH * n;
	while ((i-=8) >= 0) {
		p[0] = c2;
		p[1] = c2;
		p[2] = c2;
		p[3] = c2;
		p[4] = c2;
		p[5] = c2;
		p[6] = c2;
		p[7] = c2;
		p += 8;
	}
	i = sizeH * (sizeV-n);
	while ((i-=8) >= 0) {
		p[0] = c1;
		p[1] = c1;
		p[2] = c1;
		p[3] = c1;
		p[4] = c1;
		p[5] = c1;
		p[6] = c1;
		p[7] = c1;
		p += 8;
	}
	/* Init z-buffer */
	p = zBuffer;
	i = sizeH * sizeV;
	while ((i-=8) >= 0) {
		p[0] = 0xFFFFFFFF;
		p[1] = 0xFFFFFFFF;
		p[2] = 0xFFFFFFFF;
		p[3] = 0xFFFFFFFF;
		p[4] = 0xFFFFFFFF;
		p[5] = 0xFFFFFFFF;
		p[6] = 0xFFFFFFFF;
		p[7] = 0xFFFFFFFF;
		p += 8;
	}
	return tSym;
}

void clipLine(h1,v1,h2,v2)
double h1,v1,h2,v2;
{
	register double dh,dv,tmp;
	register unsigned outCode1, outCode2;

	outCode1 = outCode2 = 0;
	if (h1 < left)
		outCode1 |= 8;
	else if (h1 >= right)
		outCode1 |= 4;
	if (v1 < top)
		outCode1 |= 2;
	else if (v1 >= bottom)
		outCode1 |= 1;
	if (h2 < left)
		outCode2 |= 8;
	else if (h2 >= right)
		outCode2 |= 4;
	if (v2 < top)
		outCode2 |= 2;
	else if (v2 >= bottom)
		outCode2 |= 1;
	if (outCode1 & outCode2)
		return; /* Trivially reject */
	if (!(outCode1 | outCode2)) {
		MoveTo((int)h1 + orgH, (int)v1 + orgV);
		LineTo((int)h2 + orgH, (int)v2 + orgV);
		return; /* Trivially accept */
	}
	dh = h2 - h1;
	dv = v2 - v1;
	loop {
		if (!outCode1) {
			tmp = h1,  h1 = h2,  h2 = tmp;
			tmp = v1,  v1 = v2,  v2 = tmp;
			outCode1 = outCode2,  outCode2 = 0;
		}
		if (outCode1 & 1) {
			h1 += (bottom-1 - v1) * dh / dv;
			v1 = bottom - 1;
			outCode1 = 0;
			if (h1 >= right)
				outCode1 |= 4;
			else if (h1 < left)
				outCode1 |= 8;
		}
		else if (outCode1 & 2) {
			h1 += (top - v1) * dh / dv;
			v1 = top;
			outCode1 = 0;
			if (h1 >= right)
				outCode1 |= 4;
			else if (h1 < left)
				outCode1 |= 8;
		}
		else if (outCode1 & 4) {
			v1 += (right-1 - h1) * dv / dh;
			h1 = right - 1;
			outCode1 = 0;
			if (v1 >= bottom)
				outCode1 |= 1;
			else if (v1 < top)
				outCode1 |= 2;
		}
		else if (outCode1 & 8) {
			v1 += (left - h1) * dv / dh;
			h1 = left;
			outCode1 = 0;
			if (v1 >= bottom)
				outCode1 |= 1;
			else if (v1 < top)
				outCode1 |= 2;
		}
		if (!(outCode1 | outCode2)) {
			MoveTo((int)h1 + orgH, (int)v1 + orgV);
			LineTo((int)h2 + orgH, (int)v2 + orgV);
			return;
		}
		if (outCode1 & outCode2)
			return;
	}
}

void viewLine(px1,py1,pz1,px2,py2,pz2)
double px1,py1,pz1,px2,py2,pz2;
{
	double d;

	if (pz1 > 1.0  ||  pz2 > 1.0) {
		if (pz1 <= 1.0) {
			d = (1 - pz1) / (pz2 - pz1);
			clipLine(
					focLen * (px1 + d * (px2-px1)),
					focLen * (py1 + d * (py2-py1)),
					focLen*px2/pz2,
					focLen*py2/pz2 );
		}
		else if (pz2 <= 1.0) {
			d = (1 - pz2) / (pz1 - pz2);
			clipLine(
					focLen*px1/pz1,
					focLen*py1/pz1,
					focLen * (px2 + d * (px1-px2)),
					focLen * (py2 + d * (py1-py2)) );
		}
		else
			clipLine(
					focLen*px1/pz1,
					focLen*py1/pz1,
					focLen*px2/pz2,
					focLen*py2/pz2 );
	}
}

/* Draw WireFrame */
pico Wire(x)
register pico x;
{
	register vector *trans;
	register pico y;
	register int i,j;
	matHandle h;
	vector pos,v,w;
	register double dx,dy,dz;
	cell c1;

	push(EVAL1(x),c1);							/* WireObject lines */
	x = cdr(x);
	unBoxVector(EVAL1(x),&pos,CENTIMETER);	/* Position */
	x = cdr(x);
	h = NULL; 										/* Optional matrix */
	if (!isNil(x = EVAL1(x))) {
		NEEDNUM(x);
		h = (matHandle)unBox(x);
	}

	dx = pos.x - camLoc.x;
	dy = pos.y - camLoc.y;
	dz = pos.z - camLoc.z;

	x = pop(c1);
	y = car(x); /* Shadows */
	x = cdr(x);
	while (isCell(y)) {
		unBoxVector(car(car(y)), &v, MILLIMETER);
		unBoxVector(cdr(car(y)), &w, MILLIMETER);
		y = cdr(y);
		if (h) {
			transform(v.x, v.y, v.z, *h, &v);
			transform(w.x, w.y, w.z, *h, &w);
		}
		v.x += dx + v.z + pos.z;
		v.y += dy;
		w.x += dx + w.z + pos.z;
		w.y += dy;
		viewLine(
			coeff1*v.x + coeff2*v.y,
			coeff4*v.x + coeff5*v.y - coeff6*camLoc.z,
			coeff7*v.x + coeff8*v.y - coeff9*camLoc.z,
			coeff1*w.x + coeff2*w.y,
			coeff4*w.x + coeff5*w.y - coeff6*camLoc.z,
			coeff7*w.x + coeff8*w.y - coeff9*camLoc.z );
	}
	y = car(x); /* Points */
	x = cdr(x); /* Strokes */
	if (!(trans = (vector*)NewPtr(length(y) * sizeof(vector))))
		return nilSym;

	i = 0;
	while (isCell(y)) {
		unBoxVector(car(y), &v, MILLIMETER);
		y = cdr(y);
		if (h)
			transform(v.x, v.y, v.z, *h, &v);
		v.x += dx;
		v.y += dy;
		v.z += dz;
		trans[i].x = coeff1*v.x + coeff2*v.y;
		trans[i].y = coeff4*v.x + coeff5*v.y + coeff6*v.z;
		trans[i].z = coeff7*v.x + coeff8*v.y + coeff9*v.z;
		++i;
	}
	while (isCell(x)) {
		i = (int)unBox(car(car(x)));
		j = (int)unBox(cdr(car(x)));
		x = cdr(x);
		viewLine(
			trans[i].x, trans[i].y, trans[i].z,
			trans[j].x, trans[j].y, trans[j].z );
	}
	DisposPtr((Ptr)trans);
	return tSym;
}

static void mkEdge(edges,x1,y1,z1,x2,y2,z2)
edge *edges;
double x1,y1,z1,x2,y2,z2;
{
	register double a,n1,n2;
	register edge *p;
	register pixel *fPtr;
	register unsigned long *zPtr;
	short hTmp;
	register short *hp;
	unsigned long zTmp;
	register unsigned long z, *zp;
	register number h,i,j,dh,dv,dz,h1,h2,v1,v2,d1,d2;


	if (y2 < y1) {
		a = x1,  x1 = x2,  x2 = a;
		a = y1,  y1 = y2,  y2 = a;
		a = z1,  z1 = z2,  z2 = a;
	}
	/* Clip vertical */
	if (y1 > (n2 = orgV)  ||  y2 <= (n1 = -n2))
		return;
	if (y1 < n1) {
		a = (n1-y1)/(y2-y1); /* m = (x2-x1)/(y2-y1) */
		x1 += a*(x2-x1); /* x1 = x1 + m * (n1 - y1) */
		z1 += a*(z2-z1);
		y1 = n1;
	}
	if (y2 > n2) {
		a = (y2-n2)/(y2-y1);
		x2 += a*(x1-x2);
		z2 += a*(z1-z2);
		y2 = n2;
	}
	if (x1 < -CLIP)
		x1 = -CLIP;
	else if (x1 > CLIP)
		x1 = CLIP;
	if (x2 < -CLIP)
		x2 = -CLIP;
	else if (x2 > CLIP)
		x2 = CLIP;
	d1 = z1 * 10000.0;
	d2 = z2 * 10000.0;
	/* Digital Differential Analyzer */
	dh = (h2 = num(x2)) - (h1 = num(x1));
	dv = (v2 = num(y2)) - (v1 = num(y1));
	dz = d2 - d1;
	h1 += orgH;
	v1 += orgV;
	if (!dv) {
		h2 += orgH;
		if (h1 < 0)
			h1 = 0;
		else if (h1 > sizeH)
			h1 = sizeH;
		if (h2 < 0)
			h2 = 0;
		else if (h2 > sizeH)
			h2 = sizeH;
		if (h1 < h2)
			zLine(FRAMECOLOR,v1,h1,d1,h2,d2);
		else
			zLine(FRAMECOLOR,v1,h2,d2,h1,d1);
	}
	else {
		p = edges + v1;
		fPtr = fBuffer + v1*sizeH;
		zPtr = zBuffer + v1*sizeH;
		for (i = 0; i < dv; ++i) {
			if ((h = h1 + i*dh/dv) < 0)
				h = 0;
			else if (h > sizeH)
				h = sizeH;
			z = d1 + i*dz/dv; /* z not accurate when clipped in h */
			if (h != sizeH  &&  z < zPtr[h])
				zPtr[h] = z, fPtr[h] = FRAMECOLOR;
			if (p->cnt < EMAX-1) {
				hp = p->h;
				zp = p->z;
				j = 0;
				while (j < p->cnt  &&  h > *hp)
					++j, ++hp, ++zp;
				while (j++ < p->cnt) {
					hTmp = *hp, *hp++ = h, h = hTmp;
					zTmp = *zp, *zp++ = z, z = zTmp;
				}
				*hp = h;
				*zp = z;
				++p->cnt;
			}
			++p;
			fPtr += sizeH;
			zPtr += sizeH;
		}
	}
}

static void viewEdge(edges,px1,py1,pz1,px2,py2,pz2)
edge *edges;
double px1,py1,pz1,px2,py2,pz2;
{
	double n;

	if (pz1 > 1.0  ||  pz2 > 1.0) {
		if (pz1 <= 1.0) {
			n = (1 - pz1) / (pz2 - pz1);
			mkEdge(edges,
				focLen * (px1 + n * (px2-px1)),
				focLen * (py1 + n * (py2-py1)),
				pz1,
				focLen*px2/pz2,
				focLen*py2/pz2,
				pz2 );
		}
		else if (pz2 <= 1.0) {
			n = (1 - pz2) / (pz1 - pz2);
			mkEdge(edges,
				focLen*px1/pz1,
				focLen*py1/pz1,
				pz1,
				focLen * (px2 + n * (px1-px2)),
				focLen * (py2 + n * (py1-py2)),
				pz2 );
		}
		else
			mkEdge(edges,
				focLen*px1/pz1,
				focLen*py1/pz1,
				pz1,
				focLen*px2/pz2,
				focLen*py2/pz2,
				pz2 );
	}
}

/* Z-buffer processing for one line */
static void zLine(col,v,h1,z1,h2,z2)
pixel col;
long v,h1,h2;
unsigned long z1,z2;
{
	register unsigned long z;
	register long e,dh,dz,d,sz;
	register pixel *p;
	register unsigned long *q;

	if (dh = h2 - h1) {
		z = z1;
		dz = z2 - z;
		v = v*sizeH + h1;
		p = fBuffer + v;
		q = zBuffer + v;
		sz = 0;
		if (dz > 0)
			sz = 1;
		else if (dz < 0) {
			dz = -dz;
			sz = -1;
		}
		d = 0;
		if (dz > dh) {
			d = dz/dh;
			dz -= d*dh;
			d *= sz;
		}
		e = (dz *= 2) - dh;
		dh *= 2;
		while ((dh-=2) >= 0) {
			if (z < *q)
				*q = z, *p = col;
			z += d;
			if (e >= 0) {
				z += sz;
				e -= dh;
			}
			++p;
			++q;
			e += dz;
		}
	}
}

/* Paint a poly into the buffers */
pico Paint(x)
register pico x;
{
	register pico y;
	vector pos;
	register matHandle h;
	edge *edges, *shadows;
	pico rgb;
	bool shad;
	register long i;
	register pixel c;
	register edge *p;
	double dx,dy,dz;
	double px0,py0,pz0,px1,py1,pz1,px2,py2,pz2;
	double vxs,sx0,sy0,sz0,sx1,sy1,sz1,sx2,sy2,sz2;
	vector v;
	cell c1;

	push(EVAL1(x),c1);							/* Plane list */
	x = cdr(x);
	unBoxVector(EVAL1(x),&pos,CENTIMETER);	/* Position */
	x = cdr(x);
	h = NULL; 										/* Optional matrix */
	if (!isNil(x = EVAL1(x))) {
		NEEDNUM(x);
		h = (matHandle)unBox(x);
	}
	x = pop(c1);
	if (!(edges = (edge*)NewPtr(sizeV*sizeof(edge))))
		return nilSym;
	if (!(shadows = (edge*)NewPtr(sizeV*sizeof(edge)))) {
		DisposPtr((Ptr)edges);
		return nilSym;
	}

	dx = pos.x - camLoc.x;
	dy = pos.y - camLoc.y;
	dz = pos.z - camLoc.z;

	while (isCell(x)) {
		y = car(x);
		x = cdr(x);
		i = sizeV;
		while (--i >= 0)
			edges[i].cnt = 0;
		if (!isNil(rgb = car(y)))
			c = unBoxPixel(rgb);
		y = cdr(y);
		shad = !isNil(car(y));
		y = cdr(y);
		unBoxVector(car(y), &v, MILLIMETER);
		if (h)
			transform(v.x, v.y, v.z, *h, &v);
		v.x += dx;
		v.y += dy;
		v.z += dz;
		px0 = px1 = coeff1*v.x + coeff2*v.y;
		py0 = py1 = coeff4*v.x + coeff5*v.y + coeff6*v.z;
		pz0 = pz1 = coeff7*v.x + coeff8*v.y + coeff9*v.z;
		if (shad) {
			i = sizeV;
			while (--i >= 0)
				shadows[i].cnt = 0;
			vxs = v.x + v.z + camLoc.z;
			sx0 = sx1 = coeff1*vxs + coeff2*v.y;
			sy0 = sy1 = coeff4*vxs + coeff5*v.y - coeff6*camLoc.z;
			sz0 = sz1 = coeff7*vxs + coeff8*v.y - coeff9*camLoc.z;
		}
		while (isCell(y = cdr(y))) {
			unBoxVector(car(y), &v, MILLIMETER);
			if (h)
				transform(v.x, v.y, v.z, *h, &v);
			v.x += dx;
			v.y += dy;
			v.z += dz;
			px2 = coeff1*v.x + coeff2*v.y;
			py2 = coeff4*v.x + coeff5*v.y + coeff6*v.z;
			pz2 = coeff7*v.x + coeff8*v.y + coeff9*v.z;
			viewEdge(edges,px1,py1,pz1,px2,py2,pz2);
			px1 = px2;
			py1 = py2;
			pz1 = pz2;
			if (shad) {
				vxs = v.x + v.z + camLoc.z;
				sx2 = coeff1*vxs + coeff2*v.y;
				sy2 = coeff4*vxs + coeff5*v.y - coeff6*camLoc.z;
				sz2 = coeff7*vxs + coeff8*v.y - coeff9*camLoc.z;
				viewEdge(shadows,sx1,sy1,sz1,sx2,sy2,sz2);
				sx1 = sx2;
				sy1 = sy2;
				sz1 = sz2;
			}
		}
		viewEdge(edges,px2,py2,pz2,px0,py0,pz0);
		if (!isNil(rgb)) {
			p = edges + sizeV;
			while (--p >= edges)
				if (i = p->cnt)
					while ((i-=2) >= 0)
						zLine(c, p-edges, p->h[i],p->z[i],p->h[i+1],p->z[i+1]);
		}
		if (shad) {
			viewEdge(shadows,sx2,sy2,sz2,sx0,sy0,sz0);
			p = shadows + sizeV;
			while (--p >= shadows)
				if (i = p->cnt)
					while ((i-=2) >= 0)
						zLine(0, p-shadows, p->h[i],p->z[i],p->h[i+1],p->z[i+1]);
		}
	}
	DisposPtr((Ptr)edges);
	DisposPtr((Ptr)shadows);
	return tSym;
}
