;; -*- Mode:Lisp; Syntax:Common-Lisp; Package: (*SIM-I COMMON-LISP-GLOBAL); Muser: Yes -*-

(in-package :*sim-i)

;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+
;;;> 
;;;> The Thinking Machines *Lisp Simulator is in the public domain.
;;;> You are free to do whatever you like with it, including but
;;;> not limited to distributing, modifying, and copying.

;;;> Bugs, comments and revisions due to porting can be sent to:
;;;> bug-starlisp@think.com.  Other than to Thinking Machines'
;;;> customers, no promise of support is intended or implied.
;;;>
;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+

;;; Author:  JP Massar.


(defun /!! (pvar1 &rest pvars)
  (*let ((result (apply #'internal-/!! pvar1 pvars)))
    (if!! (complexp!! result)
	  result
	  (float!! result)
	  )))


(*defun *or (PVAR)
  "Returns the OR of PVAR in all selected processors"
  (simple-pvar-argument!! pvar)
  (safety-check (new-pvar-check pvar '*or))
  (incf-use-count '*or)
  (let ((pvar-array (pvar-array pvar)))
    (declare (type simple-vector pvar-array))
    ;;(1-d-array-declaration pvar-array)
    (block xyzzy
      (do-for-selected-processors-internal (processor)
	(when (aref pvar-array processor)
	  (return-from xyzzy t)
	 ))
      nil
     )))


(*defun *and (pvar)
  "returns the AND of PVAR in all selected processors"
  (simple-pvar-argument!! pvar)
  (safety-check (new-pvar-check pvar '*and))
  (incf-use-count '*and)
  (let ((pvar-array (pvar-array pvar)))
    (declare (type simple-vector pvar-array))
    ;;(1-d-array-declaration pvar-array)
    (block xyzzy
      (do-for-selected-processors-internal (processor)
	(when (not (aref pvar-array processor))
	  (return-from xyzzy nil)
	 ))
      t
     )))



(defun general-/=!! (function-name pvar-=-function scalar-/=-function pvar1 &rest pvars)

  (simple-pvar-argument!! pvar1 &rest pvars)

  (safety-check
    (new-pvar-check pvar1 function-name)
    (new-multiple-pvar-check pvars function-name)
    )
  (incf-use-count '/=!!)

  (cond ((null pvars) t!!)

	;; only two pvars, do it fast

	((= 1 (length pvars))
	 (let ((pvar2 (first pvars))
	       (return-pvar (allocate-temp-general-pvar))
	       )
	   (when
	     (with-selected-general-pvar-arrays (processor) (pvar1-array pvar2-array return-array) (pvar1 pvar2 return-pvar)
	       (setf (aref return-array processor)
		     (funcall scalar-/=-function (aref pvar1-array processor) (aref pvar2-array processor))
		     ))
	     (make-non-void return-pvar)
	     )
	   return-pvar
	   ))
	
	 ;; the general case.  we must test pvar1 for =!!
	 ;; to all the PVARS.  If any return T!!, then we
	 ;; must return NIL!!.  We must also recursively call /=!! on the pvars

	(t
	 (*let (return-pvar)
	   (*set return-pvar t!!)
	   (dolist (pvar pvars)
	     (*set return-pvar (and!! return-pvar (not!! (funcall pvar-=-function pvar1 pvar)))))
	   (and!! return-pvar (apply 'general-/=!! pvar-=-function scalar-/=-function pvars))
	   ))

	))

(defun /=!! (pvar &rest pvars)
  (apply 'general-/=!! '/=!! #'=!! #'/= pvar pvars)
  )

(defun char/=!! (pvar &rest pvars)
  (apply 'general-/=!! 'char/=!! #'char=!! #'char/= pvar pvars)
  )

(defun char-not-equal!! (pvar &rest pvars)
  (apply 'general-/=!! 'char-not-equal!! #'char-equal!! #'char-not-equal pvar pvars)
  )

(defun enumerate!! ()
  "returns a pvar that contains 0 through N-1
   where N is the number of selected processors"
  ;; The simulator always returns the enumeration in order.
  (incf-use-count 'enumerate!!)
  (LET* ((RETURN-PVAR (allocate-temp-general-pvar))
         (RETURN-ARRAY (PVAR-ARRAY RETURN-PVAR))
         (COUNTER 0)
         )
    (declare (type simple-vector return-array))
    ;;(1-d-array-declaration return-array)
    (let ((any-set nil))
      (DO-FOR-SELECTED-PROCESSORS-INTERNAL (PROCESSOR)
        (setq any-set t)
        (SETF (AREF RETURN-ARRAY PROCESSOR) COUNTER)
        (INCF COUNTER)
        )
      (when any-set (make-non-void return-pvar))
      )
    RETURN-PVAR
    ))



(defun deposit-byte!! (into-pvar position-pvar size-pvar source-pvar)
  "Returns the deposit-byte of the specified pvar arguments."
  (simple-pvar-argument!! (into-pvar position-pvar size-pvar source-pvar))
  (safety-check
    (new-pvar-check into-pvar 'load-byte!!)
    (new-pvar-check source-pvar 'load-byte!!)
    (new-pvar-check position-pvar 'load-byte!!)
    (new-pvar-check size-pvar 'load-byte!!)
    )
  (incf-use-count 'deposit-byte!!)
  (let* ((return-pvar (allocate-temp-general-pvar)))
    (when
      (with-selected-general-pvar-arrays
	(processor)
	(return-array into-array source-array position-array size-array)
	(return-pvar into-pvar source-pvar position-pvar size-pvar)
	(let ((position (aref position-array processor))
	      (size (aref size-array processor))
	      )
	  (setf (aref return-array processor)
		(dpb (aref source-array processor)
		     (byte size position)
		     (aref into-array processor)
		     ))))
      (make-non-void return-pvar)
      )
    return-pvar
    ))


(defun xor!! (&rest pvars)
  (simple-pvar-argument!! &rest pvars)
  (let ((len (length pvars)))
    (if (eql 0 len)
	nil!!
	(*let ((count (!! 0)))
	  (dolist (pvar pvars)
	    (*when pvar (*incf count))
	   )
	  (oddp!! count)
	 ))))


(*defun *xor (pvar)
  (simple-pvar-argument!! pvar)
  (safety-check (new-pvar-check pvar '*xor))
  (*when pvar (oddp (*sum (!! 1))))
  )


(defun rotate-field (integer n word-size direction)

  "Internal function used by rot!!"
  (let* ((bytespec (byte word-size 0))
	 (field (ldb bytespec integer))		       ; bits to rotate
	 (index (1- word-size))
	 (high-order-bit (expt 2 index))      
	 )
    (setf (ldb bytespec integer) 0)
    (dotimes (j n)
      (if (eq direction 'right)
	  (if (oddp field)
	      (setq field (+ high-order-bit (ash (1- field) -1)))
	      (setq field (ash field -1)))
	  (if (logbitp index field)
	      (setq field (1+ (ash (- field high-order-bit) 1)))
	      (setq field (ash field 1))
	      )))
    (+ integer field)
    ))


(defun rot!! (integer-pvar n-pvar word-size-pvar)
  (simple-pvar-argument!! (integer-pvar n-pvar word-size-pvar))
  (safety-check
   (new-pvar-check integer-pvar 'rot!!)
   (new-pvar-check n-pvar 'rot!!)
   (new-pvar-check word-size-pvar 'rot!!)
   )
  (let* ((integer-pvar-array (pvar-array integer-pvar))
         (n-pvar-array (pvar-array n-pvar))
         (word-size-pvar-array (pvar-array word-size-pvar))
         (return-pvar (allocate-temp-general-pvar))
         (return-pvar-array (pvar-array return-pvar))
         )
    (declare (type simple-vector integer-pvar-array n-pvar-array word-size-pvar-array))
    ;;(1-d-array-declaration integer-pvar-array n-pvar-array word-size-pvar-array)
    (let ((any-set nil))
      (do-for-selected-processors-internal (j)
        (setq any-set t)
        (let ((integer (aref integer-pvar-array j))
              (n (aref n-pvar-array j))
              (word-size (aref word-size-pvar-array j))
              )
          (setf (aref return-pvar-array j)
                  (rotate-field integer (abs n) word-size (if (> n 0) 'left 'right)))
          ))
      (when any-set (make-non-void return-pvar))
      )
    return-pvar
    ))

(defun equal!! (pvar1 pvar2)
  (equalp!! pvar1 pvar2))

(defun equalp!! (pvar1 pvar2)

  (simple-pvar-argument!! (pvar1 pvar2))
  
  (cond

    ((and (simple-general-pvar-p pvar1) (simple-general-pvar-p pvar2))

     (cond

       ((and (*and (numberp!! pvar1))
	     (*and (numberp!! pvar2)))
	(=!! pvar1 pvar2))
       
       (t (eql!! pvar1 pvar2))

       ))

    ((and (array-pvar-p pvar1) (array-pvar-p pvar2))
     (if (not (equal (array-pvar-dimensions pvar1) (array-pvar-dimensions pvar2)))
	 nil!!
	 (*let ((result t!!))
	   (with-many-array-elements-iterated (element1 element2) ((pvar-array pvar1) (pvar-array pvar2))
	     (*set result (and!! result (equalp!! element1 element2)))
	     )
	   result
	   )))

    ((and (structure-pvar-p pvar1) (structure-pvar-p pvar2))
     (if (not (eq (type-of (pvar-structure pvar1)) (type-of (pvar-structure pvar2))))
	 nil!!
	 (*let ((result t!!))
	   (with-structure-elements-iterated
	     ((slot1 slot2)
	      ((pvar-structure pvar1) (pvar-structure pvar2))
	      (structure-pvar-type-front-end-slot-accessors (pvar-canonical-pvar-type pvar1))
	      )
	     (*set result (and!! result (equalp!! slot1 slot2)))
	     )
	   result
	   )))

    ((or (and (simple-general-pvar-p pvar1) (or (array-pvar-p pvar2) (structure-pvar-p pvar2)))
	 (and (simple-general-pvar-p pvar2) (or (array-pvar-p pvar1) (structure-pvar-p pvar1)))
	 (and (array-pvar-p pvar1) (structure-pvar-p pvar2))
	 (and (structure-pvar-p pvar1) (array-pvar-p pvar2))
	 )
     nil!!
     )

    (t
     (*let (result)
       (do-for-selected-processors-internal (j)
	 (*setf (pref result j) (equalp (pref pvar1 j) (pref pvar2 j)))
	 )
       result
       ))

     ))


(eval-when (load)
  (setq *char-code-limit char-code-limit)
  (setq *char-bits-limit 0)
  (setq *char-font-limit 0)
  (setq *char-code-length (integer-length (1- *char-code-limit)))
  (setq *char-bits-length (integer-length (1- *char-bits-limit)))
  (setq *char-font-length (integer-length (1- *char-font-limit)))
  (setq *character-length (+ *char-code-length *char-bits-length *char-font-length))
  )

(defun initialize-character (&key code bits font front-end-p constantp)
  code bits font front-end-p constantp
  (warn "The *Lisp simulator does simulate characters accurately.~@
         It simply uses the values of char-code-limit, char-bits-limit~@
         and char-font-limit from the implementation of Common Lisp~@
         that it is running under.  Executing initialize-character~@
         thus has no effect.  Sorry."
	))


(defun code-char!! (code-pvar &optional (bits-pvar (!! 0)) (font-pvar (!! 0)))
  (simple-pvar-argument!! code-pvar &opt bits-pvar font-pvar)
  (safety-check
    (new-pvar-check code-pvar 'code-char!!)
    (new-pvar-check bits-pvar 'code-char!!)
    (new-pvar-check font-pvar 'code-char!!)
    )
  (incf-use-count 'code-char!!)
  (let ((return-pvar (allocate-temp-general-pvar)))
    (when
      (with-selected-general-pvar-arrays
	(processor)
	(return-array code-array bits-array font-array)
	(return-pvar code-pvar bits-pvar font-pvar)
	(let ((bits (aref bits-array processor))
	      (font (aref font-array processor))
	      (code (aref code-array processor))
	      )
	  (setf (aref return-array processor) (code-char code bits font))
	  ))
      (make-non-void return-pvar)
      )
    return-pvar
    ))


(defun make-char!! (char-pvar &optional (bits-pvar (!! 0)) (font-pvar (!! 0)))
  (simple-pvar-argument!! char-pvar &opt (bits-pvar font-pvar))
  (safety-check
    (new-pvar-check char-pvar 'make-char!!)
    (new-pvar-check bits-pvar 'make-char!!)
    (new-pvar-check font-pvar 'make-char!!)
    )
  (incf-use-count 'make-char!!)
  (let ((return-pvar (allocate-temp-general-pvar)))
    (when
      (with-selected-general-pvar-arrays
	(processor)
	(return-array char-array bits-array font-array)
	(return-pvar char-pvar bits-pvar font-pvar)
	(let ((bits (aref bits-array processor))
	      (font (aref font-array processor))
	      (char (aref char-array processor))
	      )
	  (setf (aref return-array processor) (make-char char bits font))
	  ))
      (make-non-void return-pvar)
      )
    return-pvar
    ))


(defun digit-char!! (weight-pvar &optional (radix-pvar (!! 10)) (font-pvar (!! 0)))
  (simple-pvar-argument!! weight-pvar &opt (radix-pvar font-pvar))
  (safety-check
    (new-pvar-check weight-pvar 'make-weight!!)
    (new-pvar-check radix-pvar 'make-weight!!)
    (new-pvar-check font-pvar 'make-weight!!)
    )
  (incf-use-count 'make-weight!!)
  (let ((return-pvar (allocate-temp-general-pvar)))
    (when
      (with-selected-general-pvar-arrays
	(processor)
	(return-array weight-array radix-array font-array)
	(return-pvar weight-pvar radix-pvar font-pvar)
	(let ((radix (aref radix-array processor))
	      (font (aref font-array processor))
	      (weight (aref weight-array processor))
	      )
	  (setf (aref return-array processor) (digit-char weight radix font))
	  ))
      (make-non-void return-pvar)
      )
    return-pvar
    ))


(defun char-bit!! (char-pvar name-pvar)
  (pvar-argument!! char-pvar character name-pvar char-bitspec)
  (safety-check
    (new-pvar-check char-pvar 'char-bit!!)
    (new-pvar-check name-pvar 'char-bit!!)
    )
  (incf-use-count 'char-bit!!)
  (let ((return-pvar (allocate-temp-general-pvar)))
    (when
      (with-selected-general-pvar-arrays (processor)
	(return-array char-array name-array)
	(return-pvar char-pvar name-pvar)
	(let* ((name (aref name-array processor)))
	  (if (and (integerp name) (<= 0 name 3))
	      (setq name (nth name '(:control :meta :super :hyper))))
	  (setf (aref return-array processor)
		(char-bit (aref char-array processor) name)))
	)
      (make-non-void return-pvar)
      )
    return-pvar
    ))

(defun set-char-bit!! (char-pvar name-pvar new-value-pvar)
  (pvar-argument!! char-pvar character name-pvar char-bitspec new-value-pvar boolean)
  (safety-check
    (new-pvar-check name-pvar 'set-char-bit!!)
    (new-two-pvar-check char-pvar new-value-pvar 'set-char-bit!!)
    )
  (incf-use-count 'set-char-bit!!)
  (let ((return-pvar (allocate-temp-general-pvar)))
    (when
      (with-selected-general-pvar-arrays
	(processor)
	(return-array char-array name-array new-value-array)
	(return-pvar char-pvar name-pvar new-value-pvar)
	(let* ((name (aref name-array processor)))
	  (if (and (integerp name) (<= 0 name 3))
	      (setq name (nth name '(:control :meta :super :hyper))))
	  (setf (aref return-array processor)
		(set-char-bit (aref char-array processor)
			      name
			      (aref new-value-array processor))))
	)
      (make-non-void return-pvar)
      )
    return-pvar
    ))



(*defun *integer-length (pvar)
  (simple-pvar-argument!! pvar)
  (safety-check (new-pvar-check pvar '*integer-length))
  (let ((return-length 0))
    (let ((pvar-array (pvar-array pvar)))
      (do-for-selected-processors-internal (processor)
	(setq return-length (max return-length (integer-length (aref pvar-array processor))))
	))
    return-length
    ))

(defun structurep!! (pvar)
  (simple-pvar-argument!! pvar)
  (!! (structure-pvar-p pvar)))

(defun twinkle-twinkle-little* (pvar)
  (declare (ignore pvar))
  (simple-pvar-argument!! pvar)
  nil)

(defun v+-constant (v x) (map 'vector #'(lambda (y) (+ y x)) v))
(defun v--constant (v x) (map 'vector #'(lambda (y) (- y x)) v))
(defun v*-constant (v x) (map 'vector #'(lambda (y) (* y x)) v))
(defun v/-constant (v x) (map 'vector #'(lambda (y) (/ y x)) v))
(defun v+ (&rest v) (apply #'map 'vector #'+ v))
(defun v- (&rest v) (apply #'map 'vector #'- v))
(defun v* (&rest v) (apply #'map 'vector #'* v))
(defun v/ (&rest v) (apply #'map 'vector #'/ v))
(defun dot-product (v1 v2)
  (reduce #'+ (concatenate 'vector (apply #'v* (list v1 v2)))))
(defun vabs-squared (v) (funcall #'dot-product v v))
(defun vabs (v) (sqrt (vabs-squared v)))
(defun cross-product (x y)
  (vector
    (- (* (aref x 1) (aref y 2)) (* (aref x 2) (aref y 1)))
    (- (* (aref x 2) (aref y 0)) (* (aref x 0) (aref y 2)))
    (- (* (aref x 0) (aref y 1)) (* (aref x 1) (aref y 0)))
    ))
(defun vscale (v x) (map 'vector #'(lambda (y) (* y x)) v))
(defun vscale-to-unit-vector (v) (vscale v (/ (vabs v))))
(defun vector-normal (v1 v2)
  (vscale-to-unit-vector (cross-product  v1 v2)))
(defun vfloor (x) (map 'vector #'floor x))
(defun vceiling (x) (map 'vector #'ceiling x))
(defun vround (x) (map 'vector #'round x))
(defun vtruncate (x) (map 'vector #'truncate x))
(defun vlist (x) (concatenate 'list x))



(defun check-all-same-float-type (float-pvar function-name)
  (let ((first-type nil) (first-value nil))
    (let ((float-pvar-array (pvar-array float-pvar)))
      (do-for-selected-processors-internal (j)
	(if (null first-type)
	    (progn
	      (setq first-value (svref float-pvar-array j))
	      (setq first-type (type-of first-value))
	      )
	    (if (not (eq first-type (type-of (svref float-pvar-array j))))
		(error "The argument given to ~S does not contain floats of the same exact type everywhere" function-name)
		))))
    first-value
    ))




(defvar *not-all-floats-error-message* "The argument given to ~S does not contain floats everywhere")

(defmacro make-float-limit-function
	  (function-name short-float-value single-float-value double-float-value long-float-value)
  `(defun ,function-name (float-pvar)
     (simple-pvar-argument!! float-pvar)
     #+SYMBOLICS
     (progn ,single-float-value ,long-float-value) ; ignore these two on Symbolics
     (when (not (*and (floatp!! float-pvar))) (error *not-all-floats-error-message* ',function-name))
     (etypecase (check-all-same-float-type float-pvar ',function-name)
       (short-float (!! ,short-float-value))
       #-SYMBOLICS
       (single-float (!! ,single-float-value))
       (double-float (!! ,double-float-value))
       #-SYMBOLICS
       (long-float (!! ,long-float-value))
       )))


(make-float-limit-function
  least-positive-float!!
  least-positive-short-float least-positive-single-float least-positive-double-float least-positive-long-float
  )

#-KCL
(make-float-limit-function
  most-positive-float!!
  most-positive-short-float most-positive-single-float most-positive-double-float most-positive-long-float
  )

#+KCL
(eval
  '(make-float-limit-function
     most-positive-float!!
     most-positive-short-float most-positive-single-float most-positive-double-float most-positive-long-float
     )
  )

(make-float-limit-function
  float-epsilon!!
  short-float-epsilon single-float-epsilon double-float-epsilon long-float-epsilon
  )

(make-float-limit-function
  negative-float-epsilon!!
  short-float-negative-epsilon single-float-negative-epsilon double-float-negative-epsilon long-float-negative-epsilon
  )

(make-float-limit-function
  least-negative-float!!
  least-negative-short-float least-negative-single-float least-negative-double-float least-negative-long-float
  )

#-KCL
(make-float-limit-function
  most-negative-float!!
  most-negative-short-float most-negative-single-float most-negative-double-float most-negative-long-float
  )

#+KCL
(eval
  '(make-float-limit-function
     most-negative-float!!
     most-negative-short-float most-negative-single-float most-negative-double-float most-negative-long-float
     ))

(defun self!! ()
  (let ((result (make-address-object!!
		  :geometry-id (!! (the fixnum (vp-set-geometry-id *current-vp-set*)))
		  :cube-address (self-address!!)
		  )))
    (setf (pvar-address-object-geometry-id result) (vp-set-geometry-id *current-vp-set*))
    (initialize-address-object-in-non-active-processors result)
    result
    ))

  
(defun set-vp-set-geometry (vp-set geometry)
  (declare (ignore vp-set geometry))
  (error "The *Lisp Simulator does not currently allow changing the geometry of a vp set")
  )


(defun any-processors-active-function () (*or t!!))


(defun coerce!! (pvar type &aux (ctype (canonical-pvar-type type)))
  (simple-pvar-argument!! pvar)
  (if (or (null ctype) (atom ctype) (not (eq (car ctype) 'pvar)))
      (error "Invalid type ~S to coerce!!." type))
  (case (canonical-pvar-element-type ctype)
    (boolean
      (assert (*and (booleanp!! pvar)) () "Cannot coerce pvar with non-boolean values into boolean pvar")
      pvar
      )
    (front-end pvar)
    (unsigned-byte
      (assert (*and (and!! (integerp!! pvar) (not!! (minusp!! pvar)))) ()
	      "Cannot coerce pvar with negative or non-integer values to unsigned-byte pvar"
	      )
      pvar
      )
    (signed-byte
      (assert (*and (integerp!! pvar)) () "Cannot coerce pvar with non-integer values to signed-byte pvar")
      pvar
      )
    (defined-float
      (assert (*and (and!! (numberp!! pvar) (not!! (complexp!! pvar)))) ()
	      "Cannot coerce pvar with non-integer/non-float values to float pvar"
	      )
      (let ((mantissa (float-pvar-type-mantissa ctype)) (exponent (float-pvar-type-exponent ctype)))
	(if (or (eq mantissa '*) (eq exponent '*))
	    (float!! pvar)
	    (if (and (<= (float-pvar-type-mantissa ctype) 23) (<= (float-pvar-type-exponent ctype) 8))
		(float!! pvar (!! 0.0))
		(float!! pvar (!! 0.0d0))
		))))
    (complex
      (assert (*and (numberp!! pvar)) () "Cannot coerce pvar with non-numeric valuue to complex pvar")
      (let* ((mantissa (complex-pvar-type-mantissa ctype))
	     (exponent (complex-pvar-type-exponent ctype))
	     (float-pvar-type `(float-pvar ,mantissa ,exponent))
	     )
	(if (or (eq mantissa '*) (eq exponent '*))
	    (complex!! (float!! (realpart!! pvar)) (float!! (imagpart!! pvar)))
	    (if!! (complexp!! pvar)
		  (complex!! (coerce!! (realpart!! pvar) float-pvar-type) (coerce!! (imagpart!! pvar) float-pvar-type))
		  (complex!! (coerce!! pvar float-pvar-type) (coerce!! (!! 0) float-pvar-type))
		  ))))
    (string-char
      (if!! (and!! (characterp!! pvar) (string-char-p!! pvar)) pvar (code-char!! pvar))
      )
    (character
      (if (eql (pvar-type pvar) :array)
	  (let ((dimensions (array-pvar-dimensions pvar))
		(rank (array-pvar-rank pvar)))
	    (assert (and (= rank 1)
			 (= (car dimensions) 1))
		    (pvar)
		    "Can only coerce character vector pvars of length 1 to character pvars.")
	    (setq pvar (aref!! pvar 0))))
      (if!! (characterp!! pvar) pvar (code-char!! pvar))
      )
    (array
      (coerce-to-array-pvar pvar ctype)
      )
    (structure
      (error "The *Lisp Simulator does not handle coercions to structure pvars.")
      )
    ((t)
     (cond
       ((array-pvar-p pvar) (error "Array pvars cannot be coerced to general pvars"))
       ((structure-pvar-p pvar) (error "Structure pvars cannot be coerced to general pvars"))
       (t pvar)
       ))
    ((* pvar) (error "Invalid type ~S to coerce!!." ctype))
    ))


(defun coerce-to-array-pvar (pvar ctype)
  (when (not (array-pvar-p pvar))
    (error "You cannot coerce a non-array pvar to an array pvar.")
    )
  (let ((array-pvar (make-pvar-based-on-canonical-pvar-type :stack ctype)))
    (setf (pvar-constant? array-pvar) nil)
    (setf (pvar-lvalue? array-pvar) t)
    (setf (pvar-name array-pvar) 'COERCE!!-RETURN)
    (when (not (eql (array-pvar-rank pvar) (array-pvar-rank array-pvar)))
      (error "You cannot coerce an array pvar of rank ~D into an array pvar of rank ~D"
	     (array-pvar-rank pvar) (array-pvar-rank array-pvar)
	     ))
    (*set array-pvar pvar)
    (setf (pvar-lvalue? array-pvar) nil)
    array-pvar
    ))



(defun integer-reverse!! (integer-pvar)
  (declare (ignore integer-pvar))
  (error "This function is unimplementable on the *Lisp Simulator.~@
          It depends on internal representations and field lengths,~@
          which the Simulator knows nothing about."
	 ))


(defun taken-as!! (pvar type)
  (declare (ignore type))
  (new-pvar-check pvar 'taken-as!!)
  (error "This function is unimplementable on the *Lisp Simulator.~@
          It depends on internal representations of data inside the ~@
          Connection Machine, which the simulator knows nothing about."
	 ))


(defun sideways-array-p (array-pvar)
  (safety-check
    (new-pvar-check array-pvar 'sideways-array-p)
    (assert (array-pvar-p array-pvar))
    )
  (pvar-sideways-p array-pvar)
  )


(defun typep!! (pvar scalar-type)
  (safety-check
    (new-pvar-check pvar 'typep!!)
    )
  (incf-use-count 'typep!!)
  (ecase (pvar-type pvar)
    (:general
      (let ((return-pvar (allocate-temp-general-pvar)))
	(when
	  (with-selected-general-pvar-arrays
	    (processor)
	    (return-array pvar-array)
	    (return-pvar pvar)
	    (setf (aref return-array processor) (typep (aref pvar-array processor) scalar-type))
	    )
	  (make-non-void return-pvar)
	  )
	return-pvar
	))
    (:structure
      (if (typep (pvar-location pvar) scalar-type) t!! nil!!)
      )
    (:array
      (error "Implementation deficiency.  Cannot do type!! correctly on array pvars.  Sorry.")
      )))


(defun *setf-realpart!! (pvar value)
  (*set pvar (complex!! value (imagpart!! pvar)))
  )

(defun *setf-imagpart!! (pvar value)
  (*set pvar (complex!! (realpart!! pvar) value))
  )


(defun *room (&key (how :by-vp-set) (print-statistics t) (stream t))
  (declare (ignore how print-statistics))
  (format stream "~%The *Lisp Simulator does not keep memory usage information.  Sorry.")
  )


(defun track-stack (&optional trace-type (trace-action :trace) (verbose t))
  (declare (ignore trace-type trace-action verbose))
  (format t "~%The *Lisp Simulator does not keep stack memory information.  Sorry.")
  (values)
  )
