;; -*- Mode:Lisp; Syntax:Common-Lisp; Package: (*SIM-I COMMON-LISP-GLOBAL); Muser: yes -*-

(in-package :*sim-i)

;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+
;;;> 
;;;> The Thinking Machines *Lisp Simulator is in the public domain.
;;;> You are free to do whatever you like with it, including but
;;;> not limited to distributing, modifying, and copying.

;;;> Bugs, comments and revisions due to porting can be sent to:
;;;> bug-starlisp@think.com.  Other than to Thinking Machines'
;;;> customers, no promise of support is intended or implied.
;;;>
;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+

;;; Author:  JP Massar.


;;; Auxiliary functions for argument checking for the addressing functions


(defun vp-set-check (vp-set function-name)
  (assert (vp-set-p vp-set) ()
	  "The vp set argument to the function ~S was not a vp set, but has the value ~A"
	  function-name vp-set
	  ))


(defun validate-all-pvars (pvar-index-list function-name)
  (new-multiple-pvar-check pvar-index-list function-name)
  )


(defun check-grid-indices-values (index-list vp-set function-name)

  ;; make sure each element of index-list is within the
  ;; dimension boundaries of its dimension.

  (let ((j 0))
    (mapc
      #'(lambda (index upper-exclusive-bound)
	  (assert (valid-integer-range-exclusive index 0 upper-exclusive-bound) ()
		  "In ~S, the value for grid dimension ~D, ~S, is out of range with respect to vp set ~S"
		  function-name j index vp-set
		  )
	  (incf j)
	  )
      index-list
      (vp-set-dimensions vp-set)
     ))
  t)


(defun grid-indices-values-valid-p (index-list vp-set)

  ;; make sure each element of index-list is within the
  ;; dimension boundaries of its dimension.

  (let ((j 0))
    (mapc
      #'(lambda (index upper-exclusive-bound)
	  (when (null (valid-integer-range-exclusive index 0 upper-exclusive-bound))
	    (return-from grid-indices-values-valid-p nil)
	    )
	  (incf j)
	  )
      index-list
      (vp-set-dimensions vp-set)
     ))
  t)


(defun check-grid-indices (index-list vp-set function-name)

  ;; make sure index-list is the same length as the
  ;; number of grid dimensions and that each element
  ;; of the list is within the dimension boundaries
  ;; of its dimension.

  (assert (eql (length index-list) (length (vp-set-dimensions vp-set))) ()
	  "In ~S, wrong number of grid addresses provided (should be ~D)"
	  function-name (length (vp-set-dimensions vp-set)))
  (check-grid-indices-values index-list vp-set function-name)
  t
 )


(defun check-cube-address (index vp-set function-name)
  (assert (valid-integer-range-exclusive index 0 (vp-set-size vp-set)) ()
	  "In ~S, value for cube address, ~S, is not between 0 and ~D, the size of vp set ~S"
	  function-name index (vp-set-size vp-set) vp-set
	  ))

(defun check-dimension (dimension vp-set function-name)
  (assert (valid-integer-range-exclusive dimension 0 (length (vp-set-dimensions vp-set))) ()
	  "In ~S, value for dimension argument, ~S, is not between 0 and ~D, the number of dimensions in vp set ~S"
	  function-name dimension (length (vp-set-dimensions vp-set)) vp-set
	  ))

(defun check-cube-address-pvar (pvar vp-set function-name)
  (let ((pvar-array (pvar-array pvar))
        (nprocessors (vp-set-size vp-set))
        )
    (declare (type simple-vector pvar-array))
    ;;(1-d-array-declaration pvar-array)
    (do-for-selected-processors (j)
      (when (not (valid-integer-range-exclusive (aref pvar-array j) 0 nprocessors))
        (error "~S: Cube address value ~S,~% at processor address ~D,~% is not valid with respect to vp set ~S."
          function-name (aref pvar-array j) j vp-set
          )))))

(defun check-dimension-pvar (pvar vp-set function-name)
  (let ((pvar-array (pvar-array pvar))
        (number-of-dimensions (length (vp-set-dimensions vp-set)))
        )
    (declare (type simple-vector pvar-array))
    ;;(1-d-array-declaration pvar-array)
    (do-for-selected-processors-internal (j)
      (when (not (valid-integer-range-exclusive (aref pvar-array j) 0 number-of-dimensions))
        (error
            "~S: Dimension value ~S,~% at processor address ~D,~% is not valid with respect to vp set ~S, which has ~D dimensions"
          function-name (aref pvar-array j) j vp-set (length (vp-set-dimensions vp-set))
          )))))


(defun internal-cube-from-vp-grid-address!!
    (function-name error-if-bad-address relativep vp-set index-pvars)
  
  "Translates a grid address into a cube address in each selected processor.
   If error-if-bad-address is nil, NIL is returned in the returned pvar
   for the processors which have an invalid grid address specified,
   otherwise an error is generated.  The grid addresses may be relative,
   in which case they are converted to absolute.
  "
  (simple-pvar-argument!! &rest index-pvars)
  
  (safety-check
   (progn
     (vp-set-check vp-set function-name)
     (new-multiple-pvar-check index-pvars function-name)
     ))
  
  (let* ((return-pvar (allocate-temp-general-pvar))
         (return-array (pvar-array return-pvar))
         (number-of-dimensions (length index-pvars))
         (temp-array-of-indices (make-sequence 'simple-vector number-of-dimensions))
         (temp-list-of-indices-for-processor (make-list number-of-dimensions))
         )
    (declare (type simple-vector return-array))
    ;;(1-d-array-declaration return-array)
    
    ;; put the pvar arrays into a vector for fast access
    
    (dotimes (j number-of-dimensions)
      (setf (aref temp-array-of-indices j) (pvar-array (nth j index-pvars)))
      )
    
    (let ((any-set nil))
      
      (do-for-selected-processors-internal (processor)
        
        (setq any-set t)
        
        ;; for each active processor, put its grid coordinate pvar
        ;; values into a vector.  If the grid coordinates are relative
        ;; add in the absolute grid coordinates for this processor.
        
        (let ((*interpreter-safety* 0))
          (dotimes (dimension number-of-dimensions)
            (setf (nth dimension temp-list-of-indices-for-processor)
                    (if relativep
                        (+ (aref (aref temp-array-of-indices dimension) processor)
                           (internal-grid-from-vp-cube-address vp-set processor dimension function-name)
                           )
                      (aref (aref temp-array-of-indices dimension) processor)
                      ))))
        
        (if error-if-bad-address
            
            ;; check that the grid coordinates are valid, then convert them
            ;; to a cube address.
            
            (setf (aref return-array processor)
                    (internal-cube-from-vp-grid-address vp-set temp-list-of-indices-for-processor function-name)
                    )
          
          ;; if it's a valid address convert the address and store it,
          ;; otherwise store NIL.
          
          (setf (aref return-array processor)
                  (if (not (grid-indices-values-valid-p temp-list-of-indices-for-processor vp-set))
                      nil
                    (internal-cube-address-from-grid-address-list
                     (vp-set-array-of-cube-addresses vp-set)
                     temp-list-of-indices-for-processor
                     )))
          
          ))
      
      (when any-set (make-non-void return-pvar))
      
      )
    
    return-pvar
    
    ))


(defun create-cube-address-array-for-vp-set (vp-set)
  (let* ((dimensions (vp-set-dimensions vp-set))
	 (number-of-dimensions (length dimensions))
	 (total-size (vp-set-size vp-set))
	 )
    (if (< number-of-dimensions array-rank-limit)
	(let* ((cube-address-array (make-array dimensions :element-type 'fixnum))
	       (displaced-array
		 (if (= 1 number-of-dimensions)
		     cube-address-array
		     (make-array total-size :displaced-to cube-address-array :element-type 'fixnum)
		     ))
	       )
	  (dotimes (j total-size)
	    (setf (aref displaced-array j) j)
	    )
	  (setf (vp-set-array-of-cube-addresses vp-set) cube-address-array)
	  )
	(error "The Simulator cannot yet deal with vp sets of more than ~D dimensions" array-rank-limit)
	)))


(defun create-grid-address-array-for-vp-set (vp-set)
  (let* ((dimensions (vp-set-dimensions vp-set))
	 (number-of-dimensions (length dimensions))
	 (total-size (vp-set-size vp-set))
	 (cube-address-array (vp-set-array-of-cube-addresses vp-set))
	 )
    (let* ((grid-address-array
	     (make-array (list total-size number-of-dimensions) :element-type 'fixnum :initial-element -1)
	     )
	   (grid-coordinate-list nil)
	   )
      (with-grid-indices-iterated
	(
	 grid-coordinate-list
	 number-of-dimensions
	 :start-index-list (make-list number-of-dimensions :initial-element 0)
	 :end-index-list dimensions
	 :check-arguments nil
	 :bind-as :list
	 )
	(let ((cube-address (internal-cube-address-from-grid-address-list cube-address-array grid-coordinate-list)))
	  (dotimes (dimension-index number-of-dimensions)
	    (assert (= -1 (aref grid-address-array cube-address dimension-index)) () "Not working right!")
	    (setf (aref grid-address-array cube-address dimension-index) (nth dimension-index grid-coordinate-list))
	    )))
      (setf (vp-set-array-of-grid-addresses vp-set) grid-address-array)
      )))


(defun iota (n)
  (let ((result nil))
    (dotimes (j n) (push j result))
    (nreverse result)
    ))

(defun verify-consistent-mapping-for-vp-set (vp-set)
  (let* ((cube-address-array (vp-set-array-of-cube-addresses vp-set))
	 (grid-address-array (vp-set-array-of-grid-addresses vp-set))
	 (size (vp-set-size vp-set))
	 (number-of-dimensions (length (vp-set-dimensions vp-set)))
	 (dimension-index-list (iota number-of-dimensions))
	 )
    (dotimes (cube-index size)
      (let ((grid-indices
	      (mapcar
		#'(lambda (grid-index) (aref grid-address-array cube-index grid-index))
		dimension-index-list
		)))
	(let ((cube-from-grid-index (apply #'aref cube-address-array grid-indices)))
	  (assert (= cube-from-grid-index cube-index) () "Not working!")
	  )))))


(defun internal-cube-address-from-grid-addresses (cube-address-array &rest grid-addresses)
  (internal-cube-address-from-grid-address-list cube-address-array grid-addresses)
  )


(defun internal-cube-address-from-grid-address-list (cube-address-array grid-address-list)
  (let ((length (length grid-address-list)))
    (cond
      ((eql 1 length) (car grid-address-list))
      ((eql 2 length)
       (aref #-KCL (the (array fixnum 2) cube-address-array) #+KCL cube-address-array
	     (first grid-address-list) (second grid-address-list)
	     ))
      ((eql 3 length)
       (aref #-KCL (the (array fixnum 3) cube-address-array) #+KCL cube-address-array
	     (first grid-address-list) (second grid-address-list) (third grid-address-list)
	     ))
      (t (apply #'aref cube-address-array grid-address-list))
      )))

(defun internal-cube-address-from-grid-address-vector (cube-address-array grid-address-vector)
  (let ((length (length grid-address-vector)))
    (cond
      ((eql 1 length) (svref grid-address-vector 0))
      ((eql 2 length)
       (aref #-KCL (the (array fixnum 2) cube-address-array) #+KCL cube-address-array
	     (svref grid-address-vector 0) (svref grid-address-vector 1)
	     ))
      ((eql 3 length)
       (aref #-KCL (the (array fixnum 3) cube-address-array) #+KCL cube-address-array
	     (svref grid-address-vector 0) (svref grid-address-vector 1) (svref grid-address-vector 2)
	     ))
      ((eql 4 length)
       (aref #-KCL (the (array fixnum 4) cube-address-array) #+KCL cube-address-array
	     (svref grid-address-vector 0) (svref grid-address-vector 1)
	     (svref grid-address-vector 2) (svref grid-address-vector 3)
	     ))
      (t (apply #'aref cube-address-array (concatenate 'list grid-address-vector)))
      )))
