;;; -*- Mode: LISP; Syntax: COMMON-LISP; Package: (*SIM-I COMMON-LISP-GLOBAL); Base: 10; Muser: yes -*-

(in-package :*sim-i)

;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+
;;;> 
;;;> The Thinking Machines *Lisp Simulator is in the public domain.
;;;> You are free to do whatever you like with it, including but
;;;> not limited to distributing, modifying, and copying.

;;;> Bugs, comments and revisions due to porting can be sent to:
;;;> bug-starlisp@think.com.  Other than to Thinking Machines'
;;;> customers, no promise of support is intended or implied.
;;;>
;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+

;;; Author:  JP Massar.


;;; A few simple functions dealing with the PVAR type.


(defun new-pvar-check (pvar function-name)
  (assert (is-pvar pvar) () "The object ~S, an argument to ~S, is not a pvar" pvar function-name)
  (assert (eq (pvar-vp-set pvar) *current-vp-set*) ()
	  "The pvar ~S,~% used in function ~S,~% does not belong to the current vp set, ~S"
	  pvar function-name *current-vp-set*
	  ))

(defun new-pvar-check-no-vp-check (pvar function-name)
  (assert (is-pvar pvar) () "The object ~S, an argument to ~S, is not a pvar" pvar function-name)
  )

(defun new-pvar-check-lvalue (pvar function-name)
  (new-pvar-check pvar function-name)
  (unless (pvar-lvalue? pvar)
    (error "~S:  A destination pvar, ~S, is an expression, not an lvalue.~@
            You are probably trying to modify a temporary pvar.  Perhaps you want to use ALIAS!!."
	   function-name pvar
	   ))
  (when (pvar-constant? pvar)
    (error "Internal error in ~S.  Pvar ~S is both a constant and an lvalue ??" function-name pvar)
    ))

(defun new-pvar-check-lvalue-no-vp-check (pvar function-name)
  (assert (is-pvar pvar) () "The object ~S, an argument to ~S, is not a pvar" pvar function-name)
  (unless (pvar-lvalue? pvar)
    (error "~S:  A destination pvar, ~S, is an expression, not an lvalue." function-name pvar)
    )
  (when (pvar-constant? pvar)
    (error "Internal error in ~S.  Pvar ~S is both a constant and an lvalue ??" function-name pvar)
    ))


(defun new-two-pvar-check (pvar1 pvar2 function-name)
  (new-pvar-check pvar1 function-name)
  (new-pvar-check pvar2 function-name)
  )

(defun new-multiple-pvar-check (pvars function-name)
  (mapc
    #'(lambda (pvar) (new-pvar-check pvar function-name))
    pvars
    ))

(defun pvar-check-lvalue (pvar function-name)

  ;; make sure a pvar can be legitimately written to.

  (pvar-check pvar)
  (unless (pvar-lvalue? pvar)
    (error "~S:  Destination pvar is an expression, not an lvalue." function-name)
    )
  (when (pvar-constant? pvar)
    (error "Internal error in ~S.  Pvar ~S is both a constant and an lvalue ??" function-name pvar)
    )
  )


;;;
;;; This function determines whether the user was returning
;;; a temporary pvar.  If the return-value is not a pvar, then
;;; we set the *TEMP-PVAR-LIST* back to its previous value and
;;; return the return-value.  If the return-value is in fact
;;; a temporary pvar, then we must arrange to return it.
;;;
;;; This is used by *DEFUN and *LET.
;;;
;;; This function is used EVERYWHERE because it controls the
;;; temporary pvar stack so it should be optimized as hell!
;;;


(defun *map-array (function-of-one-pvar-argument pvar)
  (assert (array-pvar-p pvar) () "*map-structure was called, but pvar argument was not an array pvar")
  (let ((lisp-array-holding-pvars (pvar-array pvar)))
      (with-array-elements-iterated
	lisp-array-holding-pvars
	element-pvar
	(funcall function-of-one-pvar-argument element-pvar)
	)))


(defun *map-structure (function-of-one-pvar-argument pvar)
  (assert (structure-pvar-p pvar) () "*map-structure was called, but pvar argument was not a structure pvar")
  (let* ((front-end-structure (pvar-structure pvar))
	 (type-name (pvar-structure-name pvar))
	 (front-end-slot-accessor-names (structure-pvar-type-front-end-slot-accessors type-name))
	 )
    (mapc
      #'(lambda (accessor) (funcall function-of-one-pvar-argument (funcall accessor front-end-structure)))
      front-end-slot-accessor-names
      )
    ))



(defun clean-up-stack (start end)
  (do ((pvar-list start (cdr pvar-list)))
      ((eq pvar-list end))
    (clean-up-stack-pvar (car pvar-list))
    ))


(defun clean-up-stack-pvar (pvar)
  (declare (type pvar pvar))
  (let ((class (pvar-class pvar)))
    (ecase class
      (:general
	(let ((array (pvar-array pvar)))
	  (if array
	      (progn
		(return-pvar-array-to-pool array)
		(setf (pvar-array pvar) nil)
		)
	      (error "Internal error.  Allocated general pvar on temporary list does not have pvar array")
	      )))
      (:array
	(*map-array #'(lambda (element-pvar) (*deallocate-internal element-pvar :allocate!!)) pvar)
	(setf (pvar-array pvar) nil)
	)
      (:structure
	(*map-structure #'(lambda (slot-pvar) (*deallocate-internal slot-pvar :allocate!!)) pvar)
	(setf (pvar-structure pvar) nil)
	))))


(defun handle-returning-pvar

       (return-value old-*temp-pvar-list* definitely-return-pvar)

;  (when *fuck*
;    (print 'entering)
;    (describe return-value)
;    )
;    (print (list 'return-value return-value))
;    (dotimes (j 5) (print (nth j *temp-pvar-original-list*)))
;    (print 'old-*temp-pvar-list*)
;    (dotimes (j 5) (print (nth j old-*temp-pvar-list*)))
;    )

  (cond

    ;; definitely-return-pvar is set iff the value being
    ;; returned is a pvar bound by the *LET or *LET* that
    ;; is doing this call to handle-returning-pvar.

    ;; If the pvar is an lvalue, then it was either allocated
    ;; on the heap or it was allocated by a *LET or *LET*
    ;; which has not yet returned.  Hence the pvar is not
    ;; to be regarded as temporary.

    ;; If the pvar has the property :heap-constant, then it
    ;; was created by !! as a temporary but it was put into
    ;; one of the constant hash tables and the actual pvar
    ;; and its array were allocated on the heap.  Hence it
    ;; is not to be regarded as temporary.

    ((or definitely-return-pvar
	 (and return-value
	      (pvar-p return-value)
	      (not (pvar-lvalue? (the pvar return-value)))
	      (not (eq :heap-constant (pvar-constant? (the pvar return-value))))
	      ))

;     (print 'in-here-0)

     ;; the user is returning a temporary pvar.
     ;; Lets trade it with the first thing in the OLD-*TEMP-PVAR-LIST*

     (cond

       ;; in the special case where the user happens
       ;;to be returning the first pvar in OLD-*TEMP-PVAR-LIST*,
       ;; we don't have to do anything at all except make
       ;; sure what we return has been converted into
       ;; a temporary pvar (since *LET variables are
       ;; allocated off the *TEMP-PVAR-LIST* but are not
       ;; marked as temporary) and then clean up what
       ;; is above us on the stack.

       ((eq return-value (first old-*temp-pvar-list*))
;	(print 'in-here-1)
	(setf (pvar-lvalue? (the pvar return-value)) nil)
	(clean-up-stack (cdr old-*temp-pvar-list*) *temp-pvar-list*)
	)

       (t

	;; Some random temporary pvar somewhere above where
	;; we are returning to on the stack is being returned.
	;; We are simply going to swap the contents of the
	;; pvar being returned with the pvar at the place on
	;; the stack we want to return to, then clean up the
	;; stack.

;	(print 'in-here-2)

	(let* ((return-pvar (first old-*temp-pvar-list*))
	       (old-return-pvar-array (pvar-array return-pvar))
	       (old-return-pvar-type (pvar-type return-pvar))
	       (old-return-pvar-structure-name (pvar-structure-name return-pvar))
	       )

;	  (print (list 'return-pvar return-pvar))
;	  (print (list 'old-return-pvar-array old-return-pvar-array))

	  ;; move pvar data to the one we are going
	  ;; to return from the one the user passed back
	  ;; Swap the data arrays.

	  (copy-pvar-slots return-pvar return-value)
	  (setf (pvar-array return-value) old-return-pvar-array)
	  (setf (pvar-type return-value) old-return-pvar-type)
	  (if (eq :structure old-return-pvar-type)
	      (setf (pvar-structure-name return-value) old-return-pvar-structure-name)
	      )
	  (setf (pvar-lvalue? return-pvar) nil)
	  (setq return-value return-pvar)

	  (clean-up-stack (cdr old-*temp-pvar-list*) *temp-pvar-list*)

	  ))

       )
	 
     ;; set the *TEMP-PVAR-LIST* to the element after the one we are returning

     (setq *temp-pvar-list* (cdr old-*temp-pvar-list*))

     )

    ;; If we weren't returning a PVAR, then just 'pop' the stack
    ;; and 'free up' any temporary pvars that were allocated
    ;; while the form which this function wraps around was evaluated.

    (T
     (clean-up-stack old-*temp-pvar-list* *temp-pvar-list*)
     (setq *temp-pvar-list* old-*temp-pvar-list*)
     )

    )

;  (when *fuck*
;    (print 'leaving)
;    (dotimes (j 5) (print (nth j *temp-pvar-original-list*)))
;    )

  return-value

  )


;;;
;;; This will copy the contents of a pvar into another pvar conditionally
;;;

(defun *copy-pvar (dest-pvar source-pvar)

  (simple-pvar-argument!! dest-pvar source-pvar)
  
  (safety-check (new-pvar-check source-pvar '*set))
  (safety-check (new-pvar-check-lvalue dest-pvar '*set))

  (let ((dest-is-general? (general-pvar-p dest-pvar))
	(source-is-general? (general-pvar-p source-pvar))
	)
    
    ;; The simple case.  Both pvars are general pvars
    ;; and neither contain any arrays or structures.
    
    (when (and dest-is-general? source-is-general?)
;      (when (and (null (general-pvar-array-list dest-pvar))
;		 (null (general-pvar-structure-list dest-pvar))
;		 (null (general-pvar-array-list source-pvar))
;		 (null (general-pvar-structure-list source-pvar))
;		 )
	(*copy-simple-general-pvar dest-pvar source-pvar)
	(return-from *copy-pvar nil)
	)
    
    (let ((dest-is-array? (array-pvar-p dest-pvar))
	  (dest-is-structure? (structure-pvar-p dest-pvar))
	  (source-is-array? (array-pvar-p source-pvar))
	  (source-is-structure? (structure-pvar-p source-pvar))
	  )
      
      (cond
	
	((and dest-is-array? source-is-array?) (*copy-array-pvar dest-pvar source-pvar))
	
	((and dest-is-structure? source-is-structure?) (*copy-structure-pvar dest-pvar source-pvar))
	
	((and dest-is-general? source-is-array?) (*copy-array-pvar-into-general-pvar dest-pvar source-pvar))
	
	((and dest-is-general? source-is-structure?) (*copy-structure-pvar-into-general-pvar dest-pvar source-pvar))
	
	((and dest-is-array? source-is-general?) (*copy-general-pvar-into-array-pvar dest-pvar source-pvar))
	
	((and dest-is-structure? source-is-general?) (*copy-general-pvar-into-structure-pvar dest-pvar source-pvar))
	
	((and dest-is-general? source-is-general?) (*copy-complex-general-pvar dest-pvar source-pvar))
	
	((or (and dest-is-array? source-is-structure?)
	     (and dest-is-structure? source-is-array?)
	     )
	 (error "You cannot copy an ARRAY into a STRUCTURE, or vice versa!")
	 )

	(t (error "Internal error in *copy-pvar.  This condition cannot happen!"))
	
	))))




(defun *copy-simple-general-pvar (dest-pvar source-pvar)

  ;; Copy the contents of the processors one by one
  ;; for every selected processor.

  (let ((dest-pvar-array (pvar-array dest-pvar))
	(source-pvar-array (pvar-array source-pvar))
	(any-active nil)
	)
    (declare (type simple-vector dest-pvar-array source-pvar-array))
    ;;(1-d-array-declaration dest-pvar-array source-pvar-array)
    (do-for-selected-processors-internal (processor)
      (setq any-active t)
      (setf (aref dest-pvar-array processor) (aref source-pvar-array processor))
      )
    (when any-active (make-non-void dest-pvar))
    ))


(defun arrays-the-same-shape? (array1 array2)
  (equal (array-dimensions array1) (array-dimensions array2))
  )


(defun *copy-array-pvar (dest-pvar source-pvar)

  ;; If a pvar is an array pvar, its pvar-data slot
  ;; (which aliases to be the pvar-array slot)
  ;; is a lisp array of the declared shape
  ;; of the *Lisp array.

  ;; Make sure the two arrays agree in shape.  Then
  ;; recursively copy each element of the array.

  (let ((dest-array (pvar-array dest-pvar))
	(source-array (pvar-array source-pvar))
	)

    (when (not (arrays-the-same-shape? dest-array source-array))
      (error
	"The destination pvar ~S is an array of dimensions ~S.~
       The source pvar ~S is an array of dimensions ~S.~
       Since these two shapes are not the same I cannot copy the source to the destination.~
      "
	dest-pvar
	(array-dimensions dest-array)
	source-pvar
	(array-dimensions source-array)
	))
	 
    (let ((number-of-elements (array-total-size dest-array)))

      (if (eql 1 (array-rank dest-array))

	  (dotimes (j number-of-elements)
	    (*copy-pvar (aref dest-array j) (aref source-array j))
	    )
      
	  (let ((displaced-dest-array (make-array number-of-elements :displaced-to dest-array))
		(displaced-source-array (make-array number-of-elements :displaced-to source-array))
		)
	    (dotimes (j number-of-elements)
	      (*copy-pvar (aref displaced-dest-array j) (aref displaced-source-array j))
	      ))

	  ))))


(defun *copy-structure-pvar (dest-pvar source-pvar)

  ;; If a pvar is a structure pvar its pvar-data slot
  ;; (aliased to the pvar-structure slot)
  ;; is a lisp structure corresponding slot by slot
  ;; to the *Lisp structure.

  ;; Make sure the two structures are the same type,
  ;; then recursively copy each slot.

  (let ((dest-structure (pvar-structure dest-pvar))
	(source-structure (pvar-structure source-pvar))
	)

    (when (not (eq (type-of dest-structure) (type-of source-structure)))
      (error
	"The destination pvar ~S is a structure pvar of type ~S.~
         The source pvar ~S is a structure pvar of type ~S.~
         Since these two types are not identical I cannot copy the source to the destination..~
        "
	dest-pvar
	(type-of dest-structure)
	source-pvar
	(type-of source-structure)
	))

    (let ((canonical-pvar-type (pvar-canonical-pvar-type dest-pvar)))
      (let ((slot-accessor-function-names
	      (structure-pvar-type-front-end-slot-accessors
		(structure-pvar-type-name canonical-pvar-type)
		)))
	(dolist (slot-name slot-accessor-function-names)
	  (*copy-pvar (funcall slot-name dest-structure) (funcall slot-name source-structure))
	  )))

    (when (eq 'address-object (pvar-structure-name dest-pvar))
      (if (eql (*sum (!! 1)) *number-of-processors-limit*)
	  (set-address-object-cached-geometry-id dest-pvar (address-object-cached-geometry-id source-pvar))
	  (progn
	    (decache-address-object-pvar dest-pvar)
	    (cache-address-object-pvar-if-possible dest-pvar '*set :check-legality nil)
	    )))

    ))


(defun *copy-array-pvar-into-general-pvar (dest-pvar source-pvar)
  (cond
    ((void-pvar-p dest-pvar)
     (if (not (eq :stack (allocated-pvar-p dest-pvar)))
	 (setf (vp-set-heap-pvar-arrays (pvar-vp-set dest-pvar))
	       (cons (pvar-location dest-pvar) (vp-set-heap-pvar-arrays (pvar-vp-set dest-pvar)))
	       )
	 (return-pvar-array-to-pool (pvar-array dest-pvar))
	 )
     (make-pvar-into-array-pvar
       dest-pvar
       (pvar-array-dimensions source-pvar)
       (pvar-array-canonical-element-type source-pvar)
       (if (member dest-pvar *temp-pvar-original-list* :test #'eq) :stack :heap)
       )
     (setf (pvar-lvalue? dest-pvar) t)
     (*copy-array-pvar dest-pvar source-pvar)
     )
    ((null (*or t!!)))
    (t (error "You cannot copy an array pvar into any non-array pvar other than one which is uninitialized"))
    ))


(defun *copy-structure-pvar-into-general-pvar (dest-pvar source-pvar)
  ;; (describe-pvar dest-pvar)
  (cond
    ((void-pvar-p dest-pvar)
     (if (not (eq :stack (allocated-pvar-p dest-pvar)))
	 (setf (vp-set-heap-pvar-arrays (pvar-vp-set dest-pvar))
	       (cons (pvar-location dest-pvar) (vp-set-heap-pvar-arrays (pvar-vp-set dest-pvar)))
	       )
	 (return-pvar-array-to-pool (pvar-array dest-pvar))
	 )
     (make-pvar-into-structure-pvar
       dest-pvar
       (pvar-structure-name source-pvar)
       (if (member dest-pvar *temp-pvar-original-list* :test #'eq) :stack :heap)
       )
     ;; (describe-pvar dest-pvar)
     (setf (pvar-lvalue? dest-pvar) t)
     (*copy-structure-pvar dest-pvar source-pvar)
     )
    ((null (*or t!!)))
    (t (error "You cannot copy a structure pvar into any non-array pvar other than one which is uninitialized"))
    ))



(defun *copy-general-pvar-into-array-pvar (dest-pvar source-pvar)
  
  dest-pvar source-pvar

  (if (not (*or t!!))
      (return-from *copy-general-pvar-into-array-pvar nil)
      (error "You cannot copy an array pvar into a general pvar")
      ))



(defun *copy-general-pvar-into-structure-pvar (dest-pvar source-pvar)

  dest-pvar source-pvar

  (if (not (*or t!!))
      (return-from *copy-general-pvar-into-structure-pvar nil)
      (error "You cannot copy a structure pvar into a general pvar")
      ))


(defun *copy-complex-general-pvar (dest-pvar source-pvar)
  ;; Crude, but effective.
  (error "Internal error.  This should not be able to happen anymore")
  (do-for-selected-processors-internal (j)
    (*setf (pref dest-pvar j) (pref source-pvar j))
    ))


(DEFUN !! (VALUE)

  "Returns a pvar that contains VALUE in all processors."

  (cond
    ((eq value t) t!!)
    ((eq value nil) nil!!)
    (t (!!-with-hash value))
    ))


;; Find constants in the hash table and return their
;; corresponding pvar.  If the constant isn't in the
;; hash table yet and the hash table isn't full, put it in.

(defun !!-with-hash (value)

  ;; Get the hash table for the current vp set.

  (let* ((current-vp-set *current-vp-set*)
	 (current-hash-table (vp-set-constants-hash-table current-vp-set))
	 )

    ;; If the hash table hasn't been created yet create it.

    (when (not current-hash-table)
      (setf (vp-set-constants-hash-table current-vp-set)
	    (make-hash-table :test #'eql :size *maximum-number-of-entries-in-constant-pvar-hash-table*)
	    )
      (setq current-hash-table (vp-set-constants-hash-table current-vp-set))
      )

    ;; If the value is a key in the hash table, use the
    ;; pvar value of that key.

    (let ((pvar (gethash value current-hash-table)))
      (if pvar (return-from !!-with-hash pvar))
      )

    (let ((constant-heap? nil))

      (let ((result

	      (cond
	      
		;; We only hash scalars.
	      
		((or (numberp value) (characterp value))
	       
		 ;; should we insert this constant into the hash table
		 ;; for future reference? 
	       
		 (let ((insert-into-table
			 (and (not (floatp value)) (not (complexp value))
			      (< (hash-table-count current-hash-table)
				 *maximum-number-of-entries-in-constant-pvar-hash-table*
				 )))
		       (new-pvar nil)
		       )
		 
		   ;; create the new pvar.  If it is to go into the hash table
		   ;; make it permanent, otherwise make it temporary.
		   ;; fill all its fields with 'value'.
		 
		   (setq new-pvar
			 (if insert-into-table
			     (prog1
			       (make-general-pvar :heap)
			       (setq constant-heap? t)
			       )
			     (make-general-pvar :stack)
			     ))

		   (setf (pvar-name new-pvar) (intern (format nil "CONSTANT-~S" value)))
		   (fill-array (pvar-array new-pvar) value)
		   (make-non-void new-pvar)
		 
		   ;; put the pvar into the hash table if we want to.
		 
		   (if insert-into-table (setf (gethash value current-hash-table) new-pvar))
		 
		   new-pvar
		 
		   ))

		;; handle arrays and structures separately.
	      
		((and (symbolp (type-of value))
		      (structure-pvar-type-known-*defstruct-type (type-of value))
		      )
		 (!!-structure (type-of value) value)
		 )
	      
		((arrayp value) (!!-array value))
	      
		(t (error "Cannot put values of type ~S into a pvar" (type-of value)))
	      
		)))

	(setf (pvar-constant? result) (if constant-heap? :heap-constant t))
	(setf (pvar-lvalue? result) nil)
	(setf (pvar-constant-value result) value)

	result

	))))


(defun !!-structure (type value)
  (let ((!!-function (get type '*defstruct-!!-function)))
    (when (null !!-function)
      (error "Internal error.  The type ~S is a known *DEFSTRUCT type but has no !! function" type)
      )
    (funcall !!-function value)
    ))


(defun front-end!! (value)
  (*let (result)
    (let ((result-array (pvar-array result)))
      (declare (type simple-vector result-array))
      ;;(1-d-array-declaration result-array)
      (do-for-selected-processors-internal (j) (setf (aref result-array j) value))
      )
    result
    ))

(defun front-end-p!! (pvar)
  (simple-pvar-argument!! pvar)
  (safety-check (new-pvar-check pvar 'front-end-p!!))
  (cond
   ((array-pvar-p pvar) nil!!)
   ((structure-pvar-p pvar) nil!!)
   ((general-pvar-p pvar)
    (*let (result)
          (let ((result-array (pvar-array result))
                (pvar-array (pvar-array pvar))
                )
            (declare (type simple-vector result-array pvar-array))
            ;;(1-d-array-declaration result-array pvar-array)
            (do-for-selected-processors-internal (j)
              (let ((value (aref pvar-array j)))
                (setf (aref result-array j) (and value (not (eq t value)) (not (numberp value)) (not (characterp value))))
                )))
          result
          ))))

(defun print-hash ()
  (maphash #'(lambda (key val) (format t "~%~S: ~S" key val))
	   *constant-pvar-hash-table*
   ))






(defun fill-array (array value)
  ;; Fill a vector with a value.
  ;; Usually this is a pvar array, which is a (vector t),
  ;; but it could be a context array, which is (vector bit).
  (let ((size (length array))
	(array array)
	)
    #+SYMBOLICS
    (declare (sys:array-register array))
    #-SYMBOLICS
    (declare (type simple-vector array))
    (declare (type fixnum size))
    (dotimes (j size)
      (declare (type fixnum j))
      (setf (aref array j) value)
      )))
