;; -*- Mode:Lisp; Syntax:Common-Lisp; Package: (*SIM-I COMMON-LISP-GLOBAL); Muser: Yes -*-

(in-package :*sim-i)

;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+
;;;> 
;;;> The Thinking Machines *Lisp Simulator is in the public domain.
;;;> You are free to do whatever you like with it, including but
;;;> not limited to distributing, modifying, and copying.

;;;> Bugs, comments and revisions due to porting can be sent to:
;;;> bug-starlisp@think.com.  Other than to Thinking Machines'
;;;> customers, no promise of support is intended or implied.
;;;>
;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+

;;; Author:  JP Massar.


(eval-when (compile load eval)

  (defun char-flipcase (char)
    (if (upper-case-p char)
	(char-downcase char)
	(if (lower-case-p char)
	    (char-upcase char)
	    char
	    )))

  (defun compare (x y)
    (safety-check
      (when (or (not (and (numberp x) (not (complexp x))))
		(not (and (numberp y) (not (complexp y))))
		)
	(error "Only non-complex numeric pvars may be used in compare!!")
	))
    (if (< x y) -1 (if (> x y) 1 0))
    )

  (defun starlisp-sqrt (x)
    (safety-check
      (when (and (not (complexp x)) (minusp x))
	(error "Taking the square root of a negative non-complex number is illegal in *Lisp.~@
              Use complex!! to first coerce the pvar to have complex values."
	       )))
    (sqrt x)
    )

  (defun starlisp-expt (base power)
    (safety-check
      (when (or (not (complexp base)) (not (complexp power)))
	(when (and (floatp base) (minusp base) (floatp power))
	  (error "Raising a negative floating point number to a floating point power is illegal in *Lisp.~@
                Use complex!! to first coerce the base to have complex values."
		 ))
	(when (and (integerp base) (integerp power) (minusp power))
	  (error "Raising an integer to a negative integer power is illegal in *Lisp.~@
                Use float!! to first coerce the base to have floating point values"
		 ))))
    (expt base power)
    )

  (defun starlisp-log (number &optional base)
    (safety-check
      (when (and (not (complexp number)) (minusp number))
	(error "Taking the log of a negative non-complex number is illegal in *Lisp.~@
              Use complex!! to first coerce the pvar to have complex values."
	       ))
      (when (and base (not (complexp base)) (minusp base))
	(error "Taking the log of a number to a negative non-complex base is illegal in *Lisp.~@
              Use complex!! to first coerce the base pvar to have complex values."
	       )))
    (if base
	(log number base)
	(log number)
	))


  (defun starlisp-asin (x)
    (safety-check
      (when (and (not (complexp x)) (> x 1.0))
	(error "Taking the arc-sine of a non-complex number which is greater than 1.0 is illegal in *Lisp.~@
              Use complex!! to first coerce the argument to have complex values."
	       )))
    (asin x)
    )

  (defun starlisp-acos (x)
    (safety-check
      (when (and (not (complexp x)) (> x 1.0))
	(error "Taking the arc-cosine of a non-complex number which is greater than 1.0 is illegal in *Lisp.~@
              Use complex!! to first coerce the argument to have complex values."
	       )))
    (acos x)
    )

  (defun starlisp-acosh (x)
    (safety-check
      (when (and (not (complexp x)) (< x 1.0))
	(error "Taking the acosh of a non-complex number which is less than 1.0 is illegal in *Lisp.~@
              Use complex!! to first coerce the argument to have complex values."
	       ))
      (acosh x)
      ))

  (defun starlisp-atanh (x)
    (safety-check
      (when (and (not (complexp x)) (> x 1.0))
	(error "Taking the atanh of a non-complex number which is less than 1.0 is illegal in *Lisp.~@
              Use complex!! to first coerce the argument to have complex values."
	       ))
      (atanh x)
      ))

  (defun front-end-gray-code-from-integer (integer)
    (logxor integer (ash integer -1))
    )

  (defun front-end-integer-from-gray-code (gray-code)
    (let ((answer 0))
      (dotimes (bit (integer-length (abs gray-code)))
	(setq answer (logxor answer (ash gray-code (- bit))))
	)
      answer
      ))

  (defun load-byte (source position size)
    (ldb (byte size position) source)
    )

  )


(progn


(def-trivial-one-arg-*lisp-functions (
   							       
   (1+!! 1+)
   (1-!! 1-)
   (booleanp!! booleanp)
   (floatp!! floatp)
   (integerp!! integerp)
   (complexp!! complexp)
   (numberp!! numberp)
   (characterp!! characterp)
   (signum!! signum)
   (zerop!! zerop)
   (oddp!! oddp)
   (evenp!! evenp)
   (plusp!! plusp)
   (minusp!! minusp)
   (realpart!! realpart)
   (imagpart!! imagpart)
   (phase!! phase)
   (conjugate!! conjugate)
   (cis!! cis)
   (not!! not)
   (null!! null)
   (isqrt!! isqrt)
   (lognot!! lognot)
   (random!! random)
   (abs!! abs)
   (sin!! sin)
   (cos!! cos)
   (tan!! tan)
   (exp!! exp)
   (logcount!! logcount)
   (integer-length!! integer-length)
   (standard-char-p!! standard-char-p)
   (graphic-char-p!! graphic-char-p)
   (string-char-p!! string-char-p)
   (alpha-char-p!! alpha-char-p)
   (upper-case-p!! upper-case-p)
   (lower-case-p!! lower-case-p)
   (both-case-p!! both-case-p)
   (alphanumericp!! alphanumericp)
   (char-code!! char-code)
   (char-upcase!! char-upcase)
   (char-downcase!! char-downcase)
   (char-flipcase!! char-flipcase)
   (int-char!! int-char)
   (char-bits!! char-bits)
   (char-font!! char-font)
   (character!! character)
   (char-int!! char-int)

   (sqrt!! starlisp-sqrt)
   (asin!! starlisp-asin)
   (acos!! starlisp-acos)
   (acosh!! starlisp-acosh)
   (atanh!! starlisp-atanh)

   (sinh!! sinh)
   (cosh!! cosh)
   (tanh!! tanh)
   (asinh!! asinh)
   
   (byte-size!! byte-size)
   (byte-position!! byte-position)

   (gray-code-from-integer!! front-end-gray-code-from-integer)
   (integer-from-gray-code!! front-end-integer-from-gray-code)

  ))


(def-trivial-two-arg-*lisp-functions (

  (eq!! eq)
  (eql!! eql)
  (mod!! mod)
  (rem!! rem)
  (ash!! ash)
  (expt!! starlisp-expt)
  (lognand!! lognand)
  (lognor!! lognor)
  (logandc1!! logandc1)
  (logandc2!! logandc2)
  (logorc1!! logorc1)
  (logorc2!! logorc2)
  (logtest!! logtest)
  (logbitp!! logbitp)
  (mask-field!! mask-field)
  (byte!! byte)
  (ldb!! ldb)
  (ldb-test!! ldb-test)
  (scale-float!! scale-float)
  (compare!! compare)

 ))


(def-trivial-optional-two-arg-*lisp-functions (

  (float!! float)
  (truncate!! truncate)
  (ceiling!! ceiling)
  (floor!! floor)
  (round!! round)
  (complex!! complex)
  (atan!! atan)
  (ffloor!! ffloor)
  (fceiling!! fceiling)
  (ftruncate!! ftruncate)
  (fround!! fround)
  (float-sign!! float-sign)
  (digit-char-p!! digit-char-p)
  (log!! starlisp-log)

 ))


(def-trivial-three-arg-*lisp-functions (
					
  (load-byte!! load-byte)
  (dpb!! dpb)
  (deposit-field!! deposit-field)
  (boole!! boole)

  ))

(def-trivial-nary-*lisp-functions (

   (+!! + t 0)
   (-!! - nil 0)
   (*!! * t 1)
   (internal-/!! / nil 0)
   (logand!! logand t -1)
   (logior!! logior t 0)
   (logxor!! logxor t 0)
   (logeqv!! logeqv t -1)
   (min!! min nil nil)
   (max!! max nil nil)
   (gcd!! gcd t 0)
   (lcm!! lcm nil 0)

  ))


(define-trivial-functions
  def-trivial-*lisp-reduction-function-using-initial-value

  '((*sum + 0 0)
    (*logand logand -1 -1)
    (*logior logior 0 0)
    (*logxor logxor 0 0)
   ))


(define-trivial-functions
  def-trivial-*lisp-reduction-function-using-first-element

  '((*min min nil)
    (*max max nil)
   ))



(def-trivial-*lisp-comparision-functions (
					  
  (=!! =)
  (<!! <)
  (>!! >)
  (<=!! <=)
  (>=!! >=)
  (char=!! char=)
  (char<!! char<)
  (char>!! char>)
  (char<=!! char<=)
  (char>=!! char>=)
  (char-equal!! char-equal)
  (char-lessp!! char-lessp)
  (char-greaterp!! char-greaterp)
  (char-not-lessp!! char-not-lessp)
  (char-not-greaterp!! char-not-greaterp)

 ))

)
