;;; -*- SYNTAX: COMMON-LISP; MODE: LISP; BASE: 10; PACKAGE: *SIM-I; MUSER: YES -*-

(in-package :*sim-i)

;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+
;;;> 
;;;> The Thinking Machines *Lisp Simulator is in the public domain.
;;;> You are free to do whatever you like with it, including but
;;;> not limited to distributing, modifying, and copying.

;;;> Bugs, comments and revisions due to porting can be sent to:
;;;> bug-starlisp@think.com.  Other than to Thinking Machines'
;;;> customers, no promise of support is intended or implied.
;;;>
;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+

;;; Author:  JP Massar.


#+*LISP-HARDWARE
(import '(proclaimed-type proclaimed-ftype proclaimed-declaration-p proclaimed-special-p check-paris-restriction)
	(find-package 'slc)
	)

#+*LISP-HARDWARE
(proclaim '(special slc::*compiler-options* slc::*slc-function-types*))

(defun get-function-type (function)
  (get function 
       #+*LISP-HARDWARE 'slc::function-type 
       #+*LISP-SIMULATOR 'slc-function-type))
(defun set-function-type (function type)
  (setf (get function 
             #+*LISP-HARDWARE 'slc::function-type 
             #+*LISP-SIMULATOR 'slc-function-type) 
        type))
(defun remove-function-type (function)
  (setf (get function 
             #+*LISP-HARDWARE 'slc::function-type 
             #+*LISP-SIMULATOR 'slc-function-type) 
        nil))

(defun get-variable-descriptor (variable)
  (get variable #+*LISP-HARDWARE 'slc::descriptor #+*LISP-SIMULATOR 'slc-descriptor))
(defun set-variable-descriptor (variable value)
  (setf (get variable #+*LISP-HARDWARE 'slc::descriptor #+*LISP-SIMULATOR 'slc-descriptor) value))
(defun remove-variable-descriptor (variable)
  (setf (get variable #+*LISP-HARDWARE 'slc::descriptor #+*LISP-SIMULATOR 'slc-descriptor) nil))

(defun make-pvar-for-compiler (&rest args &key type reference read-only)
  (declare (ignore reference read-only))
  #+*LISP-SIMULATOR
  (declare (ignore args))
  #+*LISP-HARDWARE
  (declare (ignore type))
  #+*LISP-HARDWARE
  (apply 'slc::make-slc-pvar args)
  #+*LISP-SIMULATOR
  type
  )

(defun make-descriptor-for-compiler (&rest args)
  #+*LISP-SIMULATOR
  (declare (ignore args))
  #+*LISP-HARDWARE
  (apply 'slc::make-descriptor args)
  #+*LISP-SIMULATOR
  nil
  )

(defun find-compiler-keyword (keyword)
  #+*LISP-SIMULATOR
  (declare (ignore keyword))
  #+*LISP-HARDWARE
  (find keyword slc::*compiler-options* :key 'slc::option-keyword)
  #+*LISP-SIMULATOR
  (error "I need a function which tells me if keyword is a legal compiler option.")
  )

(defun deal-with-compiler-option-value (option value keyword)
  #+*LISP-SIMULATOR
  (declare (ignore option value keyword))
  #+*LISP-HARDWARE
  (if (eq value :default)
      (set (funcall 'slc::option-variable option) (funcall 'slc::option-default option))
      (if (typep value (funcall 'slc::option-lisp-type option))
	  (set (funcall 'slc::option-variable option) value)
	  (cerror "Ignore declaration." "Invalid value ~S for starlisp compiler option ~S, it should be a ~S."
		  value keyword (funcall 'slc::option-lisp-type option)
		  )))
  #+*LISP-SIMULATOR
  nil
  )

#+*LISP-HARDWARE
(eval-when (compile load eval)
  (setf (symbol-function 'slc::canonical-type) 'canonical-type)
  )

(defun simplify-expression (&rest args)
  #+*LISP-HARDWARE
  (apply 'slc::simplify-expression args)
  #+*LISP-SIMULATOR
  (copy-list args)
  )

(eval-when (compile load eval)
  (defvar *maximum-integer-length* #+*LISP-HARDWARE cm:*maximum-integer-length* #+*LISP-SIMULATOR 128)
  (defvar *maximum-signficand-length* #+*LISP-HARDWARE cm:*maximum-significand-length* #+*LISP-SIMULATOR 96)
  (defvar *maximum-exponent-length* #+*LISP-HARDWARE cm:*maximum-exponent-length* #+*LISP-SIMULATOR 32)
  )

#+*LISP-SIMULATOR
(defvar nbits-per-lisp 4)

#+*LISP-HARDWARE
(eval-when (load eval compile)
  (defun *defun-maybe (function-name arguments)
    `(let* ((old-*temp-pvar-list* *temp-pvar-list*)
	    (place-holder (cm:allocate-stack-field 0)))
       (maybe-return-pvar place-holder (,function-name ,@arguments) old-*temp-pvar-list*)))

  (defun *defun-yes (function-name arguments)
    `(let ((place-holder (cm:allocate-stack-field 0)))
       (values (move-pvar-to-place-holder place-holder (,function-name ,@arguments)))))

  (defun *defun-no (function-name arguments)
    `(let ((*lisp-i::*temp-pvar-list* *lisp-i::*temp-pvar-list*)
	   (*lisp-i::place-holder (cm:allocate-stack-field 0)))
       (prog1 (,function-name ,@arguments)
	      (cm:deallocate-upto-stack-field *lisp-i::place-holder))))

  (defun *defun-type (function)
    (let* ((ftype (proclaimed-ftype function)) (return-type (if ftype (caddr ftype))))
      (if ftype
	  ;; return type must be canonical, (which it is supposed to be
	  (if (or (eq return-type 'pvar)
		  (and (consp return-type) (eq (car return-type) 'pvar)))
	      :yes 
	      (if (and (consp return-type) (member (car return-type) '(satisfies and or not member values)))
		  :dont-know
		  :no))
	  :dont-know)))

  (defun proclaim-*defun-1 (function-name)
    (let ((old-source-file-name (get function-name :source-file-name)))	
      (setf (*lisp-i::get-*defun-function function-name) (make-*defun-function function-name))
      (setf (macro-function function-name)
	    #'(lambda (%form% %environment% &aux (args (cdr %form%)))
		(or (funcall '*lisp-compiler-hook %form% %environment%)
		    (case (*defun-type function-name)
		      (:dont-know (*defun-maybe (make-*defun-function function-name) args))
		      (:no (*defun-no (make-*defun-function function-name) args))
		      (:yes (*defun-yes (make-*defun-function function-name) args))))))
      (setf (get function-name :source-file-name) old-source-file-name))))


#+*LISP-SIMULATOR
(eval-when (load eval compile)
  (defun proclaim-*defun-1 (function-name)
    (when (not (macro-function function-name))
      (let ((old-temp-pvar-list-symbol (gensym "OLD-TEMP-PVAR-LIST-")))
	(setf (macro-function function-name)
	      #'(lambda (form environment)
		  (declare (ignore environment))
		  `(let ((,old-temp-pvar-list-symbol *temp-pvar-list*))
		     (handle-returning-pvar
		       (,(make-*defun-function function-name) ,@(cdr form))
		       ,old-temp-pvar-list-symbol
		       nil
		       )))))))
  )

(defun list-of-slc-function-types () #+*LISP-HARDWARE slc::*slc-function-types* #+*LISP-SIMULATOR nil)

(defun compiler-warn (&rest args)
  #+*LISP-SIMULATOR
  (declare (ignore args))
  #+*LISP-HARDWARE
  (apply 'slc::*warn args)
  #+*LISP-SIMULATOR
  (error "Internal error.  This should never be called inside the simulator.")
  )

#+*LISP-SIMULATOR
(defun expt2 (x) (expt 2 x))
#+*LISP-SIMULATOR
(defun expt2-1- (x) (expt 2 (1- x)))

(defun expt2-symbol () #+*LISP-HARDWARE 'slc::expt2 #+*LISP-SIMULATOR 'expt2)
(defun expt2-1-symbol () #+*LISP-HARDWARE 'slc::expt2-1- #+*LISP-SIMULATOR 'expt2-1-)

(defun portable-pvar-array-element-type (pvar)
  #+*LISP-HARDWARE
  (pvar-array-element-type pvar)
  #+*LISP-SIMULATOR
  (pvar-array-canonical-element-type pvar)
  )

(defun portable-pvar-array-dimensions (pvar)
  (pvar-array-dimensions pvar)
  )

(defun portable-pvar-structure-name (pvar)
  (pvar-structure-name pvar)
  )


(defmacro proclaim-all-*defuns ()
  `(progn
     ,@(mapcan
	 #'(lambda (symbol)
	     (let ((type (get symbol :starlisp-type)))
	       (when (eq type :*defun)
		 `((*proclaim '(*defun ,symbol)))
		 )))
	 *all-external-symbols*
	 )))


(proclaim-all-*defuns)


(defun starlisp-form-not-to-macroexpand (form)
  #+*LISP-SIMULATOR
  (declare (ignore form))
  #+*LISP-HARDWARE
  (eq (car form) 'slc::allocate-temp-pvar)
  #+*LISP-SIMULATOR
  nil
  )


(defun call-starlisp-compiler (form environment)
  #+*LISP-SIMULATOR
  (declare (ignore environment))
  #+*LISP-HARDWARE
  (funcall 'slc::*compile form environment slc::*slc-environment*)
  #+*LISP-SIMULATOR
  form
  )
