

#include <stdio.h>
#include <stdlib.h>

#include "macros.h"

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#include <assert.h>

#include "db_common.h"
#include "db_records.h"

void rec_increfs (int32 rec, int n);
void rec_flush (int32 rec);
int rec_find (int32 rec);
int32 write_refcount_data(void);

/*
 * #define VERBOSE
 * Global variables
 */

static FILE *file_handle = NULL;
static unsigned char refcount[MAX_FILE_SIZE + 1];
static struct record_buffer buffered_loc[MAX_BUFF_NO];
static int num_buffered;
struct first_record header;
static int32 *free_list=NULL;
static int num_free, free_size=0;
static char *db_file_name=NULL;

void
init (void)
{
  int i;

  for (i = 0; i < MAX_FILE_SIZE; i++)
    refcount[i] = 0;

  for (i = 0; i < MAX_BUFF_NO; i++)
    {
      buffered_loc[i].recno = EMPTY;
      buffered_loc[i].dirty = NO;
      bzero (buffered_loc[i].buffer, REC_SIZE);
    }

  num_buffered = 0;
  header.recno_generator = INITIAL_DB_RECORD;

  if (free_list==NULL) {
  	  free_size = INITIAL_FREE_LIST_SIZE; 
	  free_list = (int32 *) (malloc (sizeof (int32) * free_size));
  }
  num_free = 0;

}

void
error (char *s)
{

  fprintf (stderr, "%s\n", s);
  exit (1);
}

void
rec_setrecbuf (int32 rec, unsigned char *stg, int l)
{
  int j, i, k;

#ifdef PARANOID
  for (i = 0; i < MAX_BUFF_NO; i++)
    {
      k = 0;
      for (j = 0; j < MAX_BUFF_NO; j++)
	if (buffered_loc[j].recno == buffered_loc[i].recno)
	  k++;

      assert (k == 1);
    }
#endif

  j = rec_find (rec);
  assert (j >= 0);
  assert (buffered_loc[j].recno == rec);
  bcopy (stg, buffered_loc[j].buffer, l);
  buffered_loc[j].dirty = YES;

}

void
add_to_free_list(int32 rec)
{
  num_free++;
  if (num_free > free_size)
    {
      free_size *= 2;
      free_list = (int32 *) realloc (free_list, free_size * sizeof (int32));
      assert (free_list != NULL);
    }
  free_list[num_free - 1] = rec;
}

void
rec_incref (int32 rec, int n)
{
  int j;
  int i;

#ifdef VERBOSE
  printf ("Incref %d %d \n", rec, n);
#endif
  refcount[rec] += n;
  assert(refcount[rec]>=0);
  if (refcount[rec] == 0)
    {
      add_to_free_list(rec);
      rec_increfs (rec, -1);
      if ((j = rec_find (rec)) >= 0)
	{
	  buffered_loc[j].recno = EMPTY;
	}
    }

}

void
rec_xfref (int32 x, int32 y)
{
  rec_incref (y, 1);
  rec_incref (x, -1);
}

int
rec_find (int32 rec)
{
  int i;

  for (i = 0; i < MAX_BUFF_NO; i++)
    if (buffered_loc[i].recno == rec)
      return i;

  return -1;
}

int
rec_dirtify (int32 rec)
{
  int i;

  for (i = 0; i < MAX_BUFF_NO; i++)
    if (buffered_loc[i].recno == rec)
      {
	buffered_loc[i].dirty = YES;
	return;
      }

  assert (NO);
}

void
set_rec_refcount (int32 rec,int rc)
{
  refcount[rec] = rc;
}

int
rec_refcount (int32 rec)
{

  return refcount[rec];
}

void
rec_flush (int32 rec)
{
  int i, j;

  i = rec_find (rec);
  assert (i >= 0);
  if (buffered_loc[i].dirty == NO)
    return;

#ifdef VERBOSE
  fprintf (stderr, "Flushing record %d \n", rec);
#endif
  fseek (file_handle, (long) ((rec ) * REC_SIZE), SEEK_SET);
  fwrite (buffered_loc[i].buffer, 1, REC_SIZE, file_handle);

  buffered_loc[i].dirty = NO;

}

int
find_free_buffer_pos ()
{
  int i;

  for (i = 0; i < MAX_BUFF_NO; i++)
    if (buffered_loc[i].recno == EMPTY)
      return i;

  /*
   * All records occupied 
   */

  assert (NO);

}

void
force_can_buffer ()
{
  int i;

  for (i = 0; i < MAX_BUFF_NO; i++)
    if (buffered_loc[i].recno == EMPTY)
      return;

  /*
   * All filled up 
   */

  rec_flush (buffered_loc[0].recno);
  buffered_loc[0].recno = EMPTY;

}

unsigned char *
rec_load (int32 rec)
{
  int i, j;

  open_db(NULL);

#ifdef VERBOSE
  fprintf (stderr, "Loading record %d\n", rec);
#endif

  for (i = 0; i < MAX_BUFF_NO; i++)
    if (buffered_loc[i].recno == rec)
      {
	return buffered_loc[i].buffer;
      }
#ifdef VERBOSE
  fprintf (stderr, "Not found!\n");
#endif

  force_can_buffer ();

  i = find_free_buffer_pos ();
  assert (i >= 0);

  fseek (file_handle, (long) ((rec ) * REC_SIZE), SEEK_SET);
  fread (buffered_loc[i].buffer, 1, REC_SIZE, file_handle);

  buffered_loc[i].recno = rec;
  buffered_loc[i].dirty = NO;

  return buffered_loc[i].buffer;

}

int
rec_is_compound (int32 rec)
{

  return is_compound (rec_load (rec));

}

int32
set_type (int32 rec, unsigned char typ)
{
  unsigned char *s;
  int i;

  s = rec_load (rec);
  s[type_byte - 1] = typ;

  i = rec_find (rec);
  assert (i >= 0);
  buffered_loc[i].dirty = YES;

  return rec;

}

void
set_is_compound (int32 rec, int isc)
{
  unsigned char *s;
  int i;

  s = rec_load (rec);
  if ((s[type_byte - 1] >= 128) == isc)
    return;

  if (s[type_byte - 1] >= 128)
    s[type_byte - 1] = s[type_byte - 1] - 128;
  else
    s[type_byte - 1] = s[type_byte - 1] + 128;

  i = rec_find (rec);
  assert (i >= 0);
  buffered_loc[i].dirty = YES;

}

void
rec_increfs (int32 rec, int n)
{
  unsigned char buffer[REC_SIZE];
  unsigned char *s;
  int j, num_children;

#ifdef VERBOSE
  printf ("Increfs %d %d \n", rec, n);
#endif
  bcopy (rec_load (rec), buffer, REC_SIZE);
  s = buffer;
  switch (s[type_byte - 1])
    {
    case string_record:
    case wdoccs_string_record:
    case db_index_node_ncr:
    case wd_index_node_ncr:

      break;
    case bigstr_node_record:
    case bigstr_node_ncr:
      num_children = s[nch_byte - 1];
      for (j = 0; j < num_children; j++)
	{
	  rec_incref (str_to_rec (s + bnr_ch_start + j * 4 - 1), n);
	}
      break;
    case wdoccs_str_node_record:
    case wdoccs_str_node_ncr:
      num_children = s[nch_byte - 1];
      for (j = 0; j < num_children; j++)
	{
	  rec_incref (str_to_rec (s + wo_ch_start + j * 4 - 1), n);
	}
      break;
    case wd_index_node_record:
      num_children = s[nch_byte - 1];
      for (j = 0; j < num_children; j++)
	{
	  rec_incref (str_to_rec (s + wix_ch_start + j * 4 - 1), n);
	}
      break;

    case db_index_node_record:
      num_children = s[nch_byte - 1];
      for (j = 0; j < num_children; j++)
	{
	  rec_incref (str_to_rec (s + dbix_ch_start + j * 4 - 1), n);
	}
	break;

    default:
      printf ("Unrecognized type : %d\n", s[type_byte - 1]);
      assert (NO);
      break;
    }

}

int
num_childr (int32 rec)
{
  return (rec_load (rec)[nch_byte - 1]);

}

void
set_num_childr (int32 rec, unsigned char n)
{
  int i;

  rec_load (rec)[nch_byte - 1] = n;
  i = rec_find (rec);
  buffered_loc[i].dirty = YES;

}

void
flush_all (void)
{
  int i;

  for (i = 0; i < MAX_BUFF_NO; i++)
    if (buffered_loc[i].recno != EMPTY)
      rec_flush (buffered_loc[i].recno);

}

/*
 *  Read routine
 *
 */
void
open_db(char *file_name)
{
int to_read=MAX_FILE_SIZE;
int32 i,j,refrec,nextrec,lastrec;
unsigned char *p,*q;

  if (file_handle == NULL)
    {
      init ();
      if (!file_name) 
        {
#ifdef MACINTOSH
     	  db_file_name =  malloc(strlen(db_default_file_name)+1);
     	  strcpy(db_file_name, db_default_file_name);
#else
          db_file_name = (char*)strdup(db_default_file_name);
#endif
        }
      else
	{
#ifdef MACINTOSH
   	  db_file_name = malloc(strlen(file_name)+1);
      strcpy(db_file_name, file_name);
#else
      db_file_name = (char*)strdup(file_name);
#endif

	}
      if ((file_handle = fopen (db_file_name, "rwb+")) == NULL)
	{
	  file_handle = fopen (db_file_name, "wb");

	  fseek (file_handle, 0L, SEEK_SET);
	  fwrite (&header, 1, sizeof (struct first_record), file_handle);

	  fclose (file_handle);
          file_handle = fopen (db_file_name, "rwb+");
	}
      else 
	{
		/*
		 * File is present. Read the structures from disk 
		 */

	  fseek (file_handle, 0L, SEEK_SET);
	  fread (&header, 1, sizeof (struct first_record), file_handle);
#ifdef VERBOSE
      	  printf("Reading database index structures\n");
	  printf("header : %d %d \n",header.recno_generator,header.refcount_ptr);
#endif

          if (rec_generator()>INITIAL_DB_RECORD) 
	    {
              i=0; /* refcount[] to read */
              refrec=header.refcount_ptr;
	      lastrec=0;
              while (to_read>0) 
	        {
                  p=q=rec_load(refrec);
		  if (refrec>0) lastrec=refrec;
		  if (refrec<=header.recno_generator)
	            add_to_free_list(refrec);
              	  nextrec=str_to_rec(p);
	          p=q+4;
	          for (j=i;j<i+min((REC_SIZE-4),to_read);j++) 
		    {
 	  	      set_rec_refcount(j,(int)*p++);
                    }
	  	  i=i+REC_SIZE-4;
          	  refrec= nextrec;
	          to_read -= REC_SIZE-4;
         
            } 

	    /*
	     * The records used to store the refcount list will be free
	     */

	    if (lastrec>(header.recno_generator))
              for (i=header.recno_generator+1;i<=lastrec;i++) set_rec_refcount(i,0);

	    /*
	     * Also, free the elements in the free_list 
	     */

	    for (i = 0; i < num_free; i++)
              set_rec_refcount(free_list[i],0);


#ifdef VERBOSE
	    for (i=0;i<128;i++) printf("%d ",rec_refcount(i));
	    printf("\n");
#endif

          } 
	}

    }

}

/*
 * Close routine
 */

void
close_db(char *param)
{
int i;
int32 save_recno;

  if (file_handle!=NULL)  {
	  flush_all();

          save_recno = header.recno_generator;
	  header.refcount_ptr = write_refcount_data();
	  flush_all();
          header.recno_generator=save_recno;
	  bcopy(param,header.param,32);
	  fseek (file_handle, 0L, SEEK_SET);
	  fwrite (&header, 1, sizeof (struct first_record), file_handle);
	  fclose(file_handle);
  	  file_handle=NULL;
  }


}
/*
 * -- creation routine
 */

int32
new_rec ()
{
  int32 rec;
  int i;

  open_db(NULL);

  if (num_free > 0)
    {
      rec = free_list[num_free - 1];
      num_free--;

    }
  else
    {
      rec = ++header.recno_generator;
      if (header.recno_generator > MAX_FILE_SIZE)
	{
	  error ("***** DISK_RECORDS CLASS - SPACE EXHAUSTED *****");

	}
    }

  force_can_buffer ();
  i = find_free_buffer_pos ();
  assert (i >= 0);
  buffered_loc[i].recno = rec;
  buffered_loc[i].dirty = YES;
  bzero (&buffered_loc[i].buffer, REC_SIZE);

#ifdef VERBOSE
  fprintf (stderr, "Created record %d...\n", rec);
#endif

  refcount[rec] = 1;
  return rec;
}

int
int_from_bytes (unsigned char *s)
{
  int i;

  return (*s) * 256 + (*(s + 1));

}

int
sr_length (int32 rec)
{
  unsigned char *s;

  s = rec_load (rec);
  return int_from_bytes (&s[sr_ncr_1 - 1]);

}

void
sr_set_slice (int32 rec, int i, int j, unsigned char *stg, int inplen)
{
  unsigned char *s;
  int len, erased, inserted;

  if ((i < 1) || (j < i - 1))
    error ("Illegal slice boundaries ");

  s = rec_load (rec);
  len = int_from_bytes (&s[sr_ncr_1] - 1);
  erased = j - i + 1;
  inserted = inplen;

  len += (inserted - erased);
  if (len + 2 > REC_SIZE)
    error ("Record boundary overstepped");

  bytes_from_int (&s[sr_ncr_1 - 1], len);

  bcopy (stg, s + (sr_char_start + i - 1 - 1), inplen);
  j = rec_find (rec);
  buffered_loc[j].dirty = YES;

}

unsigned char *
sr_slice (int32 rec, int i, int j)
{
  unsigned char *s, *t;
  int len, k, m;

  s = rec_load (rec);
  len = int_from_bytes (&s[sr_ncr_1]);

  m = sr_char_start + i - 1;
  k = sr_char_start + j - 1;
  if (k > REC_SIZE)
    k = REC_SIZE;

  t = (char *) (malloc (k - m + 2));
  bcopy (s + m, t, (k - m + 1));
  t[k - m + 1] = '\0';
  return t;

}

int
wo_length (int32 rec)
{
  unsigned char *s;

  s = rec_load (rec);
  return (int)(*(s+wo_nr_1 - 1));

}

int32
rec_abs_copy (int32 rec)
{
  unsigned char buffer[REC_SIZE];
  int32 newrec;

  bcopy (rec_load (rec), buffer, REC_SIZE);
  newrec = new_rec ();
  rec_setrecbuf (newrec, buffer, REC_SIZE);
  return newrec;

}

int32
rec_copy (int32 rec)
{

  if (refcount[rec] == 1)
    return rec;
  return rec_abs_copy (rec);

}

#ifdef STANDALONE

main ()
{
  int i, l;
  char spam[128];
  int32 recs[11];
  unsigned char *s;

  /*
   * Initialize the list of records 
   */

  for (i = 0; i < 11; i++)
    {
      recs[i] = new_rec ();
      printf ("Record %d is %d\n", i, recs[i]);
    }

  rec_load (1);
  flush_all ();

  for (i = 0; i < 11; i++)
    {
      printf ("Length of string %d is %d\n", i, sr_length (recs[i]));
    }

  for (i = 0; i < 11; i++)
    {
      sprintf (spam, "Hello World%d", i);
      sr_set_slice (recs[i], 1, 0, spam, strlen (spam));
    }

  for (i = 0; i < 11; i++)
    {
      l = sr_length (recs[i]);
      s = sr_slice (recs[i], 1, l);
      printf ("Length of string %d (%s) is %d\n", i, s, l);
      free (s);
    }

  flush_all ();

}

#endif

int
rec_check_mem (void)
{
  int i;

#ifdef VERBOSE
  printf ("Free List (F:%d,O:%d) ", num_free, header.recno_generator - num_free);
  for (i = 0; i < num_free; i++)
    printf (" %d", free_list[i]);
  printf ("\n");
#endif

  return header.recno_generator - num_free;

}

int
rec_generator (void)
{

  return header.recno_generator;

}

int32 
write_refcount_data(void)
{
int32 refrec=0;
int32 nextrec,currec;
unsigned char *p,*q;
int to_write=MAX_FILE_SIZE;
int32 i,j;

   /*
    * If the database is not empty, we will write a list of records
    * with the refcount structure. The free list can be obtained from
    * it when we read back the list.
    * The fist 4 bytes in the record are reserved for a pointer to the
    * the next record in the refcount list.
    */

   if (rec_generator()>INITIAL_DB_RECORD) {

      i=0; /* refcount[] to write */
      refrec=currec=new_rec();
      while (to_write>0) {
	  if (to_write>(REC_SIZE-4)) { /* Space for the 'next' counter */
              nextrec=new_rec();
          } else {
	      nextrec=0;
          }
          p=q=rec_load(currec);
          rec_to_str(p,nextrec);
	  p=q+4;
	  for (j=i;j<i+min((REC_SIZE-4),to_write);j++) {
 	  	*p++ = (unsigned char)rec_refcount(j);
          }
	  i=i+REC_SIZE-4;
          rec_dirtify(currec);
          currec=nextrec;  
	  to_write -= REC_SIZE-4;
         
      } 
   } 
   /*
    * Otherwise we will return 0
    */
   return refrec;
}

char*
db_get_param(void)
{
   return &header.param;

}
