/*
 *	Event Native Package
 */

/* SETL2 system header files */

#include "macros.h"
#include "stubs_utilities.h"
#include "str_list.h"

typedef struct setl_event {		/* Native Object Structure           */
	int32 use_count;	/* Reference Count                   */
	int32 type;			/* Encodes Type and Subtype          */

	short					stopped:1;
	plugin_item_ptr_type	setl_env;
	StrListPtr				sources_list; 

} setl_event, *setl_event_ptr;

typedef struct setl_source {

	int32 source_id;	

	int32 (*source_callback)(SETL_SYSTEM_PROTO specifier *callback);
	short has_setl_callback;
	specifier setl_callback;
} setl_source, *setl_source_ptr;


static int32 avail_source_id;	/* next source id available */		
static int32 event_type;		/* Store the type assigned to us by  */


/*
 *	Prototypes
 */

int event_seeker(StrListElem *elem, int32 *event_id);
int execute_single_event(setl_event_ptr event_instance);

int32 EVENT_CREATE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
int32 EVENT_SOURCE_ADD(SETL_SYSTEM_PROTO int argc, specifier *argv,	specifier *target);
int32 EVENT_MAINLOOP(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
int32 EVENT_SINGLE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
int32 EVENT_BREAKLOOP(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
int32 EVENT_REMOVE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);

/*
 *	Auxiliary Procedures
 */

/*
 *	Compare if the element to find 
 */

int event_seeker(StrListElem *elem, int32 *event_id)
{
	setl_source *source_elem = (setl_source_ptr)elem->str;
	return *event_id == source_elem->source_id;
}

/*
 *	int execute_single_event(setl_event_ptr event_instance)
 *
 *		execute all events for this event objects and return the number of handled 
 *		events. If an event is causing the exit from the mainloop it will return a value
 *		less than zero.
 */
int execute_single_event(setl_event_ptr event_instance)
{
	short handled = 0;
	plugin_item_ptr_type plugin_instance = event_instance->setl_env;

	StrListPtr sources_list = event_instance->sources_list;

	StrListElemPtr sources_list_it;

	if (sources_list) {
		sources_list_it = first_element_str_list(sources_list);
		while (sources_list_it) {
			setl_source_ptr source = (setl_source_ptr)sources_list_it->str;
			char type_returned;

			int32 exit_condition;
			
			if (source->source_callback) {

				exit_condition = source->source_callback(
					event_instance->setl_env, 
					source->has_setl_callback ? &source->setl_callback : NULL);

				if (exit_condition < 0)
					return exit_condition;	/* exit condition from handler, exit from here */
			
				handled++;
			}
			
			if (event_instance->stopped) {
				handled = -1;
				break;
			}
			
			sources_list_it = next_element_str_list(sources_list_it);
		}
	}

	return handled;	/* number of events handled in this event session */
}

/*
 *	Native Procedures
 */

static void internal_destructor(setl_event_ptr event_instance)
{
	if (event_instance != NULL) {
		destroy_str_list(&event_instance->sources_list);
		free(event_instance);
	}
}

int32 EVENT_CREATE(
	SETL_SYSTEM_PROTO
	int argc,                           /* number of arguments passed        */
	specifier *argv,                    /* argument vector (two here)        */
	specifier *target)                  /* return value                      */
{
	setl_event_ptr event_instance = NULL;
	
	event_instance = calloc(sizeof(setl_event), 1);
	if (!event_instance)
		goto end_event_create;
	
	event_instance->sources_list = create_str_list();
	if (!event_instance->sources_list) {
		internal_destructor(event_instance); event_instance = NULL;
		goto end_event_create;
	}
	
	event_instance->type = event_type;
	event_instance->use_count = 1;
	event_instance->stopped = 0;
	
		/* store information for main callback */
	event_instance->setl_env = plugin_instance;
	
end_event_create:

	if (!event_instance) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)event_instance;
}


int32 EVENT_SOURCE_ADD(
	SETL_SYSTEM_PROTO
	int argc,                           /* number of arguments passed        */
	specifier *argv,                    /* argument vector (two here)        */
	specifier *target)                  /* return value                      */
{
	specifier *proc_spec;
	setl_event_ptr event_instance;
	setl_source source;
	
	char type_returned;
	result_type source_callback_res;
	int32 source_callback_long;
	
	check_type_arg(SETL_SYSTEM argv, 0, "source_add", event_type);
	check_arg(SETL_SYSTEM argv, 1, ft_proc, "procedure", "source_add");
	check_arg_or_om(SETL_SYSTEM argv, 2, ft_proc, "procedure", "source_add");

	event_instance = (setl_event_ptr)(argv[0].sp_val.sp_opaque_ptr);
	
	source_callback_res = callback_caller(SETL_SYSTEM &type_returned, &argv[1], "");
	if (type_returned == 'L') source_callback_long = source_callback_res.long_value;
		else if (type_returned == 'I') source_callback_long = source_callback_res.short_value;
			else abend(SETL_SYSTEM "Unexpected type from callback");

	source.source_id = avail_source_id++;
	source.source_callback = (int32 (*)()) source_callback_long;

	if (argv[2].sp_form == ft_omega)
		source.has_setl_callback = 0;
	else {
		source.has_setl_callback = 1;
		memcpy(&source.setl_callback, &argv[2], sizeof(specifier));
	}
	
	add_element_str_list(event_instance->sources_list, (char *)&source, sizeof(source));	

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = source.source_id;
}

int32 EVENT_MAINLOOP(
	SETL_SYSTEM_PROTO
	int argc,                           /* number of arguments passed        */
	specifier *argv,                    /* argument vector (two here)        */
	specifier *target)                  /* return value                      */
{
	int ret_value = 1;
	setl_event_ptr event_instance;

	check_type_arg(SETL_SYSTEM argv, 0, "mainloop", event_type);
	event_instance = (setl_event_ptr)(argv[0].sp_val.sp_opaque_ptr);

	event_instance->stopped = 0;	
	while ((ret_value = execute_single_event(event_instance)) > 0);	/* loop untile events */

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

int32 EVENT_SINGLE(
	SETL_SYSTEM_PROTO
	int argc,                           /* number of arguments passed        */
	specifier *argv,                    /* argument vector (two here)        */
	specifier *target)                  /* return value                      */
{
	int ret_value;
	setl_event_ptr event_instance;
	
	check_type_arg(SETL_SYSTEM argv, 0, "single", event_type);
	event_instance = (setl_event_ptr)(argv[0].sp_val.sp_opaque_ptr);

	ret_value = execute_single_event(event_instance);

	if (!ret_value) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = ret_value;
}

int32 EVENT_BREAKLOOP(
	SETL_SYSTEM_PROTO
	int argc,                           /* number of arguments passed        */
	specifier *argv,                    /* argument vector (two here)        */
	specifier *target)                  /* return value                      */
{
	setl_event_ptr event_instance;

	check_type_arg(SETL_SYSTEM argv, 0, "mainloop", event_type);
	event_instance = (setl_event_ptr)(argv[0].sp_val.sp_opaque_ptr);

	event_instance->stopped = 1;

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

int32 EVENT_REMOVE(
	SETL_SYSTEM_PROTO
	int argc,                           /* number of arguments passed        */
	specifier *argv,                    /* argument vector (two here)        */
	specifier *target)                  /* return value                      */
{
	int32 event_id;
	int ret;
	setl_event_ptr event_instance;

	check_type_arg(SETL_SYSTEM argv, 0, "remove", event_type);
	check_arg(SETL_SYSTEM argv, 1, ft_short, "event_id (short)", "remove");

	event_instance = (setl_event_ptr)(argv[0].sp_val.sp_opaque_ptr);
	event_id = (argv[1].sp_val.sp_short_value);
	
	ret = delete_element_str_list(event_instance->sources_list, (char *)&event_id, (int (*)(StrListElemPtr, char *))event_seeker);

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = ret;
}

