/**************************************************************************************
/* Filename:	gr_bitplns.c
/*		Copyright  1998-99 Giuseppe Di Mauro. All rights reserved.
/*
/* Description: common general graphical routines for both sparse 
/*				and dense byteplanes images
/*
/***************************************************************************************/
 
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "jpeglib.h"
#ifndef TIFF_DISABLED
#include "tiffio.h"
#endif

#include "gr_bitplns.h"
#include "gr_dense_bitplns.h"
#include "gr_sparse_bitplns.h"
#include "gr_interpolation.h"
#include "gr_errors.h"
#include "gr_ops.h"
#include "gr_utils.h"

	/*
	 *	Not exported functions
	 */

static unsigned char	check_rect_mimage(mimage_ptr mimg, image_rect_ptr rect, int threshold);
static int				check_hline_mimage(mimage_ptr mimg, int y, int threshold);
static int				check_vline_mimage(mimage_ptr mimg, int x, int threshold);
static void				fast_fset_mimage(mimage_ptr mimg, fmpixel fvalue);
static void				fast_set_mimage(mimage_ptr mimg, mpixel value);
static mimage_ptr		fast_scale_mimage(mimage_ptr mimg, image_dimension_ptr new_dim_ptr);
static mimage_ptr		create_empty_clone(mimage_ptr mimg);

	/* 
	 *	Global Variables 
	 */

int32			gr_type;		/* used to interface with host language or interpeter */
mimg_err_no_t	mimg_err_no;	/* keep last error condition (not extensively used yet) */

	/* 
	 *	Implementation 
	 */

/**************************************************************************************
/*	Function:		mimg_lib_init
/*	Description:	perform basic initialization for the grlib
/*
/*	Parameters:
/*		none
/*
/*	Result:
/*		always 1
/*
/***************************************************************************************/

int mimg_lib_init()
{
#ifndef TIFF_DISABLED
	TIFFSetErrorHandler(NULL);	/* Remove the Tiff Library Errors handler */
#endif

	return 1; 	/* everything went ok */
}

/**************************************************************************************
/*	Function:		create_planes_array
/*	Description:	create an array of planes with the requested features
/*
/*	Parameters:
/*		<- width			width of the planes
/*		<- height			height of the planes
/*		<- kind				kind of the planes (float or discrete)
/*		<- planes_number	number of planes to be generated
/*		<- density			density of the planes (sparse or dense)
/*
/*	Result:
/*		not NULL			array of planes requested
/*		NULL				out of memory error
/*
/***************************************************************************************/

byteplane_ptr *create_planes_array(int width, int height, char kind, int planes_number, unsigned char density)
{
	int i;
	byteplane_ptr *planes_array;
	byteplane_ptr ptr;
	
	planes_array = (byteplane_ptr *) calloc(sizeof(void *) * planes_number, 1);
	if (!planes_array)
		return NULL;
		
	for (i=0; i<planes_number; i++) {
		ptr =  create_byteplane(width, height, kind, density);
		if (ptr)
			planes_array[i] = ptr;
		else {
			destroy_planes_array((void **)planes_array, kind, i, density);	/* in this way we actually
							                                           destroy the built arrays */
			return NULL;
		}
	}
		
	return planes_array;
}

/**************************************************************************************
/*	Function:		destroy_planes_array
/*	Description:	destroy an array of planes and release its memory
/*
/*	Parameters:
/*		<- planes_array		array of planes to destroy
/*		<- kind				kind of planes (float or discrete)
/*		<- planes_number	number of planes
/*		<- density			density of planes (sparse or dense)
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void destroy_planes_array(void **planes_array, char kind, int planes_number, unsigned char density)
{
	int i;

#ifdef __DEBUG__
	assert(planes_array != NULL);
#endif

	if (planes_array) {				/* check if planes_array is valid */
		i = planes_number-1;
		while (i >= 0) {			/* loop over the planes */
			if (planes_array[i])	/* and destroy the valid ones */
				destroy_byteplane(planes_array[i], kind, density);
			i--;
		}
		free(planes_array);
	}
}

/**************************************************************************************
/*	Function:		get_plane_buffer
/*	Description:	return a plane buffer
/*	Note:			this function return a void pointer and could be used
/*					for both the discrete and float buffers with no problem.
/*	Parameters:
/*		<- plane		the plane
/*		<- kind			kind of the plane
/*
/*	Result:
/*		a pointer to the buffer
/*
/***************************************************************************************/

void *get_plane_buffer(void *plane, int kind)
{
#ifdef __MWERKS__
 #pragma unused(kind)
#endif
	return ((byteplane_ptr)plane)->buffer;
}

/**************************************************************************************
/*	Function:		create_byteplane
/*	Description:	create a single byteplane
/*
/*	Parameters:
/*		<- width	width of the plane
/*		<- height	height of the plane
/*		<- kind		kind of the plane
/*		<- density	density of the plane (sparse or dense)
/*
/*	Result:
/*		not NULL	a pointer to the buffer
/*		NULL		out of memory
/*
/***************************************************************************************/

void *create_byteplane(int width, int height, char kind, unsigned char density)
{
	if (density == DENSE_IMAGE)
		return create_dense_byteplane(width, height, kind);
	else
		return create_sparse_byteplane(width, height, kind);
}

/**************************************************************************************
/*	Function:		destroy_byteplane
/*	Description:	destroy a single byteplane
/*
/*	Parameters:
/*		<- byteplane	a pointer to the plane
/*		<- kind			kind of the plane
/*		<- density		density of the plane (sparse or dense)
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void destroy_byteplane(void *byteplane, char kind, unsigned char density)
{
	if (density == DENSE_IMAGE)
		destroy_dense_byteplane(byteplane, kind);
	else
		destroy_sparse_byteplane(byteplane, kind);
	
}

/**************************************************************************************
/*	Function:		get_byteplane_from_mimage
/*	Description:	return a byteplane at a certain index of an image
/*
/*	Parameters:
/*		<- mimg			pointer to the image
/*		<- plane_index	index of the plane
/*
/*	Result:
/*		a pointer to the plane
/*
/***************************************************************************************/

byteplane_ptr get_byteplane_from_mimage(mimage_ptr mimg, char plane_index)
{
#ifdef __DEBUG__
	assert (plane_index < mimg->comp);
#endif
	return mimg->planes_array[plane_index];
}

/**************************************************************************************
/*	Function:		get_fbyteplane_from_mimage
/*	Description:	return a float byteplane at a certain index of an image
/*
/*	Parameters:
/*		<- mimg			pointer to the image
/*		<- plane_index	index of the plane
/*
/*	Result:
/*		a pointer to the float plane
/*
/***************************************************************************************/

fbyteplane_ptr get_fbyteplane_from_mimage(mimage_ptr mimg, char plane_index)
{
#ifdef __DEBUG__
	assert (plane_index < mimg->comp);
#endif
	return (fbyteplane_ptr)mimg->planes_array[plane_index];
}

/**************************************************************************************
/*	Function:		fast_zero_mimage
/*	Description:	blank out an image
/*
/*	Parameters:
/*		<- mimg			pointer to the image
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void fast_zero_mimage(mimage_ptr mimg)
{
	if (mimg->kind == FLOAT_IMAGE) 
		fast_fset_mimage(mimg, 0.0f);
	else
		fast_set_mimage(mimg, 0);
}

/**************************************************************************************
/*	Function:		clone_mimage
/*	Description:	clone an image
/*
/*	Parameters:
/*		<- mimg		pointer to the image
/*
/*	Result:
/*		not NULL	a pointer to the new image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr clone_mimage(mimage_ptr mimg)
{
	if (mimg->density == SPARSE_IMAGE)
		return clone_mimage_sparse(mimg);
	else if (mimg->density == DENSE_IMAGE)
		return clone_mimage_dense(mimg);
	
	return NULL;
}

/**************************************************************************************
/*	Function:		create_mimage
/*	Description:	create an image of the requested features
/*
/*	Parameters:
/*		<- width			width of the image
/*		<- height			height of the image
/*		<- planes_number	number of planes
/*		<- kind				kind of image (float or discrete)
/*		<- density			density of the image (sparse or dense)
/*
/*	Result:
/*		not NULL	a pointer to the new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr create_mimage(int width, int height, int planes_number, char kind, unsigned char density)
{
		/* allocate memory for the image */
	mimage_ptr mimg = (mimage_ptr)malloc(sizeof(mimage));
	if (!mimg)
		return NULL;

		/* allocate memory for the planes array */
	mimg->planes_array = create_planes_array(width, height, kind, planes_number, density);
	if (!mimg->planes_array) {
		free (mimg);
		return NULL;
	}
			
	mimg->type      = gr_type;			/* set the type (used for setl packages) */
	mimg->use_count = 1;				/* set the use_count to 1 (used for setl packages) */
	mimg->comp      = planes_number;	/* set the number of planes */
	mimg->kind      = kind;				/* set the kind of image */
	mimg->density	= density;			/* set the density */

	/* 
	 *	Note: this feature is not supported and will be removed soon
	 */
	mimg->upperleft_x = 0;		/* set the upperleft corner */
	mimg->upperleft_y = 0;

		/* 
		 *	a sparse byteplane at the very beginning 
		 *	is empty (completely sparse) 
		 */

	mimg->data_length = 0;			/* set row_start data length array to zero */
	mimg->row_start = NULL;			/* no row_start data array */
	mimg->section_length = 0;		/* set row_section data length array to zero */
	mimg->row_section = NULL;		/* no row_section data array */
	mimg->buffer_length = 0;		/* set buffer length to zero */

	return mimg;	
}

/**************************************************************************************
/*	Function:		destroy_mimage
/*	Description:	destroy an image and release its memory
/*
/*	Parameters:
/*		<- mimg		pointer to the image
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void destroy_mimage(mimage_ptr mimg)
{
	if (!mimg) return;	/* check for valid image */
	
		/* destroy the planes data */
	destroy_planes_array((void **)mimg->planes_array, mimg->kind, mimg->comp, mimg->density);

		/* and if the image is sparse, destroy the sparse data too */	
	if (IS_SPARSE(mimg)) 
	{
		SMART_FREE(mimg->row_start);
		SMART_FREE(mimg->row_section);
	}

		/* release memory used for the structure */
	free(mimg);
}

/**************************************************************************************
/*	Function:		rawline_in_planes
/*	Description:	store an interleaved rawline representation for a scanline
/*					in planes of an image
/*
/*	Parameters:
/*		<- rawline			pointer to the rawline
/*		<- planes			pointer to an array of planes
/*		<- line_index		index of the scanline
/*		<- line_width		width of line
/*		<- planes_number	number of planes
/*
/*	Result:
/*		always 0
/*
/***************************************************************************************/

int rawline_in_planes(char *rawline, byteplane_ptr *planes, int line_index, int line_width, int planes_number)
{
	int i, c;
	size_t plane_offset;
	mpixel_ptr ptr;
	char byte;
	
	plane_offset = line_index * (unsigned)line_width;	/*
														 * calculate the offset in pixels of the scanline
														 * from the upperleft corner of the picture 
														 */

	for (i=0; i<line_width; i++, plane_offset++)		/* loop over the pixels in the line */
		for (c=0; c<planes_number; c++) {				/* loop over the color components */
			byte = *(rawline + i * planes_number + c);	/* get the data from the rawline */
			ptr = planes[c]->buffer + plane_offset;		/* calculate the destination */
			*ptr = (mpixel)byte;								/* and store in the correspondend plane */
		}

	return 0;			
}

/**************************************************************************************
/*	Function:		fplanes_in_rawline
/*	Description:	store the planes of a float image in a rawline
/*
/*	Parameters:
/*		<- rawline			pointer to the rawline
/*		<- planes			pointer to an array of planes
/*		<- line_index		index of the scanline in the image
/*		<- line_width		width of line
/*		<- planes_number	number of planes
/*		<- planes_to_write	planes actually to be written
/*		<- reverse			to reverse the data in the rawline
/*
/*	Result:
/*		always 0
/*
/***************************************************************************************/

int fplanes_in_rawline(char *rawline, fbyteplane_ptr *planes, int line_index, int line_width, int planes_number, int planes_to_write, short reverse)
{
	int i, c;
	size_t plane_offset;
	char *ptr;
	char byte;
	
	plane_offset = line_index * (unsigned)line_width;	/*
														 * calculate the offset in pixels of the scanline
														 * from the upperleft corner of the plane 
														 */
	
	for (i=0; i<line_width; i++, plane_offset++)		/* loop over the pixels in the line */
		for (c=0; c<planes_to_write; c++) {				/* loop over the color components */	

 					/* if the component is greater than the 
 					   number of planes for this picture */
 			if (c > (planes_number-1))
				byte = 0;                						/* set a default value */
			else	                     						
				byte = (mpixel)planes[c]->buffer[plane_offset];	/* else get the data from the right plane */
				
			ptr = rawline + i * planes_to_write + c;	/* calculate the destination pointer */
			if (reverse)									
				*ptr = 255-byte;						/* if reverse is requested, get the reversed pixel */
			else
				*ptr = byte;							/* if not reverse, get the pixel */
		}

	return 0;
}

/**************************************************************************************
/*	Function:		planes_in_rawline
/*	Description:	store the planes of a discrete image in a rawline
/*
/*	Parameters:
/*		<- rawline			pointer to the rawline
/*		<- planes			pointer to an array of planes
/*		<- line_index		index of the scanline in the image
/*		<- line_width		width of line
/*		<- planes_number	number of planes
/*		<- planes_to_write	planes actually to be written
/*		<- reverse			to reverse the data in the rawline
/*
/*	Result:
/*		always 0
/*
/***************************************************************************************/

int planes_in_rawline(char *rawline, byteplane_ptr *planes, int line_index, int line_width, int planes_number, int planes_to_write, short reverse)
{
	int i, c;
	size_t plane_offset;
	char *ptr;
	char byte;
	
	plane_offset = line_index * line_width;			/*
													 * calculate the offset in pixels of the scanline
													 * from the upperleft corner of the plane 
													 */
	
	for (i=0; i<line_width; i++, plane_offset++)		/* loop over the pixels in the line */
		for (c=0; c<planes_to_write; c++) {				/* loop over the color components */	

 					/* if the component is greater than the
 					   number of planes for this picture */
			if (c > (planes_number-1))
				byte = 0;                					/* set a default value */
			else	                    
				byte = *(planes[c]->buffer + plane_offset);	/* else get the data from the right plane */
				
			ptr = rawline + i * planes_to_write + c;	/* calculate the destination pointer */
			if (reverse)
				*ptr = 255-byte;						/* if reverse is requested, get the reversed pixel */
			else 
				*ptr = byte;							/* if not reverse, get the pixel */
		}

	return 0;
}

#ifndef TIFF_DISABLED

/**************************************************************************************
/*	Function:		read_tiff_mimage
/*	Description:	read a tiff image from the file system and return an image
/*
/*	Parameters:
/*		<- fname	pathname of the tiff file in the filesystem
/*
/*	Result:
/*		not NULL	pointer to the image
/*		NULL		if an error condition
/*
/***************************************************************************************/

mimage_ptr read_tiff_mimage(char *fname)
{ 
	int line_index = 0;

	uint16 config; 
	uint16 nsamples;
	uint32 w, h, row;
	uint32 *raster_line;
	tsize_t scanline_size;

	byteplane_ptr *planes;
	mimage_ptr mimg = NULL;
  
	TIFF *in_file;

 		/* remove tiff warning messages display */
 	TIFFSetWarningHandler(NULL);

		/* open the tiff file */
	if ((in_file = TIFFOpen(fname, "r")) == NULL)
		return NULL;

		/* get some information from the tags of the tiff file */  
	TIFFGetField(in_file, TIFFTAG_IMAGEWIDTH, &w);
	TIFFGetField(in_file, TIFFTAG_IMAGELENGTH, &h);
	TIFFGetField(in_file, TIFFTAG_PLANARCONFIG, &config);
	TIFFGetField(in_file, TIFFTAG_SAMPLESPERPIXEL, &nsamples);

		/* get the size of a scanline */
	scanline_size = TIFFScanlineSize(in_file);
	
		/* allocate memory to store the scanline */
	raster_line = (uint32 *) malloc(scanline_size);
	if (!raster_line)
    	goto read_abort;
  
		/* create an image with requested features */
	mimg = create_mimage(w, h, nsamples, DISCRETE_IMAGE, DENSE_IMAGE);
	if (!mimg)
		goto read_abort;

		/* get the number of planes */
	planes = (byteplane_ptr *)mimg->planes_array;

		/* if the file format is the one that we require */
	if (config == PLANARCONFIG_CONTIG) {
			
			/* read scanlines, one at time and store in the image */
		for (row = 0; row < h; row++) {
			TIFFReadScanline(in_file, raster_line, row, 0);
			rawline_in_planes((char *)raster_line, planes, row, w, nsamples);
		}
	}

read_abort:

		/* 
		 * clean-up memory and close files used 
		 * during the reading of the tiff image 
		 */

	if (raster_line)
		free(raster_line);

	TIFFClose(in_file);
  
	return mimg;
}

/**************************************************************************************
/*	Function:		write_tiff_mimage
/*	Description:	write a tiff image to the file system
/*
/*	Parameters:
/*		<- fname	pathname of the tiff file in the filesystem
/*		<- mimg		pointer to the image
/*
/*	Result:
/*		0 			ok
/*		else		error number
/*
/***************************************************************************************/

int write_tiff_mimage(char *fname, mimage_ptr mimg)
{
	int err;
	
	if (mimg->density == SPARSE_IMAGE) {	/* check if the image is sparse*/

		mimg = clone_mimage(mimg);					/* clone the image */
		if (!mimg)
			return -7;
			
		err = to_dense_mimage(&mimg);				/* convert to dense */
		if (err)
			return -6;
		
		err = write_tiff_mimage_dense(fname, mimg);	/* and write the obtained dense image */
		
		destroy_mimage(mimg);						/* destroy the cloned image */
		
		return err;
	} 
	else
		return write_tiff_mimage_dense(fname, mimg); /* write the dense image */
}

/**************************************************************************************
/*	Function:		write_tiff_mimage_dense
/*	Description:	write a dense image to the file system as tiff
/*
/*	Parameters:
/*		<- fname	pathname of the tiff file in the filesystem
/*		<- mimg		pointer to the dense image
/*
/*	Result:
/*		0 			ok
/*		else		error number
/*
/***************************************************************************************/

int write_tiff_mimage_dense(char *fname, mimage_ptr mimg)
{
	int res = 0;
	int height, width, comp, kind;
	int row;
	int planes_to_write;

	short reverse;

	tsize_t scanline_size;
	uint32 *raster_line;
	TIFF *out_file;

 		/* remove tiff warning messages display */
	TIFFSetWarningHandler(NULL);

		/* obtain some useful information from the image */
	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);

#define ALPHA_SAVE
#ifndef ALPHA_SAVE
	if (comp > 3) comp = 3;			/* no more than three planes are saved */
#else
	if (comp > 4) comp = 4;			/* no more than four planes are saved */
#endif

		/* number of palnes to write */
	planes_to_write = comp;
	if (planes_to_write == 2) planes_to_write = 3;	/* if they are 2 then write 3 with a phantom one */
		
		/* open the file to write */
	if ((out_file = TIFFOpen(fname, "w")) == NULL) 
		return -4;

		/* set the tiff tags */
	TIFFSetField(out_file, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
	TIFFSetField(out_file, TIFFTAG_SAMPLESPERPIXEL, planes_to_write);
	TIFFSetField(out_file, TIFFTAG_IMAGEWIDTH, width);
	TIFFSetField(out_file, TIFFTAG_BITSPERSAMPLE, 8);
#ifdef ALPHA_SAVE
	if (planes_to_write == 4)
		TIFFSetField(out_file, TIFFTAG_EXTRASAMPLES, EXTRASAMPLE_ASSOCALPHA);
#endif
	if (planes_to_write == 1)
		TIFFSetField(out_file, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK);
	else
		TIFFSetField(out_file, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);

		/* set the size of the scanline */
	scanline_size = planes_to_write * width;
		/* allocate memory for the raster_line to write */
	raster_line = (uint32 *) malloc(scanline_size);

	if (!raster_line) {
		res = -5;
		goto abort_write_tiff;
	}
	
		/* set reverse save to false */
	reverse = FALSE;

		/* if the image is float */
	if (kind == FLOAT_IMAGE) {
			/* loop through the rows, convert to rawline and write to disk*/
		for (row = 0; row < height; row++) {
			fplanes_in_rawline((char *)raster_line, (fbyteplane_ptr *)mimg->planes_array, row, width, comp, planes_to_write, reverse);
			TIFFWriteScanline(out_file, raster_line, row, 0);
		}
	} else {	/* if the image is discrete */
			/* loop through the rows, convert to rawline and write to disk*/
		for (row = 0; row < height; row++) {
			planes_in_rawline((char *)raster_line, (byteplane_ptr *)mimg->planes_array, row, width, comp, planes_to_write, reverse);
			TIFFWriteScanline(out_file, raster_line, row, 0);
		}
	}
		
		/* release memory used for the raster_line */
	SMART_FREE(raster_line);

abort_write_tiff:

		/* close the file */
	TIFFClose(out_file);

	return res;
}

#endif /* TIFF_ENABLED */

/**************************************************************************************
/*	Function:		read_jpeg_mimage
/*	Description:	read a jpeg image from the file system and return an image
/*
/*	Parameters:
/*		<- fname	pathname of the jpeg file in the filesystem
/*
/*	Result:
/*		not NULL	pointer to the image
/*		NULL		if an error condition
/*
/***************************************************************************************/

mimage_ptr read_jpeg_mimage(char *fname)
{ 
	int line_index = 0;	/* index for line to read */

	byteplane_ptr *planes;
	mimage_ptr mimg;

	FILE *in_file;
	JSAMPARRAY row_buffer;
	int row_stride;

	struct jpeg_decompress_struct cinfo;
	struct jpeg_error_mgr jerr;

		/* initialize jpeg */
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_decompress(&cinfo);

		/* open the image file for input */
	if ((in_file = fopen(fname, "rb")) == NULL) 
		return NULL;

		/* initialize jpeg file decompression */
	jpeg_stdio_src(&cinfo, in_file);
	jpeg_read_header(&cinfo, TRUE);
	jpeg_start_decompress(&cinfo);

	row_stride = cinfo.output_width * cinfo.output_components;		/* size of a line */
	row_buffer = (*cinfo.mem->alloc_sarray)((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);	/* buffer for the line */

		/* create the image to store graphical data into */
	mimg = create_mimage(cinfo.output_width, cinfo.output_height, cinfo.output_components, DISCRETE_IMAGE, DENSE_IMAGE);
	if (!mimg)
		goto read_abort;

		/* get the planes array pointer */
	planes = (byteplane_ptr *)mimg->planes_array;

		/* loop trough the lines in the image file */
	while (cinfo.output_scanline < cinfo.output_height) {
		jpeg_read_scanlines(&cinfo, row_buffer, 1);			/* read the scanline */
			/* and write in the planes */
		rawline_in_planes((char *)row_buffer[0], planes, line_index, cinfo.output_width, cinfo.output_components);
		line_index++;	/* go to next line */
	}

read_abort:

		/* clean-up everything */
	jpeg_finish_decompress(&cinfo);
	jpeg_destroy_decompress(&cinfo);
	fclose(in_file);

	return mimg;	/* and return the image */
}

/**************************************************************************************
/*	Function:		write_jpeg_mimage
/*	Description:	write an image to the file system as jpeg
/*
/*	Parameters:
/*		<- fname	pathname of the jpeg file in the filesystem
/*		<- mimg		pointer to the dense image
/*
/*	Result:
/*		0 			ok
/*		else		error number
/*
/***************************************************************************************/

int write_jpeg_mimage(char *fname, mimage_ptr mimg)
{
		/* if the image is sparse */
	if (IS_SPARSE(mimg)) {

		int err;
		
		mimg = clone_mimage(mimg);					/* clone the image */
		if (!mimg)
			return -7;
			
		err = to_dense_mimage(&mimg);				/* convert to dense */
		if (err)
			return -6;
		
		err = write_jpeg_mimage_dense(fname, mimg);	/* write the image */
		destroy_mimage(mimg);						/* destroy the cloned image */
		
		return err;
	} 
	else
		return write_jpeg_mimage_dense(fname, mimg);	/* write the image */
}

/**************************************************************************************
/*	Function:		write_jpeg_mimage_dense
/*	Description:	write a dense image to the file system as jpeg
/*
/*	Parameters:
/*		<- fname	pathname of the jpeg file in the filesystem
/*		<- mimg		pointer to the dense image
/*
/*	Result:
/*		0 			ok
/*		else		error number
/*
/***************************************************************************************/

int write_jpeg_mimage_dense(char *fname, mimage_ptr mimg)
{
	int line_index = 0;	/* index for the line of the image to write to disk */
	int height, width, comp, kind;
	int planes_to_write;

	int row_stride;	
	JSAMPARRAY row_buffer;

	FILE *out_file;

	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;


		/* get some information from the image */
	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);

	if (comp > 3)
		comp = 3;			/* no more than three planes allowed by now */

		/* if the planes are 2 we'll write an image with 3 planes (3rd will be phantom) */
	planes_to_write = comp;
	if (planes_to_write == 2)	planes_to_write ++;

		/* we can write only 1 or 3 planes */
	if ((planes_to_write != 1) && (planes_to_write != 3))
		return -5;

		/* initialize jpeg i/o */
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_compress(&cinfo);

		/* open the file to write */
	if ((out_file = fopen(fname, "wb")) == NULL) 
		return -4;

		/* prepare jpeg i/o */
	jpeg_stdio_dest(&cinfo, out_file);

		/* set the destination info */
	cinfo.image_width      = width;
	cinfo.image_height     = height;
	cinfo.input_components = planes_to_write;

		/* set the color space depending the number of planes */
	if (planes_to_write == 1) 
		cinfo.in_color_space   = JCS_GRAYSCALE;
	else
		cinfo.in_color_space   = JCS_RGB;

		/* set the compression and other jpeg stuff */
	jpeg_set_defaults(&cinfo);
	jpeg_set_quality(&cinfo, 100, FALSE);	/* set the compression to a very low value */
	jpeg_start_compress(&cinfo, TRUE);

		/* allocate memory for one line */		
	row_stride = width * planes_to_write;
	row_buffer = (*cinfo.mem->alloc_sarray)((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

		/* if the image is float */
	if (kind == FLOAT_IMAGE) {
			/* loop through the lines */		
		while(cinfo.next_scanline < cinfo.image_height) {
				/* and write them to disk */
			fplanes_in_rawline((char *)row_buffer[0], (fbyteplane_ptr *)mimg->planes_array, line_index, width, comp, planes_to_write, FALSE);
			jpeg_write_scanlines(&cinfo, row_buffer, 1);
			line_index++;
		}
	} else {		/* if the image is discrete */
			/* loop through the lines */		
		while(cinfo.next_scanline < cinfo.image_height) {
				/* and write them to disk */
			planes_in_rawline((char *)row_buffer[0], (byteplane_ptr *)mimg->planes_array, line_index, width, comp, planes_to_write, FALSE);
			jpeg_write_scanlines(&cinfo, row_buffer, 1);
			line_index++;
		}
	}

		/* cleanup the file and buffers used */
	jpeg_finish_compress(&cinfo);
	jpeg_destroy_compress(&cinfo);
	fclose(out_file);

	return 0;
}

/**************************************************************************************
/*	Function:		get_mimage_info
/*	Description:	return requested info of an image. 
/*					pass NULL for un-necessary info
/*
/*	Parameters:
/*		<- mimg		pointer to the image
/*		-> height	return the height
/*		-> width	return the width
/*		-> comp		return the number of planes
/*		-> kind		return the kind (float or discrete)
/*		-> density	return the density (sparse or dense)
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void get_mimage_info(mimage_ptr mimg, int *height, int *width, int *comp, int *kind, unsigned char *density)
{

	if (mimg) {		/* check for a valid image */
			/* and return only the requested image info */
		if (height)	 *height = ((byteplane_ptr)(mimg->planes_array[0]))->height;
		if (width)	 *width = ((byteplane_ptr)(mimg->planes_array[0]))->width;
		if (comp)	 *comp = mimg->comp;
		if (kind)	 *kind = mimg->kind;	
		if (density) *density = mimg->density;
	}
}

/**************************************************************************************
/*	Function:		convert_mimage_to_float
/*	Description:	convert an discrete image to float kind
/*
/*	Parameters:
/*		<-> mimg		pointer to the image
/*
/*	Result:
/*		0		ok
/*		else	if an error occurs
/*
/***************************************************************************************/

int convert_mimage_to_float(mimage_ptr mimg)
{
	if (mimg->density == DENSE_IMAGE)
		return convert_dmimage_to_float(mimg);
	else if (mimg->density == SPARSE_IMAGE)
		return convert_smimage_to_float(mimg);
			
	return -10;
}

/**************************************************************************************
/*	Function:		convert_mimage_to_discrete
/*	Description:	convert a float image to discrete kind
/*
/*	Parameters:
/*		<-> mimg		pointer to the image
/*
/*	Result:
/*		0		ok
/*		else	if an error occurs
/*
/***************************************************************************************/

int convert_mimage_to_discrete(mimage_ptr mimg)
{
	if (IS_DENSE(mimg)) 
		return convert_dmimage_to_discrete(mimg);
	else
		return convert_smimage_to_discrete(mimg);
			
	return -10;
}

/**************************************************************************************
/*	Function:		crop_mimage
/*	Description:	crop an image
/*
/*	Parameters:
/*		<-> mimg	pointer to the image
/*		<- rect		rect for crop
/*
/*	Result:
/*		0		ok
/*		else	if an error occurs
/*
/***************************************************************************************/

int crop_mimage(mimage_ptr mimg, image_rect_ptr rect)
{
	if (IS_DENSE(mimg))
		return crop_mimage_dense(mimg, rect);
	else
		return crop_mimage_sparse(mimg, rect);

	return -10;	
}

/**************************************************************************************
/*	Function:		crop_from_plane_to_plane
/*	Description:	copy cropped data from one plane to another plane
/*
/*	Parameters:
/*		<-	old_plane	pointer to the source plane
/*		<->	new_plane	pointer to the dest plane
/*		<-	kind		kind of plane
/*		<-	from_rect	source rect
/*		<-	to_rect		destination rect
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void crop_from_plane_to_plane(void *old_plane, void *new_plane, char kind, image_rect_ptr from_rect, image_rect_ptr to_rect)
{
	void *new_buffer, *old_buffer;
	size_t type_size = GET_TYPE_SIZE(kind);
	int i, j;
	int l;
	int ystep  = from_rect->dx;
	int xstart = to_rect->x;		/* start x position */
	int ystart = to_rect->y;		/* start y position */
	
	old_buffer = get_plane_buffer(old_plane, kind);	/* get the old plane buffer */
	new_buffer = get_plane_buffer(new_plane, kind);	/* get the new plane buffer */
	
	if (kind == FLOAT_IMAGE) {	/* if the image is float */
			/* copy the data */
		for (l=0, j=0; j<to_rect->dy; j++)			/* for row in the crop area */
			for (i=0; i<to_rect->dx; l++, i++) 		/* for columns in the crop area */
				*(((fmpixel_ptr)new_buffer)+l) = *(((fmpixel_ptr)old_buffer)+(j+ystart)*ystep+xstart+i);
	} else {					/* if the image is discrete */
			/* copy the data */
		for (l=0, j=0; j<to_rect->dy; j++)			/* for row in the crop area */
			for (i=0; i<to_rect->dx; l++, i++) 		/* for columns in the crop area */
				*(((mpixel_ptr)new_buffer)+l) = *(((mpixel_ptr)old_buffer)+(j+ystart)*ystep+xstart+i);
	}
}

/**************************************************************************************
/*	Function:		stuff_mimage
/*	Description:	stuff an image into another at the position delta
/*
/*	Parameters:
/*		<-	mimg_a		pointer to the source image
/*		<->	mimg_b		pointer to the destination image
/*		<-	delta		offset in destination
/*
/*	Result:
/*		0		ok
/*		else	if an error occurs
/*
/***************************************************************************************/

int stuff_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b, image_displacement_ptr delta)
{
	int i, c;
	int dx, dy;
	int xstep_a, ystep_a;
	int ystep_b;
	mpixel_ptr img_a_buff_ptr, img_b_buff_ptr;
	void *buffer_a, *buffer_b;
	size_t type_size;

	int tox, toy;

	int height_a, width_a, comp_a, type_a; 
	int height_b, width_b, comp_b, type_b; 

		/* get info for both the images */
	get_mimage_info(mimg_a, &height_a, &width_a, &comp_a, &type_a, NULL);
	get_mimage_info(mimg_b, &height_b, &width_b, &comp_b, &type_b, NULL);

		/* check if the images are compatible */
	if ((comp_a != comp_b) && (type_a != type_b))
		return -1;

		/* calculate steps for x and y */
	xstep_a = type_size = type_size = GET_TYPE_SIZE(type_a);
	ystep_a = width_a * type_size;
	ystep_b = width_b * type_size;

	dx = delta->x;
	dy = delta->y;

		/* calculate max x and y for the image to be stuffed */
	tox = min( width_a - dx, width_b );
	toy = min( height_a - dy, height_b );

		/* for every plane */
	for (c=0; c<comp_a; c++) {
			
			/* get the pointer to the buffers */
		buffer_a = get_plane_buffer(mimg_a->planes_array[c], type_a);
		buffer_b = get_plane_buffer(mimg_b->planes_array[c], type_b);

			/* for every line to copy from the inner image */
		for (i=0; i<toy; i++) {
				
				/* calculate the pointer to the destination position of the buffer */
			img_a_buff_ptr = ((mpixel_ptr)buffer_a) + (dy+i)*ystep_a + dx*xstep_a;
			img_b_buff_ptr = ((mpixel_ptr)buffer_b) + i*ystep_b;

				/* and copy the data from the inner to the outer */
			memcpy(img_a_buff_ptr, img_b_buff_ptr, tox*xstep_a);
		}
	}

	return 0;
}

/**************************************************************************************
/*	Function:		shrink_mimage
/*	Description:	look for smallest enclosing rectangle for non-empty pixels
/*
/*	Parameters:
/*		<-	mimg		pointer to the image
/*		<-	threshold	threshold for black
/*		->	rect_ptr	found rectangle	
/*
/*	Result:
/*		0		ok
/*		else	if an error occurs
/*
/***************************************************************************************/

int shrink_mimage(mimage_ptr mimg, int threshold, image_rect_ptr rect_ptr)
{
	image_rect rect;
	unsigned char sides_wrong = 0; /* avoid annoying warning */

	int height, width, comp, kind; 
	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);

	threshold = 0; /* set for white */

	rect.x = rect.y = -1;
	rect.dx = width;
	rect.dy = height;

		/* find the left-most side */
	sides_wrong = 1;
	while(sides_wrong && (rect.x < width)) {
		rect.x ++;
		sides_wrong = check_vline_mimage(mimg, rect.x, threshold);
	}

		/* find the right-most side */
	sides_wrong = 1;
	while(sides_wrong && (rect.dx > 0)) {
		rect.dx--;
		sides_wrong = check_vline_mimage(mimg, rect.dx, threshold);
	}

		/* find the top-most side */
	sides_wrong = 1;
	while(sides_wrong && (rect.y < height)) {
		rect.y ++;
		sides_wrong = check_hline_mimage(mimg, rect.y, threshold);
	}

		/* find the bottom-most side */
	sides_wrong = 1;
	while(sides_wrong && (rect.dy > 0)) {
		rect.dy--;
		sides_wrong = check_hline_mimage(mimg, rect.dy, threshold);
	}

		/* store the data in the rect_ptr */
	rect_ptr->x = rect.x;
	rect_ptr->y = rect.y;
	rect_ptr->dx = rect.dx;
	rect_ptr->dy = rect.dy;

		/* and check if the obtained rectangle is valid */
	if (!is_valid_rect(rect_ptr)) return -1;

	return 0;
}

/**************************************************************************************
/*	Function:		width_and_height_mimage
/*	Description:	return width and height of an image
/*
/*	Parameters:
/*		<-	mimg		pointer to the image
/*		->	width		width of the image
/*		->	height		height of the image
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void width_and_height_mimage(mimage_ptr mimg, int *width, int *height)
{
	*width = width_mimage(mimg);	/* get the width */
	*height = height_mimage(mimg);	/* get the height */
}

/**************************************************************************************
/*	Function:		width_mimage
/*	Description:	return width of an image
/*
/*	Parameters:
/*		<-	mimg		pointer to the image
/*
/*	Result:
/*		the width
/*
/***************************************************************************************/

int width_mimage(mimage_ptr mimg)
{	
	return ((byteplane_ptr)(mimg->planes_array[0]))->width;	/* get the width */
}

/**************************************************************************************
/*	Function:		height_mimage
/*	Description:	return height of an image
/*
/*	Parameters:
/*		<-	mimg		pointer to the image
/*
/*	Result:
/*		the height
/*
/***************************************************************************************/

int height_mimage(mimage_ptr mimg)
{
	return ((byteplane_ptr)(mimg->planes_array[0]))->height; /* get the height */
}

/**************************************************************************************
/*	Function:		set_width_and_height_mimage
/*	Description:	set width and height of an image
/*
/*	Parameters:
/*		<-	mimg		pointer to the image
/*		<-	width		width for the image
/*		<-	height		height for the image
/*
/*	Result:
/*		TRUE		ok
/*		FALSE		if a plane was not found
/*
/***************************************************************************************/

int set_width_and_height_mimage(mimage_ptr mimg, int width, int height)
{
	int c;
	int ret = TRUE;

	byteplane_ptr plane;
	byteplane_ptr *planes = mimg->planes_array;
	int planes_number = mimg->comp;
	
	if (!planes)
		return FALSE;
	
	for (c=0; c<planes_number; c++) {	/* loop over the planes */

		plane = planes[c];				/* get the pointer to the plane */
		if (!plane)
			ret = FALSE;

		plane->height = height;			/* set the height */
		plane->width = width;			/* set the width */
	}
	
	return TRUE;
}

/**************************************************************************************
/*	Function:		const_mimage
/*	Description:	create a const image
/*
/*	Parameters:
/*		<-	values_array	array of values to be stored in the image
/*		<-	dim_ptr			dimensions of the image
/*		<-	kind			kind of the image (float or discrete)
/*		<-	comp			number of planes
/*
/*	Result:
/*		not NULL	pointer to the image
/*		NULL		if an error condition
/*
/***************************************************************************************/

mimage_ptr const_mimage(void *values_array, image_dimension_ptr dim_ptr, int kind, int comp)
{
	int c, i;
	fmpixel fvalue;
	fmpixel_ptr float_buffer;

	mimage_ptr mimg;
	mpixel uvalue;
	mpixel_ptr buffer;
	size_t pixels_count_plan;

	int height, width;

	width = dim_ptr->x;		/* width of target image */
	height = dim_ptr->y;	/* height of target image */

		/* create the image of the requested size, comp and kind */
	mimg = create_mimage(width, height, comp, kind, DENSE_IMAGE);
	if (!mimg) return NULL;

	pixels_count_plan = height * width;	/* calc the number of pixel per plane */

	if (mimg->kind == FLOAT_IMAGE) {		/* if the image is float */

			/* loop over the planes */
		for (c=0; c<mimg->comp; c++) {
				/* get the image data buffer for this plane */
			float_buffer = get_plane_buffer(mimg->planes_array[c], kind);
				/* get the constant value for this plane */
			fvalue = *(((fmpixel_ptr)values_array)+c);
				/* and store in all pixels in the plane */
			for (i=0; i<pixels_count_plan; i++) {
				*float_buffer = fvalue;
				float_buffer++;
			}
		}
	} else {		/* if the image is discrete */

			/* loop over the planes */
		for (c=0; c<mimg->comp; c++) {
				/* get the image data buffer for this plane */
			buffer = get_plane_buffer(mimg->planes_array[c], kind);
				/* get the constant value for this plane */
			uvalue = *(((mpixel_ptr)values_array)+c);
				/* and store in all pixels in the plane */
			for (i=0; i<pixels_count_plan; i++) {
				*buffer = uvalue;
				buffer++;
			}
		}
	}

	return mimg;
}

/**************************************************************************************
/*	Function:		are_same_format_mimages
/*	Description:	check if two images have the same format
/*
/*	Parameters:
/*		<-	mimg_a		first image
/*		<-	mimg_b		second image
/*
/*	Result:
/*		TRUE	have the same format
/*		FALSE	not have the same format
/*
/***************************************************************************************/

int are_same_format_mimages(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
	int height_a, width_a, planes_number_a, type_a; 
	int height_b, width_b, planes_number_b, type_b; 

		/* get information for both the images */
	get_mimage_info(mimg_a, &height_a, &width_a, &planes_number_a, &type_a, NULL);
	get_mimage_info(mimg_b, &height_b, &width_b, &planes_number_b, &type_b, NULL);

		/* and do the needed checks */
	if ((height_a != height_b) || (width_a != width_b) || (planes_number_a != planes_number_b) || (type_a != type_b))	
		return 0;	/* error condition */

	return 1;		/* everything is ok */
}

/**************************************************************************************
/*	Function:		binary_op_constant_mimage
/*	Description:	perform a binary operation between 
/*					an image and a constant value
/*
/*	Parameters:
/*		<-	mimg			image
/*		<-	values_array	constant data pixel array
/*		<-	gray_op			pointer to a binary operation function for discrete case
/*		<-	float_op		pointer to a binary operation function for float case
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr binary_op_constant_mimage(mimage_ptr mimg, void *values_array, mpixel (*gray_op)(mpixel, mpixel), fmpixel (*float_op)(fmpixel, fmpixel))
{
	if (mimg->density == DENSE_IMAGE)
		return binary_op_constant_dmimage(mimg, values_array, gray_op, float_op);
	else if (mimg->density == SPARSE_IMAGE)
		return binary_op_constant_smimage(mimg, values_array, gray_op, float_op);

	return NULL;	/* format not recognized */
}

/**************************************************************************************
/*	Function:		binary_op_mimage
/*	Description:	perform a binary operation between two images
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	mimg_b			second operand image
/*		<-	gray_op			pointer to a binary operation function for discrete case
/*		<-	float_op		pointer to a binary operation function for float case
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr binary_op_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b, mpixel (*gray_op)(mpixel, mpixel), fmpixel (*float_op)(fmpixel, fmpixel))
{
		/* call the right routine depending the two input images */
	if (mimg_a->density == mimg_b->density) {
		if (mimg_a->density == DENSE_IMAGE)
			return binary_op_dmimage(mimg_a, mimg_b, gray_op, float_op);
		else if (mimg_a->density == SPARSE_IMAGE)
			return binary_op_smimage(mimg_a, mimg_b, gray_op, float_op);
	} else {
		if (mimg_a->density == SPARSE_IMAGE)
			return binary_op_sdmimage(mimg_a, mimg_b, gray_op, float_op);
		else if (mimg_a->density == DENSE_IMAGE)
			return binary_op_sdmimage(mimg_b, mimg_a, gray_op, float_op);
	}
		
	return NULL;	/* format not recognized */
}

/**************************************************************************************
/*	Function:		unimath_mimage
/*	Description:	perform an "user defined" unary operation to an image
/*
/*	Parameters:
/*		<-	user_ptr		[private] used from the setl2 stubs
/*		<-	user_ptr2		[private] used from the setl2 stubs
/*		<-	mimg			operand image
/*		<-	floating_op		pointer to an external unary float operation
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr unimath_mimage(void *user_ptr, void *user_ptr2, mimage_ptr mimg, fmpixel (*floating_op)(void *user_ptr, void *user_ptr2, float, int))
{
	if (mimg->density == SPARSE_IMAGE)
		return unimath_smimage(user_ptr, user_ptr2, mimg, floating_op);
	else if (mimg->density == DENSE_IMAGE)
		return unimath_dmimage(user_ptr, user_ptr2, mimg, floating_op);

	return NULL;
}

/**************************************************************************************
/*	Function:		unary_op_mimage
/*	Description:	perform an unary operation to an image
/*
/*	Parameters:
/*		<-	mimg			operand image
/*		<-	gray_op			pointer to an unary discrete operation function for discrete case
/*		<-	float_op		pointer to an unary float operation function for float case
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr unary_op_mimage(mimage_ptr mimg, unsigned char (*gray_op)(unsigned char), float (*float_op)(float))
{
	if (mimg->density == DENSE_IMAGE)
		return unary_op_dmimage(mimg, gray_op, float_op);
	else if (mimg->density == SPARSE_IMAGE)
		return unary_op_smimage(mimg, gray_op, float_op);

	return NULL;	/* format not recognized */
}

/**************************************************************************************
/*	Function:		plus_mimage
/*	Description:	sum two images
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	mimg_b			second operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr plus_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
	return binary_op_mimage(mimg_a, mimg_b, plus_op, fplus_op);
}

/**************************************************************************************
/*	Function:		minus_mimage
/*	Description:	subtract two images
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	mimg_b			second operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr minus_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
	return binary_op_mimage(mimg_a, mimg_b, minus_op, fminus_op);
}

/**************************************************************************************
/*	Function:		times_mimage
/*	Description:	multiply two images
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	mimg_b			second operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr times_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
	return binary_op_mimage(mimg_a, mimg_b, times_op, ftimes_op);
}

/**************************************************************************************
/*	Function:		divide_mimage
/*	Description:	divide two images
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	mimg_b			second operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr divide_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
	return binary_op_mimage(mimg_a, mimg_b, divide_op, fdivide_op);
}

/**************************************************************************************
/*	Function:		maxim_mimage
/*	Description:	return an image that has as pixel values
/*					the max of the value of the pixel in the operand images
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	mimg_b			second operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr maxim_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
	return binary_op_mimage(mimg_a, mimg_b, maxim_op, fmaxim_op);
}

/**************************************************************************************
/*	Function:		minim_mimage
/*	Description:	return an image that has as pixel values
/*					the min of the value of the pixel in the operand images
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	mimg_b			second operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr minim_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
  return binary_op_mimage(mimg_a, mimg_b, minim_op, fminim_op);
}

/**************************************************************************************
/*	Function:		power_mimage
/*	Description:	return an image that has as pixel values
/*					the power of the value of the pixel in the first image
/*					by the ones of the second one
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	mimg_b			second operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr power_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
  return binary_op_mimage(mimg_a, mimg_b, power_op, fpower_op);
}

/**************************************************************************************
/*	Function:		plus_mimage_constant
/*	Description:	sum every pixel in one image with a constant value
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	values_array	constant pixel value
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr plus_mimage_constant(mimage_ptr mimg_a, void *values_array)
{
  return binary_op_constant_mimage(mimg_a, values_array, plus_op, fplus_op);
}

/**************************************************************************************
/*	Function:		minus_mimage_constant
/*	Description:	subtract every pixel in one image with a constant value
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	values_array	constant pixel value
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr minus_mimage_constant(mimage_ptr mimg_a, void *values_array)
{
  return binary_op_constant_mimage(mimg_a, values_array, minus_op, fminus_op);
}

/**************************************************************************************
/*	Function:		times_mimage_constant
/*	Description:	multiply every pixel in one image with a constant value
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	values_array	constant pixel value
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr times_mimage_constant(mimage_ptr mimg_a, void *values_array)
{
  return binary_op_constant_mimage(mimg_a, values_array, times_op, ftimes_op);
}

/**************************************************************************************
/*	Function:		divide_mimage_constant
/*	Description:	divide every pixel in one image with a constant value
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	values_array	constant pixel value
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr divide_mimage_constant(mimage_ptr mimg_a, void *values_array)
{
  return binary_op_constant_mimage(mimg_a, values_array, divide_op, fdivide_op);
}

/**************************************************************************************
/*	Function:		maxim_mimage_constant
/*	Description:	return a new image where every pixel is
/*					the maximum between the constant value and the
/*					correspondent pixel in the source image
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	values_array	constant pixel value
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr maxim_mimage_constant(mimage_ptr mimg_a, void *values_array)
{
  return binary_op_constant_mimage(mimg_a, values_array, maxim_op, fmaxim_op);
}

/**************************************************************************************
/*	Function:		maxim_mimage_constant
/*	Description:	return a new image where every pixel is
/*					the minimum between the constant value and the
/*					correspondent pixel in the source image
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	values_array	constant pixel value
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr minim_mimage_constant(mimage_ptr mimg_a, void *values_array)
{
  return binary_op_constant_mimage(mimg_a, values_array, minim_op, fminim_op);
}

/**************************************************************************************
/*	Function:		power_mimage_constant
/*	Description:	return a new image where every pixel is
/*					the power of the correspondent pixel in 
/*					the source image to the constant value
/*					
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	values_array	constant pixel value
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr power_mimage_constant(mimage_ptr mimg_a, void *values_array)
{
  return binary_op_constant_mimage(mimg_a, values_array, power_op, fpower_op);
}

/**************************************************************************************
/*	Function:		sin_mimage
/*	Description:	return a new image where every pixel is
/*					the sine of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr sin_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, sin_op, fsin_op);
}

/**************************************************************************************
/*	Function:		cosin_mimage
/*	Description:	return a new image where every pixel is
/*					the cosine of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr cosin_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, cosin_op, fcosin_op);
}

/**************************************************************************************
/*	Function:		tan_mimage
/*	Description:	return a new image where every pixel is
/*					the tangent of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr tan_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, tan_op, ftan_op);
}

/**************************************************************************************
/*	Function:		exp_mimage
/*	Description:	return a new image where every pixel is
/*					the exp of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr exp_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, exp_op, fexp_op);
}

/**************************************************************************************
/*	Function:		log_mimage
/*	Description:	return a new image where every pixel is
/*					the log of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr log_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, log_op, flog_op);
}

/**************************************************************************************
/*	Function:		abs_mimage
/*	Description:	return a new image where every pixel is
/*					the abs of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr abs_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, abs_op, fabs_op);
}

/**************************************************************************************
/*	Function:		even_mimage
/*	Description:	return a new image where every pixel is
/*					1 for even value in the correspondent pixel in 
/*					the source image, else 0 
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr even_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, even_op, feven_op);
}

/**************************************************************************************
/*	Function:		odd_mimage
/*	Description:	return a new image where every pixel is
/*					1 for odd value in the correspondent pixel in 
/*					the source image, else 0 
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr odd_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, odd_op, fodd_op);
}

/**************************************************************************************
/*	Function:		sqrt_mimage
/*	Description:	return a new image where every pixel is
/*					the square root of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr sqrt_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, sqrt_op, fsqrt_op);
}

/**************************************************************************************
/*	Function:		asin_mimage
/*	Description:	return a new image where every pixel is
/*					the arc sin of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr asin_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, asin_op, fasin_op);
}

/**************************************************************************************
/*	Function:		acosin_mimage
/*	Description:	return a new image where every pixel is
/*					the arc cosine of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr acosin_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, acosin_op, facosin_op);
}

/**************************************************************************************
/*	Function:		atan_mimage
/*	Description:	return a new image where every pixel is
/*					the arc tangent of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr atan_mimage(mimage_ptr mimg)
{
	return unary_op_mimage(mimg, atan_op, fatan_op);
}

/**************************************************************************************
/*	Function:		fix_mimage
/*	Description:	return a new image where every pixel is
/*					the fix (discrete) of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr fix_mimage(mimage_ptr mimg)
{
	mimg = unary_op_mimage(mimg, fix_op, ffix_op);
	convert_mimage_to_discrete(mimg);
	return mimg;
}

/**************************************************************************************
/*	Function:		floor_mimage
/*	Description:	return a new image where every pixel is
/*					the floor (discrete) of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr floor_mimage(mimage_ptr mimg)
{
	mimg	= unary_op_mimage(mimg, floor_op, ffloor_op);
	convert_mimage_to_discrete(mimg);
	return mimg;
}

/**************************************************************************************
/*	Function:		ceiling_mimage
/*	Description:	return a new image where every pixel is
/*					the ceiling (discrete) of the correspondent pixel in 
/*					the source image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr ceiling_mimage(mimage_ptr mimg)
{
	mimg	= unary_op_mimage(mimg, ceiling_op, fceiling_op);
	convert_mimage_to_discrete(mimg);
	return mimg;
}

/**************************************************************************************
/*	Function:		maxof_mimage
/*	Description:	return the max pixel value of an image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		the maximum
/*
/***************************************************************************************/

float maxof_mimage(mimage_ptr mimg)
{
	if (mimg->density == DENSE_IMAGE)
		return maxof_mimage_dense(mimg);
	else
		return maxof_mimage_sparse(mimg);
}

/**************************************************************************************
/*	Function:		minof_mimage
/*	Description:	return the min pixel value of an image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*
/*	Result:
/*		the maximum
/*
/***************************************************************************************/

float minof_mimage(mimage_ptr mimg)
{
	if (mimg->density == DENSE_IMAGE)
		return minof_mimage_dense(mimg);
	else
		return minof_mimage_sparse(mimg);
}

/**************************************************************************************
/*	Function:		sumall_mimage
/*	Description:	return the momentums of an image
/*					
/*	Parameters:
/*		<-	mimg	operand image
/*		<-	power	highest momentum requested
/*
/*	Result:
/*		not NULL	array of requested momentums
/*		NULL		out of memory
/*
/***************************************************************************************/

double *sumall_mimage(mimage_ptr mimg, int power)
{
	if (mimg->density == DENSE_IMAGE)
		return sumall_mimage_dense(mimg, power);
	else
		return sumall_mimage_sparse(mimg, power);
}

/**************************************************************************************
/*	Function:		sort_mimage
/*	Description:	return an array of sorted image values
/*					
/*	Parameters:
/*		<-	mimg		operand image
/*		->	elements	return the number of elements in the sorted array
/*
/*	Result:
/*		not NULL	array of sorted values
/*		NULL		out of memory
/*
/***************************************************************************************/

void *sort_mimage(mimage_ptr mimg, int *elements)
{
	if (mimg->density == DENSE_IMAGE)
		return sort_mimage_dense(mimg, elements);
	else
		return sort_mimage_sparse(mimg, elements);
}

/**************************************************************************************
/*	Function:		scale_mimage
/*	Description:	scale an image into a new image
/*					The data is interpolated using a bicubic algorithm
/*					
/*	Parameters:
/*		<-	mimg			image to be scaled
/*		<-	new_dim_ptr		pointer to a dimension structure containing the new size
/*
/*	Result:
/*		not NULL	new scaled image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr scale_mimage(mimage_ptr mimg, image_dimension_ptr new_dim_ptr)
{
	int c, i, j;
	float o_i, o_j;
	int io_i, io_j;
	mimage_ptr scaled_mimg;

	int height, width, planes_number, kind; 
	size_t pixels_count_plan;
	size_t type_size;

	unsigned char *buffer, *src_buffer;
	fmpixel_ptr float_buffer, float_src_buffer;
	float xratio, yratio;

		/* get information about the input image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	pixels_count_plan = height * width;	/* get the number of pixels per plane */
	type_size = GET_TYPE_SIZE(kind);	/* get the size of a single pixel in the image */

		/* create the destination image */
	scaled_mimg = create_mimage(new_dim_ptr->x, new_dim_ptr->y, planes_number, kind, DENSE_IMAGE);
	if (!scaled_mimg)
		return NULL;
		
		/* calculate the x and y ratio between the source and destination image */
	xratio = width / (float)new_dim_ptr->x;
	yratio = height / (float)new_dim_ptr->y;

	if (kind == FLOAT_IMAGE) {			/* if float image */
	
			/* loop over the planes */
		for (c=0; c<planes_number; c++) {
			
				/* get the buffer for the destination image */
			float_buffer = get_plane_buffer(scaled_mimg->planes_array[c], kind);
				/* get the buffer for the source image */
			float_src_buffer = get_plane_buffer(mimg->planes_array[c], kind);
			
				/* for every line in the destination image */
			for (j=0; j<new_dim_ptr->y; j++) {
			
				o_j = j * yratio;	/* get the corresponding line in the original image */
				
					/* for every column in the destination image */
				for (i=0; i<new_dim_ptr->x; i++) {

						fmpixel f00, f10, f20, f30,
							f01, f11, f21, f31,
							f02, f12, f22, f32,
							f03, f13, f23, f33;
						
						fmpixel finterpolated_value;
						
						o_i = i * xratio;	/* get the corresponding line in the original image */
						
						io_i = o_i;
						io_j = o_j;

							/* 
							 * retrieve a 4x4 matrix of pixel values around current pixel 
							 */

						f00 = *(float_src_buffer + max ( io_j - 1, 0) * width          + max ( io_i - 1, 0)); 
						f10 = *(float_src_buffer + io_j * width                        + max ( io_i - 1, 0)); 
						f20 = *(float_src_buffer + min ( io_j + 1, (height-1)) * width + max ( io_i - 1, 0)); 
						f30 = *(float_src_buffer + min ( io_j + 2, (height-1)) * width + max ( io_i - 1, 0)); 

						f01 = *(float_src_buffer + max ( io_j - 1, 0) * width          + io_i); 
						f11 = *(float_src_buffer + io_j * width                        + io_i); 
						f21 = *(float_src_buffer + min ( io_j + 1, (height-1)) * width + io_i); 
						f31 = *(float_src_buffer + min ( io_j + 2, (height-1)) * width + io_i); 
						
						f02 = *(float_src_buffer + max ( io_j - 1, 0) * width          + min ( io_i + 1, width-1)); 
						f12 = *(float_src_buffer + io_j * width                        + min ( io_i + 1, width-1)); 
						f22 = *(float_src_buffer + min ( io_j + 1, (height-1)) * width + min ( io_i + 1, width-1)); 
						f32 = *(float_src_buffer + min ( io_j + 2, (height-1)) * width + min ( io_i + 1, width-1)); 

						f03 = *(float_src_buffer + max ( io_j - 1, 0) * width          + min ( io_i + 2, width-1)); 
						f13 = *(float_src_buffer + io_j * width                        + min ( io_i + 2, width-1)); 
						f23 = *(float_src_buffer + min ( io_j + 1, (height-1)) * width + min ( io_i + 2, width-1)); 
						f33 = *(float_src_buffer + min ( io_j + 2, (height-1)) * width + min ( io_i + 2, width-1)); 


							/* calculate the interpolated value */
						finterpolated_value =interpolate_from_16(	f00,f01,f02,f03,
																	f10,f11,f12,f13,
																	f20,f21,f22,f23,
																	f30,f31,f32,f33, 
																	(float)1.0 , (float)1. + o_i - (float)io_i , 
																	(float)1. + o_j - (float)io_j);

							/* clip the interpolated value into the [0., 255.] range */
						if (finterpolated_value < 0.)
							finterpolated_value = 0.;
						else if (finterpolated_value > 255.)
							finterpolated_value = 255.;
								
						*float_buffer = finterpolated_value;
						float_buffer++;
				}
			}
		}
	} else {			/* if discrete image */

			/* loop over the planes */
		for (c=0; c<planes_number; c++) {
			
				/* get the buffer for the destination image */
			buffer = get_plane_buffer(scaled_mimg->planes_array[c], kind);
				/* get the buffer for the source image */
			src_buffer = get_plane_buffer(mimg->planes_array[c], kind);
			
				/* for every line in the destination image */
			for (j=0; j<new_dim_ptr->y; j++) {

				o_j = j * yratio;	/* get the corresponding line in the original image */

					/* for every column in the destination image */
				for (i=0; i<new_dim_ptr->x; i++) {

						float u00, u10, u20, u30,
									u01, u11, u21, u31,
									u02, u12, u22, u32,
									u03, u13, u23, u33;
					
						int interpolated_value;
						
						o_i = i * xratio;	/* get the corresponding line in the original image */			
						
						io_i = o_i;
						io_j = o_j;

							/* 
							 * retrieve a 4x4 matrix of pixel values around current pixel 
							 */

						u00 = *(src_buffer + max ( io_j - 1, 0) * width          + max ( io_i - 1, 0)); 
						u10 = *(src_buffer + io_j * width                        + max ( io_i - 1, 0)); 
						u20 = *(src_buffer + min ( io_j + 1, (height-1)) * width + max ( io_i - 1, 0)); 
						u30 = *(src_buffer + min ( io_j + 2, (height-1)) * width + max ( io_i - 1, 0)); 

						u01 = *(src_buffer + max ( io_j - 1, 0) * width          + io_i); 
						u11 = *(src_buffer + io_j * width                        + io_i); 
						u21 = *(src_buffer + min ( io_j + 1, (height-1)) * width + io_i); 
						u31 = *(src_buffer + min ( io_j + 2, (height-1)) * width + io_i); 
						
						u02 = *(src_buffer + max ( io_j - 1, 0) * width          + min ( io_i + 1, width - 1)); 
						u12 = *(src_buffer + io_j * width                        + min ( io_i + 1, width - 1)); 
						u22 = *(src_buffer + min ( io_j + 1, (height-1)) * width + min ( io_i + 1, width - 1)); 
						u32 = *(src_buffer + min ( io_j + 2, (height-1)) * width + min ( io_i + 1, width - 1)); 

						u03 = *(src_buffer + max ( io_j - 1, 0) * width          + min ( io_i + 2, width - 1)); 
						u13 = *(src_buffer + io_j * width                        + min ( io_i + 2, width - 1)); 
						u23 = *(src_buffer + min ( io_j + 1, (height-1)) * width + min ( io_i + 2, width - 1)); 
						u33 = *(src_buffer + min ( io_j + 2, (height-1)) * width + min ( io_i + 2, width - 1)); 

							/* calculate the interpolated value */
						interpolated_value = interpolate_from_16(	u00, u01, u02, u03, 
																	u10, u11, u12, u13, 
																	u20, u21, u22, u23, 
																	u30, u31, u32, u33, 
																	(float)1.0 , (float)1. + o_i - (float)io_i,
																	(float)1. + o_j - (float)io_j);

							/* clip the interpolated value into the [0., 255.] range */
						if (interpolated_value < 0)
							interpolated_value = 0;
						else if (interpolated_value > 255)
							interpolated_value = 255;

							/* write the data to the destination buffer */
						*buffer = interpolated_value;
						buffer++;
				}
			}
		}
	}

	return scaled_mimg;	/* return the scaled image */
}

/**************************************************************************************
/*	Function:		gradient_mimage
/*	Description:	create an image using a foreign function to generate a gradient
/*					
/*	Parameters:
/*		<-	user_ptr		[private] use to pass data to the foreign function
/*		<-	user_ptr2		[private] use to pass data to the foreign function
/*		<-	floating_op		float foreign function to generate the gradient
/*		<-	new_dim_ptr		image size
/*		<-	kind			image kind (float or discrete)
/*		<-	planes_number	image number of planes
/*
/*	Result:
/*		not NULL	the gradient image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr gradient_mimage(void *user_ptr, void *user_ptr2, fmpixel (*floating_op)(void *user_ptr, void *user_ptr2, int x, int y, int c), image_dimension_ptr new_dim_ptr, int kind, int planes_number)
{
	int i, j, c;

	mimage_ptr mimg;

	unsigned char *buffer;
	fmpixel_ptr float_buffer;

	int height, width; 
	size_t pixels_count_plan;
	size_t type_size;

	height = new_dim_ptr->x;
	width  = new_dim_ptr->y;

		/* create an image with requested size, planes_number and kind */
	mimg = create_mimage(width, height, planes_number, kind, DENSE_IMAGE);
	if (!mimg)
		return NULL;

	pixels_count_plan = height * width;		/* get the number of pixels per plane */
	type_size = GET_TYPE_SIZE(kind);		/* get the pixel size */

	if (kind == FLOAT_IMAGE) {	/* if float image */

			/* loop over the planes */
		for (c=0; c<planes_number; c++) {

				/* get the buffer where pixels are stored */
			float_buffer = get_plane_buffer(mimg->planes_array[c], kind);

				/* loop over lines */
			for (j=0; j<height; j++) {
					
					/* loop over columns */
				for (i=0; i<width; i++) {
							
							/* put the into the pixel the result of a user defined pattern function */
						*float_buffer = floating_op(user_ptr, user_ptr2, i, j, c);
						float_buffer++;
				}
			}
		}

	} else {	/* if discrete image */

			/* loop over the planes */
		for (c=0; c<planes_number; c++) {

				/* get the buffer where pixels are stored */
			buffer = get_plane_buffer(mimg->planes_array[c], kind);

				/* loop over lines */
			for (j=0; j<height; j++) {

					/* loop over columns */
				for (i=0; i<width; i++) {

							/* put the into the pixel the result of a user defined pattern function */
						*buffer = (unsigned char)floating_op(user_ptr, user_ptr2, i, j, c);
						buffer++;
				}
			}
		}

	}

	return mimg;
}

/**************************************************************************************
/*	Function:		mimage_2_string_format
/*	Description:	convert an image to a flat string representation
/*	Note:			doesn't work for sparse images
/*					
/*	Parameters:
/*		<-	mimg			image to convert
/*		->	str				the string 
/*		->	length			length of generated string 
/*
/*	Result:
/*		0		ok
/*		else 	error
/*
/***************************************************************************************/

int mimage_2_string_format(mimage_ptr mimg, char **str, size_t *length)
{
	int c;
	int height, width, planes_number, kind; 
	size_t pixels_count_plan;
	char *buffer_src, *buffer_dst;
	size_t type_size;
	size_t l;
	string_format_ptr sfp;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);
	pixels_count_plan = height * width;

	l = sizeof(string_format) + pixels_count_plan * type_size * planes_number;
	sfp = (string_format_ptr)malloc(l);
	if (!sfp) {
		*str = NULL;
		*length = 0;
		return -1;
	}

	sfp->signature = 0;	/* this should always be zero */
	sfp->length = l;

	sfp->height = height;
	sfp->width = width;
	sfp->kind = kind;
	sfp->comp = planes_number;

	buffer_dst = (char *)sfp+sizeof(string_format);
	for (c=0; c<planes_number; c++) {
		buffer_src = get_plane_buffer(mimg->planes_array[c], kind);
		memcpy(buffer_dst, buffer_src, pixels_count_plan * type_size);
		buffer_dst += (pixels_count_plan * type_size);
	}

	*str = (char *)sfp;
	*length = l;

	return 0;	
}

/**************************************************************************************
/*	Function:		string_format_2_mimage
/*	Description:	convert from a flat string representation to an image
/*	Note:			doesn't work for sparse images
/*					
/*	Parameters:
/*		<-	str				image in string representation
/*
/*	Result:
/*		not NULL	pointer to the image
/*		NULL		error condition occurred
/*
/***************************************************************************************/

mimage_ptr string_format_2_mimage(char *str)
{
	int c;
	size_t type_size, pixels_count_plan;
	char *buffer_src, *buffer_dst;
	mimage_ptr mimg;
	string_format_ptr sfp = (string_format_ptr)str;
	
	mimg = create_mimage(sfp->width, sfp->height, sfp->comp, sfp->kind, DENSE_IMAGE);
	if (!mimg)
		return NULL;
		
	type_size = GET_TYPE_SIZE(sfp->kind);
	pixels_count_plan = sfp->height * sfp->width;
	
	buffer_src = (char *)sfp+sizeof(string_format);
	for (c=0; c<sfp->comp; c++) {
		buffer_dst = get_plane_buffer(mimg->planes_array[c], sfp->kind);
		memcpy(buffer_dst, buffer_src, pixels_count_plan * type_size);
		buffer_src += (pixels_count_plan * type_size);
	}

	return mimg;
}

/**************************************************************************************
/*	Function:		make_compatible
/*	Description:	make two images compatible each other
/*					
/*	Parameters:
/*		<->	mimg_a		first image
/*		<->	mimg_b		second image
/*
/*	Result:
/*		0			no error
/*		else		error condition
/*
/***************************************************************************************/

int make_compatible(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
	int height_a, width_a, planes_number_a, type_a; 
	int height_b, width_b, planes_number_b, type_b; 

		/* get information about the two images */
	get_mimage_info(mimg_a, &height_a, &width_a, &planes_number_a, &type_a, NULL);
	get_mimage_info(mimg_b, &height_b, &width_b, &planes_number_b, &type_b, NULL);

		/*
		 * if they haven't the same number of 
		 * planes return an error condition 
		 */
	if (planes_number_a != planes_number_b)
		return 0;
		
		/*
		 * If images have different format convert them to the wider one
		 */
	
	if (type_a != type_b) {
		
/*		 GDM 042199 - This code is buggy because it check for same size only if it's going to convert
		the kind of the images. Although, new version works with every two images size so
		this is not needed anymore.
		
		if ((height_a == height_b) && (width_a == width_b) && (planes_number_a == planes_number_b)) {
 */
			if (mimg_a->kind == DISCRETE_IMAGE) {
				if (!convert_mimage_to_float(mimg_a))
					return 0;
				type_a = FLOAT_IMAGE;
			}	else {
				if (!convert_mimage_to_float(mimg_b))
					return 0;
				type_b = FLOAT_IMAGE;
			}
/*
		} else
			return 0;
 */
	}

	return 1;
}

/**************************************************************************************
/*	Function:		set_planes_mimage
/*	Description:	insert a set of planes from the image mimg to the image mimg_res
/*
/*	Parameters:
/*		<->	mimg_res	destintaion image
/*		<-	mimg		source image
/*		<-	from		first plane
/*		<-	to			last plane
/*
/*	Result:
/*		not NULL	the mimg_res itself with the inseted planes
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr set_planes_mimage(mimage_ptr mimg_res, mimage_ptr mimg, int from, int to)
{

	if (mimg_res->density != mimg->density)
		return NULL;
		
	if (mimg_res->density == DENSE_IMAGE)
		return set_planes_mimage_dense(mimg_res, mimg, from, to);
	else
		return set_planes_mimage_sparse(mimg_res, mimg, from, to);
	
}

/**************************************************************************************
/*	Function:		extract_planes_mimage
/*	Description:	create an image extracting a set of planes from another
/*
/*	Parameters:
/*		<-	mimg		source image
/*		<-	from		first plane
/*		<-	to			last plane
/*
/*	Result:
/*		not NULL	a pointer to the extracted image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr extract_planes_mimage(mimage_ptr mimg, int from, int to)
{
	if (mimg->density == DENSE_IMAGE)
		return extract_planes_dmimage(mimg, from, to);
	else if (mimg->density == SPARSE_IMAGE)
		return extract_planes_mimage_sparse(mimg, from, to);
	
	return NULL;
}

/**************************************************************************************
/*	Function:		convolve_mimage
/*	Description:	create an image as a convolution of another
/*
/*	Parameters:
/*		<-	mimg		source image
/*		<-	i_offset	array of offsets for x
/*		<-	j_offset	array of offsets for y
/*		<-	coefficient	array of coefficients
/*		<-	dim			kernel size
/*
/*	Result:
/*		not NULL	a pointer to the convolved image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr convolve_mimage(mimage_ptr mimg, int *i_offset, int *j_offset, float *coefficient, size_t dim)
{
	unsigned char *buffer, *res_buffer;
	float *fbuffer, *fres_buffer;
	
	float sum;
	int i, j, c, k;
	mimage_ptr mimg_res;
	
	int width, height, planes_number, kind;
	size_t type_size;

		/* get information from source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);	/* get the size of a pixel in the source image */

		/* create the destination image */
	mimg_res = create_mimage(width, height, planes_number, kind, DENSE_IMAGE);
	if (!mimg_res)
		return NULL;

	if (kind == FLOAT_IMAGE) {	/* if float image */

		for (c=0; c<planes_number; c++) {	/* loop over the planes */

			fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;			/* get the source buffer */
			fres_buffer = get_fbyteplane_from_mimage(mimg_res, c)->buffer;	/* get the destination buffer */

				/* for every line */
			for (j=0; j<height; j++) {
					/* for every column */
				for (i=0; i<width; i++) {
					
						/* reset the accumulator */
					sum = 0.;
					
						/* do the convolution using the i_offset, j_offset kernels */
					for (k=0;k<dim;k++) {
						if ( ((i-i_offset[k])>=0) 
								&& ((j-j_offset[k])>=0) 
								&& ((i-i_offset[k])<width) 
								&& ((j-j_offset[k])<height)) {
							sum += (*(fbuffer + (j-j_offset[k])*width + (i-i_offset[k])))*coefficient[k];
						}
					}
					
						/* write the convolved pixel to the destination image */
					*fres_buffer = sum;
						/* advance to next pixel */
					fres_buffer++;
				}
			}
		}
	} else {	/* if discrete image */

		for (c=0; c<planes_number; c++) {	/* loop over the planes */

			buffer = get_byteplane_from_mimage(mimg, c)->buffer;			/* get the source buffer */
			res_buffer = get_byteplane_from_mimage(mimg_res, c)->buffer;	/* get the destination buffer */

				/* for every line */
			for (j=0; j<height; j++) {
					/* for every column */
				for (i=0; i<width; i++) {
					
						/* reset the accumulator */
					sum = 0.;
					
						/* do the convolution using the i_offset, j_offset kernels */
					for (k=0;k<dim;k++) {
						if ( ((i-i_offset[k])>=0) 
								&& ((j-j_offset[k])>=0) 
								&& ((i-i_offset[k])<width) 
								&& ((j-j_offset[k])<height)) {
							sum += (*(buffer + (j-j_offset[k])*width + (i-i_offset[k])))*coefficient[k];
						}
					}
					
						/* write the convolved pixel to the destination image */
					*res_buffer = sum;
						/* advance to next pixel */
					res_buffer++;
				}
			}
		}
	}
	
	return mimg_res;	/* return the result image */
}

/**************************************************************************************
/*	Function:		minvolve_mimage
/*	Description:	create an image as a min convolution of another
/*
/*	Parameters:
/*		<-	mimg		source image
/*		<-	i_offset	array of offsets for x
/*		<-	j_offset	array of offsets for y
/*		<-	coefficient	array of coefficients
/*		<-	dim			kernel size
/*
/*	Result:
/*		not NULL	a pointer to the convolved image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr minvolve_mimage(mimage_ptr mimg, int *i_offset, int *j_offset, float *coefficient, size_t dim)
{
	unsigned char *buffer, *res_buffer;
	float *fbuffer, *fres_buffer;
	
	float sum;
	int i, j, c, k;
	mimage_ptr mimg_res;
	
	int width, height, planes_number, kind;
	size_t type_size;

		/* get information from source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);	/* get the size of a pixel in the source image */

		/* create the destination image */
	mimg_res = create_mimage(width, height, planes_number, kind, DENSE_IMAGE);
	if (!mimg_res)
		return NULL;

	if (kind == FLOAT_IMAGE) {	/* if float image */

		for (c=0; c<planes_number; c++) {	/* for each plane */

			fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;	/* get the plane of the original image */
			fres_buffer = get_fbyteplane_from_mimage(mimg_res, c)->buffer;	/* get the corresponding result plane */

			for (j=0; j<height; j++) {		/* iterate over the lines */
				for (i=0; i<width; i++) {	/* iterate over the pixels */
					
					sum = BIGFLOAT; /* (*(fbuffer + (j-j_offset[0])*width + (-i_offset[0]))) * coefficient[0]; */
					
					for (k=0;k<dim;k++) {
						if ( ((i-i_offset[k])>=0) 
								&& ((j-j_offset[k])>=0) 
								&& ((i-i_offset[k])<width) 
								&& ((j-j_offset[k])<height)) {
							sum = min ( sum, (*(fbuffer + (j-j_offset[k])*width + (i-i_offset[k]))) * coefficient[k]);
						}
					}
					
						/* write the convolved pixel to the destination image */
					*fres_buffer = sum;
						/* advance to next pixel */
					fres_buffer++;
				}
			}
		}
	} else {	/* if discrete image */

		for (c=0; c<planes_number; c++) {	/* for each plane */

			buffer = get_byteplane_from_mimage(mimg, c)->buffer;	/* get the plane of the original image */
			res_buffer = get_byteplane_from_mimage(mimg_res, c)->buffer;	/* get the corresponding result plane */

			for (j=0; j<height; j++) {		/* iterate over the lines */
				for (i=0; i<width; i++) {	/* iterate over the pixels */
					
					sum = BIGBYTE; /* (*(buffer + (j-j_offset[0])*width + (-i_offset[0]))) * coefficient[0]; */
					
					for (k=0;k<dim;k++) {
						if ( ((i-i_offset[k])>=0) 
								&& ((j-j_offset[k])>=0) 
								&& ((i-i_offset[k])<width) 
								&& ((j-j_offset[k])<height)) {
							sum = min ( sum, (*(buffer + (j-j_offset[k])*width + (i-i_offset[k]))) * coefficient[k]);
						}
					}
					
						/* write the convolved pixel to the destination image */
					*res_buffer = sum;
						/* advance to next pixel */
					res_buffer++;
				}
			}
		}
	}
	
	return mimg_res;	/* return the destination image */
}

/**************************************************************************************
/*	Function:		maxvolve_mimage
/*	Description:	create an image as a max convolution of another
/*
/*	Parameters:
/*		<-	mimg		source image
/*		<-	i_offset	array of offsets for x
/*		<-	j_offset	array of offsets for y
/*		<-	coefficient	array of coefficients
/*		<-	dim			kernel size
/*
/*	Result:
/*		not NULL	a pointer to the convolved image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr maxvolve_mimage(mimage_ptr mimg, int *i_offset, int *j_offset, float *coefficient, size_t dim)
{
	unsigned char *buffer, *res_buffer;
	float *fbuffer, *fres_buffer;
	
	float sum;
	int i, j, c, k;
	mimage_ptr mimg_res;
	
	int width, height, planes_number, kind;
	size_t type_size;

		/* get information from source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);	/* get the size of a pixel in the source image */

		/* create the destination image */
	mimg_res = create_mimage(width, height, planes_number, kind, DENSE_IMAGE);
	if (!mimg_res)
		return NULL;

	if (kind == FLOAT_IMAGE) {	/* if float image */

		for (c=0; c<planes_number; c++) {	/* for each plane */

			fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;	/* get the plane of the original image */
			fres_buffer = get_fbyteplane_from_mimage(mimg_res, c)->buffer;	/* get the corresponding result plane */

			for (j=0; j<height; j++) {		/* iterate over the lines */
				for (i=0; i<width; i++) {	/* iterate over the pixels */
					
					sum = BIGNEGFLOAT; /* (*(fbuffer + (j-j_offset[0])*width + (-i_offset[0]))) * coefficient[0]; */
					
					for (k=0;k<dim;k++) {
						if ( ((i-i_offset[k])>=0) 
								&& ((j-j_offset[k])>=0) 
								&& ((i-i_offset[k])<width) 
								&& ((j-j_offset[k])<height)) {
							sum = max ( sum, (*(fbuffer + (j-j_offset[k])*width + (i-i_offset[k]))) * coefficient[k]);
						}
					}
					
						/* write the convolved pixel to the destination image */
					*fres_buffer = sum;
						/* advance to next pixel */
					fres_buffer++;
				}
			}
		}
	} else {	/* if discrete image */

		for (c=0; c<planes_number; c++) {	/* for each plane */

			buffer = get_byteplane_from_mimage(mimg, c)->buffer;	/* get the plane of the original image */
			res_buffer = get_byteplane_from_mimage(mimg_res, c)->buffer;	/* get the corresponding result plane */

			for (j=0; j<height; j++) {		/* iterate over the lines */
				for (i=0; i<width; i++) {	/* iterate over the pixels */
					
					sum = (mpixel)0; /* (*(buffer + (j-j_offset[0])*width + (-i_offset[0]))) * coefficient[0]; */
					
					for (k=0;k<dim;k++) {
						if ( ((i-i_offset[k])>=0) 
								&& ((j-j_offset[k])>=0) 
								&& ((i-i_offset[k])<width) 
								&& ((j-j_offset[k])<height)) {
							sum = max ( sum, (*(buffer + (j-j_offset[k])*width + (i-i_offset[k]))) * coefficient[k]);
						}
					}
					
						/* write the convolved pixel to the destination image */
					*res_buffer = sum;
						/* advance to next pixel */
					res_buffer++;
				}
			}
		}
	}
	
	return mimg_res;	/* return the destination image */
}

/**************************************************************************************
/*	Function:		threshold_mimage
/*	Description:	create an image as a threshold from another
/*
/*	Parameters:
/*		<-	mimg			source image
/*		<-	fvalues			array of thresholds
/*		<-	fvalues_size	size of the array of thresholds
/*
/*	Result:
/*		not NULL	a pointer to the thresholded image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr threshold_mimage(mimage_ptr mimg, float *fvalues, size_t fvalues_size)
{
	if (mimg->density == DENSE_IMAGE)
		return threshold_mimage_dense(mimg, fvalues, fvalues_size);
	else
		return threshold_mimage_sparse(mimg, fvalues, fvalues_size);
}

/**************************************************************************************
/*	Function:		histogram_mimage
/*	Description:	return an histogram of the pixel values from an image
/*
/*	Parameters:
/*		<-	mimg			source image
/*		<-	fvalues			array of points
/*		<-	fvalues_size	size of the array of points
/*
/*	Result:
/*		not NULL	a pointer to the histogram array
/*		NULL		error condition
/*
/***************************************************************************************/

size_t *histogram_mimage(mimage_ptr mimg, float *fvalues, size_t fvalues_size)
{
	if (mimg->density == DENSE_IMAGE)
		return histogram_mimage_dense(mimg, fvalues, fvalues_size);
	else
		return histogram_mimage_sparse(mimg, fvalues, fvalues_size);
}

/**************************************************************************************
/*	Function:		perlin_mimage
/*	Description:	return an image with a sum pixels values over all
/*					pixels to the left and above at given point
/*
/*	Parameters:
/*		<-	mimg	ource image
/*
/*	Result:
/*		not NULL	a pointer to the perlin image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr perlin_mimage(mimage_ptr mimg)
{
	unsigned char *buffer, *perlin_buffer;
	float *fbuffer, *fperlin_buffer;
	mimage_ptr mimg_perlin;
	
	unsigned char sum, val;
	float fsum, fval;
	
	int i, j, c;
	
	int width, height, planes_number, kind;
	size_t type_size;

		/* get information from source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);

		/* create the destination image */
	mimg_perlin = create_mimage(width, height, planes_number, kind, DENSE_IMAGE);
	if (!mimg_perlin)
		return NULL;

	if (kind == FLOAT_IMAGE) {	/* if float image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {

			fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;	/* source buffer */
			fperlin_buffer = get_fbyteplane_from_mimage(mimg_perlin, c)->buffer;	/* target buffer */
	
				/* the perlin value across the bottom row should be 
				   the sum of all values to the left */
			for (i=0, fsum = 0.; i<width; i++) {
				fsum += fbuffer[i];
				fperlin_buffer[i] = fsum;
			}
			
				/* the perlin value up the left column should be 
				   the sum of all values above */
			for (j=0, fsum = 0.; j<height; j++) {
				fsum += fbuffer[j*width];
				fperlin_buffer[j*width] = fsum; 
			}
			
				/* the iterative formula seen below gives 
				   the remaining perlin values */
			for (j=1; j<height; j++) {
				for (i=1; i<width; i++) {
					
					fval = fbuffer[i + j*width] 
						+ fperlin_buffer[ i-1 + j*width ] 
						+ fperlin_buffer[ i + (j-1)*width ]
						- fperlin_buffer[ i-1 + (j-1)*width ];
					
					fperlin_buffer[i + j*width] = fval;
				}
			}
		}
	} else {	/* if discrete image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {

			buffer = get_byteplane_from_mimage(mimg, c)->buffer;	/* source buffer */
			perlin_buffer = get_byteplane_from_mimage(mimg_perlin, c)->buffer;	/* target buffer */

				/* the perlin value across the bottom row should be 
				   the sum of all values to the left */
			for (i=0, sum = 0; i<width; i++) {
				sum += buffer[i];
				perlin_buffer[i] = sum;
			}

				/* the perlin value up the left column should be 
				   the sum of all values above */
			for (j=0, sum = 0; j<height; j++) {
				sum += buffer[j*width];
				perlin_buffer[j*width] = sum; 
			}

				/* the perlin value up the left column should be 
				   the sum of all values above */
			for (j=1; j<height; j++) {
				for (i=1; i<width; i++) {
					
					val = buffer[i + j*width] 
						+ perlin_buffer[ i-1 + j*width ] 
						+ perlin_buffer[ i + (j-1)*width ]
						- perlin_buffer[ i-1 + (j-1)*width ];
					
					perlin_buffer[i + j*width] = val;
				}
			}
		}
	}
	
	return mimg_perlin;
}

/**************************************************************************************
/*	Function:		perlin2_mimage
/*	Description:	return an image with a sum pixels values over all
/*					pixels to the left and above at given point (variant 2)
/*
/*	Parameters:
/*		<-	mimg	source image
/*
/*	Result:
/*		not NULL	a pointer to the perlin 2 image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr perlin2_mimage(mimage_ptr mimg)
{
	unsigned char *buffer, *perlin_buffer;
	float *fbuffer, *fperlin_buffer;
	mimage_ptr mimg_perlin;
	
	unsigned char val, sum;
	float fval, fsum;
	
	int i, j, c;
	
	int width, height, planes_number, kind;
	size_t type_size;

		/* get information from source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);

		/* create the destination image */
	mimg_perlin = create_mimage(width, height, planes_number, kind, DENSE_IMAGE);
	if (!mimg_perlin)
		return NULL;

	if (kind == FLOAT_IMAGE) {	/* if float image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {

			fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;	/* source buffer */
			fperlin_buffer = get_fbyteplane_from_mimage(mimg_perlin, c)->buffer;	/* target buffer */
	
			for (i=0; i<width; i++) 
				fperlin_buffer[i] = fbuffer[i];

			for (i=1; i<width; i++) 
				fperlin_buffer[i + width] = fbuffer[i + width] + fbuffer[i] + fbuffer[i - 1];
			
				/* the perlin value up the left column should be 
				   the sum of all values above */
			for (j=0, fsum = 0.; j<height; j++) {
				fsum += fbuffer[j*width];
				fperlin_buffer[j*width] = fsum; 
			}
		
			for (j=2; j<height; j++) {
				for (i=1; i<width; i++) {
					
					fval = fbuffer[i + j*width] 
						+ fperlin_buffer[ i-1 + (j-1)*width ] 
						+ fperlin_buffer[ i + (j-1)*width ]
						- fperlin_buffer[ i-1 + (j-2)*width ];
					
					fperlin_buffer[i + j*width] = fval;
				}
			}
		}
	} else {	/* if discrete image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {

			buffer = get_byteplane_from_mimage(mimg, c)->buffer;	/* source buffer */
			perlin_buffer = get_byteplane_from_mimage(mimg_perlin, c)->buffer;	/* target buffer */
	
			for (i=0; i<width; i++) 
				perlin_buffer[i] = buffer[i];

			for (j=0, sum = 0; j<height; j++) {
				sum += buffer[j*width];
				perlin_buffer[j*width] = sum; 
			}

			for (i=0; i<width; i++) 
				perlin_buffer[i] = buffer[i];

			for (i=1; i<width; i++) 
				perlin_buffer[i + width] = buffer[i + width] + buffer[i] + buffer[i - 1];
				
			for (j=2; j<height; j++) {
				for (i=1; i<width; i++) {
					
					val = buffer[i * j*width] 
						+ perlin_buffer[ i-1 + (j-1)*width ] 
						+ perlin_buffer[ i + (j-1)*width ]
						- perlin_buffer[ i-1 + (j-2)*width ];
					
					perlin_buffer[i + j*width] = val;
				}
			}
		}
	}
	
	return mimg_perlin;	/* return the destination image */
}

/**************************************************************************************
/*	Function:		anaglyph_mimage
/*	Description:	create an anaglyph (red and blue) image
/*					from a z-buffer represented by an image
/*
/*	Parameters:
/*		<-	mimg		source image
/*		<-	scale		scale factor
/*		<-	dens		fraction factor
/*
/*	Result:
/*		not NULL	a pointer to the generated image
/*		NULL		error condition
/*
/***************************************************************************************/

#ifndef OLD_ANAGLYPH

mimage_ptr anaglyph_mimage(mimage_ptr mimg, float scale, float dens)
{
	int i = 0, j, l;
	int offs;
	int randbit, redc, cyanc;
	mimage_ptr result_mimg;
	image_dimension dim;
	
	float fvalues[] = {FLOAT_BLACK, FLOAT_BLACK, FLOAT_BLACK};
	float values[] = {DISCRETE_BLACK, DISCRETE_BLACK, DISCRETE_BLACK};

	int height, width, comp, kind;
	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);

		/* works only with 1 plane z-buffers */
	if (comp != GRAY_COMP)
		return NULL;

		/* create a constant image to anaglyph into */
	dim.x = width;
	dim.y = height;

	result_mimg = const_mimage( IS_FLOAT(mimg) ? fvalues : values, &dim, kind, RGB_COMP);
	if (!result_mimg) return NULL;

	if (!result_mimg)
		return NULL;

	l = height * width;

	if (kind == FLOAT_IMAGE) {	/* if is float */
	
		fmpixel_ptr src_buffer = get_fbyteplane_from_mimage(mimg, SINGLE_GRAY_OFFSET)->buffer;

		fmpixel_ptr dst_buffer_red = get_fbyteplane_from_mimage(result_mimg, RED_OFFSET)->buffer;
		fmpixel_ptr dst_buffer_green = get_fbyteplane_from_mimage(result_mimg, GREEN_OFFSET)->buffer;
		fmpixel_ptr dst_buffer_blue = get_fbyteplane_from_mimage(result_mimg, BLUE_OFFSET)->buffer;
		
		for (;l;l--) {

			offs = *src_buffer * scale;

			randbit = ((float)rand() / (float)RAND_MAX) > dens ?	1 : 0;
			
			if (offs < 0) {

				redc = randbit ? FLOAT_BLACK : FLOAT_WHITE;
				cyanc = i+offs >= 0 ? *(dst_buffer_red + offs) : redc;

			} else if (offs > 0) {

				cyanc = randbit ? FLOAT_BLACK : FLOAT_WHITE;
				redc = i+offs >= 0 ? *(dst_buffer_blue - offs) : cyanc;

			} else {

				cyanc = randbit ? FLOAT_BLACK : FLOAT_WHITE;
				redc = randbit ? FLOAT_BLACK : FLOAT_WHITE;
			}

			*dst_buffer_red++ = redc;
			*dst_buffer_green++ = cyanc;
			*dst_buffer_blue++ = cyanc;

			src_buffer++;
			i = (i+1) % width;
		}	

	
	
	} else { /* if is discrete */
	
		mpixel_ptr src_buffer = get_byteplane_from_mimage(mimg, SINGLE_GRAY_OFFSET)->buffer;

		mpixel_ptr dst_buffer_red = get_byteplane_from_mimage(result_mimg, RED_OFFSET)->buffer;
		mpixel_ptr dst_buffer_green = get_byteplane_from_mimage(result_mimg, GREEN_OFFSET)->buffer;
		mpixel_ptr dst_buffer_blue = get_byteplane_from_mimage(result_mimg, BLUE_OFFSET)->buffer;
		
		for (;l;l--) {

			offs = *src_buffer * scale;

			randbit = ((float)rand() / (float)RAND_MAX) > dens ?	1 : 0;
			
			if (offs < 0) {

				redc = randbit ? DISCRETE_BLACK : DISCRETE_WHITE;
				cyanc = i+offs >= 0 ? *(dst_buffer_red + offs) : redc;

			} else if (offs > 0) {

				cyanc = randbit ? DISCRETE_BLACK : DISCRETE_WHITE;
				redc = i+offs >= 0 ? *(dst_buffer_blue - offs) : cyanc;

			} else {

				cyanc = randbit ? DISCRETE_BLACK : DISCRETE_WHITE;
				redc = randbit ? DISCRETE_BLACK : DISCRETE_WHITE;
			}

			*dst_buffer_red++ = redc;
			*dst_buffer_green++ = cyanc;
			*dst_buffer_blue++ = cyanc;

			src_buffer++;
			i = (i+1) % width;
		}	
	
	}
	
	return result_mimg;	

}

#else /* OLD_ANAGLYPH */
mimage_ptr anaglyph_mimage(mimage_ptr mimg, float scale, float fraction)
{
	size_t i, j, m;

	const int right_side_margin = 100;

	int current_height, desired_height;
	int loc_in_red, loc_in_green;

	int difference;

	float *red_fbuffer, *green_fbuffer, *fbuffer;
	unsigned char *red_buffer, *green_buffer, *buffer;

	mimage_ptr result_mimg;
	image_dimension dim;

	int height, width, comp, kind;
	float fvalues[] = {FLOAT_BLACK, FLOAT_BLACK, FLOAT_BLACK};
	
		/* get source image info */
	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);
	
		/* calculate the size of the destination image */
	dim.x = width + right_side_margin;
	dim.y = height;

		/* works only with 1 plane z-buffers */
	if (comp != GRAY_COMP)
		return NULL;

		/* create a constant image to anaglyph into */
	result_mimg = const_mimage(fvalues, &dim, kind, RGB_COMP);
	if (!result_mimg)
		return NULL;

	if (kind == FLOAT_IMAGE) {	/* if is float */

		/* red = 0, green = 1, blue = 2 */
		
			/* get the pointer to the three colors */
		fbuffer = get_plane_buffer(mimg->planes_array[SINGLE_GRAY_OFFSET], kind);
		red_fbuffer = get_plane_buffer(result_mimg->planes_array[RED_OFFSET], kind);
		green_fbuffer = get_plane_buffer(result_mimg->planes_array[GREEN_OFFSET], kind);

			/* for each line */
		for (j=0; j<height; j++) {

			current_height = 0;				/* set the height to 0 */
			loc_in_red = loc_in_green = 0;	/* set the position of red and green to 0 */
		
				/* for each column */
			for (i=0; i<width; i++) {
				
					/* 
					 * calculate the desired height. 
					 * obtained as product of the value in the z-buffer times
					 * the scale factor
					 */
				
				desired_height = (*fbuffer) * scale;
				
					/* if the height is greater than desired */
				if (current_height > desired_height) {

						/* calc the difference between current and desired height */
					difference = current_height - desired_height;

						/* and store the anaglyph-ed data in to the red buffer */
					for (m=0; m<difference; m++) {
						if (((float)rand() / (float)RAND_MAX) < fraction) 
							*(red_fbuffer + dim.x * j + loc_in_red) = FLOAT_WHITE;
						
						if (loc_in_red < (dim.x-1))
							loc_in_red ++;
							
					}
					
					current_height -= difference;	/* update the current height */
										
					/* if the height is smaller than desired */
				} else if (current_height < desired_height) {

						/* calc the difference between current and desired height */
					difference = desired_height - current_height;

						/* and store the anaglyph-ed data in to the green buffer */
					for (m=0; m<difference; m++) {
						if (((float)rand() / (float)RAND_MAX) < fraction) 
							*(green_fbuffer + dim.x * j + loc_in_green) = FLOAT_WHITE;

						if (loc_in_green < (dim.x-1))
							loc_in_green ++;
					}
								
					current_height += difference;	/* update the current height */
					
				}
							
					/* and store the anaglyph-ed data in to both the buffer */
				if (((float)rand() / (float)RAND_MAX) < fraction) {
					*(red_fbuffer + dim.x * j + loc_in_red) = FLOAT_WHITE;
					*(green_fbuffer + dim.x * j + loc_in_green) = FLOAT_WHITE;
				}

				if (loc_in_red < (dim.x-1))
					loc_in_red++;

				if (loc_in_green < (dim.x-1))
					loc_in_green++;
				
					/* advance the position in buffer */
				fbuffer++;		
			}
		}		
	} else {	/* if is discrete */
		/* red = 0, green = 1, blue = 2 */
		
			/* get the pointer to the three colors */
		buffer = get_plane_buffer(mimg->planes_array[SINGLE_GRAY_OFFSET], kind);
		red_buffer = get_plane_buffer(result_mimg->planes_array[RED_OFFSET], kind);
		green_buffer = get_plane_buffer(result_mimg->planes_array[GREEN_OFFSET], kind);

			/* for each line */
		for (j=0; j<height; j++) {

			current_height = 0;				/* set the height to 0 */
			loc_in_red = loc_in_green = 0;	/* set the position of red and green to 0 */
		
				/* for each column */
			for (i=0; i<width; i++) {
				
					/* 
					 * calculate the desired height. 
					 * obtained as product of the value in the z-buffer times
					 * the scale factor
					 */
				
				desired_height = (*buffer) * scale;
				
					/* if the height is greater than desired */
				if (current_height > desired_height) {

						/* calc the difference between current and desired height */
					difference = current_height - desired_height;

						/* and store the anaglyph-ed data in to the red buffer */
					for (m=0; m<difference; m++) {
						if (((float)rand() / (float)RAND_MAX) < fraction) 
							*(red_buffer + dim.x * j + loc_in_red) = DISCRETE_WHITE;
						
						if (loc_in_red < (dim.x-1))
							loc_in_red ++;
							
					}
					
					current_height -= difference;	/* update the current height */
										
					/* if the height is smaller than desired */
			} else if (current_height < desired_height) {
		
						/* calc the difference between current and desired height */
					difference = desired_height - current_height;

						/* and store the anaglyph-ed data in to the green buffer */
					for (m=0; m<difference; m++) {
						if (((float)rand() / (float)RAND_MAX) < fraction) 
							*(green_buffer + dim.x * j + loc_in_green) = DISCRETE_WHITE;

						if (loc_in_green < (dim.x-1))
							loc_in_green ++;
					}
								
					current_height += difference;	/* update the current height */
					
				}
							
					/* and store the anaglyph-ed data in to both the buffer */
				if (((float)rand() / (float)RAND_MAX) < fraction) {
					*(red_buffer + dim.x * j + loc_in_red) = DISCRETE_WHITE;
					*(green_buffer + dim.x * j + loc_in_green) = DISCRETE_WHITE;
				}

				if (loc_in_red < (dim.x-1))
					loc_in_red++;

				if (loc_in_green < (dim.x-1))
					loc_in_green++;
				
					/* advance the position in buffer */
				buffer++;		
			}
		}		
	}	

	return result_mimg;	/* return the generated anaglyph image */
}
#endif	/* OLD_ANAGLYPH */
/**************************************************************************************
/*	Function:		get_pixel_mimage
/*	Description:	return an array with the values (if more than one plane)
/*					for a pixel at a given position in the image mimg 
/*
/*	Parameters:
/*		<-	mimg		source image
/*		<-	x			x coordinate of the pixel
/*		<-	y			y coordinate of the pixel
/*
/*	Result:
/*		not NULL	a pointer to the array (float if the image is float, else discrete)
/*		NULL		error condition
/*
/***************************************************************************************/

void *get_pixel_mimage(mimage_ptr mimg, int x, int y)
{
	if (mimg->density == DENSE_IMAGE)
		return get_pixel_mimage_dense(mimg, x, y);
	else
		return get_pixel_mimage_sparse(mimg, x, y);
}

/**************************************************************************************
/*	Function:		set_pixel_mimage
/*	Description:	set a new values for a pixel in an image
/*
/*	Parameters:
/*		<-	mimg				source image
/*		<-	x					x coordinate of the pixel
/*		<-	y					y coordinate of the pixel
/*		<-	pixel_value_array	new values for the pixels
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int set_pixel_mimage(mimage_ptr mimg, int x, int y, void *pixel_value_array)
{
	if (mimg->density == DENSE_IMAGE)
		return set_pixel_mimage_dense(mimg, x, y, pixel_value_array);
	else
		return set_pixel_mimage_sparse(mimg, x, y, pixel_value_array);
}

/**************************************************************************************
/*	Function:		flip_h
/*	Description:	create a flipped horizontally image
/*
/*	Parameters:
/*		<-	mimg	source image
/*
/*	Result:
/*		not NULL	pointer to the flipped image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr flip_h(mimage_ptr mimg)
{
	mimage_ptr res_mimg = NULL;
	
	if (mimg->density == DENSE_IMAGE)
		res_mimg = flip_h_dense(mimg);
	else
		res_mimg = flip_h_sparse(mimg);
		
	return res_mimg;
}

/**************************************************************************************
/*	Function:		flip_v
/*	Description:	create a flipped vertically image
/*
/*	Parameters:
/*		<-	mimg	source image
/*
/*	Result:
/*		not NULL	pointer to the flipped image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr flip_v(mimage_ptr mimg)
{
	mimage_ptr res_mimg = NULL;
	
	if (mimg->density == DENSE_IMAGE)
		res_mimg = flip_v_dense(mimg);
	else
		res_mimg = flip_v_sparse(mimg);
		
	return res_mimg;
}

/**************************************************************************************
/*	Function:		rotate_halfpi
/*	Description:	return a n times 90 degrees rotated image
/*	Note:			for sparse return the same image
/*
/*	Parameters:
/*		<-	mimg	source image
/*
/*	Result:
/*		not NULL	pointer to the rotated image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr rotate_halfpi(mimage_ptr mimg, short times)
{
	mimage_ptr res_mimg = NULL;

	times = times % 4;	/* angles are defined in modulo 4 space */

		/* call the right rotation routine */
	switch (times)
	{
		case 0:
			res_mimg = clone_mimage(mimg);
			break;
			
		case 1:
			if (mimg->density == DENSE_IMAGE) 
				res_mimg = rotate_90_dense(mimg);
			else
				res_mimg = rotate_90_sparse(mimg);
			break;
			
		case 2:
			if (mimg->density == DENSE_IMAGE) 
				res_mimg = rotate_180_dense(mimg);
			else
				res_mimg = rotate_180_sparse(mimg);
			break;
			
		case 3:
			if (mimg->density == DENSE_IMAGE)
				res_mimg = rotate_270_dense(mimg);
			else
				res_mimg = rotate_270_sparse(mimg);
			break;
		
	}

	return res_mimg;
}

/**************************************************************************************
/*	Function:		random_mimage
/*	Description:	random fill the pixels of an image
/*
/*	Parameters:
/*		<-	mimg	source image
/*
/*	Result:
/*		not NULL	a pointer to the result image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr random_mimage(mimage_ptr mimg, float r_min, float r_max)
{
	if (IS_DENSE(mimg))
		return random_mimage_dense(mimg, r_min, r_max);
	else
		return random_mimage_sparse(mimg, r_min, r_max);
}

/**************************************************************************************
/*	Function:		invert_mimage
/*	Description:	invert the pixels of an image [in place]
/*
/*	Parameters:
/*		<->	mimg	source image
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int	invert_mimage(mimage_ptr mimg)
{
	if (IS_DENSE(mimg))
		return invert_mimage_dense(mimg);
	else
		return invert_mimage_sparse(mimg);
}

/**************************************************************************************
/*	Function:		lut_mimage
/*	Description:	apply a Look Up Table to an image
/*
/*	Parameters:
/*		<-	mimg				source image
/*		<-	LUTs				matrix with the LUT values
/*		<-	number_of_planes	number of planes in target
/*		<-	target_kind			kind of target image
/*
/*	Result:
/*		not NULL	pointer to the generated image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr lut_mimage(mimage_ptr mimg, mpixel **LUTs, int number_of_planes, int target_kind)
{
	if (IS_FLOAT(mimg))
		return NULL;
	
	if (IS_DENSE(mimg))
		return lut_mimage_dense(mimg, LUTs, number_of_planes, target_kind);
	else
		return lut_mimage_sparse(mimg, LUTs, number_of_planes, target_kind);
}

/**************************************************************************************
/*	Function:		lex_minof_mimage
/*	Description:	return the position of the lexicographically minimum of an image
/*
/*	Parameters:
/*		<-	mimg	source image
/*		->	pos		position of the pixel that is minimum
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int lex_minof_mimage(mimage_ptr mimg, gr_image_displacement *pos)
{
	int ret;
	
	if (IS_DENSE(mimg))
		ret = lex_minof_mimage_dense(mimg, pos);
	else
		ret = lex_minof_mimage_sparse(mimg, pos);
		
	return ret;		
}

/**************************************************************************************
/*	Function:		lex_maxof_mimage
/*	Description:	return the position of the lexicographically maximum of an image
/*
/*	Parameters:
/*		<-	mimg	source image
/*		->	pos		position of the pixel that is maximum
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int lex_maxof_mimage(mimage_ptr mimg, gr_image_displacement *pos)
{
	int ret;
	
	if (IS_DENSE(mimg))
		ret = lex_maxof_mimage_dense(mimg, pos);
	else
		ret = lex_maxof_mimage_sparse(mimg, pos);
		
	return ret;		
}

/**************************************************************************************
/*	Function:		lex_sort_mimage
/*	Description:	return a lexicographically sorted image from another
/*
/*	Parameters:
/*		<-	mimg	source image
/*
/*	Result:
/*		not NULL	the sorted image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr lex_sort_mimage(mimage_ptr mimg)
{
	if (IS_DENSE(mimg))
		return lex_sort_mimage_dense(mimg);
	else
		return lex_sort_mimage_sparse(mimg);
}

/**************************************************************************************
/*	Function:		split_mimage
/*	Description:	split an image in a set of sparse images for every color
/*
/*	Parameters:
/*		<-	mimg		source image
/*		->	retNumber	number of sparse images generated
/*		->	offsets		offset in the original image of the generated sparse images
/*		->	pixels_pop	population in pixels of the generated images
/*
/*	Result:
/*		not NULL	an array of splitted sparse images
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr *split_mimage(mimage_ptr mimg, int *retNumber, image_displacement_ptr *offsets, int **pixels_pop)
{
	if (IS_DENSE(mimg)) 
		return split_dense_mimage(mimg, retNumber, offsets, pixels_pop);
	else
		return split_sparse_mimage(mimg, retNumber, offsets, pixels_pop);
	
}

/**************************************************************************************
/*	Function:		tile_mimages
/*	Description:	tile a set of images with the same size in a new one
/*
/*	Parameters:
/*		<-	mimgs			source images
/*		<-	n				number of images
/*		<-	target_height	height of the target image
/*		<-	target_width	width of the target image
/*
/*	Result:
/*		not NULL	the generated image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr tile_mimages(mimage_ptr *mimgs, int n, int target_height, int target_width)
{

	int i, k;
	int x, y;
	int planes_number = 0;
	mimage_ptr target_mimg;
	int height, width, comp, kind;
	gr_image_displacement delta;
	
		/* if the number of image to tile is zero return w/ error condition */
	if (!n)
		return NULL;
		
		/* get the info for the first image */
	get_mimage_info(mimgs[0], &height, &width, &comp, &kind, NULL);
	planes_number = comp;

		/* loop over the images to tile and check if they are compatible */
	for (i=1; i<n; i++) {
		int temp_height, temp_width, temp_comp, temp_kind;
	
		get_mimage_info(mimgs[1], &temp_height, &temp_width, &temp_comp, &temp_kind, NULL);

		if (temp_height != height || temp_width != width || temp_kind != kind) 
			return NULL;

		planes_number = max(planes_number, temp_comp);
	}
	
		/* create the target image of desired size */
	target_mimg = create_mimage(target_width, target_height, planes_number, kind, DENSE_IMAGE);
	
		/* tile all source images into the target image */
	for ( delta.y = 0, k = 0; delta.y < target_height; delta.y += height ) {
		for ( delta.x = 0 ; delta.x < target_width; delta.x += width, k = (k+1)%n) {
			if (IS_SPARSE(mimgs[k]))
				stuff_sparse_over_dense(target_mimg, mimgs[k], &delta);
			else 
				stuff_mimage(target_mimg, mimgs[k], &delta);
		}
	}

		/* return the tiled image */
	return target_mimg;
}

	/*
	 *	Static functions
	 */

/**************************************************************************************
/*	Function:		check_rect_mimage
/*	Description:	check a rect area for collision with black pixels
/*	Note:			used from shrink
/*
/*	Parameters:
/*		<-	mimg		source image
/*		<-	rect		rect in the image
/*		<-	threshold	threshold for black
/*
/*	Result:
/*		0		no collision
/*		NORTH	north collision
/*		SOUTH	south collision
/*		EAST	east collision
/*		WEST	west collision
/*
/***************************************************************************************/

static unsigned char check_rect_mimage(mimage_ptr mimg, image_rect_ptr rect, int threshold)
{
	unsigned char sides_wrong = 0;

		/* set sides_wrong w/ the collision info from each side */

	sides_wrong |= (check_hline_mimage(mimg, rect->y, threshold)  ? NORTH : 0);
	sides_wrong |= (check_hline_mimage(mimg, rect->dy, threshold) ? SOUTH : 0);
	sides_wrong |= (check_vline_mimage(mimg, rect->x, threshold)  ? EAST  : 0);
	sides_wrong |= (check_vline_mimage(mimg, rect->dx, threshold) ? WEST  : 0);

	return sides_wrong;
}

/**************************************************************************************
/*	Function:		check_hline_mimage
/*	Description:	check an horizontal line for collision with black pixels
/*	Note:			used from shrink
/*
/*	Parameters:
/*		<-	mimg		source image
/*		<-	y			line in the image
/*		<-	threshold	threshold for black
/*
/*	Result:
/*		0		no collision
/*		else	collision
/*
/***************************************************************************************/

static int check_hline_mimage(mimage_ptr mimg, int y, int threshold)
{
	mpixel_ptr img_buf_ptr;
	fmpixel_ptr float_img_buf_ptr;

	char pix = 0;
	int i, c;

	int height, width, planes_number, kind; 
	
		/* get image info */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

	if (kind == FLOAT_IMAGE) {	/* if float image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get the buffer and go to the line */
			float_img_buf_ptr = get_plane_buffer(mimg->planes_array[c], kind);
			float_img_buf_ptr += (y*width);

				/* set pix as the bitwise or of all the pixels in the line */
				/* thus only black lines will have the value zero at the end */
			for (i=0; i<width; i++) {
				pix |= ((*float_img_buf_ptr) > threshold);
				float_img_buf_ptr++;
			}
		}
	} else {	/* if discrete image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get the buffer and go to the line */
			img_buf_ptr = get_plane_buffer(mimg->planes_array[c], kind);
			img_buf_ptr += (y*width);

				/* set pix as the bitwise or of all the pixels in the line */
				/* thus only black lines will have the value zero at the end */
			for (i=0; i<width; i++) {
				pix |= ((*img_buf_ptr) > threshold);
				img_buf_ptr++;
			}
		}
	}  

	return !pix;	/* if the line is black return 1 else 0 */
}

/**************************************************************************************
/*	Function:		check_vline_mimage
/*	Description:	check a vertical line for collision with black pixels
/*	Note:			used from shrink
/*
/*	Parameters:
/*		<-	mimg		source image
/*		<-	x			column in the image
/*		<-	threshold	threshold for black
/*
/*	Result:
/*		0		no collision
/*		else	collision
/*
/***************************************************************************************/

static int check_vline_mimage(mimage_ptr mimg, int x, int threshold)
{
	mpixel_ptr img_buf_ptr;
	fmpixel_ptr float_img_buf_ptr;
	int step;

	char pix = 0;
	int i, c;

	int height, width, planes_number, kind; 

		/* get image info */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* step between two pixels in the same column */
	step = width;

	if (kind == FLOAT_IMAGE) {	/* if float image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get the buffer and go to the column */
			float_img_buf_ptr = get_plane_buffer(mimg->planes_array[c], kind);
			float_img_buf_ptr += x;

				/* set pix as the bitwise or of all the pixels in the column */
				/* thus only black columns will have the value zero at the end */
			for (i=0; i<height; i++) {
				pix |= ((*float_img_buf_ptr) > threshold);
				float_img_buf_ptr += step;
			}
		}

	} else {	/* if discrete image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get the buffer and go to the line */
			img_buf_ptr = get_plane_buffer(mimg->planes_array[c], kind);
			img_buf_ptr += x;

				/* set pix as the bitwise or of all the pixels in the column */
				/* thus only black columns will have the value zero at the end */
			for (i=0; i<height; i++) {
				pix |= ((*img_buf_ptr) > threshold);
				img_buf_ptr += step;
			}
		}
	}  

	return !pix;	/* if the column is black return 1 else 0 */
}

/**************************************************************************************
/*	Function:		fast_fset_mimage
/*	Description:	set fast the pixels of a float image
/*
/*	Parameters:
/*		<-	mimg		source image
/*		<-	fvalue		value to set
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

static void fast_fset_mimage(mimage_ptr mimg, float fvalue)
{
	int c, i;
	float *fbuffer;
	int width, height, planes_number, kind;
	size_t type_size;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);

	for (c=0; c<planes_number; c++) {		/* for every plane */
		fbuffer = get_fbyteplane_from_mimage(mimg, 0)->buffer;
		for (i=0; i<width*height; i++)		/* for every pixel */
			fbuffer[i] = fvalue;			/* set the pixel w/ the required value */
	}
}

/**************************************************************************************
/*	Function:		fast_set_mimage
/*	Description:	set fast the pixels of an discrete image
/*
/*	Parameters:
/*		<-	mimg		source image
/*		<-	fvalue		value to set
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

static void fast_set_mimage(mimage_ptr mimg, unsigned char value)
{
	int c;
	size_t plane_size;
	void *buffer;
	int width, height, planes_number, kind;
	size_t type_size;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);
	plane_size = width * height * type_size;

	for (c=0; c<planes_number; c++) {	/* for every plane */
		buffer = get_byteplane_from_mimage(mimg, 0)->buffer;
		memset(buffer, value, plane_size);	/* set all the pixels to the required value */
	}
}

/**************************************************************************************
/*	Function:		fast_scale_mimage
/*	Description:	scale an image with no interpolation
/*
/*	Parameters:
/*		<-	mimg			source image
/*		<-	new_dim_ptr		new dimensions
/*
/*	Result:
/*		not NULL	a pointer to the scaled image
/*		NULL		error condition
/*
/***************************************************************************************/

static mimage_ptr fast_scale_mimage(mimage_ptr mimg, image_dimension_ptr new_dim_ptr)
{
	int c, i, j, o_i, o_j;
	mimage_ptr scaled_mimg;
	
	int height, width, planes_number, kind; 
	size_t pixels_count_plan;
	size_t type_size;
	 
	unsigned char *buffer, *src_buffer;
	float *float_buffer, *float_src_buffer;

	float xratio, yratio;
	
		/* get info about source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
		/* get number of pixels in the source image */
	pixels_count_plan = height * width;
		/* get the size of a pixel in the source image */
	type_size = GET_TYPE_SIZE(kind);

		/* create the scaled image */
	scaled_mimg = create_mimage(new_dim_ptr->x, new_dim_ptr->y, planes_number, kind, DENSE_IMAGE);
	if (!scaled_mimg)
		return NULL;
		
		/* get the ratio between the two images */
	xratio = width / (float)new_dim_ptr->x;
	yratio = height / (float)new_dim_ptr->y;
	
	if (kind == FLOAT_IMAGE) {	/* if float image */
			
			/* loop over planes */
		for (c=0; c<planes_number; c++) {
			
				/* get source and destination buffers */
			float_buffer = get_plane_buffer(scaled_mimg->planes_array[c], kind);
			float_src_buffer = get_plane_buffer(mimg->planes_array[c], kind);
			
				/* for evey line in destination */
			for (j=0; j<new_dim_ptr->y; j++) {
					/* get the y coordinate of the corresponding pixel in the source image */
				o_j = j * yratio;
					/* for evey column in destination */
				for (i=0; i<new_dim_ptr->x; i++) {
							/* get the x coordinate of the corresponding pixel in the source image */
						o_i =  i * xratio;			

							/* and write that pixel in the new image */
						*float_buffer = *(float_src_buffer + o_j * width + o_i);
							/* advance buffer position in target image*/
						float_buffer++;
				}
			}
		}
	} else {	/* if discrete image */

			/* loop over planes */
		for (c=0; c<planes_number; c++) {
			
				/* get source and destination buffers */
			buffer = get_plane_buffer(scaled_mimg->planes_array[c], kind);
			src_buffer = get_plane_buffer(mimg->planes_array[c], kind);
			
				/* for evey line in destination */
			for (j=0; j<new_dim_ptr->y; j++) {
					/* get the y coordinate of the corresponding pixel in the source image */
				o_j = j * yratio;
					/* for evey column in destination */
				for (i=0; i<new_dim_ptr->x; i++) {
							/* get the x coordinate of the corresponding pixel in the source image */
						o_i =  i * xratio;			

							/* and write that pixel in the new image */
						*buffer = *(src_buffer + o_j * width + o_i);
							/* advance buffer position in target image*/
						buffer++;
				}
			}
		}
	}
	
	return scaled_mimg;
}

/**************************************************************************************
/*	Function:		create_empty_clone
/*	Description:	clone the shape of an image
/*
/*	Parameters:
/*		<-	mimg	source image
/*
/*	Result:
/*		not NULL	a pointer to the shape-cloned image
/*		NULL		error condition
/*
/***************************************************************************************/

static mimage_ptr create_empty_clone(mimage_ptr mimg)
{
	int width, height, planes_number, kind;
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

	return create_mimage(width, height, planes_number, kind, mimg->density);
}
