#ifndef __BITPLNS_H__
#define __BITPLNS_H__

#include <stdlib.h>

#ifdef macintosh
#include "gr_mac_debug.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/* Types definitions */

typedef unsigned char *gr_ptr;

#ifndef mpixel
typedef unsigned char mpixel;
#endif

#ifndef mpixel_ptr
typedef unsigned char * mpixel_ptr;
#endif

#ifndef fmpixel
typedef float fmpixel;
#endif

#ifndef fmpixel_ptr
typedef float * fmpixel_ptr;
#endif

#ifndef int32
typedef int int32;
#endif

/* Constants definition */

typedef enum {
	FLOAT_IMAGE		= 1,
	DISCRETE_IMAGE	= 2		/* !!! use this for not float images !!! */
} image_t;

typedef enum {
	GRAY_CHANNELS	= 1,
 	RGB_CHANNELS	= 3
} channels_number_t;

typedef enum {
	DENSE_IMAGE  = 0,
	SPARSE_IMAGE = 1
} density_t;

#define GET_TYPE_SIZE(a) ((a) == FLOAT_IMAGE ? sizeof(fmpixel) : sizeof(mpixel))

/* Constant value definitions */

#define GRAY_COMP			0x01
#define RGB_COMP			0x03

#define RED_OFFSET			0x00
#define GREEN_OFFSET		0x01
#define BLUE_OFFSET			0x02

#define SINGLE_GRAY_OFFSET	0x00

#define FLOAT_WHITE			((float)255.)
#define FLOAT_BLACK			((float)0.)

#define DISCRETE_WHITE		((mpixel)255)
#define DISCRETE_BLACK		((mpixel)0)


#ifndef BIGFLOAT
#define BIGFLOAT			1.e+38
#endif
#ifndef BIGNEGFLOAT
#define BIGNEGFLOAT			(-(BIGFLOAT))
#endif

#ifndef BIGBYTE
#define BIGBYTE				((unsigned char)(0xff))
#endif

#ifndef MAX_UCHAR
#define MAX_UCHAR			256
#endif

#define NORTH				0x01
#define SOUTH				0x02
#define EAST				0x04
#define WEST				0x08

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifndef SMART_FREE
#define SMART_FREE(a)		if((a)) free((a))
#endif

/* Types */

typedef struct {

  int height;
  int width;

  mpixel_ptr buffer;
  mpixel default_value;
  
} byteplane, *byteplane_ptr, sparse_byteplane, *sparse_byteplane_ptr;

typedef struct {

  int height;
  int width;

  fmpixel_ptr buffer;
  fmpixel default_value;
  
} fbyteplane, *fbyteplane_ptr, sparse_fbyteplane, *sparse_fbyteplane_ptr;

typedef struct {
		size_t starting_pixel_number;
		size_t number_of_pixels;
		int position_in_buffer;		
} section_data, *section_data_ptr;			/* contains three items for each section of the sparse image:
											 *  (1) is mpixel number of the first section;
											 *  (2) number of pixels of the section;
											 *  (3) equals -1 if pixels have the default color value, 
											 *		otherwise position of first mpixel in buffer.
											 *			NOTE:	the position is mpixel relative not byte-relative
											 *					so when position is 4 for a float mpixel the offset
											 *					in byte is 4*sizeof(float) = 8
											 */

typedef struct {
		/* common package part */
	int32 use_count;				/* setl use count					*/
	int32 type;					/* setl type field					*/

		/* private package part */
	char kind;						/* float or discrete				*/
	int  comp;						/* number of planes					*/
	byteplane_ptr *planes_array;		/* pointer to an array of planes	*/

		/* SPARSE_IMAGE PART */
		
	unsigned char density;	/* sparse or dense					*/

	size_t upperleft_x;
	size_t upperleft_y;		/* corner of minimal rectangle		*/
							/* containing the sparse image,		*/
							/* relative to a nominal larger		*/
							/* rectangle in wich it is embedded */

	size_t data_length;		/* is the length of the row_start array */
	size_t section_length;	/* is the length of the section array */

	int *row_start;		/* points to the start in the row_section array of the section 
							* describing the pieces of a row. The rows appear in order but
							* not necessarily successively since blank rows are handled in
							* a compressed way.
							*
							* If the value is < 0 than this indicates that one or more rows is empty:
			 				*	- if just one row is empty we place a -1
			 				*	- if n successive rows are empty we place a -n
							*/
							
	section_data *row_section;	/* contains three items for each section of the sparse image:
				  						(1) is mpixel number of the first section;
										(2) number of pixels of the section;
										(3) equals -1 if pixels have the default color value, 
											otherwise position of first mpixel in buffer.
								 */	

	size_t	buffer_length;	/* size of every buffer (number of pixels) */
							/*	NOTE: used only for sparse images */
} mimage, *mimage_ptr;



	/*
	 *	Navigation Macros
	 */

#define IS_A_SKIPPING_ROW_BY_IDX(mimg, idx)			((mimg)->row_data[(idx)] < 0)

#define FIRST_SECTION_PIXEL_BY_IDX(mimg, idx)		((mimg)->row_section[(idx)].starting_pixel_number)
#define LAST_SECTION_PIXEL_BY_IDX(mimg, idx)		(FIRST_SECTION_PIXEL_BY_IDX((mimg),(idx)) + ((mimg)->row_section[(idx)].number_of_pixels) - 1)
#define IS_DEFAULT_SECTION_BY_IDX(mimg, idx)		((mimg)->row_section[(idx)].position_in_buffer < 0)

#define IS_A_SKIPPING_ROW_BY_IDX(mimg, idx)			((mimg)->row_data[(idx)] < 0)

#define FIRST_SECTION_PIXEL(sect)					((sect)->starting_pixel_number)
#define LAST_SECTION_PIXEL(sect)					(FIRST_SECTION_PIXEL((sect)) + (sect)->number_of_pixels - 1)
#define IS_DEFAULT_SECTION(sect)					((sect)->position_in_buffer < 0)

	/*	
	 *	Query Macros
	 */

#define IS_SPARSE(mimg)						((mimg)->density == SPARSE_IMAGE)
#define IS_DENSE(mimg)						((mimg)->density == DENSE_IMAGE)
#define IS_FLOAT(mimg)						((mimg)->kind == FLOAT_IMAGE)
#define IS_DISCRETE(mimg)					((mimg)->kind == DISCRETE_IMAGE)
#define IS_RGB(mimg)						((mimg)->comp == RGB_COMP)
#define GET_PLANES_NUMBER(mimg)				((mimg)->comp)
#define GET_KIND(mimg)						((mimg)->kind)

typedef struct {
	unsigned char signature;
	int length;
	int height;
	int width;
	int kind;
	int comp;
} string_format, *string_format_ptr;

typedef struct {
  int x;
  int y;
  int dx;
  int dy;
} image_rect, *image_rect_ptr;

typedef struct {
  int x;
  int y;
} gr_image_displacement, *image_displacement_ptr, image_dimension, *image_dimension_ptr;

typedef struct {
  mpixel r;
  mpixel g;
  mpixel b;
} rgb_color, *rgb_color_ptr;

typedef struct {
  fmpixel r;
  fmpixel g;
  fmpixel b;
} rgb_fcolor, *rgb_fcolor_ptr;

enum {
	white_pixel_value = 0,
	black_pixel_value = 255
};

/* Exported functions */

int				mimg_lib_init();

mimage_ptr		create_mimage(int width, int height, int planes_number, char kind, unsigned char density);
void			destroy_mimage(mimage_ptr mimg);
mimage_ptr		read_jpeg_mimage(char *fname);
int				write_jpeg_mimage(char *fname, mimage_ptr mimg);
mimage_ptr		read_tiff_mimage(char *fname);
int				write_tiff_mimage(char *fname, mimage_ptr mimg);
void			get_mimage_info(mimage_ptr mimg, int *height, int *width, int *comp, int *kind, unsigned char *density);
byteplane_ptr	get_byteplane_from_mimage(mimage_ptr mimg, char plane_index);
fbyteplane_ptr	get_fbyteplane_from_mimage(mimage_ptr mimg, char plane_index);

mimage_ptr		clone_mimage(mimage_ptr mimg);
int				convert_mimage_to_float(mimage_ptr mimg);
int				convert_mimage_to_discrete(mimage_ptr mimg);

int				mimage_2_string_format(mimage_ptr mimg, char **str, size_t *length);
mimage_ptr		string_format_2_mimage(char *str);

int				crop_mimage(mimage_ptr img, image_rect_ptr rect);
int				stuff_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b, image_displacement_ptr delta);
int				shrink_mimage(mimage_ptr mimg, int threshold, image_rect_ptr rect_ptr);

void			width_and_height_mimage(mimage_ptr mimg, int *width, int *height);
int				width_mimage(mimage_ptr mimg);
int				height_mimage(mimage_ptr mimg);
int				set_width_and_height_mimage(mimage_ptr mimg, int width, int height);

mimage_ptr		const_mimage(void *values_array, image_dimension_ptr dim_ptr, int kind, int comp);

mimage_ptr		unimath_mimage(void *user_ptr, void *user_ptr2, mimage_ptr mimg, float (*floating_op)(void *user_ptr, void *user_ptr2, float, int));

mimage_ptr		plus_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b);
mimage_ptr		minus_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b);
mimage_ptr		times_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b);
mimage_ptr		divide_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b);
mimage_ptr		maxim_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b);
mimage_ptr		minim_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b);
mimage_ptr		power_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b);

mimage_ptr		plus_mimage_constant(mimage_ptr mimg_a, void *values_array);
mimage_ptr		minus_mimage_constant(mimage_ptr mimg_a, void *values_array);
mimage_ptr		times_mimage_constant(mimage_ptr mimg_a, void *values_array);
mimage_ptr		divide_mimage_constant(mimage_ptr mimg_a, void *values_array);
mimage_ptr		maxim_mimage_constant(mimage_ptr mimg_a, void *values_array);
mimage_ptr		minim_mimage_constant(mimage_ptr mimg_a, void *values_array);
mimage_ptr		power_mimage_constant(mimage_ptr mimg_a, void *values_array);

mimage_ptr		sin_mimage(mimage_ptr mimg);
mimage_ptr		cosin_mimage(mimage_ptr mimg);
mimage_ptr		tan_mimage(mimage_ptr mimg);
mimage_ptr		exp_mimage(mimage_ptr mimg);
mimage_ptr		log_mimage(mimage_ptr mimg);
mimage_ptr		abs_mimage(mimage_ptr mimg);
mimage_ptr		even_mimage(mimage_ptr mimg);
mimage_ptr		odd_mimage(mimage_ptr mimg);
mimage_ptr		sqrt_mimage(mimage_ptr mimg);
mimage_ptr		asin_mimage(mimage_ptr mimg);
mimage_ptr		acosin_mimage(mimage_ptr mimg);
mimage_ptr		atan_mimage(mimage_ptr mimg);
mimage_ptr		fix_mimage(mimage_ptr mimg);
mimage_ptr		floor_mimage(mimage_ptr mimg);
mimage_ptr		ceiling_mimage(mimage_ptr mimg);

float			maxof_mimage(mimage_ptr mimg);
float			minof_mimage(mimage_ptr mimg);

double			*sumall_mimage(mimage_ptr mimg, int power);
void			*sort_mimage(mimage_ptr mimg, int *elements);
mimage_ptr		scale_mimage(mimage_ptr mimg, image_dimension_ptr new_dim_ptr);
mimage_ptr		gradient_mimage(void *user_ptr, void *user_ptr2, float (*floating_op)(void *user_ptr, void *user_ptr2, int x, int y, int c), image_dimension_ptr new_dim_ptr, int kind, int planes_number);
mimage_ptr		convolve_mimage(mimage_ptr mimg, int *i_offset, int *j_offset, float *coefficient, size_t dim);
mimage_ptr		maxvolve_mimage(mimage_ptr mimg, int *i_offset, int *j_offset, float *coefficient, size_t dim);
mimage_ptr		minvolve_mimage(mimage_ptr mimg, int *i_offset, int *j_offset, float *coefficient, size_t dim);

mimage_ptr		extract_planes_mimage(mimage_ptr mimg, int from, int to);
mimage_ptr		set_planes_mimage(mimage_ptr mimg_res, mimage_ptr mimg, int from, int to);

mimage_ptr		threshold_mimage(mimage_ptr mimg, float *fvalues, size_t fvalues_size);
size_t			*histogram_mimage(mimage_ptr mimg, float *fvalues, size_t fvalues_size);
mimage_ptr		perlin_mimage(mimage_ptr mimg);
mimage_ptr		perlin2_mimage(mimage_ptr mimg);
mimage_ptr		anaglyph_mimage(mimage_ptr mimg, float scale, float fraction);

mimage_ptr		connected_components_mimage(mimage_ptr mimg);

mimage_ptr		flip_h(mimage_ptr mimg);
mimage_ptr		flip_v(mimage_ptr mimg);
mimage_ptr		rotate_halfpi(mimage_ptr mimg, short times);

mimage_ptr		random_mimage(mimage_ptr mimg, float r_min, float r_max);

void			*get_pixel_mimage(mimage_ptr mimg, int x, int y);
int				set_pixel_mimage(mimage_ptr mimg, int x, int y, void *pixel_value_array);

int				invert_mimage(mimage_ptr mimg);

/* Auxiliary functions */

int				rawline_in_planes(char *rawline, byteplane_ptr *planes, int line_index, int line_width, int planes_number);
int				planes_in_rawline(char *rawline, byteplane_ptr *planes, int line_index, int line_width, int planes_number, int planes_to_write, short reverse);
int				fplanes_in_rawline(char *rawline, fbyteplane_ptr *planes, int line_index, int line_width, int planes_number, int planes_to_write, short reverse);

byteplane_ptr	*create_planes_array(int width, int height, char kind, int planes_number, unsigned char density);
void			destroy_planes_array(void **planes_array, char kind, int planes_number, unsigned char density);

void			*create_byteplane(int width, int height, char kind, unsigned char density);

void			destroy_byteplane(void *byteplane, char kind, unsigned char density);

void			*get_plane_buffer(void *plane, int kind);

void			crop_from_plane_to_plane(void *old_plane, void *new_plane, char kind, image_rect_ptr from_rect, image_rect_ptr to_rect);
mimage_ptr		binary_op_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b, unsigned char (*gray_op)(unsigned char, unsigned char), float (*float_op)(float, float));
mimage_ptr		binary_op_constant_mimage(mimage_ptr mimg, void *values_array, unsigned char (*gray_op)(unsigned char, unsigned char), float (*float_op)(float, float));
mimage_ptr		unary_op_mimage(mimage_ptr mimg, unsigned char (*gray_op)(unsigned char), float (*float_op)(float));
int				are_same_format_mimages(mimage_ptr mimg_a, mimage_ptr mimg_b);

void			fast_zero_mimage(mimage_ptr mimg);
int				make_compatible(mimage_ptr mimg_a, mimage_ptr mimg_b);
mimage_ptr		lut_mimage(mimage_ptr mimg, mpixel *LUTs[], int number_of_planes, int target_kind);

int				lex_minof_mimage(mimage_ptr mimg, gr_image_displacement *pos);
int				lex_maxof_mimage(mimage_ptr mimg, gr_image_displacement *pos);
mimage_ptr		lex_sort_mimage(mimage_ptr mimg);
mimage_ptr		*split_mimage(mimage_ptr mimg, int *retNumber, image_displacement_ptr *offsets, int **pixels_pop);
mimage_ptr		tile_mimages(mimage_ptr *mimgs, int n, int h, int w);

#ifdef __cplusplus
}
#endif

#endif /* __BITPLNS_H__ */
