/**************************************************************************************
/* Filename:	gr_boolean_sparse_ops.c
/*		Copyright  1999 Giuseppe Di Mauro. All rights reserved.
/*
/* Description:	boolean operations for sparse byteplanes images
/*
/***************************************************************************************/

#include <string.h>

#include "gr_bitplns.h"
#include "gr_sparse_bitplns.h"
#include "gr_errors.h"
#include "gr_ops.h"

#include "gr_sparse_utils.h"
#include "gr_boolean_sparse_ops.h"

	/*
	 *	Not exported functions
	 */

static void merge_rows(mimage_ptr mimg_a, mimage_ptr mimg_b, mimage_ptr target_mimg,	int row_idx_a, int row_idx_b, int *row_idx_target, int *pixels_in_target_buffer, int *sect_idx_target, int have_same_default_color);
static void take_A_section_till(mimage_ptr mimg_a, mimage_ptr mimg_b, mimage_ptr target_mimg, int *take_pt, int pixel_no, section_data *sections_A, int *current_A_ix, int *current_new_ix, section_data *new_sections, int *pixels_in_target_buffer, 	int *current_B_ix, int num_sections_B, int num_sections_A, section_data *sections_B, int *taking_from_B, int have_same_default_color);
static void take_B_section(mimage_ptr mimg_a, mimage_ptr mimg_b, mimage_ptr target_mimg, int *current_new_ix, section_data *new_sections, int *current_B_ix, section_data *sections_B, int *pixels_in_target_buffer, int *current_A_ix, int num_sections_A, int num_sections_B, section_data *sections_A, int *take_pt, int *taking_from_B);
static short default_color_eq_check(mimage_ptr mimg_a, mimage_ptr mimg_b);

	/* 
	 *	Implementation 
	 */

/**************************************************************************************
/*	Function:		merge_rows
/*	Description:	merge two rows, given their vectors of sections and data
/*
/*	Parameters:
/*		<-	mimg_a						pointer to the first image to merge from
/*		<-	mimg_b						pointer to the second image to merge grom
/*		<->	target_mimg					target image
/*		<-	row_idx_a					row index in first image
/*		<-	row_idx_b					row index in second image
/*		<->	row_idx_target				row index for target image
/*		<->	pixels_in_target_buffer		keep count of the pixels in the target buffer
/*		<->	sect_idx_target				section index for target image
/*		<-	have_same_default_color		1 if both image have same default, 0 if not
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void merge_rows(
	mimage_ptr mimg_a, mimage_ptr mimg_b, 
	mimage_ptr target_mimg,	int row_idx_a, int row_idx_b, 
	int *row_idx_target, int *pixels_in_target_buffer, int *sect_idx_target, int have_same_default_color)
{
	int current_A_ix = 0;	/* the sections currently being examined */
	int current_B_ix = 0;
	int current_new_ix = 0;
	
	section_data *new_sections = &(target_mimg->row_section[*sect_idx_target]);	/* the new sections being formed */
	section_data *sections_A = &(mimg_a->row_section[mimg_a->row_start[row_idx_a]]);
	section_data *sections_B = &(mimg_b->row_section[mimg_b->row_start[row_idx_b]]);
	
	int num_sections_A = sections_in_row(mimg_a, row_idx_a);	/* get the number of sections in each row */
	int num_sections_B = sections_in_row(mimg_b, row_idx_b);

	int sec_A_start = sections_A[current_A_ix].starting_pixel_number;
	int sec_A_len =  sections_A[current_A_ix].number_of_pixels;
	int sec_A_data_idx = sections_A[current_A_ix].position_in_buffer;

	int sec_B_len =  sections_B[current_B_ix].number_of_pixels;
	int sec_B_start = sections_B[current_B_ix].starting_pixel_number;
	int sec_B_data_idx = sections_B[current_B_ix].position_in_buffer;
	
	int take_pt = 0;					/* the location of the next point to take, within the section being taken */
	int taking_from_B = (sec_B_start <= sec_A_start);	/* flag indicating whether we are taking from a or b */

	target_mimg->row_start[*row_idx_target] = *sect_idx_target;
	(*row_idx_target)++;	/* add one line per time */
	
	while ( (current_A_ix < num_sections_A) || (current_B_ix < num_sections_B) ) {
		if (taking_from_B) {	/* take the whole b section; advance to the next A section seen, and the next b section */
			take_B_section(mimg_a, mimg_b, target_mimg, 
				&current_new_ix, new_sections, &current_B_ix, sections_B, 
				pixels_in_target_buffer, &current_A_ix, num_sections_A, num_sections_B, sections_A, &take_pt, &taking_from_B);
		} else {	/* take as much of the current A section as precedes the current B section */
			take_A_section_till(
				mimg_a, mimg_b, target_mimg, &take_pt, 
				(current_B_ix < num_sections_B) ? sections_B[current_B_ix].starting_pixel_number : -1,
				sections_A, &current_A_ix, &current_new_ix, new_sections, pixels_in_target_buffer,
				&current_B_ix, num_sections_B, num_sections_A, sections_B, &taking_from_B, have_same_default_color);
		}
	}

	(*sect_idx_target) += current_new_ix;
}

/**************************************************************************************
/*	Function:		take_A_section_till
/*	Description:	take part of current A section up to next B point,
/*					or to sect end
/*
/*	Parameters:
/*		<-	mimg_a						pointer to the first image to merge from
/*		<-	mimg_b						pointer to the second image to merge grom
/*		<->	target_mimg					target image
/*		<->	take_pt						point to be taken
/*		<-	pixel_no					pixel number
/*		<-	sections_A					pointer to the secions of A
/*		<->	current_A_ix				current index of A sections
/*		<->	current_new_ix				current index of target image sections
/*		<->	new_sections				pointer to the sections in the target image
/*		<->	pixels_in_target_buffer		keep count of the pixels in the target buffer
/*		<->	current_B_ix				current index of B sections
/*		<-	num_sections_B				number of all the secion in B
/*		<-	num_sections_A				number of all the secion in B
/*		<-	sections_B					pointer to the sections in B
/*		->	taking_from_B				return 1 if taking from B, else 0
/*		<-	have_same_default_color		1 if both image have same default, 0 if not
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void take_A_section_till(
	mimage_ptr mimg_a, mimage_ptr mimg_b, mimage_ptr target_mimg, int *take_pt, 
	int pixel_no, section_data *sections_A, int *current_A_ix, int *current_new_ix,
	section_data *new_sections, int *pixels_in_target_buffer, int *current_B_ix, 
	int num_sections_B, int num_sections_A, section_data *sections_B, int *taking_from_B, int have_same_default_color)
{
#if macintosh && __MWERKS__
#pragma unused(mimg_b)
#endif
	int pixels_to_copy;
	int take_start;
	int rel_take_end;
	int prior_A_sec_len;
	int first_not_taken;
	
	int nns = *current_new_ix;	/* index of next available slot in the sections table */
	section_data_ptr last_sect;	/* point to last section used */
	section_data_ptr avail_sect = &new_sections[nns]; /* point to the first section available */
	section_data_ptr current_A_sect = &sections_A[*current_A_ix];

	take_start = current_A_sect->starting_pixel_number + *take_pt;	/* first mpixel to be taken */
	rel_take_end = pixel_no < 0 ? current_A_sect->number_of_pixels - 1:		/* relative position of last mpixel to be taken */
						min(current_A_sect->number_of_pixels, pixel_no - current_A_sect->starting_pixel_number) - 1;
	
	if ( *current_new_ix != 0
			&& (last_sect = &new_sections[nns - 1])->position_in_buffer == -1
			&& (last_sect->starting_pixel_number + last_sect->number_of_pixels) == take_start
			&& current_A_sect->position_in_buffer == -1 ) {	/* we can continue a default section */
		
		if (have_same_default_color) {
			last_sect->number_of_pixels += rel_take_end - *take_pt + 1;	/* just extend the length */
		} else {
			
			avail_sect->starting_pixel_number = take_start;
			avail_sect->number_of_pixels = (pixels_to_copy = rel_take_end - *take_pt + 1);
			avail_sect->position_in_buffer =  *pixels_in_target_buffer;
			(*current_new_ix)++;
			
			copy_default_value_at_offset(
				target_mimg, mimg_a, 
				*pixels_in_target_buffer, pixels_to_copy);

			(*pixels_in_target_buffer) += pixels_to_copy;
		}

	} else if ( *current_new_ix != 0
			&& (last_sect = &new_sections[nns - 1])->position_in_buffer >= 0
			&& (last_sect->starting_pixel_number + last_sect->number_of_pixels) == take_start
			&& current_A_sect->position_in_buffer >= 0 ) {	/*  we can continue a non-default section */
			
		last_sect->number_of_pixels += rel_take_end - *take_pt + 1;	/* extend the length */
		
			/* copy the pixels */
		copy_pixels_between_mimages(
			target_mimg, mimg_a, *pixels_in_target_buffer, 
			current_A_sect->position_in_buffer + *take_pt, 
			pixels_to_copy = rel_take_end - *take_pt + 1);
		(*pixels_in_target_buffer) += pixels_to_copy;

	} else if ( current_A_sect->position_in_buffer == -1) {		/* we insert a new default section */

		if (have_same_default_color) {
		
			avail_sect->starting_pixel_number = take_start;
			avail_sect->number_of_pixels = rel_take_end - *take_pt + 1;
			avail_sect->position_in_buffer = -1;
			(*current_new_ix)++;
		
		} else {
		
			avail_sect->starting_pixel_number = take_start;
			avail_sect->number_of_pixels = (pixels_to_copy = rel_take_end - *take_pt + 1);
			avail_sect->position_in_buffer = *pixels_in_target_buffer;
			(*current_new_ix)++;

			copy_default_value_at_offset(
				target_mimg, mimg_a, 
				*pixels_in_target_buffer, pixels_to_copy);

			(*pixels_in_target_buffer) += pixels_to_copy;
		}
		
	} else {		/* we insert a new non-default section, and copy its data to the data buffer */

		avail_sect->starting_pixel_number = take_start;
		avail_sect->number_of_pixels = rel_take_end - *take_pt + 1;
		avail_sect->position_in_buffer = *pixels_in_target_buffer;
		(*current_new_ix)++;
 		
			/* and copy the A data into the buffer */
		copy_pixels_between_mimages(
			target_mimg, mimg_a, *pixels_in_target_buffer, 
			current_A_sect->position_in_buffer + *take_pt, 
			pixels_to_copy = rel_take_end - *take_pt + 1);
		(*pixels_in_target_buffer) += pixels_to_copy;
	}

		/* now we must advance B to the first section whose final component exceeds the last point taken */
	first_not_taken = current_A_sect->starting_pixel_number + rel_take_end + 1;
	if (rel_take_end >= (prior_A_sec_len = current_A_sect->number_of_pixels) - 1)
		(*current_A_ix) ++;	/* we have taken the whole A section, so we advance to the next A section */

		/* determine whether the next pixels are taken from A or B */
	*taking_from_B = (*current_B_ix < num_sections_B) 
		&& ((rel_take_end < prior_A_sec_len - 1)
			|| (*current_A_ix >= num_sections_A)
			|| (sections_B[*current_B_ix].starting_pixel_number <= sections_A[*current_A_ix].starting_pixel_number));

	if (!*taking_from_B)
		*take_pt = 0;
		 		/* 
		 		 *	we must note the point of A that is taken first; this is the first point of the next A section
		 		 *	if it comes before the first point of the next B section
		 		 */
}

/**************************************************************************************
/*	Function:		take_B_section
/*	Description:	take whole of current B section
/*
/*	Parameters:
/*		<-	mimg_a						pointer to the first image to merge from
/*		<-	mimg_b						pointer to the second image to merge grom
/*		<->	target_mimg					target image
/*		<->	current_new_ix				current index of target image sections
/*		<->	new_sections				pointer to the sections in the target image
/*		<->	current_B_ix				current index of B sections
/*		<-	sections_B					pointer to the sections in B
/*		<->	pixels_in_target_buffer		keep count of the pixels in the target buffer
/*		<->	current_A_ix				current index of A sections
/*		<-	num_sections_A				number of all the secion in B
/*		<-	num_sections_B				number of all the secion in B
/*		<-	sections_A					pointer to the secions of A
/*		<->	take_pt						point to be taken
/*		->	taking_from_B				return 1 if taking from B, else 0
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void take_B_section(
	mimage_ptr mimg_a, mimage_ptr mimg_b, mimage_ptr target_mimg, 
	int *current_new_ix, section_data *new_sections, 
	int *current_B_ix, section_data *sections_B, int *pixels_in_target_buffer, 
	int *current_A_ix, int num_sections_A, int num_sections_B,
	section_data *sections_A, int *take_pt, int *taking_from_B)
{
#if macintosh && __MWERKS__
#pragma unused(mimg_a)
#endif
	int j;
	int pixels_to_copy;
	int first_not_taken;
	section_data_ptr nsns;
	int past_B, strt_A;
		

	int nsa = num_sections_A;
	int nsb = num_sections_B;
	int nns = *current_new_ix;
	section_data_ptr last_sect;	/* point to last section used */
	section_data_ptr avail_sect = &new_sections[nns]; /* point to the first section available */
	section_data_ptr current_B_sect = &sections_B[*current_B_ix];
	
	if ( *current_new_ix != 0
			&& (last_sect=&new_sections[nns-1])->position_in_buffer == -1
			&& (last_sect->starting_pixel_number + last_sect->number_of_pixels) == current_B_sect->starting_pixel_number
			&& current_B_sect->position_in_buffer == -1 ) {	/* we can continue a default section */

		last_sect->number_of_pixels += current_B_sect->number_of_pixels;	/* just extend the length */

	} else if ( *current_new_ix != 0
			&& (last_sect=&new_sections[nns-1])->position_in_buffer >= 0
			&& (last_sect->starting_pixel_number + last_sect->number_of_pixels) == current_B_sect->starting_pixel_number
			&& current_B_sect->position_in_buffer >= 0 ) {	/* we can continue a non-default section */
		
		last_sect->number_of_pixels += current_B_sect->number_of_pixels;	/* extend the length */
		
			/* copy the pixels */
		copy_pixels_between_mimages(
			target_mimg, mimg_b, *pixels_in_target_buffer, 
			current_B_sect->position_in_buffer, 
			pixels_to_copy = current_B_sect->number_of_pixels);
		(*pixels_in_target_buffer) += pixels_to_copy;

	} else if ( current_B_sect->position_in_buffer == -1) {	/* we insert a new default section */
		
		avail_sect->starting_pixel_number = current_B_sect->starting_pixel_number;
		avail_sect->number_of_pixels = current_B_sect->number_of_pixels;
		avail_sect->position_in_buffer = -1;
		(*current_new_ix)++;

	} else {	/* we insert a new non-default section, and copy its data to the data buffer */
		
		avail_sect->starting_pixel_number = current_B_sect->starting_pixel_number;
		avail_sect->number_of_pixels = current_B_sect->number_of_pixels;
		avail_sect->position_in_buffer = *pixels_in_target_buffer;
		(*current_new_ix)++;
		
			/* and copy the B data into the buffer */
		copy_pixels_between_mimages(
			target_mimg, mimg_b, *pixels_in_target_buffer, 
			current_B_sect->position_in_buffer, 
			pixels_to_copy = current_B_sect->number_of_pixels);
		(*pixels_in_target_buffer) += pixels_to_copy;

	}
	
		/* now we must advance A to the first section whose final component exceeds the last point taken */
	first_not_taken = current_B_sect->starting_pixel_number + current_B_sect->number_of_pixels;
	
	for (j=*current_A_ix ; j<nsa ; j++) {
		if ( (sections_A[j].starting_pixel_number + sections_A[j].number_of_pixels-1) >= first_not_taken ) {
			*current_A_ix = j;
			break;
		}
	}
	if (j == nsa)
		*current_A_ix = nsa;

	(*current_B_ix)++;

		/* determine whether the next pixels are taken from A or B */
	*taking_from_B = (*current_B_ix < nsb) 
		&& ((*current_A_ix >= nsa)
			|| (sections_B[*current_B_ix].starting_pixel_number 
						<= sections_A[*current_A_ix].starting_pixel_number)
			|| (sections_B[*current_B_ix].starting_pixel_number 
						== (nsns = &new_sections[*current_new_ix])->starting_pixel_number + nsns->number_of_pixels));

	if (!*taking_from_B) {	/* we must note the point of A that is taken first */
		*take_pt = (*current_A_ix >= nsa) ? 0 : 
			(*current_B_ix == 0) ? 0 : 
				( past_B = sections_B[*current_B_ix - 1].starting_pixel_number + sections_B[*current_B_ix - 1].number_of_pixels)
					<= ( strt_A = sections_A[*current_A_ix].starting_pixel_number) ? 0 : 
						past_B - strt_A;
	}
}

/**************************************************************************************
/*	Function:		default_color_eq_check
/*	Description:	returns 1 if have the same default value, 0 otherwise
/*					this procedure assumes that both the imager have the same type
/*
/*	Parameters:
/*		<- mimg_a			first image
/*		<- mimg_b			second image
/*
/*	Result:
/*		1		same default value
/*		0		different default value
/*
/***************************************************************************************/

short default_color_eq_check(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
	int c;
	
	if (IS_DISCRETE(mimg_a)) {
		byteplane_ptr plane_a;
		byteplane_ptr plane_b;
	
		for (c=0; c<mimg_a->comp; c++) {
			plane_a = get_byteplane_from_mimage(mimg_a, c);
			plane_b = get_byteplane_from_mimage(mimg_b, c);

			if ( plane_a->default_value != plane_b->default_value)
				return 0;
		}
	} else {
		fbyteplane_ptr fplane_a;
		fbyteplane_ptr fplane_b;
	
		for (c=0; c<mimg_a->comp; c++) {
			fplane_a = get_fbyteplane_from_mimage(mimg_a, c);
			fplane_b = get_fbyteplane_from_mimage(mimg_b, c);

			if ( fplane_a->default_value != fplane_b->default_value)
				return 0;
		}
	}
	
	return 1;
}

/**************************************************************************************
/*	Function:		intersect_smimage
/*	Description:	Perform the intersection operation between any two sparse images.
/*	Note:			the default value is the new one set by the function
/*
/*	Parameters:
/*		<- mimg_a			pointer to first image
/*		<- mimg_b			pointer to second image
/*		<- default_array	array of pixels for discrete default value
/*		<- fdefault_array	array of pixels for float default value
/*
/*	Result:
/*		not NULL	new intersection sparse image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr intersect_smimage(mimage_ptr mimg_a, mimage_ptr mimg_b, mpixel_ptr default_array, fmpixel_ptr fdefault_array)
{
	mimage_ptr res_mimg;		/* image result */

	int j;						/* index for scanned row at this time */
	int max_width = 0;			/* max_width written (used to calculate image width) */
	int previous_j = 0;			/* index for scanned row at precedent time */

	int	ia, ib;					/* index for columns in images a and b */

	int *row_start_a;			/* row_start array for image a */
	int *row_start_b;			/* row_start array for image b */
	
	int data_length_a;			/* row_start array length for image a */
	int data_length_b;			/* row_start array length for image b */
	
	section_data *section_data_a;	/* row_section array for image a */
	section_data *section_data_b;	/* row_section array for image a */
	
	int end_sect_a, end_sect_b;		/* indexes for last section at the j-row */

	int data_index;				/* keep track of next available index in the row_start array for result image */
	int data_length; 			/* keep track of current length in the row_start array for result image */
	int section_index;			/* keep track of next available index in the row_section array for result image */
	size_t section_length;		/* keep track of current length in the row_section array for result image */
	
	int section_written;		/* keep track of the first section for the row j */
	
	int i_start, i_end;			/* column limit at current section in result image*/

	int l_a = 0, m_a = 0;		/* l_a: row in a, m_a: col in a */
	int l_b = 0, m_b = 0;		/* l_b: row in b, m_b: col in b */

	int i_a_start, i_a_end;		/* column limits for image a */
	int j_a;					/* row index for image a */
	int i_b_start, i_b_end;		/* column limits for image b */
	int j_b;					/* row index for image b */
	
	int overlap_step;			/* number of overlapping rows processed */
	
	int width_a, height_a, planes_number_a, kind_a;		/* image information for image a */
	int width_b, height_b, planes_number_b, kind_b;		/* image information for image b */
	int width, height, planes_number, kind;				/* image information for image result */

	size_t type_size;									/* size of type used for the images */
	
	int		number_of_pixels;							/* used to count the number of pixels in a section */
	int		err;										/* keep error value for some procedure */
		
	/* 
	 *	retrieve information from source images 
	 */
	
	get_mimage_info(mimg_a, &height_a, &width_a, &planes_number_a, &kind_a, NULL);
	get_mimage_info(mimg_b, &height_b, &width_b, &planes_number_b, &kind_b, NULL);

	/*
	 *	check if images are compatible between themeselves
	 */

	if (kind_a != kind_b)
	{
		mimg_err_no = kGrKindNotMatch;
		return NULL;		/* cannot mix discrete and float */
	}
	
	if (planes_number_a != planes_number_b)
	{
		mimg_err_no = kGrPlanesNotMatch;
		return NULL;		/* cannot mix images with different planes */
	}
	
	/*
	 *	Prepare data for result image
	 */

	kind = kind_a;
	planes_number = planes_number_a;
	
	data_index = section_index = 0;
	data_length = section_length = 0;
	
	height = min(height_a, height_b);
	width = min(width_a, width_b);
	
	type_size =  GET_TYPE_SIZE(kind);
	
	/* 
	 *	create an empty sparse image
	 */
	 
	res_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
	if (!res_mimg)
	{
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}
	
	/*
	 *	setup buffers with preliminary allocations
	 *		NOTE:	more memory could be needed to
	 *				store the image. In that case
	 *				a dynamic expansion of the table
	 *				will occurr
	 */
	
	if (pre_setup_sparse_mimage_no_buffer_allocated(res_mimg, width, height) != 0) 
	{
		destroy_mimage(res_mimg);
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}
		
	data_length = res_mimg->data_length;
	section_length = res_mimg->section_length;
	
	/*
	 *	get information to process both source images
	 */ 
	 
	row_start_a = mimg_a->row_start;
	row_start_b = mimg_b->row_start;
	
	data_length_a = mimg_a->data_length;
	data_length_b = mimg_b->data_length;

	section_data_a = mimg_a->row_section;
	section_data_b = mimg_b->row_section;

	/*
	 *	translate to the y origin
	 */
	
	j_a = mimg_a->upperleft_y;
	j_b = mimg_b->upperleft_y;

		/*
		 *	Loop in row_data of a and b
		 */

	for ( ; (l_a < data_length_a) && (l_b < data_length_b) ; ) {

			/*
			 *	Loop until find the first section
			 */

		if ( row_start_a[ l_a ] < 0)
		{
			j_a -= row_start_a[ l_a ++ ];	 /*	j_a += - row_start_a[ l_a ]; l_a ++; */
			continue;
		}

		if ( row_start_b[ l_b ] < 0)
		{
			j_b -= row_start_b[ l_b ++ ];	 /*	j_b += - row_start_b[ l_b ]; l_b ++; */
			continue;
		}
		
			/*
			 *	Check if rows match
			 */

		if ( j_a < j_b ) 		/* if row for a is before row for b */
		{
			l_a ++;					/* go to the next row for a */
			j_a ++;
			continue;
		}
		else if ( j_a > j_b )		/* if row for b is before row for a */
		{
			l_b ++;
			j_b ++;					/* go to the next row for b */
			continue;
		}
		else 					/* both sections are in the same row */
		{
		
			/*
			 *	now we need to check for overlappings along the
			 *	same row. 
			 */
		
			j = j_a;	/* j_a == j_b in this case */
		
			overlap_step = 0;	/* reset overlap value */
			
				/* 
				 *	calculate last section for
				 *	this row for both a and b 
				 */

			end_sect_a = ( m_a = row_start_a[ l_a ] ) + sections_in_row( mimg_a, l_a );
			end_sect_b = ( m_b = row_start_b[ l_b ] ) + sections_in_row( mimg_b, l_b );

				/*
				 *	loop over the sections in the row 
				 */

			section_written = kNoSection;

			for ( ; (m_a < end_sect_a) && (m_b < end_sect_b) ; )
			{
					/* 
					 *	get the boundaries for the section
					 *	and translate to the x origin
					 */

				i_a_start = section_data_a[ m_a ].starting_pixel_number + mimg_a->upperleft_x;
				i_b_start = section_data_b[ m_b ].starting_pixel_number + mimg_b->upperleft_x;

				i_a_end = i_a_start + ( section_data_a[ m_a ].number_of_pixels - 1 );
				i_b_end = i_b_start + ( section_data_b[ m_b ].number_of_pixels - 1 );

					/* if a preceeds b */

				if (i_a_end < i_b_start) {
					m_a ++;						/* advance a to the next section */
					continue;
				}
					
					/* if b preceeds a */

				if (i_a_start > i_b_end) {
					m_b ++;						/* advance b to the next section */
					continue;
				}

					/*
					 *	at this point we found two overlapping section
					 */

				overlap_step = 1;				/* set the overlapping row counter to one */

				if (section_written == kNoSection)
					section_written = section_index;	/* remember the index for the first overlapping section for this row */

					/*
					 *	calc the boundaries for the destination section
					 */

				i_start	= max(i_a_start, i_b_start);
				i_end	= min(i_a_end, i_b_end);

					/*
					 *	get the offsets for the image a and b
					 */

				ia = i_start - i_a_start;
				ib = i_start - i_b_start;


					/*
					 *	calc the number of pixels
					 */

				number_of_pixels = i_end - i_start + 1;

					/*
					 *	update the data section with the added information
					 */

				write_to_section_data(
					res_mimg, 
					section_index,
					&section_length,
					i_start,
					number_of_pixels,
					-1						/* always default value for intersections */
				);
					
				section_index++;
				
					/*
					 *	update max_width
					 */
					 
				if (i_end > max_width)
					max_width = i_end;
					
					/*
					 *	advance to the next section
					 */

				if (i_a_end == i_end)
					m_a++;
					
				if (i_b_end == i_end)
					m_b++;
					
			}			


				/*
				 *	check for skipped lines
				 *
				 *		NOTE:	if overlap rows the step
				 *				must be greater than 
				 *				their number
				 */

			if ( previous_j < j - overlap_step )
			{

					/*
					 *	put the information for skipped lines
					 *	and if needed expand the table
					 */
					 
				err = expand_table_on_need(
					(void **)&res_mimg->row_start, 
					data_index,
					&data_length,
					sizeof(*res_mimg->row_start)
				);
				
				if (err) 	/* check for re-allocation errors */
				{
					destroy_mimage(res_mimg);

					mimg_err_no = kGrOutOfMemory;
					return NULL;
				}

					/* 
					 *	update row_start 
					 */
					 
				res_mimg->row_start[data_index ++ ] = previous_j - j;	/*  - (j - previous_j); with j > previous_j */

			}

			previous_j = j;		/* update previous j to the newer */

				/*
				 *	check if we found an overlap
				 */

			if (overlap_step) 
			{
					/*				
					 *	update row_start for the written sections
					 *  and if needed expand the table
					 */
					 
				err = expand_table_on_need(
					(void **)&res_mimg->row_start, 
					data_index,
					&data_length,
					sizeof(*res_mimg->row_start)
				);
				
				if (err) 
				{
					mimg_err_no = kGrOutOfMemory;
					return NULL;
				}

				res_mimg->row_start[data_index ++] = section_written;	/* update row_start */

			}
			
				/* 
				 *	advance to next section
				 */

			l_a ++;
			l_b ++;
			
			j_a ++;
			j_b ++;
			
		}
		
	}
	
	if (data_index < height)
	{
		err = expand_table_on_need(
			(void **)&res_mimg->row_start, 
			height,
			&data_length,
			sizeof(*res_mimg->row_start)
		);
		
		res_mimg->row_start[data_index] = -(height - data_index);
		data_index++;
	}

		/* 
		 *	shrink the partitions, reassignment is not needed
		 *	because data won't be moved from its position
		 */

	res_mimg->data_length = data_index;
	res_mimg->section_length = section_index;
	res_mimg->buffer_length = 0;
	
	res_mimg->row_start = realloc(res_mimg->row_start, res_mimg->data_length * sizeof( *res_mimg->row_start ) );
	res_mimg->row_section = realloc(res_mimg->row_section, res_mimg->section_length * sizeof( *res_mimg->row_section ) );

	/*
	 *	set both width and height
	 */

	set_width_and_height_mimage(res_mimg, width, height);

	if (kind == DISCRETE_IMAGE)
		set_default_color_smimage(res_mimg, default_array);
	else
		set_default_fcolor_smimage(res_mimg, fdefault_array);

	return res_mimg;
}

/**************************************************************************************
/*	Function:		superpose_smimage
/*	Description:	Perform a (super) union operation between any 
/*					two sparse images in the sense that the values of the
/*					overlapping (if any) area are the ones from the second image
/*					
/*	Parameters:
/*		<- mimg_a			pointer to first image
/*		<- mimg_b			pointer to second image
/*
/*	Result:
/*		not NULL	new superposed sparse image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr superpose_smimage(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
	int c;
	
	size_t type_size;			/* size of type used for the images */

	int	sect_idx_target = 0;
	int row_idx_target = 0;
	int data_idx_target = 0;
	
	int row_in_target, row_in_a, row_in_b;
	int col_in_target, col_in_a, col_in_b;

	int data_index, data_length;

	int section_index;			/* keep track of next available index in the row_section array for result image */
	int section_length;			/* keep track of current length in the row_section array for result image */

	int *row_start_a;			/* row_start array for image a */
	int *row_start_b;			/* row_start array for image b */
	
	int data_length_a;			/* row_start array length for image a */
	int data_length_b;			/* row_start array length for image b */
	
	section_data *section_data_a;	/* row_section array for image a */
	section_data *section_data_b;	/* row_section array for image a */

	int row_idx_a, row_idx_b;
	int section_idx_a, section_idx_b;

	mimage_ptr target_mimg;		/* image result */

	int width_a, height_a, planes_number_a, kind_a;		/* image information for image a */
	int width_b, height_b, planes_number_b, kind_b;		/* image information for image b */
	int width, height, planes_number, kind;				/* image information for image result */

	int have_same_default_color, expand_default;

	/* 
	 *	retrieve information from source images 
	 */
	
	get_mimage_info(mimg_a, &height_a, &width_a, &planes_number_a, &kind_a, NULL);
	get_mimage_info(mimg_b, &height_b, &width_b, &planes_number_b, &kind_b, NULL);

	/*
	 *	check if images are compatible between themeselves
	 */

	if (kind_a != kind_b)
	{
		mimg_err_no = kGrKindNotMatch;
		return NULL;		/* cannot mix discrete and float */
	}
	
	if (planes_number_a != planes_number_b)
	{
		mimg_err_no = kGrPlanesNotMatch;
		return NULL;		/* cannot mix images with different planes */
	}
	
	/*
	 *	Get the default colors 
	 */

	have_same_default_color = default_color_eq_check(mimg_a, mimg_b);
	expand_default = !have_same_default_color;
	/*
	 *	Prepare data for result image
	 */

	kind = kind_a;
	planes_number = planes_number_a;
	
	data_index = section_index = 0;
	data_length = section_length;
	
	height = max(height_a, height_b);
	width = max(width_a, width_b);
	
	type_size =  GET_TYPE_SIZE(kind);
	
	/* 
	 *	create an empty sparse image
	 */
	 
	target_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
	if (!target_mimg)
	{
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}
	
	if (!allocate_sparse_buffers(
			target_mimg, height /* data_length */, 
			2 * (mimg_a->section_length + mimg_b->section_length) /* section_length */, 
			height*width /* buffer_length */)) {
		destroy_mimage(target_mimg);
		return NULL;
	}

	data_length = target_mimg->data_length;
	section_length = target_mimg->section_length;

	/*
	 *	get information to process both source images
	 */ 
	
	row_idx_a = row_idx_b = 0;
	row_start_a = mimg_a->row_start;
	row_start_b = mimg_b->row_start;
	
	section_idx_a = section_idx_b = 0;
	section_data_a = mimg_a->row_section;
	section_data_b = mimg_b->row_section;

	data_length_a = mimg_a->data_length;
	data_length_b = mimg_b->data_length;

	sect_idx_target = 0;
	row_idx_target = 0;
	data_idx_target = 0;
		
	row_in_target = row_in_a = row_in_b = 0;
	col_in_target = col_in_a = col_in_b = 0;

		/* set default value to the one in the second image */
	for (c=0; c<planes_number; c++) {
		if (IS_DISCRETE(mimg_b)) {
			byteplane_ptr plane = mimg_b->planes_array[c];
			byteplane_ptr target_plane = target_mimg->planes_array[c];

			target_plane->default_value = plane->default_value;
		} else {
			fbyteplane_ptr fplane = (fbyteplane_ptr)mimg_b->planes_array[c];
			fbyteplane_ptr target_fplane = (fbyteplane_ptr)target_mimg->planes_array[c];

			target_fplane->default_value = fplane->default_value;
		}
	}
	
	while ( row_idx_a < data_length_a && row_idx_b < data_length_b )	/* put more data into the new image */
	{
			/* advance to the next non empty row in image a */
		if (row_start_a[row_idx_a] < 0)
		{
			row_in_a -= row_start_a[row_idx_a];
			row_idx_a++;
			continue;
		}

			/* advance to the next non empty row in image b */
		if (row_start_b[row_idx_b] < 0)
		{
			row_in_b -= row_start_b[row_idx_b];
			row_idx_b++;
			continue;
		}

		if (row_in_b < row_in_a) {
		
			copy_row_from_mimage(mimg_b, target_mimg, row_idx_b, &row_idx_target, &sect_idx_target, &data_idx_target, 0);
			row_idx_b++;
			row_in_b++;
		} else if (row_in_a < row_in_b) {
		
			copy_row_from_mimage(mimg_a, target_mimg, row_idx_a, &row_idx_target, &sect_idx_target, &data_idx_target, expand_default);
			row_idx_a++;
			row_in_a++;
		}
		else	/* row_in_a == row_in_b */
		{

			merge_rows(mimg_a, mimg_b, target_mimg, row_idx_a, row_idx_b, &row_idx_target, &data_idx_target, &sect_idx_target, have_same_default_color);

				/* advance now both of the indexes */
			row_idx_a++;
			row_in_a++;

			row_idx_b++;
			row_in_b++;
		}
	}
	
		/* copy remaining lines */

	while ( row_idx_a < data_length_a)
	{
		copy_row_from_mimage(mimg_a, target_mimg, row_idx_a, &row_idx_target, &sect_idx_target, &data_idx_target, expand_default);
		row_idx_a++;
		row_in_a++;
	}

	while ( row_idx_b < data_length_b)
	{
		copy_row_from_mimage(mimg_b, target_mimg, row_idx_b, &row_idx_target, &sect_idx_target, &data_idx_target, 0);
		row_idx_b++;
		row_in_b++;
	}
	
		/* prepare to cut off all superfluous data */

	target_mimg->data_length = row_idx_target;
	target_mimg->section_length = sect_idx_target;
	target_mimg->buffer_length = data_idx_target;
	
	trim_sparse(target_mimg);
	
	return target_mimg;
}
