/**************************************************************************************
/* Filename:	gr_dense_bitplns.c
/*		Copyright  1998-99 Giuseppe Di Mauro. All rights reserved.
/*
/* Description:	general graphical routines for dense byteplanes images
/*
/***************************************************************************************/
 
#include <assert.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "jpeglib.h"
#ifndef TIFF_DISABLED
#include "tiffio.h"
#endif

#include "gr_bitplns.h"
#include "gr_dense_bitplns.h"
#include "gr_interpolation.h"
#include "gr_errors.h"
#include "gr_ops.h"
#include "gr_utils.h"

	/* 
	 *	Implementation 
	 */

/**************************************************************************************
/*	Function:		create_dense_byteplane
/*	Description:	create a dense byteplane
/*
/*	Parameters:
/*		<-	width	width of requested plane
/*		<-	height	height of requested plane
/*		<-	kind	kind (float or integer) of requested plane
/*
/*	Result:
/*		not NULL	pointer to the requested plane
/*		NULL		out of memory
/*
/***************************************************************************************/

void *create_dense_byteplane(int width, int height, char kind)
{
	byteplane_ptr plane_ptr;
	fbyteplane_ptr fplane_ptr;
	void *ptr;
	 
		/* use the right byteplane kind creator */
	switch (kind) {

		case DISCRETE_IMAGE:	/* for discrete planes */

				/* alloc clean memory for plane */
			plane_ptr = (byteplane_ptr) calloc(sizeof(byteplane), 1);
			if (!plane_ptr)
				return NULL;

			plane_ptr->height = height;	/* set the height of the plane */
			plane_ptr->width = width;	/* set the width of the plane */
			plane_ptr->buffer	= (unsigned char *)
				malloc(width * height * sizeof(unsigned char));	/* alloc the buffer memory */
			if (!plane_ptr->buffer) {
				free(plane_ptr);
				return NULL;
			}
				
			ptr = plane_ptr;
			
			break;

		case FLOAT_IMAGE:	/* for float planes */

				/* alloc clean memory for plane */
			fplane_ptr	= (fbyteplane_ptr) calloc(sizeof(fbyteplane), 1);
			if (!fplane_ptr)
				return NULL;

			fplane_ptr->height = height;	/* set the height of the plane */
			fplane_ptr->width = width;		/* set the width of the plane */
			fplane_ptr->buffer = (float *)
				malloc(width * height * sizeof(float));	/* alloc the buffer memory */
			if (!fplane_ptr->buffer) {
				free(fplane_ptr);
				return NULL;
			}
			
			ptr = fplane_ptr;	/* set the return value */
			
			break;
	}

	return ptr;	/* return the pointer to the created plane */
}

/**************************************************************************************
/*	Function:		destroy_dense_byteplane
/*	Description:	destroy a dense byteplane and release its memory
/*
/*	Parameters:
/*		<-	plane	pointer to the plane
/*		<-	kind	kind (float or integer) of plane
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void destroy_dense_byteplane(void *plane, char kind)
{
	void *buffer;
		
	if (plane) {	/* if a valid plane */
		
		buffer = get_plane_buffer(plane, kind);	/* get the pointer to the buffer */
		if (buffer)			/* if a valid buffer */
			free(buffer);	/* release memory for buffer */
		free(plane);		/* release memory for plane */
	}
}

/**************************************************************************************
/*	Function:		clone_mimage_dense
/*	Description:	clone a dense image in a new one
/*
/*	Parameters:
/*		<-	img		source image
/*
/*	Result:
/*		not NULL	pointer to the new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr clone_mimage_dense(mimage_ptr mimg)
{
	int c;
	int height, width, planes_number, kind;
	size_t type_size;
	void *buffer_src, *buffer_dst;
	size_t plane_size;
	mimage_ptr cloned_mimg;
	
		/* get info for the image to clone */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);	/* get the size in bytes of a pixel */
	plane_size =  width*height*type_size;	/* get the size in bytes of a plane buffer */
	
		/* create an image with the same specs of the original one */
	cloned_mimg = create_mimage(width, height, planes_number, kind, DENSE_IMAGE);
	if (!cloned_mimg)
		return NULL;

		/* for each plane */	
	for (c=0; c<planes_number; c++) {
			/* get pointer to the buffers in both source and target */
		buffer_src = get_plane_buffer(mimg->planes_array[c], kind);
		buffer_dst = get_plane_buffer(cloned_mimg->planes_array[c], kind);
			/* and copy the pixel data */
		memcpy(buffer_dst, buffer_src, plane_size);
	}	

	return cloned_mimg;		/* return a pointer to the cloned image */	
}

/**************************************************************************************
/*	Function:		convert_dmimage_to_float
/*	Description:	convert a dense integer image in a new float one
/*
/*	Parameters:
/*		<-	img		source integer image
/*
/*	Result:
/*		not NULL	pointer to the new float converted image
/*		NULL		out of memory
/*
/***************************************************************************************/

int convert_dmimage_to_float(mimage_ptr mimg)
{
	int i, h;
	int height, width, comp, kind;
	
	fbyteplane_ptr *float_planes;

	float *plane_dst_ptr;
	unsigned char *plane_src_ptr;

		/* get info about the image to convert */
	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);

	if (kind == FLOAT_IMAGE)	/* if the image kind is already float */
		return 1; /* no conversion is necessary */

		/* create a new array of planes of float kind */
	float_planes = (fbyteplane_ptr *)create_planes_array(width, height, FLOAT_IMAGE, comp, DENSE_IMAGE);
	if (!float_planes)
		return -1;

		/* for each plane */
	for (i=0; i<comp; i++) {

			/* get both source and destination buffer pointers */
		plane_src_ptr = get_byteplane_from_mimage(mimg, i)->buffer;
		plane_dst_ptr = float_planes[i]->buffer;

			/* copy-convert the pixels between planes */
		for (h=0; h<(height*width); h++) 
			plane_dst_ptr[h] = (float)plane_src_ptr[h];
	}

		/* destroy the old planes and assign new created ones */
	destroy_planes_array((void **)mimg->planes_array, DISCRETE_IMAGE, comp, DENSE_IMAGE);
	mimg->planes_array = (byteplane_ptr *)float_planes;

	mimg->kind = FLOAT_IMAGE;

	return 0;
}

/**************************************************************************************
/*	Function:		convert_dmimage_to_discrete
/*	Description:	convert a dense float image in a new discrete one
/*
/*	Parameters:
/*		<-	img		source float image
/*
/*	Result:
/*		not NULL	pointer to the new integer converted image
/*		NULL		out of memory
/*
/***************************************************************************************/

int convert_dmimage_to_discrete(mimage_ptr mimg)
{
	int i, h;
	int height, width, comp, kind;
	
	byteplane_ptr *planes;

	unsigned char *plane_dst_ptr;
	float *plane_src_ptr;

		/* get info about the image to be converted */
	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);

	if (kind == DISCRETE_IMAGE)	/* if the image is already discrete */
		return 1; /* no conversion is necessary */
	
		/* create a new array of planes of discrete kind */
	planes = create_planes_array(width, height, DISCRETE_IMAGE, comp, DENSE_IMAGE);
	if (!planes)
		return -1;

		/* for each plane */
	for (i=0; i<comp; i++) {

			/* get both source and destination buffer pointers */
		plane_dst_ptr = planes[i]->buffer;
		plane_src_ptr = get_fbyteplane_from_mimage(mimg, i)->buffer;

			/* copy-convert the pixels between planes */
		for (h=0; h<(height*width); h++)
			plane_dst_ptr[h] = (unsigned char)plane_src_ptr[h];
	}

		/* destroy the old planes and assign new created ones */
	destroy_planes_array((void **)mimg->planes_array, DISCRETE_IMAGE, comp, DENSE_IMAGE);
	mimg->planes_array = planes;

	mimg->kind = DISCRETE_IMAGE;		/* set the new image kind */

	return 0;
}

/**************************************************************************************
/*	Function:		binary_op_constant_dmimage
/*	Description:	perform a binary operation between 
/*					a dense image and a constant value
/*
/*	Parameters:
/*		<-	mimg			dense image
/*		<-	values_array	constant data pixel array
/*		<-	gray_op			pointer to a binary operation function for integer case
/*		<-	float_op		pointer to a binary operation function for float case
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr binary_op_constant_dmimage(mimage_ptr mimg, void *values_array,
	unsigned char (*gray_op)(unsigned char, unsigned char), float (*float_op)(float, float))
{
	int c, i;
	size_t pixels_count_plan;
	float *float_buffer, *float_buffer_res;
	unsigned char *buffer, *buffer_res;
	mimage_ptr mimg_res;
	int height, width, planes_number, type; 

	/* get info about the image */
	get_mimage_info(mimg, &height, &width, &planes_number, &type, NULL);
		
	/* create a new image with the same features */
	mimg_res = create_mimage(width, height, planes_number, type, DENSE_IMAGE);
	if (!mimg_res)
	return NULL;
	  
	  /* count the number of pixels per plane */
	pixels_count_plan = height * width;

	if (type == FLOAT_IMAGE) {	/* if float image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get source and destination buffers */
			float_buffer = get_plane_buffer(mimg->planes_array[c], type); 
			float_buffer_res = get_plane_buffer(mimg_res->planes_array[c], type);

				/* and apply the operation while coping to every pixel in the plane */
			for (i=0; i<pixels_count_plan; i++) {
				*float_buffer_res = float_op(*float_buffer, *((fmpixel_ptr)values_array+c));

				++float_buffer;
				++float_buffer_res;
			}
		}
	} else {	/* if discrete image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get source and destination buffers */
			buffer = get_plane_buffer(mimg->planes_array[c], type); 
			buffer_res = get_plane_buffer(mimg_res->planes_array[c], type);

				/* and apply the operation while coping to every pixel in the plane */
			for (i=0; i<pixels_count_plan; i++) {
				*buffer_res = gray_op(*buffer, *((mpixel_ptr)values_array+c));

				++buffer;
				++buffer_res;
			}
		}
	}
	 
	return mimg_res;	/* return the operation result image */
}

/**************************************************************************************
/*	Function:		binary_op_dmimage
/*	Description:	perform a binary operation between two dense images
/*
/*	Parameters:
/*		<-	mimg_a			first operand image
/*		<-	mimg_b			second operand image
/*		<-	gray_op			pointer to a binary operation function for integer case
/*		<-	float_op		pointer to a binary operation function for float case
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr binary_op_dmimage(mimage_ptr mimg_a, mimage_ptr mimg_b, 
	unsigned char (*gray_op)(unsigned char, unsigned char), float (*float_op)(float, float))
{
	int c, i, j;

	mimage_ptr mimg_res;

	int height, width, planes_number, type;
	int height_a, width_a, planes_number_a, type_a; 
	int height_b, width_b, planes_number_b, type_b; 

		/* get info about both the operand images */
	get_mimage_info(mimg_a, &height_a, &width_a, &planes_number_a, &type_a, NULL);
	get_mimage_info(mimg_b, &height_b, &width_b, &planes_number_b, &type_b, NULL);
		
		/* convert images to the widest common type */
	if (!make_compatible(mimg_a, mimg_b))
		return NULL;

		/* set the variables for the info of the result image */
	planes_number = planes_number_a;
	type = type_a;
	width = min(width_a, width_b);
	height = min(height_a, height_b);

		/* create the result image */
	mimg_res = create_mimage(width, height, planes_number, type, DENSE_IMAGE);
	if (!mimg_res)
		return NULL;
	  
	if (IS_DISCRETE(mimg_res)) {	/* if the image is discrete */

		mpixel_ptr buffer_a, buffer_b, buffer_res;
		
			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get pointer to buffers for operand and result images */
			buffer_a = get_plane_buffer(mimg_a->planes_array[c], type_a); 
			buffer_b = get_plane_buffer(mimg_b->planes_array[c], type_b);
			buffer_res = get_plane_buffer(mimg_res->planes_array[c], type);

				/* for every pixel */
			for (i=0; i<height * width; i++) {

					/* do the operation */
				*buffer_res = gray_op(*buffer_a, *buffer_b);

					/* advance buffer pointers */
				++buffer_a;
				++buffer_b;
				++buffer_res;
			}
		}

	} else {	/* FLOAT_IMAGE */

		fmpixel_ptr fbuffer_a, fbuffer_b, fbuffer_res;
		
			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get pointer to buffers for operand and result images */
			fbuffer_a = get_plane_buffer(mimg_a->planes_array[c], type_a); 
			fbuffer_b = get_plane_buffer(mimg_b->planes_array[c], type_b);
			fbuffer_res = get_plane_buffer(mimg_res->planes_array[c], type);

				/* for every pixel */
			for (i=0; i<width * height; i++) {

					/* do the operation */
				*fbuffer_res = float_op(*fbuffer_a, *fbuffer_b);

					/* advance buffer pointers */
				++fbuffer_a;
				++fbuffer_b;
				++fbuffer_res;
			}
		}
	}
	 
	return mimg_res;	/* return the result image */
}

/**************************************************************************************
/*	Function:		unimath_dmimage
/*	Description:	perform an "user defined" unary operation to a dense image
/*
/*	Parameters:
/*		<-	user_ptr		[private] used from the setl2 stubs
/*		<-	user_ptr2		[private] used from the setl2 stubs
/*		<-	mimg			operand dense image
/*		<-	floating_op		pointer to an external unary float operation
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr unimath_dmimage(void *user_ptr, void *user_ptr2, mimage_ptr mimg, 
	float (*floating_op)(void *user_ptr, void *user_ptr2, float, int))
{
	int c, i;
	size_t pixels_count_plan;
	float *float_buffer, *float_buffer_res;
	unsigned char *buffer, *buffer_res;
	mimage_ptr mimg_res;

	int height, width, planes_number, type; 

		/* get info about the image*/
	get_mimage_info(mimg, &height, &width, &planes_number, &type, NULL);

		/* create an image w/ the same specs */
	mimg_res = create_mimage(width, height, planes_number, type, DENSE_IMAGE);
	if (!mimg_res)	return NULL;

		/* count the number of pixels in a plane */
	pixels_count_plan = height * width;

	if (type == FLOAT_IMAGE) {	/* if float image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {
		
				/* get both source and result buffer */
			float_buffer = get_plane_buffer(mimg->planes_array[c], type); 
			float_buffer_res = get_plane_buffer(mimg_res->planes_array[c], type);

				/* for each pixel */
			for (i=0; i<pixels_count_plan; i++) {
					
					/* compute the operation calling the callback */
				*float_buffer_res = floating_op(user_ptr, user_ptr2, *float_buffer, c);

					/* advance to next pixel */
				++float_buffer;
				++float_buffer_res;
			}
		}

	} else {	/* if discrete image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {
		
				/* get both source and result buffer */
			buffer = get_plane_buffer(mimg->planes_array[c], type); 
			buffer_res = get_plane_buffer(mimg_res->planes_array[c], type);

				/* for each pixel */
			for (i=0; i<pixels_count_plan; i++) {
					
					/* compute the operation calling the callback */
				*buffer_res = floating_op(user_ptr, user_ptr2, (float)*buffer, c);

					/* advance to next pixel */
				++buffer;
				++buffer_res;
			}
		}
	}

	return mimg_res;	/* return the result image */
}

/**************************************************************************************
/*	Function:		unary_op_dmimage
/*	Description:	perform an unary operation over all pixels of a dense image
/*
/*	Parameters:
/*		<-	mimg			operand dense image
/*		<-	gray_op			pointer to an unary integer operation function for integer case
/*		<-	float_op		pointer to an unary float operation function for float case
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr unary_op_dmimage(mimage_ptr mimg, unsigned char (*gray_op)(unsigned char), float (*float_op)(float))
{
	int c, i;
	size_t pixels_count_plan;
	float *float_buffer, *float_buffer_res;
	unsigned char *buffer, *buffer_res;
	mimage_ptr mimg_res;
	int height, width, planes_number, type; 

		/* get info about the operand image */
	get_mimage_info(mimg, &height, &width, &planes_number, &type, NULL);

		/* create a new image w/ the same specs */
	mimg_res = create_mimage(width, height, planes_number, type, DENSE_IMAGE);
	if (!mimg_res)
		return NULL;

		/* count the number of pixel per plane */
	pixels_count_plan = height * width;

	if (type == FLOAT_IMAGE) {	/* if float image */

			/* for each plane */
		for (c=0; c<planes_number; c++) {
		
				/* get both source and result buffer */
			float_buffer = get_plane_buffer(mimg->planes_array[c], type); 
			float_buffer_res = get_plane_buffer(mimg_res->planes_array[c], type);

				/* for each pixel */
			for (i=0; i<pixels_count_plan; i++) {

					/* compute the operation */
				*float_buffer_res = float_op(*float_buffer);

					/* advance to next pixel */
				++float_buffer;
				++float_buffer_res;
			}
		}

	} else {

		for (c=0; c<planes_number; c++) {	/* if discrete image */
		
				/* get both source and result buffer */
			buffer = get_plane_buffer(mimg->planes_array[c], type); 
			buffer_res = get_plane_buffer(mimg_res->planes_array[c], type);

				/* for each pixel */
			for (i=0; i<pixels_count_plan; i++) {

					/* compute the operation */
				*buffer_res = gray_op(*buffer);

					/* advance to next pixel */
				++buffer;
				++buffer_res;
			}
		}

	}

	return mimg_res;	/* return the result image */
}

/**************************************************************************************
/*	Function:		extract_planes_dmimage
/*	Description:	create a dense image extracting a set of planes from another
/*
/*	Parameters:
/*		<-	mimg		source dense image
/*		<-	from		first plane
/*		<-	to			last plane
/*
/*	Result:
/*		not NULL	a pointer to the extracted dense image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr extract_planes_dmimage(mimage_ptr mimg, int from, int to)
{
	int c, d;
	void *buffer, *res_buffer;
	mimage_ptr mimg_res;
	int width, height, planes_number, kind;
	size_t type_size;

		/* check if range of planes is valid (>0) */
	if (from > to)
		return NULL;

		/* get info about the image*/	
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);	/* get the pixel size */

		/* create a new image w/ the requested number of planes */
	mimg_res = create_mimage(width, height, (to-from+1), kind, DENSE_IMAGE);
	if (!mimg_res)
		return NULL;

		/* loop over the planes to be copied */
	for (d=0, c=from; c<=to; d++, c++) {

			/* get the pointers to the buffers */
		buffer = get_byteplane_from_mimage(mimg, c)->buffer;
		res_buffer = get_byteplane_from_mimage(mimg_res, d)->buffer;

			/* and copy the contents of the buffes */
		memcpy(res_buffer, buffer, width*height*type_size);
	}

	return mimg_res;	/* return the result image */
}

/**************************************************************************************
/*	Function:		flip_h_dense
/*	Description:	create a flipped horizontally dense image
/*
/*	Parameters:
/*		<-	mimg	source dense image
/*
/*	Result:
/*		not NULL	pointer to the flipped dense image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr flip_h_dense(mimage_ptr mimg)
{
	int i, j ,c;
	int width, height, planes_number, kind;
	mimage_ptr res_mimg;
	
		/* get info about the source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* create a new image w/ same specs of the original one */
	res_mimg = create_mimage(width, height, planes_number, kind, mimg->density);

	if (res_mimg) {	/* if creation on the image went fine */

		if (kind == DISCRETE_IMAGE) {	/* if the image is discrete*/
		
			mpixel_ptr buffer, dest_buffer, dest_buffer_start;
		
			for ( c=0 ; c<planes_number ; c++ ) {	/* for each plane */
			
					/* gest source and destination buffers */
				buffer = get_byteplane_from_mimage(mimg, c)->buffer;
				dest_buffer_start = get_byteplane_from_mimage(res_mimg, c)->buffer;

					/* for every line */
				for ( j=0; j<height; j++ ) {

						/* get this line dest buffer */
					dest_buffer = dest_buffer_start + (j + 1) * width;
				
						/* and make the horizontal flipping of the pixels */
					for ( i=0; i<width ; i++) 
						* -- dest_buffer = * buffer ++;
				}
			}
		} else {	/* if the image is float */
		
			fmpixel_ptr fbuffer, fdest_buffer, fdest_buffer_start;
		
			for ( c=0 ; c<planes_number ; c++ ) {	/* for each plane */
			
					/* gest source and destination buffers */
				fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
				fdest_buffer_start = get_fbyteplane_from_mimage(res_mimg, c)->buffer;

					/* for every line */
				for ( j=0; j<height; j++ ) {

						/* get this line dest buffer */
					fdest_buffer = fdest_buffer_start + (j + 1) * width;
				
						/* and make the horizontal flipping of the pixels */
					for ( i=0; i<width ; i++) 
						* -- fdest_buffer = * fbuffer ++;
				}
			}
		}
	}

	return res_mimg;	/* return the result image */
}

/**************************************************************************************
/*	Function:		flip_v_dense
/*	Description:	create a flipped vertically dense image
/*
/*	Parameters:
/*		<-	mimg	source dense image
/*
/*	Result:
/*		not NULL	pointer to the flipped dense image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr flip_v_dense(mimage_ptr mimg)
{
	int i, j ,c;
	int width, height, planes_number, kind;
	mimage_ptr res_mimg;
	
		/* get info about the source image */		
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* create a new image w/ same specs of the original one */
	res_mimg = create_mimage(width, height, planes_number, kind, mimg->density);

	if (res_mimg) {	/* if creation on the image went fine */

		if (kind == DISCRETE_IMAGE) {	/* if the image is discrete */
	
			mpixel_ptr buffer, dest_buffer, dest_buffer_start;
		
				/* loop over the planes */
			for ( c=0 ; c<planes_number ; c++ ) {
			
					/* get the pointer to the buffers */
				buffer = get_byteplane_from_mimage(mimg, c)->buffer;
				dest_buffer_start = get_byteplane_from_mimage(res_mimg, c)->buffer;

					/* for every line */
				for ( j=0; j<height; j++ ) {

						/* get this line dest buffer */
					dest_buffer = dest_buffer_start + (height - 1 - j) * width;
				
						/* and make the vertical flipping of the pixels */
					for ( i=0; i<width ; i++) 
						* dest_buffer ++ = * buffer ++;
				}
			}
		} else {	/* if the image is float */

			fmpixel_ptr buffer, dest_buffer, dest_buffer_start;
		
				/* loop over the planes */
			for ( c=0 ; c<planes_number ; c++ ) {
			
					/* get the pointer to the buffers */
				buffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
				dest_buffer_start = get_fbyteplane_from_mimage(res_mimg, c)->buffer;

					/* for every line */
				for ( j=0; j<height; j++ ) {

						/* get this line dest buffer */
					dest_buffer = dest_buffer_start + (height - 1 - j) * width;
								
						/* and make the vertical flipping of the pixels */
					for ( i=0; i<width ; i++) 
						* dest_buffer ++ = * buffer ++;
				}
			}
		}
	}

	return res_mimg;	/* return the result image */
}

/**************************************************************************************
/*	Function:		rotate_90_dense
/*	Description:	return a 90 degrees rotated image
/*
/*	Parameters:
/*		<-	mimg	source dense image
/*
/*	Result:
/*		not NULL	pointer to the rotated image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr rotate_90_dense(mimage_ptr mimg)
{
	mimage_ptr res_mimg;
	int i, j, c;
	int width, height, planes_number, kind;
	int line_length;
	
		/* get info about the image to rotate */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* note that height and width are appositely exchanged for rotation */
	res_mimg = create_mimage(height, width, planes_number, kind, mimg->density);

	if (res_mimg) {	/* if image creation was succesfull */

		if (kind == DISCRETE_IMAGE) {	/* if image is discrete */

			mpixel_ptr buffer, dest_buffer, start_buffer;
		
				/* for each plane */
			for ( c=0 ; c<planes_number ; c++ )	{
			
				line_length = height;
			
					/* get the pointer to the buffers */
				buffer = get_byteplane_from_mimage(mimg, c)->buffer;
				start_buffer = get_byteplane_from_mimage(res_mimg, c)->buffer;

					/* loop through the lines */
				for ( j=0 ; j<height ; j++ ) {

						/* calculate the position in the destination buffer */
					dest_buffer = start_buffer + (height - j - 1);

						/* and copy the source row in the destination column */
					for ( i=0; i<width ; i++ ) {
						* dest_buffer = * buffer ++;
						dest_buffer += line_length;
					}
				}

			}
		} else {	/* if image is float */
		
			fmpixel_ptr buffer, dest_buffer, start_buffer;
		
				/* for each plane */
			for ( c=0 ; c<planes_number ; c++ )	{
			
				line_length = height;
			
					/* get the pointer to the buffers */
				buffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
				start_buffer = get_fbyteplane_from_mimage(res_mimg, c)->buffer;

					/* loop through the lines */
				for ( j=0 ; j<height ; j++ ) {
					dest_buffer = start_buffer + (height - j - 1);

						/* calculate the position in the destination buffer */
					for ( i=0; i<width ; i++ ) {
						* dest_buffer = * buffer ++;
						dest_buffer += line_length;
					}
				}

			}
		}
	}

	return res_mimg;	/* return the rotated image */
}

/**************************************************************************************
/*	Function:		rotate_180_dense
/*	Description:	return a 180 degrees rotated image
/*
/*	Parameters:
/*		<-	mimg	source dense image
/*
/*	Result:
/*		not NULL	pointer to the rotated image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr rotate_180_dense(mimage_ptr mimg)
{
	mimage_ptr res_mimg;
	
	int i, c;
	int width, height, planes_number, kind;
	
		/* get info about the image to rotate */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* create the result image */
	res_mimg = create_mimage(width, height, planes_number, kind, mimg->density);

	if (res_mimg) {	/* if image creation was successful */

		if (kind == DISCRETE_IMAGE) {	/* if image is discrete */

			mpixel_ptr buffer, dest_buffer;
		
				/* for each plane */
			for ( c=0 ; c<planes_number ; c++ ) {
			
					/* get the pointer to the buffers */
				buffer = get_byteplane_from_mimage(mimg, c)->buffer;
				dest_buffer = get_byteplane_from_mimage(res_mimg, c)->buffer;

				dest_buffer += height * width - 1;	/* go to the end */

					/* loop through the pixels and make the rotation */
				for ( i=0; i<height * width ; i++ )
					* dest_buffer -- = * buffer ++;

			}
		} else {	/* if image is float */

			fmpixel_ptr buffer, dest_buffer;
		
				/* for each plane */
			for ( c=0 ; c<planes_number ; c++ )	{
			
					/* get the pointer to the buffers */
				buffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
				dest_buffer = get_fbyteplane_from_mimage(res_mimg, c)->buffer;
				
				dest_buffer += height * width - 1;	/* go to the end */

					/* loop through the pixels and make the rotation */
				for ( i=0; i<height * width ; i++ )
					* dest_buffer -- = * buffer ++;

			}
		}
	}

	return res_mimg;
}

/**************************************************************************************
/*	Function:		rotate_270_dense
/*	Description:	return a 270 degrees rotated image
/*
/*	Parameters:
/*		<-	mimg	source dense image
/*
/*	Result:
/*		not NULL	pointer to the rotated image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr rotate_270_dense(mimage_ptr mimg)
{
	mimage_ptr res_mimg;
	
	int i, j, c;
	int width, height, planes_number, kind;
	int line_length;
	
		/* get info about the image to rotate */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* note that height and width are appositely exchanged */
	res_mimg = create_mimage(height, width, planes_number, kind, mimg->density);

	if (res_mimg) {	/* if image creation was successful */

		if (kind == DISCRETE_IMAGE) {	/* if is discrete */

			mpixel_ptr buffer, dest_buffer, start_buffer;
		
				/* for each plane */
			for ( c=0 ; c<planes_number ; c++ )	{
			
				line_length = height;
			
					/* get the pointer to the buffers */
				buffer = get_byteplane_from_mimage(mimg, c)->buffer;
				start_buffer = get_byteplane_from_mimage(res_mimg, c)->buffer;

					/* loop through the lines */
				for ( j=0 ; j<height ; j++ ) {
						/* go to last row */
					dest_buffer = start_buffer + j
						+ (width - 1) * height;

						/* do the rotation */
					for ( i=0; i<width ; i++ ) {
						* dest_buffer = * buffer ++;
						dest_buffer -= line_length;
					}
				}
			}
		} else {	/* if is float */

			fmpixel_ptr buffer, dest_buffer, start_buffer;
		
				/* for each plane */
			for ( c=0 ; c<planes_number ; c++ )	{
			
				line_length = height;
			
					/* get the pointer to the buffers */
				buffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
				start_buffer = get_fbyteplane_from_mimage(res_mimg, c)->buffer;

					/* loop through the lines */
				for ( j=0 ; j<height ; j++ ) {
						/* go to last row */
					dest_buffer = start_buffer + j
						+ (width - 1) * height;

						/* do the rotation */
					for ( i=0; i<width ; i++ ) {
						* dest_buffer = * buffer ++;
						dest_buffer -= line_length;
					}
				}
			}
		}
	}

	return res_mimg;	/* return the rotated image */
}

/**************************************************************************************
/*	Function:		sumall_mimage_dense
/*	Description:	return the momentums of a dense image
/*					
/*	Parameters:
/*		<-	mimg	operand dense image
/*		<-	power	highest momentum requested
/*
/*	Result:
/*		not NULL	array of requested momentums
/*		NULL		out of memory
/*
/***************************************************************************************/

double *sumall_mimage_dense(mimage_ptr mimg, int power)
{
	int i, c, k;
	double *fsum;
	unsigned char *buffer;
	float *float_buffer;
	int height, width, comp, kind; 
	size_t pixels_count_plan;

		/* alloc memory for returning momentums array */
	fsum = calloc(power, sizeof(*fsum));
	if (!fsum)
		return NULL;	

		/* get info about the image */
	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);

		/* get the count of the pixels in the image */
	pixels_count_plan = height * width;

	if (kind == FLOAT_IMAGE) {	/* if the image is float */

			/* loop over planes */
		for (c=0; c<comp; c++) {

				/* get the pointer to the buffer in this plane */
			float_buffer = get_plane_buffer(mimg->planes_array[c], kind);

				/* for each pixel */
			for(i=0; i<pixels_count_plan; i++, ++float_buffer) 
				for (k=0;k<power;k++) 
					fsum[k] += pow(*float_buffer, k);	/* update the k-th momentum */
		}      	

	} else {	/* if the image is discrete */

			/* loop over planes */
		for (c=0; c<comp; c++) {

				/* get the pointer to the buffer in this plane */
			buffer = get_plane_buffer(mimg->planes_array[c], kind);

				/* for each pixel */
			for(i=0; i<pixels_count_plan; i++, ++buffer) 
				for (k=0;k<power;k++)
					fsum[k] += pow(*buffer, k);	/* update the k-th momentum */
		}      	

	}

	return fsum;	/* return the requested array */
}

/**************************************************************************************
/*	Function:		maxof_mimage_dense
/*	Description:	return the max pixel value of a dense image
/*					
/*	Parameters:
/*		<-	mimg	operand dense image
/*
/*	Result:
/*		the maximum
/*
/***************************************************************************************/

float maxof_mimage_dense(mimage_ptr mimg)
{ 

	int c, i;
	float *float_buffer;
	unsigned char *buffer;
	size_t pixels_count_plan;
	float fmax = 0.;
	unsigned char max = 0;
	int height, width, comp, kind; 
	
		/* get info about the image */
	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);

		/* get the number id pixels in the image */
	pixels_count_plan = height * width;

	if (kind == FLOAT_IMAGE) {	/* if the image is float */

		for (c=0; c<comp; c++) {	/* for each plane */

				/* get the pointer to the buffer in this plane*/
			float_buffer = get_plane_buffer(mimg->planes_array[c], kind);

				/* for each pixel */
			for(i=0; i<pixels_count_plan; i++) {
				if (*float_buffer > fmax)	/* update the max value */
					fmax = *float_buffer;
				++float_buffer;
			}
		}      	

	} else {	/* if the image is discrete */

		for (c=0; c<comp; c++) {	/* for each plane */

				/* get the pointer to the buffer in this plane*/
			buffer = get_plane_buffer(mimg->planes_array[c], kind);

				/* for each pixel */
			for(i=0; i<pixels_count_plan; i++) {
				if (*buffer > max)	/* update the max value */
					max = *buffer;
				++buffer;
			}
		}      	

		fmax = max;	/* set the obtained max to the return value */
	}

	return fmax;	/* return the max */
}

/**************************************************************************************
/*	Function:		minof_mimage_dense
/*	Description:	return the min pixel value of a dense image
/*					
/*	Parameters:
/*		<-	mimg	operand dense image
/*
/*	Result:
/*		the maximum
/*
/***************************************************************************************/

float minof_mimage_dense(mimage_ptr mimg)
{
	int c, i;
	float *float_buffer;
	unsigned char *buffer;
	size_t pixels_count_plan;
	float fmin = BIGFLOAT;
	unsigned char min = (char)MAX_UCHAR;
	int height, width, comp, kind; 

		/* get info about the image */	
	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);

		/* get the number of pixels in this image */
	pixels_count_plan = height * width;

	if (kind == FLOAT_IMAGE) {	/* if the image is float */

			/* loop through the planes */
		for (c=0; c<comp; c++) {

				/* get the pointer to the buffer for this plane */
			float_buffer = get_plane_buffer(mimg->planes_array[c], kind);

				/* loop over the pixels */
			for(i=0; i<pixels_count_plan; i++) {
				if (*float_buffer < fmin)	/* update the minimum value */
					fmin = *float_buffer;
				++float_buffer;
			}
		}      	

	} else {	/* if the image is integer */

			/* loop through the planes */
		for (c=0; c<comp; c++) {

				/* get the pointer to the buffer for this plane */
			buffer = get_plane_buffer(mimg->planes_array[c], kind);

				/* loop over the pixels */
			for(i=0; i<pixels_count_plan; i++) {
				if (*buffer < min)	/* update the minimum value */
					min = *buffer;
				++buffer;
			}
		}      	

		fmin = min;	/* set the found minimum to the return value */
	}

	return fmin;	/* return the minimum */
}

/**************************************************************************************
/*	Function:		sort_mimage_dense
/*	Description:	return an array of sorted dense image values
/*					
/*	Parameters:
/*		<-	mimg		operand dense image
/*		->	elements	return the number of elements in the sorted array
/*
/*	Result:
/*		not NULL	array of sorted values
/*		NULL		out of memory
/*
/***************************************************************************************/

void *sort_mimage_dense(mimage_ptr mimg, int *elements) 
{
	int i, j;
	int c;
	char *list;
	void *buffer;
	float f_old, *f_list;
	unsigned char old, *u_list;
	int height, width, comp, kind; 
	size_t pixels_count_plan;
	size_t type_size;
	 
		/* get info about the image */
	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);
	
		/* get the number of pixels in the image */
	pixels_count_plan = height * width;
	
		/* get the size in bytes of a pixel */
	type_size = GET_TYPE_SIZE(kind);

		/* allocate memory to hold the list */
	list = malloc(pixels_count_plan * type_size * comp);
	if (!list)
		return NULL;

		/* for each plane */
	for (c=0; c<comp; c++) {
			/* get the pointer to the buffer for this plane */
		buffer = get_plane_buffer(mimg->planes_array[c], kind);
			/* append the data from this buffer in the allocated list */
		memcpy( list + c * type_size * pixels_count_plan, buffer, type_size * pixels_count_plan);
	}
		
	if (kind == FLOAT_IMAGE) {	/* if the image is float */

			/* sort the data using qsort */
		qsort(list, pixels_count_plan * comp, type_size, (int (*)(const void *, const void *))float_compare);

			/* 
			 * remove duplicated elements 
			 */
			 
		f_list = (float *)list;
		f_old = f_list[0];

		for (i=1, j=1; i<pixels_count_plan * comp; i++) {
			if (f_list[i] != f_old) {
				f_old = f_list[j] = f_list[i];
				j++;
			}
		}
		
	} else {	/* if the image is discrete */

			/* sort the data using qsort */
		qsort(list, pixels_count_plan * comp, type_size, (int (*)(const void *, const void *))unsigned_char_compare);

			/*
			 * remove duplicated elements 
			 */

		u_list = (unsigned char *)list;
		old = u_list[0];

		for (i=1, j=1; i<pixels_count_plan * comp; i++) {
			if (u_list[i] != old) {
				old = u_list[j] = u_list[i];
				j++;
			}
		}
	}

	list = realloc(list, j*type_size); /* shrinks the list to the right size */
	*elements = j;	/* return the number of elements in list */

	return list;	/* return the list */
}

/**************************************************************************************
/*	Function:		histogram_mimage_dense
/*	Description:	return an histogram of the pixel values from a dense image
/*
/*	Parameters:
/*		<-	mimg			source dense image
/*		<-	fvalues			array of points
/*		<-	fvalues_size	size of the array of points
/*
/*	Result:
/*		not NULL	a pointer to the histogram array
/*		NULL		error condition
/*
/***************************************************************************************/

size_t *histogram_mimage_dense(mimage_ptr mimg, float *fvalues, size_t fvalues_size)
{
	unsigned char *buffer;
	float *fbuffer;
	size_t range_value;
	size_t *range_no;
	int c,i;
	int width, height, planes_number, kind;
	size_t type_size;

		/* get info about the image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);	/* get the size of a pixel */
	
		/* allocate an array for the return values */
	range_no = (size_t *)calloc((fvalues_size + 1) * sizeof(size_t), 1);
	if (!range_no)
		return NULL;

		/* sort the values */
	qsort(fvalues, fvalues_size, sizeof(float), 
		(int (*)(const void *, const void *))float_compare);

	if (kind == FLOAT_IMAGE) {	/* if the image is float */
	
			/* for each plane */
		for (c=0; c<planes_number; c++) {
				/* get the pointer to the buffer for this plane */
			fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;

				/* for each pixel */
			for (i=0; i<width*height; i++) {
					/* check in which range is the value */
				range_value = bfbsearch(fbuffer[i], fvalues, fvalues_size) + 1;
					/* and update that range */
				range_no[range_value]++;
			}
		}

	} else {	/* if the image is integer */
	
			/* for each plane */
		for (c=0; c<planes_number; c++) {
				/* get the pointer to the buffer for this plane */
			buffer = get_byteplane_from_mimage(mimg, c)->buffer;

				/* for each pixel */
			for (i=0; i<width*height; i++) {
					/* check in which range is the value */
				range_value = bfbsearch((float)buffer[i], fvalues, fvalues_size) + 1;
					/* and update that range */
				range_no[range_value]++;
			}
		}
	}

	return range_no;	/* return the histogram array */
}

/**************************************************************************************
/*	Function:		threshold_mimage_dense
/*	Description:	create a dense image as a threshold from another
/*
/*	Parameters:
/*		<-	mimg			source dense image
/*		<-	fvalues			array of thresholds
/*		<-	fvalues_size	size of the array of thresholds
/*
/*	Result:
/*		not NULL	a pointer to the thresholded image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr threshold_mimage_dense(mimage_ptr mimg, float *fvalues, size_t fvalues_size)
{
	unsigned char *buffer;
	unsigned char *res_buffer;
	float *fbuffer;
	float *fres_buffer;
	int c,i;
	mimage_ptr res_mimg;
	int width, height, planes_number, kind;
	size_t type_size;

		/* get info about the source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
		/* get the size of a single pixel in bytes */
	type_size = GET_TYPE_SIZE(kind);
	
		/* create the resul image */
	res_mimg = create_mimage(width, height, planes_number, kind, DENSE_IMAGE);
	if (!res_mimg)
		return NULL;

		/* sort the array of thresholdes */
	qsort(fvalues, fvalues_size, sizeof(float), (int (*)(const void *, const void *))float_compare);

	if (kind == FLOAT_IMAGE) {	/* if the image is float */
			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get the pointer to the buffers for this plane */
			fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
			fres_buffer = get_fbyteplane_from_mimage(res_mimg, c)->buffer;

				/* for each pixel */
			for (i=0; i<width*height; i++)
						/* threshold the dest pixel */
				fres_buffer[i] = threshold_pixel(fbuffer[i], fvalues, fvalues_size);
		}

	} else {	/* if the image is discrete */

			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get the poitner to the buffers for this plane */
			buffer = get_byteplane_from_mimage(mimg, c)->buffer;
			res_buffer = get_byteplane_from_mimage(res_mimg, c)->buffer;

				/* for each pixel */
			for (i=0; i<width*height; i++)
						/* threshold the dest pixel */
				res_buffer[i] = (unsigned char)threshold_pixel((float)buffer[i], fvalues, fvalues_size);
		}
	}

	return res_mimg;	/* return the destination image */
}

/**************************************************************************************
/*	Function:		get_pixel_mimage_dense
/*	Description:	return an array with the values (if more than one plane)
/*					for a pixel at a given position in the dense image mimg 
/*
/*	Parameters:
/*		<-	mimg		source dense image
/*		<-	x			x coordinate of the pixel
/*		<-	y			y coordinate of the pixel
/*
/*	Result:
/*		not NULL	a pointer to the array (float if the image is float, else integer)
/*		NULL		error condition
/*
/***************************************************************************************/

void *get_pixel_mimage_dense(mimage_ptr mimg, int x, int y)
{
	int c;
	int height, width, planes_number, kind;
 	size_t type_size;
	void *pixel_value_array;
	mpixel_ptr buffer;
	fmpixel_ptr fbuffer;

		/* get info about the image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	
		/* check if the pixel position is valid */
	if (x<0 || x>width || y<0 || y>height)
		return NULL;
  	
		/* get the pixel size */
	type_size = GET_TYPE_SIZE(kind);

		/* allocate memory to return */
	pixel_value_array = malloc(planes_number * type_size);
	if (!pixel_value_array)
		return NULL;
		
	if (kind == FLOAT_IMAGE) {	/* if the image is float */
			/* loop over planes */
		for (c=0; c<planes_number; c++) {
				/* get the pointer to the buffer for this plane */
			fbuffer = get_plane_buffer(mimg->planes_array[c], kind);
				/* and get the needed pixel */
			*((fmpixel_ptr)pixel_value_array + c) = *(fbuffer + y*width + x);
		}			
	} else {	/* if the image is discrete */
			/* loop over planes */
		for (c=0; c<planes_number; c++) {
				/* get the pointer to the buffer for this plane */
			buffer = get_plane_buffer(mimg->planes_array[c], kind);
				/* and get the needed pixel */
			*((mpixel_ptr)pixel_value_array + c) = *(buffer + y*width + x);
		}
	}

	return pixel_value_array;	/* return the array containing the pixel values */
}

/**************************************************************************************
/*	Function:		set_pixel_mimage_dense
/*	Description:	set a new values for a pixel in a dense image
/*
/*	Parameters:
/*		<-	mimg				source dense image
/*		<-	x					x coordinate of the pixel
/*		<-	y					y coordinate of the pixel
/*		<-	pixel_value_array	new values for the pixels
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int set_pixel_mimage_dense(mimage_ptr mimg, int x, int y, void *pixel_value_array)
{
	int c;
	int height, width, planes_number, kind;
 	size_t type_size;
	mpixel_ptr buffer;
	fmpixel_ptr fbuffer;

		/* get info about the image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	
		/* check if the pixel position is valid */
	if (x<0 || x>width || y<0 || y>height)
  		return -1;
  	
		/* get the pixel size */
	type_size = GET_TYPE_SIZE(kind);
		
	if (kind == FLOAT_IMAGE) {	/* if the image is float */
			/* loop over planes */		
		for (c=0; c<planes_number; c++) {
			fbuffer = get_plane_buffer(mimg->planes_array[c], kind);
				/* and set the pixel for that plane */
			*(fbuffer + y*width + x) = *((fmpixel_ptr)pixel_value_array + c);
		}			
	} else {	/* if the image is discrete */
			/* loop over planes */
		for (c=0; c<planes_number; c++) {
			buffer = get_plane_buffer(mimg->planes_array[c], kind);
				/* and set the pixel for that plane */
			*(buffer + y*width + x) = *((mpixel_ptr)pixel_value_array + c);
		}
	}

	return 0;	/* ok */	
}

/**************************************************************************************
/*	Function:		set_planes_mimage_dense
/*	Description:	insert a set of planes from the dense
/*					image mimg to the dense image mimg_res
/*
/*	Parameters:
/*		<->	mimg_res	destintaion dense image
/*		<-	mimg		source dense image
/*		<-	from		first plane
/*		<-	to			last plane
/*
/*	Result:
/*		not NULL	the dense mimg_res itself with the inseted planes
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr set_planes_mimage_dense(mimage_ptr mimg_res, mimage_ptr mimg, int from, int to)
{
	int c, d;
	void *buffer, *res_buffer;
	int width, height, planes_number, kind;
	int width_res, height_res, planes_number_res, kind_res;
	size_t type_size;

		/* if range of planes is invalid, exit */
	if (from > to)
		return NULL;

		/* get info about the images */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	get_mimage_info(mimg_res, &height_res, &width_res, &planes_number_res, &kind_res, NULL);

		/* if images are not compatible, exit */
	if ((width_res != width) || (height_res != height) || (kind_res != kind))
		return NULL;
		
		/* if number of planes are same number of planes to copy, exit */
	if ((to-from+1) != planes_number)
		return NULL;

		/* get size of pixel in bytes */
	type_size = GET_TYPE_SIZE(kind);

		/* loop through the planes should be copied */
	for (c=from, d=0; c<=to; c++, d++) {

			/* get the pointer to source and destination buffers */
		buffer = get_byteplane_from_mimage(mimg, d)->buffer;
		res_buffer = get_byteplane_from_mimage(mimg_res, c)->buffer;

			/* and copy the data of the planes */
		memcpy(res_buffer, buffer, width*height*type_size);
	}

	return mimg_res;
}

/**************************************************************************************
/*	Function:		random_mimage_dense
/*	Description:	generate a random filled dense image 
/*					after the specs of another one
/*
/*	Parameters:
/*		<-	mimg	source dense image
/*
/*	Result:
/*		not NULL	a pointer to the result dense image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr random_mimage_dense(mimage_ptr mimg, float r_min, float r_max)
{
	clock_t seed;
	int i, c;
	mimage_ptr target_mimg;
	int width, height, planes_number, kind;
	int number_of_pixels;

		/* get info about the image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* create target image */
	target_mimg = create_mimage(width, height, planes_number, kind, DENSE_IMAGE);
	if (!target_mimg)
		return NULL;

		/* calculate number of pixels */
	number_of_pixels = width * height;


		/*
		 *	Set the seed to a quasi casual value
		 */

	seed = clock();
	srand(seed);
	
	if (kind == DISCRETE_IMAGE)	{	/* if the image is discrete */
		mpixel_ptr buffer;

			/* range for generated pseudo-random values */
		int i_min = r_min;
		int i_max = r_max;
		int i_range = i_max - i_min;
		
			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get the buffer */
			buffer = get_byteplane_from_mimage(target_mimg, c)->buffer;
			
				/* and set the pixel to random values */
			for (i=0 ; i<number_of_pixels ; i++)
				*buffer++ = (((float)rand())/RAND_MAX) * i_range + i_min;
			
		}
	} else {	/* if the image is float */
		fmpixel_ptr fbuffer;

			/* range for generated pseudo-random values */
		float f_range = r_max - r_min;
		
			/* for each plane */
		for (c=0; c<planes_number; c++) {

				/* get the buffer */
			fbuffer = get_fbyteplane_from_mimage(target_mimg, c)->buffer;
			
				/* and set the pixel to random values */
			for (i=0 ; i<number_of_pixels ; i++)
				*fbuffer++ = (((float)rand())/RAND_MAX) * f_range + r_min;
			
		}
	}

	return target_mimg;	/* return the random image */
}

/**************************************************************************************
/*	Function:		crop_mimage_dense
/*	Description:	crop a dense image
/*
/*	Parameters:
/*		<-> mimg	pointer to the dense image
/*		<- rect		rect for crop
/*
/*	Result:
/*		0		ok
/*		else	if an error occurs
/*
/***************************************************************************************/

int crop_mimage_dense(mimage_ptr mimg, image_rect_ptr rect)
{
	int planes_number, width, height, kind;
	void **new_planes_array, **old_planes_array;
	image_rect from_rect;
	int i;

		/* get info  about the source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* calculate the crop margins */
	rect->dx = min(rect->dx, width - rect->x );
	rect->dy = min(rect->dy, height - rect->y );

	from_rect.dx = width;
	from_rect.dy = height;

		/* create a new array of planes of the rect size */
	new_planes_array = (void **)create_planes_array(rect->dx, rect->dy, kind, planes_number, DENSE_IMAGE);
	if (!new_planes_array)
		return -1;

		/* get a pointer to the old planes array */
	old_planes_array = (void **)mimg->planes_array;

		/* for each plane */
	for (i=0; i<planes_number; i++) 
			/* crop the plane and store in destination */
		crop_from_plane_to_plane(old_planes_array[i], new_planes_array[i], kind, &from_rect, rect);

		/* destroy old planes array */
	destroy_planes_array(old_planes_array, kind, planes_number, DENSE_IMAGE);
	mimg->planes_array = (byteplane_ptr *)new_planes_array;	/* set new planes to the image */

	return 0;
}

/**************************************************************************************
/*	Function:		invert_mimage_dense
/*	Description:	invert the pixels of a dense image [in place]
/*
/*	Parameters:
/*		<->	mimg	source dense image
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int	invert_mimage_dense(mimage_ptr mimg)
{
	int i, c;
	int planes_number, width, height, kind;
	size_t pixels_per_plane;
	fmpixel_ptr fbuffer;
	mpixel_ptr buffer;
	
		/* get info about the image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* get the number of pixels in an image */
	pixels_per_plane = height * width;

	if (kind == DISCRETE_IMAGE) {	/* if the image is discrete */
			
			/* for each plane */
		for (c=0; c<planes_number; c++) {
				/* get the pointer to the buffer for this plane */
			buffer = get_byteplane_from_mimage(mimg, c)->buffer;
				/* and invert every pixel in the plane */
			for (i=0; i<pixels_per_plane; i++) {
				*buffer = 0xff - *buffer;	/* inversion */
				buffer++;
			}
		}
	} else {	/* if the image is float */

			/* for each plane */
		for (c=0; c<planes_number; c++) {
				/* get the pointer to the buffer for this plane */
			fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
				/* and invert every pixel in the plane */
			for (i=0; i<pixels_per_plane; i++) {
				*fbuffer =  -*fbuffer;		/* inversion */
				fbuffer++;
			}
		}
	}

	return 0;
}

/**************************************************************************************
/*	Function:		lut_mimage_dense
/*	Description:	apply a Look Up Table to a dense image
/*
/*	Parameters:
/*		<-	mimg				source dense image
/*		<-	LUTs				matrix with the LUT values
/*		<-	number_of_planes	number of planes in target
/*		<-	target_kind			kind of target dense image
/*
/*	Result:
/*		not NULL	pointer to the generated dense image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr lut_mimage_dense(mimage_ptr mimg, mpixel *LUTs[], int planes_number, int target_kind)
{
	int i,c;
	int pixels_per_plane;
	int width, height, kind;
	mimage_ptr target_mimg;

		/* get information about the image */
	get_mimage_info(mimg, &height, &width, NULL, &kind, NULL);

		/* create the target image */
	target_mimg = create_mimage(width, height, planes_number, target_kind, DENSE_IMAGE);
	if (!target_mimg)
		return NULL;
		
		/* get the number of pixels per plane */
	pixels_per_plane = height * width;
	
	if (IS_DISCRETE(target_mimg)) {	/* if the image is discrete */
		
			/* loop over the planes */
		for (c=0; c<planes_number; c++) {

				/* get the pointer to the buffers for this plane */
			mpixel_ptr buffer = get_byteplane_from_mimage(mimg, 0)->buffer;
			mpixel_ptr dest_buffer = get_byteplane_from_mimage(target_mimg, c)->buffer;

				/* loop over the pixels of this plane */
			for (i=0; i<pixels_per_plane; i++) {

				*dest_buffer = LUTs[c][*buffer];	/* apply the LUT to the value */

				dest_buffer++;						/* advance to next pixel */
				buffer++;
			}
		}

	} else {	/* if the image is float */
		
			/* loop over the planes */
		for (c=0; c<planes_number; c++) {

				/* get the pointers to the buffers for this plane */
			mpixel_ptr buffer = get_byteplane_from_mimage(mimg, 0)->buffer;	/* get the buffer for this plane */
			fmpixel_ptr fdest_buffer = get_fbyteplane_from_mimage(target_mimg, c)->buffer;	/* get the buffer for this plane */

				/* loop over the pixels of this plane */
			for (i=0; i<pixels_per_plane; i++) {

				*fdest_buffer = (fmpixel)LUTs[c][*buffer];		/* apply the LUT to the value */

				fdest_buffer++;									/* advance to next pixel */
				buffer++;
			}
		}
	}

	return target_mimg;	/* return the result image */
}

/**************************************************************************************
/*	Function:		lex_minof_mimage_dense
/*	Description:	return the position of the lexicographically
/*					minimum of a dense image
/*
/*	Parameters:
/*		<-	mimg	source dense image
/*		->	pos		position of the pixel that is minimum
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int lex_minof_mimage_dense(mimage_ptr mimg, gr_image_displacement *pos)
{
	int i;
	int *d;
	int height, width, planes_number, kind; 
	size_t pixels_count_plan;
	int pix_min_pos;
	
		/* get information about the image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
		/* count the number of pixel in this plane*/
	pixels_count_plan = height * width;
	
		/* 
		 * retrieve in a vector d the distance between the planes 
		 * that is: 
		 *    if we pointers to plane: p0, p1, p2, pn
		 *    the distances (n-1) will be then: d[0] = p1-p0, d[1] = p2-p1, ..., d[n-1] = pn-p(n-1)
		 */
	d = get_planes_distances_for_mimage(mimg);
	if (!d)	
		return -1;
	
	if (IS_DISCRETE(mimg)) {	/* if the image is discrete */

		mpixel *pix_min;
		mpixel *cur_pix;
		
		pix_min = get_plane_buffer(mimg->planes_array[0], kind);	/* initialize the first pixel as minimum */
		pix_min_pos = 0;	/* starting the position of the minimum */

			/* loop through the pixels */
		for (cur_pix=pix_min+1, i=1; i<pixels_count_plan; cur_pix++,i++) {
				/* if we find a pixel that is lex-smaller than the one we found before */
			if (lex_pixels_compare(pix_min, cur_pix, planes_number, d) > 0) {
				pix_min = cur_pix;	/* set this as the new minimum */
				pix_min_pos = i;	/* get the index of the new minimum */
			}
		}
		
	} else {	/* if the image is float */

		fmpixel *fpix_min;
		fmpixel *fcur_pix;
		
		fpix_min = get_plane_buffer(mimg->planes_array[0], kind);	/* initialize the first pixel as minimum */
		pix_min_pos = 0;	/* starting position of the minimum */

			/* loop through the pixels */
		for (fcur_pix=fpix_min+1, i=1; i<pixels_count_plan; i++, fcur_pix++) {
				/* if we find a pixel that is lex-smaller than the one we found before */
			if (lex_fpixels_compare(fpix_min, fcur_pix, planes_number, d) > 0) {
				fpix_min = fcur_pix;	/* set this as the new minimum */
				pix_min_pos = i;	/* get the index of the new minimum */
			}
		}
	}

	SMART_FREE(d);	/* release memory prevously used to keep the distance between planes */

	pos->x = pix_min_pos % width;	/* get the x part of the minimum position */
	pos->y = pix_min_pos / width;	/* get the y part of the minimum position */

	return 0;	/* ok */
}

/**************************************************************************************
/*	Function:		lex_maxof_mimage_dense
/*	Description:	return the position of the lexicographically
/*					maximum of a dense image
/*
/*	Parameters:
/*		<-	mimg	source dense image
/*		->	pos		position of the pixel that is maximum
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int lex_maxof_mimage_dense(mimage_ptr mimg, gr_image_displacement *pos)
{
	int i;
	int *d;
	int height, width, planes_number, kind; 
	size_t pixels_count_plan;
	int pix_max_pos;
	
		/* get information about the image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
		/* count the number of pixel in this plane*/
	pixels_count_plan = height * width;
	
		/* 
		 * retrieve in a vector d the distance between the planes 
		 * that is: 
		 *    if we pointers to plane: p0, p1, p2, pn
		 *    the distances (n-1) will be then: d[0] = p1-p0, d[1] = p2-p1, ..., d[n-1] = pn-p(n-1)
		 */
	d = get_planes_distances_for_mimage(mimg);
	if (!d)
		return -1;

	if (IS_DISCRETE(mimg)) {	/* if the image is discrete */
		mpixel *pix_max;
		mpixel *cur_pix;
		
		pix_max = get_plane_buffer(mimg->planes_array[0], kind);	/* initialize the first pixel as minimum */
		pix_max_pos = 0;	/* starting the position of the minimum */

			/* loop through the pixels */
		for (cur_pix=pix_max+1,i=1; i<pixels_count_plan; i++, cur_pix++) {
				/* if we find a pixel that is lex-smaller than the one we found before */
			if (lex_pixels_compare(pix_max, cur_pix, planes_number, d) < 0) {
				pix_max = cur_pix;	/* set this as the new minimum */
				pix_max_pos = i;	/* get the index of the new minimum */
			}
		}

	} else {	/* if the image is float */
		fmpixel *fpix_max;
		fmpixel *fcur_pix;
		
		fpix_max = get_plane_buffer(mimg->planes_array[0], kind);	/* initialize the first pixel as minimum */
		pix_max_pos = 0;	/* starting position of the minimum */

			/* loop through the pixels */
		for (fcur_pix=fpix_max+1,i=1; i<pixels_count_plan; i++, fcur_pix++) {
				/* if we find a pixel that is lex-smaller than the one we found before */
			if (lex_fpixels_compare(fpix_max, fcur_pix, planes_number, d) < 0) {
				fpix_max = fcur_pix;	/* set this as the new minimum */
				pix_max_pos = i;	/* get the index of the new minimum */
			}
		}
	}

	SMART_FREE(d);	/* release memory prevously used to keep the distance between planes */

	pos->x = pix_max_pos % width;	/* get the x part of the minimum position */
	pos->y = pix_max_pos / width;	/* get the y part of the minimum position */

	return 0;	/* ok */
}

/**************************************************************************************
/*	Function:		lex_sort_mimage_dense
/*	Description:	return a lexicographically sorted dense image from another
/*
/*	Parameters:
/*		<-	mimg	source dense image
/*
/*	Result:
/*		not NULL	the sorted dense image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr lex_sort_mimage_dense(mimage_ptr mimg)
{
	int c, i;
	mimage_ptr res_mimg;
	int pixels_number;
	int *pixel_no_array;
	int height, width, planes_number, kind;
	
		/* get info about the image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* create the new image to keep the sorted result one */
	if (!(res_mimg = create_mimage(width, height, planes_number, kind, DENSE_IMAGE)))	goto end_lex_sort;
		/* allocate a pixel number array to use during the lex-sort */
	if (!(pixel_no_array = generate_pixel_no_array(pixels_number = height * width)))	goto end_lex_sort;
		
		/* sort the values by position and values (lexicographically) */
	sort_by_pvals(pixel_no_array, mimg, pixels_number);				

		/*
		 *	now in pixel_no_array we have the sorted array of indices
		 *	of the pixels as they should be in the result image
		 */

		/* loop through the planes */
	for (c=0; c<planes_number; c++) {

		if (IS_DISCRETE(mimg)) {	/* if the image is discrete */

				/* get the pointers to the buffers for this plane */
			mpixel_ptr src_buffer = get_byteplane_from_mimage(mimg, c)->buffer;
			mpixel_ptr dst_buffer = get_byteplane_from_mimage(res_mimg, c)->buffer;

				/* for each pixel position */
			for (i=0; i<pixels_number; i++) 
				dst_buffer[i] = src_buffer[pixel_no_array[i]];	/* get the value of pixel in sorted order */

		} else {	/* if the image is float */

				/* get the pointers to the buffers for this plane */
			fmpixel_ptr fsrc_buffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
			fmpixel_ptr fdst_buffer = get_fbyteplane_from_mimage(res_mimg, c)->buffer;

				/* for each pixel position */
			for (i=0; i<pixels_number; i++) 
				fdst_buffer[i] = fsrc_buffer[pixel_no_array[i]];	/* get the value of pixel in sorted order */
		}
	}
	
end_lex_sort:

	SMART_FREE(pixel_no_array);	/* release memory for pixel number array */
	return res_mimg;	/* return the result image */
}
