/**************************************************************************************
/* Filename:	gr_fourier.c
/*		Copyright  1999 Giuseppe Di Mauro. All rights reserved.
/*
/* Description:	fft routines for the graphic library
/*
/* Disclaimer:	part of the code from "Numerical Recipes in C" book
/*
/***************************************************************************************/

#include <math.h>
#include "gr_bitplns.h"
#include "gr_errors.h"
#include "gr_fourier.h"
#include "gr_ops.h"

	/*
	 *	Not exported functions
	 */

static void four1(float data[], unsigned long nn, int isign);
static void rlft3(float ***data, float **speq, unsigned long nn1, unsigned long nn2, unsigned long nn3, int isign);
static void fourn(float data[], unsigned long nn[], int ndim, int isign);
static void nrerror(char error_text[]);
static float **matrix(long nrl, long nrh, long ncl, long nch);
static float ***f3tensor(long nrl, long nrh, long ncl, long nch, long ndl, long ndh);
static void free_matrix(float **m, long nrl, long nrh, long ncl, long nch);
static void free_f3tensor(float ***t, long nrl, long nrh, long ncl, long nch, long ndl, long ndh);

	/*
	 *	Implementation
	 */

/**************************************************************************************
/*	Function:		fft_mimage
/*	Description:	compute the fast fourier transform of an image and return two 
/*					images for both the real and the imaginary part of the transformed
/*					image
/*
/*	Parameters:
/*		<- mimg			image we want to make the fft
/*		-> res_mimg_re	result real part of the transform
/*		-> res_mimg_im	result imaginary part of the transform
/*
/*	Result:
/*		kGrOk (0)		ok
/*		else			an error condition
/*
/***************************************************************************************/

int fft_mimage(mimage_ptr mimg, mimage_ptr *res_mimg_re, mimage_ptr *res_mimg_im)
{
	int c, i, j;
	int height, width, kind, planes_number;
	int res_width;
	fmpixel_ptr fbuffer;

	mimage_ptr mimg_re, mimg_im;
	float ***data;
	float **speq;
		
		/* return an error if the image is sparse */
	if (IS_SPARSE(mimg))
		return kGrDensityMismatch;

		/* get the image specs */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	
		/* check that the dimensions are power of two */
	if (!is_pow_of_2(height) || !is_pow_of_2(width))
		return kGrNotCorrectSizes;

		/* allocate memory for data used during the transform */

	res_width = width/2 + 1;

	*res_mimg_re = mimg_re = create_mimage(res_width, height, planes_number, FLOAT_IMAGE, DENSE_IMAGE);
	*res_mimg_im = mimg_im = create_mimage(res_width, height, planes_number, FLOAT_IMAGE, DENSE_IMAGE);
	data = f3tensor(1, 1, 1, height, 1, width);
	speq = matrix(1, 1, 1, 2*height);

		/* check if allocation went fine */
	if (!mimg_re || !mimg_im || !data || !speq) {

			/* 
			 *	this destroy is safe because 
			 *	destroy is doing the check
			 */
		destroy_mimage(mimg_re);
		destroy_mimage(mimg_im);
		
		*res_mimg_re = NULL;
		*res_mimg_im = NULL;
		
		if (data) free_f3tensor(data, 1, 1, 1, height, 1, width);
		if (speq) free_matrix(speq, 1, 1, 1, 2*height);

		return kGrOutOfMemory;	/* return the out of memory error condition */
	}

		/*
		 *	loop over the planes and 
		 *	execute the transform
		 */

	for (c=0; c<planes_number; c++) {
	
			/*
			 *	store the image data in the 
			 *	buffers to be processed
			 */

		if (kind == DISCRETE_IMAGE) {	/* discrete case */
		
			mpixel_ptr buffer = get_byteplane_from_mimage(mimg, c)->buffer;
		
			for (j=1;j<=height;j++) 
				for (i=1;i<=width;i++)
					data[1][i][j] = *buffer++;

		} else { /* float case */

			fmpixel_ptr buffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
		
			for (j=1;j<=height;j++) 
				for (i=1;i<=width;i++)
					data[1][i][j] = *buffer++;
		}

			/*
			 *	execute the fft
			 */

		rlft3(data, speq, 1, height, width, 1);

			/*
			 *	store the transform data in the 
			 *	result images buffers
			 */

		fbuffer = get_fbyteplane_from_mimage(mimg_re, c)->buffer;
		
		for (j=1 ; j<=height ; j++) {

			for (i=2;i<=width;i+=2)				/* even columns */
				*fbuffer++ = data[1][i][j];

			*fbuffer++ = speq[1][j+height];
		}

		fbuffer = get_fbyteplane_from_mimage(mimg_im, c)->buffer;

		for (j=1 ; j<=height ; j++) {

			for (i=1;i<=width;i+=2)				/* odd columns */
				*fbuffer++ = data[1][i][j];
				
			*fbuffer++ = speq[1][j];
		}
	
	}	/* for c */
	

		/*
		 *	release the buffers used to store information for the fft
		 */

	free_f3tensor(data, 1, 1, 1, height, 1, width);
	free_matrix(speq, 1, planes_number, 1, 2*height);

	return kGrOk;
}

/**************************************************************************************
/*	Function:		fft_inverse_mimage
/*	Description:	compute the inverse fast fourier transform from two images
/*					representing the real and the imaginary part of a transformed
/*					image.
/*
/*	Parameters:
/*		<- res_mimg_re	real part of the input transformed image 
/*		<- res_mimg_im	imaginary part of the input transformed image 
/*		<- mimg			result inverted transform image
/*
/*	Result:
/*		kGrOk (0)		ok
/*		else			an error condition
/*
/***************************************************************************************/

int fft_inverse_mimage(mimage_ptr mimg_re, mimage_ptr mimg_im, mimage_ptr *res_mimg)
{
	int c, i, j;
	int mul_const;

	float ***data;
	float **speq;
	
	mimage_ptr mimg;
	int height_re, width_re, planes_number_re, kind_re;
	int height_im, width_im, planes_number_im, kind_im;
	int height, width, planes_number, kind;

		/*
		 *	check for compatible sizes
		 */

	get_mimage_info(mimg_re, &height_re, &width_re, &planes_number_re, &kind_re, NULL);
	get_mimage_info(mimg_im, &height_im, &width_im, &planes_number_im, &kind_im, NULL);

	if ((height_re != height_im) 
			|| (width_re != width_im) 
			|| (planes_number_re != planes_number_im)
			|| (kind_re != kind_im)
			|| (kind_re != FLOAT_IMAGE)) {

		return kGrInvalidImages;
	}

		/*
		 *	allocate buffers
		 */

	height = height_re; 
	width = (width_re - 1) * 2;
	planes_number = planes_number_re; 
	kind = kind_re;
	
	mul_const = 1 * width * height / 2;

	data = f3tensor(1, 1, 1, height, 1, width);
	speq = matrix(1, 1, 1, 2*height);
	*res_mimg = mimg = create_mimage(width, height, planes_number, FLOAT_IMAGE, DENSE_IMAGE);

		/* check if allocation went fine */

	if (!mimg || !speq || !data) {
		if (data) free_f3tensor(data, 1, 1, 1, height, 1, width);
		if (speq) free_matrix(speq, 1, 1, 1, 2*height);
		destroy_mimage(mimg);
		*res_mimg = NULL;
		return kGrOutOfMemory;	/* return the out of memory error condition */
	}


	for (c=0; c<planes_number; c++) {

			/*
			 *	re-build the data and speq
			 */

		fmpixel_ptr fbuffer_re = get_fbyteplane_from_mimage(mimg_re, c)->buffer;
		fmpixel_ptr fbuffer_im = get_fbyteplane_from_mimage(mimg_im, c)->buffer;
		fmpixel_ptr fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;

		for (j=1; j<=height; j++) {

			for (i=1; i<=width; i+=2) {
				data[1][i][j] = *fbuffer_im++;
				data[1][i+1][j] = *fbuffer_re++;
			}

			speq[1][j+height] = *fbuffer_re++;
			speq[1][j] = *fbuffer_im++;
		}

			/*
			 *	execute the inverse fft
			 */	

		rlft3(data, speq, 1, height, width, -1);

			/*
			 *	store the transform data 
			 *	in the result image
			 */
			 
		for (j=1; j<=height; j++) 
			for (i=1; i<=width; i++)
				*fbuffer++ = data[1][i][j] / (float)mul_const;
			
	}

		/*
		 *	release allocated buffers and return ok
		 */	

	free_f3tensor(data, 1, 1, 1, height, 1, width);
	free_matrix(speq, 1, 1, 1, 2*height);

	return kGrOk;
}

	/*
	 *	Routines from "Numerical Recipes in C" book
	 */

#define SWAP(a,b) tempr=(a);(a)=(b);(b)=tempr
#define NR_END 1
#define FREE_ARG char*

static void four1(float data[], unsigned long nn, int isign)
{
	unsigned long n,mmax,m,j,istep,i;
	double wtemp,wr,wpr,wpi,wi,theta;
	float tempr,tempi;

	n=nn << 1;
	j=1;
	for (i=1;i<n;i+=2) {
		if (j > i) {
			SWAP(data[j],data[i]);
			SWAP(data[j+1],data[i+1]);
		}
		m=n >> 1;
		while (m >= 2 && j > m) {
			j -= m;
			m >>= 1;
		}
		j += m;
	}
	mmax=2;
	while (n > mmax) {
		istep=mmax << 1;
		theta=isign*(6.28318530717959/mmax);
		wtemp=sin(0.5*theta);
		wpr = -2.0*wtemp*wtemp;
		wpi=sin(theta);
		wr=1.0;
		wi=0.0;
		for (m=1;m<mmax;m+=2) {
			for (i=m;i<=n;i+=istep) {
				j=i+mmax;
				tempr=wr*data[j]-wi*data[j+1];
				tempi=wr*data[j+1]+wi*data[j];
				data[j]=data[i]-tempr;
				data[j+1]=data[i+1]-tempi;
				data[i] += tempr;
				data[i+1] += tempi;
			}
			wr=(wtemp=wr)*wpr-wi*wpi+wr;
			wi=wi*wpr+wtemp*wpi+wi;
		}
		mmax=istep;
	}
}

static void rlft3(float ***data, float **speq, unsigned long nn1, unsigned long nn2,
	unsigned long nn3, int isign)
{
	void fourn(float data[], unsigned long nn[], int ndim, int isign);
	void nrerror(char error_text[]);
	unsigned long i1,i2,i3,j1,j2,j3,nn[4],ii3;
	double theta,wi,wpi,wpr,wr,wtemp;
	float c1,c2,h1r,h1i,h2r,h2i;

	if (1+&data[nn1][nn2][nn3]-&data[1][1][1] != nn1*nn2*nn3)
		nrerror("rlft3: problem with dimensions or contiguity of data array\n");
	c1=0.5;
	c2 = -0.5*isign;
	theta=isign*(6.28318530717959/nn3);
	wtemp=sin(0.5*theta);
	wpr = -2.0*wtemp*wtemp;
	wpi=sin(theta);
	nn[1]=nn1;
	nn[2]=nn2;
	nn[3]=nn3 >> 1;
	if (isign == 1) {
		fourn(&data[1][1][1]-1,nn,3,isign);
		for (i1=1;i1<=nn1;i1++)
			for (i2=1,j2=0;i2<=nn2;i2++) {
				speq[i1][++j2]=data[i1][i2][1];
				speq[i1][++j2]=data[i1][i2][2];
			}
	}
	for (i1=1;i1<=nn1;i1++) {
		j1=(i1 != 1 ? nn1-i1+2 : 1);
		wr=1.0;
		wi=0.0;
		for (ii3=1,i3=1;i3<=(nn3>>2)+1;i3++,ii3+=2) {
			for (i2=1;i2<=nn2;i2++) {
				if (i3 == 1) {
					j2=(i2 != 1 ? ((nn2-i2)<<1)+3 : 1);
					h1r=c1*(data[i1][i2][1]+speq[j1][j2]);
					h1i=c1*(data[i1][i2][2]-speq[j1][j2+1]);
					h2i=c2*(data[i1][i2][1]-speq[j1][j2]);
					h2r= -c2*(data[i1][i2][2]+speq[j1][j2+1]);
					data[i1][i2][1]=h1r+h2r;
					data[i1][i2][2]=h1i+h2i;
					speq[j1][j2]=h1r-h2r;
					speq[j1][j2+1]=h2i-h1i;
				} else {
					j2=(i2 != 1 ? nn2-i2+2 : 1);
					j3=nn3+3-(i3<<1);
					h1r=c1*(data[i1][i2][ii3]+data[j1][j2][j3]);
					h1i=c1*(data[i1][i2][ii3+1]-data[j1][j2][j3+1]);
					h2i=c2*(data[i1][i2][ii3]-data[j1][j2][j3]);
					h2r= -c2*(data[i1][i2][ii3+1]+data[j1][j2][j3+1]);
					data[i1][i2][ii3]=h1r+wr*h2r-wi*h2i;
					data[i1][i2][ii3+1]=h1i+wr*h2i+wi*h2r;
					data[j1][j2][j3]=h1r-wr*h2r+wi*h2i;
					data[j1][j2][j3+1]= -h1i+wr*h2i+wi*h2r;
				}
			}
			wr=(wtemp=wr)*wpr-wi*wpi+wr;
			wi=wi*wpr+wtemp*wpi+wi;
		}
	}
	if (isign == -1)
		fourn(&data[1][1][1]-1,nn,3,isign);
}

static void fourn(float data[], unsigned long nn[], int ndim, int isign)
{
	int idim;
	unsigned long i1,i2,i3,i2rev,i3rev,ip1,ip2,ip3,ifp1,ifp2;
	unsigned long ibit,k1,k2,n,nprev,nrem,ntot;
	float tempi,tempr;
	double theta,wi,wpi,wpr,wr,wtemp;

	for (ntot=1,idim=1;idim<=ndim;idim++)
		ntot *= nn[idim];
	nprev=1;
	for (idim=ndim;idim>=1;idim--) {
		n=nn[idim];
		nrem=ntot/(n*nprev);
		ip1=nprev << 1;
		ip2=ip1*n;
		ip3=ip2*nrem;
		i2rev=1;
		for (i2=1;i2<=ip2;i2+=ip1) {
			if (i2 < i2rev) {
				for (i1=i2;i1<=i2+ip1-2;i1+=2) {
					for (i3=i1;i3<=ip3;i3+=ip2) {
						i3rev=i2rev+i3-i2;
						SWAP(data[i3],data[i3rev]);
						SWAP(data[i3+1],data[i3rev+1]);
					}
				}
			}
			ibit=ip2 >> 1;
			while (ibit >= ip1 && i2rev > ibit) {
				i2rev -= ibit;
				ibit >>= 1;
			}
			i2rev += ibit;
		}
		ifp1=ip1;
		while (ifp1 < ip2) {
			ifp2=ifp1 << 1;
			theta=isign*6.28318530717959/(ifp2/ip1);
			wtemp=sin(0.5*theta);
			wpr = -2.0*wtemp*wtemp;
			wpi=sin(theta);
			wr=1.0;
			wi=0.0;
			for (i3=1;i3<=ifp1;i3+=ip1) {
				for (i1=i3;i1<=i3+ip1-2;i1+=2) {
					for (i2=i1;i2<=ip3;i2+=ifp2) {
						k1=i2;
						k2=k1+ifp1;
						tempr=(float)wr*data[k2]-(float)wi*data[k2+1];
						tempi=(float)wr*data[k2+1]+(float)wi*data[k2];
						data[k2]=data[k1]-tempr;
						data[k2+1]=data[k1+1]-tempi;
						data[k1] += tempr;
						data[k1+1] += tempi;
					}
				}
				wr=(wtemp=wr)*wpr-wi*wpi+wr;
				wi=wi*wpr+wtemp*wpi+wi;
			}
			ifp1=ifp2;
		}
		nprev *= n;
	}
}

static void nrerror(char error_text[])
/* Numerical Recipes standard error handler */
{
#ifdef __MWERKS__
#pragma unused(error_text)
#endif
}

static float **matrix(long nrl, long nrh, long ncl, long nch)
/* allocate a float matrix with subscript range m[nrl..nrh][ncl..nch] */
{
	long i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
	float **m;

	/* allocate pointers to rows */
	m=(float **) malloc((size_t)((nrow+NR_END)*sizeof(float*)));
	if (!m) nrerror("allocation failure 1 in matrix()");
	m += NR_END;
	m -= nrl;

	/* allocate rows and set pointers to them */
	m[nrl]=(float *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(float)));
	if (!m[nrl]) nrerror("allocation failure 2 in matrix()");
	m[nrl] += NR_END;
	m[nrl] -= ncl;

	for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

	/* return pointer to array of pointers to rows */
	return m;
}

static float ***f3tensor(long nrl, long nrh, long ncl, long nch, long ndl, long ndh)
/* allocate a float 3tensor with range t[nrl..nrh][ncl..nch][ndl..ndh] */
{
	long i,j,nrow=nrh-nrl+1,ncol=nch-ncl+1,ndep=ndh-ndl+1;
	float ***t;

	/* allocate pointers to pointers to rows */
	t=(float ***) malloc((size_t)((nrow+NR_END)*sizeof(float**)));
	if (!t) nrerror("allocation failure 1 in f3tensor()");
	t += NR_END;
	t -= nrl;

	/* allocate pointers to rows and set pointers to them */
	t[nrl]=(float **) malloc((size_t)((nrow*ncol+NR_END)*sizeof(float*)));
	if (!t[nrl]) nrerror("allocation failure 2 in f3tensor()");
	t[nrl] += NR_END;
	t[nrl] -= ncl;

	/* allocate rows and set pointers to them */
	t[nrl][ncl]=(float *) malloc((size_t)((nrow*ncol*ndep+NR_END)*sizeof(float)));
	if (!t[nrl][ncl]) nrerror("allocation failure 3 in f3tensor()");
	t[nrl][ncl] += NR_END;
	t[nrl][ncl] -= ndl;

	for(j=ncl+1;j<=nch;j++) t[nrl][j]=t[nrl][j-1]+ndep;
	for(i=nrl+1;i<=nrh;i++) {
		t[i]=t[i-1]+ncol;
		t[i][ncl]=t[i-1][ncl]+ncol*ndep;
		for(j=ncl+1;j<=nch;j++) t[i][j]=t[i][j-1]+ndep;
	}

	/* return pointer to array of pointers to rows */
	return t;
}

static void free_matrix(float **m, long nrl, long nrh, long ncl, long nch)
/* free a float matrix allocated by matrix() */
{
	free((FREE_ARG) (m[nrl]+ncl-NR_END));
	free((FREE_ARG) (m+nrl-NR_END));
}

static void free_f3tensor(float ***t, long nrl, long nrh, long ncl, long nch,
	long ndl, long ndh)
/* free a float f3tensor allocated by f3tensor() */
{
	free((FREE_ARG) (t[nrl][ncl]+ndl-NR_END));
	free((FREE_ARG) (t[nrl]+ncl-NR_END));
	free((FREE_ARG) (t+nrl-NR_END));
}
