/**************************************************************************************
/* Filename:	gr_interpolation.c
/*		Copyright  1998-99 Giuseppe Di Mauro. All rights reserved.
/*		Portions of code by Jack Schwartz.
/*
/* Description:	bicubical interpolation (4x4 lattice) for scale routines
/*
/***************************************************************************************/

#include "gr_interpolation.h"

	/*
	 *	Definitions
	 */

#define A_EQUALS_1
#ifdef A_EQUALS_1

	/*
	 *	Implementation
	 */

/**************************************************************************************
/*	Function:		interpolate_from_4
/*	Description:	bicubic spline interpolation for 4 values in a row
/*	Note:			see comment below
/*
/*	Parameters:
/*		<-	y0		see comment below
/*		<-	y1		see comment below
/*		<-	y2		see comment below
/*		<-	y3		see comment below
/*		<-	a		see comment below
/*		<-	x		see comment below
/*
/*	Result:
/*		the interpolated value
/*
/***************************************************************************************/

float interpolate_from_4(float y0, float y1, float y2, float y3, float a, float x) {
#ifdef __MWERKS__
#pragma unused(a)
#endif

/*
	 bicubic spline interpolation for x in range between 0 and a from function values at points
	 -a,0,a,2a. The basic interpolation formula is y0 * p0(x) + y1 * p1(x) + y2 * p2(x) + y3 * p3(x)
	 where the polynomials are
	 p0(x) = x(x - a)(x - 2a)/ (-a)(-2a)(-3a) is 1 at x = -a and zero elsewhere
	 p1(x) = (x + a)(x - a)(x - 2a)/ (-a)(-2a)(-3a) is 1 at x = 0 and zero elsewhere
	 p2(x) = (x + a)x(x - 2a)/ (-a)(-2a)(-3a) is 1 at x = a and zero elsewhere
	 p3(x) = (x + a)x(x - a)/ (-a)(-2a)(-3a) is 1 at x = 2a and zero elsewhere
 */
 
	float xma_xm2a,xpa_x,xpa,xma,xm2a;

	xma_xm2a = (xma = x - (float)1.) * (xm2a = x - (float)2.);
	xpa_x = (xpa = x + (float)1.) * x;

	return  (float)0.5 * ( ( -y0*x*xma_xm2a + y3*xma*xpa_x ) 
				* (float)0.3333 + y1*xpa*xma_xm2a + -y2*xm2a*xpa_x );
}

/**************************************************************************************
/*	Function:		interpolate_from_16
/*	Description:	bicubic spline interpolation for 16 values in a matrix
/*	Note:			see comment below
/*
/*	Parameters:
/*		<-	y00		see comment below
/*		<-	y01		see comment below
/*		<-	y02		see comment below
/*		<-	y03		see comment below
/*		<-	y10		see comment below
/*		<-	y11		see comment below
/*		<-	y12		see comment below
/*		<-	y13		see comment below
/*		...
/*		...
/*		...
/*		<-	y30		see comment below
/*		<-	y31		see comment below
/*		<-	y32		see comment below
/*		<-	y33		see comment below
/*		<-	a		see comment below
/*		<-	x		see comment below
/*		<-	y		see comment below
/*
/*	Result:
/*		the interpolated value
/*
/***************************************************************************************/

float interpolate_from_16(float y00,float y01,float y02,float y03,float y10,float y11,float y12,float y13,float y20,float y21,float y22,float y23,float y30,float y31,float y32,float y33,float a,float x,float y) {
/*	
	 bicubic spline interpolation for x and y in range between 0 and a from function values at abcissae and
	 ordinates -a,0,a,2a. 
 */
	float f0,f1,f2,f3;

	f0 = interpolate_from_4(y00, y01, y02, y03, a, x);
	f1 = interpolate_from_4(y10, y11, y12, y13, a, x);
	f2 = interpolate_from_4(y20, y21, y22, y23, a, x);
	f3 = interpolate_from_4(y30, y31, y32, y33, a, x);

	return interpolate_from_4(f0, f1, f2, f3, a, y);
}

#else /* !A_EQUALS_1 */

/**************************************************************************************
/*	Function:		interpolate_from_4
/*	Description:	bicubic spline interpolation for 4 values in a row
/*	Note:			see comment below
/*
/*	Parameters:
/*		<-	y0		see comment below
/*		<-	y1		see comment below
/*		<-	y2		see comment below
/*		<-	y3		see comment below
/*		<-	a		see comment below
/*		<-	x		see comment below
/*
/*	Result:
/*		the interpolated value
/*
/***************************************************************************************/

float interpolate_from_4(float y0,float y1,float y2,float y3,float a,float x) {
/*
	 bicubic spline interpolation for x in range between 0 and a from function values at points
	 -a,0,a,2a. The basic interpolation formula is y0 * p0(x) + y1 * p1(x) + y2 * p2(x) + y3 * p3(x)
	 where the polynomials are
	 p0(x) = x(x - a)(x - 2a)/ (-a)(-2a)(-3a) is 1 at x = -a and zero elsewhere
	 p1(x) = (x + a)(x - a)(x - 2a)/ (-a)(-2a)(-3a) is 1 at x = 0 and zero elsewhere
	 p2(x) = (x + a)x(x - 2a)/ (-a)(-2a)(-3a) is 1 at x = a and zero elsewhere
	 p3(x) = (x + a)x(x - a)/ (-a)(-2a)(-3a) is 1 at x = 2a and zero elsewhere
 */
	float acro2,one_third,xma_xm2a,xpa_x,xpa,xma,xm2a;
	acro2 = 0.5 / (a*a*a);		/* 1/(2a ** 3) */
	one_third = 0.33333333333;
	xma_xm2a = (xma = x - a) * (xm2a = x - 2 * a);
	xpa_x = (xpa = x + a) * x;
	return  acro2 * (- y0 * x * xma_xm2a * one_third +
				y1 * xpa * xma_xm2a + 
				- y2 * xm2a * xpa_x + 
				y3 * xma * xpa_x* one_third);
}

/**************************************************************************************
/*	Function:		interpolate_from_16
/*	Description:	bicubic spline interpolation for 16 values in a matrix
/*	Note:			see comment below
/*
/*	Parameters:
/*		<-	y00		see comment below
/*		<-	y01		see comment below
/*		<-	y02		see comment below
/*		<-	y03		see comment below
/*		<-	y10		see comment below
/*		<-	y11		see comment below
/*		<-	y12		see comment below
/*		<-	y13		see comment below
/*		...
/*		...
/*		...
/*		<-	y30		see comment below
/*		<-	y31		see comment below
/*		<-	y32		see comment below
/*		<-	y33		see comment below
/*		<-	a		see comment below
/*		<-	x		see comment below
/*		<-	y		see comment below
/*
/*	Result:
/*		the interpolated value
/*
/***************************************************************************************/

float interpolate_from_16(float y00,float y01,float y02,float y03,float y10,float y11,float y12,float y13,float y20,float y21,float y22,float y23,float y30,float y31,float y32,float y33,float a,float x,float y) {
/*	
	 bicubic spline interpolation for x and y in range between 0 and a from function values at abcissae and
	 ordinates -a,0,a,2a. 
 */
	float f0,f1,f2,f3;			/* interpolated values for the 4 values of y */

	f0 = interpolate_from_4(y00,y01,y02,y03,a,x);
	f1 = interpolate_from_4(y10,y11,y12,y13,a,x);
	f2 = interpolate_from_4(y20,y21,y22,y23,a,x);
	f3 = interpolate_from_4(y30,y31,y32,y33,a,x);

	return interpolate_from_4(f0,f1,f2,f3, a, y);
}

#endif	/* !A_EQUALS_1 */