/**************************************************************************************
/* Filename:	gr_pkg_stubs.c
/*		Copyright  1998-99 Giuseppe Di Mauro. All rights reserved.
/*
/* Description:	stubs for the setl-language interface
/*
/***************************************************************************************/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "macros.h"

#include "jpeglib.h"
#ifndef TIFF_DISABLED
#include "tiffio.h"
#endif

#include "gr_bitplns.h"
#include "gr_sparse_bitplns.h"
#include "gr_boolean_sparse_ops.h"
#include "gr_fourier.h"
#include "gr_wavelet.h"
#include "gr_pkg_stubs.h"
	/*
	 *	Definitions
	 */

#ifdef PLUGIN
#define printf plugin_printf
#define fprintf plugin_fprintf
#endif

#ifdef macintosh 
#include <MacMemory.h>
#include "gr_mac_debug.h"
#if __MWERKS__ && __DEBUG__
void DebugStart();
#endif
#endif

/* #define TRACE_STUBS_ON /* enable to have extra stub debug info */

#ifdef TRACE_STUBS_ON
#ifdef macintosh
#define TRACE_STUBS(a)	printf(a); printf(" Mem: %ld\n", FreeMem());
#else
#define TRACE_STUBS(a)	printf(a); printf("\n");
#endif
#else /* TRACE_STUBS_ON */
#define TRACE_STUBS(a)	;
#endif

#define MIN_REF		3	/* 1 , 3 */
#define REF_BASE	2	/* 0 , 2 */

/*
 *	This module only prototypes
 */

static float gradient_callback_op(SETL_SYSTEM_PROTO void *user_ptr2, int x, int y, int c);
static float unimath_callback_op(SETL_SYSTEM_PROTO void *user_ptr2, float f, int c);

static void check_arg(SETL_SYSTEM_PROTO specifier *argv, int param, int type, char *typestr, char *routine);
static void check_type(SETL_SYSTEM_PROTO specifier *argv, int param, int type, int expected_type, char *argstr, char *errstr,char *routine);
static void check_gr_arg(SETL_SYSTEM_PROTO specifier *argv, int param, char *routine);
static void setup_rw_gr_arg(SETL_SYSTEM_PROTO mimage_ptr *mimg);

/*
 *
 */

void check_arg(SETL_SYSTEM_PROTO specifier *argv, int param, int type, char *typestr, char *routine)
{

   if (argv[param].sp_form != type)
	  abend(SETL_SYSTEM msg_bad_arg, typestr, param+1, 
	  		routine, abend_opnd_str(SETL_SYSTEM argv+param));

}

void check_type(SETL_SYSTEM_PROTO specifier *argv, int param, int type, int expected_type, char *argstr, char *errstr,char *routine)
{
	if (type!=expected_type)
		abend(SETL_SYSTEM errstr, argstr, param + 1,
			routine, abend_opnd_str(SETL_SYSTEM argv));

}

void check_gr_arg(SETL_SYSTEM_PROTO specifier *argv, int param, char *routine)
{
	if ((argv[param].sp_form != ft_opaque) ||
		(((argv[param].sp_val.sp_opaque_ptr->type) & 65535 ) != gr_type))
		abend(SETL_SYSTEM msg_bad_arg, "image", param + 1,
			routine, abend_opnd_str(SETL_SYSTEM argv+param));
}

void setup_rw_gr_arg(SETL_SYSTEM_PROTO mimage_ptr *mimg)
{
	if ((*mimg)->use_count > MIN_REF) {
		*mimg = clone_mimage(*mimg);
		if (!*mimg)
			abend(SETL_SYSTEM msg_malloc_error);
		(*mimg)->use_count = REF_BASE;
	}
}

/*
 *
 */

void create_and_return_integers_tuple_from_array(
	SETL_SYSTEM_PROTO
	int *array, 
	size_t array_size, 
	specifier *target)			/* return value */
{
	int i;
	size_t elements;

	tuple_h_ptr_type tuple_root, tuple_work_hdr, new_tuple_hdr;
	tuple_c_ptr_type tuple_cell;

	int tuple_length;
	int tuple_index, tuple_height;
	int32 expansion_trigger;

	get_tuple_header(tuple_root);
	tuple_root->t_use_count = 1;
	tuple_root->t_hash_code = 0;
	tuple_root->t_ntype.t_root.t_length = 0;
	tuple_root->t_ntype.t_root.t_height = 0;
	for (i = 0; i < TUP_HEADER_SIZE; tuple_root->t_child[i++].t_cell = NULL);
	tuple_length = 0;
	expansion_trigger = TUP_HEADER_SIZE;

	for (elements=0;elements<array_size;elements++) {
		/* expand the tuple tree if necessary */

		if (tuple_length >= expansion_trigger) {

			tuple_work_hdr = tuple_root;

			get_tuple_header(tuple_root);

			tuple_root->t_use_count = 1;
			tuple_root->t_hash_code =
				tuple_work_hdr->t_hash_code;
			tuple_root->t_ntype.t_root.t_length =
				tuple_work_hdr->t_ntype.t_root.t_length;
			tuple_root->t_ntype.t_root.t_height =
				tuple_work_hdr->t_ntype.t_root.t_height + 1;

			for (i = 1;
				i < TUP_HEADER_SIZE;
				tuple_root->t_child[i++].t_header = NULL);

			tuple_root->t_child[0].t_header = tuple_work_hdr;

			tuple_work_hdr->t_ntype.t_intern.t_parent = tuple_root;
			tuple_work_hdr->t_ntype.t_intern.t_child_index = 0;

			expansion_trigger *= TUP_HEADER_SIZE;

		}

		tuple_root->t_ntype.t_root.t_length++;

		/* descend the tree to a leaf */

		tuple_work_hdr = tuple_root;
		for (tuple_height = tuple_work_hdr->t_ntype.t_root.t_height;
			tuple_height;
			tuple_height--) {

			/* extract the element's index at this level */

			tuple_index = (tuple_length >>
				(tuple_height * TUP_SHIFT_DIST)) &
				TUP_SHIFT_MASK;

			/* if we're missing a header record, allocate one */

			if (tuple_work_hdr->t_child[tuple_index].t_header == NULL) {

				get_tuple_header(new_tuple_hdr);
				new_tuple_hdr->t_ntype.t_intern.t_parent = tuple_work_hdr;
				new_tuple_hdr->t_ntype.t_intern.t_child_index = tuple_index;
				for (i = 0;
					i < TUP_HEADER_SIZE;
					new_tuple_hdr->t_child[i++].t_cell = NULL);
				tuple_work_hdr->t_child[tuple_index].t_header =
					new_tuple_hdr;
				tuple_work_hdr = new_tuple_hdr;

			}
			else {

				tuple_work_hdr =
					tuple_work_hdr->t_child[tuple_index].t_header;

			}
		}

		/*
	   *  At this point, tuple_work_hdr points to the lowest level header
	   *  record.  We insert the new element.
	   */

		tuple_index = tuple_length & TUP_SHIFT_MASK;
		get_tuple_cell(tuple_cell);

		tuple_cell->t_spec.sp_form = ft_short;
		tuple_cell->t_spec.sp_val.sp_short_value = array[elements];

		spec_hash_code(tuple_cell->t_hash_code,&(tuple_cell->t_spec));
		tuple_root->t_hash_code ^= tuple_cell->t_hash_code;
		tuple_work_hdr->t_child[tuple_index].t_cell = tuple_cell;

		/* increment the tuple size */

		tuple_length++;

	}
	tuple_root->t_ntype.t_root.t_length = tuple_length;
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = tuple_root;

	return;
}


void create_and_return_shorts_tuple_from_array(
	SETL_SYSTEM_PROTO
	unsigned char *array, 
	size_t array_size, 
	specifier *target)			/* return value			 */
{
	int i;
	size_t elements;

	tuple_h_ptr_type tuple_root, tuple_work_hdr, new_tuple_hdr;
	tuple_c_ptr_type tuple_cell;

	int tuple_length;
	int tuple_index, tuple_height;
	int32 expansion_trigger;

	get_tuple_header(tuple_root);
	tuple_root->t_use_count = 1;
	tuple_root->t_hash_code = 0;
	tuple_root->t_ntype.t_root.t_length = 0;
	tuple_root->t_ntype.t_root.t_height = 0;
	for (i = 0;
		i < TUP_HEADER_SIZE;
		tuple_root->t_child[i++].t_cell = NULL);
	tuple_length = 0;
	expansion_trigger = TUP_HEADER_SIZE;


	for (elements=0;elements<array_size;elements++) {
		/* expand the tuple tree if necessary */

		if (tuple_length >= expansion_trigger) {

			tuple_work_hdr = tuple_root;

			get_tuple_header(tuple_root);

			tuple_root->t_use_count = 1;
			tuple_root->t_hash_code =
				tuple_work_hdr->t_hash_code;
			tuple_root->t_ntype.t_root.t_length =
				tuple_work_hdr->t_ntype.t_root.t_length;
			tuple_root->t_ntype.t_root.t_height =
				tuple_work_hdr->t_ntype.t_root.t_height + 1;

			for (i = 1;
				i < TUP_HEADER_SIZE;
				tuple_root->t_child[i++].t_header = NULL);

			tuple_root->t_child[0].t_header = tuple_work_hdr;

			tuple_work_hdr->t_ntype.t_intern.t_parent = tuple_root;
			tuple_work_hdr->t_ntype.t_intern.t_child_index = 0;

			expansion_trigger *= TUP_HEADER_SIZE;

		}

		tuple_root->t_ntype.t_root.t_length++;

		/* descend the tree to a leaf */

		tuple_work_hdr = tuple_root;
		for (tuple_height = tuple_work_hdr->t_ntype.t_root.t_height;
			tuple_height;
			tuple_height--) {

			/* extract the element's index at this level */

			tuple_index = (tuple_length >>
				(tuple_height * TUP_SHIFT_DIST)) &
				TUP_SHIFT_MASK;

			/* if we're missing a header record, allocate one */

			if (tuple_work_hdr->t_child[tuple_index].t_header == NULL) {

				get_tuple_header(new_tuple_hdr);
				new_tuple_hdr->t_ntype.t_intern.t_parent = tuple_work_hdr;
				new_tuple_hdr->t_ntype.t_intern.t_child_index = tuple_index;
				for (i = 0;
					i < TUP_HEADER_SIZE;
					new_tuple_hdr->t_child[i++].t_cell = NULL);
				tuple_work_hdr->t_child[tuple_index].t_header =
					new_tuple_hdr;
				tuple_work_hdr = new_tuple_hdr;

			}
			else {

				tuple_work_hdr =
					tuple_work_hdr->t_child[tuple_index].t_header;

			}
		}

		/*
	   *  At this point, tuple_work_hdr points to the lowest level header
	   *  record.  We insert the new element.
	   */

		tuple_index = tuple_length & TUP_SHIFT_MASK;
		get_tuple_cell(tuple_cell);

		tuple_cell->t_spec.sp_form = ft_short;
		tuple_cell->t_spec.sp_val.sp_short_value = array[elements];

		spec_hash_code(tuple_cell->t_hash_code,&(tuple_cell->t_spec));
		tuple_root->t_hash_code ^= tuple_cell->t_hash_code;
		tuple_work_hdr->t_child[tuple_index].t_cell = tuple_cell;

		/* increment the tuple size */

		tuple_length++;

	}
	tuple_root->t_ntype.t_root.t_length = tuple_length;
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = tuple_root;

	return;
}

void create_and_return_longs_tuple_from_array(
	SETL_SYSTEM_PROTO
	unsigned long *array, 
	size_t array_size, 
	specifier *target)			/* return value			 */
{
	int i;
	size_t elements;

	tuple_h_ptr_type tuple_root, tuple_work_hdr, new_tuple_hdr;
	tuple_c_ptr_type tuple_cell;

	int tuple_length;
	int tuple_index, tuple_height;
	int32 expansion_trigger;

	get_tuple_header(tuple_root);
	tuple_root->t_use_count = 1;
	tuple_root->t_hash_code = 0;
	tuple_root->t_ntype.t_root.t_length = 0;
	tuple_root->t_ntype.t_root.t_height = 0;
	for (i = 0;
		i < TUP_HEADER_SIZE;
		tuple_root->t_child[i++].t_cell = NULL);
	tuple_length = 0;
	expansion_trigger = TUP_HEADER_SIZE;


	for (elements=0;elements<array_size;elements++) {
		/* expand the tuple tree if necessary */

		if (tuple_length >= expansion_trigger) {

			tuple_work_hdr = tuple_root;

			get_tuple_header(tuple_root);

			tuple_root->t_use_count = 1;
			tuple_root->t_hash_code =
				tuple_work_hdr->t_hash_code;
			tuple_root->t_ntype.t_root.t_length =
				tuple_work_hdr->t_ntype.t_root.t_length;
			tuple_root->t_ntype.t_root.t_height =
				tuple_work_hdr->t_ntype.t_root.t_height + 1;

			for (i = 1;
				i < TUP_HEADER_SIZE;
				tuple_root->t_child[i++].t_header = NULL);

			tuple_root->t_child[0].t_header = tuple_work_hdr;

			tuple_work_hdr->t_ntype.t_intern.t_parent = tuple_root;
			tuple_work_hdr->t_ntype.t_intern.t_child_index = 0;

			expansion_trigger *= TUP_HEADER_SIZE;

		}

		tuple_root->t_ntype.t_root.t_length++;

		/* descend the tree to a leaf */

		tuple_work_hdr = tuple_root;
		for (tuple_height = tuple_work_hdr->t_ntype.t_root.t_height;
			tuple_height;
			tuple_height--) {

			/* extract the element's index at this level */

			tuple_index = (tuple_length >>
				(tuple_height * TUP_SHIFT_DIST)) &
				TUP_SHIFT_MASK;

			/* if we're missing a header record, allocate one */

			if (tuple_work_hdr->t_child[tuple_index].t_header == NULL) {

				get_tuple_header(new_tuple_hdr);
				new_tuple_hdr->t_ntype.t_intern.t_parent = tuple_work_hdr;
				new_tuple_hdr->t_ntype.t_intern.t_child_index = tuple_index;
				for (i = 0;
					i < TUP_HEADER_SIZE;
					new_tuple_hdr->t_child[i++].t_cell = NULL);

				tuple_work_hdr->t_child[tuple_index].t_header = new_tuple_hdr;
				tuple_work_hdr = new_tuple_hdr;

			}
			else {

				tuple_work_hdr =
					tuple_work_hdr->t_child[tuple_index].t_header;

			}
		}

		/*
	*  At this point, tuple_work_hdr points to the lowest level header
	*  record.  We insert the new element.
	*/

		tuple_index = tuple_length & TUP_SHIFT_MASK;
		get_tuple_cell(tuple_cell);

		tuple_cell->t_spec.sp_form = ft_short;
		tuple_cell->t_spec.sp_val.sp_short_value = (long)array[elements];

		spec_hash_code(tuple_cell->t_hash_code,&(tuple_cell->t_spec));
		tuple_root->t_hash_code ^= tuple_cell->t_hash_code;
		tuple_work_hdr->t_child[tuple_index].t_cell = tuple_cell;

		/* increment the tuple size */

		tuple_length++;

	}
	tuple_root->t_ntype.t_root.t_length = tuple_length;
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = tuple_root;

	return;
}

void create_and_return_floats_tuple_from_array(
	SETL_SYSTEM_PROTO
	float *array, 
	size_t array_size, 
	specifier *target)			/* return value			 */
{
	int i;
	size_t elements;

	i_real_ptr_type real_ptr;

	tuple_h_ptr_type tuple_root, tuple_work_hdr, new_tuple_hdr;
	tuple_c_ptr_type tuple_cell;

	int tuple_length;
	int tuple_index, tuple_height;
	int32 expansion_trigger;

	get_tuple_header(tuple_root);
	tuple_root->t_use_count = 1;
	tuple_root->t_hash_code = 0;
	tuple_root->t_ntype.t_root.t_length = 0;
	tuple_root->t_ntype.t_root.t_height = 0;
	for (i = 0;
		i < TUP_HEADER_SIZE;
		tuple_root->t_child[i++].t_cell = NULL);
	tuple_length = 0;
	expansion_trigger = TUP_HEADER_SIZE;


	for (elements=0;elements<array_size;elements++) {
		/* expand the tuple tree if necessary */

		if (tuple_length >= expansion_trigger) {

			tuple_work_hdr = tuple_root;

			get_tuple_header(tuple_root);

			tuple_root->t_use_count = 1;
			tuple_root->t_hash_code =
				tuple_work_hdr->t_hash_code;
			tuple_root->t_ntype.t_root.t_length =
				tuple_work_hdr->t_ntype.t_root.t_length;
			tuple_root->t_ntype.t_root.t_height =
				tuple_work_hdr->t_ntype.t_root.t_height + 1;

			for (i = 1;
				i < TUP_HEADER_SIZE;
				tuple_root->t_child[i++].t_header = NULL);

			tuple_root->t_child[0].t_header = tuple_work_hdr;

			tuple_work_hdr->t_ntype.t_intern.t_parent = tuple_root;
			tuple_work_hdr->t_ntype.t_intern.t_child_index = 0;

			expansion_trigger *= TUP_HEADER_SIZE;

		}

		tuple_root->t_ntype.t_root.t_length++;

		/* descend the tree to a leaf */

		tuple_work_hdr = tuple_root;
		for (tuple_height = tuple_work_hdr->t_ntype.t_root.t_height;
			tuple_height;
			tuple_height--) {

			/* extract the element's index at this level */

			tuple_index = (tuple_length >>
				(tuple_height * TUP_SHIFT_DIST)) &
				TUP_SHIFT_MASK;

			/* if we're missing a header record, allocate one */

			if (tuple_work_hdr->t_child[tuple_index].t_header == NULL) {

				get_tuple_header(new_tuple_hdr);
				new_tuple_hdr->t_ntype.t_intern.t_parent = tuple_work_hdr;
				new_tuple_hdr->t_ntype.t_intern.t_child_index = tuple_index;
				for (i = 0;
					i < TUP_HEADER_SIZE;
					new_tuple_hdr->t_child[i++].t_cell = NULL);
				tuple_work_hdr->t_child[tuple_index].t_header =
					new_tuple_hdr;
				tuple_work_hdr = new_tuple_hdr;

			}
			else {

				tuple_work_hdr =
					tuple_work_hdr->t_child[tuple_index].t_header;

			}
		}

		/*
	   *  At this point, tuple_work_hdr points to the lowest level header
	   *  record.  We insert the new element.
	   */

		tuple_index = tuple_length & TUP_SHIFT_MASK;
		get_tuple_cell(tuple_cell);

		i_get_real(real_ptr);
		tuple_cell->t_spec.sp_form = ft_real;
		tuple_cell->t_spec.sp_val.sp_real_ptr = real_ptr;
		real_ptr->r_use_count = 1;
		real_ptr->r_value = array[elements];

		spec_hash_code(tuple_cell->t_hash_code,&(tuple_cell->t_spec));
		tuple_root->t_hash_code ^= tuple_cell->t_hash_code;
		tuple_work_hdr->t_child[tuple_index].t_cell = tuple_cell;

		/* increment the tuple size */

		tuple_length++;

	}
	tuple_root->t_ntype.t_root.t_length = tuple_length;
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = tuple_root;

	return;
}

void create_and_return_doubles_tuple_from_array(
	SETL_SYSTEM_PROTO
	double *array, 
	size_t array_size, 
	specifier *target)			/* return value			 */
{
	int i;
	size_t elements;

	i_real_ptr_type real_ptr;

	tuple_h_ptr_type tuple_root, tuple_work_hdr, new_tuple_hdr;
	tuple_c_ptr_type tuple_cell;

	int tuple_length;
	int tuple_index, tuple_height;
	int32 expansion_trigger;

	get_tuple_header(tuple_root);
	tuple_root->t_use_count = 1;
	tuple_root->t_hash_code = 0;
	tuple_root->t_ntype.t_root.t_length = 0;
	tuple_root->t_ntype.t_root.t_height = 0;
	for (i = 0;
		i < TUP_HEADER_SIZE;
		tuple_root->t_child[i++].t_cell = NULL);
	tuple_length = 0;
	expansion_trigger = TUP_HEADER_SIZE;


	for (elements=0;elements<array_size;elements++) {
		/* expand the tuple tree if necessary */

		if (tuple_length >= expansion_trigger) {

			tuple_work_hdr = tuple_root;

			get_tuple_header(tuple_root);

			tuple_root->t_use_count = 1;
			tuple_root->t_hash_code =
				tuple_work_hdr->t_hash_code;
			tuple_root->t_ntype.t_root.t_length =
				tuple_work_hdr->t_ntype.t_root.t_length;
			tuple_root->t_ntype.t_root.t_height =
				tuple_work_hdr->t_ntype.t_root.t_height + 1;

			for (i = 1;
				i < TUP_HEADER_SIZE;
				tuple_root->t_child[i++].t_header = NULL);

			tuple_root->t_child[0].t_header = tuple_work_hdr;

			tuple_work_hdr->t_ntype.t_intern.t_parent = tuple_root;
			tuple_work_hdr->t_ntype.t_intern.t_child_index = 0;

			expansion_trigger *= TUP_HEADER_SIZE;

		}

		tuple_root->t_ntype.t_root.t_length++;

		/* descend the tree to a leaf */

		tuple_work_hdr = tuple_root;
		for (tuple_height = tuple_work_hdr->t_ntype.t_root.t_height;
			tuple_height;
			tuple_height--) {

			/* extract the element's index at this level */

			tuple_index = (tuple_length >>
				(tuple_height * TUP_SHIFT_DIST)) &
				TUP_SHIFT_MASK;

			/* if we're missing a header record, allocate one */

			if (tuple_work_hdr->t_child[tuple_index].t_header == NULL) {

				get_tuple_header(new_tuple_hdr);
				new_tuple_hdr->t_ntype.t_intern.t_parent = tuple_work_hdr;
				new_tuple_hdr->t_ntype.t_intern.t_child_index = tuple_index;
				for (i = 0;
					i < TUP_HEADER_SIZE;
					new_tuple_hdr->t_child[i++].t_cell = NULL);
				tuple_work_hdr->t_child[tuple_index].t_header =
					new_tuple_hdr;
				tuple_work_hdr = new_tuple_hdr;

			}
			else {

				tuple_work_hdr =
					tuple_work_hdr->t_child[tuple_index].t_header;

			}
		}

		/*
	   *  At this point, tuple_work_hdr points to the lowest level header
	   *  record.  We insert the new element.
	   */

		tuple_index = tuple_length & TUP_SHIFT_MASK;
		get_tuple_cell(tuple_cell);

		i_get_real(real_ptr);
		tuple_cell->t_spec.sp_form = ft_real;
		tuple_cell->t_spec.sp_val.sp_real_ptr = real_ptr;
		real_ptr->r_use_count = 1;
		real_ptr->r_value = array[elements];

		spec_hash_code(tuple_cell->t_hash_code,&(tuple_cell->t_spec));
		tuple_root->t_hash_code ^= tuple_cell->t_hash_code;
		tuple_work_hdr->t_child[tuple_index].t_cell = tuple_cell;

		/* increment the tuple size */

		tuple_length++;

	}
	tuple_root->t_ntype.t_root.t_length = tuple_length;
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = tuple_root;

	return;
}

void create_and_return_mimages_tuple_from_array(
	SETL_SYSTEM_PROTO
	mimage_ptr *array, 
	size_t array_size, 
	specifier *target)			/* return value			 */
{
	int i;
	size_t elements;

	tuple_h_ptr_type tuple_root, tuple_work_hdr, new_tuple_hdr;
	tuple_c_ptr_type tuple_cell;

	int tuple_length;
	int tuple_index, tuple_height;
	int32 expansion_trigger;

	get_tuple_header(tuple_root);
	tuple_root->t_use_count = 1;
	tuple_root->t_hash_code = 0;
	tuple_root->t_ntype.t_root.t_length = 0;
	tuple_root->t_ntype.t_root.t_height = 0;
	for (i = 0;
		i < TUP_HEADER_SIZE;
		tuple_root->t_child[i++].t_cell = NULL);
	tuple_length = 0;
	expansion_trigger = TUP_HEADER_SIZE;


	for (elements=0;elements<array_size;elements++) {
		/* expand the tuple tree if necessary */

		if (tuple_length >= expansion_trigger) {

			tuple_work_hdr = tuple_root;

			get_tuple_header(tuple_root);

			tuple_root->t_use_count = 1;
			tuple_root->t_hash_code =
				tuple_work_hdr->t_hash_code;
			tuple_root->t_ntype.t_root.t_length =
				tuple_work_hdr->t_ntype.t_root.t_length;
			tuple_root->t_ntype.t_root.t_height =
				tuple_work_hdr->t_ntype.t_root.t_height + 1;

			for (i = 1;
				i < TUP_HEADER_SIZE;
				tuple_root->t_child[i++].t_header = NULL);

			tuple_root->t_child[0].t_header = tuple_work_hdr;

			tuple_work_hdr->t_ntype.t_intern.t_parent = tuple_root;
			tuple_work_hdr->t_ntype.t_intern.t_child_index = 0;

			expansion_trigger *= TUP_HEADER_SIZE;

		}

		tuple_root->t_ntype.t_root.t_length++;

		/* descend the tree to a leaf */

		tuple_work_hdr = tuple_root;
		for (tuple_height = tuple_work_hdr->t_ntype.t_root.t_height;
			tuple_height;
			tuple_height--) {

			/* extract the element's index at this level */

			tuple_index = (tuple_length >>
				(tuple_height * TUP_SHIFT_DIST)) &
				TUP_SHIFT_MASK;

			/* if we're missing a header record, allocate one */

			if (tuple_work_hdr->t_child[tuple_index].t_header == NULL) {

				get_tuple_header(new_tuple_hdr);
				new_tuple_hdr->t_ntype.t_intern.t_parent = tuple_work_hdr;
				new_tuple_hdr->t_ntype.t_intern.t_child_index = tuple_index;
				for (i = 0;
					i < TUP_HEADER_SIZE;
					new_tuple_hdr->t_child[i++].t_cell = NULL);
				tuple_work_hdr->t_child[tuple_index].t_header =
					new_tuple_hdr;
				tuple_work_hdr = new_tuple_hdr;

			}
			else {

				tuple_work_hdr =
					tuple_work_hdr->t_child[tuple_index].t_header;

			}
		}

		/*
	   *  At this point, tuple_work_hdr points to the lowest level header
	   *  record.  We insert the new element.
	   */

		tuple_index = tuple_length & TUP_SHIFT_MASK;
		get_tuple_cell(tuple_cell);

		tuple_cell->t_spec.sp_form = ft_opaque;
		tuple_cell->t_spec.sp_val.sp_opaque_ptr = (opaque_item_ptr_type)array[elements];

		spec_hash_code(tuple_cell->t_hash_code,&(tuple_cell->t_spec));
		tuple_root->t_hash_code ^= tuple_cell->t_hash_code;
		tuple_work_hdr->t_child[tuple_index].t_cell = tuple_cell;

		/* increment the tuple size */

		tuple_length++;

	}
	tuple_root->t_ntype.t_root.t_length = tuple_length;
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = tuple_root;

	return;
}

/*
 *
 */

string_h_ptr_type setl2_string(SETL_SYSTEM_PROTO char *s, int slen)
{
	string_h_ptr_type string_hdr;	   /* string root			  */
	string_c_ptr_type string_cell;	  /* string cell pointer		 */
	char *string_char_ptr, *string_char_end;

	get_string_header(string_hdr);
	string_hdr->s_use_count = 1;
	string_hdr->s_hash_code = -1;
	string_hdr->s_length = 0;
	string_hdr->s_head = string_hdr->s_tail = NULL;
	string_char_ptr = string_char_end = NULL;

	/* copy the source string */

	while (slen-->0) {

		if (string_char_ptr == string_char_end) {

			get_string_cell(string_cell);
			if (string_hdr->s_tail != NULL)
				(string_hdr->s_tail)->s_next = string_cell;
			string_cell->s_prev = string_hdr->s_tail;
			string_cell->s_next = NULL;
			string_hdr->s_tail = string_cell;
			if (string_hdr->s_head == NULL)
				string_hdr->s_head = string_cell;
			string_char_ptr = string_cell->s_cell_value;
			string_char_end = string_char_ptr + STR_CELL_WIDTH;

		}

		*string_char_ptr++ = *s++;
		string_hdr->s_length++;

	}
	return string_hdr;
}

void setl2_string_to_cstring(string_h_ptr_type string_hdr,
unsigned char *p,
int buffer_len)
{
	string_c_ptr_type string_cell;	  /* string cell pointer		 */
	unsigned char *t;
	char *s;

#ifdef __DEBUG__
	assert(string_hdr->s_length<=buffer_len);
#endif

	t = p;
	for (string_cell = string_hdr->s_head;
		string_cell != NULL;
		string_cell = string_cell->s_next) {

		for (s = string_cell->s_cell_value;
			t < p + string_hdr->s_length &&
			s < (char *)(string_cell->s_cell_value + STR_CELL_WIDTH);
			*t++ = *s++);

	}
	*(p+string_hdr->s_length) = 0;
}

/*
 *
 */

callback_result d_callback_caller(
	SETL_SYSTEM_PROTO
	specifier *callback,
	char *fmt,
	...)					/* any other arguments		 */

{
	va_list argp;					 	/* current argument pointer */
	i_real_ptr_type real_ptr;			/* real pointer	*/

	static char *return_string = NULL;	 /* static return buffer */
	/* used to build tuple		 */
	specifier spare;			  		/* spare specifier */
	specifier save_callback;		 	/* saved callback handler */

	callback_result p_result;
	double p_double;
	int p_int;

	long count;

	/* make sure our callback is a procedure */

	if (callback->sp_form != ft_proc)
		abend(SETL_SYSTEM "Expected procedure in callout, but found:\n %s",
			abend_opnd_str(SETL_SYSTEM callback));

	va_start(argp, fmt);
	count = 0;
	while (*fmt) {
		count++;
		switch(*fmt++) {
		case 'I':		/* int */
			p_int = va_arg(argp, int);
			spare.sp_form = ft_short;
			spare.sp_val.sp_short_value = p_int;
			push_pstack(&spare);
			break;
		case 'D':		/* double */
			p_double = va_arg(argp, double);
			i_get_real(real_ptr);
			spare.sp_form = ft_real;
			spare.sp_val.sp_real_ptr = real_ptr;
			real_ptr->r_use_count = 1;
			real_ptr->r_value = p_double;
			push_pstack(&spare);
			break;
		}
	}
	va_end(argp);

	/* call the callback handler */

	save_callback.sp_form = callback->sp_form;
	save_callback.sp_val.sp_biggest = callback->sp_val.sp_biggest;
	spare.sp_form = ft_omega;
	call_procedure(SETL_SYSTEM &spare,
		&save_callback,
		NULL,
		count,YES,NO,0);
	callback->sp_form = save_callback.sp_form;
	callback->sp_val.sp_biggest = save_callback.sp_val.sp_biggest;

	if (spare.sp_form == ft_short) {
		p_result.value_int32 = (spare.sp_val.sp_short_value);
	} else
		if (spare.sp_form == ft_real) {
			p_result.value_float = (float)(spare.sp_val.sp_real_ptr)->r_value;
		}
		else {
			abend(SETL_SYSTEM "Expected integer or real from callback, but found:\n %s",
				abend_opnd_str(SETL_SYSTEM &spare));
		}

	unmark_specifier(&spare);

	return p_result;
}

/*
 *
 */

void gr_internal_destructor(mimage_ptr mimg)
{
	if (mimg != NULL) 
		destroy_mimage(mimg);
}

/*
 *
 */

SETL_API int32 GRLIB__INIT(
SETL_SYSTEM_PROTO_VOID)
{

TRACE_STUBS("GRLIB_INIT");

#if macintosh && __MWERKS__ && __DEBUG__
	DebugStart();
#endif

	if (mimg_lib_init())
	{
		gr_type = register_type(SETL_SYSTEM "mimage", gr_internal_destructor);
		if (gr_type == 0) return 1;
	}

	return 0;
}

/*
 *
 */

SETL_API void GR_CLONE(
	SETL_SYSTEM_PROTO
	int argc,				  /* number of arguments passed	 */
	specifier *argv,			  /* argument vector (two here)	 */
	specifier *target)			/* return value			 */

{
	mimage_ptr mimg;

TRACE_STUBS("GR_CLONE");

	check_gr_arg(SETL_SYSTEM argv, 0, "clone");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = clone_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_CREATE_IMAGE(
	SETL_SYSTEM_PROTO
	int argc,				  /* number of arguments passed	 */
	specifier *argv,			  /* argument vector (two here)	 */
	specifier *target)			/* return value			 */

{
	mimage_ptr mimg;
	int width, height, planes_number;
	char kind;

TRACE_STUBS("GR_CREATE_IMAGE");

	check_arg(SETL_SYSTEM argv, 0, ft_short, "integer", "create_image");
	check_arg(SETL_SYSTEM argv, 1, ft_short, "integer", "create_image");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "integer", "create_image");
	check_arg(SETL_SYSTEM argv, 3, ft_short, "integer", "create_image");

	height	 = (argv[0].sp_val.sp_short_value);
	width	  = (argv[1].sp_val.sp_short_value);
	planes_number = (argv[2].sp_val.sp_short_value);
	kind	   = (char)(argv[3].sp_val.sp_short_value);

	mimg = create_mimage(width, height, planes_number, kind, DENSE_IMAGE);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
}

SETL_API void GR_DESTROY_IMAGE(
	SETL_SYSTEM_PROTO
	int argc,				  /* number of arguments passed	 */
	specifier *argv,			  /* argument vector (two here)	 */
	specifier *target)			/* return value			 */
{
TRACE_STUBS("GR_DESTROY_IMAGE");

	unmark_specifier(target);		 /* just unmark the specifier, the gc 
								 will finish the job		 */
	target->sp_form = ft_omega;
	return;
}

SETL_API void GR_READ_TIFF_IMAGE(
	SETL_SYSTEM_PROTO
	int argc,				  /* number of arguments passed	 */
	specifier *argv,			  /* argument vector (two here)	 */
	specifier *target)			/* return value			 */
{
	mimage_ptr mimg;
	char fname[FNAME_MAX_SIZE];

TRACE_STUBS("GR_READ_TIFF_IMAGE");

	check_arg(SETL_SYSTEM argv, 0, ft_string, "string", "read_jpeg_image");

	setl2_string_to_cstring(argv[0].sp_val.sp_string_ptr, (unsigned char *)fname, sizeof(fname));

	mimg = read_tiff_mimage(fname);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
}

SETL_API void GR_WRITE_TIFF_IMAGE(
	SETL_SYSTEM_PROTO
	int argc,				  /* number of arguments passed	 */
	specifier *argv,			  /* argument vector (two here)	 */
	specifier *target)			/* return value			 */
{
	mimage_ptr mimg;
	char fname[FNAME_MAX_SIZE];
	int err;

TRACE_STUBS("GR_WRITE_TIFF_IMAGE");

	check_arg(SETL_SYSTEM argv, 0, ft_string, "string", "write_tiff_image");
	check_gr_arg(SETL_SYSTEM argv, 1, "write_tiff_image");

	setl2_string_to_cstring(argv[0].sp_val.sp_string_ptr, (unsigned char *)fname, sizeof(fname));

	mimg = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	err = write_tiff_mimage(fname, mimg);

	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_READ_JPEG_IMAGE(
	SETL_SYSTEM_PROTO
	int argc,				  /* number of arguments passed	 */
	specifier *argv,			  /* argument vector (two here)	 */
	specifier *target)			/* return value			 */
{
	mimage_ptr mimg;
	char fname[FNAME_MAX_SIZE];

TRACE_STUBS("GR_READ_JPEG_IMAGE");

	check_arg(SETL_SYSTEM argv, 0, ft_string, "string", "read_jpeg_image");

	setl2_string_to_cstring(argv[0].sp_val.sp_string_ptr, (unsigned char *)fname, sizeof(fname));

	mimg = read_jpeg_mimage(fname);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
}

SETL_API void GR_WRITE_JPEG_IMAGE(
	SETL_SYSTEM_PROTO
	int argc,				  /* number of arguments passed	 */
	specifier *argv,			  /* argument vector (two here)	 */
	specifier *target)			/* return value			 */
{
	mimage_ptr mimg;
	char fname[FNAME_MAX_SIZE];
	int err;

TRACE_STUBS("GR_WRITE_JPEG_IMAGE");

	check_arg(SETL_SYSTEM argv, 0, ft_string, "string", "write_jpeg_image");
	check_gr_arg(SETL_SYSTEM argv, 1, "write_jpeg_image");

	mimg = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	setl2_string_to_cstring(argv[0].sp_val.sp_string_ptr, (unsigned char *)fname, sizeof(fname));

	err = write_jpeg_mimage(fname, mimg);

	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_CONVERT_TO_FLOAT(
	SETL_SYSTEM_PROTO
	int argc,				  /* number of arguments passed	 */
	specifier *argv,			  /* argument vector (two here)	 */
	specifier *target)			/* return value			 */
{
	mimage_ptr mimg;
	int err;

	specifier return1;

TRACE_STUBS("GR_CONVERT_TO_FLOAT");

	check_gr_arg(SETL_SYSTEM argv, 0, "convert_to_float");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if (mimg->kind == FLOAT_IMAGE)
	{
		/* return the same image we passed */
		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		err = 1;	/* operation not needed */

		goto abort_to_float;
	}

	if (mimg->use_count > MIN_REF) {

		mimg = clone_mimage(mimg);

		if (!mimg)
			abend(SETL_SYSTEM msg_malloc_error);

		mimg->use_count = REF_BASE;
	}
	
	err = convert_mimage_to_float(mimg);

	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		return;
	}

abort_to_float:

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_CONVERT_TO_INT(
	SETL_SYSTEM_PROTO
	int argc,				  /* number of arguments passed	 */
	specifier *argv,			  /* argument vector (two here)	 */
	specifier *target)			/* return value			 */
{
	mimage_ptr mimg;
	int err;

	specifier return1;

TRACE_STUBS("GR_CONVERT_TO_INT");

	check_gr_arg(SETL_SYSTEM argv, 0, "convert_to_int");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if (mimg->kind == DISCRETE_IMAGE)
	{
		/* return the same image we passed */
		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		err = 1; /* op not needed */

		goto abort_to_int;
	}

	if (mimg->use_count > MIN_REF) {

		mimg = clone_mimage(mimg);

		if (!mimg)
			abend(SETL_SYSTEM msg_malloc_error);

		mimg->use_count = REF_BASE;
	}

	err = convert_mimage_to_discrete(mimg);
	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		return;
	}

abort_to_int:

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_CROP(
	SETL_SYSTEM_PROTO
	int argc,				  /* number of arguments passed	 */
	specifier *argv,			  /* argument vector (two here)	 */
	specifier *target)			/* return value			 */
{
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg;
	int i;
	int rect_array[4];
	image_rect rect;
	int err;

	specifier return1;

TRACE_STUBS("GR_CROP");

	check_gr_arg(SETL_SYSTEM argv, 0, "crop");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "string", "crop");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if (mimg->use_count > MIN_REF) {

		mimg = clone_mimage(mimg);

		if (!mimg)
			abend(SETL_SYSTEM msg_malloc_error);

		mimg->use_count = REF_BASE;
	}
	
	i=0;
	
	ITERATE_TUPLE_BEGIN(ia,argv[1])
	{
		check_type(SETL_SYSTEM argv, 1, ia_element->sp_form, ft_short, "rectangle", "the rectangle must be integer", "crop");
		if (ia_element->sp_form==ft_short)
			rect_array[i] = (ia_element->sp_val.sp_short_value);
		else 
			abend(SETL_SYSTEM "the rectangle must be integer","rectangle",1,
				"crop", abend_opnd_str(SETL_SYSTEM argv));
		++i;
	}
	ITERATE_TUPLE_END(ia)

	rect.x  = rect_array[0];
	rect.y  = rect_array[1];
	rect.dx = rect_array[2];
	rect.dy = rect_array[3];

	err = crop_mimage(mimg, &rect);

	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_STUFF(
	SETL_SYSTEM_PROTO
	int argc,				  /* number of arguments passed	 */
	specifier *argv,			  /* argument vector (two here)	 */
	specifier *target)			/* return value					  */

{
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg_a, mimg_b;
	int i;
	int delta_array[2];
	gr_image_displacement delta;
	int err;

	specifier return1;

TRACE_STUBS("GR_STUFF");

	check_gr_arg(SETL_SYSTEM argv, 0, "stuff");
	check_gr_arg(SETL_SYSTEM argv, 1, "stuff");

	mimg_a = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if (mimg_a->use_count > MIN_REF) {

		mimg_a = clone_mimage(mimg_a);

		if (!mimg_a)
			abend(SETL_SYSTEM msg_malloc_error);

		mimg_a->use_count = REF_BASE;
	}
	
	mimg_b = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	check_arg(SETL_SYSTEM argv, 2, ft_tuple, "tuple", "stuff");

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[2])
	{
		if (ia_element->sp_form==ft_short)

			delta_array[i] = (ia_element->sp_val.sp_short_value);

		else 

			abend(SETL_SYSTEM "the rectangle must be integer","rectangle",1,
				"stuff", abend_opnd_str(SETL_SYSTEM argv));

		++i;
	}
	ITERATE_TUPLE_END(ia)

	delta.x  = delta_array[0];
	delta.y  = delta_array[1];

	err = stuff_mimage(mimg_a, mimg_b, &delta);

	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg_a;
		push_pstack(&return1);

		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_SHRINK(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	mimage_ptr mimg;
	int rect_array[4];
	image_rect rect;
	int err;

TRACE_STUBS("GR_SHRINK");

	check_gr_arg(SETL_SYSTEM argv, 0, "shrink");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	err = shrink_mimage(mimg, 0, &rect);

	if (err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	rect_array[0] = rect.x;
	rect_array[1] = rect.y;
	rect_array[2] = rect.dx;
	rect_array[3] = rect.dy;

	create_and_return_integers_tuple_from_array(SETL_SYSTEM rect_array, 4, target);
}

SETL_API void GR_WIDTH(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int width;
	mimage_ptr mimg;

TRACE_STUBS("GR_WIDTH");

	check_gr_arg(SETL_SYSTEM argv, 0, "width");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	width = width_mimage(mimg);

	if (!width) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = width;
}

SETL_API void GR_HEIGHT(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int height;
	mimage_ptr mimg;

TRACE_STUBS("GR_HEIGHT");

	check_gr_arg(SETL_SYSTEM argv, 0, "height");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	height = height_mimage(mimg);

	if (!height) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = height;
}

SETL_API void GR_PLANES_NUMBER(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int planes_number;
	mimage_ptr mimg;

TRACE_STUBS("GR_PLANES_NUMBER");

	check_gr_arg(SETL_SYSTEM argv, 0, "planes_number");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	planes_number = mimg->comp;

	if (!planes_number) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = planes_number;
}

SETL_API void GR_TYPE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int kind;
	mimage_ptr mimg;

TRACE_STUBS("GR_TYPE");

	check_gr_arg(SETL_SYSTEM argv, 0, "type");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	kind = mimg->kind;

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = kind;
}

SETL_API void GR_DENSITY(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int density;
	mimage_ptr mimg;

TRACE_STUBS("GR_DENSITY");

	check_gr_arg(SETL_SYSTEM argv, 0, "density");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	density = mimg->density;

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = density;
}

SETL_API void GR_CONST(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int tuple_len;
	
	int i;
	mimage_ptr mimg = NULL;
	mpixel *multi_array = NULL;
	fmpixel *f_multi_array = NULL;
	int dim_array[4];
	image_dimension dim;
	int kind, comp;

TRACE_STUBS("GR_CONST");

	check_arg(SETL_SYSTEM argv, 0, ft_tuple, "tuple", "const");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "const");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "integer", "const");

	kind = argv[2].sp_val.sp_short_value;
	if ((kind != FLOAT_IMAGE) && (kind != DISCRETE_IMAGE))
		abend(SETL_SYSTEM "Image type should be 0 for float or 1 for integer","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	tuple_len = TUPLE_LEN(argv[0]);
	f_multi_array = malloc(tuple_len * sizeof(fmpixel));
	multi_array = malloc(tuple_len * sizeof(mpixel));
	if (!f_multi_array || !multi_array)
		goto finish_const;
		
	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[0])
	{
		if (i==16)
			abend(SETL_SYSTEM "the tuple must have less than 16 elements","tuple",1,
				"const", abend_opnd_str(SETL_SYSTEM argv));

		if (kind == FLOAT_IMAGE) {
			if (ia_element->sp_form==ft_real)
				f_multi_array[i] = (ia_element->sp_val.sp_real_ptr->r_value);
			else 
				abend(SETL_SYSTEM "the tuple must be real","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv));
		} else {
			if (ia_element->sp_form==ft_short)
				multi_array[i] = (ia_element->sp_val.sp_short_value);
			else 
				abend(SETL_SYSTEM "the tuple must be integer","rectangle",1,
					"const", abend_opnd_str(SETL_SYSTEM argv));
		}

		++i;
	}
	ITERATE_TUPLE_END(ia)
	comp = i;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_short)
			dim_array[i] = (ia_element->sp_val.sp_short_value);
		else 
			abend(SETL_SYSTEM "the rectangle must be integer","rectangle",1,
				"shrink", abend_opnd_str(SETL_SYSTEM argv+1));
		++i;
	}
	ITERATE_TUPLE_END(ia)

	dim.x  = dim_array[0];
	dim.y  = dim_array[1];

	mimg = const_mimage((kind == FLOAT_IMAGE ? (void *)f_multi_array : (void *)multi_array), &dim, kind, comp);

finish_const:

	SMART_FREE(f_multi_array);
	SMART_FREE(multi_array);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_PLUS(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_PLUS");

	check_gr_arg(SETL_SYSTEM argv, 0, "plus");
	check_gr_arg(SETL_SYSTEM argv, 1, "plus");

	mimg_a = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	mimg_b = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	mimg = plus_mimage(mimg_a, mimg_b);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_MINUS(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_MINUS");

	check_gr_arg(SETL_SYSTEM argv, 0, "minux");
	check_gr_arg(SETL_SYSTEM argv, 1, "minus");

	mimg_a = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	mimg_b = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	mimg = minus_mimage(mimg_a, mimg_b);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_TIMES(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_TIMES");

	check_gr_arg(SETL_SYSTEM argv, 0, "times");
	check_gr_arg(SETL_SYSTEM argv, 1, "times");

	mimg_a = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	mimg_b = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	mimg = times_mimage(mimg_a, mimg_b);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_DIVIDE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_DIVIDE");

	check_gr_arg(SETL_SYSTEM argv, 0, "divide");
	check_gr_arg(SETL_SYSTEM argv, 1, "divide");

	mimg_a = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	mimg_b = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	mimg = divide_mimage(mimg_a, mimg_b);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_MAXIM(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_MAXIM");

	check_gr_arg(SETL_SYSTEM argv, 0, "maxim");
	check_gr_arg(SETL_SYSTEM argv, 1, "maxim");

	mimg_a = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	mimg_b = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	mimg = maxim_mimage(mimg_a, mimg_b);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_MINIM(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_MINIM");

	check_gr_arg(SETL_SYSTEM argv, 0, "minim");
	check_gr_arg(SETL_SYSTEM argv, 1, "minim");

	mimg_a = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	mimg_b = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	mimg = minim_mimage(mimg_a, mimg_b);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_POWER(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_POWER");

	check_gr_arg(SETL_SYSTEM argv, 0, "power");
	check_gr_arg(SETL_SYSTEM argv, 1, "power");

	mimg_a = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	mimg_b = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	mimg = power_mimage(mimg_a, mimg_b);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_MAXOF(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	float f;
	i_real_ptr_type real_ptr;
	mimage_ptr mimg;

TRACE_STUBS("GR_MAXOF");

	check_gr_arg(SETL_SYSTEM argv, 0, "maxof");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	f = maxof_mimage(mimg);

	unmark_specifier(target);
	i_get_real(real_ptr);
	target->sp_form = ft_real;
	target->sp_val.sp_real_ptr = real_ptr;
	real_ptr->r_use_count = 1;
	real_ptr->r_value = f;

	return;

}

SETL_API void GR_MINOF(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	float f;
	i_real_ptr_type real_ptr;
	mimage_ptr mimg;

TRACE_STUBS("GR_MINOF");

	check_gr_arg(SETL_SYSTEM argv, 0, "minof");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	f = minof_mimage(mimg);

	unmark_specifier(target);
	i_get_real(real_ptr);
	target->sp_form = ft_real;
	target->sp_val.sp_real_ptr = real_ptr;
	real_ptr->r_use_count = 1;
	real_ptr->r_value = f;

	return;

}

SETL_API void GR_SUMALL(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	double *farray;
	i_real_ptr_type real_ptr;
	mimage_ptr mimg;
	int power;

TRACE_STUBS("GR_SUMALL");

	check_gr_arg(SETL_SYSTEM argv, 0, "sumall");

	check_arg(SETL_SYSTEM argv, 1, ft_short, "short", "sumall");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	power = (argv[1].sp_val.sp_short_value);

	farray = sumall_mimage(mimg, power);

	if (farray) {
		create_and_return_doubles_tuple_from_array(SETL_SYSTEM farray, power, target);
		free(farray);
	} else {
		unmark_specifier(target);
		target->sp_form = ft_omega;
	}
}

SETL_API void GR_SCALE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int i;
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg;
	image_dimension dim;
	int dim_array[2];

TRACE_STUBS("GR_SCALE");

	check_gr_arg(SETL_SYSTEM argv, 0, "scale");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "scale");

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_short)
			dim_array[i] = (ia_element->sp_val.sp_short_value);
		else 
			abend(SETL_SYSTEM "the tuple must be integer","dimension",1,
				"scale", abend_opnd_str(SETL_SYSTEM argv));
		++i;
	}
	ITERATE_TUPLE_END(ia)

	dim.x = dim_array[0];
	dim.y = dim_array[1];


	mimg = scale_mimage(mimg, &dim);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_SORT(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	void *array;
	int height, width, comp, kind;
	int elements;

	mimage_ptr mimg;

TRACE_STUBS("GR_SORT");

	check_gr_arg(SETL_SYSTEM argv, 0, "sort");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);

	array = sort_mimage(mimg, &elements);
	if (array) {

		if (mimg->kind == FLOAT_IMAGE)
			create_and_return_floats_tuple_from_array(SETL_SYSTEM array, elements, target);
		else
			create_and_return_shorts_tuple_from_array(SETL_SYSTEM array, elements, target);

		free(array);
	} else {
		unmark_specifier(target);
		target->sp_form = ft_omega;
	}
}

SETL_API void GR_WIDTH_AND_HEIGHT(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int array[2];
	int height, width, comp, kind;
	int elements;

	mimage_ptr mimg;

TRACE_STUBS("GR_WIDTH_AND_HEIGHT");

	check_gr_arg(SETL_SYSTEM argv, 0, "width_and_height");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	get_mimage_info(mimg, &height, &width, &comp, &kind, NULL);

	width_and_height_mimage(mimg, &array[0], &array[1]);

	create_and_return_integers_tuple_from_array(SETL_SYSTEM array, 2, target);
}

SETL_API void GR_CONVOLVE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int i;
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg;

	int *i_offset, *j_offset;
	float *coefficient;
	size_t dim;

TRACE_STUBS("GR_CONVOLVE");

	check_gr_arg(SETL_SYSTEM argv, 0, "convolve");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "convolve");
	check_arg(SETL_SYSTEM argv, 2, ft_tuple, "tuple", "convolve");
	check_arg(SETL_SYSTEM argv, 3, ft_tuple, "tuple", "convolve");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ( ( TUPLE_SIZE(argv[1]) != TUPLE_SIZE(argv[1]) ) &&
		( TUPLE_SIZE(argv[2]) != TUPLE_SIZE(argv[3]) ) )
		goto abort_convolve;

	dim = TUPLE_SIZE(argv[1]);
	i_offset = malloc(dim*sizeof(int));
	if (!i_offset)
		goto abort_convolve;

	j_offset = malloc(dim*sizeof(int));
	if (!j_offset) {
		free(i_offset);
		goto abort_convolve;
	}

	coefficient = malloc(dim*sizeof(int));
	if (!coefficient) {
		free(i_offset);
		free(j_offset);
		goto abort_convolve;
	}

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_short)
			i_offset[i] = (ia_element->sp_val.sp_short_value);
		else 
			abend(SETL_SYSTEM "the tuple must be integer","convolve",1,
				"convolve", abend_opnd_str(SETL_SYSTEM argv));
		++i;
	}
	ITERATE_TUPLE_END(ia)

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[2])
	{
		if (ia_element->sp_form==ft_short)
			j_offset[i] = (ia_element->sp_val.sp_short_value);
		else 
			abend(SETL_SYSTEM "the tuple must be integer","convolve",1,
				"convolve", abend_opnd_str(SETL_SYSTEM argv));
		++i;
	}
	ITERATE_TUPLE_END(ia)

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[3])
	{
		if (ia_element->sp_form==ft_short)
			coefficient[i] = (ia_element->sp_val.sp_short_value);
		else if (ia_element->sp_form==ft_real)
			coefficient[i] = (ia_element->sp_val.sp_real_ptr->r_value);
		else 
			abend(SETL_SYSTEM "the tuple must be either real or integer","convolve",1,
				"convolve", abend_opnd_str(SETL_SYSTEM argv));
		++i;
	}
	ITERATE_TUPLE_END(ia)

		mimg = convolve_mimage(mimg, i_offset, j_offset, coefficient, dim);

	free(i_offset);
	free(j_offset);
	free(coefficient);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort_convolve:

	unmark_specifier(target);
	target->sp_form = ft_omega;
	return;
}

SETL_API void GR_MAXVOLVE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int i;
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg;

	int *i_offset, *j_offset;
	float *coefficient;
	size_t dim;

TRACE_STUBS("GR_MAXVOLVE");

	check_gr_arg(SETL_SYSTEM argv, 0, "maxvolve");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "maxvolve");
	check_arg(SETL_SYSTEM argv, 2, ft_tuple, "tuple", "maxvolve");
	check_arg(SETL_SYSTEM argv, 3, ft_tuple, "tuple", "maxvolve");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ( ( TUPLE_SIZE(argv[1]) != TUPLE_SIZE(argv[1]) ) &&
		( TUPLE_SIZE(argv[2]) != TUPLE_SIZE(argv[3]) ) )
		goto abort_maxvolve;

	dim = TUPLE_SIZE(argv[1]);
	i_offset = malloc(dim*sizeof(int));
	if (!i_offset)
		goto abort_maxvolve;

	j_offset = malloc(dim*sizeof(int));
	if (!j_offset) {
		free(i_offset);
		goto abort_maxvolve;
	}

	coefficient = malloc(dim*sizeof(*coefficient));
	if (!coefficient) {
		free(i_offset);
		free(j_offset);
		goto abort_maxvolve;
	}

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_short)
			i_offset[i] = (ia_element->sp_val.sp_short_value);
		else {
			free(i_offset);
			free(j_offset);
			free(coefficient);
			abend(SETL_SYSTEM "the tuple must be integer","maxvolve",1,
				"maxvolve", abend_opnd_str(SETL_SYSTEM argv));
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[2])
	{
		if (ia_element->sp_form==ft_short)
			j_offset[i] = (ia_element->sp_val.sp_short_value);
		else {
			free(i_offset);
			free(j_offset);
			free(coefficient);
			abend(SETL_SYSTEM "the tuple must be integer","maxvolve",1,
				"maxvolve", abend_opnd_str(SETL_SYSTEM argv));
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[3])
	{
		if (ia_element->sp_form==ft_short)
			coefficient[i] = (ia_element->sp_val.sp_short_value);
		else if (ia_element->sp_form==ft_real)
			coefficient[i] = (ia_element->sp_val.sp_real_ptr->r_value);
		else {
			free(i_offset);
			free(j_offset);
			free(coefficient);
			abend(SETL_SYSTEM "the tuple must be integer","maxvolve",1,
				"maxvolve", abend_opnd_str(SETL_SYSTEM argv));
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = maxvolve_mimage(mimg, i_offset, j_offset, coefficient, dim);

	free(i_offset);
	free(j_offset);
	free(coefficient);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort_maxvolve:

	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_MINVOLVE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int i;
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg;

	int *i_offset, *j_offset;
	float *coefficient;
	size_t dim;

TRACE_STUBS("GR_MINVOLVE");

	check_gr_arg(SETL_SYSTEM argv, 0, "minvolve");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "minvolve");
	check_arg(SETL_SYSTEM argv, 2, ft_tuple, "tuple", "minvolve");
	check_arg(SETL_SYSTEM argv, 3, ft_tuple, "tuple", "minvolve");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ( ( TUPLE_SIZE(argv[1]) != TUPLE_SIZE(argv[1]) ) &&
		( TUPLE_SIZE(argv[2]) != TUPLE_SIZE(argv[3]) ) )
		goto abort_minvolve;

	dim = TUPLE_SIZE(argv[1]);
	i_offset = malloc(dim*sizeof(int));
	if (!i_offset)
		goto abort_minvolve;

	j_offset = malloc(dim*sizeof(int));
	if (!j_offset) {
		free(i_offset);
		goto abort_minvolve;
	}

	coefficient = malloc(dim*sizeof(*coefficient));
	if (!coefficient) {
		free(i_offset);
		free(j_offset);
		goto abort_minvolve;
	}

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_short)
			i_offset[i] = (ia_element->sp_val.sp_short_value);
		else {
			free(i_offset);
			free(j_offset);
			free(coefficient);
			abend(SETL_SYSTEM "the tuple must be integer","minvolve",1,
				"minvolve", abend_opnd_str(SETL_SYSTEM argv));
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[2])
	{
		if (ia_element->sp_form==ft_short)
			j_offset[i] = (ia_element->sp_val.sp_short_value);
		else {
			free(i_offset);
			free(j_offset);
			free(coefficient);
			abend(SETL_SYSTEM "the tuple must be integer","minvolve",1,
				"minvolve", abend_opnd_str(SETL_SYSTEM argv));
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[3])
	{
		if (ia_element->sp_form==ft_short)
			coefficient[i] = (ia_element->sp_val.sp_short_value);
		else if (ia_element->sp_form==ft_real)
			coefficient[i] = (ia_element->sp_val.sp_real_ptr->r_value);
		else {
			free(i_offset);
			free(j_offset);
			free(coefficient);
			abend(SETL_SYSTEM "the tuple must be either real or integer","minvolve",1,
				"minvolve", abend_opnd_str(SETL_SYSTEM argv));
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = minvolve_mimage(mimg, i_offset, j_offset, coefficient, dim);

	free(i_offset);
	free(j_offset);
	free(coefficient);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort_minvolve:

	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_SELF(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg;
	int i;
	int array[2];
	image_rect rect;
	int err;

TRACE_STUBS("GR_SELF");

	check_gr_arg(SETL_SYSTEM argv, 0, "self");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "self");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if (TUPLE_SIZE(argv[1]) != 2)
		goto abort_self;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_short)
			array[i] = (ia_element->sp_val.sp_short_value);
		else 
			abend(SETL_SYSTEM "the rectangle must be integer","pair",1,
				"self", abend_opnd_str(SETL_SYSTEM argv));
		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = extract_planes_mimage(mimg, array[0], array[1]);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort_self:

	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_SELF_PUT(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg, mimg_res;
	int i;
	int array[2];
	image_rect rect;
	int err;

TRACE_STUBS("GR_SELF_PUT");

	check_gr_arg(SETL_SYSTEM argv, 0, "self_put");
	check_gr_arg(SETL_SYSTEM argv, 1, "self_put");
	check_arg(SETL_SYSTEM argv, 2, ft_tuple, "tuple", "self_put");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	mimg_res = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);


	if (TUPLE_SIZE(argv[2]) != 2)
		goto abort_self_put;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[2])
	{
		if (ia_element->sp_form==ft_short)
			array[i] = (ia_element->sp_val.sp_short_value);
		else 
			abend(SETL_SYSTEM "the rectangle must be integer","self_put",1,
				"self_put", abend_opnd_str(SETL_SYSTEM argv));
		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = clone_mimage(mimg);
	if (!mimg)
		goto abort_self_put;

	mimg = set_planes_mimage(mimg, mimg_res, array[0], array[1]);
	
	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort_self_put:

	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_SIN(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

	check_gr_arg(SETL_SYSTEM argv, 0, "sin");

TRACE_STUBS("GR_SIN");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = sin_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_COSIN(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_COSIN");

	check_gr_arg(SETL_SYSTEM argv, 0, "cosin");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = cosin_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_TAN(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_TAN");

	check_gr_arg(SETL_SYSTEM argv, 0, "tan");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = tan_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_EXP(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_EXP");

	check_gr_arg(SETL_SYSTEM argv, 0, "exp");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = exp_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_LOG(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_LOG");

	check_gr_arg(SETL_SYSTEM argv, 0, "log");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = log_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_ABS(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_ABS");

	check_gr_arg(SETL_SYSTEM argv, 0, "abs");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = abs_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_EVEN(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_EVEN");

	check_gr_arg(SETL_SYSTEM argv, 0, "even");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = even_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_ODD(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_ODD");

	check_gr_arg(SETL_SYSTEM argv, 0, "odd");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = odd_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_SQRT(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_SORT");

	check_gr_arg(SETL_SYSTEM argv, 0, "sqrt");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = sqrt_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_ASIN(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_ASIN");

	check_gr_arg(SETL_SYSTEM argv, 0, "asin");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = asin_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_ACOSIN(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_ACOSIN");

	check_gr_arg(SETL_SYSTEM argv, 0, "acosin");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = acosin_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_ATAN(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_ATAN");

	check_gr_arg(SETL_SYSTEM argv, 0, "atan");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = atan_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_FIX(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_FIX");

	check_gr_arg(SETL_SYSTEM argv, 0, "fix");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = fix_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_FLOOR(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_FLOOR");

	check_gr_arg(SETL_SYSTEM argv, 0, "floor");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = floor_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_CEILING(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_CEILING");

	check_gr_arg(SETL_SYSTEM argv, 0, "ceiling");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = ceiling_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_THRESHOLD(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int i;
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg;

	float *fvalues;
	size_t dim;

TRACE_STUBS("GR_THRESHOLD");

	check_gr_arg(SETL_SYSTEM argv, 0, "threshold");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "threshold");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	dim = TUPLE_SIZE(argv[1]);

	fvalues = malloc(dim*sizeof(float));
	if (!fvalues)
		goto abort_threshold;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_real)
			fvalues[i] = (ia_element->sp_val.sp_real_ptr->r_value);
		else {
			abend(SETL_SYSTEM "the tuple must be real","threshold",1,
				"convolve", abend_opnd_str(SETL_SYSTEM argv));
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = threshold_mimage(mimg, fvalues, dim);

	free(fvalues);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort_threshold:

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

SETL_API void GR_HISTOGRAM(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int i;
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg;

	size_t *histogram;
	float *fvalues;
	size_t dim;

TRACE_STUBS("GR_HISTOGRAM");

	check_gr_arg(SETL_SYSTEM argv, 0, "histogram");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "histogram");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	dim = TUPLE_SIZE(argv[1]);

	fvalues = malloc(dim*sizeof(float));
	if (!fvalues)
		goto abort_histogram;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_real)
			fvalues[i] = (ia_element->sp_val.sp_real_ptr->r_value);
		else {
			abend(SETL_SYSTEM "the tuple must be real","histogram",1,
				"convolve", abend_opnd_str(SETL_SYSTEM argv));
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)

	histogram = histogram_mimage(mimg, fvalues, dim);

	free(fvalues);

	if (histogram) {
		create_and_return_longs_tuple_from_array(SETL_SYSTEM histogram, dim+1, target);
		free(histogram);
		return;
	}

abort_histogram:

	unmark_specifier(target);
	target->sp_form = ft_omega;
	return;

}

SETL_API void GR_GRADIENT(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg;
	int width, height, planes_number, kind;
	image_dimension dim;

TRACE_STUBS("GR_GRADIENT");

	check_arg(SETL_SYSTEM argv, 0, ft_short, "integer", "gradient");
	check_arg(SETL_SYSTEM argv, 1, ft_short, "integer", "gradient");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "integer", "gradient");
	check_arg(SETL_SYSTEM argv, 3, ft_short, "integer", "gradient");

	height		= (argv[0].sp_val.sp_short_value);
	width		 = (argv[1].sp_val.sp_short_value);
	kind		  = (argv[2].sp_val.sp_short_value);
	planes_number = (argv[3].sp_val.sp_short_value);

	dim.x = width;
	dim.y = height;

	mimg = gradient_mimage(SETL_SYSTEM &argv[4], (float(*)(void *,void *,int,int,int))gradient_callback_op, &dim, kind, planes_number);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
}

SETL_API void GR_UNIMATH(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg;

TRACE_STUBS("GR_UNIMATH");

	check_gr_arg(SETL_SYSTEM argv, 0, "unimath");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = unimath_mimage(SETL_SYSTEM &argv[1], mimg, (float(*)(void *,void *,float,int))unimath_callback_op);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
}

float gradient_callback_op(SETL_SYSTEM_PROTO void *user_ptr2, int x, int y, int c)
{
	callback_result the_callback_result;

	the_callback_result =  d_callback_caller(
		SETL_SYSTEM 
		user_ptr2,
		"III", x, y, c,
		NULL);

	return the_callback_result.value_float;
}

float unimath_callback_op(SETL_SYSTEM_PROTO void *user_ptr2, float f, int c)
{
	callback_result the_callback_result;

	the_callback_result =  d_callback_caller(
		SETL_SYSTEM 
		user_ptr2,
		"DI", f, c,
		NULL);

	return the_callback_result.value_float;
}

SETL_API void GR_PLUS_CONST(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int kind, size;
	void *values;
	mimage_ptr mimg;

TRACE_STUBS("GR_PLUS_CONST");

	check_gr_arg(SETL_SYSTEM argv, 0, "plus_const");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "plus_const");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "short", "plus_const");

	kind = argv[2].sp_val.sp_short_value;
	if ((kind<=1) && (kind>=2))
		abend(SETL_SYSTEM "Image type parameter out of range","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ((size = TUPLE_SIZE(argv[1])) != mimg->comp)
		abend(SETL_SYSTEM "Image depth and constant tuple size should be the same","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	if (kind != mimg->kind)
		abend(SETL_SYSTEM "Image and constant tuple should have the same value","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	values = malloc(size * ((kind == FLOAT_IMAGE)?sizeof(float):sizeof(unsigned char)));
	if (!values)
		goto abort;
		
	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (kind == FLOAT_IMAGE) {
			if (ia_element->sp_form==ft_real)
				((float *)values)[i] = (ia_element->sp_val.sp_real_ptr->r_value);
			else {
				abend(SETL_SYSTEM "the tuple must be real","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		} else {
			if (ia_element->sp_form==ft_short)
				((unsigned char *)values)[i] = (ia_element->sp_val.sp_short_value);
			else {
				abend(SETL_SYSTEM "the tuple must be integer","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		}

		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = plus_mimage_constant(mimg, values);

	free(values);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_MINUS_CONST(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int kind, size;
	void *values;
	mimage_ptr mimg;

TRACE_STUBS("GR_MINUS_CONST");

	check_gr_arg(SETL_SYSTEM argv, 0, "minus_const");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "minus_const");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "short", "minus_const");

	kind = argv[2].sp_val.sp_short_value;
	if ((kind<=1) && (kind>=2))
		abend(SETL_SYSTEM "Image type parameter out of range","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ((size = TUPLE_SIZE(argv[1])) != mimg->comp)
		abend(SETL_SYSTEM "Image depth and constant tuple size should be the same","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	if (kind != mimg->kind)
		abend(SETL_SYSTEM "Image and constant tuple should have the same value","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	values = malloc(size * ((kind == FLOAT_IMAGE)?sizeof(float):sizeof(unsigned char)));
	if (!values)
		goto abort;
		
	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (kind == FLOAT_IMAGE) {
			if (ia_element->sp_form==ft_real)
				((float *)values)[i] = (ia_element->sp_val.sp_real_ptr->r_value);
			else {
				abend(SETL_SYSTEM "the tuple must be real","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		} else {
			if (ia_element->sp_form==ft_short)
				((unsigned char *)values)[i] = (ia_element->sp_val.sp_short_value);
			else {
				abend(SETL_SYSTEM "the tuple must be integer","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		}

		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = minus_mimage_constant(mimg, values);

	free(values);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_TIMES_CONST(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int kind, size;
	void *values;
	mimage_ptr mimg;

TRACE_STUBS("GR_TIMES_CONST");

	check_gr_arg(SETL_SYSTEM argv, 0, "times_const");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "times_const");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "integer", "times_const");

	kind = argv[2].sp_val.sp_short_value;
	if ((kind<=1) && (kind>=2))
		abend(SETL_SYSTEM "Image type parameter out of range","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ((size = TUPLE_SIZE(argv[1])) != mimg->comp)
		abend(SETL_SYSTEM "Image depth and constant tuple size should be the same","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	if (kind != mimg->kind)
		abend(SETL_SYSTEM "Image and constant tuple should have the same value","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	values = malloc(size * ((kind == FLOAT_IMAGE)?sizeof(float):sizeof(unsigned char)));
	if (!values)
		goto abort;
		
	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (kind == FLOAT_IMAGE) {
			if (ia_element->sp_form==ft_real)
				((float *)values)[i] = (ia_element->sp_val.sp_real_ptr->r_value);
			else {
				abend(SETL_SYSTEM "the tuple must be real","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		} else {
			if (ia_element->sp_form==ft_short)
				((unsigned char *)values)[i] = (ia_element->sp_val.sp_short_value);
			else {
				abend(SETL_SYSTEM "the tuple must be integer","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		}

		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = times_mimage_constant(mimg, values);

	free(values);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_DIVIDE_CONST(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int kind, size;
	void *values;
	mimage_ptr mimg;

TRACE_STUBS("GR_DIVIDE_CONST");

	check_gr_arg(SETL_SYSTEM argv, 0, "divide_const");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "divide_const");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "integer", "divide_const");

	kind = argv[2].sp_val.sp_short_value;
	if ((kind<=1) && (kind>=2))
		abend(SETL_SYSTEM "Image type parameter out of range","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ((size = TUPLE_SIZE(argv[1])) != mimg->comp)
		abend(SETL_SYSTEM "Image depth and constant tuple size should be the same","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	if (kind != mimg->kind)
		abend(SETL_SYSTEM "Image and constant tuple should have the same value","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	values = malloc(size * ((kind == FLOAT_IMAGE)?sizeof(float):sizeof(unsigned char)));
	if (!values)
		goto abort;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (kind == FLOAT_IMAGE) {
			if (ia_element->sp_form==ft_real)
				((float *)values)[i] = (ia_element->sp_val.sp_real_ptr->r_value);
			else {
				abend(SETL_SYSTEM "the tuple must be real","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		} else {
			if (ia_element->sp_form==ft_short)
				((unsigned char *)values)[i] = (ia_element->sp_val.sp_short_value);
			else {
				abend(SETL_SYSTEM "the tuple must be integer","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		}

		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = divide_mimage_constant(mimg, values);

	free(values);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_MAXIM_CONST(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int kind, size;
	void *values;
	mimage_ptr mimg;

TRACE_STUBS("GR_MAXIM_CONST");

	check_gr_arg(SETL_SYSTEM argv, 0, "maxim_const");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "maxim_const");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "integer", "maxim_const");

	kind = argv[2].sp_val.sp_short_value;
	if ((kind<=1) && (kind>=2))
		abend(SETL_SYSTEM "Image type parameter out of range","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ((size = TUPLE_SIZE(argv[1])) != mimg->comp)
		abend(SETL_SYSTEM "Image depth and constant tuple size should be the same","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	if (kind != mimg->kind)
		abend(SETL_SYSTEM "Image and constant tuple should have the same value","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	values = malloc(size * ((kind == FLOAT_IMAGE)?sizeof(float):sizeof(unsigned char)));
	if (!values)
		goto abort;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (kind == FLOAT_IMAGE) {
			if (ia_element->sp_form==ft_real)
				((float *)values)[i] = (ia_element->sp_val.sp_real_ptr->r_value);
			else {
				abend(SETL_SYSTEM "the tuple must be real","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		} else {
			if (ia_element->sp_form==ft_short)
				((unsigned char *)values)[i] = (ia_element->sp_val.sp_short_value);
			else {
				abend(SETL_SYSTEM "the tuple must be integer","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		}

		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = maxim_mimage_constant(mimg, values);

	free(values);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_MINIM_CONST(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int kind, size;
	void *values;
	mimage_ptr mimg;

TRACE_STUBS("GR_MINIM_C0NST");

	check_gr_arg(SETL_SYSTEM argv, 0, "minim_const");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "minim_const");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "integer", "minim_const");

	kind = argv[2].sp_val.sp_short_value;
	if ((kind<=1) && (kind>=2))
		abend(SETL_SYSTEM "Image type parameter out of range","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ((size = TUPLE_SIZE(argv[1])) != mimg->comp)
		abend(SETL_SYSTEM "Image depth and constant tuple size should be the same","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	if (kind != mimg->kind)
		abend(SETL_SYSTEM "Image and constant tuple should have the same value","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	values = malloc(size * ((kind == FLOAT_IMAGE)?sizeof(float):sizeof(unsigned char)));
	if (!values)
		goto abort;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (kind == FLOAT_IMAGE) {
			if (ia_element->sp_form==ft_real)
				((float *)values)[i] = (ia_element->sp_val.sp_real_ptr->r_value);
			else {
				abend(SETL_SYSTEM "the tuple must be real","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		} else {
			if (ia_element->sp_form==ft_short)
				((unsigned char *)values)[i] = (ia_element->sp_val.sp_short_value);
			else {
				abend(SETL_SYSTEM "the tuple must be integer","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		}

		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = minim_mimage_constant(mimg, values);

	free(values);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_POWER_CONST(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int kind, size;
	void *values;
	mimage_ptr mimg;

TRACE_STUBS("GR_POWER_CONST");

	check_gr_arg(SETL_SYSTEM argv, 0, "power_const");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "power_const");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "integer", "power_const");

	kind = argv[2].sp_val.sp_short_value;
	if ((kind<=1) && (kind>=2))
		abend(SETL_SYSTEM "Image type parameter out of range","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ((size = TUPLE_SIZE(argv[1])) != mimg->comp)
		abend(SETL_SYSTEM "Image depth and constant tuple size should be the same","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	if (kind != mimg->kind)
		abend(SETL_SYSTEM "Image and constant tuple should have the same value","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	values = malloc(size * ((kind == FLOAT_IMAGE)?sizeof(float):sizeof(unsigned char)));
	if (!values)
		goto abort;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (kind == FLOAT_IMAGE) {
			if (ia_element->sp_form==ft_real)
				((float *)values)[i] = (ia_element->sp_val.sp_real_ptr->r_value);
			else {
				abend(SETL_SYSTEM "the tuple must be real","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		} else {
			if (ia_element->sp_form==ft_short)
				((unsigned char *)values)[i] = (ia_element->sp_val.sp_short_value);
			else {
				abend(SETL_SYSTEM "the tuple must be integer","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+1));
			}
		}

		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = power_mimage_constant(mimg, values);

	free(values);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_PERLIN(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg;

TRACE_STUBS("GR_PERLIN");

	check_gr_arg(SETL_SYSTEM argv, 0, "perlin");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = perlin_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_PERLIN2(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg;

TRACE_STUBS("GR_PERLIN2");

	check_gr_arg(SETL_SYSTEM argv, 0, "perlin2");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = perlin2_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_TO_STRING(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int err;
	mimage_ptr mimg;
	char *cstr;
	size_t cstr_len;
	string_h_ptr_type setl_string;

TRACE_STUBS("GR_TO_STRING");

	check_gr_arg(SETL_SYSTEM argv, 0, "to_string");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	err = mimage_2_string_format(mimg, &cstr, &cstr_len);
	setl_string = setl2_string(SETL_SYSTEM cstr, cstr_len);
	free(cstr);

	if (err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_string;
	target->sp_val.sp_string_ptr = setl_string;

}

SETL_API void GR_FROM_STRING(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int err;
	mimage_ptr mimg;
	char *cstr;
	size_t cstr_len;
	string_h_ptr_type setl_string;

TRACE_STUBS("GR_FROM_STRING");

	check_arg(SETL_SYSTEM argv, 0, ft_string, "string", "from_string");

	cstr_len = (argv[0].sp_val.sp_string_ptr->s_length)+1;

	cstr = malloc(cstr_len);
	if (!cstr)
		goto abort_from_string;

	setl2_string_to_cstring(argv[0].sp_val.sp_string_ptr, (unsigned char *)cstr, cstr_len);
	mimg = string_format_2_mimage(cstr);
	free(cstr);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort_from_string:

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

SETL_API void GR_ANAGLYPH(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

	float scale, fraction;

TRACE_STUBS("GR_ANAGLYPH");

	check_gr_arg(SETL_SYSTEM argv, 0, "anaglyph");
	check_arg(SETL_SYSTEM argv, 1, ft_real, "real", "anaglyph");
	check_arg(SETL_SYSTEM argv, 2, ft_real, "real", "anaglyph");

	scale =argv[1].sp_val.sp_real_ptr->r_value;
	fraction =argv[2].sp_val.sp_real_ptr->r_value;

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	if (mimg->comp != 1)
		abend(SETL_SYSTEM "Image should have only one plane.");

	mimg = anaglyph_mimage(mimg, scale, fraction);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_SET_PIXEL(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int err = 1;
	int size;
	int x, y;
	void *values;
	mimage_ptr mimg;

	specifier return1;

TRACE_STUBS("GR_SET_PIXEL");

	check_gr_arg(SETL_SYSTEM argv, 0, "set_pixel");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "set_pixel");
	check_arg(SETL_SYSTEM argv, 2, ft_tuple, "tuple", "set_pixel");

	if (TUPLE_SIZE(argv[1]) != 2)
		abend(SETL_SYSTEM "Coordinates should be a tuple of two elements","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ((size = TUPLE_SIZE(argv[2])) != mimg->comp)
		abend(SETL_SYSTEM "Image depth and constant tuple size should be the same","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_short)
			if (i==0)
				x = ia_element->sp_val.sp_short_value;
			else
				y = ia_element->sp_val.sp_short_value;
		else	
			abend(SETL_SYSTEM "the tuple must be integer","tuple",1,
				"set_pixel", abend_opnd_str(SETL_SYSTEM argv+1));
		++i;
	}
	ITERATE_TUPLE_END(ia)

	if (mimg->use_count > MIN_REF) {

		mimg = clone_mimage(mimg);

		if (!mimg)
			abend(SETL_SYSTEM msg_malloc_error);

		mimg->use_count = REF_BASE;
	}
	
	values = malloc(size * ((mimg->kind == FLOAT_IMAGE)?sizeof(float):sizeof(unsigned char)));
	if (!values)
		goto abort;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[2])
	{
		if (mimg->kind == FLOAT_IMAGE) {
			if (ia_element->sp_form==ft_real)
				((float *)values)[i] = (ia_element->sp_val.sp_real_ptr->r_value);
			else {
				abend(SETL_SYSTEM "the tuple must be real","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+2));
			}
		} else {
			if (ia_element->sp_form==ft_short)
				((unsigned char *)values)[i] = (ia_element->sp_val.sp_short_value);
			else {
				abend(SETL_SYSTEM "the tuple must be integer","tuple",1,
					"const", abend_opnd_str(SETL_SYSTEM argv+2));
			}
		}

		++i;
	}
	ITERATE_TUPLE_END(ia)

	err = set_pixel_mimage(mimg, x, y, values);

	free(values);

	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		return;
	}

abort:
	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;
	return;
}

SETL_API void GR_GET_PIXEL(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int err;
	int size;
	int x, y;
	void *values;
	mimage_ptr mimg;

TRACE_STUBS("GR_GET_PIXEL");

	check_gr_arg(SETL_SYSTEM argv, 0, "get_pixel");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "get_pixel");

	if (TUPLE_SIZE(argv[1]) != 2)
		abend(SETL_SYSTEM "Coordinates should be a tuple of two elements","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+3));

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_short)
			if (i==0)
				x = ia_element->sp_val.sp_short_value;
			else
				y = ia_element->sp_val.sp_short_value;
		else	
			abend(SETL_SYSTEM "the tuple must be integer","tuple",1,
				"set_pixel", abend_opnd_str(SETL_SYSTEM argv+1));
		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	values = get_pixel_mimage(mimg, x, y);

	if (values) {
		if (mimg->kind == FLOAT_IMAGE)
			create_and_return_floats_tuple_from_array(SETL_SYSTEM values, mimg->comp, target);
		else
			create_and_return_shorts_tuple_from_array(SETL_SYSTEM values, mimg->comp, target);

		free(values);
	}	 else {
		unmark_specifier(target);
		target->sp_form = ft_omega;
	}
}

SETL_API void GR_CONNECTED_COMPONENTS(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_CONNECTED_COMPONENTS");

	check_gr_arg(SETL_SYSTEM argv, 0, "connected_components");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = connected_components_mimage(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_CONVERT_TO_SPARSE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg;
	int err;

	specifier return1;

	check_gr_arg(SETL_SYSTEM argv, 0, "convert_to_sparse");

TRACE_STUBS("GR_CONVERT_TO_SPARSE");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if (mimg->use_count > MIN_REF) {

		mimg = clone_mimage(mimg);

		if (!mimg)
			abend(SETL_SYSTEM msg_malloc_error);

		mimg->use_count = REF_BASE;
	}
	
	err = to_sparse_mimage(&mimg);

	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_CONVERT_TO_DENSE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg;
	int err;

	specifier return1;

TRACE_STUBS("GR_CONVERT_TO_DENSE");

	check_gr_arg(SETL_SYSTEM argv, 0, "convert_to_dense");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if (mimg->use_count > MIN_REF) {
		mimg = clone_mimage(mimg);
		if (!mimg)
			abend(SETL_SYSTEM msg_malloc_error);
		mimg->use_count = REF_BASE;
	}
	
	err = to_dense_mimage(&mimg);

	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_GET_LEVEL_DENSE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int size;
	float *values;
	mimage_ptr mimg;

TRACE_STUBS("GR_GET_LEVEL_DENSE");

	check_gr_arg(SETL_SYSTEM argv, 0, "get_level_dense");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "get_level_dense");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ((size = TUPLE_SIZE(argv[1])) != mimg->comp)
		abend(SETL_SYSTEM "Image depth and constant tuple size should be the same","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+1));

	values = malloc(size * sizeof(fmpixel));
	if (!values)
		goto abort;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_real)
			((float *)values)[i] = (ia_element->sp_val.sp_real_ptr->r_value);
		else if (ia_element->sp_form==ft_short)
			((float *)values)[i] = (ia_element->sp_val.sp_short_value);
		else {
			abend(SETL_SYSTEM "the tuple must be a number","tuple",1,
				"const", abend_opnd_str(SETL_SYSTEM argv+1));
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)

	mimg = get_level_dense_mimage(mimg, values);

	free(values);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
		return;
	}

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

SETL_API void GR_GET_NEAR_DENSE (
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int size;
	fmpixel_ptr values, dvalues;
	mimage_ptr mimg;

TRACE_STUBS("GR_GET_NEAR_DENSE");

	check_gr_arg(SETL_SYSTEM argv, 0, "get_near_dense");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "get_level_dense");
	check_arg(SETL_SYSTEM argv, 2, ft_tuple, "tuple", "get_level_dense");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if ((size = TUPLE_SIZE(argv[1])) != mimg->comp)
		abend(SETL_SYSTEM "Image depth and constant tuple size should be the same","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+1));

	if ((size = TUPLE_SIZE(argv[2])) != mimg->comp)
		abend(SETL_SYSTEM "Image depth and constant tuple size should be the same","",1,"",
			abend_opnd_str(SETL_SYSTEM argv+2));

	values = malloc(size * sizeof(fmpixel));
	if (!values)
		goto abort;

	dvalues = malloc(size * sizeof(fmpixel));
	if (!dvalues) {
		free(values);
		goto abort;
	}

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_real)
			values[i] = ia_element->sp_val.sp_real_ptr->r_value;
		else {
			abend(SETL_SYSTEM "the tuple must be real","tuple",1,
				"const", abend_opnd_str(SETL_SYSTEM argv+1));
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[2])
	{
		if (ia_element->sp_form==ft_real)
			dvalues[i] = ia_element->sp_val.sp_real_ptr->r_value;
		else {
			abend(SETL_SYSTEM "the tuple must be real","tuple",1,
				"const", abend_opnd_str(SETL_SYSTEM argv+2));
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)


	mimg = get_near_dense_mimage(mimg, values, dvalues);


	free(values);
	free(dvalues);

	if (mimg) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

		return;
	}

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;

}

SETL_API void GR_SHRINK_AND_CUT(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	mimage_ptr mimg, res_mimg;
	int rect_array[4];
	image_rect rect;
	int err;

	specifier return1;

TRACE_STUBS("GR_SHRINK_AND_CUT");

	check_gr_arg(SETL_SYSTEM argv, 0, "shrink_and_cut");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if (mimg->use_count > MIN_REF) {
		mimg = clone_mimage(mimg);
		if (!mimg)
			abend(SETL_SYSTEM msg_malloc_error);
		mimg->use_count = REF_BASE;
	}
	
	res_mimg = shrink_and_cut_mimage(mimg, &rect);

	if (!res_mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	destroy_mimage(mimg);

	rect_array[0] = rect.x;
	rect_array[1] = rect.y;
	rect_array[2] = rect.dx;
	rect_array[3] = rect.dy;

	return1.sp_form = ft_opaque;
	return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) res_mimg;
	push_pstack(&return1);

	create_and_return_integers_tuple_from_array(SETL_SYSTEM rect_array, 4, target);
}

SETL_API void GR_VALIDATE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int number;
	mimage_ptr mimg, *res_mimg;
	int rect_array[4];
	image_rect rect;
	int err;

	char int_str_pointer[25];

TRACE_STUBS("GR_VALIDATE");

	check_arg(SETL_SYSTEM argv, 0, ft_string, "string", "validate");

	setl2_string_to_cstring(argv[0].sp_val.sp_string_ptr, (unsigned char *)int_str_pointer, sizeof(int_str_pointer));

	mimg = (mimage_ptr)atol(int_str_pointer);
	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	mimg->type = gr_type;
	mimg->use_count = 1; /* ??? */

	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
}

SETL_API void GR_FLIP_H(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_FLIP_H");

	check_gr_arg(SETL_SYSTEM argv, 0, "flip_h");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = flip_h(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_FLIP_V(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;

TRACE_STUBS("GR_FLIP_V");

	check_gr_arg(SETL_SYSTEM argv, 0, "flip_v");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = flip_v(mimg);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_ROTATE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg;
	int times;

TRACE_STUBS("GR_ROTATE");

	check_gr_arg(SETL_SYSTEM argv, 0, "rotate");
	check_arg(SETL_SYSTEM argv, 1, ft_short, "integer", "rotate");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	times = (argv[1].sp_val.sp_short_value);

	mimg = rotate_halfpi(mimg, times);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
}

SETL_API void GR_SUPERPOSE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg = NULL;

TRACE_STUBS("GR_SUPERPOSE");

	check_gr_arg(SETL_SYSTEM argv, 0, "superpose");
	check_gr_arg(SETL_SYSTEM argv, 1, "superpose");

	mimg_a = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	mimg_b = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	if (mimg_a->density == SPARSE_IMAGE && mimg_b->density == SPARSE_IMAGE) {

		mimg = superpose_smimage(mimg_a, mimg_b);
	}

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_RANDOM(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	float r_min, r_max;
	mimage_ptr mimg;

TRACE_STUBS("GR_RANDOM");

	check_gr_arg(SETL_SYSTEM argv, 0, "random");
	check_arg(SETL_SYSTEM argv, 1, ft_real, "real", "random");
	check_arg(SETL_SYSTEM argv, 2, ft_real, "real", "random");

	r_min =argv[1].sp_val.sp_real_ptr->r_value;
	r_max =argv[2].sp_val.sp_real_ptr->r_value;

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	mimg = random_mimage(mimg, r_min, r_max);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_RECTANGLE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int first_kind = -1;
	mimage_ptr mimg;
	int width, height, kind;
	int thick;
	int planes_number;
	void *values_array;
	mpixel_ptr array;
	fmpixel_ptr farray;

TRACE_STUBS("GR_RECTANGLE");

	check_arg(SETL_SYSTEM argv, 0, ft_short, "integer", "rectangle");
	check_arg(SETL_SYSTEM argv, 1, ft_short, "integer", "rectangle");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "integer", "rectangle");
	check_arg(SETL_SYSTEM argv, 3, ft_short, "integer", "rectangle");
	check_arg(SETL_SYSTEM argv, 4, ft_tuple, "tuple", "rectangle");

	planes_number = TUPLE_SIZE(argv[4]);
	values_array = malloc(planes_number * sizeof(float));
	if (!values_array) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}
	array = (unsigned char *)farray = values_array;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[4])
	{
		if (first_kind == -1)
			first_kind = ia_element->sp_form;
		else
			if (ia_element->sp_form != first_kind)
				abend(SETL_SYSTEM "all values must have the same kind","ellipse",1,
					"self", abend_opnd_str(SETL_SYSTEM argv));

		if (ia_element->sp_form==ft_short)
			array[i] = (ia_element->sp_val.sp_short_value);
		else if (ia_element->sp_form == ft_real)
			farray[i] = (ia_element->sp_val.sp_real_ptr->r_value);

		++i;
	}
	ITERATE_TUPLE_END(ia)

	width			= (argv[0].sp_val.sp_short_value);
	height			= (argv[1].sp_val.sp_short_value);
	kind			= (argv[2].sp_val.sp_short_value);
	thick			= (argv[3].sp_val.sp_short_value);

	mimg = rectangle_mimage_sparse(width, height, thick, planes_number, kind, values_array);

	free(values_array);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_ELLIPSE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)

	int i;
	int first_kind = -1;
	mimage_ptr mimg;
	int width, height, kind;
	int thick;
	int planes_number;
	void *values_array;
	mpixel_ptr array;
	fmpixel_ptr farray;

TRACE_STUBS("GR_ELLIPSE");

	check_arg(SETL_SYSTEM argv, 0, ft_short, "integer", "ellipse");
	check_arg(SETL_SYSTEM argv, 1, ft_short, "integer", "ellipse");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "integer", "ellipse");
	check_arg(SETL_SYSTEM argv, 3, ft_short, "integer", "ellipse");
	check_arg(SETL_SYSTEM argv, 4, ft_tuple, "tuple", "ellipse");

	planes_number = TUPLE_SIZE(argv[4]);
	values_array = malloc(planes_number * sizeof(float));
	if (!values_array) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}
	array = (unsigned char *)farray = values_array;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[4])
		{
		if (first_kind == -1)
			first_kind = ia_element->sp_form;
		else
			if (ia_element->sp_form != first_kind)
				abend(SETL_SYSTEM "all values must have the same kind","ellipse",1,
					"self", abend_opnd_str(SETL_SYSTEM argv));

		if (ia_element->sp_form==ft_short)
			array[i] = (ia_element->sp_val.sp_short_value);
		else if (ia_element->sp_form == ft_real)
			farray[i] = (ia_element->sp_val.sp_real_ptr->r_value);

		++i;
	}
	ITERATE_TUPLE_END(ia)

	width			= (argv[0].sp_val.sp_short_value);
	height			= (argv[1].sp_val.sp_short_value);
	kind			= (argv[2].sp_val.sp_short_value);
	thick			= (argv[3].sp_val.sp_short_value);

	mimg = ellipse_mimage_sparse(width, height, thick, planes_number, kind, values_array);

	free(values_array);

	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;
}

SETL_API void GR_INVERT(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg;
	int err;

	specifier return1;

TRACE_STUBS("GR_INVERT");

	check_gr_arg(SETL_SYSTEM argv, 0, "invert");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if (mimg->use_count > MIN_REF) {
		mimg = clone_mimage(mimg);
		if (!mimg)
			abend(SETL_SYSTEM msg_malloc_error);
		mimg->use_count = REF_BASE;
	}
	
	err = invert_mimage(mimg);

	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_OFFSET(
	SETL_SYSTEM_PROTO
	int argc,						/* number of arguments passed		*/
	specifier *argv,				/* argument vector (two here)		*/
	specifier *target)				/* return value						*/
{
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg;
	int i;
	int disp_array[2];
	gr_image_displacement disp;
	int err;

	specifier return1;

TRACE_STUBS("GR_OFFSET");

	check_gr_arg(SETL_SYSTEM argv, 0, "offset");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "offset");

	if (IS_DENSE(mimg))
	{
		/* return the same image we passed */
		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		err = 1;	/* operation not allowed yet */

		goto abort_offset;
	}

	if (mimg->use_count > MIN_REF) {
		mimg = clone_mimage(mimg);
		if (!mimg)
			abend(SETL_SYSTEM msg_malloc_error);
		mimg->use_count = REF_BASE;
	}
	
	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[1])
	{
		if (ia_element->sp_form==ft_short)
			disp_array[i] = (ia_element->sp_val.sp_short_value);
		else 
			abend(SETL_SYSTEM "the displacement must be integer","offset",1,
				"offset", abend_opnd_str(SETL_SYSTEM argv));
		++i;
	}
	ITERATE_TUPLE_END(ia)

	disp.x  = disp_array[0];
	disp.y  = disp_array[1];

	err = offset_mimage_sparse(mimg, &disp);

	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

abort_offset:

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_SPLIT(
	SETL_SYSTEM_PROTO
	int argc,			/* number of arguments passed	*/
	specifier *argv,	/* argument vector (two here)	*/
	specifier *target)	/* return value					*/
{
	/*	Hierarchy:
	 *		+ cabc
	 *			- ca [images tuple]
	 * 			- cb [tuple of tuples]
	 *				- cbb [pair offset]
	 *			- cc [tuple of populations]
	 *
	 *	[[img1, img2, ..., imgn], [[offx1, offy1], [offx2, offy2], ...[offxn, offyn]], [pop1, pop2, ..., popn]]
	 */

	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(cc)
	TUPLE_CONSTRUCTOR(cb)
	TUPLE_CONSTRUCTOR(cabc)
	TUPLE_CONSTRUCTOR(cbb)

	specifier s;

	gr_image_displacement *offsets;
	int *populations;
	
	int i;
	int number;
	mimage_ptr mimg, *res_mimgs;
	int rect_array[4];
	image_rect rect;
	int err;

TRACE_STUBS("GR_SPLIT");

	check_gr_arg(SETL_SYSTEM argv, 0, "split");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	res_mimgs = split_mimage(mimg, &number, &offsets, &populations);
	
	if (!res_mimgs || number == 0) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

		/* images tuple construction */
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (i=0; i<number; i++) {

		s.sp_form = ft_opaque;
		s.sp_val.sp_opaque_ptr = (opaque_item_ptr_type)res_mimgs[i];

		TUPLE_ADD_CELL(ca,&s);
	}
	TUPLE_CONSTRUCTOR_END(ca);
	
		/* offsets tuple construction */
	TUPLE_CONSTRUCTOR_BEGIN(cb);
	for (i=0; i<number; i++) {
		TUPLE_CONSTRUCTOR_BEGIN(cbb);
		
		s.sp_form = ft_short;
		s.sp_val.sp_short_value = offsets[i].x;
		TUPLE_ADD_CELL(cbb,&s);

		s.sp_form = ft_short;
		s.sp_val.sp_short_value = offsets[i].y;
		TUPLE_ADD_CELL(cbb,&s);
		
		TUPLE_CONSTRUCTOR_END(cbb);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(cbb);
		TUPLE_ADD_CELL(cb, &s);
	}
	TUPLE_CONSTRUCTOR_END(cb);

		/* population tuple construction */
	TUPLE_CONSTRUCTOR_BEGIN(cc);
	for (i=0; i<number; i++) {

		s.sp_form = ft_short;
		s.sp_val.sp_short_value = populations[i];

		TUPLE_ADD_CELL(cc,&s);
	}
	TUPLE_CONSTRUCTOR_END(cc);

		/* container tuple construction */
	TUPLE_CONSTRUCTOR_BEGIN(cabc);

	s.sp_form = ft_tuple;
	s.sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
	TUPLE_ADD_CELL(cabc, &s);

	s.sp_form = ft_tuple;
	s.sp_val.sp_tuple_ptr = TUPLE_HEADER(cb);
	TUPLE_ADD_CELL(cabc, &s);

	s.sp_form = ft_tuple;
	s.sp_val.sp_tuple_ptr = TUPLE_HEADER(cc);
	TUPLE_ADD_CELL(cabc, &s);

	TUPLE_CONSTRUCTOR_END(cabc);

		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(cabc);

	SMART_FREE (res_mimgs);
	SMART_FREE (offsets);
	SMART_FREE (populations);
}

SETL_API void GR_MASK(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int cloned_b = 0;
	image_rect rect;
	int height_a, width_a;
	int height_b, width_b;
	
	mimage_ptr mimg_a, mimg_b;
	mimage_ptr mimg = NULL;

TRACE_STUBS("GR_MASK");

	check_gr_arg(SETL_SYSTEM argv, 0, "mask");
	check_gr_arg(SETL_SYSTEM argv, 1, "mask");

	mimg_a = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	mimg_b = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	if (!IS_DENSE(mimg_a))
		abend(SETL_SYSTEM "first image should be dense","mask",1,
			"offset", abend_opnd_str(SETL_SYSTEM argv));
	
	if (!IS_SPARSE(mimg_b)) 
		abend(SETL_SYSTEM "mask should be sparse","mask",2,
			"offset", abend_opnd_str(SETL_SYSTEM argv+1));


	get_mimage_info(mimg_a, &height_a, &width_a, NULL, NULL, NULL);
	get_mimage_info(mimg_b, &height_b, &width_b, NULL, NULL, NULL);

	rect.x = 0;
	rect.y = 0;
	rect.dx = min(width_a, width_b);
	rect.dy = min(height_a, height_b);

	if ((rect.dx < width_b) || (rect.dy < height_b)) {
		mimg_b = clone_mimage(mimg_b);
		if (!mimg_b) {
			unmark_specifier(target);
			target->sp_form = ft_omega;
			return;
		}

		cloned_b = 1;
		crop_mimage(mimg_b, &rect);
	}
		
	mimg = mask_dense_using_sparse(mimg_a, mimg_b);

	if (cloned_b)
		destroy_mimage(mimg_b);
	
	if (!mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)mimg;

}

SETL_API void GR_LUT(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_ITERATOR(ia)
	TUPLE_ITERATOR(iaa)

	int first_type = -1;	/* not assigned yet */

	int i, j, c;
	int res_planes_number;
	mpixel *LUT = NULL;
	mpixel *FLUT = NULL;
	mpixel **LUTs = NULL;
	mimage_ptr mimg;
	mimage_ptr target_mimg = NULL;

TRACE_STUBS("GR_LUT");

	check_gr_arg(SETL_SYSTEM argv, 0, "lut");
	check_arg(SETL_SYSTEM argv, 1, ft_tuple, "tuple", "lut");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
		
	if (GET_PLANES_NUMBER(mimg) != 1 || IS_FLOAT(mimg))
		abend(SETL_SYSTEM "Source image should be an integer 1 plane image",1,
			"lut", abend_opnd_str(SETL_SYSTEM argv));
		
	res_planes_number = TUPLE_SIZE(argv[1]);
	if (res_planes_number < 1)
		abend(SETL_SYSTEM "Uncorrect number of planes",1,
			"lut", abend_opnd_str(SETL_SYSTEM argv));
	
		/*
		 *	Allocate space for the Look Up Tables
		 */

		/* create the lut big enough for both the float and discrete case */
	LUT = (mpixel *)calloc(256*res_planes_number, max(sizeof(fmpixel),sizeof(mpixel)));
	LUTs = (mpixel **)calloc(res_planes_number, sizeof(mpixel *));
	if (!LUT || !LUTs) goto abort_LUT;

	for (c=0; c<res_planes_number ; c++)
		LUTs[c] = LUT + (c * 256);
	
	c=0;
	ITERATE_TUPLE_BEGIN(iaa, argv[1]) 
	{
			
		if (TUPLE_SIZE((*iaa_element)) > 256)
			abend(SETL_SYSTEM "too many values for LUT (max 256)",1,
				"lut", abend_opnd_str(SETL_SYSTEM argv));

		i=0;
		ITERATE_TUPLE_BEGIN(ia, (*iaa_element))
		{
			if (first_type == -1)
				first_type = ia_element->sp_form;
			else {
				if (first_type != ia_element->sp_form) 
					abend(SETL_SYSTEM "All the LUT tuple elements must be the same type",1,
						"lut", abend_opnd_str(SETL_SYSTEM argv));
			}
		
			if (ia_element->sp_form==ft_real) {

				FLUT = LUTs[c];
				FLUT[i] = (ia_element->sp_val.sp_real_ptr->r_value);

			} else if (ia_element->sp_form==ft_short) {

				LUTs[c][i] = (ia_element->sp_val.sp_short_value);

			} else
				abend(SETL_SYSTEM "the LUT tuple elements must be real or integers",1,
					"lut", abend_opnd_str(SETL_SYSTEM argv));
			++i;
		}
		ITERATE_TUPLE_END(ia)
	
		c++;

	} ITERATE_TUPLE_END(iaa)

	target_mimg = lut_mimage(mimg, LUTs, res_planes_number, (first_type == ft_short ? DISCRETE_IMAGE : FLOAT_IMAGE));

abort_LUT:

	SMART_FREE (LUT);
	SMART_FREE (LUTs);

	if (!target_mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)target_mimg;

}

SETL_API void GR_LEX_MIN(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_CONSTRUCTOR(ia)
	specifier s;
	
	int err;
	mimage_ptr mimg;
	gr_image_displacement pos;
	
TRACE_STUBS("GR_LEX_MIN");

	check_gr_arg(SETL_SYSTEM argv, 0, "lex_min");
	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	err = lex_minof_mimage(mimg, &pos);
		/* container tuple construction */
	if (err)
		goto abort;

	TUPLE_CONSTRUCTOR_BEGIN(ia);

	s.sp_form = ft_short;
	s.sp_val.sp_short_value = pos.x;
	TUPLE_ADD_CELL(ia, &s);

	s.sp_form = ft_short;
	s.sp_val.sp_short_value = pos.y;
	TUPLE_ADD_CELL(ia, &s);

	TUPLE_CONSTRUCTOR_END(ia);

		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ia);

	return;

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

SETL_API void GR_LEX_MAX(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	TUPLE_CONSTRUCTOR(ia)
	specifier s;
	
	int err;
	mimage_ptr mimg;
	gr_image_displacement pos;
	
TRACE_STUBS("GR_LEX_MAX");

	check_gr_arg(SETL_SYSTEM argv, 0, "lex_max");
	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	err = lex_maxof_mimage(mimg, &pos);
		/* container tuple construction */
	if (err)
		goto abort;
		
	TUPLE_CONSTRUCTOR_BEGIN(ia);

	s.sp_form = ft_short;
	s.sp_val.sp_short_value = pos.x;
	TUPLE_ADD_CELL(ia, &s);

	s.sp_form = ft_short;
	s.sp_val.sp_short_value = pos.y;
	TUPLE_ADD_CELL(ia, &s);

	TUPLE_CONSTRUCTOR_END(ia);

		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ia);
	
	return;

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

SETL_API void GR_LEX_SORT(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	mimage_ptr mimg;
	mimage_ptr res_mimg;

TRACE_STUBS("GR_LEX_SORT");

	check_gr_arg(SETL_SYSTEM argv, 0, "lex_max");
	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	res_mimg = lex_sort_mimage(mimg);

	if (!res_mimg) {
		unmark_specifier(target);
		target->sp_form = ft_omega;
	}
	
	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)res_mimg;	
}

SETL_API void GR_STUFF_IN_PLACE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/

{
	TUPLE_ITERATOR(ia)

	mimage_ptr mimg_a, mimg_b;
	int i;
	int delta_array[2];
	gr_image_displacement delta;
	int err;

	specifier return1;

TRACE_STUBS("GR_STUFF_IN_PLACE");

	check_gr_arg(SETL_SYSTEM argv, 0, "stuff_in_place");
	check_gr_arg(SETL_SYSTEM argv, 1, "stuff_in_place");

	mimg_a = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if (mimg_a->use_count > MIN_REF) {
		mimg_a = clone_mimage(mimg_a);
		if (!mimg_a)
			abend(SETL_SYSTEM msg_malloc_error);
		mimg_a->use_count = REF_BASE;
	}
	
	mimg_b = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	check_arg(SETL_SYSTEM argv, 2, ft_tuple, "tuple", "stuff_in_place");

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[2])
	{
		if (ia_element->sp_form==ft_short)

			delta_array[i] = (ia_element->sp_val.sp_short_value);

		else 

			abend(SETL_SYSTEM "the rectangle must be integer","rectangle",1,
				"stuff_in_place", abend_opnd_str(SETL_SYSTEM argv));

		++i;
	}
	ITERATE_TUPLE_END(ia)

	delta.x  = delta_array[0];
	delta.y  = delta_array[1];

	if (!(IS_DENSE(mimg_a) && IS_SPARSE(mimg_b)) && !(IS_DENSE(mimg_a) && IS_DENSE(mimg_b)))
		abend(SETL_SYSTEM "Uncorrect kind of images for stuff_in_place");

	if (IS_DENSE(mimg_a) && IS_SPARSE(mimg_b))
		err = stuff_sparse_over_dense(mimg_b, mimg_a, &delta);
	else 
		err = stuff_mimage(mimg_b, mimg_a, &delta);

	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg_a;
		push_pstack(&return1);

		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_FLATTEN(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/

{
	mimage_ptr mimg;
	int err;

	specifier return1;

TRACE_STUBS("GR_FLATTEN");

	check_gr_arg(SETL_SYSTEM argv, 0, "flatten");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);

	if (mimg->use_count > MIN_REF) {
		mimg = clone_mimage(mimg);
		if (!mimg)
			abend(SETL_SYSTEM msg_malloc_error);
		mimg->use_count = REF_BASE;
	}
	
	err = flatten_sparse_mimage(mimg);
	
	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		return1.sp_form = ft_opaque;
		return1.sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		push_pstack(&return1);

		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;

}

SETL_API void GR_CHECK(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/

{
	mimage_ptr mimg;
	
TRACE_STUBS("GR_CHECK");

	check_gr_arg(SETL_SYSTEM argv, 0, "gr_check");
	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	
	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = mimg->use_count;
}


SETL_API void GR_CHECK_MEM(
	SETL_SYSTEM_PROTO
	int argc,						/* number of arguments passed		*/
	specifier *argv,				/* argument vector (two here)		*/
	specifier *target)				/* return value					  	*/
{
	long mem = 1 << 24;	/* if this is not a mac always return some memory */
	
TRACE_STUBS("GR_CHECK_MEM");

#ifdef macintosh
	mem =  FreeMem();
#endif

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = mem;
}

SETL_API void GR_TILE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/
{
	int i;
	TUPLE_ITERATOR(ia)

	mimage_ptr *mimages, res_mimage;
	size_t n;
	int h, w;

TRACE_STUBS("GR_TILE");

	check_arg(SETL_SYSTEM argv, 0, ft_tuple, "tuple", "tile");
	check_arg(SETL_SYSTEM argv, 1, ft_short, "integer", "tile");
	check_arg(SETL_SYSTEM argv, 2, ft_short, "integer", "tile");

	n = TUPLE_SIZE(argv[0]);

	mimages = malloc(n*sizeof(mimage_ptr));
	if (!mimages)
		goto abort_tile;

	i=0;
	ITERATE_TUPLE_BEGIN(ia, argv[0])
	{
		if ((ia_element->sp_form != ft_opaque) ||
			(((ia_element->sp_val.sp_opaque_ptr->type) & 65535 ) != gr_type)) {

			abend(SETL_SYSTEM "tuple of images expected","tile",0,
				"tile", abend_opnd_str(SETL_SYSTEM argv));

		} else {
			mimages[i] = (mimage_ptr)ia_element->sp_val.sp_opaque_ptr;
		}
		++i;
	}
	ITERATE_TUPLE_END(ia)
	
	h = (argv[1].sp_val.sp_short_value);
	w = (argv[2].sp_val.sp_short_value);

	res_mimage = tile_mimages(mimages, n, h, w);

	free(mimages);

	if (res_mimage) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)res_mimage;
		return;
	}

abort_tile:

	unmark_specifier(target);
	target->sp_form = ft_omega;
	return;

}

SETL_API void GR_FFT(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/

{
	TUPLE_CONSTRUCTOR(ca)
	specifier s;

	mimage_ptr res_mimg_re, res_mimg_im;
	mimage_ptr mimg;
	int err;

TRACE_STUBS("GR_FFT");

	check_gr_arg(SETL_SYSTEM argv, 0, "fft");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	
	err = fft_mimage(mimg, &res_mimg_re, &res_mimg_im);
	
	if (!err) {

		TUPLE_CONSTRUCTOR_BEGIN(ca);

		s.sp_form = ft_opaque;
		s.sp_val.sp_opaque_ptr = (opaque_item_ptr_type)res_mimg_re;
		TUPLE_ADD_CELL(ca, &s);

		s.sp_form = ft_opaque;
		s.sp_val.sp_opaque_ptr = (opaque_item_ptr_type)res_mimg_im;
		TUPLE_ADD_CELL(ca, &s);

		TUPLE_CONSTRUCTOR_END(ca);

			/* return everything */
		unmark_specifier(target);
		target->sp_form = ft_tuple;
		target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);

		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;
}

SETL_API void GR_FFT_INVERSE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/

{
	mimage_ptr mimg;
	mimage_ptr mimg_re, mimg_im;
	int err;

TRACE_STUBS("GR_FFT_INVERSE");

	check_gr_arg(SETL_SYSTEM argv, 0, "fft_inverse");
	check_gr_arg(SETL_SYSTEM argv, 1, "fft_inverse");

	mimg_re = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	mimg_im = (mimage_ptr)(argv[1].sp_val.sp_opaque_ptr);

	err = fft_inverse_mimage(mimg_re, mimg_im, &mimg);
	
	if (!err) {

		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type) mimg;
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;
}

SETL_API void GR_WAVELET(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/

{
	mimage_ptr mimg, res_mimg;
	int err;

TRACE_STUBS("GR_WAVELET");

	check_gr_arg(SETL_SYSTEM argv, 0, "wavelet");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	
	err = wavelet_mimage(mimg, &res_mimg, 1);
	
	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type) res_mimg;

		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;
}

SETL_API void GR_WAVELET_INVERSE(
	SETL_SYSTEM_PROTO
	int argc,							/* number of arguments passed	*/
	specifier *argv,					/* argument vector (two here)	*/
	specifier *target)				  	/* return value					*/

{
	mimage_ptr mimg, res_mimg;
	int err;

TRACE_STUBS("GR_WAVELET_INVERSE");

	check_gr_arg(SETL_SYSTEM argv, 0, "wavelet inverse");

	mimg = (mimage_ptr)(argv[0].sp_val.sp_opaque_ptr);
	
	err = wavelet_mimage(mimg, &res_mimg, -1);
	
	if (!err) {
		unmark_specifier(target);
		target->sp_form = ft_opaque;
		target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type) res_mimg;

		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = err;
}

