/**************************************************************************************
/* Filename:	gr_sparse_bitplns.c
/*		Copyright  1998-99 Giuseppe Di Mauro. All rights reserved.
/*
/* Description:	general graphical routines for dense byteplanes images
/*
/***************************************************************************************
/*
/*	History:
/*
/*	11/16/98 first release
/*	11/17/98 changed code in get_level_dense_mimage to use a dynamic table where needed
/*	11/18/98 changed format for sparse images
/*	12/04/98 code fix for first two (to_sparse_mimage, to_dense_mimage) sparse image fn
/*	12/08/98 some sparse_image are starting to work now, also float_image versions are
/*           starting to appear
/* 
/*	02/09/99 binary operations in sparse image coded (basically intersection)
/*	02/17/99 started coding for mixed sparse and dense operation
/*	02/26/99 written set_planes_mimage_dense routine
/*	04/06/99 code for macintosh debugging added
/*
/***************************************************************************************/
 
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <assert.h>

#include "gr_ops.h"
#include "gr_utils.h"
#include "gr_errors.h"
#include "gr_bitplns.h"
#include "gr_sparse_bitplns.h"
#include "gr_sparse_utils.h"

	/*
	 *	Not exported functions
	 */

static void		flip_mem(mpixel_ptr buffer, size_t size);
static void		flip_fmem(fmpixel_ptr buffer, size_t size);
static void		swap(mpixel *a, mpixel *b);
static void		fswap(fmpixel *a, fmpixel *b);
static int		ellipse_by_y(float y, float a, float b);
static int		look_for_line_in_rows(mimage_ptr mimg, int start_row_data_idx, int start_line, int line_to_find, int *new_sect_idx);
static int		look_for_pixel_in_sects(section_data *sects, int sect_idx, int sects_no, int pixel_x, short want_first);
static int		append_pixel_to_buffer(mimage_ptr mimg, void *pixel_value_array);
static void		write_pixel_at_buffer_offset(mimage_ptr mimg, int offset, void *value);

	/* 
	 *	Implementation 
	 */

/**************************************************************************************
/*	Function:		create_sparse_byteplane
/*	Description:	create a sparse byteplane
/*
/*	Parameters:
/*		<-	width	width of requested plane
/*		<-	height	height of requested plane
/*		<-	kind	kind (float or integer) of requested plane
/*
/*	Result:
/*		not NULL	pointer to the requested plane
/*		NULL		out of memory
/*
/***************************************************************************************/

void *create_sparse_byteplane(int width, int height, char kind)
{
	sparse_byteplane_ptr plane_ptr;

	size_t type_size = GET_TYPE_SIZE(kind);	/* get pixel size */

		/* allocate the plane */
	plane_ptr = (sparse_byteplane_ptr) calloc(sizeof(sparse_byteplane), 1);
	if (!plane_ptr)
		return NULL;

	plane_ptr->height = height;	/* set the height */
	plane_ptr->width = width;	/* set the width */

		/* only difference with dense byteplanes, the buffer starts to NULL */
	plane_ptr->buffer	= NULL; 
		/* and the default value is set */
	if (kind == FLOAT_IMAGE) 
		plane_ptr->default_value = FLOAT_BLACK;		/* set default to black */
	else
		plane_ptr->default_value = DISCRETE_BLACK;	/* set default to black */
	
	return plane_ptr;	/* return the pointer to the allocated plane */
}

/**************************************************************************************
/*	Function:		destroy_sparse_byteplane
/*	Description:	destroy a sparse byteplane and release its memory
/*
/*	Parameters:
/*		<-	plane	pointer to the sparse plane
/*		<-	kind	kind (float or integer) of plane
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void destroy_sparse_byteplane(sparse_byteplane_ptr byteplane, char kind)
{
#ifdef __MWERKS__
#pragma unused(kind)
#endif

	if (byteplane) {	/* if a valid byteplane */
		if (byteplane->buffer)	/* if a valid buffer */
			free(byteplane->buffer);	/* release memory for the buffer */
		free(byteplane);	/* release memory for the plane */
	}
}

/**************************************************************************************
/*	Function:		to_sparse_mimage
/*	Description:	Convert a dense image to sparse
/*
/*	Parameters:
/*		<->	img_ptr	image to be converted to sparse
/*
/*	Result:
/*		0			ok [image may not be changed if already sparse]
/*		else		error condition
/*
/***************************************************************************************/

#ifndef OLD_TO_SPARSE_IMAGE
int to_sparse_mimage(mimage_ptr *mimg_ptr)
{
	int c, j;
	int width, height, planes_number, kind;
	size_t memory_per_plane;
	mpixel_ptr  buffer;
	fmpixel_ptr fbuffer;
	sparse_byteplane_ptr *planes_array;
	mimage_ptr sparse_mimg;
	mimage_ptr mimg = *mimg_ptr;

#ifdef __MWERKS__
#pragma unused(fbuffer)
#endif
	
	if (mimg->density == SPARSE_IMAGE)
		return 0; /* image is already sparse and need not to be processed */

		/* get information from the source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	memory_per_plane = height * width * GET_TYPE_SIZE(kind);	

		/*
		 * Allocate memory for new sparse planes
		 * NOTE: allocation for sparse byteplanes creates 
		 * empty (not completely allocated) planes. 
		 */

		/* create the sparse image container */
	sparse_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
	if (!sparse_mimg)
		return -1;

		/* pre-setup (do an allocation) of the sparse structures */
	if (pre_setup_sparse_mimage(sparse_mimg, width, height, kind, planes_number) != 0) {
		destroy_mimage(sparse_mimg);
		return -2;
	}
		
		/* keep the value of the created image planes array */
	planes_array = sparse_mimg->planes_array;

		/* initialize the buffers, copying the 
		 * data from the source image to the destination image */
	for (c=0; c<planes_number; c++) {
		buffer = get_plane_buffer(mimg->planes_array[c], kind);
		memcpy(planes_array[c]->buffer, buffer, memory_per_plane);
	}
	
	sparse_mimg->data_length	= height;	/* set data length to the number of rows */
	sparse_mimg->section_length	= height;	/* set section length to the number of rows */

		/* fill each row_start slot with the index of the line */
	for (j=0; j<height; j++)
		sparse_mimg->row_start[j] = j;		

		/* fill each row_section slot with info to keep a whole image line */
	for (j=0; j<height; j++) {
			/* line start from left */
		sparse_mimg->row_section[j].starting_pixel_number = 0;
			/* number of pixels is the same of the width */
		sparse_mimg->row_section[j].number_of_pixels      = width;
			/* position in buffer is the offset of the line in the buffer */
		sparse_mimg->row_section[j].position_in_buffer    = j*width;
	}
	

	sparse_mimg->buffer_length = width * height;	/* set the size of the buffers */
	trim_sparse(sparse_mimg);						/* release extra used memory in the sparse image */

		/* remove extra data used for old image */
	for (c=0; c<mimg->comp; c++)
		destroy_byteplane(mimg->planes_array[c], mimg->kind, mimg->density);
	free(mimg->planes_array);	/* remove planes array (102799/GDM) */

		/* copy new fields in the new image */
	mimg->kind				= sparse_mimg->kind;
	mimg->comp				= sparse_mimg->comp;
	mimg->planes_array		= sparse_mimg->planes_array;
	mimg->density			= sparse_mimg->density;
	mimg->upperleft_x		= sparse_mimg->upperleft_x;
	mimg->upperleft_y		= sparse_mimg->upperleft_y;
	mimg->data_length		= sparse_mimg->data_length;
	mimg->section_length	= sparse_mimg->section_length;
	mimg->row_start			= sparse_mimg->row_start;
	mimg->row_section		= sparse_mimg->row_section;
	mimg->buffer_length		= sparse_mimg->buffer_length;

		/* release the memory used from the sparse image shell 
		 * NOTE: this could be done without any extra memory;
		 *       at this point is safer working in this way 
		 */
	free(sparse_mimg);			
	
	return 0;		/* ok */
}
#else
int to_sparse_mimage(mimage_ptr *mimg_ptr)
{
	int c, j;
	int width, height, planes_number, kind;
	size_t memory_per_plane;
	mpixel_ptr  buffer;
	fmpixel_ptr fbuffer;
	sparse_byteplane_ptr *planes_array;
	mimage_ptr sparse_mimg;
	mimage_ptr mimg = *mimg_ptr;

#ifdef __MWERKS__
#pragma unused(fbuffer)
#endif
	
	if (mimg->density == SPARSE_IMAGE)
		return 0; /* image is already sparse and need not to be processed */

		/* get information from the source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	memory_per_plane = height * width * GET_TYPE_SIZE(kind);	

		/*
		 * Allocate memory for new sparse planes
		 * NOTE: allocation for sparse byteplanes creates 
		 * empty (not completely allocated) planes. 
		 */

		/* create the sparse image container */
	sparse_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
	if (!sparse_mimg)
		return -1;

		/* pre-setup (do an allocation) of the sparse structures */
	if (pre_setup_sparse_mimage(sparse_mimg, width, height, kind, planes_number) != 0) {
		destroy_mimage(sparse_mimg);
		return -2;
	}
		
		/* keep the value of the created image planes array */
	planes_array = sparse_mimg->planes_array;

		/* initialize the buffers, copying the 
		 * data from the source image to the destination image */
	for (c=0; c<planes_number; c++) {
		buffer = get_plane_buffer(mimg->planes_array[c], kind);
		memcpy(planes_array[c]->buffer, buffer, memory_per_plane);
	}
	
	sparse_mimg->data_length	= height;	/* set data length to the number of rows */
	sparse_mimg->section_length	= height;	/* set section length to the number of rows */

		/* fill each row_start slot with the index of the line */
	for (j=0; j<height; j++)
		sparse_mimg->row_start[j] = j;		

		/* fill each row_section slot with info to keep a whole image line */
	for (j=0; j<height; j++) {
			/* line start from left */
		sparse_mimg->row_section[j].starting_pixel_number = 0;
			/* number of pixels is the same of the width */
		sparse_mimg->row_section[j].number_of_pixels      = width;
			/* position in buffer is the offset of the line in the buffer */
		sparse_mimg->row_section[j].position_in_buffer    = j*width;
	}
	
	destroy_mimage(mimg);							/* destroy the start image */
	sparse_mimg->buffer_length = width * height;	/* set the size of the buffers */
	trim_sparse(sparse_mimg);						/* release extra used memory */
	*mimg_ptr = sparse_mimg;						/* set the return image to the new one */

	return 0;		/* ok */
}
#endif /* OLD_TO_SPARSE_IMAGE */

/**************************************************************************************
/*	Function:		to_dense_mimage
/*	Description:	Convert a sparse image to dense
/*
/*	Parameters:
/*		<->	mimg_ptr	image to be converted to dense
/*
/*	Result:
/*		0			ok [image may not be changed if already dense]
/*		else		error condition
/*
/***************************************************************************************/

int to_dense_mimage(mimage_ptr *mimg_ptr)
{
	int i,j,c,k;
	mimage_ptr dense_mimg;
	size_t type_size;
	size_t line_index;
	mpixel_ptr buffer, buffer_ptr;
	size_t upperleft_x, upperleft_y;
	sparse_byteplane_ptr *planes_array;
	sparse_byteplane_ptr splane;
	section_data row_section;
	int num_sections_for_this_row, end_section_for_this_row;
	int width, height, planes_number, kind;
	mimage_ptr mimg = *mimg_ptr;
	
	if (mimg->density == DENSE_IMAGE)
		return 0; /* image is already dense and need not to be processed */
	
		/* get information from the source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
		/* get pixel size */	
	type_size = GET_TYPE_SIZE(kind);

		/* create the dense image */	
	dense_mimg = create_mimage(width, height, planes_number, kind, DENSE_IMAGE);
	if (!dense_mimg)
		return -1;

	upperleft_x = mimg->upperleft_x;			/* get the position of the sparse image conrcer in the dense image */
	upperleft_y = mimg->upperleft_y;

	line_index = upperleft_y;					/* start at the corner of the sparse image */

	planes_array = dense_mimg->planes_array;	/* get the dense image planes array */

				/* initialize planes to black */
	for (c=0; c<planes_number; c++) {
		buffer = planes_array[c]->buffer;
		memset(buffer, 0, height * width * GET_TYPE_SIZE(kind));	
	}

		/* if the sparse image is not empty */
	if (mimg->data_length) {

		for (k=0; k<mimg->data_length; k++) {	/* iterate over all the sparse image data entries */

			j = mimg->row_start[k];				/* get the number of rows to be skipped, or the start of the section data for the next row */
			if (j<0) {							/* we skip this number of rows */
				line_index -= j;
				continue;
			}

				/* get the number of sections for this row */
			num_sections_for_this_row = sections_in_row(mimg, k);
				/* get the last section index for this row */
			end_section_for_this_row = num_sections_for_this_row + j;

				/* loop through the sections for this row */
			for(; j<end_section_for_this_row; j++) {

				row_section = mimg->row_section[j];	/* pointer to the start of the section data for this row */

				for (c=0; c<planes_number; c++) {	/* iterate over all the image planes */

					buffer = planes_array[c]->buffer;				/* get pointer to output plane */
					splane = get_byteplane_from_mimage(mimg, c);	/* get pointer to the sparse image plane */

						/* determine the position to which data will be written */
					buffer_ptr = buffer + (line_index * width * type_size) 
						+ (row_section.starting_pixel_number + upperleft_x) * type_size;

					if (row_section.position_in_buffer >= 0) {		/* if there's real data in the buffer copy it to the output */
						memcpy(buffer_ptr,
							((char *)splane->buffer) 
								+ (row_section.position_in_buffer * type_size), 
							row_section.number_of_pixels * type_size);

					} else {										/* copy the default value to the proper part of the output */
					
						if (kind == FLOAT_IMAGE) 
							for (i=0; i<row_section.number_of_pixels; i++) 
								((fmpixel_ptr)buffer_ptr)[i] = ((fbyteplane_ptr)splane)->default_value;
						else
							memset(buffer_ptr, splane->default_value, 
								row_section.number_of_pixels * type_size);
					}
				}
			}
			line_index++;	/* advance to the next sparse image data entry */
		}
	}

	destroy_mimage(mimg);	/* destroy sparse image */
	*mimg_ptr = dense_mimg; /* set new image to be returned */
	
	return 0;	
}

/**************************************************************************************
/*	Function:		get_near_dense_mimage
/*	Description:	This routine returns a sparse image consisting
/*					of all pixels whose value p satisfies 
/*					abs(p - fx) < dfx (in every component)
/* 
/*	Parameters:
/*		<-	mimg_ptr	source image
/*		<-	fx			values array for center value (size = number_of_planes)
/*		<-	dfx			distance array from the value (size = number_of_planes)
/*
/*	Result:
/*		not NULL	pointer to the near values sparse image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr get_near_dense_mimage(mimage_ptr mimg, fmpixel *fx, fmpixel *dfx)
{

	int i,j,c;
	size_t row_section_size;
	size_t row_section_index;
	size_t row_start_index;
	mimage_ptr res_mimg;
	sparse_byteplane_ptr *planes_array;
	int width, height, planes_number, kind;
	short row_is_empty;
	mpixel_ptr buffer, sbuffer;
	fmpixel_ptr fbuffer, sfbuffer;

	size_t number_of_pixels_written = 0;	/* number of written pixels in the output buffer */
	size_t sparse_buffer_index = 0;
	int num_empty = 0;					/* number of completely empty rows in current run */
	int length_of_this_section = 0; 	/* number of this pixels in this section */

	/*
     * Iterate over all the rows of the dense source image,
     * collecting pixels which are identical to the value fx
     * supplied.
     * As such pixels are found they are copied to the mpixel
     * buffers for the sparse image.
     * Each continuous section in a row generates one entry
     * in the section data.
     * If a run of rows contains no such mpixel we count the 
     * number n of such rows and write -n to the row_start
     * array.
     * The row start array may grow beyond its initial length
     * and need to be doubled.
     */

	if (mimg->density == SPARSE_IMAGE)
		return NULL; /* image is sparse and so is not a proper argument */

		/* get information from the dense source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	
		/* allocate enough memory for new sparse planes */
	res_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
	if (!res_mimg)
		return NULL;
	
		/* keep a pointer to the array of planes */
	planes_array = res_mimg->planes_array;	
	
		/* 
		   pre-initialize the sparse image.
		   NOTE: this is allocating sparse memory for a complete image
		   data will need to be reallocated at the end to reduce the 
		   sparse image to what is really needed
		 */

		/* initialize indices use in loop below */
	row_start_index		= 0;	/* index to next position to write in the row_start array */
	row_section_index	= 0;	/* index to next position to write in the row_section array */

	row_section_size	= height;	/* limiting size of the row section array; this may expand as
									   the algorithm proceeds */

		/* do the guess-timate allocation */
	if (pre_setup_sparse_mimage(res_mimg, width, height, kind, planes_number) != 0)
		return NULL;

		/* the value saught defines the default value in every plane */
	if (kind == FLOAT_IMAGE)
		for (c=0; c<planes_number; c++)
			((fbyteplane_ptr *)planes_array)[c]->default_value = (fmpixel)fx[c];
	else
		for (c=0; c<planes_number; c++)
			planes_array[c]->default_value = (mpixel)fx[c];


	if (kind == FLOAT_IMAGE) {	/* if the image is float */

		for (j=0; j<height; j++) {		/* iterate over successive rows */

				/* assume initially that the row is empty */
			row_is_empty = TRUE;

			for (i=0; i<width; i++) {	/* iterate over pixels in row */

					/* 
					 * test the mpixel at this position to
					 * see if has the correct value
					 */

				short pixel_is_good = TRUE;

				for (c=0; c<planes_number; c++) {	/* iterate over planes */
					
					fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
					fbuffer += j * width + i;

						/* check if the pixel is "near" enough */
					if (abs(*fbuffer - fx[c]) > dfx[c]) {
						pixel_is_good = FALSE;
						break;
					}
				}
				
				if (pixel_is_good) {	/* if the pixel is "near" */
				
						/* copy the pixel in the buffer */
					for (c=0; c<planes_number; c++) {
						fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
		 				fbuffer += j * width + i;

						sfbuffer = get_fbyteplane_from_mimage(res_mimg, c)->buffer;
						sfbuffer += number_of_pixels_written;
						*sfbuffer = *fbuffer;
					}

					number_of_pixels_written++;	/* update the position in the buffer */
									
					if (length_of_this_section > 0)	{	/* continue the current section */

						length_of_this_section++;	/* increase the size of this section */

					} else {	/* start a new section */

							/* 
							 * if there are any accumulated empty rows, they must be
							 * written to the section_data array
							 */

						if (num_empty > 0) {
							res_mimg->row_start[row_start_index++] = -num_empty;
							num_empty = 0;
						}

						if (row_is_empty) {
							row_is_empty = FALSE;
							res_mimg->row_start[row_start_index++] = row_section_index;
						}

						length_of_this_section = 1;	/* start a new section in any case */

					}
					
				} else { /* mpixel is not good */
					
					if (length_of_this_section > 0) { /* this bad mpixel ends a run */

							/* write the section data */
						write_to_section_data(res_mimg, row_section_index, &row_section_size, 
							i-length_of_this_section, length_of_this_section, number_of_pixels_written - length_of_this_section);

						length_of_this_section = 0;	/* reset the section length */
						
						row_section_index ++;	/* note that one more section has been written */
						
					}
				}

			} /* end for i (iteration of row) */

			if (length_of_this_section > 0) { /* end of row always ends a section */
				
					/* write the section data */
				write_to_section_data(res_mimg, row_section_index, &row_section_size, 
					i-length_of_this_section, length_of_this_section, number_of_pixels_written - length_of_this_section);
				
				length_of_this_section = 0;	/* reset the section length */

				row_section_index ++;	/* note that one more section has been written */
			}

			if (row_is_empty)	/* keep track of the number of empty rows */
				num_empty++;

		} /* end for j (iteration of rows) */
	
	} else {	/* if image is discrete */

		for (j=0; j<height; j++) {		/* iterate over successive rows */

				/* assume initially that the row is empty */
			row_is_empty = TRUE;

			for (i=0; i<width; i++) {	/* iterate over pixels in row */

					/* 
					 * test the mpixel at this position to
					 * see if has the correct value
					 */

				short pixel_is_good = TRUE;

					/* check if the pixel is "near" enough */
				for (c=0; c<planes_number; c++) {
					buffer = get_byteplane_from_mimage(mimg, c)->buffer;
					buffer += j * width + i;

					if (abs((fmpixel)*buffer - fx[c]) > dfx[c]) {
						pixel_is_good = FALSE;
						break;
					}
				}
				
				if (pixel_is_good) {	/* if the pixel is "near" */
				
						/* copy the pixel in the buffer */
					for (c=0; c<planes_number; c++) {
						buffer = get_byteplane_from_mimage(mimg, c)->buffer;
		 				buffer += j * width + i;

						sbuffer = get_byteplane_from_mimage(res_mimg, c)->buffer;
						sbuffer += number_of_pixels_written;
						*sbuffer = *buffer;
					}

					number_of_pixels_written++;	/* update the position in the buffer */
									
					if (length_of_this_section > 0)	{	/* continue the current section */

						length_of_this_section++;	/* increase the size of this section */

					} else {	/* start a new section */

							/* 
							 * if there are any accumulated empty rows, they must be
							 * written to the section_data array
							 */

						if (num_empty > 0) {
							res_mimg->row_start[row_start_index++] = -num_empty;
							num_empty = 0;
						}

						if (row_is_empty) {
							row_is_empty = FALSE;
							res_mimg->row_start[row_start_index++] = row_section_index;
						}

						length_of_this_section = 1;	/* start a new section in any case */

					}
					
				} else { /* mpixel is not good */
					
					if (length_of_this_section > 0) { /* this bad mpixel ends a run */

						write_to_section_data(res_mimg, row_section_index, &row_section_size, 
							i-length_of_this_section, length_of_this_section, number_of_pixels_written - length_of_this_section);

						length_of_this_section = 0;	/* reset the section length */
						
						row_section_index ++;	/* note that one more section has been written */
						
					}
				}

			} /* end for i (iteration of row) */

			if (length_of_this_section > 0) { /* end of row always ends a section */
				
				write_to_section_data(res_mimg, row_section_index, &row_section_size, 
					i-length_of_this_section, length_of_this_section, number_of_pixels_written - length_of_this_section);
				
				length_of_this_section = 0;	/* reset the section length */

				row_section_index ++;	/* note that one more section has been written */
			}

			if (row_is_empty)	/* keep track of the number of empty rows */
				num_empty++;

		} /* end for j (iteration of rows) */
	}
	
	if (num_empty > 0)		/* if there are empty rows at the end, they must be recorded */
		res_mimg->row_start[row_start_index++] = -num_empty;
	
		/* set the buffer sizes to thei real values */
	res_mimg->data_length = row_start_index;
	res_mimg->section_length = row_section_index;
	res_mimg->buffer_length = number_of_pixels_written;

	trim_sparse(res_mimg);		/* trim extra allocated memory */

	return res_mimg;	/* return a pointer to the result image */
}

/**************************************************************************************
/*	Function:		get_level_dense_mimage
/*	Description:	This routine returns a sparse image consisting
/*					of all pixels whose value p satisfies p=x
/* 
/*	Parameters:
/*		<-	mimg_ptr	source image
/*		<-	fx			values array for level value (size = number_of_planes)
/*
/*	Result:
/*		not NULL	pointer to the generated sparse image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr get_level_dense_mimage(mimage_ptr mimg, fmpixel *fx)
{
	int i,j,c;
	size_t row_section_size;
	size_t row_section_index;
	size_t row_start_index;
	mimage_ptr res_mimg;
	sparse_byteplane_ptr *planes_array;
	int width, height, planes_number, kind;
	short row_is_empty;
	mpixel_ptr buffer;
	fmpixel_ptr fbuffer;
	
	size_t sparse_buffer_index = 0;
	int num_empty = 0;					/* number of completely empty rows in current run */
	int length_of_this_section = 0; 	/* number of this pixels in this section */

	/*
     * Iterate over all the rows of the dense source image,
     * collecting pixels which are identical to the value fx
     * supplied.
     * As such pixels are found they are copied to the mpixel
     * buffers for the sparse image.
     * Each continuous section in a row generates one entry
     * in the section data.
     * If a run of rows contains no such mpixel we count the 
     * number n of such rows and write -n to the row_start
     * array.
     * The row start array may grow beyond its initial length
     * and need to be doubled.
     */

	if (mimg->density == SPARSE_IMAGE)
		return NULL; /* image is sparse and so is not a proper argument */

		/* get information from the dense source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	
		/* allocate enough memory for new sparse planes */
	res_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
	if (!res_mimg)
		return NULL;
	
		/* keep a pointer to the array of planes for latter use */
	planes_array = res_mimg->planes_array;	
	
		/* 
		   pre-initialize the sparse image.
		   NOTE: this is allocating sparse memory for a complete image
		   data will need to be reallocated at the end to reduce the 
		   sparse image to what is really needed
		 */

		/* initialize indices use in loop below */
	row_start_index		= 0;	/* index to next position to write in the row_start array */
	row_section_index	= 0;	/* index to next position to write in the row_section array */

	row_section_size	= height;	/* limiting size of the row section array; this may expand as
									   the algorithm proceeds */

		/* do a guess-timate of the allocation size for the buffers in the sparse image */
	if (pre_setup_sparse_mimage_no_buffer(res_mimg, width, height) != 0)
		return NULL;

		/* the value saught defines the default value in every plane */
	if (mimg->kind == FLOAT_IMAGE) {
		for (c=0; c<planes_number; c++)
			((fbyteplane_ptr *)planes_array)[c]->default_value = (fmpixel)fx[c];
	} else {
		for (c=0; c<planes_number; c++)
			planes_array[c]->default_value = (mpixel)fx[c];
	}
	
	if (mimg->kind == FLOAT_IMAGE) {	/* if float image */
		
		for (j=0; j<height; j++) {		/* iterate over successive rows */

				/* assume initially that the row is empty */
			row_is_empty = TRUE;

			for (i=0; i<width; i++) {	/* iterate over pixels in row */

					/* 
					 * test the mpixel at this position to
					 * see if has the correct value
					 */

				short pixel_is_good = TRUE;

				for (c=0; c<planes_number; c++) {
					fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
					fbuffer += j * width + i;
					
					if (*fbuffer != fx[c]) {
						pixel_is_good = FALSE;
						break;
					}
				}
				
				if (pixel_is_good) {
				
					if (length_of_this_section > 0)	{	/* continue the current section */
						length_of_this_section ++;
					} else {/* start a new section */

							/* 
							 * if there are any accumulated empty rows, they must be
							 * written to the section_data array
							 */

						if (num_empty > 0) {
							res_mimg->row_start[row_start_index++] = -num_empty;
							num_empty = 0;
						}

						if (row_is_empty) {
							row_is_empty = FALSE;
							res_mimg->row_start[row_start_index++] = row_section_index;
						}

						length_of_this_section = 1;	/* start a new section in any case */

					}
					
				} else { /* mpixel is not good */
					
					if (length_of_this_section > 0) { /* this bad mpixel ends a run */

						write_to_section_data(res_mimg, row_section_index, &row_section_size, 
							i-length_of_this_section, length_of_this_section, -1);

						length_of_this_section = 0;
						
						row_section_index ++;	/* note that one more section has been written */
						
					}
				}

			} /* end for i (iteration of row) */

			if (length_of_this_section > 0) { /* end of row always ends a section */
				
				write_to_section_data(res_mimg, row_section_index, &row_section_size, 
					i-length_of_this_section, length_of_this_section, -1);
				
				length_of_this_section = 0;

				row_section_index ++;	/* note that one more section has been written */
			}

			if (row_is_empty)	/* keep track of the number of empty rows */
				num_empty++;

		} /* end for j (iteration of rows) */
	} else {	/* if discrete image */
		
		for (j=0; j<height; j++) {		/* iterate over successive rows */

				/* assume initially that the row is empty */
			row_is_empty = TRUE;

			for (i=0; i<width; i++) {	/* iterate over pixels in row */

					/* 
					 * test the mpixel at this position to
					 * see if has the correct value
					 */

				short pixel_is_good = TRUE;

				for (c=0; c<planes_number; c++) {
					buffer = get_byteplane_from_mimage(mimg, c)->buffer;
					buffer += j * width + i;
					
					if (*buffer != (mpixel)fx[c]) {
						pixel_is_good = FALSE;
						break;
					}
				}
				
				if (pixel_is_good) {
				
					if (length_of_this_section > 0)	{	/* continue the current section */
						length_of_this_section ++;
					} else {/* start a new section */

							/* 
							 * if there are any accumulated empty rows, they must be
							 * written to the section_data array
							 */

						if (num_empty > 0) {
							res_mimg->row_start[row_start_index++] = -num_empty;
							num_empty = 0;
						}

						if (row_is_empty) {
							row_is_empty = FALSE;
							res_mimg->row_start[row_start_index++] = row_section_index;
						}

						length_of_this_section = 1;	/* start a new section in any case */

					}
					
				} else { /* mpixel is not good */
					
					if (length_of_this_section > 0) { /* this bad mpixel ends a run */

						write_to_section_data(res_mimg, row_section_index, &row_section_size, 
							i-length_of_this_section, length_of_this_section, -1);

						length_of_this_section = 0;
						
						row_section_index ++;	/* note that one more section has been written */
						
					}
				}

			} /* end for i (iteration of row) */

			if (length_of_this_section > 0) { /* end of row always ends a section */
				
				write_to_section_data(res_mimg, row_section_index, &row_section_size, 
					i-length_of_this_section, length_of_this_section, -1);
				
				length_of_this_section = 0;

				row_section_index ++;	/* note that one more section has been written */
			}

			if (row_is_empty)	/* keep track of the number of empty rows */
				num_empty++;

		} /* end for j (iteration of rows) */
	}
	
	if (num_empty > 0)		/* if there are empty rows at the end, they must be recorded */
		res_mimg->row_start[row_start_index++] = -num_empty;
	
	res_mimg->data_length = row_start_index;
	res_mimg->section_length = row_section_index;
	res_mimg->buffer_length = 0;	/* this routine never writes to the buffer */
	
	trim_sparse(res_mimg);	/* trim extra allocated memory */

	return res_mimg;
}

/**************************************************************************************
/*	Function:		split_dense_mimage
/*	Description:	This routine splits a dense image in to
/*					a vector of sparse images. All pixels of 
/*					identical value are grouped into one sparse image
/* 
/*	Parameters:
/*		<-	mimg		source image
/*		->	ret_number	number of result images
/*		->	offsets		array of displacement, containing the offset of the
/*						sparse images relative to the source image
/*		->	pixels_pop	return an array of integer containing the number
/*						of pixels in each image
/*
/*	Result:
/*		not NULL	pointer to the generated sparse images array
/*		NULL		error condition	[usually out of memory]
/*
/***************************************************************************************/

mimage_ptr *split_dense_mimage(mimage_ptr mimg, int *retNumber, image_displacement_ptr *offsets, int **pixels_pop)
{
	int i, c, k;									/* miscellaneous indices */

	int min_row_number;								/* first row of the current split image */
	int max_row_number;								/* last row of the current split image */
	int min_col_number;								/* first column of the current split image */
	int max_col_number;								/* last column of the current split image */

	int level_width, level_height;

	int height,width,planes_number,kind;			/* info about dense image */
	size_t num_pixels;								/* total number of pixels */
	int num_pixel_values;							/* number of different mpixel values which appear in the dense image */
	short same_as_prior;							/* is this mpixel the same ast the prior mpixel in sorted order */
	int onp, onpp;
	int np,	npp, jp, ip, jpp, ipp;					/* various mpixel and row indices */

	int row_difference;

	mpixel_ptr buffer, bufferp, bufferpp;			/* pointers to current and prior mpixel elements in buffers */
	fmpixel_ptr fbuffer, fbufferp, fbufferpp;		/* as above but for float images */

	int *row_start;
	section_data_ptr row_section;
	
	size_t first_pixel_this_section;				/* number of the first mpixel in the current section  */
	
	int *pixel_no_array = NULL;						/* array of mpixel indices */
	mimage_ptr *the_sparse_images = NULL;			/* collection of sparse images, to be returned */
	mimage_ptr res_mimg = NULL;
	mimage_ptr sparse_mimg = NULL;
	image_displacement_ptr	the_offsets = NULL;		/* offsets to top left corner of sparse fragment 
													   from top left of original image */

	size_t num_row_entries_this_pixval, num_section_entries_this_pixval;
	
	size_t  number_of_values_seen = 0;				/* number of distinct mpixel values seen  */
	int num_omitted = 0;							/* number of rows omitted at the end of an image */
	size_t first_section_this_row;					/* number of first_section_this_row in given row */ 

#ifndef OLD_CODE2
	int *pixels_counter = NULL;						/* count the pixels in a sparse image */
#endif
										/* number of row and section entries for a mpixel value */

/* 
 *  Each mpixel of our input image has a mpixel number n, namely its position in the
 *	dense image's mpixel buffer. We begin by sorting the array of these mpixel numbers
 *	into the alphabetical order of the corresponding mpixel values. Pixel numbers
 *	with identical mpixel values are sorted into the alphabetical order of their
 *	row, column position.
 */

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);			/* get info about dense image */
	
	pixel_no_array = generate_pixel_no_array(num_pixels = (height * width));
	if (!pixel_no_array)
		goto abort_split;

	sort_by_pvals_and_posns(pixel_no_array, mimg);				/* sort in the manner described above */

/*
 * Next we iterate several times over the sorted pixel_no_array, determining the sizes of various
 * other data objects which must be allocated. Our first pass determines the number of different mpixel
 * values which appear in the dense image 
 */
	
	num_pixel_values = 1;									/* these will be counted */

	if (kind == FLOAT_IMAGE) 
	{
		for (k = 1;k < num_pixels;k++) {						/* iteration over all pixels */

			np = pixel_no_array[k];								/* number of this mpixel */

			onpp = pixel_no_array[k - 1];						/* number of prior mpixel */
			npp = (onpp < 0) ? -onpp - 1 : onpp;				/* this correction is necessary since may have changed */

			jp = np/width; ip = np - jp * width;					/* row and column number of this mpixel */
			jpp = npp/width; ipp = npp - jpp * width;					/* row and column number of prior mpixel */

			same_as_prior = TRUE;

			for (c = 0;c < planes_number;c++) {					/* look for difference with prior mpixel */
				fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
				fbufferpp = fbuffer + npp;
				fbufferp = fbuffer + np;

				if ((fmpixel) *fbufferpp != (fmpixel) *fbufferp) 
				{
					same_as_prior = FALSE; 
					break;
				}
			}

			if (!same_as_prior) {
				num_pixel_values++;				/* count up mpixel value if it differs from prior */
				pixel_no_array[k] = -pixel_no_array[k] - 1;
			} 
		}						/* end of first iteration over all pixels */
	}
	else	/* DISCRETE_IMAGE */
	{
		for (k = 1;k < num_pixels;k++) {						/* iteration over all pixels */

			np = pixel_no_array[k];								/* number of this mpixel */

			onpp = pixel_no_array[k - 1];						/* number of prior mpixel */
			npp = (onpp < 0) ? -onpp - 1 : onpp;				/* this correction is necessary since may have changed */

			jp = np/width; ip = np - jp * width;					/* row and column number of this mpixel */
			jpp = npp/width; ipp = npp - jpp * width;					/* row and column number of prior mpixel */

			same_as_prior = TRUE;

			for (c = 0;c < planes_number;c++) {					/* look for difference with prior mpixel */
				buffer = get_byteplane_from_mimage(mimg, c)->buffer;
				bufferpp = buffer + npp;
				bufferp = buffer + np;

				if ((mpixel) *bufferpp != (mpixel) *bufferp) 
				{
					same_as_prior = FALSE; 
					break;
				}
			}

			if (!same_as_prior) {
				num_pixel_values++;				/* count up mpixel value if it differs from prior */
				pixel_no_array[k] = -pixel_no_array[k] - 1;
			} 
		}						/* end of first iteration over all pixels */
	}

/* Now that we know the number of different mpixel values which appear in the image, we can allocate 
/* and initialize the number of empty sparse images needed */

	the_sparse_images = calloc(num_pixel_values, sizeof(mimage_ptr));
	if (!the_sparse_images) 
		goto abort_split;

	the_offsets = malloc(num_pixel_values * sizeof(gr_image_displacement));
	if (!the_offsets) 
		goto abort_split;

#ifndef OLD_CODE2
	pixels_counter = (int *)calloc(sizeof(*pixels_counter), num_pixel_values);
	if (!pixels_counter)
		goto abort_split;
#endif	

/* next we iterate a second time over all the pixels, counting the number of row array entries and section
/* entries needed for each mpixel value. As soon as these are known, we allocate the correxponding sparse image */
			
	num_row_entries_this_pixval = 0;						/* number of row entries for this mpixel value */
	num_section_entries_this_pixval = 0;					/* number of section entries for this mpixel value */

	np = pixel_no_array[0];									/* number of first mpixel */
	jp = np/width; ip = np - jp * width;					/* row and column number of first mpixel */
	
	min_row_number = max_row_number = jp;					/* both initial row numbers are the row number of the first mpixel */
	min_col_number = max_col_number = ip;					/* both initial column numbers are the column number of the first mpixel */

#ifdef OLD_CODE
	if (jp > 0)			/* an initial row entry will be needed if the first sparse image omits the first row */
		 num_row_entries_this_pixval = 1;
#endif

	for (k = 1;k < num_pixels;k++) {						/* second, allocation, loop over all pixels */
	
		onp = pixel_no_array[k];							/* number of this mpixel */
		np = (onp < 0) ? -onp - 1 : onp;

		onpp = pixel_no_array[k - 1];						/* number of prior mpixel */
		npp = (onpp < 0) ? -onpp - 1 : onpp;
		
		jp = np/width; ip = np - jp * width;				/* row and column number of this mpixel */
		jpp = npp/width; ipp = npp - jpp * width;			/* row and column number of prior mpixel */
		
		if (onp < 0) {								/* we have encountered a new mpixel value */
			num_row_entries_this_pixval++;	
					/* allow for the row entry representing the final nonempy row of the prior image */
			
#ifdef OLD_CODE
					/* we must count up one more row_entry if the prior entry does not belong to the final row */ 
			if (jpp < height - 1) num_row_entries_this_pixval++; /* allow for the final section of the prior image */
#endif
			num_section_entries_this_pixval++;

			level_width = max_col_number - min_col_number + 1;		/* get the width and height of the section to be created */
			level_height = max_row_number - min_row_number + 1; 

			the_offsets[number_of_values_seen].x = min_col_number;	/* catch the min row and max row information for this mpixel number */
			the_offsets[number_of_values_seen].y = min_row_number;

			min_row_number = max_row_number = jp;		/* initialize the min row and max row information for the next mpixel level */
			min_col_number = max_col_number = ip;

			the_sparse_images[number_of_values_seen++] = sparse_mimg = 			/* allocate the preceding sparse image */
#ifdef OLD_CODE
				create_mimage(width, height, planes_number, kind, SPARSE_IMAGE); 
#else
				create_mimage(level_width, level_height, planes_number, kind, SPARSE_IMAGE);
#endif /* OLD_CODE */
			if (!sparse_mimg) {
				goto abort_split;
			}
			
			if (pre_setup_sparse_mimage_no_buffer_allocated(
					sparse_mimg, 
					num_section_entries_this_pixval, 
					num_row_entries_this_pixval) != 0) {
				
				goto abort_split;
			}
			
					/* otherwise we can put the mpixel value as default into the new sparse_mimg */
			for (c=0; c<planes_number; c++) {
				if (kind == FLOAT_IMAGE) 
				{
					fbyteplane_ptr thePlane = get_fbyteplane_from_mimage(sparse_mimg, c);
					thePlane->default_value = *((fmpixel_ptr)get_plane_buffer(get_fbyteplane_from_mimage(mimg, c), kind) + npp);
				}
				else	/* DISCRETE_IMAGE */
				{
					byteplane_ptr thePlane = get_byteplane_from_mimage(sparse_mimg, c);
					thePlane->default_value = *((mpixel_ptr)get_plane_buffer(get_byteplane_from_mimage(mimg, c), kind) + npp);
				}
			}

			num_section_entries_this_pixval = num_row_entries_this_pixval = 0; 
									/* zero these out to prepare for the next mpixel value */
#ifdef OLD_CODE
			if (jp > 0)				/* allow for one row entry if the next image starts with a blank row */
				++ num_row_entries_this_pixval;
#endif
			
		} else {	/* we have the same mpixel value as the prior mpixel*/
			
			/* we must count up one row_entry if there is row change from the prior mpixel,
			/* but two if there is a row jump */ 

			num_row_entries_this_pixval += (jpp < jp - 1 ? 2 : (jpp < jp ? 1 : 0));

			max_row_number = max(max_row_number, jp);				/* keep the maximum and the minimum up to date */
			min_row_number = min(min_row_number, jp);

			max_col_number = max(max_col_number, ip);				/* keep the maximum and the minimum up to date */
			min_col_number = min(min_col_number, ip);
	
							/* we must count up one section_entry if there is either a row change */
							/* or a mpixel jump  from the prior mpixel */
			if (jpp < jp || ipp < ip - 1)
					num_section_entries_this_pixval++;

		}		/* end if  !same_as_prior */
	
	}						/* end of allocation loop */
	
	onp = pixel_no_array[num_pixels-1];								/* number of final mpixel */
	np = (onp < 0) ? -onp - 1 : onp; 
	jp = np/width;													/* row of final mpixel */
				
				/* we always write one row entry, representing the row containing the final mpixel */
				/* if the final mpixel is not in the last row, we write one more row entry, representing */
				/* the final skipped rows */
#ifdef OLD_CODE
	num_row_entries_this_pixval += (jp < height - 1 ? 2 : 1);
#else
	num_row_entries_this_pixval++;
#endif	
		/* we must count up one final section_entry */
	num_section_entries_this_pixval++;
	
	level_width = max_col_number - min_col_number + 1;		/* get the width and height of the section to be created */
	level_height = max_row_number - min_row_number + 1; 

	the_offsets[number_of_values_seen].x = min_col_number;	/* catch the min row and max row information for this mpixel number */
	the_offsets[number_of_values_seen].y = min_row_number;

	min_row_number = max_row_number = jp;		/* initialize the min row and max row information for the next mpixel level */
	min_col_number = max_col_number = ip;

	the_sparse_images[number_of_values_seen] = sparse_mimg = 			/* allocate the final sparse image */
#ifdef OLD_CODE
		create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
#else
		create_mimage(level_width, level_height, planes_number, kind, SPARSE_IMAGE);
#endif

	if (!sparse_mimg) {
		goto abort_split;
	}
	
	if (pre_setup_sparse_mimage_no_buffer_allocated(
			sparse_mimg, num_section_entries_this_pixval,
				num_row_entries_this_pixval) != 0) {
		goto abort_split;
	}
		
		/* otherwise we can put the mpixel value as defalult into the final sparse_mimg */

	for (c=0; c<planes_number; c++) {
		if (kind == FLOAT_IMAGE)
		{
			fbyteplane_ptr thePlane = get_fbyteplane_from_mimage(sparse_mimg, c);
			thePlane->default_value = *((fmpixel_ptr)get_plane_buffer(get_fbyteplane_from_mimage(mimg, c),kind) + np);
		}
		else
		{
			byteplane_ptr thePlane = get_byteplane_from_mimage(sparse_mimg, c);
			thePlane->default_value = *((mpixel_ptr)get_plane_buffer(get_byteplane_from_mimage(mimg, c),kind) + np);
		}	
	}
	
/* Having now performed exactly the allocations required, we loop one final time over the pixels,
/* filling in the row and section data*/
	
	sparse_mimg = the_sparse_images[0];
	num_pixel_values = 0;
	
	row_start = sparse_mimg->row_start;
	row_section = sparse_mimg->row_section;
			
	num_row_entries_this_pixval = 0;						/* number of row entries for this mpixel value */
	num_section_entries_this_pixval = 0;					/* number of section entries for this mpixel value */
	first_section_this_row = 0;								/* first section number of first row to be written */
	
	onp = pixel_no_array[0];									/* number of first mpixel */
	np = (onp < 0) ? -onp - 1 : onp;
		
	jp = np/width; ip = np - jp * width;					/* row and column number of first mpixel */
	
	first_pixel_this_section = ip;							/* number of the first mpixel of the first section */	
	
#ifdef OLD_CODE
	if (jp > 0)			/* an initial row entry will be needed if the first sparse image omits the first row */
		sparse_mimg->row_start[num_row_entries_this_pixval++] = -jp;
#endif
	
	for (k = 1;k < num_pixels;k++) {						/* data fill-in loop over all pixels */

		onp = pixel_no_array[k];								/* number of this mpixel */
		np = (onp < 0) ? -onp - 1 : onp;

		onpp = pixel_no_array[k - 1];						/* number of prior mpixel */
		npp = (onpp < 0) ? -onpp - 1 : onpp;

		jp = np/width; ip = np - jp * width;					/* row and column number of this mpixel */
		jpp = npp/width; ipp = npp - jpp * width;					/* row and column number of prior mpixel */
		
		if (onp < 0) {								/* we have encountered a new mpixel value */

			sparse_mimg->row_start[num_row_entries_this_pixval++] = first_section_this_row;
					/* write the row entry representing the last nonempty row of the prior mpixel value */

					/* we write one more row_entry, representing several skipped final rows, */
					/* of the prior image, if the prior entry does not belong to the final row 	*/ 
#ifdef OLD_CODE
			if ((num_omitted = height - 1 - jpp) > 0) 
					sparse_mimg->row_start[num_row_entries_this_pixval++] = -num_omitted;
#endif
					/* we also write one section_entry, representing the section just ended */
				sparse_mimg->row_section[num_section_entries_this_pixval].starting_pixel_number = 
																			first_pixel_this_section;
#ifndef OLD_CODE2
				pixels_counter[num_pixel_values] += 
#endif
				sparse_mimg->row_section[num_section_entries_this_pixval].number_of_pixels = 
																		ipp - first_pixel_this_section + 1;
				sparse_mimg->row_section[num_section_entries_this_pixval++].position_in_buffer = -1;
											/* in all these images we use the default value */

			num_section_entries_this_pixval = num_row_entries_this_pixval = 0; 
									/* zero these out to prepare for the next mpixel value */
			
						/* now we move along to the next image   */
			sparse_mimg = the_sparse_images[++num_pixel_values];
			first_section_this_row = 0;	/* first section number of the first row of the next image to be written */
			first_pixel_this_section = ip;	/* note the starting mpixel of this section */		
			
					/* we write one row_entry, representing several skipped initial rows, if */
					/* the first entry for the new mpixel value does not belong to the first row */ 
#ifdef OLD_CODE
			if (jp > 0)
					sparse_mimg->row_start[num_row_entries_this_pixval++] = -jp;
#endif			
		} else {	/* we have the same mpixel value as the prior mpixel*/

			/* we must write a row_entry representing a row of sections if there is row change */
			/* from the prior mpixel, but two (the first representing this row of sections, the */
			/* second representing the skipped rows, if there is a row jump */ 
			
			row_difference = jp - jpp;
			
			if (row_difference > 1) {		/* row jump, so write two row entries */
					sparse_mimg->row_start[num_row_entries_this_pixval++] = first_section_this_row;
					sparse_mimg->row_start[num_row_entries_this_pixval++] = -(row_difference - 1);
					first_section_this_row = num_section_entries_this_pixval + 1;
			} else {
				if (row_difference > 0) { /* row change: write a row entry representing a row of sections */
					sparse_mimg->row_start[num_row_entries_this_pixval++] = first_section_this_row;
					first_section_this_row = num_section_entries_this_pixval + 1;
				}
			}
	
					/* we must also write one section  entry, representing the section just ended */ 
					/* if there is either a row change or a mpixel jump  from the prior mpixel */
			if (jpp < jp || ipp < ip - 1) {

				sparse_mimg->row_section[num_section_entries_this_pixval].starting_pixel_number = 
																			first_pixel_this_section;
#ifndef OLD_CODE2
				pixels_counter[num_pixel_values] += 
#endif
				sparse_mimg->row_section[num_section_entries_this_pixval].number_of_pixels = 
																		ipp - first_pixel_this_section + 1;
				sparse_mimg->row_section[num_section_entries_this_pixval++].position_in_buffer = -1;
											/* in all these images we use the default value */
											
				first_pixel_this_section = ip; /* note the first mpixel of the new section */
			}
			
		}		/* end if  !same_as_prior */
	
	}						/* end of final loop */

	onp = pixel_no_array[num_pixels-1];								/* number of final mpixel */
	np = (onp < 0) ? -onp - 1 : onp; 
	jp = np/width; ip = np - jp * width;							/* row and column of final mpixel */
				
				/* we always write one row entry, representing the row containing the final mpixel */
				/* if the final mpixel is not in the last row, we write one more row entry, representing */
				/* the final skipped rows */
	sparse_mimg->row_start[num_row_entries_this_pixval++] = first_section_this_row;

#ifdef OLD_CODE
	if ((num_omitted = height - 1 - jp) > 0) 
			sparse_mimg->row_start[num_row_entries_this_pixval++] = -num_omitted;
#endif
	
										/* we must also  write the final section_entry */
	sparse_mimg->row_section[num_section_entries_this_pixval].starting_pixel_number = 
																			first_pixel_this_section;
#ifndef OLD_CODE2
	pixels_counter[num_pixel_values] += 
#endif
	sparse_mimg->row_section[num_section_entries_this_pixval].number_of_pixels = 
																ip - first_pixel_this_section + 1;
	sparse_mimg->row_section[num_section_entries_this_pixval++].position_in_buffer = -1;
											/* in all these images we use the default value */
	*retNumber = num_pixel_values+1;	/* return the number of sparse images into which the original image was split */

	SMART_FREE(pixel_no_array);

#ifndef OLD_CODE
	for (i=0; i<*retNumber; i++) {
		mimage_ptr img = the_sparse_images[i];
		section_data_ptr sects = img->row_section;
		int dx = the_offsets[i].x;
		
		for (k=0; k<img->section_length; k++)
			sects[k].starting_pixel_number -= dx;
			
	}
#endif

		/* if the offsets are requested */
	if (offsets != NULL)
		*offsets = the_offsets;	/* return them */
	else
		SMART_FREE(the_offsets); /* if not release the memory used for them */

		/* if the pixels population is requested */
	if (pixels_pop != NULL)
		*pixels_pop = pixels_counter;	/* return them */
	else
		SMART_FREE(pixels_counter); /* if not release the memory used for them */
		
	return the_sparse_images;		/* and return the list of these sparse images */
	

abort_split:

	SMART_FREE(pixels_counter);
	SMART_FREE(the_offsets);
	SMART_FREE(pixel_no_array);
	
	if (the_sparse_images) {
		int ck;
		for (ck=0 ; ck<num_pixel_values ; ck++)
			if (the_sparse_images[ck])
				destroy_mimage(the_sparse_images[ck]);
		free(the_sparse_images);
	}

	*retNumber = 0;
	
	if (offsets != NULL)
		*offsets = NULL;
	
	return NULL;			/* return the vector of sparse images that has been constructed */
}

/**************************************************************************************
/*	Function:		with_mimage
/*	Description:	This routine returns a dense image constructed
/*					by inserting all the pixels of the sparse image
/*					mimg_b into mimg_a (the inserted pixels overwrite
/*					the pixels of mimg_a)
/* 
/*	Parameters:
/*		<-	mimg_a		first source image
/*		->	mimg_b		second source image
/*
/*	Result:
/*		not NULL	pointer to the generated dense image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr with_mimage(mimage_ptr mimg_a, mimage_ptr mimg_b)
{
	int j,c,k;
	size_t type_size;
	size_t line_index;
	section_data row_section;
	mpixel_ptr buffer, buffer_ptr;
	sparse_byteplane_ptr splane;
	mimage_ptr res_mimg;
	int width_a, height_a, planes_number_a, kind_a;
	int width_b, height_b, planes_number_b, kind_b;
	int width, height, planes_number, kind;

	size_t upperleft_x = 0, upperleft_y = 0;	/* not supported anymore */
	
		/* check if both the image are sparse */
	if ((mimg_a->density != SPARSE_IMAGE) || (mimg_b->density != SPARSE_IMAGE))
		return NULL;

		/* get information from the source images */
	get_mimage_info(mimg_a, &height_a, &width_a, &planes_number_a, &kind_a, NULL);
	get_mimage_info(mimg_b, &height_b, &width_b, &planes_number_b, &kind_b, NULL);
	
		/* check if the images have the same size, kind and number of planes */
	if ( (planes_number_a != planes_number_b)
			|| (height_a != height_b) 
			|| (width_a != width_b)
			|| (kind_a != kind_b))
 		return NULL;
	
		/* set the specs for the result image */
	height = height_a;
	width = width_a;
	planes_number = planes_number_a;
	kind = kind_a;

		/* clone the first image and use this as the result image */
	res_mimg = clone_mimage(mimg_a);
	if (!res_mimg)
		return NULL;
	
		/* convert the result image to dense */
	if (!to_dense_mimage(&res_mimg)) {
		destroy_mimage(res_mimg);
		return NULL;
	}
	
		/* get the pixel size */
	type_size = GET_TYPE_SIZE(kind);	

	line_index = upperleft_y;	/* line_index will start from 0 because upperleft_y is zero */
	
		/* interate over the planes */
	for (c=0; c< planes_number; c++) {
		
			/* get the buffer for the result image */
		buffer = get_byteplane_from_mimage(res_mimg, c)->buffer;
			/* get the planes array for the second image */
		splane = get_byteplane_from_mimage(mimg_b, c);

			/* not used, will be zero */
		upperleft_x = mimg_b->upperleft_x;
		upperleft_y = mimg_b->upperleft_y;
		
			/* if the image row_start array is not empty */
		if (mimg_b->data_length) {
		
				/* for every line */
			for (k=0; k<mimg_b->data_length; k++) {
			
					/* get the line index in row */
				j = mimg_b->row_start[k];
				if (j<0) {	/* if skipping lines */
					line_index += -j;	/* collect them */
					continue;
				}
				
					/* get the first row_section for this line */
				row_section = mimg_b->row_section[j];
				
					/* get the pointer for the buffer */
				buffer_ptr = buffer	+ (line_index * width * type_size) 
					+ (row_section.starting_pixel_number + upperleft_x) * type_size;

					/* and copy the data */
				memcpy(buffer_ptr, ((mpixel_ptr)splane->buffer) + row_section.position_in_buffer, 
								row_section.number_of_pixels * type_size);
								
				line_index++;	/* increase the number of lines */

			} /* for k */
		}
	} /* for c */
	
	return res_mimg;	
}

/**************************************************************************************
/*	Function:		unary_op_smimage
/*	Description:	perform an unary operation over all pixels of a sparse image
/*
/*	Parameters:
/*		<-	mimg			operand sparse image
/*		<-	gray_op			pointer to an unary integer operation function for integer case
/*		<-	float_op		pointer to an unary float operation function for float case
/*
/*	Result:
/*		not NULL	the pointer to the result new sparse image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr unary_op_smimage(mimage_ptr mimg,  mpixel (*gray_op)(mpixel), fmpixel (*float_op)(fmpixel))
{
	int c,i;
	mpixel_ptr buffer, res_buffer;
	fmpixel_ptr fbuffer, fres_buffer;
	sparse_byteplane_ptr splane, ressplane;
	sparse_fbyteplane_ptr fsplane, fressplane;
	mimage_ptr res_mimg = clone_mimage(mimg);

	int width, height, planes_number, kind;
	
	if (res_mimg) {
		get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
		
		if (kind == FLOAT_IMAGE) {
			for (c=0; c<planes_number; c++) {
				fsplane = get_fbyteplane_from_mimage(mimg, c);
				fbuffer = fsplane->buffer;

				fressplane = get_fbyteplane_from_mimage(res_mimg, c);
				fressplane->default_value = float_op(fsplane->default_value);
				
				fres_buffer = get_fbyteplane_from_mimage(res_mimg, c)->buffer;
					
				for (i=0; i<mimg->buffer_length; i++) {
					*fres_buffer = float_op(*fbuffer);
					
					fres_buffer++;
					fbuffer++;
				}
			}
		} else { /* DISCRETE_IMAGE */
			for (c=0; c<planes_number; c++) {
				splane = get_byteplane_from_mimage(mimg, c);
				buffer = splane->buffer;
					
				ressplane = get_byteplane_from_mimage(res_mimg, c);
				ressplane->default_value = gray_op(splane->default_value);

				res_buffer = get_byteplane_from_mimage(res_mimg, c)->buffer;

				for (i=0; i<mimg->buffer_length; i++) {
					*res_buffer = gray_op(*buffer);

					res_buffer++;
					buffer++;
				}
			}
		}
	}
	
	return res_mimg;
}

/**************************************************************************************
/*	Function:		binary_op_constant_smimage
/*	Description:	perform a binary operation between 
/*					a sparse image and a constant value
/*
/*	Parameters:
/*		<-	mimg			sparse image
/*		<-	values_array	constant data pixel array
/*		<-	gray_op			pointer to a binary operation function for integer case
/*		<-	float_op		pointer to a binary operation function for float case
/*
/*	Result:
/*		not NULL	the pointer to the result new sparse image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr binary_op_constant_smimage(mimage_ptr mimg, void *values_array, unsigned char (*gray_op)(unsigned char, unsigned char), float (*float_op)(float, float))
{
	int c,i;
	mpixel_ptr buffer, res_buffer;
	fmpixel_ptr fbuffer, fres_buffer;
	sparse_byteplane_ptr splane, ressplane;
	sparse_fbyteplane_ptr fsplane, fressplane;
	mimage_ptr res_mimg = clone_mimage(mimg);

	int width, height, planes_number, kind;
	
	if (res_mimg) {
		get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
		
		if (kind == FLOAT_IMAGE) {
			for (c=0; c<planes_number; c++) {
				fsplane = get_fbyteplane_from_mimage(mimg, c);
				fbuffer = fsplane->buffer;

				fressplane = get_fbyteplane_from_mimage(res_mimg, c);
				fressplane->default_value = float_op(fsplane->default_value, *(((fmpixel_ptr)values_array)+c));
				
				fres_buffer = get_fbyteplane_from_mimage(res_mimg, c)->buffer;
					
				for (i=0; i<mimg->buffer_length; i++) {
					*fres_buffer = float_op(*fbuffer, *(((fmpixel_ptr)values_array)+c));
					
					fres_buffer++;
					fbuffer++;
				}
			}
		} else { /* DISCRETE_IMAGE */
			for (c=0; c<planes_number; c++) {
				splane = get_byteplane_from_mimage(mimg, c);
				buffer = splane->buffer;
				
				ressplane = get_byteplane_from_mimage(res_mimg, c);
				ressplane->default_value = float_op(splane->default_value, *(((mpixel_ptr)values_array)+c));
									
				res_buffer = get_byteplane_from_mimage(res_mimg, c)->buffer;

				for (i=0; i<mimg->buffer_length; i++) {
					*res_buffer = gray_op(*buffer, *(((mpixel_ptr)values_array)+c));

					res_buffer++;
					buffer++;
				}
			}
		}
	}
	
	return res_mimg;
}

/**************************************************************************************
/*	Function:		unimath_smimage
/*	Description:	perform an "user defined" unary operation to a sparse image
/*
/*	Parameters:
/*		<-	user_ptr		[private] used from the setl2 stubs
/*		<-	user_ptr2		[private] used from the setl2 stubs
/*		<-	mimg			operand dense image
/*		<-	floating_op		pointer to an external unary float operation
/*
/*	Result:
/*		not NULL	the pointer to the result new sparse image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr unimath_smimage(void *user_ptr, void *user_ptr2, mimage_ptr mimg, float (*floating_op)(void *user_ptr, void *user_ptr2, float, int))
{
	int c,i;
	mpixel_ptr buffer, res_buffer;
	fmpixel_ptr fbuffer, fres_buffer;
	sparse_byteplane_ptr splane, ressplane;
	sparse_fbyteplane_ptr fsplane, fressplane;
	mimage_ptr res_mimg = clone_mimage(mimg);

	int width, height, planes_number, kind;
	
	if (res_mimg) {
		get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
		
		if (kind == FLOAT_IMAGE) {
			for (c=0; c<planes_number; c++) {
				fsplane = get_fbyteplane_from_mimage(mimg, c);
				fbuffer = fsplane->buffer;

				fressplane = get_fbyteplane_from_mimage(res_mimg, c);
				fressplane->default_value = floating_op(user_ptr, user_ptr2, fsplane->default_value, c);
				
				fres_buffer = get_fbyteplane_from_mimage(res_mimg, c)->buffer;
					
				for (i=0; i<mimg->buffer_length; i++) {
					*fres_buffer = floating_op(user_ptr, user_ptr2, *fbuffer, c);
					
					fres_buffer++;
					fbuffer++;
				}
			}
		} else { /* DISCRETE_IMAGE */
			for (c=0; c<planes_number; c++) {
				splane = get_byteplane_from_mimage(mimg, c);
				buffer = splane->buffer;

				ressplane = get_byteplane_from_mimage(res_mimg, c);
				ressplane->default_value = (mpixel)floating_op(user_ptr, user_ptr2, splane->default_value, c);
					
				res_buffer = get_byteplane_from_mimage(res_mimg, c)->buffer;

				for (i=0; i<mimg->buffer_length; i++) {
					*res_buffer = (mpixel)floating_op(user_ptr, user_ptr2, *buffer, c);

					res_buffer++;
					buffer++;
				}
			}
		}
	}
	
	return res_mimg;
}

/**************************************************************************************
/*	Function:		extract_planes_mimage_sparse
/*	Description:	generate a new sparse image extracting some plane from
/*					another sparse one
/*
/*	Parameters:
/*		<-	mimg		source sparse image
/*		<-	from		first plane
/*		<-	to			last plane
/*
/*	Result:
/*		not NULL	the pointer to the result new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr extract_planes_mimage_sparse(mimage_ptr mimg, int from, int to)
{
	int c, d;
	int width, height, planes_number, kind;
/*	byteplane_ptr old_plane;	*/
	mimage_ptr res_mimg;
	
	if (from > to)
		return NULL;
	
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

	res_mimg = clone_mimage_sparse(mimg);

		/* if all planes have to be copied then just return the clone */
	if ((from == 0) && (to == planes_number-1))
		return res_mimg;
		
	for (c=0; c<from; c++)
		destroy_byteplane(res_mimg->planes_array[c], kind, mimg->density);
		
	if (to+1 < planes_number)
		for (c=to+1; c<planes_number; c++)
			destroy_byteplane(res_mimg->planes_array[c], kind, mimg->density);
		
	for (c=0, d=from; d<(to+1); c++, d++)
		res_mimg->planes_array[c] = res_mimg->planes_array[d];

	res_mimg->comp = to-from+1;

	return res_mimg;	
}

/**************************************************************************************
/*	Function:		clone_mimage_sparse
/*	Description:	clone a sparse image in a new one
/*
/*	Parameters:
/*		<-	img		source sparse image
/*
/*	Result:
/*		not NULL	pointer to the new sparse image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr clone_mimage_sparse(mimage_ptr mimg)
{
	int c;
	int height, width, planes_number, kind;
	size_t type_size;
	void *buffer_src, *buffer_dst;
	mimage_ptr cloned_mimg;
	byteplane_ptr *planes_array;
	byteplane_ptr res_plane;
	
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	type_size = GET_TYPE_SIZE(kind);
	
	cloned_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
	if (!cloned_mimg)
		return NULL;
	
	if (mimg->data_length)
		cloned_mimg->row_start = (int *)malloc(mimg->data_length * sizeof(int));
	else
		cloned_mimg->row_start = NULL;
				
	if (mimg->section_length)
		cloned_mimg->row_section = (section_data_ptr)malloc(mimg->section_length * sizeof(section_data));
	else
		cloned_mimg->row_section = NULL;
		
	for (c=0; c<planes_number; c++) 
	{
		res_plane = get_byteplane_from_mimage(cloned_mimg, c);
		if (mimg->buffer_length)
			res_plane->buffer = (void *) malloc(mimg->buffer_length * type_size);
		else
			res_plane->buffer = NULL;
	}

	cloned_mimg->data_length = mimg->data_length;
	cloned_mimg->section_length = mimg->section_length;
	cloned_mimg->buffer_length = mimg->buffer_length;

	memcpy(cloned_mimg->row_start, mimg->row_start, mimg->data_length * sizeof(int));
	memcpy(cloned_mimg->row_section, mimg->row_section, mimg->section_length * sizeof(section_data));

/* check all previous allocations */

	planes_array = mimg->planes_array;
	
	for (c=0; c<planes_number; c++) {

		res_plane = get_byteplane_from_mimage(cloned_mimg, c);
	
		if (kind == FLOAT_IMAGE)
			((fbyteplane_ptr)res_plane)->default_value = ((fbyteplane_ptr *)planes_array)[c]->default_value;
		else
			res_plane->default_value = planes_array[c]->default_value;

		buffer_src = planes_array[c]->buffer;
		buffer_dst = res_plane->buffer;
		
		memcpy(buffer_dst, buffer_src, mimg->buffer_length * type_size);
	}	
	
	return cloned_mimg;
}

/**************************************************************************************
/*	Function:		const_smimage
/*	Description:	create a new const sparse image
/*
/*	Parameters:
/*		<-	values_array	array of values to be stored in the image
/*		<-	dim_ptr			dimensions of the image
/*		<-	kind			kind of the image (float or integer)
/*		<-	comp			number of planes
/*
/*	Result:
/*		not NULL	pointer to the const sparse image
/*		NULL		if an error condition
/*
/***************************************************************************************/

mimage_ptr const_smimage(void *values_array, image_dimension_ptr dim_ptr, int kind, int planes_number)
{
	int c, j;
	int height, width;
	sparse_byteplane_ptr *planes_array;
	sparse_byteplane_ptr splane;
	sparse_fbyteplane_ptr fsplane;
	mimage_ptr sparse_mimg;
	
	height = dim_ptr->x;
	width = dim_ptr->y;

	sparse_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
	if (!sparse_mimg)
		return NULL;

	if (pre_setup_sparse_mimage_no_buffer(sparse_mimg, width, height) != 0) {
		destroy_mimage(sparse_mimg);
		return NULL;
	}
		
	planes_array = sparse_mimg->planes_array;
	
	if (kind == FLOAT_IMAGE) {

		for (c=0; c<planes_number; c++) {
			fsplane = get_fbyteplane_from_mimage(sparse_mimg, c);
			fsplane->default_value = ((fmpixel_ptr)values_array)[c];
		} 

	} else {

		for (c=0; c<planes_number; c++) {
			splane = get_byteplane_from_mimage(sparse_mimg, c);
			splane->default_value = ((mpixel_ptr)values_array)[c];
		}
	}
	
	sparse_mimg->data_length = height;

	for (j=0; j<height; j++) {
		sparse_mimg->row_start[j] = j;

		sparse_mimg->row_section[j].starting_pixel_number = 0;
		sparse_mimg->row_section[j].number_of_pixels = width;
		sparse_mimg->row_section[j].position_in_buffer = -1;
		
	}

	sparse_mimg->section_length = height;
	sparse_mimg->data_length = height;

	return sparse_mimg;
}

/**************************************************************************************
/*	Function:		write_to_section_data
/*	Description:	Set up data inside the section_data field at the index
/*					row_section_index for a sparse image. If index is over
/*					the size of pre-allocated array a dynamic expansion will occur.
/*
/*	Parameters:
/*		<-	mimg					image
/*		<-	row_section_index		index of the section
/*		<->	current_length			current length of the row_section area
/*		<-	starting_pixel_number	starting pixel information
/*		<-	number_of_pixels		number of pixels information
/*		<-	position_in_buffer		position in buffer information
/*
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void write_to_section_data(mimage_ptr mimg, 
	size_t row_section_index, size_t *current_length, 
		size_t starting_pixel_number, 
			size_t number_of_pixels,
				int position_in_buffer)
{
	if (row_section_index >= *current_length) {
		mimg->row_section = realloc(mimg->row_section, 2 * *current_length * sizeof(section_data));
		if (!mimg->row_section)
			abort();
		*current_length *= 2;
	}
				
	mimg->row_section[row_section_index].starting_pixel_number	= starting_pixel_number;
	mimg->row_section[row_section_index].number_of_pixels		= number_of_pixels;
	mimg->row_section[row_section_index].position_in_buffer		= position_in_buffer;
}

/**************************************************************************************
/*	Function:		convert_smimage_to_float
/*	Description:	convert an integer sparse image to float kind
/*
/*	Parameters:
/*		<-> mimg		pointer to the image
/*
/*	Result:
/*		0		ok
/*		else	if an error occurs
/*
/***************************************************************************************/

int convert_smimage_to_float(mimage_ptr mimg)
{
	int i, c;
	mpixel_ptr sbuffer;
	fmpixel_ptr sfbuffer;
	fbyteplane_ptr *sfplanes_array;
	
	int width, height, planes_number, kind;
	
		/*
		 *	get information from the source image 
		 */
		 
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
		
	if (kind == FLOAT_IMAGE)
		return 1;
	
	if (mimg->density != SPARSE_IMAGE)
		return 2;
		
		/* 
		 *	allocate a new set of planes 
		 */
			 
	sfplanes_array = (fbyteplane_ptr *)create_planes_array(width, height, kind, planes_number, mimg->density);
	if (!sfplanes_array)
		return -1;

	for (c=0; c<planes_number; c++) {
		if (mimg->buffer_length) {
			sfplanes_array[c]->buffer = (fmpixel_ptr)malloc( mimg->buffer_length / GET_TYPE_SIZE(DISCRETE_IMAGE) * GET_TYPE_SIZE(FLOAT_IMAGE));
			if (!sfplanes_array[c]->buffer) {
				destroy_planes_array((void **)sfplanes_array, kind, planes_number, SPARSE_IMAGE);
				return -3;
			}
		}
	}
	
		/* 
		 *	copy and convert the data into them 
		 */
		 
	for (c=0; c<planes_number; c++) {
	
		sbuffer = get_byteplane_from_mimage(mimg, c)->buffer;
		sfbuffer = sfplanes_array[c]->buffer;

		sfplanes_array[c]->default_value = (float)((byteplane_ptr)mimg->planes_array[c])->default_value;

		for (i=0; i<mimg->buffer_length; i++)
			*sfbuffer++ = (fmpixel)*sbuffer++;
	
	}
	
	destroy_planes_array((void **)mimg->planes_array, kind, planes_number, mimg->density);	/* in this way we actually */

	mimg->planes_array = (byteplane_ptr *)sfplanes_array;	
	mimg->kind = FLOAT_IMAGE;
	
	return 0;
}

/**************************************************************************************
/*	Function:		convert_smimage_to_discrete
/*	Description:	convert a float sparse image to integer kind
/*
/*	Parameters:
/*		<-> mimg		pointer to the image
/*
/*	Result:
/*		0		ok
/*		else	if an error occurs
/*
/***************************************************************************************/

int convert_smimage_to_discrete(mimage_ptr mimg)
{
	int i, c;
	fmpixel_ptr sfbuffer;
	mpixel_ptr sbuffer;
	byteplane_ptr *splanes_array;
	
	int width, height, planes_number, kind;
	
		/* get information from the source image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

	if (kind == DISCRETE_IMAGE)
		return 1;

	if (mimg->density != SPARSE_IMAGE)
		return 2;

	splanes_array = create_planes_array(width, height, kind, planes_number, mimg->density);
	if (!splanes_array)
		return -1;
		
	for (c=0; c<planes_number; c++) {
		if (mimg->buffer_length) {
			splanes_array[c]->buffer = (mpixel_ptr)malloc(mimg->buffer_length * GET_TYPE_SIZE(DISCRETE_IMAGE));
			if (!splanes_array[c]->buffer) {
				destroy_planes_array((void **)splanes_array, kind, planes_number, SPARSE_IMAGE);
				return -3;
			}
		} else
			splanes_array[c]->buffer = NULL;
	}

	for (c=0; c<planes_number; c++) {
		sfbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
		sbuffer = splanes_array[c]->buffer;
		
		splanes_array[c]->default_value = (mpixel)((fbyteplane_ptr)mimg->planes_array[c])->default_value;

		for (i=0; i<mimg->buffer_length; i++)
			*sbuffer++ = (mpixel)*sfbuffer++;
	}
	
	destroy_planes_array((void **)mimg->planes_array, kind, planes_number, mimg->density);	/* in this way we actually */

	mimg->planes_array = splanes_array;	
	mimg->kind = DISCRETE_IMAGE;
	
	return 0;
}

/**************************************************************************************
/*	Function:		shrink_and_cut_mimage
/*	Description:	This routine returns a pair [im,i,j] consisting of
/*						(1)	a copy of the input image mimg, 
/*							reduced to the smallest rectangle r 
/*							containing its domain
/*						(2)	the rectangle r.
/*
/*	Parameters:
/*		<-	mimg		pointer to the image
/*		->	rect_ptr	size of the rectangle
/*
/*	Result:
/*		not NULL	pointer to the generated image
/*		NULL		if an error occurs
/*
/***************************************************************************************/

mimage_ptr shrink_and_cut_mimage(mimage_ptr mimg, image_rect_ptr rect_ptr)
{
	int c;
	
	mimage_ptr res_mimg;
	int width, height, planes_number, kind;		/* basic data for image						*/
	int entry;									/* another entry in row_start array			*/

	size_t num_of_sections;

	int first_entry, last_entry;				/* initial and final entries in row_start array	*/
	size_t estimated_left, estimated_right;		/* estimated left and right of result			*/
	size_t last_section_ix;						/* index of last section in a row				*/
	int k;										/* miscellaneous iteration indices				*/

	if (mimg->density == DENSE_IMAGE)
		return NULL;					/* 	image is not sparse so operation 
											should not be applied */
	
	res_mimg = clone_mimage(mimg);
	if (!res_mimg)						/* some allocation problem */
		return NULL;

	if (res_mimg->data_length == 0) {	/* image is empty; return it with an empty rectangle */
  		rect_ptr->x = rect_ptr->y = rect_ptr->dx = rect_ptr->dy = 0;
		return res_mimg;
	}

	first_entry = res_mimg->row_start[0];						/* first entry in row_start array */
	last_entry = res_mimg->row_start[mimg->data_length - 1];	/* last entry in row_start array */

	if (res_mimg->data_length == 1 && first_entry < 0) {	
							/* image is empty; return it with an empty rectangle */

		res_mimg->data_length = 0;						/* set the data_length to 0 */

		if (res_mimg->row_start){ 					/* deallocate row_start array if any */
			free(res_mimg->row_start); 
			res_mimg->row_start = NULL;
		}
 
		res_mimg->section_length = 0;

		if (mimg->row_section){ 					/* deallocate row_section array if any */
			free(res_mimg->row_section); 
			res_mimg->row_section = NULL;
		}

 		rect_ptr->x = rect_ptr->y = rect_ptr->dx = rect_ptr->dy = 0;
		return res_mimg;
	}

	get_mimage_info(res_mimg, &height, &width, &planes_number, &kind, NULL);
	
	 										/* determine the actual starting and ending rows */
	rect_ptr->y = (first_entry < 0 ? -first_entry : 0);
	rect_ptr->dy = height - (last_entry < 0 ? -last_entry : 0) - rect_ptr->y; 

	estimated_left = width;							/* initialize quantities to be estimated */	
	estimated_right = 0;
	
	for (k=0; k<res_mimg->data_length; k++) {	/* iterate over all the sparse image data entries */

		if ((entry = res_mimg->row_start[k]) < 0) continue;		/* bypass empty rows */
		
		num_of_sections = sections_in_row(res_mimg,k);
							/* get the number of sections in this nonempty row */
		estimated_left = min(estimated_left,res_mimg->row_section[entry].starting_pixel_number);
		last_section_ix = entry + num_of_sections - 1;
		estimated_right = 
				max(estimated_right, res_mimg->row_section[last_section_ix].starting_pixel_number + 
					res_mimg->row_section[last_section_ix].number_of_pixels - 1);

	}		/* at the end of this loop we have the actual left and right of the result */

	for (c=0; c<planes_number; c++) {
						/* post the revised height and width of the result image */
		((byteplane_ptr)(res_mimg->planes_array[c]))->height = rect_ptr->dy;
		((byteplane_ptr)(res_mimg->planes_array[c]))->width = estimated_right - estimated_left + 1;
	}	
						/* if there is an empty element at the end of the input image, */
						/* reduce the data_length of the result image by 1 */
	if (last_entry < 0)
		res_mimg->data_length--; 

						/* if there is an empty element at the end of the input image,	*/
						/* reduce the data_length of the result image by 1 more, and	*/
						/* shift all the row_start data downwards						*/
	if (first_entry < 0) {
		for (k=1; k<res_mimg->data_length; k++) {
						/* iterate over all the sparse image data entries */
			res_mimg->row_start[k - 1] = res_mimg->row_start[k];
		} 

		res_mimg->data_length--; 	/* reduce the length by 1 more */
	} 
										/* we may need to loop once more,adjusting the  */
										/* starting mpixel numbers in all nonempty rows	*/
	if (estimated_left > 0) {			/* must iterate over all the sparse image data entries */

		for (k=0; k<mimg->section_length; k++) {
			res_mimg->row_section[k].starting_pixel_number -= estimated_left;
		} 

	} 
	
	rect_ptr->x = estimated_left;
	rect_ptr->dx = estimated_right - estimated_left + 1;
	
	return res_mimg;					/* return the completed result */
}

/**************************************************************************************
/*	Function:		binary_op_smimage
/*	Description:	perform a binary operation between two sparse images
/*
/*	Parameters:
/*		<-	mimg_a			first operand sparse image
/*		<-	mimg_b			second operand sparse image
/*		<-	gray_op			pointer to a binary operation function for integer case
/*		<-	float_op		pointer to a binary operation function for float case
/*
/*	Result:
/*		not NULL	the pointer to the result new sparse image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr binary_op_smimage(mimage_ptr mimg_a, mimage_ptr mimg_b, unsigned char (*gray_op)(unsigned char, unsigned char), float (*float_op)(float, float))
{
	mimage_ptr res_mimg;		/* image result */

	int c;						/* index for colour planes */
	int j;						/* index for scanned row at this time */
	int max_width = 0;			/* max_width written (used to calculate image width) */
	int previous_j = 0;			/* index for scanned row at precedent time */

	int	ia, ib;					/* index for columns in images a and b */
	int pos_in_buffer_a;		/* index for position in bitmap buffer for image a */
	int pos_in_buffer_b;		/* index for position in bitmap buffer for image b */

	int *row_start_a;			/* row_start array for image a */
	int *row_start_b;			/* row_start array for image b */
	
	int data_length_a;			/* row_start array length for image a */
	int data_length_b;			/* row_start array length for image b */
	
	section_data *section_data_a;	/* row_section array for image a */
	section_data *section_data_b;	/* row_section array for image a */
	
	int end_sect_a, end_sect_b;		/* indexes for last section at the j-row */

	int data_index;				/* keep track of next available index in the row_start array for result image */
	int data_length; 			/* keep track of current length in the row_start array for result image */
	int section_index;			/* keep track of next available index in the row_section array for result image */
	int section_length;			/* keep track of current length in the row_section array for result image */
	int buffer_index;			/* keep track of next available position in the bitmap data buffers */
	
	int section_written;		/* keep track of the first section for the row j */
	
	int i_start, i_end;			/* column limit at current section in result image*/

	int l_a = 0, m_a = 0;		/* l_a: row in a, m_a: col in a */
	int l_b = 0, m_b = 0;		/* l_b: row in b, m_b: col in b */

	int i_a_start, i_a_end;		/* column limits for image a */
	int j_a;					/* row index for image a */
	int i_b_start, i_b_end;		/* column limits for image b */
	int j_b;					/* row index for image b */
	
	int overlap_step;			/* number of overlapping rows processed */
	
	int width_a, height_a, planes_number_a, kind_a;		/* image information for image a */
	int width_b, height_b, planes_number_b, kind_b;		/* image information for image b */
	int width, height, planes_number, kind;				/* image information for image result */

	size_t type_size;									/* size of type used for the images */
	
	short	is_default_value_section;					/* true if a section has only default values */
	int		number_of_pixels;							/* used to count the number of pixels in a section */
	int		err;										/* keep error value for some procedure */
		
	/* 
	 *	retrieve information from source images 
	 */
	
	get_mimage_info(mimg_a, &height_a, &width_a, &planes_number_a, &kind_a, NULL);
	get_mimage_info(mimg_b, &height_b, &width_b, &planes_number_b, &kind_b, NULL);

	/*
	 *	check if images are compatible between themeselves
	 */

	if (kind_a != kind_b)
	{
		mimg_err_no = kGrKindNotMatch;
		return NULL;		/* cannot mix discrete and float */
	}
	
	if (planes_number_a != planes_number_b)
	{
		mimg_err_no = kGrPlanesNotMatch;
		return NULL;		/* cannot mix images with different planes */
	}
	
	/*
	 *	Prepare data for result image
	 */

	kind = kind_a;
	planes_number = planes_number_a;
	
	data_index = section_index = 0;
	data_length = section_length = buffer_index = 0;
	
	height = min(height_a, height_b);
	width = min(width_a, width_b);
	
	type_size =  GET_TYPE_SIZE(kind);
	
	/* 
	 *	create an empty sparse image
	 */
	 
	res_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
	if (!res_mimg)
	{
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}
	
	/*
	 *	setup buffers with preliminary allocations
	 *		NOTE:	more memory could be needed to
	 *				store the image. In that case
	 *				a dynamic expansion of the table
	 *				will occurr
	 */
	
	if (pre_setup_sparse_mimage(res_mimg, width, height, kind, planes_number) != 0) 
	{
		destroy_mimage(res_mimg);
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}
		
	data_length = res_mimg->data_length;
	section_length = res_mimg->section_length;
	
	/*
	 *	get information to process both source images
	 */ 
	 
	row_start_a = mimg_a->row_start;
	row_start_b = mimg_b->row_start;
	
	data_length_a = mimg_a->data_length;
	data_length_b = mimg_b->data_length;

	section_data_a = mimg_a->row_section;
	section_data_b = mimg_b->row_section;

	/*
	 *	translate to the y origin
	 */
	
	j_a = mimg_a->upperleft_y;
	j_b = mimg_b->upperleft_y;

		/*
		 *	Loop in row_data of a and b
		 */

	for ( ; (l_a < data_length_a) && (l_b < data_length_b) ; ) {

			/*
			 *	Loop until find the first section
			 */

		if ( row_start_a[ l_a ] < 0)
		{
			j_a -= row_start_a[ l_a ++ ];	 /*	j_a += - row_start_a[ l_a ]; l_a ++; */
			continue;
		}

		if ( row_start_b[ l_b ] < 0)
		{
			j_b -= row_start_b[ l_b ++ ];	 /*	j_b += - row_start_b[ l_b ]; l_b ++; */
			continue;
		}
		
			/*
			 *	Check if rows match
			 */

		if ( j_a < j_b ) 		/* if row for a is before row for b */
		{
			l_a ++;					/* go to the next row for a */
			j_a ++;
			continue;
		}
		else if ( j_a > j_b )		/* if row for b is before row for a */
		{
			l_b ++;
			j_b ++;					/* go to the next row for b */
			continue;
		}
		else 					/* both sections are in the same row */
		{
		
			/*
			 *	now we need to check for overlappings along the
			 *	same row. 
			 */
		
			j = j_a;	/* j_a == j_b in this case */
		
			overlap_step = 0;	/* reset overlap value */
			
				/* 
				 *	calculate last section for
				 *	this row for both a and b 
				 */

			end_sect_a = ( m_a = row_start_a[ l_a ] ) + sections_in_row( mimg_a, l_a );
			end_sect_b = ( m_b = row_start_b[ l_b ] ) + sections_in_row( mimg_b, l_b );

				/*
				 *	loop over the sections in the row 
				 */

			section_written = kNoSection;

			for ( ; (m_a < end_sect_a) && (m_b < end_sect_b) ; )
			{
					/* 
					 *	get the boundaries for the section
					 *	and translate to the x origin
					 */

				i_a_start = section_data_a[ m_a ].starting_pixel_number + mimg_a->upperleft_x;
				i_b_start = section_data_b[ m_b ].starting_pixel_number + mimg_b->upperleft_x;

				i_a_end = i_a_start + ( section_data_a[ m_a ].number_of_pixels - 1 );
				i_b_end = i_b_start + ( section_data_b[ m_b ].number_of_pixels - 1 );

					/* if a preceeds b */

				if (i_a_end < i_b_start) {
					m_a ++;						/* advance a to the next section */
					continue;
				}
					
					/* if b preceeds a */

				if (i_a_start > i_b_end) {
					m_b ++;						/* advance b to the next section */
					continue;
				}

					/*
					 *	at this point we found two overlapping section
					 */

				overlap_step = 1;				/* set the overlapping row counter to one */

				if (section_written == kNoSection)
					section_written = section_index;	/* remember the index for the first overlapping section for this row */

					/*
					 *	calc the boundaries for the destination section
					 */

				i_start	= max(i_a_start, i_b_start);
				i_end	= min(i_a_end, i_b_end);

					/*
					 *	calc the positions in the bitmap buffers for the source images
					 */

				pos_in_buffer_a = section_data_a[ m_a ].position_in_buffer;
				pos_in_buffer_b = section_data_b[ m_b ].position_in_buffer;
				
					/*
					 *	get the offsets for the image a and b
					 */

				ia = i_start - i_a_start;
				ib = i_start - i_b_start;


					/*
					 *	calc the number of pixels to copy
					 */

				number_of_pixels = i_end - i_start + 1;

					/*
					 *	if value is not a default value in both the images
					 */
				
				is_default_value_section = (pos_in_buffer_a < 0 && pos_in_buffer_b < 0);
				 
				if (! is_default_value_section )
				{				
						/*
						 *	do the work for discrete image
						 */
					if (kind == DISCRETE_IMAGE) 
					{
						int ii;	/* used for inner loop cycle (adiacent points loop) */

							/* 
							 *	loop over the planes
							 */
							 
						for ( c=0 ; c<planes_number ; c++ ) {

								/*
								 *	Get the planes starting points
								 */

							static mpixel_ptr buffer_a;
							static mpixel_ptr buffer_b;
							static mpixel_ptr buffer;

							buffer_a = (mpixel_ptr)get_plane_buffer(mimg_a->planes_array[c], kind) + pos_in_buffer_a;
							buffer_b = (mpixel_ptr)get_plane_buffer(mimg_b->planes_array[c], kind) + pos_in_buffer_b;
							buffer = (mpixel_ptr)get_plane_buffer(res_mimg->planes_array[c], kind) + buffer_index;


								/*
								 *	Do the operation
								 */

							for ( ii=0 ; ii<number_of_pixels ; ii++ ) 
								*buffer++ = gray_op(*buffer_a++, *buffer_b++);
													
						}
					} 
					else	/* and for float image */
					{
						int ii;	/* used for inner loop cycle (adiacent points loop) */

							/* 
							 *	loop over the planes
							 */
							 
						for ( c=0 ; c<planes_number ; c++ ) {

								/*
								 *	Get the planes starting points
								 */

							static fmpixel_ptr fbuffer_a;
							static fmpixel_ptr fbuffer_b;
							static fmpixel_ptr fbuffer;

							fbuffer_a = (fmpixel_ptr)get_plane_buffer(mimg_a->planes_array[c], kind) + ia;
							fbuffer_b = (fmpixel_ptr)get_plane_buffer(mimg_b->planes_array[c], kind) + ib;
							fbuffer = (fmpixel_ptr)get_plane_buffer(res_mimg->planes_array[c], kind) + buffer_index;
							
								/*
								 *	Do the operation
								 */

							for ( ii=0 ; ii<number_of_pixels ; ii++ ) 
								*fbuffer++ = float_op(*fbuffer_a++, *fbuffer_b++);

						}
					
					}
				}
				
					/*
					 *	update the data section with the added information
					 */

				write_to_section_data(
					res_mimg, 
					section_index,
					(size_t *)&section_length,
					i_start,
					number_of_pixels,
					is_default_value_section ? -1 : buffer_index
				);
					
				buffer_index += number_of_pixels;
				section_index++;
				
					/*
					 *	update max_width
					 */
					 
				if (i_end > max_width)
					max_width = i_end;
					
					/*
					 *	advance to the next section
					 */

				if (i_a_end == i_end)
					m_a++;
					
				if (i_b_end == i_end)
					m_b++;
					
			}			


				/*
				 *	check for skipped lines
				 *
				 *		NOTE:	if overlap rows the step
				 *				must be greater than 
				 *				their number
				 */

			if ( previous_j < j - overlap_step )
			{

					/*
					 *	put the information for skipped lines
					 *	and if needed expand the table
					 */
					 
				err = expand_table_on_need(
					(void **)&res_mimg->row_start, 
					data_index,
					&data_length,
					sizeof(*res_mimg->row_start)
				);
				
				if (err) 	/* check for re-allocation errors */
				{
					destroy_mimage(res_mimg);

					mimg_err_no = kGrOutOfMemory;
					return NULL;
				}

					/* 
					 *	update row_start 
					 */
					 
				res_mimg->row_start[data_index ++ ] = previous_j - j;	/*  - (j - previous_j); with j > previous_j */

			}

			previous_j = j;		/* update previous j to the newer */

				/*
				 *	check if we found an overlap
				 */

			if (overlap_step) 
			{
					/*				
					 *	update row_start for the written sections
					 *  and if needed expand the table
					 */
					 
				err = expand_table_on_need(
					(void **)&res_mimg->row_start, 
					data_index,
					&data_length,
					sizeof(*res_mimg->row_start)
				);
				
				if (err) 
				{
					mimg_err_no = kGrOutOfMemory;
					return NULL;
				}

				res_mimg->row_start[data_index ++] = section_written;	/* update row_start */

			}
			
				/* 
				 *	advance to next section
				 */

			l_a ++;
			l_b ++;
			
			j_a ++;
			j_b ++;
			
		}
		
	}
	
	if (data_index < height)
	{
		err = expand_table_on_need(
			(void **)&res_mimg->row_start, 
			height,
			&data_length,
			sizeof(*res_mimg->row_start)
		);
		
		res_mimg->row_start[data_index] = -(height - data_index);
		data_index++;
	}

		/* 
		 *	shrink the partitions, reassignment is not needed
		 *	because data won't be moved from its position
		 */

	res_mimg->data_length = data_index;
	res_mimg->section_length = section_index;
	res_mimg->buffer_length = buffer_index;
	
	res_mimg->row_start = realloc(res_mimg->row_start, res_mimg->data_length * sizeof( *res_mimg->row_start ) );
	res_mimg->row_section = realloc(res_mimg->row_section, res_mimg->section_length * sizeof( *res_mimg->row_section ) );

	for ( c=0; c<planes_number ; c++) 
	{
		byteplane_ptr plane = res_mimg->planes_array[c];
		plane->buffer = realloc(plane->buffer, res_mimg->buffer_length * type_size);

			/* 
			 *	while we are loopin' in the planes
			 *	let's set the default values
			 *		NOTE:	this if even if it is possible to
			 *				be moved out from the outer loop
			 *				isn't a performance issue 'cause the
			 *				number of planes usually is really small
			 *		
			 */

		if (kind == DISCRETE_IMAGE) 
		{
			mpixel default_value = gray_op ( 
						((fbyteplane_ptr)mimg_a->planes_array[c])->default_value,
						((fbyteplane_ptr)mimg_b->planes_array[c])->default_value);

			set_default_plane_color_smimage(res_mimg, default_value, c);
		} 
		else
		{
			fmpixel fdefault_value =	float_op (
						((fbyteplane_ptr)mimg_a->planes_array[c])->default_value,
						((fbyteplane_ptr)mimg_b->planes_array[c])->default_value);

			set_default_plane_fcolor_smimage(res_mimg, fdefault_value, c);
		}
	}

	/*
	 *	set both width and height
	 */

	set_width_and_height_mimage(res_mimg, width, height);

	return res_mimg;
}

/**************************************************************************************
/*	Function:		binary_op_sdmimage
/*	Description:	perform a binary operation between a sparse and a dense image
/*
/*	Parameters:
/*		<-	mimg_a			first operand sparse image
/*		<-	mimg_b			second operand dense image
/*		<-	gray_op			pointer to a binary operation function for integer case
/*		<-	float_op		pointer to a binary operation function for float case
/*
/*	Result:
/*		not NULL	the pointer to the result new dense image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr binary_op_sdmimage(mimage_ptr mimg_a, mimage_ptr mimg_b, 
	mpixel (*gray_op)(mpixel, mpixel), 
	fmpixel (*float_op)(fmpixel, fmpixel))
{
	int c;		
	int pix;
	int dest_line_offset;
	int dest_pixel_offset;

	size_t sects_in_row;
	size_t data_index, ix, section_index, data_length;

	mimage_ptr dest_mimg = NULL;
	
	int width_a, height_a, planes_number_a, kind_a;
	unsigned char density_a;

	int width_b, height_b, planes_number_b, kind_b;
	unsigned char density_b;

	int width, height, planes_number, kind;

	get_mimage_info(mimg_a, &height_a, &width_a, &planes_number_a, &kind_a, &density_a);
	get_mimage_info(mimg_b, &height_b, &width_b, &planes_number_b, &kind_b, &density_b);

	if (kind_a != kind_b) {
		mimg_err_no = kGrKindNotMatch;
		return NULL;
	}

	if (density_a != SPARSE_IMAGE || density_b != DENSE_IMAGE) 
	{
		mimg_err_no = kGrNotCorrectTypes;
		return NULL;
	}

	if ( (height_a > height_b) || (width_a > width_b) )
	{
		mimg_err_no = kGrNotCorrectSizes;
		return NULL;
	}
	
	if ( planes_number_a != planes_number_b)
	{
		mimg_err_no = kGrPlanesNotMatch;
		return NULL;
	}

	/*
	 *	initialize the variable
	 */

	width=width_b; 
	height=height_b; 
	planes_number=planes_number_b; 
	kind=kind_b;

	/*
	 *	because the image is "sparse" we just clone the data from the dense
	 *	one and do the computation only in the differing pixels
	 */

	dest_mimg = clone_mimage(mimg_b);
	if (!dest_mimg)
	{
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}

	/*
	 *	do the real computation now
	 */

	if (kind == DISCRETE_IMAGE)
	{
		mpixel_ptr buffer_a, buffer_a_ptr;
		mpixel_ptr dest_buffer, dest_buffer_ptr;
		mpixel	default_value;

		data_length = mimg_a->data_length;
		dest_line_offset = mimg_a->upperleft_y;

		for ( data_index = 0 ; data_index < mimg_a->data_length ; data_index ++ )
		{
			int section_data_for_this_row = mimg_a->row_start[data_index];
		
			if ( section_data_for_this_row < 0 ) 
			{
				dest_line_offset += -section_data_for_this_row;
				continue;
			}
			
			sects_in_row = sections_in_row(mimg_a, data_index);
			
			for ( c=0; c<planes_number ; c++)
			{
				dest_buffer = get_plane_buffer(dest_mimg->planes_array[c], kind);
				buffer_a = get_plane_buffer(mimg_a->planes_array[c], kind);
		
				default_value = get_byteplane_from_mimage(mimg_a, c)->default_value;
			
				for ( ix=0 ; ix<sects_in_row ; ix++ )
				{
					section_index = section_data_for_this_row + ix;
				
					dest_pixel_offset = 
						dest_line_offset * width									/* go to the line */
						+ mimg_a->row_section[section_index].starting_pixel_number	/* go to the mpixel */
						+ mimg_a->upperleft_x;										/* consider the offset */
		
					dest_buffer_ptr = dest_buffer + dest_pixel_offset;

					if (mimg_a->row_section[section_index].position_in_buffer < 0) 	/* this is a default value line */
					{
						for ( pix=0; pix<mimg_a->row_section[section_index].number_of_pixels; pix++ )
						{
							*dest_buffer_ptr = gray_op( *dest_buffer_ptr, default_value );
							dest_buffer_ptr ++;
						}
					} 
					else
					{
						buffer_a_ptr = buffer_a + mimg_a->row_section[section_index].position_in_buffer;

						for ( pix=0; pix<mimg_a->row_section[section_index].number_of_pixels; pix++ )
						{
							*dest_buffer_ptr = gray_op( *dest_buffer_ptr, *buffer_a_ptr );
							dest_buffer_ptr ++;
							buffer_a_ptr ++;
						}
					}
				}
			}
			
			dest_line_offset ++;
			
		}
	}
	else
	{
		fmpixel_ptr buffer_a, buffer_a_ptr;
		fmpixel_ptr dest_buffer, dest_buffer_ptr;
		fmpixel	default_value;

		data_length = mimg_a->data_length;
		dest_line_offset = mimg_a->upperleft_y;

		for ( data_index = 0 ; data_index < mimg_a->data_length ; data_index ++ )
		{
			int section_data_for_this_row = mimg_a->row_start[data_index];
		
			if ( section_data_for_this_row < 0 ) 
			{
				dest_line_offset += -section_data_for_this_row;
				continue;
			}
			
			sects_in_row = sections_in_row(mimg_a, data_index);
			
			for ( c=0; c<planes_number ; c++)
			{
				dest_buffer = get_plane_buffer(dest_mimg->planes_array[c], kind);
				buffer_a = get_plane_buffer(mimg_a->planes_array[c], kind);
		
				default_value = get_byteplane_from_mimage(mimg_a, c)->default_value;
			
				for ( ix=0 ; ix<sects_in_row ; ix++ )
				{
					section_index = section_data_for_this_row + ix;
				
					dest_pixel_offset = 
						dest_line_offset * width									/* go to the line */
						+ mimg_a->row_section[section_index].starting_pixel_number	/* go to the mpixel */
						+ mimg_a->upperleft_x;										/* consider the offset */
		
					dest_buffer_ptr = dest_buffer + dest_pixel_offset;

					if (mimg_a->row_section[section_index].position_in_buffer < 0) 	/* this is a default value line */
					{
						for ( pix=0; pix<mimg_a->row_section[section_index].number_of_pixels; pix++ )
						{
							*dest_buffer_ptr = float_op( *dest_buffer_ptr, default_value );
							dest_buffer_ptr ++;
						}
					} 
					else
					{
						buffer_a_ptr = buffer_a + mimg_a->row_section[section_index].position_in_buffer;

						for ( pix=0; pix<mimg_a->row_section[section_index].number_of_pixels; pix++ )
						{
							*dest_buffer_ptr = float_op( *dest_buffer_ptr, *buffer_a_ptr );
							dest_buffer_ptr ++;
							buffer_a_ptr ++;
						}
					}
				}
			}
			
			dest_line_offset ++;
			
		}
	}

	return dest_mimg;
}

/**************************************************************************************
/*	Function:		flip_h_sparse
/*	Description:	create a flipped horizontally sparse image
/*
/*	Parameters:
/*		<-	mimg	source sparse image
/*
/*	Result:
/*		not NULL	pointer to the flipped sparse image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr flip_h_sparse(mimage_ptr mimg)
{
	int row_idx;
	int section_idx, last_section_for_row, section_reached;
	mimage_ptr res_mimg;
	int width, height, planes_number, kind;
	unsigned char density;

	void (*flip_fn)(void *, size_t);

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);

	if (density != SPARSE_IMAGE) {
		mimg_err_no = kGrDensityMismatch;
		return NULL;
	}

	res_mimg = clone_mimage(mimg);
	if (!res_mimg)
	{
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}
	
	if (kind == DISCRETE_IMAGE)
		flip_fn = (void (*)(void *, size_t))flip_mem;
	else
		flip_fn = (void (*)(void *, size_t))flip_fmem;
	
	for ( section_reached=0, row_idx=0 ; row_idx<res_mimg->data_length; row_idx++ )
	{
		int c;
		section_data_ptr section_data;

		if ( res_mimg->row_start[row_idx] >= 0 )
		{
			last_section_for_row = section_reached + sections_in_row(res_mimg, row_idx);

			for ( section_idx=section_reached ; section_idx<last_section_for_row ; section_idx++ ) 
			{
				section_data = &res_mimg->row_section[section_idx];

				section_data->starting_pixel_number = width - 
					(section_data->starting_pixel_number + section_data->number_of_pixels);
				
				if (section_data->position_in_buffer >= 0)
				{
					for ( c=0; c<planes_number ; c++ )
					{
						void *buffer = (char *)get_plane_buffer(res_mimg->planes_array[c], kind) 
										+ section_data->position_in_buffer;

						flip_fn( buffer, section_data->number_of_pixels);
					}
				}
			}
			
			section_reached = last_section_for_row;
		}
	}
	
	return res_mimg;
}

/**************************************************************************************
/*	Function:		flip_mem
/*	Description:	flip all the pixels in a line
/*
/*	Parameters:
/*		<->	buffer	line buffer
/*		<-	size	size of the line
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void flip_mem(mpixel_ptr buffer, size_t size)
{
	size_t i;
	
	for ( i=0 ; i<size ; i++)
		swap(buffer+i, buffer+size-i);
}

/**************************************************************************************
/*	Function:		flip_fmem
/*	Description:	flip all the float pixels in a line
/*
/*	Parameters:
/*		<->	buffer	line buffer
/*		<-	size	size of the line
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void flip_fmem(fmpixel_ptr buffer, size_t size)
{
	size_t i;
	
	for ( i=0 ; i<size ; i++)
		fswap(buffer+i, buffer+size-i);
}

/**************************************************************************************
/*	Function:		swap
/*	Description:	swap two pixels
/*
/*	Parameters:
/*		<->	a	pointer to the first pixel
/*		<->	b	pointer to the second pixel
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void swap(mpixel *a, mpixel *b)
{
	mpixel c;
	
	c = *a;
	*a = *b;
	*b = c;
}

/**************************************************************************************
/*	Function:		fswap
/*	Description:	fswap two float pixels
/*
/*	Parameters:
/*		<->	a	pointer to the first float pixel
/*		<->	b	pointer to the second float pixel
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void fswap(fmpixel *a, fmpixel *b)
{
	fmpixel c;
	
	c = *a;
	*a = *b;
	*b = c;
}

/**************************************************************************************
/*	Function:		flip_v_sparse
/*	Description:	create a flipped vertically sparse image
/*
/*	Parameters:
/*		<-	mimg	source sparse image
/*
/*	Result:
/*		not NULL	pointer to the flipped sparse image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr flip_v_sparse(mimage_ptr mimg)
{
	int free_sect_idx;
	int row_idx, last_line;
	int *row_data_src, *row_data_dst;
	
	short more_lines;
	
	mimage_ptr res_mimg;
	int width, height, planes_number, kind;
	unsigned char density;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);

	/*
	 *	check if the image is ok
	 */

	if (density != SPARSE_IMAGE) {
		mimg_err_no = kGrDensityMismatch;
		return NULL;
	}

	/*
	 *	clone the image
	 */

	res_mimg = clone_mimage(mimg);
	if (!res_mimg)
	{
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}
	
	/*
	 *	free memory for row_start table
	 */

	SMART_FREE(res_mimg->row_start);

	/*
	 *	check if are there any line after the last indexed by row_data
	 */

	last_line = 0;
	for ( row_idx=0 ; row_idx<mimg->data_length ; row_idx++ )
	{
		int temp;
		
		last_line += (temp = mimg->row_start[row_idx]) < 0 ? -temp : 1;
	}
	
	if (last_line < height) 
	{
		last_line = height - last_line;
		more_lines = 1;	
	}
	else
		more_lines = 0;
		
	/*
	 *	alloc new table for row_data
	 */
	
	row_data_dst = res_mimg->row_start = malloc( (more_lines + res_mimg->data_length) * sizeof(*res_mimg->row_start) );
	if (!row_data_dst)
	{
		destroy_mimage(res_mimg);
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}
	
	/*
	 *	get the pointers of the data to be copied
	 */
	
	row_data_src = mimg->row_start;
	row_data_dst = res_mimg->row_start;

	/*
	 *	move trailing lines to the start of the new image
	 */
	
	if (more_lines) 
	{
		*row_data_dst++ = -last_line;
		res_mimg->data_length++;	
	}

	/*
	 *	simply invert row_data contents to do the vertical flip
	 */
	
	for ( free_sect_idx=0, row_idx=mimg->data_length-1 ; row_idx>=0 ; row_idx-- ) 
	{
		int row_data;
		
		row_data = row_data_src[row_idx];
		if (row_data >= 0)
		{
			int sect_idx;
			int last_section_in_row = row_data + sections_in_row(mimg, row_idx);
			
			*row_data_dst = free_sect_idx;
			
			for ( sect_idx=row_data ; sect_idx<last_section_in_row ; sect_idx++, free_sect_idx++ )
				memcpy(	res_mimg->row_section + free_sect_idx, mimg->row_section + sect_idx, sizeof(*res_mimg->row_section));
			
		}
		else
			*row_data_dst = row_data;
		
		row_data_dst++;
	}
		
	return res_mimg;
}

/**************************************************************************************
/*	Functions:		rotate_90_sparse, rotate_180_sparse, rotate_270_sparse
/*	Description:	rotate dummy stubs
/*	Note:			don't do anything
/*
/*	Parameters:
/*		<-	mimg	source image
/*
/*	Result:
/*		mimg itself
/*
/***************************************************************************************/

mimage_ptr rotate_90_sparse(mimage_ptr mimg) { return mimg; }
mimage_ptr rotate_180_sparse(mimage_ptr mimg) {	return mimg; }
mimage_ptr rotate_270_sparse(mimage_ptr mimg) { return mimg; }

/**************************************************************************************
/*	Function:		sumall_mimage_sparse
/*	Description:	return the momentums of a sparse image
/*					
/*	Parameters:
/*		<-	mimg	operand sparse image
/*		<-	power	highest momentum requested
/*
/*	Result:
/*		not NULL	array of requested momentums
/*		NULL		out of memory
/*
/***************************************************************************************/

double *sumall_mimage_sparse(mimage_ptr mimg, int power)
{
	int c, sect_idx, k, i;

	size_t number_of_pixels;
	int position_in_buffer;

	double *fsum;
	
	int width, height, planes_number, kind;
	unsigned char density;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);
	
	if (density != SPARSE_IMAGE)
	{
		mimg_err_no = kGrDensityMismatch;
		return NULL;
	}
	
	fsum = calloc(power, sizeof(*fsum));
	if (!fsum) {
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}

	if (kind == DISCRETE_IMAGE)
	{
		mpixel default_value;
		mpixel_ptr buffer_start, buffer_ptr;
		
		for ( c=0; c<planes_number; c++)
		{
			default_value = get_byteplane_from_mimage(mimg, c)->default_value;
			buffer_start = get_byteplane_from_mimage(mimg, c)->buffer;
			
			for ( sect_idx=0 ; sect_idx<mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
				position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

				if ( position_in_buffer < 0 )
				{
/*
					for ( i=0 ; i<number_of_pixels ; i++ )
						for ( k=0 ; k<power ; k++ )
							fsum[k] += pow((float)default_value, (float)k);
 */
					for ( k=0 ; k<power ; k++ )
	 					fsum[k] += number_of_pixels * pow(default_value, k);
				}
				else 
				{
					buffer_ptr = buffer_start + position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
						for ( k=0 ; k<power ; k++ )
							fsum[k] += pow(*buffer_ptr++, k);
				}
			}
		}
	}
	else /* FLOAT_IMAGE */
	{
		fmpixel fdefault_value;
		fmpixel_ptr fbuffer_start, fbuffer_ptr;
		
		for ( c=0; c<planes_number; c++)
		{
			fdefault_value = get_fbyteplane_from_mimage(mimg, c)->default_value;
			fbuffer_start = get_fbyteplane_from_mimage(mimg, c)->buffer;
			
			for ( sect_idx=0 ; sect_idx<mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
				position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

				if ( position_in_buffer < 0 )
				{
 					for ( k=0 ; k<power ; k++ ) {
	 					fsum[k] += number_of_pixels * pow(fdefault_value, k);
	 				}

				}
				else 
				{
					fbuffer_ptr = fbuffer_start + position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
						for ( k=0 ; k<power ; k++ ) {
							fsum[k] += pow(*fbuffer_ptr++, k);
						}
				}
			}
		}
	}
	
	return fsum;
}

/**************************************************************************************
/*	Function:		maxof_mimage_sparse
/*	Description:	return the max pixel value of a sparse image
/*					
/*	Parameters:
/*		<-	mimg	operand sparse image
/*
/*	Result:
/*		the maximum
/*
/***************************************************************************************/

float maxof_mimage_sparse(mimage_ptr mimg)
{
	int c, sect_idx, i;

	size_t number_of_pixels;
	int position_in_buffer;

	float fmax = 0.;
	
	int width, height, planes_number, kind;
	unsigned char density;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);
	
	if (density != SPARSE_IMAGE)
	{
		mimg_err_no = kGrDensityMismatch;
		return 0.;
	}
	
	if (kind == DISCRETE_IMAGE)
	{
		mpixel default_value;
		mpixel_ptr buffer_start, buffer_ptr;
		
		for ( c=0; c<planes_number; c++)
		{
			default_value = get_byteplane_from_mimage(mimg, c)->default_value;
			buffer_start = get_byteplane_from_mimage(mimg, c)->buffer;
			
			for ( sect_idx=0 ; sect_idx<mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
				position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

				if ( position_in_buffer < 0 )
				{
					if (fmax < default_value)
						fmax = default_value;
				}
				else 
				{
					buffer_ptr = buffer_start + position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
					{
						if (fmax < *buffer_ptr)
							fmax = *buffer_ptr;
						buffer_ptr++;
					}
				}
			}
		}
	}
	else /* FLOAT_IMAGE */
	{
		fmpixel fdefault_value;
		fmpixel_ptr fbuffer_start, fbuffer_ptr;
		
		for ( c=0; c<planes_number; c++)
		{
			fdefault_value = get_fbyteplane_from_mimage(mimg, c)->default_value;
			fbuffer_start = get_fbyteplane_from_mimage(mimg, c)->buffer;
			
			for ( sect_idx=0 ; sect_idx<mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
				position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

				if ( position_in_buffer < 0 )
				{
					if (fmax < fdefault_value)
						fmax = fdefault_value;
				}
				else 
				{
					fbuffer_ptr = fbuffer_start + position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
					{
						if (fmax < *fbuffer_ptr)
							fmax = *fbuffer_ptr;
						fbuffer_ptr++;
					}
				}
			}
		}
	}
	
	return fmax;
}

/**************************************************************************************
/*	Function:		minof_mimage_sparse
/*	Description:	return the min pixel value of a sparse image
/*					
/*	Parameters:
/*		<-	mimg	operand sparse image
/*
/*	Result:
/*		the maximum
/*
/***************************************************************************************/

float minof_mimage_sparse(mimage_ptr mimg)
{
	int c, sect_idx, i;

	size_t number_of_pixels;
	int position_in_buffer;

	float fmin = BIGFLOAT;
	
	int width, height, planes_number, kind;
	unsigned char density;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);
	
	if (density != SPARSE_IMAGE)
	{
		mimg_err_no = kGrDensityMismatch;
		return 0.;
	}
	
	if (kind == DISCRETE_IMAGE)
	{
		mpixel default_value;
		mpixel_ptr buffer_start, buffer_ptr;
		
		for ( c=0; c<planes_number; c++)
		{
			default_value = get_byteplane_from_mimage(mimg, c)->default_value;
			buffer_start = get_byteplane_from_mimage(mimg, c)->buffer;
			
			for ( sect_idx=0 ; sect_idx<mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
				position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

				if ( position_in_buffer < 0 )
				{
					if (default_value < fmin)
						fmin = default_value;
				}
				else 
				{
					buffer_ptr = buffer_start + position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
					{
						if (fmin > *buffer_ptr)
							fmin = *buffer_ptr;
						buffer_ptr++;
					}
				}
			}
		}
	}
	else /* FLOAT_IMAGE */
	{
		fmpixel fdefault_value;
		fmpixel_ptr fbuffer_start, fbuffer_ptr;
		
		for ( c=0; c<planes_number; c++)
		{
			fdefault_value = get_fbyteplane_from_mimage(mimg, c)->default_value;
			fbuffer_start = get_fbyteplane_from_mimage(mimg, c)->buffer;
			
			for ( sect_idx=0 ; sect_idx<mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
				position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

				if ( position_in_buffer < 0 ) 
				{
					if (fdefault_value < fmin)
						fmin = fdefault_value;
				}
				else 
				{
					fbuffer_ptr = fbuffer_start + position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
					{
						if (fmin > *fbuffer_ptr)
							fmin = *fbuffer_ptr;
						fbuffer_ptr++;
					}
				}
			}
		}
	}
	
	return fmin;
}

/**************************************************************************************
/*	Function:		sort_mimage_sparse
/*	Description:	return an array of sorted sparse image values
/*					
/*	Parameters:
/*		<-	mimg		operand sparse image
/*		->	elements	return the number of elements in the sorted array
/*
/*	Result:
/*		not NULL	array of sorted values
/*		NULL		out of memory
/*
/***************************************************************************************/

void *sort_mimage_sparse(mimage_ptr mimg, int *elements)
{
	void *list;
	
	int c, sect_idx, i;
	int pixels_count, j;
	
	size_t number_of_pixels;
	int position_in_buffer;

	float fmin = BIGFLOAT;
	
	int width, height, planes_number, kind;
	unsigned char density;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);
	
	if (density != SPARSE_IMAGE)
	{
		mimg_err_no = kGrDensityMismatch;
		return NULL;
	}
	
		/* upperbound for list size */
	list = malloc(width * height * GET_TYPE_SIZE(kind) * planes_number);
	if (!list)
	{
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}

	j = pixels_count = 0;
	
	if (kind == DISCRETE_IMAGE)
	{
		mpixel default_value;
		mpixel_ptr buffer_start, buffer_ptr;
		mpixel_ptr list_ptr = (mpixel_ptr)list;
		mpixel old;
		
		/*
		 *	get the values used in this image
		 */

		for ( c=0; c<planes_number; c++)
		{
			default_value = get_byteplane_from_mimage(mimg, c)->default_value;
			
			buffer_start = get_byteplane_from_mimage(mimg, c)->buffer;
			
			if (!mimg->buffer_length || !buffer_start)	/* no data */
			{
				list_ptr[pixels_count++] = default_value;
				continue;
			}
			
			for ( sect_idx=0 ; sect_idx<mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
				position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

				if ( position_in_buffer < 0 )
					list_ptr[pixels_count++] = default_value;
				else 
				{
					buffer_ptr = buffer_start + position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
					{
						list_ptr[j++] = *buffer_ptr++;
					}
				}
			}
		}
		
		/*
		 *	sort the array
		 */
		 
		qsort(list, pixels_count, GET_TYPE_SIZE(kind), (int (*)(const void *, const void *))unsigned_char_compare);

		/*
		 *	eliminate duplicates
		 */
		 
		old = list_ptr[0];
		
		for ( i=1, j=1 ; i<pixels_count ; i++)
		{
			if ( list_ptr[i] != old )
			{
				old = list_ptr[j] = list_ptr[i];
				j++;
			}
		}
	}
	else	/* FLOAT_IMAGE */
	{
		fmpixel default_value;
		fmpixel_ptr buffer_start, buffer_ptr;
		fmpixel_ptr list_ptr = (fmpixel_ptr)list;
		fmpixel old;
		
		/*
		 *	get the values used in this image
		 */

		for ( c=0; c<planes_number; c++)
		{
			default_value = get_fbyteplane_from_mimage(mimg, c)->default_value;
			buffer_start = get_fbyteplane_from_mimage(mimg, c)->buffer;
			
			if (!mimg->buffer_length || !buffer_start)	/* no data */
			{
				list_ptr[pixels_count++] = default_value;
				continue;
			}
			
			for ( sect_idx=0 ; sect_idx<mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
				position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

				if ( position_in_buffer < 0 )
					list_ptr[pixels_count++] = default_value;
				else 
				{
					buffer_ptr = buffer_start + position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
					{
						list_ptr[j++] = *buffer_ptr++;
					}
				}
			}
		}
		
		/*
		 *	sort the array
		 */
		 
		qsort(list, pixels_count, GET_TYPE_SIZE(kind), (int (*)(const void *, const void *))float_compare);

		/*
		 *	eliminate duplicates
		 */
		 
		old = list_ptr[0];
		
		for ( i=1, j=1 ; i<pixels_count ; i++)
		{
			if ( list_ptr[i] != old )
			{
				old = list_ptr[j] = list_ptr[i];
				j++;
			}
		}
	}
	
	list = realloc(list, j*GET_TYPE_SIZE(kind));
	*elements = j;

	return list;
}

/**************************************************************************************
/*	Function:		histogram_mimage_sparse
/*	Description:	return an histogram of the pixel values from a sparse image
/*
/*	Parameters:
/*		<-	mimg			source sparse image
/*		<-	fvalues			array of points
/*		<-	fvalues_size	size of the array of points
/*
/*	Result:
/*		not NULL	a pointer to the histogram array
/*		NULL		error condition
/*
/***************************************************************************************/

size_t *histogram_mimage_sparse(mimage_ptr mimg, float *fvalues, size_t fvalues_size)
{
	int c, sect_idx, i;
	size_t number_of_pixels;

	size_t *range_no;
	size_t range_value;
	
	int width, height, planes_number, kind;
	unsigned char density;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);
	
	if (density != SPARSE_IMAGE)
	{
		mimg_err_no = kGrDensityMismatch;
		return NULL;
	}

	range_no = (size_t *)calloc((fvalues_size + 1) * sizeof(size_t), 1);
	if (!range_no)
	{
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}
	
	qsort(fvalues, fvalues_size, sizeof(float), (int (*)(const void *, const void *))float_compare);

	if (kind == DISCRETE_IMAGE)
	{
		mpixel default_value;
		mpixel_ptr buffer_start, buffer_ptr;
		
		/*
		 *	get the values used in this image
		 */

		for ( c=0; c<planes_number; c++)
		{
			default_value = get_byteplane_from_mimage(mimg, c)->default_value;
			buffer_start = get_byteplane_from_mimage(mimg, c)->buffer;
			
			for ( sect_idx=0 ; sect_idx<mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;

				if ( mimg->row_section[sect_idx].position_in_buffer < 0 ) 
				{
					range_value = bfbsearch((float)default_value, fvalues, fvalues_size) + 1;
					range_no[range_value]++;
				}
				else 
				{
					buffer_ptr = buffer_start + mimg->row_section[sect_idx].position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
					{
						range_value = bfbsearch((float)*buffer_ptr++, fvalues, fvalues_size) + 1;
						range_no[range_value]++;
					}
				}
			}
		}
	}
	else	/* FLOAT_IMAGE */
	{
		fmpixel fdefault_value;
		fmpixel_ptr fbuffer_start, fbuffer_ptr;
		
		/*
		 *	get the values used in this image
		 */

		for ( c=0; c<planes_number; c++)
		{
			fdefault_value = get_fbyteplane_from_mimage(mimg, c)->default_value;
			fbuffer_start = get_fbyteplane_from_mimage(mimg, c)->buffer;
			
			for ( sect_idx=0 ; sect_idx<mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;

				if ( mimg->row_section[sect_idx].position_in_buffer < 0 ) 
				{
					range_value = bfbsearch(fdefault_value, fvalues, fvalues_size) + 1;
					range_no[range_value]++;
				}
				else 
				{
					fbuffer_ptr = fbuffer_start + mimg->row_section[sect_idx].position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
					{
						range_value = bfbsearch(*fbuffer_ptr++, fvalues, fvalues_size) + 1;
						range_no[range_value]++;
					}
				}
			}
		}
	}
	
	return range_no;
}

/**************************************************************************************
/*	Function:		threshold_mimage_sparse
/*	Description:	create a sparse image as a threshold from another
/*
/*	Parameters:
/*		<-	mimg			source sparse image
/*		<-	fvalues			array of thresholds
/*		<-	fvalues_size	size of the array of thresholds
/*
/*	Result:
/*		not NULL	a pointer to the thresholded image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr threshold_mimage_sparse(mimage_ptr mimg, float *fvalues, size_t fvalues_size)
{
	int c, sect_idx, i;
	size_t number_of_pixels;

	mimage_ptr res_mimg;
	
	int width, height, planes_number, kind;
	unsigned char density;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);

	if (density != SPARSE_IMAGE)
	{
		mimg_err_no = kGrDensityMismatch;
		return NULL;
	}

	res_mimg = clone_mimage(mimg);
	if (!res_mimg)
	{
		mimg_err_no = kGrOutOfMemory;
		return NULL;
	}
	
	qsort(fvalues, fvalues_size, sizeof(float), (int (*)(const void *, const void *))float_compare);

	if (kind == DISCRETE_IMAGE)
	{
		mpixel_ptr buffer_start, buffer_ptr;
		byteplane_ptr bplane;
		/*
		 *	get the values used in this image
		 */

		for ( c=0; c<planes_number; c++)
		{
			bplane = get_byteplane_from_mimage(res_mimg, c);

			buffer_start = bplane->buffer;
			bplane->default_value = (mpixel)threshold_pixel((float)bplane->default_value, fvalues, fvalues_size);
			
			for ( sect_idx=0 ; sect_idx<res_mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = res_mimg->row_section[sect_idx].number_of_pixels;

				if ( res_mimg->row_section[sect_idx].position_in_buffer >= 0 ) 
				{
					buffer_ptr = buffer_start + res_mimg->row_section[sect_idx].position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
						*buffer_ptr = (mpixel)threshold_pixel((float)*buffer_ptr, fvalues, fvalues_size);
				}
			}
		}
	}
	else /* FLOAT_IMAGE */
	{
		fmpixel_ptr buffer_start, buffer_ptr;
		fbyteplane_ptr bplane;
		/*
		 *	get the values used in this image
		 */

		for ( c=0; c<planes_number; c++)
		{
			bplane = get_fbyteplane_from_mimage(res_mimg, c);

			buffer_start = bplane->buffer;
			bplane->default_value = (fmpixel)threshold_pixel(bplane->default_value, fvalues, fvalues_size);
			
			for ( sect_idx=0 ; sect_idx<res_mimg->section_length ; sect_idx++ )
			{
				number_of_pixels = res_mimg->row_section[sect_idx].number_of_pixels;

				if ( res_mimg->row_section[sect_idx].position_in_buffer >= 0 ) 
				{
					buffer_ptr = buffer_start + res_mimg->row_section[sect_idx].position_in_buffer;
					for ( i=0; i<number_of_pixels ; i++ )
						*buffer_ptr = (fmpixel)threshold_pixel(*buffer_ptr, fvalues, fvalues_size);
				}
			}
		}
	}

	return res_mimg;
}

/**************************************************************************************
/*	Function:		set_planes_mimage_sparse
/*	Description:	insert a set of planes from the sparse 
/*					image mimg to the dense image mimg_res
/*
/*	Parameters:
/*		<->	mimg_res	destintaion sparse image
/*		<-	mimg		source sparse image
/*		<-	from		first plane
/*		<-	to			last plane
/*
/*	Result:
/*		not NULL	the sparse mimg_res itself with the inseted planes
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr set_planes_mimage_sparse(mimage_ptr mimg_res, mimage_ptr mimg, int from, int to)
{
	int c;

	int j, j_res;
	int l, l_res, m, m_res;

	int data_length, data_length_res;
	int *row_start, *row_start_res;
	section_data_ptr section_data, section_data_res;
	
	int width, height, planes_number, kind;
	int width_res, height_res, planes_number_res, kind_res;
	size_t type_size;

	if (from > to)
		return NULL;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	get_mimage_info(mimg_res, &height_res, &width_res, &planes_number_res, &kind_res, NULL);

		/* check if operation is possible */
	if ((width_res != width) || (height_res != height) || (kind_res != kind))
		return NULL;
		
	if ((to-from+1) != planes_number)
		return NULL;

	type_size = GET_TYPE_SIZE(kind);

	data_length = mimg->data_length;
	data_length_res = mimg_res->data_length;
	
	row_start = mimg->row_start;
	row_start_res = mimg_res->row_start;

	section_data = mimg->row_section;
	section_data_res = mimg->row_section;
	
	/* loop over row_start */
	
	for ( j=j_res=0, l=l_res=0 ; (l < data_length) && (l_res < data_length) ; )
	{
			/* jump empty lines in (guest) mimg */
		if (row_start[l] < 0)
		{
			j -= row_start[l];
			l++;
			continue;
		}
		
			/* jump empty lines in destination (host) mimg */
		if (row_start_res[m] < 0)
		{
			j_res -= row_start[m];
			m++;
			continue;
		}
		
		if ( j < j_res )
		{
			l ++;
			j ++;
		}
		else if ( j > j_res)
		{
			l_res ++;
			j_res ++;			
		}
		else /* if ( j == j_res ) */ /* rows match */
		{
			int end_sect, end_sect_res;
			
			end_sect = m = row_start[l] + sections_in_row(mimg, l);
			end_sect_res = m_res = row_start_res[l_res] + sections_in_row(mimg_res, l_res);
			
			for ( ; (m < end_sect) && (m_res < end_sect_res) ; )
			{
				int i_start, i_start_res, i_end, i_end_res;
				
				i_start = section_data[m].starting_pixel_number + mimg->upperleft_x;
				i_start_res = section_data_res[m_res].starting_pixel_number + mimg_res->upperleft_x;
				
				i_end = i_start + (section_data[m].number_of_pixels - 1);
				i_end_res = i_start_res + (section_data_res[m_res].number_of_pixels - 1);
				
				if (i_end < i_start_res)
				{
					m++;
					continue;
				}
				
				if (i_start < i_end_res)
				{
					m_res++;
					continue;
				}
				
					/* 
					 *	do not get default value sections 
					 *	*** this is a simplifing limitation ***
					 */
				
				if (section_data[m].position_in_buffer < 0)
					m++;
				else if (section_data_res[m_res].position_in_buffer < 0)
					m_res++;
				else
				{
					int start, end;
					
					start = max(i_start, i_start_res);
					end = min(i_end, i_end_res);

					for ( c=from ; c<=to ; c++)
					{
						int idx;
						int l = end-start;
						int d = section_data[m].position_in_buffer + (i_start - section_data[m].starting_pixel_number);
						int d_res = section_data_res[m_res].position_in_buffer + (i_start_res - section_data_res[m_res].starting_pixel_number);

						if (kind == DISCRETE_IMAGE)
						{
							mpixel_ptr buffer = ((byteplane_ptr)(mimg->planes_array[c]))->buffer;
							mpixel_ptr buffer_res = ((byteplane_ptr)(mimg_res->planes_array[c]))->buffer;
							
							buffer += d;
							buffer_res += d_res;

							for ( idx=0 ; idx<l ; idx++ )
								*buffer_res++ = *buffer++;
						}
						else
						{
							fmpixel_ptr buffer = ((fbyteplane_ptr)(mimg->planes_array[c]))->buffer;
							fmpixel_ptr buffer_res = ((fbyteplane_ptr)(mimg_res->planes_array[c]))->buffer;
							
							buffer += d;
							buffer_res += d_res;

							for ( idx=0 ; idx<l ; idx++ )
								*buffer_res++ = *buffer++;
						}
					}
					
					if (end == i_end)
						m++;

					if (end == i_end_res)
						m_res++;
				}
					
				
			}
		}
	}

	return mimg_res;
}

/**************************************************************************************
/*	Function:		get_pixel_mimage_sparse
/*	Description:	return an array with the values (if more than one plane)
/*					for a pixel at a given position in the sparse image mimg 
/*
/*	Parameters:
/*		<-	mimg		source sparse image
/*		<-	x			x coordinate of the pixel
/*		<-	y			y coordinate of the pixel
/*
/*	Result:
/*		not NULL	a pointer to the array (float if the image is float, else integer)
/*		NULL		error condition
/*
/***************************************************************************************/

void *get_pixel_mimage_sparse(mimage_ptr mimg, int x, int y)
{
	int c;
	int found;
	void *result = NULL;
	
	int row_idx, sect_idx, offset, position_in_buffer, previous_reached_line;
	
	int width, height, planes_number, kind;
	unsigned char density;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);

	if (density != SPARSE_IMAGE)
	{
		mimg_err_no = kGrDensityMismatch;
		return NULL;
	}
	
	found = locate_pixel_mimage_sparse(mimg, x, y, &row_idx, &sect_idx, &offset, &position_in_buffer, &previous_reached_line);

	if (found)
	{
		int type_size = GET_TYPE_SIZE(kind);
		
		result = calloc(planes_number, type_size);
		if (!result)
		{
			mimg_err_no = kGrOutOfMemory;
			return NULL;
		}
		
		if (kind == DISCRETE_IMAGE)
		{

			if (position_in_buffer < 0)
			{
				for ( c=0 ; c<planes_number ; c++ )
					((mpixel_ptr)result)[c] = get_byteplane_from_mimage(mimg, c)->default_value;
			}
			else
			{
				for ( c=0 ; c<planes_number ; c++ )
				{
					offset += position_in_buffer;
					((mpixel_ptr)result)[c] = *(get_byteplane_from_mimage(mimg, c)->buffer + offset);
				}
			}
		}
		else	/* FLOAT_IMAGE */
		{
			if (position_in_buffer < 0)
			{
				for ( c=0 ; c<planes_number ; c++ )
					((fmpixel_ptr)result)[c] = get_fbyteplane_from_mimage(mimg, c)->default_value;
			}
			else
			{
				for ( c=0 ; c<planes_number ; c++ )
				{
					offset += position_in_buffer;
					((fmpixel_ptr)result)[c] = *(get_fbyteplane_from_mimage(mimg, c)->buffer + offset);
				}
			}
		}
	}
	else
		mimg_err_no = kGrPixelNotFound;
		
	return result;
}


/**************************************************************************************
/*	Function:		locate_pixel_mimage_sparse
/*	Note:			auxiliary function for set_pixel_mimage_sparse not in use
/*
/*	Parameters:
/*		n/a
/*
/*	Result:
/*		n/a
/*
/***************************************************************************************/

int locate_pixel_mimage_sparse(mimage_ptr mimg, int x, int y, 
	int *r_row_idx, int *r_sect_idx, int *r_offset_in_sect, int *r_position_in_buffer, int *r_previous_reached_line)
{
/*
 *	Locates the specified mpixel in the sparse image, Returns 1 if mpixel found, 0 otherwise. 
 *	If the mpixel  is found, r_row_idx is the index of the data row in which it was found; 
 *	r_sect_idx is the index of the section in which the mpixel was found
 *	r_offset_in_sect is the offset of the mpixel from the start of its section
 *	r_position_in_buffer is the position of the mpixel data in the mpixel values buffer 
 *	r_previous_reached_line is not used, and is set to 0.
 *
 *	There are three cases in which the mpixel might not be found. It might lie in an inter-row gap, or
 *	might lie in an inter-section gap within its row. In the first case, 
 *	r_row_idx is the index of the last preceding data row, or -1 if no row preceded;
 *	r_previous_reached_line is the image line number of the line indexed by r_row_idx
 *	r_sect_idx, r_offset_in_sect, and r_position_in_buffer are not used, and are set to 0;
 *
 *	if the mpixel does not lie in an inter-row gap but does lie in an inter-section gap, then 
 *	r_row_idx is the index of the data row in which it lies;
 *	r_previous_reached_line is the image line number of this line;
 *	r_sect_idx is the index of the last preceding section (not necessarily in this row), or -1 if no section preceded;
 *	r_offset_in_sect is not used, and is set to 0; 
 *	r_position_in_buffer is used as a flag indicating that this case has been reached  and is set to -1;
 */

	int image_line_reached;
	int row_idx, sect_idx, last_sect_for_row;

	int *row_start;						/* get pointer to the row_start array */
	section_data_ptr row_section;		/* get pointer to the row_section array */

	size_t type_size;
	
	int width, height, planes_number, kind;
	unsigned char density;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);

	type_size = GET_TYPE_SIZE(kind);
 
	row_start = mimg->row_start;
	row_section = mimg->row_section;
	
		/*
		 *	Iterate over the rows
		 */
	
	for ( image_line_reached=0, row_idx=0 ; row_idx<mimg->data_length ; row_idx++ )
	{

#ifdef I_WANNA_ERRORS
		if (image_line_reached>y)	/* no row found, return */
		{
				/*
				 *	see comment above 
				 */
				 
			*r_row_idx = row_idx - 1;
			*r_previous_reached_line = image_line_reached + row_start[row_idx - 1];	/* note that this row_start value is negative */
			
				/* other vars are not used in this case */
			*r_sect_idx = 0;
			*r_offset_in_sect = 0;
			*r_position_in_buffer = 0;		
			
			return 0;	/* mpixel has not been found */
		}
#endif /* I_WANNA_ERRORS */
			
		if (row_start[row_idx] < 0)	/* this negative value gives the size of an inter-row gap */
		{
			image_line_reached -= row_start[row_idx];	/* advance this many rows */
			
			if (image_line_reached > y)
			{
				*r_row_idx = row_idx - 1;
				*r_previous_reached_line = image_line_reached + row_start[row_idx] ;	/* note that this row_start value is negative */
				
					/* other vars are not used in this case */
				*r_sect_idx = 0;
				*r_offset_in_sect = 0;
				*r_position_in_buffer = 0;		
				
				return 0;	/* mpixel has not been found */
			}
			
			continue;	/* and continue */
		}
		
		if (image_line_reached==y)
		{
			/* 
			 *	the row exists, now look for the column inside the sections
			 */

			last_sect_for_row = (sect_idx = row_start[row_idx]) + sections_in_row(mimg, row_idx);

				/*
				 *	iterate over all sections for this row
				 */
				 
			for ( ; sect_idx<last_sect_for_row ; sect_idx++ )
			{
				int offset = x - row_section[sect_idx].starting_pixel_number;
				
					/*
					 *	we have fallen in an inter-section gap
					 */

/* 				if (offset > row_section[sect_idx].number_of_pixels)	*/
				if (offset < 0)
				{
						/*
						 *	see comment above 
						 */

					*r_row_idx = row_idx;							/* return row_idx */
					*r_previous_reached_line = image_line_reached;	/* we found the right line but not the right column*/
					*r_sect_idx = sect_idx - 1;
					*r_position_in_buffer = -1;

						/* other vars are not used in this case */
					*r_offset_in_sect = 0;
					return 0;	/* mpixel has not been found */
				}

					/*
					 *	we have found the right section
					 */

				if (offset >= 0 && offset < row_section[sect_idx].number_of_pixels)
				{
					int position_in_buffer = row_section[sect_idx].position_in_buffer;
					
					*r_row_idx = row_idx;
					*r_previous_reached_line = 0;
					*r_sect_idx = sect_idx;
					*r_offset_in_sect = offset;
					*r_position_in_buffer = position_in_buffer;		

					return 1;	/* mpixel has been found */
				}
			}

				/*	
				 *	if we fall outside the loop the mpixel is not in any 
				 *	section for this line. Return after setting
				 */

			*r_row_idx = row_idx;
			*r_previous_reached_line = image_line_reached;  /* we found the right line but not the right column*/
			*r_sect_idx = sect_idx - 1;
			*r_position_in_buffer = -1;

				/* other vars are not used in this case */
			*r_offset_in_sect = 0;
			
			return 0;	/* mpixel has not been found */
		}
		
		image_line_reached++;
	}
	
		/*
		 *	see comment above 
		 */				 

	*r_row_idx = row_idx - 1;
	*r_previous_reached_line = image_line_reached + row_start[row_idx - 1];	/* note that this row_start value is negative */
	
		/* other vars are not used in this case */
	*r_sect_idx = 0;
	*r_offset_in_sect = 0;
	*r_position_in_buffer = 0;		
	
	return 0;	/* mpixel has not been found */
}

/**************************************************************************************
/*	Function:		append_pixel_to_buffer
/*	Note:			auxiliary function for set_pixel_mimage_sparse not in use
/*
/*	Parameters:
/*		n/a
/*
/*	Result:
/*		n/a
/*
/***************************************************************************************/

int append_pixel_to_buffer(mimage_ptr mimg, void *pixel_value_array)
{
	int c;
	int type_size = GET_TYPE_SIZE(mimg->kind);

	if (mimg->kind == DISCRETE_IMAGE)
	{
		mpixel_ptr buffer;

		for ( c=0 ; c<mimg->comp ; c++)
		{
				/* reallocate ... */
				
			buffer = get_byteplane_from_mimage(mimg, c)->buffer;
			buffer = realloc(buffer, (mimg->buffer_length+1) * type_size );
			if (!buffer) {
				for ( c-- ; c>=0 ; c-- )
				{
					buffer = get_byteplane_from_mimage(mimg, c)->buffer;
					realloc ( buffer, mimg->buffer_length * type_size );	/* restore changed buffers to their previous values */
					
					return -1;
				}
			}
				
				/* ... and set the mpixel */
			
			*(buffer + mimg->buffer_length) = *((mpixel_ptr)pixel_value_array + c);
		}
	}
	else	/* FLOAT_IMAGE */
	{
		fmpixel_ptr fbuffer;

		for ( c=0 ; c<mimg->comp ; c++)
		{
				/* reallocate ... */
				
			fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
			fbuffer = realloc(fbuffer, (mimg->buffer_length+1) * type_size );
			if (!fbuffer) {
				for ( c-- ; c>=0 ; c-- )
				{
					fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
					realloc ( fbuffer, mimg->buffer_length * type_size );	/* restore changed buffers to their previous values */
					
					return -1;
				}
			}
				
				/* ... and set the mpixel */
			
			*(fbuffer + mimg->buffer_length) = *((fmpixel_ptr)pixel_value_array + c);
		}
	}
	
	return ++mimg->buffer_length;
}

/**************************************************************************************
/*	Function:		set_pixel_mimage_sparse
/*	Description:	set a new values for a pixel in a sparse image
/*	Note:			never completed
/*
/*	Parameters:
/*		<-	mimg				source sparse image
/*		<-	x					x coordinate of the pixel
/*		<-	y					y coordinate of the pixel
/*		<-	pixel_value_array	new values for the pixels
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int set_pixel_mimage_sparse(mimage_ptr mimg, int x, int y, void *pixel_value_array)
{
	int found;
	int r_row_idx, r_sect_idx, r_offset_in_sect, r_position_in_buffer, r_previous_reached_line;

	int width, height, planes_number, kind;
	unsigned char density;

	int number_of_pixels;
	
	int *row_start;
	section_data *row_section;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);

	if (density != SPARSE_IMAGE)
	{
		mimg_err_no = kGrDensityMismatch;
		return -1;
	}
	
	row_start = mimg->row_start;
	row_section = mimg->row_section;
		
		/*
		 *	see if the mpixel belongs to the image
		 */
	found = locate_pixel_mimage_sparse(mimg, x, y, &r_row_idx, &r_sect_idx, &r_offset_in_sect, &r_position_in_buffer, &r_previous_reached_line);

	if (found)	/* the mpixel belonbgs to the image */
	{

		if (r_position_in_buffer >= 0)	/* if the mpixel is in the buffer */
		{
			write_pixel_at_buffer_offset(mimg, r_position_in_buffer, pixel_value_array);	/* just write the mpixel */
			return 1;
		}
		
		/* 
		 *	otherwise it is not in the buffer, then must have a default value 
		 */

		if (is_pixel_default_value(mimg, pixel_value_array))	/* if the mpixel has the default value, simply return */
		{
			return 1;
		}
		
		/* 
		 *	otherwise it is not in the buffer, value being overwritten must be default
		 */
		 
		if ((number_of_pixels = row_section[r_sect_idx].number_of_pixels) == 1)
		{
			int new_pixel_offset = append_pixel_to_buffer(mimg, pixel_value_array);
		}
		
	}
		/* otherwise the mpixel does not belong to the image */

	return 1;
}

/**************************************************************************************
/*	Function:		copy_pixels_between_mimages
/*	Description:	copy pixel data from one image to another one
/*
/*	Parameters:
/*		<->	mimg_dst		destination image
/*		<-	mimg_src		source image
/*		<-	offset_dst		offset in destination
/*		<-	offset_src		offset in source
/*		<-	num_of_pixels	number of pixels to copy
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void copy_pixels_between_mimages(mimage_ptr mimg_dst, mimage_ptr mimg_src, int offset_dst, int offset_src, int num_of_pixels)
{
	int c;
	gr_ptr src_buffer, dst_buffer;

	int type_size = GET_TYPE_SIZE(mimg_src->kind);
	
	for (c=0 ; c<mimg_src->comp; c++) {
		src_buffer = get_byteplane_from_mimage(mimg_src, c)->buffer;
		dst_buffer = get_byteplane_from_mimage(mimg_dst, c)->buffer;
		
		src_buffer += offset_src * type_size;
		dst_buffer += offset_dst * type_size;
		
		memcpy(dst_buffer, src_buffer, num_of_pixels * type_size);
	}
}

/**************************************************************************************
/*	Function:		copy_default_value_at_offset
/*	Description:	copy the default value at a certain offset in the image buffer
/*
/*	Parameters:
/*		<->	target_mimg						destination image
/*		<-	src_mimg						source image (where the default comes from)
/*		<-	position_in_target_buffer		offset in destination
/*		<-	times 							number of default pixels to put
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void copy_default_value_at_offset(
	mimage_ptr target_mimg, mimage_ptr src_mimg, 
	int position_in_target_buffer, int times)
{
	int n, c;
	int planes_number = GET_PLANES_NUMBER(src_mimg);

	if (IS_DISCRETE(src_mimg)) {
		mpixel_ptr dst_buffer;
		mpixel default_value;

		for (c=0; c<planes_number; c++) {
			default_value = get_byteplane_from_mimage(src_mimg, c)->default_value;
			dst_buffer = get_byteplane_from_mimage(target_mimg, c)->buffer;

			dst_buffer += position_in_target_buffer;

			for (n=times; n ; n--) 
				*dst_buffer++ = default_value;
		}
	} else {
		fmpixel_ptr fdst_buffer;
		fmpixel fdefault_value;

		for (c=0; c<planes_number; c++) {
			fdefault_value = get_fbyteplane_from_mimage(src_mimg, c)->default_value;
			fdst_buffer = get_fbyteplane_from_mimage(target_mimg, c)->buffer;

			fdst_buffer += position_in_target_buffer;

			for (n=times; n ; n--) 
				*fdst_buffer++ = fdefault_value;
		}
	}
}

/**************************************************************************************
/*	Function:		copy_default_value_at_offset
/*	Description:	copy the default value at a certain offset in the image buffer
/*					of the destination image
/*
/*	Parameters:
/*		<->	mimg		destination image
/*		<-	offset		offset in destination
/*		<-	value		array (float or unsigned char) 
/*						containing the value of the pixel
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void write_pixel_at_buffer_offset(mimage_ptr mimg, int offset, void *value)
{
	int c;
	
	if (mimg->kind == DISCRETE_IMAGE)
	{
		mpixel_ptr buffer;
		
		for (c=0 ; c<mimg->comp; c++)
		{
			buffer = get_byteplane_from_mimage(mimg, c)->buffer;
			*(buffer + offset) = *((mpixel_ptr)value + c);
		}
	}
	else	/* FLOAT_IMAGE */
	{
		fmpixel_ptr fbuffer;
		
		for (c=0 ; c<mimg->comp; c++)
		{
			fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
			*(fbuffer + offset) = *((fmpixel_ptr)value + c);
		}
	}
}

/**************************************************************************************
/*	Function:		random_mimage_sparse
/*	Description:	random fill the pixels of a sparse image
/*
/*	Parameters:
/*		<-	mimg	source sparse image
/*
/*	Result:
/*		not NULL	a pointer to the result sparse image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr random_mimage_sparse(mimage_ptr mimg, float r_min, float r_max)
{
	clock_t seed;
	
	int i, c;
	int number_of_pixels;
	int type_size;
	
	mimage_ptr target_mimg;
	
	target_mimg = clone_mimage_sparse(mimg);
	if (!target_mimg)
		return NULL;
				
	type_size = GET_TYPE_SIZE(target_mimg->kind);

		/* 
		 *	let's release the buffers first 
		 */
	for (c=0; c<target_mimg->comp; c++) {	
		byteplane_ptr plane = target_mimg->planes_array[c];
		if (plane->buffer)
			free(plane->buffer);

		plane->buffer = NULL;
	}
	target_mimg->buffer_length = 0;

		/*
		 *	calculate size of the buffers
		 */
	number_of_pixels = 0;	
	for (i=0 ; i<target_mimg->section_length ; i++) {
		section_data_ptr sect = &(target_mimg->row_section[i]);
		
		sect->position_in_buffer = number_of_pixels;
		number_of_pixels += sect->number_of_pixels;
	}

		/*
		 *	allocate buffers of the right size
		 */
	for (c=0; c<target_mimg->comp; c++) {	
		byteplane_ptr plane = target_mimg->planes_array[c];
		plane->buffer = malloc(number_of_pixels * type_size);
		if (!plane->buffer) {
			destroy_mimage(target_mimg);
			return NULL;
		}
	}
	target_mimg->buffer_length = number_of_pixels;

		/*
		 *	set the random generator seed to a more casual value
		 */
	seed = clock();
	srand(seed);

	if (IS_DISCRETE(target_mimg))
	{
		mpixel_ptr buffer;

		int i_min = r_min;
		int i_max = r_max;
		int i_range = i_max - i_min;
		
		for (c=0; c<target_mimg->comp; c++) {
			buffer = get_byteplane_from_mimage(target_mimg, c)->buffer;
			
			for (i=0 ; i<number_of_pixels ; i++)
				*buffer++ = (((float)rand())/RAND_MAX) * i_range + i_min;
			
		}
	} else {	/* ! DISCRETE_IMAGE */
		fmpixel_ptr fbuffer;

		float f_range = r_max - r_min;
		
		for (c=0; c<target_mimg->comp; c++) {
			fbuffer = get_fbyteplane_from_mimage(target_mimg, c)->buffer;
			
			for (i=0 ; i<number_of_pixels ; i++)
				*fbuffer++ = (((float)rand())/RAND_MAX) * f_range + r_min;
			
		}
	}
	
	return target_mimg;
}


/**************************************************************************************
/*	Function:		rectangle_mimage_sparse
/*	Description:	create a sparse rectangle image
/*
/*	Parameters:
/*		<-	width			rectangle width
/*		<-	height			rectangle height
/*		<-	thick			rectangle thick
/*		<-	planes_number	image planes number
/*		<-	kind			image kind
/*		<-	values_array	array (float or unsigned char) 
/*							containing the value of the pixels
/*							of the rectangle
/*
/*	Result:
/*		not NULL	a pointer to the generated sparse image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr rectangle_mimage_sparse(int width, int height, int thick, int planes_number, int kind, void *values_array)
{
	int i;
	int ok;
	int sect_idx = 0, row_idx = 0;
	int section_length;
	mimage_ptr target_mimg;
	
	if ((thick * 2 > width) || (thick * 2 > height))
		return NULL;

	target_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
	if (!target_mimg)
		return NULL;
		
	section_length = 2 * height - 2 * thick;

	ok = allocate_sparse_buffers(target_mimg, height, section_length, 0);	
	if (!ok) {
		destroy_mimage(target_mimg);
		return NULL;
	}
	
	for (i=0; i<thick; i++) {
		section_data_ptr sect = &(target_mimg->row_section[sect_idx]);
		target_mimg->row_start[row_idx++] = sect_idx;
		
		sect->starting_pixel_number = 0;
		sect->number_of_pixels = width;
		sect->position_in_buffer = -1; 	/* only default values */
		
		sect_idx++;
	}

	for (i=0; i<(height - 2 * thick); i++) {
		section_data_ptr sect = &(target_mimg->row_section[sect_idx]);
		target_mimg->row_start[row_idx++] = sect_idx;
	
			/* first section in line */
		sect->starting_pixel_number = 0;
		sect->number_of_pixels = thick;
		sect->position_in_buffer = -1; 	/* only default values */
		
		sect_idx++;

		sect = &(target_mimg->row_section[sect_idx]);
		
			/* second section in line */
		sect->starting_pixel_number = width - thick;
		sect->number_of_pixels = thick;
		sect->position_in_buffer = -1; 	/* only default values */
		
		sect_idx++;
	}
	
	for (i=0; i<thick; i++) {
		section_data_ptr sect = &(target_mimg->row_section[sect_idx]);
		target_mimg->row_start[row_idx++] = sect_idx;
	
		sect->starting_pixel_number = 0;
		sect->number_of_pixels = width;
		sect->position_in_buffer = -1; 	/* only default values */
		
		sect_idx++;
	}

	/*
	 *	Set now the default value
	 */
	if (kind == DISCRETE_IMAGE)
		set_default_color_smimage(target_mimg, values_array);
	else	/* FLOAT_IMAGE */
		set_default_fcolor_smimage(target_mimg, values_array);
	
	return target_mimg;
}

/**************************************************************************************
/*	Function:		ellipse_mimage_sparse
/*	Description:	create a sparse ellipse image
/*
/*	Parameters:
/*		<-	width			ellipse width
/*		<-	height			ellipse height
/*		<-	thick			ellipse thick
/*		<-	planes_number	image planes number
/*		<-	kind			image kind
/*		<-	values_array	array (float or unsigned char) 
/*							containing the value of the pixels
/*							of the rectangle
/*
/*	Result:
/*		not NULL	a pointer to the generated sparse image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr ellipse_mimage_sparse(int width, int height, int thick, int planes_number, int kind, void *values_array)
{
	int size;

	float y, y2;
	int start, end, start2, end2;
	float a, b, a2, b2;
	
	int row_idx, sect_idx;
	
	int ok;
	int section_length;
	mimage_ptr target_mimg;
	section_data_ptr sect;

	if ((thick * 2 > width) || (thick * 2 > height))
		return NULL;

	target_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);
	if (!target_mimg)
		return NULL;

	section_length = 2 * height;	/* uppercase in number of sections */

	ok = allocate_sparse_buffers(target_mimg, height, section_length, 0);	
	if (!ok) {
		destroy_mimage(target_mimg);
		return NULL;
	}

	a = width / 2.;
	b = height / 2.;

	a2 = (width-2*thick) / 2.;
	b2 = (height-2*thick) / 2.;

	y = -(b - 0.5);
	y2 = -(b2 - 0.5);
		
	for (row_idx=0, sect_idx=0; row_idx<height; row_idx++, y++) {
		
		target_mimg->row_start[row_idx] = sect_idx;
		sect = &(target_mimg->row_section[sect_idx]);

		start = ROUND(a - ellipse_by_y(y, a, b));
		end = width - start;
		
		if (abs(y) <= abs(y2)) {			/* if is there any ellipse inside */

			start2 = ROUND(a - ellipse_by_y(y, a2, b2));
			end2 = width - start2;

			sect->starting_pixel_number = start;
			sect->number_of_pixels = (size = start2 - start) < 0 ? 0 : size;
			sect->position_in_buffer = -1;	/* only default values */
			
			sect_idx++;

			sect = &(target_mimg->row_section[sect_idx]);
			
			sect->starting_pixel_number = end2;
			sect->number_of_pixels = (size = end - end2) < 0 ? 0 : size;
			sect->position_in_buffer = -1;	/* only default values */

			sect_idx++;
			
		} else {	/* if only outer ellipse */

			sect->starting_pixel_number = start;
			sect->number_of_pixels = (size = end-start);
			sect->position_in_buffer = -1;	/* only default values */
			
			sect_idx++;
		}
	}

	/*
	 *	Set now the default value
	 */
	if (kind == DISCRETE_IMAGE)
		set_default_color_smimage(target_mimg, values_array);
	else	/* FLOAT_IMAGE */
		set_default_fcolor_smimage(target_mimg, values_array);
	
	target_mimg->section_length = sect_idx;

	trim_sparse(target_mimg);

	return target_mimg;
}

/**************************************************************************************
/*	Function:		ellipse_by_y
/*	Description:	return the x coordinate from the y in an ellipse
/*
/*	Parameters:
/*		<-	y		y coordinate
/*		<-	a		a axe
/*		<-	b		b axe
/*
/*	Result:
/*		the x coordinate
/*
/***************************************************************************************/

int ellipse_by_y(float y, float a, float b)
{
	float f = sqrt( a*a * (1 - (y*y) / (b*b)));
	return ROUND(f);
}

/**************************************************************************************
/*	Function:		invert_mimage_sparse
/*	Description:	invert all the pixels inside a sparse image
/*
/*	Parameters:
/*		<-<	mimg	the image
/*
/*	Result:
/*		0		ok
/*		else	error condition
/*
/***************************************************************************************/

int invert_mimage_sparse(mimage_ptr mimg)
{
	int i, c;

	mpixel_ptr buffer;
	fmpixel_ptr fbuffer;
	byteplane_ptr plane;
	fbyteplane_ptr fplane;

	int planes_number = mimg->comp;
	size_t pixels_per_plane = mimg->buffer_length;
	
	if (mimg->kind == DISCRETE_IMAGE) {
		for (c=0; c<planes_number; c++) {
		
			plane = get_byteplane_from_mimage(mimg, c);
			buffer = plane->buffer;
			
			plane->default_value = 0xff - plane->default_value;			
			
			for (i=0; i<pixels_per_plane; i++) {
				*buffer = 0xff - *buffer;
				buffer++;
			}
		}
	} else {
		for (c=0; c<planes_number; c++) {
		
			fplane = get_fbyteplane_from_mimage(mimg, c);
			fbuffer = fplane->buffer;
			
			fplane->default_value = 0xff - fplane->default_value;			
			
			for (i=0; i<pixels_per_plane; i++) {
				*fbuffer = 0xff - *fbuffer;
				fbuffer++;
			}
		}
	}
		
	return 0;	
}

/**************************************************************************************
/*	Function:		offset_mimage_sparse
/*	Description:	offset a sparse image
/*
/*	Parameters:
/*		<->	mimg	source dense image
/*		<-	d		offset
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int offset_mimage_sparse(mimage_ptr mimg, image_displacement_ptr d)
{
	int i;
	int width, height;

	int sect_data_idx;
	int *row_start;
		
	if (d->x < 0 || d->y < 0) {
		return -1;
	}

		/* 
		 *	y offset 
		 */

		/* if the first row is negative */
	if (mimg->row_start[0] < 0) {
		mimg->row_start[0] -= d->y;
	} else {
		row_start = realloc(mimg->row_start, (mimg->data_length + 1) * sizeof( *mimg->row_start ));
		if (!row_start)
			return -2;	/* not enough memory to expand the row start buffer */
		mimg->data_length ++;
		mimg->row_start = row_start;
		
			/* move up the lines */
		for (i=mimg->data_length-1; i>0; i--)
			mimg->row_start[i] = mimg->row_start[i-1];
			
		mimg->row_start[0] = -d->y;
	}
	
		/* 
		 *	x offset 
		 */

	for (sect_data_idx=0 ; sect_data_idx<mimg->section_length ; sect_data_idx++ ) {
		mimg->row_section[sect_data_idx].starting_pixel_number += d->x;
	}
	
	width_and_height_mimage(mimg, &width, &height);
	set_width_and_height_mimage(mimg, width+d->x, height+d->y);

	return 0;
}

/**************************************************************************************
/*	Function:		look_for_line_in_rows
/*	Description:	look for a line in row data
/*
/*	Parameters:
/*		<-	mimg					image
/*		<-	start_row_data_idx		row data idx to start from
/*		<-	start_line				line to start from
/*		<-	line_to_find			line to find
/*		<->	new_sect_idx			last used section
/*
/*	Result:
/*		index in row data
/*
/***************************************************************************************/

int look_for_line_in_rows(mimage_ptr mimg, int start_row_data_idx, int start_line, int line_to_find, int *new_sect_idx)
{
	int the_row_data_idx;
	int *row_start = mimg->row_start;
	int row_data_idx = start_row_data_idx;
	int line = start_line;

	if (start_line == line_to_find) {
		the_row_data_idx = start_row_data_idx;
	} else {
	
		the_row_data_idx = -1;
	
		for (; row_data_idx<mimg->data_length ; row_data_idx++) {

			if (line == line_to_find) {
			
				the_row_data_idx = row_data_idx;
				*new_sect_idx = mimg->row_start[the_row_data_idx];
				
				break;
			}

			if (line > line_to_find) {
			
				the_row_data_idx = row_data_idx - 1;
				*new_sect_idx = ( -row_start[the_row_data_idx] ) - (line - line_to_find);
				break;
			}

			if (row_start[row_data_idx] < 0)
				line -= row_start[row_data_idx];
			else
				line ++;	/* go to the next line */
				
		}
	}
			
		/* we reached this only if last line was skipped
		   because is part of a group of empty lines */
	if (row_data_idx == mimg->data_length) {

		the_row_data_idx = row_data_idx - 1;
		*new_sect_idx = ( -row_start[the_row_data_idx] ) - (line - line_to_find);
	}
	
	return the_row_data_idx;
}
 
/**************************************************************************************
/*	Function:		crop_mimage_sparse
/*	Description:	crop a sparse image
/*
/*	Parameters:
/*		<-> mimg	pointer to the sparse image
/*		<- rect		rect for crop
/*
/*	Result:
/*		0		ok
/*		else	if an error occurs
/*
/***************************************************************************************/

int crop_mimage_sparse(mimage_ptr mimg, image_rect_ptr rect)
{
	int c,i,j;
	
	int					target_data_length = 0;
	int					target_section_length = 0;
	int					target_buffer_length = 0;
	
	int					*target_row_start = NULL;
	section_data_ptr	target_row_section = NULL;
	gr_ptr				*target_buffers = NULL;

	int line = 0;
	int row_data_idx = 0;
	int row_sect_idx = 0;

	int new_first_row_data_idx;	/* index of new image first line in the old image */
	int new_last_row_data_idx;	/* index of new image last line in the old image */
	int new_first_row_sect_idx;	/* index for the section to be replaced in
								   the first row data of the destination image */
	int new_last_row_sect_idx;	/* index for the section to be replaced in
								   the last row data of the destination image */
	
	int height, width, planes_number, kind;
	
	int *row_start;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
		
		/* pre-allocate all of the tables */
		
	target_row_start = malloc(sizeof(*target_row_start) * mimg->data_length);
	if (!target_row_start)
		goto abort_crop_sparse;
	target_row_section = malloc(sizeof(*target_row_section) * mimg->section_length);
	if (!target_row_section)
		goto abort_crop_sparse;
	target_buffers = calloc(planes_number, sizeof(gr_ptr));
	if (!target_buffers)
		goto abort_crop_sparse;
	for (i=0; i<planes_number; i++) {
		if (mimg->buffer_length) {
			target_buffers[i] = malloc(GET_TYPE_SIZE(kind) * mimg->buffer_length);
			if (!target_buffers[i])
				goto abort_crop_sparse;
		}
	}	

	rect->dx = min(rect->dx, width - rect->x);
	rect->dy = min(rect->dy, height - rect->y);

	row_start = mimg->row_start;
	
		/* first of all jump to the line */
	new_first_row_data_idx = look_for_line_in_rows(mimg, row_data_idx, line, rect->y, &new_first_row_sect_idx);
	if (new_first_row_data_idx < 0) {
		/* line not found - empty image to be returned */
		goto abort_crop_sparse;
	}

	line = rect->y;
	new_last_row_data_idx = look_for_line_in_rows(mimg, new_first_row_data_idx, line, rect->dy, &new_last_row_sect_idx);

		/* 
		 *	at this point we have two indexes:
		 *		new_first_row_data_idx: index of new image first line in the old image
		 *		new_last_row_data_idx: index of new image last line in the old image
		 */

		/* loop between rows and set up the sections in the target image */
		
	for (i=new_first_row_data_idx ; i <= new_last_row_data_idx ; i++) {

		int last_sect_in_row_idx;		
				
		int first_sect_in_row_idx = mimg->row_start[i];
		int sects_no = sections_in_row(mimg, i);

		int skipped_lines = mimg->row_start[i];
		skipped_lines = skipped_lines >= 0 ? 0 : -skipped_lines;	/* get the positive number of skipped lines if any else 0 */

		if (!skipped_lines) {
		
				/* look for the section after or conteining the first mpixel */
			first_sect_in_row_idx = look_for_pixel_in_sects(mimg->row_section, first_sect_in_row_idx, sects_no, rect->x, 1);
			
			if (first_sect_in_row_idx != -1) { /* start mpixel found */
		
					/* look for the section after or conteining the last mpixel */
				last_sect_in_row_idx = look_for_pixel_in_sects(mimg->row_section, first_sect_in_row_idx, sects_no, rect->dx, 0);
				
				if (last_sect_in_row_idx != -1) {	/* end sections not found; empty row */
				
					target_row_start[target_data_length++] = target_section_length;

						/* copy found sections in the target */
					for (j=first_sect_in_row_idx; j <= last_sect_in_row_idx ; j++) {
						
						int offset = 0;
						section_data_ptr sect_j;
						section_data_ptr orig_sect = &mimg->row_section[j];
											
						int start = (int)orig_sect->starting_pixel_number - rect->x;
						int num_pixels =  orig_sect->number_of_pixels;
						
						if (start < 0) {
							num_pixels -= (offset = -start);
							start = 0;
						}
						
						if ((start + num_pixels + rect->x - 1) > rect->dx)
							num_pixels -= (start + num_pixels + rect->x - 1) - rect->dx;

						sect_j = &target_row_section[target_section_length++];
						
						sect_j->starting_pixel_number = start;
						sect_j->number_of_pixels = num_pixels;

						if (!IS_DEFAULT_SECTION_BY_IDX(mimg, j)) {
							for (c=0; c<planes_number; c++) {
								gr_ptr src_buffer = get_byteplane_from_mimage(mimg, c)->buffer + (sect_j->position_in_buffer + offset)* GET_TYPE_SIZE(kind);
								memcpy(target_buffers[c], src_buffer, GET_TYPE_SIZE(kind) * sect_j->number_of_pixels);
								
								sect_j->position_in_buffer = target_buffer_length;
								target_buffer_length += sect_j->number_of_pixels;
							}
						} else {
							sect_j->position_in_buffer = -1;
						}
					}
					
					continue;	/* restart the loop */
				}
			}
			
			skipped_lines = -1;	/* no more than a skipped line for this time */
			
		}
		
			/* count the skipped lines */
		if (target_data_length && target_row_start[target_data_length - 1] < 0)
			target_row_start[target_data_length++] += -skipped_lines;
		else
			target_row_start[target_data_length++] = -skipped_lines;
	}

	SMART_FREE(mimg->row_start);
	SMART_FREE(mimg->row_section);

	mimg->buffer_length = target_buffer_length;	
	mimg->section_length = target_section_length;
	mimg->data_length = target_data_length;	

	mimg->row_start = target_row_start;
	mimg->row_section = target_row_section;

	for (c=0; c<planes_number; c++) {
		byteplane_ptr plane = get_byteplane_from_mimage(mimg, c);
		SMART_FREE(plane->buffer);
		plane->buffer = target_buffers[c];
	}

	set_width_and_height_mimage(mimg, rect->dx - rect->x + 1, rect->dy - rect->y + 1);

	SMART_FREE(target_buffers);
	
	trim_sparse(mimg);	/* remove superfluous memory */

	return 0;

abort_crop_sparse:

	SMART_FREE(target_row_start);
	SMART_FREE(target_row_section);
	if (target_buffers) {
		for (i=0; i<planes_number; i++)
			SMART_FREE(target_buffers[i]);
		SMART_FREE(target_buffers);
	}		

	return -1;
}

	/* want_first:	if true return the first section that preceeds or contains the mpixel	*/
	/*				if false return the last one that preceeds or contains the mpixel		*/
	
/**************************************************************************************
/*	Function:		look_for_pixel_in_sects
/*	Description:	look for pixel in sections
/*
/*	Parameters:
/*		<-	sects		pointer to the section data
/*		<-	sect_idx	sect index where to start
/*		<-	sects_no	number of sections to look inside
/*		<-	pixel_x		pixel abscissa we are looking for
/*		<-	want_first	if TRUE return the first section that
/*						preceeds or contains the mpixel
/*						if FALSE return the last one that
/*						preceeds or contains the mpixel
/*
/*	Result:
/*		0		ok
/*		else	if an error occurs
/*
/***************************************************************************************/

int look_for_pixel_in_sects(section_data *sects, int sect_idx, int sects_no, int pixel_x, short want_first)
{
	int j, idx = -1;
	int last_sect = sect_idx + sects_no - 1;

	for (j=sect_idx; j<=last_sect; j++) {

		section_data_ptr sect = &sects[j];
		int first_pixel_in_sect = sect->starting_pixel_number;
		int last_pixel_in_sect = first_pixel_in_sect + sect->number_of_pixels - 1;
		
		if (first_pixel_in_sect > pixel_x) {
			if (want_first) return j;	/*  */
			else break;
		}

		if (last_pixel_in_sect < pixel_x) {
			if (!want_first) idx = j;
			continue;
		}

		if (want_first) return j;
		else idx = j;
	}
	
	return idx;
}

/**************************************************************************************
/*	Function:		mask_dense_using_sparse
/*	Description:	return a new sparse image masking a dense image using a sparse one
/*
/*	Parameters:
/*		<-	mimg		the dense image
/*		<-	mask_mimg	the sparse mask image
/*
/*	Result:
/*		not NULL	pointer to the image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr mask_dense_using_sparse(mimage_ptr mimg, mimage_ptr mask_mimg)
{
	int c;
	int i, j;
	int x, y;
	
	int type_size;
	int planes_number;
	int width, height;
	int mask_width, mask_height;

	int rsj;
	int *row_start;
	int sections_at_this_row;
	section_data_ptr row_section;
	section_data_ptr sects;
	int data_length;
	int pixels_to_copy;
	int buffer_top = 0;
	
	gr_ptr src_buffer, dst_buffer;
	
	mimage_ptr target_mimg;
	
	get_mimage_info(mimg, &height, &width, NULL, NULL, NULL);
	get_mimage_info(mask_mimg, &mask_height, &mask_width, NULL, NULL, NULL);
/*
	if (mask_height > height || mask_width > width)
		return NULL;
*/
	if (!IS_DENSE(mimg) || !IS_SPARSE(mask_mimg))
		return NULL;

	target_mimg = clone_mimage(mask_mimg);
	if (!target_mimg)
		return NULL;
		
	type_size = GET_TYPE_SIZE(mimg->kind);

	data_length = target_mimg->data_length;
	row_start = target_mimg->row_start;
	row_section = target_mimg->row_section;

	target_mimg->kind = mimg->kind;

	destroy_planes_array((void **)target_mimg->planes_array, target_mimg->kind, target_mimg->comp, target_mimg->density);
	target_mimg->planes_array = NULL;
	
	planes_number = target_mimg->comp = mimg->comp;

	target_mimg->planes_array = malloc(sizeof(byteplane_ptr) * target_mimg->comp);
	
	if (!target_mimg->planes_array) {
		destroy_mimage(target_mimg);
		return NULL;
	}
	
	for (c=0; c<planes_number; c++) {
	
		byteplane_ptr plane = malloc(IS_FLOAT(target_mimg) ? sizeof(fbyteplane) : sizeof(byteplane));
		
		if (!plane) {
			destroy_mimage(mask_mimg);
			return NULL;
		}
		
		plane->buffer = malloc(mask_width * mask_height * type_size);
		
		if (!plane->buffer) {
			SMART_FREE(plane);
			destroy_mimage(mask_mimg);
			return NULL;
		}
		
		if (IS_FLOAT(target_mimg))
			((fbyteplane_ptr)plane)->default_value = 255.;
		else
			plane->default_value = 255;
		
		target_mimg->planes_array[c] = plane;
	}
	set_width_and_height_mimage(target_mimg, mask_width, mask_height);
	
	for (y=0, j=0; j<data_length; j++) {

		if ((rsj = row_start[j]) < 0) {
			y -= rsj;
			continue;
		}
		
		sects = &(row_section[rsj]);
		sections_at_this_row = sections_in_row(target_mimg, j);
		
		for (i=0; i<sections_at_this_row; i++) {

			x = sects[i].starting_pixel_number;
			pixels_to_copy = sects[i].number_of_pixels;
			
			for (c=0; c<planes_number; c++) {
				src_buffer = get_byteplane_from_mimage(mimg, c)->buffer + ((y * width + x) * type_size);
				dst_buffer = get_byteplane_from_mimage(target_mimg, c)->buffer + (buffer_top * type_size);
				
				memcpy(dst_buffer, src_buffer, pixels_to_copy * type_size);
			}
			
			sects[i].position_in_buffer = buffer_top;
			buffer_top += pixels_to_copy;
		}

		y ++;

	}
 
 
	target_mimg->buffer_length = buffer_top;

	trim_sparse(target_mimg);

	return target_mimg;
}

/**************************************************************************************
/*	Function:		create_sparse_mimage
/*	Description:	create a sparse image of the requested features
/*
/*	Parameters:
/*		<-	width			width of the image
/*		<-	height			height of the image
/*		<-	planes_number	number of planes
/*		<-	kind			kind of image (float or discrete)
/*		<-	section_length	size of the section buffer data
/*		<-	data_length		size of the row data buffer
/*		<-	buffer_length	size of the plane buffers
/*
/*	Result:
/*		not NULL	a pointer to the new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr create_sparse_mimage(int width, int height, int planes_number, int kind, int section_length, int data_length, int buffer_length)
{
	int err;
	mimage_ptr target_mimg;

	target_mimg = create_mimage(width, height, planes_number, kind, SPARSE_IMAGE);

	if (!target_mimg)
		return NULL;
	
	err = pre_setup_sparse_mimage_no_buffer_allocated(target_mimg, section_length, data_length);
	if (err) { 
		destroy_mimage(target_mimg); 
		return NULL; 
	}
	
	err = allocate_buffers_for_sparse_mimage(target_mimg, kind, buffer_length);
	if (err) { 
		destroy_mimage(target_mimg); 
		return NULL; 
	}

	return target_mimg;
}

/**************************************************************************************
/*	Function:		lut_mimage_sparse
/*	Description:	apply a Look Up Table to a new sparse image
/*
/*	Parameters:
/*		<-	mimg			source image
/*		<-	LUTs			array of LUT (256 values * planes_number)
/*		<-	planes_number	number of planes
/*		<-	target_kind		kind of image (float or discrete)
/*
/*	Result:
/*		not NULL	a pointer to the new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr lut_mimage_sparse(mimage_ptr mimg, mpixel *LUTs[], int planes_number, int target_kind)
{
	int i, c, err;
	int pixels_per_plane = 0;
	int width, height, kind;
	fmpixel *FLUT;
	mimage_ptr target_mimg;
	byteplane_ptr plane;
	
	get_mimage_info(mimg, &height, &width, NULL, &kind, NULL);

		/*
		 *	Create the target sparse image
		 */

	target_mimg = create_sparse_mimage(
			width, height, planes_number, 
			target_kind, 
			mimg->section_length, 
			mimg->data_length, 
			mimg->buffer_length);

	if (!target_mimg) 
		return NULL;

		/*
		 *	Copy the buffers now
		 */

	memcpy(target_mimg->row_start, mimg->row_start, mimg->data_length * sizeof(int));
	memcpy(target_mimg->row_section, mimg->row_section, mimg->section_length * sizeof(section_data));

	plane = get_byteplane_from_mimage(mimg, 0);

	if (IS_DISCRETE(target_mimg)) {
		
		for (c=0; c<planes_number; c++) {

			mpixel_ptr buffer = plane->buffer;

			byteplane_ptr dest_plane = get_byteplane_from_mimage(target_mimg, c);
			mpixel_ptr dest_buffer = dest_plane->buffer;	/* get the buffer for this plane */

			dest_plane->default_value = LUTs[c][plane->default_value];

			for (i=0; i<pixels_per_plane; i++) {

				*dest_buffer = LUTs[c][*buffer];	/* apply the LUT to the value */
				
				dest_buffer++;
				buffer++;							/* go to next mpixel */
			}
		}
	} else {
		
		for (c=0; c<planes_number; c++) {

			mpixel_ptr buffer = plane->buffer;

			fbyteplane_ptr fdest_plane = get_fbyteplane_from_mimage(target_mimg, c);
			fmpixel_ptr fdest_buffer = fdest_plane->buffer;	/* get the buffer for this plane */

			FLUT = (fmpixel *)LUTs[c];

			fdest_plane->default_value = FLUT[plane->default_value];

			for (i=0; i<pixels_per_plane; i++) {

				*fdest_buffer = FLUT[*buffer];				/* apply the LUT to the value */

				fdest_buffer++;
				buffer++;									/* go to next mpixel */
			}
		}
	}

	return target_mimg;
}

/**************************************************************************************
/*	Function:		lex_minof_mimage_sparse
/*	Description:	return the position of the lexicographically
/*					minimum of a sparse image
/*
/*	Parameters:
/*		<-	mimg	source sparse image
/*		->	pos		position of the pixel that is minimum
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int lex_minof_mimage_sparse(mimage_ptr mimg, gr_image_displacement *pos)
{
	void *def_ptr;
	int c, sect_idx, row_idx, i;
	int *d;
	int current_row = 0;
	int last_sect_per_row;
	
	size_t number_of_pixels;
	int position_in_buffer;

	int width, height, planes_number, kind;
	unsigned char density;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);
	
	if (density != SPARSE_IMAGE)
	{
		mimg_err_no = kGrDensityMismatch;
		return -1;
	}
	
	def_ptr = (void *)malloc(max(sizeof(mpixel), sizeof(fmpixel)) * planes_number);
	if (!def_ptr)
		return -2;
			
	d = get_planes_distances_for_mimage(mimg);
	if (!d) { SMART_FREE(def_ptr); return -3; }
	
	if (kind == DISCRETE_IMAGE) {

		mpixel *pix_min;
		mpixel *defpix_min = def_ptr;
		mpixel_ptr buffer_start, buffer_ptr;
	
			/* store in a pointer the default value */
		for (c=0; c<(planes_number - 1); c++) 
			defpix_min[c] = get_byteplane_from_mimage(mimg, c)->default_value;

		if (mimg->buffer_length) {
			pix_min = get_byteplane_from_mimage(mimg, 0)->buffer;	/* get the value of the first mpixel */
		
			buffer_start = get_byteplane_from_mimage(mimg, c)->buffer;
			
			for ( row_idx=0 ; row_idx<mimg->data_length ; row_idx++) { 

				if ((sect_idx = mimg->row_start[row_idx]) < 0) {
					current_row -= sect_idx;
					continue;
				}
				
				last_sect_per_row = sect_idx + sections_in_row(mimg, row_idx);
	
				for ( ; sect_idx<last_sect_per_row ; sect_idx++ )
				{
					number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
					position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

					if ( position_in_buffer < 0 )
						continue;		/* deafult case is handled at the end */
					else 
					{
						buffer_ptr = buffer_start + position_in_buffer;
						for ( i=0; i<number_of_pixels ; i++ )
						{
							if (lex_pixels_compare(pix_min, buffer_ptr, planes_number, d) > 0) {
								pix_min = buffer_ptr;
								pos->x = mimg->row_section[sect_idx].starting_pixel_number + i;
								pos->y = current_row;
							}
							buffer_ptr++;
						}
					}
				}
				
				current_row++;
			}	
				
				/* compare now the obtained value with the default one */			
			if (lex_pixels_default_compare(pix_min, defpix_min, planes_number, d) > 0) {
				pos->x = -1;	/* this convention means the default value is minimum */
				pos->y = -1;
			}
			
		} else {
			pos->x = -1;	/* this convention means the default value is minimum */
			pos->y = -1;
		}
	} else {
		
		fmpixel *fpix_min;
		fmpixel *fdefpix_min = def_ptr;
		fmpixel_ptr fbuffer_start, fbuffer_ptr;
	
			/* store in a pointer the default value */
		for (c=0; c<(planes_number - 1); c++) 
			fdefpix_min[c] = get_byteplane_from_mimage(mimg, c)->default_value;

		if (mimg->buffer_length) {
			fpix_min = get_fbyteplane_from_mimage(mimg, 0)->buffer;	/* get the value of the first mpixel */
		
			fbuffer_start = get_fbyteplane_from_mimage(mimg, c)->buffer;
			
			for ( row_idx=0 ; row_idx<mimg->data_length ; row_idx++) { 

				if ((sect_idx = mimg->row_start[row_idx]) < 0) {
					current_row -= sect_idx;
					continue;
				}
				
				last_sect_per_row = sect_idx + sections_in_row(mimg, row_idx);
	
				for ( ; sect_idx<last_sect_per_row ; sect_idx++ )
				{
					number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
					position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

					if ( position_in_buffer < 0 )
						continue;		/* deafult case is handled at the end */
					else 
					{
						fbuffer_ptr = fbuffer_start + position_in_buffer;
						for ( i=0; i<number_of_pixels ; i++ )
						{
							if (lex_fpixels_compare(fpix_min, fbuffer_ptr, planes_number, d) > 0) {
								fpix_min = fbuffer_ptr;
								pos->x = mimg->row_section[sect_idx].starting_pixel_number + i;
								pos->y = current_row;
							}
							fbuffer_ptr++;
						}
					}
				}
				
				current_row++;
			}	
				
				/* compare now the obtained value with the default one */			
			if (lex_fpixels_default_compare(fpix_min, fdefpix_min, planes_number, d) > 0) {
				pos->x = -1;	/* this convention means the default value is minimum */
				pos->y = -1;
			}
			
		} else {
			pos->x = -1;	/* this convention means the default value is minimum */
			pos->y = -1;
		}
	}

	SMART_FREE(d);
	SMART_FREE(def_ptr);
	
	return 0;
}

/**************************************************************************************
/*	Function:		lex_maxof_mimage_sparse
/*	Description:	return the position of the lexicographically
/*					maximum of a sparse image
/*
/*	Parameters:
/*		<-	mimg	source sparse image
/*		->	pos		position of the pixel that is maximum
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int lex_maxof_mimage_sparse(mimage_ptr mimg, gr_image_displacement *pos)
{
	void *def_ptr;
	int c, sect_idx, row_idx, i;
	int *d;
	int current_row = 0;
	int last_sect_per_row;
	
	size_t number_of_pixels;
	int position_in_buffer;

	int width, height, planes_number, kind;
	unsigned char density;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, &density);
	
	if (density != SPARSE_IMAGE)
	{
		mimg_err_no = kGrDensityMismatch;
		return -1;
	}
	
	def_ptr = (void *)malloc(max(sizeof(mpixel), sizeof(fmpixel)) * planes_number);
	if (!def_ptr)
		return -2;
			
	d = get_planes_distances_for_mimage(mimg);
	if (!d) { SMART_FREE(def_ptr); return -3; }
			
	if (kind == DISCRETE_IMAGE) {

		mpixel *pix_max;
		mpixel *defpix_max = def_ptr;
		mpixel_ptr buffer_start, buffer_ptr;
	
			/* store in a pointer the default value */
		for (c=0; c<(planes_number - 1); c++) 
			defpix_max[c] = get_byteplane_from_mimage(mimg, c)->default_value;

		if (mimg->buffer_length) {
			pix_max = get_byteplane_from_mimage(mimg, 0)->buffer;	/* get the value of the first mpixel */
		
			buffer_start = get_byteplane_from_mimage(mimg, c)->buffer;
			
			for ( row_idx=0 ; row_idx<mimg->data_length ; row_idx++) { 

				if ((sect_idx = mimg->row_start[row_idx]) < 0) {
					current_row -= sect_idx;
					continue;
				}
				
				last_sect_per_row = sect_idx + sections_in_row(mimg, row_idx);
	
				for ( ; sect_idx<last_sect_per_row ; sect_idx++ )
				{
					number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
					position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

					if ( position_in_buffer < 0 )
						continue;		/* deafult case is handled at the end */
					else 
					{
						buffer_ptr = buffer_start + position_in_buffer;
						for ( i=0; i<number_of_pixels ; i++ )
						{
							if (lex_pixels_compare(pix_max, buffer_ptr, planes_number, d) > 0) {
								pix_max = buffer_ptr;
								pos->x = mimg->row_section[sect_idx].starting_pixel_number + i;
								pos->y = current_row;
							}
							buffer_ptr++;
						}
					}
				}
				
				current_row++;
			}	
				
				/* compare now the obtained value with the default one */			
			if (lex_pixels_default_compare(pix_max, defpix_max, planes_number, d) > 0) {
				pos->x = -1;	/* this convention means the default value is maximum */
				pos->y = -1;
			}
			
		} else {
			pos->x = -1;	/* this convention means the default value is maximum */
			pos->y = -1;
		}
	} else {
		
		fmpixel *fpix_max;
		fmpixel *fdefpix_max = def_ptr;
		fmpixel_ptr fbuffer_start, fbuffer_ptr;
	
			/* store in a pointer the default value */
		for (c=0; c<(planes_number - 1); c++) 
			fdefpix_max[c] = get_byteplane_from_mimage(mimg, c)->default_value;

		if (mimg->buffer_length) {
			fpix_max = get_fbyteplane_from_mimage(mimg, 0)->buffer;	/* get the value of the first mpixel */
		
			fbuffer_start = get_fbyteplane_from_mimage(mimg, c)->buffer;
			
			for ( row_idx=0 ; row_idx<mimg->data_length ; row_idx++) { 

				if ((sect_idx = mimg->row_start[row_idx]) < 0) {
					current_row -= sect_idx;
					continue;
				}
				
				last_sect_per_row = sect_idx + sections_in_row(mimg, row_idx);
	
				for ( ; sect_idx<last_sect_per_row ; sect_idx++ )
				{
					number_of_pixels = mimg->row_section[sect_idx].number_of_pixels;
					position_in_buffer = mimg->row_section[sect_idx].position_in_buffer;

					if ( position_in_buffer < 0 )
						continue;		/* deafult case is handled at the end */
					else 
					{
						fbuffer_ptr = fbuffer_start + position_in_buffer;
						for ( i=0; i<number_of_pixels ; i++ )
						{
							if (lex_fpixels_compare(fpix_max, fbuffer_ptr, planes_number, d) > 0) {
								fpix_max = fbuffer_ptr;
								pos->x = mimg->row_section[sect_idx].starting_pixel_number + i;
								pos->y = current_row;
							}
							fbuffer_ptr++;
						}
					}
				}
				
				current_row++;
			}	
				
				/* compare now the obtained value with the default one */			
			if (lex_fpixels_default_compare(fpix_max, fdefpix_max, planes_number, d) > 0) {
				pos->x = -1;	/* this convention means the default value is maximum */
				pos->y = -1;
			}
			
		} else {
			pos->x = -1;	/* this convention means the default value is maximum */
			pos->y = -1;
		}
	}

	SMART_FREE(d);
	SMART_FREE(def_ptr);
	
	return 0;
}

/**************************************************************************************
/*	Function:		lex_sort_mimage
/*	Description:	make a lexicographic sort of the elements 
/*					in the buffer of the sparse image
/*	Note:			the default values won't be sorted in this case,
/*					causing when the image is displayed holes in the
/*					pixels sequence
/*
/*	Parameters:
/*		<-	mimg	source image
/*
/*	Result:
/*		not NULL	the sorted image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr lex_sort_mimage_sparse(mimage_ptr mimg)
{
	int c, i;
	mimage_ptr res_mimg = NULL;

	int pixels_number;
	int *pixel_no_array;
	int height, width, planes_number, kind;
	
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

	if (!(res_mimg = create_sparse_mimage(width, height, planes_number, kind, mimg->section_length, mimg->data_length, pixels_number = mimg->buffer_length)))	goto end_lex_sort;
	if (!(pixel_no_array = generate_pixel_no_array(pixels_number)))	goto end_lex_sort;

	memcpy(res_mimg->row_start, mimg->row_start, res_mimg->data_length * sizeof(*mimg->row_start));
	memcpy(res_mimg->row_section, mimg->row_section, res_mimg->section_length * sizeof(*mimg->row_section));
		
	sort_by_pvals(pixel_no_array, mimg, pixels_number);				

	/*
	 *	now in pixel_no_array we have the sorted array of indices
	 *	of the pixels as they should be in the result image
	 */

		/* loop through the planes */
	for (c=0; c<planes_number; c++) {

		if (IS_DISCRETE(mimg)) {

			mpixel_ptr src_buffer = get_byteplane_from_mimage(mimg, c)->buffer;
			mpixel_ptr dst_buffer = get_byteplane_from_mimage(res_mimg, c)->buffer;

			for (i=0; i<pixels_number; i++) 
				dst_buffer[i] = src_buffer[pixel_no_array[i]];
		} else {

			fmpixel_ptr fsrc_buffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
			fmpixel_ptr fdst_buffer = get_fbyteplane_from_mimage(res_mimg, c)->buffer;

			for (i=0; i<pixels_number; i++) 
				fdst_buffer[i] = fsrc_buffer[pixel_no_array[i]];
		}
	}
	
end_lex_sort:

	SMART_FREE(pixel_no_array);
	return res_mimg;
}

/**************************************************************************************
/*	Function:		count_numbers_of_colors_sparse
/*	Description:	Return the number of colors in the buffers of a sparse image. 
/*	Note:			the default_value is not counted and that it may or not be 
/*					inside the buffer
/*
/*	Parameters:
/*		<-	mimg	source image
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int count_numbers_of_colors_sparse(mimage_ptr mimg)
{
	int i, cmp;
	int *pixel_no_array, *d;

	int number = 0;
	int planes_number = GET_PLANES_NUMBER(mimg);
	int buffer_length = mimg->buffer_length;

	if (!(d = get_planes_distances_for_mimage(mimg)))
		goto end_count_colors;

	if (!(pixel_no_array = generate_pixel_no_array(buffer_length)))	
		goto end_count_colors;

	sort_by_pvals(pixel_no_array, mimg, buffer_length);				
	
	number = 1;
	if (IS_DISCRETE(mimg)) {
		mpixel_ptr buffer = get_byteplane_from_mimage(mimg, 0)->buffer;
		
		for (i=0; i<buffer_length-1; i++) {
			cmp = lex_pixels_compare(buffer+pixel_no_array[i], buffer+pixel_no_array[i+1], planes_number, d);
			if (cmp != 0)
				number ++;
		}
	} else {
		fmpixel_ptr fbuffer = get_fbyteplane_from_mimage(mimg, 0)->buffer;
		
		for (i=0; i<buffer_length-1; i++) {
			cmp = lex_fpixels_compare(fbuffer+pixel_no_array[i], fbuffer+pixel_no_array[i+1], planes_number, d);
			if (cmp != 0)
				number ++;
		}
	}

end_count_colors:

	SMART_FREE(d);
	SMART_FREE(pixel_no_array);
	return number;
}

/**************************************************************************************
/*	Function:		stuff_sparse_over_dense
/*	Description:	Stuff the mimg_sparse in the mimg_dense at the offset delta
/*
/*	Parameters:
/*		<-	mimg_sparse		source sparse image
/*		<->	mimg_dense		destination dense image
/*		<-	delta			offset where to store the sparse image
/*
/*	Result:
/*		0		ok
/*		else	error condition
/*
/***************************************************************************************/

int stuff_sparse_over_dense(mimage_ptr mimg_sparse, mimage_ptr mimg_dense, gr_image_displacement *delta)
{
	int c;		
	int pix;
	int dest_line_offset;
	int dest_pixel_offset;

	section_data *sect_it;
	size_t sects_in_row;
	size_t data_index, ix, section_index, data_length;

	mimage_ptr dest_mimg = NULL;
	
	int width_sparse, height_sparse, planes_number_sparse, kind_sparse;
	unsigned char density_sparse;

	int width_dense, height_dense, planes_number_dense, kind_dense;
	unsigned char density_dense;

	int width, height, planes_number, kind;

	get_mimage_info(mimg_sparse, &height_sparse, &width_sparse, &planes_number_sparse, &kind_sparse, &density_sparse);
	get_mimage_info(mimg_dense, &height_dense, &width_dense, &planes_number_dense, &kind_dense, &density_dense);

	if (kind_sparse != kind_dense) {
		mimg_err_no = kGrKindNotMatch;
		return NULL;
	}

	if (density_sparse != SPARSE_IMAGE && density_dense != DENSE_IMAGE) 
	{
		mimg_err_no = kGrNotCorrectTypes;
		return kGrNotCorrectTypes;
	}

	if ( (delta->y > height_dense) || (delta->x > width_dense) )
	{
		mimg_err_no = kGrNotCorrectSizes;
		return kGrNotCorrectSizes;
	}
	
	if ( planes_number_sparse != planes_number_dense)
	{
		mimg_err_no = kGrPlanesNotMatch;
		return kGrPlanesNotMatch;
	}

	/*
	 *	initialize the variable
	 */

	width=width_dense; 
	height=height_dense; 
	planes_number=planes_number_dense; 
	kind=kind_dense;

	dest_mimg = mimg_dense;

	/*
	 *	do the real computation now
	 */

	if (kind == DISCRETE_IMAGE)
	{
		mpixel_ptr buffer_sparse, buffer_sparse_ptr;
		mpixel_ptr dest_buffer, dest_buffer_ptr;
		mpixel	default_value;

		data_length = mimg_sparse->data_length;
		dest_line_offset = delta->y;

		for ( data_index = 0 ; data_index < mimg_sparse->data_length ; data_index ++ )
		{
			int section_data_for_this_row = mimg_sparse->row_start[data_index];
		
			if ( section_data_for_this_row < 0 ) 
			{
				dest_line_offset += -section_data_for_this_row;
				continue;
			}
			
			sects_in_row = sections_in_row(mimg_sparse, data_index);
			
			for ( c=0; c<planes_number ; c++)
			{
				dest_buffer = get_plane_buffer(dest_mimg->planes_array[c], kind);
				buffer_sparse = get_plane_buffer(mimg_sparse->planes_array[c], kind);
		
				default_value = get_byteplane_from_mimage(mimg_sparse, c)->default_value;
			
				sect_it = mimg_sparse->row_section + section_data_for_this_row;
			
				for ( ix=0 ; ix<sects_in_row ; ix++ )
				{
					int start_to_copy;
					int length_to_copy;
					
					if ((start_to_copy = delta->x + sect_it->starting_pixel_number) > width_dense)
						continue;
				
					if ((start_to_copy + sect_it->number_of_pixels) > width_dense)
						length_to_copy = width_dense - (start_to_copy + sect_it->number_of_pixels);
					else
						length_to_copy = sect_it->number_of_pixels;
				
					dest_pixel_offset = dest_line_offset * width + start_to_copy;
					dest_buffer_ptr = dest_buffer + dest_pixel_offset;

					if (sect_it->position_in_buffer < 0) {	/* this is a default value line */

						for ( pix=0; pix < length_to_copy; pix++)
							*dest_buffer_ptr++ = default_value;

					} else {

						buffer_sparse_ptr = buffer_sparse + sect_it->position_in_buffer;

						for ( pix=0; pix < length_to_copy ; pix++ )
							*dest_buffer_ptr++ = *buffer_sparse_ptr++;
					}
				
					sect_it++;	/* advance to next section */
				}
			}
			
			dest_line_offset ++;
			
		}
	} else {	/* FLOAT_IMAGE */
	
		fmpixel_ptr buffer_sparse, buffer_sparse_ptr;
		fmpixel_ptr dest_buffer, dest_buffer_ptr;
		fmpixel	default_value;

		data_length = mimg_sparse->data_length;
		dest_line_offset = delta->y;

		for ( data_index = 0 ; data_index < mimg_sparse->data_length ; data_index ++ )
		{
			int section_data_for_this_row = mimg_sparse->row_start[data_index];
		
			if ( section_data_for_this_row < 0 ) 
			{
				dest_line_offset += -section_data_for_this_row;
				continue;
			}
			
			sects_in_row = sections_in_row(mimg_sparse, data_index);
			
			for ( c=0; c<planes_number ; c++)
			{
				dest_buffer = get_plane_buffer(dest_mimg->planes_array[c], kind);
				buffer_sparse = get_plane_buffer(mimg_sparse->planes_array[c], kind);
		
				default_value = get_fbyteplane_from_mimage(mimg_sparse, c)->default_value;
			
				sect_it = mimg_sparse->row_section + section_data_for_this_row;
			
				for ( ix=0 ; ix<sects_in_row ; ix++ )
				{
					int start_to_copy;
					int length_to_copy;
					
					if ((start_to_copy = delta->x + sect_it->starting_pixel_number) > width_dense)
						continue;
				
					if ((start_to_copy + sect_it->number_of_pixels) > width_dense)
						length_to_copy = width_dense - (start_to_copy + sect_it->number_of_pixels);
					else
						length_to_copy = sect_it->number_of_pixels;
				
					dest_pixel_offset = dest_line_offset * width + start_to_copy;
					dest_buffer_ptr = dest_buffer + dest_pixel_offset;

					if (sect_it->position_in_buffer < 0) {	/* this is a default value line */

						for ( pix=0; pix < length_to_copy; pix++)
							*dest_buffer_ptr++ = default_value;

					} else {

						buffer_sparse_ptr = buffer_sparse + sect_it->position_in_buffer;

						for ( pix=0; pix < length_to_copy ; pix++ )
							*dest_buffer_ptr++ = *buffer_sparse_ptr++;
					}
				
					sect_it++;	/* advance to next section */
				}
			}
			
			dest_line_offset ++;
			
		}
	}

	return kGrNoErr;
}

/**************************************************************************************
/*	Function:		flatten_sparse_mimage
/*	Description:	remove all data buffer from a sparse image leaving only the
/*					default values
/*
/*	Parameters:
/*		<->	mimg	source image
/*
/*	Result:
/*		0			ok
/*		else		error condition
/*
/***************************************************************************************/

int flatten_sparse_mimage(mimage_ptr mimg)
{
	int j;
	int c;
	int planes_number = mimg->comp;
	
	byteplane_ptr *planes_array;
	section_data *sects;
	void *buffer;
	
	mimg->buffer_length = 0;
	planes_array = mimg->planes_array;
	if (!planes_array)
		return kPlanesNotFound;
	
	for (c=0; c<planes_number; c++) {

		if (planes_array[c]->buffer) {
			free(planes_array[c]->buffer);
			planes_array[c]->buffer = NULL;
		}
	}

	sects = mimg->row_section;
	
	for (j=0; j<mimg->section_length; j++, sects++ ) 
		sects->position_in_buffer = -1;			/* set to the default value */

	return 0;
}

/**************************************************************************************
/*	Function:		split_sparse_mimage
/*	Description:	split a sparse image in a set of sparse images for every color
/*
/*	Parameters:
/*		<-	mimg		source image
/*		->	retNumber	number of sparse images generated
/*		->	offsets		offset in the original image of the generated sparse images
/*		->	pixels_pop	population in pixels of the generated images
/*
/*	Result:
/*		not NULL	an array of splitted sparse images
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr *split_sparse_mimage(mimage_ptr mimg, int *retNumber, image_displacement_ptr *offsets, int **pixels_pop)
{
	int i, c;
	int def_pop;							/* used to store the default layer mpixel population */
	gr_image_displacement def_offset;		/* used to store the default layer offset */
	int height, width, planes_number, kind;

	int *d = NULL;										/* array for distances between planes */
	mimage_ptr *res_mimgs = NULL;						/* array containing returned array of pixels */

	int images_number = 0;								/* count the number of images */
	int *thePixels_pop = NULL;							/* count the pixels population of every image */
	gr_image_displacement *theOffsets = NULL;			/* offsets of the top-left corner of every image */

	gr_image_displacement *pixel_disp_array = NULL;		/* buffer 2 displacement indexes map */
	int	*pixel_no_array = NULL;							/* auxiliary array to lex sort */

	int buffer_length = mimg->buffer_length;			/* cache the default length value */
	
		
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* 
		 *	setup the image displacement array
		 */

	res_mimgs = calloc(sizeof(mimage_ptr), buffer_length + 1);
	theOffsets = calloc(sizeof(image_displacement_ptr), buffer_length + 1);
	thePixels_pop = calloc(sizeof(int), buffer_length + 1);
	
	if (!res_mimgs || !theOffsets || !thePixels_pop)
			goto end_split_mimage_sparse;

	if (buffer_length) { /* if not default values are present in this image */

		d = get_planes_distances_for_mimage(mimg);
		pixel_disp_array = generate_displacements_for_sparse_mimage(mimg);
		pixel_no_array = generate_pixel_no_array(buffer_length);

		if (!d || !pixel_disp_array || !pixel_no_array) 
			goto end_split_mimage_sparse;
		

			/* 
			 *	sort lexicographically the position array, taking care of the displacements
			 */	

		sort_sparse_by_pvals_and_posns(pixel_no_array, pixel_disp_array, mimg);
		
			/* now every data is ready and we can start generating the images */
		if (IS_DISCRETE(mimg) || IS_FLOAT(mimg)) {	/* work for both the cases */

			int j;
			
			section_data *sect;

			int first_i_for_image, last_i_for_image;
			int first_j_for_section, last_j_for_section;
			
			int row_idx, sect_idx;
			int first_sect_for_row;
			int sects_no = 1;	/* minimum one section per image */
			
			int min_x = width, max_x = 0;
			int min_y = height, max_y = 0;
			
			int cmp;
			mpixel_ptr first_buffer = get_byteplane_from_mimage(mimg, 0)->buffer;
			fmpixel_ptr first_fbuffer = get_fbyteplane_from_mimage(mimg, 0)->buffer;	

				/*
				 *	loop over the pixels, storing information for found images, 
				 *	and generate all of them
				 */	

			first_i_for_image = 0;
			first_j_for_section = 0;

			min_x = min(pixel_disp_array[pixel_no_array[0]].x, width);
			max_x = max(pixel_disp_array[pixel_no_array[0]].x, 0);

			min_y = min(pixel_disp_array[pixel_no_array[0]].y, height);
			max_y = max(pixel_disp_array[pixel_no_array[0]].y, 0);

			for (i=1; i <= buffer_length; i++) {

					/* note that we loop over all pixels plus one */
					/* this is because we save the image just when */
					/* a new one is presented, in this case when we */
					/* arrive at the end save the last image as it is*/

				int pixel_no_at_i;
				int pixel_no_at_previous_i = pixel_no_array[i-1];

				if (i < buffer_length) {

						/* if this is not the last image case */
						/* get the information for the pixel_no_at_i */			
					pixel_no_at_i = pixel_no_array[i];
					
					if (IS_DISCRETE(mimg)) {
						cmp = lex_pixels_compare(first_buffer + 
							pixel_no_at_i, first_buffer + pixel_no_at_previous_i, planes_number, d);

					} else {
						cmp = lex_fpixels_compare(first_fbuffer + 
							pixel_no_at_i, first_fbuffer + pixel_no_at_previous_i, planes_number, d);

					}
				} else 
					cmp = 1;		/* if this is the last image case, force to save the image, */
									/* without compare to any other */


				if (cmp != 0) {	/* new color found, generate a new image */

					mimage_ptr mimg_for_color;

					last_i_for_image = i-1;
					
					mimg_for_color = create_sparse_mimage(
						max_x - min_x + 1,	/* width */
						max_y - min_y + 1,	/* height */
						planes_number,
						kind,
						sects_no,
						max_y - min_y + 1,	/* data_lenght (for xss) */
						0					/* buffer_length (no buffer needed) */
					);

					if (!mimg_for_color) {
						int k;
						
						for (k=0; k<images_number ; k++) 
							destroy_mimage(res_mimgs[k]);

						images_number = 0;
							
						goto end_split_mimage_sparse;
					}
					
						/*
						 *	set the default color for the 
						 *	generated image as a representant 
						 *	of the source image 
						 */
					for (c=0; c<planes_number; c++) 
						mimg_for_color->planes_array[c]->default_value = 
							*(mimg->planes_array[c]->buffer + pixel_no_array[first_i_for_image]);
					
					res_mimgs[images_number] = mimg_for_color;
					thePixels_pop[images_number] = last_i_for_image - first_i_for_image + 1;
					theOffsets[images_number].x = min_x;
					theOffsets[images_number].y = min_y;
					
					images_number++;
					
					sect = mimg_for_color->row_section;

					sect_idx = 0;
					row_idx = 0;
					first_j_for_section = first_i_for_image;
					first_sect_for_row = 0;

					for (j=first_i_for_image; j<=last_i_for_image; j++) {

						int pixel_no_at_j = pixel_no_array[j];
						
						if (j>first_i_for_image) {	/* second loop */

							int jumped_lines;
							int pixel_no_at_previous_j = pixel_no_array[j - 1];
							
								/* 
								 *	if the sequence is interrupted or we are in different lines 
								 */
							if ((pixel_disp_array[pixel_no_at_j].x != pixel_disp_array[pixel_no_at_previous_j].x + 1) ||
								(pixel_disp_array[pixel_no_at_j].y != pixel_disp_array[pixel_no_at_previous_j].y)) {

								last_j_for_section = j-1;
								
								sect->starting_pixel_number = pixel_disp_array[pixel_no_array[first_j_for_section]].x - min_x;
								sect->number_of_pixels = 
									pixel_disp_array[pixel_no_array[last_j_for_section]].x - min_x -
									sect->starting_pixel_number + 1;
								sect->position_in_buffer = -1;	/* always default planes are generated */

								sect_idx++;

									/* if we are jumping some line */								
								if ((jumped_lines = pixel_disp_array[pixel_no_at_j].y 
									- pixel_disp_array[pixel_no_at_previous_j].y) != 0) {

									mimg_for_color->row_start[row_idx++] = first_sect_for_row;
									first_sect_for_row = sect_idx;

									if (jumped_lines > 1)
										mimg_for_color->row_start[row_idx++] = -(jumped_lines - 1);

								}
								
								first_j_for_section = j;
								
								sect++;	/* advance to next section */
								
								continue;

							}	
						} 
					}
					
						/* 
						 *	if this is last section for this image
						 *	we can save the section 
						 */
						 
					last_j_for_section = j - 1;
					
					sect->starting_pixel_number = pixel_disp_array[pixel_no_array[first_j_for_section]].x - min_x;
					sect->number_of_pixels = 
						pixel_disp_array[pixel_no_array[last_j_for_section]].x - min_x -
							sect->starting_pixel_number + 1;
					sect->position_in_buffer = -1;	/* always default planes are generated */
					
					sect_idx++;

					mimg_for_color->row_start[row_idx++] = first_sect_for_row;
			

						/*
						 *	finish the image
						 */

					mimg_for_color->data_length = row_idx;
					mimg_for_color->section_length = sect_idx;
					
					trim_sparse(mimg_for_color);	/* remove superfluous data */
					
						/* reset values for next image */
					min_x = min(pixel_disp_array[pixel_no_at_i].x, width);
					max_x = max(pixel_disp_array[pixel_no_at_i].x, 0);

					min_y = min(pixel_disp_array[pixel_no_at_i].y, height);
					max_y = max(pixel_disp_array[pixel_no_at_i].y, 0);

					sects_no = 1;
					
					first_i_for_image = i;
					
					continue;
				} else {
					min_x = min(pixel_disp_array[pixel_no_at_i].x, min_x);
					max_x = max(pixel_disp_array[pixel_no_at_i].x, max_x);

					min_y = min(pixel_disp_array[pixel_no_at_i].y, min_y);
					max_y = max(pixel_disp_array[pixel_no_at_i].y, max_y);
				
						/* 
						 *	if the sequence is interrupted or we are in different lines 
						 */
					if ((pixel_disp_array[pixel_no_at_i].x != pixel_disp_array[pixel_no_at_previous_i].x + 1) ||
						(pixel_disp_array[pixel_no_at_i].y != pixel_disp_array[pixel_no_at_previous_i].y)) {
						
						sects_no ++;
					} 
				}
			}	

		} /* DISCRETE_MIMAGE */
	}

	/* save now the default image */

	res_mimgs[images_number] = clone_default_data_sparse(mimg, &def_pop, &def_offset);
	if (!res_mimgs[images_number]) {
		int k;
		
		for (k=0; k<images_number ; k++) 
			destroy_mimage(res_mimgs[k]);

		images_number = 0;
			
		goto end_split_mimage_sparse;
	}
	
	if (res_mimgs[images_number]->section_length == 0)	/* if the image is empty */
		destroy_mimage(res_mimgs[images_number]);
	else {
		thePixels_pop[images_number] = def_pop;
		images_number ++;
	}
	
end_split_mimage_sparse:

	if (images_number && theOffsets && thePixels_pop) {
		realloc(res_mimgs, images_number * sizeof(mimage_ptr));	/* shrink to the real number of images */
		realloc(theOffsets, images_number * sizeof(gr_image_displacement));
		realloc(thePixels_pop, images_number * sizeof(int));
	} else {

		if (res_mimgs) {
			free (res_mimgs);
			res_mimgs = NULL;
		}

		if (theOffsets) {
			free (theOffsets);
			theOffsets = NULL;
		}

		if (thePixels_pop) {
			free (thePixels_pop);
			theOffsets = NULL;
		}
	}

	if (d) free(d);
	if (pixel_no_array) free(pixel_no_array);
	if (pixel_disp_array) free(pixel_disp_array);

	if (offsets)	*offsets = theOffsets;
	if (retNumber)	*retNumber = images_number;
	if (pixels_pop)	*pixels_pop = thePixels_pop;

	return res_mimgs;
}

/**************************************************************************************
/*	Function:		clone_default_data_sparse
/*	Description:	clone the only default values if a sparse image
/*
/*	Parameters:
/*		<- mimg				source image
/*		-> pop				population copied
/*		-> offset			offset
/*
/*	Result:
/*		not NULL	a pointer to the new image
/*		NULL		out of memory
/*
/***************************************************************************************/

mimage_ptr clone_default_data_sparse(mimage_ptr mimg, int *pop, gr_image_displacement *offset) 
{
	int i, j, c;

	int height, width, planes_number, kind;
	
	int data_length = mimg->data_length;
	int section_length = mimg->section_length;
	
	int max_width = 0;
	int the_pop = 0;
	int inserted_sections = 0;
	int target_row_idx = 0, section_row_idx = 0;
	
	mimage_ptr target_mimg;

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);
	
	target_mimg = create_sparse_mimage(width, height, planes_number, kind, mimg->section_length, mimg->data_length, 0);
	if (!target_mimg)
		return NULL;
	
	offset->x = width;
	offset->y = 0;

	for (i=0; i<data_length; i++) {

		int first_section_per_row;
		int last_section_per_row;

		if (mimg->row_start[i] < 0) {
			
			if (target_row_idx) {
				if (target_mimg->row_start[target_row_idx - 1] < 0)
					target_mimg->row_start[target_row_idx - 1] += mimg->row_start[i];
				else {
					target_mimg->row_start[target_row_idx] = mimg->row_start[i];
					target_row_idx ++;
				}
			} else {
				target_mimg->row_start[target_row_idx] = mimg->row_start[i];
				target_row_idx ++;
			}
			
			continue;
		}
			
		first_section_per_row = mimg->row_start[i];
		last_section_per_row = first_section_per_row + sections_in_row(mimg, i);
		
		inserted_sections = 0;

		for ( j=first_section_per_row ; j<last_section_per_row ; j++) {
		
			if (mimg->row_section[j].position_in_buffer < 0) {	/* this is a default value */

				the_pop += (mimg->row_section + j)->number_of_pixels;
				offset->x = min(offset->x, (mimg->row_section + j)->starting_pixel_number);
				max_width = max(max_width, (mimg->row_section + j)->starting_pixel_number + (mimg->row_section + j)->number_of_pixels);
				
				memcpy(target_mimg->row_section + section_row_idx, mimg->row_section + j, sizeof(section_data));
				inserted_sections ++;
				section_row_idx ++;
			}
		}
		
		if (!inserted_sections) {

			if (target_row_idx) {
				if (target_mimg->row_start[target_row_idx - 1] < 0)
					target_mimg->row_start[target_row_idx - 1] += -1;
				else {
					target_mimg->row_start[target_row_idx] = -1;
					target_row_idx ++;
				}
			} else {
				target_mimg->row_start[target_row_idx] = -1;
				target_row_idx ++;
			}
		} else {
			target_mimg->row_start[target_row_idx] = section_row_idx - inserted_sections;
			target_row_idx ++;

		}
	}
	
		/* clone the default value */
	for (c=0; c<planes_number; c++)
		target_mimg->planes_array[c]->default_value = mimg->planes_array[c]->default_value;

	target_mimg->data_length = target_row_idx;
	target_mimg->section_length = section_row_idx;

		/* offset x */
	for (i=0; i<target_mimg->section_length; i++)
		target_mimg->row_section[i].starting_pixel_number -= offset->x;

		/* offset y and remove extra lines */
	if (target_mimg->data_length) {
		if (target_mimg->row_start[0] < 0) {
		
			offset->y = -target_mimg->row_start[0];
			height += target_mimg->row_start[0];	/* reduce the height */
			
			for (i=1; i<target_mimg->data_length; i++)
				target_mimg->row_start[i-1] = target_mimg->row_start[i];
			
			target_mimg->data_length--;
		}
		
			/* cut off extra empty lines */
		if (target_mimg->data_length)
			if (target_mimg->row_start[target_mimg->data_length - 1] < 0) {
				height += target_mimg->row_start[target_mimg->data_length - 1];	/* reduce the height */
				target_mimg->data_length--;
			}				
	}

	set_width_and_height_mimage(target_mimg, max_width, height);	/* set the new image size */
	trim_sparse(target_mimg);

	if (pop) *pop = the_pop;
	
	return target_mimg;
}