/**************************************************************************************
/* Filename:	gr_sparse_utils.c
/*		Copyright  1998-99 Giuseppe Di Mauro. All rights reserved.
/*
/* Description:	utilities for sparse byteplanes images
/*
/***************************************************************************************/

#include <string.h>
#include "gr_sparse_utils.h"

/**************************************************************************************
/*	Function:		sections_in_row
/*	Description:	returns the number of sections in the k-th row_start 
/*					row of an image, assuming that the row is not empty
/*
/*	Parameters:
/*		<- mimg				pointer to the image
/*		<- k				index in row_start pointing to the required row
/*
/*	Result:
/*		the number of rows
/*
/***************************************************************************************/

int sections_in_row(mimage_ptr mimg, int k)		/* get number of sections in k-th row	*/
{
	size_t num_sections_for_this_row;

	if (k < mimg->data_length - 1) 
	{
		if (mimg->row_start[k+1] >= 0) 
		{
			num_sections_for_this_row = mimg->row_start[k+1] - mimg->row_start[k];
		}
		else 
		{
			if (k+2 < mimg->data_length) 
			{
				num_sections_for_this_row = mimg->row_start[k+2] - mimg->row_start[k];
			}
			else
			{
				num_sections_for_this_row = mimg->section_length - mimg->row_start[k];
			}
		}
	}
	else
	{
		num_sections_for_this_row = mimg->section_length - mimg->row_start[k];
	}
	
	return num_sections_for_this_row;
}

/**************************************************************************************
/*	Function:		allocate_sparse_buffers
/*	Description:	allocate memory for sparse buffers for a sparse image
/*
/*	Parameters:
/*		<->	mimg				pointer to the image
/*		<-	data_length			required size for row_start buffer
/*		<-	section_length		required size for section_data buffer
/*		<-	buffer_length		required size for pixels buffer
/*
/*	Result:
/*		0		error condition
/*		1		ok
/*
/***************************************************************************************/

int allocate_sparse_buffers(mimage_ptr mimg, int data_length, int section_length, int buffer_length)
{
	int c;
	
		/* allocate memory for result image */
	if (data_length) {

		mimg->row_start = (int *) malloc( data_length * sizeof(int) );
		if (!mimg->row_start)
			return 0;
	}

	if (section_length) {
	
		mimg->row_section = (section_data_ptr) malloc( section_length * sizeof(section_data) );
		if (!mimg->row_section)
			return 0;
	}

	mimg->data_length = data_length;
	mimg->section_length = section_length;

	if (buffer_length) {

		for (c=0; c<mimg->comp ; c++)
		{
			byteplane_ptr plane = get_byteplane_from_mimage(mimg, c);
			plane->buffer = malloc(buffer_length * GET_TYPE_SIZE(mimg->kind));
			
			if (!plane->buffer) 
				return 0;
		}
	}
	
	mimg->buffer_length = buffer_length;

	return 1;
}

/**************************************************************************************
/*	Function:		expand_table_on_need
/*	Description:	Expand a dynamic table if needed
/*
/*	Parameters:
/*		<->	table		pointer to the table
/*		<-	index		index required	
/*		<->	length		lenght of the dynamic table
/*		<-	type_size	size of the type
/*
/*	Result:
/*		0		if ok
/*		-1		error condition
/*
/***************************************************************************************/

int expand_table_on_need(void **table, int index, int *length, short type_size)
{
	void *temp_table;
	
	if (index >= *length) {
		temp_table = realloc(*table, type_size * (*length <<= 1));
		if (!*table)
			return -1;

		*table = temp_table;
	}
	
	return 0;
}

/**************************************************************************************
/*	Function:		shrink_table_to_size
/*	Description:	shrink a  dynamic table to required size
/*
/*	Parameters:
/*		<->	table		pointer to the table
/*		<-	length		required lenght for the dynamic table
/*		<-	type_size	size of the type
/*
/*	Result:
/*		0		if ok
/*		-1		error condition
/*
/***************************************************************************************/

int shrink_table_to_size(void **table, int length, short type_size)
{
	void *temp_table;
	temp_table = realloc(*table, type_size * length);

#ifdef PARANOIC
	if (temp_table != *table)
		return -1;
#endif	
	
	return 0;
}

/**************************************************************************************
/*	Function:		set_default_plane_color_smimage
/*	Description:	Set the default color in a plane of a discrete image
/*
/*	Parameters:
/*		<->	mimg			pointer to the image
/*		<-	pixel_value		default value to be set
/*		<-	plane_index		index of the plane
/*
/*	Result:
/*		always FALSE
/*
/***************************************************************************************/

int set_default_plane_color_smimage(mimage_ptr mimg, mpixel pixel_value, short plane_index)
{
	byteplane_ptr *planes = mimg->planes_array;
	planes[plane_index]->default_value = pixel_value;

	return FALSE;
}

/**************************************************************************************
/*	Function:		set_default_plane_fcolor_smimage
/*	Description:	Set the default color in a plane of a float image
/*
/*	Parameters:
/*		<->	mimg			pointer to the image
/*		<-	pixel_value		float default value to be set
/*		<-	plane_index		index of the plane
/*
/*	Result:
/*		always FALSE
/*
/***************************************************************************************/

int set_default_plane_fcolor_smimage(mimage_ptr mimg, fmpixel fpixel_value, short plane_index)
{
	fbyteplane_ptr *planes = (fbyteplane_ptr *)mimg->planes_array;
	planes[plane_index]->default_value = fpixel_value;

	return FALSE;
}

/**************************************************************************************
/*	Function:		set_default_color_smimage
/*	Description:	Set the default color in a discrete image
/*
/*	Parameters:
/*		<->	mimg			pointer to the image
/*		<-	default_array	array of default values to be set
/*
/*	Result:
/*		TRUE	if ok
/*		FALSE	error condition
/*
/***************************************************************************************/

int set_default_color_smimage(mimage_ptr mimg, mpixel_ptr default_array)
{
	int c;
	int ret = FALSE;

	byteplane_ptr plane;
	byteplane_ptr *planes = mimg->planes_array;
	int planes_number = mimg->comp;
	
	if (!planes)
		return TRUE;
	
	for (c=0; c<planes_number; c++) 
	{
		plane = planes[c];
		if (!plane)
			ret = TRUE;

		plane->default_value = default_array[c];
	}
	
	return ret;
}

/**************************************************************************************
/*	Function:		set_default_fcolor_smimage
/*	Description:	Set the default color in a float imag
/*
/*	Parameters:
/*		<->	mimg			pointer to the image
/*		<-	default_array	array of default values to be set
/*
/*	Result:
/*		TRUE	if ok
/*		FALSE	error condition
/*
/***************************************************************************************/

int set_default_fcolor_smimage(mimage_ptr mimg, fmpixel_ptr fdefault_array)
{
	int c;
	int ret = FALSE;

	fbyteplane_ptr plane;
	fbyteplane_ptr *planes = (fbyteplane_ptr *)mimg->planes_array;
	int planes_number = mimg->comp;
	
	if (!planes)
		return TRUE;
	
	for (c=0; c<planes_number; c++) 
	{
		plane = planes[c];
		if (!plane)
			ret = TRUE;

		plane->default_value = fdefault_array[c];
	}
	
	return ret;
}

/**************************************************************************************
/*	Function:		trim_sparse
/*	Description:	Trim the allocation of an image to what is really needed
/*
/*	Parameters:
/*		<->	mimg			pointer to the image
/*
/*	Result:
/*		used only for debug
/*
/***************************************************************************************/

int trim_sparse(mimage_ptr mimg)
{
	int ret = 0;
	
	int c;
	int kind;
	void *buffer;
	byteplane_ptr plane;

	void *row_start = mimg->row_start;
	void *row_section = mimg->row_section;

	size_t type_size = GET_TYPE_SIZE(kind = mimg->kind);	
	
	if (row_start) {
		mimg->row_start = realloc(mimg->row_start, mimg->data_length * sizeof(int));
/*
		if (!mimg->data_length)
			mimg->row_start = NULL;
 */
	}

	if (row_section) {
		mimg->row_section = realloc(mimg->row_section, mimg->section_length * sizeof(section_data));
/*
		if (!mimg->section_length)
			mimg->row_section = NULL;
 */
	}
				
#ifdef PARANOIC_CHECK
	if ( ((row_start != mimg->row_start) || (row_section != mimg->row_section)) && mimg->section_length)
		return ret++; 
#endif	

	for (c=0; c<mimg->comp; c++) {
		plane = get_byteplane_from_mimage(mimg, c);
		buffer = plane->buffer;
		
		if (buffer) {
			plane->buffer = realloc(plane->buffer, mimg->buffer_length * type_size);

/*
			if (!mimg->buffer_length)
				plane->buffer = NULL;
 */
		}
		
#ifdef PARANOIC_CHECK
		if (buffer != plane->buffer && mimg->buffer_length)
			ret++;
#endif	
			
	}

#ifdef PARANOIC_CHECK
#ifdef macintosh
	if (ret)
		SysBeep(0);
#endif
#endif	

	return ret;	
}

/**************************************************************************************
/*	Function:		is_pixel_default_value
/*	Description:	check if a pixel value is a default value for an image
/*
/*	Parameters:
/*		<-	mimg			pointer to the image
/*		<-	value			array of values (size equals number of planes)
/*
/*	Result:
/*		1	yes
/*		0	not
/*
/***************************************************************************************/

int is_pixel_default_value(mimage_ptr mimg, void *value)
{
	int c;
	
	if (mimg->kind == DISCRETE_IMAGE)
	{
		mpixel default_value;
		
		for (c=0; c<mimg->comp; c++) {
			default_value = get_byteplane_from_mimage(mimg, c)->default_value;
			if (default_value != *((mpixel_ptr)value+c))
				return 0;
		}
	}
	else	/* FLOAT_IMAGE */
	{
		fmpixel fdefault_value;
		
		for (c=0; c<mimg->comp; c++) {
			fdefault_value = get_fbyteplane_from_mimage(mimg, c)->default_value;
			if (fdefault_value != *((fmpixel_ptr)value+c))
				return 0;
		}
	}
	
	return 1;	
}

/*
 ************************ pre_setup_sparse_mimage_no_buffer *****************************
 *
 *	Allocate preliminary buffers when number of section is equal to the numbers of lines
 *
 */

/**************************************************************************************
/*	Function:		pre_setup_sparse_mimage_no_buffer
/*	Description:	Allocate preliminary buffers when number of section 
/*					is equal to the numbers of lines
/*	Note:			Buffers are not allocated
/*
/*	Parameters:
/*		<->	mimg			pointer to the image
/*		<-	width			width of the image
/*		<-	height			height of the image
/*
/*	Result:
/*		0		ok
/*		else	error condition
/*
/***************************************************************************************/

int pre_setup_sparse_mimage_no_buffer(mimage_ptr mimg, int width, int height)
{
#ifdef __MWERKS__
#pragma unused(width)
#endif
	
	mimg->row_start = malloc(height * sizeof(int));						/* allocate enough memory for the array of row starts */

	if (!mimg->row_start)
		return -1;	/* return an error signal */

	mimg->row_section  = malloc(height * sizeof(section_data));		/* allocate enough memory for the row data array; 
																	 * since in this case each row will contain only one 
																	 * section, we need only as many section data blocks as
																	 * there are rows
																	 */
	if (!mimg->row_section) {
		free (mimg->row_start);
		mimg->row_start = NULL;
		return -2;	/* return an error signal */
	}
	
	mimg->data_length = height;
	mimg->section_length = height;
	
	return 0;
}

/**************************************************************************************
/*	Function:		pre_setup_sparse_mimage_no_buffer
/*	Description:	Allocate preliminary buffers when number of section 
/*					is not equal to the numbers of lines
/*	Note:			Buffers are not allocated
/*
/*	Parameters:
/*		<->	mimg			pointer to the image
/*		<-	num_sections	number of sections to allocate
/*		<-	nominal_height	height of the image
/*
/*	Result:
/*		0		ok
/*		else	error condition
/*
/***************************************************************************************/

int pre_setup_sparse_mimage_no_buffer_allocated(mimage_ptr mimg, int num_sections, int nominal_height)
{
	mimg->row_start = malloc(nominal_height * sizeof(int));	/* allocate enough memory for the array of row starts */

	if (!mimg->row_start && nominal_height)
		return -1;	/* return an error signal */

	mimg->row_section  = malloc(num_sections * sizeof(section_data));
						/* allocate enough memory for the section data array */
						
	if (!mimg->row_section && num_sections) {
		if (mimg->row_start) free(mimg->row_start);
		mimg->row_start = NULL;
		return -2;	/* return an error signal */
	}
	
	mimg->data_length = nominal_height;
	mimg->section_length = num_sections;
	
	return 0;
}

/**************************************************************************************
/*	Function:		pre_setup_sparse_mimage_no_buffer
/*	Description:	Perform all allocations for a new sparse image
/*					with just one section per row,	except the allocation
/*					of the basic planes array held at the logical top of the
/*					image structure.
/*
/*	Parameters:
/*		<->	mimg			pointer to the image
/*		<-	width			height of the image
/*		<-	height			height of the image
/*		<-	kind			kind of the image
/*		<-	planes_number	number of planes in the image
/*
/*	Result:
/*		0		ok
/*		else	error condition
/*
/***************************************************************************************/

int pre_setup_sparse_mimage(mimage_ptr mimg, int width, int height, int kind, int planes_number)
{
	size_t pixels_per_plane;
	byteplane_ptr *planes_array;
	int err;
	
	pixels_per_plane = height * width;	

	planes_array = mimg->planes_array;

		/* Allocate memory and check it */
	err = pre_setup_sparse_mimage_no_buffer(mimg, width, height);
	if (err)									
		return err;
								 
	return allocate_buffers_for_sparse_mimage(mimg, kind, pixels_per_plane);
}

/**************************************************************************************
/*	Function:		allocate_buffers_for_sparse_mimage
/*	Description:	Allocate the buffers for a sparse image
/*
/*	Parameters:
/*		<->	mimg			pointer to the image
/*		<-	kind			kind of buffers
/*		<-	buffer_length	lenght of the buffers
/*
/*	Result:
/*		0		ok
/*		else	error condition
/*
/***************************************************************************************/

int allocate_buffers_for_sparse_mimage(mimage_ptr mimg, int kind, int buffer_length)
{
	int c, k;
	byteplane_ptr *planes_array = mimg->planes_array;
	int planes_number = mimg->comp;
	
		/* allocate enough memory for full planes of the requested size */
	for (c=0; c<planes_number; c++) {

		if (buffer_length) {
			planes_array[c]->buffer = malloc(buffer_length * GET_TYPE_SIZE(kind));

					/* check */
			if (!planes_array[c]->buffer && buffer_length)  {
				for (k=0; k<planes_number; k++) {
					destroy_planes_array((void **)planes_array, kind, planes_number, SPARSE_IMAGE);
				}
				return -3;	/* return an error signal */
			}
		} else
			planes_array[c]->buffer = NULL;	/* no buffer allocated */
	}

	mimg->buffer_length = buffer_length;
	
	return 0;
}

/**************************************************************************************
/*	Function:		write_section_in_buffer_at_offset
/*	Description:	write the contents of the sections in a target image buffers
/*					at a certain offset
/*
/*	Parameters:
/*		<-	mimg				pointer to the source image
/*		<-	target_mimg			pointer to the target image
/*		<-	sect				pointer to the sections in the source image
/*		<-	position_in_buffer	position of the buffers in the source image
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void write_section_in_buffer_at_offset(mimage_ptr mimg, mimage_ptr target_mimg, section_data_ptr sect, int position_in_buffer)
{
	int c, pix_idx;
	
	if (sect->position_in_buffer >= 0)	{ /* normal value */			

			/*
			 *	Copy the memory for the data if is not default
			 */
			
		for (c=0 ; c<target_mimg->comp ; c++) {

			gr_ptr buffer, target_buffer;
			
			buffer = get_byteplane_from_mimage(mimg, c)->buffer + sect->position_in_buffer * GET_TYPE_SIZE(mimg->kind);
			target_buffer = get_byteplane_from_mimage(target_mimg, c)->buffer + position_in_buffer * GET_TYPE_SIZE(target_mimg->kind);
			
			memcpy(target_buffer, buffer, sect->number_of_pixels * GET_TYPE_SIZE(mimg->kind)); 
		}
	} else {	/* default value */
		if (mimg->kind == DISCRETE_IMAGE) {
		
			mpixel_ptr target_buffer;
			mpixel default_value;
			
			for (c=0 ; c<target_mimg->comp ; c++) {
			
				default_value = get_byteplane_from_mimage(mimg, c)->default_value;

				target_buffer = get_byteplane_from_mimage(target_mimg, c)->buffer + position_in_buffer;

				for (pix_idx = 0 ; pix_idx < sect->number_of_pixels ; pix_idx++)
					*target_buffer++ = default_value;
			}
		} else {	/* FLOAT_IMAGE */
		
			fmpixel_ptr target_fbuffer;
			fmpixel default_fvalue;
			
			for (c=0 ; c<target_mimg->comp ; c++) {
			
				default_fvalue = get_fbyteplane_from_mimage(mimg, c)->default_value;
				target_fbuffer = get_fbyteplane_from_mimage(target_mimg, c)->buffer + position_in_buffer;
				
				for (pix_idx = 0 ; pix_idx < sect->number_of_pixels ; pix_idx++)
					*target_fbuffer++ = default_fvalue;
			}
		}
	}
}

/**************************************************************************************
/*	Function:		copy_section_from_mimage
/*	Description:	copy sections from one image to another
/*
/*	Parameters:
/*		<-	mimg				pointer to the source image
/*		<-	target_mimg			pointer to the target image
/*		<-	sect				pointer to the sections in the source image
/*		<-	target_sect			pointer to the secionts in the target image
/*		<-	data_idx_target		index of the row_start data in the target image
/*		<-	expand_default		1 if expand the default value, else 0
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void copy_section_from_mimage(mimage_ptr mimg, mimage_ptr target_mimg, section_data_ptr sect, section_data_ptr target_sect, int *data_idx_target, int expand_default)
{
		/*
		 *	Copy the section
		 */

	memcpy(target_sect, sect, sizeof(section_data));
	if (sect->position_in_buffer < 0 && !expand_default)
		return;		

	write_section_in_buffer_at_offset(mimg, target_mimg, sect, target_sect->position_in_buffer = *data_idx_target);
	*data_idx_target += target_sect->number_of_pixels;
}

/**************************************************************************************
/*	Function:		copy_row_from_mimage
/*	Description:	copy rows from one image to another
/*
/*	Parameters:
/*		<-	mimg				pointer to the source image
/*		<-	target_mimg			pointer to the target image
/*		<-	row_idx				pointer to the row data in the source image
/*		<-	row_idx_target		pointer to the row data in the target image
/*		<-	sect_idx_target		index to the section data in the target image
/*		<-	data_idx_target		index of the row_start data in the target image
/*		<-	expand_default		1 if expand the default value, else 0
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void copy_row_from_mimage(mimage_ptr mimg, mimage_ptr target_mimg, int row_idx, int *row_idx_target, int *sect_idx_target, int *data_idx_target, int expand_default)
{
	int i;
	int sect_idx;
	int n_sects;
	section_data_ptr sect, target_sect;


	sect_idx = mimg->row_start[row_idx];

	if (sect_idx >= 0) {		/* check if we have to skip rows */
		
		target_mimg->row_start[*row_idx_target] = *sect_idx_target;

		n_sects = sections_in_row(mimg, row_idx);	
		
		for (i=0 ; i<n_sects ; i++)
		{
			target_sect	= &target_mimg->row_section[*sect_idx_target];
			sect		= &mimg->row_section[sect_idx];
			
				/*
				 *	Copy the section
				 */

			copy_section_from_mimage(mimg, target_mimg, sect, target_sect, data_idx_target, expand_default);

				/*
				 *	Update the indexes
				 */

			(*sect_idx_target)++;
			sect_idx++;
		}		

	} else {
	
		target_mimg->row_start[*row_idx_target] = sect_idx;	/*	skip the same number of rows in the target
																that was skipped in the source */
	}
	
	(*row_idx_target)++;
	return;
}
