/**************************************************************************************
/* Filename:	gr_utils.c
/*		Copyright  1998-99 Giuseppe Di Mauro. All rights reserved.
/*
/* Description:	common utilitiesfor both sparse and dense byteplanes images
/*
/***************************************************************************************/

#include "gr_utils.h"
#include "gr_sparse_utils.h"

	/*
	 *	Definitions for lexicographically image ordering utilities
	 */

#define USE_BUFFERS_OPTIMIZATION
#define STATIC_BUFFERS_OPTIMIZATION
#define COLOR_POINTER_OPTIMIZATION
#define AGGRESSIVE_OPTIMIZATION

	/*
	 *	Globals for lexicographically image ordering utilities
	 */

gr_ptr					ic_buffers_array[256];	/* set a limit of 256 planes */
byteplane_ptr			*ic_planes_array;
short					ic_planes_number;
size_t					ic_width;
gr_image_displacement	*ic_disp_array;
 
	/*
	 *	Function prototypes for lexicographically image ordering utilities
	 */

static int integer_val_and_pos_compare(int *x, int *y);
static int integer_val_and_pos_compare_for_float(int *x, int *y);
static int integer_val_and_pos_sparse_compare(int *x, int *y);
static int integer_val_and_pos_sparse_compare_for_float(int *x, int *y);
static int integer_val_compare(int *x, int *y);
static int integer_val_compare_for_float(int *x, int *y);

	/*
	 *	Implementation
	 */

/**************************************************************************************
/*	Function:		is_valid_rect
/*	Description:	check if boundaries of the rect are valid
/*
/*	Parameters:
/*		<- rect_ptr		pointer to the rect
/*
/*	Result:
/*		1	ok
/*		0	not ok
/*
/***************************************************************************************/

int is_valid_rect(image_rect_ptr rect_ptr)
{
		/* 
		 * check for negative boundaries
		 */
		 
  if ((rect_ptr->x < 0) || (rect_ptr->y < 0) || (rect_ptr->dx < 0) ||
      (rect_ptr->dy < 0))
    return 0;

		/*
		 * check for negative size
		 */
  if ((rect_ptr->dx < rect_ptr->x) || (rect_ptr->dy < rect_ptr->y))
    return 0;

  return 1;
}

/**************************************************************************************
/*	Function:		float_compare
/*	Description:	compare real function used for sorting
/*
/*	Parameters:
/*		<- x		first float number
/*		<- y		second float number
/*
/*	Result:
/*		-1	first is smaller than second
/*		0	the same
/*		1	first is greater than second
/*
/***************************************************************************************/

int float_compare(float *x, float *y)
{
	float res = *x-*y;
	return ( (res > (float)0.) ? (int)1 : ( (res < (float)0.) ? (int)-1 : (int)0 ) );
}

/**************************************************************************************
/*	Function:		unsigned_char_compare
/*	Description:	compare function used for sorting
/*
/*	Parameters:
/*		<- x		first number
/*		<- y		second number
/*
/*	Result:
/*		-1	first is smaller than second
/*		0	the same
/*		1	first is greater than second
/*
/***************************************************************************************/

int unsigned_char_compare(unsigned char *x, unsigned char *y)
{
	return ((int)*x-(int)*y);
}

/**************************************************************************************
/*	Function:		lex_pixels_default_compare
/*	Description:	compare function used for lexical sort between a pixel
/*					and a default value
/*
/*	Parameters:
/*		<- a		pixel pointer
/*		<- def		default pixel array
/*		<- n		number of planes
/*		<- d		distance between planes in memory
/*
/*	Result:
/*		-1	first is smaller than second
/*		0	the same
/*		1	first is greater than second
/*
/***************************************************************************************/

int lex_pixels_default_compare(mpixel *a, mpixel *def, int n, int *d)
{
	int i;
		
		/* for every plane */
	for (i=0; i<n; i++) {
			/* if a is less than the default value */
		if (*a < *def)
			return -1;	/* return -1 */
		else if (*a > *def)	/* if is bigger */
			return 1;	/* return 1 */
		
			/* advance pixel to next plane; */
		a = (mpixel_ptr)((char *)a + d[i]);
			/* advance default value to next plane */
		def ++;
	}
	
	return 0;	/* pixels match */
}

/**************************************************************************************
/*	Function:		lex_fpixels_default_compare
/*	Description:	compare function used for lexical sort between a float pixel
/*					and a default float value
/*
/*	Parameters:
/*		<- a		float pixel pointer
/*		<- def		default float pixel array
/*		<- n		number of planes
/*		<- d		distance between planes in memory
/*
/*	Result:
/*		-1	first is smaller than second
/*		0	the same
/*		1	first is greater than second
/*
/***************************************************************************************/

int lex_fpixels_default_compare(fmpixel *a, fmpixel *def, int n, int *d)
{
	int i;
	
		/* for every plane */
	for (i=0; i<n; i++) {
			/* if a is less than the default value */
		if (*a < *def)
			return -1	/* return -1 */;
		else if (*a > *def)	/* if is bigger */
			return 1;	/* return 1 */
		
			/* advance pixel to next plane; */
		a = (fmpixel_ptr)((char *)a + d[i]);
			/* advance default value to next plane */
		def ++;
	}
	
	return 0;	/* pixels match */
}

/**************************************************************************************
/*	Function:		lex_pixels_compare
/*	Description:	compare function used for lexical sort between two pixels
/*
/*	Parameters:
/*		<- a		first pixel pointer
/*		<- b		second pixel pointer
/*		<- n		number of planes
/*		<- d		distance between planes in memory
/*
/*	Result:
/*		-1	first is smaller than second
/*		0	the same
/*		1	first is greater than second
/*
/***************************************************************************************/

int lex_pixels_compare(mpixel *a, mpixel *b, int n, int *d)
{
	int i;
	
		/* for every plane */
	for (i=0; i<n; i++) {
		if (*a < *b)	/* if b is greater than a */
			return -1;
		else if (*a > *b)	/* viceversa */
			return 1;
		
			/* if this plane has same value */
		
			/* advance both pointer to next plane */
		a = (mpixel_ptr)((char *)a + d[i]);
		b = (mpixel_ptr)((char *)b + d[i]);
	}
	
	return 0;	/* pixels match */
}

/**************************************************************************************
/*	Function:		lex_fpixels_compare
/*	Description:	compare function used for lexical sort between two float pixels
/*
/*	Parameters:
/*		<- a		first float pixel pointer
/*		<- b		second float pixel pointer
/*		<- n		number of planes
/*		<- d		distance between planes in memory
/*
/*	Result:
/*		-1	first is smaller than second
/*		0	the same
/*		1	first is greater than second
/*
/***************************************************************************************/

int lex_fpixels_compare(fmpixel *a, fmpixel *b, int n, int *d)
{
	int i;
	
		/* for every plane */
	for (i=0; i<n; i++) {
		if (*a < *b)	/* if b is greater than a */
			return -1;
		else if (*a > *b)	/* viceversa */
			return 1;
		
			/* if this plane has same value */
		
			/* advance both pointer to next plane */
		a = (fmpixel_ptr)((char *)a + d[i]);
		b = (fmpixel_ptr)((char *)b + d[i]);
	}
	
	return 0;	/* pixels match */
}

/**************************************************************************************
/*	Function:		get_planes_distances_for_mimage
/*	Description:	create an array containing the distance between the planes in
/*					memory. If pointer to three planes are p0, p1, p2 the distances
/*					will be p1-p0, p2-p1.
/*
/*	Parameters:
/*		<- mimg		pointer to the image
/*
/*	Result:
/*		not NULL	array of distances calculated
/*		NULL		error condition
/*
/***************************************************************************************/

int *get_planes_distances_for_mimage(mimage_ptr mimg)
{
	int c;
	int *d;
	int kind = GET_KIND(mimg);
	int planes_number = GET_PLANES_NUMBER(mimg);

		/* allocate memory for distances */
	d = (int *)malloc(sizeof(int) * planes_number);
	if (!d) 
		return NULL;

	 	/* calculate the distance between the buffers */
	for (c=0; c<(planes_number - 1); c++) {
		
		char *buffer1, *buffer2;
		
			/* get the two buffers */
		buffer1 = get_plane_buffer(mimg->planes_array[c], kind);
		buffer2 = get_plane_buffer(mimg->planes_array[c+1], kind);
		
			/* and save the distance */
		d[c] = buffer2 - buffer1;
	}
	
	d[c] = 0;	/* extra distance, not used */
	
	return d;	/* return the distance */
}

/**************************************************************************************
/*	Function:		bfbsearch
/*	Description:	This routine returns the index of the last item in
/*					a sorted float vector which is not larger than x or
/*					if there is none such, return -1
/*
/*	Parameters:
/*		<- value	value to look for
/*		<- values	array of values where to look for
/*		<- size		size of the array
/*
/*	Result:
/*		index of the last item which is not larger than x
/*		-1		no such element
/*
/***************************************************************************************/

int bfbsearch(float value, float *values, size_t size)
{
	size_t low, high;
	size_t mid;
	
	mid = 0;
	low = 0; 
	high = size-1;
	
	while ((low + 1) < high) {

		mid = (low + high)/2;
		
		if (values[mid] < value)
			low = mid;
		else
			high = mid;
	}
	
		/* at this point low and high differ by at most 1, and fvalues[low] < fvalue if there is any j
		   such that fvalues[j] < fvalue; also fvalues[high] >= fvalue if there is any j such 
		   that fvalues[j] >= fvalue.
		 */


	return ( (values[low] > value) ? -1 : ((values[high] > value) ? low : high));
}

/**************************************************************************************
/*	Function:		bibsearch
/*	Description:	This routine returns the index of the last item in
/*					a sorted float vector which is not larger than x or
/*					if there is none such, return -1
/*
/*	Parameters:
/*		<- value	value to look for
/*		<- values	array of values where to look for
/*		<- size		size of the array
/*
/*	Result:
/*		index of the last item which is not larger than x
/*		-1		no such element
/*
/***************************************************************************************/

int bibsearch(int value, int *values, size_t size)
{
	size_t low, high;
	size_t mid;
	
	mid = 0;
	low = 0; 
	high = size-1;
	
	while ((low + 1) < high) {

		mid = (low + high)/2;
		
		if (values[mid] < value)
			low = mid;
		else
			high = mid;
	}
	
		/* 
		 *	at this point low and high differ by at most 1, and fvalues[low] < fvalue if there is any j
		 *	such that fvalues[j] < fvalue; also fvalues[high] >= fvalue if there is any j such 
		 *	that fvalues[j] >= fvalue.
		 */

	return ( (values[low] > value) ? -1 : ((values[high] > value) ? low : high));
}

/**************************************************************************************
/*	Function:		threshold_pixel
/*	Description:	This routine returns the value of a pixel that is closer to the
/*					values in a threshold array
/*
/*	Parameters:
/* 		<-	value			value to threshold
/*		<-	fvalues:		threshold array (should be already sorted)
/*		<-	fvalues_size:	number of elements in the threshold list
/*
/*	Result:
/*		the thresholded value
/*
/***************************************************************************************/

float threshold_pixel(float value, float *fvalues, size_t fvalues_size)
{
	int index;

	index = bfbsearch(value, fvalues, fvalues_size);
	if (index < 0)
		return (float)0.;
		
	return fvalues[index];	
}

/**************************************************************************************
/*	Function:		generate_pixel_no_array
/*	Description:	generate an array which position in array
/*					are the values of the array
/*
/*	Parameters:
/*		<-	num_pixels	number of pixels for the pixel number array
/*
/*	Result:
/*		NULL		not enough memory to generate the array
/*		not NULL	return such an array
/*
/***************************************************************************************/

int *generate_pixel_no_array(int num_pixels)
{
	int i;
	int *pixel_no_array;
	
		/* allocate the array of mpixel indices */
	pixel_no_array = malloc(num_pixels * sizeof(*pixel_no_array));
	if (pixel_no_array) {

			/* initialize the array */
		for (i=0 ; i<num_pixels ; i++)	pixel_no_array[i] = i;		
	}

	return pixel_no_array;		
}

/**************************************************************************************
/*	Function:		generate_pixel_disp_array
/*	Description:	generate an array of pairs (x,y) which positions (x, y)
/*					in array are the values of the array 
/*
/*	Parameters:
/*		<-	num_pixels	number of pixels for the pixel displacement array
/*
/*	Result:
/*		NULL		not enough memory to generate the array
/*		not NULL	return such an array
/*
/***************************************************************************************/

gr_image_displacement *generate_pixel_disp_array(int size)
{
	int i;
	gr_image_displacement *disp_array;
	
		/* allocate the displacement array */
	if (!(disp_array = malloc( size * sizeof(gr_image_displacement)))) {

			/* initialize every coordinate to -1 */
		for (i=0; i<size; i++) {
			disp_array[i].x = -1;
			disp_array[i].y = -1;
		}
	}

	return disp_array;
}

/**************************************************************************************
/*	Function:		sort_sparse_by_pvals_and_posns
/*	Description:	sort values and positions buffer
/*
/*	Parameters:
/*		<->	pixel_no_array	array with pixel numbers
/*		<->	disp_array		array with pixel positions
/*		<-	mimg			image which pixels belong
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void sort_sparse_by_pvals_and_posns(int *pixel_no_array, gr_image_displacement *disp_array, mimage_ptr mimg)
{
	int c;
	size_t ic_height;

	for (c=0; c<mimg->comp; c++)
		ic_buffers_array[c] = get_byteplane_from_mimage(mimg, c)->buffer;

	ic_disp_array		= disp_array;
	ic_planes_array		= mimg->planes_array;
	ic_planes_number	= mimg->comp;
	
	if (IS_DISCRETE(mimg))
		qsort(pixel_no_array, mimg->buffer_length, sizeof(int), (int (*)(void const *, void const *))integer_val_and_pos_sparse_compare);
	else
		qsort(pixel_no_array, mimg->buffer_length, sizeof(int), (int (*)(void const *, void const *))integer_val_and_pos_sparse_compare_for_float);
}				

/**************************************************************************************
/*	Function:		sort_by_pvals_and_posns
/*	Description:	This routine sorts its first argument, an array of mpixel numbers
/*					into the alphabetical order of the corresponding mpixel values. Pixel numbers
/*					with identical mpixel values are sorted into the alphabetical order of their
/*					row, column position.
/*
/*	Parameters:
/*		<->	pixel_no_array	array with pixel numbers
/*		<-	mimg			image which pixels belong
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void sort_by_pvals_and_posns(int *pixel_no_array, mimage_ptr mimg) 
{
	int c;
	size_t ic_height;

	for (c=0; c<mimg->comp; c++)
		ic_buffers_array[c] = get_byteplane_from_mimage(mimg, c)->buffer;

	ic_planes_array =  mimg->planes_array;
	ic_planes_number = mimg->comp;
	ic_width =         ((byteplane_ptr)(mimg->planes_array[0]))->width;
	ic_height =         ((byteplane_ptr)(mimg->planes_array[0]))->height;
	
	if (IS_DISCRETE(mimg))
		qsort(pixel_no_array, ic_width * ic_height, sizeof(int), (int (*)(void const *, void const *))integer_val_and_pos_compare);
	else
		qsort(pixel_no_array, ic_width * ic_height, sizeof(int), (int (*)(void const *, void const *))integer_val_and_pos_compare_for_float);
}				

/**************************************************************************************
/*	Function:		sort_by_pvals
/*	Description:	This routine sorts its first argument, an array of mpixel numbers
/*					into the alphabetical order of the corresponding mpixel values.
/*
/*	Parameters:
/*		<->	pixel_no_array	array with pixel numbers
/*		<-	mimg			image which pixels belong
/*		<-	pixels_number	number of pixels
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void sort_by_pvals(int *pixel_no_array, mimage_ptr mimg, int pixels_number)
{
	int c;
	size_t ic_height;

	for (c=0; c<mimg->comp; c++)
		ic_buffers_array[c] = get_byteplane_from_mimage(mimg, c)->buffer;

	ic_planes_array =  mimg->planes_array;
	ic_planes_number = mimg->comp;
	
	if (IS_DISCRETE(mimg))
		qsort(pixel_no_array, pixels_number, sizeof(int), (int (*)(void const *, void const *))integer_val_compare);
	else
		qsort(pixel_no_array, pixels_number, sizeof(int), (int (*)(void const *, void const *))integer_val_compare_for_float);
}				

/**************************************************************************************
/*	Function:		integer_val_and_pos_compare_for_float
/*	Description:	Compare function for sort_by_pvals_and_posns for float images
/*
/*	Parameters:
/*		<-	x	pointer to the first position value
/*		<-	y	pointer to the second position value
/*
/*	Result:
/*		-1	value at position x is smaller than second
/*		0	the same
/*		1	value at position y is greater than second
/*
/***************************************************************************************/

static int integer_val_and_pos_compare_for_float(int *x, int *y)
{
	int c;
	
	size_t row_of_x, row_of_y;
	fmpixel pixel_val_x, pixel_val_y;

	for (c=0; c<ic_planes_number; c++) {
		pixel_val_x = *((fmpixel_ptr)(ic_planes_array[c]->buffer) + *x);
		pixel_val_y = *((fmpixel_ptr)(ic_planes_array[c]->buffer) + *y);
	
		if (pixel_val_x > pixel_val_y)	/* x is larger than y */
			return 1;

		if (pixel_val_x < pixel_val_y) /* y is larger than x */
			return -1;
	}

	row_of_x = *x/ic_width;
	row_of_y = *y/ic_width;
	
	if (row_of_x > row_of_y)
		return 1;

	if (row_of_x < row_of_y)
		return -1;
	
	if (*x > *y)
		return 1;
		
	if (*x < *y)
		return -1;

	return  0;
}

/**************************************************************************************
/*	Function:		integer_val_and_pos_compare
/*	Description:	Compare function for sort_by_pvals_and_posns
/*
/*	Parameters:
/*		<-	x	pointer to the first position value
/*		<-	y	pointer to the second position value
/*
/*	Result:
/*		-1	value at position x is smaller than second
/*		0	the same
/*		1	value at position y is greater than second
/*
/***************************************************************************************/

static int integer_val_and_pos_compare(int *x, int *y)
{
	register int _x = *x, _y = *y;
	register int c;
	
	size_t row_of_x, row_of_y;
	mpixel pixel_val_x, pixel_val_y;
	register mpixel_ptr buffer;
	
	for (c=0; c<ic_planes_number; c++) {
		buffer = *(ic_buffers_array + c);

		pixel_val_x = *(buffer + _x);
		pixel_val_y = *(buffer + _y);
	
		if (pixel_val_x > pixel_val_y)	/* x is larger than y */
			return 1;

		if (pixel_val_x < pixel_val_y) /* y is larger than x */
			return -1;
	}

	row_of_x = _x/ic_width;
	row_of_y = _y/ic_width;
	
	if (row_of_x > row_of_y)
		return 1;

	if (row_of_x < row_of_y)
		return -1;
	
	return _x - _y;
}

/**************************************************************************************
/*	Function:		integer_val_and_pos_sparse_compare_for_float
/*	Description:	Compare function for sort_by_pvals_and_posns
/*
/*	Parameters:
/*		<-	x	pointer to the first position value
/*		<-	y	pointer to the second position value
/*
/*	Result:
/*		-1	value at position x is smaller than second
/*		0	the same
/*		1	value at position y is greater than second
/*
/***************************************************************************************/

static int integer_val_and_pos_sparse_compare_for_float(int *x, int *y)
{
	int c;
	
	size_t row_of_x, row_of_y;
	fmpixel pixel_val_x, pixel_val_y;

	for (c=0; c<ic_planes_number; c++) {
		pixel_val_x = *((fmpixel_ptr)(ic_planes_array[c]->buffer) + *x);
		pixel_val_y = *((fmpixel_ptr)(ic_planes_array[c]->buffer) + *y);
	
		if (pixel_val_x > pixel_val_y)	/* x is larger than y */
			return 1;

		if (pixel_val_x < pixel_val_y) /* y is larger than x */
			return -1;
	}

	row_of_x = ic_disp_array[*x].y;
	row_of_y = ic_disp_array[*y].y;
	
	if (row_of_x > row_of_y)
		return 1;

	if (row_of_x < row_of_y)
		return -1;
	
	if (*x > *y)
		return 1;
		
	if (*x < *y)
		return -1;

	return  0;
}

/**************************************************************************************
/*	Function:		integer_val_and_pos_sparse_compare
/*	Description:	Compare function for sort_by_pvals_and_posns for sparse
/*
/*	Parameters:
/*		<-	x	pointer to the first position value
/*		<-	y	pointer to the second position value
/*
/*	Result:
/*		-1	value at position x is smaller than second
/*		0	the same
/*		1	value at position y is greater than second
/*
/***************************************************************************************/

static int integer_val_and_pos_sparse_compare(int *x, int *y)
{
	register int _x = *x, _y = *y;
	register int c;
	
	size_t row_of_x, row_of_y;
	mpixel pixel_val_x, pixel_val_y;
	register mpixel_ptr buffer;
	
	for (c=0; c<ic_planes_number; c++) {
		buffer = *(ic_buffers_array + c);

		pixel_val_x = *(buffer + _x);
		pixel_val_y = *(buffer + _y);
	
		if (pixel_val_x > pixel_val_y)	/* x is larger than y */
			return 1;

		if (pixel_val_x < pixel_val_y) /* y is larger than x */
			return -1;
	}

	row_of_x = ic_disp_array[_x].y;
	row_of_y = ic_disp_array[_y].y;
	
	if (row_of_x > row_of_y)
		return 1;

	if (row_of_x < row_of_y)
		return -1;
	
	return _x - _y;
}

/**************************************************************************************
/*	Function:		integer_val_and_pos_compare_for_float
/*	Description:	Compare function for sort_by_pvals_and_posns for float images
/*
/*	Parameters:
/*		<-	x	pointer to the first position value
/*		<-	y	pointer to the second position value
/*
/*	Result:
/*		-1	value at position x is smaller than second
/*		0	the same
/*		1	value at position y is greater than second
/*
/***************************************************************************************/

static int integer_val_compare_for_float(int *x, int *y)
{
	int c;
	
	size_t row_of_x, row_of_y;
	fmpixel pixel_val_x, pixel_val_y;

	for (c=0; c<ic_planes_number; c++) {
		pixel_val_x = *((fmpixel_ptr)(ic_planes_array[c]->buffer) + *x);
		pixel_val_y = *((fmpixel_ptr)(ic_planes_array[c]->buffer) + *y);
	
		if (pixel_val_x > pixel_val_y)	/* x is larger than y */
			return 1;

		if (pixel_val_x < pixel_val_y) /* y is larger than x */
			return -1;
	}

	return  0;
}

/**************************************************************************************
/*	Function:		integer_val_compare
/*	Description:	Compare function for sort_by_pvals_and_posns
/*
/*	Parameters:
/*		<-	x	pointer to the first position value
/*		<-	y	pointer to the second position value
/*
/*	Result:
/*		-1	value at position x is smaller than second
/*		0	the same
/*		1	value at position y is greater than second
/*
/***************************************************************************************/

static int integer_val_compare(int *x, int *y)
{
	register int c;
	
	size_t row_of_x, row_of_y;
	mpixel pixel_val_x, pixel_val_y;
	
	for (c=0; c<ic_planes_number; c++) {

		pixel_val_x = *((mpixel_ptr)(ic_planes_array[c]->buffer) + *x);
		pixel_val_y = *((mpixel_ptr)(ic_planes_array[c]->buffer) + *y);
	
		if (pixel_val_x > pixel_val_y)	/* x is larger than y */
			return 1;

		if (pixel_val_x < pixel_val_y) /* y is larger than x */
			return -1;
	}

	return  0;
}

/**************************************************************************************
/*	Function:		generate_displacements_for_sparse_mimage
/*	Description:	generate the displacement array for the pixels in
/*					a sparse image. This is how it works: not default
/*					image data for sparse images is stored in buffers
/*					contigously. Being the image is sparse pixel at
/*					position 0 in buffer could be different than pixel
/*					at position (0,0) in the image. Thus, the displacements
/*					array at position n	contains the coordinates (displacement) 
/*					of the n-th pixel in buffer in the image.
/*
/*	Parameters:
/*		<-	mimg	image whom we need to calculate the displacement array
/*
/*	Result:
/*		not NULL	the displacement array
/*		NULL		error condition
/*
/***************************************************************************************/

gr_image_displacement *generate_displacements_for_sparse_mimage(mimage_ptr mimg)
{
	int x, y = 0;
	int i,j;
	int l;
	int pos;
	int sect_idx;
	int sects_per_row;
	section_data *sects;
	section_data_ptr sect;
	int *row_start;
	gr_image_displacement *pixel_disp_array;
	
		/* 
		 * allocate first the displacement array
		 * and initialize it
		 */
	pixel_disp_array = generate_pixel_disp_array(mimg->buffer_length);
	if (pixel_disp_array) {	/* if generation went ok */

		row_start = mimg->row_start;	/* get the row_start */

		for (j=0; j<mimg->data_length; j++) {	/* for every row */

			if ((sect_idx = row_start[j]) < 0) {	/* if is one or more skipping rows */
				y -= sect_idx;	/* keep track of it */
				continue;
			}

			sects_per_row = sections_in_row(mimg, j);	/* get the number of sections per row */
			sects = mimg->row_section + sect_idx;	/* get the section for this row */
			
			for (i=0; i<sects_per_row; i++) {	/* for every section in this row */
			
				sect = sects;		/* use sect to cache currect section */
			
				if ((pos = sect->position_in_buffer) >= 0) {			/* if a real value (not default) */
					
					x = sect->starting_pixel_number;			/* get the starting pixel number */

					for (l=0; l<sect->number_of_pixels; l++) {	/* for every pixel in the section */

						pixel_disp_array[pos].x = x;	/* get the displacement */
						pixel_disp_array[pos].y = y;
						
						pos++;		/* advance to next position */
						x++;		/* increase x */
					}
				}
				sects++;	/* advance to next section */
			}
			y++;	/* increase y */
		}
	}
	
	return pixel_disp_array;	/* return the generated displacement array */
}
