#include <math.h>
#include <stdlib.h>
#include <string.h>

#include <NTL/RR.h>
#include <NTL/ZZ.h>
#include <NTL/ZZ_pE.h>
#include <NTL/ZZ_pEX.h>
#include <NTL/GF2.h>
#include <NTL/GF2X.h>
#include <NTL/GF2E.h>
#include <NTL/GF2EX.h>

#include <NTL/mat_ZZ.h>
#include <NTL/mat_ZZ_pE.h>
#include <NTL/mat_RR.h>
#include <NTL/mat_GF2.h>

#include <NTL/vec_ZZ.h>
#include <NTL/vec_ZZ_p.h>
#include <NTL/vec_RR.h>
#include <NTL/vec_GF2.h>

#include <NTL/ZZXFactoring.h>
#include <NTL/ZZ_pXFactoring.h>
#include <NTL/ZZ_pEXFactoring.h>
#include <NTL/GF2XFactoring.h>
#include <NTL/GF2EXFactoring.h>

#include <NTL/tools.h>

#ifdef __cplusplus
extern "C" {
#endif

#include "macros.h"	/* SETL2 system header files */
#include "stubs.h"


#define INT_BIT_WIDTH	INT_CELL_WIDTH

	/*
	 *	Globals
	 */

int32 ntl_type;

void abort(void);
void abort(void) 
{ printf("\n"); }


	/*
	 *	Prototypes
	 */

		// Setl2 external
extern int register_type(SETL_SYSTEM_PROTO char *, void *);

		// Internal use
static inline void return_setl_boolean(SETL_SYSTEM_PROTO specifier *target, int32 int_value);
static void get_tuple_matrix_size(SETL_SYSTEM_PROTO specifier *argv, int &n_col, int &n_row);
static void get_tuple_vector_size(SETL_SYSTEM_PROTO specifier *argv, int &n_col);
static ntl_struct *check_ntl_arg(SETL_SYSTEM_PROTO specifier *argv, uint32 sub_type, int i, char *expected_type, char *fn_name);
static long check_long_arg(SETL_SYSTEM_PROTO specifier *argv, int i, char *fn_name);
static double check_real_arg(SETL_SYSTEM_PROTO specifier *argv, int i, char *fn_name);
static unsigned char *check_setl_long_arg(SETL_SYSTEM_PROTO specifier *argv, int i, char *expected_type, char *fn_name, int32& number_of_bytes, short &is_negative);
static inline void smart_delete(void *ptr);
static inline void smart_free(void *ptr);
static integer_h_ptr_type setl_long_from_ZZ(SETL_SYSTEM_PROTO ZZ &ZZ);
static integer_h_ptr_type setl_short_to_long(SETL_SYSTEM_PROTO int32 source);
static unsigned char *convert_setl_2_uarray(SETL_SYSTEM_PROTO integer_h_ptr_type source, int32 *number_of_bytes, short *is_negative);
static integer_h_ptr_type convert_uarray_2_setl(SETL_SYSTEM_PROTO unsigned char *source, int32 number_of_bytes, short is_negative = 0);
static void NewZZFromBytes(ZZ& x, const unsigned char *p, long n, short is_negative);

		// Packages needed
int32 LIBNTL__INIT(SETL_SYSTEM_PROTO_VOID);
void internal_destructor(struct ntl_struct *spec);
int get_tuple_count(SETL_SYSTEM_PROTO tuple_h_ptr_type tuple_root, int form);

		// Exported
void NTL_GET_REAL(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_REAL_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_REAL_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_INT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MODINT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_INT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MODINT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MODINT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_INT_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MODINT_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MODEXTINT_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_INT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MODINT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MODEXTINT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MODEXTINT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MODEXTINT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MODEXTINT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MOD2INT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MOD2INT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MOD2EXTINT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MOD2EXTINT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MOD2INT_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MOD2INT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MOD2INT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MOD2EXTINT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GET_MOD2EXTINT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);

void NTL_TO_SETL_OBJ(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);

void NTL_EQ(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_LT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_LE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_GE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_PROBPRIME(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_ISZERO(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_ISONE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_PROBIRREDTEST(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_DETIRREDTEST(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);

	/*
	 *	Conversion to setl auxiliary functions
	 */

static void NTL_GET_SETL_REAL(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_REAL_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_REAL_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_POLY_MODULO(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_POLY_MODULO_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_EXT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_EXT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_POLY_EXT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_POLY_EXT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_2(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_2_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_2_EXT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_2_POLY_EXT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_2_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_2_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_MOD2INT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_MOD2EXTINT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_MOD2EXTINT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);

static void NTL_GET_SETL_INTEGER_POLY_LONG_PAIR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_POLY_MODULO_LONG_PAIR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_POLY_EXT_MODULO_LONG_PAIR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_2_POLY_LONG_PAIR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
static void NTL_GET_SETL_INTEGER_MODULO_2_POLY_EXT_LONG_PAIR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);

static void NTL_GET_SETL_MODEXTINT_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
	/*
	 *	Auxiliary useful routines
	 */
	 
void NTL_SET_MODULO(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_BIT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_NUMBITS(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_RANDOMBND(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_INVMOD(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
void NTL_SET_EXT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);

	/*
	 *	Shell routines
	 */

	/*
	 *	Implementation: Package Utilities
	 */

inline void return_setl_boolean(SETL_SYSTEM_PROTO specifier *target, int32 int_value)
{
	unmark_specifier(target);
	target->sp_form = ft_atom;
	target->sp_val.sp_atom_num = int_value ? spec_true->sp_val.sp_atom_num : spec_false->sp_val.sp_atom_num;
}

void get_tuple_matrix_size(SETL_SYSTEM_PROTO specifier *argv, int &n_col, int &n_row)
{
	TUPLE_ITERATOR(ia)
	TUPLE_ITERATOR(iaa)

	n_col = 0;
	n_row = 0;
	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0])	
	{	// loop over outer tuple elements
		n++;
		int m=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element)) 
		{// loop over inner tuple elements
			m++;
		}
		ITERATE_TUPLE_END(iaa)
		if (n_row) {
			if (n_row != m) 
				abend(SETL_SYSTEM "Rows have different size in tuple matrix");
		} else {
			n_row = m;
		}
	}
	ITERATE_TUPLE_END(ia)
	n_col = n;
}

void get_tuple_vector_size(SETL_SYSTEM_PROTO specifier *argv, int &n_col)
{
	TUPLE_ITERATOR(ia)
	TUPLE_ITERATOR(iaa)

	n_col = 0;
	ITERATE_TUPLE_BEGIN(ia,argv[0])	
		n_col++;
	ITERATE_TUPLE_END(ia)
}


ntl_struct *check_ntl_arg(SETL_SYSTEM_PROTO specifier *argv, uint32 sub_type, int i, char *expected_type, char *fn_name)
{
	if ((argv->sp_form != ft_opaque)) 
		goto abend_label;

	ntl_struct *ntl_data = (ntl_struct *)argv->sp_val.sp_opaque_ptr;
	
	if  ((ntl_data->type != ntl_type) || ((sub_type != ntl_any_type) && (ntl_data->sub_type != sub_type)))
		goto abend_label;

	return ntl_data; // check is ok

abend_label:

	abend(SETL_SYSTEM msg_bad_arg, expected_type,i+1,fn_name,abend_opnd_str(SETL_SYSTEM argv));
	return NULL;	// never reached, just to avoid a warning
}

long check_long_arg(SETL_SYSTEM_PROTO specifier *argv, int i, char *fn_name)
{
	if (argv->sp_form != ft_short)
		goto abend_label;
				
	return argv->sp_val.sp_short_value;

abend_label:

	abend(SETL_SYSTEM msg_bad_arg, "long",i+1,fn_name,abend_opnd_str(SETL_SYSTEM argv));
	return 0l;	// never reached, just to avoid a warning
}

double check_real_arg(SETL_SYSTEM_PROTO specifier *argv, int i, char *fn_name)
{
	if (argv->sp_form != ft_real)
		goto abend_label;
				
	return argv->sp_val.sp_real_ptr->r_value;

abend_label:

	abend(SETL_SYSTEM msg_bad_arg, "real",i+1,fn_name,abend_opnd_str(SETL_SYSTEM argv));
	return 0.;	// never reached, just to avoid a warning
}

unsigned char *check_setl_long_arg(SETL_SYSTEM_PROTO specifier *argv, int i, char *expected_type, char *fn_name, int32& number_of_bytes, short &is_negative)
{
	unsigned char *byte_array;
	integer_h_ptr_type setl_int;

	if (argv->sp_form == ft_long)
		setl_int = argv->sp_val.sp_long_ptr;
	else if (argv->sp_form == ft_short) {
		setl_int = setl_short_to_long(SETL_SYSTEM argv->sp_val.sp_short_value);
	} else
		goto abend_label;
	
	byte_array = convert_setl_2_uarray(SETL_SYSTEM setl_int, &number_of_bytes, &is_negative);
	return byte_array;		

abend_label:

	abend(SETL_SYSTEM msg_bad_arg, expected_type,i+1,fn_name,abend_opnd_str(SETL_SYSTEM argv));
	return NULL;
}

inline void smart_free(void *ptr)
{
	if (ptr)
		free(ptr);
}

inline void smart_delete(void *ptr)
{
	if (ptr)
		delete ptr;
}

integer_h_ptr_type setl_long_from_ZZ(SETL_SYSTEM_PROTO ZZ &ZZ)
{
	integer_h_ptr_type integer_item;
	int32 ZZ_bytes_size = (NumBits(ZZ)+7)/8;

	if (ZZ_bytes_size) {	// if not zero

		unsigned char *byte_array = (unsigned char *)calloc(sizeof(unsigned char), (uint32)ZZ_bytes_size);
		if (!byte_array)
			return NULL;
						
		short is_negative = (ZZ < 0l);
		BytesFromZZ(byte_array, ZZ, ZZ_bytes_size);

		integer_item = convert_uarray_2_setl(SETL_SYSTEM byte_array, ZZ_bytes_size, is_negative);
		smart_free(byte_array);

	} else {	// zero

		integer_c_ptr_type(t1);
		
		get_integer_header(integer_item);
		get_integer_cell(t1)

		t1->i_cell_value = 0;
		t1->i_prev = NULL;
		t1->i_next = NULL;
		
		integer_item->i_use_count = 1;
		integer_item->i_hash_code = -1;
		integer_item->i_cell_count = 1;
		integer_item->i_is_negative = 0;	// 0 is not negative
		integer_item->i_head = t1;
		integer_item->i_tail = t1;
	}	
	
	return integer_item;
}


integer_h_ptr_type setl_short_to_long(SETL_SYSTEM_PROTO int32 source)
{
	integer_c_ptr_type t1, t2;
	integer_h_ptr_type target_hdr;

		/* create a long integer */

	get_integer_header(target_hdr);
	target_hdr->i_use_count = 1;
	target_hdr->i_cell_count = 0;
	target_hdr->i_hash_code = -1;

		/* set the sign of the integer */

	if (source < 0L) {
		target_hdr->i_is_negative = YES;
		source = -source;
	}
	else {
		target_hdr->i_is_negative = NO;
	}

		/* keep adding cells until we use up the source */

	if (source) {
		t2 = NULL;
		while (source) {

			/* allocate a new cell node */

			get_integer_cell(t1);
			if (t2 != NULL)
				t2->i_next = t1;
			else
				target_hdr->i_head = t1;

			t1->i_cell_value = source & MAX_INT_CELL;
			source >>= INT_CELL_WIDTH;
			t1->i_prev = t2;
			t2 = t1;

			target_hdr->i_cell_count++;

		}

		t1->i_next = NULL;
		target_hdr->i_tail = t1;

	} else {	// if source == 0

		get_integer_cell(t1);

		target_hdr->i_head = t1;
		target_hdr->i_cell_count=1;	

		t1->i_cell_value = 0;
		t1->i_next = NULL;
		t1->i_prev = NULL;
	}
	
	return target_hdr;
}

int get_tuple_count(
	SETL_SYSTEM_PROTO
	tuple_h_ptr_type tuple_root,
	int form
)
{
TUPLE_ITERATOR(ia) 
int count;
specifier spare;

  spare.sp_form=ft_tuple;
  spare.sp_val.sp_tuple_ptr = tuple_root;
  count = 0;
   
	   /* loop over the elements of source */

	ITERATE_TUPLE_BEGIN(ia,spare)
	{
		if (count < ia_number) 
			return -1;

		if ((form>0)&&(ia_element->sp_form != form))
			return -1;

		count++;
	}
   ITERATE_TUPLE_END(ia)

   return count;
}

	/*
	 *	convert the unsigned part of a setl long integer to a byte array
	 */

unsigned char *convert_setl_2_uarray(SETL_SYSTEM_PROTO integer_h_ptr_type source, int32 *number_of_bytes, short *is_negative)
{
	unsigned char *ntl_dest, *ntl_start;	/* pointers for the byte array */
	integer_c_ptr_type s1;
	
	uint32 w;
	uint32 v = 0;			/* can be up to 22 bits (15+7) */
	unsigned short bits_in_v = 0;	/* keep count of bits */

	int cell_count = source->i_cell_count;
		/* allocate the byte array */
	ntl_start = ntl_dest = (unsigned char *)calloc(cell_count, 2u);
	if (!ntl_dest) {
		*number_of_bytes = 0;
		*is_negative = 0;
		return NULL;
	}

		/* loop over the setl cells */
	for (s1 = source->i_head; s1 != NULL; s1 = s1->i_next) {

		w = (uint32)s1->i_cell_value;	/* get the value of the setl cell */
		w = w << bits_in_v;		/* shift left to make room for the next piece */
		v = v | w;				/* insert next piece */
		bits_in_v += 15;		/* take the count of the newly inserted bits */
		*ntl_dest++ = v & 0xff;	/* store the lower byte to the byte array */

		v = v >> 8;				/* shift out the stored byte */
		bits_in_v -= 8;			/* update number of bits in v */

		if (bits_in_v >= 8) {		/* if more than 8 bits in v left */
			*ntl_dest++ = v & 0xff;	/* store another byte */
			v = v >> 8;				/* shift out the stored byte */
			bits_in_v -= 8;			/* update the count */
		}					 
	}
	
	if (bits_in_v) 			/* if any final fragment is left */
		*ntl_dest++ = v;	/* store it */

	realloc(ntl_start, *number_of_bytes = ntl_dest - ntl_start);	/* shrink the memory partition to fit excatly the array */
	*is_negative = source->i_is_negative;
	
	return ntl_start;	/* return the byte array */
}

	/*
	 *	convert the unsigned part of a setl long integer to a byte array
	 */

integer_h_ptr_type convert_uarray_2_setl(SETL_SYSTEM_PROTO unsigned char *source, int32 number_of_bytes, short is_negative)
{
	int i;
	integer_c_ptr_type t1,t2;
	integer_h_ptr_type target;

	unsigned long w;
	unsigned long v = 0;
	unsigned short bits_in_v = 0;

	get_integer_header(target);
	
	target->i_use_count = 1;
	target->i_hash_code = -1;
	target->i_cell_count = 0;
	target->i_is_negative = is_negative;
	
	t2 = NULL;
	for (i=0; i<number_of_bytes; i++) {	// work up from least significant end

		w = *source++;		// gets next byte of source
		w = w << bits_in_v;	// shift left by number of bits in v
		v = v | w;			// or in the bits of v into low position
		bits_in_v += 8;		// note that have 8 bits more

		if (bits_in_v < 15)	// if not yes 15 bits
			continue;		// get more
			
		get_integer_cell(t1);	// get a setl integer cell
		target->i_cell_count++;	// increment the cell count

		if (t2 != NULL)				// if there is a previous cell
			t2->i_next = t1;		// this is the following cell
		else 						// otherwise
			target->i_head = t1;	// set the head of the list
			
		t1->i_cell_value = v & 0x7fff;	// take 15 bits
		t1->i_prev = t2;				// put in the back link
		t2 = t1;						// keep track of previous link
		
		v = v >> 15;					// shift out 15 bits
		bits_in_v -= 15;				// note lost of 15 bits
	}
	
	if (bits_in_v && v) {		// if any final fragment is left

		get_integer_cell(t1);		// get another cell
		target->i_cell_count++;		// count up the final cell

		if (t2 != NULL)				// if there is a previous cell
			t2->i_next = t1;		// this is the following cell
		else  						// otherwise
			target->i_head = t1;	// set the head of the list
			
		t1->i_cell_value = v;		// store the value of v
		t1->i_prev = t2;			// put in the back link
		t2 = t1;					// keep track of previous link
	}
	
	t1->i_next = NULL;				// set the next node as NULL to finish the list
	target->i_tail = t1;			// set the tail to the last node
	
	return target;					// return the value
}

void NewZZFromBytes(ZZ& x, const unsigned char *p, long n, short is_negative)
{
	if (p && n) {
		ZZFromBytes(x, p, n);	// convert to the ZZ type
		if (is_negative)
			x = -x;
		free((void *)p);				// release memory used for byte_array
	} else
		x = 0;
}

	/*
	 *	Implementation: Common package routines
	 */
	 
void internal_destructor(struct ntl_struct *ntl_data)
{
	if (ntl_data)
		delete ntl_data;
}

int32 LIBNTL__INIT(
   SETL_SYSTEM_PROTO_VOID)
{
	ZZ_p::init(to_ZZ(17));	// init the modulo math to avoid latter crash

	if (!(ntl_type=register_type(SETL_SYSTEM "struct ntl_struct", internal_destructor)))
		return 1;
	return 0;
}

	/*
	 *	Implementation: This package routines
	 */
	 
void NTL_TO_SETL_OBJ(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "ntl integer modulo", "ntl_get_setl_obj");
	
	switch (ntl_data->sub_type) {
		case ntl_RR_type:
			NTL_GET_SETL_REAL(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_mat_RR_type:
			NTL_GET_SETL_REAL_MATRIX(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_RR_type:
			NTL_GET_SETL_REAL_VECTOR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_ZZ_type:
			NTL_GET_SETL_INTEGER(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_ZZ_p_type:
			NTL_GET_SETL_INTEGER_MODULO(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_GF2_type:
			NTL_GET_SETL_INTEGER_MODULO_2(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_GF2E_type:
			NTL_GET_SETL_INTEGER_MODULO_2_EXT(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_GF2X_type:
			NTL_GET_SETL_INTEGER_MODULO_2_POLY(SETL_SYSTEM argc, argv, target);
			break;		
		case ntl_GF2EX_type:
			NTL_GET_SETL_INTEGER_MODULO_2_POLY_EXT(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_ZZ_type:
			NTL_GET_SETL_INTEGER_VECTOR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_ZZ_p_type:
			NTL_GET_SETL_INTEGER_MODULO_VECTOR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_GF2_type:
			NTL_GET_SETL_INTEGER_MODULO_2_VECTOR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_GF2X_type:
			NTL_GET_SETL_MOD2INT_POLY_VECTOR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_GF2E_type:
			NTL_GET_SETL_MOD2EXTINT_VECTOR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_GF2EX_type:
			NTL_GET_SETL_MOD2EXTINT_POLY_VECTOR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_ZZX_type:
			NTL_GET_SETL_INTEGER_POLY(SETL_SYSTEM argc, argv, target);
			break;
   		case ntl_vec_pair_ZZX_long_type:
			NTL_GET_SETL_INTEGER_POLY_LONG_PAIR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_ZZ_pX_type:
			NTL_GET_SETL_INTEGER_POLY_MODULO(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_mat_ZZ_type:
			NTL_GET_SETL_INTEGER_MATRIX(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_mat_ZZ_p_type:
			NTL_GET_SETL_INTEGER_MODULO_MATRIX(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_mat_ZZ_pE_type:
			NTL_GET_SETL_MODEXTINT_MATRIX(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_mat_GF2_type:
			NTL_GET_SETL_INTEGER_MODULO_2_MATRIX(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_ZZ_pE_type:
			NTL_GET_SETL_INTEGER_MODULO_EXT(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_ZZ_pE_type:
			NTL_GET_SETL_INTEGER_MODULO_EXT_VECTOR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_ZZ_pEX_type:
			NTL_GET_SETL_INTEGER_MODULO_POLY_EXT(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_ZZ_pX_type:
			NTL_GET_SETL_INTEGER_POLY_MODULO_VECTOR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_ZZ_pEX_type:
			NTL_GET_SETL_INTEGER_MODULO_POLY_EXT_VECTOR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_pair_ZZ_pX_long_type:
			NTL_GET_SETL_INTEGER_POLY_MODULO_LONG_PAIR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_pair_GF2EX_long_type:
			NTL_GET_SETL_INTEGER_MODULO_2_POLY_EXT_LONG_PAIR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_pair_ZZ_pEX_long_type:
			NTL_GET_SETL_INTEGER_POLY_EXT_MODULO_LONG_PAIR(SETL_SYSTEM argc, argv, target);
			break;
		case ntl_vec_pair_GF2X_long_type:
			NTL_GET_SETL_INTEGER_MODULO_2_POLY_LONG_PAIR(SETL_SYSTEM argc, argv, target);
			break;
	}
}

void NTL_GET_REAL(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	if (argv[0].sp_form != ft_real)
		abend(SETL_SYSTEM msg_bad_arg, "real",1,"ntl_get_real",abend_opnd_str(SETL_SYSTEM argv));
	double double_data = argv[0].sp_val.sp_real_ptr->r_value;
	
	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_RR_type);
	*ntl_data->var.RR = double_data;	

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_REAL(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_RR_type, 0, "ntl float", "ntl_to_setl_obj");
	RR& RR = *ntl_data->var.RR;

	i_real_ptr_type	i_real;
	i_get_real(i_real);
	i_real->r_use_count = 1;
	i_real->r_value = to_double(RR);
	
	unmark_specifier(target);
	target->sp_form = ft_real;
	target->sp_val.sp_real_ptr = i_real;
}

void NTL_GET_REAL_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)
	TUPLE_ITERATOR(iaa)
	
	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_real_matrix",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_mat_RR_type);	
	mat_RR &mat_RR = *ntl_data->var.mat_RR;

	int n_col, n_row;
	get_tuple_matrix_size(SETL_SYSTEM argv, n_col, n_row);

	mat_RR.SetDims(n_col, n_row);
	
	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) // loop over outer tuple elements
	{
		int m=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element))	// loop over inner tuple elements
		{
			mat_RR[n][m] = check_real_arg(SETL_SYSTEM iaa_element, 0, "ntl_get_real_matrix");
			m++;
		}
		ITERATE_TUPLE_END(iaa)
		n++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
	
}

void NTL_GET_SETL_REAL_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target) 
{
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_mat_RR_type, 
			0, "ntl real matrix", "ntl_to_setl_obj");

	mat_RR &mat_RR = *ntl_data->var.mat_RR;
	vec_vec_RR *ibuf = (vec_vec_RR *)&rep(mat_RR);

	int32 elms_in_vector_i = ibuf->length();
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector_i; i++) {
		
		vec_RR &jbuf = mat_RR[i];
		
		int32 elms_in_vector_j = jbuf.length();
		TUPLE_CONSTRUCTOR_BEGIN(caa);
		for (int32 j=0; j<elms_in_vector_j; j++) {
			
			RR &RR = jbuf[j];

			i_real_ptr_type i_real;
			i_get_real(i_real);
			i_real->r_use_count = 1;
			i_real->r_value = to_double(RR);
			
			s.sp_form = ft_real;
			s.sp_val.sp_real_ptr = i_real;
			TUPLE_ADD_CELL(caa,&s);

		}
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
		TUPLE_ADD_CELL(ca,&s);
				
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_REAL_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_real_vector",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_vec_RR_type);	// create the big real vector
	vec_RR &vec_RR = *ntl_data->var.vec_RR;

	int n_col;
	get_tuple_vector_size(SETL_SYSTEM argv, n_col);

	vec_RR.SetLength(n_col);

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) 
	{
		vec_RR[n] = check_real_arg(SETL_SYSTEM ia_element, 0, "ntl_get_real_vector");
		n++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_REAL_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_RR_type, 0, "ntl real vector", "ntl_to_setl_obj");

	vec_RR &ibuf = *ntl_data->var.vec_RR;
	int32 elms_in_vector = ibuf.length();

	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector; i++) {
		
		RR &RR = ibuf[i];
				
		i_real_ptr_type i_real;
		i_get_real(i_real);
		i_real->r_use_count = 1;
		i_real->r_value = to_double(RR);
		
		s.sp_form = ft_real;
		s.sp_val.sp_real_ptr = i_real;
		TUPLE_ADD_CELL(ca,&s);
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_INT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	int32 number_of_bytes;
	short is_negative;
	
	unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM &argv[0], 0, "setl integer", "ntl_get_int", number_of_bytes, is_negative);

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_ZZ_type);

	NewZZFromBytes(*ntl_data->var.ZZ, byte_array, number_of_bytes, is_negative);

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;

	return;

abort:

	if (ntl_data) { 
		smart_delete(ntl_data->var.ZZ);
		delete ntl_data;
	}
	smart_free(byte_array);
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_SETL_INTEGER(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_type, 0, "ntl integer", "ntl_to_setl_obj");

	ZZ& ZZ = *ntl_data->var.ZZ;
	
	int n_of_bits;
	
	if ((n_of_bits = NumBits(ZZ)) > INT_BIT_WIDTH) {
		integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
		if (!integer_item) goto abort;

		unmark_specifier(target);
		target->sp_form = ft_long;
		target->sp_val.sp_long_ptr = integer_item;
	} else {
		int32 long_value;

		long_value = to_long(ZZ);
		
		unmark_specifier(target);
		target->sp_form = ft_short;
		target->sp_val.sp_short_value = long_value;
	}

	return;
	
abort:

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MODINT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	int32 number_of_bytes;
	short is_negative;

	unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM &argv[0], 0, "setl integer", "ntl_get_modint", number_of_bytes, is_negative);

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_ZZ_p_type);

	ZZ ZZ;
	NewZZFromBytes(ZZ, byte_array, number_of_bytes, is_negative);
//	ZZ_p &ZZ_p = *ntl_data->var.ZZ_p;
	*ntl_data->var.ZZ_p = to_ZZ_p(ZZ);

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;

	return;

abort:

	if (ntl_data) { 
		smart_delete(ntl_data->var.ZZ);
		delete ntl_data;
	}
	smart_free(byte_array);
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_SETL_INTEGER_MODULO(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_p_type, 0, "ntl integer modulo", "ntl_to_setl_obj");

	ZZ& ZZ = ntl_data->var.ZZ_p->LoopHole();
	
	if (NumBits(ZZ) > INT_BIT_WIDTH) {
		integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
		if (!integer_item) goto abort;

		unmark_specifier(target);
		target->sp_form = ft_long;
		target->sp_val.sp_long_ptr = integer_item;
	} else {
		int32 long_value;

		long_value = to_long(ZZ);
		
		unmark_specifier(target);
		target->sp_form = ft_short;
		target->sp_val.sp_short_value = long_value;
	}
		
	return;
	
abort:

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MOD2INT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	long long_arg = check_long_arg(SETL_SYSTEM &argv[0], 0, "ntl_get_integer_mod2int");

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_GF2_type);

	*ntl_data->var.GF2 = to_GF2(long_arg % 2);

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;

	return;

abort:

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_SETL_INTEGER_MODULO_2(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_GF2_type, 0, "ntl integer", "ntl_to_setl_obj");

	long long_arg = rep(*ntl_data->var.GF2);
	
	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = long_arg;

	return;
}

void NTL_GET_MOD2INT_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)
	TUPLE_ITERATOR(iaa)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_mod2int_matrix",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_mat_GF2_type);	// create the big integer vector
	mat_GF2 &mat_GF2 = *ntl_data->var.mat_GF2;

	int n_col, n_row;
	get_tuple_matrix_size(SETL_SYSTEM argv, n_col, n_row);

	mat_GF2.SetDims(n_col, n_row);

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) // loop over outer tuple elements
	{
		int m=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element))	// loop over inner tuple elements
		{	
			long long_arg = check_long_arg(SETL_SYSTEM iaa_element, m, "ntl_get_mod2int_matrix");
			mat_GF2[n][m] = long_arg % 2;
			
			m++;
		}
		ITERATE_TUPLE_END(iaa)
		n++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_MODULO_2_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_mat_GF2_type, 
			0, "ntl integer modulo 2 matrix", "ntl_to_setl_obj");

	mat_GF2 &mat_GF2 = *ntl_data->var.mat_GF2;
	vec_vec_GF2 *ibuf = (vec_vec_GF2 *)&rep(mat_GF2);

	int32 elms_in_vector_i = ibuf->length();
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector_i; i++) {
		
		vec_GF2 &jbuf = mat_GF2[i];
		
		int32 elms_in_vector_j = jbuf.length();
		TUPLE_CONSTRUCTOR_BEGIN(caa);
		for (int32 j=0; j<elms_in_vector_j; j++) {
			
			s.sp_form = ft_short;
			s.sp_val.sp_short_value = rep(jbuf[j]);
			TUPLE_ADD_CELL(caa,&s);

		}
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
		TUPLE_ADD_CELL(ca,&s);
				
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MOD2INT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_mod2int_vector",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_vec_GF2_type);	// create the big real vector
	vec_GF2 &vec_GF2 = *ntl_data->var.vec_GF2;

	int n_col;
	get_tuple_vector_size(SETL_SYSTEM argv, n_col);

	vec_GF2.SetLength(n_col);

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) 
	{
		long long_arg = check_long_arg(SETL_SYSTEM ia_element, 0, "ntl_get_mod2int_vector");
		vec_GF2[n] = long_arg % 2;
		n++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_MOD2INT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)		// outer tuple
	TUPLE_ITERATOR(iaa)		// inner tuple

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_mod2int_poly_vector",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_vec_GF2X_type);	// create the big real vector
	vec_GF2X &vec_GF2X = *ntl_data->var.vec_GF2X;

	int n_col;
	get_tuple_vector_size(SETL_SYSTEM argv, n_col);

	vec_GF2X.SetLength(n_col);

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) 
	{
		
		GF2X &GF2X = vec_GF2X[n];
//		WordVector &w_vector = GF2X.xrep;
//		int n_row;
//		get_tuple_vector_size(SETL_SYSTEM ia_element, n_row);
//		w_vector.SetLength(n_row);
		
		int i=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element))
		{
			long long_arg = check_long_arg(SETL_SYSTEM iaa_element, 0, "ntl_get_mod2int_poly_vector");
//			w_vector[i] = (u_long)long_arg % 2;
			SetCoeff(GF2X, i, (long_arg % 2));
			i++;			
		}
		ITERATE_TUPLE_END(iaa)

		n++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_MOD2EXTINT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	// vec_GF2E_type
	TUPLE_ITERATOR(ia)		// outer tuple
	TUPLE_ITERATOR(iaa)		// inner tuple

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_mod2extint_vector",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_vec_GF2E_type);	// create the big real vector
	vec_GF2E &vec_GF2E = *ntl_data->var.vec_GF2E;

	int n_col;
	get_tuple_vector_size(SETL_SYSTEM argv, n_col);

	vec_GF2E.SetLength(n_col);

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) 
	{
		
		GF2E &GF2E = vec_GF2E[n];
//		WordVector &w_vector = GF2E.LoopHole().xrep;
		GF2X &GF2X = GF2E.LoopHole();

//		int n_row;
//		get_tuple_vector_size(SETL_SYSTEM ia_element, n_row);
//		w_vector.SetLength(n_row);
		
		int i=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element))
		{
			long long_arg = check_long_arg(SETL_SYSTEM iaa_element, 0, "ntl_get_mod2int_poly_vector");
//			w_vector[i] = (u_long)long_arg % 2;
			SetCoeff(GF2X, i, (long_arg % 2) );
			i++;			
		}
		ITERATE_TUPLE_END(iaa)

		n++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_MOD2EXTINT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	// vec_GF2EX_type

	TUPLE_ITERATOR(ia)		// outer tuple
	TUPLE_ITERATOR(iaa)		// middle tuple
	TUPLE_ITERATOR(iaaa)	// inner tuple

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_mod2extint_poly_vector",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_vec_GF2EX_type);	// create the big real vector
	vec_GF2EX &vec_GF2EX = *ntl_data->var.vec_GF2EX;

	int n_col;
	get_tuple_vector_size(SETL_SYSTEM argv, n_col);

	vec_GF2EX.SetLength(n_col);

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) 
	{
		
		GF2EX &GF2EX = vec_GF2EX[n];
		vec_GF2E &vec_GF2E = GF2EX.rep;
	
		int n_mid;
		get_tuple_vector_size(SETL_SYSTEM ia_element, n_mid);
		vec_GF2E.SetLength(n_mid);
		
		int l=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element))
		{
			GF2E &GF2E = vec_GF2E[l];
//			WordVector &w_vector = GF2E.LoopHole().xrep;
			GF2X &GF2X = GF2E.LoopHole();			
//			int n_row;
//			get_tuple_vector_size(SETL_SYSTEM ia_element, n_row);
//			w_vector.SetLength(n_row);
			
			int i=0;
			ITERATE_TUPLE_BEGIN(iaaa,(*iaa_element))
			{
				long long_arg = check_long_arg(SETL_SYSTEM iaaa_element, 0, "ntl_get_mod2extint_poly_vector");
//				w_vector[i] = (u_long)long_arg % 2;
				SetCoeff(GF2X, i, (long_arg % 2));
				i++;			
			}
			ITERATE_TUPLE_END(iaaa)

			l++;
		}		
		ITERATE_TUPLE_END(iaa)

		n++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_MODULO_2_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_GF2_type, 0, "ntl integer modulo 2 vector", "ntl_to_setl_obj");

	vec_GF2 &ibuf = *ntl_data->var.vec_GF2;
	int32 elms_in_vector = ibuf.length();

	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector; i++) {
		
		s.sp_form = ft_short;
		s.sp_val.sp_short_value = rep(ibuf[i]);
		TUPLE_ADD_CELL(ca,&s);
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}


void NTL_GET_MOD2INT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_mod2int_poly",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_GF2X_type);	// create the big integer vector
	GF2X &GF2X = *ntl_data->var.GF2X;
//	WordVector &w_vector = ntl_data->var.GF2X->xrep;

//	int n_col;
//	get_tuple_vector_size(SETL_SYSTEM argv, n_col);
//	w_vector.SetLength(n_col);

	int i=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0])
	{
		long long_arg = check_long_arg(SETL_SYSTEM ia_element, i, "ntl_get_mod2int_poly");
		SetCoeff(GF2X, i, (long_arg % 2));
//		w_vector[i] = (u_long)long_arg % 2;
		i++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_MODULO_2_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_GF2X_type, 0, "ntl integer modulo 2 vector", "ntl_to_setl_obj");
	
//	WordVector &w_vector = ntl_data->var.GF2X->xrep;
//	int32 elms_in_vector = w_vector.length();
	
	GF2X &GF2X = *(ntl_data->var.GF2X);
	int32 elms_in_vector = deg(GF2X)+1;

		// remove trailing zeros from count of elements in poly
	short last_zeros = 0;
	for (int32 i=0; i<elms_in_vector; i++) {
//		if (w_vector[i] == 0L)
//			last_zeros ++;
//		else
//			last_zeros = 0;

		if (coeff(GF2X, i) == 0)
			last_zeros ++;
		else
			last_zeros = 0;
	}
	elms_in_vector -= last_zeros;

	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector; i++) {

//		long long_value = w_vector[i];
		long long_value = (coeff(GF2X, i) == 0 ? 0 : 1);

		s.sp_form = ft_short;
		s.sp_val.sp_short_value = long_value;
		TUPLE_ADD_CELL(ca,&s);
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MOD2EXTINT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_mod2extint",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_GF2E_type);	// create the big integer vector
//	WordVector &w_vector = ntl_data->var.GF2E->LoopHole().xrep;
	GF2X &GF2X = ntl_data->var.GF2E->LoopHole();
//	int n_col;
//	get_tuple_vector_size(SETL_SYSTEM argv, n_col);
//	w_vector.SetLength(n_col);

	int i=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0])
	{
		long long_arg = check_long_arg(SETL_SYSTEM ia_element, i, "ntl_get_mod2extint");
//		w_vector[i] = (u_long)long_arg % 2;
		SetCoeff(GF2X, i, (long_arg % 2));
		i++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_MODULO_2_EXT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_GF2E_type, 0, "ntl integer modulo 2 vector", "ntl_to_setl_obj");
	
//	WordVector &w_vector = ntl_data->var.GF2E->LoopHole().xrep;
//	int32 elms_in_vector = w_vector.length();
	GF2X &GF2X = ntl_data->var.GF2E->LoopHole();
	int32 elms_in_vector = deg(GF2X)+1;

	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector; i++) {
//		long long_value = w_vector[i];
		long long_value = (coeff(GF2X, i) == 0 ? 0 : 1);

		s.sp_form = ft_short;
		s.sp_val.sp_short_value = long_value;
		TUPLE_ADD_CELL(ca,&s);
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MOD2EXTINT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)		/* outer iterator */
	TUPLE_ITERATOR(iaa)		/* inner iterator */

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_mod2extint_poly",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_GF2EX_type);
	vec_GF2E &vec_GF2E = ntl_data->var.GF2EX->rep;
	
	int n_row;
	get_tuple_vector_size(SETL_SYSTEM argv, n_row);	// get the number of elements in the vector
	vec_GF2E.SetLength(n_row);
	
	int m=0;
	ITERATE_TUPLE_BEGIN(ia, argv[0])
	{
		GF2X &GF2X = vec_GF2E[m].LoopHole();
//		WordVector &w_vector = GF2X.xrep;

//		int n_col;
//		get_tuple_vector_size(SETL_SYSTEM ia_element, n_col);
//		w_vector.SetLength(n_col);
		
		int n=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element)) 
		{
			int32 number_of_bytes;
			short is_negative;
			
			long long_arg = check_long_arg(SETL_SYSTEM iaa_element, n, "ntl_get_mod2extint_poly");
//			w_vector[n] = (u_long)long_arg % 2;
			
			SetCoeff(GF2X, n, (long_arg % 2));
			
			n++;
		}
		ITERATE_TUPLE_END(iaa)
		m++;
	}
	ITERATE_TUPLE_END(ia)
		
	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;

}

void NTL_GET_SETL_INTEGER_MODULO_2_POLY_EXT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	specifier s;

	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_GF2EX_type, 0, "ntl polynomial over integer modulo 2", "ntl_to_setl_obj");
	vec_GF2E &vec_GF2E = ntl_data->var.GF2EX->rep;
	int elms_in_out_vector = vec_GF2E.length();

	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 j=0; j<elms_in_out_vector; j++) {
		
		GF2X &GF2X = vec_GF2E[j].LoopHole();
//		WordVector &w_vector = GF2X.xrep;		

//		int32 elms_in_inner_vector = w_vector.length();
		int32 elms_in_inner_vector = deg(GF2X)+1;

			// remove trailing zeros from count of elements in poly
		short last_zeros = 0;
		for (int32 i=0; i<elms_in_inner_vector; i++) {
//			if (w_vector[i] == 0L)
//				last_zeros ++;
//			else
//				last_zeros = 0;
			if (coeff(GF2X, i) == 0)
				last_zeros ++;
			else
				last_zeros = 0;
		}
		elms_in_inner_vector -= last_zeros;

		TUPLE_CONSTRUCTOR_BEGIN(caa);
		for (int32 i=0; i<elms_in_inner_vector; i++) {

			s.sp_form = ft_short;
//			s.sp_val.sp_short_value = w_vector[i];
			s.sp_val.sp_short_value = (coeff(GF2X, i) == 0 ? 0 : 1);

			TUPLE_ADD_CELL(caa,&s);
		}
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
		TUPLE_ADD_CELL(ca, &s);

	}
	TUPLE_CONSTRUCTOR_END(ca);
	
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_SET_MODULO(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_type, 0, "ntl integer", "ntl_set_modulo");
	ZZ& ZZ = *ntl_data->var.ZZ;
	
	ZZ_p::init(ZZ);

	unmark_specifier(target);
	target->sp_form = ft_omega;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_INT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_int_poly",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_ZZX_type);	// create the big integer vector
	vec_ZZ &ibuf = ntl_data->var.ZZX->rep;

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) 
	{	
		int32 number_of_bytes;
		short is_negative;
		unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM ia_element, n, "setl tuple integer element", "ntl_get_int_poly", number_of_bytes, is_negative);

		if (n % NTL_VectorInputBlock == 0)
			ibuf.SetMaxLength(n + NTL_VectorInputBlock);
		n++;
		ibuf.SetLength(n);

		NewZZFromBytes(ibuf[n-1], byte_array, number_of_bytes, is_negative);
	}
	ITERATE_TUPLE_END(ia)


	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZX_type, 0, "ntl integer vector", "ntl_to_setl_obj");

	vec_ZZ &ibuf = ntl_data->var.ZZX->rep;
	int32 elms_in_vector = ibuf.length();

		// remove trailing zeros from count of elements in poly
	short last_zeros = 0;
	for (int32 i=0; i<elms_in_vector; i++) {
		ZZ &ZZ = ibuf[i];
		if (ZZ == to_ZZ(0))
			last_zeros ++;
		else
			last_zeros = 0;
	}
	elms_in_vector -= last_zeros;
	
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector; i++) {
		
		ZZ &ZZ = ibuf[i];

		if (NumBits(ZZ) > INT_BIT_WIDTH) {
			integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
			if (!integer_item) goto abort;

			s.sp_form = ft_long;
			s.sp_val.sp_long_ptr = integer_item;
		} else {
			int32 long_value;

			long_value = to_long(ZZ);
			
			s.sp_form = ft_short;
			s.sp_val.sp_short_value = long_value;
		}
		TUPLE_ADD_CELL(ca,&s);
				
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_SETL_INTEGER_POLY_LONG_PAIR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	specifier s;
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	TUPLE_CONSTRUCTOR(caaa)
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_pair_ZZX_long_type, 0, "ntl pair big integer and long vector", "ntl_to_setl_obj");

	vec_pair_ZZX_long &vec = *ntl_data->var.vec_pair_ZZX_long;
	int32 elms_in_vector = vec.length();	// get the number of pairs in ibuf

	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 j=0; j<elms_in_vector; j++) {	// loop over the pairs
		
		TUPLE_CONSTRUCTOR_BEGIN(caa);

		vec_ZZ &ibuf = vec[j].a.rep;
		int32 elms_in_pol = ibuf.length();

			// remove trailing zeros from count of elements in poly
		short last_zeros = 0;
		for (int32 i=0; i<elms_in_pol; i++) {
			ZZ &ZZ = ibuf[i];
			if (ZZ == to_ZZ(0))
				last_zeros ++;
			else
				last_zeros = 0;
		}
		elms_in_pol -= last_zeros;

		TUPLE_CONSTRUCTOR_BEGIN(caaa);
		for (int32 i=0; i<elms_in_pol; i++) {

			ZZ &ZZ = ibuf[i];
			if (NumBits(ZZ) > INT_BIT_WIDTH) {
				integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
				if (!integer_item) goto abort;

				s.sp_form = ft_long;
				s.sp_val.sp_long_ptr = integer_item;
			} else {
				int32 long_value;

				long_value = to_long(ZZ);
				
				s.sp_form = ft_short;
				s.sp_val.sp_short_value = long_value;
			}
			TUPLE_ADD_CELL(caaa,&s);
					
		}
		TUPLE_CONSTRUCTOR_END(caaa);

		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caaa);	// this is a tuple
		TUPLE_ADD_CELL(caa,&s);

			// second pair element		
		s.sp_form = ft_short;
		s.sp_val.sp_short_value = vec[j].b;	// this is a short integer
		TUPLE_ADD_CELL(caa,&s);

		TUPLE_CONSTRUCTOR_END(caa);
				
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);	// this is a tuple
		TUPLE_ADD_CELL(ca,&s);

	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_SETL_INTEGER_POLY_MODULO_LONG_PAIR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	specifier s;
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	TUPLE_CONSTRUCTOR(caaa)

	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_pair_ZZ_pX_long_type, 0, "ntl pair big integer modulo poly and long vector", "ntl_to_setl_obj");
	vec_pair_ZZ_pX_long &vec = *ntl_data->var.vec_pair_ZZ_pX_long;
	
	int32 elms_in_vector = vec.length();	 // get the size of the vector
	
		// create external tuple
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 j=0; j<elms_in_vector; j++) {
		
		ZZ_pX &ZZ_pX_el = vec[j].a;
		long &long_el = vec[j].b;
				
			// create internal pair
		TUPLE_CONSTRUCTOR_BEGIN(caa);

			// first pair element	(ZZ_pX)
		{
			vec_ZZ_p &ibuf = ZZ_pX_el.rep;
			int32 elms_in_pol = ibuf.length();
			
				// remove trailing zeros from count of elements in poly
			short last_zeros = 0;
			for (int32 i=0; i<elms_in_pol; i++) {
				ZZ_p &ZZ_p = ibuf[i];
				if (ZZ_p == to_ZZ_p(0))
					last_zeros ++;
				else
					last_zeros = 0;
			}		
			elms_in_pol -= last_zeros;

			TUPLE_CONSTRUCTOR_BEGIN(caaa);
			for (int32 i=0; i<elms_in_pol; i++) {
				
				ZZ &ZZ = ibuf[i].LoopHole();
				if (NumBits(ZZ) > INT_BIT_WIDTH) {
					integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
					if (!integer_item) goto abort;

					s.sp_form = ft_long;
					s.sp_val.sp_long_ptr = integer_item;
				} else {
					int32 long_value;

					long_value = to_long(ZZ);
					
					s.sp_form = ft_short;
					s.sp_val.sp_short_value = long_value;
				}
				TUPLE_ADD_CELL(caaa,&s);
			}
			TUPLE_CONSTRUCTOR_END(caaa);
		}
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caaa);	// this is a tuple
		TUPLE_ADD_CELL(caa,&s);

			// second pair element	(long)
		{
			s.sp_form = ft_short;
			s.sp_val.sp_short_value = long_el;		
			TUPLE_ADD_CELL(caa, &s);
		}
		
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);	// this is a tuple
		TUPLE_ADD_CELL(ca,&s);
	}
	TUPLE_CONSTRUCTOR_END(ca);

		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_SETL_INTEGER_MODULO_2_POLY_EXT_LONG_PAIR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	specifier s;
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	TUPLE_CONSTRUCTOR(caaa)
	TUPLE_CONSTRUCTOR(caaaa)

	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_pair_GF2EX_long_type, 0, "ntl pair big integer modulo 2 poly ext and long vector", "ntl_to_setl_obj");
	vec_pair_GF2EX_long &vec = *ntl_data->var.vec_pair_GF2EX_long;
	
	int32 elms_in_vector = vec.length();	 // get the size of the vector
	
		// create external tuple
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 j=0; j<elms_in_vector; j++) {
		
		GF2EX &GF2EX_el = vec[j].a;
		long &long_el = vec[j].b;
				
		vec_GF2E &vec_GF2E = GF2EX_el.rep;
		int elms_in_gf2e = vec_GF2E.length();

			// create internal pair
		TUPLE_CONSTRUCTOR_BEGIN(caa);

		TUPLE_CONSTRUCTOR_BEGIN(caaa);
		for (int32 l=0; l<elms_in_gf2e; l++) 	// first pair element	(GF2EX)
		{

			GF2X &GF2X = vec_GF2E[l].LoopHole();

			int32 elms_in_inner_vector = deg(GF2X)+1;			
			
				// remove trailing zeros from count of elements in poly
			short last_zeros = 0;
			
			for (int32 i=0; i<elms_in_inner_vector; i++) {
				if (coeff(GF2X, i) == 0)
					last_zeros ++;
				else
					last_zeros = 0;
			}
			elms_in_inner_vector -= last_zeros;


			TUPLE_CONSTRUCTOR_BEGIN(caaaa);
			for (int32 i=0; i<elms_in_inner_vector; i++) {

				s.sp_form = ft_short;
				s.sp_val.sp_short_value = (coeff(GF2X, i) == 0 ? 0 : 1);

				TUPLE_ADD_CELL(caaaa,&s);
			}
			TUPLE_CONSTRUCTOR_END(caaaa);
			
			s.sp_form = ft_tuple;
			s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caaaa);
			TUPLE_ADD_CELL(caaa, &s);
		}
		TUPLE_CONSTRUCTOR_END(caaa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caaa);	// this is a tuple
		TUPLE_ADD_CELL(caa,&s);

			// second pair element	(long)
		{
			s.sp_form = ft_short;
			s.sp_val.sp_short_value = long_el;		
			TUPLE_ADD_CELL(caa, &s);
		}
		
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);	// this is a tuple
		TUPLE_ADD_CELL(ca,&s);
	}
	TUPLE_CONSTRUCTOR_END(ca);

		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_SETL_INTEGER_POLY_EXT_MODULO_LONG_PAIR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	specifier s;
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	TUPLE_CONSTRUCTOR(caaa)
	TUPLE_CONSTRUCTOR(caaaa)

	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_pair_ZZ_pEX_long_type, 0, "ntl pair big integer modulo poly ext and long vector", "ntl_to_setl_obj");
	vec_pair_ZZ_pEX_long &vec = *ntl_data->var.vec_pair_ZZ_pEX_long;
	
	int32 elms_in_vector = vec.length();	 // get the size of the vector
	
		// create external tuple
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 m=0; m<elms_in_vector; m++) {
		
		ZZ_pEX &ZZ_pEX_el = vec[m].a;
		long &long_el = vec[m].b;
				
			// create internal pair
		TUPLE_CONSTRUCTOR_BEGIN(caa);

			// first pair element	(ZZ_pEX)
		{
			vec_ZZ_pE &vec_ZZ_pE = ZZ_pEX_el.rep;
			int elms_in_out_vector = vec_ZZ_pE.length();

			TUPLE_CONSTRUCTOR_BEGIN(caaa);
			for (int32 j=0; j<elms_in_out_vector; j++) {
				vec_ZZ_p &vec_ZZ_p = vec_ZZ_pE[j].LoopHole().rep;
				int32 elms_in_inner_vector = vec_ZZ_p.length();
				
					// remove trailing zeros from count of elements in poly
				short last_zeros = 0;
				for (int32 i=0; i<elms_in_inner_vector; i++) {
					ZZ &ZZ = vec_ZZ_p[i].LoopHole();
					if (ZZ == to_ZZ(0))
						last_zeros ++;
					else
						last_zeros = 0;
				}
				elms_in_inner_vector -= last_zeros;
				
					// build the poly
				TUPLE_CONSTRUCTOR_BEGIN(caaaa);
				for (int32 i=0; i<elms_in_inner_vector; i++) {

					ZZ &ZZ = vec_ZZ_p[i].LoopHole();

					if (NumBits(ZZ) > INT_BIT_WIDTH) {
						integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
						if (integer_item)	goto abort;
						
						s.sp_form = ft_long;
						s.sp_val.sp_long_ptr = integer_item;
					} else {
						int32 long_value;
						
						long_value = to_long(ZZ);
						
						s.sp_form = ft_short;
						s.sp_val.sp_short_value = long_value;
					}
					
					TUPLE_ADD_CELL(caaaa, &s);

				}
				TUPLE_CONSTRUCTOR_END(caaaa);
				
				s.sp_form = ft_tuple;
				s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caaaa);
				TUPLE_ADD_CELL(caaa, &s);
				
			}
			TUPLE_CONSTRUCTOR_END(caaa);
			
			
		}
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caaa);	// this is a tuple
		TUPLE_ADD_CELL(caa,&s);

			// second pair element	(long)
		{
			s.sp_form = ft_short;
			s.sp_val.sp_short_value = long_el;		
			TUPLE_ADD_CELL(caa, &s);
		}
		
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);	// this is a tuple
		TUPLE_ADD_CELL(ca,&s);
	}
	TUPLE_CONSTRUCTOR_END(ca);

		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_SETL_INTEGER_MODULO_2_POLY_LONG_PAIR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	specifier s;
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	TUPLE_CONSTRUCTOR(caaa)

	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_pair_GF2X_long_type, 0, "ntl pair big integer modulo 2 poly and long vector", "ntl_to_setl_obj");
	vec_pair_GF2X_long &vec = *ntl_data->var.vec_pair_GF2X_long;
	
	int32 elms_in_vector = vec.length();	 // get the size of the vector
	
		// create external tuple
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 j=0; j<elms_in_vector; j++) {
		
		GF2X &GF2X = vec[j].a;
		long &long_el = vec[j].b;
				
		int elms_in_inner_vector = deg(GF2X)+1;

		TUPLE_CONSTRUCTOR_BEGIN(caa);

			// first pair element
		{
				// remove trailing zeros from count of elements in poly
			short last_zeros = 0;
			
			for (int32 i=0; i<elms_in_inner_vector; i++) {

				if (coeff(GF2X, i) == 0)
					last_zeros ++;
				else
					last_zeros = 0;
			}
			elms_in_inner_vector -= last_zeros;

			TUPLE_CONSTRUCTOR_BEGIN(caaa);
			for (int32 i=0; i<elms_in_inner_vector; i++) {

				s.sp_form = ft_short;
				s.sp_val.sp_short_value = (coeff(GF2X, i) == 0 ? 0 : 1);

				TUPLE_ADD_CELL(caaa,&s);
			}
			TUPLE_CONSTRUCTOR_END(caaa);
			
			s.sp_form = ft_tuple;
			s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caaa);
			TUPLE_ADD_CELL(caa, &s);
		}
		
			// second pair element	(long)
		{
			s.sp_form = ft_short;
			s.sp_val.sp_short_value = long_el;		
			TUPLE_ADD_CELL(caa, &s);
		}
		
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);	// this is a tuple
		TUPLE_ADD_CELL(ca,&s);
	}
	TUPLE_CONSTRUCTOR_END(ca);

		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:

	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MODINT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_modint_poly",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_ZZ_pX_type);	// create the big integer vector
	vec_ZZ_p &ibuf = ntl_data->var.ZZ_pX->rep;

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) 
	{	
		int32 number_of_bytes;
		short is_negative;
		unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM ia_element, n, "setl tuple integer element", "ntl_get_integer_poly", number_of_bytes, is_negative);

		if (n % NTL_VectorInputBlock == 0)
			ibuf.SetMaxLength(n + NTL_VectorInputBlock);
		n++;
		ibuf.SetLength(n);

		ZZ ZZ;
		NewZZFromBytes(ZZ, byte_array, number_of_bytes, is_negative);
		ibuf[n-1] = to_ZZ_p(ZZ);
		
	}
	ITERATE_TUPLE_END(ia)


	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_POLY_MODULO(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_pX_type, 0, "ntl integer vector", "ntl_to_setl_obj");

	vec_ZZ_p &ibuf = ntl_data->var.ZZ_pX->rep;
	int32 elms_in_vector = ibuf.length();

		// remove trailing zeros from count of elements in poly
	short last_zeros = 0;
	for (int32 i=0; i<elms_in_vector; i++) {
		ZZ &ZZ = ibuf[i].LoopHole();
		if (ZZ == to_ZZ(0))
			last_zeros ++;
		else
			last_zeros = 0;
	}
	elms_in_vector -= last_zeros;

	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector; i++) {
		
		ZZ &ZZ = ibuf[i].LoopHole();
		if (NumBits(ZZ) > INT_BIT_WIDTH) {
			integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
			if (!integer_item) goto abort;

			s.sp_form = ft_long;
			s.sp_val.sp_long_ptr = integer_item;
		} else {
			int32 long_value;

			long_value = to_long(ZZ);
			
			s.sp_form = ft_short;
			s.sp_val.sp_short_value = long_value;
		}
		TUPLE_ADD_CELL(ca,&s);
				
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

//
//
//	General Add operation schema
//
//

#ifdef ORIGINALS
void NTL_ADD(SETL_SYSTEM_PROTO int argc, specifier *argv,specifier* target);

void NTL_ADD(SETL_SYSTEM_PROTO int argc, specifier *argv,specifier* target)
{
	int32 sub_type;

	ntl_struct *ntl_data1 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "ntl type", "ntl_add");
	ntl_struct *ntl_data2 = check_ntl_arg(SETL_SYSTEM &argv[1], ntl_any_type, 1, "ntl type", "ntl_add");

	if ((sub_type = ntl_data1->sub_type) != ntl_data2->sub_type)
		abend(SETL_SYSTEM "not allowed mixed type in %s operation", "add");
		
	ntl_struct *ntl_data0 = new ntl_struct(SETL_SYSTEM sub_type);

	switch (sub_type) {
		case ntl_ZZ_type:
			add(*ntl_data0->var.ZZ, *ntl_data1->var.ZZ, *ntl_data2->var.ZZ);		// call the function
			break;
		case ntl_ZZX_type:
			add(*ntl_data0->var.ZZX, *ntl_data1->var.ZZX, *ntl_data2->var.ZZX);		// call the function
			break;
		case ntl_ZZ_p_type:
			add(*ntl_data0->var.ZZ_p, *ntl_data1->var.ZZ_p, *ntl_data2->var.ZZ_p);		// call the function
			break;
		case ntl_ZZ_pX_type:
			add(*ntl_data0->var.ZZ_pX, *ntl_data1->var.ZZ_pX, *ntl_data2->var.ZZ_pX);		// call the function
			break;
		default:
			delete ntl_data0;
			abend(SETL_SYSTEM "not allowed type in %s", "add");
			break;
	}

			/*return one item*/
	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data0;
}
#endif // ORIGINALS

void NTL_GET_INT_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)
	TUPLE_ITERATOR(iaa)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_int_matrix",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_mat_ZZ_type);	// create the big integer vector
	mat_ZZ &mat_ZZ = *ntl_data->var.mat_ZZ;

	int n_col, n_row;
	get_tuple_matrix_size(SETL_SYSTEM argv, n_col, n_row);

	mat_ZZ.SetDims(n_col, n_row);

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) // loop over outer tuple elements
	{
		int m=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element))	// loop over inner tuple elements
		{	
			int32 number_of_bytes;
			short is_negative;
			unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM iaa_element, m, 
					"setl tuple integer element", "ntl_get_int_matrix", number_of_bytes, is_negative);

			NewZZFromBytes(mat_ZZ[n][m], byte_array, number_of_bytes, is_negative);
			m++;
		}
		ITERATE_TUPLE_END(iaa)
		n++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_mat_ZZ_type, 
			0, "ntl integer matrix", "ntl_to_setl_obj");

	mat_ZZ &mat_ZZ = *ntl_data->var.mat_ZZ;
	vec_vec_ZZ *ibuf = (vec_vec_ZZ *)&rep(mat_ZZ);

	int32 elms_in_vector_i = ibuf->length();
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector_i; i++) {
		
		vec_ZZ &jbuf = mat_ZZ[i];
		
		int32 elms_in_vector_j = jbuf.length();
		TUPLE_CONSTRUCTOR_BEGIN(caa);
		for (int32 j=0; j<elms_in_vector_j; j++) {
			
			ZZ &ZZ = jbuf[j];
			if (NumBits(ZZ) > INT_BIT_WIDTH) {
				integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
				if (!integer_item) goto abort;

				s.sp_form = ft_long;
				s.sp_val.sp_long_ptr = integer_item;
			} else {
				int32 long_value;

				long_value = to_long(ZZ);
				
				s.sp_form = ft_short;
				s.sp_val.sp_short_value = long_value;
			}
			TUPLE_ADD_CELL(caa,&s);

		}
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
		TUPLE_ADD_CELL(ca,&s);
				
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MODINT_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)
	TUPLE_ITERATOR(iaa)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_modint_matrix",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_mat_ZZ_p_type);	// create the big integer vector
	mat_ZZ_p &mat_ZZ_p = *ntl_data->var.mat_ZZ_p;
	vec_vec_ZZ_p *ibuf = (vec_vec_ZZ_p *)&rep(mat_ZZ_p);

	int n_col, n_row;
	get_tuple_matrix_size(SETL_SYSTEM argv, n_col, n_row);
	mat_ZZ_p.SetDims(n_col, n_row);

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) // loop over outer tuple elements
	{
		vec_ZZ_p &jbuf = mat_ZZ_p[n];		

		int m=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element))	// loop over inner tuple elements
		{	
			int32 number_of_bytes;
			short is_negative;

			unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM iaa_element, m, 
					"setl tuple integer element", "ntl_get_integer_matrix", number_of_bytes, is_negative);

			ZZ ZZ;
			NewZZFromBytes(ZZ, byte_array, number_of_bytes, is_negative);
			jbuf[m] = to_ZZ_p(ZZ);
			
			m++;
		}
		ITERATE_TUPLE_END(iaa)
		n++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_MODULO_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_mat_ZZ_p_type, 
			0, "ntl integer matrix", "ntl_to_setl_obj");

	mat_ZZ_p &mat_ZZ_p = *ntl_data->var.mat_ZZ_p;
	vec_vec_ZZ_p *ibuf = (vec_vec_ZZ_p *)&rep(mat_ZZ_p);

	int32 elms_in_vector_i = ibuf->length();
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector_i; i++) {
		
		vec_ZZ_p &jbuf = mat_ZZ_p[i];
		
		int32 elms_in_vector_j = jbuf.length();
		TUPLE_CONSTRUCTOR_BEGIN(caa);
		for (int32 j=0; j<elms_in_vector_j; j++) {
			
			ZZ &ZZ = jbuf[j].LoopHole();
			if (NumBits(ZZ) > INT_BIT_WIDTH) {
				integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
				if (!integer_item) goto abort;

				s.sp_form = ft_long;
				s.sp_val.sp_long_ptr = integer_item;
			} else {
				int32 long_value;

				long_value = to_long(ZZ);
				
				s.sp_form = ft_short;
				s.sp_val.sp_short_value = long_value;
			}
			TUPLE_ADD_CELL(caa,&s);

		}
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
		TUPLE_ADD_CELL(ca,&s);
				
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}
 
void NTL_NUMBITS(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_type, 0, "ntl integer", "ntl_num_bits");
	ZZ& ZZ = *ntl_data->var.ZZ;
	
	long nBits = NumBits(ZZ);

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = (int32)nBits;
}

void NTL_RANDOMBND(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	ntl_struct *ntl_data1 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_type, 0, "ntl integer", "ntl_RandomBnd");
	ZZ& ZZ = *ntl_data1->var.ZZ;
	
	ntl_struct *ntl_data0 = new ntl_struct(SETL_SYSTEM ntl_ZZ_type);	// generate a big integer for the result

	RandomBnd(*ntl_data0->var.ZZ,ZZ);

			/*return one item*/
	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data0;
}

void NTL_INVMOD(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	ntl_struct *ntl_data1 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_type, 0, "ntl integer", "ntl_InvMod");
	ntl_struct *ntl_data2 = check_ntl_arg(SETL_SYSTEM &argv[1], ntl_ZZ_type, 1, "ntl integer", "ntl_InvMod");

	ZZ& ZZ1 = *ntl_data1->var.ZZ;
	ZZ& ZZ2 = *ntl_data2->var.ZZ;
	
	ntl_struct *ntl_data0 = new ntl_struct(SETL_SYSTEM ntl_ZZ_type);	// generate a big integer for the result

	InvMod(*ntl_data0->var.ZZ, ZZ1, ZZ2);

			/*return one item*/
	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data0;
}

void NTL_BIT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	ntl_struct *ntl_data1 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_type, 0, "ntl integer", "ntl_bit");
	if (argv[1].sp_form != ft_short) 
		abend(SETL_SYSTEM msg_bad_arg, "integer",2,"ntl_bit",abend_opnd_str(SETL_SYSTEM argv));
	long long_data = argv[1].sp_val.sp_short_value;
	
	long bit_value = bit(*ntl_data1->var.ZZ, long_data);

	unmark_specifier(target);
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = (int32)bit_value;

}

void NTL_GET_INT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_int_vector",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_vec_ZZ_type);	// create the big integer vector
	vec_ZZ &ibuf = *ntl_data->var.vec_ZZ;

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) 
	{	
		int32 number_of_bytes;
		short is_negative;
		unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM ia_element, n, "setl tuple integer element", "ntl_get_int_vector", number_of_bytes, is_negative);

		if (n % NTL_VectorInputBlock == 0)
			ibuf.SetMaxLength(n + NTL_VectorInputBlock);
		n++;
		ibuf.SetLength(n);

		NewZZFromBytes(ibuf[n-1], byte_array, number_of_bytes, is_negative);
	}
	ITERATE_TUPLE_END(ia)


	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_ZZ_type, 0, "ntl integer vector", "ntl_to_setl_obj");

	vec_ZZ &ibuf = *ntl_data->var.vec_ZZ;
	int32 elms_in_vector = ibuf.length();

	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector; i++) {
		
		ZZ &ZZ = ibuf[i];
		if (NumBits(ZZ) > INT_BIT_WIDTH) {
			integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
			if (!integer_item) goto abort;

			s.sp_form = ft_long;
			s.sp_val.sp_long_ptr = integer_item;
		} else {
			int32 long_value;

			long_value = to_long(ZZ);
			
			s.sp_form = ft_short;
			s.sp_val.sp_short_value = long_value;
		}
		TUPLE_ADD_CELL(ca,&s);
				
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MODINT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_modint_vector",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_vec_ZZ_p_type);	// create the big integer vector
	vec_ZZ_p &ibuf = *ntl_data->var.vec_ZZ_p;

	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) 
	{	
		int32 number_of_bytes;
		short is_negative;
		unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM ia_element, n, "setl tuple integer element", "ntl_get_modint_vector", number_of_bytes, is_negative);

		if (n % NTL_VectorInputBlock == 0)
			ibuf.SetMaxLength(n + NTL_VectorInputBlock);
		n++;
		ibuf.SetLength(n);

		ZZ ZZ;
		NewZZFromBytes(ZZ, byte_array, number_of_bytes, is_negative);	
		ibuf[n-1] = to_ZZ_p(ZZ);
	}
	ITERATE_TUPLE_END(ia)


	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_MODULO_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_ZZ_p_type, 0, "ntl integer modulo vector", "ntl_to_setl_obj");

	vec_ZZ_p &ibuf = *ntl_data->var.vec_ZZ_p;
	int32 elms_in_vector = ibuf.length();

	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector; i++) {

		ZZ &ZZ = ibuf[i].LoopHole();
		if (NumBits(ZZ) > INT_BIT_WIDTH) {
			integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
			if (!integer_item) goto abort;

			s.sp_form = ft_long;
			s.sp_val.sp_long_ptr = integer_item;
		} else {
			int32 long_value;

			long_value = to_long(ZZ);
			
			s.sp_form = ft_short;
			s.sp_val.sp_short_value = long_value;
		}
		TUPLE_ADD_CELL(ca,&s);
				
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MODEXTINT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_modextint",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_ZZ_pE_type);	// create the big integer vector
	vec_ZZ_p &vec_ZZ_p = ntl_data->var.ZZ_pE->LoopHole().rep;

	int n_col;	
	get_tuple_vector_size(SETL_SYSTEM argv, n_col);
	vec_ZZ_p.SetLength(n_col);	// expand the representing vector to needed size

		/*
		 *	Insert the elements in the representing vector
		 */
	int n=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) 
	{	
		int32 number_of_bytes;
		short is_negative;
		unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM ia_element, n, 
				"setl tuple integer element", "ntl_get_modextint", number_of_bytes, is_negative);

		ZZ ZZ;
		NewZZFromBytes(ZZ, byte_array, number_of_bytes, is_negative);
		vec_ZZ_p[n] = to_ZZ_p(ZZ);
		
		n++;
	}
	ITERATE_TUPLE_END(ia)

	ZZ_pE one = to_ZZ_pE(1);
	ZZ_pE &ZZ_pE = *ntl_data->var.ZZ_pE;
	ZZ_pE = ZZ_pE * one;

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_MODULO_EXT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_pE_type, 0, "ntl modulo int extension", "ntl_to_setl_obj");

	vec_ZZ_p &vec_ZZ_p = ntl_data->var.ZZ_pE->LoopHole().rep;
	int32 elms_in_vector = vec_ZZ_p.length();



	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector; i++) {
		
		ZZ &ZZ = vec_ZZ_p[i].LoopHole();

		if (NumBits(ZZ) > INT_BIT_WIDTH) {
			integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
			if (!integer_item) goto abort;

			s.sp_form = ft_long;
			s.sp_val.sp_long_ptr = integer_item;
		} else {
			int32 long_value;

			long_value = to_long(ZZ);
			
			s.sp_form = ft_short;
			s.sp_val.sp_short_value = long_value;
		}
		TUPLE_ADD_CELL(ca,&s);
				
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MODEXTINT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ivect)
	TUPLE_ITERATOR(ia)

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_modextint_vector",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_vec_ZZ_pE_type);	// create the big integer vector
	vec_ZZ_pE &vec_ZZ_pE = *ntl_data->var.vec_ZZ_pE;

	int n_col;	
	get_tuple_vector_size(SETL_SYSTEM argv, n_col);
	vec_ZZ_pE.SetLength(n_col);	// expand the representing vector to needed size

	ZZ_pE one = to_ZZ_pE(1);
	int m=0;
	ITERATE_TUPLE_BEGIN(ivect, argv[0])
	{
		ZZ_pE &ZZ_pE = vec_ZZ_pE[m];
		vec_ZZ_p &vec_ZZ_p = ZZ_pE.LoopHole().rep;	// dummy

		int n_row;	
		get_tuple_vector_size(SETL_SYSTEM ivect_element, n_row);
		vec_ZZ_p.SetLength(n_row);	// expand the representing vector to needed size

			/*
			 *	Insert the elements in the representing vector
			 */
		int n=0;
		ITERATE_TUPLE_BEGIN(ia,(*ivect_element)) 
		{	
			int32 number_of_bytes;
			short is_negative;
			unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM ia_element, n, 
					"setl tuple integer element", "ntl_get_modextint_vector", number_of_bytes, is_negative);

			ZZ ZZ;
			NewZZFromBytes(ZZ, byte_array, number_of_bytes, is_negative);
			vec_ZZ_p[n] = to_ZZ_p(ZZ);
			
			n++;
		}
		ITERATE_TUPLE_END(ia);

		ZZ_pE = ZZ_pE * one;
		m++;
	}
	ITERATE_TUPLE_END(ivect)
	
	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_MODULO_EXT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
//	ntl_vec_ZZ_pE_type
	TUPLE_CONSTRUCTOR(cvect)
	TUPLE_CONSTRUCTOR(ca)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_ZZ_pE_type, 0, "ntl modulo int extension", "ntl_to_setl_obj");

	vec_ZZ_pE &vec_ZZ_pE = *ntl_data->var.vec_ZZ_pE;
	int32 elms_in_outer_vector = vec_ZZ_pE.length();

	TUPLE_CONSTRUCTOR_BEGIN(cvect);
	for (int32 j=0; j<elms_in_outer_vector; j++) {

		vec_ZZ_p &vec_ZZ_p = vec_ZZ_pE[j].LoopHole().rep;
		int32 elms_in_vector = vec_ZZ_p.length();

		TUPLE_CONSTRUCTOR_BEGIN(ca);
		for (int32 i=0; i<elms_in_vector; i++) {
			
			ZZ &ZZ = vec_ZZ_p[i].LoopHole();

			if (NumBits(ZZ) > INT_BIT_WIDTH) {
				integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
				if (!integer_item) goto abort;

				s.sp_form = ft_long;
				s.sp_val.sp_long_ptr = integer_item;
			} else {
				int32 long_value;

				long_value = to_long(ZZ);
				
				s.sp_form = ft_short;
				s.sp_val.sp_short_value = long_value;
			}
			TUPLE_ADD_CELL(ca,&s);
					
		}
		TUPLE_CONSTRUCTOR_END(ca);

		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		TUPLE_ADD_CELL(cvect,&s);
	}
	TUPLE_CONSTRUCTOR_END(cvect);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(cvect);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_SET_EXT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "ntl polynomial over integer modulo p or 2", "ntl_set_ext_poly");

	switch(ntl_data->sub_type) {
		case ntl_ZZ_pX_type:
			ZZ_pX &ZZ_pX = *ntl_data->var.ZZ_pX;
			ZZ_pE::init(ZZ_pX);
			break;
		case ntl_GF2X_type:
			GF2X &GF2X = *ntl_data->var.GF2X;
			GF2E::init(GF2X);
			break;
		default:
			abend(SETL_SYSTEM "Not allowed type in set modulo ext");
			break;
	}

//	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_pX_type, 0, "ntl polynomial over integer modulo p", "ntl_set_modulo_ext");
//	ZZ_pX &ZZ_pX = *ntl_data->var.ZZ_pX;
//	
//	ZZ_pE::init(ZZ_pX);
}

void NTL_GET_MODEXTINT_POLY(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)		/* outer iterator */
	TUPLE_ITERATOR(iaa)		/* inner iterator */

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_modextint_poly",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_ZZ_pEX_type);
	vec_ZZ_pE &vec_ZZ_pE = ntl_data->var.ZZ_pEX->rep;
	
	int n_row;
	get_tuple_vector_size(SETL_SYSTEM argv, n_row);	// get the number of elements in the vector
	vec_ZZ_pE.SetLength(n_row);
	
	int m=0;
	ITERATE_TUPLE_BEGIN(ia, argv[0])
	{
		vec_ZZ_p &vec_ZZ_p = vec_ZZ_pE[m].LoopHole().rep;
		
		int n_col;
		get_tuple_vector_size(SETL_SYSTEM ia_element, n_col);
		vec_ZZ_p.SetLength(n_col);
		
		int n=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element)) 
		{
			int32 number_of_bytes;
			short is_negative;
			unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM iaa_element, n,
				"setl tuple integer element", "ntl_get_modextint_poly", number_of_bytes, is_negative);
				
			ZZ ZZ;
			NewZZFromBytes(ZZ, byte_array, number_of_bytes, is_negative);
			vec_ZZ_p[n] = to_ZZ_p(ZZ);
			
			n++;
		}
		ITERATE_TUPLE_END(iaa)
		m++;
	}
	ITERATE_TUPLE_END(ia)
		
	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_MODULO_POLY_EXT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	specifier s;

	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_pEX_type, 0, "ntl polynomial over integer modulo p", "ntl_to_setl_obj");
	vec_ZZ_pE &vec_ZZ_pE = ntl_data->var.ZZ_pEX->rep;
	int elms_in_out_vector = vec_ZZ_pE.length();

	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 j=0; j<elms_in_out_vector; j++) {

		vec_ZZ_p &vec_ZZ_p = vec_ZZ_pE[j].LoopHole().rep;
		int32 elms_in_inner_vector = vec_ZZ_p.length();

			// remove trailing zeros from count of elements in poly
		short last_zeros = 0;
		for (int32 i=0; i<elms_in_inner_vector; i++) {
			ZZ &ZZ = vec_ZZ_p[i].LoopHole();
			if (ZZ == to_ZZ(0))
				last_zeros ++;
			else
				last_zeros = 0;
		}
		elms_in_inner_vector -= last_zeros;

			// build the poly
		TUPLE_CONSTRUCTOR_BEGIN(caa);
		for (int32 i=0; i<elms_in_inner_vector; i++) {
		
			ZZ &ZZ = vec_ZZ_p[i].LoopHole();
			
			if (NumBits(ZZ) > INT_BIT_WIDTH) {
				integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
				if (!integer_item) goto abort;

				s.sp_form = ft_long;
				s.sp_val.sp_long_ptr = integer_item;
			} else {
				int32 long_value;

				long_value = to_long(ZZ);
				
				s.sp_form = ft_short;
				s.sp_val.sp_short_value = long_value;
			}
		
			TUPLE_ADD_CELL(caa,&s);
		
		}
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
		TUPLE_ADD_CELL(ca, &s);

	}
	TUPLE_CONSTRUCTOR_END(ca);
	
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MODINT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)		/* outer iterator */
	TUPLE_ITERATOR(iaa)		/* inner iterator */

	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_modint_poly_vector",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_vec_ZZ_pX_type);	// create the big integer vector
	vec_ZZ_pX &vec_ZZ_pX = *ntl_data->var.vec_ZZ_pX;

	int n_row;
	get_tuple_vector_size(SETL_SYSTEM argv, n_row);	// get the number of elements in the vector
	vec_ZZ_pX.SetLength(n_row);	// set the size of the vector

	int m=0;
	ITERATE_TUPLE_BEGIN(ia, argv[0])
	{
		ZZ_pX &ZZ_pX = vec_ZZ_pX[m];
		vec_ZZ_p &vec_ZZ_p = ZZ_pX.rep;
		
		int n_col;
		get_tuple_vector_size(SETL_SYSTEM ia_element, n_col);
		vec_ZZ_p.SetLength(n_col);	// set the size of the poly
		
		int n=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element)) 
		{
			int32 number_of_bytes;
			short is_negative;
			unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM iaa_element, n, 
					"setl tuple integer element", "ntl_get_integer_poly", number_of_bytes, is_negative);
		
			ZZ ZZ;
			NewZZFromBytes(ZZ, byte_array, number_of_bytes, is_negative);
			vec_ZZ_p[n] = to_ZZ_p(ZZ);
			n++;
		}
		ITERATE_TUPLE_END(iaa)
		m++;
	}
	ITERATE_TUPLE_END(ia)
		
	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_POLY_MODULO_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_ZZ_pX_type, 0, "ntl integer vector", "ntl_to_setl_obj");

	vec_ZZ_pX &ibuf = *ntl_data->var.vec_ZZ_pX;
	int32 elms_in_outer_vector = ibuf.length();

	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 j=0; j<elms_in_outer_vector; j++) {
		
		ZZ_pX &ZZ_pX = ibuf[j];
		vec_ZZ_p &vec_ZZ_p = ZZ_pX.rep;
		
		int32 elms_in_inner_vector = vec_ZZ_p.length();
		
			// remove trailing zeros from count of elements in poly
		short last_zeros = 0;
		for (int32 i=0; i<elms_in_inner_vector; i++) {
			ZZ &ZZ = vec_ZZ_p[i].LoopHole();
			if (ZZ == to_ZZ(0))
				last_zeros ++;
			else
				last_zeros = 0;
		}
		elms_in_inner_vector -= last_zeros;
		
		TUPLE_CONSTRUCTOR_BEGIN(caa);
		for (int32 i=0; i<elms_in_inner_vector; i++) {
			
			ZZ &ZZ = vec_ZZ_p[i].LoopHole();
			
			if (NumBits(ZZ) > INT_BIT_WIDTH) {
				integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
				if (!integer_item) goto abort;

				s.sp_form = ft_long;
				s.sp_val.sp_long_ptr = integer_item;
			} else {
				int32 long_value;

				long_value = to_long(ZZ);
				
				s.sp_form = ft_short;
				s.sp_val.sp_short_value = long_value;
			}
			TUPLE_ADD_CELL(caa,&s);
		}
		TUPLE_CONSTRUCTOR_END(caa);
				
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
		TUPLE_ADD_CELL(ca,&s);

	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MODEXTINT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ivect)	/* vector iterator */
	TUPLE_ITERATOR(ia)		/* outer iterator */
	TUPLE_ITERATOR(iaa)		/* inner iterator */
	
	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_modextint_poly_vector",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_vec_ZZ_pEX_type);
	vec_ZZ_pEX &vec_ZZ_pEX = *ntl_data->var.vec_ZZ_pEX;
	
	int n_vec;
	get_tuple_vector_size(SETL_SYSTEM argv, n_vec);
	vec_ZZ_pEX.SetLength(n_vec);

		// loop through the elements in the vector
	int l=0;
	ITERATE_TUPLE_BEGIN(ivect, argv[0])
	{
			ZZ_pEX &ZZ_pEX = vec_ZZ_pEX[l];
			vec_ZZ_pE &vec_ZZ_pE = ZZ_pEX.rep;
						
			int n_row;
			get_tuple_vector_size(SETL_SYSTEM ivect_element, n_row);
			vec_ZZ_pE.SetLength(n_row);
			
			int m=0;
			ITERATE_TUPLE_BEGIN(ia, (*ivect_element))
			{
				vec_ZZ_p &vec_ZZ_p = vec_ZZ_pE[m].LoopHole().rep;
				
				int n_col;
				get_tuple_vector_size(SETL_SYSTEM ia_element, n_col);
				vec_ZZ_p.SetLength(n_col);
				
				int n=0;
				ITERATE_TUPLE_BEGIN(iaa,(*ia_element)) 
				{
					int32 number_of_bytes;
					short is_negative;
					unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM iaa_element, n,
						"setl tuple integer element", "ntl_get_modextint_poly_vector", number_of_bytes, is_negative);
						
					ZZ ZZ;
					NewZZFromBytes(ZZ, byte_array, number_of_bytes, is_negative);
					vec_ZZ_p[n] = to_ZZ_p(ZZ);
					
					n++;
				}
				ITERATE_TUPLE_END(iaa)
				m++;
			}
			ITERATE_TUPLE_END(ia)
			
			l++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

void NTL_GET_SETL_INTEGER_MODULO_POLY_EXT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(cvect)
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_ZZ_pEX_type, 0, "ntl integer vector", "ntl_to_setl_obj");

	vec_ZZ_pEX &vec_ZZ_pEX = *ntl_data->var.vec_ZZ_pEX;
	int32 elms_in_vector = vec_ZZ_pEX.length();

	TUPLE_CONSTRUCTOR_BEGIN(cvect);
	for (int32 l=0; l<elms_in_vector; l++) {
		ZZ_pEX &ZZ_pEX = vec_ZZ_pEX[l];
		vec_ZZ_pE &vec_ZZ_pE = ZZ_pEX.rep;

		int32 elms_in_outer_vector = vec_ZZ_pE.length();
		
		TUPLE_CONSTRUCTOR_BEGIN(ca);
		for (int32 j=0; j<elms_in_outer_vector; j++) {
			
			vec_ZZ_p &vec_ZZ_p = vec_ZZ_pE[j].LoopHole().rep;

			int32 elms_in_inner_vector = vec_ZZ_p.length();
			
				// remove trailing zeros from count of elements in poly
			short last_zeros = 0;
			for (int32 i=0; i<elms_in_inner_vector; i++) {
				ZZ &ZZ = vec_ZZ_p[i].LoopHole();
				if (ZZ == to_ZZ(0))
					last_zeros ++;
				else
					last_zeros = 0;
			}			
			
			TUPLE_CONSTRUCTOR_BEGIN(caa);
			for (int32 i=0; i<elms_in_inner_vector; i++) {
				
				ZZ &ZZ = vec_ZZ_p[i].LoopHole();
				
				if (NumBits(ZZ) > INT_BIT_WIDTH) {
					integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
					if (!integer_item) goto abort;

					s.sp_form = ft_long;
					s.sp_val.sp_long_ptr = integer_item;
				} else {
					int32 long_value;

					long_value = to_long(ZZ);
					
					s.sp_form = ft_short;
					s.sp_val.sp_short_value = long_value;
				}
				TUPLE_ADD_CELL(caa,&s);
			}
			TUPLE_CONSTRUCTOR_END(caa);
						
			s.sp_form = ft_tuple;
			s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
			TUPLE_ADD_CELL(ca,&s);		

		}		
		TUPLE_CONSTRUCTOR_END(ca);	
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		TUPLE_ADD_CELL(cvect,&s);		
	
			
	}
	TUPLE_CONSTRUCTOR_END(cvect);

		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);

	return;

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_SETL_MOD2INT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_GF2X_type, 0, "ntl integer modulo 2 poly vector", "ntl_to_setl_obj");

	vec_GF2X &vec_GF2X = *ntl_data->var.vec_GF2X;
	int32 elms_in_vector = vec_GF2X.length();
	
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 i=0; i<elms_in_vector; i++) {

		GF2X &GF2X = vec_GF2X[i];
//		WordVector &w_vector = GF2X.xrep;
//		int32 elms_in_inner_vector = w_vector.length();
		int32 elms_in_inner_vector = deg(GF2X) + 1;
				
			// remove trailing zeros from count of elements in poly
		short last_zeros = 0;
		for (int32 j=0; i<elms_in_inner_vector; i++) {
//			if (w_vector[j] == 0L)
//				last_zeros ++;
//			else
//				last_zeros = 0;
			if (coeff(GF2X, j) == 0)
				last_zeros ++;
			else
				last_zeros = 0;
		}
		elms_in_inner_vector -= last_zeros;
		
		TUPLE_CONSTRUCTOR_BEGIN(caa);
		for (int32 j=0; j<elms_in_inner_vector; j++) {
//			long long_value = w_vector[j];
			long long_value = (coeff(GF2X, i) == 0 ? 0 : 1);
			
			s.sp_form = ft_short;
			s.sp_val.sp_short_value = long_value;
			TUPLE_ADD_CELL(caa, &s);
		}
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
		TUPLE_ADD_CELL(ca, &s);
	}
	TUPLE_CONSTRUCTOR_END(ca);

		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_SETL_MOD2EXTINT_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca);
	TUPLE_CONSTRUCTOR(caa);
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_GF2E_type, 0, "ntl integer modulo 2 extension vector", "ntl_to_setl_obj");

	vec_GF2E &vec_GF2E = *ntl_data->var.vec_GF2E;
	int32 elms_in_vector = vec_GF2E.length();
	
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 j=0; j<elms_in_vector; j++) {
		GF2E &GF2E = vec_GF2E[j];

//		WordVector &w_vector = GF2E.LoopHole().xrep;
//		int32 elms_in_inner_vector = w_vector.length();
		GF2X &GF2X = GF2E.LoopHole();
		int32 elms_in_inner_vector = deg(GF2X)+1;

		TUPLE_CONSTRUCTOR_BEGIN(caa);
		for (int32 i=0; i<elms_in_vector; i++) {
//			long long_value = w_vector[i];
			long long_value = (coeff(GF2X, i) == 0 ? 0 : 1);

			s.sp_form = ft_short;
			s.sp_val.sp_short_value = long_value;
			TUPLE_ADD_CELL(caa,&s);
		}
		TUPLE_CONSTRUCTOR_END(caa);

		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
		TUPLE_ADD_CELL(ca, &s);

	}
	TUPLE_CONSTRUCTOR_END(ca);

		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
	
	return;

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_SETL_MOD2EXTINT_POLY_VECTOR(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_CONSTRUCTOR(ca_vect);
	TUPLE_CONSTRUCTOR(ca);
	TUPLE_CONSTRUCTOR(caa);
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_vec_GF2EX_type, 0, "ntl integer modulo 2 extension poly vector", "ntl_to_setl_obj");
	
	vec_GF2EX &vec_GF2EX =  *ntl_data->var.vec_GF2EX;
	int32 elms_in_vector = vec_GF2EX.length();
	
	TUPLE_CONSTRUCTOR_BEGIN(ca_vect);
	for (int32 l=0; l<elms_in_vector; l++) {
		GF2EX &GF2EX = vec_GF2EX[l];

		vec_GF2E &vec_GF2E = GF2EX.rep;
		int elms_in_out_vector = vec_GF2E.length();

		TUPLE_CONSTRUCTOR_BEGIN(ca);
		for (int32 j=0; j<elms_in_out_vector; j++) {
			
			GF2X &GF2X = vec_GF2E[j].LoopHole();
//			WordVector &w_vector = GF2X.xrep;		
//			int32 elms_in_inner_vector = w_vector.length();
			int32 elms_in_inner_vector = deg(GF2X)+1;

				// remove trailing zeros from count of elements in poly
			short last_zeros = 0;
			for (int32 i=0; i<elms_in_inner_vector; i++) {
//				if (w_vector[i] == 0L)
				if (coeff(GF2X,i) == 0)
					last_zeros ++;
				else
					last_zeros = 0;
			}
			elms_in_inner_vector -= last_zeros;

			TUPLE_CONSTRUCTOR_BEGIN(caa);
			for (int32 i=0; i<elms_in_inner_vector; i++) {

				s.sp_form = ft_short;
//				s.sp_val.sp_short_value = w_vector[i];
				s.sp_val.sp_short_value = (coeff(GF2X, i) == 0 ? 0 : 1);

				TUPLE_ADD_CELL(caa,&s);
			}
			TUPLE_CONSTRUCTOR_END(caa);
			
			s.sp_form = ft_tuple;
			s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
			TUPLE_ADD_CELL(ca, &s);

		}
		TUPLE_CONSTRUCTOR_END(ca);

		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		TUPLE_ADD_CELL(ca_vect, &s);
		
	}
	TUPLE_CONSTRUCTOR_END(ca_vect);

		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca_vect);
	
	return;

abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

void NTL_GET_MODEXTINT_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
	TUPLE_ITERATOR(ia)
	TUPLE_ITERATOR(iaa)
	TUPLE_ITERATOR(iaaa)
	
	if (argv[0].sp_form != ft_tuple)
		abend(SETL_SYSTEM msg_bad_arg,"setl tuple",1,"ntl_get_modextint_matrix",
			abend_opnd_str(SETL_SYSTEM argv));

	ntl_struct *ntl_data = new ntl_struct(SETL_SYSTEM ntl_mat_ZZ_pE_type);	// create the big integer vector
	mat_ZZ_pE &mat_ZZ_pE = *ntl_data->var.mat_ZZ_pE;
	vec_vec_ZZ_pE *ibuf = (vec_vec_ZZ_pE *)&rep(mat_ZZ_pE);

	int n_col, n_row;
	get_tuple_matrix_size(SETL_SYSTEM argv, n_col, n_row);
	mat_ZZ_pE.SetDims(n_col, n_row);

	int l=0;
	ITERATE_TUPLE_BEGIN(ia,argv[0]) // loop over outer tuple elements
	{
		vec_ZZ_pE &vec_ZZ_pE = mat_ZZ_pE[l];		

		int m=0;
		ITERATE_TUPLE_BEGIN(iaa,(*ia_element))	// loop over inner tuple elements
		{	

			ZZ_pE &ZZ_pE = vec_ZZ_pE[m];

			int n=0;
			ITERATE_TUPLE_BEGIN(iaaa,(*iaa_element)) 
			{	
				int32 number_of_bytes;
				short is_negative;
				unsigned char *byte_array = check_setl_long_arg(SETL_SYSTEM iaaa_element, n, 
						"setl tuple integer element", "ntl_get_modextint_matrix", number_of_bytes, is_negative);

				ZZ ZZ;
				NewZZFromBytes(ZZ, byte_array, number_of_bytes, is_negative);
				
				vec_ZZ_p &vec_ZZ_p = ZZ_pE.LoopHole().rep;
				vec_ZZ_p[n] = to_ZZ_p(ZZ);
				
				n++;
			}
			ITERATE_TUPLE_END(iaaa)

		}
		ITERATE_TUPLE_END(iaa)

		l++;
	}
	ITERATE_TUPLE_END(ia)

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)ntl_data;
}

static void NTL_GET_SETL_MODEXTINT_MATRIX(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
//	ntl_mat_ZZ_pE_type
	TUPLE_CONSTRUCTOR(ca)
	TUPLE_CONSTRUCTOR(caa)
	TUPLE_CONSTRUCTOR(caaa)
	specifier s;
	
	ntl_struct *ntl_data = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_mat_ZZ_pE_type, 
			0, "ntl integer modulo ext matrix", "ntl_to_setl_obj");

	mat_ZZ_pE &mat_ZZ_pE = *ntl_data->var.mat_ZZ_pE;
	vec_vec_ZZ_pE *ibuf = (vec_vec_ZZ_pE *)&rep(mat_ZZ_pE);

	int32 elms_in_vector_l = ibuf->length();
	TUPLE_CONSTRUCTOR_BEGIN(ca);
	for (int32 l=0; l<elms_in_vector_l; l++) {
		
		vec_ZZ_pE &jbuf = mat_ZZ_pE[l];
		
		int32 elms_in_vector_j = jbuf.length();
		TUPLE_CONSTRUCTOR_BEGIN(caa);
		for (int32 j=0; j<elms_in_vector_j; j++) {

			ZZ_pE ZZ_pE = jbuf[j];
			vec_ZZ_p &vec_ZZ_p = ZZ_pE.LoopHole().rep;

			int32 elms_in_vector = vec_ZZ_p.length();
			TUPLE_CONSTRUCTOR_BEGIN(caaa);
			for (int32 i=0; i<elms_in_vector; i++) {
				
				ZZ &ZZ = vec_ZZ_p[i].LoopHole();

				if (NumBits(ZZ) > INT_BIT_WIDTH) {
					integer_h_item *integer_item = setl_long_from_ZZ(SETL_SYSTEM ZZ);
					if (!integer_item) goto abort;

					s.sp_form = ft_long;
					s.sp_val.sp_long_ptr = integer_item;
				} else {
					int32 long_value;

					long_value = to_long(ZZ);
					
					s.sp_form = ft_short;
					s.sp_val.sp_short_value = long_value;
				}
				TUPLE_ADD_CELL(caaa,&s);
						
			}
			TUPLE_CONSTRUCTOR_END(caaa);

			s.sp_form = ft_tuple;
			s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caaa);
			TUPLE_ADD_CELL(caa,&s);

		}
		TUPLE_CONSTRUCTOR_END(caa);
		
		s.sp_form = ft_tuple;
		s.sp_val.sp_tuple_ptr = TUPLE_HEADER(caa);
		TUPLE_ADD_CELL(ca,&s);
				
	}
	TUPLE_CONSTRUCTOR_END(ca);
		
		/* return everything */
	unmark_specifier(target);
	target->sp_form = ft_tuple;
	target->sp_val.sp_tuple_ptr = TUPLE_HEADER(ca);
		
	return;
	
abort:
	unmark_specifier(target);
	target->sp_form = ft_omega;
}

	/*
	 *	Comparison Operators
	 */

void NTL_EQ(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
static char *fn_name = "ntl_eq";
	int32 sub_type;
	bool areEqual = false;

	ntl_struct *ntl_data0 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "ntl type", fn_name);
	ntl_struct *ntl_data1 = check_ntl_arg(SETL_SYSTEM &argv[1], ntl_any_type, 1, "ntl type", fn_name);
	
	if ((sub_type = ntl_data0->sub_type) != ntl_data1->sub_type)
		abend(SETL_SYSTEM "Not allowed mixed type in equality test");

	switch (sub_type) {
		case ntl_RR_type:
			areEqual = (*(ntl_data0->var.RR) == *(ntl_data1->var.RR));
			break;
		case ntl_ZZ_type:
			areEqual = (*(ntl_data0->var.ZZ) == *(ntl_data1->var.ZZ));
			break;
		case ntl_ZZ_p_type:
			areEqual = (*(ntl_data0->var.ZZ_p) == *(ntl_data1->var.ZZ_p));
			break;
		case ntl_ZZ_pX_type:
			areEqual = (*(ntl_data0->var.ZZ_pX) == *(ntl_data1->var.ZZ_pX));
			break;
		case ntl_ZZX_type:
			areEqual = (*(ntl_data0->var.ZZX) == *(ntl_data1->var.ZZX));
			break;
		case ntl_ZZ_pE_type:
			areEqual = (*(ntl_data0->var.ZZ_pE) == *(ntl_data1->var.ZZ_pE));
			break;
		case ntl_ZZ_pEX_type:
			areEqual = (*(ntl_data0->var.ZZ_pEX) == *(ntl_data1->var.ZZ_pEX));
			break;
		case ntl_GF2_type:
			areEqual = (*(ntl_data0->var.GF2) == *(ntl_data1->var.GF2));
			break;
				// Matrices
		case ntl_mat_ZZ_type:
			areEqual = (*(ntl_data0->var.mat_ZZ) == *(ntl_data1->var.mat_ZZ));
			break;
		case ntl_mat_ZZ_p_type:
			areEqual = (*(ntl_data0->var.mat_ZZ_p) == *(ntl_data1->var.mat_ZZ_p));
			break;
		case ntl_mat_RR_type:
			areEqual = (*(ntl_data0->var.mat_RR) == *(ntl_data1->var.mat_RR));
			break;
		case ntl_mat_GF2_type:
			areEqual = (*(ntl_data0->var.mat_GF2) == *(ntl_data1->var.mat_GF2));
			break;
				// Vectors
		case ntl_vec_ZZ_type:
			areEqual = (*(ntl_data0->var.vec_ZZ) == *(ntl_data1->var.vec_ZZ));
			break;
		case ntl_vec_ZZ_p_type:
			areEqual = (*(ntl_data0->var.vec_ZZ_p) == *(ntl_data1->var.vec_ZZ_p));
			break;
		case ntl_vec_RR_type:
			areEqual = (*(ntl_data0->var.vec_RR) == *(ntl_data1->var.vec_RR));
			break;
		case ntl_vec_GF2_type:
			areEqual = (*(ntl_data0->var.vec_GF2) == *(ntl_data1->var.vec_GF2));
			break;
		case ntl_vec_pair_ZZX_long_type:
			areEqual = (*(ntl_data0->var.vec_pair_ZZX_long) == *(ntl_data1->var.vec_pair_ZZX_long));
			break;
	}

	return_setl_boolean(SETL_SYSTEM target, areEqual);
}

void NTL_GT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
static char *fn_name = "ntl_gt";
	int32 sub_type;
	bool res = false;

	ntl_struct *ntl_data0 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "ntl type", fn_name);
	ntl_struct *ntl_data1 = check_ntl_arg(SETL_SYSTEM &argv[1], ntl_any_type, 1, "ntl type", fn_name);
	
	if ((sub_type = ntl_data0->sub_type) != ntl_data1->sub_type)
		abend(SETL_SYSTEM "Not allowed mixed type in compare test");

	switch (sub_type) {
		case ntl_ZZ_type:
			res = (*(ntl_data0->var.ZZ) > *(ntl_data1->var.ZZ));
			break;
		case ntl_RR_type:
			res = (*(ntl_data0->var.RR) > *(ntl_data1->var.RR));
			break;
	}

	return_setl_boolean(SETL_SYSTEM target, res);
}

void NTL_GE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
static char *fn_name = "ntl_ge";
	int32 sub_type;
	bool res = false;

	ntl_struct *ntl_data0 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "ntl type", fn_name);
	ntl_struct *ntl_data1 = check_ntl_arg(SETL_SYSTEM &argv[1], ntl_any_type, 1, "ntl type", fn_name);
	
	if ((sub_type = ntl_data0->sub_type) != ntl_data1->sub_type)
		abend(SETL_SYSTEM "Not allowed mixed type in compare test");

	switch (sub_type) {
		case ntl_ZZ_type:
			res = (*(ntl_data0->var.ZZ) >= *(ntl_data1->var.ZZ));
			break;
		case ntl_RR_type:
			res = (*(ntl_data0->var.RR) >= *(ntl_data1->var.RR));
			break;
	}

	return_setl_boolean(SETL_SYSTEM target, res);
}

void NTL_LT(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
static char *fn_name = "ntl_lt";
	int32 sub_type;
	bool res = false;

	ntl_struct *ntl_data0 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "ntl type", fn_name);
	ntl_struct *ntl_data1 = check_ntl_arg(SETL_SYSTEM &argv[1], ntl_any_type, 1, "ntl type", fn_name);
	
	if ((sub_type = ntl_data0->sub_type) != ntl_data1->sub_type)
		abend(SETL_SYSTEM "Not allowed mixed type in compare test");

	switch (sub_type) {
		case ntl_ZZ_type:
			res = (*(ntl_data0->var.ZZ) < *(ntl_data1->var.ZZ));
			break;
		case ntl_RR_type:
			res = (*(ntl_data0->var.RR) < *(ntl_data1->var.RR));
			break;
	}

	return_setl_boolean(SETL_SYSTEM target, res);
}

void NTL_LE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
static char *fn_name = "ntl_le";
	int32 sub_type;
	bool res = false;

	ntl_struct *ntl_data0 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "ntl type", fn_name);
	ntl_struct *ntl_data1 = check_ntl_arg(SETL_SYSTEM &argv[1], ntl_any_type, 1, "ntl type", fn_name);
	
	if ((sub_type = ntl_data0->sub_type) != ntl_data1->sub_type)
		abend(SETL_SYSTEM "Not allowed mixed type in compare test");

	switch (sub_type) {
		case ntl_ZZ_type:
			res = (*(ntl_data0->var.ZZ) <= *(ntl_data1->var.ZZ));
			break;
		case ntl_RR_type:
			res = (*(ntl_data0->var.RR) <= *(ntl_data1->var.RR));
			break;
	}

	return_setl_boolean(SETL_SYSTEM target, res);
}

void NTL_PROBPRIME(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
static char *fn_name = "ntl_probprime";
	long res;

	ntl_struct *ntl_data0 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_ZZ_type, 0, "big integer", fn_name);
	long NumTrials = check_long_arg(SETL_SYSTEM &argv[1], 1, fn_name);

	res = ProbPrime(*ntl_data0->var.ZZ, NumTrials);

	return_setl_boolean(SETL_SYSTEM target, (bool) res);
}

void NTL_ISZERO(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
static char *fn_name = "ntl_iszero";
	long res;

	ntl_struct *ntl_data0 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "big integer", fn_name);
	int32 sub_type = ntl_data0->sub_type;
	
	switch(sub_type) {
		case ntl_RR_type:
			res = IsZero(*ntl_data0->var.RR);
			break;
		case ntl_ZZ_type:
			res = IsZero(*ntl_data0->var.ZZ);
			break;
		case ntl_ZZ_p_type:
			res = IsZero(*ntl_data0->var.ZZ_p);
			break;
		case ntl_ZZX_type:
			res = IsZero(*ntl_data0->var.ZZX);
			break;
		case ntl_ZZ_pX_type:
			res = IsZero(*ntl_data0->var.ZZ_pX);
			break;
		case ntl_ZZ_pE_type:
			res = IsZero(*ntl_data0->var.ZZ_pE);
			break;
		case ntl_ZZ_pEX_type:
			res = IsZero(*ntl_data0->var.ZZ_pEX);
			break;
		case ntl_GF2_type:
			res = IsZero(*ntl_data0->var.GF2);
			break;
		case ntl_GF2X_type:
			res = IsZero(*ntl_data0->var.GF2X);
			break;
		case ntl_GF2E_type:
			res = IsZero(*ntl_data0->var.GF2E);
			break;
		case ntl_GF2EX_type:
			res = IsZero(*ntl_data0->var.GF2EX);
			break;
		case ntl_mat_RR_type:
			res = IsZero(*ntl_data0->var.mat_RR);
			break;
		case ntl_mat_ZZ_type:
			res = IsZero(*ntl_data0->var.mat_ZZ);
			break;
		case ntl_mat_ZZ_p_type:
			res = IsZero(*ntl_data0->var.mat_ZZ_p);
			break;
		case ntl_mat_GF2_type:
			res = IsZero(*ntl_data0->var.GF2);
			break;
		case ntl_vec_RR_type:
			res = IsZero(*ntl_data0->var.vec_RR);
			break;
		case ntl_vec_ZZ_type:
			res = IsZero(*ntl_data0->var.vec_ZZ);
			break;
		case ntl_vec_ZZ_p_type:
			res = IsZero(*ntl_data0->var.vec_ZZ_p);
			break;
		case ntl_vec_ZZ_pE_type:
			res = IsZero(*ntl_data0->var.vec_ZZ_pE);
			break;
		case ntl_vec_GF2_type:
			res = IsZero(*ntl_data0->var.vec_GF2);
			break;
	}

	return_setl_boolean(SETL_SYSTEM target, (bool) res);
}

void NTL_ISONE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
static char *fn_name = "ntl_iszero";
	long res;

	ntl_struct *ntl_data0 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "big integer", fn_name);
	int32 sub_type = ntl_data0->sub_type;
	
	switch(sub_type) {
		case ntl_RR_type:
			res = IsOne(*ntl_data0->var.RR);
			break;
		case ntl_ZZ_type:
			res = IsOne(*ntl_data0->var.ZZ);
			break;
		case ntl_ZZ_p_type:
			res = IsOne(*ntl_data0->var.ZZ_p);
			break;
		case ntl_ZZX_type:
			res = IsOne(*ntl_data0->var.ZZX);
			break;
		case ntl_ZZ_pX_type:
			res = IsOne(*ntl_data0->var.ZZ_pX);
			break;
		case ntl_ZZ_pE_type:
			res = IsOne(*ntl_data0->var.ZZ_pE);
			break;
		case ntl_ZZ_pEX_type:
			res = IsOne(*ntl_data0->var.ZZ_pEX);
			break;
		case ntl_GF2_type:
			res = IsOne(*ntl_data0->var.GF2);
			break;
		case ntl_GF2X_type:
			res = IsOne(*ntl_data0->var.GF2X);
			break;
		case ntl_GF2E_type:
			res = IsOne(*ntl_data0->var.GF2E);
			break;
		case ntl_GF2EX_type:
			res = IsOne(*ntl_data0->var.GF2EX);
			break;
		case ntl_mat_GF2_type:
			res = IsOne(*ntl_data0->var.GF2);
			break;
	}

	return_setl_boolean(SETL_SYSTEM target, (bool) res);
}

void NTL_PROBIRREDTEST(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
static char *fn_name = "ntl_probirredtest";
	long res;

	ntl_struct *ntl_data0 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "ntl type", fn_name);
	int32 sub_type = ntl_data0->sub_type;
	
	switch(sub_type) {
		case ntl_ZZ_pX_type:
			res = ProbIrredTest(*ntl_data0->var.ZZ_pX);
			break;
		case ntl_ZZ_pEX_type:
			res = ProbIrredTest(*ntl_data0->var.ZZ_pEX);
			break;
		case ntl_GF2EX_type:
			res = ProbIrredTest(*ntl_data0->var.GF2EX);
			break;
	}

	return_setl_boolean(SETL_SYSTEM target, (bool) res);
}

void NTL_DETIRREDTEST(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target)
{
static char *fn_name = "ntl_detirredtest";
	long res;

	ntl_struct *ntl_data0 = check_ntl_arg(SETL_SYSTEM &argv[0], ntl_any_type, 0, "ntl type", fn_name);
	int32 sub_type = ntl_data0->sub_type;
	
	switch(sub_type) {
		case ntl_ZZ_pX_type:
			res = DetIrredTest(*ntl_data0->var.ZZ_pX);
			break;
		case ntl_ZZ_pEX_type:
			res = DetIrredTest(*ntl_data0->var.ZZ_pEX);
			break;
		case ntl_GF2EX_type:
			res = DetIrredTest(*ntl_data0->var.GF2EX);
			break;
	}

	return_setl_boolean(SETL_SYSTEM target, (bool) res);
}
	/*
	 *	Shell Routines
	 */

static void eval2(ZZ_pE& b, const ZZ_pX& f, const ZZ_pE& a);
static void eval3(vec_ZZ_p& b, const ZZ_pX& f, const vec_ZZ_p& a);
static void eval4(vec_ZZ_pE& b, const ZZ_pEX& f, const vec_ZZ_pE& a);

void eval2(ZZ_pE& b, const ZZ_pX& f, const ZZ_pE& a)
{
	eval(b, f, a);
}

void eval3(vec_ZZ_p& b, const ZZ_pX& f, const vec_ZZ_p& a)
{
	eval(b, f, a);
}

void eval4(vec_ZZ_pE& b, const ZZ_pEX& f, const vec_ZZ_pE& a)
{
	eval(b, f, a);
}

void BuildIrred_mod2extint_poly(GF2EX& a, long l);
void BuildIrred_mod2int_poly(GF2X& a, long l);
void BuildIrred_modextint_poly(ZZ_pEX& a, long l);
void BuildIrred_modint_poly(ZZ_pX& a, long l);

void BuildIrred_mod2extint_poly(GF2EX& a, long l)
{
	BuildIrred(a, l);
}

void BuildIrred_mod2int_poly(GF2X& a, long l)
{
	BuildIrred(a, l);
}

void BuildIrred_modextint_poly(ZZ_pEX& a, long l)
{
	BuildIrred(a, l);
}

void BuildIrred_modint_poly(ZZ_pX& a, long l)
{
	BuildIrred(a, l);
}

void ComputeDegree(ZZ &d, const ZZ_pX& a, const ZZ_pX &f);
void ProbComputeDegree(ZZ &d, const ZZ_pX& a, const ZZ_pX &f);

void ComputeDegree(ZZ &d, const ZZ_pX& a, const ZZ_pX &f)
{
	ZZ_pXModulus F(f);

	d = to_ZZ(ComputeDegree(a, F));
}

void ProbComputeDegree(ZZ &d, const ZZ_pX& a, const ZZ_pX &f)
{
	ZZ_pXModulus F(f);

	d = to_ZZ(ProbComputeDegree(a, F));
}

#ifdef __cplusplus
}
#endif

void TraceMap(ZZ_pX& w, const ZZ_pX& a, long d, const ZZ_pX& f, const ZZ_pX& b);
void TraceMap(ZZ_pEX& w, const ZZ_pEX& a, long d, const ZZ_pEX& f, const ZZ_pEX& b);
void TraceMap(GF2X& w, const GF2X& a, long d, const GF2X& f, const GF2X& b);
void TraceMap(GF2EX& w, const GF2EX& a, long d, const GF2EX& f, const GF2EX& b);

void TraceMap(ZZ_pX& w, const ZZ_pX& a, long d, const ZZ_pX& f, const ZZ_pX& b)
{
	ZZ_pXModulus F(f);

	TraceMap(w, a, d, F, b);
}

void TraceMap(ZZ_pEX& w, const ZZ_pEX& a, long d, const ZZ_pEX& f, const ZZ_pEX& b)
{
	ZZ_pEXModulus F(f);

	TraceMap(w, a, d, F, b);
}

void TraceMap(GF2X& w, const GF2X& a, long d, const GF2X& f, const GF2X& b)
{
	GF2XModulus F(f);

	TraceMap(w, a, d, F, b);
}

void TraceMap(GF2EX& w, const GF2EX& a, long d, const GF2EX& f, const GF2EX& b)
{
	GF2EXModulus F(f);

	TraceMap(w, a, d, F, b);
}

#ifdef __cplusplus
extern "C" {
#endif

	/*
	 *	Other Stubs
	 */

#include "ntl_stubs.c"

#ifdef __cplusplus
}
#endif

