/* 
 * tkRastport.c --
 *
 *	This module implements "rastport" widgets.  A "rastport" is
 *	a widget that displays a single rastport that can be moved
 *	around and resized. 
 *
 */

#include "tkrasterport.h"

/*
 * A data structure of the following type is kept for each rastport
 * widget managed by this file:
 */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the rastport.  NULL
							 * means window has been deleted but
							 * widget record hasn't been cleaned up yet. */
    Display *display;		/* X's token for the window's display. */
    Tcl_Interp *interp;		/* Interpreter associated with widget. */
    Tcl_Command widgetCmd;	/* Token for rastport's widget command. */
    int x, y;				/* Position of rastport's upper-left corner
							 * within widget. */
    int width, height;		/* Width and height of rastport. */

    /*
     * Information used when displaying widget:
     */

	XImage *image;			/* Displaying image */

    GC gc;					/* Graphics context for copying from
							 * off-screen pixmap onto screen. */
    int updatePending;		/* Non-zero means a call to RastportDisplay
							 * has already been scheduled. */
} Rastport;

/*
 * Information used for argv parsing.
 */

static Tk_ConfigSpec configSpecs[] = {
	{TK_CONFIG_INT, "-width", "width", "Width", "320", Tk_Offset(Rastport, width), NULL},
	{TK_CONFIG_INT, "-height", "height", "Height", "240", Tk_Offset(Rastport, height), NULL},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL, (char *) NULL, 0, 0}
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static int RastportWidgetCmd(ClientData clientData, Tcl_Interp *interp, int argc, char **argv);
static int RastportConfigure(Tcl_Interp *interp, Rastport *rastportPtr, int argc, char **argv, int flags);
static void RastportEventProc(ClientData clientData, XEvent *eventPtr);
static void RastportCmdDeletedProc(ClientData clientData);
static void RastportDisplay(ClientData clientData);
static void RastportDestroy(char *memPtr);

static void KeepInWindow(Rastport *rastportPtr);
static XImage *CreateRastportImage(int width, int height, Tk_Window tkwin);
static void FastRedisplay(Tk_Window tkwin, Rastport *rastportPtr, int x, int y, int mimg_width, int mimg_height);
static void GetAbsoluteRastportPosition(Rastport *rastportPtr, int *dest_x, int *dest_y);
static int stuff_rastport_over_sparse_ex(mimage_ptr mimg_sparse, Rastport *rastportPtr, int x, int y, float c1, float c2, float (*op )(float, float ));
static int stuff_sparse_over_rastport_ex(mimage_ptr mimg_sparse, Rastport *rastportPtr, int x, int y, float c1, float c2, float (*op )(float, float ));
static int StuffMimageInRastport_rotate_ex(Tcl_Interp *interp, mimage_ptr mimg, char *rastport_path, int x, int y, float c1, float c2, float (*op)(float, float), int rotate);
static int stuff_sparse_over_rastport_rotate_ex(mimage_ptr mimg_sparse, Rastport *rastportPtr, int x, int y, float c1, float c2, float (*op )(float, float), int rotate);

/*
 *	Global Variables
 */

#undef MIN
#define MIN(a, b)	((a) < (b)? (a): (b))
#undef MAX
#define MAX(a, b)	((a) > (b)? (a): (b))

#define PLANES_IN_IMAGE	4

Tcl_HashTable	rastportTable;


/*
 *--------------------------------------------------------------
 *
 * RastportCmd --
 *
 *	This procedure is invoked to process the "rastport" Tcl
 *	command.  It creates a new "rastport" widget.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	A new widget is created and configured.
 *
 *--------------------------------------------------------------
 */

int
RastportCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
							 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    Tk_Window main = (Tk_Window) clientData;
    Rastport *rastportPtr;
    Tk_Window tkwin;
    Tcl_HashEntry *entryPtr;
	int newEntry;
	
    if (argc < 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"",
			argv[0], " pathName ?options?\"", (char *) NULL);
		return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, main, argv[1], (char *) NULL);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }
    Tk_SetClass(tkwin, "Rastport");

    /*
     * Allocate and initialize the widget record.
     */

    rastportPtr = (Rastport *) ckalloc(sizeof(Rastport));

	entryPtr = Tcl_CreateHashEntry(&rastportTable, argv[1], &newEntry);
	if (!newEntry) {
		Tcl_AppendResult(interp, "rastport ",
			argv[0], " already exists", (char *) NULL);
		return TCL_ERROR;
    }
	Tcl_SetHashValue(entryPtr, rastportPtr);

    rastportPtr->tkwin = tkwin;
    rastportPtr->display = Tk_Display(tkwin);
    rastportPtr->interp = interp;
    rastportPtr->widgetCmd = Tcl_CreateCommand(interp,
	    Tk_PathName(rastportPtr->tkwin), RastportWidgetCmd,
	    (ClientData) rastportPtr, RastportCmdDeletedProc);
    rastportPtr->x = 0;
    rastportPtr->y = 0;
    rastportPtr->width = 0;			/* configure will set default width */
    rastportPtr->height = 0;		/* configure will set default height */
    rastportPtr->gc = None;
    rastportPtr->updatePending = 0;

    Tk_CreateEventHandler(rastportPtr->tkwin, ExposureMask|StructureNotifyMask, RastportEventProc, (ClientData) rastportPtr);
    if (RastportConfigure(interp, rastportPtr, argc-2, argv+2, 0) != TCL_OK) {
		Tk_DestroyWindow(rastportPtr->tkwin);
		return TCL_ERROR;
    }

		/* after the configure has been executed we can create the image */
	rastportPtr->image = CreateRastportImage(rastportPtr->width, rastportPtr->height, tkwin);	/* create the rastport buffer image */

    interp->result = Tk_PathName(rastportPtr->tkwin);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * RastportWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
RastportWidgetCmd(clientData, interp, argc, argv)
    ClientData clientData;		/* Information about rastport widget. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    Rastport *rastportPtr = (Rastport *) clientData;
    int result = TCL_OK;
    size_t length;
    char c;

    if (argc < 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"",
			argv[0], " option ?arg arg ...?\"", (char *) NULL);
		return TCL_ERROR;
    }
    Tcl_Preserve((ClientData) rastportPtr);
    c = argv[1][0];
    length = strlen(argv[1]);

    if ((c == 'c') && (strncmp(argv[1], "cget", length) == 0)
	    && (length >= 2)) {

		if (argc != 3) {
		    Tcl_AppendResult(interp, "wrong # args: should be \"",
			    argv[0], " cget option\"",
			    (char *) NULL);
		    goto error;
		}

		result = Tk_ConfigureValue(interp, rastportPtr->tkwin, configSpecs,
			(char *) rastportPtr, argv[2], 0);

    } else if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)
	    && (length >= 2)) {

		if (argc == 2) {

		    result = Tk_ConfigureInfo(interp, rastportPtr->tkwin, configSpecs,
			    (char *) rastportPtr, (char *) NULL, 0);

		} else if (argc == 3) {

		    result = Tk_ConfigureInfo(interp, rastportPtr->tkwin, configSpecs,
			    (char *) rastportPtr, argv[2], 0);

		} else {

		    result = RastportConfigure(interp, rastportPtr, argc-2, argv+2,
			    TK_CONFIG_ARGV_ONLY);
		}

    } else if ((c == 'p') && (strncmp(argv[1], "position", length) == 0)) {

		if ((argc != 2) && (argc != 4)) {
		    Tcl_AppendResult(interp, "wrong # args: should be \"",
			    argv[0], " position ?x y?\"", (char *) NULL);
		    goto error;
		}

		if (argc == 4) {

		    if ((Tk_GetPixels(interp, rastportPtr->tkwin, argv[2],
			    &rastportPtr->x) != TCL_OK) || (Tk_GetPixels(interp,
			    rastportPtr->tkwin, argv[3], &rastportPtr->y) != TCL_OK)) {
		
				goto error;
		    }
		    KeepInWindow(rastportPtr);
		}
		
		sprintf(interp->result, "%d %d", rastportPtr->x, rastportPtr->y);

    } else if ((c == 's') && (strncmp(argv[1], "size", length) == 0)) {

		if ((argc != 2) && (argc != 3)) {
		    Tcl_AppendResult(interp, "wrong # args: should be \"",
			    argv[0], " size ?amount?\"", (char *) NULL);
		    goto error;
		}

		if (argc == 3) {
		    int i;

		    if (Tk_GetPixels(interp, rastportPtr->tkwin, argv[2], &i) != TCL_OK) {
			goto error;
		    }
		    if ((i <= 0) || (i > 100)) {
				Tcl_AppendResult(interp, "bad size \"", argv[2],
					"\"", (char *) NULL);
				goto error;
		    }
		    rastportPtr->width = i;
		    rastportPtr->height = i;
		    KeepInWindow(rastportPtr);
		}

		sprintf(interp->result, "%d %d", rastportPtr->width, rastportPtr->height);

    } else {
		Tcl_AppendResult(interp, "bad option \"", argv[1],
			"\": must be cget, configure, position, or size",
			(char *) NULL);
		goto error;
    }

    if (!rastportPtr->updatePending) {
		Tcl_DoWhenIdle(RastportDisplay, (ClientData) rastportPtr);
		rastportPtr->updatePending = 1;
    }

    Tcl_Release((ClientData) rastportPtr);
    return result;

error:
    Tcl_Release((ClientData) rastportPtr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * RastportConfigure --
 *
 *	This procedure is called to process an argv/argc list in
 *	conjunction with the Tk option database to configure (or
 *	reconfigure) a rastport widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width,
 *	etc. get set for rastportPtr;  old resources get freed,
 *	if there were any.
 *
 *----------------------------------------------------------------------
 */

static int
RastportConfigure(interp, rastportPtr, argc, argv, flags)
    Tcl_Interp *interp;			/* Used for error reporting. */
    Rastport *rastportPtr;			/* Information about widget. */
    int argc;				/* Number of valid entries in argv. */
    char **argv;			/* Arguments. */
    int flags;				/* Flags to pass to
							 * Tk_ConfigureWidget. */
{
    if (Tk_ConfigureWidget(interp, rastportPtr->tkwin, configSpecs,
	    argc, argv, (char *) rastportPtr, flags) != TCL_OK) {
		return TCL_ERROR;
    }

    /*
     * Register the desired geometry for the window.  Then arrange for
     * the window to be redisplayed.
     */

    Tk_GeometryRequest(rastportPtr->tkwin, rastportPtr->width, rastportPtr->height);
    if (!rastportPtr->updatePending) {
	Tcl_DoWhenIdle(RastportDisplay, (ClientData) rastportPtr);
	rastportPtr->updatePending = 1;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * RastportEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on rastports.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
RastportEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
    Rastport *rastportPtr = (Rastport *) clientData;

    if (eventPtr->type == Expose) {
	if (!rastportPtr->updatePending) {
	    Tcl_DoWhenIdle(RastportDisplay, (ClientData) rastportPtr);
	    rastportPtr->updatePending = 1;
	}
    } else if (eventPtr->type == ConfigureNotify) {
	KeepInWindow(rastportPtr);
	if (!rastportPtr->updatePending) {
	    Tcl_DoWhenIdle(RastportDisplay, (ClientData) rastportPtr);
	    rastportPtr->updatePending = 1;
	}
    } else if (eventPtr->type == DestroyNotify) {
	if (rastportPtr->tkwin != NULL) {
	    rastportPtr->tkwin = NULL;
	    Tcl_DeleteCommandFromToken(rastportPtr->interp,
		    rastportPtr->widgetCmd);
	}
	if (rastportPtr->updatePending) {
	    Tcl_CancelIdleCall(RastportDisplay, (ClientData) rastportPtr);
	}
	Tcl_EventuallyFree((ClientData) rastportPtr, RastportDestroy);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RastportCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */

static void
RastportCmdDeletedProc(clientData)
    ClientData clientData;	/* Pointer to widget record for widget. */
{
    Rastport *rastportPtr = (Rastport *) clientData;
    Tk_Window tkwin = rastportPtr->tkwin;
	XImage *image = rastportPtr->image;
    Tcl_HashEntry *entry;
    
    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */

    if (tkwin != NULL) {

	    entry = Tcl_FindHashEntry(&rastportTable, Tk_PathName(tkwin));
		if (entry)
			Tcl_DeleteHashEntry(entry);    
		rastportPtr->tkwin = NULL;

#ifdef UNIX
		XFree(image);				/* destroy memory used for put image */
#else
	
		if (image)	ckfree(image);	/* release the memory for the image */
#endif

		Tk_DestroyWindow(tkwin);	/* destroy the window */
    }
	
	if (image != NULL)	rastportPtr->image = NULL;

}

/*
 *--------------------------------------------------------------
 *
 * RastportDisplay --
 *
 *	This procedure redraws the contents of a rastport window.
 *	It is invoked as a do-when-idle handler, so it only runs
 *	when there's nothing else for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */

static void
RastportDisplay(clientData)
    ClientData clientData;	/* Information about window. */
{
	int dest_x, dest_y;
	GC gc;
	XGCValues gcValues;

	Rastport *rastportPtr = (Rastport *) clientData;
	Tk_Window tkwin = rastportPtr->tkwin;
	XImage *img = rastportPtr->image;
	Pixmap pm = None;
	Drawable d;

    rastportPtr->updatePending = 0;

    if (!Tk_IsMapped(tkwin)) 
		return;

		/* create a graphic context */
	gcValues.function = GXcopy;
	gcValues.graphics_exposures = False;

	gc = Tk_GetGC(tkwin, GCFunction|GCGraphicsExposures, &gcValues);

	KeepInWindow(rastportPtr);

	d = Tk_WindowId(tkwin);

	GetAbsoluteRastportPosition(rastportPtr, &dest_x, &dest_y);

	TkPutImage(
		NULL, 				/* unsigned long *colors	NOT USED ON MAC */
		0,					/* int colors				NOT USED ON MAC */
		Tk_Display(tkwin), 	/* Display *display							*/
		d,					/* Drawable d								*/
		gc,					/* GC gc									*/
		rastportPtr->image,	/* XImage *image							*/
		0, 0,				/* int src_x, src_y							*/
		dest_x, dest_y,		/* int x, y									*/
		rastportPtr->width, rastportPtr->height	/* unsigned int width, unsigned int height	*/
	);

	Tk_FreeGC(Tk_Display(tkwin), gc);
	
}

/*
 *----------------------------------------------------------------------
 *
 * RastportDestroy --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of a rastport at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the rastport is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
RastportDestroy(memPtr)
    char *memPtr;		/* Info about rastport widget. */
{
    Rastport *rastportPtr = (Rastport *) memPtr;

    Tk_FreeOptions(configSpecs, (char *) rastportPtr, rastportPtr->display, 0);
    if (rastportPtr->gc != None) {
	Tk_FreeGC(rastportPtr->display, rastportPtr->gc);
    }
    ckfree((char *) rastportPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * KeepInWindow --
 *
 *	Adjust the position of the rastport if necessary to keep it in
 *	the widget's window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The x and y position of the rastport are adjusted if necessary
 *	to keep the rastport in the window.
 *
 *----------------------------------------------------------------------
 */

static void
KeepInWindow(rastportPtr)
    register Rastport *rastportPtr;		/* Pointer to widget record. */
{
    int i, bd;
	Tk_Window tkwin = rastportPtr->tkwin;
	
    bd = 0;
    i = (Tk_Width(tkwin) - bd) - (rastportPtr->x + rastportPtr->width);
    if (i < 0) {
		rastportPtr->x += i;
    }
    i = (Tk_Height(tkwin) - bd) - (rastportPtr->y + rastportPtr->height);
    if (i < 0) {
		rastportPtr->y += i;
    }
    if (rastportPtr->x < bd) {
		rastportPtr->x = bd;
    }
    if (rastportPtr->y < bd) {
		rastportPtr->y = bd;
    }
}

int Rastport_Init(Tcl_Interp *interp)
{
	Tcl_InitHashTable(&rastportTable, TCL_STRING_KEYS);

	Tcl_CreateCommand(interp, "rastport", RastportCmd,
			(ClientData) Tk_MainWindow(interp),
			(Tcl_CmdDeleteProc *) NULL);
			
	return TCL_OK;
}

int Rastport_Destroy(Tcl_Interp *interp)
{
	Tcl_DeleteHashTable(&rastportTable);
}

static XImage *CreateRastportImage(int width, int height, Tk_Window tkwin)
{
	int i;
	XImage *image;

	image = XCreateImage (
		Tk_Display(tkwin), 
		NULL,
		32,
		ZPixmap,
		0,
		NULL,
		width,
		height,
		32,
		PLANES_IN_IMAGE * width	
	);

	image->data = (char *)malloc( width * height * PLANES_IN_IMAGE);
	if (!image->data) {
#ifdef UNIX
		XFree(image);				/* destroy memory used for put image */
#else
	
		if (image)	ckfree(image);	/* release the memory for the image */
#endif
		return NULL;

	}
	
	memset(image->data, 0xaf, width * height * PLANES_IN_IMAGE);
	
	for ( i=0 ; i< width * height * PLANES_IN_IMAGE ; i+=4 ) {
		image->data[i+0] = 0x00;
		image->data[i+1] = 0x00;
		image->data[i+2] = 0x00;
		image->data[i+3] = 0x00;
		
		i+=PLANES_IN_IMAGE;

		image->data[i+0] = 0x00;
		image->data[i+1] = 0xff;
		image->data[i+2] = 0xff;
		image->data[i+3] = 0xff;
	} 
	
	return image;
}

/* 
 *	Not widget functions
 */

int StuffMimageInRastport_ex(Tcl_Interp *interp, mimage_ptr mimg, char *rastport_path, int x, int y, float c1, float c2, float (*op)(float, float), int rotate)
{
	int i, j, c;
	int mimg_width, mimg_height;
	mpixel_ptr image_base, image_ptr;
	int pixels_number;
	Tcl_HashEntry *entry;
	Tk_Window tkmainwin, tkwin;
	Rastport *rastportPtr;

	if (!mimg)
		return 0;

	if (rotate)
		return StuffMimageInRastport_rotate_ex(interp, mimg, rastport_path, x, y, c1, c2, op, rotate);

	tkmainwin = Tk_MainWindow(interp);
	if (!tkmainwin)
		return 0;

	tkwin = Tk_NameToWindow(interp, rastport_path, tkmainwin);
	if (!tkwin)
		return 0;
		
	entry = Tcl_FindHashEntry(&rastportTable, rastport_path);
	if (entry)
		rastportPtr = Tcl_GetHashValue(entry);
	else
		return 0;

	if (x<0 || y<0)
		return 1;
		
	if (y > rastportPtr->height)
		return 1;	/* nothing to do image is outside */

	if (x > rastportPtr->width)
		return 1;	/* nothing to do image is outside */

	pixels_number = (mimg_height = mimg->planes_array[0]->height) * (mimg_width = mimg->planes_array[0]->width);
	image_base = rastportPtr->image->data;

	if (mimg->density == SPARSE_IMAGE) {	/* SPARSE */
		stuff_sparse_over_rastport_ex(mimg, rastportPtr, x, y, c1, c2, op);
	} else {

		if (!rotate) {

			if (mimg->kind == DISCRETE_IMAGE) {
				
				for (c=0; c<mimg->comp; c++) {

					byteplane_ptr plane = mimg->planes_array[c];
					mpixel_ptr buffer;

					for (j=0; j<MIN(mimg_height, rastportPtr->height - y) ; j++) {

						image_ptr = image_base + (((j + y) * rastportPtr->width + x) * PLANES_IN_IMAGE) + c + 1;
						buffer = plane->buffer + (j * mimg_width);
				
						for (i=0; i<MIN(mimg_width, rastportPtr->width - x); i++) {
							*image_ptr = op(*image_ptr * c1, *buffer * c2);

							buffer ++;
							image_ptr += PLANES_IN_IMAGE;
						}
					}
				}
			} else {
				
				for (c=0; c<mimg->comp; c++) {

					fbyteplane_ptr plane = mimg->planes_array[c];
					fmpixel_ptr buffer;

					for (j=0; j<MIN(mimg_height, rastportPtr->height - y) ; j++) {

						image_ptr = image_base + (((j + y) * rastportPtr->width + x) * PLANES_IN_IMAGE) + c + 1;
						buffer = plane->buffer + (j * mimg_width);
				
						for (i=0; i<MIN(mimg_width, rastportPtr->width - x); i++) {
							*image_ptr = op(*image_ptr * c1, *buffer * c2);

							buffer ++;
							image_ptr += PLANES_IN_IMAGE;
						}
					}
				}
			}

		}
	}	/* DENSE */
	 
#ifdef TK_GR_SLOW_COPY
	RastportDisplay(rastportPtr);
#else
	FastRedisplay(tkwin, rastportPtr, x, y, mimg_width, mimg_height);
#endif

	return 1;
}

static int StuffMimageInRastport_rotate_ex(Tcl_Interp *interp, mimage_ptr mimg, char *rastport_path, int x, int y, float c1, float c2, float (*op)(float, float), int rotate)
{
	int i, j, c;
	int mimg_width, mimg_height;
	mpixel_ptr image_base, image_ptr;
	int pixels_number;
	Tcl_HashEntry *entry;
	Tk_Window tkmainwin, tkwin;
	Rastport *rastportPtr;

	if (!(tkmainwin = Tk_MainWindow(interp)))
		return 0;

	if (!(tkwin = Tk_NameToWindow(interp, rastport_path, tkmainwin)))
		return 0;
		
	if ((entry = Tcl_FindHashEntry(&rastportTable, rastport_path)) != NULL)
		rastportPtr = Tcl_GetHashValue(entry);
	else
		return 0;

	if (x<0 || y<0)
		return 1;
		
	if (y > rastportPtr->height)
		return 1;	/* nothing to do image is outside */

	if (x > rastportPtr->width)
		return 1;	/* nothing to do image is outside */

	pixels_number = (mimg_height = mimg->planes_array[0]->height) * (mimg_width = mimg->planes_array[0]->width);
	image_base = rastportPtr->image->data;

	if (mimg->density == SPARSE_IMAGE) {	/* SPARSE */
		stuff_sparse_over_rastport_rotate_ex(mimg, rastportPtr, x, y, c1, c2, op, rotate);
	} else {

		if (mimg->kind == DISCRETE_IMAGE) {
			
			for (c=0; c<mimg->comp; c++) {

				byteplane_ptr plane = mimg->planes_array[c];
				mpixel_ptr buffer;

				for (j=0; j<MIN(mimg_width, rastportPtr->height - y) ; j++) {

					image_ptr = image_base + (((j + y) * rastportPtr->width + x) * PLANES_IN_IMAGE) + c + 1;
					buffer = plane->buffer + j;
			
					for (i=0; i<MIN(mimg_height, rastportPtr->width - x); i++) {
						*image_ptr = op(*image_ptr * c1, *buffer * c2);

						buffer += (mimg_width);
						image_ptr += PLANES_IN_IMAGE;
					}
				}
			}
		} else {
			
			for (c=0; c<mimg->comp; c++) {

				fbyteplane_ptr plane = mimg->planes_array[c];
				fmpixel_ptr buffer;

				for (j=0; j<MIN(mimg_width, rastportPtr->height - y) ; j++) {

					image_ptr = image_base + (((j + y) * rastportPtr->width + x) * PLANES_IN_IMAGE) + c + 1;
					buffer = plane->buffer + j;
			
					for (i=0; i<MIN(mimg_height, rastportPtr->width - x); i++) {
						*image_ptr = op(*image_ptr * c1, *buffer * c2);

						buffer += (mimg_width);
						image_ptr += PLANES_IN_IMAGE;
					}
				}
			}
		}
		
	}	/* DENSE */
	 
#ifdef TK_GR_SLOW_COPY
	RastportDisplay(rastportPtr);
#else
	FastRedisplay(tkwin, rastportPtr, x, y, mimg_height, mimg_width);
#endif

	return 1;
}

int StuffRastportInMimage_ex(Tcl_Interp *interp, mimage_ptr mimg, char *rastport_path, int x, int y, float c1, float c2, float (*op )(float, float), int rotate)
{
	int i, j, c;
	mpixel_ptr image_base, image_ptr;
	Tcl_HashEntry *entry;
	Tk_Window tkmainwin, tkwin;
	Rastport *rastportPtr;

	if (x<0 || y<0)
		return 1;

	if (!mimg)
		return 0;
		
	tkmainwin = Tk_MainWindow(interp);
	if (!tkmainwin)
		return 0;

	tkwin = Tk_NameToWindow(interp, rastport_path, tkmainwin);
	if (!tkwin)
		return 0;
		
	entry = Tcl_FindHashEntry(&rastportTable, rastport_path);
	if (entry)
		rastportPtr = Tcl_GetHashValue(entry);
	else
		return 0;

	image_base = rastportPtr->image->data;
	if (!image_base)
		return 0;

	if (mimg->density == DENSE_IMAGE) {
		if (mimg->kind == DISCRETE_IMAGE) {
			for (c=0; c<mimg->comp; c++) {
				
				mpixel_ptr buffer_base = mimg->planes_array[c]->buffer;
				
				for (j=0; j<MIN(rastportPtr->height - y, mimg->planes_array[0]->height); j++) {

					mpixel_ptr buffer = buffer_base + (j * mimg->planes_array[0]->width) * sizeof(mpixel);
					image_ptr = image_base + ((y + j) * rastportPtr->width + x) * (PLANES_IN_IMAGE) + c + 1;

					for (i=0; i<MIN(rastportPtr->width - x, mimg->planes_array[0]->width); i++) {
					
						*buffer = op(c1 * *buffer, c2 * *image_ptr);
						
						buffer++;
						image_ptr += PLANES_IN_IMAGE;
					}
				}
			}
		} else {
			for (c=0; c<mimg->comp; c++) {
				
				fmpixel_ptr buffer_base = mimg->planes_array[c]->buffer;
				
				for (j=0; j<MIN(rastportPtr->height - y, mimg->planes_array[0]->height); j++) {

					fmpixel_ptr buffer = buffer_base + (j * mimg->planes_array[0]->width) * sizeof(fmpixel);
					image_ptr = image_base + ((y + j) * rastportPtr->width + x) * (PLANES_IN_IMAGE) + c + 1;

					for (i=0; i<MIN(rastportPtr->width - x, mimg->planes_array[0]->width); i++) {
					
						*buffer = op(c1 * *buffer, c2 * *image_ptr);
						
						buffer++;
						image_ptr += PLANES_IN_IMAGE;
					}
				}
			}
		}
	} else {	/* SPARSE */
		return stuff_rastport_over_sparse_ex(mimg, rastportPtr, x, y, c1, c2, op);
	}
	
	return 1;
}

static void FastRedisplay(Tk_Window tkwin, Rastport *rastportPtr, int x, int y, int mimg_width, int mimg_height)
{
	int dest_x, dest_y;
	GC gc;
	XGCValues gcValues;
	Drawable d;
	
	if (!Tk_IsMapped(tkwin))
		return;

	GetAbsoluteRastportPosition(rastportPtr, &dest_x, &dest_y);

		/* create a graphic context */
	gcValues.function = GXcopy;
	gcValues.graphics_exposures = False;

	gc = Tk_GetGC(tkwin, GCFunction|GCGraphicsExposures, &gcValues);

	KeepInWindow(rastportPtr);

	d = Tk_WindowId(tkwin);

#ifdef UNIX

	/* unix */

	XPutImage(
		Tk_Display(tkwin),								/* Display *display		*/
		d,												/* Drawable d			*/
		gc,												/* GC gc				*/
		rastportPtr->image,								/* XImage *image		*/
		x, y,											/* int x, y				*/
		dest_x + x,										/* int dst_x			*/
		dest_y + y,										/* int dst_y			*/
		MIN(mimg_width, rastportPtr->width - x), 		/* unsigned int width	*/
		MIN(mimg_height, rastportPtr->height - y)		/* unsigned int height	*/
		);
#else
		TkPutImage(
		NULL, 											/* unsigned long *colors	*/
		0,												/* int colors				*/
		Tk_Display(tkwin),								/* Display *display			*/
		d,												/* Drawable d				*/
		gc,												/* GC gc					*/
		rastportPtr->image,								/* XImage *image			*/
		x, y,											/* int src_x, src_y			*/
		dest_x + x,										/* int dst_x				*/
		dest_y + y,										/* int dst_y				*/
		MIN(mimg_width, rastportPtr->width - x), 		/* unsigned int width		*/
		MIN(mimg_height, rastportPtr->height - y)		/* unsigned int height		*/
	);
#endif

	Tk_FreeGC(Tk_Display(tkwin), gc);
	
}

static void GetAbsoluteRastportPosition(Rastport *rastportPtr, int *dest_x, int *dest_y)
{
/*	volatile int top_level = 0; */
	Tk_Window tkwin = rastportPtr->tkwin;

	*dest_x=*dest_y=0;

	*dest_x = rastportPtr->x;
	*dest_y = rastportPtr->y;

/*	while (Tk_Parent(tkwin)) { */
	while (!Tk_IsTopLevel(tkwin)) {

/*		top_level = Tk_IsTopLevel(tkwin); */

		*dest_x += Tk_X(tkwin) + Tk_Changes(tkwin)->border_width;
		*dest_y += Tk_Y(tkwin) + Tk_Changes(tkwin)->border_width;

		tkwin = Tk_Parent(tkwin);
	}
}

static int stuff_rastport_over_sparse_ex(mimage_ptr mimg_sparse, Rastport *rastportPtr, int x, int y, float c1, float c2, float (*op )(float, float))
{
	int c;
	int xx,yy;
	int i,j, l;
	int pos;
	int lmax;
	
	int sect_idx;
	int sects_per_row;
	section_data *sects;
	section_data_ptr sect;
	int *row_start;
	
	int planes_number;
	byteplane_ptr *planes_array;
	int width_sparse, height_sparse, planes_number_sparse, kind_sparse;
	unsigned char density_sparse;
	int next_idx;

	get_mimage_info(mimg_sparse, &height_sparse, &width_sparse, &planes_number_sparse, &kind_sparse, &density_sparse);
	
	if (x > rastportPtr->width || y > rastportPtr->height)	/* out of area */
		return 0;
		
	next_idx = mimg_sparse->buffer_length;
	planes_array = mimg_sparse->planes_array;
	planes_number = mimg_sparse->comp;
	
	for (c=0; c<planes_number; c++) {
		void *buffer;
		buffer = realloc(planes_array[c]->buffer, mimg_sparse->section_length * width_sparse * GET_TYPE_SIZE(mimg_sparse->kind));
		if (buffer)
			planes_array[c]->buffer = buffer;
		else {
			for (--c; c>=0; c--) 
				planes_array[c]->buffer = realloc(planes_array[c]->buffer, mimg_sparse->buffer_length * GET_TYPE_SIZE(mimg_sparse->kind));
				
			return 0;
		}
	}

	next_idx = mimg_sparse->buffer_length;
	row_start = mimg_sparse->row_start;	

	for (j=0; j<mimg_sparse->data_length; j++) {

		if ((sect_idx = row_start[j]) < 0) {
			yy -= sect_idx;
			continue;
		}

		if (yy >= (rastportPtr->height - y))	/* no more lines to add */
			break;

		sects_per_row = sections_in_row(mimg_sparse, j);

		sects = mimg_sparse->row_section + sect_idx;
		
		for (i=0; i<sects_per_row; i++) {
		
			sect = sects;				
		
			if ((pos = sect->position_in_buffer) >= 0) {
				
				xx = sect->starting_pixel_number;

				if (xx > (rastportPtr->width - x))	/* no more sections to add */
					break;

				if (mimg_sparse->kind == FLOAT_IMAGE) {
					for (c=0; c<planes_number; c++) {
						mpixel_ptr image_buffer = rastportPtr->image->data + ((y + yy) * rastportPtr->width + (x + xx)) * PLANES_IN_IMAGE + (c + 1);


						lmax = MIN(sect->number_of_pixels, rastportPtr->width - (x+xx));
						
						if (sect->position_in_buffer >= 0) {
							fmpixel_ptr buffer = planes_array[c]->buffer + sect->position_in_buffer;

							for (l=0; l<lmax; l++)
								*buffer = op(c1 * *buffer, c2 * *image_buffer);
						} else {
							fmpixel_ptr buffer = planes_array[c]->buffer + pos;

							for (l=0; l<lmax; l++)
								*buffer = op(c1 * *buffer, c2 * *image_buffer);
							for (;l<sect->number_of_pixels;l++)
								*buffer = ((fbyteplane_ptr)(planes_array[c]))->default_value;
							
							pos += sect->number_of_pixels;
						}
						
					}
				} else {
					for (c=0; c<planes_number; c++) {
						mpixel_ptr image_buffer = rastportPtr->image->data + ((y + yy) * rastportPtr->width + (x + xx)) * PLANES_IN_IMAGE + (c + 1);


						lmax = MIN(sect->number_of_pixels, rastportPtr->width - (x+xx));
						
						if (sect->position_in_buffer >= 0) {
							mpixel_ptr buffer = planes_array[c]->buffer + sect->position_in_buffer;

							for (l=0; l<lmax; l++)
								*buffer = op(c1 * *buffer, c2 * *image_buffer);
						} else {
							mpixel_ptr buffer = planes_array[c]->buffer + pos;

							for (l=0; l<lmax; l++)
								*buffer = op(c1 * *buffer, c2 * *image_buffer);
							for (;l<sect->number_of_pixels;l++)
								*buffer = planes_array[c]->default_value;
							
							pos += sect->number_of_pixels;
						}
						
					}
				}
				
				xx += sect->number_of_pixels;
			}
			sects++;	/* advance to next section */
		}

		yy++;
	}

	mimg_sparse->buffer_length = next_idx;
	for (c=0; c<planes_number; c++)
		planes_array[c]->buffer = realloc(planes_array[c]->buffer, mimg_sparse->buffer_length * GET_TYPE_SIZE(mimg_sparse->kind));

	return 1;
}

static int stuff_sparse_over_rastport_rotate_ex(mimage_ptr mimg_sparse, Rastport *rastportPtr, int x, int y, float c1, float c2, float (*op )(float, float), int rotate)
{
	int c;		
	int pix;
	int dest_column_offset;
	int dest_pixel_offset;

	section_data *sect_it;
	size_t sects_in_row;
	size_t data_index, ix, section_index, data_length;

	int width_sparse, height_sparse, planes_number_sparse, kind_sparse;
	unsigned char density_sparse;

	int width, height, planes_number, kind;

	get_mimage_info(mimg_sparse, &height_sparse, &width_sparse, &planes_number_sparse, &kind_sparse, &density_sparse);

	width = rastportPtr->width;
	height = rastportPtr->height;
	planes_number = 3;
	
	if ( (y > height) || (x > width) ) {
		return 0;
	}
	
	if ( planes_number_sparse > planes_number) {
		return 0;
	}

	/*
	 *	do the real computation now
	 */

	if (kind_sparse == DISCRETE_IMAGE) {

		mpixel_ptr buffer_sparse, buffer_sparse_ptr;
		mpixel_ptr dest_buffer, dest_buffer_ptr;
		mpixel	default_value;

		data_length = mimg_sparse->data_length;
		dest_column_offset = x;

		for ( data_index = 0 ; data_index < mimg_sparse->data_length ; data_index ++ )
		{
			int section_data_for_this_row = mimg_sparse->row_start[data_index];
		
			if ( section_data_for_this_row < 0 ) 
			{
				dest_column_offset += -section_data_for_this_row;
				continue;
			}
			
			sects_in_row = sections_in_row(mimg_sparse, data_index);
			
			for ( c=0; c<planes_number ; c++)
			{
				dest_buffer = rastportPtr->image->data;
				buffer_sparse = get_plane_buffer(mimg_sparse->planes_array[c], kind);
		
				default_value = mimg_sparse->planes_array[c]->default_value;
			
				sect_it = mimg_sparse->row_section + section_data_for_this_row;
			
				for ( ix=0 ; ix<sects_in_row ; ix++ )
				{
					int start_to_copy;
					int length_to_copy;
					
					if ((start_to_copy = y + sect_it->starting_pixel_number) > height)
						continue;
				
					if ((start_to_copy + sect_it->number_of_pixels) > height)
						length_to_copy = height - (start_to_copy + sect_it->number_of_pixels);
					else
						length_to_copy = sect_it->number_of_pixels;
				
					dest_pixel_offset = (start_to_copy * width + dest_column_offset) * PLANES_IN_IMAGE + c + 1;
					dest_buffer_ptr = dest_buffer + dest_pixel_offset;

					if (sect_it->position_in_buffer < 0) {	/* this is a default value line */

						for ( pix=0; pix < length_to_copy; pix++) {
							*dest_buffer_ptr = op (c1 * *dest_buffer_ptr, c2 * default_value);
							dest_buffer_ptr += (width * PLANES_IN_IMAGE);
						}

					} else {

						buffer_sparse_ptr = buffer_sparse + sect_it->position_in_buffer;

						for ( pix=0; pix < length_to_copy ; pix++ ) {
							*dest_buffer_ptr = op(c1 * *dest_buffer_ptr, c2 * *buffer_sparse_ptr++);
							dest_buffer_ptr += (width * PLANES_IN_IMAGE);
						}
					}
				
					sect_it++;	/* advance to next section */
				}
			}
			
			dest_column_offset ++;
			
		}

	} else {

		fmpixel_ptr buffer_sparse, buffer_sparse_ptr;
		fmpixel	default_value;
		mpixel_ptr dest_buffer, dest_buffer_ptr;

		data_length = mimg_sparse->data_length;
		dest_column_offset = x;

		for ( data_index = 0 ; data_index < mimg_sparse->data_length ; data_index ++ )
		{
			int section_data_for_this_row = mimg_sparse->row_start[data_index];
		
			if ( section_data_for_this_row < 0 ) 
			{
				dest_column_offset += -section_data_for_this_row;
				continue;
			}
			
			sects_in_row = sections_in_row(mimg_sparse, data_index);
			
			for ( c=0; c<planes_number ; c++)
			{
				dest_buffer = rastportPtr->image->data;
				buffer_sparse = (fmpixel_ptr)get_plane_buffer(mimg_sparse->planes_array[c], kind);
		
				default_value = mimg_sparse->planes_array[c]->default_value;
			
				sect_it = mimg_sparse->row_section + section_data_for_this_row;
			
				for ( ix=0 ; ix<sects_in_row ; ix++ )
				{
					int start_to_copy;
					int length_to_copy;
					
					if ((start_to_copy = y + sect_it->starting_pixel_number) > height)
						continue;
				
					if ((start_to_copy + sect_it->number_of_pixels) > height)
						length_to_copy = height - (start_to_copy + sect_it->number_of_pixels);
					else
						length_to_copy = sect_it->number_of_pixels;
				
					dest_pixel_offset = (start_to_copy * width + dest_column_offset) * PLANES_IN_IMAGE + c + 1;
					dest_buffer_ptr = dest_buffer + dest_pixel_offset;

					if (sect_it->position_in_buffer < 0) {	/* this is a default value line */

						for ( pix=0; pix < length_to_copy; pix++) {
							*dest_buffer_ptr = op (c1 * *dest_buffer_ptr, c2 * default_value);
							dest_buffer_ptr += (width * PLANES_IN_IMAGE);
						}

					} else {

						buffer_sparse_ptr = buffer_sparse + sect_it->position_in_buffer;

						for ( pix=0; pix < length_to_copy ; pix++ ) {
							*dest_buffer_ptr = op(c1 * *dest_buffer_ptr, c2 * *buffer_sparse_ptr++);
							dest_buffer_ptr += (width * PLANES_IN_IMAGE);
						}
					}
				
					sect_it++;	/* advance to next section */
				}
			}
			
			dest_column_offset ++;
			
		}

	}

	return 1;
}

static int stuff_sparse_over_rastport_ex(mimage_ptr mimg_sparse, Rastport *rastportPtr, int x, int y, float c1, float c2, float (*op )(float, float))
{
	int c;		
	int pix;
	int dest_line_offset;
	int dest_pixel_offset;

	section_data *sect_it;
	size_t sects_in_row;
	size_t data_index, ix, section_index, data_length;

	int width_sparse, height_sparse, planes_number_sparse, kind_sparse;
	unsigned char density_sparse;

	int width, height, planes_number, kind;

	get_mimage_info(mimg_sparse, &height_sparse, &width_sparse, &planes_number_sparse, &kind_sparse, &density_sparse);

	width = rastportPtr->width;
	height = rastportPtr->height;
	planes_number = 3;
	
	if ( (y > height) || (x > width) ) {
		return 0;
	}
	
	if ( planes_number_sparse > planes_number) {
		return 0;
	}

	/*
	 *	do the real computation now
	 */

	if (kind_sparse == DISCRETE_IMAGE) {

		mpixel_ptr buffer_sparse, buffer_sparse_ptr;
		mpixel_ptr dest_buffer, dest_buffer_ptr;
		mpixel	default_value;

		data_length = mimg_sparse->data_length;
		dest_line_offset = y;

		for ( data_index = 0 ; data_index < mimg_sparse->data_length ; data_index ++ )
		{
			int section_data_for_this_row = mimg_sparse->row_start[data_index];
		
			if ( section_data_for_this_row < 0 ) 
			{
				dest_line_offset += -section_data_for_this_row;
				continue;
			}
			
			if (dest_line_offset >= rastportPtr->height)	/* GDM 051399 */
				break;	/* no more to write */
			
			sects_in_row = sections_in_row(mimg_sparse, data_index);
			
			for ( c=0; c<planes_number ; c++)
			{
				dest_buffer = rastportPtr->image->data;
				buffer_sparse = get_plane_buffer(mimg_sparse->planes_array[c], kind);
		
				default_value = mimg_sparse->planes_array[c]->default_value;
			
				sect_it = mimg_sparse->row_section + section_data_for_this_row;
			
				for ( ix=0 ; ix<sects_in_row ; ix++ )
				{
					int start_to_copy;
					int length_to_copy;
					
					if ((start_to_copy = x + sect_it->starting_pixel_number) >= width) {
						continue;
					}
					
					if ((start_to_copy + sect_it->number_of_pixels) >= width)
						length_to_copy = width - start_to_copy;
					else
						length_to_copy = sect_it->number_of_pixels;
				
					dest_pixel_offset = (dest_line_offset * width + start_to_copy) * PLANES_IN_IMAGE + c + 1;
					dest_buffer_ptr = dest_buffer + dest_pixel_offset;

					if (sect_it->position_in_buffer < 0) {	/* this is a default value line */

						for ( pix=0; pix < length_to_copy; pix++) {
							*dest_buffer_ptr = op (c1 * *dest_buffer_ptr, c2 * default_value);
							dest_buffer_ptr += PLANES_IN_IMAGE;
						}

					} else {

						buffer_sparse_ptr = buffer_sparse + sect_it->position_in_buffer;

						for ( pix=0; pix < length_to_copy ; pix++ ) {
							*dest_buffer_ptr = op(c1 * *dest_buffer_ptr, c2 * *buffer_sparse_ptr++);
							dest_buffer_ptr += PLANES_IN_IMAGE;
						}
					}
				
					sect_it++;	/* advance to next section */
				}
			}
			
			dest_line_offset ++;
			
		}

	} else {

		fmpixel_ptr buffer_sparse, buffer_sparse_ptr;
		mpixel_ptr dest_buffer, dest_buffer_ptr;
		fmpixel	default_value;

		data_length = mimg_sparse->data_length;
		dest_line_offset = y;

		for ( data_index = 0 ; data_index < mimg_sparse->data_length ; data_index ++ )
		{
			int section_data_for_this_row = mimg_sparse->row_start[data_index];
		
			if ( section_data_for_this_row < 0 ) 
			{
				dest_line_offset += -section_data_for_this_row;
				continue;
			}
			
			if (dest_line_offset >= rastportPtr->height)	/* GDM 051399 */
				break;	/* no more to write */
			
			sects_in_row = sections_in_row(mimg_sparse, data_index);
			
			for ( c=0; c<planes_number ; c++)
			{
				dest_buffer = rastportPtr->image->data; // get_plane_buffer(dest_mimg->planes_array[c], kind);
				buffer_sparse = get_plane_buffer(mimg_sparse->planes_array[c], kind);
		
				default_value = mimg_sparse->planes_array[c]->default_value;
			
				sect_it = mimg_sparse->row_section + section_data_for_this_row;
			
				for ( ix=0 ; ix<sects_in_row ; ix++ )
				{
					int start_to_copy;
					int length_to_copy;
					
					if ((start_to_copy = x + sect_it->starting_pixel_number) >= width) {
						continue;
					}
									
					if ((start_to_copy + sect_it->number_of_pixels) >= width)
						length_to_copy = width - start_to_copy;
					else
						length_to_copy = sect_it->number_of_pixels;
				
					dest_pixel_offset = (dest_line_offset * width + start_to_copy) * PLANES_IN_IMAGE + c + 1;
					dest_buffer_ptr = dest_buffer + dest_pixel_offset;

					if (sect_it->position_in_buffer < 0) {	/* this is a default value line */

						for ( pix=0; pix < length_to_copy; pix++) {
							*dest_buffer_ptr = op (c1 * *dest_buffer_ptr, c2 * default_value);
							dest_buffer_ptr += PLANES_IN_IMAGE;
						}

					} else {

						buffer_sparse_ptr = buffer_sparse + sect_it->position_in_buffer;

						for ( pix=0; pix < length_to_copy ; pix++ ) {
							*dest_buffer_ptr = op(c1 * *dest_buffer_ptr, c2 * *buffer_sparse_ptr++);
							dest_buffer_ptr += PLANES_IN_IMAGE;
						}
					}
				
					sect_it++;	/* advance to next section */
				}
			}
			
			dest_line_offset ++;
			
		}

	}

	return 1;
}

