/*\
 *  %
 *  %  Copyright (c) W. Kirk Snyder
 *  %  All Rights Reserved
 *  %
 *  %  This is unpublished source code for SETL2.  There should be no
 *  %  copies which are not in the possession of the author.  If you
 *  %  somehow come across a copy, please return or destroy it.
 *  %
 *
 *  \package{The Imported Package Table}
 *
 *  This imported packages table is unusually simple -- all we need is a
 *  place to store the packages imported by a program or package.  We
 *  keep a linked list of these nodes attached to the procedure table.
 *
 *  \texify{import.h}
 *
 *  \packagebody{Imported Package Table}
\*/

/* SETL2 system header files */

#include "system.h"                    /* SETL2 system constants            */
#include "compiler.h"                  /* SETL2 compiler constants          */
#include "giveup.h"                    /* severe error handler              */
#include "messages.h"                  /* error messages                    */
#include "import.h"                    /* imported packages and classes     */

/* performance tuning constants */

#define IMPORT_BLOCK_SIZE     50       /* import block size                 */

/* generic table item structure (import use) */

struct table_item {
   union {
      struct table_item *ti_next;      /* next free item                    */
      struct import_item ti_data;      /* data area when in use             */
   } ti_union;
};

/* block of table items */

struct table_block {
   struct table_block *tb_next;        /* next block of data                */
   struct table_item tb_data[IMPORT_BLOCK_SIZE];
                                       /* array of table items              */
};

/* package-global data */

static struct table_block *table_block_head = NULL;
                                       /* list of table blocks              */
static struct table_item *table_next_free = NULL;
                                       /* list of free table items          */

/*\
 *  \function{init\_import()}
 *
 *  This procedure initializes the import table. All we do is push
 *  everything onto the free list.
\*/

void init_import()

{
struct table_block *tb_ptr;            /* work name table pointer           */

   while (table_block_head != NULL) {
      tb_ptr = table_block_head;
      table_block_head = table_block_head->tb_next;
      free((void *)tb_ptr);
   }
   table_next_free = NULL;
}

/*\
 *  \function{get\_import()}
 *
 *  This procedure allocates a import node. It is just like most of the
 *  other dynamic table allocation functions in the compiler.
\*/

import_ptr_type get_import(SETL_SYSTEM_PROTO_VOID)

{
struct table_block *old_head;          /* name table block list head        */
import_ptr_type return_ptr;            /* return pointer                    */

   if (table_next_free == NULL) {

      /* allocate a new block */

      old_head = table_block_head;
      table_block_head = (struct table_block *)
                         malloc(sizeof(struct table_block));
      if (table_block_head == NULL)
         giveup(SETL_SYSTEM msg_malloc_error);
      table_block_head->tb_next = old_head;

      /* link items on the free list */

      for (table_next_free = table_block_head->tb_data;
           table_next_free <=
              table_block_head->tb_data + IMPORT_BLOCK_SIZE - 2;
           table_next_free++) {
         table_next_free->ti_union.ti_next = table_next_free + 1;
      }

      table_next_free->ti_union.ti_next = NULL;
      table_next_free = table_block_head->tb_data;
   }

   /* at this point, we know the free list is not empty */

   return_ptr = &(table_next_free->ti_union.ti_data);
   table_next_free = table_next_free->ti_union.ti_next;

   /* initialize the new entry */

   clear_import(return_ptr);

   return return_ptr;

}

/*\
 *  \function{free\_import()}
 *
 *  This function is the complement to \verb"get_import()". All we do is
 *  push the passed import table pointer on the free list.
\*/

void free_import(
   import_ptr_type discard)            /* item to be discarded              */

{

   ((struct table_item *)discard)->ti_union.ti_next = table_next_free;
   table_next_free = (struct table_item *)discard;

}

