#include <memory.h>
#include <stdlib.h>
#include <string.h>
#include "barbados.h"
#include "ipr.h"



Expr* debuge;


/*--------------------- Misc functions: --------------------*/

interface bool IsComparison(o_enum o)
{
        return o >= oLT and o <= oGE;
}


static o_enum AssignopToOp(o_enum o)
{
	switch (o) {
	    case oAADD:	return oADD;
	    case oASUB:	return oSUB;
	    case oAMUL:	return oMUL;
	    case oADIV:	return oDIV;
	    case oASHR:	return oSHR;
	    case oASHL:	return oSHL;
	    case oAMOD:	return oMOD;
	    case oAAND:	return oAND;
	    case oAOR:	return oOR;
	    case oAXOR:	return oXOR;
	    default:	assert(false);
			return oNOP1;
	}
}


interface str OtoString(int o)
{
#define O(s)	case o##s: return #s
	switch (o) {
	    O(Unknown);
	    O(Constant);
	    O(LocalVar);
	    O(NOP1);
	    O(CONVERT);
	    O(NEG);
	    O(NOT);
	    O(NOTT);
	    O(DER);
	    O(SIN);
	    O(COS);
	    O(TAN);
	    O(ATAN);
	    O(SQRT);
	    O(ADD);
	    O(SUB);
	    O(MUL);
	    O(DIV);
	    O(MOD);
	    O(LT);
	    O(GT);
	    O(EQ);
	    O(NE);
	    O(LE);
	    O(GE);
	    O(AND);
	    O(OR);
	    O(XOR);
	    O(ANND);
	    O(ORR);
	    O(SHL);
	    O(SHR);
	    O(COMMA);
	    O(TERNARY);
	    O(BITFIELD);
	    O(ASSIGN);
	    O(AADD);
	    O(ASUB);
	    O(AMUL);
	    O(ADIV);
	    O(AMOD);
	    O(AXOR);
	    O(AOR);
	    O(AAND);
	    O(ASHL);
	    O(ASHR);
	    O(AADDAFTER);
	    O(FUNC);
	    O(MEMIA);
	    default:	return "???";
#undef O
        }
}


interface str OtoString(Expr* e)
{
	return OtoString((o_enum)e->o);
}





/*--------------- Registering statements and expressions: ---------------*/

interface Statement* first_statement;


interface void IprRegisterStatement(Statement* S)
/* Add this statement to a linear list of statements */
/* used by the current function.  We use this to     */
/* detect dead code and also to simplify iteration   */
/* through statements. */
{
        if (S == NULL) {
            /* This is the signal to initialise Ipr. */
            first_statement = NULL;
            return;
        }
        S->next_registered = first_statement;
        first_statement = S;
}








/*================ Optimisations: ================*/

#define MAX_TERMS   50

typedef struct opt_term_node {
    int sign;
    int complexity;
    Expr* e;
} *opt_term_type;


interface void OptimiseExpr(Expr* *ep, bool need_result);

static struct opt_term_node global_Term[MAX_TERMS];
static o_enum global_Add, global_Sub, global_Neg;
static int global_idx;
    

static int PowerOf2(int n)
/* Is this integer a power of 2? 0=no. */
{	int l, p;

	l = 1;
	p = 0;
	while (l < n)
	    l <<= 1, p++;
	if (l == n)
	    return p;
	else return 0; 
}


static void OptCommCollect(Expr* e, int sign)
/* Recursively collect terms into 'term'. */
{	opt_term_type ocTrm;

	if (e->o == oNOP1)
	    OptCommCollect(e->left, sign);
	else if (e->o == global_Add) {
	    OptCommCollect(e->left, sign);
	    OptCommCollect(e->right, sign);
	}
	else if (e->o == global_Sub) {
	    if (e->tp != tp_double and sign == -1 and global_Sub == oDIV)
		goto ATOMIC;    /* Integer division has a truncation operation */
		/* secretly embedded in it. */
	    OptCommCollect(e->left, sign);
	    OptCommCollect(e->right, -sign);
	}
	else if (e->o == global_Neg) {
	    OptCommCollect(e->left, -sign);
	}
	else {
	    /* An atomic term: */
	    ATOMIC:
	    if (global_idx >= MAX_TERMS) {
		ErrorParse(e->src, "Too many terms");
		return;
	    }
	    ocTrm = &global_Term[global_idx++];
	    ocTrm->sign = sign;
	    ocTrm->e = e;
	}
}


static bool CanCombinePointer(Expr* a, Expr* b)
/* Is it possible to combine these two expressions?  One of them may be a pointer. */
{
	if (a->tp == tp_pointer and b->tp == tp_pointer) {
	    if (a->right == b->right) {
		a->tp = tp_int;
		return yes;
	    }
	    else return no;
	    /* Can't combine two pointers with different bases. */
	}
	if (b->tp == tp_pointer) {
	    a->tp = tp_pointer;
	    a->right = b->right;
	}
	return yes;
}


static int OptCommComplexity(Expr* e)
/* Returns the complexity of this expression. */
{       exprf_type ef;
        int i,r,m;

        switch (e->o) {
            case oUnknown:  return 0;
            case oConstant: return e->tp == tp_double ? 1 : 0;
            case oPUSHSPPLUS:
            case oLocalVar: return 0;

            case Unaries:   return OptCommComplexity(e->left);

            case oFUNC:     ef = e->u.func;
                            m = 1;
                            for (i=0; i < ef->arity; i++) {
                                r = OptCommComplexity(ef->param[i]);
                                if (r > m)
                                    m = r;
                            }
                            return m;

            case oBITFIELD: m = OptCommComplexity(e->u.bitfield->addr);
                            r = OptCommComplexity(e->u.bitfield->operand);
                            return (r > m) ? r : m;

            default:        m = OptCommComplexity(e->left);
                            r = OptCommComplexity(e->right) + (e->tp==tp_double?1:5);
                            return (r > m) ? r : m;
        }
}


static int compar1(opt_term_type a, opt_term_type b)
{
        if (b->complexity != a->complexity)
            return b->complexity - a->complexity;
	if (b->e->o != a->e->o)
	    return b->e->o - a->e->o;
	return b->e->tp - a->e->tp;
}


static void OptimiseCommutative(Expr* e, o_enum _ADD,
                                o_enum _SUB, o_enum _NEG)
/* We have an expression 'e' which has one of these 3 instructions */
/* at the root.  We flatten the tree into a set of positive and    */
/* negative terms (or factors) to optimise it.  The instructions   */
/* can be addition or multiplication instructions, integer/long or */
/* float. */
{	struct opt_term_node Term[MAX_TERMS], tmp_term;
	str src=e->src;
	Expr* f;
	tp_enum tp;
	int i, idx;

	/* Collect the terms: */
	global_Add = _ADD;
	global_Sub = _SUB;
	global_Neg = _NEG;
	global_idx = 0;
	OptCommCollect(e, 1);
	if (global_idx == 1) {
	    OptimiseExpr(&global_Term[0].e, yes);
	    return;
	}
	memcpy(Term, global_Term, global_idx*sizeof(struct opt_term_node));
	idx = global_idx;


	/* Optimise them and evaluate their resulting complexity: */
	for (i=0; i < idx; i++) {
	    OptimiseExpr(&Term[i].e, yes);
	    Term[i].complexity = OptCommComplexity(Term[i].e);
	}


	/* Sort them to get the most complex first. */
	qsort(Term, idx, sizeof(Term[0]), (cmp_fn)compar1);


	/* Combine all the constant terms: */
    #define COMBINE_WITH_NEG(field,Zero,_op_,inv,cancombine)           \
	{                                           \
	    for (i=idx-1; i >= 0; i--) {		\
		if (Term[i].e->o != oConstant)      \
		    break;                          \
		if (Term[i].sign == -1) {           \
		    Term[i].e->field = inv Term[i].e->field;\
		    Term[i].sign = 1;               \
		}                                   \
	    }                                       \
	    if (++i < idx) {			\
		while (idx - 1 > i and cancombine)      \
		    Term[i].e->field _op_##= Term[--idx].e->field;\
		if (Term[i].e->field == Zero and idx > 1)\
		    idx--;				\
	    }                                       \
	}

    #define COMBINE_UNIDIR(field,Zero,_op_)         \
	{                                           \
	    for (i=idx-1; i >= 0; i--) {		\
		if (Term[i].e->o != oConstant)      \
		    break;                          \
	    }                                       \
	    if (++i < idx) {			\
		while (idx - 1 > i)			\
		    Term[i].e->field _op_##= Term[--idx].e->field;\
		if (Term[i].e->field == Zero and idx > 1)\
		    idx--;				\
	    }                                       \
	}

	tp = TpToSigned(e->tp);
	if (_ADD == oADD) {
	    if (tp == tp_int or tp == tp_short or tp == tp_char)
		COMBINE_WITH_NEG(u.i, 0, +, -, true)
	    else if (e->tp == tp_pointer)
		COMBINE_WITH_NEG(u.i, 0, +, -, CanCombinePointer(Term[i].e, Term[i+1].e))
	    else if (e->tp == tp_float or e->tp == tp_double)
		COMBINE_WITH_NEG(u.f, 0, +, -, true)
	    else if (e->tp == tp_long)
		COMBINE_WITH_NEG(u.l, 0, +, -, true)
	    if (idx >= 2) {
		/* Combine together a local offset and a constant. */
		if (Term[idx-1].e->o == oConstant and Term[idx-2].e->o == oLocalVar)
		    (int&)(Term[idx-2].e->right) += Term[idx-1].e->u.i, idx--;
	    }
	}
	else if (_ADD == oMUL) {
	    if (tp == tp_int or e->tp == tp_short or e->tp == tp_char)
		COMBINE_UNIDIR(u.i, 1, *)
	    else if (e->tp == tp_float or e->tp == tp_double)
		COMBINE_WITH_NEG(u.f, 1.0, *, 1.0/, true)
	    else if (e->tp == tp_long)
		COMBINE_UNIDIR(u.i, 1, *)
	}
	else if (_ADD == oOR) {
	    if (tp == tp_int or tp == tp_short or tp == tp_short)
		COMBINE_UNIDIR(u.i, 0, |)
	    else if (e->tp == tp_long)
		COMBINE_UNIDIR(u.l, 0, |)
	}
	else if (_ADD == oAND) {
	    if (tp == tp_int or tp == tp_short or tp == tp_short)
		COMBINE_UNIDIR(u.i, ~(int)0, &)
	    else if (e->tp == tp_long)
		COMBINE_UNIDIR(u.l, ~(long)0, &)
	}

	/* Re-combine it into an expression: */
	assert(idx > 0);
	if (Term[0].sign == -1) {
	    for (i=1; i < idx; i++)
		if (Term[i].sign == 1) {
		    tmp_term = Term[i];
		    Term[i] = Term[0];
		    Term[0] = tmp_term;
		    break;
		}
	}
	if (Term[0].sign == 1)
	    f = Term[0].e;
	else {
	    f = NewExpr(e->opr, Term[0].e, NULL);
	    if (_NEG != oUnknown) {
		f->o = _NEG;
		f->tp = e->tp;
		f->src = Term[0].e->src;
	    }
	    else {
		f->o = _SUB;
		f->tp = e->tp;
		f->src = Term[0].e->src;
		f->right = f->left;
		assert(e->tp == tp_double or e->tp == tp_float);
		f->left = NewExpr(0, NULL, NULL);
		f->left->o = oConstant;
		f->left->u.f = 1.0;
		f->left->tp = e->tp;
	    }
	}
	i = 0;
	while (++i < idx) {
	    f = NewExpr(e->opr, f, Term[i].e);
	    if (Term[i].sign == 1)
		f->o = _ADD;
	    else f->o = _SUB;
	    f->src = Term[i].e->src;
	    f->tp = e->tp;
	}


	/* Place it back into the original: */
	*e = *f;
	e->src = src;
}


static bool ConstantExpr(Expr* *ep)
/* Returns 'yes' if this expression optimises to a constant. */
/* Involves optimising the sub-tree.  */
{
	OptimiseExpr(ep, yes);
	if ((*ep)->o == oConstant)
	    return yes;
	else return no;
}


static Expr* boolToTernary(Expr* e)
/* Convert a bool expression, i.e. an expression that doesn't */
/* generate a result in a register, (only flags) into a ternary  */
/* expression of the form:   e ? 1 : 0                           */
{
        if (e->o == oNOTT)
            e = NewExprTernary(e->left, NewExprInt(0), NewExprInt(1));
        else e = NewExprTernary(e, NewExprInt(1), NewExprInt(0));
        e->o = oTERNARY;
        e->tp = tp_int;
        return e;
}


#if 0
static Expr* NewExprAssignToTemp(void* temp, Expr* rvalue)
/* Create a new subexpression of the form:  localval = rvalue. */
/* 'rvalue' is decorated already.  The return must be decorated. */
{       Expr* e, lvalue;

        lvalue = NewExpr(string_const, temp, NULL);
        lvalue->type = rvalue->type;
        lvalue->ref = 0;
        lvalue->o = oLocalVar;
        lvalue->tp = rvalue->tp;
        e = NewExpr(op_sing('='), lvalue, rvalue);
        e->o = oASSIGN;
        e->tp = rvalue->tp;
        e->type = rvalue->type;
}
#endif


interface void OptimiseExpr(Expr **ep, bool need_result)
{	Expr* e;
	int i;

	do {
	    e = *ep;
	    if (e->o != oNOP1)
		break;
	    *ep = e->left;
	} forever;
	switch (e->o) {
	    case oUnknown:
	    case oLocalVar:
            case oPUSHSPPLUS:
			break;

	    case oConstant:
			if (e->tp == tp_pointer) {
			    if (e->right != localstatic_base)
				e->right = e->left;
			}
			/* This is to set up the 'base' pointer for this pointer. */
			/* The 'base' pointer is the pointer to the beginning of  */
			/* the malloc block that e->u.p refers to.  We need to    */
			/* onow it in order to be able to swizzle pointers even   */
			/* after constants have been folded into them. */
			break;

	    case oADD:
	    case oSUB:  OptimiseCommutative(e, oADD, oSUB, oNEG);
			break;

	    case oNEG:  OptimiseCommutative(e, oADD, oSUB, oNEG);
			if (e->left->o == oConstant) {
			    e->o = oConstant;
			    if (e->tp == tp_double)
				e->u.f = -e->left->u.f;
			    else {
				e->right = NULL;
				e->u.i = -e->left->u.i;
			    }
			}
			break;

	    case oMUL:  OptimiseCommutative(e, oMUL, oMUL, oUnknown);
			/* Integer division is not rearrangeable. */
			if (e->o == oMUL and e->tp != tp_double and e->tp != tp_float) {
			    if (e->right->o == oConstant) {
				i = PowerOf2(e->right->u.i);
				if (i > 0)
				    e->o = oSHL, e->right->u.i = i;
			    }
			    else if (e->left->o == oConstant) {
				i = PowerOf2(e->left->u.i);
				if (i > 0) {
				    Expr* t;

				    e->o = oSHL;
				    t = e->right;
				    e->right = e->left;
				    e->left = t;
				    e->right->u.i = i;
				    e->right_first = no;
				}
			    }
			}
			break;

	    case oDIV:  if (e->tp == tp_float or e->tp == tp_double)
			    OptimiseCommutative(e, oMUL, oDIV, oUnknown);
			else if (e->left->o == oConstant and e->right->o == oConstant) {
			    e->o = oConstant;
                            if (e->right->u.i == 0)
                                ErrorParse(e->src, "Divide by zero");
			    else e->u.i = e->left->u.i / e->right->u.i;
			}
			else if (e->right->o == oConstant and e->right->u.i == 1) {
			    e->o = oNOP1;	// Division by 1
			    e->right = NULL;
			}
			else if (e->right->o == oConstant and (i=PowerOf2(e->right->u.i)) != 0) {
			    /* Is it a power of 2? */
			    e->o = oSHR;
			    e->right->u.i = i;
			}
			break;

	    case oOR:   OptimiseCommutative(e, oOR, oOR, oUnknown);
			break;

	    case oAND:  OptimiseCommutative(e, oAND, oAND, oUnknown);
			break;

	    case oXOR:  OptimiseCommutative(e, oXOR, oXOR, oNOT);
			break;

	    case oNOT:  OptimiseCommutative(e, oXOR, oXOR, oNOT);
			if (e->left->o == oConstant) {
			    e->o = oConstant;
			    e->u.i = ~e->left->u.i;
			    e->right = NULL;
			}
			break;

	    case oMOD:  if (ConstantExpr(&e->left)) {
			    int a,b;
			    a = e->left->u.i;
			    if (ConstantExpr(&e->right)) {
				b = e->right->u.i;
				e->o = oConstant;
				if (b == 0) {
				    ErrorParse(e->src, "Divide by zero");
				}
				else {
				    e->u.i = a % b;
				    e->o = oConstant;
				    e->tp = tp_int;
				}
			    }
			}
			break;

	    case oFUNC: OptimiseExpr(&e->u.func->func, yes);
			for (i=0; i < e->u.func->arity; i++)
			    OptimiseExpr(&e->u.func->param[i], yes);
			break;

	    case oDER:	if (e->left->o == oBITFIELD) {
			    e->left->u.bitfield->o = oDER;
			    e->o = oBITFIELD;
			    e->u.bitfield = e->left->u.bitfield;
			    OptimiseExpr(&e->u.bitfield->addr, yes);
			}
			else {
			    OptimiseExpr(&e->left, yes);
			}
			break;

	    case oSIN: case oCOS: case oTAN: case oATAN: case oSQRT:
			OptimiseExpr(&e->left, yes);
			break;

	    case oCONVERT:
			OptimiseExpr(&e->left, yes);
			if (e->left->o == oConstant) {
			    e->o = oConstant;
			    if (e->left->tp == tp_double or e->left->tp == tp_float) {
				if (e->tp == tp_long)
				    e->u.l = (long)e->left->u.f;
				else if (e->tp == tp_double or e->tp == tp_float)
				    e->u.f = e->left->u.f;
				else e->u.i = (int)e->left->u.f;
			    }
			    else if (e->left->tp == tp_long) {
				if (e->tp == tp_double or e->tp == tp_float)
				    e->u.f = e->left->u.l;
				else if (e->tp == tp_long)
				    e->u.l = e->left->u.l;
				else e->u.i = e->left->u.l;
			    }
			    else if (TpIsIntegral(e->left->tp) or e->left->tp == tp_pointer) {
				if (e->tp == tp_double or e->tp == tp_float)
				    e->u.f = e->left->u.i;
				else if (e->tp == tp_long)
				    e->u.l = e->left->u.i;
				else e->u.i = e->left->u.i;
			    }
			}
			break;

	    case oLT:
	    case oGT:
	    case oEQ:
	    case oNE:
	    case oLE:
	    case oGE:   OptimiseExpr(&e->left, yes);
			OptimiseExpr(&e->right, yes);
			if (e->left->o == oConstant and e->right->o == oConstant) {
			    bool eq,lt;
			    double af,bf;
			    int ai,bi;

			    if (e->left->tp == tp_double) {
				af = e->left->u.f;
				bf = e->right->u.f;
				eq = (af == bf);
				lt = (af < bf);
			    }
			    else {
				ai = e->left->u.i;
				bi = e->right->u.i;
				eq = (ai == bi);
				lt = (ai < bi);
			    }
			    switch (e->o) {
				case oLT:   e->u.i = lt; break;
				case oGT:   e->u.i = not lt and not eq; break;
				case oEQ:   e->u.i = eq; break;
				case oNE:   e->u.i = not eq; break;
				case oLE:   e->u.i = lt or eq; break;
				case oGE:   e->u.i = not lt; break;
			    }
			    e->o = oConstant;
			}
			else if (need_result) {
			    /* We need to convert this to a ternary construct:   e ? 1 : 0. */
			    *ep = boolToTernary(e);
			}
			break;

	    case oSHL:	OptimiseExpr(&e->left, yes);
			OptimiseExpr(&e->right, yes);
			if (e->left->o == oConstant and e->right->o == oConstant) {
			    e->o = oConstant;
			    e->u.i = e->left->u.i << e->right->u.i;
			}
			break;

	    case oSHR:	OptimiseExpr(&e->left, yes);
			OptimiseExpr(&e->right, yes);
			if (e->left->o == oConstant and e->right->o == oConstant) {
			    e->o = oConstant; 
			    e->u.i = e->left->u.i >> e->right->u.i;
			}
			break;

	    case oTERNARY:
			OptimiseExpr(&e->left, no);
			if (e->left->o == oConstant) {
			    *ep = (e->left->u.i ? e->right->left : e->right->right);
			    OptimiseExpr(ep, need_result);
			}
			else {
			    OptimiseExpr(&e->right->left, need_result);
			    OptimiseExpr(&e->right->right, need_result);
			}
			break;

	    case oANND: OptimiseExpr(&e->left, no);
			OptimiseExpr(&e->right, no);
			if (e->left->o == oConstant) {
			    if (e->left->u.i)
				e = e->right;
			    else e->o = oConstant, e->u.i = 0;
			}
			else if (e->right->o == oConstant) {
			    if (e->right->u.i)
				e = e->left;
			    else e->o = oConstant, e->u.i = 0;
			}
			if (e->o == oConstant) {
			    e->u.i = (e->u.i != 0);
			}
			else if (need_result) {
			    /* We need to convert this to a ternary construct:   e ? 1 : 0. */
			    e = boolToTernary(e);
			}
			*ep = e;
			break;
                                        
	    case oORR:  OptimiseExpr(&e->left, no);
			OptimiseExpr(&e->right, no);
			if (e->left->o == oConstant) {
			    if (e->left->u.i == 0)
				e = e->right;
			    else e->o = oConstant, e->u.i = 1;
			}
			else if (e->right->o == oConstant) {
			    if (e->right->u.i == 0)
				e = e->left;
			    else e->o = oConstant, e->u.i = 1;
			}
			if (e->o == oConstant) {
			    e->u.i = (e->u.i != 0);
			}
			else if (need_result) {
			    /* We need to convert this to a ternary construct:   e ? 1 : 0. */
			    e = boolToTernary(e);
			}
			*ep = e;
			break;
        
	    case oNOTT: OptimiseExpr(&e->left, no);
			if (e->left->o == oConstant) {
			    e->o = oConstant;
			    e->u.i = not e->left->u.i;
			}
			if (need_result) {
			    /* We need to convert this to a ternary construct:   e ? 1 : 0. */
			    *ep = boolToTernary(e);
			}
			break;

	    case oCOMMA:OptimiseExpr(&e->left, no);
			OptimiseExpr(&e->right, yes);
			break;

	    case oBITFIELD:
			ErrorType(e->src, "You're not allowed to take the address of bit-fields.");
			break;

	    case oPUSH:	OptimiseExpr(&e->left, yes);
            		if (e->right)
                            OptimiseExpr(&e->right, yes);
			break;

	    case oASSIGN:
	    case oAADD:
	    case oASUB:
	    case oAMUL:
	    case oADIV:
	    case oAMOD:
	    case oAXOR:
	    case oAOR:
	    case oAAND:
	    case oASHL:
	    case oASHR:
	    case oAADDAFTER:
			if (e->left->o != oBITFIELD) {
			    OptimiseExpr(&e->left, yes);
			    OptimiseExpr(&e->right, yes);
			}
			else {
			    e->left->u.bitfield->o = e->o;
			    e->left->u.bitfield->operand = e->right;
			    e->o = oBITFIELD;
			    e->u.bitfield = e->left->u.bitfield;
			    e->right = NULL;
			    OptimiseExpr(&e->u.bitfield->addr, yes);
			    OptimiseExpr(&e->u.bitfield->operand, yes);
			}
			break;

	    default:	assert(false);
			break;
	}
	if (e->tp == tp_enumerated or e->tp == tp_bool)
	    e->tp = tp_int;
	else if (e->tp == tp_array)
	    e->tp = tp_pointer;
}



static void OptimiseShortCut(IfStatement* S_if)
{	Statement* S;
	IfStatement* S2;
	Expr* e;

	LOOP:
	e = S_if->Test;
	if (e->o == oUnknown and e->right == NULL) {
	    S_if->Test = e->left;
	    goto LOOP;
	}
	else if (e->o == oUnknown and e->left == NULL) {
	    S_if->Test = e->right;
	    goto LOOP;
	}
	else if (e->o == oNOTT) {
	    S = S_if->True;
	    S_if->True = S_if->Fals;
	    S_if->Fals = S;
	    S_if->Test = e->left;
	    goto LOOP;
	}
	else if (e->o == oANND) {
	    S2 = (IfStatement*)NewStatement(st_if);
	    S2->Test = e->right;
	    S2->True = S_if->True;
	    S2->Fals = S_if->Fals;
            S2->src = e->right->src;
            OptimiseShortCut(S2);
	    S_if->True = (Statement*)S2;
	    S_if->Test = e->left;
	    goto LOOP;
	}
	else if (e->o == oORR) {
	    S2 = (IfStatement*)NewStatement(st_if);
	    S2->Test = e->right;
	    S2->True = S_if->True;
	    S2->Fals = S_if->Fals;
            S2->src = e->right->src;
            OptimiseShortCut(S2);
	    S_if->Fals = (Statement*)S2;
	    S_if->Test = e->left;
	    goto LOOP;
	}
}


static str ExprSrc(Expr* e)
/* Find a source-ptr for this expression. */
{       exprf_type ef;
	str s;
        int i;

        if (e->src)
            return e->src;
        switch (e->o) {
            case oUnknown:
            case oConstant:
            case oLocalVar:
            case oPUSHSPPLUS:
                return NULL;
            case Unaries:
                return ExprSrc(e->left);
            case oFUNC:     
                ef = e->u.func;
                for (i=0; i < ef->arity; i++)
                    if ((s=ExprSrc(ef->param[i])) != NULL)
                        return s;
                return NULL;
	    case oBITFIELD:
                return ExprSrc(e->u.bitfield->addr);
            default:        
                if ((s=ExprSrc(e->left)) != NULL)
		    return s;
                if ((s=ExprSrc(e->right)) != NULL)
		    return s;
		return NULL;
        }
}


#if 0
static Statement* PreExpression_root;

static void StashPreExpression(Statement* S)
/* Add a statement to the list:  PreExpression_root. */
{
        S->next = PreExpression_root;
        PreExpression_root = S;
}


static void InsertPreExpression(Statement* Orig)
/* Convert this to a null statement, pointing to PreExpression_root's, */
/* followed by 'S'. */
{       Statement* S, Newbie, *Sp1, *Sp2;
        IfStatement* S_if;
        int size;

        switch (Orig->st) {
            case st_expr:   size = sizeof(ExprStatement); break;
            case st_switch: size = sizeof(SwitchStatement); break;
            case st_return: size = sizeof(ReturnStatement); break;
            case st_if:     size = sizeof(IfStatement); break;
            default:        assert(false);
        }
        Newbie = qmalloc(size);
        memcpy(Newbie, Orig, size);
        IprRegisterStatement(Newbie);
        Orig->st = st_null;
        Sp2 = NULL;
        Sp1 = &Orig->next;
        for (S=PreExpression_root; S; S=S->next) {
            if (Sp1)
                *Sp1 = S, Sp1 = NULL;
            if (Sp2)
                *Sp2 = S, Sp2 = NULL;
            if (S->st == st_expr)
                Sp1 = &S->next;
            else if (S->st == st_if) {
                S_(S_if);
                Sp1 = &S_if->True;
                Sp2 = &S_if->False;
            }
            else assert(false);
        }
        if (Sp1)
            *Sp1 = Newbie;
        if (Sp2)
            *Sp2 = Newbie;
}
#endif


static void CheckExprIsUseful(Expr* e)
/* Check if this is a valid top-level expression.  If it doesn't do anything, */
/* then it's an error. */
{
        if (e->o == oCOMMA) {
            CheckExprIsUseful(e->left);
            CheckExprIsUseful(e->right);
        }
	else if (e->o == oNOP1)
            CheckExprIsUseful(e->left);
        else if (e->o < oASSIGN or e->o > oFUNC) {
            ErrorType(ExprSrc(e), "Code has no effect");
        }
}


static void OptimiseExpressions(void)
/* Cycle through all expressions in this statement and optimise them. */
{	ReturnStatement* S_return;
	SwitchStatement* S_switch;
	ExprStatement* S_expr;
	IfStatement* S_if;
	Statement* S;

	for (each_statement) {
	    switch (S->st) {
		case st_expr:
			S_expr = (ExprStatement*)S;
			CheckExprIsUseful(S_expr->e);
			OptimiseExpr(&S_expr->e, no);
			break;

		case st_if:
			S_if = (IfStatement*)S;
			OptimiseExpr(&S_if->Test, no);
			OptimiseShortCut(S_if);
			if (S_if->Test->o == oConstant) {
			    S_if->st = st_null;
			    S_if->next = S_if->Test->u.i ? S_if->True : S_if->Fals;
			}
			break;

		case st_switch:
			S_switch = (SwitchStatement*)S;
			OptimiseExpr(&S_switch->Test, yes);
			break;

		case st_return:
                        S_return = (ReturnStatement*)S;
			if (S_return->e and S_return->e != (Expr*)0x1)
			    OptimiseExpr(&S_return->e, yes);
			break;

		case st_null:
			break;

		default:assert(false);
			break;
	    }
	}
}


interface int IprEvaluateIntConstant(Expr* e, bool *successp)
/* Hopefully this expression evaluates to an integer constant. */
/* If so, return it. */
{
        OptimiseExpr(&e, yes);
        if (e->o != oConstant) {
            *successp = no;
            return 0;
        }
        *successp = yes;
        return e->u.i;
}


interface bool IprEvaluateConstant(Expr* *ep)
/* Optimise this expression.  Hopefully it evaluates to a constant. */
/* If so, leave it in 'e->u'. */
{
        OptimiseExpr(ep, yes);
	return (*ep)->o == oConstant;
}





/*======================== Variable graphs: ===========================*/

/* 
In this section, we construct graphs of local variable accesses.  Each expr 
where a local variable is read or written is given a 'vargraph_node'.  These
nodes are then linked together in a graph, each line representing the fact
that execution can flow from node 1 to node 2 without hitting any nodes for
this variable in the middle.
	This analysis then allows us to identify first uses, last uses, and
when the variable holds a constant value.
*/

typedef struct vargraph_node {
	AutoNamedobj *obj;
	Statement* S;
	Expr* parent;
	int4 minm, maxm;
	int n_idx, p_idx, na_idx, pa_idx;	// The used and allocated indexes for next and prev.
	uchar assignment, straight_assignment, flag1;
	struct vargraph_node **next;
	struct vargraph_node **prev;
	struct vargraph_node *next_in_list;
} *vargraph_type;


static Statement* CurrentStatement;
static vargraph_type CurrentVg;
static AutoNamedobj* CurrentLocal;
static bool PassedOurself;

static void VargraphLink(vargraph_type vg, Statement* S);



static void VargraphMakeLink(vargraph_type parent, vargraph_type child)
{	vargraph_type *List;

	/* The link from the parent to the child: */
	if (parent->n_idx + 1 >= parent->na_idx) {
	    if (parent->next == NULL) {
		parent->na_idx = 10;
		parent->next = (vargraph_type*)qmalloc(parent->na_idx * sizeof(vargraph_type));
	    }
	    else {
		parent->na_idx *= 2;
		List = (vargraph_type*)qmalloc(parent->na_idx * sizeof(vargraph_type));
		memcpy(List, parent->next, parent->n_idx * sizeof(vargraph_type));
	    }
	}
	parent->next[parent->n_idx++] = child;

	/* The link from the child to the parent: */
	if (child->p_idx + 1 >= child->pa_idx) {
	    if (child->prev == NULL) {
		child->pa_idx = 10;
		child->prev = (vargraph_type*)qmalloc(child->pa_idx * sizeof(vargraph_type));
	    }
	    else {
		child->pa_idx *= 2;
		List = (vargraph_type*)qmalloc(child->pa_idx * sizeof(vargraph_type));
		memcpy(List, child->prev, child->p_idx * sizeof(vargraph_type));
	    }
	}
	child->prev[child->p_idx++] = parent;
}


static vargraph_type VargraphNew(AutoNamedobj *obj, Expr* e, Expr* parent)
{	vargraph_type vg;

	vg = (vargraph_type)qcalloc(sizeof(struct vargraph_node));
	if (obj->vargraph == NULL)
	    obj->vargraph = vg;
	vg->parent = parent;
	vg->S = CurrentStatement;
	vg->next = vg->prev = NULL;
	vg->next_in_list = obj->root;
	obj->root = vg;
	vg->assignment = vg->straight_assignment = no;
	if (parent) {
	    vg->assignment = (parent->o >= oASSIGN and parent->o <= oAADDAFTER and parent->left == e);
	    vg->straight_assignment = (parent->o == oASSIGN and parent->left == e);
	}
	assert(e->o == oLocalVar);
	e->right = (Expr*)vg;
	vg->obj = obj;
	return vg;
}


static bool RecurseVargraphLinkExpr(Expr* e, Expr* parent, Statement* S)
/* Recursively traverse this expression tree, searching for occurrences */
/* of CurrentLocal being accessed.  If you find such an occurrence,     */
/* return 'yes'. */
{       AutoNamedobj *obj;
	vargraph_type vg;
	exprf_type ef;
	int i;

        switch (e->o) {
            case oUnknown:
            case oPUSHSPPLUS:
            case oConstant: return no;

            case oLocalVar: if (e->u.obj != CurrentLocal)
				return no;
			    if ((vargraph_type)e->right == CurrentVg) {
				PassedOurself = yes;
				return no;	// We're just passing ourselves.
			    }
			    if (S == CurrentStatement and not PassedOurself)
				return no;
			    if (e->right) {
				VargraphMakeLink(CurrentVg, (vargraph_type)e->right);
				return yes;
			    }
			    obj = e->u.obj;
			    vg = VargraphNew(obj, e, parent);
			    VargraphMakeLink(CurrentVg, vg);
			    VargraphLink(vg, S);
			    return yes;

            case Unaries:   return RecurseVargraphLinkExpr(e->left, e, S);

	    case Binaries:  if (RecurseVargraphLinkExpr(e->left, e, S))
				return yes;
			    if (RecurseVargraphLinkExpr(e->right, e, S))
				return yes;
			    return no;

	    case oBITFIELD: if (RecurseVargraphLinkExpr(e->u.bitfield->addr, e, S))
				return yes;
			    if (RecurseVargraphLinkExpr(e->u.bitfield->operand, e, S))
				return yes;
			    return no;

            case oFUNC:     ef = e->u.func;
                            for (i=ef->arity-1; i >= 0; i--) {
                                if (RecurseVargraphLinkExpr(ef->param[i], e, S))
				    return yes;
			    }
                            return no;

            default:        assert(false);
			    return no;
        }
}


static void RecurseVargraphLinkStatement(Statement* S)
/* Recursively traverse this program graph, identifying all possible */
/* 'next' nodes for CurrentVg. */
{	ReturnStatement* S_return;
	SwitchStatement* S_switch;
	switchcase_type scase;
	ExprStatement* S_expr;
	IfStatement* S_if;

	LOOP:
	if (S->flag2)
	    return;	/* already done this one. */
	S->flag2 = yes;
	CurrentStatement = S;
	switch (S->st) {
	    case st_expr:
		    S_expr = (ExprStatement*)S;
		    if (RecurseVargraphLinkExpr(S_expr->e, NULL, S))
			return;
		    S = S->next;
		    goto LOOP;

	    case st_if:
		    S_if = (IfStatement*)S;
		    if (RecurseVargraphLinkExpr(S_if->Test, NULL, S))
			return;
		    RecurseVargraphLinkStatement(S_if->True);
		    RecurseVargraphLinkStatement(S_if->Fals);
		    break;

	    case st_switch:
		    S_switch = (SwitchStatement*)S;
		    if (RecurseVargraphLinkExpr(S_switch->Test, NULL, S))
			return;
		    for (scase=S_switch->Cases; scase; scase=scase->next)
			RecurseVargraphLinkStatement(scase->Body);
		    RecurseVargraphLinkStatement(S_switch->Default);
		    break;

	    case st_return:
		    S_return = (ReturnStatement*)S;
		    RecurseVargraphLinkExpr(S_return->e, NULL, S);
		    break;

	    case st_null:
		    S = S->next;
		    goto LOOP;

	    default:assert(false);
		    break;
	}
}


static void VargraphLink(vargraph_type vg, Statement* S)
/* Recurse from CurrentStatement down, identifying children for this vargraph. */
{	Statement* old_S;
	vargraph_type old_vg;

	old_vg = CurrentVg;
	old_S = CurrentStatement;
	CurrentVg = vg;
	CurrentStatement = S;
	PassedOurself = no;
	for (each_statement)
	    S->flag2 = no;
	RecurseVargraphLinkStatement(CurrentStatement);
	CurrentVg = old_vg;
	CurrentStatement = old_S;
}


static void VargraphExamine(Expr* e, Expr* parent)
/* We have found a oLocalVar expression.  Give it a vargraph_node */
/* if it doesn't already have one, and search for all children.   */
{	AutoNamedobj* obj;
	vargraph_type vg;

	assert(e->o == oLocalVar);
	if (e->right)
	    return;	/* Already has one. */
	obj = e->u.obj;
	vg = VargraphNew(obj, e, parent);
	e->right = (Expr*)vg;
	CurrentLocal = obj;
	VargraphLink(vg, CurrentStatement);
}


static void RecurseVargraphExpr(Expr* e, Expr* parent)
{       exprf_type ef;
        int i;

        switch (e->o) {
            case oUnknown:
            case oPUSHSPPLUS:
            case oConstant: return;
            case oLocalVar: VargraphExamine(e, parent);
			    return;

            case Unaries:   RecurseVargraphExpr(e->left, e);
			    return;

	    case Binaries:  RecurseVargraphExpr(e->left, e);
			    RecurseVargraphExpr(e->right, e);
			    return;

	    case oBITFIELD: RecurseVargraphExpr(e->u.bitfield->addr, e);
			    RecurseVargraphExpr(e->u.bitfield->operand, e);
			    return;

            case oFUNC:     ef = e->u.func;
                            for (i=ef->arity-1; i >= 0; i--)
                                RecurseVargraphExpr(ef->param[i], e);
                            return;

            default:        assert(false);
        }
}


static void RecurseVargraphStatement(Statement* S)
{	ReturnStatement* S_return;
	SwitchStatement* S_switch;
	switchcase_type scase;
	ExprStatement* S_expr;
	IfStatement* S_if;

	LOOP:
	if (S->flag1)
	    return;	/* already done this one. */
	S->flag1 = yes;
	CurrentStatement = S;
	switch (S->st) {
	    case st_expr:
		    S_expr = (ExprStatement*)S;
		    RecurseVargraphExpr(S_expr->e, NULL);
		    S = S->next;
		    goto LOOP;

	    case st_if:
                    S_if = (IfStatement*)S;
		    RecurseVargraphExpr(S_if->Test, NULL);
		    RecurseVargraphStatement(S_if->True);
		    RecurseVargraphStatement(S_if->Fals);
		    break;

	    case st_switch:
		    S_switch = (SwitchStatement*)S;
		    RecurseVargraphExpr(S_switch->Test, NULL);
		    for (scase=S_switch->Cases; scase; scase=scase->next)
			RecurseVargraphStatement(scase->Body);
		    RecurseVargraphStatement(S_switch->Default);
		    break;

	    case st_return:
		    S_return = (ReturnStatement*)S;
		    RecurseVargraphExpr(S_return->e, NULL);
		    break;

	    case st_null:
		    S = S->next;
		    goto LOOP;

	    default:assert(false);
		    break;
	}
}


static void VargraphGenerateAll(Statement* Prog)
/* Generate a vargraph for each local variable accessed in this statement.    */
/* Works by traversing the statement graph, and each time we find a oLocalVar */
/* without a vargraph_node, we separately create this variable's vargraph.    */
{	Statement* S;

	for (each_statement)
	    S->flag1 = no;
	RecurseVargraphStatement(Prog);
}


static bool VargraphAllNextAreAssignments(vargraph_type vg)
/* Is this a terminal node, or a node whose only antecedents assign to it? */
{	int i;

	for (i=0; i < vg->n_idx; i++)
	    if (vg->next[i]->parent->o != oASSIGN)
		return no;
	return yes;
}


static bool CheckReducedExpr(Expr* e)
/* This expression may have been reduced.  Return 'no' if it's now useless. */
{
        if (e->o == oCOMMA) {
            if (not CheckReducedExpr(e->left) and not CheckReducedExpr(e->right))
		return no;
	    else return yes;
        }
        else if (e->o < oASSIGN or e->o > oFUNC)
	    return no;
	else return yes;
}


static void CheckReducedStatement(Statement* S)
/* This statement has been reduced.  Check its top-level expression for usefulness. */
{	ExprStatement* S_expr;
	IfStatement* S_if;

	if (S->st == st_expr) {
	    S_expr = (ExprStatement*)S;
	    if (not CheckReducedExpr(S_expr->e))
		S->st = st_null;
	}
	else if (S->st == st_if) {
            S_if = (IfStatement*)S;
	    if (S_if->Test->o == oConstant) {
		S->st = st_null;
		if (S_if->Test->u.i)
		    S->next = S_if->True;
		else S->next = S_if->Fals;
	    }
	}
}




#define MIN_INT	    (1<<31)
#define MAX_INT	    (~(1<<31))


static bool VargraphValues(vargraph_type vg)
/* Do we have all this vg's parents' values?  If so, evaluate this vg's values. */
/* Return 'yes' if we manage to deduce something. */
{	int i,mi,ma,c;

	if (vg->flag1 == 2)
	    return no;
	c = 2;
	for (i=0; i < vg->p_idx; i++) {
	    if (vg->prev[i]->flag1 == 0)
		return no;
	    else if (vg->prev[i]->flag1 == 1)
		c = 1;
	}
	if (vg->flag1 == 0)
	    mi = MAX_INT, ma = MIN_INT;
	else mi = vg->minm, ma = vg->maxm;
	for (i=0; i < vg->p_idx; i++) {
	    if (mi > vg->prev[i]->minm)
		mi = vg->prev[i]->minm;
	    if (ma < vg->prev[i]->maxm)
		ma = vg->prev[i]->maxm;
	}
	if (vg->assignment) {
	    if (vg->parent->o == oAADD or vg->parent->o == oASUB) {
		if (vg->parent->right->o == oConstant) {
		    c = vg->parent->right->u.i;
		    if (vg->parent->o == oASUB)
			c = -c;
		    mi += c;
		    ma += c;
		}
		else mi = MIN_INT, ma = MAX_INT;
	    }
	    else mi = MIN_INT, ma = MAX_INT;
	}
	if (vg->flag1 == 0)
	    vg->flag1 = c;
	else if (vg->minm == mi and vg->maxm == ma)
	    return no;
	vg->minm = mi;
	vg->maxm = ma;
	assert(mi <= ma);
	return yes;
}


static void VargraphCombine(vargraph_type vg, int mi, int ma)
{
	if (vg->flag1 == 2)
	    return;
	vg->flag1 = 1;
	vg->minm = mi;
	vg->maxm = ma;
}


static bool VargraphCheckIfComparison(Expr* e, vargraph_type vg)
{
	return (IsComparison(e->o) and (e->left == vg->parent or e->right == vg->parent));
}


static void VargraphCheckIf(IfStatement* S_if, vargraph_type vg)
/* Does this expression tell us anything about children of 'vg'? */
{	int mit, mat, mif, maf;
	vargraph_type child;
	Expr* e;
	o_enum o;
	int i,c;

	/* We need the variable to be read: */
	if (vg->parent->o != oDER)
	    return;

	/* We need the 'if' to imply a pure comparison. */
	e = S_if->Test;
	if (VargraphCheckIfComparison(e, vg))
	    ;
	else if (e->o == oANND and VargraphCheckIfComparison(e, vg))
	    e = e->left;
	else if (e->o == oANND and VargraphCheckIfComparison(e, vg))
	    e = e->right;
	else return;

	/* We need to have a constant to compare with: */
	o = e->o;
	if (e->right->o == oConstant)
	    ;	/* go on */
	else if (e->left->o == oConstant) {
	    if (o == oLT) o = oGT;
	    else if (o == oLE) o = oGE;
	    if (o == oGT) o = oLT;
	    else if (o == oGE) o = oLE;
	}
	else return;

	/* What are the bounds if true?  If false? */
	mit = mif = MIN_INT;
	mat = maf = MAX_INT;
	c = e->right->u.i;
	if (o == oEQ)
	    mit = mat = c;
	else if (o == oNE)
	    mif = maf = c;
	else if (o == oGT)
	    mit = c+1, maf = c;
	else if (o == oGE)
	    mit = c, maf = c-1;
	else if (o == oLT)
	    mat = c-1, mif = c;
	else if (o == oLE)
	    mat = c, mif = c+1;

	/* Can we find a child down 'true'? Down 'false'? */
	for (i=0; i < vg->n_idx; i++) {
	    child = vg->next[i];
	    if (child->p_idx > 1)
		continue;
	    if (S_if->True == S_if->Fals)
		continue;
	    if (child->S == S_if->True)
		VargraphCombine(child, mit, mat);
	    else if (child->S == S_if->Fals)
		VargraphCombine(child, mif, maf);
	}
}


interface vargraph_type debug_vg;

static void VargraphAnalyse(AutoNamedobj *obj)
{	vargraph_type vg, tmproot, vgnext;
	bool something_reduced;

	/* Flip the linked list around: */
	tmproot = NULL;
	for (vg=obj->root; vg; vg=vgnext) {
	    vgnext = vg->next_in_list;
	    vg->next_in_list = tmproot;
	    tmproot = vg;
	}
	obj->root = tmproot;

	/* What can we learn from the graph? */
	for (vg=obj->root; vg; vg=vg->next_in_list) {
	    if (vg->p_idx == 0) {
		/* This is an initial node. */
		if (not vg->straight_assignment) {
		    ErrorParse(vg->parent->src, "Variable '%s' is undefined at this point.", obj->name);
		}
	    }
	    if (VargraphAllNextAreAssignments(vg) and vg->S->st != st_return) {
		/* This is a last-use. */
		if (vg->assignment) {
		    if (vg->straight_assignment) {
			vg->parent->o = oNOP1;
			vg->parent->left = vg->parent;
		    }
		    else {
			vg->parent->o = AssignopToOp(vg->parent->o);
			vg->parent->left = NewExpr('*', vg->parent->left, NULL);
			vg->parent->left->o = oDER;
			vg->parent->left->src = vg->parent->left->left->src;
			vg->parent->left->type = vg->parent->type;
		    }
		    CheckReducedStatement(vg->S);
		}
	    }
	}

	/* Can we deduce variable values? */
	if (TypeSize(obj->type) <= 4) {
	    for (vg=obj->root; vg; vg=vg->next_in_list) {
		vg->flag1 = no;
		if (vg->straight_assignment) {
		    vg->flag1 = 2;
		    if (vg->parent->right->o == oConstant)
			vg->minm = vg->maxm = vg->parent->right->u.i;
		    else vg->minm = MIN_INT, vg->maxm = MAX_INT;
		}
	    }
	    for (vg=obj->root; vg; vg=vg->next_in_list) {
		if (vg->S->st == st_if) {
		    IfStatement* S_if=(IfStatement*)vg->S;
		    VargraphCheckIf(S_if, vg);
		    if (S_if->Test->o == oLT)
			debug_vg = vg->next[0];
		}
	    }
	    do {
		something_reduced = no;
		for (vg=obj->root; vg; vg=vg->next_in_list) {
		    if (VargraphValues(vg))
			something_reduced = yes;
		}
	    } while (something_reduced);
	    for (vg=obj->root; vg; vg=vg->next_in_list) {
		if (not vg->flag1)
		    vg->minm = MIN_INT, vg->maxm = MAX_INT;
	    }
	}
}


static void VargraphAnalyseAll(Statement* S)
{	AutoNamedobj *obj;

	VargraphGenerateAll(S);
	for (obj=localvar_root; obj; obj=obj->nextlocal)
	    VargraphAnalyse(obj);
}


static void DumpVargraph(AutoNamedobj* obj)
{	vargraph_type vg;
	char buf[512];
	str s;
	int i,j;

	printf("Vargraph for %s:\n", obj->name);
	i = 0;
	for (vg=obj->root; vg; vg=vg->next_in_list)
	    vg->flag1 = ++i;
	for (vg=obj->root; vg; vg=vg->next_in_list) {
	    if (vg->parent->src == NULL)
		strcpy(buf, "(null)");
	    else memcpy(buf, vg->parent->src, 8);
	    buf[8] = '\0';
	    for (s=buf; *s; s++)
		if (*s == '\n')
		    *s = '\0';
	    printf("%d: [%s]  ", vg->flag1, buf);
	    if (vg->minm != MIN_INT or vg->maxm != MAX_INT)
		printf("(%d..%d)  ", vg->minm, vg->maxm);
	    if (vg->n_idx == 0)
		printf("Terminates.\n");
	    else {
		printf("Goes to ");
		for (j=0; j < vg->n_idx; j++)
		    printf("%d,", vg->next[j]->flag1);
		printf("\n");
	    }
	}
}


static void DumpVargraphs(void)
/* Print all vargraphs for this function. */
{	AutoNamedobj *obj;

	for (obj=localvar_root; obj; obj=obj->nextlocal)
	    DumpVargraph(obj);
}





/*========================= Common subexpressions: ======================*/

typedef struct comon_node {
	Expr* e;
	Expr* *E;
	uint E_idx;
	struct comon_node *next;
} *comon_type;

static comon_type *Comon;
static uint ComonSize;


static uint RecurseExprHash(Expr* e, int depth)
{	exprf_type ef;
	uint h,i;

	h = e->o;
	if (depth-- == 0)
	    return h;
	switch (e->o) {
	    case oConstant:
            case oPUSHSPPLUS:
	    case oLocalVar: h += (int)e->left;
			    return h;

            case Unaries:   h += RecurseExprHash(e->left, depth)*7;
			    return h;

	    case Binaries:  h = (RecurseExprHash(e->left, depth) * 13) ^ 
				RecurseExprHash(e->right, depth);
			    return h;

	    case oBITFIELD: h = (RecurseExprHash(e->u.bitfield->addr, depth) * 13) ^
				RecurseExprHash(e->u.bitfield->addr, depth);
			    return h;

	    case oPUSH:     h = RecurseExprHash(e->left, depth) * 13;
            		    if (e->right)
                            	h ^= RecurseExprHash(e->right, depth);
			    return h;

            case oFUNC:     ef = e->u.func;
                            for (i=ef->arity-1; i >= 0; i--)
                                h = (h << 1) ^ RecurseExprHash(ef->param[i], depth);
                            return h;

	    default:	    assert(false);
			    return h;
	}
}


static uint ExprHash(Expr* e)
{
	return RecurseExprHash(e, 3) % ComonSize;
}


static bool ExprEqual(Expr* a, Expr* b)
/* Are these two expressions identical? */
{	exprf_type efa, efb;
	uint i,sz;

	if (a->tp != b->tp)
	    return no;
	if (a->o != b->o)
	    return no;
	switch (a->o) {
	    case oLocalVar: return a->left == b->left;

            case oPUSHSPPLUS:
	    case oConstant: sz = TpSize(a->tp);
			    if (sz <= 4)
				return a->u.i == b->u.i;
			    else return a->u.f == b->u.f;

            case Unaries:   return ExprEqual(a->left, b->left);

	    case Binaries:  return ExprEqual(a->left, b->left)
				and ExprEqual(a->right, b->right);

	    case oBITFIELD: if (not ExprEqual(a->u.bitfield->addr, b->u.bitfield->addr))
				return no;
			    if (a->u.bitfield->operand == NULL and b->u.bitfield->operand == NULL)
				;   // Ok
			    else if (a->u.bitfield->operand == NULL or b->u.bitfield->operand == NULL)
				return no;
			    if (a->u.bitfield->bit_offset != b->u.bitfield->bit_offset)
				return no;
			    if (a->u.bitfield->bit_width != b->u.bitfield->bit_width)
				return no;
			    return yes;

            case oPUSH:	    if (not ExprEqual(a->left, b->left))
            			return no;
            		    if (a->right == b->right/*==NULL*/)
                            	return yes;
                            if (a->right == NULL or b->right == NULL)
                            	return no;
                            return ExprEqual(a->right, b->right);
                            
            case oFUNC:     efa = a->u.func;
			    efb = b->u.func;
			    if (efa->arity != efb->arity)
				return no;
                            for (i=efa->arity-1; i >= 0; i--)
                                if (not ExprEqual(efa->param[i], efb->param[i]))
				    return no;
                            return yes;

	    default:	    assert(false);
			    return yes;
	}
}


static void RecurseComon(Expr* e)
{       comon_type com;
        exprf_type ef;
	Expr* *E;
	uint i,h;

	/* Don't bother about constants and variables: */
	if (e->o == oConstant or e->o == oLocalVar
		or (e->o == oDER and (e->left->o == oConstant or e->left->o == oLocalVar))) {
	    e->idx = 0;
	    return;
	}

	/* Look for an entry for this equivalence class: */
	h = ExprHash(e);
	for (com=Comon[h]; com; com=com->next) {
	    if (ExprEqual(e, com->e)) {
		if (com->E_idx == 1) {
		    com->E = (Expr**)qmalloc(sizeof(Expr*) * 10);
		    com->E[0] = com->e;
		}
		else if (com->E_idx == 10) {
		    E = (Expr**)qmalloc(sizeof(Expr*) * 100);
		    memcpy(E, com->E, com->E_idx * sizeof(Expr*));
		    com->E = E;
		}
		else if (com->E_idx >= 99) {
		    ErrorParse(e->src, "There are more than 100 occurrences of this common subexpression.\n");
		}
		com->E[com->E_idx++] = e;
		goto FOUND;
	    }
	}
	com = (comon_type)qmalloc(sizeof(struct comon_node));
	com->e = e;
	com->E = &com->e;
	com->E_idx = 1;
	com->next = Comon[h];
	Comon[h] = com;
	FOUND:

	/* Now do the subexpressions too: */
        switch (e->o) {
            case Unaries:   RecurseComon(e->left);
			    return;

	    case Binaries:  RecurseComon(e->left);
			    RecurseComon(e->right);
			    return;

	    case oBITFIELD: RecurseComon(e->u.bitfield->addr);
			    RecurseComon(e->u.bitfield->operand);
			    return;

	    case oPUSH:     RecurseComon(e->left);
            		    if (e->right)
			    	RecurseComon(e->right);
			    return;

            case oFUNC:     ef = e->u.func;
                            for (i=ef->arity-1; i >= 0; i--)
                                RecurseComon(ef->param[i]);
                            return;

            default:        e->idx = 0;
			    return;
        }
}


static void ComonConstruct(void)
/* Construct a hash-table of all common subexpressions. */
{	ReturnStatement* S_return;
	SwitchStatement* S_switch;
	ExprStatement* S_expr;
	Statement* S;
	comon_type com;
	IfStatement* S_if;
	uint h,i,c;

	/* Initialise a blank hash-table: */
	ComonSize = 0;
	for (S=first_statement; S; S=S->next_registered)
	    ComonSize++;
	ComonSize = ComonSize*4;
	ComonSize -= ComonSize % 10;
	ComonSize += 7;
	Comon = (comon_type*)calloc(anon_heap, ComonSize, sizeof(comon_type));

	/* Iterate over all statements: */
	for (S=first_statement; S; S=S->next_registered) {
	    switch (S->st) {
		case st_expr:
			S_expr = (ExprStatement*)S;
			RecurseComon(S_expr->e);
			break;

		case st_if:
                        S_if = (IfStatement*)S;
			RecurseComon(S_if->Test);
			break;

		case st_switch:
                        S_switch = (SwitchStatement*)S;
			RecurseComon(S_switch->Test);
			break;

		case st_return:
                        S_return = (ReturnStatement*)S;
			if (S_return->e and S_return->e != (Expr*)0x1)
			    RecurseComon(S_return->e);
			break;
	    }
	}

	/* Allocate 'comon numbers' to each expression: */
	c = 1;
	for (h=0; h < ComonSize; h++) {
	    for (com=Comon[h]; com; com=com->next) {
		printf("Set %d:\n", c);
		for (i=0; i < com->E_idx; i++) {
		    com->E[i]->idx = c;
		    DumpExpr(com->e);
		}
		printf("\n");
		c++;
	    }
	}
	free(anon_heap, Comon);
}





/*========================= Jump Analysis: ======================*/

static void NextValidStatement(Statement **To_p)
/* We have a statement pointer.  It should point to the next */
/* valid statement, skipping null statements (and goto's).   */
/* At the same time, we examine this transition in case it   */
/* needs destructors (or a warning about skipping a constru- */
/* ctor). */
{       Statement* To=*To_p;
        int i=0;

        while (To->st == st_null) {
            To = To->next;
            if (++i >= 20) {
                ErrorParse(To->src, "Infinite loop");
                break;
            }
        }
        *To_p = To;
}


static str GetStatementSrc(Statement* S)
{       int i=0;

        do {
            if (S == NULL)
                return NULL;
            if (S->src)
                return S->src;
            if (++i >= 20)
                return NULL;
            if (S->st == st_null or S->st == st_expr)
                S = S->next;
            else return NULL;
        } forever;
}


static void AnalyseJumps(Statement* EntryStatement)
/* Cycle through all expressions in this statement and optimise them. */
{	SwitchStatement* S_switch;
	switchcase_type scase;
	IfStatement* S_if;
	Statement* S;

	/* Detect dead code: */
	for (each_statement)
	    S->flag1 = no;
	EntryStatement->flag1 = yes;
	for (each_statement) {
	    if (S->st == st_expr or S->st == st_null) {
		S->next->flag1 = yes;
	    }
	    else if (S->st == st_if) {
		S_if = (IfStatement*)S;
		S_if->True->flag1 = yes;
		S_if->Fals->flag1 = yes;
	    }
	    else if (S->st == st_switch) {
		S_switch = (SwitchStatement*)S;
		for (scase=S_switch->Cases; scase; scase=scase->next)
		    scase->Body->flag1 = yes;
                if (S_switch->Default)
                    S_switch->Default->flag1 = yes;
	    }
	}
        if (Error.err)
            return;
	for (each_statement) {
	    if (not S->flag1 and S->st != st_null) {
                if (S->src == NULL)
                    continue;
		/*s = GetStatementSrc(S);
                if (s == NULL or Error.err)
                    continue;*/
                else if (S->st == st_return and ((ReturnStatement*)S)->e == NULL)
                    continue;
                else {
                    ErrorParse(S->src, "Dead code");
                    return;
                }
	    }
	}

	/* Optimise jumps: */
	for (each_statement) {
	    if (S->st == st_expr or S->st == st_null) {
		NextValidStatement(&S->next);
	    }
	    else if (S->st == st_if) {
		S_if = (IfStatement*)S;
		NextValidStatement(&S_if->True);
		NextValidStatement(&S_if->Fals);
		if (S_if->True == S_if->Fals) {
		    S->next = S_if->True;
		    S->st = st_expr;
		    ((ExprStatement*)S)->e = S_if->Test;
		}
	    }
	    else if (S->st == st_switch) {
		S_switch = (SwitchStatement*)S;
		for (scase=S_switch->Cases; scase; scase=scase->next)
		    NextValidStatement(&scase->Body);
                if (S_switch->Default == NULL)
                    S_switch->Default = S->next;
		NextValidStatement(&S_switch->Default);
	    }
	}
}


static void OptimiseEntity(Statement* S)
{
	OptimiseExpressions();
	AnalyseJumps(S);
}





/*=================================================*/

static Statement* SA[512];
static int SA_idx;


static int SA_Find(Statement* S)
{       int i;

        for (i=0; i < SA_idx; i++)
            if (SA[i] == S)
                return i;
        return -1;
}


static void RecurseIpr(Statement* S)
{       SwitchStatement* S_switch;
        switchcase_type scase;
        IfStatement* S_if;

        LOOP:
        if (S == NULL)
            return;
        if (SA_Find(S) != -1)
            return;
        SA[SA_idx++] = S;
        if (S->st == st_if) {
            S_if = (IfStatement*)S;
            RecurseIpr(S_if->True);
            RecurseIpr(S_if->Fals);
        }
        else if (S->st == st_switch) {
            S_switch = (SwitchStatement*)S;
            for (scase=S_switch->Cases; scase; scase=scase->next)
                RecurseIpr(scase->Body);
            RecurseIpr(S_switch->Default);
        }
        else if (S->st == st_return)
            return;
        else {
            S = S->next;
            goto LOOP;
        }
}


interface str OpToString(char op)
{
        switch (op) {
            case oADD:          return "+";
            case oSUB:          return "-";
            case oMUL:          return "*";
            case oDIV:          return "/";
            case oMOD:          return "%";
            case oLT:           return "<";
            case oGT:           return ">";
            case oLE:           return "<=";
            case oGE:           return ">=";
            case oEQ:           return "==";
            case oNE:           return "!=";
            case oSHL:          return "<<";
            case oSHR:          return ">>";
            case oAND:          return "&";
            case oOR:           return "|";
            case oNOT:          return "~";
            case oANND:         return "&&";
            case oORR:          return "||";
            case oXOR:          return "^";
            case oNOTT:         return "!";
            case oNEG:          return "unry-";
            case oASSIGN:       return "=";
            case oAADDAFTER:    return "+=after";
            case oAADD:         return "+=";
            case oASUB:         return "-=";
            case oAMUL:         return "*=";
            case oADIV:         return "/=";
            case oAMOD:         return "%=";
            case oAOR:          return "|=";
            case oAAND:         return "&=";
            case oAXOR:         return "^=";
            case oDER:          return "**";
            case oCOMMA:        return "; then ";
            case oTERNARY:      return "?";
            case oCONVERT:      return "convert";
            case oFUNC:         return "Func";
            case oConstant:     return "Constant";
            case oLocalVar:     return "LocalVar";
            case oPUSHSPPLUS:	return "SUBSPPUSHSP";
            case oSIN:          return "sin";
            case oCOS:          return "cos";
            case oTAN:          return "tan";
            case oATAN:         return "atan";
            case oSQRT:         return "sqrt";
	    case oNOP1:		return "NOP";
	    case oMEMIA:	return "MEMIA";
	    case oPUSH:		return "PUSH";
	    case oBITFIELD:	return "BITFIELD";
            default:		return "?op?";
        }
}


interface void DumpExpr(Expr* e, int depth)
{       AutoNamedobj *obj;
	struct memia_node {
	    Expr *e1, *e2;	/* It is 'e2' that is scaled. */
	    uint K;
	    void* Kbase;
	    uint scaler;
	} *memia;
        int i;

	printf("\n%*s", depth*4, "");
	switch (e->o) {
	    case oConstant:
		    if (e->storereg)
			printf("<%s>", RegisterToString(e->storereg));
		    if (e->tp == tp_int or e->tp == tp_short or e->tp == tp_char)
			printf("%d", e->u.i);
		    else if (e->tp == tp_long)
			printf("%ld", e->u.l);
		    else if (e->tp == tp_float or e->tp == tp_double)
			printf("%f", e->u.f);
		    else if (e->tp == tp_pointer)
			printf("%x", e->u.p);
		    else if (e->tp == tp_uint or e->tp == tp_uchar or e->tp == tp_ushort)
			printf("%u", e->u.i);
		    else printf("???Constant");
		    break;

	    case oLocalVar:
		    obj = e->u.obj;
		    printf("%s", obj->name);
		    break;

            case oPUSHSPPLUS:
            	    printf("SUBSPPUSHSP %d", e->u.i);
                    break;

	    case oUnknown:
		    printf("Unknown expr (e->opr=%d)", e->opr);
		    break;

	    case oNOP1:
		    printf(" -nop-");
		    DumpExpr(e->left, depth+1);
		    break;

	    case oFUNC:
		    if (e->storereg)
			printf("<%s>", RegisterToString(e->storereg));
		    if (e->u.func->func and e->u.func->func->o != oConstant) {
			DumpExpr(e->u.func->func, depth+1);
			printf(" call(");
		    }
		    else printf("f(");
		    for (i=0; i < e->u.func->arity; i++) {
			if (i != 0)
			    printf(",");
			DumpExpr(e->u.func->param[i], depth+1);
		    }
		    printf("\n%*s", depth*4, "");
		    printf(") ");
		    break;

	    case oTERNARY:
		    printf("(");
		    if (e->storereg)
			printf("<%s>", RegisterToString(e->storereg));
		    DumpExpr(e->left, depth+1);
		    printf(" ? ");
		    DumpExpr(e->right->left, depth+1);
		    printf(" : ");
		    DumpExpr(e->right->right, depth+1);
		    printf(")");
		    break;

	    case oBITFIELD:
		    if (e->storereg)
			printf("<%s>", RegisterToString(e->storereg));
		    printf("bitfield[\"%s\" addr=", OpToString(e->u.bitfield->o));
		    DumpExpr(e->u.bitfield->addr, depth+1);
		    if (e->u.bitfield->operand) {
			printf(" operand=");
			DumpExpr(e->u.bitfield->operand, depth+1);
		    }
		    printf(" off=%d wid=%d]", e->u.bitfield->bit_offset, e->u.bitfield->bit_width);
		    break;

            case oPUSH:
            	    printf("Push ");
            	    DumpExpr(e->left);
                    if (e->right) {
                    	printf(" and ");
                        DumpExpr(e->right);
                    }
                    break;

	    case oMEMIA:
		    memia = (struct memia_node*)e->u.p;
		    if (e->storereg)
			printf("<%s> ", RegisterToString(e->storereg));
		    printf("[MEMIA:  scale=%d  K=0x%x", 1<<memia->scaler, memia->K);
		    if (memia->e1) {
			extern Expr frame_register;
			printf(" e1=");
			if (memia->e1 == &frame_register)
			    printf("BP");
			else DumpExpr(memia->e1, depth+1);
		    }
		    if (memia->e2) {
			printf(" e2=");
			DumpExpr(memia->e2, depth+1);
		    }
		    printf("\n%*s]\n", depth*4, "");
		    break;

	    default:
		    printf("(");
		    if (e->right_first) {
			if (e->right)
			    DumpExpr(e->right, depth+1);
			if (e->storereg)
			    printf(" <%s>", RegisterToString(e->storereg));
			else printf(" ");
			printf("rev%s ", OpToString(e->o));
			if (e->left)
			    DumpExpr(e->left, depth+1);
		    }
		    else {
			if (e->left)
			    DumpExpr(e->left, depth+1);
			if (e->storereg)
			    printf(" <%s>", RegisterToString(e->storereg));
			else printf(" ");
			printf("%s ", OpToString(e->o));
			if (e->right)
			    DumpExpr(e->right, depth+1);
		    }
		    printf(")");
		    break;

        }
}


interface void DumpIpr(Statement* S)
{       SwitchStatement* S_switch;
        ReturnStatement* S_return;
        switchcase_type scase;
        ExprStatement* S_expr;
        IfStatement* S_if;
        int i,j;

        SA_idx = 0;
        RecurseIpr(S);
        for (i=0; i < SA_idx; i++) {
            S = SA[i];
            printf("%d: ", i);
            switch (S->st) {
                case st_if:     printf("if ");
                                S_if = (IfStatement*)S;
                                DumpExpr(S_if->Test);
                                printf(" then %d else %d\n",
                                        SA_Find(S_if->True), SA_Find(S_if->Fals));
                                continue;

                case st_expr:   S_expr = (ExprStatement*)S;
                                DumpExpr(S_expr->e);
                                break;

                case st_switch: S_switch = (SwitchStatement*)S;
                                printf("switch ");
                                for (scase=S_switch->Cases; scase; scase=scase->next)
                                    printf("%d-->%d  ", scase->tag, SA_Find(scase->Body));
                                printf("Otherwise %d\n", SA_Find(S_switch->Default));
                                continue;

                case st_return: S_return = (ReturnStatement*)S;
                                printf("return");
                                if (S_return->e and S_return->e != (Expr*)0x1) {
                                    printf(" ");
                                    DumpExpr(S_return->e);
                                }
                                break;

                case st_null:   break;

                /* Unallowables: */
                default:        printf("Unknown statement: %d", S->st);
                                break;
            }
            if (S->next == NULL)
                printf("; --> NULL\n");
            else {
                j = SA_Find(S->next);
                if (j == i + 1)
                    printf("\n");
                else if (S->st == st_return)
                    printf(";\n");
                else printf("; goto %d\n", j);
            }
        }
}


interface funcblock_type IprCompile(Statement* S, void *preferred_location)
/* Generate Pentium machine-code from this intermediate program representation. */
{       funcblock_type funcblock;

        if (S == NULL)
            return NULL;
        OptimiseEntity(S);
	if (Error.err)
	    return NULL;
//	VargraphAnalyseAll(S);
//	ComonConstruct();
        if (not PredefinitionPhase) {
	    //DumpIpr(S);
	    //DumpVargraphs();
	    //return NULL;
	}
        funcblock = GenerateCode(S, preferred_location);
        InitLocalVars();
        return funcblock;
}



