#include <string.h>
#include <math.h>
#include "barbados.h"
#include "bitmap.h"

#pragma warning ( disable : 4244 )
#pragma warning ( disable : 4761 )


typedef short int2;




typedef struct edge_node {
	int col;
	struct vertex_node *v;
} *edge_type;


typedef struct vertex_node {
	void* id;
	str name;
	struct vertex_node *next;
	edge_type arrow;
	int a_idx;
	int col;
	int2 x, y;
	struct vertex_node *next2d;
} *vertex_type;


#define MAX_HASH    107

typedef struct graph_node {
	vertex_type V[MAX_HASH];
} *graph_type;


static graph_type G;
static str undef_name = "<>";


interface graph_type GraphNew(void)
{       graph_type G;
	
	G = new graph_node;
	clearS(*G);
	return G;
}


static vertex_type FindId(void* id, bool create)
/* Find or create a vertex corresponding to this id. */
{       unsigned int h;
	vertex_type v;

	/* Get the hash: */
	h = (uint)id;
	h += 7*(h&0xff);
	h %= MAX_HASH;

	/* Is there an existing one? */
	for (v=G->V[h]; v; v=v->next)
	    if (v->id == id)
		return v;

	/* Don't necessarily create it: */
	if (not create)
	    return NULL;
	    
	/* Create one: */
	v = new vertex_node;
	clearS(*v);
	v->id = id;
	v->next = G->V[h];
	G->V[h] = v;
	v->name = undef_name;
	return v;
}

	
interface bool GraphAddVertex(graph_type Gv, void* id, str name, int col)
/* Create a vertex with this id and this name. Overwrite any */
/* previous vertex with the same id.   Returns yes if new.   */
{       vertex_type v;
	str s;
	
	G = Gv;
	assert(name != NULL);
	if (strlen(name) < 2) {
	    s = " ? ";
	    s[1] = name[0];
	    name = s;
	}
	v = FindId(id, yes);
	v->col = col;
	if (v->name and v->name != undef_name) {
	    if (strcmp(v->name, name) == 0)
		return no;
	    free(anon_heap, v->name);
	    v->name = strdup(anon_heap, name);
	    return no;
	}
	else {
	    v->name = strdup(anon_heap, name);
	    return yes;
	}   
}


interface void GraphAddEdge(graph_type Gv, void* id_from, void* id_to, int col)
/* Add an arrow from 'id_from' to 'id_to'. */
{       vertex_type v1, v2;

	G = Gv;
	if (col <= 0)
	    col = 15;
	v1 = FindId(id_from, yes);
	v2 = FindId(id_to, yes);
	v1->arrow = (edge_type)realloc(anon_heap, v1->arrow, (v1->a_idx+1) * sizeof(struct edge_node));
	v1->arrow[v1->a_idx].v = v2;
	v1->arrow[v1->a_idx++].col = col > 0 ? col : -col;
	v2->arrow = (edge_type)realloc(anon_heap, v2->arrow, (v2->a_idx+1) * sizeof(struct edge_node));
	v2->arrow[v2->a_idx].v = v1;
	v2->arrow[v2->a_idx++].col = -col;
}


interface void GraphFree(graph_type Gv)
{       vertex_type v;
	int h;

	for (h=0; h < MAX_HASH; h++) {
	    while (G->V[h]) {
		v = G->V[h];
		G->V[h] = v->next;
		free(anon_heap, v->name);
		free(anon_heap, v->arrow);
		free(anon_heap, v);
	    }
	}
	free(anon_heap, G);
	G = NULL;
}




/*--------------------- Rendering: ----------------------*/

#define x0      my_x0
#define y0      my_y0
#define x1      my_x1
#define y1      my_y1
#define UNDEFD      -32768
#define SCALEY  30.0
#define SCALEX  (SCALEY*0.866*1.6)

typedef struct {
	int x,y;
} Place;

static vertex_type *Plane;
static int x0,y0,x1,y1;
static int NoVertices;
static Place C1, C2, P1, P2, Shift;



static vertex_type Find2D(int x, int y, vertex_type vnew)
/* Look for a vertex existing at this location. */
/* If the location is empty, and v is non-NULL, */
/* place 'v' at this location. */
{       vertex_type v;
	int h;

	/* Get the hash: */
	h = 11*x + 7*y;
	h %= MAX_HASH;
	if (h < 0)
	    h += MAX_HASH;

	/* Is there an existing one? */
	for (v=Plane[h]; v; v=v->next2d)
	    if (v->x == x and v->y == y)
		return v;

	/* Don't necessarily create it: */
	if (vnew == NULL)
	    return NULL;
	    
	/* Put it there: */
	assert(vnew->x == UNDEFD);
	vnew->x = x;
	vnew->y = y;
	vnew->next2d = Plane[h];
	assert(vnew->name != NULL);
	Plane[h] = vnew;
	if (x < x0)
	    x0 = x;
	else if (x > x1)
	    x1 = x;
	if (y < y0)
	    y0 = y;
	else if (y > y1)
	    y1 = y;
	return vnew;
}



static double VerScore(vertex_type v)
/* How 'important' is this vertex? */
{       vertex_type v2;
	double score;
	int i;

	score = v->a_idx;
	for (i=0; i < v->a_idx; i++) {
	    v2 = v->arrow[i].v;
	    if (v2->x != UNDEFD)
		score += 5;
	}
	return score;
}


static vertex_type NextMostImportant(void)
/* Returns the most important unplaced vertex. */
{       struct {
	    int score;
	    vertex_type v;
	} best;
	vertex_type v;
	int score;
	int h;

	best.score = -9999;
	best.v = NULL;
	for (h=0; h < MAX_HASH; h++) {
	    for (v=G->V[h]; v; v=v->next) {
		if (v->x != UNDEFD)
		    continue;
		if (v->name == NULL)
		    continue;
		score = VerScore(v);
		if (score > best.score)
		    best.score = score, best.v = v;
	    }
	}
	return best.v;
}


static double Sgn(int s, double val)
/* Return +val, 0 or -val, depending on 's'. */
{
	if (s == 0)
	    return 0;
	else if (s < 0)
	    return -val;
	else return val;
}


static int PosScore(int x, int y, vertex_type v)
/* How suitable is 'v' for this position? */
{       bool width_bigger, height_bigger;
	double d, density, score, dx, dy;
	vertex_type v2; 
	int i;
	
	/* Preferably don't increase the size. */
	width_bigger = (x < x0 or x > x1);
	height_bigger = (y < y0 or y > y1);
	score = 0;
	if (width_bigger)
	    score -= 2;
	if (height_bigger)
	    score -= 2;

	/* Don't produce long, thin ones or very sparse ones. */
	if (width_bigger or height_bigger) {
	    if (x < x0) dx = x1 - x;
	    else if (x > x1) dx = x - x0;
	    else dx = x1 - x0;
	    if (y < y0) dy = y1 - y;
	    else if (y > y1) dy = y - y0;
	    else dy = y1 - y0;
	    if (dy > dx)
		score -= (dy - dx) * 10;
	    dx++, dy++;
	    density = NoVertices / (dx*dy);
	    if (density < 0.25)
		score -= 5 / density;
	}

	/* Look at adjoining vertices: */
	for (i=0; i < v->a_idx; i++) {
	    v2 = v->arrow[i].v;
	    if (v2->x == UNDEFD)
		continue;
	    dx = (x - v2->x) * 0.866;
	    dy = (y - v2->y);
	    
	    // Minimise distances:
	    d = sqrt(dx*dx + dy*dy);
	    score += 20 - 5*d;      

	    // Prefer certain directions:
	    score += Sgn(-v->arrow[i].col, Sgn((int)dx, 1.5) + Sgn((int)dy, 1.5));
	    if (dy == 0 and fabs(dx) < 2)
		score += 3;
	}
	return score * 10;
}


static void PlaceVertex(vertex_type v)
/* Put this vertex somewhere on the plane. */
{       struct {
	    int score;
	    int x,y;
	} best;
	int x,y,score;

	best.score = -9999;

	/* Iterate through all possible coordinates. */
	for (y=y0-1; y <= y1+1; y++) {
	    x = x0 - 2;
	    if ((y&1) != (x&1))
		x++;
	    for (; x <= x1+2; x += 2) {
		if (Find2D(x,y,NULL))       
		    continue;           // There's already something there
		score = PosScore(x,y,v);
		if (score > best.score)
		    best.score = score, best.x = x, best.y = y;
	    }
	}

	/* Place it. */
	assert(best.score > -9999);
	Find2D(best.x, best.y, v);
}


static void GraphToPlane(void)
/* Render this graph in 2 dimensions. */
{       vertex_type v;
	int h;

	/* Create a blank plane: */
	NoVertices = 0;
	for (h=0; h < MAX_HASH; h++)
	    for (v=G->V[h]; v; v=v->next)
		v->x = v->y = UNDEFD, NoVertices++;
	if (NoVertices == 0)
	    return;
	x0 = x1 = 0;
	y0 = y1 = 0;
	
	/* Place each vertex on the plane: */
	v = NextMostImportant();
	Find2D(0,0, v);
	while ((v=NextMostImportant()) != NULL)
	    PlaceVertex(v);
}


static Place VertexToPlace(vertex_type v)
/* Map this vertex to a bitmap x,y value. */
{       Place P;

	P.x = (v->x - x0) * SCALEX + Shift.x;
	P.y = (v->y - y0) * SCALEY + Shift.y;
	return P;
}


static vertex_type IsLineClear(Place *A, Place *B)
/* Is the line from A to B clear?  If yes, return NULL. */
/* If not, return the vertex that's in the way.         */
{       vertex_type v;
	int x,y,d;
	float f;
	
	if (abs(A->x - B->x) > abs(A->y - B->y)) {
	    /* Do the x's: */
	    d = B->x > A->x ? 1 : -1;
	    for (x=A->x+d; x != B->x; x += d) {
		f = A->y + (float)(B->y-A->y)*(x-A->x)/(B->x-A->x);
		if (f != (int)f)
		    continue;
		y = f;
		v = Find2D(x,y,NULL);
		if (v)
		    return v;
	    }
	}
	else {
	    /* Do the y's: */
	    d = B->y > A->y ? 1 : -1;
	    for (y=A->y+d; y != B->y; y += d) {
		f = A->x + (float)(B->x-A->x)*(y-A->y)/(B->y-A->y);
		if (f != (int)f)
		    continue;
		x = f;
		v = Find2D(x,y,NULL);
		if (v)
		    return v;
	    }
	}
	return NULL;
}


static int LineDiversion(Place *A, Place *B)
/* In order to draw a line from A to B, we may have to divert the */
/* line to avoid vertices.  This function tells us how much to    */
/* divert by. */
{       vertex_type v;
	
	v = IsLineClear(A,B);
	if (v == NULL)
	    return 0;
	return 15;
#if 0
	int d, dx, dy, sx, sy;
	double t;

	dx = abs(B->x - A->x);
	dy = abs(B->y - A->y);
	BM_BubbleSize(bm, v->name, &sx, &sy);
	t = (double)dx / (dx + dy);
	d = (int)(t*sx + (1-t)*sy);
	assert(d > 0);
	return d;
#endif
}


static void DrawLine(bitmap_type bm, vertex_type a, Place A, vertex_type b, Place B, int colour)
/* Draw a line, straight or diverted, from A to B, in line 'colour'. */
{       Place Pa, Pb;
	int d;

	if (a == b) {
	    /* Loop: */
	    BM_BubbleIntersect(bm, a->name, 
			    C1.x, C1.y, C1.x-40, C1.y-40,
			    (int*)&P1);
	    BM_BubbleIntersect(bm, b->name, 
			    C2.x, C2.y, C1.x-40, C1.y+40,
			    (int*)&P2);
	    BM_ArrowArc(bm, P1.x, P1.y, P2.x, P2.y, -20, colour);
	    return;
	}

	BM_BubbleIntersect(bm, a->name, 
			    C1.x, C1.y, C2.x, C2.y,
			    (int*)&P1);
	BM_BubbleIntersect(bm, b->name, 
			    C2.x, C2.y, C1.x, C1.y,
			    (int*)&P2);
	Pa.x = a->x;
	Pa.y = a->y;
	Pb.x = b->x;
	Pb.y = b->y;
	d = LineDiversion(&Pa, &Pb);
	if (d == 0) {
	    /* Straight: */
	    BM_Arrow(bm, P1.x, P1.y, P2.x, P2.y, colour);
	    return;
	}
	else {
	    /* Curved: */
	    BM_ArrowArc(bm, P1.x, P1.y, P2.x, P2.y, d, colour);
	    return;
	}
}
				    

static int PaletteMap[16] = {
	    0x000000,
	    0x880000,
	    0x008800,
	    0x888800,
	    0x000088,
	    0x880088,
	    0x008888,
	    0x888888,
	    0x0060FF,
	    0xFF0000,
	    0x00FF00,
	    0xFFFF00,
	    0x0000FF,
	    0xFF00FF,
	    0x00FFFF,
	    0xFFFFFF
};


static bitmap_type PlaneToBitmap(void)
/* Render this plane on a bitmap. */
{       int width, height, h, i, left, right;
	int fore_rgb, peri_rgb, col;
	vertex_type v, v2;
	bitmap_type bm;
	Place S;

	/* What are the dimensions of the bitmap? */
	left = right = 0;
	for (i=y0; i <= y1; i++) {
	    v = Find2D(x0,i,NULL);
	    if (v) {
		BM_BubbleSize(NULL, v->name, &S.x, &S.y);
		left = max(left, S.x / 2);
	    }
	    v = Find2D(x1,i,NULL);
	    if (v) {
		BM_BubbleSize(NULL, v->name, &S.x, &S.y);
		right = max(right, S.x / 2);
	    }
	}
	Shift.x = left;
	width = (x1 - x0) * SCALEX + left + right;
	height = (y1 - y0 + 1) * SCALEY;
	Shift.y = 12;
	    

	/* Create the bitmap: */
	bm = BM_New(width, height);


	/* Print the vertices on it: */
	for (h=0; h < MAX_HASH; h++) {
	    for (v=G->V[h]; v; v=v->next) {
		if (v->x == UNDEFD)
		    continue;
		C1 = VertexToPlace(v);
		col = v->col;
		fore_rgb = PaletteMap[col&15]|0x02000000;
		if (col < 16) 
		    peri_rgb = PaletteMap[col]|0x02000000;
		else peri_rgb = PaletteMap[col>>4]|0x02000000;
		BM_Bubble(bm, v->name, C1.x, C1.y, 
			0x00ffff, fore_rgb, peri_rgb);
		for (i=0; i < v->a_idx; i++) {
		    if (v->arrow[i].col < 0)
			continue;           // It's a back-pointer. 
		    v2 = v->arrow[i].v;
		    if (v2->x == UNDEFD)
			continue;           // It's undefined. 
		    C2 = VertexToPlace(v2);
		    DrawLine(bm, v, C1, v2, C2, PaletteMap[v->arrow[i].col]); 
		}
	    }
	}
	return bm;
}


interface bitmap_type GraphToBitmap(graph_type Gv)
{	bitmap_type bm;

	G = Gv;
	Plane = (vertex_type*)calloc(anon_heap, MAX_HASH, sizeof(vertex_type));
	GraphToPlane();
	bm = PlaneToBitmap();
	free(anon_heap, Plane);
	return bm;
}

