/* container_id.cpp
        This module takes care of the directory structure of the Win OS files
        associated to each container.
*/



#include <string.h>
#include <stdio.h>
#include <windows.h>
#ifdef __BORLANDCPP__
#include <dir.h>
#else
#include <direct.h>
#endif
#include <math.h>
#include "barbados.h"
#include "modules.h"
#include "memory.h"

#define DEBUG(S)
#define ASSERT(C,S)


#ifndef MAXLEN
#define MAXLEN 1024
#endif


// Identifier of the Top container
  const container_id ROOT                  = 1;
  const container_id CORRUPTLEVEL          = 100;

// Constant literals
  const str PERSISTENT_ROOT  = "C:\\PS";
  const str DATA_FILE        = "\\OWNERSHIP.DAT";
#define CONTAINER_EXT       ".CTR"
  const str FILESURETOFAIL   = "$&$";
  const str LOSTDIR          = "C:\\PS\\lost+found";


/////////////////////////////////////

/* Internal array needed when DiskListofChildren() is call. It
will provide the container_id's from the files in the directory
*/
    static container_id *A     = NULL;
    static int     A_idx = 0;

/* This variables are globals for this translation unit. When
the file OWNERSHIP.DAT is mapped in memory, the pointer with
descriptive name points to the initials of the table, while
SM_len holds the number of container_ids present.
    The cell number into this table will become the container
id, while its contents will be the parent of the container.
*/
    static container_id *SM = NULL;
    static int SM_len;

/* This variables are needed in order to manage the file mapping
Win32 API, creating the handle for the file to be mapped.
*/
    static HANDLE IdxFile = NULL;
    static HANDLE Mapping = NULL;

    static const int MAXTRIESTOFINDFREE  = 50;
    static const unsigned int MAXWAIT    = 1000L;// One Second, time to
                                                 // wait for mutexes.

    static HANDLE hmutex;                        // Handle of mutex
    static DWORD  waitformutex;                  // Used in order to gain
                                                 // access to mutex.


//--------------------------------------------------------------------Common

static bool LockIt(void);
static void ReleaseIt(void);
static bool HaveIt(void);

//////////////////////////////////////////////////// InitMyMutex()
inline
bool InitMyMutex(void)
{
    hmutex = CreateMutex(NULL,      // We do not need the handle
                                    // to be inherited by child
                                    // processes.
                        false,      // We do not need to claim ownership now
                        "limitedSM" // Name of the mutex object
    );

    if (hmutex==NULL)
	 return false;
    else return true;
}

//////////////////////////////////////////////////// ObtainMutex()
inline
bool ObtainMutex(void)
{
    waitformutex = WaitForSingleObject(hmutex,MAXWAIT);
    
    if (waitformutex==WAIT_TIMEOUT)
         return false;
    else return true;
}

//////////////////////////////////////////////////// FreeMutex()
inline
void FreeMutex(void)
{
    ReleaseMutex(hmutex);
}

//////////////////////////////////////////////////// MandatorilyRemoveDirectory()
static
void MandatorilyRemoveDirectory(str dir)
{
    char buf[MAXLEN];
    WIN32_FIND_DATA x;

    strcpy(buf, dir);
    strcat(buf, "\\*");

    if (FindFirstFile(buf,&x)!=INVALID_HANDLE_VALUE)
        do {
            DeleteFile(x.cFileName);
        } while (FindNextFile(buf, &x));


    RemoveDirectory(dir);
}

//////////////////////////////////////////////////// CreateDirectory()

inline
bool CreateNewDirectory(const str newparentname)
// Exists the new directory ? Create it if it doesn't
{
      bool toret            = true;
      bool directory_exists = true;
      container_id newparent      = PathToCid(newparentname);

      if (newparent!=ROOT)
      {
        // Does the directory exist ?
        {
            WIN32_FIND_DATA x;

	    DEBUG("Testing if directory (below) already exists");
            DEBUG((str)newparentname);
            if (FindFirstFile(newparentname,&x)==INVALID_HANDLE_VALUE)
            {
                directory_exists = false;
                DEBUG("directory does not exist.");
                DEBUG((str)newparentname);
            }
            else {DEBUG("Directory already exists");
                  DEBUG((str)newparentname);}
        }

        if (!directory_exists)
         if (!CreateDirectory(newparentname,NULL))  // Create directory WIN32
         {
           b_errno = E_CREATE_WINDIR;
           DEBUG("Error while creating Windows directory");
           toret = false;
         }
         else {DEBUG("Directory created:"); DEBUG((str)newparentname); }
        else DEBUG("Directory creation skipped because it already exists");
      } else DEBUG("Create Directory skipped because parent is ROOT");

    return toret;
}

//////////////////////////////////////////////////// PathToCid()
container_id PathToCid(str name)
// Extract the cid from the filename
{
    if (streq(name,PERSISTENT_ROOT))
        // It is the ROOT !
        return 1;
    else
    {
	container_id cid   = 0;
	str t, s = name;
	while ((t=strchr(s,'.')) != NULL)
	    s = t + 1;
	while (s > name and isdigit(s[-1]))
	    s--;
	cid = 0;
        sscanf(s,"%d",&cid);
	return cid;
    }
}


///////////////////////////////////////////////// CleanSM()
inline
void CleanSM(container_id pos)
{
      SM =
        (container_id*)MapViewOfFile(Mapping,
                               FILE_MAP_READ|FILE_MAP_WRITE, 0, 0, 0);

      memset(SM+(pos*sizeof(container_id)),0,SM_len*sizeof(container_id)-(pos*sizeof(container_id)));

      UnmapViewOfFile(SM);
}

//////////////////////////////////////////////////// ValidContainerId()
inline
bool ValidContainerId(container_id place)
{
     return (place  < SM_len
        and  place  > 0);
}

//////////////////////////////////////////////////// Initialised()
inline
bool Initialised(void)
{
    return (SM != NULL);
}

//////////////////////////////////////////////////// ReturnNextFree()
static
container_id ReturnNextFree(void)
{
    container_id ret = 0,aux,prev_len;

    if (!HaveIt())
        return 0;

    if (!ObtainMutex())
    {
        b_errno = E_MUTEX_BUSY;
        return 0;
    }
    else b_errno = E_NOERROR;

    for (unsigned int n=0;n<MAXTRIESTOFINDFREE;n++)
    {
        aux = (rand()%(SM_len-1)) + 1;
        if (   ValidContainerId(aux)
            && SM[aux] == 0
            && aux != ROOT)
        {
            ret = aux;
            break;
        }
    }


    if (ret!=0)
        return ret;
    else
    {
        unsigned long int size;

	// Extend the file
        LockIt();
        UnmapViewOfFile(SM);

        // Extend the size of the file
        prev_len = SM_len;
        size    = (SM_len * sizeof(container_id)) + 4096;
        SM_len  = size / sizeof(container_id);
        Mapping = CreateFileMapping(IdxFile,
                                    NULL, PAGE_READWRITE, 
                                    0, size, NULL);

        CleanSM(prev_len+1);
        
        // Return the situation of the mapping to the previous state
        ReleaseIt();
    }

    FreeMutex();

    return prev_len + 1; // First at the new file
}

//--------------------------------------------------------------------RetParent()

 container_id RetParent(container_id cid)
{
    container_id toret = 0;
    if (cid == -1 or cid == 0)
        return 0;

    if (HaveIt())
    {
      if (ObtainMutex())
      {
        toret   = SM[cid];
        b_errno = E_NOERROR;
      } else b_errno = E_MUTEX_BUSY;

      FreeMutex();
    }

    return toret;
}


 container_id  MaxId()
{
    if (HaveIt())
         return SM_len;
    else return 0;
}



//--------------------------------------------------------------------HaveIt()

  static bool HaveIt(void)
  // Open SM in read mode if it's not already open.
  // Initialises the system container id if it is not.
  { int size;

    // Are we already initialised ?
    if (SM)
	return true;

    // Prepare mutex control
   if (!InitMyMutex())
    {
        b_errno = E_COULDNT_INIT_MUTEX;
        return false;
    }

   if (!ObtainMutex())
    {
        b_errno = E_MUTEX_BUSY;
        return false;
    }

    // Open the OWNERSHIP.DAT file:
    if (IdxFile == NULL) {
	char path[MAXLEN];
	strcpy(path, PERSISTENT_ROOT);
	strcat(path, DATA_FILE);
        IdxFile =
          CreateFile(path,
            GENERIC_READ | GENERIC_WRITE,
            FILE_SHARE_READ | FILE_SHARE_WRITE,
            NULL,
            OPEN_EXISTING,
            FILE_ATTRIBUTE_NORMAL,
            NULL);

        if (IdxFile == NULL
         or IdxFile == INVALID_HANDLE_VALUE)
        {   int result = GetLastError();

          if (result == ERROR_PATH_NOT_FOUND or result == ERROR_FILE_NOT_FOUND) {
            _mkdir(PERSISTENT_ROOT);
            IdxFile =
             CreateFile(path,
		GENERIC_READ | GENERIC_WRITE,
                FILE_SHARE_READ | FILE_SHARE_WRITE,
                NULL,
                OPEN_ALWAYS,
                FILE_ATTRIBUTE_NORMAL,
                NULL);
          }
          if (IdxFile == NULL or IdxFile == INVALID_HANDLE_VALUE) {
            FreeMutex();
	    b_errno = E_NO_OWNERSHIP_FILE;
            return false;
          }
        }
    }

    // Map the file into memory:
    if (Mapping == NULL) {
        bool needcleaning = false;

        size = GetFileSize(IdxFile, NULL);

        if (size == 0) {
            size = 4096;
            needcleaning = true;
        }

        SM_len = size / sizeof(container_id);

        Mapping = CreateFileMapping(IdxFile, 
                                        NULL, PAGE_READWRITE, 
                                    0, size, NULL);

        if (Mapping <= 0)
        {
	    //int result = GetLastError();
            b_errno = E_NO_OWNERSHIP_FILE;
	    return false;
	}

        if (needcleaning) 
            CleanSM(0);
    }

    // Map a view of the file in read-only mode:
    SM =
	(container_id*)MapViewOfFile(Mapping, FILE_MAP_READ, 0, 0, 0);

    FreeMutex();    // Free the mutex.

    if (!Initialised()) {
        b_errno = E_NO_OWNERSHIP_FILE;
        return false;
    }

    b_errno = E_NOERROR;        // All was good.
    return true;
  }

//---------------------------------------------------------------------LockIt()

  static bool LockIt()
  // Get the OWNERSHIP.DAT file in read/write mode.
  // If it can't lock it, then it returns false.
  {
    if (Initialised())
        UnmapViewOfFile(SM);

    SM = 
        (container_id*)MapViewOfFile(Mapping, 
                               FILE_MAP_READ|FILE_MAP_WRITE, 0, 0, 0);

    return Initialised();
  }

//------------------------------------------------------------------ReleaseIt()

  static void ReleaseIt()
  // Release the exclusive write lock on the OWNERSHIP.DAT file.
  {
    if (Initialised())
        UnmapViewOfFile(SM);

    SM = (container_id*)MapViewOfFile(Mapping,FILE_MAP_READ, 0, 0, 0);
  }


//-------------------------------------------------------------------Fix()

 bool TestContainerId(container_id x)
{
    bool toret = true;
    container_id p   = x;

    if (HaveIt())
    {
      while (p!=ROOT)
        if (ValidContainerId(p))
            p = SM[p];
        else { toret = false;
	       break;
             }
    } else toret = false;

    return toret;
}

//-------------------------------------------------------------------- Fix() --

 bool Fix(void)
{
   bool toret = true;

   if (HaveIt())
   {
    if (!LockIt()
     || !ObtainMutex())
         b_errno = E_MUTEX_BUSY;
    else {
        // Run over the vector in order to find dangling id's
        for (unsigned int n=ROOT+1;n<SM_len;n++)
        {
            if (SM[n]!=0)
            {
             DEBUG("Testing (first) with parent (second)");
             DEBUG((int)n);
             DEBUG((int)SM[n]);

             if (TestContainerId(SM[n]))
             {
                WIN32_FIND_DATA fnd;
		char buf[MAXLEN];
                char *filename = CidToPath(n,buf,sizeof(buf));
                if (FindFirstFile(filename,&fnd)==INVALID_HANDLE_VALUE)
                {
                    SM[n] = 0;
		    toret = false;
                    DEBUG("Error found.");
                }
                else DEBUG("Done.");
                delete filename;
             } else {
                DEBUG("Error found.");
                SM[n] = 0;
                toret = false;
             }
            }
        }
        b_errno = E_NOERROR;
    }
   }

    return toret;
}

//-------------------------------------------------------------------CheckContainer()
 bool CheckContainer(container_id x)
{
    bool toret  = true;
    container_id *ptr = DiskListOfChildren(x),*list;

    CreateDirectory(LOSTDIR,NULL);  // Win32 CreateDirectory

    if (ptr!=NULL)
    {
     if (ObtainMutex())
     {
        for (list = ptr;*list;list++)
        {
            if (!ValidContainerId(*list))
            {
                WIN32_FIND_DATA x;
		char file[MAXLEN];
                CidToPath(*list, file, sizeof(file));
                str onlyfilename = strrchr(file,'\\');
                char finaldestname[MAXLEN];
		char directory[MAXLEN];
		strcpy(finaldestname, LOSTDIR);
		strcat(finaldestname, "\\");
		strcat(finaldestname,onlyfilename + 1);
		strcpy(directory, file);
		strcat(directory, onlyfilename-1);
		strcat(directory, "\\*");
		strcat(directory, CONTAINER_EXT);

                MoveFile(file, finaldestname);

                if (FindFirstFile(directory,&x)
                    ==INVALID_HANDLE_VALUE)
                    RemoveDirectory(directory);
	    }
        }

        FreeMutex();
        b_errno = E_NOERROR;
        free(anon_heap, ptr);
     } else b_errno = E_MUTEX_BUSY, toret = false;
    } else b_errno = E_BAD_CONTAINER_ID,toret = false;

    return toret;
}

//-------------------------------------------------------------------FreeRes()
 void FreeRes(void)
{
    // Free all resources

    if (Initialised())
    {
      // Free the mutexes
      CloseHandle(hmutex);

      // Free the mapping
      UnmapViewOfFile(SM);
      CloseHandle(Mapping);
      CloseHandle(IdxFile);
    }
}

//-------------------------------------------------------------------NewContainerId()

 container_id NewContainerId(container_id parent)
  // Allocate a new, unique container id with 'parent' as the parent.
  {
     char aux[MAXLEN];
     container_id cid=0;

    if (!HaveIt())
        return 0;

    if (parent> 0)
    {
     if    (parent!=ROOT
      and !(ValidContainerId(SM[parent])))
     {
	b_errno = E_INVALID_PARENT;
        return 0;
     }

     cid = ReturnNextFree();

     if (!ValidContainerId(cid))
     {
        DEBUG((int)cid);
	ASSERT(false, "Not valid container id (above) !");
	return 0;
     }
     else
     {
      if (not LockIt())
      {
        b_errno = E_MUTEX_BUSY;
        return 0;
      }

      if (!ObtainMutex())
      {
        b_errno = E_MUTEX_BUSY;
        return 0;
      }

      // Do the work
      SM[cid] = parent;
      // a) Create the file
      // a.1) Create the directory of the parent

      CidToPath(parent, aux, sizeof(aux));
      if (parent!=ROOT)
        *(aux+(strlen(aux)-strlen(CONTAINER_EXT))) = '\0';

      if (!CreateNewDirectory(aux))
           SM[cid] = 0,cid     = 0;

      // a.2) Create the container file itself.
      if (cid!=0)
      {
        HANDLE h;
        CidToPath(cid, aux, sizeof(aux));

        DEBUG("(below) to be created");
        DEBUG(aux);

        h    = CreateFile(aux,GENERIC_WRITE,0,NULL,OPEN_ALWAYS,
                         FILE_ATTRIBUTE_NORMAL,NULL);  // Create file Win32 API

        if (h == INVALID_HANDLE_VALUE)
        {
         SM[cid] = 0,cid     = 0;
         b_errno = E_CREATE_WINDIR;
         DEBUG("Error while creating file.");
        }
        else {
          CloseHandle(h);
          DEBUG("File succesfully created:");
          DEBUG(aux);
        };

      } else DEBUG("Skipping create file because a previous error");


      // We're finished.
      if (cid>1)    b_errno = E_NOERROR;        // All was good.

      ReleaseIt();
      FreeMutex();


      return cid;
    }
   }
   else {
        // We're creating the ROOT container
      if (LockIt())
        if (ObtainMutex())
        {
        HANDLE h;
        CidToPath(ROOT, aux, sizeof(aux));

        DEBUG("ROOT container associated file to be created");
        DEBUG(aux);

        h    = CreateFile(aux,GENERIC_WRITE,0,NULL,OPEN_ALWAYS,
                         FILE_ATTRIBUTE_NORMAL,NULL);  // Create file Win32 API

        if (h == INVALID_HANDLE_VALUE)
        {
         SM[cid] = 0,cid     = 0;
         b_errno = E_CREATE_WINDIR;
         DEBUG("Error while creating ROOT file.");
        }
        else {
          cid     = ROOT;
          SM[cid] = ROOT;
          b_errno = E_NOERROR;

          CloseHandle(h);
          DEBUG("ROOT File succesfully created:");
          DEBUG(aux);
        };

        }
        else b_errno = E_MUTEX_BUSY;
      else b_errno = E_MUTEX_BUSY;

      FreeMutex();
      return cid;
   }
  }

//----------------------------------------------------------------DeleteContainerId()

  bool DeleteContainerId(container_id cid)
// Delete this id from shared memory (SM)
  {
    bool toret = true;
    char parent_filename[MAXLEN];
    char filename[MAXLEN];

    // Init
    if (!HaveIt())
        return false;


    DEBUG("DeleteContainerId(). Trying to delete container id:");
    DEBUG((int)cid);

    // Check
    if (!ObtainMutex())
    {
        b_errno = E_MUTEX_BUSY;
        toret   = false;
    } else
    {
      if (!ValidContainerId(cid)            // Check correctness of given data
         ||!ValidContainerId(SM[cid])
         || cid == ROOT)              // You can't delete the root !
      {
        b_errno = E_BAD_CONTAINER_ID;
        toret   = false;
        DEBUG("The Cid (first) with parent (second) is not valid. Results (3) (4)");
        DEBUG(cid);
        DEBUG(SM[cid]);
        DEBUG(ValidContainerId(cid));
        DEBUG(ValidContainerId(cid));
      }
      else
      {
      // Get the Windows path of the cid
      WIN32_FIND_DATA px;
      CidToPath(cid, filename, sizeof(filename));
      DEBUG("The container filename to delete is");
      DEBUG(filename);

      if (strchr(filename,'.'))
        *(filename+strlen(filename)-strlen(CONTAINER_EXT)) = '\0';
      DEBUG("The container filename to delete is");
      DEBUG(filename);
      
      // Check if the directory exists (therefore it has childs)
      DEBUG("Cheking if it has childs");
      if (FindFirstFile(filename,&px)   // Find First File Win32
            != INVALID_HANDLE_VALUE)
      {
        b_errno = E_HASCHILD;
        FreeMutex();
        toret   = false;
      } else
      {
        DEBUG("No, it has not childs");
        if (not LockIt())
        {
            b_errno = E_MUTEX_BUSY;
            FreeMutex();
            toret   = false;
        }
        else
        {

            // Delete File of the container
            {char aux[MAXLEN];
             strcpy(aux, filename);
             strcat(aux, CONTAINER_EXT);
             if (!DeleteFile(aux))// Delete File Win32
             {
                DEBUG("Delete of file was not succesful");
                DEBUG(aux);
                b_errno = E_BAD_CONTAINER_ID;
                toret   = false;
             }
             else
             if (SM[cid]!=ROOT)
             {
            // Check if it was the last file in the parent container
                CidToPath(SM[cid], parent_filename, sizeof(parent_filename));
                *(parent_filename+strlen(parent_filename)-strlen(CONTAINER_EXT)) = '\0';
                strcpy(aux, parent_filename);
                strcat(aux, "\\*");
                strcat(aux, CONTAINER_EXT);

                DEBUG("Cheking if it was the last file in");
                DEBUG(aux);
                if (FindFirstFile(aux,&px)==INVALID_HANDLE_VALUE)
                {
                    // It was the last file of the parent directory,
                    // so the parent directory must be deleted
                    DEBUG("Yes, it was the last file in:");
                    DEBUG(parent_filename);
                    
                    if (!RemoveDirectory(parent_filename))
                        MandatorilyRemoveDirectory(parent_filename);
                } else DEBUG("No, it was not the last file.");
             } else DEBUG("Passing parent dir because it is ROOT.");
            }
            // Delete entry in the container_id table
            if (toret)
            {
              SM[cid] = 0;
              b_errno = E_NOERROR;
            }
              
            ReleaseIt();
            FreeMutex();
        }
      }
      }
    }

    return toret;
}

//------------------------------------------------------------------MoveContainerId()

 bool MoveContainerId(container_id cid, container_id newparent)
  // Move this cid so that its parent becomes 'newparent'.
  {
    bool toret = true;

    if (!HaveIt())
        return false;

    if (!ObtainMutex())
    {
        b_errno = E_MUTEX_BUSY;
        return false;
    }

    if (!ValidContainerId(cid)
     || !ValidContainerId(newparent)
     || cid == ROOT)
    {
        b_errno = E_BAD_CONTAINER_ID;
        return false;
    }

  // Check that 'from' is not an ancestor of 'to'
  // This process is currently moving a container
    {
    int level = 0;
    container_id p  = newparent;

    do {
        if (p == cid) {
            b_errno = E_CYCLE;
            FreeMutex();
            return false;
        }
        if (level > CORRUPTLEVEL) {
            b_errno = E_CORRUPT;
            FreeMutex();
            return false;
        }
        if (p == ROOT)
            break;
        if (!ValidContainerId(p)) {
            b_errno = E_CORRUPT;
            FreeMutex();
            return false;
        }

        p = SM[p];
        level++;
    } forever;
    }

    if (not LockIt())
    {
        FreeMutex();
        return false;
    }

    // Now moving
    char filename[MAXLEN];
    char onlyfilename[MAXLEN];
    char newparentname[MAXLEN];

    // Get the name of cid, without extension
    CidToPath(cid, filename, sizeof(filename));
    *(filename+strlen(filename)-strlen(CONTAINER_EXT)) = '\0';
    // Get only the container filename in char *
    {
        char* posbar;

        if ((posbar=strrchr(filename,'\\'))!=NULL)
             strcpy(onlyfilename,posbar+1);
        else strcpy(onlyfilename,filename);
    }


    DEBUG("The filename of the origin container is:");
    DEBUG(filename);

    // Get the name of newparent
    CidToPath(newparent, newparentname, sizeof(newparentname));
    if (strchr(newparentname,'.')
     && newparent != ROOT)
        *(newparentname+strlen(newparentname)-strlen(CONTAINER_EXT)) = '\0';

    DEBUG("The name of the destination container is:");
    DEBUG(newparentname);


    if (!CreateNewDirectory(newparentname))
        toret = false;

    // Now move it:
    if (toret)
    {
        char origfile[1024];
        char destfile[MAXLEN];
	sprintf(origfile, "%s"CONTAINER_EXT, filename);
	sprintf(destfile, "%s\\%s"CONTAINER_EXT, newparentname, onlyfilename);

        // Move the file
        DEBUG("Move (first) to (second) toret is (third)");
        DEBUG(origfile);
        DEBUG(destfile);
        DEBUG((int)toret);

        if (!MoveFile(origfile, destfile))
        {
            b_errno = E_FILE_MOVE_ERROR;
            toret   = false;
        }

	sprintf(destfile, "%s\\%s", newparentname, onlyfilename);

        // Move the related directory (perhaps it does no exist)
        DEBUG("Move (first) to (second) toret is (tercero)");
        DEBUG(filename);
        DEBUG(destfile);
        DEBUG((int)toret);

        if (toret)
            MoveFile(filename,destfile);

        // Delete the original directory (if it not exists, nothing happens)
        // If it is empty, it must be deleted
        char directoryparent[MAXLEN];
	CidToPath(SM[cid], directoryparent, sizeof(directoryparent));
        if (strchr(directoryparent,'.')
         && SM[cid]!= ROOT)
            *(directoryparent+strlen(directoryparent)-strlen(CONTAINER_EXT)) = '\0';

        DEBUG("Deleting ...");
        DEBUG(directoryparent);

        RemoveDirectory(directoryparent);

        // Finally, change the shared memory
        if (toret)
        {
            b_errno = E_NOERROR;
            SM[cid] = newparent;
        }
    }

    ReleaseIt();
    FreeMutex();

    return toret;
  }

//--------------------------------------------------------CidToPathRecursive()
  static
  str CidToPathRecurse(container_id cid, str dest, str end)
  {
    if (cid == -1 or cid == 0) {
        strcpy(dest,FILESURETOFAIL);
        return dest + strlen(FILESURETOFAIL);
        //
        // That segment is an error
        // Return a filename which is sure to fail.
        //
    }
    else if (cid == ROOT) {
        strcpy(dest,PERSISTENT_ROOT);
        return dest + strlen(PERSISTENT_ROOT);
        // Top of recursion. Copy ROOT address
    }
    else {
        dest = CidToPathRecurse(SM[cid], dest, end);

        if (dest<end)
        {
          if (*(dest-1)!='\\')
            *(dest++) = '\\';   // Next Level == Next directory

          if ((dest+((int)log(double(cid)))+1)<end)
          {
            itoa(cid,dest,10);

            while((*dest++)!='\0');
            dest--;
          }
          else strcpy(dest-strlen(FILESURETOFAIL),FILESURETOFAIL);
        } else
            strcpy(dest-strlen(FILESURETOFAIL),FILESURETOFAIL);

        return dest;
    }
  }

//-----------------------------------------------------------------CidToPath()

  str CidToPath(container_id cid, str buf, int sizeofbuf)
// Retrieve the full Windows path name for this cid.
  {

    if (!HaveIt())
        return NULL;

    if (not ValidContainerId(cid))
    {
        b_errno = E_BAD_CONTAINER_ID;
        return NULL;
    };

    if (!ObtainMutex())
    {
        b_errno = E_MUTEX_BUSY;
        return NULL;
    }

    if (buf==NULL)
    {
        buf  = new char[MAXLEN];
        *buf = '\0';
        sizeofbuf = MAXLEN;
    }
    else *buf = '\0';

    if (sizeofbuf==0)
        sizeofbuf = sizeof(buf);

    // Now working
    char *end = CidToPathRecurse(cid, buf, buf + sizeofbuf);

    if (cid!=ROOT)
    {
        if  ((end+strlen(CONTAINER_EXT))<(buf+sizeofbuf))
            strcpy(end, CONTAINER_EXT);
        else strcpy(buf,FILESURETOFAIL);
    } else strcat(buf, "\\1"CONTAINER_EXT);

    b_errno = E_NOERROR;
    FreeMutex();
    return buf;
 }

//-------------------------------------------------------------------AddFile()

  static
  void AddFile(const str &name)
  // Add the cid of this filename to the dynamic array 'A'.
  { container_id cid;

     // Extract the Path from the file name
     cid = PathToCid(const_cast<char*>(name));

     // Add it to the list:
     A = (container_id*)realloc(A, Heap::RoundUp((A_idx+2) * sizeof(container_id)));
     A[A_idx++] = cid;
  }

//--------------------------------------------------------DiskListOfChildren()

  container_id* DiskListOfChildren(container_id cid)
  {
    WIN32_FIND_DATA Fnd;
    char path[1024];
    str s;
    HANDLE Search;

    A      = NULL;
    A_idx  = 0;

    DEBUG("Listing childs of ...");
    DEBUG((int)cid);

    CidToPath(cid, path, sizeof(path));

    if (path!=NULL)
    {

      DEBUG("Result of CidToPath() is ...");
      DEBUG(path);

      if (cid!=ROOT)
           s      = path + (strlen(path) - strlen(CONTAINER_EXT));
      else s = path + strlen(path);

      strcpy(s, "\\*");  // Substitute the .CTR by \\* in order to get
                       // access to the files
      strcat(s, CONTAINER_EXT);

      DEBUG("Searching for ");
      DEBUG(path);

      A = (container_id*) malloc(anon_heap, sizeof(container_id));

      Search = FindFirstFile(path, &Fnd);
      if (Search == INVALID_HANDLE_VALUE)
        goto RETURN;

      AddFile(Fnd.cFileName);
      while (FindNextFile(Search, &Fnd))
        AddFile(Fnd.cFileName);
    }

    RETURN:
    A[A_idx++] = 0;

    return A;
  }






