/* uInspect.cpp

        The kernel of the program coded in C++.
        This code uses the Win32 functions in order to store in memory
        container's with the file mapped in memory mechanism.

        loaDll() and releaseDll() are the only ones that depend
        on Win32 functions. Plus, of course the win32 HANDLEs in the private data
        of the class
*/

#include "windows.h"
#include <sstream>
#include <iomanip>
#include <cctype>
#include <ctime>
#include "uinspect.h"

// Constants -------------------------------------------------------------------
const unsigned int Inspect::version  = 11007;

const char * inspHEADER[] = { "Container Header",
                              "Main Container's Directory",
                              "Public Objects of this container",
                              "C-N Swizzling Entries",
                              "\0"
};
const std::string Inspect::inspERROR = "An error ocurred while processing archive.";
const char *Inspect::kinds[] = {
        "This is not a Barbados container.",
        "This is a Barbados container without a C-N Swizzling table.",
        "This is a Barbados container with a C-N Swizzling table."
};

const char BARBADOS_LETTER_FOR_CID =                'B';
const int  BARBADOS_ROOT_CONTAINER =                 1;
const void *TABLE_FINISH_MARK   =                 (void* ) 0xFFFFFFFF;

// External functions  ---------------------------------------------------------
extern int LengthOfTypeString(type_type);
extern size_t TypeSize(type_type type);


//--------------------------------------------------------------- loadDll()
void *Inspect::loadDll(void)
{
        // Using the CreateFileMapping(), MapViewOfFile(), UnMapViewOfFile()
        // and CloseHandle() functions of the windows win32 API
        location = NULL;

        // Open the file
        file = CreateFile( filename.c_str(), GENERIC_READ | GENERIC_WRITE,
                        FILE_SHARE_READ, NULL,
                        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL,
                        NULL
                );

        if (file!=INVALID_HANDLE_VALUE)
        {
                unsigned long int read;

                // Assign the length of file
                filelength = GetFileSize(file, NULL);

                // Prepare the mapping
                memory = CreateFileMapping(file, NULL,  // file handle Security
                                  PAGE_READWRITE,       // Access in memory
                                  0,0,                  // size of  file
                                  "InspectorFM"         // name of this resource
                        );
                if (memory!=NULL)
                {
                        // All is correct, map
                        location = MapViewOfFile(memory,
                                    FILE_MAP_READ | FILE_MAP_WRITE,
                                    0,0,                // Start
                                    0                   // Entire File to map
                                );

                        if (location==NULL)
                                errorhappened = true;
                }
                else errorhappened = true;
        }
        else errorhappened = true;

        return location;
}

//--------------------------------------------------------------- releaseDll()
void Inspect::releaseDll(void)
{
        UnmapViewOfFile(location);
        CloseHandle(memory);
        CloseHandle(file);
}

//--------------------------------------------------------------- Constructor
Inspect::Inspect(const std::string &name): filename(name),errorhappened(false)
{
        loadDll();
}

//--------------------------------------------------------------- totalDump()
void Inspect::totalDump(void)
{
        if (!errorhappened)
        {
                headerDump();
                directoryDump();
                CNTableDump();
        }
        else puttext(inspERROR);
}

//--------------------------------------------------------------- headerDump
void Inspect::headerDump(void)
{

/*
INFO
====

        At the very beginning, passing the first 4 bytes, we find the container
header: (this info may be out-to-date, please check typedef.h)

struct ContainerHeader {
        int container_signature;
        void * base_address;            // The base address this container was loaded last
                                        // at.
	unsigned int lin_start;		// The offset from the header to the list of
					// (ptr, container id, name) pointers
	time_t interface_change;	// What time did the interface last change?
	time_t remake;			// What time was the container last recompiled?
	directory dir;			// The root object
};


        The directory is a part of the container header:

class Directory {
	Namedobj **Hash;
	uint Hash_len, NumEntries;
	Directory *parent;
	container_id parent_cid;
};

*/
        using std::cout;
        using std::endl;
        using std::hex;

        std::ostringstream out;

        // Get Container header
        header = ((ContainerHeader *) (((char *)location)+4));

        // Get kindoffile
        if (header->container_signature == BARBADOS_SIGNATURE)
        {
          // The tables at the end of the container are empty
          // if we have only two marks (MARK_FINISH_TABLE)
          if (header->lin_start==(filelength-(sizeof(int)*2)))
               kindoffile = kinds[1];
          else kindoffile = kinds[2];
        } else {
                kindoffile = kinds[0];
                errorhappened = true;
        }

        out         << "Kind of file:       "
                    << kindoffile << endl
                    << "Version:            "
                    << ((header->container_signature) & 0x000000FF)
                    << endl                << endl;

        // Perhaps we do not need more processing
        if (kindoffile!=kinds[1])
        {
         // Get directory
         direc = &header->dir;

         // Gathering all the information
         // Container Header
              out   << inspHEADER[0]           << endl
                    << "Start of C-N Swizzling Table,"
                                               << endl
                    << "Container Length:   "
                    << header->lin_start       << endl
                    << "Base Address:       "  << header->base_address
                    << endl
                    << "Interface changed:  "  << extime(header->interface_change)
                    << endl
                    << "Remaked at:         "  << extime(header->remake)
                                               << endl
                    << "Saved at:           "  << extime(header->timestamp)
                                               << endl
                    << endl                    << endl
                    << inspHEADER[1]           << endl
        // Directory summary
                    << "Number of Entries:  "  << direc->getHashLen() << endl
                    << "Parent Cont.'s id:  "  << printCid(direc->parent_cid)
                    << endl
                ;

               if (direc->parent_cid==0)
                out << "This container is the ROOT." << endl;
        }

        // Print & Return
        puttext(out.str());
}

//-------------------------------------------------------------------- swizzle()
void *Inspect::swizzle(void * x) const
{
        char *aux = (char *) x;

        aux += getDelta();

        return ((void *) aux);
}

//----------------------------------------------------------------- getStorage()
static
std::string getStorage(unsigned char x)
{
        const char * store[] = { "unknown_storage",
		"static_storage", "perprocess_storage", "perthread_storage",
                "local_static", "member_storage", "inherit_storage",
		"auto_storage", "parameter_storage", "const_storage",
		"straight_fn", "library_fn", "member_fn",
		"oneinstr_fn", "virtual_fn", "inline_fn",
		"typedef_storage", "macro_storage", "keyword_storage"
        };

        return store[x];
}

//-------------------------------------------------------------------- getType()
static
std::string getType(char x, void *ptr)
{
        switch (x)
        {
                case tp_void            : return "void";
                case tp_bool            : return "boolean";
                case tp_char            : return "char";
                case tp_enumerated      : return "enum";
                case tp_short           : return "short int";
                case tp_int             : return "int";
                case tp_long            : return "long int";
                case tp_float           : return "float";
                case tp_double          : return "double";
                case tp_uint            : return "unsigned int";
                case tp_uchar           : return "unsigned char";
                case tp_ushort          : return "unsigned short int";
                case tp_class           : if (ptr==NULL)
                                                return "class";
                                          else  return "object";
                case tp_pointer         : return "pointer";
                case tp_reference       : return "reference";
                case tp_array           : return "array";
                case tp_dynarray        : return "dynamic array";
                case tp_function        : return "function";
                case tp_terminated      : return "";
                case tp_container       : return "container";
                case tp_const           : return "const";
                case tp_volatile        : return "volatile";
                default                 : return "";
        }
}

static
std::string interpretType(const namedobj &obj)
{
        char *that;
        std::ostringstream out;
        int length;

        out << '[' << getType(obj.type[0], obj.u.location);
        if(obj.type[0]!=tp_class)
        {
          that   = ((char *)&(obj.type)) + 1;
          length = LengthOfTypeString(const_cast<unsigned char*>(obj.type)) - 1;
          for(unsigned int n=0; n<length; ++n)
          {
                out << ',' << ' '
                    << getType(*(that++), obj.u.location)
                    << (((*(that-1)) == tp_pointer) ? "*" : "")
                ;

                if ((*(that-1))==tp_class)
                                break;
          }
        }

        out << ']';
        return out.str();
}

//-------------------------------------------------------------- directoryDump()
void Inspect::directoryDump()
{
        namedobj_type obj;
        type_type that;
        std::ostringstream out;
        unsigned int length;
        bool isSimple;

        using std::cout;
        using std::endl;
        using std::endl;

        // List namedobj's in this directory
        if (!errorhappened
         && direc->getHashLen()>0)
        {
         namedobj **h = (namedobj **) swizzle(*((namedobj ***) direc));
                // Hash is the first field

         out << endl
             << inspHEADER[2]           << endl;

         if (!itsHere(h))
                out << "*** Oops ... error reading the directory ! ... strange ..." << endl;
         else
          for (unsigned int n=0; n<direc->getHashLen(); ++n)
          {
           if (h[n]!= NULL)
           {
             obj= (namedobj *) swizzle(h[n]);
             if (!itsHere(obj))
                out << "*** Oops .. error reading one entry ! ... strange ..." << endl;
             else {
               do
               {
                // Print the tile
                out
                    << "Name:               "  << (char *)swizzle(obj->name)
                    << '\n'
                    << "Storage:            "  << getStorage(obj->storage)
                    << '\n'
                    << "Type:               "  << interpretType(*obj)
                    << '\n'
                    << "Value:              "  << printVal(*obj, isSimple)
                    <<'\n'                     << endl
                ;

                // Store it, if it is a simple object
                if (isSimple)
                      objlist[(char *)swizzle(obj->name)] = obj;

                // Get next tile
                if (obj->next!=NULL)
                        obj = (namedobj *) swizzle(obj->next);
                else    obj = NULL;

                // Show progress (if any ;-) )
                putProgress(((int)obj) - ((int)location), filelength);
               }
               while(obj!=NULL);
             }
           }
          }

         puttext(out.str());
        }
}

//-------------------------------------------------------------- CNTableDump()
void Inspect::CNTableDump() const
{
        std::ostringstream out;

        using std::cout;
        using std::endl;
        using std::endl;
        using std::hex;

        CNInfo rec;
        int length;
        char *where;

        if (!errorhappened)
        {
                // C-N Swizzling Entries
                out << endl
                    << inspHEADER[3]                  << endl
                    << "Pointer positions are relative to the base_address" << endl
                    << endl;
                ;

                where       = ((char *) location + header->lin_start);
                rec.pointer = *((void ****) where);
                where       = (((char *) where) + 4);
                while(rec.pointer!=TABLE_FINISH_MARK
                  && ((char *)where) < ((char *) location + filelength))
                {
                   // Get the container_id
                   rec.container = *((container_id *) where);
                   where         = (((char *) where) + sizeof(container_id));

                   // Get the name of the Namedobj being referenced
                   length   = *((int *) where);
                   where   += sizeof(length);
                   rec.name = new char[length + 1];
                   memcpy(rec.name, where, length);
                   *(rec.name + length) = 0;
                   where   += length;

                   // Get the reducedclassdef of this entry
                   length   = *((int *) where);
                   where   += sizeof(length);
                   rec.ReducedClassdef = new char[length + 1];
                   memcpy(rec.ReducedClassdef, where, length);
                   *(rec.ReducedClassdef + length) = 0;
                   where   += length;
                   if (strlen(rec.ReducedClassdef) == 0)
                   {
                        delete rec.ReducedClassdef;
                        rec.ReducedClassdef = new char[25];
                        strcpy(rec.ReducedClassdef, "No reduced classdef.");
                   }

                   // Get the pointers list
                   length   = *((int *) where);
                   where   += sizeof(length);
                   rec.ptrs = (void **) new char[length];
                   memcpy(rec.ptrs, where, length);
                   where   += length;

                   // Print the entry's information
                   // *ptr
                   out << "Namedobj's addrs :" << rec.pointer          << endl
                       << "Kind of target:   ";

                   // TypeOfPointer, ReducedClassdef when needed
                   if (*rec.name=='A')
                        out << "absolute address";
                        else if (*rec.name=='R')
                                  out << "self-relative address";
                             else {
                                out << "TypeClass address"  << endl
                                    << "ReducedClassdef:  " << '\n'
                                    << rec.ReducedClassdef;
                             }

                   // Container id, namedobj name
                   out << endl
                       << "Container id:     " << rec.container      << endl
                       << "Namedobj ref.:    " << '\"' << rec.name +1<< '\"'
                       << endl
                       << "List of Ptrs:     ";
                   ;

                   // list of pointers
                   for (int n=0; n<length; n += sizeof(void*))
                   {
                        out << *((void **)(((char *)rec.ptrs) + n));

                        if ((n + sizeof(void*)) < length)
                                out << ',' << ' ';
                   }

                   // Next entry
                   rec.pointer = *((void ****) where);
                   where       = (((char *) where) + 4);
                   out << '\n' << endl;
                   
                   delete rec.ReducedClassdef;
                   delete rec.name;
                   delete rec.ptrs;

                   // Show progress (if any ;-) )
                   putProgress(((int)where) - ((int)location), filelength);
                }
        }

        puttext(out.str());
}

/*-------------------------------------------------------------- RCTableDump()
void Inspect::RCTableDump(void *where) const
{
        using std::cout;
        using std::endl;
        using std::hex;
        using std::string;

        string name, descriptor;
        char strindex[256];
        int index = 0;
        int lengthindexstr;

        strcpy(strindex, "Index:            ");
        lengthindexstr = strlen(strindex);
        if (!errorhappened)
        {
                // Reduced Classdef Entries

                puttext(inspHEADER[4]);

                while( *((void **)where) != TABLE_FINISH_MARK
                  && ((char *)where) < ((char *) location + filelength))
                {
                   // Get the class name
                   name = "";
                   do {
                        name.push_back(*((char*) where));
                        where   = (((char *) where) + 1);
                   } while(*(((char*)where)-1)!='|');
                   name.push_back('\0');

                   // Get the descriptor
                   descriptor = "";
                   do {
                        descriptor.push_back(*((char*) where));
                        where   = (((char *) where) + 1);
                   } while(*(((char*)where)-1)!='\0');

                   // Print the entry's information
                   itoa(index++, strindex + lengthindexstr, 10);
                   puttext(strindex, NOTCR);
                   puttext("Class name:       " + name, NOTCR);
                   puttext("Class descriptor: " + descriptor);

                   // Show progress (if any ;-) )
                   putProgress(((int)where) - ((int)location), filelength);
                }
        }
}
*/

// ------------------------------------------------------------------ printCid()
std::string Inspect::printCid(int cid) const
{
        std::ostringstream toret;

        toret << BARBADOS_LETTER_FOR_CID
              << cid;

        if (cid==BARBADOS_ROOT_CONTAINER)
                toret << " (ROOT container)";

        return toret.str();
}

// -------------------------------------------------------------------- PrintVal
std::string Inspect::printVal(const namedobj &obj, bool &simple, void *val) const
{
        std::ostringstream toret;
        std::string aux;
        simple    = true;
        val       = ((val==NULL)? obj.u.location : val);
        int    n  = 0;
        bool done = false;

        while(!done)
         switch(obj.type[n++])
         {
                case tp_bool:
                        if (itsHere(swizzle(val)))
                                toret << (*((bool *)(swizzle(val)))? true:false);
                        else    toret << "(foreign)";
                        done = true;
                        break;
                case tp_char:
                        if (!std::isprint(*((char *)swizzle(val))))
                                toret << "%c"
                                      << (*((int*)swizzle(val)));
                        else
                                toret << *((char *)swizzle(val));
                        done = true;
                        break;
                case tp_long:
                        if (itsHere(swizzle(val)))
                                toret << *((long int *)swizzle(val));
                        else    toret << "(foreign)";
                        done = true;
                        break;
                case tp_short:
                        if (itsHere(swizzle(val)))
                                toret << *((short int *)swizzle(val));
                        else    toret << "(foreign)";
                        done = true;
                        break;
                case tp_enumerated:
                case tp_int:
                        if (itsHere(swizzle(val)))
                                toret << *((int *)swizzle(val));
                        else    toret << "(foreign)";
                        done = true;
                        break;
                case tp_float:
                        if (itsHere(swizzle(val)))
                                toret << *((float *)swizzle(val));
                        else    toret << "(foreign)";
                        done = true;
                        break;
                case tp_double:
                        if (itsHere(swizzle(val)))
                                toret << *((double *)swizzle(val));
                        else    toret << "(foreign)";
                        done = true;
                        break;
                case tp_uint:
                        if (itsHere(swizzle(val)))
                                toret << *((unsigned int *)swizzle(val));
                        else    toret << "(foreign)";
                        done = true;
                        break;
                case tp_uchar:
                        if (itsHere(swizzle(val)))
                                toret << *((unsigned char *)swizzle(val));
                        else    toret << "(foreign)";
                        done = true;
                        break;
                case tp_ushort:
                        if (itsHere(swizzle(val)))
                                toret << *((unsigned short int *)swizzle(val));
                        else    toret << "(foreign)";
                        done = true;
                        break;
                case tp_pointer:
                case tp_reference:
                        if (itsHere(swizzle(val)))
                                toret << *((void **)swizzle(val));
                        else    toret << "(foreign)";
                        done = true;
                        break;
                case tp_class:
                        if (val==NULL)
                                toret << interpretClass(obj);
                        else
                                toret << interpretObject(obj);
                        simple = false;
                        done = true;
                        break;
                case tp_const:
                        break;
                default:
                        toret << "not a direct type";
                        done = true;
                        simple = false;
         }

        return toret.str();
}

bool inline Inspect::itsHere(void *v) const
{
        char *c = (char *) v;

        if ((c<location
         ||  c>(((char*)location) + filelength)))
             return false;
        else return true;
}

std::string Inspect::interpretClass(const namedobj &obj) const
{
        Classdef* cl = (Classdef*) swizzle(((void *)*((Classdef* *)(obj.type+1))));
        namedobj_type m;
        std::ostringstream toret;

        if (!itsHere(cl) || cl->signature!=HELLO_BABE)
                toret << "foreign class";
        else
        {
                m = (namedobj *) swizzle(cl->member);
                toret << '{';
                while(m != NULL)
                {
                        // Type
                        toret << interpretType(*m);

                        // Next entry
                        m = m->next;
                        if (m != NULL
                         && !itsHere(((char *)m)))
                        {
                                m = NULL;
                                toret << '...';
                        }
                        else
                          if (m!=NULL)
                          {
                                m = (namedobj *) this->swizzle(m);
                                toret << ',';
                          }
                }
                toret << '}';
        }

        return toret.str();
}

std::string Inspect::interpretObject(const namedobj &obj) const
{
        Classdef* cl = (Classdef*) swizzle(((void *)*((Classdef* *) (obj.type+1))));
        namedobj_type m;
        std::ostringstream toret;
        char *pos;

        if (!itsHere(cl) || cl->signature != HELLO_BABE)
                toret << "foreign class";
        else
        {
                bool isSimple = true;
                m   = (namedobj *) swizzle(cl->member);
                pos = (char *) obj.u.location;

                toret << '{';
                while(m != NULL
                   && isSimple)
                {
                        // Type
                        toret << printVal(*m, isSimple, pos);

                        // Next one
                        pos += TypeSize(m->type);

                        if(!isSimple)
                        {
                                toret << "...";
                                continue;
                        }

                        // Next entry
                        m = m->next;
                        if (m != NULL
                         && !itsHere(((char *)m)))
                        {
                                m = NULL;
                                toret << '...';
                        }
                        else
                         if (m!=NULL)
                         {
                                m = (namedobj *) swizzle(m);
                                toret << ',';
                         }

                }
                toret << '}';
        }

        return toret.str();
}

void Inspect::Modify(namedobj_type obj, const std::string &newvalue)
{
        if (obj != NULL
         && itsHere(obj))
           switch(*(obj->type))
           {
                case tp_enumerated:
                case tp_int:
                        *((int*) swizzle(obj->u.location)) = atoi(newvalue.c_str());
                        break;
                case tp_bool:
                        if (newvalue=="true")
                             *((short int*) swizzle(obj->u.location)) = 1;
                        else if (newvalue=="false")
                                *((short int*) swizzle(obj->u.location)) = 0;
                        break;
                case tp_uchar:
                case tp_char:
                        *((char*) swizzle(obj->u.location)) = atoi(newvalue.c_str());
                        break;
                case tp_long:
                        *((long int*) swizzle(obj->u.location)) = atol(newvalue.c_str());
                        break;
                case tp_ushort:
                case tp_short:
                        *((short int*) swizzle(obj->u.location)) = atoi(newvalue.c_str());
                        break;
                case tp_float:
                        *((float*) swizzle(obj->u.location)) = atof(newvalue.c_str());
                        break;
                case tp_double:
                        *((double *) swizzle(obj->u.location)) = atof(newvalue.c_str());
                        break;
                case tp_pointer:
                case tp_reference:
                        *((int*) swizzle(obj->u.location)) = atoi(newvalue.c_str());
                        break;
           }
}

// -------------------------------------------------------------------- extime()
std::string Inspect::extime(const time_t &x) const
{
        std::ostringstream toret;

        toret << x
              << std::endl
              << '\t' << ctime(&x);

        return toret.str();
}

// --------------------------- Useless function provided because of the needs of type.cpp
void assert_failed(char *x, int i)
{
        return;
}
