#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>
#include "barbados.h"
#include "memory.h"
#include "debug.h"
#include "ipr.h"




        /* We have 6 general-purpose registers and 8 floating-point registers. */
        /* We also have some other non-register enums for describing operands. */

        /* AX is listed last of the general-purpose registers because we want to */
        /* reserve it for special functions, e.g. multiplication/division and */
        /* function calls.  If we leave it for last then we know it's always */
        /* available when we need it. */

/*
	The 'e->storereg' value tells us where we want to store the result
	of each expression.

	reg_discard:  means we don't need the result: but still do the operation
		because of side-effects (eg. often assignments will have reg_discard)
		and also we might want the flags result of the expression. So you
		need to do the operation.

	reg_operand:	means the value is suitable to be used directly as the
		operand for some instruction, and it's a non-register operand.

*/


#define first_general	    reg_BX
#define last_general	    reg_AX
#define code_base	    (void*)0x3


static int localstatic_offset;
static char* localstatic_buf;
interface Expr frame_register;



/*---------------------- Misc functions: ---------------------*/

interface str RegisterToString(uchar reg)
{
        switch (reg) {
            case reg_operand:	return "";
            case reg_AX:	return "AX";
            case reg_BX:	return "BX";
            case reg_CX:	return "CX";
            case reg_DX:	return "DX";
            case reg_SI:	return "SI";
            case reg_DI:	return "DI";
            case reg_R0:	return "R0";
            case reg_R1:	return "R1";
            case reg_R2:	return "R2";
            case reg_R3:	return "R3";
            case reg_R4:	return "R4";
            case reg_R5:	return "R5";
            case reg_R6:	return "R6";
            case reg_R7:	return "R7";
            case reg_SP:	return "SP";
            case reg_BP:	return "BP";
            case reg_stack:	return "stack";
            case reg_fpstack:   return "fpstack";
            case reg_discard:   return "discard";
            case reg_mem:	return "mem";
            case reg_imm:	return "imm";
            case reg_somegeneral:return "somegeneral";
            default:		return "?reg?";
        }
}


static bool IsGeneral(reg_enum reg)
{
        return reg >= first_general and reg <= last_general;
}


static bool IsFloatingReg(reg_enum reg)
{
        return reg >= reg_R0 and reg <= reg_R7;
}


static bool IsFloat(Expr* e)
/* Is this expression something that needs the floating-point unit? */
{
	return e->tp == tp_double or e->tp == tp_long or e->tp == tp_float;
}


static bool IsIntegral(Expr* e)
/* Is this expression something that can fit into a general register? */
{
	return e->tp == tp_pointer or 
		e->tp == tp_int or e->tp == tp_short or e->tp == tp_char or
		e->tp == tp_uint or e->tp == tp_ushort or e->tp == tp_uchar;
}


static int ReverseCondition(int tttn)
/* Turn the comparison around to the mirror image. */
{
        switch (tttn) {
            case 2:     return 7;
            case 3:     return 6;
            case 6:     return 3;
            case 7:     return 2;
            case 12:    return 15;
            case 13:    return 14;
            case 14:    return 13;
            case 15:    return 12;
            default:    return tttn;
        }
}


static int ReverseFlop(int op)
/* Turn this floating-point op into the reverse. */
{
	if (op >= 4 and op <= 7)
	    op ^= 1;
	return op;
}


static int NormalFlop(int op)
/* Turn this floating-point op into the normal form. */
{
	if (op >= 4 and op <= 7)
	    op &= ~1;
	return op;
}




/*--------------- Recognising the MEMIA pattern: --------------*/

/* The Pentium processor has a sophisticated addressing mode which allows you   */
/* to encode the expression:  [Reg1 + (1|2|4|8)*Reg2 + K]  as a single machine- */
/* code operand.   Here we detect this pattern.  If found, we replace the	*/
/* expression with a 'kMEMIA' expression. */

typedef struct memia_node {
	Expr *e1, *e2;	/* It is 'e2' that is scaled. */
	uint K;
	void* Kbase;		/* The base (for swizzling pointers) */
	uint scaler;		/* 0= *1,  1= *2,  2= *4,   3= *8   */
} *memia_type;


static bool IsMemiaMultiplication(Expr* e)
{
	if (e->o != oSHL)
	    return no;
	return (e->right->o == oConstant and e->right->u.i >= 0 and e->right->u.i <= 3);
}


static bool RecogniseMemia(Expr* e)
/* Note that we are looking for the address expression, without the dereference. */
{	memia_type memia;

	/* Does the expression fit the MEMIA template? */
	if (e->o == oMEMIA)
	    return yes;
	if (e->o == oConstant)
	    return yes;
	if (e->o == oNOP1) {
	    if (RecogniseMemia(e->left)) {
		e->o = oMEMIA;
		e->u.p = e->left->u.p;
		return yes;
	    }
	}
	if (e->o == oLocalVar) {
	    AutoNamedobj *obj;

	    obj = (AutoNamedobj*)e->u.p;
	    memia = (memia_type)qmalloc(sizeof(struct memia_node));
	    memia->e1 = &frame_register;
	    memia->K = obj->offset + (int)(e->right);
	    memia->Kbase = NULL;
	    memia->e2 = NULL;
	    memia->scaler = 0;
	    e->o = oMEMIA;
	    e->u.p = memia;
	    if (frame_register.o == 0) {
		frame_register.o = oLocalVar;
		frame_register.tp = tp_int;
		frame_register.u.p = NULL;
	    }
	    return yes;
	}
	if (not IsIntegral(e->left) or not IsIntegral(e->right))
	    return no;
	if (e->o != oADD)
	    return no;


	/*** Okay, as of now we have recognised it. Now set it up: ***/
	memia = (memia_type)qmalloc(sizeof(struct memia_node));
	if (IsMemiaMultiplication(e->left)) {
	    memia->e2 = e->left->left;
	    memia->scaler = e->left->right->u.i;
	    if (e->right->o == oConstant) {
		memia->K = e->right->u.i;
		memia->Kbase = e->right->right;
		memia->e1 = NULL;
	    }
	    else {
		memia->K = 0;
		memia->Kbase = NULL;
		memia->e1 = e->right;
	    }
	}
	else if (IsMemiaMultiplication(e->right)) {
	    memia->e2 = e->right->left;
	    memia->scaler = e->right->right->u.i;
	    if (e->left->o == oConstant) {
		memia->K = e->left->u.i;
		memia->Kbase = e->left->right;
		memia->e1 = NULL;
	    }
	    else {
		memia->K = 0;
		memia->Kbase = NULL;
		memia->e1 = e->left;
	    }
	}
	else if (e->right->o == oConstant and e->left->o == oADD and IsMemiaMultiplication(e->left->left)) {
	    memia->K = e->right->u.i;
	    memia->Kbase = e->right->right;
	    memia->e1 = e->left->right;
	    memia->e2 = e->left->left->left;
	    memia->scaler = e->left->left->right->u.i;
	}
	else if (e->right->o == oConstant and e->left->o == oADD and IsMemiaMultiplication(e->left->right)) {
	    memia->K = e->right->u.i;
	    memia->Kbase = (void*)e->right->left;
	    memia->e1 = e->left->left;
	    memia->e2 = e->left->right->left;
	    memia->scaler = e->left->right->right->u.i;
	}
	else if (e->right->o == oConstant) {
	    memia->K = e->right->u.i;
	    memia->Kbase = e->right->right;
	    memia->scaler = 0;
	    if (e->left->o == oADD) {
		memia->e1 = e->left->left;
		memia->e2 = e->left->right;
		e->right_first = e->left->right_first;
	    }
	    else {
		memia->e1 = e->left;
		memia->e2 = NULL;
	    }
	}
	else {
	    memia->K = 0;
	    memia->Kbase = NULL;
	    memia->scaler = 0;
	    memia->e1 = e->left;
	    memia->e2 = e->right;
	}
	e->u.p = memia;
	e->o = oMEMIA;
	return yes;
}




/*------------------ Assigning registers to expresssions: -------------------*/
#define NUM_IREGS       5
#define NUM_FREGS       8


/*  In assigning registers to expressions, we have a 3-pass algorithm:

    Pass 1:  Here we ask each expression how many registers it will need to
        compute its value, and thereby decide what order to compute the
	child expressions.  We want to minimise use of registers by computing
	the sub-expression that uses more registers first.  The only purpose
	of this pass is to decide whether to compute the right child or
	left child first.  If this pass is performed incorrectly, the worst
	that can happen is suboptimal use of registers.

    Pass 2:  Here we allocate registers to subexpressions.  We simulate the
	actual running of the program: we traverse the expression tree in the
	same order that execution will go in the compiled code.   On the way
	down in the recursion, we designate our 'preferred destination', and
	on the way back up we actually allocate either this or another destination
	to the expression.  The parent then has the option of overriding this.
		We oeep track of which registers are holding results of previous 
	subexpressions.  If we ever need a specific register, we 'throw' that
	subexpression 'out' of that register, by changing its destination to
	the stack.  If we ever need a register (any general register) and all
	general registers are occupied, then we 'throw out' the oldest
	expression to occupy a register.

    Pass 3:  This pass is less important:  it goes through expressions that
	have been thrown out of specialist registers because another expression
	needed that register for a specific purpose, (e.g. the AX register)
	and tries to give them another general-purpose register instead of
	the stack.


*/



static void RegisterRecursePass1(Expr* e)
/* Pass 1:  Work out how many registers of each type, and   */
/* what special registers each sub-expression needs, and    */
/* therefore what's the optimum order to do the operands.   */
/*							    */
/* During this pass, we also recognise any 'kMEMIA'	    */
/* patterns:  where we can use an SIB byte to optimise	    */
/* the code. */
{       Expr *sub, *sub1, *sub2;
	int i,air,afr,bir,bfr;
	exprf_type ef;

        switch (e->o) {
            case oUnknown:
            case oPUSHSPPLUS:
            case oConstant: e->numiregs = 0;
                            e->numfregs = 0;
                            e->needregs = 0;
                            break;

            case oLocalVar: if (e == &frame_register)
				return;
			    RecogniseMemia(e);
			    RegisterRecursePass1(e);
                            break;

            case Unaries:   RegisterRecursePass1(e->left);
                            e->numiregs = e->left->numiregs;
                            e->numfregs = e->left->numfregs;
                            e->needregs = e->left->needregs;
                            break;

            case oFUNC:     ef = e->u.func;
			    for (i=ef->arity-1; i >= 0; i--)
				RegisterRecursePass1(ef->param[i]);
			    RegisterRecursePass1(ef->func);
			    e->numfregs = 8;
			    e->numiregs = 6;
			    e->needregs = 255;
			    /* A function call could potentially overwrite all registers. */
			    break;

                            /* else e->numiregs = 1, e->numfregs = 0;
                            e->needregs = (1<<reg_AX);
                            for (i=0; i < ef->arity; i++) {
                                RegisterRecursePass1(sub=ef->param[i]);
                                if (sub->numiregs > e->numiregs)
                                    e->numiregs = sub->numiregs;
                                if (sub->numfregs > e->numfregs)
                                    e->numfregs = sub->numfregs;
                                e->needregs |= sub->needregs;
                            }
                            break;*/

            case oCOMMA:    sub = e->left;
                            RegisterRecursePass1(sub);
                            air = sub->numiregs;
                            afr = sub->numfregs;
                            sub = e->right;
                            RegisterRecursePass1(sub);
                            bir = sub->numiregs;
                            bfr = sub->numfregs;
                            e->numiregs = air > bir ? air : bir;
                            e->numfregs = afr > bfr ? afr : bfr;
                            e->needregs = e->left->needregs | e->right->needregs;
                            break;

            case oTERNARY:  RegisterRecursePass1(e->left);
                            e->numiregs = e->left->numiregs;
                            e->numfregs = e->left->numfregs;
                            e->needregs = e->left->needregs;
                            RegisterRecursePass1(e->right->left);
                            e->numiregs = max(e->left->numiregs, e->right->left->numiregs);
                            e->numfregs = max(e->left->numfregs, e->right->left->numfregs);
                            e->needregs |= e->right->left->needregs;
                            RegisterRecursePass1(e->right->right);
                            e->numiregs = max(e->left->numiregs, e->right->left->numiregs);
                            e->numfregs = max(e->left->numfregs, e->right->left->numfregs);
                            e->needregs |= e->right->left->needregs;
                            break;

            case oBITFIELD: if (e->u.bitfield->operand == NULL) {
				RegisterRecursePass1(e->u.bitfield->addr);
				e->needregs = e->u.bitfield->addr->needregs;
				e->numiregs = e->u.bitfield->addr->numiregs;
				e->numfregs = e->u.bitfield->addr->numfregs;
				break;
			    }
			    RegisterRecursePass1(e->u.bitfield->addr);
                            RegisterRecursePass1(e->u.bitfield->operand);
                            e->needregs = e->u.bitfield->addr->needregs | e->u.bitfield->operand->needregs;
			    air = e->u.bitfield->addr->numiregs;
			    bir = e->u.bitfield->operand->numiregs;
                            e->numiregs = air > bir ? air : bir;
                            e->numfregs = e->u.bitfield->operand->numfregs;
                            break;

            case oPUSH:	    RegisterRecursePass1(e->left);
                            e->needregs = e->left->needregs;
            		    e->numiregs = e->left->numiregs;
            		    e->numfregs = e->left->numfregs;
                            if (e->right) {
            			RegisterRecursePass1(e->right);
                                e->needregs |= e->right->needregs;
                                if (e->right->numiregs > e->numiregs)
                                    e->numiregs = e->right->numiregs;
                                if (e->right->numiregs > e->numiregs)
                                    e->numfregs = e->right->numfregs;
                            }
                            e->right_first = no;
                            break;

            case oMOD:
            case oDIV:
            case oAMOD:
            case oADIV:     if (not IsFloat(e)) {
                                /* Integer division must be AX to register or AX to memory. */
                                sub = e->left;
                                if (sub->o == oConstant) {
                                    sub->numiregs = 1;
                                    RegisterRecursePass1(e->right);
                                    e->numfregs = 0;
                                    e->right_first = yes;
                                    e->numiregs = max(e->right->numiregs, 3);
                                    e->needregs = (1<<reg_AX) | (1<<reg_DX);
                                    break;
                                }
                                else {
                                    sub = e->right;
                                    if (sub->o == oConstant) {
                                        sub->numiregs = 1;
                                        RegisterRecursePass1(e->left);
                                        e->right_first = no;
                                        e->numfregs = 0;
                                        e->numiregs = max(e->left->numiregs, 3);
                                        e->needregs = (1<<reg_AX) | (1<<reg_DX);
                                        break;
                                    }
                                }
                            }
                            else goto DEFAULT_BINARY;

	    case oADD:	    if (RecogniseMemia(e))
				RegisterRecursePass1(e);
                            else goto DEFAULT_BINARY;
			    break;

	    case oMEMIA:    memia_type memia;
			    memia = (memia_type)e->u.p;
			    sub1 = memia->e1;
			    sub2 = memia->e2;
			    if (sub1 and sub2)
				goto HAVE_SUBS;
			    if (sub1 == NULL)
				sub1 = sub2, sub2 = NULL;
			    RegisterRecursePass1(sub1);
                            e->numiregs = sub1->numiregs ? sub1->numiregs : 1;
                            e->numfregs = sub1->numfregs;
                            e->needregs = sub1->needregs;
			    break;

            default:        DEFAULT_BINARY:
			    sub1 = e->left;
			    sub2 = e->right;
			    HAVE_SUBS:
			    RegisterRecursePass1(sub1);
                            air = sub1->numiregs;
                            afr = sub1->numfregs;
                            e->needregs = sub1->needregs;
			    RegisterRecursePass1(sub2);
                            bir = sub2->numiregs;
                            bfr = sub2->numfregs;
                            e->needregs |= sub2->needregs;
                            e->right_first = (5*bir + bfr > 5*air + afr);
                            /* General-purpose registers are in higher demand. */
                            if (IsFloat(e)) {
                                if (e->right_first)
                                    bfr++;
                                else afr++;
                            }
                            else {
                                if (e->right_first)
                                    bir++;
                                else air++;
                                if (e->o == oMUL)
                                    e->needregs |= (1<<reg_AX);
                            }
                            e->numiregs = air > bir ? air : bir;
                            e->numfregs = afr > bfr ? afr : bfr;
                            break;
        }
}




static Expr* Occupier[reg_maxrealregs];
static int e_idx;


static void SelectRegister(Expr* e, reg_enum prefers, reg_enum requires)
/* Select a suitable register for this general or floating-point value. */
/* If the register we want is occupied, then set its occupier to use    */
/* the stack instead. */
{       reg_enum reg, reg0, reg1, best_reg;

        assert(prefers != reg_discard);
	if (requires == reg_operand)
	    requires = reg_unknown;
        e->idx = ++e_idx;
        if (requires and requires < reg_somegeneral) {
            /* Do we need a special-purpose register? */
            if (requires == reg_stack or requires == reg_fpstack) {
                e->storereg = e->workreg = requires;
                return;
            }
            if (Occupier[requires])
                Occupier[requires]->storereg = reg_stack;
            Occupier[requires] = e;
            e->storereg = e->workreg = requires;
            return;
        }

        /* Cycle thru to see if there are any free registers. */
        if (prefers and prefers != reg_somegeneral and Occupier[prefers] == NULL) {
            if (prefers == reg_stack) {
                e->storereg = reg_stack;
                return;
            }
            Occupier[prefers] = e;
            e->storereg = e->workreg = prefers;
            return;
        }
        if (IsFloat(e))
            reg0 = reg_R0, reg1 = reg_R7;
        else reg0 = reg_BX, reg1 = reg_AX;
        for (reg=reg0; reg <= reg1; ((uchar&)reg)++) {
            if (Occupier[reg] == NULL) {
                Occupier[reg] = e;
                e->storereg = e->workreg = reg;
                return;
            }
        }

        /* No? Well then, find the oldest one and steal it. */
        if (prefers and prefers != reg_somegeneral) {
            Occupier[prefers]->storereg = reg_stack;
            Occupier[prefers] = e;
            e->storereg = e->workreg = prefers;
            return;
        }
        best_reg = reg0;
        for (reg=(reg_enum)(reg0+1); reg <= reg1; ((uchar&)reg)++) {
            if (Occupier[reg]->idx < Occupier[best_reg]->idx)
                best_reg = reg;
        }
        reg = best_reg;
        Occupier[reg]->storereg = reg_stack;
        Occupier[reg] = e;
        e->storereg = e->workreg = reg;
}


static void ReleaseRegister(Expr* e)
/* In this function, we release all the registers used by the */
/* instruction corresponding to this Expr. Most commonly */
/* it's just a single register, but in the case of MEMIA's,   */
/* there can be up to 2 registers we need to release. */
{
	if (e->storereg == reg_operand) {
	    if (e->o == oMEMIA) {
		memia_type memia=(memia_type)e->u.p;
		if (memia->e1 and memia->e1 != &frame_register)
		    ReleaseRegister(memia->e1);
		if (memia->e2)
		    ReleaseRegister(memia->e2);
	    }
	    else if (e->o == oDER or e->o == oCONVERT) {
                ReleaseRegister(e->left);
            }
	    else if (e->o == oConstant)
		;
	    else assert(false);
	    return;
	}
	else if (e->storereg == reg_stack or e->storereg == reg_discard
		or e->storereg == reg_fpstack)
	    ;
	else if (IsGeneral(e->storereg)) {
            assert(Occupier[e->storereg] == e);
	    Occupier[e->storereg] = NULL;
	}
	else assert(false);
}


static void ClearRegister(reg_enum reg)
{
        if (Occupier[reg]) {
            Occupier[reg]->storereg = reg_stack;
            Occupier[reg] = NULL;
	}
}


static void TransferRegister(Expr* from, Expr* to)
{
	if (from->storereg == reg_operand) {
	    to->storereg = to->workreg = reg_operand;
	    return;
	}
        else if (from->storereg == reg_stack or from->storereg == reg_discard 
	    or from->storereg == reg_fpstack)
            return;
        assert(Occupier[from->storereg] == from);
        to->storereg = to->workreg = from->storereg;
        Occupier[to->storereg]  = to;
}


static reg_enum VacantRegister(void)
/* Choose a vacant general register. */
{	reg_enum reg;

        for (reg=first_general; reg <= last_general; ((int&)reg)++) {
            if (Occupier[reg] == NULL)
		return reg;
        }
	assert(false);	    /* We've run out of registers! This should not happen. */
	return reg_unknown;
}


extern Expr* debuge;

static void RegisterRecursePass2(Expr* e, reg_enum dest)
/* Pass 2:  Actually allocate the registers.  Each expressions decides	*/
/* for itself what place to store its value in, based on the instruct-	*/
/* ions given in 'dest'.  If dest==reg_discard, then we don't need to	*/
/* store the value anywhere (just perform the requested operation).	*/
/* If dest==reg_somegeneral, then it means we need some	general		*/
/* register.  If dest==reg_operand, then it means we want either a	*/
/* register or a oMEMIA or a oConstant.  If dest=regAX etc., then this	*/
/* means we require the result to be put in this specific register. If	*/
/* dest==reg_stack, then we want the result put on the stack.		*/
{       Expr *sub1, *sub2;
	reg_enum subdest;
        exprf_type ef;
        int i;

        switch (e->o) {
            case oUnknown:
            case oConstant: if (dest == reg_discard or dest == reg_operand or dest == reg_stack
				or dest == reg_fpstack)
				e->storereg = dest;
			    else if (IsFloat(e))
                                e->storereg = reg_fpstack;
			    else SelectRegister(e, dest, reg_unknown);
                            break;

            case oLocalVar: if (e == &frame_register)
				return;
			    assert(false);	// Should have been converted to a memia.
			    break;

            case oPUSHSPPLUS:
            		    if (e->u.i == 0) {
                            	e->storereg = reg_stack;
                                break;
                            }
            		    SelectRegister(e, reg_somegeneral, reg_unknown);
                            ReleaseRegister(e);		// We only need it temporarily
                            e->workreg = e->storereg;
                            e->storereg = reg_stack;
                            break;

            case oDER:      if (IsFloat(e)) {
                                if (e->left->o == oConstant) {
				    e->left->storereg = reg_operand;
				}
                                else if (e->left->o == oMEMIA) {
                                    RegisterRecursePass2(e->left, reg_operand);
				    e->left->storereg = reg_operand;
                                }
				else {
                                    RegisterRecursePass2(e->left, reg_somegeneral);
				    ReleaseRegister(e->left);
				}
				e->storereg = (dest == reg_operand or dest == reg_stack)
				     ? dest : reg_fpstack;
                            }
                            else if (e->left->o == oConstant) {
				e->left->storereg = reg_operand;
				if (dest == reg_operand)
				    e->storereg = reg_operand;
				else if (dest == reg_stack)
				    e->storereg = reg_stack;
                                else SelectRegister(e, dest, reg_unknown);
                            }
                            else if (TpSize(e->tp) > 4 and TypeSize(e->type) > 4) {
                                SelectRegister(e, reg_unknown, reg_SI);
				ReleaseRegister(e);
                                SelectRegister(e, reg_unknown, reg_DI);
                                ReleaseRegister(e);
                                SelectRegister(e, reg_unknown, reg_CX);
                                ReleaseRegister(e);
				RegisterRecursePass2(e->left, reg_SI);
				if (e->left->storereg)
				    TransferRegister(e->left, e);
				else SelectRegister(e, reg_unknown, reg_SI);
				e->storereg = (dest == reg_stack) ? reg_stack : reg_discard;
                            }
			    else if (dest == reg_discard) {
				RegisterRecursePass2(e->left, reg_somegeneral);
				e->storereg = reg_discard;
				break;	    // This should only happen with comparison operators (where we take 
				// the flags results only) because of "Code has no effect" checking.
			    }
			    else if (dest == reg_operand and (e->left->o == oMEMIA or
					e->left->o == oConstant)) {
				RegisterRecursePass2(e->left, reg_operand);
				e->storereg = e->workreg = reg_operand;
			    }
			    else if (IsGeneral(dest) or dest == reg_somegeneral) {
				if (e->left->o == oConstant or e->left->o == oMEMIA)
				    subdest = reg_operand;
				else subdest = dest;
				RegisterRecursePass2(e->left, subdest);
				if (IsGeneral(e->left->storereg))
				    TransferRegister(e->left, e);
				else if (IsGeneral(dest))
				    SelectRegister(e, reg_unknown, dest);
				else SelectRegister(e, reg_unknown, reg_unknown);
			    }
			    else if (dest == reg_operand) {
				RegisterRecursePass2(e->left, reg_somegeneral);
				if (dest == reg_operand)
                                    ;   // We'll need to oeep it until
                                    // the operand is used.
                                else
                                    ReleaseRegister(e->left);
				e->storereg = dest;
			    }
			    else if (dest == reg_stack) {
				RegisterRecursePass2(e->left, 
					(e->left->o == oMEMIA or e->left->o == oConstant) ? reg_operand : reg_somegeneral);
				ReleaseRegister(e->left);
				e->storereg = dest;
			    }
			    else if (IsGeneral(dest)) {
				RegisterRecursePass2(e->left, reg_operand);
				ReleaseRegister(e->left);
				SelectRegister(e, reg_unknown, dest);	// Must have it.
			    }
			    else assert(false);
                            break;

            case oNEG: case oNOT:
            case oSIN: case oCOS: case oTAN: case oATAN: case oSQRT:
                            if (dest == reg_stack or dest == reg_discard)
                                subdest = reg_somegeneral;
                            RegisterRecursePass2(e->left, subdest);
			    if (dest == reg_stack)
				e->storereg = dest;
			    else if (e->left->storereg == reg_fpstack)
				e->storereg = reg_fpstack;
			    else {
				if (e->left->storereg == reg_operand)
				    SelectRegister(e->left, dest, reg_unknown);
				TransferRegister(e->left, e);
			    }
			    break;

	    case oCONVERT:  if (IsFloat(e) and IsFloat(e->left)) {
				/* FPU to FPU: */
				RegisterRecursePass2(e->left, reg_fpstack);
				if (dest == reg_fpstack or dest == reg_stack
					or dest == reg_discard)
				    e->storereg = dest;
				else assert(false);
			    }
			    else if (IsFloat(e)) {
				/* ALU to FPU: */
				assert(e->left->tp != tp_char);
				RegisterRecursePass2(e->left, reg_operand);
				if (dest == reg_fpstack or dest == reg_stack
					or dest == reg_discard)
				    e->storereg = dest;
                                else if (dest == reg_operand)
                                    stop();
				else assert(false);
			    }
			    else if (IsFloat(e->left))  {
				/* FPU to ALU: */
				assert(e->tp != tp_char);
				RegisterRecursePass2(e->left, reg_fpstack);
				SelectRegister(e, reg_unknown, dest);
			    }
			    else {
				/* ALU to ALU: */
				if (IsGeneral(dest)) {
				    if (e->left->o == oDER)
					subdest = reg_operand;
				    else subdest = dest;
				}
				else if (dest == reg_operand and TpSize(e->tp) > TpSize(e->left->tp))
				    subdest = reg_somegeneral;
				else if (dest == reg_stack or dest == reg_discard)
				    subdest = reg_somegeneral;
				else subdest = dest;
				RegisterRecursePass2(e->left, subdest);
				if (dest == reg_stack or dest == reg_discard) {
				    ReleaseRegister(e->left);
				    e->storereg = dest;
				}
				else if (e->left->storereg == reg_operand) {
				    if (IsGeneral(dest))
				    	SelectRegister(e, dest, dest);
				    else SelectRegister(e, reg_unknown, subdest);
				}
				else if (IsGeneral(dest) and e->left->storereg != dest) {
				    ReleaseRegister(e->left);
				    SelectRegister(e, reg_unknown, dest);
				}
				else TransferRegister(e->left, e);
				if (TpSize(e->tp) > TpSize(e->left->tp))
				    assert(IsGeneral(e->storereg) or IsGeneral(e->left->storereg));
			    }
			    break;

	    case oNOP1:     RegisterRecursePass2(e->left, dest);
                            if (e->left->storereg == reg_discard)
                                e->storereg = reg_discard;
			    else TransferRegister(e->left, e);
			    break;

	    case oFUNC:     ef = e->u.func;
			    for (i=first_general; i <= last_general; i++)
				ClearRegister((reg_enum)i);
				/* When we make a function call, we make no assumptions */
				/* about registers oeeping their values.  So everything */
				/* we're storing must go onto the stack. */
                            for (i=ef->arity-1; i >= 0; i--) {
                                sub1 = ef->param[i];
				RegisterRecursePass2(sub1, reg_stack);
				assert(sub1->storereg == reg_stack);
				if (IsGeneral(sub1->storereg))
				    sub1->workreg = sub1->storereg;
				sub1->storereg = reg_stack;
				if (sub1->o == oCOMMA and sub1->right->storereg == reg_operand)
				    sub1->right->storereg = reg_stack;
                            }
			    RegisterRecursePass2(ef->func, reg_operand);
			    ReleaseRegister(ef->func);
			    if (e->tp == tp_void or e->tp == 0)
				e->storereg = reg_discard;
                            else if (dest == reg_SI and e->tp == tp_class
                            	and TypeSize(e->type) > 4) {
                                SelectRegister(e, reg_unknown, reg_SI);
                            }
			    else {
                                SelectRegister(e, reg_unknown,
                                        (IsFloat(e)) ? reg_fpstack : reg_AX);
				e->workreg = e->storereg;
				if (dest == reg_discard)
				    ReleaseRegister(e), e->storereg = reg_discard;
				else if (dest == reg_stack)
				    ReleaseRegister(e), e->storereg = reg_stack;
			    }
                            break;

            case oMOD:
            case oDIV:      if (IsFloat(e))
                                goto BINARY_FP;
                            if (e->right->o == oConstant) {
                                RegisterRecursePass2(e->left, reg_AX);
                                assert(IsGeneral(e->left->storereg));
                                SelectRegister(e->right, reg_unknown, reg_unknown);
                            }
                            else {
                                if (e->right_first) {
                                    RegisterRecursePass2(e->right, reg_operand);
                                    RegisterRecursePass2(e->left, reg_AX);
                                }
                                else {
                                    RegisterRecursePass2(e->left, reg_AX);
                                    RegisterRecursePass2(e->right, reg_operand);
                                }
                            }
                            ReleaseRegister(e->right);
                            ReleaseRegister(e->left);
			    if (e->o == oDIV) {
				ClearRegister(reg_DX);
				SelectRegister(e, reg_unknown, reg_AX);
			    }
			    else {
				ClearRegister(reg_AX);
				SelectRegister(e, reg_unknown, reg_DX);
			    }
			    if (e->storereg != dest and (dest == reg_discard or dest == reg_stack)) {
				ReleaseRegister(e);
				e->storereg = dest;
			    }
                            break;

            case oAMOD:
            case oADIV:     if (IsFloat(e))
                                goto BINARY_FP;
                            if (e->right->o == oConstant) {
				e->right->storereg = reg_operand;
                                RegisterRecursePass2(e->left, reg_operand);
                                //SelectRegister(e->right, reg_unknown, reg_unknown);
				/* We can do this either with a register or with local-static memory. */
                            }
                            else if (e->left->o == oConstant) {
				e->left->storereg = reg_operand;
                                RegisterRecursePass2(e->right, reg_operand);
                            }
                            else {
                                if (e->right_first) {
                                    RegisterRecursePass2(e->right, reg_operand);
                                    RegisterRecursePass2(e->left, reg_operand);
                                }
                                else {
                                    RegisterRecursePass2(e->left, reg_operand);
                                    RegisterRecursePass2(e->right, reg_operand);
                                }
                            }
                            ClearRegister(reg_AX);
                            ClearRegister(reg_DX);
                            ReleaseRegister(e->right);
			    ReleaseRegister(e->left);
			    if (e->o == oADIV) {
				SelectRegister(e, reg_unknown, reg_AX);
			    }
			    else {
				SelectRegister(e, reg_unknown, reg_DX);
			    }
			    if (dest == reg_discard or dest == reg_stack) {
				ReleaseRegister(e);
				e->storereg = dest;
			    }
                            break;

	    case oBITFIELD: if (dest == reg_stack or dest == reg_discard)
                                subdest = reg_unknown;
			    else subdest = dest;
			    if (e->u.bitfield->operand == NULL) {
                                RegisterRecursePass2(e->u.bitfield->addr, subdest);
                                if (IsGeneral(e->u.bitfield->addr->storereg))
				    TransferRegister(e->u.bitfield->addr, e);
				else SelectRegister(e, subdest, reg_unknown);
				break;
			    }
                            if (e->right_first) {
                                RegisterRecursePass2(e->u.bitfield->operand, subdest);
                                RegisterRecursePass2(e->u.bitfield->addr, subdest);
                            }
                            else {
                                RegisterRecursePass2(e->u.bitfield->addr, subdest);
                                RegisterRecursePass2(e->u.bitfield->operand, subdest);
                            }
                            if (dest == reg_discard) {
                                e->storereg = reg_discard;
                                ReleaseRegister(e->u.bitfield->addr);
                                ReleaseRegister(e->u.bitfield->operand);
                            }
                            else {
				e->storereg = reg_stack;
			    }
                            break;

            case oPUSH:	    e->storereg = reg_stack;
            		    e->workreg = reg_unknown;
                            RegisterRecursePass2(e->left,reg_stack);
                            if (e->right)
                            	RegisterRecursePass2(e->right,reg_stack);
                            break;

	    case oASSIGN:   if (e->o == oASSIGN and e->tp == tp_class and TypeSize(e->type) > 4) {
				if (e->right_first) {
				    if (e->right->o == oDER) {
                                    	e->right->storereg = reg_operand;
					RegisterRecursePass2(e->right->left, reg_SI);
                                    }
				    else if (e->right->o == oFUNC or e->right->o == oPUSH)
					RegisterRecursePass2(e->right, reg_SI);
				    else assert(false);
				    RegisterRecursePass2(e->left, reg_DI);
				}
				else {
				    RegisterRecursePass2(e->left, reg_DI);
				    if (e->right->o == oDER) {
					RegisterRecursePass2(e->right->left, reg_SI);
                                        e->right->storereg = reg_operand;
                                    }
				    else if (e->right->o == oFUNC or e->right->o == oPUSH)
					RegisterRecursePass2(e->right, reg_SI);
				    else assert(false);
				}
                                SelectRegister(e, reg_unknown, reg_CX);
                                ReleaseRegister(e);
                                ReleaseRegister(e->left);
                                if (e->right->o == oDER)
                                    ReleaseRegister(e->right->left);
                                else ReleaseRegister(e->right);
				if (dest == reg_discard)
				    e->storereg = e->workreg = reg_discard;
				else e->storereg = e->workreg = reg_stack;
				break;
                            }
			    else goto ASSIGN_OP;

	    case oAADD:
            case oASUB:
            case oAMUL:
            case oAXOR:
            case oAOR:
            case oAAND:
            case oAADDAFTER:ASSIGN_OP:
			    if (IsFloat(e))
                                goto BINARY_AFP;
                            if (dest == reg_stack or dest == reg_discard)
                                subdest = reg_somegeneral;
			    else subdest = dest;
                            if (e->right_first)
				sub1 = e->right, sub2 = e->left;
			    else sub2 = e->right, sub1 = e->left;
                            if (sub1->o == oConstant) {
				if (e->right_first and e->o == oASSIGN and
					(dest == reg_somegeneral or IsGeneral(dest)))
				    RegisterRecursePass2(sub1, subdest);
				else sub1->storereg = reg_operand;
			    }
			    else if (sub1->o == oMEMIA)
				RegisterRecursePass2(sub1, sub1 == e->left ? reg_operand : subdest);
			    else RegisterRecursePass2(sub1, subdest);
                            if (sub2->o == oConstant) {
				if (not e->right_first and e->o == oASSIGN and
					(dest == reg_somegeneral or IsGeneral(dest)))
				    RegisterRecursePass2(sub2, subdest);
				else sub2->storereg = reg_operand;
			    }
			    else if (sub2->o == oMEMIA)
				RegisterRecursePass2(sub2, sub2 == e->left ? reg_operand : subdest);
			    else RegisterRecursePass2(sub2, subdest);
                            if (e->o == oAMUL) {
                                /* We need to allocate a working register. */
                                if (IsGeneral(e->right->storereg))
                                    TransferRegister(e->right, e);
                                else if (e->right->o == oConstant) {
                                    SelectRegister(e->right, IsGeneral(subdest) ? subdest : reg_unknown, reg_unknown);
                                    TransferRegister(e->right, e);
                                }
                                else {
                                    SelectRegister(e, IsGeneral(subdest) ? subdest : reg_unknown, reg_unknown);
				    ReleaseRegister(e->right);
                                }
                                ReleaseRegister(e->left);
                            }
                            else if (dest == reg_discard) {
                                e->storereg = reg_discard;
                                ReleaseRegister(e->left);
                                ReleaseRegister(e->right);
                            }
                            else {
                                if (e->o == oASSIGN and IsGeneral(e->right->storereg)) {
                                    TransferRegister(e->right, e);
                                    ReleaseRegister(e->left);
                                }
				else if (e->o == oAADDAFTER) {
                                    SelectRegister(e, reg_unknown, dest);
                                    ReleaseRegister(e->left);
                                    ReleaseRegister(e->right);
				}
                                else {
                                    if (IsGeneral(e->right->storereg)) {
                                        TransferRegister(e->right, e);
                                        ReleaseRegister(e->left);
                                    }
                                    else if (IsGeneral(e->left->storereg)) {
                                        TransferRegister(e->left, e);
					// don't need to release e->right, it's not a general.
                                    }
                                    else {
                                        SelectRegister(e, reg_unknown, dest);
                                        ReleaseRegister(e->left);
                                        ReleaseRegister(e->right);
                                    }
                                }
                            }
			    if (dest == reg_stack) {
				e->workreg = e->storereg;
				e->storereg = reg_stack;
			    }
                            break;

	    case oMEMIA:    memia_type memia;
			    memia = (memia_type)e->u.p;
			    if (e->right_first)
				sub1 = memia->e2, sub2 = memia->e1;
			    else sub1 = memia->e1, sub2 = memia->e2;
			    if (sub1 and sub2)
				goto HAVE_SUBS;
			    else if (sub1 == NULL)
				sub1 = sub2, sub2 = NULL;
			    if (sub1 != &frame_register) {
				RegisterRecursePass2(sub1, IsGeneral(dest) ? dest : reg_somegeneral);
				if (dest == reg_discard or dest == reg_operand)
				    e->storereg = reg_operand;
				    // The memia's e1 & e2 hold onto the registers until
				    // we call ReleaseRegister on the memia.
				else if (dest == reg_stack) {
				    e->storereg = reg_stack;
				    ReleaseRegister(sub1);
				}
				else TransferRegister(sub1, e);
			    }
			    else {
				if (dest == reg_discard)
				    ;
				else if (dest == reg_operand)
				    e->storereg = e->workreg = reg_operand;
				else {
				    SelectRegister(e, reg_unknown, IsGeneral(dest) ? dest : reg_unknown);
				    if (dest == reg_stack) {
					e->workreg = e->storereg;
					ReleaseRegister(e);
					e->storereg = reg_stack;
				    }
				}
			    }
                            break;

	    case oLT: case oGT:
            case oEQ: case oNE:
            case oLE: case oGE:
			    if (not IsFloat(e->left))
                            	goto BINARY_INT;
                            if (e->right_first)
                                sub1 = e->right, sub2 = e->left;
                            else sub1 = e->left, sub2 = e->right;
                            RegisterRecursePass2(sub1, reg_fpstack);
                            RegisterRecursePass2(sub2, reg_fpstack);
			    ReleaseRegister(sub2);
			    ReleaseRegister(sub1);
			    if (dest == reg_stack or dest == reg_discard)
                            	e->storereg = dest;
                            else SelectRegister(e, dest, reg_unknown);
                            break;

            case oADD:
            case oSUB:
            case oAND:
            case oOR:
            case oXOR:
            case oMUL:	    if (IsFloat(e))
				goto BINARY_FP;
            		    else goto BINARY_INT;

                            BINARY_INT:
                            if (e->right_first)
                                sub1 = e->right, sub2 = e->left;
                            else sub1 = e->left, sub2 = e->right;

			    /* Our rules for allocating registers are:

			    1. We can completely avoid the use of registers in a very special set
				of circumstances:
				The parent wants to discard the result (must be a comparison)
				AND it's "mem cmp imm" or "imm cmp mem"
				AND it's an 8-bit or 32-bit integer operation.
					    (NB:- see also the OP_ASSIGN section).

			    2. Otherwise, we need a register for sub1 and for the result,

			    3. And we need one for sub2 unless sub2 is mem or imm AND
				we're doing an 8-bit or 32-bit operation.

			    */

			    HAVE_SUBS:
			    if (dest == reg_discard and
				    (sub1->o == oDER and sub1->left->o == oConstant and sub2->o == oConstant)
				    and (TpSize(e->tp) == 4 or TpSize(e->tp) == 1)) {
				/* It's a registerless operation! */
				e->storereg = sub1->storereg = reg_discard;
				sub1->left->storereg = sub2->storereg = reg_operand;
				break;
			    }
			    RegisterRecursePass2(sub1, IsGeneral(dest) ? dest : reg_somegeneral);
			    if ((sub2->o == oConstant or (sub2->o == oDER and sub2->left->o == oConstant))
				and (TpSize(e->tp) == 4 or TpSize(e->tp) == 1)
				and not dest) {
				sub2->storereg = reg_operand;	// This doesn't need a register -
				// it'll be a registerless instruction.
			    }
			    else {
				RegisterRecursePass2(sub2,
					e->o == oMUL or e->o == oMEMIA or sub2->o == oMEMIA
						? reg_somegeneral : reg_operand);
					/* Why do these 3 cases require a register?
					1. Multiplication is always done register-to-register
					2. Memia's are always of the form:  reg1 + scale*reg2 + constant
					3. We can't use "x+k" as an operand, only "[x+k]".
					*/
			    }
                            if (e->o == oSUB) {
				/* oSUB is the only non-commutative operation: */
                                if (e->left->storereg) {
                                    ReleaseRegister(e->right);
                                    TransferRegister(e->left, e);
                                }
                                else {
                                    ReleaseRegister(e->left);
                                    TransferRegister(e->right, e);
                                }
                            }
			    else {
				if (sub1->storereg == reg_stack) {
       				    TransferRegister(sub2, e);
				}
				else {
       				    TransferRegister(sub1, e);
				    ReleaseRegister(sub2);
				}
			    }
			    if (dest == reg_discard or dest == reg_stack) {
				e->workreg = e->storereg;
				ReleaseRegister(e);
				e->storereg = dest;
			    }
                            break;


			    BINARY_FP:
			    /* Allocate floating-point registers to a non-assignment */
			    /* arithmetic operation. */
                            if (e->right_first)
                                sub1 = e->right, sub2 = e->left;
                            else sub1 = e->left, sub2 = e->right;
                            RegisterRecursePass2(sub1, reg_fpstack);
                            RegisterRecursePass2(sub2, reg_fpstack);
			    ReleaseRegister(sub2);
			    ReleaseRegister(sub1);
			    e->storereg = (dest == reg_stack) ? reg_stack : reg_fpstack;
			    break;


			    BINARY_AFP:
			    /* Allocate floating-point registers to an assignment arithmetic operation. */
                            if (e->right_first) {
				RegisterRecursePass2(e->right, reg_fpstack);
				if (e->left->o == oConstant)
                                    e->left->storereg = reg_operand;
                                else RegisterRecursePass2(e->left,
                                	e->left->o == oMEMIA or e->left->o == oLocalVar ?
                                        reg_operand : reg_somegeneral);
			    }
			    else {
				if (e->left->o == oConstant)
                                    e->left->storereg = reg_operand;
                                else RegisterRecursePass2(e->left,
                                	e->left->o == oMEMIA or e->left->o == oLocalVar ?
                                        reg_operand : reg_somegeneral);
				RegisterRecursePass2(e->right, reg_fpstack);
			    }
			    ReleaseRegister(e->left);
			    if (dest == reg_fpstack or dest == reg_stack or
                                dest == reg_operand or dest == reg_discard) {
				ReleaseRegister(e->right);
				e->storereg = dest;
			    }
			    else TransferRegister(e->right, e);
			    break;

            case oSHL:
            case oSHR:      if (e->right_first) {
				if (e->right->o == oConstant)
				    e->right->storereg = reg_operand;
				else {
				    if (e->right->storereg != reg_CX)
					SelectRegister(e, reg_unknown, reg_CX);
				}
                                RegisterRecursePass2(e->left, IsGeneral(dest) ? dest : reg_somegeneral);
                            }
                            else {
                                RegisterRecursePass2(e->left, IsGeneral(dest) ? dest : reg_somegeneral);
				if (e->right->o == oConstant)
				     e->right->storereg = reg_operand;
				 else {
				    RegisterRecursePass2(e->right, reg_CX);
				    if (e->right->storereg != reg_CX)
					SelectRegister(e, reg_unknown, reg_CX);
				}
                            }
                            if (IsGeneral(e->left->storereg))
                                TransferRegister(e->left, e);
                            else SelectRegister(e, dest, reg_unknown);
                            ReleaseRegister(e->right);
			    if (dest == reg_stack) {
				e->workreg = e->storereg;
				e->storereg = reg_stack;
			    }
                            break;

            case oASHL:
            case oASHR:     if (e->right_first) {
                                RegisterRecursePass2(e->right, reg_CX);
                                if (e->right->workreg != reg_CX)
                                    SelectRegister(e, reg_unknown, reg_CX);
                                if (e->left->o == oConstant)
				    e->left->storereg = reg_operand;
                                else RegisterRecursePass2(e->left, (dest == reg_stack) ? reg_operand : dest);
                            }
                            else {
                                if (e->left->o == oConstant)
				    e->left->storereg = reg_operand;
                                else RegisterRecursePass2(e->left, (dest == reg_stack) ? reg_operand : dest);
                                RegisterRecursePass2(e->right, reg_CX);
                                if (e->right->workreg != reg_CX)
                                    SelectRegister(e, reg_unknown, reg_CX);
                            }
                            ReleaseRegister(e->right);
                            ReleaseRegister(e->left);
                            if (dest == reg_discard)
				e->storereg = reg_discard;
                            else SelectRegister(e, dest, reg_unknown);
                            break;

            case oCOMMA:    RegisterRecursePass2(e->left, reg_discard);
                            ReleaseRegister(e->left);
                            RegisterRecursePass2(e->right, dest);
			    if (e->right->storereg == reg_stack or 
				e->right->storereg == reg_discard or
				e->right->storereg == reg_operand)
				e->storereg = e->right->storereg;
                            else TransferRegister(e->right, e);
                            break;

            case oTERNARY:  RegisterRecursePass2(e->left,
                                IsComparison(e->left->o) ? reg_discard : reg_somegeneral);
                            ReleaseRegister(e->left);
                            RegisterRecursePass2(e->right->left,
                                                dest == reg_discard ? reg_operand
                                                : dest == reg_operand ? reg_somegeneral
                                                : dest);
                            if (e->right->left->storereg == reg_operand)
                                SelectRegister(e->right->left, dest, reg_unknown);
                            ReleaseRegister(e->right->left);
                            RegisterRecursePass2(e->right->right, e->right->left->storereg);
                            if (e->right->right->storereg == reg_operand)
                                SelectRegister(e->right->right, e->right->left->storereg, reg_unknown);
                            ReleaseRegister(e->right->right);
                            SelectRegister(e, reg_unknown, e->right->left->storereg);
                            break;

            case oANND:
            case oORR:      /* Only occur inside oTERNARY expressions. */
                            RegisterRecursePass2(e->left, reg_discard);
                            RegisterRecursePass2(e->right, reg_discard);
                            if (dest == reg_somegeneral)
                                SelectRegister(e, reg_somegeneral, reg_unknown);
                            else e->storereg = dest;
			    break;

            case oNOTT:     /* Only occurs inside oTERNARY expressions. */
                            RegisterRecursePass2(e->left, reg_discard);
                            if (dest == reg_somegeneral)
                                SelectRegister(e, reg_somegeneral, reg_unknown);
                            else e->storereg = dest;
                            break;

            default:        assert(false);
                            break;
        }
	assert(e->storereg != reg_unknown and e->storereg != reg_somegeneral);
}


static void UseRegister(Expr* e)
{       reg_enum reg=e->storereg;

        if (reg != reg_stack and reg < reg_maxrealregs and reg != reg_unknown)
            Occupier[reg] = e;
}


static void UnuseRegister(Expr* e)
{       reg_enum reg=e->storereg;

        if (reg != reg_stack and reg < reg_maxrealregs and reg != reg_unknown)
            Occupier[reg] = NULL;
}


static bool RegisterOccursInExpr(Expr* e, reg_enum reg)
/* Is 'reg' used within 'e'? */
{
	if (e == NULL)
	    return no;
        if (e->storereg == reg)
            return yes;
        switch (e->o) {
            case Unaries:   return RegisterOccursInExpr(e->left, reg);

            case Atomics:   return no;

            case Binaries:  if (e->o == oDIV and (reg == reg_AX or reg == reg_DX))
                                return yes;
                            /*if (e->o == oAMUL and reg == e->storereg)
                                return yes;*/
                            return RegisterOccursInExpr(e->left, reg) or
                                RegisterOccursInExpr(e->right, reg);

            case oBITFIELD: return RegisterOccursInExpr(e->u.bitfield->addr, reg) or
                                (e->u.bitfield->operand and
				    RegisterOccursInExpr(e->u.bitfield->operand, reg));

            case oPUSH:     return RegisterOccursInExpr(e->left, reg)
            			or (e->right and RegisterOccursInExpr(e->right, reg));

            case oFUNC:     return yes;
			    // We must assume that a function call destroys all
			    // values in registers, because we compile functions
			    // to use all registers (and not save any).
			    /*for (i=0; i < e->u.func->arity; i++)
                                if (RegisterOccursInExpr(e->u.func->param[i], reg))
                                    return yes;
                            return no;*/

	    case oMEMIA:    memia_type memia;
			    memia = (memia_type)e->u.p;
			    return RegisterOccursInExpr(memia->e1, reg) or RegisterOccursInExpr(memia->e2, reg);

            case oTERNARY:  if (RegisterOccursInExpr(e->left, reg))
                                return yes;
                            if (RegisterOccursInExpr(e->right->left, reg))
                                return yes;
                            if (RegisterOccursInExpr(e->right->right, reg))
                                return yes;
                            return no;

            default:        assert(false);
                            return no;
        }
}


static bool SpecialRegister(Expr* e)
/* This expression has a certain storereg.  Is it forced to use */
/* this register, or can we completely assign it to another	*/
/* register? */
{
	if (e->o == oFUNC)
	    return yes;
	if (e->o == oConstant or e->o == oLocalVar)
	    return no;
	if (IsFloat(e))
	    return no;
	if (e->o == oDIV or e->o == oMOD or e->o == oADIV or e->o == oAMOD)
	    return yes;
	return no;
}


static void UseSomeRegister(Expr* e, Expr* brother)
/* Try to select a register for 'e' that is not occupied */
/* and will not be overwritten by 'brother'. */
{       reg_enum reg, reg1, reg0;

        if (IsFloat(e))
            reg0 = reg_R0, reg1 = reg_R7;
        else reg0 = first_general, reg1 = last_general;
        for (reg=reg0; reg <= reg1; ((uchar&)reg)++) {
            if (Occupier[reg] == NULL) {
                if (not RegisterOccursInExpr(brother, reg)) {
                    Occupier[reg] = e;
                    e->storereg = reg;
                    return;
                }
            }
        }
}


static void RegisterRecursePass3(Expr* e)
/* Pass 3:  Try to minimise use of the stack resulting from clashes */
/* in the use of specialist registers.  This is also the pass in    */
/* which we allocate work registers. */
{       Expr *sub1, *sub2;
        exprf_type ef;
        int i;

        switch (e->o) {
            case Atomics:   break;

            case Unaries:   RegisterRecursePass3(e->left);
                            break;

            case Binaries:  if (e->right_first)
                                sub1 = e->right, sub2 = e->left;
                            else sub1 = e->left, sub2 = e->right;
			    HAVE_SUBS:
                            if (sub1->storereg == reg_stack and TpSize(sub1->tp) <= 4)
				UseSomeRegister(sub1, sub2);	    // Hopefully, we can
				// find some other register to be used instead of the stack.
			    else UseRegister(sub1);
                            assert(sub2->storereg != reg_stack or e->o == oCOMMA
                            		or sub2->o == oPUSH);
                            RegisterRecursePass3(sub1);
                            if (sub1->storereg == reg_stack and sub1->o != oBITFIELD)
                                /* This is the crux of the whole pass. */
                                /* Try to replace the destination of 'sub1' */
                                /* with a register not currently reserved   */
                                /* and not overwritten by 'sub2'. */
                                UseSomeRegister(sub1, sub2);
                            RegisterRecursePass3(sub2);
                            if (sub1->storereg == reg_stack and TpSize(sub1->tp) <= 4)
				sub1->workreg = VacantRegister();	// We are unfortunately
				// forced to use the stack. We will need a 'workreg' to pop
				// the value into after computing the operands but before doing
				// the actual operation.
                            UnuseRegister(sub1);
                            UnuseRegister(sub2);
                            break;

            case oBITFIELD: if (e->right_first)
				sub1 = e->u.bitfield->operand, sub2 = e->u.bitfield->addr;
                            else sub1 = e->u.bitfield->addr, sub2 = e->u.bitfield->operand;
                            UseRegister(sub1);
			    if (sub2 == NULL) {
				RegisterRecursePass3(sub1);
				UnuseRegister(sub1);
				sub1->storereg = IsGeneral(sub1->storereg) ? sub1->storereg : VacantRegister();
				break;
			    }
                            RegisterRecursePass3(sub1);
                            if (not IsGeneral(sub1->storereg))
                                UseSomeRegister(sub1, sub2);
                            if (not IsGeneral(e->u.bitfield->operand->storereg))
                                UseSomeRegister(e->u.bitfield->operand,
					e->u.bitfield->addr);
                            RegisterRecursePass3(sub2);
                            UnuseRegister(sub1);
                            UnuseRegister(sub2);
			    sub1->storereg = IsGeneral(sub1->storereg) ? sub1->storereg : VacantRegister();
                            break;

            case oPUSH:     RegisterRecursePass3(e->left);
                            if (e->right)
                            	RegisterRecursePass3(e->right);
                            break;

            case oFUNC:     ef = e->u.func;
                            for (i=ef->arity-1; i >= 0; i--)
                                RegisterRecursePass3(ef->param[i]);
                            break;

	    case oMEMIA:    memia_type memia;
			    memia = (memia_type)e->u.p;
			    if (e->right_first)
                                sub1 = memia->e2, sub2 = memia->e1;
                            else sub1 = memia->e1, sub2 = memia->e2;
			    if (sub1 and sub2)
				goto HAVE_SUBS;
			    else if (sub1 == NULL)
				sub1 = sub2, sub2 = NULL;
			    if (sub1 == &frame_register)
				break;
			    RegisterRecursePass3(sub1);
                            break;

            case oTERNARY:  RegisterRecursePass3(e->left);
                            UnuseRegister(e->left);
                            RegisterRecursePass3(e->right->left);
                            UnuseRegister(e->right->left);
                            RegisterRecursePass3(e->right->right);
                            UnuseRegister(e->right->right);
                            break;

            default:        assert(false);
        }
}



static void RegistersIntoExpr(Expr* e, reg_enum destination)
{
        /* Pass 1:  Work out how many registers each sub-expression needs: */
        RegisterRecursePass1(e);

        /* Pass 2:  Tell each sub-expression what registers you want them to */
        /* put their results in: */
        e_idx = 0;
        clearA(Occupier);
        RegisterRecursePass2(e, destination);
        /* If we request 'reg_operand' but instead are given a general */
	/* register, then we mustn't change it (eg. MOVSXZX can't handle that). */
	/* But if we're asked to put the result in eg. AX, then we must do this.*/
	if (IsGeneral(destination))
	    assert(e->storereg == destination);

        /* Pass 3:  Try to minimise use of the stack resulting from clashes */
        /* in the use of specialist registers. */
        clearA(Occupier);
        RegisterRecursePass3(e);
}


static void AllocateRegistersToExprs(void)
{       SwitchStatement* S_switch;
        ReturnStatement* S_return;
        ExprStatement* S_expr;
        IfStatement* S_if;
        Statement* S;

        for (each_statement) {
            if (S->st == st_if) {
                S_if = (IfStatement*)S;
                RegistersIntoExpr(S_if->Test, IsComparison(S_if->Test->o) ? reg_discard : reg_operand);
            }
            else if (S->st == st_expr) {
                S_expr = (ExprStatement*)S;
                RegistersIntoExpr(S_expr->e, reg_discard);
            }
            else if (S->st == st_return) {
                S_return = (ReturnStatement*)S;
                if (S_return->e and S_return->e != (Expr*)0x1)
                    RegistersIntoExpr(S_return->e,
                            IsFloat(S_return->e) ? reg_fpstack :
			    TpSize(S_return->e->tp) > 4 ? reg_discard : reg_AX);
                if (S_return->destructors)
                    RegistersIntoExpr(S_return->destructors, reg_discard);
            }
            else if (S->st == st_switch) {
		S_switch = (SwitchStatement*)S;
		RegistersIntoExpr(S_switch->Test, reg_somegeneral);
            }
        }
}






/*--------------- Allocating offsets to local variables: -----------------*/

static int frame_size;
static bool UseFrame;


static void AllocateLocalVars(void)
{       AutoNamedobj *obj, *tmp, *tmp2;

	/* First reverse the list, to return to the declaration order: */
	obj = localvar_root;
	tmp = NULL;
	while (obj) {
	    tmp2 = obj->nextlocal;
	    obj->nextlocal = tmp;
	    tmp = obj;
	    obj = tmp2;
	}
	localvar_root = tmp;

	/* Now allocate them properly: */
	frame_size = 0;
	UseFrame = no;
        for (obj=localvar_root; obj; obj=obj->nextlocal) {
	    if (obj->storage == parameter_storage) {
		UseFrame = yes;
		continue;	/* It must be a parameter, already allocated. */
	    }
	    else if (obj->storage == local_static) {
		continue;
	    }
	    else if (obj->storage == auto_storage) {
		frame_size += TypeSizeWord(obj->type);
		frame_size = ((frame_size-1)|3)+1;	// Word-align
		obj->offset = -frame_size;
                if ((*obj->type == tp_class) and
                                (*(Classdef**)(obj->type+1))->VirtualFns)
                     frame_size += sizeof(void*);
		UseFrame = yes;
	    }
	    else assert(false);
        }
}






/*--------------- Storing info about pointers: --------------*/

typedef struct pb_node {
	int code_vs_ls:1;	// Is this offset relative to codebuf or localstatic_buf?
				// Switch tables are relative to localstatic_buf.
                                // I'd use a 'bool' but due to a bug in Borland, this
                                // makes the struct 12 bytes in size.
	uint offset:30;		// The address of the pointer, relative to either
        			// codebuf or localstatic_buf.
	void* base;		// The base of the ptr.  Either equals a pointer in
				// the in-memory persistent store, or special values:
				// localstatic_buf or codebuf.  Only relevant in
                                // the case where constants have been folded into
                                // the pointer value so that the target now points
                                // into the 'wrong' malloc block.
} *pb_type;

static uint pointerbuf_idx, pointerbuf_msize;
static pb_type pointerbuf;


static void PointerbufInit(void)
{
	if (pointerbuf)
	    free(anon_heap, pointerbuf);
	pointerbuf = NULL;
	pointerbuf_msize = 0;
	pointerbuf_idx = 0;
}


static void PointerWasEmitted(bool code_vs_ls, uint offset, void* base)
/* Make a note of the fact that a pointer was just written. 	*/
/* The address of the pointer is given by 'offset', an offset	*/
/* relative to either 'codebuf' or 'localstatic_buf'. 		*/
/* The target of the pointer is either (a) the code, in which   */
/* case base==code_base;  (b) local statics, in which case 	*/
/* base==localstatic_base;  or (c) somewhere else in the conim, */
/* in which case base==somewhere inside the malloc block for 	*/
/* the target.  So altogether there are 6 cases.  */
{	pb_type pb;

        if (pointerbuf == NULL) {
	    pointerbuf_msize = 200;
	    pointerbuf = (pb_type)malloc(anon_heap, pointerbuf_msize);
	}
	if ((pointerbuf_idx + 1)*sizeof(struct pb_node) >= pointerbuf_msize) {
	    pointerbuf_msize *= 2;
	    pointerbuf = (pb_type)realloc(anon_heap, pointerbuf, pointerbuf_msize);
	}
	pb = &pointerbuf[pointerbuf_idx++];
	pb->code_vs_ls = code_vs_ls;
	pb->offset = offset;
	pb->base = base;
}






/*--------------- Low-level emitting instructions: -----------------*/

/* 
The functions in this section generally output a single instruction each.
It is up to the caller to pass them arguments (i.e. addressing modes etc)
for which an instruction exists.
	It is important to be able to identify pointers in the generated
code.  This is why constants passed around are always passed around as
"kConstant Expr*'s".  The Expr* contains the information we need
to identify it as a pointer or integer, and if it's a pointer then what
it's 'base' is.  The 'base' is some pointer into a malloc block that the
value is tied to:  for example, with something like:   A['0' - *s];
where &A+'0' are folded together during optimisation, the pointer embedded 
in the code might actually point outside the malloc block of the pointer
you get during execution.  This means pointer-swizzling wouldn't work
unless we can identify the base of each pointer embedded in the code.
	The 'base' for local-statics is set as NULL during code generation.
This is because the local-static block oeeps changing location. We fix
them up after putting the code in its final location.
*/


static uchar codebuf[4096] = { 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90 };
static int cb;
static int last_add_sp;

#define dereference		((Expr*)0x1)


static void EmitByte(int byte)
{
        codebuf[cb++] = (uchar)byte;
}


static void EmitInt2(int4 i)
{
        *((short*)(codebuf+cb)) = i;
	cb += 2;
}


static void EmitInt4(int4 i)
{
        *((int*)(codebuf+cb)) = i;
	cb += 4;
}


static void EmitPtr(void* p, void* base)
{
	if (base)
	    PointerWasEmitted(yes, cb, base);
        *((void**)(codebuf+cb)) = p;
	cb += 4;
}


static void EmitLocalStaticPtr(int4 offset)
{
	PointerWasEmitted(yes, cb, localstatic_base);
        *((int4*)(codebuf+cb)) = offset;
	cb += 4;
}


static void Emit124(int4 arg, int size)
{
        if (size == 1)
            EmitByte(arg);
        else if (size == 2)
            EmitInt2(arg);
        else EmitInt4(arg);
}


static int RegEncoding(reg_enum reg)
/* Return the 3-bit code for this register. */
{
        switch (reg) {
            case reg_AX:    return 0;
            case reg_CX:    return 1;
            case reg_DX:    return 2;
            case reg_BX:    return 3;
            case reg_SP:    return 4;
            case reg_BP:    return 5;
            case reg_SI:    return 6;
            case reg_DI:    return 7;
	    default:        assert(false);
			    return -1;
        }
}


static void EmitConstant(Expr* e, uint size)
/* We have a oConstant expression-type which we want to emit as an immediate */
/* argument.  e->tp tells us if it's a pointer or not, and if it is then     */
/* what is its base (in particular, is it a local static?). */
{
	if (e == NULL) {
	    Emit124(0, size);
	    return;
	}
	assert(e->o == oConstant or e->o == oPUSHSPPLUS);
	if (e->tp != tp_pointer)
	    Emit124(e->u.i, size);
	else {
	    PointerWasEmitted(yes, cb, (void*)(e->right));
 	    EmitInt4(e->u.i);
	}
}


static void EmitConstant4(Expr* e)
/* Like the above but for when you onow it's 4 bytes. */
{
	if (e == NULL) {
	    EmitInt4(0);
	    return;
	}
	assert(e->o == oConstant);
	if (e->tp != tp_pointer)
	    Emit124(e->u.i, 4);
	else {
	    PointerWasEmitted(yes, cb, (void*)(e->right));
 	    EmitInt4(e->u.i);
	}
}


static void EmitMemOperand(int rg, Expr* e)
/* 'e' is the oConstant or oMEMIA expression just underneath a	*/
/* oDER expression (the only exception being expressions which	*/
/* will be part of a LEA instruction).				*/
/* Emit a normal mod-r/m byte + optional SIB byte + arguments.	*/
/* The middle 3 bits of the mod-r/m byte, which denotes either	*/
/* a 2nd register or extra opcode information, is filled out	*/
/* from the 'rg' parameter.  This function shifts it left	*/
/* (don't you do it). */
{	int SIB;

	while (e->o == oNOP1)
	    e = e->left;
	if (e->o == oConstant) {
	    /* The 'mode 5' - just a displacement. */
	    EmitByte(0x05 | (rg<<3));
	    EmitPtr(e->u.p, e->right);
	}
	else if (e->storereg != reg_operand) {
	    /* This may well be a oMEMIA, but if it has a storereg then we onow */
	    /* it's already computed the address into a register. */
	    assert(IsGeneral(e->storereg));
            EmitByte(0x00 | (rg<<3) | RegEncoding(e->storereg));
	}
	else if (e->o == oMEMIA) {
	    memia_type memia;

	    memia = (memia_type)e->u.p;
            if (memia->e1) assert(IsGeneral(memia->e1->workreg) or memia->e1 == &frame_register);
            if (memia->e2) assert(IsGeneral(memia->e2->workreg));
	    if ((int)memia->K == (char)memia->K and memia->e2 == NULL) {
		if (memia->e1 == &frame_register)
		    EmitByte(0x40 | (rg<<3) | 5);
		else EmitByte(0x40 | (rg<<3) | RegEncoding(memia->e1->workreg));
		EmitByte(memia->K);
	    }
	    else if (memia->e2 == NULL) {
		if (memia->e1 == &frame_register)
		    EmitByte(0x80 | (rg<<3) | 5);
		else EmitByte(0x80 | (rg<<3) | RegEncoding(memia->e1->workreg));
		EmitPtr((void*)memia->K, memia->Kbase);
	    }
	    else {
		/* We'll need an SIB byte. */
		if (memia->K == 0 or memia->e1 == NULL)
		    EmitByte(0x00 | (rg<<3) | 4);
		else if ((int)memia->K == (char)memia->K)
		    EmitByte(0x40 | (rg<<3) | 4);
		else
		    EmitByte(0x80 | (rg<<3) | 4);

		/* The SIB byte itself: */
		SIB = memia->scaler << 6;
		if (memia->e2 == NULL)
		    SIB |= 4 << 3;
		else SIB |= RegEncoding(memia->e2->workreg) << 3;
		if (memia->e1 == NULL)
		    SIB |= 5;
		else if (memia->e1 == &frame_register)
		    SIB |= 4;
		else SIB |= RegEncoding(memia->e1->workreg);
		EmitByte(SIB);
		if (memia->K == 0)
		    ;	 /* No displacement was requested */
		else if ((int)memia->K == (char)memia->K)
		    EmitByte(memia->K);	    /* 8-byte (signed) displacement */
		else EmitPtr((void*)memia->K, memia->Kbase);    /* 32-bit displacement */
	    }
	}
	else assert(false);
}


static void EmitMOV(reg_enum to, Expr* to_arg, reg_enum from, Expr* from_arg, int size)
/* 'size' must be 1,2 or 4.  Move 'size' bytes from (from,from_arg) to (to,to_arg)'. */
/* 'to' or 'from' can specify an immediate value, an absolute memory-location,       */
/* the stack, or a register.  If the stack is involved, it emits a PUSH or POP.      */
/* If a register is specified, the corresponding argument tells you whether we mean  */
/* the register itself or the memory pointed to by the register. */
{       int size_bit;

        if (size == 4)
            size_bit = 1;
	else if (size == 2)
            size_bit = 1;
        else if (size == 1)
            size_bit = 0;
        else assert(false);
        if (from == reg_imm) {
            if (to == reg_mem) {
                // Immediate to memory
		if (size == 2)
		    EmitByte(0x66);
                EmitByte(0xc6 | size_bit);
                EmitMemOperand(0, to_arg);
                EmitConstant(from_arg, size);
            }
            else if (to == reg_stack) {
                // Immediate to stack
                EmitByte(0x68);
                EmitConstant4(from_arg);
            }
            else if (to_arg == dereference) {
                // Immediate to [reg]
                assert(IsGeneral(to));
                if (size == 4)
		    EmitByte(0xc7);
                else if (size == 1)
                    EmitByte(0xc6);
		else EmitByte(0x66), EmitByte(0xc7);
                EmitByte(0x00 | RegEncoding(to));
		EmitConstant(from_arg, size);
            }
            else {
                // Immediate to register
                size_bit <<= 3;
                EmitByte(0xb0 | size_bit | RegEncoding(to));
                EmitConstant(from_arg, size);
            }
        }
        else if (from == reg_mem) {
            if (to == reg_stack) {
                // Memory to stack
                EmitByte(0xff);
                EmitMemOperand(6, from_arg);
            }
            else {
                // Memory to register
                assert(IsGeneral(to) and to_arg == 0);
                EmitByte(0x8a | size_bit);              // MOV [mem] to reg
                EmitMemOperand(RegEncoding(to), from_arg);
            }
        }
        else if (from == reg_stack) {
            if (to == reg_mem) {
                // Stack to memory
		if (size == 2)
		    EmitByte(0x66);
		else if (size == 1)
		    assert(false);	// Not implemented - does this ever occur?
                EmitByte(0x8f);
                EmitMemOperand(0, to_arg);
            }
            else if (to_arg == dereference) {
                // Stack to [reg]
                assert(IsGeneral(to));
		if (size == 2)
		    EmitByte(0x66);
		else if (size == 1)
		    assert(false);	// Not implemented - does this ever occur?
                EmitByte(0x8f);
                EmitByte(0x00 | RegEncoding(to));
            }
            else {
                // Stack to register
                assert(IsGeneral(to));
                EmitByte(0x8f);
                EmitByte(0xc0 | RegEncoding(to));
            }
        }
        else if (from_arg == 0) {
            assert(IsGeneral(from) or from == reg_BP or from == reg_SP);
            if (to == reg_mem) {
                // Register to memory
                if (size == 4)
		    EmitByte(0x89);
                else if (size == 1)
		    EmitByte(0x88);
		else //if (size == 2)
		    EmitByte(0x66), EmitByte(0x89);
		EmitMemOperand(RegEncoding(from), to_arg);
            }
            else if (to == reg_stack) {
                // Register to stack
                EmitByte(0x50 | RegEncoding(from));
            }
            else {
                assert(IsGeneral(to));
                if (to_arg == dereference) {
                    // Register to [Register]
		    if (size == 2)
			EmitByte(0x66);
                    EmitByte(0x88 | size_bit);      // MOV reg to [reg]
                    EmitByte(0x00 | RegEncoding(from)*8 | RegEncoding(to));
                }
                else {
                    // Register to register
                    EmitByte(0x88 | size_bit);      // MOV reg to reg
                    EmitByte(0xc0 | RegEncoding(from)*8 | RegEncoding(to));
                }
            }
        }
        else {
            assert(IsGeneral(from));
            if (to == reg_stack) {
		// [reg] to stack
                EmitByte(0xff);
                EmitByte(0x00 | 6*8 | RegEncoding(from));
            }
            else {
                assert(IsGeneral(to) and to_arg == 0);
                // [Register] to Register
		bool Unsigned;
		if (from_arg == NULL or from_arg == dereference)
		    Unsigned = no;
		else Unsigned = TpIsUnsigned(from_arg->tp);
		if (size == 4)
		    EmitByte(0x8b);      // MOV [reg] to reg
		else if (size == 2)
		    EmitByte(0x0f), EmitByte(0xb7 | (Unsigned ? 0 : 8));// MOVSX/MOVZX
		else if (size == 1)
		    EmitByte(0x0f), EmitByte(0xb6 | (Unsigned ? 0 : 8));// MOVSX/MOVZX
                EmitByte(0x00 | RegEncoding(to)*8 | RegEncoding(from));
            }
        }
}


static void EmitMOVSXZX(reg_enum to_reg, uint to_size, Expr* from)
{
	bool Unsigned = TpIsUnsigned(from->tp);
	uint from_size = TpSize(from->tp);
	if (IsGeneral(from->storereg)) {
	    if (to_size == 2)
		EmitByte(0x66);
	    EmitByte(0x0f);
	    EmitByte(0xb6 | (from_size >= 2?1:0) | (Unsigned?0:8));
	    EmitByte(0xc0 | RegEncoding(to_reg)*8 | RegEncoding(from->storereg));
	}
	else if (from->storereg == reg_operand) {
	    if (to_size == 2)
		EmitByte(0x66);
	    EmitByte(0x0f);
	    EmitByte(0xb6 | (from_size >= 2?1:0) | (Unsigned?0:8));
	    assert(from->o == oDER);
	    EmitMemOperand(RegEncoding(to_reg), from->left);
	}
	else assert(false);
}


static void EmitMOVSP(reg_enum to, reg_enum from)
/* We don't regard the stack ptr as a general-purpose register. */
/* This function allows you to copy to/from it.  One of the two */
/* parameters must be 'reg_stack'. */
{
	if (from == reg_stack and IsGeneral(to)) {
	    EmitByte(0x89);
	    EmitByte(0xc0 | 4*8 | RegEncoding(to));
	}
	else if (IsGeneral(from) and to == reg_stack) {
	    EmitByte(0x89);
	    EmitByte(0xc0 | RegEncoding(from)*8 | 4);
	}
        else assert(false);
}


static void EmitLEA(reg_enum reg, Expr* arg)
{	reg_enum tmp=arg->storereg;

	if (arg->o == oConstant) {
            EmitByte(0xb0 | (1<<3) | RegEncoding(reg));
            EmitConstant(arg, 4);
            return;
        }
	arg->storereg = reg_operand;	    // This is to signal to EmitMemOperand
		// that this is where we're computing the address, don't assume
		// the address is already computed.
	assert(IsGeneral(reg));
	EmitByte(0x8d);
	EmitMemOperand(RegEncoding(reg), arg);
	arg->storereg = tmp;
}


static void EmitPUSH(reg_enum reg, Expr* arg)
{
        if (reg == reg_imm) {
	    if (TpSize(arg->tp) <= 4) {
		EmitByte(0x68);
		EmitConstant4(arg);
	    }
	    else if (arg->tp == tp_double) {
		union {
		    int i[2];
		    double f;
		} u;
		u.f = arg->u.f;
		EmitByte(0x68);
		EmitInt4(u.i[1]);
		EmitByte(0x68);
		EmitInt4(u.i[0]);
	    }
	    else assert(false);
        }
        else if (reg == reg_mem) {
            EmitByte(0xff);
            EmitMemOperand(6, arg);
        }
        else EmitByte(0x50 | RegEncoding(reg));
}


static void EmitPOP(reg_enum reg)
{
        EmitByte(0x58 | RegEncoding(reg));
}


static void EmitXCHG(reg_enum a, reg_enum b)
{
        if (a == reg_AX)
            EmitByte(0x90 | RegEncoding(b));
        else if (b == reg_AX)
            EmitByte(0x90 | RegEncoding(a));
        else {
            EmitByte(0x87);
            EmitByte(0xc0 | RegEncoding(a)*8 | RegEncoding(b)*8);
        }
}


static void EmitCMOV(int condition, reg_enum to, reg_enum from)
{
        EmitByte(0x0f);
        EmitByte(0x40 | condition);
        EmitByte(0xc0 | RegEncoding(to)*8 | RegEncoding(from));
}


static void EmitNEG(reg_enum reg)
{
        EmitByte(0xf7);
        EmitByte(0xd8 | RegEncoding(reg));
}


static void EmitNOT(reg_enum reg)
{
        EmitByte(0xf7);
        EmitByte(0xe0 | RegEncoding(reg));
}


static void EmitFloatNEG(void)
{
        EmitByte(0xd9);
        EmitByte(0xee);
        EmitByte(0xDE);
        EmitByte(0xC1 | 4*8);
}


static void EmitArithI(int op, reg_enum a, Expr* a_arg, reg_enum b, Expr* b_arg, int4 size)
/* Emit an ADD, SUB, CMP, AND, OR or XOR.   The destination is 'a'. */
{       int size_bit;

        if (size == 4 or size == 2)
            size_bit = 1;
        else if (size == 1)
            size_bit = 0;
        else assert(false);

        if (a == reg_mem) {
	    if (b == reg_imm) {
                if ((op == 0 or op == 5) and (b_arg->u.i == 1 or b_arg->u.i == -1)) {
                    if (b_arg->u.i == -1)
                        op = 5 - op;    /* toggle between add and sub */
                    // [mem] ++ / --
		    if (size == 2)
			EmitByte(0x66);
                    EmitByte(0xfe | size_bit);
                    EmitMemOperand((op == 0) ? 0 : 1, a_arg);
                }
                else {
                    // [mem] *= imm
		    if (size == 2)
			EmitByte(0x66);
                    EmitByte(0x80 | size_bit);
                    EmitMemOperand(op, a_arg);
                    EmitConstant(b_arg, size);
                }
            }
            else if (IsGeneral(b)) {
                // [mem] *= reg
		if (size == 2)
		    EmitByte(0x66);
                EmitByte(op*8 | size_bit);
                EmitMemOperand(RegEncoding(b), a_arg);
            }
            else assert(false);
        }
        else if (IsGeneral(a) and a_arg == dereference) {
            if (b == reg_imm) {
                if ((op == 0 or op == 5) and (b_arg->u.i == 1 or b_arg->u.i == -1)) {
                    if (b_arg->u.i == -1)
                        op = 5 - op;    /* toggle between add and sub */
                    // [reg] ++ / --
		    if (size == 2)
			EmitByte(0x66);
                    EmitByte(0xfe | size_bit);
                    EmitByte(((op == 0) ? 0x00 : 0x08) | RegEncoding(a));
                }
                else {
                    // [reg] *= imm
		    if (size == 2)
			EmitByte(0x66);
                    EmitByte(0x80 | size_bit);
                    EmitByte(0x00 | op*8 | RegEncoding(a));
					EmitConstant(b_arg, size);
                }
            }
            else {
                assert(IsGeneral(b));
                // [reg] *= reg
		if (size == 2)
		    EmitByte(0x66);
                EmitByte(op*8 | size_bit);
                EmitByte(0x00 | RegEncoding(b)*8 | RegEncoding(a));
            }
        }
        else {
            /* We have a working register: */
            assert(IsGeneral(a) and a_arg == NULL);
	    if (b == reg_imm) {
		if ((op == 0 or op == 5) and (b_arg->u.i == 1 or b_arg->u.i == -1)) {
		    if (b_arg->u.i == -1)
			op = 5 - op;	    /* Toggle between add and sub */
		    // reg ++/--
		    EmitByte(((op == 0) ? 0x40 : 0x48) | RegEncoding(a));
		}
		else {
		    // reg *= imm
		    EmitByte(0x80 | size_bit);
		    EmitByte(0xc0 | op*8 | RegEncoding(a));
		    EmitConstant(b_arg, size);
		}
	    }
	    else if (b == reg_mem) {
		// reg *= [mem]
		if (size == 2)
		    EmitByte(0x66);
		EmitByte(0x02 | op*8 | size_bit);
		EmitMemOperand(RegEncoding(a), b_arg);
	    }
	    else if (IsGeneral(b)) {
		// reg *= reg
		EmitByte(op*8 | size_bit);
		EmitByte(0xc0 | RegEncoding(b)*8 | RegEncoding(a));
	    }
	    else assert(false);
	}
}


static void EmitArithF(int op, tp_enum tp, reg_enum from, Expr* mem)
/* Emit a floating-point operation that takes the left-hand-side from the */
/* top of the floating-pt stack, the right-hand-side from (tp,from,mem)   */
/* and stores the result in the same place as the left-hand-side.         */
{
        if (from == reg_fpstack) {
            assert(tp == tp_double);
	    op = ReverseFlop(op);
            EmitByte(0xDE);
            EmitByte(0xC1 | op*8);
            return;
        }
        if (tp == tp_double)
            EmitByte(0xDE);
        else if (tp == tp_float)
            EmitByte(0xD8);
        else if (tp == tp_int or tp == tp_uint)
            EmitByte(0xDA);
        else if (tp == tp_short or tp == tp_ushort)
            EmitByte(0xDE);
        else assert(false);
        if (from == reg_mem) {
            EmitMemOperand(op, mem);
        }
        else {
            assert(IsGeneral(from));
            EmitByte(op*8 | 8*RegEncoding(from));       
        }
}


static void EmitIMUL(reg_enum d, reg_enum s, Expr* s_arg)
{
        if (d == reg_AX) {
            if (s == reg_imm) {
                EmitByte(0x69);
                EmitByte(0xc0 | RegEncoding(d)*8 | RegEncoding(d));
                EmitConstant4(s_arg);
            }
            else if (s == reg_mem) {
                EmitByte(0x0f);
                EmitByte(0xaf);
                EmitMemOperand(RegEncoding(d), s_arg);
            }
            else {
                EmitByte(0xf7);
                EmitByte(0xe8 | RegEncoding(s));
            }
        }
        else {
            if (s == reg_imm) {
                EmitByte(0x69);
                EmitByte(0xc0 | RegEncoding(d)*8 | RegEncoding(d));
                EmitConstant4(s_arg);
            }
            else if (s == reg_mem) {
                EmitByte(0x0f);
                EmitByte(0xaf);
                EmitMemOperand(RegEncoding(d), s_arg);
            }
	    else if (s_arg == dereference) {
                EmitByte(0x0f);
                EmitByte(0xaf);
                EmitByte(0x00 | RegEncoding(d)*8 | RegEncoding(s));
	    }
            else {
                EmitByte(0x0f);
                EmitByte(0xaf);
                EmitByte(0xc0 | RegEncoding(d)*8 | RegEncoding(s));
            }
        }
}


static void EmitIDIV(reg_enum reg, Expr* arg, bool Unsigned)
{
        if (reg == reg_mem) {
            EmitByte(0xf7);
            EmitMemOperand(7, arg);
        }
        else if (arg == dereference) {
            EmitByte(0xf7);
            EmitByte(0x00 | 0x38);
	}
	else {
            EmitByte(0xf7);
            EmitByte(0xf8 | RegEncoding(reg));
        }
}


static void EmitShift(int op, reg_enum shiftee, Expr* shiftee_arg, reg_enum howmuch, Expr* hm_arg, tp_enum tp)
{       int size_bit, size;

	size = TpSize(tp);
        if (size == 4)
            size_bit = 1;
        else if (size == 2)
            size_bit = 0;
        else assert(false);
        assert(op == 4 or op == 5);
	if (op == 5 and not TpIsUnsigned(tp))
	    op = 7;	/* SAR: Copy the sign bit along. */
        if (shiftee == reg_mem or shiftee == reg_operand) {
            if (howmuch == reg_CX) {
		if (shiftee_arg == dereference) {
		    EmitByte(0xd2 | size_bit);
		    EmitByte(0x00 | RegEncoding(shiftee));
		}
		else {
		    EmitByte(0xd2 | size_bit);
		    EmitMemOperand(op, shiftee_arg);
		}
            }
            else {
                assert(howmuch == reg_imm);
                if (hm_arg->u.i == 1) {
                    EmitByte(0xd0 | size_bit);
                    EmitMemOperand(op, shiftee_arg);
                }
                else {
                    EmitByte(0xc0 | size_bit);
                    EmitMemOperand(op, shiftee_arg);
		    assert(hm_arg->o == oConstant);
                    EmitByte(hm_arg->u.i);
                }
            }           
        }
        else if (shiftee_arg == 0) {
            // Shifting a register
            assert(IsGeneral(shiftee));
            if (howmuch == reg_CX) {
                EmitByte(0xd2 | size_bit);
                EmitByte(0xc0 | op*8 | RegEncoding(shiftee));
            }
            else {
                assert(howmuch == reg_imm);
                if (hm_arg->u.i == 1) {
                    EmitByte(0xd0 | size_bit);
                    EmitByte(0xc0 | op*8 | RegEncoding(shiftee));
                }
                else {
                    EmitByte(0xc0 | size_bit);
                    EmitByte(0xc0 | op*8 | RegEncoding(shiftee));
					assert(hm_arg->o == oConstant);
                    EmitByte(hm_arg->u.i);
                }
            }
        }
        else {
            // Shifting memory pointed to by a register:
            if (howmuch == reg_CX) {
                EmitByte(0xd2 | size_bit);
                EmitByte(0x00 | op*8 | RegEncoding(shiftee));
            }
            else {
                assert(howmuch == reg_imm);
                if (hm_arg->u.i == 1) {
                    EmitByte(0xd0 | size_bit);
                    EmitByte(0x00 | op*8 | RegEncoding(shiftee));
                }
                else {
                    EmitByte(0xc0 | size_bit);
                    EmitByte(0x00 | op*8 | RegEncoding(shiftee));
					assert(hm_arg->o == oConstant);
                    EmitByte(hm_arg->u.i);
                }
            }
        }
}


static void EmitCLEAR(reg_enum reg)
{
        assert(IsGeneral(reg));
        EmitByte(0x31);
        EmitByte(0xc0 | RegEncoding(reg)*8 | RegEncoding(reg));
}


static void EmitCDQ(void)
{
        EmitByte(0x99);
}


static void EmitFFlagsToEFlags(void)
{
        EmitByte(0xdf);
        EmitByte(0xe0);
        EmitByte(0x9e);
}


static void EmitFCOMIP(void)
{
        EmitByte(0xdf);
        EmitByte(0xf1);
}


static void EmitFPOP(void)
{       static double dummy;

        /*EmitByte(0xd9);               <-- I thought d9f6 was an FPOP,
        EmitByte(0xf6);                 but it's not! It's a FDECSTP, which
                                        has a different effect to the dummy
                                        FSTP below.
        */
        EmitByte(0xdd);
        EmitByte(0x1d);
        EmitInt4((int)&dummy);
}


static uint EmitJMP(uint location)
{       int instruction, offset;

	instruction = cb;
        offset = location - cb - 2;
        if (offset == (char)offset and location) {
            EmitByte(0xeb);
            EmitByte(offset);
        }
        else {
            offset -= 3;
            EmitByte(0xe9);
            EmitInt4(offset);
        }
	return instruction;
}


static uint EmitConditionalJump(int condition, uint location)
/* Emit a conditional jump to this location.  We return the address of the */
/* instruction (relative to 'codebuf') in case we need to backpatch it. */
{       uint instruction=cb;
        int offset;

        offset = location - instruction - 2;
	if (offset == (char)offset and location) {
	    EmitByte(0x70 | condition);
	    EmitByte(offset);
	}
	else {
	    offset -= 4;
	    EmitByte(0x0f);
	    EmitByte(0x80 | condition);
	    EmitInt4(offset);
	}
        return instruction;
}


static void EmitTableJump(reg_enum reg, uint ls_offset)
/* Emit a "JMP to base+reg*4" instruction. */
/* The 'base' is a local-static pointer.   */
{
	EmitByte(0xff);
	EmitByte(0x04 | (4<<3));
	EmitByte(0x85 | (RegEncoding(reg)<<3));
	EmitLocalStaticPtr(ls_offset);
}


static void EmitBackPatch(uint instruction, uint location)
/* Now that we onow the location, insert this back into the instruction. */
{       int offset, condition;

        if (codebuf[instruction] == 0x0f) {
            // A conditional jump
            offset = location - (instruction + 2);
            if (offset == (char)offset) {
                condition = codebuf[instruction+1] & 0x0f;
                codebuf[instruction+0] = 0x70 | condition;
                codebuf[instruction+1] = offset;
                codebuf[instruction+2] = (uchar)0x90;   // NOP
                codebuf[instruction+3] = (uchar)0x90;   // NOP
                codebuf[instruction+4] = (uchar)0x90;   // NOP
                codebuf[instruction+5] = (uchar)0x90;   // NOP
            }
            else {
		offset = location - (instruction + 6);
                instruction += 2;
                *(int4*)(codebuf+instruction) = offset;
            }
        }
        else {
            // An unconditional jump
            assert(codebuf[instruction] == (uchar)0xe9);
            offset = location - instruction - 2;
            if (offset == (char)offset) {
                codebuf[instruction+0] = (uchar)0xeb;
                codebuf[instruction+1] = offset;
                codebuf[instruction+2] = (uchar)0x90;   // NOP
                codebuf[instruction+3] = (uchar)0x90;   // NOP
                codebuf[instruction+4] = (uchar)0x90;   // NOP
            }
            else {
		offset = location - (instruction + 5);
                instruction++;
                *(int4*)(codebuf+instruction) = offset;
            }
        }
}


static void EmitExamine(reg_enum reg, Expr* arg)
/* Emit an "or reg,reg" instruction, whose only purpose is to set the */
/* flags according to 'reg'. */
{
	if (IsGeneral(reg)) {
	    EmitByte(0x09);
	    EmitByte(0xc0 | RegEncoding(reg)*8 | RegEncoding(reg));
	}
	else if (reg == reg_operand) {
	    if (arg->o == oCONVERT and arg->left->o == oDER and arg->left->storereg == reg_operand) {
		EmitByte(0x83);
		EmitMemOperand(7, arg->left->left);
		EmitByte(0x00);
	    }
	    else if (arg->o == oCONVERT and IsGeneral(arg->left->storereg)) {
		reg = arg->left->storereg;
		EmitByte(0x09);
		EmitByte(0xc0 | RegEncoding(reg)*8 | RegEncoding(reg));
	    }
	    else if (arg->o == oDER) {
		EmitByte(0x83);
		EmitMemOperand(7, arg->left);
		EmitByte(0x00);
	    }
	    else assert(false);
	}
	else assert(false);
}


static void EmitCALL(reg_enum reg)
/* Emit a call to 'reg'. */
{
        EmitByte(0xff);
        EmitByte(0xd0 | RegEncoding(reg));
}


static void EmitCALL(Expr* e)
/* Emit a call to the expression denoted by 'e'. */
{
        EmitByte(0xff);
        EmitMemOperand(2, e);
}


static void EmitCALL(void* direct)
/* Emit a call to location 'direct'. */
{
        EmitByte(0xe8);
        EmitInt4((int4)direct);
		    /* The destination should be a relative 4-byte quantity, */
		    /* but we'll fix this up when we allocate the final space */
		    /* for the function. */
}


static void EmitEnter(int frame_size)
{
        EmitByte(0xc8);
        EmitInt2(frame_size);
        EmitByte(0);
}


static void EmitLeave(void)
{
        EmitByte(0xc9);
}


static void EmitADD_SP(int size)
{
	if (last_add_sp == cb - 6 and last_add_sp) {
	    cb -= 6;
	    size += *(int*)(codebuf + cb + 2);
	}
        if (size > 0) {
	    last_add_sp = cb;
            EmitByte(0x81);
            EmitByte(0xc4);
            EmitInt4(size);
        }
        else if (size < 0) {
	    last_add_sp = cb;
            EmitByte(0x81);
            EmitByte(0xec);
            EmitInt4(-size);
        }
}


static void EmitLocalAddress(reg_enum dest, int offset)
{
	if (dest == reg_stack) {
	    /* Push BP and then add to it in situ. */
	    EmitPUSH(reg_BP, NULL);
	    if ((char)offset == offset) {
		EmitByte(0x83);
		EmitByte(0x04);
		EmitByte(0x24);
		EmitByte(offset);
	    }
	    else {
		EmitByte(0x81);
		EmitByte(0x04);
		EmitByte(0x24);
		EmitInt4(offset);
	    }
	}
	else {
	    /* Emit an LEA instruction. */
	    EmitByte(0x8d);
	    if ((char)offset == offset) {
		EmitByte(0x45 | RegEncoding(dest)*8);
		EmitByte(offset);
	    }
	    else {
		EmitByte(0x85 | RegEncoding(dest)*8);
		EmitInt4(offset);
	    }
	}
}


static void EmitRET(void)
{
        EmitByte(0xc3);
}


static void EmitFLoad(tp_enum tp, reg_enum from, Expr* arg)
/* Load a constant, or a value from absolute memory, */
/* or a value from [reg] onto the floating-pt stack. */
{
        if (from == reg_mem) {
            switch (tp) {
                case tp_double: EmitByte(0xdd);
				EmitMemOperand(0, arg);
                                break;
                case tp_float:  EmitByte(0xd9);
				EmitMemOperand(0, arg);
                                break;
                case tp_long:   EmitByte(0xdf);
				EmitMemOperand(5, arg);
                                break;
		case tp_uint:	EmitByte(0x6a);	    // PUSH 0
				EmitByte(0x00);
				EmitByte(0xff);	    // PUSH [from]
				EmitMemOperand(6, arg);	    
				EmitByte(0xdf);	    // FILD [SP]
				EmitByte(0x2c);
				EmitByte(0x24);
				EmitADD_SP(8);
                                break;
                case tp_int:    EmitByte(0xdb);
				EmitMemOperand(0, arg);
                                break;
		case tp_ushort:
                case tp_short:  EmitByte(0xdf);
				EmitMemOperand(0, arg);
                                break;
                default:        assert(false);
                                break;
            }
        }
        else if (from == reg_imm) {
            switch (tp) {
                case tp_double: if (arg->u.f == 1.0) {
                                    EmitByte(0xd9);     /* FLD1 */
                                    EmitByte(0xe8);
                                }
                                else if (arg->u.f == 0) {
                                    EmitByte(0xd9);     /* FLDZ */
                                    EmitByte(0xee);
                                }
                                else {
                                    EmitByte(0xdd);     EmitByte(0x05);
                                    EmitLocalStaticPtr(AllocateAndCopyLocalStatic(
                                                8, &arg->u.f, (Type)"g"));
                                }
                                break;
                case tp_float:  {   float f=(float)arg->u.f;

				    EmitByte(0xd9);     EmitByte(0x05);
				    EmitLocalStaticPtr(AllocateAndCopyLocalStatic(
                                                4, &f, (Type)"f"));
				}
                                break;
                case tp_long:   EmitByte(0xdf);     EmitByte(0x05 | 5*8);
                                EmitLocalStaticPtr(AllocateAndCopyLocalStatic(
                                                8, &arg->u.l, (Type)"l"));
                                break;
		case tp_uint:	{   double f=(double)(unsigned int)arg->u.i;

				    EmitByte(0xdd);     EmitByte(0x05);
				    EmitLocalStaticPtr(AllocateAndCopyLocalStatic(
                                                8, &f, (Type)"u"));
				}
                                break;
                case tp_int:    if (arg->u.i == 1) {
                                    EmitByte(0xd9);     /* FLD1 */
                                    EmitByte(0xe8);
                                }
                                else if (arg->u.i == 0) {
                                    EmitByte(0xd9);     /* FLDZ */
                                    EmitByte(0xee);
                                }
                                else {
                                    EmitByte(0xdb);     EmitByte(0x05);
				    EmitLocalStaticPtr(AllocateAndCopyLocalStatic(
                                                4, &arg->u.i, (Type)"i"));
                                }
                                break;
		case tp_ushort:
                case tp_short:  EmitByte(0xdf);     EmitByte(0x05);
				EmitLocalStaticPtr(AllocateAndCopyLocalStatic(
                                                2, &arg->u.i, (Type)":"));
                                break;
                default:        assert(false);
                                break;
            }
        }
        else {
            assert(IsGeneral(from));
            switch (tp) {
                case tp_double: EmitByte(0xdd);     EmitByte(0x00 | RegEncoding(from));
                                break;
                case tp_float:  EmitByte(0xd9);     EmitByte(0x00 | RegEncoding(from));
                                break;
                case tp_long:   EmitByte(0xdf);     EmitByte(0x00 | 5*8 | RegEncoding(from));
                                break;
		case tp_uint:	EmitByte(0x6a);	    // PUSH 0
				EmitByte(0x00);
				EmitByte(0x50 | RegEncoding(from));// PUSH from
				EmitByte(0xdf);	    // FILD [SP]
				EmitByte(0x2c);
				EmitByte(0x24);
				EmitADD_SP(8);
                                break;
                case tp_int:    EmitByte(0xdb);     EmitByte(0x00 | RegEncoding(from));
                                break;
		case tp_ushort:
                case tp_short:  EmitByte(0xdf);     EmitByte(0x00 | RegEncoding(from));
                                break;
                default:        assert(false);
                                break;
            }
        }
}


static void EmitFStore(tp_enum tp, reg_enum to, Expr* arg, bool keep_value)
/* Store the top-of-fpstack into 'arg', of type 'tp'. */
{       int op;

        op = keep_value ? 2 : 3;
        if (to == reg_mem) {
            switch (tp) {
                case tp_double: EmitByte(0xdd);
				EmitMemOperand(op, arg);
                                break;
                case tp_float:  EmitByte(0xd9);
				EmitMemOperand(op, arg);
                                break;
                case tp_long:   if (keep_value)
                                    EmitByte(0xd9);     EmitByte(0xC0);// Duplicate the top-of-stack.
                                EmitByte(0xdf);
				EmitMemOperand(7, arg);
                                break;
                case tp_uint:
                case tp_int:    EmitByte(0xdb);
				EmitMemOperand(op, arg);
                                break;
                case tp_ushort:
                case tp_short:  EmitByte(0xdf);
				EmitMemOperand(op, arg);
                                break;
                default:        assert(false);
                                break;
            }
        }
        else if (to == reg_stack) {
            EmitADD_SP(-8);
            EmitByte(0xdd);
            EmitByte(0x1c);
            EmitByte(0x24);
        }
        else {
	    assert(IsGeneral(to));
            op = op*8 | RegEncoding(to);
            switch (tp) {
                case tp_double: EmitByte(0xdd);     EmitByte(op);
                                break;
		case tp_float:  EmitByte(0xd9);     EmitByte(op);
                                break;
                case tp_long:   if (keep_value)
                                    EmitByte(0xd9);     EmitByte(0xC0);// Duplicate the top-of-stack.
                                EmitByte(0xdf);     EmitByte(7*8 | 0x05);
                                break;
                case tp_int:    EmitByte(0xdb);     EmitByte(op);
                                break;
                case tp_short:  EmitByte(0xdf);     EmitByte(op);
                                break;
                default:        assert(false);
                                break;
            }
        }
}


static void EmitFStoreOnStack(tp_enum tp)
/* Store the top-of-fpstack on top of the normal stack. */
/* Oh - and pop the top-of-fpstack. */
{
	if (tp == tp_double) {
	    EmitADD_SP(-8);
	    EmitByte(0xdd);
	    EmitByte(0x1c);
	    EmitByte(0x24);
	}
	else if (tp == tp_float) {
	    EmitADD_SP(-4);
	    EmitByte(0xd9);
	    EmitByte(0x1c);
	    EmitByte(0x24);
	}
	else if (tp == tp_long) {
	    EmitADD_SP(-8);
	    EmitByte(0xdf);
	    EmitByte(0xfc);
	    EmitByte(0x24);
	}
	else assert(false);
}


static void EmitFCopyTop(void)
/* Copy the top-of-stack. */
{
        EmitByte(0xd9);
        EmitByte(0xc0);
}


static void EmitFSpecial(o_enum o)
{
        if (o == oCOS)
            EmitByte(0xd9), EmitByte(0xff);
        else if (o == oSIN)
            EmitByte(0xd9), EmitByte(0xfe);
        else if (o == oTAN)
            EmitByte(0xd9), EmitByte(0xf2);
        else if (o == oATAN)
            EmitByte(0xd9), EmitByte(0xf3);
        else if (o == oSQRT)
            EmitByte(0xd9), EmitByte(0xfa);
        else assert(false);
}


static void EmitREPMOVS(uint size)
/* Emit a repeat-move data str to str. */
{
	if (size == 1) {
	    EmitByte(0xf3);
	    EmitByte(0xa4);
	}
	else if (size == 4) {
	    EmitByte(0xf3);
	    EmitByte(0xa5);
	}
	else assert(false);
}





/*--------------- Emitting Code: -----------------*/

static void EmitSwitch(SwitchStatement* S_switch);
static void RecurseEmitExpr(Expr* e);
static void EmitTernary(Expr* e);


static Expr* ArgTempFloat(void)
/* Allocate 8 bytes of memory for ultra-temporary storage. */
{       static Expr e;
	static double d;

	if (e.o)
	    return &e;
	e.o = oConstant;
	e.left = (Expr*)&d;
	e.right = (Expr*)&d;
	e.tp = tp_pointer;
	return &e;
}


static Expr* ArgImmediate(int4 c)
/* Convert this constant integer to a constant integer expression */
/* suitable for the Emit functions. */
{       static Expr e;

	e.o = oConstant;
	e.tp = tp_int;
	e.u.i = c;
	return &e;
}


static Expr* ArgLocalStatic(uint size, void* s)
{       static Expr e;

	e.o = oConstant;
	e.tp = tp_pointer;
	e.left = (Expr*)AllocateAndCopyLocalStatic(size, s, (Type)"i");
	e.right = (Expr*)localstatic_base;
	return &e;
}


static reg_enum FindWorkreg(Expr* e)
/* This is an expression with a reg_stack or reg_discard as the	storereg. */
/* But if we need a register to perform this single instruction, then	*/
/* which register can we use? */
{
	if (IsGeneral(e->storereg))
	    return e->storereg;
	if (e->o == oMEMIA) {
	    memia_type memia=(memia_type)e->u.p;
	    if (memia->e1 and IsGeneral(memia->e1->storereg))
		return memia->e1->storereg;
	    if (memia->e2 and IsGeneral(memia->e2->storereg))
		return memia->e2->storereg;
	}
	else {
	    if (e->left and IsGeneral(e->left->storereg))
		return e->left->storereg;
	    if (e->right and IsGeneral(e->right->storereg))
		return e->right->storereg;
	}
	if (e->workreg)
	    return e->workreg;
	assert(false);
	return reg_discard;
}


static unsigned int BitfieldMask(int bit_width)
/* Identify if this bit-vector is all 1's or all 0's for this width. */
{	unsigned int mask=0;

	while (bit_width-- > 0)
	    mask = (mask << 1) | 1;
	return mask;
}


static bool BitfieldIsAllOnOff(unsigned int x, int bit_width, bool b)
/* Identify if this bit-vector is all 1's or all 0's for this width. */
{
	if (b)
	    return x == BitfieldMask(bit_width);
	else return (x & BitfieldMask(bit_width)) == 0;
}


static void RecurseEmitExpr(Expr* e)
/* If any function can be regarded as the 'heart' of the compiler, then */
/* this is it.  The purpose of this function is to emit all code for    */
/* the given expression-tree.
Note:

Each invocation of this function must [emit code to] take data from
the operands via the e->left->workreg and e->right->workreg fields,
and perform the operation, leaving the result in e->storereg.

Usually, e->workreg == e->storereg, however if we ever run out of
registers to compute some expression and are forced to use the
stack, then e->storereg represents where the value is stored
in-between being computed and being used, and e->workreg represents
a free register that we can pop the value into for computing the
operation.  This is why we must take the values out of e->left->workreg
and e->right->workreg and put it into e->storereg.

Note also that we're not allowed to use any register other than those 3.

If we have some expression with a side-effect, e.g. an assignment, then
e->storereg will be 'reg_discard' if we don't need the value (after
storing it in the memory location), but e->storereg will be some register
if the parent expression is relying on being able to find the value in
this register.  In other words, if e->storereg says we must put the
value somewhere, then we must put it there!

Special values of e->storereg are:
reg_discard	: don't need the result
reg_stack	: put it onto the stack
reg_fpstack	: put it in an FPU register
reg_operand	: the operand comes from a MEMIA expression or constant expr.
*/
{       reg_enum dreg, tmpreg;
        Expr *sub1, *sub2;
        memia_type memia;
	exprf_type ef;
        int i, op, x;

	if (e == NULL)
	    return;

	/* Generate code for the operands: */
	switch (e->o) {
	    case oConstant:
	    case oLocalVar:
	    case oPUSHSPPLUS:
			break;

	    case oNOP1:	RecurseEmitExpr(e->left);
			return;

	    case oDER: case oNEG: case oNOT:
	    case oSIN: case oCOS: case oTAN: case oATAN: case oSQRT:
			RecurseEmitExpr(e->left);
			break;

	    case oCONVERT:
			if (TpIsIntegral(e->tp) and TpIsIntegral(e->left->tp)
				and TpSize(e->tp) > TpSize(e->left->tp)
				and e->left->o == oDER and IsGeneral(e->storereg)
				and e->left->storereg == reg_operand) {
			    RecurseEmitExpr(e->left->left);
			    EmitMOVSXZX(e->storereg, TpSize(e->tp), e->left);
			    return;
			}
			RecurseEmitExpr(e->left);
			break;

	    case oNOTT: break;

	    case oMEMIA:memia = (memia_type)e->u.p;
			if (e->right_first)
			    sub1 = memia->e2, sub2 = memia->e1;
			else sub1 = memia->e1, sub2 = memia->e2;
                        if (sub1)
                            assert(sub1->storereg != reg_discard);

			HAVE_SUBS:
			RecurseEmitExpr(sub1);
			RecurseEmitExpr(sub2);
			if (sub1 and sub1->storereg == reg_stack)
			    EmitPOP(sub1->workreg);
			if (sub2) {
			    assert(e->o == oCOMMA or sub2->o == oPUSH or
                            		sub2->storereg != reg_stack);
                            //assert(sub2->storereg != reg_discard);
                        }
			break;

	    case Binaries:
            case oBITFIELD:
			if (e->right_first)
			    sub1 = e->right, sub2 = e->left;
			else sub1 = e->left, sub2 = e->right;
			goto HAVE_SUBS;

            case oPUSH: RecurseEmitExpr(e->left);
			if (e->right)
                            RecurseEmitExpr(e->right);
                        break;

	    case oFUNC:	break;

	    case oTERNARY:
			break;

	    default:	assert(false);
			break;
	}


	/* At this point, we have emitted instructions which will guarantee */
	/* that all the operands for the instruction have been computed,    */
	/* and (apart from oFUNC) are in the subexpressions' "workreg"	    */
	/* field.  Our object now is to take the operands out of the	    */
	/* child->workregs, perform the operation, and write the value into */
	/* e->storereg. */


	/* Emit the instruction itself, given that all prerequisite values have */
	/* been emitted. */
        switch (e->o) {

            case oConstant: if (e->storereg == reg_operand)
				break;
			    else if (e->storereg == reg_stack) {
                                EmitPUSH(reg_imm, e);
                            }
                            else if (e->storereg == reg_fpstack) {
                                EmitFLoad(e->tp, reg_imm, e);
                            }
                            else if (e->storereg == reg_discard) {
				if (e->src > CurrentSource and e->src[-1] == ',')
				    ErrorParse(e->src, "You are ignoring this constant (don't put commas in numbers)");
				else ErrorParse(e->src, "You are ignoring this constant.");
			    }
			    else if (e->storereg == reg_operand) {
			    }
			    else {
                                EmitMOV(e->storereg, NULL, reg_imm, e, TpSize(e->tp));
                            }
                            break;

            case oLocalVar: if (e == &frame_register)
				break;
			    assert(false);
			    break;

            case oPUSHSPPLUS:
            		    tmpreg = e->workreg;
                            if (e->u.i == 0)
                            	EmitPUSH(reg_SP, NULL);
                            else {
                                assert(IsGeneral(tmpreg));
                                // If it was assigned and immediately released
                                // then it can't be on the stack. We just want
                                // a purely temporary register here.
                                EmitMOV(tmpreg, NULL, reg_SP, NULL, 4);
                                EmitArithI(0/*ADD*/,
                                            tmpreg, NULL,
                                            reg_imm, e,
                                            4);
                                EmitPUSH(tmpreg, NULL);
                            }
            		    break;

            case oMEMIA:    if (e->storereg == reg_operand)
				break;
			    else if (IsGeneral(e->storereg))
				EmitLEA(e->storereg, e);
			    else if (e->storereg == reg_stack) {
				reg_enum reg = FindWorkreg(e);
				EmitLEA(reg, e);
				EmitPUSH(reg, NULL);
			    }
			    else if (e->storereg == reg_discard)
				;
			    else assert(false);
			    break;

            case oDER:      if (e->storereg == reg_operand or e->storereg == reg_discard)
				break;	    /* It will be dereferenced as an operand. */
			    else if (IsFloat(e)) {
                                EmitFLoad(e->tp, reg_mem, e->left);
                                if (e->storereg == reg_stack)
                                    EmitFStoreOnStack(e->tp);
                            }
                            else if (TpSize(e->tp) <= 4) {
				if (e->left->o == oMEMIA or e->left->o == oConstant) {
				    EmitMOV(e->storereg, 0, reg_mem, e->left, TpSize(e->tp));
				}
				else if (IsGeneral(e->left->storereg)) {
				    EmitMOV(e->storereg, NULL, e->left->storereg, dereference, TpSize(e->tp));
				}
				else if (e->left->storereg == reg_operand) {
				    EmitMOV(e->storereg, NULL, reg_mem, e->left, TpSize(e->tp));
				}
				else assert(false);
			    }
			    else {
                                /* Put a struct onto the stack. */
                                i = TypeSizeWord(e->type);
                                EmitADD_SP(-i);
                                EmitMOV(reg_CX, NULL, reg_imm, ArgImmediate((i+3) / 4), 4);
				if (IsGeneral(e->left->storereg))
				    EmitMOV(reg_SI, NULL, e->left->storereg, NULL, 4);
				else if (e->left->o == oConstant)
				    EmitMOV(reg_SI, NULL, reg_imm, e->left, 4);
				else assert(false);
                                EmitMOVSP(reg_DI, reg_stack);
                                EmitREPMOVS(4);
                            }
                            break;

            case oADD:      if (IsFloat(e)) {
                                op = 0;
                                goto SIMPLE_FLOAT_OP;
                            }
			    else {
                                op = 0;
                                goto SIMPLE_INTEGER_OP;
			    }

            case oSUB:      if (IsFloat(e)) {
                                op = 4;
                                goto SIMPLE_FLOAT_OP;
                            }
                            else {
                                op = 5;
                                goto SIMPLE_INTEGER_OP;
                            }

            case oAND:      op = 4;
                            goto SIMPLE_INTEGER_OP;

	    case oOR:       op = 1;
                            goto SIMPLE_INTEGER_OP;

            case oXOR:      op = 6;
                            goto SIMPLE_INTEGER_OP;

                            SIMPLE_INTEGER_OP:
                            if (e->left->storereg == reg_operand) {
                                if (e->storereg == reg_discard and e->right->o == oDER
                                            and e->right->left->o == oConstant) {
				    /* A registerless operation */
                                    assert(IsComparison(e->o));
                                    EmitArithI(op, 
                                                reg_mem, e->right->left, 
                                                reg_imm, e->left, 
                                                TpSize(e->left->tp));
                                    e->numiregs = ReverseCondition(e->numiregs);            
                                }
                                else if (e->storereg == reg_discard and e->right->o == oConstant
					    and e->left->o == oDER and e->left->left->o == oConstant) {
				    /* A registerless operation */
                                    assert(IsComparison(e->o));
                                    EmitArithI(op, 
                                                reg_mem, e->left->left, 
                                                reg_imm, e->right,
                                                TpSize(e->right->tp));
                                    // don't reverse:   e->numiregs = ReverseCondition(e->numiregs);
                                }
                                else {
                                    assert(IsGeneral(e->right->workreg));
                                    if (e->o == oSUB) {
                                        EmitNEG(e->right->workreg);
                                        op = 0;
                                    }
				    if (e->left->o == oConstant)
					EmitArithI(op,
						e->right->workreg, NULL,
                                                reg_imm, e->left, 
                                                TpSize(e->left->tp));
				    else if (e->left->o == oDER)
					EmitArithI(op,
                                                e->right->workreg, NULL,
                                                reg_mem, e->left->left, 
                                                TpSize(e->left->tp));
				    else assert(false);
				    if (e->storereg != e->right->workreg and e->storereg != reg_discard)
					EmitMOV(e->storereg, NULL, e->right->workreg, NULL, TpSize(e->tp));
                                }
                            }
                            else if (e->right->o == oConstant) {
				if (e->storereg == reg_discard and e->left->o == oDER and
					(e->left->left->o == oMEMIA or e->left->left->o == oConstant)) {
				    /* A registerless operation */
                                    assert(IsComparison(e->o));
                                    EmitArithI(op,
                                            reg_mem, e->left->left, 
                                            reg_imm, e->right, 
                                            TpSize(e->left->tp));
                                }
				else {
				    assert(IsGeneral(e->left->workreg));
                                    EmitArithI(op,
                                            e->left->workreg, NULL,
                                            reg_imm, e->right, 
                                            TpSize(e->left->tp));
				    if (e->storereg != reg_discard and e->storereg != e->left->workreg)
					EmitMOV(e->storereg, NULL, e->left->workreg, NULL, TpSize(e->tp));
                                }
                            }
                            else if (e->right->o == oDER and e->right->left->o == oConstant) {
                                assert(e->left->workreg);
                                EmitArithI(op,
					e->left->workreg, NULL,
                                        reg_mem, e->right->left, 
                                        TpSize(e->left->tp));
				if (e->storereg != reg_discard and e->storereg != e->left->workreg)
				    EmitMOV(e->storereg, NULL, e->left->workreg, NULL, TpSize(e->tp));
                            }
                            else if (e->left->o == reg_operand and IsGeneral(e->right->workreg)) {
				assert(e->left->o == oDER and 
				    (e->left->left->o == oConstant or e->left->left->o == oMEMIA));
                                EmitArithI(op,
                                        e->right->workreg, NULL,
                                        reg_mem, e->left->left, 
                                        TpSize(e->left->tp));                      
				if (e->storereg != reg_discard and e->storereg != e->right->workreg)
				    EmitMOV(e->storereg, NULL, e->right->workreg, NULL, TpSize(e->tp));
                            }
                            else if (IsGeneral(e->left->workreg) and e->right->storereg == reg_operand) {
				assert(e->right->o == oDER and 
				    (e->right->left->o == oConstant or e->right->left->o == oMEMIA
                                    or IsGeneral(e->right->left->storereg)));
                                    // These are the conditions for the value being
                                    // expressible as a simple Pentium operand.
                                if (e->o == oSUB) {
                                    EmitNEG(e->left->workreg);
                                    op = 0;
                                }
				else if (IsComparison(e->o)) {
				    EmitArithI(op,
                                        e->left->workreg, NULL,
                                        reg_mem, e->right->left, 
                                        TpSize(e->left->tp));
				    assert(e->storereg == reg_discard);
				    break;
				}
                                EmitArithI(op,
                                        e->left->workreg, NULL,
                                        reg_mem, e->right->left, 
                                        TpSize(e->left->tp));                      
				if (e->storereg != reg_discard and e->storereg != e->left->workreg)
				    EmitMOV(e->storereg, NULL, e->left->workreg, NULL, TpSize(e->tp));
                            }
                            else {
                                assert(IsGeneral(e->left->workreg));
                                assert(IsGeneral(e->right->workreg));
				if (e->right->workreg == e->storereg and op != 5) {
				    EmitArithI(op,
					    e->right->workreg, NULL, 
					    e->left->workreg, NULL,
					    TpSize(e->left->tp));
				}
                                else {
				    EmitArithI(op,
                                        e->left->workreg, NULL,
                                        e->right->workreg, NULL, 
                                        TpSize(e->left->tp));
				    if (e->storereg != reg_discard and e->storereg != e->left->workreg)
					EmitMOV(e->storereg, NULL, e->left->workreg, NULL, TpSize(e->tp));
				}
                            }
                            break;

                            SIMPLE_FLOAT_OP:
			    if (e->right_first)
				sub1 = e->right, sub2 = e->left, op = ReverseFlop(op);
			    else sub2 = e->right, sub1 = e->left;
			    assert(sub1->storereg == reg_fpstack);
                            if (sub2->storereg == reg_operand and sub2->o == oConstant) {
                                EmitArithF(op, sub2->tp, reg_mem, 
                                            ArgLocalStatic(TpSize(sub2->tp), &sub2->u.f));
                            }
			    else if (sub2->storereg == reg_fpstack and sub1->storereg == reg_fpstack) {
			        EmitArithF(op, sub2->tp, reg_fpstack, NULL);
			    }
                            else if (sub2->o == oDER and sub2->left->o == oConstant) {
                                EmitArithF(op, sub2->tp, reg_mem, sub2->left);
			    }
                            else if (sub2->o == oDER) {
				assert(sub2->left->workreg);
				if (sub2->left->o == oMEMIA) {
				    EmitArithF(op, sub2->tp, reg_mem, sub2->left);
				}
				else {
				    EmitArithF(op, sub2->tp, reg_mem, sub2->left);
				}
                            }
                            else if (sub2->o == oCONVERT) {
                                if (sub2->left->o == oDER and 
					    (sub2->left->left->o == oConstant or sub2->left->left->o == oMEMIA)) {
                                    EmitArithF(op, sub2->left->tp, reg_mem, sub2->left->left);
                                }
                                else if (sub2->left->o == oConstant) {
                                    EmitArithF(op, sub2->left->tp, reg_mem, 
                                            ArgLocalStatic(TpSize(sub2->left->tp), &sub2->left->u.f));
                                }
                                else {
                                    Expr* tmp=ArgTempFloat();
				    assert(sub2->left->workreg != reg_discard);
                                    EmitMOV(reg_mem, tmp, sub2->left->workreg, NULL, 8);
                                    EmitFLoad(sub2->left->tp, reg_mem, tmp);
                                    EmitArithF(op, sub2->left->tp, reg_mem, tmp);
                                }
                            }
                            else {
				assert(false);
				/*assert(sub2->storereg == reg_fpstack);
                                EmitArithF(op, sub2->tp, reg_fpstack, NULL);*/
                            }
                            if (e->storereg == reg_stack)
                                EmitFStoreOnStack(e->tp);
                            break;

	    case oMUL:      if (IsFloat(e)) {
                                op = 1;
                                goto SIMPLE_FLOAT_OP;
                            }
                            else if (e->right->o == oConstant) {
				assert(e->left->workreg != reg_discard);
                                EmitIMUL(e->left->workreg, reg_imm, e->right);
                                if (e->storereg != reg_discard and e->storereg != e->left->workreg)
                                    EmitMOV(e->storereg, NULL, e->left->workreg, NULL, TpSize(e->tp));
                            }
                            else if (e->left->o == oConstant) {
				assert(IsGeneral(e->right->workreg));
				if (e->right->workreg != e->right->workreg)
				    EmitMOV(e->right->workreg, NULL, e->right->workreg, NULL, TpSize(e->tp));
                                EmitIMUL(e->right->workreg, reg_imm, e->left);
                                if (e->storereg != reg_discard and e->storereg != e->right->workreg)
                                    EmitMOV(e->storereg, NULL, e->right->workreg, NULL, TpSize(e->tp));
                            }
                            else if (e->right->o == oDER and
                                        e->right->left->o == oConstant) {
				assert(IsGeneral(e->left->workreg));
                                EmitIMUL(e->left->workreg, reg_mem, e->right->left);
                                if (e->storereg != reg_discard and e->storereg != e->left->workreg)
                                    EmitMOV(e->storereg, NULL, e->left->workreg, NULL, TpSize(e->tp));
                            }
                            else {
                                if (e->left->workreg == reg_AX) {
                                    EmitIMUL(e->left->workreg, e->right->workreg, NULL);
				    if (e->storereg != reg_discard and e->storereg != e->left->workreg)
					EmitMOV(e->storereg, NULL, e->left->workreg, NULL, TpSize(e->tp));
                                }
                                else {
                                    EmitIMUL(e->right->workreg, e->left->workreg, NULL);
				    if (e->storereg != reg_discard and e->storereg != e->right->workreg)
					EmitMOV(e->storereg, NULL, e->right->workreg, NULL, TpSize(e->tp));
                                }
			    }
                            break;

            case oMOD:      dreg = reg_DX;
                            goto MODDIV;

            case oDIV:      if (IsFloat(e)) {
                                op = 6;
                                goto SIMPLE_FLOAT_OP;
                            }
                            dreg = reg_AX;
                            MODDIV:
                            if (e->right->o == oDER and 
                                        e->right->left->o == oConstant) {
				assert(e->left->workreg == reg_AX);
                                if (e->left->workreg != reg_AX)
                                    EmitMOV(reg_AX, NULL, e->left->workreg, NULL, 4);
				if (TpIsUnsigned(e->left->tp))
				    EmitCLEAR(reg_DX);
                                else EmitCDQ();      /* We need to convert AX to DX:AX by extending the sign. */
                                EmitIDIV(reg_mem, e->right->left, TpIsUnsigned(e->tp));
                            }
                            else {
				assert(e->left->workreg == reg_AX);
				assert(IsGeneral(e->right->workreg));
				if (TpIsUnsigned(e->left->tp))
				    EmitCLEAR(reg_DX);
                                else EmitCDQ();      /* We need to convert AX to DX:AX by extending the sign. */
                                EmitIDIV(e->right->workreg, NULL, TpIsUnsigned(e->tp));
                            }
                            if (e->storereg != reg_discard and e->storereg != dreg)
                                EmitMOV(e->storereg, NULL, dreg, NULL, 4);
			    break;

	    case oASSIGN:   if (IsFloat(e)) {
				if (e->right->o == oMEMIA) {
				    assert(e->right->storereg == reg_fpstack);
				    EmitFStore(e->tp, reg_mem, e->left, e->storereg != reg_discard);
				}
				else {
				    if (e->left->storereg != reg_operand)
					EmitFStore(e->tp, e->left->storereg, NULL, e->storereg != reg_discard);
				    else EmitFStore(e->tp, reg_mem, e->left, e->storereg != reg_discard);
                                }
				if (e->storereg == reg_stack)
                                    EmitFStoreOnStack(e->tp);
                            }
			    else if (e->tp == tp_class and TypeSize(e->type) > 4) {
				if (e->right->o == oDER)
				    e->right = e->right->left;
				else if (e->right->o == oFUNC) {
                                    if (e->right->storereg != reg_SI) {
                                        EmitMOV(reg_SI, NULL, reg_BP, NULL, 4);
                                        EmitArithI(0, reg_SI, NULL, reg_imm, ArgImmediate(TypeSize(e->type)), 4);
                                    }
				}
                                else if (e->right->o == oPUSH)
                                    ;
				else assert(false);
				i = (TypeSize(e->type)+3) / 4;
				EmitMOV(reg_CX, NULL, reg_imm, ArgImmediate(i), 4);
				assert(IsGeneral(e->left->workreg));
                                if (e->right->workreg == reg_SI)
                                    ;		// We already have the source in SI
				else if (e->right->o == oMEMIA or e->right->o == oConstant)
				    EmitLEA(reg_SI, e->right);
                                else if (e->right->o == oPUSH)
                                    EmitMOV(reg_SI, NULL, reg_SP, NULL, 4);
				else if (e->right->workreg != reg_SI)
				    EmitMOV(reg_SI, NULL, e->right->workreg, NULL, 4);
                                if (e->left->workreg == reg_DI)
                                    ;		// We already have the dest in DI
				else if (e->left->o == oMEMIA or e->left->o == oConstant)
				    EmitLEA(reg_DI, e->left);
				else if (e->left->workreg != reg_DI)
				    EmitMOV(reg_DI, NULL, e->left->workreg, NULL, 4);
				EmitREPMOVS(4);
                                if (e->right->storereg == reg_stack) {
                                    if (e->storereg == reg_stack)
                                    	;			// Leave it on the stack
                                    else EmitADD_SP(i*4);	// Take it off the stack
                                }
				else if (e->storereg == reg_stack) {
				    EmitADD_SP(-i*4);		// Put it on the stack
				    EmitMOV(reg_CX, NULL, reg_imm, ArgImmediate(i), 4);
                                    EmitArithI(5, reg_SI, NULL, reg_imm, ArgImmediate(i*4), 4);
				    EmitMOVSP(reg_DI, reg_stack);
				    EmitREPMOVS(4);
				}
                                // Else it was not on the stack and we don't want it
                                // on the stack.
			    }
                            else if (e->right->storereg == reg_operand) {
                            	x = (e->tp == tp_class) ? TypeSize(e->type) : TpSize(e->tp);
                                if (e->right->o == oConstant and (e->left->o == oMEMIA or e->left->o == oConstant)) {
				    EmitMOV(reg_mem, e->left, reg_imm, e->right, x);
                                }
                                else if (e->left->o == oConstant or e->left->o == oMEMIA) {
				    EmitMOV(reg_imm, e->left, reg_mem, e->right, x);
				}
                                else {
                                    assert(IsGeneral(e->left->workreg));
                                    EmitMOV(e->left->workreg, dereference, reg_imm, e->right, x);
                                }
				if (e->storereg == reg_stack)
				    EmitPUSH(reg_imm, e->right);
				else if (e->storereg == reg_discard)
				    ;
				else if (IsGeneral(e->storereg) and e->storereg != e->right->workreg) {
                                    if (e->left->workreg == reg_operand)
				        EmitMOV(e->storereg, NULL, reg_mem, e->left, x);
                                    else
				        EmitMOV(e->storereg, NULL, e->left->workreg, dereference, x);
                                }
				else assert(false);
                            }
                            else {
                                assert(IsGeneral(e->right->workreg));
                                if (e->left->o == oMEMIA or e->left->o == oConstant) {
                                    EmitMOV(reg_mem, e->left, e->right->workreg, NULL, TpSize(e->tp));
                                    if (e->storereg != reg_discard and e->storereg != e->right->workreg)
                                        EmitMOV(e->storereg, NULL, e->right->workreg, NULL, TpSize(e->tp));
                                }
                                else {
                                    assert(IsGeneral(e->left->workreg));
                                    EmitMOV(e->left->workreg, dereference, e->right->workreg, NULL, TpSize(e->tp));
                                    if (e->storereg != reg_discard and e->storereg != e->right->workreg)
                                        EmitMOV(e->storereg, NULL, e->right->workreg, NULL, TpSize(e->tp));
                                }
                            }
                            break;

            case oAADD:     op = 0;
                            goto OP_ASSIGN;

            case oASUB:     op = IsFloat(e) ? 4 : 5;
                            goto OP_ASSIGN;

            case oAAND:     op = 4;
                            goto OP_ASSIGN;

            case oAOR:      op = 1;
                            goto OP_ASSIGN;

	    case oAXOR:     op = 6;
                            goto OP_ASSIGN;

                            OP_ASSIGN:
                            if (IsFloat(e)) {
                                if (e->left->o == oConstant or e->left->o == oMEMIA) {
                                    e->right->workreg = reg_fpstack;
                                    if (op == 7)        // Reverse the order for non-commutative operations
                                        op = 6;
                                    else if (op == 6)
                                        op = 7;
                                    else if (op == 5)
                                        op = 4;
                                    else if (op == 4)
                                        op = 5;
                                    EmitArithF(op, e->tp, reg_mem, e->left);
                                    EmitFStore(e->tp, reg_mem, e->left, e->storereg != reg_discard);
                                }
                                else {
                                    Expr* tmp=ArgTempFloat();
                                    EmitArithF(op, e->tp, reg_mem, e->left);
                                    EmitFStore(e->tp, reg_mem, tmp, e->storereg != reg_discard);
                                    //EmitMOV(e->left->reg, 0, reg_mem, tmp, 8);
                                }
                                if (e->storereg == reg_stack)
                                    EmitFStoreOnStack(e->tp);
                            }
                            else if (e->right->o == oConstant) {
                                if (e->left->o == oConstant or e->left->o == oMEMIA) {
                                    if (e->storereg == reg_discard)
                                        EmitArithI(op,
                                                reg_mem, e->left,
                                                reg_imm, e->right, 
                                                TpSize(e->tp));
                                    else if (e->storereg == reg_stack) {
                                        EmitArithI(op,
						reg_mem, e->left,
                                                reg_imm, e->right,
                                                TpSize(e->tp));
                                        EmitPUSH(reg_mem, e->left);
                                    }
                                    else {
                                        assert(IsGeneral(e->storereg));
                                        EmitArithI(op,
                                                reg_mem, e->left,
                                                reg_imm, e->right, 
                                                TpSize(e->tp));
                                        EmitMOV(e->storereg, NULL, reg_mem, e->left, TpSize(e->tp));
                                    }
                                }
                                else {
                                    assert(IsGeneral(e->left->workreg));
                                    EmitArithI(op,
                                                e->left->workreg, dereference,
                                                reg_imm, e->right,
                                                TpSize(e->tp));
				    if (e->storereg != reg_discard)
					EmitMOV(e->storereg, NULL, e->left->workreg, dereference, TpSize(e->tp));
                                }
                            }
                            else {
                                assert(IsGeneral(e->right->workreg));
                                if (e->left->o == oConstant or e->left->o == oMEMIA) {
                                    EmitArithI(op,
                                                reg_mem, e->left, 
                                                e->right->workreg, NULL,
                                                TpSize(e->tp));
				    if (e->storereg != reg_discard)
					EmitMOV(e->storereg, NULL, reg_mem, e->left, TpSize(e->tp));
                                }
                                else {
                                    EmitArithI(op,
						e->left->workreg, dereference,
                                                e->right->workreg, NULL, 
                                                TpSize(e->tp));
				    if (e->storereg != reg_discard)
					EmitMOV(e->storereg, NULL, e->left->workreg, dereference, TpSize(e->tp));
                                }
                            }
                            break;

            case oAADDAFTER:op = 0;
                            if (not e->storereg)
                                goto OP_ASSIGN;
                            assert(e->right->o == oConstant);
                            if (not IsFloat(e)) {
                                if (e->left->o == oMEMIA or e->left->o == oConstant) {
                                    if (e->storereg != reg_discard)
					EmitMOV(e->storereg, NULL, reg_mem, e->left, TpSize(e->tp));
                                    EmitArithI(op,
                                                reg_mem, e->left,
                                                reg_imm, e->right,
                                                TpSize(e->tp));
                                }
                                else {
                                    assert(IsGeneral(e->left->storereg) and e->storereg != e->left->storereg);
                                    if (e->storereg != reg_discard)
					EmitMOV(e->storereg, NULL, e->left->storereg, dereference, TpSize(e->tp));
                                    EmitArithI(op,
                                                e->left->storereg, dereference,
                                                reg_imm, e->right,
                                                TpSize(e->tp));
                                }
                            }
                            else {
                                if (e->left->o == oMEMIA or e->left->o == oConstant) {
                                    EmitFLoad(e->tp, reg_mem, e->left);
                                    EmitFCopyTop();
				    //assert(e->right->tp == e->tp);
                                    if (e->right->u.f >= 0) {
                                        EmitFLoad(e->tp, reg_imm, e->right);
                                        EmitArithF(0, tp_double, reg_fpstack, NULL);
                                    }
                                    else {
                                        e->right->u.f = - e->right->u.f;		/* $$$ We're destroying evidence */
                                        EmitFLoad(e->tp, reg_imm, e->right);
                                        EmitArithF(5, tp_double, reg_fpstack, NULL);
                                    }
                                    EmitFStore(e->tp, reg_mem, e->left, no);
                                }
                                else {
                                    assert(IsGeneral(e->left->workreg));
                                    EmitFLoad(e->tp, e->left->workreg, NULL);
                                    EmitFCopyTop();
                                    //assert(e->right->tp == e->tp);
                                    if (e->right->u.f >= 0) {
                                        EmitFLoad(tp_int, reg_imm, e->right);
                                        EmitArithF(0, tp_double, reg_fpstack, NULL);
                                    }
                                    else {
                                        e->right->u.f = - e->right->u.f;		/* $$$ We're destroying evidence */
                                        EmitFLoad(tp_int, reg_imm, e->right);
                                        EmitArithF(5, tp_double, reg_fpstack, NULL);
                                    }
                                    EmitFStore(e->tp, e->left->workreg, NULL, no);
                                }
                                if (e->storereg == reg_stack)
                                    EmitFStoreOnStack(e->tp);
                            }
                            break;

            case oADIV:     if (IsFloat(e)) {
                                op = 6;
                                goto OP_ASSIGN;
			    }
			    dreg = reg_AX;
			    goto AMODDIV;

            case oAMOD:     dreg = reg_DX;
			    AMODDIV:
			    assert(not IsFloat(e));
                            if (e->storereg == 0)
                                e->storereg = reg_AX;

                            /* Step 1: Get the numerator: */
                            if (e->left->o == oConstant)
                                EmitMOV(reg_AX, NULL, reg_mem, e->left, TpSize(e->tp));
                            else if (e->left->o == oMEMIA)
                                EmitMOV(reg_AX, NULL, reg_mem, e->left, TpSize(e->tp));
                            else EmitMOV(reg_AX, NULL, e->left->workreg, dereference, TpSize(e->tp));
			    if (TpSize(e->tp) < 4)
				EmitMOVSXZX(reg_AX, 4, e);
			    if (TpIsUnsigned(e->tp))
				EmitCLEAR(reg_DX);
                            else EmitCDQ();      /* We need to convert AX to DX:AX by extending the sign. */

                            /* Step 2: Do the division: */
                            if (e->right->o == oDER and e->right->left->o == oConstant)
                                EmitIDIV(reg_mem, e->right->left, TpIsUnsigned(e->tp));
                            else if (IsGeneral(e->right->workreg))
                                EmitIDIV(e->right->workreg, NULL, TpIsUnsigned(e->tp));
                            else if (e->right->o == oConstant)
                                EmitIDIV(reg_mem, ArgLocalStatic(4, &e->right->u.i), TpIsUnsigned(e->tp));
                            else if (e->right->o == oDER)
				EmitIDIV(reg_mem, e->right->left, TpIsUnsigned(e->tp));
			    else assert(false);

                            /* Step 3: Put the value back. */
                            if (e->left->o == oConstant or e->left->o == oMEMIA)
                                EmitMOV(reg_mem, e->left, dreg, NULL, TpSize(e->tp));
			    else {
                                assert(IsGeneral(e->left->workreg) and e->left->workreg != reg_AX);
                                EmitMOV(e->left->workreg, dereference, dreg, NULL, TpSize(e->tp));
                            }

                            /* Step 4: The return value */
                            if (e->storereg != reg_discard and e->storereg != reg_operand
					and e->storereg != dreg)
                                EmitMOV(e->storereg, NULL, dreg, NULL, TpSize(e->tp));
                            break;

            case oAMUL:     if (IsFloat(e)) {
                                op = 1;
                                goto OP_ASSIGN;
                            }

                            /* Step 1: Work out what register to hold the result. */
			    if (IsGeneral(e->storereg))
				dreg = e->storereg;
			    else if (IsGeneral(e->right->workreg))
				dreg = e->right->workreg;
			    else if (IsGeneral(e->storereg))
				dreg = e->storereg;
			    else assert(false);

			    /* Step 2: Do the multiplication. */
			    if (dreg == e->right->workreg) {
                                if (e->left->o == oConstant or e->left->o == oMEMIA) {
                                    EmitIMUL(dreg, reg_mem, e->left);
                                }
                                else {
                                    EmitIMUL(dreg, e->left->workreg, dereference);
                                }
                            }
			    else if (dreg == e->left->workreg) {
                                if (e->right->o == oConstant) {
				    EmitIMUL(dreg, reg_imm, e->right);
                                }
                                else if (e->right->o == oDER and e->right->left->o == oMEMIA) {
                                    EmitIMUL(dreg, reg_mem, e->right->left);
                                }
                                else {
                                    EmitIMUL(dreg, e->right->workreg, NULL);
                                }
                            }
			    else {
                                EmitMOV(dreg, NULL, e->left->workreg, dereference, TpSize(e->tp));
				EmitIMUL(dreg, e->right->workreg, NULL);
                            }

                            /* Step 3: Move the product back into memory. */
                            if (e->left->o == oConstant or e->left->o == oMEMIA)
                                EmitMOV(reg_mem, e->left, dreg, NULL, TpSize(e->tp));
                            else {
                                assert(IsGeneral(e->left->workreg));
                                EmitMOV(e->left->workreg, dereference, dreg, NULL, TpSize(e->tp));
                            }

                            /* Step 4: The return value */
                            if (e->storereg != reg_discard and e->storereg != dreg)
                                EmitMOV(e->storereg, NULL, dreg, NULL, TpSize(e->tp));
                            break;


            case oNEG:      if (IsFloat(e)) {
                                EmitFloatNEG();
                                if (e->storereg == reg_stack)
                                    EmitFStoreOnStack(e->tp);
                            }
                            else {
				assert(IsGeneral(e->left->workreg));
			        EmitNEG(e->left->workreg);
				if (e->storereg != e->left->workreg)
                                    EmitMOV(e->storereg, NULL, e->left->workreg, NULL, TpSize(e->tp));
                            }
                            break;

            case oNOT:      assert(not IsFloat(e));
                            EmitNOT(e->left->workreg);
                            if (e->storereg != e->left->workreg)
                                EmitMOV(e->storereg, NULL, e->left->workreg, NULL, TpSize(e->tp));
                            break;

            case oSIN:
            case oCOS:
            case oTAN:
            case oATAN:
            case oSQRT:     assert(e->tp == tp_double);
                            EmitFSpecial(e->o);
                            if (e->storereg == reg_stack)
                                EmitFStoreOnStack(e->tp);
                            break;

            case oFUNC:     ef = e->u.func;

            		    /* Does the function return a struct to the stack? */
                            if (ef->arity and ef->param[0]->o == oPUSHSPPLUS)
                            	EmitADD_SP(-(int)ef->param[0]->right);

			    /* Output all parameters onto the stack: */
                            for (i=ef->arity-1; i >= 0; i--) {
                                assert(ef->param[i]->storereg == reg_stack);
                                RecurseEmitExpr(ef->param[i]);
                            }

			    /* Emit the CALL instruction: */
                            if (ef->func->o == oConstant)
                                EmitCALL((void*)ef->func->u.p);
			    else if (ef->func->o == oDER) {
				RecurseEmitExpr(ef->func->left);
                                EmitCALL(ef->func->left);
			    }
                            else {
                                RecurseEmitExpr(ef->func);
				assert(IsGeneral(ef->func->storereg));
                                EmitCALL(ef->func->storereg);
                            }

			    /* Get rid of the parameter stack frame: */
                            x = ef->stack_size;
                            if (ef->flags & 1) {
                            	// Pop the 1st param into AX
                            	EmitPOP(reg_AX), x -= 4;
                            }
                            else if (e->storereg == reg_discard and ef->arity and
                            		ef->param[0]->o == oPUSHSPPLUS)
                                x += (int)ef->param[0]->right;
                            EmitADD_SP(x);

			    /* Move the return value into the appropriate place: */
			    if (e->tp == tp_void or e->tp == 0)
				assert(e->storereg == reg_discard);
			    else if (e->storereg == reg_discard) {
				if (IsFloat(e))
				    EmitFPOP();
			    }
			    else if (e->storereg == reg_stack) {
				if (IsFloat(e))
				    EmitFStoreOnStack(e->tp);
				else if (e->ref or (i=TypeSizeWord(e->type)) <= 4)
				    EmitPUSH(reg_AX, 0);
				else {
                                    if (ef->arity and ef->param[0]->o == oPUSHSPPLUS) {
                                    	// It's already been placed on the stack
                                    }
				    else {
                                        //EmitADD_SP(-i);  This is done above.
                                        EmitMOV(reg_CX, NULL, reg_imm, ArgImmediate((i+3) / 4), 4);
                                        EmitMOV(reg_SI, NULL, reg_SP, NULL, 4);
                                        EmitByte(0x8b); EmitByte(0xb6); EmitInt4(-x); // MOV SI, [SI-x]
                                        EmitMOVSP(reg_DI, reg_stack);
                                        EmitREPMOVS(4);
                                    }
				}
			    }
			    else if (IsFloat(e)) {
				assert(e->storereg == reg_fpstack);
			    }
                            else if (e->storereg == reg_SI and e->tp == tp_class
                            	and (i=TypeSize(e->type)) > 4) {
                                // LEA SI,[DI-i]
                                EmitByte(0x8d);
                                EmitByte(0x80 | (RegEncoding(reg_SI)<<3) | RegEncoding(reg_DI));
                                EmitInt4(-i);
                            }
			    else if (e->storereg != reg_AX and e->storereg != reg_discard) {
				EmitMOV(e->storereg, NULL, reg_AX, NULL, 4);
			    }
                            break;

            case oCOMMA:    break;

	    case oNOP1:	    break;

	    case oLT:       op = 12;
                            goto COMPARISON;

            case oGE:       op = 13;
                            goto COMPARISON;

            case oEQ:       op = 4;
			    goto COMPARISON;

            case oNE:       op = 5;
                            goto COMPARISON;

            case oLE:       op = 14;
                            goto COMPARISON;

            case oGT:       op = 15;
                            goto COMPARISON;

                            COMPARISON:
                            if (not IsFloat(e->left)) {
                                if (e->right_first)
                                    op = ReverseCondition(op);
                                e->numiregs = op;       // Store the condition in e->numiregs.
                                assert(e->storereg == reg_discard);
                                op = 7;
                                goto SIMPLE_INTEGER_OP;
                            }
                            else {
				assert(e->right->workreg != reg_discard and e->left->workreg != reg_discard);
                                if (not e->right_first)
                                    op = ReverseCondition(op);
                                /* The FPU 'FCOMIP' instruction sets the EFLAGS
                                register directly, but only the CF & ZF flags.
                                So the JGT/JLT etc. instructions don't work,
                                instead we need JA/JB etc. (Jump if above/below). */
                                if (op == 14)
                                    op = 6;
                                else if (op == 15)
                                    op = 7;
                                else if (op == 13)
                                    op = 3;
                                else if (op == 12)
                                    op = 2;
                                e->numiregs = op;       // Store the condition in e->numiregs.
                                EmitFCOMIP();
                                EmitFPOP();
                            }
                            break;

            case oCONVERT:  if (IsFloat(e)) {
                                /* It's easy to convert anything (except a char) to a double. */
				if (e->left->storereg == reg_fpstack)
				    ;
				else if (e->left->o == oConstant)
				    EmitFLoad(e->left->tp, reg_imm, e->left);
				else if (e->left->o == oDER) {
				    EmitFLoad(e->left->tp, reg_mem, e->left->left);
				}
				else {
				    Expr* tmp=ArgTempFloat();
				    assert(e->left->workreg);
				    EmitMOV(reg_mem, tmp, e->left->workreg, NULL, TpSize(e->left->tp));
				    EmitFLoad(e->left->tp, reg_mem, tmp);
				}
				if (e->storereg == reg_stack)
				    EmitFStoreOnStack(e->tp);
			    }
			    else if (IsFloat(e->left)) {
				/* Converting FROM a double: */
				Expr* tmp=ArgTempFloat();
				assert(e->left->storereg);
				EmitFStore(e->tp, reg_mem, tmp, no);
				EmitMOV(e->storereg,NULL, reg_mem,tmp, TpSize(e->tp));
			    }
			    else if (e->storereg == reg_discard)
				;	// We must be doing an EmitExamine().
                            else {
				/* Converting to and from integer values */
				assert(TpIsIntegral(e->tp) or e->tp == tp_pointer);
				if (TpSize(e->tp) <= TpSize(e->left->tp)) {
				    /* Going from a larger size to a smaller size: no change. */
                                    if (e->storereg == e->left->storereg or e->storereg == reg_discard)
                                        break;
                                    if (e->left->storereg == reg_operand) {
                                        if (e->left->o == oDER)
                                            EmitMOV(e->storereg,NULL, reg_mem,e->left->left, TpSize(e->tp));
                                        else assert(false);
                                    }
                                    else EmitMOV(e->storereg,NULL, e->left->storereg,NULL, TpSize(e->tp));
                                }
				else if (IsGeneral(e->storereg)) {
				    EmitMOVSXZX(e->storereg, TpSize(e->tp), e->left);
				}
				else if (IsGeneral(e->left->workreg)) {
				    EmitMOVSXZX(e->left->workreg, TpSize(e->tp), e->left);
				    if (e->storereg == reg_stack)
					EmitPUSH(e->left->workreg, NULL);
				    else {
					assert(false);
				    }
				}
				else {
				    dreg = e->left->workreg;
				    assert(IsGeneral(dreg));
				    // The MOVSX/MOVZX instructions
				    // require a register to be used.
				    EmitMOVSXZX(dreg, TpSize(e->tp), e->left);
				    assert(e->storereg == reg_stack);
				    EmitPUSH(dreg, NULL);
				}
                            }
                            break;

            case oSHL:      op = 4;
                            goto SHIFT;

            case oSHR:      op = 5;
                            goto SHIFT;

                            SHIFT:
                            if (e->right->o == oConstant) {
                                assert(e->right->tp == tp_int);
                                if (IsGeneral(e->left->workreg))
				    EmitShift(op, e->left->workreg, NULL, reg_imm, e->right, e->tp);
				else assert(false);
                            }
                            else {
                                assert(e->right->workreg == reg_CX);
                                EmitShift(op, e->left->workreg, NULL, reg_CX, NULL, e->tp);
                            }
                            if (e->storereg != e->left->workreg) {
				if (IsGeneral(e->left->workreg))
				    EmitMOV(e->storereg, NULL, e->left->workreg, NULL, TpSize(e->left->tp));
				else assert(false);
			    }
                            break;

            case oASHL:     op = 4;
                            goto ASSIGN_SHIFT;

            case oASHR:     op = 5;
                            goto ASSIGN_SHIFT;

                            ASSIGN_SHIFT:
                            if (e->right->o == oConstant) {
                                assert(e->right->tp == tp_int);
                                if (e->left->o == oMEMIA or e->left->o == oConstant) {
                                    EmitShift(op, reg_mem, e->left, reg_imm, e->right, e->tp);
                                    if (e->storereg != reg_discard)
                                        EmitMOV(e->storereg, NULL, reg_mem, e->left, TpSize(e->tp));
                                    break;
                                }
                                else {
				    assert(e->left->workreg);
                                    EmitShift(op, e->left->workreg, dereference, reg_imm, e->right, e->tp);
                                }
                            }
                            else {
                                assert(e->right->workreg == reg_CX);
                                if (e->left->o == oMEMIA) {
				    assert(e->right->workreg);
                                    EmitShift(op, reg_mem, e->left, reg_CX, NULL, e->tp);
                                    if (e->storereg != reg_discard)
                                        EmitMOV(e->storereg, NULL, reg_mem, e->left, TpSize(e->tp));
                                    break;
                                }
                                else if (IsGeneral(e->left->workreg)) {
                                    EmitShift(op, e->left->workreg, dereference, reg_CX, NULL, e->tp);
                                }
                                else if (e->left->storereg == reg_operand) {
                                    EmitShift(op, reg_mem, e->left, reg_CX, NULL, e->tp);
                                }
				else assert(false);
                            }
                            if (e->storereg != reg_discard and e->storereg != e->left->workreg) {
				if (e->left->storereg == reg_operand)
				    EmitMOV(e->storereg, NULL, reg_mem, e->left, TpSize(e->right->tp));
				else EmitMOV(e->storereg, NULL, e->left->workreg, NULL, TpSize(e->right->tp));
			    }
                            break;

            case oTERNARY:  /* The ternary operator is compiled as an expression (instead of being */
                            /* converted to an 'if' construct with temporary variables) in order   */
                            /* that it is computed only when needed and thereby usually won't need */
                            /* any storage to stash it. */
                            EmitTernary(e);
                            break;

	    case oBITFIELD: if (e->right_first) {
				sub1 = e->u.bitfield->addr;
				sub2 = e->u.bitfield->operand;
			    }
			    else {
				sub2 = e->u.bitfield->operand;
				sub1 = e->u.bitfield->addr;
			    }
			    if (sub2 == NULL)
				;
			    else if (sub2->o == oConstant)
				;
			    else if (sub2->o == oMEMIA) {
				if (sub2 == e->u.bitfield->operand)
				    EmitLEA(sub2->storereg, sub2);
			    }
			    else {
				assert(sub2->storereg == sub2->storereg);
			    }
			    if (sub1->storereg == reg_stack) {
				assert(IsGeneral(sub1->storereg));
				EmitMOV(sub1->storereg, NULL, reg_stack, NULL, TpSize(e->tp));
			    }

			    /* Now: is it a read, write, or op-assg? */
			    if (e->u.bitfield->o == oDER) {
				if (IsGeneral(e->storereg))
				    dreg = e->storereg;
				else dreg = e->u.bitfield->addr->storereg;
				if (e->u.bitfield->addr->o == oConstant or e->u.bitfield->addr->o == oMEMIA)
				    EmitMOV(dreg, NULL, reg_mem, e->u.bitfield->addr, TpSize(e->tp));
				else EmitMOV(dreg, NULL, e->u.bitfield->addr->storereg, dereference, TpSize(e->tp));
                                EmitShift(4/*left*/, dreg, NULL, reg_imm,
					ArgImmediate(32 - e->u.bitfield->bit_offset - e->u.bitfield->bit_width), e->tp);
                                EmitShift(5/*right*/, dreg, NULL, reg_imm,
					ArgImmediate(32 - e->u.bitfield->bit_width), e->tp);
				if (e->storereg != reg_discard and e->storereg != dreg)
				    EmitMOV(e->storereg, NULL, dreg, NULL, 4);
			    }
			    else if (e->u.bitfield->o == oASSIGN) {
				i = ((unsigned int)-1 >> (32 - e->u.bitfield->bit_width)) << e->u.bitfield->bit_offset;
				i = ~i;
				if (e->u.bitfield->addr->o == oConstant or e->u.bitfield->addr->o == oMEMIA)
				    tmpreg = reg_mem, sub1 = e->u.bitfield->addr;
				else tmpreg = e->u.bitfield->addr->storereg, sub1 = dereference;
				if (e->u.bitfield->operand->o == oConstant) {
				    x = e->u.bitfield->operand->u.i;
				    if (not BitfieldIsAllOnOff(x, e->u.bitfield->bit_width, 1))
					EmitArithI(4/*AND*/, tmpreg, sub1, reg_imm, ArgImmediate(i), 4);
				    if (not BitfieldIsAllOnOff(x, e->u.bitfield->bit_width, 0))
					EmitArithI(1/*OR*/, tmpreg, sub1,
					    reg_imm, ArgImmediate(
						    (x & BitfieldMask(e->u.bitfield->bit_width))
						    << e->u.bitfield->bit_offset), 4);
				    if (e->storereg != reg_discard)
					EmitMOV(e->storereg, NULL, reg_imm,
						    ArgImmediate(x), 4);
				}
				else {
				    EmitArithI(4/*AND*/, tmpreg, sub1, reg_imm, ArgImmediate(i), 4);
				    dreg = e->u.bitfield->operand->storereg;
				    EmitArithI(4/*AND*/, dreg, NULL, reg_imm,
					    ArgImmediate(BitfieldMask(e->u.bitfield->bit_width)), 4);
				    if (e->storereg != reg_discard)
					EmitMOV(e->storereg, NULL, e->u.bitfield->operand->storereg, NULL, 4);
				    EmitShift(4/*left*/, dreg, NULL, reg_imm,
					ArgImmediate(e->u.bitfield->bit_offset), e->tp);
				    EmitArithI(1/*OR*/, tmpreg, sub1, dreg, NULL, 4);
				}
			    }
			    else if (e->u.bitfield->o == 0)
				assert(false);
			    else {
				/*** An op-assign: ***/

				/* Which register holds the result? */
				dreg = e->u.bitfield->operand->storereg;

				/* Get the left-hand-side: */
				if (e->u.bitfield->addr->o == oConstant or e->u.bitfield->addr->o == oMEMIA)
				    EmitMOV(dreg, NULL, reg_mem, e->u.bitfield->addr, TpSize(e->tp));
				else EmitMOV(dreg, NULL, e->u.bitfield->addr->storereg, dereference, TpSize(e->tp));
                                EmitShift(4/*left*/, dreg, NULL, reg_imm, 
					ArgImmediate(32 - e->u.bitfield->bit_offset - e->u.bitfield->bit_width), e->tp);
                                EmitShift(5/*right*/, dreg, NULL, reg_imm, 
					ArgImmediate(32 - e->u.bitfield->bit_width), e->tp);

				/* Operate with the right-hand-side: */
				op = e->u.bitfield->o;
				if (op == oAADD)
				    op = 0;
				else if (op == oASUB)
				    op = 5;
				else if (op == oAAND)
				    op = 4;
				else if (op == oAOR)
				    op = 1;
				else if (op == oAXOR)
				    op = 6;
				else if (op == oAADDAFTER) {
				    if (e->storereg != reg_discard and e->storereg != dreg)
					EmitMOV(e->storereg, NULL, dreg, NULL, 4);
				    op = 0;
				}
				else if (op == oMUL) {
				    EmitIMUL(dreg, e->u.bitfield->operand->storereg, NULL);
				    goto DONE;
				}
				else assert(false);
				if (e->u.bitfield->operand->o == oConstant)
				    EmitArithI(op, dreg, NULL, reg_imm, 
					    ArgImmediate(e->u.bitfield->operand->u.i), 4);
				else if (e->u.bitfield->operand->o == oMEMIA)
				    EmitArithI(op, dreg, NULL, reg_mem,
					    e->u.bitfield->operand, 4);
				else if (e->u.bitfield->operand->o == oDER
					and e->u.bitfield->operand->left->o == oConstant)
				    EmitArithI(op, dreg, NULL, reg_mem,
					    e->u.bitfield->operand->left, 4);
				else if (e->u.bitfield->operand->o == oDER)
				    EmitArithI(op, dreg, NULL,
					    e->u.bitfield->operand->storereg, dereference, 4);
				else {
				    assert(IsGeneral(e->u.bitfield->operand->storereg));
				    EmitArithI(op, dreg, NULL,
					    e->u.bitfield->operand->storereg, NULL, 4);
				}
				DONE:

				/* Put the result into the expression return place: */
				if (e->storereg != reg_discard and e->storereg != dreg and e->u.bitfield->o != oAADDAFTER)
				    EmitMOV(e->storereg, NULL, dreg, NULL, 4);

				/* Put the result back into the memory location: */
				i = ((unsigned int)-1 >> (32 - e->u.bitfield->bit_width)) << e->u.bitfield->bit_offset;
				i = ~i;
				if (e->u.bitfield->addr->o == oConstant or e->u.bitfield->addr->o == oMEMIA)
				    tmpreg = reg_mem, sub1 = e->u.bitfield->addr;
				else tmpreg = e->u.bitfield->addr->storereg, sub1 = dereference;
				EmitArithI(4/*AND*/, tmpreg, sub1, reg_imm, ArgImmediate(i), 4);
				EmitArithI(4/*AND*/, dreg, NULL, reg_imm,
					ArgImmediate(BitfieldMask(e->u.bitfield->bit_width)), 4);
				EmitShift(4/*left*/, dreg, NULL, reg_imm,
				    ArgImmediate(e->u.bitfield->bit_offset), e->tp);
				EmitArithI(1/*OR*/, tmpreg, sub1, dreg, NULL, 4);
			    }
			    break;

            case oPUSH:     if (e->left->storereg != reg_stack)
            			EmitPUSH(e->left->storereg, e);
            		    if (e->right and e->right->storereg != reg_stack)
                            	EmitPUSH(e->right->storereg, e);
            		    break;

            default:        assert(false);
                            break;
        }
}


static void RecurseEmitTernary(Expr* e, bool JumpCondition, uint *Jumps, int *j_idxp)
{       int condition, i;

        if (e->o == oANND) {
            if (JumpCondition) {
		uint LocalJumps[10];
		int j_idx=0;

                /* Jump to 'Jumps' if A AND B is true. */
                RecurseEmitTernary(e->left, no, LocalJumps, &j_idx);
                RecurseEmitTernary(e->right, yes, Jumps, j_idxp);
		for (i=0; i < j_idx; i++)
                    EmitBackPatch(LocalJumps[i], cb);
            }
            else {
                /* Jump to 'Jumps' if A AND B is false. */
                RecurseEmitTernary(e->left, no, Jumps, j_idxp);
                RecurseEmitTernary(e->right, no, Jumps, j_idxp);
            }
        }
        else if (e->o == oORR) {
            if (JumpCondition) {
                /* Jump to 'Jumps' if A OR B is true. */
                RecurseEmitTernary(e->left, yes, Jumps, j_idxp);
                RecurseEmitTernary(e->right, yes, Jumps, j_idxp);
            }
            else {
		uint LocalJumps[10];
		int j_idx=0;

                /* Jump to 'Jumps' if A OR B is false. */
                RecurseEmitTernary(e->left, yes, LocalJumps, &j_idx);
                RecurseEmitTernary(e->right, no, Jumps, j_idxp);
                for (i=0; i < j_idx; i++)
                    EmitBackPatch(LocalJumps[i], cb);
            }
	}
        else if (e->o == oNOTT) {
            RecurseEmitTernary(e->left, not JumpCondition, Jumps, j_idxp);
        }
        else if (IsComparison(e->o)) {
	    assert(e->storereg == reg_discard);
            RecurseEmitExpr(e);
            condition = e->numiregs;
            EMIT_CONDITIONAL:
            if (JumpCondition)
                Jumps[(*j_idxp)++] = EmitConditionalJump(condition, 0);
	    else Jumps[(*j_idxp)++] = EmitConditionalJump(condition^1, 0);
        }
        else if (e->o == oDER or e->o == oDIV or e->o == oMOD or e->o == oMUL) {
	    /* These operations don't affect the ZF flag. */
	    assert(e->storereg != reg_discard);
	    RecurseEmitExpr(e);
	    EmitExamine(e->storereg, e);
            condition = 5;
            goto EMIT_CONDITIONAL;
        }
        else {
            /* Hopefully, e->o is some arithmetic expression */
            /* that affects the flags. */
            RecurseEmitExpr(e);
            condition = 5;
            goto EMIT_CONDITIONAL;
        }
}


static void EmitTernary(Expr* e)
{       uint FalseJumps[100], AfterTrue;
        int i, fj_idx;

        fj_idx = 0;
        RecurseEmitTernary(e->left, no, FalseJumps, &fj_idx);   /* Jump if false */
	RecurseEmitExpr(e->right->left);
        AfterTrue = EmitJMP(0);
        for (i=0; i < fj_idx; i++)
            EmitBackPatch(FalseJumps[i], cb);
        RecurseEmitExpr(e->right->right);
        EmitBackPatch(AfterTrue, cb);
}


static void EmitExpr(Expr* e)
{
        RecurseEmitExpr(e);
}


static void RecurseEmitStatement(Statement* S)
{       SwitchStatement* S_switch;
        ReturnStatement* S_return;
        ExprStatement* S_expr;
        uint backpatch;
        IfStatement* S_if;
        int condition;

        do {
            if (S->location) {
                EmitJMP(S->location);
                return;
            }
	    S->location = cb;
	    last_add_sp = 0;
            if (S->st == st_if) {
                S_if = (IfStatement*)S;
		if (IsComparison(S_if->Test->o)) {
		    EmitExpr(S_if->Test);
                    condition = S_if->Test->numiregs;
                }
                else {
		    EmitExpr(S_if->Test);
                    EmitExamine(S_if->Test->storereg, S_if->Test);
                    condition = 5;
                }
                if (S_if->True->location) {
                    EmitConditionalJump(condition, S_if->True->location);
                    if (S_if->Fals->location)
                        EmitJMP(S_if->Fals->location);
		    else RecurseEmitStatement(S_if->Fals);
                }
                else {
                    if (S_if->Fals->location) {
                        EmitConditionalJump(condition^1/*negate*/, S_if->Fals->location);
                        RecurseEmitStatement(S_if->True);
                    }
                    else {
                        // Neither branch has been emitted yet.
                        backpatch = EmitConditionalJump(condition^1, 0);
                        RecurseEmitStatement(S_if->True);
                        if (S_if->Fals->location == 0)
                            RecurseEmitStatement(S_if->Fals);
                        EmitBackPatch(backpatch, S_if->Fals->location);
                    }
                }
                return;
            }
            else if (S->st == st_expr) {
                S_expr = (ExprStatement*)S;
                EmitExpr(S_expr->e);
                S = S->next;
                continue;
            }
            else if (S->st == st_return) {
                S_return = (ReturnStatement*)S;
		if (S_return->e == (Expr*)0x1)
		    ErrorParse(S_return->src, "You need to return a value from this function.");
                else if (S_return->e)
                    EmitExpr(S_return->e);
                if (S_return->destructors) {
                    EmitPUSH(reg_AX,NULL);
                    EmitExpr(S_return->destructors);
                    EmitPOP(reg_AX);
                    /* If we can rely on the destructors to leave EAX unmodified
                    (by not using it, or by PUSHing it), then we won't need this
                    PUSH/POP. */
                }
                if (UseFrame)
                    EmitLeave();
                EmitRET();
                return;
            }
            else if (S->st == st_switch) {
                S_switch = (SwitchStatement*)S;
                EmitExpr(S_switch->Test);
		assert(IsGeneral(S_switch->Test->storereg));
		EmitSwitch(S_switch);
                return;
            }
            else if (S->st == st_null) {
                S = S->next;
            }
            else assert(false);
        } forever;
}


static int compar2(switchcase_type *ap, switchcase_type *bp)
{	switchcase_type a=*ap, b=*bp;

	return a->tag - b->tag;
}


static str switch_src;


static void MarkLineno(str src)
/* Set up a dummy statement which will later be used to create */
/* (code,lineNo) mappings for debugging purposes. */
{       Statement* S;

        S = (Statement*)NewStatement(st_null);
        S->location = cb;
        S->src = src;
}


static void RecurseEmitJumpTable(reg_enum reg, Statement* Default, switchcase_type *A, int a, int b, int A_idx)
/* By halving-the-interval, emit a binary-tree switch for the (label,Body)'s in A.  */
/* Emit the actual code of the statements at the same time.  Do from A[a] to A[b-1].*/
{	switchcase_type scase;
	uint backpatch;
	int m;

	if (b == a + 1) {
	    scase = A[a];

	    /* Do we need to test for equality? */
	    if (a > 0 and b < A_idx and A[a-1]->tag == scase->tag-1 and scase->tag+1 == A[b]->tag) {
		/* No - just emit the code. */
		if (scase->Body->location)
		    MarkLineno(switch_src),
                    EmitJMP(scase->Body->location);
		else RecurseEmitStatement(scase->Body);
	    }
	    else {
		EmitArithI(7/*CMP*/, reg, NULL, reg_imm, ArgImmediate(scase->tag), 4);
		if (scase->Body->location) {
		    MarkLineno(switch_src);
		    EmitConditionalJump(4/*JEQ*/, scase->Body->location);
		    if (Default->location) {
                        MarkLineno(switch_src);
			EmitJMP(Default->location);
                    }
		    else RecurseEmitStatement(Default);
		}
		else if (Default->location) {
		    MarkLineno(switch_src);
		    EmitConditionalJump(5/*JNE*/, Default->location);
		    RecurseEmitStatement(scase->Body);
		}
		else {
		    MarkLineno(switch_src);
		    backpatch = EmitConditionalJump(5/*JNE*/, 0);
		    RecurseEmitStatement(scase->Body);
		    if (Default->location == 0)
			RecurseEmitStatement(Default);
		    EmitBackPatch(backpatch, Default->location);
		}
	    }
	}
	else {
	    assert(b > a + 1);
	    m = (a + b) / 2;
	    scase = A[m];
            MarkLineno(switch_src);
	    EmitArithI(7/*CMP*/, reg, NULL, reg_imm, ArgImmediate(scase->tag), 4);
            MarkLineno(switch_src);
	    backpatch = EmitConditionalJump(13/*JGE*/, 0);
	    RecurseEmitJumpTable(reg, Default, A, a, m, A_idx);
	    EmitBackPatch(backpatch, cb);
	    RecurseEmitJumpTable(reg, Default, A, m, b, A_idx);
	}
}


static void EmitSwitch(SwitchStatement* S_switch)
{	uint DefaultBackpatch, table;
	int highest, lowest, n;
	switchcase_type scase;
	reg_enum reg;

	/* What range are we covering? */
	highest = lowest = S_switch->Cases->tag;
	n = 0;
	for (scase=S_switch->Cases; scase; scase=scase->next) {
	    if (scase->tag > highest)
		highest = scase->tag;
	    if (scase->tag < lowest)
		lowest = scase->tag;
	    n++;
	}

	/* The figure should be in a register: */
	reg = S_switch->Test->storereg;
	assert(IsGeneral(reg));

	/* There are 2 types of switches:  compact ones with a jump-table, */
	/* and less compact ones with a binary tree of comparisons. */
	if (highest - lowest <= n * 3) {
	    /* It's compact: emit a jump-table. */
	    if (lowest)
		EmitArithI(5/*SUB*/, reg, NULL, reg_imm, ArgImmediate(lowest), 4); // Subtract the lowest.
	    highest -= lowest;
	    EmitArithI(7/*CMP*/, reg, NULL, reg_imm, ArgImmediate(highest), 4);	// cmp with highest
	    DefaultBackpatch = EmitConditionalJump(7/*JA*/, 0);
	    table = AllocateLocalStatic((highest+1) * 4, (Type)"i");
	    EmitTableJump(reg, table);
	    for (scase=S_switch->Cases; scase; scase=scase->next)
		scase->backpatch = table + 4*(scase->tag - lowest);

	    /*** Emit the code of the statements: ***/
	    for (scase=S_switch->Cases; scase; scase=scase->next)
		RecurseEmitStatement(scase->Body);
	    RecurseEmitStatement(S_switch->Default);

	    /*** Back-patch the jump table: ***/
	    for (n=0; n <= highest; n++) {
		((int*)(localstatic_buf + table))[n] = S_switch->Default->location;
		PointerWasEmitted(no, table + 4*n, code_base);
	    }
	    for (scase=S_switch->Cases; scase; scase=scase->next)
		*(int4*)(localstatic_buf+scase->backpatch) = scase->Body->location;
	    EmitBackPatch(DefaultBackpatch, (int)S_switch->Default->location);
	}
	else {
	    switchcase_type *A;

	    /*** It's sparse: emit a binary tree. ***/

	    /*** Get a sorted array of tags: ***/
	    A = (switchcase_type*)qcalloc(n * sizeof(switchcase_type));
	    n = 0;
	    for (scase=S_switch->Cases; scase; scase=scase->next)
		A[n++] = scase;
	    qsort(A, n, sizeof(A[0]), (cmp_fn)compar2);

	    /*** Emit a binary tree. This does the EmitStatements()'s for you. ***/
            switch_src = S_switch->src;
	    RecurseEmitJumpTable(reg, S_switch->Default, A, 0, n, n);
	}
}


static void EmitCode(Statement* S)
{
	cb = 8;
	if (UseFrame)
            EmitEnter(frame_size);
	RecurseEmitStatement(S);
}



/*------------------ Disassembling: -----------------*/


typedef struct opcode_node {
	char imm;	    // How many bytes of immediate data?
	char modregrm;	    // Is there a modregrm byte?   0=no, 1=after 1-byte opcode, 2=after 2-byte opcode
	str mnemonic;
	struct opcode_node *_2nd;
} *opcode_type;


static struct opcode_node Instruction[256];

#define each_instruction(start, finish)     ins=start; ins < finish; ins = NextInstruction(ins)



static void InstrX(str mnemonic, int opcode, uchar rg, uchar imm)
/* This instruction has 1-byte opcode followed by a mod-r/m byte with a 3-bit */
/* extra-opcode info in 'rg'. */
{	int i,mod;

	assert(rg >= 0 and rg < 8);
	if (Instruction[opcode]._2nd == NULL)
	    Instruction[opcode]._2nd = (opcode_type)calloc(anon_heap, 256, sizeof(struct opcode_node));
	for (mod=0x00; mod <= 0xc0; mod += 0x40) {
	    for (i=0; i < 8; i++) {
		Instruction[opcode]._2nd[rg*8+i+mod].imm = imm;
		Instruction[opcode]._2nd[rg*8+i+mod].modregrm = 1;
		Instruction[opcode]._2nd[rg*8+i+mod].mnemonic = mnemonic;
	    }
	}
}


static void Instr1(str mnemonic, int opcode, uchar imm, bool modregrm)
/* This instruction has a 1-byte opcode.  If 'modregrm', then a modregrm byte */
/* follows the opcode.  'imm' is how many bytes of immediate data we have.  */
{
        Instruction[opcode].imm = imm;
        Instruction[opcode].modregrm = modregrm;
        Instruction[opcode].mnemonic = mnemonic;
	assert(Instruction[opcode]._2nd == NULL);
}


static void Instr1r(str mnemonic, int opcode, uchar imm, bool modregrm)
/* This instruction has a 1-byte opcode, of which the bottom 3 bits denote the register. */
{	int i;

	for (i=0; i < 8; i++)
	    Instr1(mnemonic, opcode+i, imm, modregrm);
}


static void Instr2(str mnemonic, int opcode1, int opcode2, uchar imm, uchar modregrm)
/* This instruction has a 2-byte opcode.  If 'modregrm', then a modregrm byte        */
/* follows the 2nd opcode byte.  'imm' is how many bytes of immediate data we have.  */
{
	if (Instruction[opcode1]._2nd == NULL)
	    Instruction[opcode1]._2nd = (opcode_type)calloc(anon_heap, 256,sizeof(struct opcode_node));
	Instruction[opcode1]._2nd[opcode2].modregrm = modregrm ? 2 : 0;
	Instruction[opcode1]._2nd[opcode2].imm = imm;
	Instruction[opcode1]._2nd[opcode2].mnemonic = mnemonic;
}


static void InstrArith(str mnemonic, uchar _3bit)
{
	Instr1(mnemonic, _3bit*8,     0, yes);
	Instr1(mnemonic, _3bit*8 | 1, 0, yes);
	Instr1(mnemonic, _3bit*8 | 2, 0, yes);
	Instr1(mnemonic, _3bit*8 | 3, 0, yes);
	Instr1(mnemonic, _3bit*8 | 4, 1, no);
	Instr1(mnemonic, _3bit*8 | 5, 4, no);
	InstrX(mnemonic, 0x80, _3bit, 1);
	InstrX(mnemonic, 0x81, _3bit, 4);
	InstrX(mnemonic, 0x82, _3bit, 1);
	InstrX(mnemonic, 0x83, _3bit, 1);
}


static void InstrFArith(str mnemonic, str mnemonicp, str imnemonic, uchar _3bit)
{
        InstrX(mnemonic,  0xd8, _3bit, 0);
        InstrX(mnemonic,  0xdc, _3bit, 0);
        InstrX(mnemonic,  0xdc, _3bit, 0);
	InstrX(mnemonicp, 0xde, _3bit, 0);
        InstrX(imnemonic, 0xda, _3bit, 0);
}


static void InitDisassembler(void)
{
        if (Instruction[0].mnemonic)
	    return;     /* It's already been initialised. */
	InstrArith("ADD", 0);
	InstrArith("OR" , 1);
	InstrArith("ADC", 2);
	InstrArith("SBB", 3);
	InstrArith("AND", 4);
	InstrArith("SUB", 5);
	InstrArith("XOR", 6);
	InstrArith("CMP", 7);
	InstrFArith("FADD", "FADDP", "FIADD", 0);
	InstrFArith("FSUB", "FSUBP", "FISUB", 4);
	InstrFArith("FSUBR", "FSUBRP", "FISUBR", 5);
	InstrFArith("FCOM", "FCOMP", "FCOMI", 2);
	InstrFArith("FMUL", "FMULP", "FIMUL", 1);
	InstrFArith("FDIV", "FDIVP", "FIDIV", 6);
	InstrFArith("FDIVR", "FDIVRP", "FIDIVR", 7);
	Instr1("CALL", 0xe8, 4, no);
	Instr1("CALL", 0x9a, 4, no);
	Instr1("CMOV", 0x0f, 0, yes);
	Instr1("IMUL", 0x68, 4, yes);
	Instr1("IMUL", 0x69, 4, yes);
	Instr1("ENTER", 0xc8, 3, no);
	Instr1("JO",  0x70, 1, no);
	Instr1("JNO", 0x71, 1, no);
	Instr1("JB",  0x72, 1, no);
	Instr1("JNB", 0x73, 1, no);
	Instr1("JEQ", 0x74, 1, no);
	Instr1("JNE", 0x75, 1, no);
	Instr1("JBE", 0x76, 1, no);
	Instr1("JA",  0x77, 1, no);
        Instr1("JS",  0x78, 1, no);
        Instr1("JNS", 0x79, 1, no);
        Instr1("JPE", 0x7a, 1, no);
        Instr1("JPO", 0x7b, 1, no);
        Instr1("JLT", 0x7c, 1, no);
	Instr1("JGE", 0x7d, 1, no);
        Instr1("JLE", 0x7e, 1, no);
	Instr1("JGT", 0x7f, 1, no);
        Instr1("JMP", 0xeb, 1, no);
        Instr1("JMP", 0xea, 4, no);
        Instr1("JMP", 0xe9, 4, no);
	Instr1("HLT", 0xf4, 0, yes);
	Instr1("LEA", 0x8d, 0, yes);
	Instr1("LEAVE", 0xc9, 0, no);
	Instr1("LOOP", 0xe2, 1, no);
	Instr1("LOOPZ", 0xe1, 1, no);
	Instr1("LOOPNZ", 0xe0, 1, no);
	Instr1("MOV", 0x88, 0, yes);
	Instr1("MOV", 0x89, 0, yes);
	Instr1("MOV", 0x8a, 0, yes);
	Instr1("MOV", 0x8b, 0, yes);
	Instr1("MOV", 0xa0, 4, no);
	Instr1("MOV", 0xa1, 4, no);
	Instr1("MOV", 0xa2, 4, no);
	Instr1("MOV", 0xa3, 4, no);
	Instr1("MOVS", 0xa4, 0, no);
	Instr1("MOVS", 0xa5, 0, no);
	Instr1("NOP", 0x90, 0, no);
	Instr1("CDQ", 0x99, 0, no);
	Instr1("INT3", 0xcc, 0, no);
	Instr1("PUSH", 0x68, 4, no);
	Instr1("PUSH", 0x69, 1, no);
	Instr1("RET", 0xc3, 0, no);
	Instr1("RET", 0xc2, 2, no);
	Instr1("RET", 0xcb, 0, no);
	Instr1("RET", 0xca, 2, no);
	Instr1r("INC", 0x40, 0, no);
	Instr1r("DEC", 0x48, 0, no);
        Instr1r("MOV", 0xb0, 1, no);
        Instr1r("MOV", 0xb8, 4, no);
	Instr1r("POP", 0x58, 0, no);
	Instr1r("PUSH", 0x50, 0, no);
	InstrX("JMP", 0xff, 5, 4);
	InstrX("JMP", 0xff, 4, 0);
	InstrX("CALL", 0xff, 2, 0);
        InstrX("DIV",  0xf6, 6, 0);
        InstrX("DIV",  0xf7, 6, 0);
	InstrX("IDIV", 0xf6, 7, 0);
        InstrX("IDIV", 0xf7, 7, 0);
        InstrX("IMUL", 0xf6, 5, 0);
        InstrX("IMUL", 0xf7, 5, 0);
	InstrX("MUL", 0xf6, 4, 0);
        InstrX("MUL", 0xf7, 4, 0);
        InstrX("INC", 0xfe, 0, 0);
        InstrX("INC", 0xff, 0, 0);
        InstrX("DEC", 0xfe, 1, 0);
        InstrX("DEC", 0xff, 1, 0);
        InstrX("MOV", 0xc6, 0, 1);
        InstrX("MOV", 0xc7, 0, 4);
        InstrX("NEG", 0xf6, 3, 0);
        InstrX("NEG", 0xf7, 3, 0);
        InstrX("NOT", 0xf6, 2, 0);
        InstrX("NOT", 0xf7, 2, 0);
        InstrX("POP", 0x8f, 0, 0);
        InstrX("PUSH", 0xff, 6, 0);
	InstrX("SHL", 0xd0, 4, 0);
        InstrX("SHL", 0xd1, 4, 0);
        InstrX("SHL", 0xd2, 4, 0);
        InstrX("SHL", 0xd3, 4, 0);
        InstrX("SHL", 0xc0, 4, 1);
        InstrX("SHL", 0xc1, 4, 1);
        InstrX("SHR", 0xd0, 5, 0);
        InstrX("SHR", 0xd1, 5, 0);
        InstrX("SHR", 0xd2, 5, 0);
        InstrX("SHR", 0xd3, 5, 0);
	InstrX("SHR", 0xc0, 5, 1);
        InstrX("SHR", 0xc1, 5, 1);
        InstrX("SAR", 0xd0, 7, 0);
        InstrX("SAR", 0xd1, 7, 0);
	InstrX("SAR", 0xd2, 7, 0);
        InstrX("SAR", 0xd3, 7, 0);
        InstrX("SAR", 0xc0, 7, 1);
        InstrX("SAR", 0xc1, 7, 1);
	InstrX("FCOMIP", 0xdf, 6, 0);
	Instr2("FDECSTP", 0xd9, 0xf6, 0, no);
	InstrX("FILD", 0xdf, 0, 0);
	InstrX("FILD", 0xdb, 0, 0);
	InstrX("FILD", 0xdf, 5, 0);
	InstrX("FIST", 0xdf, 2, 0);
	InstrX("FIST", 0xdb, 2, 0);
	InstrX("FISTP",0xdf, 3, 0);
	InstrX("FISTP",0xdb, 3, 0);
	InstrX("FISTP",0xdf, 7, 0);
	InstrX("FST",  0xd9, 2, 0);
	InstrX("FST",  0xdd, 2, 0);
	InstrX("FSTP", 0xd9, 3, 0);
	InstrX("FSTP", 0xdd, 3, 0);
	InstrX("FSTP", 0xdb, 7, 0);
	InstrX("FLD",  0xd9, 0, 0);
	InstrX("FLD",  0xdd, 0, 0);
	InstrX("FLD",  0xdb, 5, 0);
        Instr2("IMUL", 0x0f, 0xaf, 0, yes);
        Instr2("JO",  0x0f, 0x80, 4, no);
        Instr2("JNO", 0x0f, 0x81, 4, no);
        Instr2("JB",  0x0f, 0x82, 4, no);
        Instr2("JNB", 0x0f, 0x83, 4, no);
        Instr2("JEQ", 0x0f, 0x84, 4, no);
        Instr2("JNE", 0x0f, 0x85, 4, no);
        Instr2("JBE", 0x0f, 0x86, 4, no);
	Instr2("JA",  0x0f, 0x87, 4, no);
        Instr2("JS",  0x0f, 0x88, 4, no);
        Instr2("JNS", 0x0f, 0x89, 4, no);
        Instr2("JPE", 0x0f, 0x8A, 4, no);
	Instr2("JPO", 0x0f, 0x8B, 4, no);
        Instr2("JLT", 0x0f, 0x8c, 4, no);
        Instr2("JGE", 0x0f, 0x8d, 4, no);
        Instr2("JLE", 0x0f, 0x8e, 4, no);
        Instr2("JGT", 0x0f, 0x8f, 4, no);
        Instr2("MOVSX", 0x0f, 0xbe, 0, yes);
        Instr2("MOVSX", 0x0f, 0xbf, 0, yes);
        Instr2("MOVZX", 0x0f, 0xb6, 0, yes);
        Instr2("MOVZX", 0x0f, 0xb7, 0, yes);
        Instr2("REP MOVS", 0xf3, 0xa4, 0, no);
	Instr2("REP MOVS", 0xf3, 0xa5, 0, no);
        Instr2("REP STOS", 0xf3, 0xaa, 0, no);
        Instr2("REP STOS", 0xf3, 0xab, 0, no);
        Instr2("REPE CMPS", 0xf3, 0xa6, 0, no);
        Instr2("REPE CMPS", 0xf3, 0xa7, 0, no);
        Instr2("REPNE CMPS", 0xf2, 0xa6, 0, no);
        Instr2("REPNE CMPS", 0xf2, 0xa7, 0, no);
        Instr2("FLD1",  0xd9, 0xe8, 0, no);
        Instr2("FLDZ",  0xd9, 0xee, 0, no);
        Instr2("FLDPI", 0xd9, 0xeb, 0, no);
        Instr2("FLDL2E",0xd9, 0xea, 0, no);
        Instr2("FLDL2T",0xd9, 0xe9, 0, no);
        Instr2("F2XM1", 0xd9, 0xf0, 0, no);
        Instr2("FABS",  0xd9, 0xe1, 0, no);
	Instr2("FSIN",  0xd9, 0xfe, 0, no);
        Instr2("FCOS",  0xd9, 0xff, 0, no);
        Instr2("FPTAN", 0xd9, 0xf2, 0, no);
        Instr2("FPATAN",0xd9, 0xf3, 0, no);
        Instr2("FSQRT", 0xd9, 0xfa, 0, no);
}


interface ins_type NextInstruction(ins_type instruction)
/* Given an instruction, work out where the next instruction begins. */
{       int mod, reg, rm, sib, modregrm;
	ins_type p=instruction;
	int operand_size=4;
        opcode_type op;

	/* Skip prefixes: */
	if (*p == 0x66)
	    p++, operand_size = 2;

	/* Is it a 1-byte or 2-byte opcode? */
	op = &Instruction[*p++];
	if (op->_2nd) {
	    op = &op->_2nd[*p++];
	    if (op->modregrm == 2)
		modregrm = *p++;
	    else modregrm = p[-1];
	    reg = (modregrm & 0x38) >> 3;
	    mod = modregrm >> 6;
	    rm = modregrm & 7;
	}
	else {
	    if (op->modregrm) {
		modregrm = *p++;
		reg = (modregrm & 0x38) >> 3;
		mod = modregrm >> 6;
		rm = modregrm & 7;
	    }
	}

	/* Instructions with no modregrm byte: */
	if (not op->modregrm)
	    return p + op->imm;

	/* The modregrm byte: */
        if (mod == 3)
            ;		// the operand is a register, not memory at all.
        else if (mod == 0 and rm == 5)
            p += 4;	// the basic 'disp32' mode
        else if (rm != 4)
            p += (mod==0 ? 0 : mod==1 ? 1 : 4);	// No SIB byte
	else {
	    /* An SIB byte */
	    sib = *p++;
            p += (mod==0 and (sib&0x7) == 5 ? 4 : 0);
	}

	/* The immediate data: */
	if (op->imm == 4)
	    p += operand_size;
	else p += op->imm;

	return p;
}


interface str Reg8ToString[8] = { "AX", "CX", "DX", "BX", "SP", "BP", "SI", "DI" };
static str Scaler4ToString[4] = { "", "*2", "*4", "*8" };

interface str Disassemble(char buf[], ins_type ins)
/* Display this instruction as a str. */
{	int i, modregrm, reg, mod, rm, sib, index, base, scaler;
	char _prefix[20], _mem[20], _reg[10], _imm[20];
	uchar opcode1, opcode2;
        opcode_type op, _2nd;
	bool direction=0;
	ins_type p=ins;
	str s,d;

	/* Instruction prefixes: */
	_prefix[0] = '\0';
	while ((*p|1) == 0x67) {
	    if (*p == 0x66)
		strcat(_prefix, " (16bit operand)");
	    p++;
	}

	/* Is it a 1-byte or 2-byte opcode? */
	opcode1 = *p++;	
	op = &Instruction[opcode1];
	_2nd = op->_2nd;
	if (op->_2nd) {
	    opcode2 = *p++;
	    op = &op->_2nd[opcode2];
	    if (op->modregrm == 2) {
		modregrm = *p++;
		direction = 1;
		_2nd = NULL;
	    }
	    else modregrm = p[-1];
	    reg = (modregrm & 0x38) >> 3;
	    mod = modregrm >> 6;
	    rm = modregrm & 7;
	}
	else {
	    opcode2 = 0;
	    if (op->modregrm) {
		direction = ((p[-1] & 2) >> 1) or (opcode1 == 0x8d);
		modregrm = *p++;
		reg = (modregrm & 0x38) >> 3;
		mod = modregrm >> 6;
		rm = modregrm & 7;
	    }
	}
	if (op->mnemonic == NULL)
	    return strcpy(buf, "???");

	/* Is there a modregrm byte? */
	if (op->modregrm) {
	    s = _mem;
	    if (mod == 0 and rm == 5)
		s += sprintf(s, "[0x%x]", *(*(int**)&p)++);
	    else if (rm == 4 and mod != 3) {
		// An SIB byte
		sib = *p++;
		scaler = sib >> 6;
		index = (sib >> 3) & 7;
		base = sib & 7;
		*s++ = '[';
		if (index != 4)
		    s += sprintf(s, "%s%s+", Reg8ToString[index], Scaler4ToString[scaler]);
		if (base == 5) {
		    if (mod == 1)
			s += sprintf(s, "0x%x]", *(*(char**)&p)++);/* $$$ I don't think this is correct. */
		    else s += sprintf(s, "0x%x]", *(*(int**)&p)++);
		}
		else s += sprintf(s, "%s]", Reg8ToString[base]);
	    }
	    else {
		if (mod == 0)
		    s += sprintf(s, "[%s]", Reg8ToString[rm]);
		else if (mod == 1)
		    s += sprintf(s, "[%s+%d]", Reg8ToString[rm], *(*(char**)&p)++);
		else if (mod == 2)
		    s += sprintf(s, "[%s+%d]", Reg8ToString[rm], *(*(int**)&p)++);
		else 
		    s += sprintf(s, "%s", Reg8ToString[rm]);
	    }

	    /* Is the 'reg' 3bit a real reg denoter, or extra opcode info? */
	    if (_2nd) {
		for (i=0; i < 7; i++) {
		    if (_2nd[(mod<<6)|(i<<3)|rm].mnemonic != op->mnemonic)
			reg = -1;
		}
	    }
	}
	else if (not _2nd and op->mnemonic == Instruction[opcode1^7].mnemonic) {
	    reg = opcode1 & 7;
	    _mem[0] = '\0';
        }
        else {
	    reg = -1;
	    _mem[0] = '\0';
	}

	/* Is there a register argument? */
	if (reg >= 0)
	    strcpy(_reg, Reg8ToString[reg]);
	else _reg[0] = '\0';

	/* Are there any immediate arguments? */
	if (op->imm) {
	    s = _imm;
	    if ((opcode1 & 0xf0) == 0x70 or opcode1 == 0xeb)
		s += sprintf(s, "0x%x", (char*)ins + 2 + *(*(char**)&p)++);    // 8-bit JMP or Jcc
	    else if ((opcode1 == 0x0f and (opcode2 & 0xf0) == 0x80)
		    or (opcode1 == 0xe9)
		    or (opcode1 == 0xe8))
		s += sprintf(s, "0x%x", (char*)ins + 6 + *(*(int**)&p)++);	    // 32-bit JMP or Jcc
	    else if (op->imm == 1)
		s += sprintf(s, "%d", *(*(char**)&p)++);
	    else if (op->imm == 2)
		s += sprintf(s, "%d", *(*(short**)&p)++);
	    else if (op->imm == 4)
		s += sprintf(s, "%d", *(*(int**)&p)++);
	}
	else _imm[0] = '\0';

	/* Assemble the mnemonics: */
	if (direction == 0)
	    sprintf(buf, "%s %s, %s, %s%s", op->mnemonic, _mem, _reg, _imm, _prefix);
	else sprintf(buf, "%s %s, %s, %s%s", op->mnemonic, _reg, _mem, _imm, _prefix);
	for (s=d=buf; *s; ) {
	    if (*s == ',' and s[2] == ',')
		s += 2;
	    else if (*s == ' ' and s[1] == ',')
		s += 2;
	    else if (*s == '+' and s[1] == '-')
		s++;
	    else if (*s == ' ' and s[1] == ' ')
		s++;
	    else *d++ = *s++;
	}
	while (d > buf and (d[-1] == ' ' or d[-1] == ','))
	    d--;
	*d = '\0';
	return buf;
}


interface str InstrToString(ins_type ins)
/* Disassemble this instruction. */
{	static char buf[40];

	Disassemble(buf, ins);
	return buf;
}


interface void DisassembleFuncToCout(machinecode_type func)
/* Disassemble this function to cout. */
{	static char buf[40];
        ins_type ins;
        int cnt=0;

        ins = (ins_type)func;
        if (ins == NULL)
            printf("<null>\n");
        else do {
            Disassemble(buf, ins);
            printf("%s\n", buf);
            if (*ins == 0xc3)
                break;
            if (++cnt > 10)
                break;
            ins = NextInstruction(ins);
        } forever;
}


interface void DisassembleToStruct(ins_type ins, disassembledins_type I)
/* Break this instruction up into its components. */
{	uchar opcode1, opcode2;
        opcode_type op, _2nd;
	int i, modregrm;
	ins_type p=ins;

	/* Clear the struct: */
	clearS(*I);
	I->reg = -1;

	/* Instruction prefixes: */
	while ((*p|1) == 0x67) {
	    if (*p == 0x66)
		I->_16bit = yes;
	    p++;
	}

	/* Is it a 1-byte or 2-byte opcode? */
	opcode1 = *p++;	
	op = &Instruction[opcode1];
	_2nd = op->_2nd;
	if (op->_2nd) {
	    opcode2 = *p++;
	    op = &op->_2nd[opcode2];
	    if (op->modregrm == 2) {
		modregrm = *p++;
		I->direction = 1;
		_2nd = NULL;
	    }
	    else modregrm = p[-1];
	    I->reg = (modregrm & 0x38) >> 3;
	    I->mod = modregrm >> 6;
	    I->rm = modregrm & 7;
	}
	else {
	    opcode2 = 0;
	    if (op->modregrm) {
		I->direction = ((p[-1] & 2) >> 1) != 0;
		modregrm = *p++;
		I->reg = (modregrm & 0x38) >> 3;
		I->mod = modregrm >> 6;
		I->rm = modregrm & 7;
	    }
	}
	if (op->mnemonic == NULL) {
	    I->mnemonic = "???";
	    return;
	}
	else I->mnemonic = op->mnemonic;

	/* Is there a modregrm byte? */
	if (op->modregrm) {
	    I->modregrm_present = yes;
	    if (I->mod == 0 and I->rm == 5)
		I->disp = *(*(int**)&p)++;
	    else if (I->rm == 4 and I->mod != 3) {
		// An SIB byte
		I->sib_present = yes;
		I->sib = *p++;
		I->scaler = 1 << (I->sib >> 6);
		I->index = (I->sib >> 3) & 7;
		I->base = I->sib & 7;
		/*if (I->index == 4)
		    I->index = -1, I->scaler = 0;*/
		if (I->base == 5) {
		    if (I->mod == 1)
			I->disp = *(*(char**)&p)++;
		    else I->disp = *(*(int**)&p)++;
		}
	    }
	    else {
		if (I->mod == 1)
		    I->disp = *(*(char**)&p)++;
		else if (I->mod == 2)
		    I->disp = *(*(int**)&p)++;
		I->base = I->rm;
	    }

	    /* Is the 'reg' 3bit a real reg denoter, or extra opcode info? */
	    if (_2nd) {
		for (i=0; i < 7; i++) {
		    if (_2nd[(I->mod<<6)|(i<<3)|I->rm].mnemonic != op->mnemonic)
			I->reg = -1;
		}
	    }
	}
	else {
	    I->reg = -1;
	}

	/* Are there any immediate arguments? */
	if (op->imm) {
	    I->imm_present = yes;
	    if ((opcode1 & 0xf0) == 0x70 or opcode1 == 0xeb)
		I->imm = (int)ins + 2 + *(*(char**)&p)++;	    // 8-bit JMP or Jcc
	    else if ((opcode1 == 0x0f and (opcode2 & 0xf0) == 0x80)
		    or (opcode1 == 0xe9)
		    or (opcode1 == 0xe8))
		I->imm = (int)ins + 6 + *(*(int**)&p)++;	    // 32-bit JMP or Jcc
	    else if (op->imm == 1)
		I->imm = *(*(char**)&p)++;
	    else if (op->imm == 2)
		I->imm = *(*(short**)&p)++;
	    else if (op->imm == 4)
		I->imm = *(*(int**)&p)++;
	}
	else I->imm_present = no;
}




/*--------------- Allocating space for local static data: --------------*/

static void NotifyLsptr(Type type, int offset)
{       Type type0=type;

        do {
            switch (*type++) {
                case tp_reference:
                case tp_pointer:    continue;
                case tp_array:      type += sizeof(int);
                                    continue;
                case tp_class:      PointerWasEmitted(no, offset+type-type0,
                                                *(Classdef**)type);
                                    return;
                default:            return;
            }
        } forever;
}


interface void InitLocalStatic(void)
{
	if (localstatic_buf) {
            free(anon_heap, localstatic_buf);
            localstatic_buf = NULL;
        }
	localstatic_offset = 0;
        PointerbufInit();
}


interface int AllocateLocalStatic(uint size, Type type)
/* Allocate some data storage space inside the current compileable */
/* entity.  The data lasts as long as the compileable entity does, */
/* so if you're inside a function then it should last a while.     */
{       int r;

        /* Determine the offset and create the memory: */
        WordAlign(size);
        r = localstatic_offset;
        localstatic_offset += size;
        localstatic_buf = (str)realloc(anon_heap, localstatic_buf, localstatic_offset);
	memset(localstatic_buf + r, 0, size);

        return r;
}


interface int AllocateAndCopyLocalStatic(uint size, void* s, Type type)
/* Allocate space with the above function and then copy this data into it. */
{       int offset;

        offset = AllocateLocalStatic(size, type);
	memcpy(localstatic_buf + offset, s, size);

        /* If it's a type-type, then notify the local-static-pointer list: */
        if (type == typetype_typstr)
            NotifyLsptr((Type)s, offset);

        return offset;
}


interface int CopyLocalStatic(int offset, uint size, void* s)
/* Copy this data into a local-static variable that has already been created. */
{
        memcpy(localstatic_buf + offset, s, size);
        return offset;
}


interface void* RetrieveLocalStatic(int offset)
/* Get the ptr corresponding to this offset in the localstatic_buf. */
{
	return localstatic_buf + offset;
}






/*------------------ Generating Debug information: -----------------*/

static DebugInfo::linemap_node *CodeToLine;
static uint CL_idx;
static DebugInfo* dinfo;


static int StringToLine(str src, str *LineToString, int idx)
/* Map this str to a line-number by looking up LineToString, */
/* a sorted array of line-start-pointers. */
{
	assert(idx > 0);
	if (idx == 1)
	    return 0;
	if (src < LineToString[idx/2])
	    return StringToLine(src, LineToString, idx/2);
	else return idx/2 + StringToLine(src, LineToString + idx/2, idx - idx/2);
}


static int compar1(DebugInfo::linemap_node *a, DebugInfo::linemap_node *b)
{
	return a->Code - b->Code;
}


static void CreateDebugLines(void)
{	str s, Source, *LineToString;
	int n, L_idx, f, d;
	Statement* S;

	/* Step 1: Get an array mapping line-numbers to points in CurrentSource. */
	Source = CurrentSource;
	n = 1;
	s = Source-1;
	do {
	    s++;
	    n++;
	} while ((s=strchr(s, '\n')) != NULL);
	LineToString = (str*)qcalloc(n * sizeof(int));
	n = 0;
	s = Source-1;
	do {
	    LineToString[n++] = ++s;
	} while ((s=strchr(s, '\n')) != NULL);
	L_idx = n;

	/* Step 2: Get an unsorted list of (LineNo, Code) mappings: */
	n = 1;
	for (each_statement)
	    if (S->location and S->src)
		n++;
	CodeToLine = (DebugInfo::linemap_node*)qcalloc(n * sizeof(DebugInfo::linemap_node));
	n = 0;
	for (each_statement) {
	    if (S->location and S->src) {
		CodeToLine[n].LineNo = StringToLine(S->src, LineToString, L_idx);
		CodeToLine[n].Code = (uchar*)S->location;
		n++;
	    }
	}

	/* Step 3: Sort this list: */
	qsort(CodeToLine, n, sizeof(CodeToLine[0]), (cmp_fn)compar1);

	/* Step 4: Eliminate duplicates: */
	f = d = 1;
	while (f < n) {
	    if (CodeToLine[f].LineNo != CodeToLine[d-1].LineNo)
		CodeToLine[d++] = CodeToLine[f];
	    f++;
	}
	CodeToLine[d].LineNo = -1;
	CodeToLine[d].Code = (uchar*)cb;
	CodeToLine[0].Code = (uchar*)(UseFrame ? 12 : 8);
	d++;
	CL_idx = d;
}


static bool StorageIsLocal(storage_enum storage)
{
	return storage == auto_storage or storage == parameter_storage or
		storage == local_static;
}


static void CreateDebugInfo(void* localstatic_start)
/* Create a block containing all debug info for the current compileable. */
{	DebugInfo::localvarmap_node *dvar;
        LocalStaticNamedobj *objLS;
	AutoNamedobj *obj;
	int size, ts;
	str s;

	/* Get the size of the object: */
	size = sizeof(DebugInfo);
	size += CL_idx * sizeof(DebugInfo::linemap_node);
	for (obj=localvar_root; obj; obj=obj->nextlocal) {
	    size += sizeof(DebugInfo::localvarmap_node) +
		    LengthOfTypeString((Type)obj->type)
		    + strlen(obj->name) + 1;
	    size = ((size-1) | 3) + 1;
	}
	for (objLS=localstaticvar_root; objLS; objLS=objLS->nextlocal) {
	    size += sizeof(DebugInfo::localvarmap_node) +
		    LengthOfTypeString((Type)objLS->type)
		    + strlen(objLS->name) + 1;
	    size = ((size-1) | 3) + 1;
	}
	size += 4;

	/* Create it: */
	if (dinfo) {
	    Heap* heap = Ptr_to_conim(dinfo);
	    if (heap)
		heap->free(dinfo);
	}
	dinfo = (DebugInfo*)default_heap->malloc(size);
	dinfo->size = size;
	dinfo->linemap = (DebugInfo::linemap_node*)(dinfo+1);
	dinfo->localvarmap = (DebugInfo::localvarmap_node*)(dinfo->linemap + CL_idx);
	memcpy(dinfo->linemap, CodeToLine, CL_idx * sizeof(CodeToLine[0]));

	/* Put in the local variables too: */
	dvar = dinfo->localvarmap;
	for (obj=localvar_root; obj; obj=obj->nextlocal) {
	    dvar->storage = obj->storage;
	    dvar->offset = obj->offset;
	    memcpy(dvar->type, obj->type, (ts=LengthOfTypeString((Type)obj->type)));
	    s = dvar->type + ts;
	    dvar->name = strcpy(s, obj->name);
	    s += strlen(s) + 1;
	    until (((int)s & 3) == 0)
		*s++ = 0;
	    dvar = (DebugInfo::localvarmap_node*)s;
	}
	for (objLS=localstaticvar_root; objLS; objLS=objLS->nextlocal) {
	    dvar->storage = objLS->storage;
	    dvar->offset = int(objLS->location) + (int)localstatic_start;
	    memcpy(dvar->type, objLS->type, (ts=LengthOfTypeString((Type)objLS->type)));
	    s = dvar->type + ts;
	    dvar->name = strcpy(s, objLS->name);
	    s += strlen(s) + 1;
	    until (((int)s & 3) == 0)
		*s++ = 0;
	    dvar = (DebugInfo::localvarmap_node*)s;
	}
	dvar->name = NULL;
	assert((char*)dvar->name + 4 < (char*)dinfo + size);
}


interface DebugInfo* IprGetDebugInfo(DebugInfo *old_dinfo)
/* Return a malloc'd block with all debug info for the current compileable. */
/* The block will have been malloc'd in the default container. */
{	DebugInfo* r;

        if (old_dinfo)
            default_heap->free(old_dinfo);
	r = dinfo;
	dinfo = NULL;
	return r;
}


interface void IprClearDebugInfo(void)
/* Free the malloc'd block which we create with any compilation. */
{
	default_heap->free(dinfo);
	dinfo = NULL;
}





/*------------------ Post-processing the code: -----------------*/

typedef struct jmp_node {
	uchar* instruction;
	uchar* destination;
	uint size;
} *jmp_type;


static jmp_type J;
static uint J_idx, J_midx;


typedef struct debugheader_node {
	uchar Signature[8];
	Namedobj* obj;
} *debugheader_type;


typedef struct funcblockheader_node {
	int debugheader_offset;
	int ptrlist_offset;
} *funcblockheader_type;


static void RegisterJump(ins_type instruction, uint size)
{
	if (J_idx >= J_midx) {
	    if (J_midx == 0)
		J_midx = 32;
	    J_midx *= 2;
	    J = (jmp_type)realloc(anon_heap, J, J_midx * sizeof(struct jmp_node));
	}
	J[J_idx].instruction = instruction;
	if (size == 1)
	    J[J_idx].destination = instruction + 1 + *(char*)instruction;
	else J[J_idx].destination = instruction + 4 + *(int4*)instruction;
	J[J_idx].size = size;
	J_idx++;
}


static void RemoveBytes(ins_type ins, int n)
/* Update our jumps table with information about removing 'n' bytes from location 'ins'. */
{	pb_type pb;
	uint i;

	for (i=0; i < J_idx; i++) {
	    if (J[i].instruction > ins)
		J[i].instruction -= n;
	    if (J[i].destination > ins)
		J[i].destination -= n;
	}
	for (i=0; i < pointerbuf_idx; i++) {
	    pb = &pointerbuf[i];
	    if (pb->code_vs_ls) {
		if (codebuf + pb->offset > ins)
		    pb->offset -= n;
	    }
	    else if (pb->base == code_base) {
		if (codebuf + *(uint*)(localstatic_buf + pb->offset) > ins)
		    *(ins_type*)(localstatic_buf + pb->offset) -= n;
	    }
	}

        /* Now update debugging info too: */
	for (i=0; i < CL_idx; i++) {
	    if (codebuf + (int)CodeToLine[i].Code > ins)
		CodeToLine[i].Code -= n;
	}
}


static void RemoveNOPs(void)
/* Take out the NOP's left by shortened jumps. */
{	ins_type d, s, s0, ins, start, finish;
	int offset;
	uint i;


	/*** Get a list of all jumps and their destinations: ***/
        InitDisassembler();
	J = NULL;
	J_midx = J_idx = 0;
	start = codebuf + 8;
	finish = codebuf + cb;
	for (each_instruction(start, finish)) {
	    if (*ins == 0xeb or (*ins & 0xf0) == 0x70) {
		/* An 8-bit jump */
		RegisterJump(ins+1, 1);
	    }
	    else if (*ins == 0xe9) {
		/* A 32-bit jump */
		RegisterJump(ins+1, 4);
	    }
	    else if (*ins == 0x0f and (ins[1] & 0xf0) == 0x80) {
		/* A 32-bit jump */
		RegisterJump(ins+2, 4);
	    }
	}


	/*** Physically delete the NOPs: ***/
	d = s = s0 = codebuf + 8;
	while (s < finish) {
	    if (*s == 0x90) {
		if (s > s0) {
		    /* Flush whatever data we have so far: */
		    if (d != s0)
			memcpy(d, s0, s - s0);
		    d += s - s0;
		    s0 = s;
		}
		while (*s == 0x90 and s < finish)
		    s++;
		RemoveBytes(d, s - s0);
		s0 = s;
	    }
	    s = NextInstruction(s);
	}
	if (d != s0)
	    memcpy(d, s0, s - s0);
	d += s - s0;


	/*** Write the jumps back in: ***/
	for (i=0; i < J_idx; i++) {
	    if (J[i].size == 1) {
		offset = J[i].destination - J[i].instruction - 1;
		assert(offset == (char)offset);
		*J[i].instruction = offset;
	    }
	    else if (J[i].size == 4) {
		offset = J[i].destination - J[i].instruction - 4;
		*(int4*)J[i].instruction = offset;
	    }
	}


	/*** Finished! ***/
	cb = d - codebuf;
	free(anon_heap, J);
	J = NULL;
}


static funcblock_type CodebufToFuncbloc(void* preferred_location)
/* We have compiled the function into 'codebuf'. We must*/
/* copy it into an independent malloc'd block with the	*/
/* funcblock information. */
{       uint code_size, ptrlist_size, funcblock_size;
        ins_type ins, start, finish, ls_start;
	debugheader_type debugheader;
	funcblockheader_type header;
	funcblock_type funcblock;
	pb_type pb, ptrlist;
        uint *sr, i;


        /*** Set up (lineno,code) pairs: ***/
	CreateDebugLines();


	/*** Remove NOP's: ***/
	RemoveNOPs();


	/*** Set up the funcbloc: ***/
	code_size = cb - 8;
	while (code_size&3)
	    code_size++;
	while (localstatic_offset&3)
	    localstatic_offset++;
	ptrlist_size = pointerbuf_idx * sizeof(pointerbuf[0]);
	funcblock_size = code_size
		    + localstatic_offset
		    + sizeof(struct debugheader_node)
		    + ptrlist_size
		    + sizeof(struct funcblockheader_node);
	/* The format of a funcblock is as follows:

	    code | [debugheader] | localstatic_data | gap | pointerlist | [funcblockheader]

	Note that all items are variable-length unless in square brackets.
	'gap' means any leftover space from what malloc allocates us. This
	normally goes at the end of the block, but in this case we put the
	funcblockheader at the end of the block because this is the only
	way we can find it, given that we want the start of the funcblock
	to equal the entrypoint for the function.

	The funcblockheader uses negative integers to point to the debugheader
	and pointerlist. These are offsets expressed relative to the
	funcblockheader.
	*/

	if (preferred_location and funcblock_size <= (uint)msize(preferred_location)) {
	    uchar buf[5];
	    assert(TypeEqual(Heap::Typestr(preferred_location, buf), voidvoidfn_typstr));
	    funcblock = (funcblock_type)preferred_location;
	}
	else {
	    funcblock = (funcblock_type)default_heap->NewPlus(voidvoidfn_typstr, funcblock_size);
	    int c = msize(funcblock);
	    assert(c >= funcblock_size);
	}
	header = (funcblockheader_type)
	    ((char*)funcblock + msize(funcblock) - sizeof(struct funcblockheader_node));
	debugheader = (debugheader_type)((char*)funcblock + code_size);
	ls_start = (ins_type)((char*)debugheader + sizeof(struct debugheader_node));
	ptrlist = (pb_type)((char*)header - ptrlist_size);
	header->debugheader_offset = (char*)debugheader - (char*)header;
	header->ptrlist_offset = (char*)ptrlist - (char*)header;


	/*** Copy the code: ***/
	start = (uchar*)funcblock;
	finish = start + code_size;
	memcpy(start, codebuf + 8, code_size);


	/*** Adjust all the CALL instructions: ***/
	for (each_instruction(start, finish)) {
	    if (*ins == 0xe8) {		// a CALL instruction
		sr = (uint*)(ins + 1);
		if (*sr)
		    *sr -= (uint)(sr + 1);
		/* otherwise it's a call to a function that hasn't been defined yet. */
	    }
	}


	/*** Create and adjust the debug info: ***/
	for (i=0; i < CL_idx; i++)
	    CodeToLine[i].Code += (uint)funcblock - 8;


	/*** Write the debug header: ***/
	memset(debugheader->Signature, 0x90, sizeof(debugheader->Signature));
	debugheader->Signature[7] = 0xcc;
	debugheader->obj = CompilerCurrentFuncObj;
	/* The signature is full of NOPs, which won't occur in any code we generate. */
	/* i.e. to find the debugheader, just search forward for this signature. */


	/*** Copy all the local-static data in: ***/
	memcpy(ls_start, localstatic_buf, localstatic_offset);


	/*** Adjust all the embedded pointers: ***/
	for (i=0; i < pointerbuf_idx; i++) {
	    pb = &pointerbuf[i];
            /* There are 6 cases, depending on whether the ptr is in */
            /* (1) the code or (2) local.statics; and depending on   */
            /* whether the target is (a) in the code, (b) in the     */
            /* local statics or (c) somewhere else.  But only 4 cases*/
            /* need the pointers to be adjusted. */
	    if (pb->base == localstatic_base) {
		/* It's a local-static. */
		if (pb->code_vs_ls)
		    *(uint*)&start[pb->offset-8] += (int4)ls_start;
		else *(uint*)&ls_start[pb->offset] += (int4)ls_start;
	    }
	    else if (pb->base == code_base) {
		/* It's a pointer into the code. */
		if (pb->code_vs_ls)
		    *(uint*)&start[pb->offset-8] += (int4)funcblock;
		else *(uint*)&ls_start[pb->offset] += (int4)funcblock - 8;
                        // This last line is used by switch jump tables,
                        // (anything else?)
	    }
	}


	/*** Create the pointerlist (for swizzling): ***/
	for (i=0; i < pointerbuf_idx; i++) {
	    pb = &pointerbuf[i];
	    ptrlist[i].code_vs_ls = no;		// By this stage, it makes no difference.
	    ptrlist[i].offset = pb->code_vs_ls ? pb->offset : pb->offset + ((str)ls_start - (str)funcblock) + 8;
	    ptrlist[i].base = pb->base;
	}
	PointerbufInit();
	assert((char*)(ptrlist + i) <= (char*)header);


	/*** Create debug info before all the information disappears: */
	if (not PredefinitionPhase)
	    CreateDebugInfo(ls_start);


	return funcblock;
}


interface void FindPtrsInFuncblock(funcblock_type funcblock, PtrProcessor_fn fn)
/* Interpret 'code' as Intel86 machine-code, and pass each pointer */
/* found through the function 'fn'. */
{	debugheader_type debugheader;
	funcblockheader_type header;
	pb_type pb, pointerlist;
	ins_type ins;
	void** ptr;

	/* Break down the funcblock: */
	header = (funcblockheader_type)
	    ((char*)funcblock + MsizeSwizzled(funcblock)
		    - sizeof(struct funcblockheader_node));
	pointerlist = (pb_type)((char*)header + header->ptrlist_offset);

        /* Find all ptrs in the debug header: */
        if (header->debugheader_offset) {
            debugheader = (debugheader_type)((char*)header + header->debugheader_offset);
            fn((void**)&debugheader->obj);
        }
        // Otherwise it's probably a ptrfinder function.

	/* Find all ptrs from pointerlist: */
	for (pb=pointerlist; (char*)pb < (char*)header; pb++) {
	    ptr = (void**)((char*)funcblock + pb->offset - 8);
	    fn(ptr);
	}

	/* Find all relative CALLs: */
	for (each_instruction(funcblock, (ins_type)debugheader)) {
	    if (*ins == 0xe8) {
		ProcessRelativePtr((int*)(ins+1), fn);
		/* If we're in the LOAD phase, AND it's a local pointer (local to this container), */
		/* then we don't need to do anything.  But often it's a foreign pointer. */
            }
            else if (*ins == 0x90) {
                if (*(unsigned int*)(ins+1) == 0x90909090)
                    break;
            }
	}
}


static Namedobj* *FuncblocksObjp(ins_type ip)
/* Find the address of the back-pointer from the funcblock to the obj. */
{	uchar Signature[8];
	ins_type ins;

	memset(Signature, 0x90, sizeof(Signature));
	Signature[7] = 0xcc;
	ins = ip - 1;
	do {
            // Allow the first 4 bytes of the signature to be overwritten,
            // in case we have a 5-byte breakpoint on the terminating 1-byte
            // RET instructions.
	    ins = (ins_type)memchr(ins+1, Signature[4], 9999);
	    if (ins == NULL or ins > ip + 9999)
		return NULL;
	} until (memcmp(ins, Signature+4, sizeof(Signature)-4) == 0);
        ins += 4;
	return (Namedobj**)ins;
}


interface void FuncblockAdjustObj(funcblock_type funcblock, Namedobj* obj)
/* This obj has been moved, so update the back-pointer to it from the debug */
/* header inside the funcblock. */
{       extern void PureVirtual();
        Namedobj* *objp;

        if (funcblock == (funcblock_type)PureVirtual)
            return;
        objp = FuncblocksObjp(funcblock);
        if (objp == NULL) {
            assert(false);
            return;
        }
        *objp = obj;
}


interface Namedobj* InsToObj(ins_type ip)
{       Namedobj* *objp;

        objp = FuncblocksObjp(ip);
        return objp ? *objp : NULL;
}





/*------------------ The supervisor: -----------------*/

interface funcblock_type GenerateCode(Statement* S, void* preferred_location)
/* We have an analysed and optimised IPR in 'S'. */
/* Emit Pentium code from it. */
{
        AllocateLocalVars();
        AllocateRegistersToExprs();
	if (not PredefinitionPhase) {
	    //DumpIpr(S);
	    //return NULL;
	}
	EmitCode(S);
        return CodebufToFuncbloc(preferred_location);
}


