/*****************************************************\
** COMPDECL.C:                                       **
**          The declaration parser module.  Takes a  **
** stream of tokens for a declaration and performs   **
** the necessary calls to DeclareVariable() etc. in  **
** then Kernel.                                      **
\*****************************************************/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "barbados.h"
#include "namedecl.h"
#include "source.h"
#include "array.h"
#include "name.h"
#include "make.h"
#include "compiler.h"

#define DECBUF_SIZE     1000


interface str param_name[100];			// The parameter names
interface unsigned int pn_idx;			// An index into the above
interface Namedobj *context_fn;			// What function are we now defining?
interface Classdef* context_class;		// What class's namespace are we 'in'?
interface Classdef* declarator_class;	        // What class are we doing a member fn of?
interface storage_enum default_storage;
interface int auto_offset, max_auto_offset;     // For storage of auto variables
interface int member_offset;                    // For storage of class members
interface int member_bit_offset;                // For storage of bit-fields
interface InsideAggregateEnum InsideAggregate;	// Are we inside a struct, class, union or nothing?
interface visibility_enum visibility;           // public, private or protected.
interface int enum_value;                       // The current value for an enum-const decl.
interface int BitField;				// The width of the current field in bits.
interface Directory* declaration_directory;	// =curdir or NULL (for locals/members)
interface AutoNamedobj* localvar_root;          // Allows us to iterate over all local
                                                // vars, even those out of scope.
interface LocalStaticNamedobj* localstaticvar_root; // Like above, but for local_static.
interface Namedobj* dbg_obj;

static Namedobj* fund_obj;			// The base type in a declaration (first type word)
static storage_enum storage;
static Classdef* UnprocessedClassdef;	        // A classdef which has been compiled
						// but not compacted.
static scopemarker_type enclosingclass_scope;   // The 'scope' of the immediate enclosing class
struct DefArg_node {                            // For default arguments
        int size;
        char *data;
        int min_arity;
};


static Type ParseDeclarator(Type *basep, Type decs, char name[], DefArg_node *DefArg);
static Type ParseParameter(Type type, char name[]);
static Classdef* ParseAggregateType(token_enum aggr, Type type);
static Namedobj* ParseEnum(void);
static void PrintDeclaration(Namedobj* obj);
static Classdef* PostProcessClassdef(Classdef* classdef, Classdef* prevclass);
static Namedobj* FindMemberFunction(Classdef* cl, str name, Type type);
static Type ReplacePointers(Type type, void* oldie, void* newie);





interface void DeclareInit(void)
{
	default_storage = static_storage;
	InsideAggregate = not_inside;
	UnprocessedClassdef = NULL;
}


interface Namedobj* CreateAutoObj(str name, Type type, storage_enum storage, int offset)
{       AutoNamedobj *obj;
        uint size,tsize;
        str s;

        /* Create the basic object: */
	tsize = LengthOfTypeString(type);
        size = sizeof(AutoNamedobj) + strlen(name)+1 + tsize;
        obj = (AutoNamedobj*)qmalloc(size);
        obj->storage = storage;
        obj->type = (Type)(obj + 1);
	memcpy(obj->type, type, tsize);
	s = (str)obj->type + tsize;
	obj->name = strcpy(s, name);
        obj->offset = offset;
        obj->initialisation = NULL;

        /* Create the vargraph and localvar list data-structures: */
	obj->vargraph = NULL;
	obj->last_use = NULL;
        obj->nextlocal = localvar_root;
        localvar_root = obj;

        return obj;
}


interface LocalStaticNamedobj* CreateLocalStaticObj(str name, Type type)
{       LocalStaticNamedobj *obj;
        uint size,tsize;
        str s;

        /* Create the basic object: */
	tsize = LengthOfTypeString(type);
        size = sizeof(LocalStaticNamedobj) + strlen(name)+1 + tsize;
        obj = (LocalStaticNamedobj*)qmalloc(size);
        obj->storage = local_static;
        obj->type = (Type)(obj + 1);
	memcpy(obj->type, type, tsize);
	s = (str)obj->type + tsize;
	obj->name = strcpy(s, name);
        obj->location = NULL;

        /* The special fields: */
        obj->nextlocal = localstaticvar_root;
        localstaticvar_root = obj;

        return obj;
}


interface AutoNamedobj* CreateTemporaryVariable(Type type)
/* Create a temporary local variable of this type. */
{       extern int max_auto_offset;
	AutoNamedobj *obj;

	max_auto_offset -= TypeSize(type);
        obj = (AutoNamedobj*)CreateAutoObj("",type,auto_storage, max_auto_offset);
        return obj;
}


interface void InitLocalVars(void)
{
	InitLocalStatic();
	localvar_root = NULL;
        localstaticvar_root = NULL;
}


static Classdef* IsTypedefOwner(Namedobj *obj)
/* Is this object the owner of any classdef? */
{	Type type=obj->type;
	Classdef* classdef;
	int dimension;

	do switch (*type++) {
	    case tp_class:
		classdef = *(Classdef**)type;
		if (classdef->typedef_obj == obj)
		    return classdef;
		else return NULL;

	    case tp_pointer:
	    case tp_reference:
	    case tp_const:
	    case tp_volatile:
	    case tp_dynarray:
		continue;

	    case tp_array:
		GetDimension(dimension, type);  // macro
		continue;

	    default:
		return NULL;
	} forever;
}


static void* ClassdefVirtualFn(Classdef* classdef, int virtualfn_idx)
/* Look up this virtual function in this installed class. */
/* Recall that the representation of virtual-fn tables    */
/* changes when it is installed. */
{       void** vftable;

	vftable = (void**)((char*)&classdef->VirtualFns + 4);
        if (virtualfn_idx >= Array_Size(vftable))
            return NULL;
        else return vftable[virtualfn_idx];
}


interface void ParseOperatorName(char dest[])
/* We require an operator in tok.buf. Construct an operator name */
/* out of it into 'dest[]'.   We assume the word 'operator' has  */
/* already been gobbled, and we want the rest of the phrase to   */
/* be gobbled by the time we leave this function. */
{	str s;

	s = strcpy(dest, "operator");
        s += strlen(s);
	switch (tok.en) {
            case op_sing('+'):
            case op_sing('-'):
            case op_sing('*'):
            case op_sing('/'):
            case op_sing('%'):
            case op_sing('&'):
            case op_sing('|'):
            case op_sing('^'):
            case op_doub('+'):
            case op_doub('-'):
            case op_doub('&'):
            case op_doub('|'):
            case op_unry('!'):
            case op_sing('='):
            case op_sing('<'):
            case op_sing('>'):
            case op_doub('='):
            case op_assg('<'):
            case op_assg('>'):
            case op_doub('<'):
            case op_doub('>'):
            case op_assg('+'):
            case op_assg('-'):
            case op_assg('*'):
            case op_assg('/'):
            case op_assg('%'):
            case op_assg('&'):
            case op_assg('|'):
            	strcpy(s, tok.buf);
                NextToken();
                return;

            case open_round:
            	NextToken();
                if (tok.en != close_round)
                    ErrorParse(tok.src, "Expecting:  operator()");
            	strcpy(s, "()");
                NextToken();
                return;

            case open_square:
            	NextToken();
                if (tok.en != close_square)
                    ErrorParse(tok.src, "Expecting:  operator[]");
            	strcpy(s, "[]");
                NextToken();
                return;

            case kw_new:
            case kw_delete:
            	*s++ = ' ';
                strcpy(s, tok.buf);
                NextToken();
                return;

            default:
            	ErrorParse(tok.src, "You need to have an operator after "
                	"the word 'operator' (one of the allowable operators)");
                return;
        }
}


interface str AbstractClassExplanation(Classdef* classdef, char dest[])
/* Create an explanation of why this classdef is abstract. */
/* If it was declared with a "fn() = 0" syntax, then no    */
/* explanation is necessary, otherwise explain further.    */
{	Namedobj* obj;

	/* Was a pure virtual function declared in this classdef? */
	for (obj=classdef->member; obj; obj=obj->next) {
            if (obj->storage == virtual_fn) {
                FunctionNamedobj* fobj=(FunctionNamedobj*)obj;
            	if (ClassdefVirtualFn(classdef, fobj->u.virtualfn_idx) == PureVirtual) {
                    dest[0] = '\0';
                    return dest;
                }
            }
        }

        /* Okay, which base class has a pure virtual function not defined? */
	for (obj=classdef->member; obj; obj=obj->next) {
            if (obj->storage == inherit_storage) {
            	Classdef* baseclass = TypeToClassdef(obj->type);
                if (baseclass == NULL)
                    continue;
                if (baseclass->RoundupSize != -1)
                    continue;
                for (Namedobj* bobj=baseclass->member; bobj; bobj=bobj->next) {
                    if (bobj->storage == virtual_fn and
                        ClassdefVirtualFn(classdef,
                                ((FunctionNamedobj*)bobj)->u.virtualfn_idx) == PureVirtual) {
                        sprintf(dest, "\"%s\" is abstract because e.g. the pure "
                        	"virtual \"%s\" has never been defined.",
                                classdef->typedef_obj->name,
                                bobj->name);
                        return dest;
                    }
                }
            }
        }
        sprintf(dest, "Actually, I don't know why %s is abstract.",
        	classdef->typedef_obj->name);
        return dest;
}


interface Type ParseBaseType(Type d)
/* Parse a base type into this buffer.  Contains only basic types, */
/* e.g. tp_int, tp_char etc., or modifiers e.g. unsigned. */
{	Classdef* classdef;
	token_enum fund_tok;
	bool Unsigned=no;
	str src;

	LOOP:
	fund_tok = tok.en;
        fund_obj = NULL;
	src = tok.src;
	if (fund_tok != identifier and fund_tok != kws_identifier_path)
	    NextToken();

	switch (fund_tok) {
	    case kw_void:   *d++ = tp_void; return d;
	    case kw_bool:   *d++ = tp_bool; return d;
	    case kw_char:   *d++ = Unsigned ? tp_uchar : tp_char; return d;
	    case kw_short:  if (tok.en == kw_int) {
				NextToken();
				*d++ = Unsigned ? tp_ushort : tp_short;
			    }
			    else *d++ = tp_short;
                            return d;
	    case kw_int:    *d++ = Unsigned ? tp_uint : tp_int;  return d;
	    case kw_long:   if (tok.en == kw_int) {
				NextToken();
				*d++ = tp_long;
			    }
			    else if (tok.en == kw_double) {
				NextToken();
				*d++ = tp_double;
			    }
			    else *d++ = tp_long;
			    return d;
	    case kw_int64:  *d++ = tp_int64; return d;
	    case kw_float:  *d++ = tp_float; return d;
	    case kw_double: *d++ = tp_double; return d;
	    case kw_container:
			    *d++ = tp_container; return d;

	    case kw_unsigned:
			    Unsigned = yes;
			    if (tok.en == kw_char or tok.en == kw_short or tok.en == kw_int
				or tok.en == kw_long or tok.en == kw_enum or tok.en == kw_bool)
				goto LOOP;
			    else if (tok.en == kw_float or tok.en == kw_double or tok.en == kw_struct
				    or tok.en == kw_union)
				ErrorParse(src, "You can only have unsigned char's, short's and int's.");
			    *d++ = tp_uint;
			    return d;

            case kw_signed:
             		    Unsigned = no;
                            goto LOOP;

	    case kw_const:
			    *d++ = tp_const;
			    goto LOOP;

	    case kw_volatile:
			    *d++ = tp_volatile;
			    goto LOOP;

	    case identifier:fund_obj = ResolveTok(declarator_class);
			    if (fund_obj == NULL) {
				ErrorParse(src, "Unknown type : %s", tok.buf);
				*d++ = tp_error;
                                SkipTo(semi_colon);
                                return d;
			    }
			    NextToken();

			    PATH_FOUND:
                            /* At this point, we've gobbled up the token that */
                            /* gave rise to 'fund_obj': */
			    if (fund_obj->storage != typedef_storage) {
				ErrorParse(src, "\"%s\" is not a type name.", fund_obj->name);
				*d++ = tp_error;
			    }
			    else {
				memcpy(d, fund_obj->type, LengthOfTypeString(fund_obj->type));
				MakeDepends(fund_obj->make);
			    }
                            if (*d == tp_class)
                                classdef = *(Classdef**)(d+1);  //$$$ debugging
                            if (tok.en == double_colon) {
                            	Gobble(double_colon);
                                if (fund_obj->type[0] != tp_class)
                                    ErrorParse(tok.src, "%s is not a class/struct/union.",
                                    		fund_obj->name);
                                else if (tok.en != identifier)
                                    ErrorParse(tok.src, "Expecting an identifier");
                                else {
                                    Classdef* classdef = *(Classdef**)(fund_obj->type+1);
				    fund_obj = ResolveMember(classdef, tok.buf, private_visibility);
                                    if (fund_obj == NULL) {
                                        ErrorParse(src, "Unknown member typedef : %s", tok.buf);
                                        *d++ = tp_error;
                                        SkipTo(semi_colon);
                                        return d;
                                    }
                                    NextToken();
                                    goto PATH_FOUND;
                                }
                            }
			    return d;

	    case op_sing('/'):
	    case dot:       PushTokenBack(fund_tok, NULL, src);
			    fund_obj = ParsePath();
			    if (Error.err) {
				*d++ = tp_error;
                                return d;
                            }
			    goto PATH_FOUND;

	    case kws_identifier_path:
			    fund_obj = tok.pathobj;
                            NextToken();
			    memcpy(d, fund_obj->type, LengthOfTypeString(fund_obj->type));
			    if (fund_obj->storage == member_fn or fund_obj->storage ==
					virtual_fn) {
			    	declarator_class = (Classdef*)fund_obj->owner;
                                if (IsDestructor(fund_obj)) {
                                    /* A special case for destructor declarations. */
                                    if (tok.en == identifier)
                                        NextToken();
                                    else goto DESTRUCTOR_ERROR;
                                    if (tok.en == open_round)
                                        NextToken();
                                    else goto DESTRUCTOR_ERROR;
                                    if (tok.en == close_round)
                                        NextToken();
                                    else {
                                        DESTRUCTOR_ERROR:
                                        ErrorParse(src, "Strange destructor syntax - "
                                        "I was expecting:  %s::~%s()",
                                        declarator_class->typedef_obj->name,
                                        declarator_class->typedef_obj->name);
                                    }
                                    PushTokenBack(identifier, NULL, src);
                                    strcpy(tok.buf, "~");
                                }
                            }
			    return d;

	    case kw_enum:   *d++ = tp_enumerated;
			    fund_obj = ParseEnum();
			    PutPtr(fund_obj, d);
			    return d;

	    case kw_union:
	    case kw_class:
	    case kw_struct: *d++ = tp_class;
			    classdef = ParseAggregateType(fund_tok, d);
			    PutPtr(classdef, d);
			    return d;

	    case kw_typedef:
	    case kw_static:
	    case kw_virtual:
            case kw_inline:
	    case kw_type:   ErrorParse(src, "You can't have storage types here.");
			    return d;

	    default:        ErrorParse(src, "Illegal declaration : you need a base type, e.g. int, char, struct S, etc.");
			    *d++ = tp_error;
			    return d;
	}
}


interface void ParseDeclaration(void)
/* Parse a full declaration. Does typedefs, initialisations, etc. */
/* The syntax is:  [<storageclass>] [<basetype>] {<declarator>}   */
/* The declaration goes into 'dec_buf[]'. */
{       Classdef* owner, *UnnamedClassdef, *orig_context_class;
	uchar base_buf[DECBUF_SIZE];	    // The base type
	uchar dec_buf[DECBUF_SIZE];	    // The full type
        int overload_version;
        DefArg_node DefArg;
	Type type, base;
	Namedobj* obj;
        str src;

        if (tok.en == semi_colon) {
            Gobble(semi_colon);
            return;		// Allow 'null' declarations, e.g.
            // what you get after the '}' in this example:
            // int foo() { return 5; };
        }


	/*** Parse the storage specifier: ***/
	storage = default_storage;
	if (not InsideAggregate)
	    visibility = private_visibility;
        orig_context_class = context_class;
	switch (tok.en) {
	    case kw_static: if (storage == auto_storage)
				storage = local_static;
			    NextToken();
			    break;
	    case kw_typedef:storage = typedef_storage;
			    NextToken();
			    break;
	    case kw_virtual:if (storage != member_storage) {
				if (storage == auto_storage)
				    ErrorParse(tok.src, "You're not allowed to declare a virtual function "
					    "outside a class definition.");
				else if (storage == static_storage)
				    ErrorParse(tok.src, "The keyword 'virtual' belongs only in the class definition.");
				else assert(false);
			    }
			    else storage = virtual_fn;
			    NextToken();
			    break;
            case kw_inline: //storage = inline_storage;
              		    NextToken();
                            break;
	    case kw_type:   NextToken();	    // keywords to ignore.
			    break;
	    default:	    break;
	}


	/*** Parse the base type. This can cause classdef's and enum's to be parsed. ***/
	ParseBaseType((Type)base_buf);
	if (Error.err)
	    return;


	/* Do the declarator list. */
	do {

	    /* Get the declarator: */
	    char declarator_name[512];
	    declarator_name[0] = '\0';
	    base = base_buf;
            src = tok.src;
	    type = ParseDeclarator(&base, (Type)dec_buf, declarator_name, &DefArg);
	    if (Error.err) {
		SkipTo(semi_colon);
		Gobble(semi_colon);
		return;
	    }


	    /* Append the base type: */
	    type = TypeAppend(type, base);


	    /* Process any newly-created class, if needed: */
	    UnnamedClassdef = NULL;
	    if (UnprocessedClassdef) {
		Classdef* classdef, *prevclass;

		// We would have processed it immediately except that we
		// first need to know the declarator name, in case there's
		// a previous version of this class under that name.
		prevclass = NULL;
		if (curdir and default_storage == static_storage) {
		    obj = curdir->Find(declarator_name);
		    if (obj and *obj->type == tp_class)
			prevclass = *(Classdef**)(obj->type+1);
		}
		classdef = UnprocessedClassdef;
		if (classdef->member) {
		    classdef = PostProcessClassdef(classdef, prevclass);
                    ReplacePointers(dec_buf, UnprocessedClassdef, classdef);
                    ReplacePointers(base_buf, UnprocessedClassdef, classdef);
		    UnprocessedClassdef = NULL;
		}
		if (classdef->typedef_obj == NULL)
		    UnnamedClassdef = classdef;
	    }


            /* Now that we have both the name and the type, perform some checks: */
	    if (declarator_name[0] == '\0') {
            	// This is probably an error
		if (tok.en == semi_colon and (*base_buf == tp_class or
                				*base_buf == tp_enumerated)) {
                    // We're allowed to be without a declarator in
                    // the case of:   "class X { .. };"  (Note that X is the
                    // class name, not a declarator in this case).
                    if (UnnamedClassdef)
		    	ErrorParse(tok.src, "Pointless class/struct/union declaration");
                        // "class { .. };" is not allowed.
                }
                else if (tok.en == op_sing('~')) {
                    if (declarator_class == NULL or
                    		declarator_class->typedef_obj == NULL)
                    	ErrorParse(tok.src, "What's the '~' doing? It's obviously "
                        	"not part of a destructor declaration.");
                    else ErrorParse(tok.src, "Destructors are declared as follows: "
                    	"\"~%s();\"", declarator_class->typedef_obj->name);
                }
                else
		    ErrorParse(tok.src, "You're missing an identifier declarator.");
		break;
	    }
            if (*dec_buf == tp_class and storage != typedef_storage) {
            	// Check: is the user trying to declare an object of an abstract type?
            	Classdef* classdef = *(Classdef**)(dec_buf+1);
		if (classdef->RoundupSize == -1) {
                    char buf[512];
                    ErrorParse(tok.src, "\"%s\" is an abstract class, so you can't "
                    	"declare any object of this type. %s",
                        classdef->typedef_obj->name,
                        AbstractClassExplanation(classdef, buf));
                    break;
                }
                else if (classdef->RoundupSize == -2) {
                    ErrorParse(tok.src, "\"%s\" is an incomplete class, so you "
                    	"can't declare instances of it.",
                        classdef->typedef_obj->name);
                }
            }
            if (default_storage == member_storage and declarator_class->typedef_obj
                and streq(declarator_name, declarator_class->typedef_obj->name)) {
                /* It's a constructor: */
		if (dec_buf[0] != tp_function)
		    ErrorParse(tok.src, "You can't have a data field with the "
                        "same name as the class.");
		else {
		    Type type = TypeFunctionReturnType(dec_buf);
		    if (*type != tp_void)
			ErrorParse(tok.src, "Constructors cannot have return types");
		}
            }
            if (default_storage == member_storage) {
            	overload_version = 0;
            	for (obj=NamePartialMembers();
                                obj and obj != enclosingclass_scope;
                                obj=obj->next) {
                    if (streq(obj->name, declarator_name)) {
                    	char tmp[512], *s1, *s2;
                    	if (*obj->type != tp_function or *dec_buf != tp_function)
                            ErrorParse(tok.src, "There is already a '%s' declared "
                            		"in this class - you can't have two.",
                        		declarator_name);
                        else if (EqualFootprint(obj->type, dec_buf)) {
                            TypeToString(dec_buf, tmp, sizeof(tmp));
                            s1 = strchr(tmp, '(');
                            s2 = strchr(tmp, ')');
                            s2[1] = '\0';
                            ErrorParse(tok.src, "There is already a '%s%s' in this class.",
                        		declarator_name,s1);
                        }
                        else {
                            if (obj->overload_version >= overload_version)
                        	overload_version = obj->overload_version + 1;
                        }
                    }
                }
            }
            if (*dec_buf == tp_reference and tok.en != op_sing('=')
            	and storage != parameter_storage) {
            	ErrorParse(tok.src, "Whenever you declare a reference variable, "
                	"you must initialise it to some variable (otherwise how can "
                        "you ever attach it to a value?)");
            }


	    /*** Declare the object: ***/
	    MakeCanBeClosed = (tok.en != open_brace);
            if (*dec_buf == tp_array and *(int*)(dec_buf+1) == 0) {
                // A conforming array: the sizeof, and therefore the type,
                // depends on how many initialisation items we find.
                if (tok.en != op_sing('='))
                    ErrorParse(tok.src, "The array size is missing.");
                NextToken();
                obj = ParseConformingArray(
                        default_storage == static_storage ? declaration_directory : NULL,
                        declarator_name,
                        (Type)dec_buf,
                        storage,
                        NAME_CREATEMAKE);
            }
            else if (storage == virtual_fn) {
            	// Compare this function with virtual functions in the base classes.
                char tmp1[80],tmp2[80];
                FunctionNamedobj *vobj;
                Type oldrett,newrett;
                int virtualfn_idx;

                vobj = FindMemberFn(declarator_class, declarator_name, dec_buf);
                if (vobj) {
                    // Link in with an existing virtual function
                    if (vobj->storage != virtual_fn)
                    	ErrorType(tok.src, "Confusion: Is \"%s\" a virtual "
                        	"function or a %s?",
                        	declarator_name, StorageToString(vobj->storage));
                    oldrett = TypeFunctionReturnType(vobj->type);
                    newrett = TypeFunctionReturnType(dec_buf);
                    if (not TypeEqual(oldrett, newrett))
                        ErrorType(tok.src, "\"%s\" was supposed to have returned %s,"
                        	"but here it's redeclared to return a %s.\n",
                                declarator_name,
                                TypeToString(oldrett,tmp1,sizeof(tmp1)),
                                TypeToString(newrett,tmp2,sizeof(tmp2)));
                    virtualfn_idx = vobj->u.virtualfn_idx;
                    declarator_class->VirtualFns[virtualfn_idx] = NULL;
                }
                else {
                    // Create a new virtual function
                    virtualfn_idx = Array_Size(declarator_class->VirtualFns);
                    Array_Add(declarator_class->VirtualFns, NULL);
                }
		obj = NameDeclare(NULL,
				declarator_name,
				(Type)dec_buf, storage, NAME_OVERLOAD);
                ((FunctionNamedobj*)obj)->u.virtualfn_idx = virtualfn_idx;
                obj->owner = declarator_class;
            }
	    else if (declarator_class == NULL or default_storage != static_storage) {
            	bool NeedMake;
                if (PredefinitionPhase)
                    NeedMake = no;
                else if (default_storage == member_storage)
                    NeedMake = (*dec_buf == tp_function or storage == static_storage);
                else NeedMake = yes;
                if (default_storage == static_storage and
                        (obj=declaration_directory->Find(declarator_name)) != NULL) {
                    if ((*obj->type == tp_function) != (*dec_buf == tp_function)) {
                        ErrorParse(tok.src, (*obj->type == tp_function) ?
                                "There is a function called \"%s\", and you can't override it "
                                "with a data object except by explicitly deleting the function "
                                "first (I'm afraid you'll wipe out some valuable work). Type "
                                "\"del(%s);\"." :
                                "There is a data-object called \"%s\", are you sure you want "
                                "to override it with a function?  If so, delete it first using "
                                "the \"del(%s);\" command.",
                                declarator_name, declarator_name);
                        return;
                    }
                }
                /* The payload!: */
		obj = NameDeclare(default_storage == static_storage ? declaration_directory : NULL,
				declarator_name,
				(Type)dec_buf, storage,
                                NeedMake ? NAME_OVERLOAD | NAME_CREATEMAKE : NAME_OVERLOAD);
                /* */
                if (declarator_class) {
                    obj->owner = declarator_class;
                    obj->overload_version = 0;//overload_version;
                }
	    }
	    else {
		obj = FindMemberFunction(declarator_class, declarator_name, (Type)dec_buf);
		if (obj == NULL) {
		    ErrorParse(tok.src, "No \"%s\" function of this type was declared.", declarator_name);
		    return;
		}
		if (obj->make == NULL) {
		    /* The only way this can occur is if we open an old */
		    /* container, and we're in the process of linking in the  */
		    /* source right now. */
		    obj->make = MakeEntityNew(obj);
		}
	    }
	    MakeCanBeClosed = yes;
            context_class = orig_context_class;


            /* Functions have extra stuff: */
            if (*dec_buf == tp_function) {
                FunctionNamedobj *fobj=(FunctionNamedobj*)obj;
                fobj->DefaultParameters = DefArg.data;
                fobj->MinimumArity = DefArg.min_arity;
                fobj->Throws = NULL;
                if (DefArg.size) {
                    free(DefArg.data);
                    DefArg.data = NULL;
                    DefArg.size = 0;
                }
            }


	    /* Classdefs each need a typedef_obj: */
	    if (UnnamedClassdef) {
		UnnamedClassdef->typedef_obj = obj;
                UnnamedClassdef = NULL;
            }

	    if (tok.en == op_sing('=')) {

		/*** Initialisations: ***/
		src = tok.src;
                NextToken();
                if (*obj->type == tp_function) {
                    if (tok.en != int_const or tok.int_val != 0) {
                    	ErrorParse(src, "You can't assign anything to a function "
                        	"(except in the case \"fn()=0\").");
                        return;
                    }
                    NextToken();
                    if (obj->storage != virtual_fn or default_storage != member_storage) {
                    	ErrorParse(src, "Only virtual functions inside class definitions "
                        	"are allowed the \"= 0;\" syntax.");
                        return;
                    }
                    int virtualfn_idx = ((FunctionNamedobj*)obj)->u.virtualfn_idx;
                    declarator_class->VirtualFns[virtualfn_idx] = PureVirtual;
                }
                else {
                    ParseInitialisation(obj);
                }
	    }
            else if (TypeHasConstructor(obj->type) and default_storage != member_storage) {
                Expr* e = NewExprIdentifier(obj);
                e->src = src;
                e = ParseConstructorExpr(obj->type,e);
            	AddInitialisingExpr(e);
                if (obj->storage == auto_storage)
                    ((AutoNamedobj*)obj)->initialisation = e;
            }
	    else if (tok.en == open_brace or tok.en == colon) {
		if (*obj->type != tp_function) {
                    ErrorParse(tok.src, "If you're trying to define a function, "
                    	"you need the round-bracket notation.");
                    return;
                }

		/*** Function Definitions: ***/
		owner = declarator_class;
		if (default_storage == auto_storage) {
		    ErrorParse(tok.src, "You can't define a function inside another function!");
		    return;
		}
                assert(default_storage == static_storage or default_storage == member_storage);
		/*if (storage == virtual_fn) {
                    if (obj->make == NULL and not PredefinitionPhase) {
                        obj->make = MakeEntityNew(obj);
                        MakeChanged(obj->make, yes);
                    }
		}*/
		if (obj->make)
		    obj->make->compile_stamp -= 2;
		    // The body is not compiled, so mark it as not up-to-date.
		CompilerCurrentFuncObj = (FunctionNamedobj*)obj;
		ParseFunctionDefinition(CompilerCurrentFuncObj);
		CompilerCurrentFuncObj = MainEntity;
		MakeEntityReset();
		if ((default_storage == static_storage or *obj->type == tp_function)
                        and not PredefinitionPhase and Error.err == 0) {
		    SourceStash(obj, owner);
                    // All objects that belong in directories get source-code.
                    // Also, member functions get source-code.
                    if (*obj->type == tp_function and obj->make)
                        obj->make->source.dinfo = IprGetDebugInfo(
                                /*old object=*/obj->make->source.dinfo);
		}
		if (Error.err == 0 and CompilerInvoker == 'I')
		    PrintDeclaration(obj);
		return;
	    }
	    else if (tok.en == open_round) {
            	LookAhead();
            	ErrorParse(tok.next->src, "What is: %s ?  I don't know if you're "
                	"trying to use a constructor or declare a "
                        "function.", tok.next->buf);
            }


	    /* If we're declaring at the top level, print it. */
	    if (default_storage == static_storage) {
		if (CompilerInvoker == 'I')
		    PrintDeclaration(obj);
                if (*obj->type == tp_container or TypeEqual(obj->type, direc_typstr))
                    ;   // No need to store e.g. "mkdir(myStuff);".
		else SourceStash(obj, NULL);
	    }


	    if (tok.en == comma) {
		NextToken();
		continue;
	    }
	    else break;
	} forever;

	/* It's either a semi-colon, or a function definition. */
	Gobble(semi_colon);
}





/*---------------- Parsing Declarators: -------------------*/

static Type ParseDeclarator(Type *basep, Type type, char name[], DefArg_node *DefArg)
/* Parse a declarator, including pointers, references, arrays,	*/
/* lgo-id's and functions.  Insert it into 'type' and return	*/
/* the end of this type-str. Set the global variable 'name'	*/
/* to be the declarator's identifier, or NULL if there is no	*/
/* identifier.  Return the _end_ of the partial type-str. */
{       Namedobj* obj;
	int dimension;
        str src;


        /* What does a double-colon mean? */
	if (tok.en == double_colon) {
            if (LookAhead() == op_sing('*')) {
                /* It's a ptr to a member function! */
                Gobble(double_colon);
                Gobble(op_sing('*'));
                *type++ = tp_ptrmemberfn;
                if (tok.en != identifier)
                    ErrorParse(tok.src, "This is ptr-to-member-fn syntax, so "
                                "you need an identifier here.");
                strcpy(name, tok.buf);
                Gobble(identifier);
                Gobble(open_round);
                goto INSIDE_FN;
            }
            else if (declarator_class and streq(tok.next->buf,
            			declarator_class->typedef_obj->name)) {
                /* Constructor definitions that have this syntax:   T::T(...); */
                strcpy(name, fund_obj->name);
                **basep = tp_void;
                obj = fund_obj;
                goto DOUBLE_COLON;
            }
            else {
            	ErrorParse(tok.src, "I wasn't expecting a double-colon here.");
            }
	}

	/* Constructor prototypes that have this syntax:   class T {  ... T(...); } */
        if (fund_obj == NULL)
            ;	// can't possibly be a constructor
	else if (tok.en == open_round and LookAhead() != op_sing('*')) {
	    strcpy(name, fund_obj->name);
	    **basep = tp_void;
	    goto POST_OPERATORS;
	}
        else if (tok.en == kws_identifier_path and IsConstructor(tok.pathobj)) {
            NextToken();
	    strcpy(name, fund_obj->name);
	    **basep = tp_void;
	    goto POST_OPERATORS;
	}

	/* Pre-operators: */
	PRE_OPERATORS:
	switch (tok.en) {
	    case identifier:
              	    if (LookAhead() == double_colon) {
                        obj = ResolveTok(declarator_class);
                        if (obj == NULL) {
                            ErrorType(tok.src, "There's no such class: %s", tok.buf);
                            NextToken();
                            goto PRE_OPERATORS;
                        }
                        DOUBLE_COLON:
                        if (obj->storage != typedef_storage or *obj->type != tp_class)
                            ErrorType(tok.src, "\"%s\" is not a class.", tok.buf);
                        else declarator_class = context_class = TypeToClassdef(obj->type);
                        NextToken();
                        Gobble(double_colon);
                        if (tok.en == kw_operator) {
                            NextToken();
                            ParseOperatorName(name);
                        }
                        else if (tok.en == identifier)
                            goto PRE_OPERATORS;
                        else {
                            ErrorType(tok.src, "I was expecting an identifier after the ::.");
                            goto PRE_OPERATORS;
                        }
                    }
                    else {
                        strcpy(name, tok.buf);
		    	NextToken();
                    }
		    break;

	    case op_sing('*'):
		    NextToken();
		    type = ParseDeclarator(basep, type, name, DefArg);
		    *type++ = tp_pointer;
		    return type;

	    case op_sing('&'):
		    NextToken();
		    type = ParseDeclarator(basep, type, name, DefArg);
		    *type++ = tp_reference;
		    return type;

	    case open_round:
		    /* These are a grouping brackets, not function brackets. */
		    Gobble(open_round);
		    type = ParseDeclarator(basep, type, name, DefArg);
		    Gobble(close_round);
		    break;

	    case kw_operator:
		    NextToken();
		    ParseOperatorName(name);
		    break;

	    case kw_bool:
	    case kw_char:
	    case kw_int:
	    case kw_long:
	    case kw_float:
	    case kw_double:
	    case kw_container:
	    case kws_directory:
                    src = tok.src;
		    strcpy(name, tok.buf);
		    NextToken();
                    if (tok.en != open_round) {
                        ErrorParse(src, "\"%s\" is a keyword; it can't be declared.", name);
                        *name = '\0';
                    }
		    break;

            case int_const:
            	    ErrorParse(tok.src, "You can't have an integer where we're "
                    	"expecting a declarator (identifier)");
            	    break;
	}


	/* Post-operators: */
	POST_OPERATORS:
	do {
	    if (tok.en == open_round) {
		unsigned char *no_params_ptr;
		unsigned char no_params;
                bool success;
                int size;
                Expr* e;
                str src;

                /* Are we in the middle of a function declaration or a 	*/
                /* variable declaration with round-bracket-for-		*/
                /* constructor. It can sometimes be a very subtle 	*/
                /* distinction, for example:  C c(C g); is a fn	   	*/
                /* declaration but C c(g); is a declaration-with-	*/
                /* constructor. */
                src = tok.src;
                NextToken();
                if (tok.en != close_round and not AtTypeExpression()) {
                    PushTokenBack(open_round, NULL, src);
                    return type;
                }

		/* Declaring functions: */
		*type++ = tp_function;
                INSIDE_FN:
		no_params_ptr = type++;
		no_params = 0;
                if (DefArg) {
                    DefArg->size = 0;
                    DefArg->data = NULL;
                    DefArg->min_arity = 0;
                }
		if (tok.en != close_round) {
		    do {
			char pname[512];

			if (tok.en == dot) {
			    /* Ellipsis dots for vararg functions. */
			    Gobble(dot);
			    Gobble(dot);
			    Gobble(dot);
			    no_params |= 128;
			    break;
			}
			type = ParseParameter(type, pname);
			if (no_params_ptr[1] == tp_void) {
			    /* A function declared as:  f(void).  */
			    if (no_params != 0)
				ErrorParse(tok.src, "Illegal use of `void'");
			    type = no_params_ptr + 1;
			    break;
			}
			if (no_params >= 100)
			    break;
			if (param_name[no_params])
			    free(anon_heap, param_name[no_params]);
			param_name[no_params++] = strdup(anon_heap, pname);
                        if (DefArg == NULL) {
                            if (tok.en == op_sing('='))
                                ErrorParse(tok.src, "Default parameters cannot "
                                "be used in a function pointer declaration.");
                        }
                        else if (tok.en == op_sing('=')) {
                            /* Default (Optional) parameters: */
                            Gobble(tok.en);
                            e = ParseExpr(yes);
                            success = IprEvaluateConstant(&e);
                            if (not success)
                                ErrorParse(tok.src, "Default parameters does not "
                                        "evaluate to a constant.  (ANSI C++ can "
                                        "handle this but Barbados can't yet).");
                            size = e->ref ? 4 : TypeSizeWord(e->type);
                            if (DefArg->size == 0) {
                                DefArg->data = (char*)malloc(default_heap,
                                                DefArg->size=size);
                            }
                            else {
                                DefArg->data = (char*)realloc(default_heap,
                                                DefArg->data, DefArg->size+size);
                                memmove(DefArg->data+size,DefArg->data,DefArg->size);
                                DefArg->size += size;
                            }
                            memcpy(DefArg->data, &e->u, size);
                        }
                        else {
                            DefArg->min_arity = no_params;
                            if (DefArg->size)
                                ErrorParse(tok.src, "Default parameters must be "
                                "declared after all non-default parameters.");
                        }
			if (tok.en == close_round or tok.en == EOF)
			    break;
			else Gobble(comma);
		    } forever;
		}
		Gobble(close_round);
		*no_params_ptr = pn_idx = no_params;
		*type++ = tp_terminated;
		return type;
	    }
	    else if (tok.en == open_square) {
		/* Declaring arrays: */
		NextToken();
		if (tok.en == ternary) {
		    /* Dynamic arrays: */
		    *type++ = tp_dynarray;
		    NextToken();
		    Gobble(close_square);
		    continue;
		}
		else if (tok.en == close_square)
		    dimension = 0;
		else dimension = ParseAndEvaluateIntExpr(no);
		Gobble(close_square);
		*type++ = tp_array;
		PutDimension(dimension, type);
		continue;
	    }
	    else if (tok.en == colon) {
		if (default_storage != member_storage)
		    ErrorParse(tok.src, "You can only have bit-fields inside struct/union/class's.");
		NextToken();
		BitField = ParseAndEvaluateIntExpr(yes);
		if (BitField < 0 or BitField >= 32)
		    ErrorParse(tok.src, "Bad bit-field : it must be from 0 to 31.");
		if (BitField == 0)
		    BitField = -1;
		return type;
	    }
	    else {
		return type;
	    }
	} forever;
}


static Type ParseParameter(Type type, char name[])
/* Parse a parameter-expression, from the fundamental type */
/* up to but not including the comma or close_round.  Can  */
/* accept either a named parameter or just a type expr.    */
{	Type base, typeend;
	uchar base_buf[512];

	ParseBaseType((Type)base_buf);
	base = base_buf;
	typeend = ParseDeclarator(&base, type, name, NULL);
	if (typeend > type and *type == tp_array) {
	    type[0] = tp_const;
	    type[1] = tp_pointer;
	    memcpy(type+2, type+5, typeend-type-5);
	    typeend -= 3;
	}
	if (base)
	    typeend = TypeAppend(typeend, base);
	return typeend;
}


interface Type ParseTypeExpression(void)
/* Parse a pure type expression. This is like a full */
/* declaration but without storage-class specifiers  */
/* and without an identifier. */
{	static uchar dec_buf[512];
	Type base, typeend;
	uchar base_buf[512];
	char name[512];

	ParseBaseType((Type)base_buf);
	base = base_buf;
	typeend = ParseDeclarator(&base, (Type)dec_buf, name, NULL);
	if (base)
	    typeend = TypeAppend(typeend, base);
	return (Type)dec_buf;
}



/*---------------- Aggregate Declarations ---------------*/

static Classdef* DeclareStubClass(str name)
/* Declare an empty classdef to be used in a forward declaration. */
{	static uchar decbuf[6] = { 0, tp_class, 0, 0, 0, 0 };
	Type type=decbuf+2;
	Classdef* classdef;
	Namedobj* obj;

	classdef = (Classdef*)default_heap->malloc(sizeof(Classdef));
	classdef->size = 0;
	classdef->RoundupSize = -2;
	classdef->def_len = 0;
	classdef->member = NULL;
	classdef->signature = HELLO_BABE;
	classdef->typedef_obj = NULL;
	classdef->VirtualFns = NULL;
        classdef->Friends = NULL;
        classdef->ptrfinder = NULL;
	PutPtr(classdef, type);
	obj = NameDeclare(curdir, name, decbuf+1, typedef_storage, NAME_CREATEMAKE);
	// We want it always within a function's name-space
	// but outside an enclosing classdef's name-space.
	if (obj->make)
	    MakeEntityClear(obj->make);
	classdef->typedef_obj = obj;
	return classdef;
}


static void ParseFriend(Classdef* classdef)
/* Parse a 'friend' declaration up to and including the semi-colon. */
{	Classdef* friendclass;
	Namedobj *obj;

	Gobble(kw_friend);
        if (tok.en == kw_class) {
            // Friend classes:
            Gobble(kw_class);
            obj = ResolveTok(declarator_class);
            if (obj == NULL)
            	friendclass = DeclareStubClass(tok.buf);
            else if (obj->storage == typedef_storage and *obj->type == tp_class)
		friendclass = TypeToClassdef(obj->type);
            else if (obj->owner == declarator_class or obj->owner == curdir) {
            	ErrorParse(tok.src, "Friends must be either classes or functions.");
                NextToken();
                return;
            }
            else friendclass = DeclareStubClass(tok.buf);
            Array_Add(classdef->Friends, friendclass);
        }
        else {
            // Friend functions:
            uchar base_buf[DECBUF_SIZE];    // The base type
            uchar dec_buf[DECBUF_SIZE];	    // The full type
            char declarator_name[512];
            Type base,type;
            Namedobj *friendfn;
            str src;

            ParseBaseType((Type)base_buf);
            if (Error.err)
                return;

            /* Do the declarator list. */
            do {
                /* Get the declarator: */
                declarator_name[0] = '\0';
                base = base_buf;
                src = tok.src;
                type = ParseDeclarator(&base, (Type)dec_buf, declarator_name, NULL);
                if (Error.err) {
                    SkipTo(semi_colon);
                    Gobble(semi_colon);
                    return;
                }

                /* Append the base type: */
                type = TypeAppend(type, base);

                /* Declare it and denote it as a friend of the current class: */
                if (*dec_buf != tp_function) {
                    ErrorParse(src, "There's no such thing as a friend non-function.");
                }
                else {
                    friendfn = NameDeclare(curdir,
				declarator_name,
				(Type)dec_buf,
                                static_storage,
                                NAME_OVERLOAD);
                    Array_Add(classdef->Friends, friendfn);
                }

                /* Additional functions? */
                if (tok.en == comma) {
                    NextToken();
                    continue;
	    	}
	    	else break;
            } forever;
            Gobble(semi_colon);
        }
}


static Classdef* ParseAggregateType(token_enum aggr, Type type)
/* Parse an aggregate type (struct/union/class). Return */
/* the resulting classdef, either in processed form     */
/* (signature=HELLOBABE) or unprocessed form (signature */
/* == 0) depending on whether or not we have a class    */
/* name (typedef_obj). */
{       storage_enum old_storage, old_default_storage;
	InsideAggregateEnum old_InsideAggregate;
        Classdef* old_declarator_class;
	Classdef* classdef, *prevclass;
	visibility_enum old_visibility;
	Directory* old_decldir;
	scopemarker_type scope;
	int old_member_offset;
	Type base_type;
	Namedobj* obj;
	char name[512];
	str src;


	/*** Get the thing's name ***/
	if (tok.en == identifier) {
	    strcpy(name, tok.buf);
	    NextToken();
	}
	else *name = '\0';


	/*** If they specify something of the form: "class <name> var;" ***/
	/*** deal with it here.            ***/
	if (tok.en != open_brace and tok.en != colon) {
	    if (*name == '\0') {
		ErrorParse(tok.src, "Expecting a class-name or '{' here");
		return NULL;
	    }
            // In this case we look for the
	    // class name anywhere in the name space.
	    obj = Resolve(name);
            if (*ResolveErrormsg) {
            	ErrorParse(tok.src, "%s", ResolveErrormsg);
                return NULL;
            }
	    if (obj and obj->storage == typedef_storage and *obj->type == tp_class)
		classdef = *(Classdef**)(obj->type+1);
	    else classdef = DeclareStubClass(name), obj = classdef->typedef_obj;
	    MakeDepends(obj->make);
	    return classdef;
	}


	/*** Establish the classdef (part I) ***/
	WordAlign(member_offset);
	prevclass = NULL;
	classdef = (Classdef*)anon_heap->malloc(sizeof(Classdef));
	classdef->member = NULL;
	classdef->def_len = 0;
        classdef->Friends = NULL;
	classdef->typedef_obj = NULL;
	classdef->signature = HELLO_BABE+1;     // Denotes a half-created classdef
	classdef->VirtualFns = NULL;
        classdef->RoundupSize = -2;
	if (*name == NULL)
	     obj = NULL;
	else {
	    if (default_storage == static_storage)
		obj = curdir->Find(name);	// In this case we look just in curdir.
	    else obj = NULL;
	    if (obj and *obj->type == tp_class and obj->storage == typedef_storage) {
		prevclass = *(Classdef**)(obj->type + 1);
		if (obj->make == NULL and not PredefinitionPhase)
		    obj->make = MakeEntityNew(obj);
	    }
	    else {
		uchar typebuf[5] = { tp_class, 0, 0, 0, 0 };
		*(Classdef**)(typebuf+1) = classdef;
		obj = NameDeclare(
			default_storage == static_storage ? curdir : NULL,
			name, typebuf, typedef_storage, NAME_CREATEMAKE);
		if (default_storage == static_storage and CompilerInvoker == 'I') {
		    classdef->typedef_obj = obj;
		    PrintDeclaration(obj);
		}
                if (declarator_class)
                    obj->owner = declarator_class;
	    }
	}
	classdef->typedef_obj = obj;


	/*** Save the old values & set them up for member declaration ***/
	old_default_storage = default_storage;
	old_storage = storage;
	scope = enclosingclass_scope = NameMarkScope();
	old_member_offset = member_offset;
	old_InsideAggregate = InsideAggregate;
	old_visibility = visibility;
	old_decldir = declaration_directory;
        old_declarator_class = declarator_class;
	storage = default_storage = member_storage;
	member_offset = 0;
	member_bit_offset = 0;
	InsideAggregate = (aggr == kw_class) ? in_class :
			  (aggr == kw_struct) ? in_struct : in_union;
	visibility = (InsideAggregate == in_class) ?
			    private_visibility : public_visibility;
	declaration_directory = NULL;
	declarator_class = classdef;


	/*** Base classes? ***/
	if (tok.en == colon) {
            visibility_enum orig_visibility = visibility;
	    do {
            	visibility = orig_visibility;
                Gobble(tok.en);
                if (tok.en == kw_public)
                    visibility = public_visibility, Gobble(tok.en);
                else if (tok.en == kw_private)
                    visibility = private_visibility, Gobble(tok.en);
                else if (tok.en == kw_protected)
                    visibility = protected_visibility, Gobble(tok.en);
                if (tok.en != identifier) {
                    ErrorParse(tok.src, "Bad base class declaration");
                    return NULL;
                }
                obj = ResolveTok(declarator_class);
                if (obj == NULL or obj->storage != typedef_storage or
                        *obj->type != tp_class) {
                    ErrorParse(tok.src, "Base class: %s is not a class.", tok.buf);
                    return NULL;
                }
                base_type = obj->type;
                Classdef* parentclass = *(Classdef**)(obj->type + 1);
                NameDeclare(declaration_directory, "", base_type, inherit_storage, NAME_CREATEMAKE);
                Gobble(identifier);
                void **v = (void**)((char**)&parentclass->VirtualFns + 1);
                Array_Concat(classdef->VirtualFns, v);
            } while (tok.en == comma or tok.en == colon);
            visibility = orig_visibility;
	}


	/*** Parse the members ***/
	Gobble(open_brace);
	until (tok.en == close_brace or tok.en == EOF) {

            /* Temporarily allow classdef to point to those members already */
            /* parsed.  This is necessary for resolving symbols defined in  */
            /* the current class or parent classes. */
            classdef->member = NamePartialMembers();

	    if (tok.en == kw_public) {
		visibility = public_visibility;
                GOBBLE_COLON:
                src = tok.src + strlen(tok.buf);
		NextToken();
                if (tok.en != colon)
                    ErrorParse(src, "You need a colon ':' after public/private/protected");
		NextToken();
                continue;
	    }
	    else if (tok.en == kw_private) {
		visibility = private_visibility;
                goto GOBBLE_COLON;
	    }
	    else if (tok.en == kw_protected) {
		visibility = protected_visibility;
                goto GOBBLE_COLON;
	    }
            else if (tok.en == kw_friend) {
            	ParseFriend(classdef);
            	continue;
            }
            else if (tok.en == op_sing('~')) {
            	str src=tok.src;
            	NextToken();
                if (tok.en != identifier or not streq(tok.buf, name)) {
                    ErrorParse(src, "If you're trying to declare a destructor "
                    	"here, use:  \"~%s\".  Otherwise, what's the '~' doing?",
                        name);
                }
                NextToken();
                PushTokenBack(identifier, NULL, src);
                strcpy(tok.buf, "~");
                PushTokenBack(kw_void, NULL, src);
                strcpy(tok.buf, "void");
                /* and carry on: */
            }
	    ParseDeclaration();
	}
	Gobble(close_brace);
	if (tok.en == EOF)
	    ErrorParse(NULL, "You need a ';' after class declarations!");

	/* Establish the classdef (part II) */
	if (member_offset == 0)
	    member_offset = 1;	    // According to ANSI C++, you're allowed to declare empty classes,
				    // and they must have sizeof() == 1.
	classdef->size = member_offset;
	classdef->RoundupSize = Heap::RoundUp(member_offset) + 4;
	classdef->member = NameGrabList(scope);

	/* Restore the saved values. */
	default_storage = old_default_storage;
	storage = old_storage;
	member_offset = old_member_offset;
	    // NameDeclare() will add the size of the classdef to it.
	visibility = old_visibility;
	InsideAggregate = old_InsideAggregate;
	declaration_directory = old_decldir;
	declarator_class = old_declarator_class;

	/* If we have a class name, then we can process it immediately. */
	if (classdef->typedef_obj)
	    classdef = PostProcessClassdef(classdef, prevclass);
        else UnprocessedClassdef = classdef;

	/* Return the classdef */
	return classdef;
}


static Namedobj* ParseEnum(void)
{       uchar typebuf[] = { tp_enumerated, 0, 0, 0, 0 };
	Namedobj* tag_obj, *obj, *first_enum;
	char name[512];
	int next_value;

	/* Is there a tag? */
	if (tok.en == identifier) {
	    tag_obj = NameDeclare(curdir, tok.buf, typebuf, typedef_storage, NAME_CREATEMAKE);
	    SourceStash(tag_obj, NULL);
	    Gobble(tok.en);
	}
	else tag_obj = NULL;

	/* Setting up: */
	Gobble(open_brace);
	next_value = 0;
	first_enum = NULL;
	NameMarkScope();

	until (tok.en == close_brace) {
	    strcpy(name, tok.buf);
	    Gobble(identifier);

	    if (tok.en == op_sing('=')) {
		Gobble(tok.en);
		enum_value = ParseAndEvaluateIntExpr(yes);
	    }
	    else enum_value = next_value++;
	    obj = NameDeclare(default_storage == static_storage ? curdir : NULL,
            		name, (Type)typebuf, const_storage, NAME_CREATEMAKE);
	    if (first_enum == NULL) {
		first_enum = obj;
		AssgPtr(typebuf+1, &first_enum);
		AssgPtr(obj->type+1, &first_enum);
		if (tag_obj)
		    AssgPtr(tag_obj->type+1, &first_enum);
	    }
            if (default_storage == member_storage)
            	obj->owner = declarator_class;
            else if (default_storage == auto_storage)
            	obj->owner = NULL;
            else obj->owner = curdir;
	    if (tok.en == comma)
		Gobble(tok.en);
	}

	Gobble(close_brace);

	return first_enum;
}




/*-------------------- Post-Processing Classdefs: --------------------*/

static Type ReplacePointers(Type type, void *oldie, void *newie)
/* Patch the new value of a pointer. */
{	Classdef* classdef;
	int arity;

	do switch (*type++) {
	    case tp_class:
		classdef = *(Classdef**)type;
		if (classdef == oldie)
		    *(void**)type = newie;
		type += sizeof(void*);
		return type;

	    case tp_pointer:
	    case tp_reference:
	    case tp_const:
	    case tp_volatile:
	    case tp_dynarray:
		continue;

	    case tp_array:
		GetDimension(arity, type);  // macro
		continue;

	    case tp_function:
		arity = *type++;
		while (arity-- > 0) {
		    type = ReplacePointers(type, oldie,newie);
		}
		if (*type++ != tp_terminated)
		    assert(false);
		continue;

            case tp_enumerated:
                if (*(void**)type == oldie)
                    *(void**)type = newie;
            	return type + 4;

	    default:
		return type;
	} forever;
}


static void ReplaceEnumPointersInClass(Classdef* classdef,
		Namedobj *oldie, Namedobj* newbie)
{	Namedobj *obj;
	int i;

        for (obj=classdef->member; obj; obj=obj->next)
            ReplacePointers(obj->type, oldie, newbie);
        for (each_aeli(obj, (Namedobj**)classdef->Friends))
            ReplacePointers(obj->type, oldie, newbie);
}


struct PtrTable_node {
        int offset;
        Classdef* classdef;
        int arraysize;
        /* If it's a pointer member, then 'classdef==NULL'.
        If it's a class/struct/union member, then 'classdef != NULL'.
        */
};


static int ScoreTypeForLinkedlistedness(Type type, Classdef* thisclass,
                int depth)
{       Classdef* classdef;
        int score, maxscore;
        Namedobj *obj;

        while (*type == tp_const or *type == tp_volatile)
            type++;
        if (*type++ != tp_pointer)
            return 0;
        switch (*type++) {
            case tp_pointer:
                return ScoreTypeForLinkedlistedness(type, thisclass, depth+1);

            case tp_class:
                classdef = *(Classdef**)type;
                if (classdef == thisclass)
                    return 999; // A recursive link
                if (depth > 6)
                    return 2;
                maxscore = 0;
                for (obj=classdef->member; obj; obj=obj->next) {
                    score = ScoreTypeForLinkedlistedness(obj->type, thisclass, depth+1);
                    if (score > maxscore)
                        maxscore = score;
                }
                return maxscore + 1;

            case tp_array:
                type += sizeof(int);
                return ScoreTypeForLinkedlistedness(type, thisclass, depth+1) + 1;

            default:
                return 0;
        }
}


typedef struct funcblockheader_node {
	int debugheader_offset;
	int ptrlist_offset;
} *funcblockheader_type;


class Assembler {
        ins_type p, func;
        int allocd;

public:
        Assembler(int estimate) { 
                    p = func = (ins_type)malloc(anon_heap, allocd=estimate); }
        void CheckSize(int buffer) {        // only realloc at certain well-defined points.
            if (p + 28 > func + allocd) {
                ins_type newfunc;
                allocd *= 2;
                newfunc = (ins_type)realloc(anon_heap, func, allocd);
                p += newfunc - func;
                func = newfunc;
            }
        }
        int GrabLocation() { return p - func; }
        ins_type Finished(int *lenp) { assert(p <= func + allocd); 
                    *lenp = allocd;
                    p = func; func = NULL; return p; }
        ~Assembler() { assert(func == NULL); }

        void ADD_EAX(int n) {
                if (n == (char)n)
                    *p++ = 0x83, *p++ = 0xc0, *p++ = n;     
                else *p++ = 0x05, *(int*)p = n, p += sizeof(int);
        }
        void MOV_ECX(int n) {
                *p++ = 0xb9, *(int*)p = n, p += sizeof(int);
        }
        void PUSH_EAX() { *p++ = 0x50; }
        void PUSH_ECX() { *p++ = 0x51; }
        void POP_EAX() { *p++ = 0x58; }
        void POP_ECX() { *p++ = 0x59; }
        void CALL_indirect(void *fnp) {
                    *p++ = 0xFF, *p++ = 0x15, *(void**)p = fnp, p += sizeof(str);
        }
        void CALL_relative_deferred(void *fn) {
                    // It'll be a relative CALL, but we we'll put an absolute
                    // address in there for now until we know where the function
                    // will be placed.
                    *p++ = 0xe8, *(void**)p = fn, p += sizeof(str);
        }
        void LOOP(int LoopStart) {
                int delta = (func+LoopStart) - (p+2);
                assert(delta == (char)delta);
                *p++ = 0xe2;
                *p++ = delta;
        }
        void RET() {
                *p++ = 0xc3;    // RET
        }
        void NOP() {
                *p++ = 0x90;
        }
};


static PtrFinder_fn BuildPtrFinder(Classdef* classdef, PtrFinder_fn prevfn)
/* Build a pointer-finder function. The previous ptr-finder function    */
/* is stored in 'classdef->ptrfinder'.  If possible, store the new one  */
/* in the same place, otherwise free the old one - this will avoid us   */
/* having to recompile other pointer-finder functions.                  */
/*      Pointer-finder functions have the usual calling convention:     */
/* * Address of object passed in top-of-stack.                          */
{       PtrTable_node *Table=NULL, *ptel, pnode;
        int score, best_score, best_i, size, i;
        extern PtrProcessor_fn PtrProcessor;
        uchar *func,*p,*finalfunc;
        Classdef* subclass;
        Namedobj *obj;
        int func_size;
        int offset;
        Type type;

        /* Collect all the members that are pointers or include pointers: */
        best_score = 0;
	for (obj=classdef->member; obj; obj=obj->next) {
            type = obj->type;
            while (*type == tp_const or *type == tp_volatile)
                type++;
            if (*type == tp_pointer) {
                ptel = (PtrTable_node*)Array_Next(Table);
                ptel->classdef = NULL;
                ptel->arraysize = 0;
                SCORE:
                assert(obj->storage == member_storage
                        or obj->storage == inherit_storage);
                ptel->offset = ((FieldNamedobj*)obj)->offset;
                score = ScoreTypeForLinkedlistedness(type, classdef, 0);
                if (strchr(obj->name, 'x') or strchr(obj->name, 'X'))
                    score++;    // This help discriminate between 'next' ptrs
                    // and 'prev'.
                if (score > best_score)
                    best_score = score, best_i = Array_Size(Table)-1;
            }
            else if (*type == tp_class) {
                subclass = *(Classdef**)(type+1);
                if (subclass->ptrfinder) {
                    ptel = (PtrTable_node*)Array_Next(Table);
                    ptel->classdef = subclass;
                    ptel->arraysize = 0;
                    goto SCORE;
                }
            }
            else if (*type == tp_array) {
                size = 1;
                do {
                    type++;
                    size *= *(int*)type;
                    type += sizeof(int);
                    while (*type == tp_const)
                        type++;
                } while (*type == tp_array);
                if (*type == tp_pointer) {
                    ptel = (PtrTable_node*)Array_Next(Table);
                    ptel->classdef = NULL;
                    ptel->arraysize = size;
                    goto SCORE;
                }
                else if (*type == tp_class) {
                    subclass = *(Classdef**)(type+1);
                    if (subclass->ptrfinder != NULL) {
                        ptel = (PtrTable_node*)Array_Next(Table);
                        ptel->classdef = subclass;
                        ptel->arraysize = size;
                        goto SCORE;
                    }
                }
            }
        }

        /* Is there any pointer inside this class at all? */
        if (Table == NULL) {
            if (classdef->ptrfinder)
                free(NULL, classdef->ptrfinder);
            return NULL;
        }

        /* If there's a possible linked list link, shuffle it to last position: */
        if (best_score > 0 and best_i < Array_Size(Table)-1) {
            pnode = Table[best_i];
            Array_DelN(Table, best_i);
            Array_Add(Table, pnode);
        }

        /* Start generating code: */
        /*
        Note that the auto-generated ptr-finders only use 6 types of instructions:

        ADD EAX,<n>  | 83 C0 <1-byte n>    or   05 <4-byte n>
        JMP EDI      | FF E7
        CALL [var]   | FF 15
        MOV CX,<n>   | B9 <1-byte n>         or
        LOOP         | E2 <1-byte negative displacement>
        PUSH EAX     | 50
        PUSH ECX     | 51
        POP EAX      | 58
        POP ECX      | 59
        RET          | C3

        Note the absence of any stack-frame instructions.  Auto-generated
        ptr-finders don't use any stack frame.  
        */
        Assembler asmb(Array_Size(Table) * 8 + 48);
        offset = 0;
        for (each_oeli(ptel,Table)) {
            if (ptel->offset != offset) {
                asmb.ADD_EAX(ptel->offset - offset);
                offset = ptel->offset;
            }
            if (ptel->arraysize) {
                asmb.MOV_ECX(ptel->arraysize);
                int LoopStart = asmb.GrabLocation();
                asmb.PUSH_ECX();
                if (ptel->classdef == NULL) {
                    asmb.PUSH_EAX();
                    asmb.CALL_indirect(&PtrProcessor);
                    asmb.POP_EAX();
                    asmb.ADD_EAX(ptel->arraysize * 4);
                    offset += ptel->arraysize * 4;
                }
                else {
                    asmb.PUSH_EAX();
                    asmb.CALL_relative_deferred(ptel->classdef->ptrfinder);
                    asmb.POP_EAX();
                    asmb.ADD_EAX(ptel->classdef->size);
                    offset += ptel->classdef->size;
                }
                asmb.POP_ECX();
                asmb.LOOP(LoopStart);
            }
            else if (ptel->classdef) {
                asmb.PUSH_EAX();
                asmb.CALL_relative_deferred(ptel->classdef->ptrfinder);
                asmb.POP_EAX();
                asmb.ADD_EAX(ptel->classdef->size);
                offset += ptel->classdef->size;
            }
            else {
                asmb.PUSH_EAX();
                asmb.CALL_indirect(&PtrProcessor);
                asmb.POP_EAX();
                asmb.ADD_EAX(4);
                offset += 4;
            }
            asmb.CheckSize(56);
        }
        asmb.RET();

        /* Add the Barbados function-trailer: */
        for (i=0; i < 8; i++)
            asmb.NOP();

        /* Copy it into the final location: */
        func = asmb.Finished(&func_size);
        func_size += sizeof(funcblockheader_node);
        static uchar voidvoidfn[4] = { tp_function, 0, tp_terminated, tp_void };
        finalfunc = (ins_type)default_heap->realloc(prevfn,func_size,voidvoidfn);
        memcpy(finalfunc, func, func_size);
        for (p=finalfunc; p < finalfunc+func_size; ) {
            switch (*p) {
                case 0x83:
                case 0x80:
                case 0x8b:      p += 3; break;
                case 0x05:
                case 0xb9:      p += 5; break;
                case 0xc3:
                case 0x50:
                case 0x51:
                case 0x58:
                case 0x59:      p++;    break;
                case 0xff:      if (p[1] == 0x15)
                                    p += 6;
                                else assert(false);
                                break;
                case 0xe2:
                case 0x75:      p += 2; break;
                case 0xe8:
                case 0xe9:      p++; *(int*)p -= int(p+4); p += 4; break;
                case 0x0f:      p += 6; break;
                case 0x90:      goto BREAK;
                default:        assert(false);
            }
        }
        BREAK:
        free(anon_heap, func);

        /* Finalise it as a function in the Barbados format, i.e. will be */
        /* correctly interpreted by 'FindPtrsInFuncblock()'. */
        funcblockheader_node *header = (funcblockheader_node*)
                ((str)finalfunc + msize(finalfunc) - sizeof(funcblockheader_node));
        header->debugheader_offset = 0;
	header->ptrlist_offset = 0;

        return (PtrFinder_fn)finalfunc;
}


static Classdef* PostProcessClassdef(Classdef* classdef,
		Classdef* prevclass)
/* The task of this routine is to:
	1. Compact the distributed 'classdef' into a compact 'newclass'
	2. Free the objects in the distributed representation
	3. Compare it with 'prevclass' (if any) to (a) determine if
	   the class has changed, and (b) recover any makenodes and
	   (c) transfer virtual functions and static objects from the
	   old to the new.
        4. Construct the 'ptrfinder' fn.

'classdef' is the newly compiled classdef, 'newclass' is the
compacted class that we will return, and 'prevclass' is the previous
version of this class (if any) .

We hold off running this function until we have a typedef_obj.
Once we have a typedef_obj (owner Namedobj), we can look up to
find a previous version of the class and thereby do step 3.
*/
{	Namedobj *oldobj, *newobj, **newobjp;
	Classdef* newclass, *memberclass;
        PtrFinder_fn prevptrfinder;
        funcblock_type *VirtualFns;
	int i, def_len, len;
        int VirtualFns_idx;
	Namedobj *obj;
	bool Changed;
	str d;

	/* How many bytes do we need for the classdef? */
	def_len = sizeof(Classdef);
	def_len += Array_Size(classdef->VirtualFns) * sizeof(void*);
	for (obj=classdef->member; obj; obj=obj->next) {
	    len = SizeofObj(obj->name, obj->type, obj->storage);
	    WordAlign(len);
	    def_len += len;
	}

	/* Has it changed since the previous version? */
	Changed = no;
	if (prevclass) {
	    if (classdef->size != prevclass->size)
		Changed = yes;
	    for (oldobj=prevclass->member, newobj=classdef->member;
		oldobj;
		oldobj=oldobj->next, newobj=newobj ? newobj->next : NULL) {
		if (newobj == NULL)
		    Changed = yes;
		else if (not Changed) {
		    if (not streq(oldobj->name, newobj->name)
			or not TypeEqual(oldobj->type, newobj->type)
			or oldobj->visibility != newobj->visibility
			or oldobj->storage != newobj->storage)
			Changed = yes;
                    else {
                        if (oldobj->storage == member_storage and
                                ((FieldNamedobj*)oldobj)->offset !=
                                ((FieldNamedobj*)newobj)->offset)
                            Changed = yes;
                        else if (oldobj->storage == static_storage and
                                ((StaticNamedobj*)oldobj)->location !=
                                ((StaticNamedobj*)newobj)->location)
                            Changed = yes;
                        else if (oldobj->storage == virtual_fn and
                                ((FunctionNamedobj*)oldobj)->u.virtualfn_idx !=
                                ((FunctionNamedobj*)newobj)->u.virtualfn_idx)
                            Changed = yes;
                    }
		}
		// While we're at it, transfer member functions and make nodes
		// from the previous version to the new version.
		if (oldobj->storage == member_fn
			and newobj and newobj->storage == member_fn
			and streq(oldobj->name, newobj->name)) {
                    if (((FunctionNamedobj*)newobj)->u.fn == NULL)
		    	((FunctionNamedobj*)newobj)->u.fn =
                                ((FunctionNamedobj*)oldobj)->u.fn;
		    ((FunctionNamedobj*)oldobj)->u.fn = NULL;
                    if (oldobj->make) {
                        if (newobj->make)
                            MakeDelete(newobj->make);  // We'd rather get the make
                            // node from the previous version than the newly created
                            // one.
		        newobj->make = oldobj->make;
                        newobj->make->obj = newobj;
                        //newobj->make->source = oldobj->make->source;
		        oldobj->make = NULL;
                    }
		}
                if (oldobj->storage == virtual_fn) {
                    int idx = ((FunctionNamedobj*)oldobj)->u.virtualfn_idx;
                    void* prevfn = ClassdefVirtualFn(prevclass, idx);
                    if (prevfn == PureVirtual)
                        ;	// No need to transfer the "= 0" marker
                    else if (idx < Array_Size(classdef->VirtualFns))
                        classdef->VirtualFns[idx] = prevfn;
                        // Processed classes have a different way of representing
                        // the VirtualFns array, remember.
                    else free(NULL, prevfn);
                }
	    }
	}

	/* Create newclass: */
	if (prevclass and def_len <= prevclass->def_len
			and classdef->RoundupSize == prevclass->RoundupSize)
	    newclass = prevclass;
	    // We use the old memory if (a) it's big enough, and (b) it won't
	    // corrupt the heap by changing the RoundupSize.
	else newclass = (Classdef*)default_heap->NewPlus(classdef_typstr,
			def_len-sizeof(Classdef));

	/* Copy the header from *classdef: */
        prevptrfinder = prevclass ? prevclass->ptrfinder : NULL;
	*newclass = *classdef;
	newclass->signature = HELLO_BABE;
	newclass->myself = newclass;
	newclass->def_len = def_len;
        newclass->ptrfinder = NULL;
	d = (char*)(newclass + 1);
	memcpy(d, classdef->VirtualFns, Array_Size(classdef->VirtualFns)*sizeof(void*));
	newclass->VirtualFns = (void**)Array_Size(classdef->VirtualFns);
        VirtualFns = ((funcblock_type*)(&newclass->VirtualFns+1));
        VirtualFns_idx = Array_Size(classdef->VirtualFns);
        for (i=0; i < VirtualFns_idx; i++)
            VirtualFns[i] = (funcblock_type)classdef->VirtualFns[i];
	d += VirtualFns_idx * sizeof(void*);
	if (newclass->typedef_obj)	// Back-patch the new classdef to the typedef_obj
	    *(Classdef**)(newclass->typedef_obj->type + 1) = newclass;

	/* Copy the members into it: */
	newobjp = &newclass->member;
	for (oldobj=classdef->member; oldobj; oldobj=oldobj->next) {
	    newobj = *newobjp = (Namedobj*)d;
	    newobjp = &newobj->next;
	    len = SizeofObj(oldobj->name, oldobj->type, oldobj->storage);
	    memcpy(newobj, oldobj, len);
            if (oldobj->storage == member_fn)
                ((FunctionNamedobj*)oldobj)->u.fn = NULL;
                // To avoid unwanted 'free's just below.
	    newobj->name = oldobj->name + ((char*)newobj - (char*)oldobj);
	    newobj->type = oldobj->type + ((char*)newobj - (char*)oldobj);
	    newobj->owner = newclass;
	    if (oldobj->make) {
		newobj->make = oldobj->make;
                newobj->make->obj = newobj;
		MakeChanged(newobj->make, yes);
	    }
            if (*oldobj->type == tp_function)
                SourceStashRename(oldobj, newobj);
	    if ((memberclass=IsTypedefOwner(oldobj)) != NULL)
		memberclass->typedef_obj = newobj;
	    if (newobj->make != NULL)
		newobj->make->obj = newobj, oldobj->make = NULL;
	    WordAlign(len);
	    d += len;
	}

        /* Update the pointer-finder: */
        newclass->ptrfinder = BuildPtrFinder(newclass, prevptrfinder);
        /*extern void DisassembleFuncToCout(machinecode_type ins);
        if (not PredefinitionPhase)
            DisassembleFuncToCout(newclass->ptrfinder);*/

	/* Now free the distributed representation: */
	while (classdef->member) {
	    obj = classdef->member;
	    classdef->member = obj->next;
	    DestroyObj(obj);
	    free(NULL, obj);
	}
	free(NULL, classdef);

	/* Check some details: */
	for (obj=newclass->member; obj; obj=obj->next) {

	    /* Update any recursive references to this class: */
	    ReplacePointers(obj->type, classdef, newclass);
	    // NB:- this is a special case. Mutually recursive classes
	    // are dealt with using the make algorithm just like recursive
	    // functions.

	    /* Check constructor details: */
	    if (newclass->typedef_obj and streq(obj->name, newclass->typedef_obj->name)) {
		if (obj->type[0] != tp_function)
		    ErrorParse(tok.src, "Bad constructor");
		else if (*TypeFunctionReturnType(obj->type) != tp_void)
                    ErrorParse(tok.src, "Constructors cannot have return types");
		else if (obj->storage == virtual_fn)
                    ErrorParse(tok.src, "Constructors cannot be virtual or static.");
	    }
            else if (streq(obj->name, "~")) {	// Destructors too...
		if (obj->type[0] != tp_function)
		    ErrorParse(tok.src, "Bad destructor - where's the () brackets?");
                else if (obj->type[1])
		    ErrorParse(tok.src, "Destructors are not allowed to have "
                    		"any parameters.");
            }

            /* Check for nonvirtuals overriding virtuals: */
	    if (obj->storage == member_fn and newclass->VirtualFns) {
		extern FunctionNamedobj* RecurseFindVirtualInBase(Classdef* classdef, Namedobj *obj);
            	Namedobj *basefn = RecurseFindVirtualInBase(newclass, obj);
                if (basefn and basefn->storage == virtual_fn) {
                    int virtualfn_idx=((FunctionNamedobj*)basefn)->u.virtualfn_idx;
                    assert(virtualfn_idx < VirtualFns_idx);
            	    VirtualFns[virtualfn_idx] = (uchar*)((FunctionNamedobj*)obj)->u.fn;
                }
            }

            /* Replace pointers with enums defined in this class: */
            if (*obj->type == tp_enumerated) {
            	Type type = obj->type+1;
                Namedobj *prevfirstenum = *(Namedobj**)(obj->type+1);
                if (prevfirstenum->owner == NULL or prevfirstenum->owner == classdef) {
		    ReplaceEnumPointersInClass(newclass, prevfirstenum, obj);
                }
            }
            else if (*obj->type == tp_function) {
                /* Replace the back-pointer in the funcblock's debug header: */
                funcblock_type funcblock;
                if (obj->storage == virtual_fn)
                    funcblock = VirtualFns[((FunctionNamedobj*)obj)->u.virtualfn_idx];
                else funcblock = (uchar*)((FunctionNamedobj*)obj)->u.fn;
                if (funcblock)
                    FuncblockAdjustObj(funcblock, obj);
            }
	}

        /* Is it an abstract class, or not? */
        for (i=0; i < VirtualFns_idx; i++) {
            if (VirtualFns[i] == (funcblock_type)PureVirtual) {
            	newclass->RoundupSize = -1;	// It's an abstract class.
                // No objects of this type are allowed to be declared.
                break;
            }
        }

        /* Update friend functions: */
        if (classdef->Friends == NULL)
            newclass->Friends = NULL;
        else {
            Namedobj *friendfn;
            newclass->Friends = classdef->Friends;
            classdef->Friends = NULL;
            for (each_aeli(friendfn, (Namedobj**)newclass->Friends)) {
            	if (((Classdef*)friendfn)->ValidClassdef())
                    continue;	// It's a friend class
                else ReplacePointers(friendfn->type, classdef, newclass);
            }
        }

	/* Update the make information and source-code. */
	obj = newclass->typedef_obj;
	if (obj and obj->make) {
	    MakeChanged(obj->make, Changed);
	    SourceStash(obj, NULL);
	    if (Changed)
		CurdirInterfaceChange(no);
	    /* If there are other declarators following, they */
	    /* will all depend on this object.  */
	    MakeDepends(obj->make);
	}

	/* Echo the class: */
	extern void OutputClassdef(Ostream* cout, Classdef* classdef);
	if (CompilerInvoker == 'I')
	    OutputClassdef(cout, newclass);

	/* Return 'newclass': */
	return newclass;
}




/*---------------- Macros and the name-space: --------------*/

interface void MacroEnter(str name, int no_params, str value)
/* Enter the macro with this name & # of parameters into the database. */
{       static char null_type[] = { tp_void };
	Namedobj* obj;
	int len;
	str val;


	len = strlen(value) + 2;
	if (not InsideCompiler)
	    MakeEntityReset();


	/* Do we have an old macro object to replace? */
        if (tok.curdir_obj)
            obj = tok.curdir_obj;
        else obj = NULL;
	if (default_storage == static_storage and
		obj and obj->storage == macro_storage) {
	    val = ((MacroNamedobj*)obj)->macro;
	    assert(val != NULL);
	    if (*val != no_params or strcmp(val+1, value) != 0)
		if (obj->make) {
		    MakeChanged(obj->make, yes);
		    /* This code is naughty.  We need to do for macros */
		    /* some of the things we do for other named objects, */
		    /* such as link in the source LGO. */
		    SourceLinkSource(curconim,yes);
		    CurdirInterfaceChange(no);
		}
	    val = (str)realloc(default_heap, val, len);
	}
	else {
	    /* Otherwise declare a new one: */
	    InsideCompiler++;
	    obj = NameDeclare(curdir, name, (Type)null_type, macro_storage, NAME_CREATEMAKE);
	    InsideCompiler--;
	    if (obj == NULL)
		return;
	    val = (str)default_heap->malloc(len);
	}


	/* Set it up with this str: */
	((MacroNamedobj*)obj)->macro = val;
	*val++ = no_params;
	strcpy(val, value);

	/* Store its source: */
	if (not PredefinitionPhase)
	    SourceStash(obj, NULL);
}


interface void MacroDelete(str name)
/* 'undef' this macro. */
{	Namedobj* obj;

	obj = curdir->Find(name);
	if (obj and obj->storage == macro_storage)
	    NameDelete(curdir, obj);
}


interface void MacroUsed(Namedobj* obj)
/* Register the fact that this macro has been used in the */
/* current compilation. */
{
	assert(obj != NULL);
	if (obj->make != NULL)
	    MakeDepends(obj->make);
}





/*------------------- Misc functions: -----------------*/

static Namedobj* FindMemberFunction(Classdef* classdef, str name, Type type)
/* Search for a member function matching (h,type) */
/* in the set of members of 'cl'. */
{       Namedobj* obj;

	for (obj=classdef->member; obj; obj=obj->next)
	    if (streq(obj->name, name) and TypeEqual(obj->type, type))
		return obj;
	return NULL;
}


interface str StorageToString(storage_enum storage)
{
	switch (storage) {
	    case static_storage:    return "static";
	    case local_static:	    return "localstatic";
	    case typedef_storage:   return "typedef";
	    case const_storage:	    return "const (enum)";
	    case macro_storage:	    return "macro";
	    case member_storage:    return "member";
	    case inherit_storage:   return "inherit";
	    case auto_storage:	    return "auto";
	    case parameter_storage: return "parameter";
	    case straight_fn:	    return "straight function";
	    case oneinstr_fn:	    return "one-instruction function";
	    case member_fn:	    return "member function";
	    default:		    return "unknown";
	}
}


static void PrintDeclaration(Namedobj* obj)
/* Display a message to the console providing feedback of */
/* the declaration of this object. */
{
	if (InsideAggregate or PredefinitionPhase)
	    return;	// Suppress output in these cases.

	InitStdio();
	if (obj->name == NULL)
	    ;
	else if (declarator_class)
	    OstreamPrintf(cout, "  %s::%s : ", declarator_class->typedef_obj->name, obj->name);
	else OstreamPrintf(cout, "  %s : ", obj->name);
	OutputType(cout, obj->type);
	OstreamPuts(cout, ";\n");
}


interface str ObjDeclarationString(Namedobj* obj, char dest[], int sizeofdest)
/* Output this object as it might have been declared in C++ syntax. */
{	Ostream out;

        /* Actually, for now let's content ourselves with my pet notation. */
	out.s = out.buf = dest;
	out.buf_end = dest + sizeofdest;
	out.flush = NULL;
        OstreamPrintf(&out, "%s : ", obj->name);
	OutputType(&out, obj->type);
	*out.s = '\0';
	return dest;
}


