/*****************************************************\
** COMPEXPR.C:                                       **
**          The parser module sends information to this  **
** module which sets up expression trees.  Then this **
** module goes along the tree, 'decorating' it with  **
** type and instruction information.                 **
\*****************************************************/


#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include "barbados.h"
#include "memory.h"
#include "runlib.h"
#include "array.h"
#include "compiler.h"
#include "name.h"
#include "make.h"



/* Note:
	We build the IPR graph directly out of the parse-graph.
	The parser creates a tree of Expr*'s, and by adding
	extra info to this tree we get the IPR.
	    Adding the extra info is called 'Decorating'.  We
	need to assign a nonzero value to the 'type' field and
	the 'o' (and 'tp' field too if applicable).  An Expr*
	with these 3 fields assigned to is called 'decorated' and
	is ready to be passed to the code-generator.
	    We decorate as we go, i.e. we decorate each expression
	as it is parsed.  This is because C allows us to do this,
	(all attributes are formed bottom-up), and because it
	allows us to detect errors as soon as possible.
*/


interface str UndefinedFunctionCallSrc;
interface Namedobj* UndefinedFunctionCall;
	// Are there calls to undef'd funcs?

interface StaticNamedobj* two_dots_obj;
interface AutoNamedobj* this_obj;

interface void DecorateExpr(Expr* e);
static void DecorateExplicitCast(Expr* e);
static void DecorateMemberAccess(Expr* e);
static void DecorateElementAccess(Expr* e);
static void DecorateIdentifier(Expr* e);

static bool CanConvert(Expr* e, type_plus_ref tr, bool is_explicit);
int CanConvertBuiltIn(Expr* e, type_plus_ref tr);
Namedobj* CanConvertUserDefined(Expr* e, type_plus_ref tr);
Expr* DoUserDefinedCoerce(Expr* a, type_plus_ref b, Namedobj *coercer);
Namedobj* ClassFindOriginal(Namedobj* o);
void ArrayToPointer(Expr* e);



/*------- Setting up the fundamental types: --------*/

uchar err_typstr[] = { tp_error };
uchar void_typstr[] = { tp_void };
uchar bool_typstr[] = { tp_bool };
uchar char_typstr[] = { tp_char };
uchar short_typstr[] = { tp_short };
uchar int_typstr[] = { tp_int };
uchar long_typstr[] = { tp_long };
uchar float_typstr[] = { tp_float };
uchar double_typstr[] = { tp_double };
uchar int64_typstr[] = { tp_int64 };
uchar string_typstr[] = { tp_pointer, tp_char };
uchar typetype_typstr[] = { tp_class, 0, 0, 0, 0 };
uchar any_typstr[] = { tp_class, 0, 0, 0, 0 };
uchar ostream_typstr[] = { tp_class, 0, 0, 0, 0 };
uchar container_typstr[] = { tp_container };
uchar direc_typstr[] = { tp_class, 0, 0, 0, 0 };
uchar dirref_typstr[] = { tp_reference, tp_class, 0, 0, 0, 0 };
uchar namedobj_typstr[] = { tp_class, 0, 0, 0, 0 };
uchar staticnamedobj_typstr[] = { tp_class, 0, 0, 0, 0 };
uchar macronamedobj_typstr[] = { tp_class, 0, 0, 0, 0 };
uchar functionnamedobj_typstr[] = { tp_class, 0, 0, 0, 0 };
uchar ptrclassdef_typstr[] = { tp_pointer, tp_class, 0, 0, 0, 0 };
uchar *classdef_typstr = ptrclassdef_typstr + 1;
uchar containerheader_typstr[] = { tp_pointer, tp_class, 0, 0, 0, 0 };
uchar bitmap_typstr[] = { tp_class, 0, 0, 0, 0 };
uchar voidvoidptr_typstr[] = { tp_pointer, tp_pointer, tp_void };
uchar voidptr_typstr[] = { tp_pointer, tp_void };
uchar file_typstr[] = { tp_class, 0, 0, 0, 0 };
uchar tyex_typstr[] = { tp_class, 0, 0, 0, 0 };
uchar uchar_typstr[] = { tp_uchar };
uchar ushort_typstr[] = { tp_ushort };
uchar uint_typstr[] = { tp_uint };
uchar voidvoidfn_typstr[] = { tp_function, 0, tp_terminated, tp_void };
uchar virtualfntable_typstr[] = { tp_pointer, tp_pointer, tp_function, 0, tp_terminated, tp_void };
uchar ambiguous_typstr[] = { tp_void };

/* Setting up the basic type-refs: */
type_plus_ref CHR_VAL   = { char_typstr, 0 };
type_plus_ref SHO_VAL   = { short_typstr, 0 };
type_plus_ref INT_VAL   = { int_typstr, 0 };
type_plus_ref LNG_VAL   = { long_typstr, 0 };
type_plus_ref FLT_VAL   = { float_typstr, 0 };
type_plus_ref DUB_VAL   = { double_typstr, 0 };
type_plus_ref STR_VAL   = { string_typstr, 0 };
type_plus_ref BOO_VAL   = { bool_typstr, 0 };
type_plus_ref I64_VAL   = { int64_typstr, 0 };
type_plus_ref CHR_REF   = { char_typstr, 1 };
type_plus_ref SHO_REF   = { short_typstr, 1 };
type_plus_ref INT_REF   = { int_typstr, 1 };
type_plus_ref LNG_REF   = { long_typstr, 1 };
type_plus_ref FLT_REF   = { float_typstr, 1 };
type_plus_ref DUB_REF   = { double_typstr, 1 };
type_plus_ref I64_REF   = { int64_typstr, 1 };
type_plus_ref STR_REF   = { string_typstr, 1 };
type_plus_ref OST_REF   = { ostream_typstr, 1 };
type_plus_ref BOO_REF   = { bool_typstr, 1 };
type_plus_ref UCH_VAL   = { uchar_typstr, 0 };
type_plus_ref USH_VAL   = { ushort_typstr, 0 };
type_plus_ref UIN_VAL   = { uint_typstr, 0 };
type_plus_ref UCH_REF   = { uchar_typstr, 1 };
type_plus_ref USH_REF   = { ushort_typstr, 1 };
type_plus_ref UIN_REF   = { uint_typstr, 1 };
type_plus_ref VPT_VAL   = { voidptr_typstr, 0 };

char print_fn_type[] = { tp_function, 1, tp_reference, tp_class, 0,0,0,0, tp_terminated, 
	    /* ret-val= */ tp_reference, tp_class, 0,0,0,0, };





/*----------------- Misc functions: ----------------*/

static void* Dynarray(void)
{
	assert(false);
	return NULL;
}


static int DynarrayLen(void* A)
{
	return 0;
}


static void* any_project(Any* a)
{
	return NULL;
}







/*----------------- Decorating support routines: ----------------*/

static tp_enum TypeToTp(Type type)
/* Convert this type to a tp by taking the first byte, but */
/* converting tp_long's to tp_int's etc.: */
{       tp_enum tp;

        tp = (tp_enum)*type;
        if (tp == tp_long)
            return tp_int;
        else if (tp == tp_ulong)
            return tp_int;
        else if (tp == tp_const or tp == tp_volatile)
            return TypeToTp(++type);
        else return tp;
}


static void ExprFunctionCall(Expr* e, Type return_type)
/* You can have the operator syntax that maps to CALL instructions, */
/* and you can have function syntax that maps to single Pentium     */
/* instructions.  Here we insert the structure for a CALL           */
/* instruction in something that might have the operator syntax. */
{       exprf_type ef;

	if (e->opr != op_funccall) {
	    ef = (exprf_type)qmalloc(sizeof(struct exprf_node) + 2 * sizeof(Expr*));
	    ef->func = NULL;
	    ef->owner = NULL;
	    ef->arity = 0;
	    if (e->left)
		ef->param[ef->arity++] = e->left;
	    if (e->right)
		ef->param[ef->arity++] = e->right;
	    e->u.func = ef;
	}
	e->o = oFUNC;
	e->ref = 0;
	e->tp = TypeToTp(return_type);
	while (*return_type == tp_reference)
	    return_type++, e->ref++;
	e->type = return_type;
}



interface Expr* ExprDeref(Expr* e)
/* Generate a dereference for this expression, to convert a */
/* reference to a value. */
{
	assert(e->ref > 0);
	e = NewExpr(op_sing('*'), e, NULL);
	e->type = e->left->type;
	e->ref = e->left->ref - 1;
	e->o = oDER;
	e->tp = (e->ref >= 1) ? tp_pointer : TypeToTp(e->type);
	e->src = e->left->src;
	if (e->tp == tp_class and TypeSize(e->type) <= 4)
	    e->tp = tp_int;
	return e;
}


static Expr* ExprWordDeref(Expr* e)
/* Generate a dereference of a pointer */
/* which points to at least a word. */
{
	e = NewExpr(op_unry('*'), e, NULL);
	e->o = oDER;
	e->ref--;
        e->tp = TypeToTp(e->type);
	if (e->tp == tp_char or e->tp == tp_short)
	    e->tp = tp_int;
	else if (e->tp == tp_uchar or e->tp == tp_ushort)
	    e->tp = tp_uint;
	return e;
}


interface Expr* ExprGetRvalue(Expr* e)
/* Convert this expression to an r-value. */
{
	assert(e->type != NULL);
	while (e->ref > 0)
	    e = ExprDeref(e);
	return e;
}


interface Expr* ExprGetLvalue(Expr* e)
/* Convert this expression to an l-value. */
{
	if (e->ref == 1)
	    return e;
	assert(e->type != NULL and e->ref < 10);
	if (*e->type == tp_error)
	    ;
	else if (e->ref == 0) {
	    ErrorType(e->src, "You need an l-value here");
	}
	else while (e->ref > 1)
	    e = ExprDeref(e);
	return e;
}


static Expr* ExprInsertConversion(Expr* e, tp_enum tp)
/* Insert a decorated conversion operation into a decorated expression. */
{	str orig_src=e->src;

	assert(e->type);
	e = NewExpr(kw_typedef, e, NULL);
	e->o = oCONVERT;
        if (tp == tp_long)
            tp = tp_int;
        else if (tp == tp_ulong)
            tp = tp_uint;
	e->tp = tp;
	e->type = double_typstr;    /* This may not be correct, but it doesn't */
			/* matter because here on in we look at the tp field only. */
	e->src = orig_src;
	return e;
}


static void DecoratePop(Expr* e)
/* Generate code to pop this value off the stack. */
{       
	if (*e->type == tp_error)
	    return;
	e->type = void_typstr;
        e->tp = tp_void;
	e->ref = 0;
}


static void GetFrameSize(exprf_type ef)
/* Set up the 'ef->stack_size' field for this function-call. */
/* The parameters must have been decorated already. */
{       Expr* e;
	int i;

	ef->stack_size = 0;
	for (i=0; i < ef->arity; i++) {
	    e = ef->param[i];
	    if (e->ref)
		ef->stack_size += 4;
	    else if (*e->type == tp_array)
		ef->stack_size += 4;
	    else ef->stack_size += TypeSizeWord(ef->param[i]->type);
	}
}


interface Expr* ExprDuplicate(Expr* e)
/* Generate a duplicate copy of this expression.  It's used for */
/* things like bit-field +=, -= etc.; and virtual function      */
/* accesses by expressions.  For optimal code, and also to	*/
/* avoid side-effects getting executed twice, it is important	*/
/* that we don't do a deep-copy of the expression but rather we */
/* store the value in a temporary variable and then access it.	*/
/*	We return the value which is calculated _after_ the	*/
/* side-effects of the value are computed, i.e. the original	*/
/* expression contains the real computations and we return an	*/
/* access to the temporary variable used to store the result.	*/
/*	So you must take care that 'e' is evaluated before the	*/
/* expression we return. The expression we return here assumes  */
/* that 'e' will already have been computed. */
{	AutoNamedobj *obj;
	Expr *r,*v;

	assert(e->type);
	if (e->opr == identifier or e->opr == int_const) {
	    r = NewExpr(e->opr, NULL, NULL);
	    *r = *e;
	    return r;
	}
	else if (e->opr == op_unry('*') and (e->left->opr == identifier or e->left->opr == int_const)) {
	    r = NewExpr(op_unry('*'), NewExpr(e->left->opr, NULL, NULL), NULL);
	    *r = *e;
	    *r->left = *e->left;
	    return r;
	}
	else {
	    v = NewExpr(e->opr, NULL, NULL);
	    *v = *e;
	    e->opr = op_sing('=');
	    e->type = v->type;
	    //e->ref = 0;   Let's keep the same ref it had originally.
	    e->right = v;
	    e->left = NewExprIdentifier(obj=CreateTemporaryVariable(e->type));
            e->src = v->src;
	    e->o = oASSIGN;
	    e->tp = v->tp;
	    r = NewExprIdentifier(obj);
	    r->type = v->type;
	    r->ref = 1;
	    r->o = oLocalVar;
	    r->tp = v->tp;
            r = ExprDeref(r);
            r->ref = e->ref;
            r->type = e->type;
	    return r;
	}
}







/*---------------- Type manipulations --------------*/

interface Type GetPointerToType(Type type)
/* Generate a Type by putting 'tp_pointer' in front of 'type'. */
{       Type r;
	int c;

	c = LengthOfTypeString(type);
	r = (Type)qmalloc(c+1);
	*r = tp_pointer;
	memcpy(r+1, type, c);
	return r;
}



interface Type GetArrayOfType(Type type, int dim)
/* Generate a type putting the array back in a pointer. */
{       Type r, t;
	int c;

	assert(*type == tp_pointer);
	c = LengthOfTypeString(type);
	r = t = (Type)qmalloc(c+sizeof(int)+1);
	*t++ = tp_array;
	PutDimension(dim, t);
	type++;
	memcpy(t, type, c);
	return r;
}


static void ArrayToPointer(Expr* e)
/* Convert this from an array expression to a pointer expression. */
{       Type type=e->type;
	int dimension;

	type++;
	GetDimension(dimension, type);
	e->type = GetPointerToType(type);
        e->tp = tp_pointer;
	e->ref--;
}


interface type_plus_ref GetTypePlusRef(Type type)
/* Return a type-plus-ref based on this type description. */
{       type_plus_ref tr;

	tr.type = type;
	tr.ref = 0;
	while (*tr.type == tp_reference)
	    tr.type++, tr.ref++;
	return tr;
}


static int ParamSize(type_plus_ref tr)
{       int size;

	if (tr.ref)
	    return sizeof(char*);
	size = TypeSize(tr.type);
	return max(size, sizeof(int));
}





/*---------------- Lifting the function veil: ---------------*/

/* In order to provide a programming environment good for beginners,	*/
/* we need to be able to report errors not just in syntax or language-	*/
/* level constructs, but also errors in usage of the standard library.	*/
/* A prime example is reporting errors in the usage of the 'printf()'	*/
/* family and the 'scanf()' family. We need to 'lift the library veil'.	*/
/* To do that, we have a hash-table of functions which have 'param	*/
/* checker functions which check the usage of the function at the 'expr'*/
/* level. */

static struct checkerpair_node {
	Namedobj* method;
	ParamChecker_fn CheckFn;
} *CheckerList;


static ParamChecker_fn FindParamCheckerFn(Namedobj* method)
{	struct checkerpair_node *p;
	int i;

	for (each_oeli(p, CheckerList)) {
	    if (p->method == method)
		return p->CheckFn;
	}
	return NULL;
}


static void SetParamCheckerFn(Namedobj* method, ParamChecker_fn CheckFn)
{	struct checkerpair_node *p;
	int i;

	for (each_oeli(p, CheckerList)) {
	    if (p->method == method) {
		p->CheckFn = CheckFn;
		return;
	    }
	}
	p = (checkerpair_node*)Array_Next(CheckerList);
	p->method = method;
	p->CheckFn = CheckFn;
}


static void PrintfChecker(exprf_type ef)
{	char buf[512], typebuf[512], ch;
	str fmt,s,expecting,t,d;
	Namedobj* obj;
	Expr* fmte;
	Type type;
	int arity,n,b;
        tp_enum tp;

	if (ef->func->o != oConstant)
	    return;
	obj = (Namedobj*)ef->func->u.idfier.resolve->obj;
	if (obj == NULL)
	    return;
	arity = TypeFunctionArity(obj->type);
	n = (arity&127)-1;
	fmte = ef->param[n++];
	if (fmte->o != oConstant)
	    return;
	if (fmte->right == localstatic_base)
	    fmt = (str)RetrieveLocalStatic((int)fmte->left);
	else fmt = (str)fmte->u.p;
	for (s=fmt; *s; ) {
	    if (*s++ == '%') {
		STILL_INSIDE:
		switch (*s++) {
		    case '%':	break;

		    case 'c':
		    case 'u':
		    case 'x':
		    case 'd':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
                                tp = ef->param[n++]->tp;
				if (not TpIsIntegral(tp)) {
				    expecting = "int";
				    goto WRONG_PARAM;
				}
				break;

		    case 's':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
				type = ef->param[n++]->type;
				if (*type == tp_pointer and
                                        (type[1] == tp_char or type[1] == tp_uchar))
                                    ;   // A string, it's ok
                                else if (*type == tp_array and
                                        (type[5] == tp_char or type[5] == tp_uchar))
                                    ;   // A char array, also ok
                                else {
				    expecting = "char* ie. string";
				    goto WRONG_PARAM;
				}
				break;

		    case 'p':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
                                tp = ef->param[n++]->tp;
				if (tp != tp_pointer) {
				    expecting = "pointer";
				    goto WRONG_PARAM;
				}
				break;

		    case 'e':
		    case 'g':
		    case 'f':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
                                tp = ef->param[n++]->tp;
				if (tp != tp_double) {
				    expecting = "double or float";
				    goto WRONG_PARAM;
				}
				break;

		    case 'l':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
                                tp = ef->param[n++]->tp;
				ch = *s++;
				if (ch == 'd' or ch == 'u' or ch == 'x') {
				    if (tp != tp_long and tp != tp_int
				    	and tp != tp_ulong and tp != tp_uint) {
					expecting = "long int";
					goto WRONG_PARAM;
				    }
				}
				else if (ch == 'f' or ch == 'g' or ch == 'e') {
				    if (tp != tp_double) {
					expecting = "double";
					goto WRONG_PARAM;
				    }
				}
				break;

		    case '0': case '1': case '2': case '3': case '4':
		    case '5': case '6': case '7': case '8': case '9':
		    case '-': case '.':
				goto STILL_INSIDE;

		    case '*':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
				tp = ef->param[n++]->tp;
				if (tp != tp_int and tp != tp_uint and tp != tp_char
					    and tp != tp_uchar) {
				    expecting = "int";
				    goto WRONG_PARAM;
				}
				goto STILL_INSIDE;

		    case '\0':	ErrorParse(fmte->src + (s-fmt),
					"The format string can't end in a single '%' - perhaps use two.");
				return;

		    default:	ErrorParse(fmte->src + (s-fmt),
					"Unknown character in format string - check the printf format string syntax.");
				return;

				RAN_OUT_OF_PARAMS:
				ErrorParse(fmte->src + (s-fmt),
					"The parameters run out before the format string runs out.");
				return;

				WRONG_PARAM:
				n--;
				t = ef->param[n]->src;
				if (t == NULL)
				    sprintf(buf, "number %d", n+1);
				else {
				    d = buf;
				    b = 0;
				    while (*t and *t != ',' and *t != ';' and d - buf < 10) {
					if (*t == '(')
					    b++;
					else if (*t == ')') {
					    if (--b < 0)
						break;
					}
					*d++ = *t++;
				    }
				    *d = '\0';
				}
				ErrorParse(fmte->src + (s-fmt),
					"The parameter \"%s\" of type \"%s\" is incompatible with '%c' (=%s).",
						    buf, TypeToString(ef->param[n]->type, typebuf, 50),
						    s[-1], expecting);
				return;
		}
	    }
	}

	if (n < ef->arity)
	    ErrorParse(ef->param[n]->src, "You have more parameters than this format string will display.");
}


static void ScanfChecker(exprf_type ef)
{	char buf[512], typebuf[512], ch;
	str fmt,s,expecting,t,d;
	Namedobj* obj;
	Expr* fmte;
	Type type;
	int arity,n,b;

	if (ef->func->o != oConstant)
	    return;
	obj = (Namedobj*)ef->func->u.idfier.resolve->obj;
	if (obj == NULL)
	    return;
	arity = TypeFunctionArity(obj->type);
	n = (arity&127)-1;
	fmte = ef->param[n++];
	if (fmte->o != oConstant)
	    return;
	if (fmte->right == localstatic_base)
	    fmt = (str)RetrieveLocalStatic((int)fmte->left);
	else fmt = (str)fmte->u.p;
	for (s=fmt; *s; ) {
	    if (*s++ == '%') {
		STILL_INSIDE:
		switch (*s++) {
		    case '%':	break;

		    case 'c':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
				type = ef->param[n++]->type;
				if (*type != tp_pointer or (type[1] != tp_char and type[1] != tp_uchar)) {
				    expecting = "address of char";
				    goto WRONG_PARAM;
				}
				break;

		    case 'u':
		    case 'x':
		    case 'i':
		    case 'o':
		    case 'n':
		    case 'd':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
				type = ef->param[n++]->type;
				if (*type != tp_pointer or (type[1] != tp_int and type[1] != tp_uint)) {
				    expecting = "address of int";
				    goto WRONG_PARAM;
				}
				break;

		    case '[':	while (*s != ']') {
				    if (*s == '\0')
					goto BAD_FORMAT;
				    s++;
				}
				/* and carry on: */

		    case 's':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
				type = ef->param[n++]->type;
				if (*type == tp_pointer and (type[1] == tp_char or type[1] == tp_uchar))
				    ;	    // okay
				else if (*type == tp_array and (type[5] == tp_char or type[5] == tp_uchar))
				    ;	    // okay
				else {
				    expecting = "char array or char*";
				    goto WRONG_PARAM;
				}
				break;

		    case 'p':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
				type = ef->param[n++]->type;
				if (*type != tp_pointer or type[1] != tp_pointer) {
				    expecting = "address of pointer";
				    goto WRONG_PARAM;
				}
				break;

		    case 'e':
		    case 'g':
		    case 'f':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
				type = ef->param[n++]->type;
				if (*type != tp_pointer or type[1] != tp_float) {
				    expecting = "address of float";
				    goto WRONG_PARAM;
				}
				break;

		    case 'l':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
				type = ef->param[n++]->type;
				if (*type != tp_pointer) {
				    expecting = "pointer";
				    goto WRONG_PARAM;
				}
				ch = *s++;
				if (ch == 'd' or ch == 'u' or ch == 'x') {
				    if (type[1] != tp_long) {
					expecting = "pointer to long int";
					goto WRONG_PARAM;
				    }
				}
				else if (ch == 'f' or ch == 'g' or ch == 'e') {
				    if (type[1] != tp_double) {
					expecting = "pointer to double";
					goto WRONG_PARAM;
				    }
				}
				break;

		    case '0': case '1': case '2': case '3': case '4': 
		    case '5': case '6': case '7': case '8': case '9':
		    case '-': case '.':
				goto STILL_INSIDE;

		    case '*':	if (n >= ef->arity)
				    goto RAN_OUT_OF_PARAMS;
				type = ef->param[n++]->type;
				if (*type != tp_int and *type != tp_uint and *type != tp_char
					    and *type != tp_uchar) {
				    expecting = "int";
				    goto WRONG_PARAM;
				}
				goto STILL_INSIDE;

		    case '\0':	ErrorParse(fmte->src + (s-fmt), 
					"The format string can't end in a single '%' - you're missing something.");
				return;

		    default:	BAD_FORMAT:
				ErrorParse(fmte->src + (s-fmt), 
					"Unknown character in format string - check the scanf format string syntax.");
				return;

				RAN_OUT_OF_PARAMS:
				ErrorParse(fmte->src + (s-fmt), 
					"The parameters run out before the format string runs out.");
				return;

				WRONG_PARAM:
				n--;
				t = ef->param[n]->src;
				if (t == NULL)
				    sprintf(buf, "number %d", n+1);
				else {
				    d = buf;
				    b = 0;
				    while (*t and *t != ',' and *t != ';' and d - buf < 10) {
					if (*t == '(')
					    b++;
					else if (*t == ')') {
					    if (--b < 0)
						break;
					}
					*d++ = *t++;
				    }
				    *d = '\0';
				}   
				ErrorParse(fmte->src + (s-fmt), 
					"The parameter \"%s\" of type \"%s\" is incompatible with '%c' (=%s).",
						    buf, TypeToString(ef->param[n]->type, typebuf, sizeof(typebuf)),
						    s[-1], expecting);
				return;
		}
	    }
	}

	if (n < ef->arity)
	    ErrorParse(ef->param[n]->src, "You have more parameters than this format string will scan.");
}


static void NullTerminatorChecker(exprf_type ef)
{	Expr* finale;

	if (ef->func->o != oConstant)
	    return;
	finale = ef->param[ef->arity-1];
	if (finale->o != oConstant or finale->u.p != NULL or finale->right != NULL)
	    ErrorParse(finale->src, "This function has a variable argument list and needs to terminate in a NULL.");
}


static void CheckFunctionParameters(exprf_type ef, FunctionNamedobj* obj)
{	ParamChecker_fn Checker;

	/* Find the checker function: */
	if (obj == NULL or obj->u.fn == NULL)
	    return;
	Checker = FindParamCheckerFn(obj);
	if (Checker == NULL)
	    return;

	/* Call it: */
	Checker(ef);
}


interface void SetParamCheckerFn(Namedobj* obj, char Ch)
/* Set this function to have a parameter-checker function. */
/* Ch: 'f'=printf format checker;  'N'=null-terminator.	   */
{
	if (Ch == 'f')
	    SetParamCheckerFn(obj, PrintfChecker);
	else if (Ch == 's')
	    SetParamCheckerFn(obj, ScanfChecker);
	else if (Ch == 'N')
	    SetParamCheckerFn(obj, NullTerminatorChecker);
	else assert(false);
}




/*======== Choosing methods: =======*/

static void MethodNative(str name, int arity, o_enum o, tp_enum tp,
	    type_plus_ref returntype, ...)
/* Create this function (method) in the generic function linked list.	*/
/* It generates expressions with IPR operators, which covers all the	*/
/* built-in functions and function calls. */
{	FunctionNamedobj* fobj;
	uchar typebuf[512];
	char newname[20];
	type_plus_ref tr;
	va_list ap;
	Type d;
	int i;


	/*** Construct the type: ***/
	d = typebuf;
	*d++ = tp_function;
	*d++ = arity;
	va_start(ap, returntype);
	for (i=0; i < arity; i++) {
	    tr = va_arg(ap, type_plus_ref);
	    while (tr.ref-- > 0)
		*d++ = tp_reference;
	    d = TypeAppend(d, tr.type);
	}
	*d++ = tp_terminated;
	while (returntype.ref-- > 0)
	    *d++ = tp_reference;
	d = TypeAppend(d, returntype.type);


	/*** For operators, we insert the word 'operator' here: ***/
	if (not isalpha(*name)) {	// (It makes the code look neater below).
	    strcpy(newname, "operator");
	    strcat(newname, name);
	    name = newname;
	}


	/*** Create a function object: ***/
	fobj = (FunctionNamedobj*)NameDeclare(
                        curdir, name, typebuf, oneinstr_fn,
                        NAME_OVERLOAD | NAME_CREATEMAKE);


	/*** Fill it out: ***/
	fobj->u.ipr.o = o;
	fobj->u.ipr.tp = tp;
        fobj->MinimumArity = arity;
	va_end(ap);
}


interface void MethodInit(void)
{
	/* Set up the methods for the operators: */
	MethodNative("+", 2, oADD,tp_double, DUB_VAL, DUB_VAL, DUB_VAL);
	MethodNative("+", 2, oADD,tp_long,   LNG_VAL, LNG_VAL, LNG_VAL);
	MethodNative("+", 2, oADD,tp_int,    INT_VAL, INT_VAL, INT_VAL);
	MethodNative("+", 2, oADD,tp_uint,   UIN_VAL, UIN_VAL, UIN_VAL);
	MethodNative("+", 2, oADD,tp_char,   CHR_VAL, CHR_VAL, CHR_VAL);
	MethodNative("+", 2, oADD,tp_uchar,  UCH_VAL, UCH_VAL, UCH_VAL);
	MethodNative("+", 2, oADD,tp_int64,  I64_VAL, I64_VAL, I64_VAL);
	MethodNative("-", 2, oSUB,tp_double, DUB_VAL, DUB_VAL, DUB_VAL);
	MethodNative("-", 2, oSUB,tp_long,   LNG_VAL, LNG_VAL, LNG_VAL);
	MethodNative("-", 2, oSUB,tp_int,    INT_VAL, INT_VAL, INT_VAL);
	MethodNative("-", 2, oSUB,tp_uint,   UIN_VAL, UIN_VAL, UIN_VAL);
	MethodNative("-", 2, oSUB,tp_char,   CHR_VAL, CHR_VAL, CHR_VAL);
	MethodNative("-", 2, oSUB,tp_uchar,  UCH_VAL, UCH_VAL, UCH_VAL);
	MethodNative("-", 2, oSUB,tp_int64,  I64_VAL, I64_VAL, I64_VAL);
	MethodNative("*", 2, oMUL,tp_double, DUB_VAL, DUB_VAL, DUB_VAL);
	MethodNative("*", 2, oMUL,tp_long,   LNG_VAL, LNG_VAL, LNG_VAL);
	MethodNative("*", 2, oMUL,tp_int,    INT_VAL, INT_VAL, INT_VAL);
	MethodNative("*", 2, oMUL,tp_int64,  I64_VAL, I64_VAL, I64_VAL);
	MethodNative("/", 2, oDIV,tp_double, DUB_VAL, DUB_VAL, DUB_VAL);
	MethodNative("/", 2, oDIV,tp_long,   LNG_VAL, LNG_VAL, LNG_VAL);
	MethodNative("/", 2, oDIV,tp_int,    INT_VAL, INT_VAL, INT_VAL);
	MethodNative("/", 2, oDIV,tp_int64,  I64_VAL, I64_VAL, I64_VAL);
	MethodNative("%", 2, oMOD,tp_long,   LNG_VAL, LNG_VAL, LNG_VAL);
	MethodNative("%", 2, oMOD,tp_int,    INT_VAL, INT_VAL, INT_VAL);
	MethodNative("-", 1, oNEG,tp_double, DUB_VAL, DUB_VAL);
	MethodNative("-", 1, oNEG,tp_long,   LNG_VAL, LNG_VAL);
	MethodNative("-", 1, oNEG,tp_int,    INT_VAL, INT_VAL);
	MethodNative("-", 1, oNEG,tp_char,   CHR_VAL, CHR_VAL);
	MethodNative("-", 1, oNEG,tp_short,  SHO_VAL, SHO_VAL);
	MethodNative("-", 1, oNEG,tp_float,  FLT_VAL, FLT_VAL);
	MethodNative("-", 1, oNEG,tp_int64,  I64_VAL, I64_VAL);
	MethodNative("~", 1, oNOT,tp_long,   LNG_VAL, LNG_VAL);
	MethodNative("~", 1, oNOT,tp_int,    INT_VAL, INT_VAL);
	MethodNative("~", 1, oNOT,tp_short,  SHO_VAL, SHO_VAL);
	MethodNative("~", 1, oNOT,tp_char,   CHR_VAL, CHR_VAL);
	MethodNative("!", 1, oNOTT,tp_bool,  BOO_VAL, BOO_VAL);
	MethodNative("<<",2, oSHL,tp_long,   LNG_VAL, LNG_VAL, LNG_VAL);
	MethodNative("<<",2, oSHL,tp_int,    INT_VAL, INT_VAL, INT_VAL);
	MethodNative(">>",2, oSHR,tp_long,   LNG_VAL, LNG_VAL, LNG_VAL);
	MethodNative(">>",2, oSHR,tp_int,    INT_VAL, INT_VAL, INT_VAL);
	MethodNative("&", 2, oAND,tp_long,   LNG_VAL, LNG_VAL, LNG_VAL);
	MethodNative("&", 2, oAND,tp_int,    INT_VAL, INT_VAL, INT_VAL);
	MethodNative("|", 2, oOR,tp_long,    LNG_VAL, LNG_VAL, LNG_VAL);
	MethodNative("|", 2, oOR,tp_int,     INT_VAL, INT_VAL, INT_VAL);
	MethodNative("^", 2, oXOR,tp_long,   LNG_VAL, LNG_VAL, LNG_VAL);
	MethodNative("^", 2, oXOR,tp_int,    INT_VAL, INT_VAL, INT_VAL);
	MethodNative("&&",2, oANND,tp_int,   BOO_VAL, BOO_VAL, BOO_VAL);
	MethodNative("&&",2, oANND,tp_int,   BOO_VAL, INT_VAL, INT_VAL);
	MethodNative("||",2, oORR,tp_int,    BOO_VAL, BOO_VAL, BOO_VAL);
	MethodNative("||",2, oORR,tp_int,    BOO_VAL, INT_VAL, INT_VAL);

	MethodNative(">", 2, oGT,tp_int,     BOO_VAL, DUB_VAL, DUB_VAL);
	MethodNative(">", 2, oGT,tp_int,     BOO_VAL, LNG_VAL, LNG_VAL);
	MethodNative(">", 2, oGT,tp_int,     BOO_VAL, INT_VAL, INT_VAL);
	MethodNative(">", 2, oGT,tp_int,     BOO_VAL, UIN_VAL, UIN_VAL);
	MethodNative(">", 2, oGT,tp_int,     BOO_VAL, CHR_VAL, CHR_VAL);
	MethodNative(">", 2, oGT,tp_int,     BOO_VAL, UCH_VAL, UCH_VAL);
	MethodNative(">=",2, oGE,tp_int,     BOO_VAL, DUB_VAL, DUB_VAL);
	MethodNative(">=",2, oGE,tp_int,     BOO_VAL, LNG_VAL, LNG_VAL);
	MethodNative(">=",2, oGE,tp_int,     BOO_VAL, INT_VAL, INT_VAL);
	MethodNative(">=",2, oGE,tp_int,     BOO_VAL, UIN_VAL, UIN_VAL);
	MethodNative(">=",2, oGE,tp_int,     BOO_VAL, CHR_VAL, CHR_VAL);
	MethodNative(">=",2, oGE,tp_int,     BOO_VAL, UCH_VAL, UCH_VAL);
	MethodNative("<", 2, oLT,tp_int,     BOO_VAL, DUB_VAL, DUB_VAL);
	MethodNative("<", 2, oLT,tp_int,     BOO_VAL, LNG_VAL, LNG_VAL);
	MethodNative("<", 2, oLT,tp_int,     BOO_VAL, INT_VAL, INT_VAL);
	MethodNative("<", 2, oLT,tp_int,     BOO_VAL, UIN_VAL, UIN_VAL);
	MethodNative("<", 2, oLT,tp_int,     BOO_VAL, CHR_VAL, CHR_VAL);
	MethodNative("<", 2, oLT,tp_int,     BOO_VAL, UCH_VAL, UCH_VAL);
	MethodNative("<=",2, oLE,tp_int,     BOO_VAL, DUB_VAL, DUB_VAL);
	MethodNative("<=",2, oLE,tp_int,     BOO_VAL, LNG_VAL, LNG_VAL);
	MethodNative("<=",2, oLE,tp_int,     BOO_VAL, INT_VAL, INT_VAL);
	MethodNative("<=",2, oLE,tp_int,     BOO_VAL, UIN_VAL, UIN_VAL);
	MethodNative("<=",2, oLE,tp_int,     BOO_VAL, CHR_VAL, CHR_VAL);
	MethodNative("<=",2, oLE,tp_int,     BOO_VAL, UCH_VAL, UCH_VAL);
	MethodNative("==",2, oEQ,tp_int,     BOO_VAL, DUB_VAL, DUB_VAL);
	MethodNative("==",2, oEQ,tp_int,     BOO_VAL, LNG_VAL, LNG_VAL);
	MethodNative("==",2, oEQ,tp_int,     BOO_VAL, INT_VAL, INT_VAL);
	MethodNative("==",2, oEQ,tp_int,     BOO_VAL, UIN_VAL, UIN_VAL);
	MethodNative("==",2, oEQ,tp_int,     BOO_VAL, CHR_VAL, CHR_VAL);
	MethodNative("==",2, oEQ,tp_int,     BOO_VAL, UCH_VAL, UCH_VAL);
	MethodNative("!=",2, oNE,tp_int,     BOO_VAL, DUB_VAL, DUB_VAL);
	MethodNative("!=",2, oNE,tp_int,     BOO_VAL, LNG_VAL, LNG_VAL);
	MethodNative("!=",2, oNE,tp_int,     BOO_VAL, INT_VAL, INT_VAL);
	MethodNative("!=",2, oNE,tp_int,     BOO_VAL, UIN_VAL, UIN_VAL);
	MethodNative("!=",2, oNE,tp_int,     BOO_VAL, CHR_VAL, CHR_VAL);
	MethodNative("!=",2, oNE,tp_int,     BOO_VAL, UCH_VAL, UCH_VAL);

	MethodNative("=", 2, oASSIGN,tp_double, DUB_VAL, DUB_REF, DUB_VAL);
	MethodNative("=", 2, oASSIGN,tp_float,  FLT_VAL, FLT_REF, FLT_VAL);
	MethodNative("=", 2, oASSIGN,tp_long,   LNG_VAL, LNG_REF, LNG_VAL);
	MethodNative("=", 2, oASSIGN,tp_int,    INT_VAL, INT_REF, INT_VAL);
	MethodNative("=", 2, oASSIGN,tp_uint,   UIN_VAL, UIN_REF, UIN_VAL);
	MethodNative("=", 2, oASSIGN,tp_short,  SHO_VAL, SHO_REF, SHO_VAL);
	MethodNative("=", 2, oASSIGN,tp_ushort, USH_VAL, USH_REF, USH_VAL);
	MethodNative("=", 2, oASSIGN,tp_char,   CHR_VAL, CHR_REF, CHR_VAL);
	MethodNative("=", 2, oASSIGN,tp_uchar,  UCH_VAL, UCH_REF, UCH_VAL);
	MethodNative("=", 2, oASSIGN,tp_bool,   BOO_VAL, BOO_REF, BOO_VAL);
	MethodNative("+=",2, oAADD,tp_double,   DUB_VAL, DUB_REF, DUB_VAL);
	MethodNative("+=",2, oAADD,tp_float,    FLT_VAL, FLT_REF, FLT_VAL);
	MethodNative("+=",2, oAADD,tp_int,      INT_VAL, INT_REF, INT_VAL);
	MethodNative("+=",2, oAADD,tp_uint,     UIN_VAL, UIN_REF, INT_VAL);
	MethodNative("+=",2, oAADD,tp_long,     LNG_VAL, LNG_REF, LNG_VAL);
	MethodNative("+=",2, oAADD,tp_short,    SHO_VAL, SHO_REF, SHO_VAL);
	MethodNative("+=",2, oAADD,tp_char,     CHR_VAL, CHR_REF, CHR_VAL);
	MethodNative("-=",2, oASUB,tp_double,   DUB_VAL, DUB_REF, DUB_VAL);
	MethodNative("-=",2, oASUB,tp_float,    FLT_VAL, FLT_REF, FLT_VAL);
	MethodNative("-=",2, oASUB,tp_long,     LNG_VAL, LNG_REF, LNG_VAL);
	MethodNative("-=",2, oASUB,tp_int,      INT_VAL, INT_REF, INT_VAL);
	MethodNative("-=",2, oASUB,tp_uint,     UIN_VAL, UIN_REF, INT_VAL);
	MethodNative("-=",2, oASUB,tp_short,    SHO_VAL, SHO_REF, SHO_VAL);
	MethodNative("-=",2, oASUB,tp_char,     CHR_VAL, CHR_REF, CHR_VAL);
	MethodNative("*=",2, oAMUL,tp_double,   DUB_VAL, DUB_REF, DUB_VAL);
	MethodNative("*=",2, oAMUL,tp_float,    FLT_VAL, FLT_REF, FLT_VAL);
	MethodNative("*=",2, oAMUL,tp_long,     LNG_VAL, LNG_REF, LNG_VAL);
	MethodNative("*=",2, oAMUL,tp_int,      INT_VAL, INT_REF, INT_VAL);
	MethodNative("*=",2, oAMUL,tp_short,    SHO_VAL, SHO_REF, SHO_VAL);
	MethodNative("*=",2, oAMUL,tp_char,     CHR_VAL, CHR_REF, CHR_VAL);
	MethodNative("/=",2, oADIV,tp_double,   DUB_VAL, DUB_REF, DUB_VAL);
	MethodNative("/=",2, oADIV,tp_float,    FLT_VAL, FLT_REF, FLT_VAL);
	MethodNative("/=",2, oADIV,tp_long,     LNG_VAL, LNG_REF, LNG_VAL);
	MethodNative("/=",2, oADIV,tp_int,      INT_VAL, INT_REF, INT_VAL);
	MethodNative("/=",2, oADIV,tp_uint,     UIN_VAL, UIN_REF, UIN_VAL);
	MethodNative("/=",2, oADIV,tp_short,    SHO_VAL, SHO_REF, SHO_VAL);
	MethodNative("/=",2, oADIV,tp_char,     CHR_VAL, CHR_REF, CHR_VAL);
	MethodNative("%=",2, oAMOD,tp_long,     LNG_VAL, LNG_REF, LNG_VAL);
	MethodNative("%=",2, oAMOD,tp_int,      INT_VAL, INT_REF, INT_VAL);
	MethodNative("%=",2, oAMOD,tp_uint,     UIN_VAL, UIN_REF, INT_VAL);
	MethodNative("%=",2, oAMOD,tp_short,    SHO_VAL, SHO_REF, SHO_VAL);
	MethodNative("%=",2, oAMOD,tp_char,     CHR_VAL, CHR_REF, CHR_VAL);
	MethodNative("<<=",2,oASHL,tp_long,     LNG_VAL, LNG_REF, LNG_VAL);
	MethodNative("<<=",2,oASHL,tp_int,      INT_VAL, INT_REF, INT_VAL);
	MethodNative("<<=",2,oASHL,tp_short,    SHO_VAL, SHO_REF, SHO_VAL);
	MethodNative("<<=",2,oASHL,tp_char,     CHR_VAL, CHR_REF, CHR_VAL);
	MethodNative(">>=",2,oASHR,tp_long,     LNG_VAL, LNG_REF, LNG_VAL);
	MethodNative(">>=",2,oASHR,tp_int,      INT_VAL, INT_REF, INT_VAL);
	MethodNative(">>=",2,oASHR,tp_short,    SHO_VAL, SHO_REF, SHO_VAL);
	MethodNative(">>=",2,oASHR,tp_char,     CHR_VAL, CHR_REF, CHR_VAL);
	MethodNative("&=",2, oAAND,tp_long,     LNG_VAL, LNG_REF, LNG_VAL);
	MethodNative("&=",2, oAAND,tp_int,      INT_VAL, INT_REF, INT_VAL);
	MethodNative("&=",2, oAAND,tp_short,    SHO_VAL, SHO_REF, SHO_VAL);
	MethodNative("&=",2, oAAND,tp_char,     CHR_VAL, CHR_REF, CHR_VAL);
	MethodNative("|=",2, oAOR,tp_long,      LNG_VAL, LNG_REF, LNG_VAL);
	MethodNative("|=",2, oAOR,tp_int,       INT_VAL, INT_REF, INT_VAL);
	MethodNative("|=",2, oAOR,tp_short,     SHO_VAL, SHO_REF, SHO_VAL);
	MethodNative("|=",2, oAOR,tp_char,      CHR_VAL, CHR_REF, CHR_VAL);
	MethodNative("^=",2, oAXOR,tp_long,     LNG_VAL, LNG_REF, LNG_VAL);
	MethodNative("^=",2, oAXOR,tp_int,      INT_VAL, INT_REF, INT_VAL);
	MethodNative("^=",2, oAXOR,tp_short,    SHO_VAL, SHO_REF, SHO_VAL);
	MethodNative("^=",2, oAXOR,tp_char,     CHR_VAL, CHR_REF, CHR_VAL);

	MethodNative("sin", 1, oSIN, tp_double, DUB_VAL, DUB_VAL);
	MethodNative("cos", 1, oCOS, tp_double, DUB_VAL, DUB_VAL);
	MethodNative("tan", 1, oTAN, tp_double, DUB_VAL, DUB_VAL);
	MethodNative("atan",1, oATAN,tp_double, DUB_VAL, DUB_VAL);
	MethodNative("sqrt",1, oSQRT,tp_double, DUB_VAL, DUB_VAL);
}


interface void CheckFunctionReturningBigStruct(Expr* e)
/* If this expression is a oFUNC expression, and it returns */
/* a big struct, and its return-val reference (parameter 0) */
/* hasn't been filled out yet, then fill it out to be a     */
/* temporary. */
{       AutoNamedobj *obj;

	if (e->o != oFUNC)
            return;
        if (not TypeIsBigStruct(e->type))
            return;
        if (e->u.func->param[0] != NULL)
            return;
        obj = CreateTemporaryVariable(e->type);
	e->u.func->param[0] = NewExprIdentifier(obj);
}


static Expr* BigStructToLvalue(Expr* e)
/* We have an Expr* which evaluates to a big struct. It is */
/* needed by another expression which is expecting an l-value. */
/* Construct an expression with the required properties. */
{
	if (e->o == oASSIGN) {
            e = NewExpr(comma, e, ExprDuplicate(e->left));
            DecorateExpr(e);
            return e;
        }
        else if (e->o == oFUNC) {
            CheckFunctionReturningBigStruct(e);
            e = NewExpr(comma, e, ExprDuplicate(e->u.func->param[0]));
            DecorateExpr(e);
            return e;
        }
        else if (e->o == oCOMMA) {
            /* Now, in case the value is used in some further expression:
            Expr* X3;
            X3 = ExprDuplicate(e->right->left->left);
            e->right = NewExpr(comma, e->right, X3);
            e->right->src = "\1";
            DecorateExpr(e->right);
            e->ref = e->right->ref; */

            e->right = BigStructToLvalue(e->right);
            e->ref = 1;
        }
        else if (e->o == oNOP1)
            return BigStructToLvalue(e->left);
        else assert(false);
        return e;
}


static int ParameterMatchScore(Expr* param, Type type)
/* Does this parameter match this type? Returns 6=no, otherwise 1..5 is */
/* the quality of the match. To paraphrase Bjarne Stroustrup:           */
/* "...a series of criteria are tried in order:
[1] Exact match; that is, match using no or only trivial conversions
        (for example, array name to pointer, function name to pointer
        to function and T to const T)
[2] Match using promotions; that is, integral promotions (bool to int,
        char to int, short to int and their unsigned counterparts), and
        float to double
[3] Match using standard conversions (for example, int to double, double
        to int, double to long double, Derived* to Base*, T* to void*,
        int to unsigned int)
[4] Match using user-defined conversions
[5] Match using the ellipsis ... in a function declaration.
*/
{	type_plus_ref tr;
        int level;

        tr.type = type;
        tr.ref = 0;
        LOOP:
	while (*tr.type == tp_reference)
            tr.ref++, tr.type++;
        if (*tr.type == tp_const) {
            tr.type++;
            goto LOOP;
        }

        level = CanConvertBuiltIn(param, tr);
        if (level < 4)
            return level;

        if (CanConvertUserDefined(param, tr))
            return 4;

        return 6;
}


static FunctionNamedobj* ChooseBestMethod(Resolved_node *res,
		Expr* *params, int call_arity, str src, bool ReportError)
/* Try to find a method that fits the parameters given in 'e'.  */
/* (If we have just 1 function i.e. no overloads function, then */
/* the choice is obvious - we just check that the function      */
/* matches the parameters).  Otherwise it gets more complicated:*/
/* To quote Bjarne Stroustrup:  "...a best match is found for   */
/* each argument using the rules from 7.4 [the 5 levels]. A     */
/* function that is the best match for one argument and a better*/
/* than or equal match for all other arguments is called. If no */
/* such function exists, the call is rejected as ambiguous.     */
/*                                                              */
/*      If we find a suitable candidate, we simply return it:   */
/* we don't do any of the conversions required. */
{       FunctionNamedobj *best_obj, *fobj, **candidates;
        str *Matches, BestMatch;
        Namedobj* obj;
        int obj_arity;
        Type type;
	int i,p;

        /* If there are no matching functions, then the caller must catch this. */
        obj = res->obj;
        assert(obj);

        /* A short-cut if there's just a single candidate: */
	if (res->list == NULL) {
            type = obj->type;
            if (*type++ != tp_function) {
                if (ReportError)
                    ErrorType(src, "%s is not a function.", obj->name);
                return NULL;
            }
            fobj = (FunctionNamedobj*)obj;
            obj_arity = *type++;
            if (obj_arity & 128) {
                if (call_arity < fobj->MinimumArity) {
                    if (not ReportError)
                        return NULL;
                    ErrorType(src, "You need at least %d parameters, but you have %d",
                        fobj->MinimumArity, call_arity);
                    return NULL;
                }
            }
            else {
                if (call_arity < fobj->MinimumArity) {
                    if (not ReportError)
                        return NULL;
                    if (call_arity == 0 and IsConstructor(obj))
                    	ErrorType(src, "There is no default constructor for "
                        	"this type (i.e. you need to specify constructor "
                                "parameters)");
                    else
                        ErrorType(src, "You need %d parameters, but you have only %d",
                                fobj->MinimumArity, call_arity);
                    return NULL;
                }
                else if (call_arity > obj_arity) {
                    if (not ReportError)
                        return NULL;
                    ErrorType(src, "You need only %d parameters, but you have %d",
                        obj_arity, call_arity);
                    return NULL;
                }
            }
            obj_arity &= 127;
            if (call_arity < obj_arity)
                obj_arity = call_arity;
            for (p=0; p < obj_arity; p++) {
                char buf[512];
                if (ParameterMatchScore(params[p], type) >= 6) {
                    if (not ReportError)
                        return NULL;
                    ErrorType(params[p]->src, "Parameter %d doesn't match \"%s\"",
                                p+1, TypeToString(type, buf, sizeof(buf)));
                    return NULL;
                }
                type += LengthOfTypeString(type);
            }
            return fobj;
        }

        /* Compute each candidate's match vector: */
        candidates = (FunctionNamedobj**)res->list;
        Matches = NULL;
        for (each_aeli(fobj, candidates)) {
            Array_Add(Matches, NULL);
            if (fobj->storage == keyword_storage)
                continue;
            type = fobj->type;
            if (*type++ != tp_function)
                continue;
            obj_arity = *type++;
            if (call_arity < fobj->MinimumArity or
                (call_arity > obj_arity and (obj_arity&128)==0))
                continue;
            obj_arity &= 127;
            if (call_arity < obj_arity)
                obj_arity = call_arity;
            Matches[i] = (str)malloc(anon_heap, obj_arity);
            for (p=0; p < obj_arity; p++) {
                /* What is the match for this argument? */
                Matches[i][p] = ParameterMatchScore(params[p], type);
                if (Matches[i][p] >= 6) {
                    // One of the params completely fails to match.
                    free(anon_heap, Matches[i]);
                    Matches[i] = NULL;
                    break;
                }
                type += LengthOfTypeString(type);
            }
        }

        /* What are the best matches for each argument? */
        BestMatch = (str)malloc(anon_heap, call_arity);
        memset(BestMatch, 6, call_arity);
        for (each_aeli(fobj, candidates)) {
            if (Matches[i] == NULL)
                continue;
            obj_arity = min(call_arity, obj->type[1] & 127);
            for (p=0; p < obj_arity; p++) {
                if (BestMatch[p] > Matches[i][p])
                    BestMatch[p] = Matches[i][p];
            }
        }

        /* Look for a single best candidate: */
        best_obj = NULL;
        for (each_aeli(fobj, candidates)) {
            if (Matches[i] == NULL)
                continue;
            obj_arity = min(call_arity, obj->type[1] & 127);
            for (p=0; p < obj_arity; p++) {
                if (Matches[i][p] != BestMatch[p])
                    goto REJECT;
            }
            if (best_obj == NULL)
                best_obj = fobj;
            else {
                char buf1[512], buf2[512];
                ErrorType(src, "Ambiguous function call: your call matches "
                        "both:\t%s\t and:\t%s",
                        ObjDeclarationString(best_obj, buf1, sizeof(buf1)),
                        ObjDeclarationString(fobj, buf2, sizeof(buf2)));
                best_obj = NULL;
                goto FREE;
            }
            REJECT:;
        }
        if (best_obj == NULL and ReportError) {
            if (IsConstructor(candidates[0])) {
            	if (call_arity == 0)
                    ErrorType(src, "This class has no default constructor - "
                    	"i.e. you need to specify constructor parameters.");
            	else ErrorType(src, "None of the constructors match your call.");
            }
            else ErrorType(src, "None of the functions of this name match your call.");
        }

	/* Free our temporary buffers: */
	FREE:
	free(anon_heap, BestMatch);
	for (each_aeli(fobj, candidates)) {
	    if (Matches[i])
		free(anon_heap, Matches[i]);
	}
	Array_Free(Matches);
        res->obj = best_obj;
	return best_obj;
}


static FunctionNamedobj* MethodFind(str name, Expr* e, int arity, bool ReportError)
/* Find a function in the current name-space of this name that  */
/* that matches this function call/operator expression.         */
/* If nothing matches the name, or we don't find a suitable     */
/* function, then maybe report an error.  If we have an         */
/* ambiguous call, then definitely report an error. */
{	Namedobj* obj;

	obj = Resolve(name);
	if (obj == NULL) {
	    if (not ReportError)
		return NULL;
	    ErrorType(e->src, "There are no functions with the name '%s' "
		"in the current name-space.", name);
	    return NULL;
	}
	return ChooseBestMethod(&ResolveInfo,
		(e->opr == op_funccall) ? e->u.func->param : &e->left,
		arity, e->src, ReportError);
}


static Expr* DefArgToExpr(char* data, Type type, int size)
{       Expr* e;

        if (size >= 8)
            e = NewExprFloat(*(double*)data);
        else e = NewExprInt(*(int*)data);
        e->tp = (tp_enum)*type;
	e->type = type;
        return e;
}


static void MethodDecorate(FunctionNamedobj* obj, Expr* e, int arity)
/* Once we have identified a specific method for computing some */
/* expression, decorate it and its operands.    'e' is either 	*/
/* an operator binary expression or an 'op_funccall' expression.*/
/* If 'obj' is NULL, then we assume it's a call by pointer to 	*/
/* function. */
{       int frame_size, i, arity2;
	Type t, return_type;
	type_plus_ref tr;
        bool Ellipsis;
	exprf_type ef;
	Expr* *param;


	/*** Be aware of undefined functions. ***/
	if (obj and obj->u.fn == NULL and obj->storage != virtual_fn) {
	    if (UndefinedFunctionCall == NULL) {
		UndefinedFunctionCall = obj;	    // Take the first one only.
		UndefinedFunctionCallSrc = e->src;
	    }
	    // Don't raise it as an error here: that would make it impossible
	    // to define recursive functions.
	}


	/*** Set things up and convert the 'e->opr' to one of ***/
	/*** 3 possibilities, (which among other things will  ***/
	/*** determine the order things will be generated).   ***/
	if (e->opr == op_funccall) {
	    ef = e->u.func;
	    param = ef->param;
            assert(ef->func->type and ef->func->o != oUnknown);
            /*ef->func->o = oConstant;
            ef->func->type = obj->type;*/
	}
	else if (arity == 1 or arity == 2) {
	    param = &e->left;
	    ef = NULL;
	}
	else assert(false);


	/*** Process default parameters: ***/
	t = obj ? obj->type : ef->func->type + 1;
	assert(*t == tp_function);
	t++;
	frame_size = 0;
	arity2 = *t++;
        Ellipsis = (arity2 & 128);
        arity2 &= 127;
        if (arity < arity2 and ef) {
            char* data=obj->DefaultParameters;
            Type type=t;
            int size;

            NewEnlargedExprFn(e, arity2);
            ef = e->u.func;
            param = ef->param;
            for (int i=0; i < arity; i++)
                type += LengthOfTypeString(type);
            while (arity < arity2) {
                size = TypeSizeWord(type);
                param[arity++] = DefArgToExpr(data,type,size);
                data += size;
                type += LengthOfTypeString(type);
            }
        }


	/*** Process ellipsis parameters: ***/
	if (Ellipsis) {
	    for (i=arity2; i < arity; i++) {
		if (*param[i]->type == tp_array) {
		    /* Arrays have to get converted to pointers. */
		    param[i] = ExprGetRvalue(param[i]);
		    frame_size += sizeof(str);
		}
		else if (*param[i]->type == tp_float) {
		    /* Floats have to get converted to doubles. */
		    param[i] = ExprGetRvalue(param[i]);
		    param[i] = ExprInsertConversion(param[i], tp_double);
		    DecorateExpr(param[i]);
		    frame_size += sizeof(double);
		}
                else if (TpSize((tp_enum)*param[i]->type) < 4) {
		    /* char's, shorts and enum's get converted to int's. */
		    param[i] = ExprGetRvalue(param[i]);
		    param[i] = ExprInsertConversion(param[i], tp_int);
		    DecorateExpr(param[i]);
		    frame_size += sizeof(int);
                }
		else {
		    /* Ordinary types must at least be r-values. */
		    param[i] = ExprGetRvalue(param[i]);
		    frame_size += ParamSize(expr_to_tr(param[i]));
		}
	    }
	}


	/*** Check the number of parameters: ***/
	if (arity == arity2)
	    ;
	else if (arity < arity2) {
	    ErrorType(e->src, "Too few parameters to %s.", obj->name);
	    e->type = err_typstr;
	    return;
	}
	else if (arity > arity2 and not Ellipsis) {
	    ErrorType(e->src, "Too many parameters to %s.", obj->name);
	    e->type = err_typstr;
	    return;
	}



	/*** Do any necessary conversions, and get the frame size: ***/
	for (i=0; i < arity2; i++) {
	    tr.ref = 0;
	    while (*t == tp_reference)
		t++, tr.ref++;
	    tr.type = t;
	    param[i] = ExprConvert(param[i], tr);
	    /* Errors will get reported from inside ExprConvert(). */
            if (TypeIsBigStruct(param[i]->type)) {
            	if (param[i]->o == oFUNC) {
                    exprf_node *subef;
                    subef = param[i]->u.func;
                    assert(subef->param[0] == NULL);
                    subef->param[0] = PushSPPlusExpr(subef->stack_size-4, TypeSize(e->type));
                }
            }
	    frame_size += ParamSize(expr_to_tr(param[i]));
	    t += LengthOfTypeString(t);
	}
	assert(*t == tp_terminated);
	t++;


	/*** Check the function parameters if we're "lifting the function veil": ***/
	if (ef)
	    CheckFunctionParameters(ef, obj);


	/*** Deal with owners: ***/
	if (ef != NULL and ef->owner != NULL) {

	    /* Insert the owner onto the front of the parameter-list: */
	    assert(ef->owner->ref >= 1);
	    ef->owner = ExprGetLvalue(ef->owner);
	    memmove(param+1, param, arity*sizeof(*param));
	    param[0] = ef->owner;
	    frame_size += 4;
	    ef->arity++;
            arity++;
	}


	/*** Set up the return value: ***/
	return_type = t;
	e->ref = 0;
	e->type = return_type;
        if (TypeIsBigStruct(e->type)) {
	    /* If the function returns a struct, then we insert here a pointer */
            /* to the return value onto the front of the parameter-list: */
	    memmove(param+1, param, arity*sizeof(*param));
	    param[0] = NULL;	// It is the responsibility of the calling
            // expression to fill this out to something meaningful, such as
            // an l-value (in the case of assignment), a kPUSHSPPLUS (in the
            // case of a parameter) or the address of a temporary.
	    frame_size += 4;
	    ef->arity++;
            arity++;
        }


	/*** Prepare the tree: ***/
	if (obj and obj->storage == oneinstr_fn) {
	    /* Built-in operator: */
	    e->o = (o_enum)obj->u.ipr.o;
	    e->tp = (tp_enum)obj->u.ipr.tp;
            if (e->tp == tp_long)
                e->tp = tp_int;
	    if (e->opr == op_funccall) {
		e->left = param[0];
		e->right = param[1];
	    }
	    while (*return_type == tp_reference)
		return_type++, e->ref++;
	}
	else if (ef != NULL and ef->func->o != oConstant) {
	    /* A function-ptr call: */
	    assert(ef != NULL);
	    e->o = oFUNC;
	    e->tp = TypeToTp(e->type);
	    ef->stack_size = frame_size;
	    //assert(e->left->o == oConstant);
	    while (*return_type == tp_reference)
		return_type++, e->ref++;
	}
        else if (obj == NULL) {
            assert(false);
            // Why wasn't this identified as a function-ptr call?
        }
	else {
	    if (obj->make and obj->storage != virtual_fn) {
                assert(obj->make->obj == obj);
		MakeDepends(obj->make);
            }
	    /* if (g->obj->u.location == NULL), then we'll catch it in the Make algorithm. */
	    /* If we raise it as an error here, then we make it impossible to define     */
	    /* recursive functions. */
	    ExprFunctionCall(e, return_type);
	    ef = e->u.func;
	    ef->func = NewExprIdentifier(obj);
	    ef->func->o = oConstant;
	    ef->func->tp = tp_pointer;
	    ef->stack_size = frame_size;
	    e->tp = TypeToTp(e->type);
	}
}



static bool Method(str name, Expr* e, int arity, bool ReportError=no)
/* Looks for an appropriate method.  If it finds one, */
/* it applies the method (decorates the expression    */
/* with the appropriate conversions + instructions).  */
{	Namedobj* obj;

	/* When operators are used with class instances, we search first */
	/* for an operator??() member function. */
	if (e->opr != op_funccall and e->left and e->left->type and
		    *e->left->type == tp_class and e->left->ref > 0) {
	    Classdef* classdef = *(Classdef**)(e->left->type + 1);
	    Namedobj* obj;
	    obj = ResolveMember(classdef, name, public_visibility);
	    if (obj) {
		Expr *func_e, *newe;
		func_e = NewExprIdentifier(obj);
		func_e->src = e->src;
		assert(arity == 1 or arity == 2);
		newe = NewExprFn(func_e, &e->right, arity - 1);
		newe->u.func->owner = e->left;
		newe->u.func->func = func_e;
		newe->src = e->src;
		obj = ChooseBestMethod(&ResolveInfo,
                		newe->u.func->param, arity-1, e->src, no);
		if (obj) {
		    func_e->u.p = obj;
		    func_e->type = obj->type;
		    func_e->ref = 0;
		    func_e->o = oConstant;
		    func_e->tp = tp_pointer;
		    MethodDecorate((FunctionNamedobj*)obj, newe, arity-1);
		    e->opr = kw_typedef;
		    e->o = oNOP1;
		    e->type = newe->type;
		    e->ref = newe->ref;
		    e->left = newe;
		    e->right = NULL;
                    e->tp = TypeToTp(e->type);
		    return yes;
		}
	    }
	}

	/* Search the directory name-space: */
	obj = MethodFind(name, e, arity, ReportError);
        if (obj == NULL)
            return no;
	MethodDecorate((FunctionNamedobj*)obj, e, arity);
	return yes;
}









/*----------------- Decorating the tree: ----------------*/

/* Compiling expressions works as follows:      */
/* 1. The parser generates an expression tree.  */
/* 2. We 'decorate' the tree with type info and */
/*        instructions.                             */
/* 3. We walk the tree to linearise the code.   */

static void EnterTypeIntoExpression(Expr* a);


static Type qtypedup(Type type)
/* Duplicate this type on the 'q' heap and return the type. */
/* When local variables go out of scope, they and their     */
/* type-strings are freed, so we need to take copies of     */
/* those type-strings. */
{	Type r;
	int size;

	size = LengthOfTypeString(type);
	r = (Type)qmalloc(size);
	return (Type)memcpy(r, type, size);
}


static Expr* DecorateCreateThis(str src, bool for_function)
/* Create and decorate the 'this' object: */
{       Expr* e;

	if (this_obj == NULL) {
            ErrorType(src, for_function ?
                "You have no 'this' object to call this member function." :
                "You have no 'this' object to look up this member.");
	    e = ErrorExpr();
	    return e;
	}
	e = NewExprIdentifier(this_obj);
	e->tp = tp_pointer;
	e->u.p = this_obj;
	assert(*this_obj->type == tp_pointer);
	e->type = this_obj->type+1;
	e->ref = 2;     /* So it's already Decorated(). */
	e->o = oLocalVar;
        e->src = "this";
	return e;
}


interface Expr* ReturnRefExpr()
/* Create and decorate a return-reference expression.  This refers to */
/* the parameter "%returnref" which is automatically created in any   */
/* function which returns a struct > 4 bytes. */
{	AutoNamedobj* obj;
	Expr* e;

	obj = (AutoNamedobj*)NameLocalResolve("%returnref");
        if (obj == NULL) {
            assert(false);
            return ErrorExpr();
        }
        e = NewExpr(identifier, (Expr*)obj, NULL);
        assert(*obj->type == tp_reference);
        e->type = obj->type+1;
        e->ref = 2;
        e->o = oLocalVar;
        e->tp = tp_pointer;
        e->src = "%returnref%";
        return e;
}


static Type FunctionToPtrmemberfn(Type type, Namedobj* obj)
/* Convert this function type to a pointer-to-member-function type. */
{       Classdef* classdef;
        Type pmf, d;
        int len;

        len = LengthOfTypeString(type);
        if (*type++ != tp_function)
            return err_typstr;
        pmf = d = (Type)qmalloc(len + 6);
        *d++ = tp_ptrmemberfn;
        *d++ = *type++ + 1;             // Arity; add 1 for the 'this'
        *d++ = tp_reference;
        *d++ = tp_class;
        classdef = (Classdef*)obj->owner;
        assert(classdef->ValidClassdef());
        *(Classdef**)d = classdef;
        d += sizeof(Classdef*);
        memcpy(d, type, len - 2);
        return pmf;
}


static Namedobj* FindTypedefObj(Type type)
/* Find the typedef object corresponding to this type. */
{       Classdef* classdef;

	/* Get the 'classdef': */
	classdef = TypeToClassdef(type);
	if (classdef == NULL)
	    return NULL;
	else return classdef->typedef_obj;
}


static void DecorateIdentifier(Expr* e)
/* Generate the code for an appearance of a variable. */
/* Must deal with all the storage classes, etc.       */
/* If there's a function call to a generic function,  */
/* it parses that function call.                      */
{       Resolved_node *res;
        Namedobj *obj;

	res = e->u.idfier.resolve;
        obj = res->obj;

	switch (obj->storage) {
	    case oneinstr_fn:
		    e->o = oUnknown; //(o_enum)obj->u.ipr.k;
		    e->tp = (tp_enum)0; //obj->u.ipr.tp
		    e->ref = 0;
		    e->type = obj->type;
		    e->u.p = NULL;
		    break;

	    case const_storage:
		    assert(*obj->type != tp_class);
		    /* We only support single-datum constants. */
		    e->o = oConstant;
		    e->ref = 0;
		    e->type = qtypedup(obj->type);
		    e->tp = TypeToTp(e->type);
		    e->u.i = ((IntNamedobj*)obj)->constval;
		    MakeDepends(obj->make);
		    break;

	    case parameter_storage:
	    case auto_storage:
		    e->o = oLocalVar;
		    e->u.p = obj;
                    e->right = 0;
		    e->ref = (*obj->type == tp_array) ? 0 : 1;
		    e->type = qtypedup(obj->type);
		    e->tp = tp_pointer;
		    //e->u.i = local->location;
		    break;

	    case local_static:
		    e->o = oConstant;
		    e->ref = (*obj->type == tp_array) ? 0 : 1;
		    e->type = qtypedup(obj->type);
		    e->tp = tp_pointer;
		    e->u.p = ((StaticNamedobj*)obj)->location;
		    e->right = (Expr*)localstatic_base;
		    break;

	    case static_storage:
		    e->o = oConstant;
		    e->ref = (*obj->type == tp_array) ? 0 : 1;
		    e->type = obj->type;
		    e->u.p = ((StaticNamedobj*)obj)->location;
		    e->tp = tp_pointer;
		    MakeDepends(obj->make);
		    break;

	    case member_storage:
		    e->left = DecorateCreateThis(e->src, no);
		    e->opr = dot;
		    e->right = NewExprIdentifier(res);
                    e->right->src = e->src;
		    e->type = obj->type;
                    e->tp = TypeToTp(e->type);
		    DecorateMemberAccess(e);
		    break;

	    case member_fn:
                    if (this_obj == NULL) {
                        /* The only way it can be legal to refer to a member_fn */
                        /* without an owner object is if we're accessing */
                        /* a pointer to a member function. */
                        e->type = FunctionToPtrmemberfn(obj->type, obj);
		        e->o = oConstant;
		        e->ref = 0;
		        e->tp = tp_pointer;
		        e->u.p = ((FunctionNamedobj*)obj)->u.fn;
		        e->right = (Expr*)obj;
		        MakeDepends(obj->make);
                        break;
                    }
                    if (ResolveInfo.offset != 0)
                    	assert(false);	// not yet implemented
                    e = NewExpr(dot, DecorateCreateThis(e->src, yes), e);
		    DecorateMemberAccess(e);
		    break;

	    case straight_fn:
		    e->o = oConstant;
                    e->tp = tp_pointer;
		    e->ref = 0;
		    e->type = obj->type;
		    MakeDepends(obj->make);
		    e->u.p = ((FunctionNamedobj*)obj)->u.fn;
		    break;

	    case typedef_storage:
		    ErrorParse(e->src, "%s is a typedef.", obj->name);
		    e->type = err_typstr;
		    break;

	    case macro_storage:
		    ErrorParse(e->src, "%s is a macro.", obj->name);
		    e->type = err_typstr;
		    break;

	    default:assert(false);
	}

	EnterTypeIntoExpression(e);
}



static void EnterTypeIntoExpression(Expr* e)
/* Given an identifier or other l-value expression, process */
/* its type, reference for use in an expression. */
{
	while (*e->type == tp_reference)
	    e->type++, e->ref++;
}


interface Expr* NewMemberAddress(Namedobj *obj, str src)
/* 'obj' represents a member_storage or inherit_storage member  */
/* of the current object ("this").  Return a decorated expr	*/
/* representing the address of this member. */
{       Resolved_node Res;
	Expr* e;

        Res.list = NULL;
        Res.obj = obj;
        Res.offset = 0;
	e = NewExpr(dot, DecorateCreateThis(src, no),
			NewExprIdentifier(&Res));
        e->src = e->right->src = src;
        e->type = obj->type;
        e->tp = TypeToTp(e->type);
        e->ref = 1;
        DecorateMemberAccess(e);
        return e;
}


static void DecorateIndex(Expr* *ap, Expr* *bp, bool NeedLvalue)
/* Process this expression for use as an array dereference (a[b]) */
/* or for pointer arithmetic, with `type'.  Must scale it and     */
/* ensure that it is the correct size (size of index = size of    */
/* pointer).  If it's for an assignment operator, a must be an    */
/* l-value, otherwise an r-value. */
{       Expr *a=*ap, *b=*bp;
	int size;

	if (*a->type == tp_error or *b->type == tp_error)
	    return;

	/* Make 'b' an int or a long. */
	if (*b->type == tp_long)
	    ;
	else b = ExprConvert(b, INT_VAL);
	//assert(sizeof(str) == sizeof(long));

	/* Scale 'b': */
	size = TypeSize(a->type+1);
	if (size == 0) {
	    ErrorType(a->src, "Size == 0");
	    return;
	}
	else if (size == 1)
	    ;
	else {
	    b = NewExpr(op_sing('*'), b, NewExprInt(size));
	    DecorateExpr(b);
	}

	/* Prepare 'a' as a l-value or r-value: */
	if (NeedLvalue)
	    a = ExprGetLvalue(a);
	else
	    a = ExprGetRvalue(a);

	/* Return the expressions: */
	*ap = a;
	*bp = b;
}


static void DecorateTernary(Expr* e)
{       type_plus_ref tr;
	Expr *a,*b,*c;

	a = e->left;
	b = e->right->right;
	c = e->right->left;
	DecorateExpr(a);
	DecorateExpr(b);
	DecorateExpr(c);
	a = ExprConvert(a, BOO_VAL);
	if (TypeEqual(b->type, c->type)) {
	    while (b->ref > c->ref)
		b = ExprDeref(b);
	    while (c->ref > b->ref)
		c = ExprDeref(c);
	    e->type = b->type;
	    e->ref = b->ref;
	    e->tp = TypeToTp(e->type);
	}
	else {
	    if (CanConvert(b, tr=expr_to_tr(c), no))
		b = ExprConvert(b, tr);
	    else if (CanConvert(c, tr=expr_to_tr(b), no))
		c = ExprConvert(c, tr);
	    else if (CanConvert(b, tr=GetTypePlusRef(c->type), no))
		b = ExprConvert(b, tr);
	    else if (CanConvert(c, tr=GetTypePlusRef(b->type), no))
		c = ExprConvert(c, tr);
	    else ErrorType(e->src, "Incompatible values in ? :");
	    e->type = tr.type;
	    e->ref = tr.ref;
	    e->tp = TypeToTp(e->type);
	}
	e->left = a;
	e->right->right = b;
	e->right->left = c;
	e->o = oTERNARY;
}


static Expr* DefaultOutputter(Expr* ostr, Expr* value)
/* Combines these two expressions into an expression which */
/* calls PrintValue() for this value. */
{       Expr *P[4], *type_e, *size_e, *func_e, *e;
	Type type;
	int size;

	type = value->type;
	if (*type == tp_array) {
	    Type s;
	    int i, len;

	    /* Leave it as an l-value if it is one. */
	    len = LengthOfTypeString(value->type);
	    s = type = (Type)qmalloc(len + value->ref);
	    for (i=0; i < value->ref; i++)
		*s++ = tp_reference;
	    memcpy(s, value->type, len);
	    value->ref = 0;
	}
	size = TypeSizeWord(type);

	/* Marshall the 4 parameters: */
        extern Ostream* PrintAny(Ostream* ostr, int size, Type type, int value);
	func_e = NewExpr(string_const, (Expr*)PrintAny, NULL);
	ostr = ExprGetLvalue(ostr);
	value = ExprGetRvalue(value);
        if (value->o == oFUNC and TypeIsBigStruct(type)) {
            exprf_node *subef = value->u.func;
            if (subef->param[0] == NULL)
            	subef->param[0] = PushSPPlusExpr(subef->stack_size-4, TypeSize(value->type));
            else value = ExprGetRvalue(BigStructToLvalue(value));
        }
	size_e = NewExprInt(size);
	type_e = ExprCopyOfTypestring(type, yes);
	DecorateExpr(func_e);
	DecorateExpr(ostr);
	DecorateExpr(size_e);
	DecorateExpr(type_e);
	DecorateExpr(value);
	P[0] = ostr;
	P[1] = size_e;
	P[2] = type_e;
	P[3] = value;

	/* Create the function expression: */
	e = NewExprFn(func_e, P, 4);
	e->type = ostr->type;
	e->ref = ostr->ref;
	GetFrameSize(e->u.func);
	e->src = "<default outputter>";
	e->tp = e->ref ? tp_pointer : tp_class;

	return e;
}


static Expr* PrintMethod(Expr* value, Expr* ostr)
/* Attempt to generate an 'X.Print(ostream)' method for this value. */
{       Classdef* classdef;
	Namedobj* obj;
	Expr* e;
	Type type;

	type = value->type;
	if (*type++ != tp_class)
	    return NULL;                    // We need a class
	GetPtr(classdef, type);

	for (obj=classdef->member; obj; obj=obj->next)
	    if (streq(obj->name, "Print") and TypeEqual(obj->type, (Type)print_fn_type))
		break;
	if (obj == NULL)
	    return NULL;                    // No matching function found.
	if (((FunctionNamedobj*)obj)->u.fn == NULL)
	    return NULL;                    // It's been declared but not defined.

	if (value->ref == 0) {
	    /* Make this r-value into an l-value: */
	    return NULL;
	    /*e = NewExpr(op_funccall, value, ostr);
	    ExprMultipleInstr(e,
		PUSH_SP, sizeof(str),
		CALL, obj->u.location,
		POPDX, 2*sizeof(str)+classdef->size,
		NOP
	    );*/
	}
	else {
	    /* We're given an l-value: */
	    value = ExprGetLvalue(value);
	    e = NewExprFn(NewExprIdentifier(obj), &ostr, 1);
	    e->u.func->owner = value;
	    e->u.func->func->o = oConstant;
	    e->u.func->func->tp = tp_pointer;
	    e->u.func->func->type = voidptr_typstr;
            e->u.func->func->u.idfier.resolve =
            		(Resolved_node*)qmalloc(sizeof(Resolved_node));
            e->u.func->func->u.idfier.resolve->obj = obj;
            e->u.func->func->u.idfier.resolve->list = NULL;
            e->src = value->src;
	    DecorateExpr(e);
	}

	return e;
}


static Expr* NewPrintString(str s)
/* Create an expression for printing this str. */
{       Expr *e, *func_e, *s_e;

	func_e = NewExpr(string_const, (Expr*)CoutPuts, NULL);
	func_e->o = oConstant;
	func_e->tp = tp_pointer;
	s_e = NewExpr(string_const, (Expr*)s, NULL);
	s_e->o = oConstant;
	s_e->tp = tp_pointer;
	s_e->type = string_typstr;
	e = NewExprFn(func_e, &s_e, 1);
	e->type = void_typstr;
	e->ref = 0;
	e->o = oFUNC;
	GetFrameSize(e->u.func);
	e->tp = tp_void;
	return e;
}


static void DecorateOutput(Expr* e)
/* Generate expressions to output this value in the appropriate form. */
/* If the value is void, it must return.  Otherwise it must print the */
/* value in the format:  "\r = %s;\n", where the %s is replaced by a  */
/* str representation of the value.  The str representation is  */
/* given by the default outputter if (a) the value's type is char,    */
/* int, long or str, or (b) if there is no custom outputter.  The  */
/* str representation is given by the custom outputter if there is */
/* one and if the value is not char, int, long or str.  The reason */
/* for this is that we want strings, chars etc in the language-       */
/* compatible format, e.g. "hello\n" or '\\'. */
{       Expr *value, *prt_value, *ostr;
	bool success;
	Type type;

	/*** Get something to generate the value. ***/
	/* Leave it on the stack in r-value or      */
	/* l-value form in case a custom outputter  */
	/* requires an l-value.   Print the "\r = " */
	/* str after the value has been computed */
	/* but before it has been printed, (include */
	/* it in 'value'). */
	value = e->left;
	DecorateExpr(value);
	if (*value->type == tp_void) {      // If it's a void, do nothing else.
	    e->type = void_typstr;
            e->tp = tp_void;
	    e->ref = 0;
	    e->o = oNOP1;
	    e->left = value;
	    return;
	}

	/*** Get something to output the value: ***/
	type = value->type;
	ostr = NewExpr(ostr_const, (Expr*)cout, NULL);
	ostr = NewExpr(comma, NewPrintString("\r = "), ostr);
	DecorateExpr(ostr);

	// These are fundamental types with language equivalents:
	while (*type == tp_const or *type == tp_volatile)
	    type++;
	if (*type == tp_short or *type == tp_long or *type == tp_char or *type == tp_int)
	    goto USE_DEFAULT;
	if (*type == tp_pointer and type[1] == tp_char)
	    goto USE_DEFAULT;
        if (*type == tp_array)  // We want to avoid the "ostream operator(void*)".
            goto USE_DEFAULT;
	if (*type == tp_reference)
	    goto USE_DEFAULT;
	if (*type == tp_function) {
	    value->type = voidptr_typstr;
	    value->tp = tp_pointer;
	    value->o = oConstant;
	    if (value->left == (void*)0x8)
		value->left = NULL;
	}
        if (*type == tp_enumerated)
            goto USE_DEFAULT;

	// Try a "ostream& <<" -operator:
	prt_value = NewExpr(op_doub('<'), ostr, value);
	success = Method("operator<<", prt_value, 2);
	if (success) {
	    DecorateExpr(prt_value);
	    goto DONE;
	}

	// Try the "*.Print()" method:
	prt_value = PrintMethod(value, ostr);
	if (prt_value != NULL) {
	    *e = *prt_value;
	    return; //goto DONE;	I did this because we don't want the terminating ";\n" for directory printing.
	}
	else goto USE_DEFAULT;


	USE_DEFAULT:
	value = ExprGetRvalue(value);
	prt_value = DefaultOutputter(ostr, value);
	DecorateExpr(prt_value);
	goto DONE;


	/*** Put this expression into 'e': ***/
	DONE:
	e->o = oCOMMA;
	e->tp = (tp_enum)0;
	e->type = void_typstr;
	e->ref = 0;
	e->left = prt_value;
	DecorateExpr(e->left);
	e->right = NewPrintString(";\n");
}



static void DecorateFunctionCall(Expr* e)
/* We have a function-call syntax expression. Decorate the */
/* expression by selecting an appropriate method.	   */
/* If it's a normal function call, ef->owner == NULL, and  */
/* if it's a member function call, ef->owner is the owner. */
/* In this function, we first choose an appropriate method */
/* for the call (in case there are more than one), and     */
/* then we call MethodDecorate() which prepares the oFUNC  */
/* expression (and which is also used for the call-by-	   */
/* operator-syntax function). */
{	FunctionNamedobj* obj;
        Namedobj *orig_obj;
	exprf_type ef;
	int i;

	/*** Get 'ef' and decorate its components: ***/
	ef = e->u.func;
	if (ef->owner != NULL)
	    DecorateExpr(ef->owner);
	for (i=0; i < ef->arity; i++)
	    DecorateExpr(ef->param[i]);


	/*** Select a method for doing the function call: ***/
	if (ef->func->opr == identifier) {

            orig_obj = (Namedobj*)ef->func->u.idfier.resolve->obj;  // This is often
                        // the very one we want anyway.
	    if (*orig_obj->type == tp_pointer and orig_obj->type[1] == tp_function)
                goto FUNCTION_POINTER;
            /* orig_obj points to the first object in the current directory */
            /* or the first member of the relevant class with this name. */
            obj = ChooseBestMethod(ef->func->u.idfier.resolve,
                			ef->param, ef->arity, e->src, yes);
	    if (obj == NULL) {
		e->type = err_typstr;
	        return;
            }
	    if (obj->storage == member_fn or obj->storage == virtual_fn) {
		if (ef->owner == NULL)		// Member fn's called from member fn's
		    ef->owner = DecorateCreateThis(e->src, yes);
                if (Error.err) {
                    e->type = err_typstr;
                    return;
                }
                if (ef->owner->ref == 0)
                    ef->owner = BigStructToLvalue(ef->owner);
		if (obj->storage == member_fn) {
		    /* A non-virtual function call: */
                    ef->func->o = oConstant;
                    ef->func->type = obj->type;
		    MethodDecorate(obj, e, ef->arity);
                }
		else {
		    /* A virtual function call: */
		    Expr* func;
                    assert(ef->owner->ref == 1);
		    func = ExprDuplicate(ef->owner);
		    assert(func->ref > 0);
		    func->ref--;
		    func = NewExpr(op_unry('*'),
				NewExpr(op_sing('+'),
				    func,
				    NewExprInt(-4)
				),
				NULL);
		    func->left->left->type = string_typstr;
		    DecorateExpr(func);
		    func->type = string_typstr;
		    func = NewExpr(op_sing('+'),
				    func,
				    NewExprInt(obj->u.virtualfn_idx*sizeof(str) + sizeof(Classdef))
				);
		    DecorateExpr(func);
		    func->type = virtualfntable_typstr;
		    ef->func = NewExpr(op_unry('*'),
				func,
				NULL
			    );
		    DecorateExpr(ef->func);
		    assert(ef->func->ref);
		    ef->func = ExprRvalue(ef->func);
		    MethodDecorate(obj, e, ef->arity);
		}
	    }
	    else if (obj->storage == straight_fn or obj->storage == oneinstr_fn) {

		/* A normal C-style function call: */
                ef->func->o = oConstant;
                ef->func->type = obj->type;
		MethodDecorate(obj, e, ef->arity);
	    }
	    else {
		assert(false);	    // There's no other storage class.
	    }
	}
	else if ((*ef->func->type == tp_pointer and ef->func->type[1] == tp_function)
		or *ef->func->type == tp_function) {

	    /* A function pointer: */
	    FUNCTION_POINTER:
	    DecorateExpr(ef->func);
	    if (*ef->func->type == tp_function) {
		assert(ef->func->opr == op_unry('*'));
		ef->func = ef->func->left;
	    }
	    ef->func = ExprGetRvalue(ef->func);
	    MethodDecorate(NULL, e, ef->arity);
	}
	else {

	    /* Our only hope left is an 'operator()' method. */
	    obj = MethodFind("operator()", e, ef->arity, no);
	    if (obj)
		MethodDecorate(obj, e, ef->arity);
	    else ErrorType(e->src, "You have the () function-call notation, "
		"but your expression is not a function or function pointer.");
	}
}


static bool TryArithmetic(Expr* e, o_enum k)
/* Try to decorate 'e' by giving it a method for an arithmetic */
/* operation.  See Bjarne Stroustrup's book, section C.6.3,    */
/* 'Usual Arithmetic Conversions'. */
/* Returns 'yes' for success. */
{	tp_enum atp,btp;
	Expr *a, *b;

	a = e->left;
	b = e->right;
	atp = (tp_enum)*a->type;
	btp = (tp_enum)*b->type;
	if (not TpGroup(atp) or not TpGroup(btp))
	    return no;

	/* Determine the type of the operation: */
	if (atp == tp_double or btp == tp_double)
	    e->type = double_typstr;
	else if (atp == tp_float or btp == tp_float)
	    e->type = float_typstr;
	else if (atp == tp_uint or btp == tp_uint)
	    e->type = uint_typstr;
	else e->type = int_typstr;

	/* Decorate 'e': */
	e->tp = TypeToTp(e->type);
	e->o = k;

	/* Convert operands if necessary: */
        e->left = ExprGetRvalue(e->left);
        e->right = ExprGetRvalue(e->right);
	if (atp != e->tp)
	    e->left = ExprInsertConversion(e->left, e->tp);
	if (btp != e->tp)
	    e->right = ExprInsertConversion(e->right, e->tp);

	return yes;
}


static bool TryArithmeticOrUserdefined(Expr* e, o_enum k, str opname,
                bool ReportError=yes)
/* 'e' is a binary operation.  Try to decorate 'e' by either:   */
/* 1. Using a built-in arithmetic operation, or                 */
/* 2. Finding a user-defined operator method.                   */
{       Namedobj* obj;
        char buf[512];
        str s;

        /* Ensure the operands are decorated: (usually they are already) */
        DecorateExpr(e->left);
        DecorateExpr(e->right);

        /* Try a built-in arithmetic operation: */
        if (TryArithmetic(e, k))
            return yes;

        /* Try a user-defined method: */
	obj = Resolve(opname);
	if (obj == NULL) {
            if (not ReportError)
                return no;
            if (TpGroup(e->left->tp) == '\0')
		s = "left ", TypeToString(e->left->type, buf, sizeof(buf));
            else if (TpGroup(e->right->tp) == '\0')
                s = "right ", TypeToString(e->left->type, buf, sizeof(buf));
            else s = "?";
            ErrorType(e->src, "Bad %soperand to operator %s: %s", s, opname+8, buf);
            e->type = err_typstr;
	    return no;
	}
	obj = ChooseBestMethod(&ResolveInfo,
		(e->opr == op_funccall) ? e->u.func->param : &e->left,
		2, e->src, ReportError);
        if (obj == NULL) {
            e->type = err_typstr;
            return no;
        }
	MethodDecorate((FunctionNamedobj*)obj, e, 2);
        return yes;
}


static void DecorateRelational(str name, o_enum o, Expr* e)
/* Generate a relational operation.   Try Method(), and */
/* if it comes up blank, try a pointer comparison.      */
{       int dimension;
	Type l,r;

        if (TryArithmetic(e, o)) {
            e->type = bool_typstr;
            e->tp = tp_int;
            return;
        }

	/* Special case for enum comparisons: */
        // Note the difficulties with enums:  if we call Method() first,
        // then it will try to match this operation with double==double,
        // long==long and int==int, and get an ambiguous fn call error.
	l = e->left->type;
	r = e->right->type;
        while (*l == tp_const) l++;
        while (*r == tp_const) r++;
	if (*l == tp_enumerated and *r == tp_enumerated) {
	    if (TypeEqual(l,r)) {
		e->left = ExprGetRvalue(e->left);
		e->right = ExprGetRvalue(e->right);
		e->o = o;
		e->tp = tp_int;
		e->type = bool_typstr;
		e->ref = 0;
	    }
	    else {
		char buf[30], buf2[30];
		ErrorType(e->src, "Comparing enums of different types: %s vs %s",
			TypeToString(l,buf,sizeof(buf)-1),
			TypeToString(r,buf2,sizeof(buf2)-1));
	    }
	    return;
	}

	/* Then try an existing method: */
	if (Method(name, e, 2))
	    return;

	/* No?  Try a pointer comparison. */
	if (*l == tp_pointer) {
	    l++;
	    e->left = ExprGetRvalue(e->left);
	}
	else if (*l == tp_array) {
	    if (e->ref == 0)
		goto ERROR;
	    else e->left = ExprGetLvalue(e->left);
	    GetDimension(dimension, l);
	}
	else goto ERROR;
	if (*r == tp_pointer) {
	    r++;
	    e->right = ExprGetRvalue(e->right);
	}
	else if (*r == tp_array) {
	    if (e->ref == 0)
		goto ERROR;
	    else e->right = ExprGetLvalue(e->right);
	    GetDimension(dimension, r);
	}
	else if (CanConvert(e->right, VPT_VAL, no)) {
	    e->right = ExprConvert(e->right, VPT_VAL);
	    r = voidptr_typstr + 1;
	}
	else goto ERROR;
	if (*l == tp_void or *r == tp_void or TypeEqual(l,r)) {
	    /* Generate a pointer comparison */
	    e->type = bool_typstr;
	    e->ref = 0;
	    e->o = o;
	    e->tp = tp_pointer;
	}
	else {
	    ERROR:
	    ErrorType(e->src, "Invalid operand");
	    e->type = err_typstr;
	}
}


static Expr* ConstructAnyAssignment(Expr* e)
/* We have an assignment of the form "any = non-any". Convert it to */
/* "X.val = Y, X.type = typeof(Y)". */
{	Expr *X, *Y, *X2, *e2;

	X = e->left;
        Y = e->right;
        assert(Y->ref == 1);
        X2 = ExprDuplicate(X);
        Y->ref = 0;		// Take the address of it.
        e->opr = comma;
        e->o = oCOMMA;
        ResolveInfo.list = NULL;
        ResolveInfo.obj = ResolveMember(AnyStruct, "value", private_visibility);
        e->left = NewExpr(op_sing('='), NewExpr(dot, X, NewExprIdentifier(&ResolveInfo)), Y);
        ResolveInfo.obj = ResolveMember(AnyStruct, "t", private_visibility);
        e->right = NewExpr(op_sing('='), e2=NewExpr(dot, X2, NewExprIdentifier(&ResolveInfo)),
        		NewExprVoidptr(Y->type));
        Y->type = voidptr_typstr;
        DecorateExpr(e2);
        e2->type = voidptr_typstr;
	DecorateExpr(e);
        e->type = any_typstr;
        e->ref = 0;
        e->tp = tp_class;

        return e;
}


interface void DecorateExpr(Expr* e)
#define GenMeth(op)         opname = "operator"op; goto GEN_METH
#define GenMethUnary(op)    opname = "operator"op; goto GEN_METH_UNARY
/* Recursively decorate 'e' and the tree it represents. */
/* This means adding type information and instruction   */
/* information to each expression.   */
{       bool success, before;
	type_plus_ref tr;
	Expr *a,*b;
	str opname;
	int size;

	assert(e != NULL);
	if (e->type != NULL)
	    return;     /* It's already been decorated. */
	assert(qptrcheck(e));
	a = e->left;
	b = e->right;

	switch (e->opr) {

	    /* Terminal nodes: */
	    case identifier:    DecorateIdentifier(e);
				break;

	    case char_const:    e->o = oConstant;
				e->tp = tp_char;
				e->type = char_typstr;
				break;

	    case int_const:     e->o = oConstant;
				e->tp = tp_int;
				e->type = int_typstr;
				break;

	    case long_const:    e->o = oConstant;
				e->tp = tp_int; // long==int
				e->type = long_typstr;
				break;

	    case float_const:   e->o = oConstant;
				e->tp = tp_double;
				e->type = double_typstr;
				break;

	    case kw_typeof:
	    case string_const:  // If the left is defined, it's a static.
				// If the right is defined, it's a local
				// static with the right as the offset.
				if (e->left) {
				    e->o = oConstant;
				    e->u.p = e->left;
				    e->tp = tp_pointer;
				}
				else {
				    e->o = oConstant;
				    e->u.p = e->right;
				    e->tp = tp_pointer;
				}
				e->type = (e->opr == kw_typeof) ?
					typetype_typstr : string_typstr;
				e->ref = 0;
				break;

	    case ostr_const:    e->o = oConstant;
				e->u.p = a;
				e->tp = tp_pointer;
				e->type = ostream_typstr;
				e->ref = 1;
				break;

	    case direc_const:   e->o = oConstant;
				e->u.p = a;
				e->tp = tp_pointer;
				e->type = direc_typstr;
				e->ref = 1;
				break;

	    case kw_container:	e->o = oConstant;
				e->u.p = e->left;
				e->tp = tp_int;
				e->type = container_typstr;
				e->ref = 0;
				e->opr = int_const;
				break;



	    /* Top-level operators: */
	    case semi_colon:
	    case comma:         DecorateExpr(a);
				DecorateExpr(b);
				e->o = oCOMMA;
                                e->tp = b->tp;
				e->type = b->type;
				e->ref = b->ref;
				if (a->o == oFUNC)
                                    CheckFunctionReturningBigStruct(a);
				break;

                                
	    /* Normal Operators: */
	    case kw_typedef:    // Casts:
				DecorateExpr(b);
				DecorateExplicitCast(e);
				break;

	    case kw_if:         // bool expressions:
				DecorateExpr(a);
				e->right = a;
				e->left = (Expr*)bool_typstr;
				e->opr = kw_typedef;
				/* Implement it as an explicit cast. */
				DecorateExplicitCast(e);
				break;

	    case op_funccall:   DecorateFunctionCall(e);
				break;

	    case dot:           DecorateExpr(a);
				if (*a->type != tp_class) {
				    if (*a->type == tp_pointer and a->type[1] == tp_class)
					ErrorType(a->src, "Use the arrow operator: \"->\"");
				    else ErrorType(a->src, "You need a struct here");
				    goto ERROR;
				}
				DecorateMemberAccess(e);
				break;

	    case arrowtok:      DecorateExpr(a);
				if (*a->type != tp_pointer) {
				    if (*a->type == tp_class)
					ErrorType(a->src, "Use the dot operator");
				    else ErrorType(a->src, "You need a struct pointer here");
				    goto ERROR;
				}
				a = ExprGetRvalue(a);
				a->type++;
				a->ref = 1;
                                e->left = a;
				DecorateMemberAccess(e);
				break;

	    case op_sing('\\'): DecorateExpr(a);
				DecorateElementAccess(e);
				break;

	    case op_sing('+'):  if (TryArithmeticOrUserdefined(e, oADD, "operator+", no))
                                    break;

                                /* Try pointer arithmetic: */
				if (*a->type == tp_array)
				    ArrayToPointer(a);
				if (*b->type == tp_array)
				    ArrayToPointer(b);
				if (*a->type == tp_pointer and
					    CanConvert(b, INT_VAL, no)) {
				    // Pointer arithmetic
				    DecorateIndex(&a, &b, no);
				    e->left = a;
				    e->right = ExprConvert(b, INT_VAL);
				    e->o = oADD;
				    e->type = a->type;
				    e->ref = 0;
				    e->tp = tp_pointer;
				}
				else if (*b->type == tp_pointer and
					    CanConvert(a, INT_VAL, no)) {
				    // Pointer arithmetic.
				    DecorateIndex(&b, &a, no);
				    e->left = ExprConvert(a, INT_VAL);
				    e->right = b;
				    e->o = oADD;
				    e->type = b->type;
				    e->ref = 0;
				    e->tp = tp_pointer;
				}
				else {
                                    TryArithmeticOrUserdefined(e, oADD, "operator+", yes);
                                    // This will generate a good error message.
				}
                                break;

	    case op_sing('-'):  if (TryArithmeticOrUserdefined(e, oSUB, "operator-", no))
                                    break;

                                /* Try pointer arithmetic: */
				if (*a->type == tp_array)
				    ArrayToPointer(a);
				a = ExprGetRvalue(a);
				DecorateExpr(a);
				if (*b->type == tp_array)
				    ArrayToPointer(b);
				b = ExprGetRvalue(b);
				DecorateExpr(b);

				if (*a->type != tp_pointer) {
                                    TryArithmeticOrUserdefined(e, oSUB, "operator-", yes);
                                    // This will generate a good error message.
                                    break;
				}
				else if (CanConvert(b, INT_VAL, no)) {
				    // Pointer - integer
				    DecorateIndex(&a, &b, no);
				    e->left = a;
				    e->right = ExprConvert(b, INT_VAL);
				    e->o = oSUB;
				    e->type = a->type;
                                    e->tp = tp_pointer;
				    e->ref = 0;
				    break;
				}
				else if (CanConvert(b, expr_to_tr(a), no)) {
                                    // Pointer - pointer
				    int size1, size2;
                                    e->left = ExprGetRvalue(a);
                                    e->right = ExprGetRvalue(b);
				    size1 = TypeSize(e->left->type+1);
				    size2 = TypeSize(e->right->type+1);
				    if (size1 != size2) {
					ErrorType(e->src, "Invalid Pointer Subtraction (different sizes)");
					goto ERROR;
				    }
				    e->o = oSUB;
				    e->type = int_typstr;
				    e->right = ExprConvert(b, expr_to_tr(a));
				    if (size1 > 1) {
					e = NewExpr(op_sing('/'), e, NewExprInt(size1));
					DecorateExpr(e);
					e->o = oDIV;
				    }
				    e->ref = 0;
                                    e->tp = tp_int;
				    break;
				}
				else {
				    ErrorType(e->src, "Invalid Pointer subtraction");
				    goto ERROR;
				}

	    case open_square:   DecorateExpr(a);
				DecorateExpr(b);
				if (*a->type == tp_pointer or *a->type == tp_array) {
				    e->opr = op_unry('*');
				    e->left = NewExpr(op_sing('+'), a, b);
				    e->right = NULL;
				    e->src = a->src;
				    DecorateExpr(e);
				}
				else if (*a->type == tp_dynarray) {
				    /* Accessing a dynamic array: */
				    if (not CanConvert(b, INT_VAL, no)) {
					ErrorType(e->src, "Invalid Operands");
					goto ERROR;
				    }
				    a = ExprGetLvalue(a);
				    b = NewExpr(op_sing('+'), ExprConvert(b, INT_VAL),
					    NewExprInt(TypeSize(a->type+1))
					);
				    DecorateExpr(b->right);
				    e->left = a;
				    e->right = b;
				    e->src = a->src;
				    e->o = oConstant;
				    e->u.p = Dynarray;
				    e->type = a->type + 1;
                                    e->tp = TypeToTp(e->type);
				    e->ref = 1;
				}
				else {
				    GenMeth("[]");
				}
				break;

	    case op_sing('*'):  TryArithmeticOrUserdefined(e, oMUL, "operator*");
                                break;

	    case op_sing('/'):  TryArithmeticOrUserdefined(e, oDIV, "operator/");
				break;

	    case op_sing('%'):  TryArithmeticOrUserdefined(e, oMOD, "operator%");
                                break;

	    case op_sing('|'):  TryArithmeticOrUserdefined(e, oOR, "operator|");
                                break;

	    case op_sing('&'):  TryArithmeticOrUserdefined(e, oAND, "operator&");
                                break;

	    case op_sing('^'):  TryArithmeticOrUserdefined(e, oXOR, "operator^");
                                break;

	    case op_doub('>'):  TryArithmeticOrUserdefined(e, oSHR, "operator>>");
                                break;

	    case op_doub('<'):  if (Method("operator<<", e, 2, no))
                                    break;
				/* A special case for an 'ostream <<' operation: */
                                if (CanConvert(e->left, OST_REF, yes)) {
				    /* With an ostream, any value can */
				    /* be outputted to it. */
				    e->left = ExprConvert(e->left, OST_REF);
				    *e = *DefaultOutputter(e->left, e->right);
				}
                                else {
                                    TryArithmeticOrUserdefined(e, oSHL, "operator<<", no);
                                    // This will give us a good error-message.
                                }
                                break;

	    case op_output:     /* This does the default outputting of */
				/* values to the screen.  It includes  */
				/* the " = %s;\n" stuff and includes   */
				/* quote marks etc. */
				DecorateOutput(e);
				break;

	    case op_unry('-'):  GenMethUnary("-");
	    case op_sing('!'):  GenMethUnary("!");
	    case op_sing('~'):  DecorateExpr(a);
				if (*a->type != tp_dynarray) {
				    GenMethUnary("~");
				}
				else {
				    e->o = oConstant;
				    e->u.p = DynarrayLen;
				    e->type = int_typstr;
                                    e->tp = tp_int;
				    e->ref = 0;
				}
				break;

	    case op_unry('&'):  DecorateExpr(a);
				if (a->ref == 0) {
                                    if (*a->type == tp_function or *a->type == tp_ptrmemberfn)
                                        a->ref++;
                                    else {
                                        success = Method("operator&_", e, 1);
                                        if (not success) {
                                            ErrorType(e->src, "You need an l-value");
                                            goto ERROR;
                                        }
				        break;
                                    }
				}
				e->ref = --a->ref;
				e->type = GetPointerToType(a->type);
                                e->tp = TypeToTp(e->type);
				e->o = oNOP1;
				break;

	    case op_unry('*'):  DecorateExpr(a);
				success = Method("operator*_", e, 1);
				if (success)
				    break;  // There was a method.
				// Otherwise invoke dereferencing by default.
				if (*a->type == tp_pointer) {
                                    e->type = a->type + 1;
                                    e->ref = ++a->ref;
                                    e->o = oNOP1;
                                    e->tp = tp_pointer;
                                }
                                else if (*a->type == tp_array) {
                                    e->type = a->type + 5;
                                    e->ref = ++a->ref;
                                    e->o = oNOP1;
                                    e->tp = tp_pointer;
                                }
                                else {
				    ErrorType(e->src, "You need a pointer");
				    goto ERROR;
				}
				break;

	    case op_doub('+'):  before = (b == NULL);
				if (before)
				    opname = "++operator";
				else {
				    opname = "operator++";
				    e->left = a = b;  // a is the expr for post & pre -.
				    e->right = NULL;
				}
				DecorateExpr(a);
                                if (*a->type == tp_const)
                                    ErrorType(e->src, "You cannot modify a const object");
				success = Method(opname, e, 1);
				if (success)
				    break;
				if (*a->type == tp_function or *a->type == tp_class or a->ref == 0) {
				    ErrorType(e->src, "Bad operand to ++");
				    goto ERROR;
				}
				e->left = a = ExprGetLvalue(e->left);
				if (*a->type == tp_double or *a->type == tp_float)
				    e->right = NewExprFloat(1.0);
				else {
				    if (*a->type == tp_pointer)
					size = TypeSize(a->type+1);     /* Scale the increment */
				    else size = 1;
				    e->right = NewExprInt(size);
				}
				DecorateExpr(e->right);
				e->o = before ? oAADD : oAADDAFTER;
				e->type = a->type;
				e->ref = 0;
				e->tp = TypeToTp(a->type);
				break;

	    case op_doub('-'):  before = (b == NULL);
				if (before)
				    opname = "--operator";
				else {
				    opname = "operator++";
				    e->left = a = b;  // a is the expr for post & pre -.
				    e->right = NULL;
				}
				DecorateExpr(a);
                                if (*a->type == tp_const)
                                    ErrorType(e->src, "You cannot modify a const object");
				success = Method(opname, e, 1);
				if (success)
				    break;
				if (*a->type == tp_function or *a->type == tp_class or a->ref == 0) {
				    ErrorType(e->src, "Bad operand to --");
				    goto ERROR;
				}
				e->left = a = ExprGetLvalue(e->left);
				if (*a->type == tp_pointer)
				    size = TypeSize(a->type+1);     /* Scale the increment */
				else size = 1;
				if (before)
				    e->o = oASUB;
				else e->o = oAADDAFTER, size = -size;
				if (*a->type == tp_double or *a->type == tp_float)
				    e->right = NewExprFloat(1.0);
				else e->right = NewExprInt(size);
				DecorateExpr(e->right);
				e->type = a->type;
				e->ref = 0;
				e->tp = TypeToTp(a->type);
				break;

	    case op_assg('+'):  DecorateExpr(a);
				DecorateExpr(b);
                                if (*a->type == tp_const)
                                    ErrorType(e->src, "You cannot modify a const object");
				success = Method("operator+=", e, 2);
				if (success)
				    break;
				if (*a->type == tp_pointer)
				    if (CanConvert(b, INT_VAL, no)) {
					// pointer += integer
					DecorateIndex(&a, &b, yes);
					e->left = a;
					e->right = b;
					e->o = oAADD;
					e->type = a->type;
					e->tp = tp_pointer;
					e->ref = 0;
					break;
				    }
				ErrorType(e->src, "Invalid operand");
				goto ERROR;

	    case op_assg('-'):  DecorateExpr(a);
				DecorateExpr(b);
                                if (*a->type == tp_const)
                                    ErrorType(e->src, "You cannot modify a const object");
				success = Method("operator-=", e, 2);
				if (success)
				    break;
				if (*a->type == tp_pointer)
				    if (CanConvert(b, INT_VAL, no)) {
					// pointer -= integer
					DecorateIndex(&a, &b, yes);
					e->left = a;
					e->right = b;
					e->o = oASUB;
					e->type = a->type;
					e->tp = tp_pointer;
					e->ref = 0;
					break;
				    }
				ErrorType(e->src, "Invalid operand");
				goto ERROR;

	    case op_sing('>'):  DecorateExpr(a);
				DecorateExpr(b);
				DecorateRelational("operator>", oGT, e);
				break;
	    case op_assg('>'):  DecorateExpr(a);
				DecorateExpr(b);
				DecorateRelational("operator>=", oGE, e);
				break;
	    case op_sing('<'):  DecorateExpr(a);
				DecorateExpr(b);
				DecorateRelational("operator<", oLT, e);
				break;
	    case op_assg('<'):  DecorateExpr(a);
				DecorateExpr(b);
				DecorateRelational("operator<=", oLE, e);
				break;

	    case op_assg('='):
	    case op_doub('='):  DecorateExpr(a);
				DecorateExpr(b);
				DecorateRelational("operator==", oEQ, e);
				break;

	    case op_assg('!'):  DecorateExpr(a);
				DecorateExpr(b);
				DecorateRelational("operator!=", oNE, e);
				break;

	    case op_doub('&'):  DecorateExpr(a);
				DecorateExpr(b);
				GenMeth("&&");

	    case op_doub('|'):  DecorateExpr(a);
				DecorateExpr(b);
				GenMeth("||");

	    case op_init:       /* Variable initialisations: */
				DecorateExpr(a);
				DecorateExpr(b);
				if (a->ref > 2) {
				    ErrorType(e->src, "Too many references");
				    goto ERROR;
				}
				if (a->ref >= 2) {
				    tr.type = a->type;
				    tr.ref = a->ref - 1;
				    e->right = ExprConvert(b, tr);
				    e->o = oASSIGN;
				    e->type = tr.type;
				    e->ref = tr.ref;
				    e->tp = tp_pointer;
				    break;
				}
				else goto ASSIGNMENT;

	    case op_sing('='):  DecorateExpr(a);
				DecorateExpr(b);

				ASSIGNMENT:
				if (*a->type == tp_enumerated)
				    goto DEFAULT_ASG;
				success = Method("operator=", e, 2);
				if (success)
				    break;

				/* Otherwise generate the default assignment. */
				DEFAULT_ASG:
                                if (*a->type == tp_const and e->opr == op_sing('='))
                                    ErrorType(e->src, "You cannot modify a const object");
				e->left = a = ExprGetLvalue(a);
				tr.type = a->type;
				tr.ref = 0;
                                if (b->type == NULL)
                                    break;
				if (TypeEqual(a->type, b->type))
				    e->right = b = ExprGetRvalue(b);
                                else if (TypeEqual(a->type, any_typstr)) {
                                    ConstructAnyAssignment(e);
                                    break;
                                }
				else if (not CanConvert(b, tr, no)) {
                                    if (*b->type == tp_pointer and b->type[1] == tp_class and
                                    	*tr.type == tp_pointer and tr.type[1] == tp_class) {
                                    	Classdef* lclass,*rclass;
                                        rclass = TypeToClassdef(b->type+1);
                                        lclass = TypeToClassdef(tr.type+1);
                                    	ErrorType(e->src, "Incompatible types: class %s is "
                                        	"not a base class of %s",
                                                lclass->typedef_obj->name,
                                                rclass->typedef_obj->name);
                                    }
				    else ErrorType(e->src, "Incompatible assignment");
				    goto ERROR;
				}
				else e->right = b = ExprConvert(b, tr);
                                if (TypeIsBigStruct(b->type)) {
                                    if (b->o == oFUNC) {
                                        if (b->u.func->param[0] == NULL) {
                                            b->u.func->param[0] = a;
                                            *e = *b;
                                            break;
                                        }
                                        // else carry on.
                                    }
                                    else if (b->o == oDER)
                                    	;
                                    else e->right = b = ExprGetRvalue(
                                    	BigStructToLvalue(b));
                                }
				e->o = oASSIGN;
				e->type = a->type;
				e->tp = TypeToTp(e->type);
				if (e->tp == tp_ptrmemberfn)
                                    e->tp = tp_pointer;
				e->ref = 0;
				break;

	    case op_assg('*'):  GenMeth("*=");
	    case op_assg('/'):  GenMeth("/=");
	    case op_assg('%'):  GenMeth("%=");
	    case op_assg('&'):  GenMeth("&=");
	    case op_assg('|'):  GenMeth("|=");
	    case op_assg('^'):  GenMeth("^=");
	    case op_shft('<'):  GenMeth("<<=");
	    case op_shft('>'):  GenMeth(">>=");

	    case op_sing('?'):  DecorateTernary(e);
				break;

				/* GEN_METH & GEN_METH_UNARY: */
				GEN_METH:
				/* For binary operations without a default op */
				DecorateExpr(a);
				DecorateExpr(b);
				success = Method(opname, e, 2);
				if (success)
				    break;
				ErrorType(e->src, "Invalid operand");
				goto ERROR;

				GEN_METH_UNARY: // For unary operations.
				DecorateExpr(a);
				success = Method(opname, e, 1);
				if (success)
				    break;
				ErrorType(e->src, "Invalid operand");
				goto ERROR_UNARY;

	    default:            assert(false);
				ERROR:
				ERROR_UNARY:
				e->type = err_typstr;
				break;
	}

	assert(Error.err or (e->type != NULL and e->o != oUnknown));
}




/*----------------- Interfacing expression compilation: ----------------*/

interface Expr* ExprRvalue(Expr* e)
/* Add this expression to the expression-list, */
/* but compile so that it leaves its value on  */
/* the stack, e.g. for return values.  It is   */
/* converted to an r-value if requested. */
{
	DecorateExpr(e);
	return ExprGetRvalue(e);
}




/*--------------- Member Access: ----------------*/

static int member_offset;
static Classdef* member_class;


static Namedobj* ClassFindObj(Classdef* classdef,
		str name, Type type)
/* Find the object matching this description from this class definition. */
{       Classdef* base_class;
	Namedobj* obj, *obj2;

	/* Is there a direct match in this type-definition? */
	for (obj = classdef->member; obj; obj=obj->next)
	    if (streq(obj->name, name) and TypeEqual(obj->type, type))
		return obj;

	/* Is there a match in some base class? */
	for (obj = classdef->member; obj; obj=obj->next) {
	    if (obj->storage == inherit_storage) {
		base_class = TypeToClassdef(obj->type);
		obj2 = ClassFindObj(base_class, name, type);
		if (obj2)
		    return obj2;
	    }
	}

	return NULL;
}


static Namedobj* ClassFindOriginal(Namedobj* o)
/* Each member function/field is mapped into the current name-space */
/* as a local variable.  Map the local 'obj' back to the one inside */
/* the type-definition.  This is needed for the 'obj->make' field.  */
{       Classdef* classdef;
	Type type;

	type = this_obj->type;
	assert(type[0] == tp_pointer and type[1] == tp_class);
	type += 2;
	GetPtr(classdef, type);
	return ClassFindObj(classdef, o->name, o->type);
}



static Expr* ExprToBase(Expr* e, int offset)
/* Add this offset to this expression, in order to */
/* convert it to some base class.   Define the type*/
/* of 'e' to be the 'current_class'. */
{       Expr* offset_e;
	Type type;
	uchar buf[5];

        if (offset == 0)
            return e;
	if (e->ref == 0)
	    assert(false);      /* Don't know how to do this one. */
	e = ExprGetLvalue(e);
	offset_e = NewExprInt(offset);
	e = NewExpr(op_sing('+'), e, offset_e);
	type = (Type)buf;
	*type++ = tp_class;
	PutPtr(member_class, type);
	type = (Type)qmalloc(5);
	memcpy(type, buf, 5);
	e->type = type;
	e->ref = 1;
        e->o = oADD;
        e->tp = TypeToTp(type);
	return e;
}


static void DecorateMemberAccess(Expr* e)
/* We have a reference to (hopefully) a struct in `a'. */
/* (It may have been converted from a pointer.)        */
/* And we are accessing `member'.                      */
{       Expr *a=e->left, *b=e->right;
	Classdef* classdef;
	type_plus_ref tr;
	Namedobj* obj;
	exprf_type ef;
        int offset;

	if (Error.err)
	    return;

	/**** Get the classdef: ****/
	if (*a->type != tp_class) {
	    ErrorType(a->src, "You need a class, struct or union here.");
	    e->type = err_typstr;
	    return;
	}
	else {
	    tr.type = a->type + 1;
	    GetPtr(classdef, tr.type);
	    if (classdef == NULL) {
		e->type = err_typstr;
		return;
	    }
	}
	if (a->ref > 1)
	    e->left = a = ExprGetLvalue(a);
        if (a->o == oNOP1) {
            if (a->left->src == NULL)
            	a->left->src = a->src;
            a->left->ref = a->ref;
            e->left = a = a->left;
        }
	if (a->o == oFUNC and a->ref == 0 and TypeIsBigStruct(a->type)) {
            e->left = a = BigStructToLvalue(a);
        }


	/**** Find the member: ****/
	if (b->opr == identifier) {
	    obj = b->u.idfier.resolve->obj;
            offset = b->u.idfier.resolve->offset;   // Nearly always 0
        }
	else if (b->opr == op_funccall) {
	    ef = b->u.func;
	    obj = ef->func->u.idfier.resolve->obj;
            offset = ef->func->u.idfier.resolve->offset;   // Nearly always 0
	}
	else {
            assert(false);
            return;
        }


	if (b->opr == op_funccall and
        	   (obj->storage == member_fn or obj->storage == virtual_fn)) {

	    /**** Function Access by function syntax ****/
	    ef = b->u.func;
	    if (obj->storage != virtual_fn)
		MakeDepends(obj->make);
	    ef->owner = ExprToBase(a, offset);
	    ef->func->type = obj->type;
	    ef->func->ref = 0;
	    ef->func->o = oConstant;
	    DecorateFunctionCall(b);
	    e->left = b;
	    e->right = NULL;
	    e->type = e->left->type;
	    e->ref = e->left->ref;
	    e->tp = TypeToTp(e->type);
	    e->o = oNOP1;
	}
	else if (*obj->type == tp_function) {

	    /**** Function Access by object type: ****/
	    ErrorType(b->src, "Member function %s accessed without function syntax", obj->name);
	    e->type = obj->type;
	    e->o = oConstant;
	    e->u.p = ((FunctionNamedobj*)obj)->u.fn;
	    e->ref = 0;
	    e->right = NULL;
	}
	else if (obj->storage == member_storage and
                ((FieldNamedobj*)obj)->offset < 0) {

	    /**** A bit-field access ****/
	    e->o = oBITFIELD;
	    e->u.bitfield = (bitfield_type)qcalloc(sizeof(struct bitfield_node));
	    DecodeBitField(offset, &offset, &e->u.bitfield->bit_offset, &e->u.bitfield->bit_width);
	    if (offset) {
		b->o = oConstant;
		b->tp = tp_int;
		b->u.i = offset;
		e->u.bitfield->addr = NewExpr(op_sing('+'), a, b);
		e->u.bitfield->addr->o = oADD;
		e->u.bitfield->addr->tp = tp_pointer;
		e->u.bitfield->addr->type = obj->type;
	    }
	    else {
		e->u.bitfield->addr = a;
	    }
	    e->tp = tp_int;
	    e->ref = 1;
	    e->right = NULL;
	    e->type = obj->type;
            e->tp = TypeToTp(e->type);
	    EnterTypeIntoExpression(e);
	}
	else if (b->opr != op_funccall) {

	    /**** Data-Field access ****/
	    e->o = oADD;
	    e->tp = tp_int;
	    e->ref = (*obj->type == tp_array) ? 0 : 1;
	    b->o = oConstant;
	    b->tp = tp_int;
	    b->u.i = offset + ((FieldNamedobj*)obj)->offset;
	    e->type = obj->type;
	    EnterTypeIntoExpression(e);

        }
        else {
            /*** Function-pointer: ***/
            if (obj->type[0] != tp_pointer or obj->type[1] != tp_function) {
            	ErrorParse(e->src, "This data member can't be used with function notation.");
                return;
            }

            /* Juggle the expr's around: Instead of (a.b()), we want to see it
            represented as  (a.b)().  */
            str src=e->src;
            e->o = b->o;
            e->tp = b->tp;
            e->src = b->src;
            e->ref = b->ref;
            e->left = b->left;
            e->right = b->right;
            e->opr = op_funccall;
	    ef = b->u.func;
            ef->owner = NULL;
            ef->func = b;
            b->o = oADD;
            b->tp = tp_pointer;
            b->src = src;
            b->type = obj->type;
            b->opr = op_sing('+');
            b->right = a;
            b->left = NewExprInt(offset + ((FieldNamedobj*)obj)->offset);
            b->ref = 1;
            ef->func = ExprDeref(b);
	    DecorateFunctionCall(e);
	}
}


static bool ClassInheritsRecurse(Classdef* classdef,
		    Classdef* base)
{       Namedobj* obj;

	if (classdef == base)
	    return yes;
	for (obj=classdef->member; obj; obj=obj->next) {
	    if (obj->storage == inherit_storage) {
		classdef = TypeToClassdef(obj->type);
		if (ClassInheritsRecurse(classdef, base)) {
		    member_offset += ((FieldNamedobj*)obj)->offset;
		    return yes;
		}
	    }
	}
	return no;
}


static bool ClassInherits(Classdef* classdef, Classdef* base)
/* Does this classdef inherit from this base class? */
{
	member_offset = 0;
	return ClassInheritsRecurse(classdef, base);
}


static bool TypeInherits(Type a, Type b)
/* Is this a type of something that inherits from this type? */
{	Classdef* A, *B;

	A = TypeToClassdef(a);
	B = TypeToClassdef(b);
	if (A == NULL or B == NULL)
	    return no;
	return ClassInherits(A, B);
}






/*-------------------- Element Access: ----------------------*/

static void DecorateElementAccess(Expr* e)
/* We hopefully have a directory constant on the left, */
/* and an identifier on the right.  We have to convert */
/* it to a simple identifier. */
{       Directory* direc;
	Namedobj* obj;
	str name;

	/* Get the directory: */
	if (not TypeEqual(e->left->type, direc_typstr)) {
	    ERROR:
	    ErrorType(e->src, "Expecting a directory name");
	    return;
	}
	if (e->left->opr == direc_const)
	    direc = (Directory*)e->left->u.p;
	else if (e->left->opr == identifier)
	    direc = (Directory*)e->left->u.p;
	else goto ERROR;

	/* Get the identifier: */
	name = (str)e->right;

	/* Search for the element: */
	if (streq(name, "..")) {
	    obj = two_dots_obj;
	    two_dots_obj->location = direc->parent;
	    goto FOUND;
	}
	obj = direc->Find(name);
	ErrorType(e->src, "Element %s not found", name);
	e->type = err_typstr;
	return;

	/* Replace this node with this element. */
	FOUND:
	e->opr = identifier;
	e->left = (Expr*)obj;
	DecorateIdentifier(e);
}





/*------------------ Coersions & Conversions ----------------*/

/* We keep a database of type-coersion functions.       */
/* We look up a generic_type (corresponding to the set  */
/* of coersion functions to that type)  given an        */
/* expanded type definition, by looking up a symbol     */
/* table. Later, we can have them directly off the type */
/* definition objects, but this requires that the type  */
/* system deals with non-expanded type definitions.     */

static char CanConvertTp(tp_enum from, tp_enum to)
/* Is it possible to convert from 'from' to 'to'? */
/* Return 6=no, 2=yes via promotion, 3=yes via    */
/* standard conversion, 4=only if it's an explicit*/
/* cast. */
{	static char Idx[12] = { tp_bool, tp_uchar, tp_char, tp_ushort, tp_short,
	    tp_uint, tp_int, tp_long, tp_float, tp_double, tp_int64, tp_enumerated };
	static char M[12][12] = {
	    /* to|    from->	b B c : 2 u i l f d 8 e */
	    /* bool	*/	1,3,3,3,3,3,3,3,3,3,3,4,
	    /* uchar	*/	3,1,3,3,3,3,3,3,3,3,3,4,
	    /* char	*/	3,3,1,3,3,3,3,3,3,3,3,4,
	    /* ushort	*/	3,3,3,1,3,3,3,3,3,3,3,4,
	    /* short	*/	3,3,3,3,1,3,3,3,3,3,3,4,
	    /* uint	*/	3,3,3,3,3,1,3,3,3,3,3,4,
	    /* int	*/	2,2,2,2,2,3,1,3,3,3,3,4,
	    /* long	*/	3,3,3,3,3,3,3,1,3,3,3,4,
	    /* float	*/	3,3,3,3,3,3,3,3,1,3,3,6,
	    /* double	*/	3,3,3,3,3,3,3,3,2,1,3,6,
	    /* int64	*/	3,3,3,3,3,3,3,3,2,2,1,4,
	    /* enum	*/	3,3,3,2,2,2,2,2,2,2,2,4
	};
	int f,t;

	for (f=0; f < sizeof(Idx); f++)
	    if (Idx[f] == from)
		goto FOUND_F;
	return 6;
	FOUND_F:
	for (t=0; t < sizeof(Idx); t++)
	    if (Idx[t] == to)
		goto FOUND_T;
	return 6;
	FOUND_T:
	return M[t][f];
}


static tp_enum LowestTp(int i)
/* What is the lowest type we can represent this integer */
/* constant inside? */
{
	if ((uchar)i == i)
	    return tp_uchar;
	if ((char)i == i)
	    return tp_char;
	if ((unsigned short)i == i)
	    return tp_ushort;
	if ((short)i == i)
	    return tp_short;
	if (i >= 0)
	    return tp_uint;
	return tp_int;
}


static int CanConvertBuiltIn(Expr* e, type_plus_ref tr)
/* Can we convert 'e' to 'tr'?  Returns 6 if not, otherwise:    */
/* 1=they're already the same (perhaps a trivial conversion),   */
/* 2=match using promotions, 3=match using standard conversions,*/
/* 4=only if it's explicit. */
{       Classdef* classdef;
	int dimension,level;
	Type a,b;

	a = e->type;
	b = tr.type;


	/*** L-value casts: ***/
	if (tr.ref) {
	    if (e->ref < tr.ref)
		return 6;
	    if (TypeEqual(a,b))
		return 1;
	    if (TpIsIntegral((tp_enum)*a) and TpToSigned((tp_enum)*a) == TpToSigned((tp_enum)*b))
		return 2;
	    if ((tp_enum)*a == tp_pointer and (tp_enum)*b == tp_pointer)
		return 4;
	    return 6;
	}


	/*** R-value casts: ***/
	if (a == b)                 // Short-cut:
	    return 1;
        if (*a == tp_const and *b != tp_const)
            a++;
	if (TypesNearlyEqual(a,b))  // Equal or equivalent types
	    return 1;
	if (TypeEqual(a, any_typstr))
	    return 4;               // An 'any' can be cast as anything.
	while (*a == tp_const or *a == tp_volatile)
	    a++;
	while (*b == tp_const or *b == tp_volatile)
	    b++;

	switch (*b) {               // What do we need to make a 'b'?

	    case tp_bool:   if (*e->type == tp_pointer)
				 return 3;
			    /* else carry on */

	    case tp_enumerated:
	    case tp_double:
	    case tp_ushort:
	    case tp_float:
            case tp_int64:
	    case tp_uchar:
	    case tp_short:
	    case tp_long:
	    case tp_char:
	    case tp_uint:
	    case tp_int:    if (e->opr == int_const and CanConvertTp(LowestTp(e->u.i), (tp_enum)*b) == 'Y')
			        return 2;
                            if (*a == tp_enumerated)
                                return (*b == tp_bool) ? 4 : 3;
			    level = CanConvertTp((tp_enum)*a, (tp_enum)*b);
			    if (level < 6)
                                return level;
			    if (*a == tp_pointer)
				return CanConvertTp(tp_int, (tp_enum)*b) < 4 ? 4 : 6;
                            return 6;

	    case tp_pointer:if (e->opr == identifier and e->u.p == NULL) {
            			/* 'e' represents NULL. */
            			assert(e->o == oConstant and e->tp == tp_pointer);
                                return yes;
            		    }
             		    else if ((tp_enum)*a == tp_pointer) {
				a++, b++;
                                if ((tp_enum)*b == tp_void)
				    return 2;
                                else if (*a == tp_class and *b == tp_class) {
                                    Classdef* aclass,*bclass;
                                    int offset;
                                    aclass = TypeToClassdef(a);
                                    bclass = TypeToClassdef(b);
                                    return BisBaseOfD(bclass, aclass, &offset) ? 2 : 4;
				}
                                else return 4;
			    }
			    else if ((tp_enum)*a == tp_array) {
				a++;
				b++;
				GetDimension(dimension, a);
                                if ((tp_enum)*b == tp_void)
				    return 2;
				return TypesNearlyEqual(a,b) ? 1 : 4;
			    }
			    else if ((tp_enum)*a == tp_function) {
				return TypesNearlyEqual(a,b+1) ? 1 : 4;
			    }
			    else if ((tp_enum)*a == tp_int) {
				if (e->opr == int_const and e->u.i == 0)
				    return 1;	    // This is the special token '0'.
				return 4;
			    }
			    else return 6;

	    case tp_ptrmemberfn:
                            return *a == tp_ptrmemberfn or *a == tp_pointer ? 4 : 6;

	    case tp_class:  b++;
            		    classdef = *(Classdef**)b;
                            if (classdef == AnyStruct)
                            	return 3;		// Any object can be converted to an any.
            		    return 6;

	    case tp_container:
	    case tp_void:
	    default:        return 6;
			    /* It's only convertible if they're type-equivalent,*/
			    /* and this was checked above. */
	    }
}


static Namedobj *CanConvertUserDefined(Expr* e, type_plus_ref tr)
/* Can we convert 'e' to 'tr' using a user-defined coersion operation? */
/* If we can, it returns the specific conversion function.      */
{       Classdef* classdef;
	Namedobj* obj;
	Type type;
	str name;

	type = tr.type;
	if (*type == tp_class) {
	    /* First check for constructor functions in 'tr's class: */
	    type++;
	    classdef = *(Classdef**)type;
	    name = classdef->typedef_obj->name;
	    for (obj=classdef->member; obj; obj=obj->next) {
		if (not streq(obj->name, classdef->typedef_obj->name))
		    continue;
		type = obj->type;
		assert(*type == tp_function);
		type++;
		if (*type++ != 1)                       // Arity should be one.
		    continue;
		type_plus_ref tr = GetTypePlusRef(type);
		if (TypeEqual(e->type, tr.type) and e->ref >= tr.ref)
		    return obj;
	    }
	}
	else {
	    if (*type == tp_int)
		name = "int";
	    else if (*type == tp_double)
		name = "double";
	    else return NULL;
	}


	/* Now check for any method in the current name-space. */
	obj = Resolve(name);
	if (obj == NULL or obj->type == NULL)
	    return NULL;
	obj = ChooseBestMethod(&ResolveInfo, &e, 1, e->src, no);
	return obj;
}


static bool CanConvert(Expr* e, type_plus_ref tr, bool is_explicit)
{       int level;

	level = CanConvertBuiltIn(e, tr);
	if (level <= 3)
	    return yes;
	if (level == 4 and is_explicit)
	    return yes;
	if (CanConvertUserDefined(e, tr))
	    return yes;
	return no;
}


static Expr* DoUserDefinedCoerce(Expr* a, type_plus_ref tr, Namedobj *coercer)
/* Generate a call to the user-defined coersion function, */
/* to convert 'a' to a 'b'. */
{	Expr *e, *func_e;

	func_e = NewExprIdentifier(coercer);
	a = ExprGetRvalue(a);
	e = NewExprFn(func_e, &a, 1);
	e->src = a->src;
	e->u.func->stack_size = TypeSizeWord(tr.type);
	func_e->type = coercer->type;
	func_e->ref = 0;
	func_e->o = oConstant;
	func_e->tp = tp_pointer;
	e->type = tr.type;
	e->tp = TypeToTp(e->type);
	return e;
}


interface Expr* ExprConvert(Expr* e, type_plus_ref target)
/* Convert this expression to this type, r-value or l-value. */
{	Namedobj *coercer;
	tp_enum from, to;

	if (*e->type == tp_error)
	    return e;


	/*** Attempt l-value conversions: ***/
	if (target.ref) {
	    // Will we allow a cast to an l-value?
	    if (e->ref < target.ref) {
		ErrorType(e->src, "Need an l-value here");
		return e;
	    }
	    while (e->ref > target.ref)
		e = ExprDeref(e);
	    if (e->type == target.type)
		goto REF_OK;
	    else if (TypesNearlyEqual(e->type, target.type))
		goto REF_OK;
	    else if (TpIsIntegral((tp_enum)*e->type) and TpIsIntegral((tp_enum)*target.type)) {
		if (TypeSize(e->type) == TypeSize(target.type))
		    goto REF_OK;
		else goto REF_ERROR;
	    }
	    else if ((tp_enum)*e->type == tp_pointer and (tp_enum)*target.type == tp_pointer)
		goto REF_OK;
	    else if (TypeEqual(e->type, any_typstr))
		goto ANY_PROJECTION;
	    else if ((tp_enum)*e->type == tp_class)
		goto INHERIT_CONVERT;
	    else goto REF_ERROR;

	    REF_ERROR:
	    ErrorType(e->src, "Bad cast to l-value");
	    e->tp = tp_pointer;
	    return e;

	    REF_OK:
	    e->type = target.type;
	    e->tp = tp_pointer;
	    return e;
	}


	/*** Short-cuts: ***/
	if (e->type == target.type)
	    return ExprGetRvalue(e);
        if (*e->type == tp_const)
            e->type++;
        if (*target.type == tp_const)
            target.type++;
	if (TypeEqual(e->type, target.type))
	    return ExprGetRvalue(e);
	else if ((coercer=CanConvertUserDefined(e, target)) != NULL) {
	    /* There is a user-defined coersion operation */
	    return DoUserDefinedCoerce(e, target, coercer);
	}


	/*** Projections out of 'any's: ***/
	if (*e->type == tp_class and CompPtr(e->type+1, &AnyStruct) == 0) {
	    /* Projection out of an 'any': */
	    ANY_PROJECTION:
	    e = NewExpr(op_sing('+'),
			ExprGetRvalue(e),
			ExprCopyOfTypestring(target.type, no));
	    assert(false);      // I'm sure this code doesn't work.
	    DecorateExpr(e->right);
	    e->o = oConstant;
	    e->u.p = any_project;
	    e->type = target.type;
	    e->ref = 1;
            e->tp = TypeToTp(e->type);
	    /* if (target.ref == 0)
		e = ExprDeref(e); */
	    return e;
	}


	/*** Generate conversions: ***/
        LOOP:
	to = (tp_enum)*target.type;
	switch (to) {

	    case tp_enumerated:
	    case tp_ushort:
	    case tp_short:
	    case tp_uchar:
	    case tp_long:
	    case tp_uint:
	    case tp_char:
	    case tp_bool:
	    case tp_int:
		e = ExprGetRvalue(e);
		from = (tp_enum)*e->type;
		if (TpIsIntegral(from)) {
		    if (TpSize(from) == TpSize(to)) {
			// Nothing to do.
			break;
		    }
		    else e = ExprInsertConversion(e, to);
		}
		else if (from == tp_float or from == tp_double) {
		    e = ExprInsertConversion(e, to);
		}
		else if (from == tp_pointer and to == tp_bool)
		    e = ExprInsertConversion(e, to);
		else if (from == tp_pointer and (to == tp_int or to == tp_long))
		    e->tp = tp_int;
		else if (from == tp_pointer and (to == tp_uint or to == tp_ulong))
		    e->tp = tp_uint;
		else ErrorType(e->src, "Bad conversion to integral type");
		break;


	    case tp_double:
	    case tp_float:
		e = ExprGetRvalue(e);
		from = (tp_enum)*e->type;
		if (TpIsIntegral(from) or from == tp_float or from == tp_double
			    or from == tp_pointer)
		    e = ExprInsertConversion(e, to);
		else ErrorType(e->src, "Bad conversion to float");
		break;

	    case tp_class:
		if (CompPtr(target.type+1, &AnyStruct) == 0) {
		    /* Target == ANY: */

		    /* Is it already an 'any'? */
		    if (*e->type == tp_class and CompPtr(e->type+1, &AnyStruct) == 0)
			return e;

		    /* Is it an l-value? */
		    if (e->ref) {
			/* A cast from an lvalue to an Any is always allowed. */
			e = ExprGetLvalue(e);
			e = NewExpr(comma, e, NewExprVoidptr(e->type));
                        e->o = oPUSH;
                        e->tp = tp_class;
                        e->src = e->left->src;
			e->type = any_typstr;
                        e->tp = tp_class;
			e->ref = 0;
			return e;
		    }

		    /* Allow functions to be 'any's: */
		    if (*e->type == tp_function) {
			e = NewExpr(comma, NewExprVoidptr(e->type), e);
                        e->o = oPUSH;
                        e->tp = tp_class;
                        e->src = e->left->src;
			e->type = any_typstr;
			e->ref = 0;
                        e->tp = tp_class;
			return e;
		    }

		    /* Is it an identifier? */
		    if (e->opr == identifier) {
			/* It might be an array: */
			Type type;

			e = ExprGetRvalue(e);
			type = e->type;
			assert(type != NULL);       // It should be decorated already.
			if (*type == tp_array) {
			    /*e = ExprAddInstr(e, PUSHP, (op_type)type);*/
			    e->type = any_typstr;
			    e->ref = 0;
                            e->tp = tp_class;
			    return e;
			}
		    }

		    /* Ok.  All we have is an r-value, so let's make  */
		    /* an 'any' by allocating from the volatile heap. */
		    assert(false);
#if 0
		    e = ExprAddInstr(e, NOP, NULL);
		    ExprMultipleInstr(e, PUSH, TypeSize(e->type),
					CALLSYS, GetFrval2lval(),
					PUSHP, e->type,
					NOP);
#endif
		}
		else {
		    INHERIT_CONVERT:
		    if (e->ref == 0)
			goto REF_ERROR;
		    if (not TypeInherits(e->type, target.type)) {
			ErrorType(e->src, "Bad conversion to struct");
			break;
		    }
		    e = ExprToBase(e, member_offset);
		}
		break;


	    case tp_void:
		DecorateExpr(e);
		break;


	    case tp_ptrmemberfn:
	    case tp_pointer:
		e = ExprGetRvalue(e);
		if (*e->type == tp_pointer or *e->type == tp_ptrmemberfn) {
		    if (*e->type == tp_pointer and to == tp_pointer) {
                    	Classdef *bclass,*dclass;
                        int offset;
                        bclass = TypeToClassdef(target.type+1);
                        dclass = TypeToClassdef(e->type+1);
                        if (bclass and dclass and BisBaseOfD(bclass,dclass,&offset)) {
                            if (offset) {
                                e = NewExpr(op_sing('+'), e, NewExprInt(offset));
                                e->type = target.type;
                                e->ref = 0;
                                e->o = oADD;
                                e->tp = tp_pointer;
                                return e;
                            }
                        }
                    }
                }
		else if (*e->type == tp_array) {
		    ArrayToPointer(e);
		}
		else if (*e->type == tp_function) {
		    if (target.type[1] != tp_void and target.type[1] != tp_function)
			ErrorType(e->src, "Bad conversion: fn to ptr");
		}
		else if (*e->type == tp_int)
		    ;
		else ErrorType(e->src, "Bad conversion to pointer");
		break;


	    case tp_container:
		e = ExprGetRvalue(e);
		if (*e->type == tp_container)
		    break;
		else ErrorType(e->src, "Bad conversion");
		break;


            case tp_const:
            case tp_volatile:
            	target.type++;
            	goto LOOP;


	    default:
		ErrorType(e->src, "Unknown type in conversion");
		break;
	}

	e->type = target.type;
	e->ref = target.ref;
        e->tp = TypeToTp(e->type);
	return e;
}


static void DecorateExplicitCast(Expr* e)
{       Expr* a=e->right;
	type_plus_ref tr;

	/*** Initialise: ***/
	tr = GetTypePlusRef((Type)e->left);
	DecorateExpr(a);


	/*** Is it already that type? ***/
	if (a->ref == tr.ref and TypeEqual(a->type, tr.type)) {
	    e->left = a;
	    e->type = e->left->type;
	    e->ref = e->left->ref;
	    e->o = oNOP1;
            e->tp = TypeToTp(e->type);
	    return;
	}


	/*** Are we allowed to do it? ***/
	if (a->ref and tr.ref == 0) {       // E.g. consider:
	    tr.ref++;                       //   (str)vp = "Hello";
	    if (CanConvert(a, tr, yes))
		goto CONVERT_OK;
	    tr.ref--;
	}
	if (not CanConvert(a, tr, yes)) {
	    ErrorType(a->src, "Illegal cast");
	    e->type = err_typstr;
	    return;
	}


	/*** Do it: ***/
	CONVERT_OK:
	e->left = ExprConvert(a,tr);
	e->type = e->left->type;
	e->ref = e->left->ref;
        e->tp = TypeToTp(e->type);
	e->o = oNOP1;
}




/*----------------------- Inserting Destructor calls: ---------------------*/

static str ShortSrc(str src, char tmp[25])
{       str s,d;

        if (src == NULL)
            return "(null)";
        s = src;
        d = tmp;
        while (*s and d-tmp < 15) {
            if (*s == '\n')
                s++;
            else if (*s == '\t')
                *d++ = ' ', s++;
            else *d++ = *s++;
        }
        while (d-tmp < 15)
            *d++ = ' ';
        *d = '\0';
        return tmp;
}


static void InsertDestructorCalls(Statement* A, Statement* *Bp)
/* Examine this transition from A to B (B==A->next or A->IfTrue etc).   */
/* If variables go out of scope, then insert destructors for them.      */
/* If variables enter scope and we skip a relevant constructor, then    */
/* output a warning. */
{       AutoNamedobj *objA, *objB, *join;
        Statement *B=*Bp;
        ExprStatement* S;
        Expr* e;

        /* A short-cut of B==NULL: */
        if (B == NULL) {
            join = NULL;
            goto DESTRUCTORS;
        }

        /* Find the closest common ancestor (could be NULL): */
        objA = A->scope;
        objB = B->scope;
        while (objA != objB) {
#define objoffset(obj)  (obj?obj->offset:(1<<30))
            if (objoffset(objA) < objoffset(objB))
                objA = (AutoNamedobj*)objA->next;
            else objB = (AutoNamedobj*)objB->next;
        }
        join = objA;

        /* Check for variables having entered scope: */
        for (objB=B->scope; objB != join; objB=(AutoNamedobj*)objB->next) {
            /*printf("<<%s>>   to   <<%s>>:   '%s' came into scope.\n",
                        ShortSrc(A->src, tmp1),
                        ShortSrc(B->src, tmp2),
                        objB->name);*/
            if (objB->initialisation/*TypeHasConstructor(objB->type)*/) {
                /* objB needs a constructor. Are we right on its constructor? */
                if (B->st != st_expr)
                    goto MISSING;
                e = ((ExprStatement*)B)->e;
                if (e == objB->initialisation)
                    continue;
                MISSING:
                ErrorParse(A->src, "You're bypassing the initialisation for '%s'",
                                objB->name);
                        // This is a warning, not error, on most other compilers
                /*
                Namedobj *funcobj;
                exprf_type ef;
                if (e->o != oFUNC)
                    goto MISSING;
                ef = e->u.func;
                if (ef->owner == NULL or ef->owner->o != oLocalVar)
                    goto MISSING;
                if (ef->owner->u.obj != objB)
                    goto MISSING;       // Looking good: it's a member fn for 'obj'
                e = ef->func;
                if (e->o != oConstant)
                    goto MISSING;
                funcobj = e->u.idfier.resolve->obj;
                if (funcobj == NULL)
                    goto MISSING;
                if (not IsConstructor(funcobj)) {
                    MISSING:
                    ErrorParse(A->src, "You're skipping the constructor for '%s'",
                                objB->name);     // A warning on other compilers
                } */
            }
        }

        /* Check for variables having left scope: */
        DESTRUCTORS:
        for (objA=A->scope; objA != join; objA=(AutoNamedobj*)objA->next) {
            /*printf("<<%s>>   to   <<%s>>:   '%s' went out of scope.\n",
                        ShortSrc(A->src, tmp1),
                        ShortSrc(B->src, tmp2),
                        objA->name);*/
            if (TypeHasDestructor(objA->type)) {
                /* We need a destructor call here. */

                /* First create a decorated '&variable' expression: */
                e = NewExprIdentifier(objA);
                e->src = A->src;
                DecorateExpr(e);
                e->ref--;
                e->type = GetPointerToType(e->type);
                e->tp = tp_pointer;

                /* Now create the destructor call: */
                e = DestructorCall(objA->type, e, NULL, no);

                /* Optimise it now, because we've already finished the main */
                /* optimisation phase. */
                OptimiseExpr(&e, no);

                /* Now set it up as a statement: */
                S = (ExprStatement*)NewStatement(st_expr);
                S->e = e;
                S->src = e->src = A->src;
                assert(S->next == NULL);
                *Bp = S;
                Bp = &S->next;
            }
        }
        *Bp = B;
}


static void ExamineTransition(Statement *From, Statement **To_p)
{       Statement *To=*To_p;

        if (To->scope != From->scope)
            InsertDestructorCalls(From, To_p);
}


interface void InsertDestructorCalls()
/* Examine each transition from statement A to statement B, in order to */
/* (a) insert destructor calls, and (b) check for skipping initialisations. */
{       SwitchStatement* S_switch;
        ReturnStatement* S_return;
	switchcase_type scase;
	Statement *S,*save;
	IfStatement* S_if;
        ExprStatement *D;

	for (each_statement) {
	    if (S->st == st_expr or S->st == st_null) {
		ExamineTransition(S, &S->next);
	    }
	    else if (S->st == st_if) {
		S_if = (IfStatement*)S;
		ExamineTransition(S, &S_if->True);
		ExamineTransition(S, &S_if->Fals);
		if (S_if->True == S_if->Fals) {
		    S->next = S_if->True;
		    S->st = st_expr;
		    ((ExprStatement*)S)->e = S_if->Test;
		}
	    }
	    else if (S->st == st_switch) {
		S_switch = (SwitchStatement*)S;
		for (scase=S_switch->Cases; scase; scase=scase->next)
		    ExamineTransition(S, &scase->Body);
                if (S_switch->Default == NULL)
                    S_switch->Default = S->next;
		ExamineTransition(S, &S_switch->Default);
	    }
            else if (S->st == st_return) {
                if (S->scope == NULL)
                    continue;
                D = NULL;
                save = first_statement;
		InsertDestructorCalls(S, (Statement**)&D);
                if (D) {
                    // Convert 'D' from a linked list of statements to
                    // a linked list of expressions.
                    S_return = (ReturnStatement*)S;
                    S_return->destructors = D->e;
                    while (D->next) {
                        D = (ExprStatement*)D->next;
                        S_return->destructors = NewExpr(comma,
                                S_return->destructors, D->e);
                        D->e = NULL;
                        D->src = NULL;
                    }
                    first_statement = save;     // This effectively deletes
                    // the ExprStatements we just created, so we don't need
                    // to worry about them getting compiled.  If the
                    // destructor-call-expressions are taken out of them
                    // and put into the ReturnStatement, then we shouldn't
                    // compile the ExprStatements.
                }
            }
	}
}

