/********************************************************\
** COMPLEX.C:                                           **
**          The lexical analyser for the Compiler.      **
** Takes a null-terminated string and provides a stream **
** of tokens.  This involves lexical analysis,          **
** preprocessing and symbol-table functions.            **
\********************************************************/

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <stdarg.h>
#include "barbados.h"
#include "tfc.h"
#include "modules.h"
#include "compiler.h"
#include "name.h"



#pragma warning ( disable : 4102 )
#define EOFC    -1



interface int StatLineCount;


static void ProcessPreprocessor(void);
static void ExpandMacro(str macro);







/*===================== Serialising Source Text: ===================*/

#define MAX_MACLEVELS   15

interface str source;		// Should be static, but we need it global for debugging.
static char tok_ch;                     // The next character to be read
static str sub_source[MAX_MACLEVELS];
static int ss_idx;
static int LineNo;
static bool LineWrap;
static str* MacroExpansions;
static int MI_idx;


static char nextch(void)
/* Return the next character in the stream. */
{
	RETRY:
	tok_ch = *source++;                  // The ordinary case
	if (tok_ch == '\\')
	    if (*source == '\0' or *source == '\n')
		goto SLOSH;
	if (tok_ch)
	    return tok_ch;

	EOS:
	if (ss_idx) {                       // Returning from macro expansions
	    source = sub_source[--ss_idx];
	    goto RETRY;
	}
	else goto EOL;

	EOL:
	source--;
	return tok_ch = EOFC;                    // We're compiling a '\0'-termd str.

	SLOSH:
	if (*source == '\0' and ss_idx)
	    goto EOS;
	else {
	    source++;
	    goto RETRY;
	}
}


interface void CompileBegin(str s)
/* Initialise the source stream to do this str. */
{
	ss_idx = 0;
	Error.err = err_none;
	LineWrap = no;
	LineNo = 0;
	if (tok.en != EOF) {
	    ErrorParse(NULL, "Error: Compiler re-entered");
	    tok.en = EOF;
	    return;
	}
	source = s;
	CurrentSource = source;
        MI_idx = 0;
        MacroExpansions = NULL;
	nextch();
        tok.next = NULL;
	NextToken();
}


interface str LexGetSourcePos(void)
{
	if (ss_idx == 0)
	    return source;
	else return sub_source[0];
}


interface str LexGetSourceStart()
{
	return source;
}





/*================== Tokenising: ==================*/

interface struct token_struct tok;      // The current token


static char GetChar(void)
/* Get a character out of a character str that started with '\\'. */
{	static char s1[] = "\n\r\t\b\f\"\'\\";
	static char s2[] = "nrtbf\"\'\\";
	char ch;
	str s;
	int r;

	ch = nextch();
	s = strchr(s2, ch);
	if (s) {
	    nextch();
	    return s1[s-s2];
	}
	else if (ch == 'x') {
	    r = 0;
	    while (isxdigit(ch=nextch())) {
		if (isdigit(ch))
		    ch -= '0';
		else if (isupper(ch))
		    ch -= 'A' - 10;
		else ch -= 'a' - 10;
		r = r * 16 + ch;
	    }
	    return r;
	}
	else if (isdigit(ch)) {
	    r = 0;
	    do {
		if (ch < '0' or ch > '7')
		    return r;
		r = r * 8 + (ch - '0');
		ch = nextch();
	    } forever;
	}
	else {
	    nextch();
	    return ch;
	}
}


static token_enum ConstructToken(void)
/* Physically construct the next token.  The first character to be */
/* examined should be tok_ch. */
#define Return(t) nextch(); return (t)
#define state(s) s: nextch(); keep##s
{	Namedobj* obj;

	goto keepSTART;

	state(START):   tok.x = 0;
			tok.src = source-1;
			assert(tok.x < 100);
			tok.buf[tok.x++] = tok_ch;
			tok.buf[tok.x] = '\0';
			switch (tok_ch) {
			    case '#':	goto PREPROCESSOR;
			    case '/':	goto AWAIT_ASTERISK;
			    case '\'':	goto CHAR_CONST;
			    case '\"':	tok.x = 0;
					goto STRING_CONST;
			    case '\n':
			    case '\t':
			    case '\f':
			    case '\r':
			    case ' ':	goto START;
			    case ']':	Return (close_square);
			    case ')':	Return (close_round);
			    case '(':	Return (open_round);
			    case '[':	Return (open_square);
			    case '{':	Return (open_brace);
			    case '}':	Return (close_brace);
			    case ';':	Return (semi_colon);
			    case ':':	nextch();
					if (tok_ch == ':') {
					    nextch();
					    return double_colon;
					}
					else return colon;
			    case ',':	Return (comma);
			    case '.':	Return (dot);
			    case '~':	Return (op_sing('~'));
			    case '?':	Return (ternary);
			    case '`':	tok.x = 0;
					goto LITERAL_IDENTIFIER;
			    case '+':
			    case '-':
			    case '&':
			    case '|':	goto DOUBLE_OPERATOR;
			    case '>':
			    case '<':	goto SHIFT_OPERATOR;
			    case '!':
			    case '=':
			    case '*':
			    case '%':
			    case '^':	goto ASSIGN_OPERATOR;
			    case '0':	goto OCTAL_OR_HEX;
			    case '1': case '2': case '3':
			    case '4': case '5': case '6':
			    case '7': case '8': case '9':
					tok.x--;
					goto keepDECIMAL;
			    case EOFC:	tok.x = 1;
					goto keepEND_OF_FILE;
			    case '_':	tok.x--;
					goto keepIDENTIFIER;
			    default:	if (isalpha(tok_ch)) {
					    tok.x--;
					    goto keepIDENTIFIER;
					}
					else goto ERROR;
			}


	state(IDENTIFIER):  if (isalnum(tok_ch) or tok_ch == '_') {
				tok.buf[tok.x++] = tok_ch;
				goto IDENTIFIER;
			    } else if (tok_ch == EOFC) {
				tok.x--;
				return (EOF);
			    }

			    tok.buf[tok.x] = '\0';

                            /* Is it a macro? */
                            tok.local_obj = obj = NameLocalResolve(tok.buf);
                            if (obj and obj->storage == macro_storage) {
                            	MACRO:
				MacroUsed(obj);
				ExpandMacro(((MacroNamedobj*)obj)->macro);
				goto START;
			    }
                            if (curdir) {
                            	tok.curdir_obj = obj = curdir->Find(tok.buf);
                            	if (obj and obj->storage == macro_storage)
                                    goto MACRO;
                            }

                            /* Is it a keyword? */
                            if (stdlib_dir) {
                            	tok.stdlib_obj = obj = stdlib_dir->Find(tok.buf);
	    			if (obj and obj->storage == keyword_storage)
                                    return (token_enum)((IntNamedobj*)obj)->constval;
                            	if (obj and obj->storage == macro_storage)
                                    goto MACRO;
                            }

                            /* Okay, it's just a normal typed C++ identifier. */
                            /* Let's leave further resolving for later. */
			    return identifier;

	state(LITERAL_IDENTIFIER):
			    // Construct an identifier from whatever we find between `` marks.
			    // It will not be considered a macro, label or keyword.
			    if (tok_ch == '`') {
				tok.buf[tok.x] = '\0';
				nextch();
				return identifier;
			    }
			    else if (tok_ch == '\n' or tok_ch == EOFC or tok.x > 60) {
				ErrorParse(tok.src, "Unterminated back-quote (`)");
				goto START;
			    }
			    tok.buf[tok.x++] = tok_ch;
			    goto LITERAL_IDENTIFIER;

	state(DECIMAL):     if (isdigit(tok_ch)) {
				tok.buf[tok.x++] = tok_ch;
				goto DECIMAL;
			    } else if (tok_ch == '.') {
				tok.buf[tok.x++] = tok_ch;
				goto FLOAT_DEC;
			    } else if (toupper(tok_ch) == 'E') {
				goto FLOAT_EXP;
			    } else {
				tok.buf[tok.x] = '\0';
				tok.long_val = atol(tok.buf);

				RETURN_INTEGRAL:
				tok.int_val = tok.long_val;
                                tok.x = -1;     // denotes signed.
				if (tok_ch == 'L') {
				    nextch();
				    return long_const;
				}
                                else if (tok_ch == 'U') {
                                    nextch();
                                    tok.x = 0;   // denotes unsigned.
                                }
				if (tok.int_val == tok.long_val)
				    return int_const;
				else return long_const;
			    }

	state(OCTAL_OR_HEX):tok.int_val = 0;
			    tok.long_val = 0;
			    tok.buf[tok.x++] = tok_ch;
			    if (tok_ch == 'x')
				goto HEX;
			    else if (tok_ch == '.')
				goto FLOAT_DEC;
			    else if (tok_ch < '0' or tok_ch > '7') {
				// The '0' token:
				tok.buf[1] = '\0';
				tok.long_val = 0;
                                tok.x = -1;
				return int_const;
			    }
			    else goto OCTAL;

	state(HEX):         if (isxdigit(tok_ch)) {
				tok.buf[tok.x++] = tok_ch;
				tok.long_val *= 16;
				if (isdigit(tok_ch))
				    tok.long_val += (tok_ch - '0');
				else tok.long_val += (toupper(tok_ch) - 'A' + 10);
				goto HEX;
			    }
			    else
				goto RETURN_INTEGRAL;

	state(OCTAL):       if (tok_ch >= '0' and tok_ch <= '7') {
				tok.buf[tok.x++] = tok_ch;
				tok.long_val *= 8;
				tok.long_val += (tok_ch - '0');
				goto OCTAL;
			    }
			    else
				goto RETURN_INTEGRAL;

	state(FLOAT_DEC):   if (isdigit(tok_ch)) {
				tok.buf[tok.x++] = tok_ch;
				goto FLOAT_DEC;
			    } else if (toupper(tok_ch) == 'E') {
				tok.buf[tok.x++] = 'E';
				nextch();
				if (tok_ch == '-' or tok_ch == '+') {
				    tok.buf[tok.x++] = tok_ch;
				    nextch();
				}
				goto FLOAT_EXP;
			    } else {
				tok.float_val = (float)atof(tok.buf);
				return float_const;
			    }

	state(FLOAT_EXP):   if (isdigit(tok_ch)) {
				tok.buf[tok.x++] = tok_ch;
				goto FLOAT_DEC;
			    }
			    tok.float_val = (float)atof(tok.buf);
			    return float_const;

	state(CHAR_CONST):  if (tok_ch == '\\')
				tok.buf[0] = GetChar();
			    else tok.buf[0] = tok_ch, nextch();
			    if (tok_ch != '\'')
				ErrorParse(tok.src, "You can only have a single "
                                        "char in single-quotes (perhaps you "
                                        "want double-quotes: \")");
			    nextch();
			    return char_const;

	state(STRING_CONST):if (tok_ch == '\"') {
				tok.buf[tok.x] = '\0';
				nextch();
				return (string_const);
			    }
			    else if (tok.x >= 158) {
				tok.buf[tok.x] = '\0';
				tok_ch = '"';
				return string_const;
			    }
			    else if (tok_ch == EOFC) {
				ErrorParse(tok.src, "You're missing the end-quote.");
				tok.x--;
				return (EOF);
			    } else if (tok_ch == '\\') {
				tok.buf[tok.x++] = GetChar();
				goto keepSTRING_CONST;
			    }
			    else {
				tok.buf[tok.x++] = tok_ch;
				goto STRING_CONST;
			    }

	state(SHIFT_OPERATOR):
			    if (tok_ch == tok.buf[0]) {
				tok.buf[tok.x++] = tok_ch;
				goto ASSIGN_OPERATOR;
			    }
			    else goto keepASSIGN_OPERATOR;

	state(DOUBLE_OPERATOR):
			    if (tok_ch == tok.buf[0]) {
				tok.buf[tok.x++] = tok_ch;
				tok.buf[tok.x] = '\0';
				nextch();
				return (op_doub(tok.buf[0]));
			    }
			    else if (tok.buf[0] == '-' and tok_ch == '>') {
				tok.buf[1] = tok_ch;
				tok.buf[2] = '\0';
				nextch();
				return arrowtok;
			    }
			    else goto keepASSIGN_OPERATOR;

	state(ASSIGN_OPERATOR):
			    if (tok_ch == '=') {
				tok.buf[tok.x++] = tok_ch;
				tok.buf[tok.x] = '\0';
				nextch();
				return ((tok.x==2) ? op_assg(tok.buf[0]) : op_shft(tok.buf[0]));
			    }
			    else {
				tok.buf[tok.x] = '\0';
				return ((tok.x==1) ? op_sing(tok.buf[0]) : op_doub(tok.buf[0]));
			    }

	state(AWAIT_ASTERISK):
			    if (tok_ch == '*')
				goto REMARK;
			    else if (tok_ch == '/')
				goto REMARK_II;
			    else
				goto keepASSIGN_OPERATOR;

	state(REMARK):      if (tok_ch == EOFC)
				return EOF;
			    else if (tok_ch == '*')
				goto AWAIT_SLASH;
			    goto REMARK;

	state(AWAIT_SLASH): if (tok_ch == '/')
				goto START;
			    else goto keepREMARK;

	state(REMARK_II):   /* This does the C++ double-slash remarks. */
			    if (tok_ch == '\n')
				goto START;
			    else if (tok_ch == EOFC) {
				tok.x = 0;
				return (EOF);
			    }
			    goto REMARK_II;

	state(PREPROCESSOR):ProcessPreprocessor();
			    tok.x = 0;
			    goto keepSTART;

	state(ERROR):       ErrorParse(tok.src, "Illegal token");
			    goto START;

	state(END_OF_FILE): return (EOF);
}


interface token_enum NextToken(void)
/* Provide the caller with the next token in the stream. */
{
        if (tok.next) {
            token_struct *toknext = tok.next;
            tok = *toknext;
            free(anon_heap, toknext);
            return tok.en;
        }
        else return tok.en = ConstructToken();
}


interface void Gobble(token_enum expected)
/* Eat the current token if it is 'expected', otherwise */
/* send an error. */
{
	if (tok.en != expected) {
	    if (expected == semi_colon)
		ErrorParse(tok.src, "I was expecting a semi-colon.");
	    else if (expected == colon)
		ErrorParse(tok.src, "I was expecting a colon.");
            else if (expected == close_round)
                ErrorParse(tok.src, "I was expecting a close-round ')'");
            else if (expected == close_brace)
                ErrorParse(tok.src, "I was expecting a close-brace '}'");
            else if (expected == close_square)
                ErrorParse(tok.src, "I was expecting a close-square ']'.");
            else if (expected == comma)
                ErrorParse(tok.src, "I was expecting a comma.");
	    else ErrorParse(tok.src, "I was expecting something else.");
	}
	NextToken();
}


interface void SkipTo(token_enum t)
/* Skip input until token `t'. Used for error recovery. */
{
	until (tok.en == t or tok.en == EOF)
	    NextToken();
}


interface token_enum LookAhead(void)
/* Returns the next token which will be read, without actually going to it. */
{
        if (tok.next == NULL) {
            token_struct tmp=tok, *next;
            tok.en = ConstructToken();
            next = (token_struct*)malloc(anon_heap, sizeof(tok));
            *next = tok;
            tmp.next = next;
            tok = tmp;
            assert(tok.next != NULL);
        }
        return tok.next->en;
}


interface void PushTokenBack(token_enum new_tok, Namedobj* pathobj, str src)
/* Push this token back into the stream so that it appears      */
/* as the current token (and what was the current token becomes */
/* the following token). */
{       token_struct *next;

        next = (token_struct*)malloc(anon_heap, sizeof(tok));
        *next = tok;
        tok.en = new_tok;
        tok.local_obj = NULL;
        tok.stdlib_obj = NULL;
        tok.curdir_obj = NULL;
        tok.pathobj = pathobj;
        if (pathobj)
            strcpy(tok.buf, pathobj->name);
        tok.src = src;
        tok.next = next;
}


bool EndOfParse(void)
/* Returns yes iff the current token is EOF, i.e. we're at the */
/* end of the source text sequence.  Usually means we've       */
/* reached the current line in the editor. */
{
	return (tok.en == EOF);
}



/*--------------------- Error reporting: --------------------*/

static void GetErrorMess(str src, str mess, va_list argptr)
/* Construct an error-reporting line.   Put a strdup'd copy */
/* into 'error.mess'. */
{	char buf[512];
	str start, s;
        int len;

	strcpy(buf, "<<< ");
	vsprintf(buf+4, mess, argptr);
	if (src and src < CurrentSource or src >= CurrentSource + strlen(CurrentSource)) {
	    for (start=src; start[-1] and src - start < 80; start--)
		;
	    strcat(buf, " in macro expansion:  ");
            /* It might be nice to display the expansion in context.
            This would require storing inside each macro expansion string
            (which are allocated via qmalloc()) a pointer to the beginning
            and the end of the macro expression it's replacing. We could
            search backwards for a '\0' char and then subtract 2*sizeof(str)
            to get the address of the two pointers.  Then we would somehow
            indicate which parts are original and which parts are expansion.
            So I haven't bothered as yet. */
            s = buf + strlen(buf);
            len = strlen(start);
            if (len > 160)
                len = 160;
	    memcpy(s, start, len);
            s += len;
            *s++ = ' ';
            *s = '\0';
	    Error.src = CurrentSource;
	    Error.pos = sub_source[0];
	}
	else {
	    Error.src = CurrentSource;
	    Error.pos = src ? src : CurrentSource + strlen(CurrentSource) - 2;
	    if (PredefinitionPhase) {
		strcat(buf, "\n\t");
		strcat(buf, CurrentSource);
	    }
	}
	strcat(buf, " >>>");
	if (Error.message != NULL)
	    free(anon_heap, Error.message);
	Error.message = strdup(anon_heap, buf);

	if (PredefinitionPhase)
	    TfcMessage("Predefinition Error", '!', Error.message);
}


interface void ErrorParse(str src, str mess, ...)
/* Display this parsing error message. */
{       va_list argptr;

	if (Error.err)			// Ignore it because we already have an error.
	    return;                     // (We want of course the first error).
	Error.err = err_certain;
	va_start(argptr, mess);
	GetErrorMess(src, mess, argptr);
	va_end(argptr);
	tok.en = EOF;
}


interface void ErrorType(str src, str mess, ...)
/* We've got an error of types or methods. It might resolve itself */
/* when we run make, so don't report it just yet. */
{       va_list argptr;

	if (Error.err >= err_maybe)     // Ignore it because we already have an error.
	    return;                     // (We want of course the first error).
	Error.err = err_maybe;
	va_start(argptr, mess);
	GetErrorMess(src, mess, argptr);
	va_end(argptr);
}




/*==================== Macros ==================*/

#define MAX_MACPARAMS   20


static void ProcessMacroDefine(void)
/* When we enter this function, the input stream is */
/* at the first character of the macro name.  We    */
/* process it as a macro definition. */
{	str name, value, param[MAX_MACPARAMS];
        char tmp[1024];
	int no_params;
	str d;
	int i;

        tok.curdir_obj = tok.stdlib_obj = NULL;
	d = name = tmp;
	while (isalnum(tok_ch) or tok_ch == '_') {            // Get the name
	    *d++ = tok_ch;
	    nextch();
	}
	*d++ = 0;

	if (tok_ch == '(') {         // Get the parameters
	    nextch();
	    no_params = 0;
	    do {
		while (isspace(tok_ch))
		    nextch();
		if (tok_ch == ')') {
		    nextch();
		    break;
		}
		if (no_params >= MAX_MACPARAMS) {
		    ErrorParse(tok.src, "Too many parameters");
		    return;
		}
		param[no_params++] = d;
		while (isalnum(tok_ch) or tok_ch == '_') {
		    *d++ = tok_ch;
		    nextch();
		}
		*d++ = '\0';
		while (tok_ch == ' ')
		    nextch();
		if (tok_ch == ')') {
		    nextch();
		    break;
		}
		else if (tok_ch == ',') {
		    nextch();
		    continue;
		}
		else {
		    ErrorParse(tok.src, "Funny thing in macro");
		    break;
		}
	    } forever;
	}
	else no_params = -1;    // Signifies a macro without brackets

	while (isspace(tok_ch))          // Get the value
	    nextch();
	value = d;

	while (tok_ch != '\n' and tok_ch != EOFC) {
	    /* Substitute small chars for the params: */
	    if ((isalnum(tok_ch) or tok_ch == '_') and no_params) {
		tok.x = 0;
		while (isalnum(tok_ch) or tok_ch == '_') {
		    tok.buf[tok.x++] = tok_ch;
		    nextch();
		}
		tok.buf[tok.x] = 0;
		for (i=0; i < no_params; i++)
		    if (strcmp(tok.buf,param[i]) == 0)
			goto FOUND;

		/* Not found */
		strcpy(d, tok.buf);
		while (*d)
		    d++;
		continue;

		/* Found: */
		FOUND:              // Make a substitution:
		*d++ = i + 1;       //  The small char
	    }
	    else if (tok_ch == '\t') {
		*d++ = ' ';
		nextch();
	    }
	    else {
		*d++ = tok_ch;
		nextch();
	    }
	}

	*d++ = '\0';

        /* Enter it as a namedobj in the relevant directory: */
	MacroEnter(name, no_params, value);
}


static void ExpandMacro(str value)
/* We've just seen an identifier which expands to macro value 'value'. */
/* The input stream points to the character after the macro name.      */
/* Expand the macro, whether it has arguments or not, and get the      */
/* source stream to point to the expanded value. */
{       char buf1[512], expansion[1024];
	int no_params, bracket_level;
        str param[MAX_MACPARAMS];
	bool non_space_found;
	str s;

	if (*value == EOFC) {           // Macros without parameters or ().
            assert(source[-1] == tok_ch);
            source--;
            strcpy(expansion, value+1);
            s = expansion + strlen(expansion);
	    goto HAVE_EXPANSION;
	}

	while (tok_ch == ' ' or tok_ch == '\t')   // Parse the '('.
	    nextch();
	if (tok_ch != '(') {
	    ErrorParse(tok.src, "Expecting arguments to this macro");
	    return;
	}
	nextch();

        /* Parse the macro parameters: */
	bracket_level = 0;
	no_params = 0;
        s = buf1;
	do {
	    param[no_params++] = s;
	    non_space_found = no;
	    while (tok_ch == ' ' or tok_ch == '\t')
		nextch();
	    until (bracket_level == 0 and (tok_ch == ',' or tok_ch == ')')) {
		if (tok_ch == '(')
		    bracket_level++;
		else if (tok_ch == ')')
		    bracket_level--;
		else if (tok_ch == EOFC)
		    break;
		else if (not isspace(tok_ch))
		    non_space_found = yes;
		*s++ = tok_ch;
		nextch();
	    }
	    *s++ = '\0';
	    if (tok_ch == ',') {
		nextch();
		continue;
	    }
	    else if (tok_ch == ')') {
		if (not non_space_found)
		    no_params--;
		/* Empty () */
		break;
	    }
	    else {
		ErrorParse(tok.src, "Missing macro ')'");
		return;
	    }
	} forever;

	if (no_params != *value++)      // Check the # of parameters
	    ErrorParse(tok.src, "Expecting a different number of parameters.");

        /* Construct the expanded string: */
	s = expansion;
	while (*value) {
	    if (*value < MAX_MACPARAMS) {
		// It's a macro parameter to be substituted.
		strcpy(s, param[*value++ - 1]);
		while (*s)
		    s++;
	    }
	    else if (*value == '#') {
		value++;
		if (*value == '#')
		    // Token pasting. Ignore both #'s.
		    value++;
		else if (*value < MAX_MACPARAMS) {
		    // Stringising operator:
		    str t=param[*value++ - 1];
		    *s++ = '"';
		    while (*t) {
			if (*t == '"')
			    *s++ = '\\';
			*s++ = *t++;
		    }
		    *s++ = '"';
		}
		else *s++ = '#';
	    }
	    else *s++ = *value++;
            if (s >= expansion+sizeof(expansion)) {
                ErrorParse(source, "Macro expansion is too big.");
                return;
            }
	}
	*s++ = '\0';

        /* Set up the expansion in the compiler heap: */
        HAVE_EXPANSION:
        sub_source[ss_idx++] = source;
        source = (str)malloc(anon_heap,s-expansion+2);
        MacroExpansions = (str*)realloc(anon_heap, MacroExpansions,
                        (MI_idx+1)*sizeof(str));        // Our heap
                        // automatically rounds the size up to avoid
                        // overly frequent memcpy's.
        MacroExpansions[MI_idx++] = source;
        *source++ = '\0';
        strcpy(source, expansion);
}


interface void FreeMacroExpansions()
/* Macro expansions are used in error-reporting late in the compile  */
/* algorithm, and they can contain multiple compileables.  Therefore */
/* they must be freed not at the end of a CompileEntity() but at the */
/* end of a Compile(). */
{
        while (MI_idx)
            free(anon_heap, MacroExpansions[--MI_idx]);
        free(anon_heap, MacroExpansions);
        MacroExpansions = NULL;
        MI_idx = 0;
}


static void ProcessPreprocessor(void)
/* Process a '#preprocessor' command. The command is on the */
/* input stream. */
{       enum { pp_unknown, pp_define, pp_undef, pp_include, pp_ifdef,
		    pp_else, pp_elif, pp_endif } pp;
        char buf[512], *d=buf;
        str src;

        /* What preprocessor statement is it? */
        src = source-1;
        while (isalnum(tok_ch)) {
            *d++ = tok_ch;
            nextch();
        }
        *d = '\0';
	if (strieq(buf, "define"))
	    pp = pp_define;
	else if (strieq(buf, "undef"))
	    pp = pp_undef;
	else if (strieq(buf, "include"))
	    pp = pp_include;
	else {
            ErrorParse(src, "Bad preprocessor statement: "
                        "I don't understand \"%s\".",
                        buf);
            return;
        }
	while (isspace(tok_ch))
	    nextch();

	switch (pp) {
	    case pp_define: ProcessMacroDefine();
			    break;

	    case pp_undef:  tok.x = 0;
			    while (isalnum(tok_ch) or tok_ch == '_') {
				tok.buf[tok.x++] = tok_ch;
				nextch();
			    }
			    tok.buf[tok.x++] = '\0';
			    MacroDelete(tok.buf);
			    break;

	    case pp_include:if (tok_ch == '"')
				nextch();
			    tok.x = 0;
			    while (strchr("\" \t\n", tok_ch) == NULL) {
				tok.buf[tok.x++] = tok_ch;
				nextch();
			    }
			    tok.buf[tok.x++] = '\0';
			    while (tok_ch != '\n')
				nextch();
			    /*if (tok.buf[0]) {
				extern bool EditorInclude(str filename);
				EditorInclude(tok.buf);
			    }*/
                            printf("<<<In Barbados, there is no need "
                                "to #include any header files. All the standard "
                                "library functions are automatically available, "
                                "(and to use functions in other directories you "
                                "create a link (a directory reference) to that "
                                "directory).>>>");
			    break;

	    default:        ErrorParse(tok.src, "Unimplemented");
			    break;
	}
}






/*------------- Module Initialisation -------------*/

static struct {
	int num;
	str key;
} keyword_array[] = {
	  kw_type,"auto", kw_bool,"bool", kw_break, "break", kw_case,"case",
          kw_char,"char", kw_const, "const", kw_container, "container",
          kw_class,"class", kw_type,"cdecl", kw_continue, "continue",
	  kw_do,"do", kw_default, "default", kw_delete,"delete",
          kw_double,"double", kw_else,"else", kw_enum,"enum",
	  kw_extern, "extern", kw_for,"for", kw_float,"float",
	  kw_type,"far", kw_friend,"friend",
          kw_goto,"goto", kw_if,"if", kw_inline,"inline",
          kw_int,"int", kw_type,"interrupt", kw_sizeof,"sizeof",
	  kw_long,"long", kw_type,"near", kw_new,"new",
	  kw_operator,"operator", kw_int64,"int64",
	  kw_protected,"protected", kw_public,"public", kw_private,"private",
	  kw_type,"register", kw_return,"return", kw_switch,"switch",
	  kw_struct,"struct", kw_short,"short", kw_static,"static",
	  kw_type,"signed", kw_typedef,"typedef", kw_union,"union",
	  kw_unsigned,"unsigned", kw_virtual,"virtual", kw_void,"void",
	  kw_volatile,"volatile", kw_while,"while", kw_typeof,"typeof",
	  kw_true,"true", kw_false,"false",
	  0,NULL
};


interface void ConstructKeywordsInStdlib(Heap *Stdlib)
{       IntNamedobj *obj;

	for (int i=0; keyword_array[i].num; i++) {
            obj = (IntNamedobj*)malloc(Stdlib, sizeof(IntNamedobj));
            clearS(*obj);
            obj->name = keyword_array[i].key;
            obj->storage = keyword_storage;
            obj->constval = keyword_array[i].num;
	    stdlib_dir->Insert(obj);
	}
}


interface void LexicalInit(void)
/* Initialises this module. */
{
	tok.en = EOF;
}


interface void LexicalClose(void)
{
}

