/********************************************************\
** COMPPARS.CPP:					**
**          The parser module.  Takes a stream of	**
** tokens and does a recursive parsing of it with the	**
** result being an expression-tree.  This tree is	**
** then passed to COMPEXPR.CPP for the next phase.	**
\********************************************************/


#include <string.h>
#include <stdlib.h>
#include "barbados.h"
#include "modules.h"
#include "memory.h"
#include "runlib.h"
#include "source.h"
#include "compiler.h"
#include "make.h"
#include "name.h"


#define DEFAULT_TAG     -(1<<31)

Type debug_type;
GotoStatement* debugS;


interface FunctionNamedobj* MainEntity; // 'Make' for the entered expr.
interface Classdef* this_class;	        // A member function's class
interface int parameter_location;       // Offset of the next parameter
interface int InsideCompiler;		// Are we compiling stuff?
interface Type expression_type;         // The top-level type
interface int debugging;
interface FunctionNamedobj* CompilerCurrentFuncObj;
interface int CompilerVersion=202;

static Expr error_expr;
static Statement* DestructorS;
static Expr* decl_init_root;
static Type typeof_string;
static Type return_type;
static int size;
static Statement* ParseThing();
static Expr* ParseFunctionCall(Expr* a);
static void DecorateEntity(Statement *S, bool NeedReturn);







/*==================== Quick Malloc =====================*/

/* This sub-module implements an optimised 'malloc' module.  */
/* Memory is allocated with successive calls to 'qmalloc()', */
/* but is only freed with a call to 'qfreeall()' which resets*/
/* all the memory this module allocates. */


#define Q_SIZE      16384

typedef struct qnode {
	struct qnode *next;
	char buf[Q_SIZE];
} *qtype;


/*static*/ qtype qn=NULL;
/*static*/ int qidx;


interface void qinit(void)
/* Initialise the quick-heap. */
{
	//assert(qn == NULL);
	qn = (qnode*)anon_heap->malloc(sizeof(qnode));
	qn->next = NULL;
	qidx = 0;
	IprRegisterStatement(NULL);
}


interface bool qptrcheck(void* p)
/* Check that this is a valid quick-heap pointer. */
{       qtype q;

	for (q=qn; q; q=q->next)
	    if ((qtype)p >= q and (qtype)p < q+1)
		return yes;
	return no;
}


interface void* qmalloc(int n)
/* Allocates 'n' bytes from the quick-heap. */
{       qtype qn2;
	void *p;

        /* We prefer multiples of 4 bytes: */
        n = ((n-1)|3)+1;

	/* Can we fit it into this qnode? */
	if (qidx + n < Q_SIZE) {
	    p = qn->buf + qidx;
	    qidx += n;
	    return p;
	}

	/* Allocate a new qnode: */
	qn2 = new(qnode);
	if (qn2 == NULL) {
	    ErrorParse(NULL, "Out of memory");
	    assert(false);
	    return malloc(anon_heap, n);
	}
	qn2->next = qn;
	qn = qn2;
	assert(n < Q_SIZE);
	qidx = n;
	return qn->buf;
}


interface void* qcalloc(int n)
/* Allocates and clears 'n' bytes from the quick-heap. */
{
	return memset(qmalloc(n), 0, n);
}


static void qfreeall(void)
{       qtype qn2;

	while (qn) {
	    qn2 = qn->next;
	    free(anon_heap, qn);
	    qn = qn2;
	}
	qn = NULL;
}






/*================ Creating Program Trees: ==============*/


interface void ExprTreeClear(void)
{
	error_expr.type = err_typstr;
	error_expr.ref = 0;
	error_expr.o = oUnknown;
}


interface Expr* NewExpr(int opr, Expr* _left, Expr* _right)
{       Expr* r;

	r = (Expr*)qcalloc(sizeof(Expr));
	r->opr = (char)opr;
	r->right = _right;
	r->left = _left;
	return r;
}


interface Expr* ErrorExpr(void)
/* Create an expression corresponding to an erroneous expression. */
{
	return &error_expr;
}


interface Expr* PushSPPlusExpr(int x, int typesize)
/* Create a kPUSHSPPLUS expression. */
{	Expr* e;

	e = NewExprInt(x);
        e->o = oPUSHSPPLUS;
        e->src = "%bigpush%";
        e->right = (Expr*)typesize;
        return e;
}


interface Expr* NewExprFloat(double f)
/* Create an expression for this 'float'. */
{       Expr* r;

	r = NewExpr(float_const, 0, 0);
	r->u.f = f;
	return r;
}


interface Expr* NewExprInt(int i)
/* Create a constant-integer expression. */
{       Expr* r;

	r = NewExpr(int_const, (Expr*)i, 0);
	r->o = oConstant;
	r->tp = tp_int;
	r->type = int_typstr;
	return r;
}


interface Expr* NewExprVoidptr(void *v)
/* Create a constant void-pointer expression. */
{       Expr* r;

	r = NewExpr(int_const, (Expr*)v, 0);
	r->o = oConstant;
	r->tp = tp_pointer;
	r->type = voidptr_typstr;
	return r;
}


interface Expr* NewExprIdentifier(Resolved_node *res)
/* Returns a decorated expression representing a possibly */
/* overloaded function. */
{       Expr* r;

	r = (Expr*)qcalloc(sizeof(Expr));
	r->opr = identifier;
        r->u.idfier.x = NULL;
        r->u.idfier.resolve = (Resolved_node*)memcpy(
        	qmalloc(sizeof(*res)), res, sizeof(*res));
        return r;
}


interface Expr* NewExprIdentifier(Namedobj *obj)
/* Returns a decorated expression representing this specific obj. */
/* e->u.idfier.resolve == NULL. */
{	Expr* r;

        r = NewExpr(identifier, NULL, NULL);
        r->type = obj->type;
        if (obj->storage == static_storage) {
            r->ref = 1;
            r->o = oConstant;
            r->tp = tp_pointer;
            r->u.idfier.x = ((StaticNamedobj*)obj)->location;
        }
        else if (obj->storage == auto_storage or obj->storage == parameter_storage) {
            r->ref = 1;
            r->o = oLocalVar;
            r->tp = tp_pointer;
            r->u.obj = (AutoNamedobj*)obj;
        }
        else if (obj->storage == straight_fn or obj->storage == member_fn) {
            r->ref = 0;
            r->o = oConstant;
            r->tp = tp_pointer;
            r->u.idfier.x = ((FunctionNamedobj*)obj)->u.fn;
            r->u.idfier.resolve = (Resolved_node*)qmalloc(sizeof(Resolved_node));
            r->u.idfier.resolve->obj = obj;
            r->u.idfier.resolve->offset = 0;
            r->u.idfier.resolve->list = NULL;
        }
        else assert(false);
        return r;
}


interface Expr* NewExprIdentifier(AutoNamedobj *obj)
/* Return a decorated expression representing a local variable. */
{       Expr* e;

        e = NewExpr(identifier, NULL, NULL);
        e->o = oLocalVar;
        e->u.obj = obj;
        e->right = 0;
        e->ref = 1;
        e->type = obj->type;
        e->tp = tp_pointer;
        return e;
}


interface Expr* NewExprVirtualFnTable(void** VirtualFns)
/* Create a constant-virtual-table expression. */
{       Expr* r;

	r = NewExpr(int_const, (Expr*)VirtualFns, 0);
	r->o = oConstant;
	r->tp = tp_pointer;
	r->type = virtualfntable_typstr;
	return r;
}


interface void* qmallocCopy(void *source, int len)
/* Do a 'memcpy' of this sequence into the q-heap. */
{
	return memcpy(qmalloc(len), source, len);
}


/*interface Expr* NewExprContainer(container_id cid)
// Create a container constant value. 
{	Expr* r;

	r = NewExpr(kw_container, *(Expr**)&cid, 0);
	return r;
}*/


interface Expr* NewExprTernary(Expr* a, Expr* b, Expr* c)
/* Generate the ternary operator with these three operands. */
{       Expr* r;

	r = (Expr*)qmalloc(sizeof(Expr) + sizeof(str));
	r->type = NULL;
	r->ref = 0;
	r->o = oUnknown;
	r->tp = (tp_enum)0;
	r->opr = op_sing('?');
	r->left = a;
	r->right = NewExpr(op_sing(':'), b, c);
	r->src = a->src;
	return r;
}


interface Expr* NewExprFn(Expr* func, Expr **parameters, int arity)
/* Creates a new node for function calls. */
{       exprf_type ef;
	Expr* r;

	r = (Expr*)qmalloc(sizeof(Expr) +
		    sizeof(struct exprf_node) +
		    (arity+1) * sizeof(Expr*));
		    // Leave space in ef->param for us to insert a 'this' parameter.
	r->type = NULL;
	r->ref = 0;
	r->o = oFUNC;
	r->tp = (tp_enum)0;
	r->opr = op_funccall;
	r->src = func->src;
	ef = (exprf_type)(r+1);
	ef->func = func;
	ef->owner = NULL;
	ef->arity = arity;
        ef->flags = 0;
	memcpy(&ef->param, parameters, arity * sizeof(Expr*));
	r->u.func = ef;
	return r;
}


interface Expr* NewEnlargedExprFn(Expr *e, int arity)
{       exprf_node *old, *ef;

        old = e->u.func;
        assert(arity > old->arity);
	ef = (exprf_node*)qmalloc(
		    sizeof(struct exprf_node) +
		    (arity+1) * sizeof(Expr*));
        *ef = *old;
        memcpy(ef->param, old->param, old->arity*sizeof(Expr*));
        memset(ef->param+old->arity, 0, (arity-old->arity)*sizeof(Expr*));
        ef->arity = arity;
        e->u.func = ef;
        return e;
}


interface void* NewStatement(token_enum kw)
/* Creates an object for statements of this kind. */
{       Statement *S;
	int size;

	switch (kw) {
	    case kw_for:    size = sizeof(ForStatement);
			    break;
	    case kw_while:  size = sizeof(WhileStatement);
			    break;
	    case kw_do:     size = sizeof(DoStatement);
			    break;
	    case kws_instruction:
	    case kw_case:   size = sizeof(CaseStatement);
			    break;
	    case kw_break:
	    case kw_continue:
	    case kw_goto:   size = sizeof(GotoStatement);
			    break;
	    default:        size = sizeof(Statement);
			    break;
	}

	S = (Statement*)qcalloc(size);
	S->st = (st_enum)kw;
	S->next = NULL;
        S->scope = (AutoNamedobj*)NameMarkScope();
	return S;
}


interface void* NewStatement(st_enum st)
/* Creates an object for statements of this kind. */
{       Statement *S;
	int size;

	switch (st) {
	    case st_if:     size = sizeof(IfStatement);
			    break;
	    case st_switch: size = sizeof(SwitchStatement);
			    break;
	    case st_return: size = sizeof(ReturnStatement);
			    break;
	    case st_expr:   size = sizeof(ExprStatement);
			    break;
	    case st_null:   size = sizeof(Statement);
			    break;
	    default:        size = sizeof(Statement);
			    break;
	}

	S = (Statement*)qcalloc(size);
	S->st = st;
	S->next = NULL;
	if (st == st_if or st == st_expr or st == st_return or st == st_null
		or st == st_switch)
	    IprRegisterStatement(S);
        S->scope = (AutoNamedobj*)NameMarkScope();
        assert(S->scope == NULL or S->scope->storage == auto_storage or
                                S->scope->storage == parameter_storage or
                                S->scope->storage == local_static or
                                S->scope->storage == static_storage // <-- during debugging
                 );
	return S;
}




/*================ Operator-precedence parser ===============*/

typedef enum { unknown_op,
	    comma_op, conditional_right, assignment_op, conditional_left,
	    logical_or, logical_and, bitwise_or, bitwise_xor, bitwise_and,
	    equality_op, relational_op, shift_op, addition_op,
	    multiplication_op, unary_op, highest } precedence_type;

static Expr* ParEx(precedence_type UpTo);




static precedence_type GetPrecedence(token_enum token)
{
	switch (token) {
	    case op_sing('~'):
	    case op_sing('!'):
	    case op_doub('+'):
	    case op_doub('-'):  return unary_op;
	    case op_sing('*'):
	    case op_sing('/'):
	    case op_sing('%'):  return multiplication_op;
	    case op_sing('+'):
	    case op_sing('-'):  return addition_op;
	    case op_doub('>'):
	    case op_doub('<'):  return shift_op;
	    case op_assg('<'):
	    case op_assg('>'):
	    case op_sing('<'):
	    case op_sing('>'):  return relational_op;
	    case op_doub('='):
	    case op_assg('='):
	    case op_assg('!'):  return equality_op;
	    case op_sing('|'):  return bitwise_or;
	    case op_sing('&'):  return bitwise_and;
	    case op_sing('^'):  return bitwise_xor;
	    case op_doub('&'):  return logical_and;
	    case op_doub('|'):  return logical_or;
	    case op_sing('='):
	    case op_assg('+'):
	    case op_assg('-'):
	    case op_assg('*'):
	    case op_assg('/'):
	    case op_assg('%'):
	    case op_shft('<'):
	    case op_shft('>'):
	    case op_assg('|'):
	    case op_assg('&'):
	    case op_assg('^'):  return assignment_op;
	    case colon       :  return unknown_op;      // (Could be in a case xxx:  statement)
	    case ternary     :  return conditional_left;
	    case comma:         return comma_op;

	    default:            return unknown_op;  /* e.g. ';'. */
	}
}




static Expr* NewStringOrTs(Type type, void *s, int len, bool store_in_code)
/* Allocate space either in the 'shell memory' or in the 'code local-static' */
/* section for this sequence of length 'len'. Copy the sequence into this    */
/* place. Return an expression containing it.                                */
/* (And if it's a typestring, find any pointers within and add them to the   */
/* fn's pointer-list). */
{       int offset;
	Expr* r;
	str m;

	if (store_in_code) {
	    /* Put it in the local static area.    */
	    offset = AllocateAndCopyLocalStatic(len, s, type);
	    r = NewExpr(string_const, (Expr*)offset, (Expr*)localstatic_base);
	}
	else {
	    /* Put it in the 'shell memory'. */
	    m = (str)malloc(anon_heap, len);
	    memcpy(m, s, len);
	    r = NewExpr(string_const, (Expr*)m, NULL);
	}
	r->o = oConstant;
	r->tp = tp_pointer;
	r->type = type;
	return r;
}



interface Expr* ExprCopyOfTypestring(Type type, bool store_in_code)
/* Returns an expression corresponding to a duplicate of */
/* this type, allocated in the static area or in the     */
/* shell memory.  Use the static area if you want it to  */
/* last as long as the code, use the shell memory if you */
/* want it to last as long as the shell. */
{	int len;

	len = LengthOfTypeString(type);
	return NewStringOrTs(typetype_typstr, type, len, store_in_code);
}



static Expr* ParseStringConst(void)
{       int len, len2;
	Expr* r;
	str s;

	/* Get the whole str, even if it's in seperate tokens. */
	len = tok.x + 1;
	s = (str)memcpy(malloc(anon_heap, len),tok.buf,len);
	NextToken();
	while (tok.en == string_const) {
	    len2 = len;
	    s = (str)realloc(anon_heap, s, len += tok.x);
	    strcpy(s+len2-1, tok.buf);
	    NextToken();
	}

	/* Move it to the final destination: */
	r = NewStringOrTs(string_typstr, s, len, (default_storage == auto_storage));
	// Store the str inside the code iff we're inside a function.
	// We know we're inside a function iff default_storage == auto_storge.
	free(anon_heap, s);

	return r;
}


interface Namedobj* ParsePath(void)
/* Parse a single identifier or a path-name, and return the   */
/* object denoted.                                            */
/* If multiple objects are denoted, we return them via the    */
/* global 'ResolveInfo'. */
{	static container_id RootPid=1;
	container_id path_pid;
	Namedobj *obj, *obj1;
	token_enum lookahead;
        StaticNamedobj *sobj;
        IntNamedobj *iobj;
	Directory* dir;


	/*** A shortcut for simple identifiers: ***/
	if (tok.en == identifier) {
            obj = ResolveTok(declarator_class);
	    if (obj == NULL) {
		ErrorParse(tok.src, "Undefined identifier: %s", tok.buf);
		return NULL;
	    }
	    NextToken();
	    if (tok.en != op_sing('/') and tok.en != double_colon)
		return obj;

	    if (not TypeEqual(obj->type, direc_typstr) and
		not TypeEqual(obj->type, container_typstr) and
		tok.en != double_colon)
		return obj;

	    path_pid = 0L;
	    goto HAVE_OBJ;
	}


	/*** The path must start with  /  or .. . */
	dir = curdir;
	path_pid = 0L;

	do {
	    /* At this point, we are relying on 'dir' to have */
	    /* a valid directory value corresponding to the   */
	    /* current path-name position. */

	    switch (tok.en) {
		case dot:   NextToken();
			    obj = sobj = (StaticNamedobj*)qmalloc(
                                SizeofObj("...", direc_typstr, static_storage));
			    clearS(*sobj);
			    sobj->storage = static_storage;
			    if (tok.en != dot) {       // Single dot
				sobj->location = dir;
				sobj->type = direc_typstr;
				sobj->name = ".";
				break;
			    }
			    NextToken();              // Double dot
			    if (dir->parent) {
				sobj->location = dir->parent;
				sobj->type = direc_typstr;
				sobj->name = "..";
			    }
			    else if (dir->parent_cid) {
				sobj->location = &dir->parent_cid;
				sobj->type = container_typstr;
				sobj->name = "..";
			    }
			    else {
				ErrorParse(tok.src, "You're at the root.");
				return NULL;
			    }
			    break;

		case op_sing('/'):
			    lookahead = LookAhead();
			    if (lookahead != identifier and lookahead != dot)
				NextToken();
			    iobj = (IntNamedobj*)qmalloc(SizeofObj("/", container_typstr, const_storage));
			    clearS(*iobj);
			    iobj->type = container_typstr;
			    iobj->storage = const_storage;
			    iobj->constval = RootPid;
			    iobj->name = "/";
                            obj = iobj;
			    break;

		case identifier:
			    obj = dir->Find(tok.buf);
			    if (obj) {
                            	if (obj->type == ambiguous_typstr)
                                    ErrorParse(tok.src, "You can't use member \"%s\" "
                                    	"because it's ambiguous: I don't know which "
                                        "base class to take it from. You need to "
                                        "disambiguate it using the :: notation.",
                                        obj->name);
				NextToken();
				goto HAVE_OBJ;
			    }
			    ErrorParse(tok.src, "Element %s not found in path name",
					tok.buf);
			    return NULL;

		default:    ErrorParse(tok.src, "Syntax error in path name");
			    return NULL;
	    }

	    HAVE_OBJ:

	    /* The scope-resolution operator: */
	    if (tok.en == double_colon) {
		if (obj->storage != typedef_storage or *obj->type != tp_class) {
		    ErrorParse(tok.src, "%s is not a class name.", obj->name);
		    return NULL;
		}
		str src=tok.src;
                if (LookAhead() == op_sing('*'))
                    break;
		Gobble(double_colon);
		Classdef* classdef;
		classdef = *(Classdef**)(obj->type+1);
                if (streq(classdef->typedef_obj->name, tok.buf)) {
                    for (obj=classdef->member; obj; obj=obj->next) {
                    	if (IsConstructor(obj))
                            break;
                    }
                }
                else obj = ResolveMember(classdef, tok.buf, public_visibility);
                if (obj == NULL) {
                    ErrorParse(tok.src, "There is no member of %s called '%s'",
                            classdef->typedef_obj ? classdef->typedef_obj->name : "<unnamed>",
                            tok.buf);
                    return NULL;
                }
		NextToken();
		if (tok.en == double_colon)
		    continue;
		else break;
	    }

	    /* Is it the end of path name? */
	    if (tok.en != op_sing('/'))
		break;

	    /* The path name continues, so we'd better get */
	    /* a valid directory pointer: */
	    if (TypeEqual(obj->type, direc_typstr)) {
		if (obj->storage != static_storage) {
		    ERROR:
		    ErrorParse(tok.src, "Path names must have constant values");
		    return NULL;
		}
		dir = (Directory*)((StaticNamedobj*)obj)->location;
		Gobble(op_sing('/'));
		continue;
	    }
	    else if (TypeEqual(obj->type, dirref_typstr)) {
		if (obj->storage != static_storage)
		    goto ERROR;
		dir = *(Directory**)((StaticNamedobj*)obj)->location;
		Gobble(op_sing('/'));
		continue;
	    }
	    else if (TypeEqual(obj->type, container_typstr)) {
		container_id new_pid;
		if (obj->storage == static_storage)
		    new_pid = *(container_id*)((StaticNamedobj*)obj)->location;
		else if (obj->storage == const_storage)
		    new_pid = (container_id)((IntNamedobj*)obj)->constval;
		else goto ERROR;
		/* Move the path cursor from 'path_pid' to 'new_pid'. */
		if (path_pid)
		    CloseContainer(path_pid);
		dir = OpenContainer(new_pid, READONLY);
		path_pid = new_pid;
		Gobble(op_sing('/'));
		continue;
	    }
	    else break;
		// We might have something like "x / y", meaning division.
	} forever;

	/* End of the path: */
	if (path_pid) {
	    if (path_pid != curconim->cid)
		curconim->PublicLink(Conim::FindConim(path_pid));
	    CloseContainer(path_pid);
	    // The 'CloseContainer()' is to match with the 'OpenContainer()'.
	    // This pair of brackets is to ensure that the container will stay in memory.
	}
        if (obj == NULL)
            return NULL;
        ResolveInfo.list = NULL;
        if (obj->type == NULL or obj->type[0] != tp_function)
            return ResolveInfo.obj = obj;
        obj1 = obj;
        do {
            obj = obj->next;
            if (obj == NULL)
            	break;
            if (streq(obj->name, obj1->name))
            	Array_Add(ResolveInfo.list, obj);
        } forever;
        if (ResolveInfo.list) {
            Array_Add(ResolveInfo.list, obj1);
            ResolveInfo.list = qcopyobjlist(ResolveInfo.list);
        }
	return ResolveInfo.obj = obj1;
}


interface bool AtTypeExpression(void)
/* Returns true if we are at the start of a type expression, */
/* e.g. tok.en == kw_int.   */
{
	if (tok.en >= kw_declare and tok.en <= kw_type)
	    return yes;
	/*if (tok.en == identifier) {
            Namedobj *obj = ResolveTok(declarator_class);
	    if (obj and obj->storage == typedef_storage)
		return yes;
        }*/
        else if (tok.en == identifier or tok.en == op_sing('/') or tok.en == dot) {
            Namedobj* obj;
            str src = tok.src;
            obj = ParsePath();
            PushTokenBack(kws_identifier_path, obj, src);
            return obj and obj->storage == typedef_storage;
        }
	return no;
}


static Type ParseExprOrTypedef(void)
/* Parse an expression or typedef object so that we can pass the */
/* type back to 'sizeof()' or 'typeof()'. */
{       Namedobj* obj;
	Expr* e;

	/* Do we have a typedef type or a fundamental type (e.g. "int")? */
	if (AtTypeExpression())
	    return ParseTypeExpression();

	/* Does the expression begin with an identifier or path? */
	if (tok.en == identifier or tok.en == op_sing('/') or tok.en == dot) {
	    str src=tok.src;
	    obj = ParsePath();
	    if (obj == NULL)
		return ErrorExpr()->type;
	    if (tok.en == close_round)             // Simple objects (only '/'s)
		return obj->type;
	    if (obj->storage == typedef_storage)
		return obj->type;               // Will probably cause syntax err
	    PushTokenBack(kws_identifier_path, obj, src);
	}

	/* Parse an expression: */
	e = ParEx(unknown_op);
	/*DecorateExpr(e);*/
	return e->type;
}


static Expr* NewExprLibraryFunc(str funcname)
/* Look up a single, unoverloaded function of this name in Stdlib, */
/* and return it as a simple decorated Expr*. */
{	Resolved_node Res;
        Expr* func;

	Res.obj = Stdlib->directory()->Find(funcname);
        assert(Res.obj);
        Res.list = NULL;
	func = NewExprIdentifier(&Res);
	DecorateExpr(func);
        return func;
}


static Expr* ParseNew()
/* We have reached the keyword 'new'; parse a 'new' expression. */
{	FunctionNamedobj *malloc_obj, *constructor;
        Expr *e, *func, *params[4], *dim_e, *e2;
	str src, srcnew=tok.src;
	int len, dim, dimension;
        Classdef* classdef;
        Type typewithptr;
        uchar type[128];

        /* Gobble the 'new' keyword: */
        Gobble(kw_new);

        /* What type do they want? */
        src = tok.src;
	ParseBaseType(type);

        /* Arrays are dealt with in a special way: */
        if (tok.en == open_square) {
            classdef = TypeToClassdef(type);
            NextToken();
            dim_e = ParseIntExpr(no);	// The 1st dimension can be variable
            if (tok.en != close_square)
            	ErrorParse(tok.src, "Expecting a close-square (']') here");
            NextToken();
            dimension = 1;
            while (tok.en == open_square) {
                NextToken();
                dim = ParseAndEvaluateIntExpr(no);
                if (tok.en != close_square)
                    ErrorParse(tok.src, "Expecting a close-square (']') here");
                NextToken();
                len = LengthOfTypeString(type);
                memmove(type+5,type,len);
                type[0] = tp_array;
                *(int*)(type+1) = dim;
                dimension *= dim;
            }
            params[0] = NewExprVoidptr(classdef ? classdef : (void*)*type);
            params[0]->src = src;
            if (dimension == 1)
            	params[1] = dim_e;
            else params[1] = NewExpr(op_sing('*'), dim_e, NewExprInt(dimension));
            DecorateExpr(params[1]);
            if (classdef) {
            	Namedobj *constructor2;
            	constructor = ClassDefaultConstructor(classdef, &constructor2);
                if (constructor2 and not constructor)
                    ErrorParse(src, "You need a default constructor for \"%s\""
                    	" if you're going to create arrays of it.",
                        classdef->typedef_obj->name);
            }
            else constructor = NULL;
            if (constructor) {
            	MakeDepends(constructor->make);
            	params[2] = NewExprVoidptr(constructor->u.fn);
            }
            else params[2] = NewExprVoidptr(NULL);
            func = NewExprLibraryFunc("GenericArrayConstructor2");
            func->src = srcnew;
            e = NewExprFn(func,params,3);
            e->src = srcnew;
            e->u.func->stack_size = 12;
            e->type = (Type)qmalloc(len=LengthOfTypeString(type) + 1);
            memcpy(e->type + 1, type, len);
            *e->type = tp_pointer;
            e->tp = tp_pointer;
            typewithptr = e->type;
            DecorateExpr(e);
            return e;
        }

        /* Set up the call to 'malloc()': */
	params[0] = NewStringOrTs(typetype_typstr, type, LengthOfTypeString(type), yes);
        params[0]->src = src;
	DecorateExpr(params[0]);
	malloc_obj = (FunctionNamedobj*)Resolve("operator new");
	assert(malloc_obj);
	func = NewExprIdentifier(&ResolveInfo);
        func->src = srcnew;
	DecorateExpr(func);
	e = NewExprFn(func,params,1);
        e->src = srcnew;
	e->u.func->stack_size = 4;
	e->type = (Type)qmalloc(len=LengthOfTypeString(type) + 1);
	memcpy(e->type + 1, type, len);
	*e->type = tp_pointer;
	e->tp = tp_pointer;
        typewithptr = e->type;

	/* Now, do we need a constructor? */
	if (TypeHasConstructor(type)) {
            e->ref++;
            e2 = e;
            e = ParseConstructorExpr(type, e);
            if (e == NULL)
            	return ErrorExpr();
            e->type = typewithptr;
            e->tp = tp_pointer;
            assert(e->o == oFUNC);
            if (e == e2)
                e->ref--;  // This means that the only 'constructor' was the
                // vftable ptr
            else e->u.func->flags |= 1;	// This signifies to pop
            // the 1st parameter into AX after the call.  We use
            // it to send the address of the object to the caller
            // of the 'new' operator.  This way we avoid the need
            // to set up temporary variables.
        }
        else if (tok.en == open_round) {
            char buf[50];
            ErrorType(tok.src, "There is no constructor for: %s",
            		TypeToString(type,buf,sizeof(buf)));
        }
	return e;
}


interface Expr* DestructorCall(Type type, Expr* ptre,
                        Expr* calle, bool array_delete)
/* There's no parsing required here.  Just take 'ptre' and if necessary,*/
/* create a destructor call expression for it.  'ptre' represents a ptr */
/* to the object.  If 'array_delete', then it means we've received a    */
/* 'delete[] x' expression that requires a run-time identification of   */
/* the size of the array.                                               */
/* 'calle' represents the 'delete' call if that's where we're coming    */
/* from, or NULL if it's a variable going out-of-scope.  It returns     */
/* a new 'calle' if we're coming from a 'delete' or the destructor call */
/* if it's a variable going out-of-scope. */
{       Expr *func, *dcall, *params[4];
        FunctionNamedobj *destructor;
        Classdef* classdef;
        str src=tok.src;
        int dimension;
        tp_enum tp;

        do {
            tp = (tp_enum)*type++;
            if (tp == tp_const)
                continue;
            else if (tp == tp_class) {
                classdef = *(Classdef**)type;
                destructor = (FunctionNamedobj*)ResolveMember(classdef, "~", private_visibility);
                if (destructor == NULL)
                    break;
                if (array_delete)
                    goto ARRAY_DELETE;

                /* We need to call the destructor: */
                func = NewExprIdentifier(&ResolveInfo);
                func->src = src;
                func = NewExprFn(func, NULL, 0);
                func->src = src;
                dcall = NewExpr(arrowtok, ptre, func);
                dcall->src = tok.src;
                if (calle) {
                    calle->u.func->param[0] = ExprRvalue(ExprDuplicate(ptre));
                    // 'ptre' is used first by the destructor and second by the
                    // 'free'. So the destructor gets the original and the 'free'
                    // gets the ExprDuplicate().
                    DecorateExpr(dcall);
                    calle = NewExpr(comma, dcall, calle);
                }
                else {
                    DecorateExpr(dcall);
                }
                break;
            }
            else if (tp == tp_array) {
            	dimension = 1;
                type--;
                while (*type == tp_array) {
                    type++;
                    dimension *= *((int*&)type)++;
                    while (*type == tp_const)
                    	type++;
                }
                if (*type == tp_class) {
                    classdef = *(Classdef**)++type;
                    destructor = (FunctionNamedobj*)ResolveMember(classdef, "~", private_visibility);
                    if (destructor == NULL)
                        break;

                    // Get ArrayDestructor address as a decorated expr:
                    ARRAY_DELETE:
                    func = NewExprLibraryFunc("GenericArrayDestructor");

                    // Get the 3 parameters to GenericArrayDestructor:
                    params[0] = ptre;
                    if (calle)
                        calle->u.func->param[0] = ptre;
                    params[1] = NewExprVoidptr(destructor->u.fn);

                    // Get the function call expression:
                    dcall = NewExprFn(func, params, 2);
                    DecorateExpr(dcall);
                    /*e = NewExpr(comma, dcall, e);*/
                    // The GenericArrayDestructor does the 'free', so we can
                    // just ignore the 'e' expression.
                    return dcall;
                }
                else break;
            }
            else break;
        } forever;

	return calle ? calle : dcall;
}


static Expr* ParseDelete()
/* We have reached the keyword 'delete'; now parse a 'delete' expression. */
{	Expr *e, *func, *ptre;
        Namedobj *free_obj;
	bool array_delete;
	char buf[512];
        Type type;

        /* Gobble the 'delete' keyword: */
        Gobble(kw_delete);
        array_delete = (tok.en == open_square);
        if (array_delete) {
            NextToken();
            if (tok.en != close_square) {
            	ErrorParse(tok.src, "The correct syntax is simply: \"delete [] X\"");
                return ErrorExpr();
            }
            Gobble(close_square);
        }

        /* Get the pointer expression: */
	ptre = ExprGetRvalue(ParseExpr(yes));
	DecorateExpr(ptre);
	while (ptre->type[0] == tp_const)
	    ptre->type++;
	if (ptre->type[0] != tp_pointer)
	    ErrorType(ptre->src, "The 'delete' operator requires a pointer expression. "
		"Your expression is a \"%s\".",
		TypeToString(ptre->type, buf, sizeof(buf)));

        /* Set up the call to 'free()': */
	free_obj = Resolve("operator delete");
	assert(free_obj);
	func = NewExprIdentifier(&ResolveInfo);
	DecorateExpr(func);
	e = NewExprFn(func,&ptre,1);
	e->u.func->stack_size = 4;
	e->type = void_typstr;
	e->tp = tp_void;

	/* Is there a destructor to call? */
        type = ptre->type+1;
        if (TypeHasDestructor(type))
            e = DestructorCall(type, ptre, e, array_delete);
        return e;
}


#if 0
static Statement* PopScopeDestructors(Namedobj* scope)
{       Expr *e, *es=NULL;
        ExprStatement* S;
        Namedobj *obj;

        /* Create any destructor calls that are needed: */
        for (obj=NameMarkScope(); obj != scope; obj=obj->next) {

            /* Is there a destructor? */
            if (TypeHasDestructor(obj->type)) {
                e = NewExprIdentifier(obj);
                e->src = tok.src;
                //Instead of this:  e = NewExpr(op_unry('&'), e, NULL);
                //We do:
                DecorateExpr(e);
                e->ref--;
                e->type = GetPointerToType(e->type);
                e->tp = tp_pointer;
                //
                es = DestructorCall(obj->type, e, es, no);
            }
        }

        /* If there were any destructor calls, then create the relevant */
        /* statement. */
        if (es == NULL)
            return NULL;
        S = (ExprStatement*)NewStatement(st_expr);
        S->e = es;
        S->src = tok.src;
        return (Statement*)S;
}
#endif


static Expr* ParseFunctionCast(void)
/* Set up a cast, given something of the form:  type(params). */
{	uchar type_buf[512];
	Type type;
	Expr* a;

	type = ParseBaseType((Type)type_buf);
        if (tok.en == open_round)
            NextToken();
        else {
            ErrorParse(tok.src, "This looks like a declaration but I thought "
                "we were inside an expression. Check that this expression "
                "starts with an appropriate declaration keyword/identifier.");
            return ErrorExpr();
        }
	a = ParEx(unknown_op);
	Gobble(close_round);
	type = (Type)qmallocCopy((Type)type_buf, type-type_buf);
	return NewExpr(kw_typedef, (Expr*)type, a);
}


static Expr* ParEx(precedence_type UpTo)
/* Compile an expression. Stop when we reach an operator   */
/* of precedence 'UpTo' or less, and then leave that token */
/* in the input stream. */
{	static precedence_type prec;
        visibility_enum visibility;
	str src=tok.src, src2;
	Classdef* classdef;
	token_enum this_op;
	Namedobj* obj;
        char buf[512];
        Expr *a, *b;

	/*************** Determine the first operand.  **********************/
	switch (tok.en) {
	    case identifier:
		obj = ParsePath();

		HAVE_OBJS:
		if (obj == NULL) {
		    ErrorParse(tok.src, "Undeclared variable: %s", tok.buf);
		    a = ErrorExpr();
		    NextToken();
		    break;
		}
		else if (obj->storage == typedef_storage)
		    goto CAST_FUNCTION;
                else if (obj->storage == member_storage and declarator_class == NULL)
                    ErrorType(src, "%s is a data member, but you don't have anything "
			"for it to be a member of.", obj->name);
		a = NewExprIdentifier(&ResolveInfo);
		a->src = src;
		break;

	    case double_colon:
		Gobble(tok.en);
		if (tok.en != identifier) {
		    ErrorParse(tok.src, "Need identifier");
		    break;
		}
		obj = ResolveGlobal(tok.buf);
		NextToken();
		goto HAVE_OBJS;

	    case kws_identifier_path:
		/* This is for path names occurring at the start of a line */
		/* which have already been parsed into PathEntities.       */
                obj = tok.pathobj;
		NextToken();
                ResolveInfo.obj = obj;
                goto HAVE_OBJS;

	    case op_sing('/'):
	    case dot:
		a = NewExpr(identifier, NULL, NULL);
		obj = ParsePath();
                goto HAVE_OBJS;

	    case kw_char: case kw_long: case kw_int: case kw_bool:
	    case kw_float: case kw_double: case kw_container:
		CAST_FUNCTION:
		a = ParseFunctionCast();
		break;

	    case kw_operator:
            	ParseOperatorName(buf);
		obj = Resolve(buf);
                if (*ResolveErrormsg) {
                    ErrorParse(tok.src, "%s", ResolveErrormsg);
                    return ErrorExpr();
                }
		assert(obj != NULL);
		NextToken();
                goto HAVE_OBJS;

	    case char_const:
		a = NewExpr(char_const, NULL, NULL);
		a->u.i = tok.buf[0];
		NextToken();
		break;

	    case int_const:
		a = NewExpr(int_const, NULL, NULL);
		a->u.i = tok.int_val;
                if (tok.x < 0)
                    a->type = int_typstr;
                else a->type = uint_typstr;
                a->tp = (tp_enum)*a->type;
                a->o = oConstant;
		NextToken();
		break;

	    case long_const:
		a = NewExpr(long_const, NULL, NULL);
		a->u.l = tok.long_val;
		NextToken();
		break;

	    case float_const:
		a = NewExprFloat(tok.float_val);
		NextToken();
		break;

	    case string_const:
		a = ParseStringConst();
		break;

	    case kw_true:
	    case kw_false:
		a = NewExpr(int_const, NULL, NULL);
		a->u.i = (tok.en == kw_true);
		DecorateExpr(a);
		a->type = bool_typstr;
		NextToken();
		break;

	    case kw_sizeof:
		NextToken();
		Gobble(open_round);
		size = TypeSize(ParseExprOrTypedef());
		a = NewExpr(int_const, (Expr*)size, NULL);
		Gobble(close_round);
		break;

	    case kw_typeof:
		NextToken();
		Gobble(open_round);
		typeof_string = ParseExprOrTypedef();
		a = ExprCopyOfTypestring(typeof_string, (default_storage == auto_storage));
		// We store the type-str inside the code iff we're inside a function.
		// We know we're inside a function if default_storage == auto_storage.
		debug_type = (Type)a->left;
		Gobble(close_round);
		break;

	    case open_round:
		NextToken();
		if (AtTypeExpression() and LookAhead() != open_round) {
		    Type type;
		    Expr* e;

		    type = ParseTypeExpression();
		    type = (Type)qmallocCopy(type, LengthOfTypeString(type));
		    Gobble(close_round);
		    e = ParEx(unary_op);
		    a = NewExpr(kw_typedef, (Expr*)type, e);
		}
		else {
		    a = ParEx(unknown_op);
		    Gobble(close_round);
		}
		break;

	    case op_sing('-'):
		NextToken();
                a = ParEx(unary_op);
                if (a->opr == int_const)
                    a->u.i = -a->u.i;
		else a = NewExpr(op_unry('-'),a,NULL);
		break;
	    case op_sing('+'):
		NextToken();
		a = ParEx(highest);
		break;

	    case op_doub('+'):
	    case op_doub('-'):
	    case op_sing('~'):
	    case op_sing('!'):
		this_op = tok.en;
		NextToken();
		a = NewExpr(this_op, ParEx(highest), NULL);
		break;

	    case op_sing('&'):
		NextToken();
		a = NewExpr(op_unry('&'), ParEx(highest), NULL);
		break;

	    case op_sing('*'):
		NextToken();
		a = NewExpr(op_unry('*'), ParEx(highest), NULL);
		break;

	    case kw_new:
		a = ParseNew();
		break;

	    case kw_delete:
		a = ParseDelete();
		break;

	    default:
		ErrorParse(tok.src, "I don't understand the token \"%s\"  ", tok.buf);
		NextToken();
		return ErrorExpr();
	}

	/*************** We now have the first operand! **********************/
	if (a->src == NULL)
	    a->src = src;
        if (tok.en != open_round) {
            /* If we're going to use function-call notation, then we'd */
            /* better wait for the full parameter list before decorating */
            /* the identifier. */
            DecorateExpr(a);
        }


	/* Unary post-operators */
	do {
	    if (Error.err == err_certain)
		return a;
	    switch (tok.en) {
	    case open_round:            // Function calls
		a = ParseFunctionCall(a);
		break;

	    case open_square:           // Array dereferences
		NextToken();
		b = ParEx(unknown_op);
		a = NewExpr(open_square,a,b);
		a->src = src;
		Gobble(close_square);
		break;

	    case op_doub('+'):          // Post-increment
		NextToken();
		a = NewExpr(op_doub('+'),NULL,a);
		a->src = src;
		break;

	    case op_doub('-'):          // Post-decrement
		NextToken();
		a = NewExpr(op_doub('-'),NULL,a);
		a->src = src;
		break;

	    case dot:                   // Member-access
	    case arrowtok:
		this_op = tok.en;
                if (tok.en == dot)
                    classdef = TypeToClassdef(a->type);
                else if (tok.en == arrowtok and *a->type == tp_pointer)
                    classdef = TypeToClassdef(a->type+1);
                else classdef = NULL;
                if (classdef == NULL) {
                    ErrorParse(tok.src, tok.en == dot ? "You need a class "
                    	"instance on the left of a '.'" :
                        "You need a class pointer on the left of a '->'");
                    return ErrorExpr();
                }
                else if (not classdef->ValidClassdef()) {
                    ErrorParse(a->src, "Internal error - I got a corrupt "
                        "classdef.");
                    return ErrorExpr();
                }
		NextToken();
                SCOPE_RESOLUTION:
		if (tok.en != identifier)
		    ErrorParse(tok.src, "Expecting a member name");
		src2 = tok.src;
                if (CompilerInvoker == 'D')
                    visibility = debug_visibility;
                else if (context_fn == NULL or (context_fn->storage != member_fn
                	and context_fn->storage != virtual_fn))
                    visibility = public_visibility;
		else if (context_fn->owner == classdef)
                    visibility = private_visibility;
                else if (IsPublicBaseClass(classdef, (Classdef*)context_fn->owner))
		    visibility = protected_visibility;
                else visibility = public_visibility;
                obj = ResolveMember(classdef, tok.buf, visibility);
                if (obj == NULL and LookAhead() == double_colon) {
                    obj = ResolveTok(declarator_class);
                    if (obj == NULL) {
                    	ErrorParse(tok.src, "Unknown scope:  %s", tok.buf);
                    	return ErrorExpr();
                    }
                    else if (obj->storage != typedef_storage or
                    	*obj->type != tp_class) {
                    	ErrorParse(tok.src, "\"%s\" is not a class name, "
                        	"so you can't use it for scope resolution.", tok.buf);
                    	return ErrorExpr();
                    }
		    classdef = TypeToClassdef(obj->type);
                    NextToken();
                    Gobble(double_colon);
                    goto SCOPE_RESOLUTION;
                }
                else if (ResolveErrormsg[0]) {
                    ErrorType(tok.src, "%s", ResolveErrormsg);
                    return ErrorExpr();
                }
                else if (obj == NULL) {
                    obj = ResolveMember(classdef, tok.buf,
                		context_fn and classdef == context_fn->owner ?
                        	private_visibility : public_visibility);
                    if (ResolveMember(classdef, tok.buf, debug_visibility) == NULL)
                        ErrorType(tok.src, "There is no member \"%s\" of class %s.",
                        	tok.buf, classdef->typedef_obj->name);
                    else ErrorType(tok.src, "Member \"%s\" of class %s is not accessible",
                    		tok.buf, classdef->typedef_obj->name);
                    obj = ResolveMember(classdef, tok.buf,
                		context_fn and classdef == context_fn->owner ?
                        	private_visibility : public_visibility);
                    return ErrorExpr();
                }
		NextToken();
		if (tok.en == open_round) {
		    /* Member function calls: */
		    b = NewExprIdentifier(&ResolveInfo);
		    b->src = src2;
		    a = NewExpr(this_op, a, ParseFunctionCall(b));
		}
		else {
		    /* Variable member access: */
		    b = NewExprIdentifier(&ResolveInfo);
		    b->src = src2;
		    a = NewExpr(this_op, a, (Expr*)b);
		}
		a->src = src;
		break;

	    case identifier:
		ErrorParse(tok.src, "Unexpected identifier. Do you need a type-name?");
		return ErrorExpr();

	    default:
		goto BREAK_OUT;

	    }
	    DecorateExpr(a);
	} forever;
	BREAK_OUT:

	do {
	    prec = GetPrecedence(tok.en);
	    if (prec == assignment_op) {
		if (prec < UpTo)
		    return a;
	    }
	    else if (prec <= UpTo)
		return a;
	    this_op = tok.en;
	    src = tok.src;
	    NextToken();

	    if (this_op == ternary) {   // The ternary operator
		Expr* c;
		b = ParEx(conditional_right);
		Gobble(colon);
		c = ParEx(conditional_right);
		a = NewExprTernary(a,b,c);
		DecorateExpr(a);
		continue;
	    }

	    /* Get the second operand. */
	    b = ParEx(prec);

	    /* Create the expression: */
	    a = NewExpr(this_op, a, b);
	    a->src = src;
	    DecorateExpr(a);
	    if (Error.err == err_certain)
		return a;

	} forever;
}



/*----------- 4 interfaces to the expression code ---------*/

static Expr* ParseBooleanExpr(void)
/* Compile this expression to an integer expression.	*/
/* Used for bool conditions in 'if', 'while' etc. 	*/
/* Install it in the expression list, ready for code	*/
/* generation. */
{   	Expr* e;

	e = ParEx(unknown_op);
	e = ExprRvalue(e);

	/* Check for the =/== mistake: */
	if (e->opr == op_sing('=')) {
	    str src = tok.src;
	    if (tok.en == close_round and tok.src and tok.src[-1] == ')')
		;	// Well in this case we'll believe that they mean it.
	    else ErrorParse(e->src, "I think you mean for this to be a '==' equals "
		"operator. (If you really want an assignment here, put an extra "
		"set of brackets around it).");
	}

	/* Do we need to convert it to bool? */
	DecorateExpr(e);
	if (*e->type != tp_bool) {
	    e = NewExpr(kw_if, e, NULL);
	    e->src = e->left->src;
	    DecorateExpr(e);
	}
	return e;
}


interface Expr* ParseIntExpr(bool StopAtComma)
/* Compile an expression into a cardinal type (char/int/enum etc). */
{       Expr* e;

	e = ExprGetRvalue(ParEx(StopAtComma ? comma_op : unknown_op));
	if (*e->type != 'i' and *e->type != 'u') {
            /* Even char's, short's and enums will need to be zero-extended: */
	    e = NewExpr(kw_typedef, (Expr*)int_typstr, e);
	    DecorateExpr(e);
	    e = ExprGetRvalue(e);
	}

	return e;
}


interface int ParseAndEvaluateIntExpr(bool StopAtComma)
/* For various things, we need to get constant integer expressions. */
/* E.g. array declarations, #if conditions, bit fields, enums.      */
/* Note that it does not allow commas in the expression except in   */
/* brackets, (think about enum declarations).                       */
/* This cannot be called in the middle of parsing an expression.    */
/* It might be possible to reimplement this using the optimisation  */
/* which reduces an expression to an integer, but I haven't tried   */
/* that. */
{       bool success;
	Expr* e;
	int i;

	e = ParseIntExpr(StopAtComma);
	if (Error.err)
	    return 1;
	if (e->opr == int_const)    /* Short-cut */
	    return e->u.i;
	i = IprEvaluateIntConstant(e, &success);
	if (not success)
	    ErrorParse(e->src, "This expression is not constant.");
	return i;
}


interface Expr* ParseExpr(bool StopAtCommas)
/* Parses code for an expression into an expression tree. */
{
	return ParEx(StopAtCommas ? comma_op : unknown_op);
}





/*------------------- The 'datainit' data-structure: ---------------*/

/* We allocate temporary storage for arrays being initialised. */

static uchar* databuf;
static int db_midx, db_idx;	    // Allocated size and used size


static void DatainitBegin(void)
/* Set up a temporary buffer for writing data into. */
{
	db_midx = 1024;
	databuf = (uchar*)malloc(anon_heap, db_midx);
	db_idx = 0;
}


static void DatainitAddData(void* data, uint size)
/* Add this sequence of bytes to the growing data-sequence. */
{
	db_idx += size;
	while (db_idx > db_midx) {
	    db_midx *= 2;
	    databuf = (uchar*)realloc(anon_heap, databuf, db_midx);
	}
	memcpy(databuf + db_idx - size, data, size);
}


static int DatainitSize(void)
/* Returns the current number of bytes written. */
{
	return db_idx;
}


static uchar* DatainitSource(void)
/* Returns the location of the initialised data. */
{
	return databuf;
}


static void DatainitAlign(uint offset)
/* Pad out the data so that we get to 'offset'. */
{	int extra;

	extra = offset - db_idx;
	assert(extra >= 0);
	if (extra > 0) {
	    db_idx += extra;
	    while (db_idx > db_midx) {
		db_midx *= 2;
		databuf = (uchar*)realloc(anon_heap, databuf, db_midx);
	    }
	    memset(databuf + db_idx, 0, extra);
	}
}


static void DatainitReset(void)
/* Release all this memory again - we've got the data. */
{
	free(anon_heap, databuf);
	databuf = NULL;
}





/*=============== Variable Initialisation routines =============*/

static Type ConformingType;


static void ParseAggrInit(Type type)
/* Parse this as an aggregate initialisation of type 'type'. */
/* The destination for the data is 'data_dest'.              */
/* This function is recursive. */
{	int size, dec_bound, bound;
	Classdef* classdef;
	int base, len;
	Namedobj* obj;
	Expr* e;
	Type type0;

	switch (*type) {
	    case tp_class:
		    /* Initialise a struct */
		    Gobble(open_brace);
		    type++;
		    GetPtr(classdef, type);
		    base = DatainitSize();
		    for (obj=classdef->member; obj; obj=obj->next) {
                        if (obj->storage != member_storage)
                            continue;
			DatainitAlign(base + ((FieldNamedobj*)obj)->offset);
			ParseAggrInit(obj->type);
			// Uses the assumption that fields are in location order.
			if (tok.en == comma)
			    Gobble(tok.en);
			else break;
		    }
		    DatainitAlign(base + classdef->size);
		    Gobble(close_brace);
		    break;

	    case tp_array:
		    /* Initialise an array */
		    Gobble(open_brace);
		    type0 = type++;
		    GetDimension(dec_bound, type);
		    size = TypeSize(type);
		    bound = 0;
		    base = DatainitSize();
		    until (tok.en == close_brace) {
			ParseAggrInit(type);
			if (++bound > dec_bound and dec_bound != 0)
			    ErrorType(tok.src, "Too many initialisors");
			if (tok.en == comma)
			    Gobble(tok.en);
			else break;
		    }
		    if (bound < dec_bound)
			DatainitAlign(base + dec_bound * size);
		    else if (dec_bound == 0) {
			len = LengthOfTypeString(type0);
			ConformingType = (Type)memcpy(qmalloc(len), type0, len);
			type = ConformingType + 1;
			PutDimension(bound, type);
		    }
		    Gobble(close_brace);
		    break;

	    case tp_dynarray:
		    /* Initialise a dynamic array: */
		    Gobble(open_brace);
		    type0 = type++;
		    size = TypeSize(type);
		    base = DatainitSize();
		    until (tok.en == close_brace) {
			ParseAggrInit(type);
			if (tok.en == comma)
			    Gobble(tok.en);
			else break;
		    }
		    ErrorParse(tok.src, "Dynarray initialisation not finished yet.");
		    Gobble(close_brace);
		    break;

	    default:
		    /* Initialise a fundamental type */
		    e = ParseExpr(yes);
		    e = NewExpr(kw_typedef, (Expr*)type, e);
		    DecorateExpr(e);
		    if (Error.err)
			return;
		    if (not IprEvaluateConstant(&e))
			ErrorParse(tok.src, "This expression is not constant.");
		    size = TypeSize(e->type);
		    DatainitAddData(&e->u, size);
		    break;

	}
}


static void PutInitialisationDataIntoObj(Namedobj *obj)
/* We have just parsed a structure/array initialisation, and the data */
/* is stored in the Datainit data-structure. Copy it out of there into */
/* 'obj'. */
{
        if (obj->storage == local_static) {
	    /* The storage has already been allocated. */
	    CopyLocalStatic((int)((StaticNamedobj*)obj)->location, DatainitSize(), DatainitSource());
	    DatainitReset();
        }
	else if (obj->storage == auto_storage) {
            ErrorParse(tok.src, "Can't initialise a local array/struct (not implemented).");
#if 0
	    offset = AllocateLocalStatic(data_size);
	    value = local_static_buf + offset;
	    memcpy(value, DatainitSource(), data_size);
	    DatainitReset();

	    /* To make this code work, set this up as an expression */
	    /* to be emitted later (after the 'ENTER_FN').  */
	    AssemblerAddInstr(PUSH_LS, (op_type)offset);
	    AssemblerAddInstr(DEREFX, (op_type)data_size);
	    AssemblerAddInstr(PUSH_FP, (op_type)obj->u.location);
	    AssemblerAddInstr(STOX, (op_type)data_size);
	    AssemblerAddInstr(POPDX, (op_type)data_size);
#endif
        }
        else if (obj->storage == static_storage) {
            memcpy(((StaticNamedobj*)obj)->location, DatainitSource(), DatainitSize());
        }
        else assert(false);
}


interface void AddInitialisingExpr(Expr* e)
/* Add this expression to the list of things we need to execute */
/* at the start of the frame. */
{
	if (e == NULL)
            return;
        if (decl_init_root == NULL)
            decl_init_root = e;
        else {
            decl_init_root = NewExpr(comma, decl_init_root, e);
            DecorateExpr(decl_init_root);
            decl_init_root->src = e->src;
        }
}


interface void ParseInitialisation(Namedobj* obj)
/* After having gobbled the assignment operator, parse */
/* and compile an initialisation.  */
{       uint data_size;
	Expr *v,*e;
	str src;

	if (tok.en != open_brace and obj->storage != local_static) {

	    /*** Initialising a non-structured value on entry into the function: ***/
            ResolveInfo.obj = obj;
            ResolveInfo.list = NULL;
            ResolveInfo.offset = 0;
	    v = NewExprIdentifier(&ResolveInfo);
            src = tok.src;
	    v->src = src - 4;
	    e = ParEx(comma_op);                // Do the initialising expression
	    v = NewExpr(op_init, v, e);
	    v->src = (src[-1] == ' ') ? src-2 : src-1;
	    DecorateExpr(v);
            AddInitialisingExpr(v);
            if (obj->storage == auto_storage)
                ((AutoNamedobj*)obj)->initialisation = v;

	}
	else if (obj->storage != static_storage and obj->storage != auto_storage
		and obj->storage != local_static) {
	    ErrorParse(tok.src, "This thing cannot be initialised (bad storage class)");
	    return;
	}
	else {

	    /*** Initialising a struct or array or static object: ***/

	    /* Get the data: */
	    DatainitBegin();
	    ParseAggrInit(obj->type);
	    data_size = DatainitSize();
	    assert(data_size == TypeSize(obj->type));


            /* Put the data into 'obj': */
            PutInitialisationDataIntoObj(obj);
        }
}


static Expr* InsertVftptr(Expr* var_e, Classdef* classdef)
/* 'e' represents a oLocalVar expression (i.e. a stack variable). */
/* Insert the virtual function table pointer into it. */
{       Expr *e, *vftval, *vftaddr, *var2_e;
        str src=var_e->src;

        vftval = NewExprVoidptr(classdef);
        assert(var_e->o == oLocalVar and var_e->ref == 1);
        assert(var_e->type != NULL);
        var2_e = ExprDuplicate(var_e);
        var_e->src = src;
        var_e->ref = 0;
        var_e->type = voidvoidptr_typstr;
        vftaddr = NewExpr(op_sing('+'), var_e, NewExprInt(-1));
        vftaddr->src = src;
        DecorateExpr(vftaddr);
        vftaddr->ref = 1;
        vftaddr->type = voidptr_typstr;
        e = NewExpr(op_sing('='), vftaddr, vftval);
        e->src = src;
        DecorateExpr(e);
        e = NewExpr(comma, e, var2_e);
        e->src = src;
        DecorateExpr(e);
        return e;
}


interface Namedobj* ParseConformingArray(Directory* declaration_directory,
                str name, Type type, storage_enum storage, int options)
/* Like above, but we don't have 'obj' yet because we don't know */
/* what size the array will be.  Parse the data and then declare */
/* 'obj' accordingly. */
{       int array_size, element_size;
        Namedobj *obj;
        Type t;

        /* Get the data: */
	DatainitBegin();
	ParseAggrInit(type);
	array_size = DatainitSize();


        /* Fill out the type: */
        t = type;
        assert(*t == tp_array);
        t++;
        assert(*(int*)t == 0);
        element_size = TypeSize(t + sizeof(int));
        *(int*)t = (array_size + element_size - 1) / element_size;       // Round up
	assert(array_size == TypeSize(type));


        /* Do the declaration: */
        obj = NameDeclare(declaration_directory, name, type, storage, options);


        /* Put the data into 'obj': */
        PutInitialisationDataIntoObj(obj);
        return obj;
}


interface Expr* ParseConstructorExpr(Type type, Expr* this_expr)
/* This object needs to have a constructor. If it's a 	*/
/* constructor with parameters then we'll be starting 	*/
/* at the open-round token. Otherwise emit the code   	*/
/* for the default constructor but don't gobble any   	*/
/* tokens. 						*/
{	Expr *e, *params[256], *func;
        FunctionNamedobj *conobj;
	Classdef* classdef;
        str src=tok.src;
	int arity;

        classdef = TypeToClassdef(type);
        if (classdef == NULL) {
            FunctionNamedobj *DefaultConstructor;
            Classdef* classdef;
            Expr* params[4];
            Namedobj *obj;
	    int dimension;

            // How big is the array?
            this_expr->ref--;
            assert(*type == tp_array);
            dimension = 1;
            while (*type == tp_array) {		// Deal with arrays of arrays...
            	type++;
	    	dimension *= *((int*&)type)++;
            }
            assert(*type == tp_class);

            // What's the default constructor?
            classdef = TypeToClassdef(type);
            conobj = ClassDefaultConstructor(classdef, &obj);
            if (conobj == NULL and obj) {
            	ErrorType(src, "There's no default constructor for type: \"%s\". "
            		"This means you can't declare arrays of it.",
                        classdef->typedef_obj->name);
            	return ErrorExpr();
            }
            DefaultConstructor = conobj;

            // Get ArrayConstructor address as a decorated expr:
            obj = Stdlib->directory()->Find("GenericArrayConstructor");
            assert(obj != NULL);
            func = NewExprIdentifier(obj);

            // Get the 4 parameters to ArrayConstructor:
            params[0] = this_expr;
            params[1] = NewExprInt(dimension);
            params[2] = NewExprInt(TypeSize(type));
            params[3] = NewExprVoidptr(DefaultConstructor->u.fn);

            // Get the function call expression:
            e = NewExprFn(func, params, 4);
            DecorateExpr(e);
            return e;
        }

        /* Build up an expression representing the parameter list: */
        arity = 0;
        if (tok.en == open_round) {
            NextToken();
            until (tok.en == close_round) {
            	if (arity >= arraymax(params)) {
                    ErrorParse(tok.src, "> 256 params");
                    return NULL;
                }
		params[arity++] = ParseExpr(yes/*stop at commas*/);
                if (tok.en == comma)
                    NextToken();
                else break;
            }
            Gobble(close_round);
        }
        if (Error.err)
            return NULL;

        /* If it's a static or a heap object, then the vft-pointer      */
        /* (virtual function table pointer) is already in place.        */
        /* But we must put it in for local variables. */
        if (this_expr->o == oLocalVar and classdef->VirtualFns)
            this_expr = InsertVftptr(this_expr, classdef);


        /* Marshall all the candidate constructors: */
        ResolveInfo.obj = NULL;
        ResolveInfo.list = NULL;
        ResolveInfo.offset = 0;
        for (Namedobj* obj=classdef->member; obj; obj=obj->next) {
            if (IsConstructor(obj))
            	Array_Add(ResolveInfo.list, obj);
        }
        if (ResolveInfo.list) {
            ResolveInfo.obj = ResolveInfo.list[0];
            if (Array_Size(ResolveInfo.list) == 1)
            	Array_Free(ResolveInfo.list);
        }
        else {
            if (this_expr->o == oCOMMA and this_expr->right->o == oLocalVar)
                this_expr = this_expr->left;  // Undo the duplication
                // from the InsertVftptr();
            return this_expr;		// We didn't actually find any constructor.
            // We probably only thought there was one because of the vftable.
        }
        func = NewExprIdentifier(&ResolveInfo);
        func->src = src;

        /* Create the function-call expression: */
	e = NewExprFn(func,params,arity);
        e->src = src;
        e->u.func->owner = this_expr;

        /* Decorate it: */
        DecorateExpr(e);
        if (Error.err)
            return NULL;
        return e;
}




/*============= Function Definition =============*/

static Type DeclareParameters(Type type)
/* Declare all the parameters for this function as local*/
/* variables.  The types come from interpreting 'type',	*/
/* and the names come from 'parameter_name[]'.		*/
/* Returns the return type.				*/
{       int i, extra, no_params;
	Namedobj* obj;
	int offset;
	struct {
	    Type type;
	    str name;
	    int offset;
	} param[100];

	/* Set things up */
	offset = 2 * sizeof(int);	// Leave space on the stack for PC & BP.
	assert(*type == tp_function);
	type++;
	i = *type++;
	assert(i == (int)pn_idx);
	no_params = pn_idx & ~128;
	default_storage = parameter_storage;

	/* Do the 'this' parameter if applicable. */
	extra = 0;
	if (declarator_class) {
	    static uchar class_type[6] = { tp_pointer, tp_class, 0, 0, 0, 0 };
	    Type type=class_type+2;

	    PutPtr(declarator_class, type);
	    param[0].offset = offset;
	    param[0].type = class_type;
	    param[0].name = "this";
	    offset += sizeof(str);
	    extra++;
	}


	/* Get the parameter types & offsets from the start. */
	for (i=0; i < no_params; i++) {
	    param[i+extra].type = type;
	    param[i+extra].offset = offset;
	    param[i+extra].name = param_name[i];
	    offset += TypeSizeWord(type);
	    type += LengthOfTypeString(type);
	}
	no_params += extra;
	assert(*type == tp_terminated);
	type++;


	/* Create a fictitional '__ellipsis' parameter: */
	if (pn_idx & 128) {
	    param[no_params+extra].type = char_typstr;
	    param[no_params+extra].offset = offset;
	    param[no_params+extra].name = "%ellipsis";
	    no_params++;
	}

        /* If the fn returns a struct, then we need a return-val reference param: */
        if (TypeIsBigStruct(type)) {
            /* Insert it to the front of the list: */
            for (i=0; i < no_params; i++)
            	param[i].offset += 4;
            offset += 4;
            memmove(param+1, param, sizeof(param[0])*no_params);
            no_params++;
            i = LengthOfTypeString(type);
            param[0].type = (Type)qmalloc(i+1);
            memcpy(param[0].type+1, type, i);
            param[0].type[0] = tp_reference;
            param[0].offset = 2*sizeof(int);
            param[0].name = "%returnref";
        }

	/* Declare the variables with locations relative to the stack frame. */
	for (i=0; i < no_params; i++) {
	    parameter_location = param[i].offset;
	    obj = NameDeclare(NULL, param[i].name, param[i].type, parameter_storage, 0);
	}

	/* Get the local 'this' object if it's a member function: */
	if (declarator_class) {
	    this_obj = (AutoNamedobj*)NameLocalResolve("this");
	    assert(this_obj != NULL);
	    this_class = declarator_class;
	}

	return type;
}


interface FunctionNamedobj* RecurseFindVirtualInBase(Classdef* classdef, Namedobj *obj)
{	Classdef* base_classdef;
        Namedobj *bobj;

	for (Namedobj *base=classdef->member; base; base=base->next) {
            if (base->storage != inherit_storage)
            	continue;
            if (base->visibility == private_visibility)
            	continue;
	    base_classdef = TypeToClassdef(base->type);
            for (bobj=base_classdef->member; bobj; bobj=bobj->next) {
            	if (streq(bobj->name, obj->name) and TypeEqual(bobj->type, obj->type))
                    return (FunctionNamedobj*)bobj;
            }
	    bobj = RecurseFindVirtualInBase(base_classdef, obj);
            if (bobj)
            	return (FunctionNamedobj*)bobj;
        }
        return NULL;
}


static void CheckMemberfnOverridesVirtual(Namedobj *obj, funcblock_type body)
/* We've just compiled a function definition.  In some cases, */
/* a function is both a virtual and non-virtual function:  it */
/* was declared as non-virtual but it provides a definition   */
/* for something declared as virtual in a base class. In this */
/* case, we compile it as a non-virtual but assign a ptr here.*/
{	FunctionNamedobj *basefn;

	if (not declarator_class or declarator_class->VirtualFns == NULL)
            return;
        if (obj->storage == virtual_fn)
            return;
        basefn = RecurseFindVirtualInBase(declarator_class, obj);
        if (basefn and basefn->storage == virtual_fn) {
            funcblock_type *fnlist;
            if (declarator_class->RoundupSize == -2) {
            	// An incomplete (middle of compiling it) class.
                // It has a different representation of VirtualFns.
            	fnlist = (funcblock_type*)declarator_class->VirtualFns;
            }
            else {
            	fnlist = (funcblock_type*)((str)&declarator_class->VirtualFns + 4);
            }
            fnlist[basefn->u.virtualfn_idx] = body;
        }
}


static str BaseName(Namedobj *obj)
/* This object may represent an inherit_storage object.  If so, */
/* return its name. */
{
        if (obj->storage != inherit_storage)
            return obj->name;
        assert(*obj->type == tp_class);
        Classdef* baseclass = *(Classdef**)(obj->type+1);
        assert(baseclass->typedef_obj);
	return baseclass->typedef_obj->name;
}


static Expr* NewExprConstructorList(Type type)
{	Namedobj *obj, **list=NULL;
	Classdef* classdef;
	Expr* a;

        if (*type++ != tp_class)
            return NULL;

        /* Form a list of constructors: */
        classdef = *(Classdef**)type;
        for (obj=classdef->member; obj; obj=obj->next) {
            if (streq(obj->name, classdef->typedef_obj->name))
            	Array_Add(list, obj);
        }
        assert(list != NULL);

        /* A Resolved_node for this list: */
        a = NewExpr(identifier,NULL,NULL);
        a->u.idfier.resolve = (Resolved_node*)qmalloc(sizeof(Resolved_node));
        a->u.idfier.resolve->obj = list[0];
        a->u.idfier.resolve->offset = 0;
        a->u.idfier.resolve->list = list;
        return a;
}


static Statement* ParseAndProcessMemberConstructors(Namedobj *fnobj)
/* We are at the beginning of parsing a constructor function. 	*/
/* This class may contain members and base classes which also 	*/
/* have constructors.  We will need to either call the default	*/
/* constructors in this case or parse expressions of the form 	*/
/* ": obj(params)".  In the latter case, we can assume the    	*/
/* token we're on upon entry is the first colon. 		*/
/*     Return the first statement in the list of constructor    */
/* calls. */
{	Namedobj *obj, **NeedsConstructor=NULL, **HasConstructor=NULL;
	Expr *a, *e, **ConstructorList=NULL;
	Statement *S, *S0=NULL;
        ExprStatement* S_expr;
        str src;
        int i;

        /* Construct the list of members (including base classes) */
        /* needing constructors: */
	for (obj=declarator_class->member; obj; obj=obj->next) {
            if ((obj->storage == inherit_storage or obj->storage == member_storage)
                	and TypeHasConstructor(obj->type))
                Array_Add(NeedsConstructor, obj);
        }

        /* A shortcut: */
        if (NeedsConstructor == NULL) {
            if (tok.en == colon)
            	ErrorParse(tok.src, "What's the colon for?  There are no members "
                	"or base classes that need constructors.");
            return NULL;
        }

        /* Parse any explicit constructor calls: */
        if (tok.en == colon) {
            do {
                NextToken();
                if (tok.en != identifier) {
                    ErrorParse(tok.src, "I'm expecting an identifier (representing "
                            "a base class or member constructor) here.");
                    goto RETURN;
                }

                /* Find the constructor: */
                for (each_aeli(obj, NeedsConstructor)) {
                    if (strieq(BaseName(obj), tok.buf))
                        goto PARSE_FUNCCALL;
                }

                /* Why doesn't it need a constructor? */
		for (obj=declarator_class->member; obj; obj=obj->next) {
                    if (streq(BaseName(obj), tok.buf)) {
                        if (Array_HasP(HasConstructor, obj))
                            ErrorParse(tok.src, "'%s' already has a constructor "
                                "call in this function", tok.buf);
                	else ErrorParse(tok.src, "'%s' doesn't have a "
                        	"constructor.", tok.buf);
                    	goto RETURN;
                    }
                }
                assert(false);
                goto RETURN;

                /* Parse it as a function call: */
                PARSE_FUNCCALL:
                src = tok.src;
                NextToken();
                a = NewExprConstructorList(obj->type);
                a->src = src;
                e = ParseFunctionCall(a);
                e->u.func->owner = NewMemberAddress(obj, src);
                e->src = src;
                DecorateExpr(e);

                /* Move it from the list of members needing to those having: */
                Array_Add(HasConstructor, obj);
                Array_DelP(NeedsConstructor, obj);
                Array_Add(ConstructorList, e);
            } while (tok.en == comma);
        }

        /* Deal with any remaining members needing constructors: */
	for (each_aeli(obj, NeedsConstructor)) {
            /* Create a fn-call expr to the default constructor. */
            Namedobj *anyconstructor, *defconstructor;
            Expr *func, *e, *mem_e;
            Type type=obj->type;
            Classdef* memclass;

            if (*type == tp_class)
                memclass = *(Classdef**)++type;
            else assert(false);
            defconstructor = ClassDefaultConstructor(memclass, &anyconstructor);
            if (defconstructor == NULL)
                continue;
            func = NewExprIdentifier(defconstructor);
            mem_e = NewMemberAddress(obj, tok.src);
            e = NewExprFn(func, NULL, 0);
            e->u.func->owner = mem_e;
            DecorateExpr(e);
            Array_Add(ConstructorList, e);
        }

        /* Put all the constructor calls into the statement list: */
        S = S0 = NULL;
        for (each_aeli(e, ConstructorList)) {
            S_expr = (ExprStatement*)NewStatement(st_expr);
            S_expr->e = e;
            S_expr->src = tok.src;
            if (S0 == NULL)
                S0 = S = (Statement*)S_expr;
            else S->next = (Statement*)S_expr, S = S->next;
        }

        /* Clean up: */
        RETURN:
        Array_Free(NeedsConstructor);
        Array_Free(HasConstructor);
        Array_Free(ConstructorList);
        return S0;
}


static Namedobj* ClassDestructor(Classdef* classdef)
{
	for (Namedobj *obj=classdef->member; obj; obj=obj->next) {
            if ((obj->storage == member_fn or obj->storage == virtual_fn)
            	and streq(obj->name, "~"))
                return obj;
        }
        return NULL;
}


static Statement* BuildMemberDestructors(Namedobj *fnobj)
/* The current class may contain members and base classes which also 	*/
/* have destructors. If so, construct the statements to call them.	*/
{	Namedobj *obj, *destructor;
        Expr *func, *e, *mem_e;
        ExprStatement* S_expr;
        Classdef* memclass;
        Statement *S, *S0;
	Type type;

        /* Construct the list of members (including base classes) */
        /* needing destructors: */
        S = S0 = NULL;
	for (obj=declarator_class->member; obj; obj=obj->next) {
            if (obj->storage == inherit_storage or obj->storage == member_storage) {
                type = obj->type;
                if (*type != tp_class)
                    continue;
                memclass = *(Classdef**)++type;
                destructor = ClassDestructor(memclass);
                if (destructor == NULL)
                    continue;
                func = NewExprIdentifier(destructor);
                mem_e = NewMemberAddress(obj, tok.src);
                e = NewExprFn(func, NULL, 0);
                e->u.func->owner = mem_e;
                DecorateExpr(e);
                S_expr = (ExprStatement*)NewStatement(st_expr);
                S_expr->e = e;
                S_expr->src = tok.src;
                if (S0 == NULL)
                    S0 = S = (Statement*)S_expr;
                else S->next = (Statement*)S_expr, S = S->next;
            }
        }
        if (S)
            S->next = (Statement*)NewStatement(st_return);
        return S0;
}


static void PatchSrcIntoStatement(Statement *S, str src)
{
	while (S) {
            S->src = src;
            S = S->next;
        }
}


interface void ParseFunctionDefinition(FunctionNamedobj* obj)
/* Parse a function definition. Starts with `tok.en' pointing */
/* at the open brace. */
{       funcblock_type body, *orig_bodyp;
	storage_enum old_storage;
	Namedobj* scope;
	Statement *S;

	/* Set things up */
        context_fn = obj;
	scope = NameMarkScope();
	old_storage = default_storage;
	UndefinedFunctionCall = NULL;
        DestructorS = NULL;
	InitLocalVars();
        IprRegisterStatement(NULL);
	if (obj->storage == virtual_fn) {
            if (declarator_class->RoundupSize == -2) {
            	// It's an incomplete class, i.e. we must be declaring the
                // function inside the actual class definition.
                assert(obj->u.virtualfn_idx >= 0 and obj->u.virtualfn_idx
                	< Array_Size(declarator_class->VirtualFns));
            	orig_bodyp = (funcblock_type*)&declarator_class->VirtualFns[obj->u.virtualfn_idx];
            }
	    else {
                funcblock_type *fnlist;
                fnlist = (funcblock_type*)((str)&declarator_class->VirtualFns + 4);
	    	orig_bodyp = &fnlist[obj->u.virtualfn_idx];
            }
	}
	else orig_bodyp = (uchar**)&obj->u.fn;
        if (*orig_bodyp)
            assert(Heap::AssertTile(tile_type(*orig_bodyp - 4)));

	/* Put the parameters into the name-space and setup the return value */
	return_type = DeclareParameters(obj->type);
	assert(*obj->type == tp_function);

        /* Is it a constructor/destructor with members/base classes */
        /* that need construction/destruction? */
        if (IsConstructor(obj)) {
            S = ParseAndProcessMemberConstructors(obj);
            if (Error.err)
            	goto ERROR;
            if (S) {
                Statement* S2=S;
                while (S2->next)
                    S2 = S2->next;
                S2->next = ParseThing();
                goto HAVE_S;
            }
        }
        else if (streq(obj->name, "~")) 		// A destructor
            DestructorS = BuildMemberDestructors(obj);

	/* Generate the function code */
	S = ParseThing();
        if (DestructorS)
            PatchSrcIntoStatement(DestructorS, tok.src-1);
	NamePopScope(scope);
        HAVE_S:
	if (Error.err == err_certain or Error.err == err_maybe)
	    goto ERROR;
	DecorateEntity(S,no);
	if (Error.err == err_certain or Error.err == err_maybe)
	    goto ERROR;
        InsertDestructorCalls();
	body = IprCompile(S, *orig_bodyp);

        /* Finish up: */
	default_storage = old_storage;
	return_type = NULL;
	this_obj = NULL;
	if (Error.err == err_certain) {
	    if (body and body != *orig_bodyp)
		default_heap->free(body), body = NULL;
	    ERROR:
	    if (*orig_bodyp) {
		DebugBodyDeleted(obj);
		default_heap->free(*orig_bodyp);
		*orig_bodyp = NULL;
	    }
	    if (Error.err == err_maybe) {
		MakeEntityUpdate(obj);
		MakeChanged(obj->make, yes);
	    }

	    /* ABORT: */
	    InitLocalVars();
	    declarator_class = NULL;
            context_fn = NULL;
	    return;
	}

	/* Set up the make-entity information */
	MakeEntityUpdate(obj);
	if (*orig_bodyp == body) {
	    MakeChanged(obj->make, no);
	}
	else {
	    free(NULL, *orig_bodyp);
	    *orig_bodyp = body;
	    MakeChanged(obj->make, yes);
	}

	/* Other stuff: */
	DebugBodyRecompiled(obj);
	CurdirInterfaceChange(yes);
        CheckMemberfnOverridesVirtual(obj, body);
        context_fn = NULL;
}




/*================== Function Calls: ==================*/

interface int SizeOfFunction(FunctionNamedobj* obj)
/* Return the size of this function in bytes. */
{       void* p;

	p = obj->u.fn;
	if (p == NULL)
	    return 0;
	else return *(int*)p;
}


static Expr* ParseFunctionCall(Expr* a)
/* The expression 'a', being a member function, static function or */
/* function pointer expression, is being used with the function    */
/* notation.  Parse the round-brackets and parameters and generate */
/* a oFUNC expression. */
{       Expr *parameters[30], *e;
	int arity;

	Gobble(open_round);
	arity = 0;

	if (tok.en != close_round)
	    do {
		parameters[arity++] = e = ParEx(comma_op);
		if (Error.err)
		    break;
		else if (tok.en == comma) {
		    NextToken();
		    continue;
		}
		else break;
	    } forever;

	if (tok.en != close_round) {
	    ErrorParse(tok.src, "Expecting a `)'");
	    SkipTo(close_round);
	}

	Gobble(close_round);
	e = NewExprFn(a, parameters, arity);
	e->src = a->src;
	return e;
}




/*================= Label Routines ===============*/

typedef struct gotolabel_node {
	Statement *S;
	struct gotolabel_node *next;
	char name[1];
} *gotolabel_type;


static gotolabel_type gotolabel_list;

static void LabelClear(void)
/* Clear all the labels: */
{	gotolabel_type glab;

	while (gotolabel_list) {
	    glab = gotolabel_list->next;
	    free(anon_heap, gotolabel_list);
	    gotolabel_list = glab;
	}
}


static void LabelDefine(str name, Statement *S)
/* Define this label at this point: */
{	gotolabel_type glab;

	assert(S != NULL);
	glab = (gotolabel_type)malloc(anon_heap, sizeof(struct gotolabel_node) + strlen(name));
	glab->S = S;
	strcpy(glab->name, name);
        glab->next = gotolabel_list;
        gotolabel_list = glab;
}


interface Statement* LabelFind(str name)
/* Find this label in the list: */
{       gotolabel_type glab;

	for (glab=gotolabel_list; glab; glab=glab->next) {
	    if (streq(glab->name, name))
		return glab->S;
	}
	return NULL;
}







/*================= Grammatical Routines ===============*/

#define end_of_statement    0


static bool SetUpStackFrame(void)
/* If we're not already in a stack frame, create one and return */
/* the location of its start (otherwise return NULL).           */
/* This function signifies that we are entering the world of    */
/* local variables. */
{
	if (default_storage == auto_storage)
	    return no;
	else {
	    default_storage = auto_storage;
	    return yes;
	}
}


static Statement* ReleaseStackFrame(bool needed, void* S)
/* If we created a stack frame at this address, release it now. */
/* This function signifies that we are leaving the world of     */
/* local variables. */
{       Statement *F;

	F = (Statement*)S;
	return F;
}


static Statement* ParseThing()
/* Parse a declaration, expression or statement. */
{       extern int TimeOfThisMake, depth;
	Statement *S, **S_old, *S_root;
	ReturnStatement* S_return;
	SwitchStatement* S_switch;
	WhileStatement* S_while;
	ExprStatement* S_expr;
	CaseStatement* S_case;
	GotoStatement* S_goto;
	int temp_auto_offset;
	token_enum lookahead;
	ForStatement* S_for;
	DoStatement* S_do;
	IfStatement* S_if;
	Namedobj *PathObj;
	Namedobj *scope;
	bool frame;
	Expr* e;
	str src;

	src = tok.src;

	switch (tok.en) {

	    case open_brace:frame = SetUpStackFrame();
			    // == NULL if there's already a stack frame.
			    Gobble(open_brace);
			    temp_auto_offset = auto_offset; // Save it
			    scope = NameMarkScope();
			    S_root = NULL;
			    S_old = &S_root;
			    until (tok.en == close_brace or tok.en == EOF or Error.err) {
				S = ParseThing();
				if (S) {
				    *S_old = S;
				    while (S->next)
					S = S->next;
				    S_old = &S->next;
				}
			    }
			    if (frame) {
				/* We're falling off the end of a function. */
				if (return_type == NULL)
				    ;
				else if (*return_type == tp_void) {
                                    if (DestructorS)    // base class destructor calls
                                    	*S_old = DestructorS;
				    else {
                                        S_return = (ReturnStatement*)NewStatement(st_return);
				    	*S_old = (Statement*)S_return;
                                        S_return->e = NULL;
                                        S_return->src = tok.src;
                                        S_return->scope = NULL;
                                    }
				}
				else {
				    S_return = (ReturnStatement*)NewStatement(st_return);
				    *S_old = (Statement*)S_return;
				    S_return->e = (Expr*)0x1;
				    /* This will cause an error during the 'JumpAnalysis' stage */
				    /* if it's reachable. */
				}
			    }
			    else {
				*S_old = NULL;
			    }
                            NamePopScope(scope);
			    auto_offset = temp_auto_offset;
			    Gobble(close_brace);
			    if (S_root == NULL)
				return NULL;
			    S = ReleaseStackFrame(frame, S_root);
			    break;


	    case close_brace:
			    ErrorParse(tok.src, "Unexpected '}'  - do you need to insert a null statement ';' ?");
			    return NULL;

	    case kw_while:  frame = SetUpStackFrame();
			    scope = NameMarkScope();
			    S_while = (WhileStatement*)NewStatement(kw_while);
			    S_while->src = src;
			    Gobble(kw_while);
			    Gobble(open_round);

			    S_while->Test = ParseBooleanExpr();

			    Gobble(close_round);
			    S_while->Body = ParseThing();

			    NamePopScope(scope);
			    S = ReleaseStackFrame(frame, S_while);
			    break;


	    case kw_for:    frame = SetUpStackFrame();
			    scope = NameMarkScope();
			    S_for = (ForStatement*)NewStatement(kw_for);
			    S_for->src = src;
			    Gobble(kw_for);
			    Gobble(open_round);

			    S_for->Init = ParseThing();   // The initialisor

			    /* (for breaks & continues) */
			    if (tok.en == semi_colon)          // The test
				S_for->Test = NULL;
			    else S_for->Test = ParseBooleanExpr();
			    Gobble(semi_colon);

			    if (tok.en == close_round)         // The incrementor
				S_for->Incr = NULL;
			    else {
				e = ParseExpr(no);
				S_for->Incr = e;
			    }

			    Gobble(close_round);
			    S_for->Body = ParseThing();   // The body

			    NamePopScope(scope);
			    S = ReleaseStackFrame(frame, S_for);
			    break;


	    case kw_do:     frame = SetUpStackFrame();
			    scope = NameMarkScope();
			    S_do = (DoStatement*)NewStatement(kw_do);
			    Gobble(kw_do);

			    S_do->Body = ParseThing();

			    Gobble(kw_while);
			    Gobble(open_round);
			    S_do->src = tok.src;

			    S_do->Test = ParseBooleanExpr();

			    Gobble(close_round);
			    Gobble(semi_colon);
			    NamePopScope(scope);
			    S = ReleaseStackFrame(frame, S_do);
			    break;


	    case kw_if:     frame = SetUpStackFrame();
			    scope = NameMarkScope();
			    S_if = (IfStatement*)NewStatement(st_if);
			    S_if->src = src;
			    Gobble(kw_if);
			    Gobble(open_round);

			    S_if->Test = ParseBooleanExpr();

			    Gobble(close_round);

			    S_if->True = ParseThing();

			    if (tok.en == kw_else) {
				Gobble(kw_else);

				S_if->Fals = ParseThing();
			    }
			    else {
				S_if->Fals = NULL;
			    }

			    NamePopScope(scope);
			    S = ReleaseStackFrame(frame, S_if);
			    break;


	    case kw_else:   ErrorParse(tok.src, "This word can't begin a statement.");
			    return NULL;


	    case kw_switch: frame = SetUpStackFrame();
			    scope = NameMarkScope();
			    S_switch = (SwitchStatement*)NewStatement(st_switch);
			    S_switch->src = src;
			    Gobble(kw_switch);
			    Gobble(open_round);

			    S_switch->Test = ParseIntExpr(no);
			    Gobble(close_round);

			    S_switch->Body = ParseThing();

			    NamePopScope(scope);
			    S = ReleaseStackFrame(frame, S_switch);
			    break;


	    case kw_case:   S_case = (CaseStatement*)NewStatement(kw_case);
			    Gobble(kw_case);

			    S_case->tag = ParseAndEvaluateIntExpr(no);
			    if (S_case->tag == DEFAULT_TAG)
				ErrorParse(tok.src, "Case value is out of range.");

			    Gobble(colon);
			    S_case->Body = ParseThing();
			    S = (Statement*)S_case;
			    break;


	    case kw_default:S_case = (CaseStatement*)NewStatement(kw_case);
			    Gobble(kw_default);

			    S_case->tag = DEFAULT_TAG;
			    Gobble(colon);
			    S_case->Body = ParseThing();
			    S = (Statement*)S_case;
			    break;

	    case kw_break:
			    Gobble(kw_break);
			    S = (Statement*)NewStatement(kw_break);
			    S->src = src;
			    Gobble(semi_colon);
			    break;

	    case kw_continue:
			    Gobble(kw_continue);
			    S = (Statement*)NewStatement(kw_continue);
			    S->src = src;
			    Gobble(semi_colon);
			    break;

	    case kw_goto:   Gobble(kw_goto);
			    S_goto = (GotoStatement*)NewStatement(kw_goto);
			    S_goto->label = strcpy((str)qmalloc(strlen(tok.buf)+1), tok.buf);
			    S_goto->src = src;
                            debugS = S_goto;
			    Gobble(identifier);
			    Gobble(semi_colon);
			    S = (Statement*)S_goto;
			    break;

	    case kw_return: Gobble(kw_return);
			    S_return = (ReturnStatement*)NewStatement(st_return);
			    S_return->src = src;
			    S = (Statement*)S_return;
			    if (tok.en == semi_colon) {
				S_return->e = NULL;
				if (return_type and *return_type != tp_void)
				    ErrorParse(tok.src, "This is not a void function. "
					"You need to return something.");
                                if (DestructorS)
                                    S = DestructorS;	// If there are member
                                    // destructors that we need to call, then
                                    // they have already been set up and we'll
                                    // ignore our S_return and instead jump
                                    // to the destructor calls.
			    }
			    else {
				e = ParEx(comma_op);
				if (Error.err)
				    return S;
				DecorateExpr(e);
				if (return_type == NULL or *return_type == tp_void) {
				    if (e->type[0] != tp_void)
					ErrorType(e->src,
						"This is a void function - you can't "
						"return a value here.");
				    else {
					S_expr = (ExprStatement*)NewStatement(st_expr);
					S_expr->e = e;
					S_expr->next = (Statement*)S_return;
					S_return->e = NULL;
					S = (Statement*)S_expr;
				    }
				}
				else {
				    S_return->e = ExprConvert(e, GetTypePlusRef(return_type));
				    DecorateExpr(S_return->e);
				}
			    }
			    Gobble(semi_colon);
                            if (S_return->e == NULL)
                                S_return->scope = NULL;
                                /* If S_return->e != NULL, then we'll need
                                the local variables in order to compute the
                                return expression, and so the 'destructors'
                                method applies instead. */
			    if (TypeIsBigStruct(return_type)) {
				/* If we're returning a struct which is > 4 bytes, */
				/* then we can't do it in the normal way using the */
				/* EAX register. We need to use the 'pointer to    */
				/* destination' method. */
				Expr* retref = ReturnRefExpr();
				Expr* e = S_return->e;
				if (e->o == oFUNC) {
				    assert(e->u.func->param[0] == NULL);
				    e->u.func->param[0] = retref;
				}
				else {
				    S_return->e = e = NewExpr(op_sing('='), retref, S_return->e);
				    DecorateExpr(e);
				    e->type = return_type;
				    e->src = src;
				}
			    }
			    break;

	    case kw_char: case kw_short: case kw_int: case kw_double: case kw_bool:
	    case kw_float: case kw_long: case kws_directory: case kw_container:
			    /*if (ItLooksLikeWereAtACastFn())
				goto EXPRESSION;*/
			    /* Note that some C++ expressions can be ambiguous:
			    e.g. "int (*fp);"  -  Are we declaring 'fp' to be
			    a pointer to an int, or are we dereferencing 'fp'
			    and casting it to an integer?   My experiments show
			    that ANSI C++ always takes it as a declaration in
			    this case.  This means that if you want to see the
			    result of a cast function in Barbados, you need to
			    specify either:  cout << int(*fp);  or  (int(*fp));
			    */
			    goto DECLARATION;

	    case kw_void: case kw_enum:
	    case kw_class : case kw_struct: case kw_union:
	    case kw_type: case kw_unsigned:
	    case kw_typedef: case kw_static:
	    case kw_extern: case kw_virtual:
            case kw_const: case kw_volatile:
            case kw_inline:
			    DECLARATION:
			    if (default_storage == static_storage and not PredefinitionPhase
					and SourceNeedsLinkSource()) {
				Error.err = err_pleaselinksrc;
				S = NULL;
				break;
			    }
			    ParseDeclaration();
			    if (decl_init_root == NULL)
				return NULL;
			    S_expr = (ExprStatement*)NewStatement(st_expr);
			    S_expr->e = decl_init_root;
			    S_expr->src = src;
			    decl_init_root = NULL;
			    S = (Statement*)S_expr;
			    break;


	    case op_sing('/'):
	    case dot:       PATH_NAME:
			    PathObj = ParsePath();
			    if (PathObj == NULL)
				return NULL;
			    PushTokenBack(kws_identifier_path, PathObj, src);
			    if (PathObj->storage == typedef_storage)
				goto DECLARATION;
			    else if (IsConstructor(PathObj) or IsDestructor(PathObj))
				goto DECLARATION;
			    else goto EXPRESSION;


	    case semi_colon:NextToken();
			    S = (Statement*)NewStatement(st_null);
			    return S;

	    case EOF:       return NULL;

	    case identifier:lookahead = LookAhead();
			    if (lookahead == colon) {
				/* Goto Labels */
				char label[512];

				strcpy(label, tok.buf);
				Gobble(identifier);
				Gobble(colon);
				S = ParseThing();
				if (S == NULL)
				    return NULL;
				LabelDefine(label, S);
				return (Statement*)S;
			    }
                            PathObj = Resolve(tok.buf);
			    /* See the discussion at kw_char: et al. */
			    if (PathObj == NULL)
				goto EXPRESSION;	// generate error
			    else if (lookahead == op_sing('/') or lookahead == double_colon)
				goto PATH_NAME;
			    else if (PathObj->storage == typedef_storage)
				goto DECLARATION;
			    /* else carry on */

	    default:        EXPRESSION:
			    e = ParseExpr(no);
			    if (Error.err) {
				SkipTo(semi_colon);
				goto SEMI_COLON;
			    }
			    if (Error.err)
				goto SEMI_COLON;
			    else if (default_storage == static_storage and
                              (CompilerInvoker == 'I' or CompilerInvoker == 'D')
                                        and e->type[0] != tp_void) {
				e = NewExpr(op_output, e, NULL);
				DecorateExpr(e);
			    }
			    SEMI_COLON:
			    Gobble(semi_colon);
                            if (e->o == oFUNC)
                            	CheckFunctionReturningBigStruct(e);
			    S_expr = (ExprStatement*)NewStatement(st_expr);
			    S_expr->e = e;
			    S_expr->src = src;
			    S = (Statement*)S_expr;
	}
	return S;
}



/*================= Decoration of program trees ==============*/


#define MAX_LOOPS   30
/* The maximum number of nested loops */

struct {
	Statement *Sbreak, *Scontinue;
} Loop[MAX_LOOPS];

int lp_idx;


static void DecorateSwitch(SwitchStatement* S_switch);


static void DecorateCheckIf(IfStatement* S_if)
/* Check this 'if' statement for having a constant test expression. */
/* If so, convert it into a 'goto'. */
{       Expr* e;
	bool val;

	/* Can we get a constant value out? */
	if (S_if->st != kw_if)
	    return;
	e = S_if->Test;
	if (e == NULL)
	    val = yes;
	else {
	    if (e->o == oConstant and e->tp == tp_int)
		val = (e->u.i != 0);
	    else return;
	}

	/* Is it true or false? */
	S_if->st = st_null;
	if (val)
	    S_if->next = S_if->True;
	else S_if->next = S_if->Fals;
}


static void DecorateStatement(Statement *S, Statement *after)
{       Statement *S_next, *Test, *Incr, *next;
	SwitchStatement* S_switch;
        WhileStatement* S_while;
	CaseStatement* S_case;
	GotoStatement* S_goto;
	ExprStatement* S_expr;
	AutoNamedobj* scope;
        ForStatement* S_for;
	DoStatement* S_do;
	IfStatement* S_if;

	if (S == NULL)
	    return;         /* Can occur with errors. */

	do {
	    S_next = S->next;
	    if (S_next)
		next = S_next;
	    else next = after;

	    switch (S->st) {
		case kw_for:
			/* Initialisations: */
			S_for = (ForStatement*)S;


			/* Rearranging: */
                        scope = S_for->Init ? S_for->Init->scope : S_for->scope;
			if (S_for->Body == NULL) {
			    S_for->Body = (Statement*)NewStatement(st_null);
                            S_for->Body->scope = scope;
                        }
			if (S_for->Test) {
			    S_if = (IfStatement*)NewStatement(st_if);
			    S_if->Test = S_for->Test;
			    S_if->True = S_for->Body;
			    S_if->Fals = next;
			    S_if->src = S_if->Test->src;
                            S_if->scope = scope;
			    Test = (Statement*)S_if;
			}
			else Test = S_for->Body;
			if (S_for->Incr) {
			    S_expr = (ExprStatement*)NewStatement(st_expr);
			    S_expr->e = S_for->Incr;
			    S_expr->next = Test;
			    S_expr->src = S_expr->e->src;
                            S_expr->scope = scope;
			    Incr = (Statement*)S_expr;
			}
			else Incr = Test;
			S->st = st_null;
			IprRegisterStatement(S);
			if (S_for->Init == NULL)
			    S->next = Test;
			else S->next = S_for->Init;


			/* Setting up the loop: */
			lp_idx++;
			Loop[lp_idx].Scontinue = Incr;
			Loop[lp_idx].Sbreak = next;


			/* Decorating: */
			if (S_for->Init)
			    DecorateStatement(S_for->Init, Test);
			DecorateStatement(S_for->Body, Incr);
			DecorateCheckIf((IfStatement*)Test);


			/* Closing: */
			lp_idx--;
			break;


		case kw_while:
			/* Initialising: */
			S_while = (WhileStatement*)S;
			lp_idx++;
			Loop[lp_idx].Scontinue = S;
			Loop[lp_idx].Sbreak = next;
			if (S_while->Body == NULL)
			    S_while->Body = (Statement*)NewStatement(st_null);


			/* Rearranging: */
			S_if = (IfStatement*)NewStatement(st_if);
			S_if->Test = S_while->Test;
			S_if->True = S_while->Body;
			S_if->Fals = next;
			S_if->src = S_if->Test->src;
                        S_if->scope = S_while->scope;
			S->st = st_null;
			IprRegisterStatement(S);
			S->next = S_if;


			/* Decorating: */
			DecorateStatement(S_while->Body, (Statement*)S_if);
			DecorateCheckIf(S_if);


			/* Closing: */
			lp_idx--;
			break;


		case kw_do:
			/* Initialising: */
			S_do = (DoStatement*)S;
			lp_idx++;
			Loop[lp_idx].Scontinue = S;
			Loop[lp_idx].Sbreak = next;
			if (S_do->Body == NULL) {
			    S_do->Body = (Statement*)NewStatement(st_null);
                            S_do->Body->scope = S_do->scope;
                        }


			/* Rearranging: */
			S_if = (IfStatement*)NewStatement(st_if);
			S_if->Test = S_do->Test;
			S_if->True = S_do->Body;
			S_if->Fals = next;
                        S_if->src = S_do->src;
                        S_if->scope = S_do->scope;
			S->st = st_null;
			IprRegisterStatement(S);
			S->next = S_do->Body;


			/* Decorating: */
			DecorateStatement(S_do->Body, (Statement*)S_if);
			DecorateCheckIf(S_if);


			/* Closing: */
			lp_idx--;
			break;


		case st_if:
			/* Initialising: */
			S_if = (IfStatement*)S;

			/* Decorating: */
			if (S_if->True == NULL)
			    S_if->True = next;
			else DecorateStatement(S_if->True, next);
			if (S_if->Fals == NULL)
			    S_if->Fals = next;
			else DecorateStatement(S_if->Fals, next);
			DecorateCheckIf(S_if);

			break;


		case st_switch:
			/* Initialising: */
			S_switch = (SwitchStatement*)S;
			lp_idx++;
			Loop[lp_idx].Scontinue = Loop[lp_idx-1].Scontinue;
                        Loop[lp_idx].Sbreak = next;

			DecorateSwitch(S_switch);
			DecorateStatement(S_switch->Body, next);

			lp_idx--;
			break;


		case kw_break:
			if (lp_idx <= 0)
			    ErrorParse(S->src, "Not inside loop");
			S->next = Loop[lp_idx].Sbreak;
			if (S->next == NULL)
			    S->next = after;
			S->st = st_null;
			IprRegisterStatement(S);
			break;


		case kw_continue:
			if (lp_idx <= 0 or Loop[lp_idx].Scontinue == NULL)
			    ErrorParse(S->src, "Not inside loop");
			else S->next = Loop[lp_idx].Scontinue;
			S->st = st_null;
			IprRegisterStatement(S);
			break;


		case kw_goto:
			S_goto = (GotoStatement*)S;
			S_goto->next = LabelFind(S_goto->label);
			if (S_goto->next == NULL)
			    ErrorParse(S->src, "Label %s is undefined.", S_goto->label);
			S->st = st_null;
			IprRegisterStatement(S);
			break;


		case st_expr:
			break;


		case st_null:
			break;


		case st_return:
			break;


		case kws_instruction:
		case kw_case:
			S_case = (CaseStatement*)S;
			DecorateStatement(S_case->Body, next);
			S->st = st_null;
			IprRegisterStatement(S);
			S->next = S_case->Body;
			break;


		case kws_code:
		case kw_void:
			break;


		default:assert(false);
			break;

	    }
	    if (S_next)
		S = S_next;
	    else break;
	} forever;

	if (S->next == NULL)
	    S->next = after;
}


static void DecorateEntity(Statement *S, bool NeedReturn)
/* Iterate over this whole thing, converting all loops to 'if's, */
/* putting in jump locations to statement pointers, putting in   */
/* all the 'next' fields and decorating expressions.       */
{       ReturnStatement* S_return;

	lp_idx = 0;
	if (S == NULL)
	    return;
        if (NeedReturn) {
            /* Code fragments come without a st_return and need one,
            whereas function definitions already have an st_return. */
            S_return = (ReturnStatement*)NewStatement(st_return);
            S_return->e = NULL;
            S_return->src = NULL;
            S_return->scope = NULL;
        }
        else S_return = NULL;
	DecorateStatement(S, (Statement*)S_return);
}






/*================= Decorating Switches ===============*/

static SwitchStatement* CurSwitch;

static void SwitchAddCase(CaseStatement* S_case)
/* Add this case statement to the linked list. */
{	switchcase_type r;

	if (S_case->tag == DEFAULT_TAG) {
	    if (CurSwitch->Default == CurSwitch->next)
		CurSwitch->Default = S_case->Body;
	    else ErrorParse(S_case->src, "You have two \"default:\"s. ");
	}
	else {
	    r = (switchcase_type)qmalloc(sizeof(struct switchcase_node));
	    r->tag = S_case->tag;
	    r->Body = S_case->Body;
	    r->next = CurSwitch->Cases;
	    CurSwitch->Cases = r;
	}
}


static void RetrieveCases(Statement *S)
/* Extract all the 'cases' out of this switch. */
{       CaseStatement* S_case;

	S_case = (CaseStatement*)S;
	while (S_case) {
	    if (S_case->st == kw_case) {
		SwitchAddCase(S_case);
		RetrieveCases(S_case->Body);
	    }
	    else if (S_case->st == kws_instruction)
		RetrieveCases(S_case->Body);
	    S_case = (CaseStatement*)S_case->next;
	}
}


static void DecorateSwitch(SwitchStatement* S_switch)
/* Set up the fields in this switch-type.  If there's no 'default' */
/* case, control will go to 'defS'. */
{
	CurSwitch = S_switch;
	CurSwitch->Default = CurSwitch->next;
	RetrieveCases(S_switch->Body);
}


static bool ObjIsOverloaded(Namedobj *obj)
{       str origname=obj->name;

        if (*obj->type != tp_function)
            return no;
        for (obj=obj->next; obj; obj=obj->next) {
            if (streq(obj->name, origname))
                return yes;
        }
        return no;
}


static void ReportUndefinedFunction()
{       char buf[512];

	if (IsConstructor(UndefinedFunctionCall))
            ErrorType(UndefinedFunctionCallSrc, "The constructor  \"%s\"  is undefined.",
                ObjDeclarationString(UndefinedFunctionCall, buf, sizeof(buf)));
        else if (ObjIsOverloaded(UndefinedFunctionCall)) {
            ErrorType(UndefinedFunctionCallSrc, "The function  \"%s\"  is undefined.",
                ObjDeclarationString(UndefinedFunctionCall, buf, sizeof(buf)));
        }
	else ErrorType(UndefinedFunctionCallSrc, "The function  \"%s\"  is undefined.",
			UndefinedFunctionCall->name);
}




/*================== Printing Program Graphs: ==================*/

static void Pri(int indent)
{	int i;

	for (i=0; i < indent; i++)
	    Pr("    ");
}


static void PrintE(Expr* e)
{
	Pr("expr");
}


str ParseOpToString(char op)
{
	switch (op) {
	    case op_sing('+'):  return "+";
	    case op_sing('-'):  return "-";
	    case op_sing('*'):  return "*";
	    case op_sing('/'):  return "/";
	    case op_sing('<'):  return "<";
	    case op_sing('>'):  return ">";
	    case op_sing('='):  return "=";
	    case op_doub('<'):  return "<<";
	    case op_doub('>'):  return ">>";
	    default:    return "?op?";
	}
}


static void PrintS(Statement *S, int indent)
{       ReturnStatement* S_return;
	WhileStatement* S_while;
	CaseStatement* S_case;
	ExprStatement* S_expr;
	ForStatement* S_for;
	DoStatement* S_do;
	IfStatement* S_if;

	START:
	switch (S->st) {
	    case kw_for:
		    Pri(indent);
		    S_for = (ForStatement*)S;
		    Pr("for (");
		    PrintS(S_for->Init, indent + 1);
		    PrintE(S_for->Test);
		    Pr("; ");
		    PrintE(S_for->Incr);
		    Pr(")\n");
		    PrintS(S_for->Body, indent + 1);
		    Pr("\n");
		    break;

	    case kw_while:
		    Pri(indent);
		    S_while = (WhileStatement*)S;
		    Pr("while (");
		    PrintE(S_while->Test);
		    Pr(")\n");
		    PrintS(S_while->Body, indent + 1);
		    Pr("\n");
		    break;

	    case kw_do:
		    Pri(indent);
		    S_do = (DoStatement*)S;
		    Pr("do\n");
		    PrintS(S_do->Body, indent + 1);
		    Pri(indent);
		    Pr("while (");
		    PrintE(S_do->Test);
		    Pr(");\n");
		    break;

	    case st_if:
		    Pri(indent);
		    S_if = (IfStatement*)S;
		    Pr("if (");
		    PrintE(S_if->Test);
		    Pr(")\n");
		    PrintS(S_if->True, indent + 1);
		    if (S_if->Fals) {
			Pri(indent);
			Pr("else\n");
			Pri(indent);
			PrintS(S_if->Fals, indent + 1);
		    }
		    break;

	    case kw_break:
		    Pri(indent);
		    Pr("break\n");
		    break;

	    case kw_continue:
		    Pri(indent);
		    Pr("continue\n");
		    break;

	    case st_return:
		    Pri(indent);
		    S_return = (ReturnStatement*)S;
		    Pr("return ");
		    PrintE(S_return->e);
		    Pr(";\n");
		    break;

	    case kw_case:
		    Pri(indent);
		    S_case = (CaseStatement*)S;
		    Pr("case %d:\n", S_case->tag);
		    break;

	    case st_expr:
		    Pri(indent);
		    S_expr = (ExprStatement*)S;
		    PrintE(S_expr->e);
		    Pr(";\n");
		    break;

	    default:Pri(indent);
		    Pr("???;\n");
		    break;
	}

	if (S->next) {
	    S = S->next;
	    goto START;
	}
}


static void PrintStatement(Statement *S)
{
	PrintS(S,0);
}




/*============== Module Initialisation ================*/

static void CompilerInit(void)
/* Initialises the compiler. */
{	static bool initialised;

	if (initialised)
	    return;

	LexicalInit();
	DeclareInit();
	MethodInit();

	CurrentSource = NULL;
	debugging = 0;

	MainEntity = (FunctionNamedobj*)NameDeclare(curdir, "Barbados>",
                        void_typstr, straight_fn, NAME_CREATEMAKE);
	two_dots_obj = (StaticNamedobj*)NameDeclare(
                curdir, "..", direc_typstr, static_storage, NAME_CREATEMAKE);
        context_fn = NULL;
	initialised = yes;
}







/*==================== Main Parse routines =================*/

interface char CompilerInvoker;
	/* Is this compilation happening directly at the Barbados */
	/* prompt?  (As opposed to inside Make, inside a runtime  */
	/* call to compile(str s), or in the predefinition phase).*/


static void* CompileEntity(void)
/* Parse one compileable entity from the text set up for us. */
{       Statement *S;
	void *block;

	MakeEntityReset();
	ExprTreeClear();
	LabelClear();
        if (qn == NULL)
            qinit();
	InsideCompiler = 1;
	expression_type = NULL;
	auto_offset = 0;
	max_auto_offset = 0;
	decl_init_root = NULL;
	default_storage = static_storage;
	declarator_class = NULL;
	return_type = NULL;
	UndefinedFunctionCall = NULL;
	block = NULL;

	S = ParseThing();
	if (S) {
	    DecorateEntity(S,yes);
	    if (UndefinedFunctionCall)
                ReportUndefinedFunction();
                // We do it at this late stage because it solves complications
                // to do with defining recursive functions.
	    else if (not Error.err) {
                InsertDestructorCalls();
		block = IprCompile(S, NULL);
            }
	}

	if (expression_type and *expression_type == tp_void)
	    expression_type = NULL;
	InsideCompiler = 0;
	qfreeall();

	if (Error.err)
	    tok.en = EOF;      // To check for the compiler being re-entered.
	return block;
}


interface machinecode_type Compile(str source, Directory* context, char Invoker)
/* Compile this one str.  Return a "void f(void)" function ptr*/
/* which you can call if it's an executable code fragment, or */
/* is NULL if you're just doing a variable declaration or     */
/* function definition. */
{	void* block;

        CompilerInvoker = Invoker;
	CompilerInit();
        NameInit();
	declaration_directory = context;
	SourceClearStashed();
	qinit();
        if (Invoker == 'D')
            DebuggerEnterLocalsIntoScope();
	CompileBegin(source);
	block = NULL;
	until (tok.en == EOF) {
	    if (block)
		free(default_heap, block);
	    block = CompileEntity();
	    if (Error.err)
		tok.en = EOF;
	}
        qfreeall();
	SourceStoreStashed();
	IprClearDebugInfo();
        FreeMacroExpansions();
	return (machinecode_type)block;
}


interface void* UserCompileString(str source)
/* Compile this str and return a pointer to the resulting code. */
/* Used for user calls of the compiler.    Careful - we don't want */
/* to overwrite the current source.  */
{
	CurrentSourceIsPersistent = no;
	return Compile(source, curdir, 'R');
}


interface void CompileAndRun(str source)
{       machinecode_type fn;

        fn = (machinecode_type)Compile(source, curdir, 'I');
        fn();
}


