#include <windows.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include "barbados.h"
#include "exception.h"
#include "modules.h"
#include "textwin.h"
#include "runlib.h"
#include "source.h"
#include "array.h"
#include "debug.h"
#include "join.h"
#include "name.h"



#define each_localvar(dinfo)	    dvar=dinfo->localvarmap; dvar->name; dvar=dvar->Next()

extern CONTEXT *RegisterImage;
	/* A struct telling you the values of all the CPU registers. */

static SplitWindow *MainWin;
        /* The container of subwins: */

static struct {
        int x, y;
        ScrollWin *origWin;
} Pmen;
        /* Information about selecting subwindows */

void DebugPopupMenu(ScrollWin *owner, int x, int y);

static str DebuggerHelp = "\1The Debugger:\1\n"
"Right-click for the menu.\n\n"
"Keys:\n"
"ALT-DOWN = F8 = Step over line\n"
"ALT-RIGHT = F7 = Step into line\n"
"ALT-LEFT = Step out of function\n"
"ALT-UP = F9 = Continue running (Go)\n"
"CTRL-PAGEUP = Move one function up the call stack\n"
"CTRL-PAGEDOWN = Move one function down the call stack\n"
"FULL STOP = Toggle breakpoint on the current line (use UP/DOWN to "
"move to different lines)\n\n"
"You can also toggle breakpoints by clicking in the left-hand side of "
"the source-window.\n\n"
"Ordinary typing + ENTER will enter a C++ expression into the Watch window.\n\n";





/*--------------------- SrcLine's: -----------------------*/

// A SrcLine represents one line of C++ source-code,
// namely a string, plus a set of instructions that
// are compiled from it.

class SrcLine {
public:
	str s;
	ins_type *Instructions;
};






/*--------------------- ProgRow's: -----------------------*/

// A ProgRow is a single line of source-code or disassembly code,
// i.e. something you might be able to step over or set a breakpoint at.


class ProgRow : public Row {
public:
	int LineNo;
	ins_type ins;
        bool Breakpointable;

	void Paint(int x1, int y1, int x2, int y2);
	void Measure();
	void Measure(int *widthp, int *heightp)
		{ Measure(); *widthp = width; *heightp = height; }
	virtual bool Mousestroke(int op, int x, int y);
};




class ProgWin : public RowWin {

public:
	enum { mo_unknown, mo_src, mo_dis, mo_mixed, mo_displus } Mode;
	Namedobj* this_obj, *debug_obj;
	DebugInfo* dinfo;
	TfcRect CursorRect;
        FrameNode *Frame;
	ProgRow *Cursor;
	str this_source;
	int FontPHeight;
	SrcLine* Lines;
	TfcFont font;
	int TabSize;
	struct {
	    ins_type ins;
	    int LineNo;
	} Execution;


	virtual void PointerHasChanged(Row* Old, Row* New);
		// To update all references in the application

	virtual void CursorPaint();
		// How to paint a cursor.

	ProgWin(str caption, int width, int height);
		// Create the window

	~ProgWin();
		// Destructor

	void FreeRowsAndClear();
		// Free all rows and clear the screen.

	void FreeLines();
		// Free 'Lines'

	void InstallSource(str source, DebugInfo* dinfo);
		// Install this source-str in the debug window.

	void EnterDebugger(Namedobj* obj, DebugInfo* _dinfo, int LineNo,
			FrameNode *Frame);
		// Display the debug window with this context and get input.

        virtual bool Mousestroke(int op, int x, int y);
        virtual bool Keystroke(int key);
                //

        void SetCursor();
                // Set the cursor rectangle according to the cursor line
};

static ProgWin    *SrcWindow;



static str MyDisassemble(char buf[], ins_type ins)
{
	sprintf(buf, "0x%x| ", ins);
	if (ins == NULL)
	    return buf;
	DebugDisassemble(buf + strlen(buf), ins);
	return buf;
}


void ProgRow::Paint(int x1, int y1, int x2, int y2)
{	ProgWin *W=(ProgWin*)parent;
	int foreground, background;
	bool breakpoint;
	char buf[1024];
	bool current;

	current = (ins == W->Execution.ins) or
		    (W->Mode == ProgWin::mo_src and LineNo == W->Execution.LineNo
                    and LineNo != -1);
	if (W->Mode == ProgWin::mo_src)
	    breakpoint = FindBreakpoint(W->this_obj, LineNo);
	else breakpoint = FindCodeBreakpoint(ins);
	if (current) {
	    foreground = BLACK;
	    background = GREEN;
	}
	else if (breakpoint) {
	    foreground = WHITE;
	    background = DARK(RED);
	}
	else {
	    foreground = YELLOW;
	    background = ins ? DARK(GREEN) : DARK(BLUE);
	}
	if (ins and LineNo >= 0 and LineNo < Array_Size(W->Lines)) {
            str s = W->Lines[LineNo].s;
            while (isspace(*s))
                s++;
	    DrawString(ExpandTabs(s, buf, W->TabSize), -1, W->font,
		    30*8, 0, x2, y2, YELLOW, DARK(BLUE));
	    DrawString(MyDisassemble(buf, ins), -1, W->font,
		    0, 0, 30*8, y2, foreground, background);
	}
	else if (ins) {
	    DrawString(MyDisassemble(buf, ins), -1, W->font,
		    0, 0, x2, y2, foreground, background);
	}
	else if (LineNo >= 0 and LineNo < Array_Size(W->Lines)) {
	    DrawString(ExpandTabs(W->Lines[LineNo].s, buf, W->TabSize), -1, W->font,
		    10, 0, x2, y2, foreground, background);
            DrawRectangle(0,0,10,y2,background);
            int v=4,w=12,c=3;
            if (breakpoint)
                DrawPolygon(RED, WHITE, v,0, 0,v, 0,w-v, v,w, w-v,w,
                                w,w-v, w,v, w-v,0, TFC_ENDOFPOLY);
            else if (Breakpointable)
                DrawEllipse(w/2-c, w/2-c, w/2+c+1, w/2+c+1, GREEN, BLACK);
	}
	else DrawString("", -1, W->font, x1, y1, x2, y2, foreground, background);
}


void ProgRow::Measure()
/* Measure and set the display dimensions of this row. */
{	ProgWin *W=(ProgWin*)parent;
	int width4, height4;
	char buf[1024];

	if (W == NULL) {
	    width = 60;
	    height = 13;
	    return;
	}
	if (ins and LineNo >= 0 and LineNo < Array_Size(W->Lines)) {
	    TextDimensions(ExpandTabs(W->Lines[LineNo].s, buf, W->TabSize), -1, W->font,
		    &width4, &height4);
	    width4 += 30*8;
	}
	else if (ins) {
	    TextDimensions(MyDisassemble(buf, ins), -1, W->font,
		    &width4, &height4);
	}
	else if (LineNo >= 0 and LineNo < Array_Size(W->Lines)) {
	    TextDimensions(ExpandTabs(W->Lines[LineNo].s, buf, W->TabSize), -1, W->font,
		    &width4, &height4);
	}
	else {
	    width4 = height4 = 0;
	}
        if (height4 == 0)
            height4 = 13;
	width = width4;
	height = height4;
}


void ProgWin::CursorPaint()
{	int x1=CursorRect.left;
	int y1=CursorRect.top;
	int x2=CursorRect.right;
	int y2=CursorRect.bottom;

	DrawPolygon(RED, NOCOLOUR, x1, y1, x1, y2, x2, (y1+y2)/2, TFC_ENDOFPOLY);
}


void ProgWin::SetCursor()
{
        CursorRect.left = 0;
        CursorRect.top = Cursor->y;
        CursorRect.right = 8;
        CursorRect.bottom = Cursor->y + Cursor->height;
        ScrollWin::SetCursor(CursorRect);
}


void ProgWin::PointerHasChanged(Row* Old, Row* New)
/* To update all references in the application. */
{
	if (Cursor == Old)
	    Cursor = (ProgRow*)New;
}


ProgWin::ProgWin(str caption, int width, int height)
	: RowWin(caption, width, height)
{
	Mode = mo_src;
	Lines = NULL;
	this_obj = debug_obj = NULL;
	dinfo = NULL;
	Cursor = NULL;
	this_source = NULL;
	FontPHeight = 13;
	TabSize = 4;
	Execution.ins = NULL;
	Execution.LineNo = 0;
	font = TfcFindFont(13, no, no, no, "Courier");
        CursorRect.left = 0;
        CursorRect.top = 0;
        CursorRect.right = 0;
        CursorRect.bottom = 0;
}


ProgWin::~ProgWin()
{
	FreeRowsAndClear();
	FreeLines();
}


void ProgWin::FreeRowsAndClear()
{	Row *ln;

	SuppressPaints();
	while (root) {
	    ln = root->next;
	    free(anon_heap, root);
	    root = ln;
	}
	Cursor = NULL;
        focus = NULL;
	Clear();
}


void ProgWin::FreeLines()
{	int i;

	for (i=0; i < Array_Size(Lines); i++) {
	    if (Lines[i].Instructions)
		Array_Free(Lines[i].Instructions);
	}
	Array_Free(Lines);
}


bool ProgWin::Mousestroke(int op, int x, int y)
{
        if (op == MOUSE2_PRESS) {
            Pmen.x = x;
            Pmen.y = y;
            Pmen.origWin = this;
            DebugPopupMenu(this,x,y);
        }
        else RowWin::Mousestroke(op,x,y);
        return yes;
}


void ProgWin::InstallSource(str source, DebugInfo* _dinfo)
/* Install this source-str in the debug window. */
{	ins_type ins, ins_end, *Instructions;
	DebugInfo::linemap_node* linemap;
	ProgRow *row, *newbie;
	int i,j;
	str s;

	/* Free whatever we had before in SrcWindow[]: */
	FreeRowsAndClear();
	FreeLines();


	/** Tie this window to this debuginfo: */
	dinfo = _dinfo;


        /* The pure machine-code form used if we have no dinfo: */
        if (dinfo == NULL or source == NULL) {
            ins_end = (ins_type)RegisterImage->Eip + 512;
            row = NULL;
            for (ins=(ins_type)RegisterImage->Eip;
                        ins < ins_end; ins=DebugNextInstruction(ins)) {
                newbie = new ProgRow;
                newbie->LineNo = -1;
                newbie->ins = ins;
                newbie->Breakpointable = no;
                row = (ProgRow*)Insert(row, newbie);
                if (ins[0] == (uchar)0x90 and ins[1] == (uchar)0x90)
                    break;	// This is the end-of-function signature.
            }
            return;
        }


	/*** Put 'source' into 'Line': ***/
	i = 0;
	s = source;
	Lines = NULL;
	do {
	    Array_Idx(Lines, i);
	    Lines[i].Instructions = NULL;
	    Lines[i].s = s;
	    s = strchr(s, '\n');
	    if (s == NULL)
		break;
	    i++;
	    s++;
	} forever;
        for (i=0; (linemap=&dinfo->linemap[i])->LineNo >= 0; i++) {
            ins_end = dinfo->linemap[i+1].Code;
            for (ins=linemap->Code; ins < ins_end; ins = DebugNextInstruction(ins)) {
                if (ins[0] == (uchar)0x90 and ins[1] == (uchar)0x90)
                    break;	// This is the end-of-function signature.
                Array_Add(Lines[linemap->LineNo].Instructions, ins);
            }
        }


	/*** Build 'SrcWindow' from this dinfo: ***/
	row = NULL;
	if (Mode == mo_src) {
	    s = source;
	    i = 0;
	    do {
		newbie = new ProgRow;
		newbie->LineNo = i++;
		newbie->ins = NULL;
                newbie->Breakpointable = no;
		row = (ProgRow*)Insert(row, newbie);
		while (*s and *s != '\n')
		    s++;
		if (*s == '\n')
		    s++;
	    } while (*s);
	    for (i=0; (linemap=&dinfo->linemap[i])->LineNo >= 0; i++) {
                ProgRow *guess=(ProgRow*)root;
                row = guess;
                for (row=(ProgRow*)guess->next; row != guess; row=row ? (ProgRow*)row->next : (ProgRow*)root) {
                    if (row->LineNo == linemap->LineNo) {
                        row->Breakpointable = yes;
                        guess = row;
                        break;
                    }
                }
            }
	}
	else if (Mode == mo_dis or Mode == mo_displus) {
	    for (i=0; (linemap=&dinfo->linemap[i])->LineNo >= 0; i++) {
		ins_end = dinfo->linemap[i+1].Code;
		for (ins=linemap->Code; ins < ins_end; ins = DebugNextInstruction(ins)) {
		    newbie = new ProgRow;
		    newbie->LineNo = (ins == linemap->Code and Mode == mo_displus)
				? linemap->LineNo : -1;
		    newbie->ins = ins;
		    row = (ProgRow*)Insert(row, newbie);
		    if (ins[0] == (uchar)0x90 and ins[1] == (uchar)0x90)
			break;	// This is the end-of-function signature.
		}
	    }
	}
	else if (Mode == mo_mixed) {
	    for (i=0; i < Array_Size(Lines); i++) {
		newbie = new ProgRow;
		newbie->LineNo = i;
		newbie->ins = NULL;
		row = (ProgRow*)Insert(row, newbie);
		Instructions = Lines[i].Instructions;
		for (each_ael(ins, Instructions, j)) {
		    newbie = new ProgRow;
		    newbie->LineNo = -1;
		    newbie->ins = ins;
		    row = (ProgRow*)Insert(row, newbie);
		}
	    }
	}
}





/*--------------------- The text-based windows: --------------------*/

class TextDbgWin : public TextWin {
public:
	TextDbgWin(str name);
        bool Mousestroke(int op, int x, int y);
        bool Keystroke(int key);
};

static TextDbgWin *VarWindow;
static TextDbgWin *RegWindow;



TextDbgWin::TextDbgWin(str name)
        : TextWin(30,30, NULL, 14,yes,"Courier")
{
        SetTitle(name);
}


bool TextDbgWin::Mousestroke(int op, int x, int y)
{
        if (op == MOUSE2_PRESS) {
            Pmen.x = x;
            Pmen.y = y;
            Pmen.origWin = this;
            DebugPopupMenu(this,x,y);
        }
        else TextWin::Mousestroke(op,x,y);
        return yes;
}


bool TextDbgWin::Keystroke(int key)
{
        if (key == F1) {
            ViewHelp(DebuggerHelp);
            return yes;
        }
        if (MainWin->Keystroke(key))
            return yes;
        if (EditorWin::Keystroke(key))
            return yes;
        return response = key;
}






/*----------------- Local variables: ---------------*/

static void FrameDisplay(FrameNode *Frame)
/* Display a window of all local variables. */
{	DebugInfo::localvarmap_node* dvar;
	char buf[1024];
	char *valp;

	/* The assembler mode: */
	if (SrcWindow->dinfo == NULL)
	    return;

	/* Prepare VarWindow: */
	if (VarWindow) {
            if (VarWindow->ShowState != tfc_shown)
                return;
	    VarWindow->Clear();
            VarWindow->Show();
        }
	else {
            VarWindow = new TextDbgWin("Locals");
        }

	/* Print them: */
	VarWindow->Printf("Frame pointer = 0x%08X\n", Frame->BP);
	for (each_localvar(SrcWindow->dinfo)) {
	    if (dvar->storage == local_static)
		valp = (char*)dvar->offset;
	    else valp = (char*)Frame->BP + dvar->offset;
	    VarWindow->Printf("%s : %s  \t[offset=%d]",
		    dvar->name,
		    ValueToString(buf, sizeof(buf), valp, (Type)dvar->type, yes),
		    dvar->offset);
	}
        VarWindow->UpdateScrollbars(0,0);
        VarWindow->PaintWhole();
	VarWindow->Refresh();
}





/*----------------- The Watch Window: ---------------*/

#if 0

static int InstructionsBetween(ins_type a, ins_type b)
/* Count how many machine-code instructions there are between */
/* 'a' and 'b'  (a < b). */
{	int c=0;

	until (a >= b)
	    a = DebugNextInstruction(a), c++;
	return c;
}


static ins_type GuessPrevInstruction(ins_type ins, int c)
/* Subtract 'c' instructions from 'ins'. */
{	ins_type p;

	if (InstructionsBetween(ins - 2*c - 5, ins) >= c)
	    p = ins - 2*c;
	while (InstructionsBetween(p, ins) < c)
	    p--;
	while (InstructionsBetween(p-1, ins) == c)
	    p--;
	return p;
}


static ins_type PrevInstruction(ins_type ins1)
{	ins_type ins, ins2;

	if (dinfo)
	    ins = dinfo->linemap[0].Code;
	else ins = Execution.ins;
	if (ins < ins1) {
	    while ((ins2 = DebugNextInstruction(ins)) != NULL)
		ins = ins2;
	    return ins;
	}
	else {
	    return GuessPrevInstruction(ins, 1);
	}
}

#endif



DebugInfo::localvarmap_node* DebugInfo::localvarmap_node::Next()
/* Move 'var' to the next variable. */
{       DebugInfo::localvarmap_node *dvar;

	//dvar = (localvarmap_type)(dvar->type + LengthOfTypeString((Type)dvar->type) + strlen(dvar->name));
	dvar = (localvarmap_node*)(name + strlen(name) + 1);
	dvar = (localvarmap_node*)((((int)dvar - 1) | 3) + 1);
	return dvar;
}


static void PutLocalVariablesIntoNameSpace(void* BP)
{	DebugInfo::localvarmap_node *dvar;
        extern void qinit();
        StaticNamedobj* obj;

	if (SrcWindow->dinfo == NULL)
	    return;
        for (each_localvar(SrcWindow->dinfo)) {
	    obj = (StaticNamedobj*)NameDeclare(NULL, dvar->name, (Type)dvar->type, local_static, 0);
            obj->storage = static_storage;
	    if (dvar->storage == local_static)
		obj->location = (void*)(dvar->offset);
	    else
		obj->location = (void*)(dvar->offset + (char*)BP);
	}
}


interface void DebuggerEnterLocalsIntoScope()
{
        PutLocalVariablesIntoNameSpace(Frame.BP);
}





/*----------------- Registers: ---------------*/

static void DisplayReg(str name, unsigned int value)
{	
	RegWindow->Printf("%s : 0x%08X   = %d", name, value, value);
}


static int SafeDeref(void *mem, bool *Undefinedp)
{
	__try {
	    *Undefinedp = no;
	    return *(int*)mem;
	}
	__except(yes) {
	    *Undefinedp = yes;
	    return 0;
	}
}


static void RegisterDisplay(void)
/* Display a window of all local variables. */
{	struct disassembledins_node I;
	char buf[1024], *s;
	bool Undefined;
	int value;
	void *mem;

	/* Clear what we had before: */
	if (RegWindow == NULL) {
	    RegWindow = new TextDbgWin("Registers");
        }
	else {
            if (RegWindow->ShowState != tfc_shown)
                return;
            RegWindow->Clear();
            RegWindow->Show();
        }

	/* Print the general registers: */
        DisplayReg("AX", RegisterImage->Eax);
        DisplayReg("BX", RegisterImage->Ebx);
        DisplayReg("CX", RegisterImage->Ecx);
        DisplayReg("DX", RegisterImage->Edx);
        DisplayReg("SI", RegisterImage->Esi);
        DisplayReg("DI", RegisterImage->Edi);
	DisplayReg("BP", RegisterImage->Ebp);
	DisplayReg("SP", RegisterImage->Esp);
	DisplayReg("IP", RegisterImage->Eip);

	/* Print the next instruction: */
	mem = DebugDisassembleToStruct((ins_type)RegisterImage->Eip, &I);
	s = buf;
	s += sprintf(buf, "%s ", I.mnemonic);
	if (I.reg >= 0 and I.direction)
	    s += sprintf(s, "%s, ", Reg8ToString[I.reg]);
	if (I.modregrm_present)
            s += sprintf(s, "[%p] ", mem);
	if (I.reg >= 0 and I.direction == 0) {
	    if (I.modregrm_present)
		s[-1] = ',', *s++ = ' ';
	    s += sprintf(s, "%s ", Reg8ToString[I.reg]);
	}
	if (I.imm_present) {
	    if (I.modregrm_present)
		s[-1] = ',', *s++ = ' ';
	    if ((int)I.imm == (short)I.imm)
		s += sprintf(s, "%d ", I.imm);
	    else s += sprintf(s, "0x%x ", I.imm);
	}
	while (s[-1] == ' ' or s[-1] == ',')
	    s--;
	if (I.modregrm_present and I.mod != 3) {
	    Undefined = no;
	    value = SafeDeref(mem, &Undefined);
	    s += sprintf(s, "     [%p] = ", mem);
	    if (Undefined)
		s += sprintf(s, "Unreadable");
	    else if ((int)value == (short)value)
		s += sprintf(s, "%d", value);
	    else s += sprintf(s, "0x%x", (void*)value);
	}
	*s = '\0';
	RegWindow->Printf("");
	RegWindow->Printf("%s", buf);
        RegWindow->PaintWhole();

	/* Display the window: */
	RegWindow->Refresh();
}




/*----------------- The Call Chain: ---------------*/

class InvocationRow : public Row {
public:
        FrameNode Frame;        // The instruction ptr + BP
        Namedobj *obj;          // The named object
        str args;               // A string form of the arguments

        virtual void Paint(int x1, int y1, int x2, int y2);
        virtual bool Mousestroke(int op, int x, int y);
        virtual bool Keystroke(int key);
        virtual void Measure(int *widthp, int *heightp);
        InvocationRow(ins_type ip, void** bp, Namedobj *obj);
        InvocationRow(str msg);
        ~InvocationRow() { free(args); }
};


class CallChainWin : public RowWin {
public:
        TfcFont font;
        CallChainWin();
};


static CallChainWin *CalWindow;


InvocationRow::InvocationRow(ins_type ip, void** bp, Namedobj *obj)
{
        Frame.ins = ip;
        Frame.BP = bp;
        this->obj = obj;

        /* Prepare the 'args' string representation: */
        char buf[512];
	DebugInfo* dinfo;
        dinfo = obj->make ? obj->make->source.dinfo : NULL;
        if (dinfo == NULL or bp == NULL)
            sprintf(buf, "%s()   bp=%p", obj->name, bp);
        else {
            DebugInfo::localvarmap_node *dvar;
            str s;

            s = buf;
            s += sprintf(s, "%s(", obj->name);
            for (each_localvar(dinfo)) {
                if (dvar->storage == parameter_storage) {
                    ValueToString(s, s+sizeof(buf)-s,
                            (char*)bp + dvar->offset, (Type)dvar->type, yes);
                    s += strlen(s);
                    *s++ = ',';
                }
            }
            if (s[-1] == ',')
                s--;
            s += sprintf(s, ")");
        }
        args = strdup(buf);
}


InvocationRow::InvocationRow(str msg)
{
        Frame.ins = NULL;
        Frame.BP = NULL;
        obj = NULL;
        args = strdup(msg);
}


void InvocationRow::Paint(int x1, int y1, int x2, int y2)
{       CallChainWin *ccwin=(CallChainWin*)parent;
        bool isCurrent;

        isCurrent = (Frame.ins == SrcWindow->Frame->ins);
        DrawString(args,-1,ccwin->font,
                0,0,ccwin->clientWidth,height,
                YELLOW, isCurrent ? DARK(CYAN) : ccwin->Background);
}


bool InvocationRow::Mousestroke(int op, int x, int y)
{       extern void WatchUpdate();
        DebugInfo* dinfo;

        if (op != MOUSE_PRESS)
            return yes;
        if (obj == NULL or SrcWindow->Frame->ins == Frame.ins)
            return yes;
        dinfo = obj->make ? obj->make->source.dinfo : NULL;
	SrcWindow->EnterDebugger(obj,
                        dinfo,
                        DebugInsToLineNo(dinfo, Frame.ins),
                        &Frame);
	SrcWindow->PaintWhole();
        FrameDisplay(&Frame);
        CalWindow->PaintWhole();
        WatchUpdate();
        return yes;
}


bool InvocationRow::Keystroke(int key)
{       CallChainWin *ccwin=(CallChainWin*)parent;

        if (key == CTRL_PG_DOWN) {
            if (next and ((InvocationRow*)next)->obj) {
                ccwin->focus = next;
                ((InvocationRow*)ccwin->focus)->Mousestroke(MOUSE_PRESS,0,0);
            }
        }
        else if (key == CTRL_PG_UP) {
            if (prev and ((InvocationRow*)prev)->obj) {
                ccwin->focus = prev;
                ((InvocationRow*)ccwin->focus)->Mousestroke(MOUSE_PRESS,0,0);
            }
        }
        else MainWin->Keystroke(key);
        return yes;
}


void InvocationRow::Measure(int *widthp, int *heightp)
{       CallChainWin *ccwin=(CallChainWin*)parent;

        TextDimensions(args,-1,ccwin->font,widthp,heightp);
}


CallChainWin::CallChainWin()
        : RowWin(NULL,200,200)
{
        font = TfcFindFont(16);
        Background = BLACK;
        SetTitle("Call Chain");
}


static bool LooksLikeReturnAddress(ins_type ins, ins_type funcstart)
/* Does this address look like the return address of a CALL instruction? */
{       bool r;

        ins -= 5;
        r = no;
	__try {
	    r = (*ins == 0xe8) and ins + 5 + *(int*)(ins+1) == funcstart;
	}
	__except(yes) {
	}
        return r;
}


static void CallChainDisplay(void)
/* Display a window of all local variables. */
{	InvocationRow *invo;
        funcblock_type fn;
	void **bp, **sp;
	Namedobj* obj;
	ins_type ip;


	/* Prepare CalWindow: */
	if (CalWindow) {
            if (CalWindow->ShowState != tfc_shown)
                return;
	    CalWindow->Clear();
        }
	else {
            CalWindow = new CallChainWin();
        }


        /* Display the exception message: */
        invo = new InvocationRow(exception_message);
        CalWindow->Insert(NULL, invo);


	/* Get the list of calls: */
        CalWindow->focus = NULL;
	bp = (void**)RegisterImage->Ebp;
	sp = (void**)RegisterImage->Esp;
	ip = (ins_type)RegisterImage->Eip;
	do {
	    if (bp > StartOfUserStack or bp < (void*)RegisterImage->Ebp
	    	or bp < (void*)&obj)
		break;
	    obj = InsToObj(ip);
            if (obj == NULL)
                break;
            invo = new InvocationRow(ip,bp,obj);
	    if (obj == MainEntity)
		break;
            if (obj->type[0] != tp_function)
                break;
            fn = ((FunctionNamedobj*)obj)->u.fn;
            if (fn == NULL)
                break;
            CalWindow->Insert(NULL, invo);
            if (CalWindow->focus == NULL)
                CalWindow->focus = invo;
            if (fn[0] == 0xc8) {
                USE_BP:
                ip = (unsigned char*)bp[1];
                sp = bp;
                bp = (void**)bp[0];
                if (ip == NULL or bp == NULL)
                    break;
            }
            else {
                /* We don't have any perfect way of identifying the caller. */
                /* So instead search the stack for likely 'return address'  */
                /* values, and at the very worst we'll just skip the caller */
                /* and go to the last function that had a proper BP stack.  */
                for ( ; sp < bp; sp++) {
                    if (LooksLikeReturnAddress((ins_type)*sp, fn)) {
                        ip = (ins_type)*sp;
                        goto FOUND;
                    }
                }
                goto USE_BP;
                FOUND:;
            }
	} forever;
        CalWindow->Show();
}



/*---------------- Memory: ---------------*/

class MemViewer : public ScrollWin {
        void CalcCursorRect();

public:
	ins_type top;           // The address at the top of the window
        ins_type cur;           // The address of the current location
        int bytes_per_row;
        int fontheight, fontwidth;
        TfcFont font;

	MemViewer(void* mem);

	void GoTo(void* mem);		// Go to this location

	void Paint(int x1, int y1, int x2, int y2);
					// How to paint it
	void CursorPaint();
					// How to paint a cursor.
	void Measure(int *widthp, int *heightp)
			{ *widthp = clientHeight; *heightp = clientHeight; }
					// Get the width and height
	bool Keystroke(int key);
        void SetCursor();               // Calc cursor rect
};

static MemViewer *MemWindow;



MemViewer::MemViewer(void* mem)
	: ScrollWin(NULL, 600, 300)
{
	bytes_per_row = 16;
	fontheight = 16;
	font = TfcFindFont(fontheight, no, no, yes);
	fontwidth = Litewin::TextWidth("H",-1,font);
	Background = BLACK;
	realWidth = fontwidth*(12 + 3*bytes_per_row);
	realHeight = 1500;
        SetTitle("Memory Viewer");
	GoTo(mem);
}


void MemViewer::GoTo(void* mem)
{       
	cur = (ins_type)mem;
	top = cur - 512;
	top -= (int)top % bytes_per_row;
	scrollY = (cur - top) / bytes_per_row * fontheight - clientHeight/2;
	if (scrollY < 0)
	    scrollY = 0;
	PaintWhole();
}


void MemViewer::Paint(int x1, int y1, int x2, int y2)
{       char buf[512], buf2[512];
	ins_type mem, mem1, mem2;
	int x,y,i,v,y3,ey;
	bool Undefined;

	y3 = y2;
	ey = clientHeight+scrollY-fontheight-2;
	if (y2 > ey)
	    y2 = ey;
	DrawRectangle(realWidth,y1,x2,y3,BLACK);
	mem1 = top + (y1/fontheight) * bytes_per_row;
	mem2 = top + ((y2+fontheight-1)/fontheight) * bytes_per_row;
	y = y1/fontheight * fontheight;
	for (mem=mem1; mem < mem2; mem += bytes_per_row) {
	    sprintf(buf, "0x%08X| ", mem);
	    DrawString(buf, -1, font,
			0, y, fontwidth*12, y+fontheight,
			GREY, Background);
	    x = fontwidth * 12;
	    for (i=0; i < bytes_per_row; i+=4) {
		v = SafeDeref(mem+i, &Undefined);
	        for (int j=0; j < 4; j++) {
                    if (Undefined)
                        sprintf(buf, "?? ");
                    else sprintf(buf, "%02X ", ((unsigned char*)&v)[j]);
                    DrawString(buf,-1,font,
                            x,y,x+3*fontwidth,y+fontheight,
                            WHITE, Background);
                    x += 3*fontwidth;
                }
	    }
	    y += fontheight;
	}
	DrawRectangle(0,ey,realWidth,ey+2,BLUE);
	v = SafeDeref(cur, &Undefined);
	sprintf(buf, "     [0x%08X] = %02x   %s", cur, v&0xff,
		Disassemble(buf2, (ins_type)cur));
	DrawString(buf,-1,font,
		0,ey+2,clientWidth,ey+fontheight+2,
		RED,BLACK);
}


void MemViewer::CursorPaint()
{       int x,y;

	y = (cur-top) / bytes_per_row * fontheight;
	x = fontwidth*12 + ((cur-top) % bytes_per_row) * fontwidth*3;
	DrawRectangle(x,y,x+fontwidth*3-1,y+fontheight-1,NOCOLOUR,RED);
}


void MemViewer::SetCursor()
{       TfcRect Rect;
        int x,y;

	y = (cur-top) / bytes_per_row * fontheight;
	x = fontwidth*12 + ((cur-top) % bytes_per_row) * fontwidth*3;
	Rect.top = y;
	Rect.left = x;
	Rect.bottom = y + fontheight - 1;
        Rect.right = x + fontwidth*3 - 1;
        ScrollWin::SetCursor(Rect);
}


bool MemViewer::Keystroke(int key)
{       char buf[512];

        switch (key) {
            case UP:    cur -= bytes_per_row;
                        break;

            case DOWN:  cur += bytes_per_row;
                        break;

            case LEFT:  cur--;
			break;

            case RIGHT: cur++;
                        break;

	    case PG_UP: cur -= bytes_per_row * 16;
			break;

	    case PG_DOWN:
			cur += bytes_per_row * 16;
			break;

	    case 'i':
	    case 'I':	cur = DebugNextInstruction((ins_type)cur);
			break;

	    case 'G':
	    case 'g':	GoTo(*(void**)cur);
			break;

	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
			sprintf(buf, "0x%X", cur);
			DoDialog("GoTo",
				Control(buf, sizeof(buf),
					"Enter the new address", 10)
						-
					OkButton());
			sscanf(buf, "0x%p", &cur);
			GoTo(cur);
			break;

	    case WINDOW_QUIT:
			MemWindow = NULL;
			delete this;
			return yes;

            case F1:    ViewHelp(DebuggerHelp);
                        break;

	    default:    break;
	}
	if (cur < top) {
	    top -= 8*bytes_per_row;
	    PaintWhole();
	}
	else if (((cur-top) / bytes_per_row + 1) * fontheight > realHeight-4*fontheight) {
	    top += 8*bytes_per_row;
	    PaintWhole();
	}
        SetCursor();
        return yes;
}




/*----------------------- Watch expressions: ----------------------*/

class WatchLine : public Row {
        str source;
        str value;
        void (*fn)();           // It must output to 'cout' (which will be redirected)
        void *Frame;            // This == NULL if there is no reference
                                // made to locals.
        struct {
            str source;
            void *Frame;
            Namedobj* obj;
        } fncontext;            // What were the inputs to 'fn'?
public:
        WatchLine();
        virtual void Paint(int x1, int y1, int x2, int y2);
        virtual bool Keystroke(int key);
        virtual bool Mousestroke(int op, int x, int y);
        virtual void Measure(int *w, int *h);
        virtual void CursorPaint();
        friend class WatchWin;
        int Paragrapher(char mode);
        void Changed();

        /* Engine: */
        bool CompileAndEvaluate();
        bool Evaluate();
};


class WatchWin : public RowWin {
        int fontheight;
        TfcFont *font;
        int c;
public:
        WatchWin();
        virtual void CursorPaint()
                        { if (focus) focusln()->CursorPaint(); }
        virtual bool Mousestroke(int op, int x, int y);
        void SetCursor();
        WatchLine* focusln() { return (WatchLine*)focus; }

        friend class WatchLine;

} *WatWindow;


WatchLine::WatchLine()
{
        source = strdup(anon_heap, "");
        value = NULL;
        fn = NULL;
        width = 100;
        height = 16;
        fncontext.source = NULL;
        fncontext.Frame = NULL;
        fncontext.obj = NULL;
}


void WatchLine::Measure(int *wp, int *hp)
{
        WatchWin* Win = (WatchWin*)parent;
        *wp = Win->clientWidth;
        *hp = Paragrapher('M');
}


void WatchLine::Paint(int x1, int y1, int x2, int y2)
{
        Paragrapher('P');
}


void WatchWin::SetCursor()
{
        if (focusln() == NULL)
            return;
        ((WatchLine*)focusln())->Paragrapher('C');
}


void WatchLine::CursorPaint()
{
        Paragrapher('c');
}


int WatchLine::Paragrapher(char mode)
/* Determine how to layout this line, for various purposes ('mode'): */
/* 'P'=paint,  'M'=measure,  'C'=determine cursor position, */
/* 'c'=paint cursor. */
{       int w1,w2,w,rw1,centre,quarter,I,h1,h2,h,n;
        WatchWin *Win;

        /* What is the 1-line width of the two halves? */
        Win = (WatchWin*)parent;
        w1 = rw1 = TextWidth(source,-1,Win->font) + 3;
        if (value)
            w2 = TextWidth(value,-1,Win->font);
        else w2 = 0;
        quarter = Win->clientWidth / 4;
        if (w1 < quarter)
            w1 = quarter;
        if (w2 < quarter)
            w2 = quarter;

        /* Where will we put the break between them? */
        centre = Win->clientWidth / 2;
        if (w1 < centre and w2 < centre)
            I = centre;
        else if (w1 + w2 < Win->clientWidth)
            I = (w1 > centre) ? w1 : Win->clientWidth - w2;
        else {
            I = w1 * (Win->clientWidth / (w1+w2));
            if (I < quarter)
                I = quarter;
            else if (I > Win->clientWidth - quarter)
                I = Win->clientWidth - quarter;
        }

        /* How high will it be? */
        if (rw1 < I - 3) {
            h1 = Win->fontheight;
            if (mode == 'P')
                DrawString(source,-1,Win->font,0,0,I,height,YELLOW,DARK(BLUE));
            else if (mode == 'C' or mode == 'c') {
                w = TextWidth(source, Win->c, Win->font);
                if (mode == 'C')
                    SetCursor(w,0,w+3,Win->fontheight);
                else
                    DrawRectangle(w,0,w+3,Win->fontheight,RED);
            }
        }
        else {
            int best_n, best_score, score;
            bool punct1, punct2;
            str s = source;
            h1 = 0;
            while (*s) {
                /* A shortcut: maybe we're on the final line. */
                TextDimensions(s, -1, Win->font, &w1, &h, 0);
                if (w1 <= I) {
                    n = strlen(s);
                    goto HAVE_N;
                }
                /* No? Okay, do the next line. */
                best_n = n = 0;
                best_score = 0;
                w1 = 0;
                punct1 = yes;
                do {
                    TextDimensions(s, n, Win->font, &w1, &h, 0);
                    if (w1 > I - 3 and n >= 5)
                        break;
                    punct2 = (isalnum(s[n]) or s[n] == '_');
                    score = punct1 or punct2 ?
                            ((punct1 and punct2) ? 1 : 2)
                            : 0;
                    /* The best places to break are those on a border between */
                    /* punctuation and identifier. The worst place is in the  */
                    /* middle of an identifier. */
                    if (score >= best_score)
                        best_score = score, best_n = n;
                    if (s[n] == '\0')
                        break;
                    n++;
                    punct1 = punct2;
                } forever;
                n = best_n;
                HAVE_N:
                if (mode == 'P')
                    DrawString(s,n,Win->font,0,h1,I,height,YELLOW,DARK(BLUE));
                else if ((mode == 'C' or mode == 'c')
                                and s+n >= source+Win->c) {
                    TextDimensions(s, source+Win->c-s, Win->font, &w1, &h, 0);
                    if (mode == 'C')
                        SetCursor(w1,h1,w1+3,h1+Win->fontheight);
                    else DrawRectangle(w1,h1,w1+3,h1+Win->fontheight,RED);
                    return 0;
                }
                h1 += Win->fontheight;
                s += n;
            }
        }
        TextDimensions(value?value:"", -1, Win->font, &w2, &h2,
                        TFC_WORDBREAK, Win->clientWidth-I);
        h = (h1 > h2) ? h1 : h2;
        if (mode == 'M')
            return h;

        /* Are we painting? */
        if (mode == 'P') {
            DrawString(value ? value : "",-1,Win->font,
                        I,0,Win->clientWidth,height,RED,BROWN,
                        TFC_WORDBREAK);
        }
        return 0;
}


bool WatchLine::Keystroke(int key)
{       WatchWin *Win;
        char buf[512];

        Win = (WatchWin*)parent;
        WatWindow->CursorOff();
        switch (key) {
            case INS:   Win->Insert(this,new WatchLine);
                        break;

            case UP:    if (Win->focusln())
                            Win->focusln()->CompileAndEvaluate();
                        if (prev) {
                            Win->focus = prev;
                            Win->c = 0;
                        }
                        break;

            case DOWN:  if (Win->focusln())
                            Win->focusln()->CompileAndEvaluate();
                        if (next) {
                            Win->focus = next;
                            if (Win->c > strlen(Win->focusln()->source))
                                Win->c = 0;
                        }
                        break;

            case ENTER: Win->c = 0;
                        if (Win->focusln())
                            Win->focusln()->CompileAndEvaluate();
                        if (next == NULL)
                            Win->Insert(this, new WatchLine);
                        Win->focus = next;
                        break;

            case RIGHT: Win->c++;
                        if (Win->c > strlen(source))
                            Win->c = strlen(source);
                        break;

            case LEFT:  if (Win->c)
                            Win->c--;
                        break;

            case BACKSPACE:
                        if (*source == '\0')
                            goto DELETE_ROW;
                        if (Win->c > strlen(source))
                            Win->c = strlen(source);
                        if (Win->c == 0)
                            break;
                        Win->c--;
                        strcpy(source+Win->c, source+Win->c+1);
                        goto SOURCE_CHANGED;

            case DEL:   if (*source == '\0')
                            goto DELETE_ROW;
                        if (Win->c >= strlen(source))
                            break;
                        strcpy(source+Win->c, source+Win->c+1);
                        goto SOURCE_CHANGED;

            case SHIFT_DEL:
            case CTRL('Y'):
                        /* Delete the whole row: */
                        DELETE_ROW:
                        if (next)
                            Win->focus = next,
                            Win->Delete(this);
                        else if (prev)
                            Win->focus = prev,
                            Win->Delete(this);
                        if (Win->c > strlen(Win->focusln()->source))
                            Win->c = strlen(Win->focusln()->source);
                        break;

            case HOME:  Win->c = 0;
                        break;

            case END:   Win->c = strlen(source);
                        break;

            default:    if (key < ' ' or key >= 127)
                            return MainWin->Keystroke(key);
                        strcpy(buf, source);
                        free(anon_heap, source);
                        if (Win->c > strlen(buf))
                            Win->c = strlen(buf);
                        memmove(buf+Win->c+1,buf+Win->c,strlen(buf+Win->c)+1);
                        buf[Win->c++] = key;
                        source = strdup(anon_heap, buf);
                        SOURCE_CHANGED:
                        Changed();
                        break;
        }
        Win->SetCursor();
        return yes;
}


bool WatchLine::Mousestroke(int op, int x, int y)
{
        WatchWin *Win = (WatchWin*)parent;
        if (op == MOUSE_PRESS) {
            Win->focus = this;
            int n = strlen(Win->focusln()->source);
            if (Win->c > n)
                Win->c = n;
            Win->SetCursor();
        }
        return yes;
}


bool WatchWin::Mousestroke(int op, int x, int y)
{
        if (op == MOUSE2_PRESS) {
            Pmen.x = x;
            Pmen.y = y;
            Pmen.origWin = this;
            DebugPopupMenu(this,x,y);
            return yes;
        }
        else return RowWin::Mousestroke(op, x, y);
}


void WatchLine::Changed()
/* Measure the line, update the height in RowWin, and */
/* paint the whole window. */
{       int w,h;

        Measure(&w,&h);
        height = h;
        WatchWin *Win = (WatchWin*)parent;
        Win->Replace(this,this);
        PaintWhole();
}


WatchWin::WatchWin()
        : RowWin(NULL,100,100)
{
        Background = 0x400000;
        font = NULL;
        fontheight = TfcFontHeight(font);
        c = 0;
        SetTitle("Watch");
        Insert(NULL,new WatchLine);
        SetCursor();
}


bool WatchLine::CompileAndEvaluate()
/* Compile this source-string and then evaluate the expression. */
/* If we've compiled it already, and we've got the same string  */
/* in the same frame, then skip the recompilation part and just */
/* do the evaluation. */
{       Directory* dir;
        Heap* olddefheap;
        str s,semicolon;

        /* Has anything changed? */
        if (fncontext.source and streq(fncontext.source, source)
                and fncontext.Frame == SrcWindow->Frame
                and fncontext.obj == SrcWindow->this_obj)
            goto EVALUATE;
        if (fncontext.source)
            free(anon_heap, fncontext.source);
        fncontext.source = strdup(anon_heap, source);
        fncontext.Frame = SrcWindow->Frame;
        fncontext.obj = SrcWindow->this_obj;

        /* Clear old values: */
        if (fn)
            free(anon_heap, fn), fn = NULL;
        if (value)
            free(anon_heap, value), value = NULL;

        /* Short-cut: */
        for (s=source; *s == ' '; s++)
            ;
        if (*s == '\0')
            return no;

        /* Set up the context: */
        dir = ObjectToDirectory(SrcWindow->this_obj);

        /* Append the semicolon: */
        source = (str)realloc(anon_heap, source, strlen(source)+1);
        semicolon = source + strlen(source);
        *semicolon = ';';
        semicolon[1] = '\0';

        /* Compile: */
        olddefheap = default_heap;
        default_heap = anon_heap;
        fn = (machinecode_type)::Compile(source, dir, 'D');
        default_heap = olddefheap;
        assert(fn == NULL or Heap::Ptr_to_heap(fn) == anon_heap);
        *semicolon = '\0';
        if (Error.err) {
            value = strdup(anon_heap, Error.message);
            Changed();
        }

        /* In case the user assigned to any of the locals: */
        FrameDisplay(SrcWindow->Frame);

        /* Evaluate it: */
        EVALUATE:
        if (not Evaluate())
            return no;

        return (fn != NULL);
}


bool WatchLine::Evaluate()
/* Construct the 'value' field from the 'fn' field. Then display it. */
{       Ostream old_cout;
        char dest[4096];
        bool success;

        if (fn == NULL)
            return no;
        if (value)
            free(anon_heap, value), value = NULL;

        /* Set up a dummy 'cout': */
        *dest = '\0';
        old_cout = *cout;
        cout->buf = dest;
        cout->buf_end = dest + sizeof(dest);
        cout->s = cout->buf;
        cout->line_buffering = yes;
        cout->flush = NULL;

        /* Call the function: */
        success = yes;
        try {
            fn();
            *cout->s = '\0';
        } catch(...) {
            dest[100] = '\0';
            strcat(dest, "Can't evaluate");
            success = no;
        }

        /* Prepare the returned string: */
        dest[sizeof(dest)-1] = '\0';
        str s = dest + strlen(dest) - 1;
        while (*s == '\n' or *s == '\r' or *s == ' ')
            *s-- = '\0';
        value = strdup(anon_heap, dest);
        *cout = old_cout;

        /* Paint the new row: */
        Changed();
        return success;
}


void WatchUpdate()
{       WatchLine *ln;

        HeapCheck(0);
        if (WatWindow == NULL) {
            WatWindow = new WatchWin();
        }
        for (ln=(WatchLine*)WatWindow->root; ln; ln=(WatchLine*)(ln->next)) {
            ln->CompileAndEvaluate();
        }
        HeapCheck(0);
}







/*--------------------- The various debugging windows: ---------------------*/

void AddWin(ScrollWin *subwin)
{
        MainWin->Split(Pmen.origWin, subwin, Pmen.x, Pmen.y);
}


void HideWin(ScrollWin *subwin)
{
        MainWin->Remove(subwin);
}


void AddCalWin()
{
        MainWin->Split(Pmen.origWin, CalWindow, Pmen.x, Pmen.y);
        CallChainDisplay();
}


void DebugPopupMenu(ScrollWin *owner, int x, int y)
{
#define SubwinIt(s,win)\
		TfcMenuItem(s,  TfcCallback(AddWin,(ScrollWin*)win), \
                win == owner ? TFC_GREYED : 0)
        owner->PopupMenu(x,y,
                SubwinIt("Source code/machine code",  SrcWindow),
		TfcMenuItem("Call Stack",  AddCalWin),
		SubwinIt("Registers",  RegWindow),
		SubwinIt("Memory viewer",  MemWindow),
		SubwinIt("Local variables",  VarWindow),
		SubwinIt("Watch expressions",  WatWindow),
                TfcSeparator(),
		TfcMenuItem("Help",  TfcCallback(ViewHelp,DebuggerHelp)),
		TfcMenuItem("Hide",  TfcCallback(HideWin,owner)),
                NULL);
}


interface void MemDisplay(void *mem)
{
	/* Prepare VarWindow: */
	if (MemWindow == NULL)
	    MemWindow = new MemViewer(mem);
        else {
            if (MemWindow->ShowState != tfc_shown)
                return;
        }
	MemWindow->GoTo(mem);
	MemWindow->Show();
        MemWindow->CursorPaint();
}





/*--------------- The debugger top level: --------------------*/

static struct {
	ins_type ins;		    // The location	|  One or the other
	int LineNo;		    // The line number	|  is defined
	Namedobj* obj;		    // obj and
	DebugInfo* dinfo;	    // dinfo
	bool temporarily_on;	    // Temporarily on or temporarily off?
} TempBP[100];

static int TempBP_idx;
static int TempShouldContinue;
	    /* 0=no, 1=yes, 2=(for "step into"): do the call but then break. */


static void InsertTempBPSameObj(ins_type ins)
/* Insert a temporary breakpoint at this location */
/* in the same function. */
{
	if (FindCodeBreakpoint(ins)) {
	    /* There's already one there! */
	    return;
	}
	if (TempBP_idx >= 100) {
	    assert(false);
	    return;
	}
	TempBP[TempBP_idx].ins = ins;
	TempBP[TempBP_idx].obj = SrcWindow->this_obj;
	TempBP[TempBP_idx].dinfo = SrcWindow->dinfo;
	TempBP[TempBP_idx].LineNo = -1;
	TempBP[TempBP_idx].temporarily_on = yes;
	TempBP_idx++;
	DebugInsertCodeBreakpoint(ins, SrcWindow->this_obj, SrcWindow->dinfo);
}


static void InsertTempBP(ins_type ins)
/* Insert a temporary breakpoint at this location */
/* in possibly a different function. */
{
	if (FindCodeBreakpoint(ins)) {
	    /* There's already one there! */
	    return;
	}
	if (TempBP_idx >= 100) {
	    assert(false);
	    return;
	}
	TempBP[TempBP_idx].ins = ins;
	TempBP[TempBP_idx].obj = NULL;
	TempBP[TempBP_idx].dinfo = NULL;
	TempBP[TempBP_idx].LineNo = -1;
	TempBP[TempBP_idx].temporarily_on = yes;
	TempBP_idx++;
	DebugInsertCodeBreakpoint(ins, NULL, NULL);
}


static void DisableCurrentInstructionBreakpoint(void)
/* Temporarily turn off the breakpoint at Execution.ins. */
{
	assert(SrcWindow->Execution.ins);
	if (FindBreakpoint(SrcWindow->this_obj, SrcWindow->Execution.LineNo)) {
	    TempBP[TempBP_idx].temporarily_on = no; // it's temporarily off
	    TempBP[TempBP_idx].ins = NULL;
	    TempBP[TempBP_idx].LineNo = SrcWindow->Execution.LineNo;
	    TempBP[TempBP_idx].obj = SrcWindow->this_obj;
	    TempBP[TempBP_idx].dinfo = SrcWindow->dinfo;
	    TempBP_idx++;
	    DebugDeleteBreakpoint(SrcWindow->this_obj, SrcWindow->Execution.LineNo);
	}
	else if (FindCodeBreakpoint(SrcWindow->Execution.ins)) {
	    TempBP[TempBP_idx].temporarily_on = no; // it's temporarily off
	    TempBP[TempBP_idx].ins = SrcWindow->Execution.ins;
	    TempBP[TempBP_idx].LineNo = -1;
	    TempBP[TempBP_idx].obj = SrcWindow->this_obj;
	    TempBP[TempBP_idx].dinfo = SrcWindow->dinfo;
	    TempBP_idx++;
	    DebugDeleteCodeBreakpoint(SrcWindow->Execution.ins);
	}
}


static ProgRow* ExecutionToRow(void)
/* Update (and return) the row corresponding to the execution point. */
{	ProgRow* row;

	if (SrcWindow->Execution.ins) {
	    for (row=(ProgRow*)SrcWindow->root; row; row=(ProgRow*)row->next) {
		if (row->ins == SrcWindow->Execution.ins)
		    return row;
	    }
	}
	for (row=(ProgRow*)SrcWindow->root; row; row=(ProgRow*)row->next) {
	    if (row->LineNo == SrcWindow->Execution.LineNo)
		return row;
	}
        TfcMessage("debugger", '!',
                "I couldn't find which row corresponds to 0x%p!",SrcWindow->Execution.ins);
	assert(false);
	return (ProgRow*)SrcWindow->root;
}


void ProgWin::EnterDebugger(Namedobj* obj, DebugInfo* _dinfo, int LineNo,
		FrameNode *Frame)
/* Display the debug window with this context and get input. */
{
	/* Do we need to reinstall the source-code/debug-info? */
	if (obj != this_obj) {
	    this_obj = obj;
	    this_source = obj ? SourceFromObject(obj) : NULL;
	    if (this_source)
		InstallSource(this_source, _dinfo);
	    else InstallSource("Can't find the source.", _dinfo);
	}

	/* Set up our lines: */
	Execution.LineNo = LineNo;
	Execution.ins = Frame->ins;
	Cursor = ExecutionToRow();
        this->Frame = Frame;
}


bool ProgWin::Keystroke(int key)
{       ins_type *temp;

        switch (key) {
            case UP:    if (Cursor->prev)
                            Cursor = (ProgRow*)Cursor->prev, SetCursor();
                        break;

            case DOWN:  if (Cursor->next)
                            Cursor = (ProgRow*)Cursor->next, SetCursor();
                        break;

            case PG_UP: for (int i=0; i < 16 and Cursor->prev; i++)
                            Cursor = (ProgRow*)Cursor->prev;
                        SetCursor();
                        break;

            case PG_DOWN:
                        for (int i=0; i < 16 and Cursor->next; i++)
                            Cursor = (ProgRow*)Cursor->next;
                        SetCursor();
                        break;

            case WINDOW_QUIT:
                        ErrorRun("You exited from the debugger before completion.");
                        QuitEventLoop = yes;
                        break;

            case '1':   Mode = mo_src;
                        InstallSource(this_source, dinfo);
                        Cursor = ExecutionToRow();
                        SetCursor();
                        break;

            case '2':   Mode = mo_dis;
                        InstallSource(this_source, dinfo);
                        Cursor = ExecutionToRow();
                        SetCursor();
                        break;

            case '3':   Mode = mo_mixed;
                        InstallSource(this_source, dinfo);
                        Cursor = ExecutionToRow();
                        SetCursor();
                        break;

            case '4':   Mode = mo_displus;
                        InstallSource(this_source, dinfo);
                        Cursor = ExecutionToRow();
                        SetCursor();
                        break;

            case '.':   if (Cursor->ins) {
                            if (FindCodeBreakpoint(Cursor->ins))
                                DebugDeleteCodeBreakpoint(Cursor->ins);
                            else DebugInsertCodeBreakpoint(Cursor->ins, this_obj, SrcWindow->dinfo);
                        }
                        else if (Cursor->LineNo >= 0) {
                            if (FindBreakpoint(this_obj, Cursor->LineNo))
                                DebugDeleteBreakpoint(this_obj, Cursor->LineNo);
                            else DebugInsertBreakpoint(this_obj, Cursor->LineNo, SrcWindow->dinfo);
                        }
                        Cursor->PaintWhole();
                        break;

            case F10:
            case ALT_DOWN:
                        /* Step over: */
                        // Turn off the current breakpoint:
                        DisableCurrentInstructionBreakpoint();
                        // Turn on all follower breakpoints:
                        if (Execution.LineNo >= 0 and Mode == mo_src)
                            DebugFindFollowers(Lines[Execution.LineNo].Instructions, InsertTempBP);
                        else {
                            temp = NULL;
                            Array_Add(temp, Execution.ins);
                            DebugFindFollowers(temp, InsertTempBP);
                            Array_Free(temp);
                        }
                        // Turn on the caller's return-point:
                        if (this_obj and not strieq(this_obj->name, "Barbados>"))
                            DebugFindParent(Frame, InsertTempBP);
                        QuitEventLoop = yes;
                        break;

            case F11:
            case ALT_RIGHT:
                        /* Step into: */
                        // Turn off the current breakpoint:
                        DisableCurrentInstructionBreakpoint();
                        // Turn on all follower breakpoints:
                        if (Execution.LineNo >= 0 and Mode == mo_src) {
                            DebugFindFollowers(Lines[Execution.LineNo].Instructions, InsertTempBPSameObj);
                            // Turn on all function-call breakpoints:
                            DebugFindCalls(Lines[Execution.LineNo].Instructions, InsertTempBP);
                        }
                        else {
                            temp = NULL;
                            Array_Add(temp, Execution.ins);
                            DebugFindFollowers(temp, InsertTempBPSameObj);
                            // Turn on all function-call breakpoints:
                            DebugFindCalls(temp, InsertTempBP);
                            Array_Free(temp);
                        }
                        // Turn on the caller's return-point:
                        if (this_obj and not strieq(this_obj->name, "Barbados>"))
                            DebugFindParent(Frame, InsertTempBP);
                        TempShouldContinue = 2;
                        QuitEventLoop = yes;
                        break;

            case ALT_LEFT:
                        // Turn on the caller's return-point:
                        DebugFindParent(Frame, InsertTempBP);
                        QuitEventLoop = yes;
                        break;

            case ALT_UP:
            case F5:    /* Continue: */
                        if (not FindCodeBreakpoint(Execution.ins)) {
                            QuitEventLoop = yes;
                            break;
                        }
                        /* In order to continue, we need to disable the
                        current breakpoint, continue past it, and then
                        re-enable it. */
                        DisableCurrentInstructionBreakpoint();
                        // Turn on all followers:
                        if (Execution.LineNo >= 0 and Mode == mo_src)
                            DebugFindFollowers(Lines[Execution.LineNo].Instructions, InsertTempBPSameObj);
                        else {
                            temp = NULL;
                            Array_Add(temp, Execution.ins);
                            DebugFindFollowers(temp, InsertTempBPSameObj);
                            Array_Free(temp);
                        }
                        // As soon as we hit a follower, we'll turn the breakpoint at this line back on.
                        TempShouldContinue = yes;
                        QuitEventLoop = yes;
                        break;

            case ALT_END:
            case ALT_HOME:
                        /* Continue to current position: */

                        /* Step 1: Make sure the new place has a breakpoint. */
                        if (Cursor->LineNo >= 0 and Mode == mo_src) {
                            if (not FindBreakpoint(this_obj, Cursor->LineNo)) {
                                temp = Lines[Cursor->LineNo].Instructions;
                                for (int i=0; i < Array_Size(temp); i++)
                                    InsertTempBPSameObj(temp[i]);
                            }
                        }
                        else if (Cursor->ins and Mode != mo_src) {
                            if (not FindCodeBreakpoint(Cursor->ins))
                                InsertTempBPSameObj(Cursor->ins);
                        }
                        else break;

                        /* Step 2: Temporarily Turn off the current breakpoint: */
                        DisableCurrentInstructionBreakpoint();

                        /* Step 3: Continue. */
                        QuitEventLoop = yes;
                        break;

            case TAB:   if (WatWindow) {
                            WatWindow->Focus();
                            WatWindow->SetCursor();
                        }
                        break;

            case F4:    MemDisplay((void*)RegisterImage->Ebp);
                        break;

            case F1:    ViewHelp(DebuggerHelp);
                        break;

            default:    if (key >= ' ' and key <= 127 and isalnum(key) or key == '_') {
                            if (WatWindow) {
                                WatWindow->Focus();
                                WatWindow->SetCursor();
                                WatWindow->Keystroke(key);
                            }
                        }
                        else if (key == CTRL_PG_UP or key == CTRL_PG_DOWN) {
                            if (CalWindow) {
                                CalWindow->Keystroke(key);
                            }
                        }
                        else break;
	}
        return yes;
}


bool ProgRow::Mousestroke(int op, int x, int y)
{
	if (op == MOUSE_PRESS and this) {
            /* Move the cursor to this row: */
	    ProgWin *W=(ProgWin*)parent;
	    if (y < W->realHeight) {
		W->Cursor = this;
                W->SetCursor();
            }

            if (x >= 0 and x < 12) {
                /* Toggle the breakpoint: */
                if (ins) {
                    if (FindCodeBreakpoint(ins))
                        DebugDeleteCodeBreakpoint(ins);
                    else DebugInsertCodeBreakpoint(ins, W->this_obj, SrcWindow->dinfo);
                }
                else if (LineNo >= 0) {
                    if (FindBreakpoint(W->this_obj, LineNo))
                        DebugDeleteBreakpoint(W->this_obj, LineNo);
                    else DebugInsertBreakpoint(W->this_obj, LineNo, SrcWindow->dinfo);
                }
                PaintWhole();
            }
	}
	return yes;
}





/*--------------------- A customised SplitWindow: --------------------*/

class DbgSplitWindow : public SplitWindow {
public:
        bool Keystroke(int key);
        DbgSplitWindow(str title, int width, int height, ScrollWin *first)
                : SplitWindow(title,width,height,first)
                { }
};


bool DbgSplitWindow::Keystroke(int key)
{
        if (key == CTRL_PG_UP or key == CTRL_PG_DOWN) {
            if (CalWindow)
                return CalWindow->Keystroke(key);
        }
        else if (SrcWindow)
            return SrcWindow->Keystroke(key);
        return no;
}





/*--------------------- Interface functions: --------------------*/

static void DebuggerTurnWindowOn(void)
{
	/* Flush all stdout: */
	flush_stdout();

	/* Ensure SrcWindow is created: */
	if (SrcWindow == NULL)
	    SrcWindow = new ProgWin("SrcWindow", 400, 400);
	if (WatWindow == NULL)
	    WatWindow = new WatchWin();
	if (CalWindow == NULL)
	    CalWindow = new CallChainWin();

        /* Ensure we have the MainWin: */
        if (MainWin == NULL) {
            MainWin = new DbgSplitWindow(
                        "Debugger - right-click for menu, F1 for help",
                        800, 500, SrcWindow);
            MainWin->Split(SrcWindow, WatWindow, 0.7, 'R');
            MainWin->Split(WatWindow, CalWindow, 0.7, 'B');
        }
        else MainWin->Show();

	/* Display changes: */
        if (WatWindow)
            WatWindow->SetCursor();
	SrcWindow->PaintWhole();
}


interface void DebuggerExit(void)
/* Clean up all windows reserved by the debugger. */
/* Currently never called. */
{
	delete VarWindow;
	delete RegWindow;
	delete CalWindow;
	delete SrcWindow;
        delete WatWindow;
        VarWindow = NULL;
        RegWindow = NULL;
        CalWindow = NULL;
        SrcWindow = NULL;
        WatWindow = NULL;
}


interface void DebugHideWindows(void)
/* Hide all debug windows. */
{
        if (MainWin)
            MainWin->Hide();
}


interface void DebugGuiDeleteObj(Namedobj* obj)
/* Uninstalls the source-code from the cache */
/* when the source-code changes. */
{
	if (SrcWindow and obj == SrcWindow->this_obj)
	    SrcWindow->this_obj = NULL;
}


interface void Debugger(Namedobj* obj, DebugInfo* dinfo, int LineNo, FrameNode *Frame)
/* Display the debug window with this context and get input. */
{       ScrollWin *prevFocus;

	/* Undo the temporary changes to BP's: */
	while (TempBP_idx > 0) {
	    TempBP_idx--;
	    if (TempBP[TempBP_idx].temporarily_on)
		DebugDeleteCodeBreakpoint(TempBP[TempBP_idx].ins);
	    else if (TempBP[TempBP_idx].LineNo == -1)
		DebugInsertCodeBreakpoint(TempBP[TempBP_idx].ins, TempBP[TempBP_idx].obj, TempBP[TempBP_idx].dinfo);
	    else
		DebugInsertBreakpoint(TempBP[TempBP_idx].obj, TempBP[TempBP_idx].LineNo, TempBP[TempBP_idx].dinfo);
	}
	if (TempShouldContinue) {
	    if (TempShouldContinue == 2) {
	       if ((LineNo >= 0 and SrcWindow and LineNo == SrcWindow->Execution.LineNo)
		    or (Frame->ins and Frame->ins == SrcWindow->Execution.ins)
		  ) {
		    /* We're trying to step into a function and we have to do it in 2 steps. */
		    TempShouldContinue = no;
		    DisableCurrentInstructionBreakpoint();
		    DebugFindCalls(SrcWindow->Lines[LineNo].Instructions, InsertTempBP);
		    return;
		}
	    }
	    else {
		TempShouldContinue = no;
		return;
	    }
	}
	if (obj and strieq(obj->name, "Barbados>"))
	    return;

	/* Go to the GUI: */
        prevFocus = CursorSw;
	DebuggerTurnWindowOn();
	SrcWindow->EnterDebugger(obj, dinfo, LineNo, Frame);
	SrcWindow->PaintWhole();
        FrameDisplay(Frame);
        RegisterDisplay();
        CallChainDisplay();
        WatchUpdate();
        MemDisplay((void*)RegisterImage->Ebp);
        HeapCheck(0);

	/* Interact: */
        SrcWindow->SetCursor();
        if (prevFocus == WatWindow)
            WatWindow->SetCursor();
        else if (prevFocus == MemWindow)
            MemWindow->SetCursor();
        else SrcWindow->Focus();
        ScrollWin::EventLoop();
}

