/************************************************\
** JOIN.C:                                      **
**    This module interfaces the compiler and   **
** the editor/user interface.  This includes    **
** setting up transfer of source text to the    **
** compiler, error messages, and initialising   **
** the environment via compiling standard       **
** declarations, macros and functions.          **
\************************************************/

#include <windows.h>	    // Just for the exception stuff.
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include "barbados.h"
#include "tfc.h"
#include "modules.h"
#include "make_container.h"
#include "exception.h"
#include "source.h"
#include "memory.h"
#include "runlib.h"
#include "debug.h"
#include "make.h"
#include "name.h"
#include "join.h"



interface str CurrentSource;
interface bool CurrentSourceIsPersistent;
interface struct error_node Error;
interface void* StartOfUserStack;



/*----------------- Module Initialisation and Tidying: ---------------------*/

interface void JoinInit(void)
{
	InitMake();
	MakeTimeAdvance();
	StdlibInit();
	CurdirInit();
	DebugInit();
	MakeTimeAdvance();
	ContainerCheckSource(curconim);
}






/*--------------- Error Handling ------------------*/

static void GetErrorMess(str mess, va_list argptr)
/* Construct an error-reporting line.   Put a strdup'd copy */
/* into 'error.mess'. */
{	char buf[1024], *s;

	s = buf;
	s += sprintf(s, "<<< ");
	if (PredefinitionPhase) {
	    extern str LexGetSourceStart();
	    s += sprintf(s, "%s\n", LexGetSourceStart());
	}
	s += vsprintf(s, mess, argptr);
	strcat(s, " >>>");
	if (Error.message != NULL)
	    free(anon_heap, Error.message);
	Error.message = strdup(anon_heap, buf);
	if (PredefinitionPhase) {
	    TfcMessage("Predefinition Error", '!', Error.message);
	    exit(1);
	}
}


interface void ErrorRun(str mess, ...)
/* We've got a run-time error message to display. */
{       va_list argptr;

	if (Error.err >= err_certain)
	    return;
	Error.err = err_runtime;
	Error.src = CurrentSource;
	Error.pos = NULL;

	va_start(argptr, mess);
	GetErrorMess(mess, argptr);
	va_end(argptr);

	if (ClosingDownPhase)
	    TfcMessage("Closing down error", '!', Error.message);
	else RaiseException(CONTROL_C_EXIT, 0, 0, NULL);
}


interface void ErrorWarning(str mess, ...)
/*
        We've got a run-time error message to display. We don't want the application
        to stop, only to print the message and return
*/
{       va_list argptr;

	if (Error.err >= err_certain)
	    return;
	Error.err = err_runtime;
	Error.src = CurrentSource;
	Error.pos = NULL;

	va_start(argptr, mess);
	GetErrorMess(mess, argptr);
	va_end(argptr);

	if (ClosingDownPhase)
	    TfcMessage("Closing down error", '!', Error.message);
}



interface void ErrorMake(str mess, ...)
/* We've got a 'make'-time error message to display. */
{       va_list argptr;

	if (Error.err >= err_make)
	    return;
	Error.err = err_make;
	Error.src = NULL;
	Error.pos = NULL;

	va_start(argptr, mess);
	GetErrorMess(mess, argptr);
	va_end(argptr);
}


interface void ErrorException(str mess, ...)
/* Just like ErrorRuntime but don't try to raise an exception. */
{       va_list argptr;

	if (Error.err >= err_certain)
	    return;
	Error.err = err_runtime;
	Error.src = CurrentSource;
	Error.pos = NULL;

	va_start(argptr, mess);
	GetErrorMess(mess, argptr);
	va_end(argptr);
}


interface void** baderr;

interface void JoinCompileAndRun(str src, error_type err)
/* Compile & then execute the current text.  May include several */
/* compileable entities, e.g.   "command; command;"              */
/* If there's an error, return the error in 'err'. */
{       EXCEPTION_POINTERS *dbg;
	container_id cid;

	ContainerLinkOldies();
	ClearOutputLines();
	InitStdio();
	StartOfUserStack = &dbg;

	/*** Compile the command for the first time: ***/
	if (MainEntity)
	    assert(MainEntity->u.fn == NULL);
	CurrentSourceIsPersistent = no;
	baderr = (void**)&err;

	HeapCheck(0);
	MainEntity->u.fn = (funcblock_type)Compile(src, curdir, 'I');
	HeapCheck(0);


	/*** If we aborted immediately due to lack of source-code ***/
	/* being linked, (i.e. no make_nodes on referenced namedobj's), */
	/* then link source and recompile. */
	if (Error.err == err_pleaselinksrc) {
	    SourceLinkSource(curconim,no);
	    CurrentSourceIsPersistent = no;
	    MainEntity->u.fn = (funcblock_type)Compile(src, curdir, 'I');
	    assert(Error.err != err_pleaselinksrc);
	}


	if (Error.err != err_certain) {
	    /*** Run 'make' on the command. In some cases, the command itself ***/
	    /* will be recompiled. */
	    MakeThis(NULL);
	    if (MainEntity->u.fn and not Error.err) {
		machinecode_type fn;
		Conim* curdir_conim;

		/*** We want to execute the command: ***/
		cid = curconim->cid;
		curdir_conim = curconim;
		OpenContainer(cid, READONLY);       // Increment refcount - lock it in place
		HeapCheck(0);
		HeapCheck(-1);
		HeapCheck(cid);
		fn = (machinecode_type)MainEntity->u.fn;
		MainEntity->u.fn = NULL;
		//$$$__try {
                    __asm CALL fn
		    //fn();    Visual Studio requires us to save ESI, but we don't save it,
                    // so we do the __asm thing instead of fn().
		/*}		// fn() might cause curconim to change.
		__except(DebugException(GetExceptionInformation())) { };*/
		HeapCheck(0);
		HeapCheck(-1);
		HeapCheck(cid);

		curdir_conim->free(fn);
		CloseContainer(cid);      // Decrement refcount
		HeapCheck(0);

		JoinReturnToTopLevel();
		MainEntity->u.fn = NULL;
	    }
	    else goto DIDNT_EXECUTE;
	}
	else {
	    DIDNT_EXECUTE:
	    if (MainEntity->u.fn)
		free(NULL, MainEntity->u.fn), MainEntity->u.fn = NULL;
	}

	/*** Clean up: ***/
	flush_stdout();
	if (Error.err == err_maybe)
	    Error.err = err_certain;
	if (err == NULL)
	    TfcMessage("debug", 'i', "err = %p, baderr=%p, &err=%p",
			err, baderr, &err);
	*err = Error;
	ContainerLinkOldies();
}


interface void JoinEditSource(str name)
/* Find this object's source-code and insert it into the editor. */
/* (Also allow the caller to specify a string to be pasted	 */
/* into the editor). */
{	Namedobj* obj;
	str source;

	obj = NamePathToObj(name);
	if (obj == NULL) {
	    ErrorRun("Object %s not found", name);
	    return;
	}

	source = SourceFromObject(obj);
	if (source == NULL) {
	    if (obj->storage == static_storage and
		    ((*obj->type == tp_array and obj->type[5] == tp_char)
		    or TypeEqual(obj->type, string_typstr)))
		source = (str)((StaticNamedobj*)obj)->location;
	    else {
		ErrorRun("Source not found");
		return;
	    }
	}

	EditorAddSource(source);
}


interface void JoinReturnToTopLevel(void)
{
	DebugHideWindows();
}

