/*****************************************************\
** MAKE.C:                                           **
**          The make module.  Contains all the functions **
** for doing 'make's.                                **
\*****************************************************/

#include <memory.h>
#include "barbados.h"
#include "modules.h"
#include "trace.h"
#include "name.h"
#include "join.h"
#include "make.h"




interface timestamp_type
	    TimeOfCurrent,
	    TimeOfLastInterfaceChange,
	    TimeOfLastCompile,
	    TimeOfThisRecurse,
	    TimeOfThisMake;
interface bool PredefinitionPhase;
interface bool ClosingDownPhase;
static Make *free_list;





/*=============== Module Initialisation ==============*/

interface void InitMake(void)
{
	TimeOfCurrent = 0;
	MakeEntityReset();
	free_list = NULL;
}




/*=============== Recompilation of one entity: ================*/

static void Recompile(Make *A)
/* Recompile this entity.  That involves changing to the */
/* appropriate directory, (so the name-space will be     */
/* correct) and calling the compiler, and changing back. */
{       Directory* tempdir;

	printf("Recompiling: %s\n", A->obj->name);
	if (A->obj == MainEntity) {
	    assert(A->source.s != NULL);
	    if (MainEntity->u.fn)
		free(NULL, MainEntity->u.fn);
		// NULL means we don't know what container it would have been in.
	    CurrentSourceIsPersistent = yes;
	    MainEntity->u.fn = (funcblock_type)Compile(A->source.s, curdir, 'M');
	    MakeChanged(A, no);
	}
	else if (A->source.s == NULL)
	    ErrorMake("\"%s\" has a make-node but no source-code.\n", A->obj->name);
	else {
	    tempdir = curdir;
	    Chdir(A->source.dir);
	    CurrentSourceIsPersistent = yes;
	    Compile(A->source.s, curdir, 'M');
	    Chdir(tempdir);
	}
}





/*=============== The allocation of make nodes: ================*/


static Make *MakeNew(void)
/* Create a new make node either from 'malloc' or from the free_list. */
{       Make *A;

	if (free_list) {
	    A = free_list;
	    free_list = free_list->next;
	}
	else {
	    A = new(Make);
	    clearS(*A);
	}
	A->interface_stamp = TimeOfCurrent;
	return A;
}


interface void MakeDelete(Make *A)
/* Deal with the fact that we have deleted this Make's object. */
/* We want to put it into the free_list. Don't free it, because other */
/* make_entities will be pointing to it. */
{
	assert(A != NULL);
	A->next = free_list;
	free_list = A;
	free(anon_heap, A->depends_on);
	A->obj = NULL;
	A->depends_on = NULL;
	A->interface_stamp = TimeOfCurrent;
	// This ensures that anything referencing it will be recompiled. */
}




/*=============== Setting up make_entities & dependencies ================*/

static int dobuf_idx;
static Make make_current;            // The current make info


interface void MakeEntityReset(void)
/* Initialise `make_current' at the start of a compilation.   */
/* The 'source' pointer is set at the current parse position. */
{       static Make* depends_on_buf[100];

	make_current.source.s = NULL;
	make_current.obj = NULL;
	make_current.check_stamp = 0;
	make_current.compile_stamp = TimeOfCurrent;
	make_current.interface_stamp = 0;
	make_current.depends_on = depends_on_buf;
	make_current.depends_on[0] = NULL;
	dobuf_idx = 0;
}



interface void MakeDepends(Make *B)
/* Assert that the current entity depends on B. */
{	Make **p;

	if (B == NULL)
	    return;         // A pre-defined entity, e.g. "str".
	for (p=make_current.depends_on; *p; p++)
	    if (*p == B)
		return;     // It's already there.
	make_current.depends_on[dobuf_idx++] = B;
	make_current.depends_on[dobuf_idx] = NULL;
}



interface Make* MakeEntityNew(Namedobj* obj)
/* Return a copy of the make_current information. */
/* Used for initialising new make-entities.       */
{       Make *A;
	int len;

	A = MakeNew();
	A->obj = obj;
	A->source.s = NULL;
	A->NoOfCompilations = 0;
	A->interface_stamp = TimeOfCurrent;
	A->compile_stamp = TimeOfCurrent;
	A->check_stamp = 0;
	if (dobuf_idx == 0)
	    A->depends_on = NULL;
	else {
	    len = (dobuf_idx+1) * sizeof(Make*);
	    A->depends_on = (Make**)malloc(anon_heap, len);
	    memcpy(A->depends_on, make_current.depends_on, len);
	}
	return A;
}


interface void MakeEntityUpdate(Namedobj* obj)
/* Update the dependencies for this entity, which */
/* already exists.  It copies the depends_on info */
/* from make_current into this entity.  It is     */
/* important that the entity stays at the same    */
/* address.    You can have multiple calls to     */
/* M..Update() following a single M..Open() if you*/
/* so desire. */
{       Make *A=obj->make;
	int len;

	if (A == NULL)
            return;	/* This happens in 2 situations: (a) Predefinition */
        /* phase, and (b) functions defined inside a class definition. */
	free(anon_heap, A->depends_on);
	A->obj = obj;
	A->compile_stamp =  make_current.compile_stamp;
	A->check_stamp =    make_current.check_stamp;
	if (dobuf_idx == 0)
	    A->depends_on = NULL;
	else {
	    len = (dobuf_idx+1) * sizeof(Make*);
	    A->depends_on = (Make**)malloc(anon_heap, len);
	    memcpy(A->depends_on, make_current.depends_on, len);
	}
}


interface void MakeChanged(Make *A, bool InterfaceChanged)
/* Assert that this entity has just changed - which means updating */
/* the time-stamps.  The global time-stamp is advanced only when   */
/* an interface has changed - everything compiled in-between these */
/* events is being compiled in an identical environment. */
{
	if (A == NULL)
            return;	/* Predefinition phase or function defined inside */
            /* class definition. */
	if (InterfaceChanged) {
	    MakeTimeAdvance();
	    A->compile_stamp = TimeOfCurrent;
	    A->interface_stamp = TimeOfCurrent;
	    TimeOfLastInterfaceChange = TimeOfCurrent;
	    MakeTimeAdvance();
	}
	else {
	    if (A->compile_stamp < TimeOfThisMake)
		A->NoOfCompilations = 0;
	    A->compile_stamp = TimeOfCurrent;
	}
}


interface void MakeEntityClear(Make *A)
/* Wipe away the dependencies, but keep the node itself valid. */
{
	if (A == NULL) {
	    assert(PredefinitionPhase);
	    return;
	}
	if (A->depends_on)
	    free(anon_heap, A->depends_on);
	A->depends_on = (Make**)malloc(anon_heap,
	sizeof(str));
	A->depends_on[0] = NULL;
	A->compile_stamp = 0;
}





/*=================== Misc Make functions: ========================*/

interface timestamp_type MakeTimeAdvance(void)
{
	return TimeOfCurrent++;
}



interface bool MakeEntityInUse(Make *A)
/* Is this object in use?  We know that it is being used */
/* by the current compileable entity if its check_stamp  */
/* is equal to the current time-stamp. */
{
	return A and (A->check_stamp >= TimeOfThisMake);
}


interface timestamp_type MakeCurrentStamp(void)
/* Returns the current time-stamp. */
{
	return TimeOfCurrent;
}





/*================ The Make Algorithm ================*/

interface int depth;                    // The depth of the make recursion $$$


static void RecurseMake(Make *A)
/* Recursively get this entity up-to-date. */
{	timestamp_type TimeThisCompiled;
	Make **p, *B;
	bool CompileThis;
	static int L;


	EXAMINE_A:
	/* Examine the dependencies, recompile if out-of-date, */
	/* and loop if it was recompiled. */


	/* Has the object been deleted? */
	if (A->obj == NULL)
	    return;
	    // Just return. You never know - it might have been recompiled
	    // under the same name since, and if not then we'll just generate
	    // the appropriate error.
	assert(A->obj->make == A);
	TimeThisCompiled = A->compile_stamp;


	/* Test and set the check-stamp: */
	if (A->check_stamp > TimeOfLastInterfaceChange)
	    return;                             // Checked very recently
	if (A->check_stamp >= TimeOfThisRecurse)
	    return;                             // Checked this recursion
	else A->check_stamp = TimeOfCurrent;    // Mark it as visited


	/* Short-cut for initial entities: */
	if (A->depends_on == NULL)
	    goto RETURN;
	if (A->depends_on[0] == NULL)
	    goto RETURN;


	/* For every entity B that A depends upon: */
	CompileThis = no;
	for (p=A->depends_on; *p; p++) {
	    B = *p;

	    /* Bring 'B' up-to-date: */
	    if (B->obj == NULL)
		continue;       // It's a freed make-node.
	    if (B->obj->make != B) {
		Pr("Make bug!  %s :  %p %p %s\n", A->obj->name, B, B->obj, B->obj->name);
		continue;
	    }
	    if (debugging & 24)
		Trace("%*s%s<-%s:\n", depth*2, "",
			    A->obj->name, B->obj->name);
	    if (depth++ > 50) 
		ErrorMake("Make is recursing infinitely.");
	    else 
		RecurseMake(B);
	    depth--;
	    if (Error.err == err_make)
		return;


	    /* Has 'A' just been inadvertently recompiled (because */
	    /* 'B' is created in the same compilation as 'A')?     */
	    if (A->compile_stamp != TimeThisCompiled)
		goto EXAMINE_A;


	    /* Is 'A' out-of-date? */
	    if (B->interface_stamp - A->compile_stamp >= 0) {
		CompileThis = yes;
		if (debugging & 24)
		    Trace("%*s%s<-%s recompile:\n", depth*2, "",
			    A->obj->name, B->obj->name);
	    }
	}


	/* Is it up-to-date even without recompilation? */
	if (not CompileThis)
	    goto RETURN;


	/* Ok.  We're going to have to recompile. */
	if (A->compile_stamp < TimeOfThisMake)
	    A->NoOfCompilations = 0;
	if (debugging & 24)
	    Trace("%*sRecompiling %s\n", depth*2, "", A->obj->name);
	if (A->NoOfCompilations++ >= 2) {
	    ErrorMake("Make is going too many times");
	    return;
	}
	TimeOfLastCompile = TimeOfCurrent;
	Recompile(A);
	if (Error.err) {
	    if (Error.err == err_certain)
		return;
	}
	else if (A->compile_stamp - TimeOfLastCompile < 0) {
	    Recompile(A);
	    ErrorMake("Source for %s has disappeared!", A->obj->name);
	}
	goto EXAMINE_A;


	RETURN:
	/* If it's a function, check that the body has been defined. */
	if (not NameBodyDefined(A->obj))
	    ErrorMake("Function %s() is declared but undefined.",
			    A->obj->name);
}



interface void MakeThis(Make *A)
/* Make this entity current and compile the make_list. */
/* If 'A' is NULL, work on a copy of make_current.     */
{
	/* If 'A' is NULL, work on a copy of make_current. */
	if (A == NULL) {
	    A = MainEntity->make = MakeEntityNew(MainEntity);
	    A->source.s = CurrentSource;
	}


	/* Assert that the make node is not a deleted one. */
	assert(A->obj != NULL);


	/* Reset the `visited' and `NoOfCompilations' fields for each object. */
	TimeOfThisMake = TimeOfCurrent;


	/* Loop for as long as we get recompilations: */
	do {
	    MakeTimeAdvance();
	    TimeOfThisRecurse = TimeOfCurrent;
	    depth = 0;
	    RecurseMake(A);
	    if (debugging & 24)
		Trace("\n");
	    if (Error.err == err_certain)
		break;
	} while (TimeOfLastCompile >= TimeOfThisRecurse);


	/* Delete 'A' if it was created temporarily. */
	if (A->obj == MainEntity)
	    MakeDelete(A);


	/* If we had an error, mark everything as unchecked. */
	if (Error.err)
	    MakeTimeAdvance();
}


#if 0
interface void MakeAssert(Make *A)
/* Check this Make for consistency (i.e. for debugging purposes). */
{       Make *B, **p;

	assert(A->obj->make == A);
	if (A->depends_on == NULL)
	    return;
	for (p=A->depends_on; *p; p++) {
	    B = *p;
	    assert(B->obj->make == B);
	    assert(*B->obj->name != '\0');
	}
}

#endif

