/********************** MEM_HEAP.C: *****************************\
** Summary:   This module implements Barbados Heaps and also	**
**            memory access checking.				**
** (a) We must support the ordinary heap operations, but now	**
**     on a set of heaps rather than a single heap.		**
** (b) We must be able to expand any of the heaps in any	**
**     order.							**
** (c) We must support a function to 'free' an entire heap.	**
** (d) We must support the operation of the Saver.		**
**								**
**								**
** I have redefined malloc, free, calloc, realloc, strdup,	**
** operator new and operator delete to use these functions.	**
** They all use a special heap called the 'anon_heap' - this	**
** is a special heap that is not associated with any container	**
** and therefore nothing in it gets automatically saved. So by	**
** default, any object Barbados creates goes into the		**
** anon_heap, but any object the user creates goes into the	**
** current container's heap.					**
**								**
**								**
\****************************************************************/


#include <string.h>
#include <stdlib.h>
#include <windows.h>
#include "barbados.h"
#include "memory.h"
//#include "mem.h"

#define each_region     WholesaleRegion* region=wholesaleregions; region; region=region->next




interface Heap AnonHeap, *anon_heap=&AnonHeap;
interface Heap *default_heap;
static int DisableAssert;

uint Heap::size2power[512];
uint Heap::power2size[HEAP_NUMPOWERS];
int Heap::LowestMem, Heap::HighestMem;
Heap** Heap::PageToHeap;

void* RawRegion(int numbytes);  // Wholesale memory allocation
void FreeRawRegion(void* mem);





/*------------- Mapping pointers to heaps: ----------------*/

void Heap::PageMapAddRegion(void* _mem, uint size)
/* Add this region to PageToHeap[]. */
{       int mem=(int)_mem/PAGE_SIZE, end=mem + size/PAGE_SIZE;
        char *addr1, *addr2;

        assert((((int)_mem|size) & 65535) == 0);
        if (PageToHeap == NULL)
            PageToHeap = (Heap**)VirtualAlloc(NULL, 4*65536,
				MEM_RESERVE, PAGE_READWRITE);
        if (LowestMem == NULL or mem < LowestMem) {
            LowestMem = mem;
            if (end > HighestMem)
                HighestMem = end;
            COMMIT:
            addr1 = (char*)&PageToHeap[mem];
            addr2 = (char*)&PageToHeap[end];
            VirtualAlloc(addr1, addr2-addr1,
                                    MEM_COMMIT, PAGE_READWRITE);
        }
        if (end > HighestMem) {
            HighestMem = end;
            goto COMMIT;
        }
        for (int i=mem; i < end; i++)
            PageToHeap[i] = this;
}


void Heap::PageMapDelRegion(void* _mem, uint size)
/* Remove this region from PageToHeap[]. */
{	int mem=(int)_mem/PAGE_SIZE, end=mem + size/PAGE_SIZE;

        assert((((int)_mem|size) & 65535) == 0);
        for (int i=mem; i < end; i++)
            PageToHeap[i] = NULL;
}


Heap* Heap::Ptr_to_heap(void* _ptr)
/* Map this pointer to a heap. */
{       int ptr=(int)_ptr / PAGE_SIZE;

        if (ptr < LowestMem or ptr >= HighestMem)
            return NULL;
        return PageToHeap[ptr];
}









/*---------------------- Heaps: malloc & free -----------------------*/


/* The representation is as follows:

* There is a 4-byte header for all tiles.

* Free tiles are denoted by this header bit0=1.

* Allocated tiles have header bit0=0.

* Free blocks have bit0=1, and to get the size of the block
    just ignore this bit.

* There are 3 types of allocated tiles: untyped tiles,
    non-class-instances typed tiles, and class-instances.

* Untyped tiles have header bit1=1, and the header (minus this 
    bit) represents the size of the block in bytes including
    the 4-byte header.

* Class-instance tiles have header bit1=0, and the header
    represents a pointer to the Classdef that defines
    the class.   'classdef->RoundupSize' then gives the size.

* Non-class-instance typed tiles are the most complex kind of
    tile.  They also have header bit1=0, however this pointer
    points _inside_ the tile.  (To identify whether it's a
    class-instance tile or non-class-instance tile you need
    to do a pointer range comparison). This pointer points to 
    a 4-byte integer which gives the _full_ size of the tile,
    (in order to be as compatible as possible with class-instance
    tiles), however following this 4-byte integer is the type 
    information (a type-string).  This complexity is justified
    on the basis that 99% of all typed tiles will hopefully be
    class instances.  These tiles are the result either of 
    types such as 'int', 'double' or 'int[57]' being new'd, or
    the use of the NewPlus() function which allows you to create
    a typed tile with extra space at the end for variable-length
    data.  In both other types of tiles, the user is free to 
    use 'realsize-4' bytes of the tile, however in this tile 
    the user's data size is more than 4 bytes less than the total
    size.

* All 'sizes' include the 4 bytes of the header.

* There are only certain allowable sizes of tile.  A tile
    can only have a size which is either a power of 2 or 150%
    of a power of 2, starting from 8, i.e. 8, 12, 16, 24, 32, 
    48, 64, 96, 128, 192, ... . Tiles are always rounded up to 
    one of these sizes.  If the caller requests e.g. 25 bytes, 
    then we allocate 32 bytes and 7 bytes are wasted (we don't
    attempt to set it up as a free block).

* In addition, you can have free-tiles of size 4 bytes (i.e. 
    only a header).  However you can't have allocated tiles of 
    this size. 4-byte free blocks are too small to worry about, 
    so we just let them lie there wasting space until being reclaimed.

* All free blocks (except for these 4-byters) have, in addition 
    to the 4-byte header, a 4-byte pointer which forms a link to 
    the next free block of the same size, with the exception of 
    free blocks of size 4 bytes (which don't have any space for 
    anything other than the header). 

*/



uint Heap::TileRoundUp(uint size)
{   
	return power2size[Size2Power(size)];
}


uint Heap::AllocdTileToSize(tile_type tile)
/* Returns the size in bytes of this tile, including the header. */
{
	if ((*(uint*)tile & 2) == 0)
	    return *((uint**)tile)[0];		// A class instance
	else return *((uint*)tile) & ~2;	// A non-class instance
}


uint Heap::FreeTileToSize(tile_type tile)
{
	if (((int*)tile)[0] == -1)
	    return 4;
	else return ((int*)tile)[0] & ~1;
}


bool Heap::TileTypeIsClass(tile_type tile)
{	Classdef* classdef;

	if (*(uint*)tile & 3)
	    return no;
	classdef = *(Classdef**)tile;
	return ((char*)classdef < tile or (char*)classdef > tile + classdef->RoundupSize);
}


Type Heap::TileTypestr(tile_type tile)
/* We assume that this ptr points to a non-class-instance typed block. */
{
	return *(uchar**)tile + 4;
}


Type Heap::Typestr(void* ptr, unsigned char dest[5])
/* Return or construct a typestr for this ptr. */
{	tile_type tile=(char*)ptr-4;

	if (not TileIsTyped(tile))
	    return NULL;
	if (TileTypeIsClass(tile)) {
	    *dest = tp_class;
	    *(Classdef**)(dest + 1) = *(Classdef**)tile;
	    return (Type)dest;
	}
	return TileTypestr(tile);
}


uint Heap::CalcSize2Power(uint size)
{	uint c,h;

	c = size - 1;
	h = 0;
	while (c > 3)
	    c >>= 1, h+=2;
	if (c == 3)
	    h++;
	if (size < power2size[h])
	    h++;
	if (h >= 4)
	    h -= 4;
	else h = 0;
        assert(h < HEAP_NUMPOWERS);
	return h;
}


uint Heap::Size2Power(uint size)
{
	if (size < 512)
	    return size2power[size];
	else return CalcSize2Power(size);
}


Heap::Heap(uint init_size, void* insistAddr)
{
	/* Clear 'freechain': */
	clearA(freechain);

	/* Initialise fields: */
	wholesaleregions = NULL;
	NumFreedSinceMerge = NumUsed = 0;
        Lock = 0;

	/* In VC++, the heap is needed long before we enter WinMain(), because	*/
	/* there are many Windows system-level functions which call 'malloc()'	*/
	/* which is a function we re-define.  In this case, the very first	*/
	/* malloc() call will initialise the heap.  However, in BC++, there is	*/
	/* no such initialisation, which leads to problems if the heap functions*/
	/* are needed before the first call to malloc. So we ensure in any	*/
	/* case that the heap is initialised. */
        CheckInitialised();

	/* Initialise the first (big) tile: */
	AddNewRegion(init_size, insistAddr);
}


Heap::Heap(char* mem, uint numbytes, bool writeable)
{	WholesaleRegion *region;
        int numbytes64K;

	clearA(freechain);
	NumFreedSinceMerge = NumUsed = 0;
        Lock = 0;
        assert((numbytes&3) == 0);
        numbytes64K = ((numbytes-1)|(PAGE_SIZE-1))+1;
        if (writeable) {
            region = (WholesaleRegion*)(mem + numbytes64K - sizeof(WholesaleRegion));
            region->numbytes = numbytes64K - sizeof(WholesaleRegion);
            region->next = NULL;
            wholesaleregions = region;
            if (region->numbytes > numbytes)
	        FreeTile(mem+numbytes, region->numbytes - numbytes);
	}
        else {
            wholesaleregions = NULL;
        }

	/* Add it to the page-map: */
	PageMapAddRegion(mem, numbytes64K);
        Assert();
}


int Heap::MemTotalBytes()
{       int n=0;

	for (each_region)
            n += region->numbytes + sizeof(WholesaleRegion);
        return n;
}


void *Heap::SuggestedBaseAddress(void)
{
        return wholesaleregions->FirstTile();
}


void Heap::WaitForLock()
{
        do {
            Lock++;
            if (Lock == 1)
                return;
            Lock--;
            Sleep(10/*ms*/);
            /* This is guaranteed to work if:
            1. The compiler implements the 'volatile' keyword correctly
            2. The compiler compiles 'Lock++' into:  INC [mem]
            Both are pretty safe assumptions on Intel hardware.
            */
        } while (1);
}


void Heap::Unlock()
{
        Lock--;
}


bool Heap::AssertTile(tile_type tile)
/* Check that this block is allocated and Ok in every respect. */
{
	if (*(uint*)tile & 1)
	    return no;
	if ((*(uint*)tile & 2) == 0) {
	    Classdef* classdef;	// A class instance
	    classdef = *(Classdef**)tile;
	    if ((char*)classdef > tile and (char*)classdef < (char*)tile + classdef->RoundupSize)
		return yes;
	    return classdef->signature == HELLO_BABE;
	}
	else {
	    // A non-class instance
	    return *((uint*)tile) > 0 and *(uint*)tile < 1<<26;
	}
}


static void insitusort(void **Q, int n)
/* A quicksort algorithm that doesn't need to allocate memory. */
{       void *pivot, *p;
        int a,b,c;

        if (n <= 1)
            return;
        pivot = Q[n/2];
        a = b = 0;
        c = n-1;
        while (b <= c) {
            if (Q[b] == pivot)
                b++;
            else if (Q[b] < pivot) {
                if (a < b)
                    p=Q[a], Q[a]=Q[b], Q[b]=p;
                a++;
                b++;
            }
            else {
                p=Q[c], Q[c]=Q[b], Q[b]=p;
                c--;
            }
        }
        insitusort(Q, a);
        insitusort(Q+c, n-c);
}


bool Heap::Assert(void)
/* A detailed validation of the heap, to search for various types of    */
/* corruption.  If it finds an error, it will assert(false) and return  */
/* 'no'. Occasionally, if the heap is corrupt it will crash in here     */
/* instead. */
{	struct {
            tile_type *list;
            int a_idx;                  // allocated size
            int l_idx;                  // real size of list
            int i;                      // Where are we up to?
        } sorted[HEAP_NUMPOWERS];
	tile_type tile, t;
	char* special, *a;
        int NumFreeTiles;
	uint size, h;
	char *end;
	int i,n;

        WaitForLock();
        /*if (MemTotal() > 1024*1024)
            return yes;*/

        /* Build a _sorted_ list of free tiles from the free-list: */
        /* Note that it is unfortunate that we have to do memory   */
        /* allocation inside the memory check algorithm, but it's  */
        /* unavoidable if we don't want an O(n^2) algorithm.  In   */
	/* order to avoid interfering with the heap itself, we     */
	/* allocate this memory using a special low-level call.	   */
	NumFreeTiles = 0;
        for (h=0; h < HEAP_NUMPOWERS; h++) {
            n = 0;
            for (t=(char*)freechain[h]; t; t=((char**)t)[1]) {
                if (++n > 1<<26) {
		    assert(false);
                    return no;         // We must be in a cycle.
		}
            }
            NumFreeTiles += n;
            sorted[h].a_idx = n;
        }
	a = special = (char*)RawRegion(NumFreeTiles*sizeof(tile_type));
        for (h=0; h < HEAP_NUMPOWERS; h++) {
            sorted[h].list = (tile_type*)a;
	    a += sorted[h].a_idx * sizeof(tile_type);
            sorted[h].i = 0;
            i = 0;
            for (t=(char*)freechain[h]; t; t=((char**)t)[1]) {
                assert(i < sorted[h].a_idx);
                sorted[h].list[i++] = t;
            }
            insitusort((void**)sorted[h].list, i);
	    for (int j=1; j < i; j++)
		assert(sorted[h].list[j-1] < sorted[h].list[j]);
            sorted[h].l_idx = i;
        }

        /* Walk through the heap: */
	for (each_region) {
	    tile = (char*)region - region->numbytes;
	    end = (char*)region;
	    while (tile < end) {
		if (IsFree(tile)) {
		    size = FreeTileToSize(tile);
		    if (size == 4)
			;
		    else {
			h = Size2Power(size);
			if (power2size[h] != size) {
                            assert(false);
                            return no;
                        }
			if (tile != sorted[h].list[sorted[h].i]) {
                            if (tile < sorted[h].list[sorted[h].i]) {
                                // This free tile not found in the free-chain,
				// or sorted[h].list[i] not found in the
				// heap walk.
                                assert(false);
                                return no;
                            }
                            else {
                                // 'sorted[h].list[sorted[h].i]' is in the 
				// free-chain but not in the heap walk.
                                assert(false);
                                return no;
                            }
                        }
                        if (++sorted[h].i > sorted[h].l_idx) {
                            assert(false);
                            return no;
                        }
		    }
		}
		else {
		    size = AllocdTileToSize(tile);
		    if (! AssertTile(tile)) {
			assert(false);
			return no;
		    }
		}
		tile += size;
	    }
	    assert(tile == end);
	}
	for (h=0; h < HEAP_NUMPOWERS; h++) {
	    if (sorted[h].i != sorted[h].l_idx) {
                assert(false);
                return no;
                // The free-chain has more entries that we found in the heap.
            }
	}

	/* Return our special memory region to the operating system: */
	FreeRawRegion(special);
        Unlock();

	return yes;
}


static int SuitableBreak(void* p)
/* We try to optimise the application's use of the Intel chips' 128-byte */
/* primary cache lines.  How suitable is 'p' as a place to break two tiles? */
{	int i = (int)p & 127;

	if (i == 0)
	    return 9;
	else if ((i & 63) == 0)
	    return 8;
	else if ((i & 31) == 0)
	    return 7;
	else if ((i & 15) == 0)
	    return 6;
	else if ((i & 7) == 0)
	    return 5;
	else return 4;
}


void Heap::FreeTile(char* p, uint size, uint good_h/*=0*/)
/* We have some free memory which might not have one of the */
/* allowable sizes.  Break it up into pieces if necessary   */
/* and free it.  Insofar as we have choice in the way we    */
/* break it up, try to break it into multiples of 'good_h's */
/* size (this is a hint as to likely next size to be	    */
/* allocated). */
{	uint h, hsz;
	char* p2;

        if (size > MAX_HEAP_SIZE) {
            assert(false);
            return;
        }

	/* Try to make it a multiple of 'good_h': */
	if (good_h and power2size[good_h] <= size) {
	    h = good_h;
	    while (power2size[h+2] <= size)
	    	h += 2;
	    goto HAVE_H;
	}

	/* Just use the standard algorithm: cut it into the biggest */
	/* possible block with the smallest possible remainder, and */
	/* repeat. */
	while (size > 4) {
	    h = Size2Power(size);
	    if (power2size[h] > size)
		h--;
	    HAVE_H:
	    hsz = power2size[h];
	    if (SuitableBreak(p+hsz) > SuitableBreak(p+size-hsz))
		p2 = p + hsz;
	    else
		p2 = p, p = p + size - hsz;
	    *((int*)p) = hsz | 1;
	    ((void**)p)[1] = freechain[h];
	    freechain[h] = p;
	    p = p2;
	    size -= hsz;
	}

	/* Free tiles of size 4 are not put into any free chain; */
	/* they are simply wasted (until the next MergeTiles).   */
	if (size)
	    *(int*)p = -1;

	/* I did try to coalesce 2 blocks of size 4 until I realised */
	/* that this would often cause page faults as we write	     */
	/* outside the allocated region. */
}


void Heap::MergeTiles()
/* Reclaim space lost to fragmented memory by merging */
/* adjacent free blocks. */
{	tile_type p, q;

	memset(freechain, 0, sizeof(freechain));
	for (each_region) {
	    p = region->FirstTile();
	    while (p < (char*)region) {
		if (IsFree(p)) {
		    q = p;
		    do {
			q += FreeTileToSize(q);
		    } while (q < (char*)region and IsFree(q));
		    FreeTile(p, q-p);
		    p = q;
		}
		else {
		    p += AllocdTileToSize(p);
		}
	    }
	}
	//Assert();
}


void* Heap::RawMalloc(uint size, uint h, uint header)
/* Allocate a block of this size, this power and with this header. */
/* The size must include the 4-byte header, and we also require    */
/* the caller to have checked the size and computed the header.    */
{	static int cnt, prevmergecnt=-1000;
        uint *p;

        NumUsed++;
        WaitForLock();                          // To make it thread-safe

	RETRY:
	p = (uint*)freechain[h];		// Do we have a block the right size?
	if (p) {
	    freechain[h] = ((tile_type*)p)[1];
	    *p = header;
            Unlock();
	    return (tile_type)(p + 1);		// Mark the block as used and return it.
	}

	while (++h < HEAP_NUMPOWERS) {		// Break up larger blocks.
	    p = (uint*)freechain[h];
	    if (p) {
		freechain[h] = ((tile_type*)p)[1];
		*p = header;
		FreeTile((char*)p + size, power2size[h] - size);
                Unlock();
		return (tile_type)(p + 1);	// Mark the block as used and return it.
	    }
	}

        if (cnt - prevmergecnt > 100) {         // Don't merge too frequently.
            prevmergecnt = cnt;
            MergeTiles();
            for (h=Size2Power(size); h < HEAP_NUMPOWERS; h++) {
                if (freechain[h]) {
                    p = (uint*)freechain[h];
                    freechain[h] = ((tile_type*)p)[1];
                    *p = header;
                    FreeTile((char*)p + size, power2size[h] - size);
                    // ^ If power2size[h] == size, this just returns immediately.
                    Unlock();
                    return (tile_type)(p + 1);	// Mark the block as used and return it.
                }
            }
        }

	/* Request more memory from the OS */
        int bucket;
        bucket = MemTotalBytes() * 512;
        if (bucket == 0)
            bucket = PAGE_SIZE;
	if (size + 32 >= bucket)        // consider the WholesaleRegion header.
            bucket = (size+32+PAGE_SIZE-1)/PAGE_SIZE*PAGE_SIZE;
        assert(bucket <= MAX_HEAP_SIZE);
        AddNewRegion(bucket);
	h = Size2Power(size);
	goto RETRY;
}


void* Heap::malloc(uint size)
/* Allocate a non-class-instance block. */
{	uint h;

	size += 4;			// The size should include the header.

	h = Size2Power(size);		// What power are we in?

	size = power2size[h];		// Round the size up to this power.

	return RawMalloc(size, h, size|2);
}


void* Heap::New(Classdef* classdef)
/* Allocate a class-instance block. */
{	uint size, h;

	size = classdef->RoundupSize;	// The size should include the header.

	h = Size2Power(size);		// What power are we in?

	size = power2size[h];		// Round the size up to this power.
	assert(size);

	return RawMalloc(size, h, (uint)classdef);
}


void* Heap::New(Type type)
/* Allocate a typed block. */
{	uint tlen, size, h, *p;
	char *mem;

	if (*type == tp_class) {	// Class instances are treated as a special case
	    type++;
	    return New(*(Classdef**)type);
	}

	tlen = LengthOfTypeString(type);// We need to allocate space for the typestr
	while (tlen & 3)		// Both tlen and size must be multiples of 4.
	    tlen++;
	size = TypeSizeWord(type) + tlen + 8;// Add in 4 bytes for the header
                                        // and an additional 4 bytes for the
                                        // 'size' ptr.
	h = Size2Power(size);		// What power are we in?
	size = power2size[h];		// Round the size up to this power.
	assert((tlen & 3) == 0 and (size & 3) == 0);

	mem = (char*)RawMalloc(size, h, size|2);
	p = (uint*)(mem - 4 + size - tlen);
	p[-1] = size;
	memcpy(p, type, tlen);
	((uint**)mem)[-1] = p - 1;
        memset(mem, 0x57, TypeSize(type));

	return mem;
}


void Heap::free(void* p)
{	tile_type tile;
	uint size, h;

	if (p == NULL)
	    return;
	if (not DisableAssert and Ptr_to_heap(p) != this) {
            assert(false);
            return;
        }
	tile = (char*)p - 4;
	if (IsFree(tile)) {
	    assert(false);
	    return;
	}
	size = AllocdTileToSize(tile);
	h = Size2Power(size);
        WaitForLock();
	*((int*)tile) = size | 1;
	((void**)tile)[1] = freechain[h];
	freechain[h] = tile;
	NumFreedSinceMerge++;
	NumUsed--;
        Unlock();
}


void* Heap::realloc(void* oldv, uint newsize)
{	uint oldsize;
	void *newv;

	if (oldv == NULL)
	    return malloc(newsize);
	oldsize = AllocdTileToSize((char*)oldv-4);
	newsize = TileRoundUp(newsize+4);
	if (oldsize == newsize)
	    return oldv;
	if (oldsize > newsize) {
            /* Method 1: use this as an opportunity to reduce fragmentation. */
	    newv = memcpy(malloc(newsize-4), oldv, newsize-4);
	    free(oldv);
	    return newv;
            /* Method 2: minimise 'memcpy()'s: (I find the cost of
            memcpy() to be insignificant) */
	    /*((int*)oldv)[-1] = newsize | 2;
            WaitForLock();
	    FreeTile((char*)oldv - 4 + newsize, oldsize - newsize);
            NumFreedSinceMerge++;
            Unlock();
	    return oldv;*/
	}
	else {
	    newv = memcpy(malloc(newsize-4), oldv, oldsize-4);
	    free(oldv);
	    return newv;
	}
}


void* Heap::calloc(uint a, uint b)
{
	a *= b;
	return memset(this->malloc(a), 0, a);
}


void* Heap::NewPlus(Type type, int extra)
/* Create a new object of this class or typestr and with extra space at the end. */
{	uint tlen, tlen2, inner_size, outer_size, h, *p;
	char *mem;

	if (extra < 0)
	    extra = 0;
	if (extra == 0) {		// Class instances are treated as a special case
	    type++;
	    return New(*(Classdef**)type);
	}

	inner_size = TypeSize(type) + extra;
	while (inner_size & 3)
	    inner_size++;
	tlen = LengthOfTypeString(type);// We need to allocate space for the typestr
	tlen2 = tlen;
	while (tlen2 & 3)
	    tlen2++;
	outer_size = inner_size + tlen2 + 4 + 4;
					// 4 bytes for header and 4 bytes for 'RoundupSize'
	h = Size2Power(outer_size);	// What power are we in?
	outer_size = power2size[h];	// Round the size up to this power.
	assert((tlen2 & 3) == 0 and (outer_size & 3) == 0);

	mem = (char*)RawMalloc(outer_size, h, outer_size|2);
	p = (uint*)(mem - 4 + outer_size - tlen2);
	p[-1] = outer_size;
	memcpy(p, type, tlen);
	*(uint**)((char*)mem-4) = p - 1;

	return mem;
}


void* Heap::realloc(void* oldv, uint size, Type type)
/* A typed realloc, related to NewPlus().  */
{	uint oldsize, newsize, tlen, h;
	void* newv;

	/* Is it really a malloc? */
	if (oldv == NULL)
	    return NewPlus(type, size - TypeSize(type));

	/* What will be the new size? */
	tlen = LengthOfTypeString(type);// We need to allocate space for the typestr
	while (tlen & 3)		// Both tlen and size must be multiples of 4.
	    tlen++;
	newsize = size + tlen + 4 + 4;	// 4 bytes for header and 4 for 'RoundupSize'
	h = Size2Power(newsize);	// What power are we in?
	newsize = power2size[h];	// Round the size up to this power.
	assert((tlen & 3) == 0 and (newsize & 3) == 0);

	/* Is the size changing? */
	oldsize = AllocdTileToSize((char*)oldv-4);
	if (newsize == oldsize)
	    return oldv;
	else if (newsize < oldsize) {	// Shrinking
	    uint *new_pi;
	    tile_type tile = (char*)oldv - 4;
	    new_pi = (uint*)(tile + newsize - tlen - 4);
	    *(uint**)tile = new_pi;
	    *new_pi = newsize;
	    memcpy(new_pi + 1, type, tlen);
	    FreeTile(tile + newsize, oldsize - newsize);
	    return oldv;
	}
	else {				// Expanding
	    newv = NewPlus(type, size - TypeSize(type));
	    memcpy(newv, oldv, msize(oldv));
	    free(oldv);
	    return newv;
	}
}


uint Heap::msize(void* p)
/* What is the inner size of this block?  Inner size means the size that */
/* the user requested, rounded up as appropriate.  So we exclude the	 */
/* bytes occupied by the header or by the type-string. */
{	uint i = *((uint*)p - 1), *pi;

	if ((i & 3) == 0) {
	    pi = (uint*)i;
	    if (pi < (uint*)p or (char*)pi > (char*)p + i)
		return *pi - 4;			// A class instance
	    return (char*)pi - (char*)p;	// A non-class instance typed tile
	}
	if (i & 2)
	    return i - 6;			// An untyped tile
	assert(false);
	return 0;
}


interface uint msize(void* ptr)
{
	return Heap::msize(ptr);
}


interface void* operator new(uint size)
{
        Heap::CheckInitialised();
	return AnonHeap.malloc(size);
}


interface void operator delete(void* mem)
{
	AnonHeap.free(mem);
}


interface void* malloc(Heap* heap, uint size)
{
        Heap::CheckInitialised();
	return heap->malloc(size);
}


interface void* realloc(Heap* heap, void* ptr, uint size)
{
	return heap->realloc(ptr, size);
}


interface void* calloc(Heap* heap, uint size, uint n)
{
        Heap::CheckInitialised();
	return heap->calloc(size, n);
}


interface void free(Heap* heap, void *ptr)
{
	if (ptr == NULL)
	    return;
	if (heap == NULL) {
	    heap = Heap::Ptr_to_heap(ptr);
	    if (heap == NULL) {
		assert(false);
		return;
	    }
	}
	heap->free(ptr);
}


interface str strdup(Heap* heap, str s)
{
	return strcpy((str)heap->malloc(strlen(s)+1), s);
}







/*--------------------- Iterating thru allocked blocks: ----------------*/

void HeapIterator::reset()
{
	tile = NULL;
        region = NULL;
}


HeapIterator::HeapIterator(Heap* _heap)
{
	heap = _heap;
	reset();
}


tile_type HeapIterator::operator++(int dummy)
/* Coming in here, 'tile' points to some unprocessed, possibly free tile. */
/* Return 'tile' (if allocated, otherwise the next allocated tile). */
{
	if (tile == NULL)
	    goto NEXT_REGION;
	tile += Heap::AllocdTileToSize(tile);
	do {
	    if (tile >= (char*)region) {
		NEXT_REGION:
		if (region == NULL)
                    region = heap->wholesaleregions;
                else region = region->next;
                if (region == NULL)
                    return NULL;
		tile = region->FirstTile();
	    }
	    if (not Heap::IsFree(tile))
		return tile;
	    tile += Heap::FreeTileToSize(tile);
	} forever;
#ifndef __BORLANDC__
	return NULL;	// Stupid Windows compiler!
#endif
}





/*-------------------- Module Interface routines: -------------------*/

interface bool MemorySetDefaultHeap(container_id cid)
/* Set 'default_heap' to this region's heap. */
{       Heap *conim;

	if (cid == 0) {
	    default_heap = NULL;
	    return yes;
	}
	conim = Conim::FindConim(cid);
	if (conim == NULL)
	    return no;
	default_heap = conim;
	return yes;
}


interface void HeapCheck(container_id cid)
{       Heap *heap;

	if (cid == 0)
	    heap = default_heap;
	else if (cid == -1)
	    heap = &AnonHeap;
	else {
	    heap = Conim::FindConim(cid);
	    assert(heap != NULL);
	}
	heap->Assert();
}






/*----------- WholesaleRegion's: interfacing with virtual memory ------------*/

/* All Heap objects, except anon_heap, are stored inside the anon_heap.
The anon_heap is a global variable:  the object itself is in global
memory, but its data-structures are in itself. This works by using a
careful ordering of operations.
*/

bool Heap::AddNewRegion(uint numbytes, void* insistAddr)
/* Create a new WholesaleRegion for this heap. Initialise it for */
/* immediate use with malloc/free. */
{	WholesaleRegion *region, **regionp;
	char* mem;

	/* VirtualAlloc only works with a multiple of 64k: */
	assert((numbytes & 65535) == 0);

        /* We insist that Stdlib starts at 0xb10000: */
        if (insistAddr) {
	    mem = (char*)VirtualAlloc((char*)0xb10000, numbytes,
				MEM_COMMIT|MEM_RESERVE, PAGE_READWRITE);
            if (mem == NULL)
                throw 1;
            assert(wholesaleregions == NULL);
            region = wholesaleregions = (WholesaleRegion*)(mem + numbytes) - 1;
            region->numbytes = numbytes - sizeof(WholesaleRegion);
            region->next = NULL;
            FreeTile(mem, region->numbytes);
            goto OKAY;
        }


	/* Try to get memory adjoining an existing region: */
	for (regionp=&wholesaleregions; (region=*regionp) != NULL;
				regionp=&region->next) {
	    mem = (char*)VirtualAlloc((char*)(region + 1), numbytes,
				MEM_COMMIT|MEM_RESERVE, PAGE_READWRITE);
	    if (mem) {
		region = (WholesaleRegion*)(mem + numbytes) - 1;
		region->numbytes = (*regionp)->numbytes + numbytes;
		region->next = (*regionp)->next;
		FreeTile((char*)*regionp, numbytes);
		*regionp = region;
		goto OKAY;
	    }
	}

        /* Get memory from anywhere: */
        mem = (char*)VirtualAlloc(NULL, numbytes,
				MEM_COMMIT|MEM_RESERVE, PAGE_READWRITE);
	for (regionp=&wholesaleregions; (region=*regionp) != NULL;
				regionp=&region->next) {
	    if ((char*)region > mem)
		break;
	}
	region = (WholesaleRegion*)(mem + numbytes) - 1;
	region->numbytes = numbytes - sizeof(WholesaleRegion);
	region->next = *regionp;
	*regionp = region;
	FreeTile(mem, region->numbytes);

	/* Now update wholesaleregions. */
        OKAY:
	DisableAssert++;
	PageMapAddRegion(mem, numbytes);
	DisableAssert--;
	return yes;
}


static void* RawRegion(int numbytes)
/* Get a new uninitialised large memory block from the OS */
{
        return VirtualAlloc(NULL, numbytes,
				MEM_COMMIT|MEM_RESERVE, PAGE_READWRITE);
}


static void FreeRawRegion(void* mem)
{
	VirtualFree(mem, 0, MEM_RELEASE);
}


Heap::~Heap()
{	MEMORY_BASIC_INFORMATION info;
        WholesaleRegion *region;
        char* mem;
	int size;

        if (this == anon_heap)
            return;

        /* Return all memory to Windows and clear PageToHeap[]. */
        HeapCheck(-1);
	for (region=wholesaleregions; region; ) {
            mem = region->FirstTile();
	    PageMapDelRegion(mem, region->numbytes+sizeof(WholesaleRegion));
            size = region->numbytes;
            region = region->next;
            while (size > 0) {
                VirtualQuery(mem, &info, sizeof(info));
 	        VirtualFree(mem, 0, MEM_RELEASE);
                mem += info.RegionSize;
                size -= info.RegionSize;
            }
	}
        HeapCheck(-1);
}


void Heap::InitialiseAnon()
/* Initialise the static variables and the 'anon_heap'. */
/* 'anon_heap' is not constructed using a normal        */
/* constructor because this would occur before WinMain()*/
/* and lots of confusing ordering issues would arise.   */
{	uint b,c,i;

	/*** We need to initialise the static variables. ***/
	/* Initialise 'power2size[]'. */
	for (c=0; c < HEAP_NUMPOWERS; c++) {
	    b = c & 1;
	    i = c >> 1;
	    power2size[c] = b ? (12<<i) : (8<<i);
	}

	/* Initialise 'size2power[]'. */
	for (i=1; i < 512; i++)
	    size2power[i] = CalcSize2Power(i);
	size2power[0] = 0;


	/*** Initialise anon_heap: ***/
	AnonHeap.AddNewRegion(65536*2);
        default_heap = &AnonHeap;
}


void Heap::PageProtection(bool _writeable)
/* Mark all the pages in this heap as read-only. */
{       MEMORY_BASIC_INFORMATION info;
	DWORD OldProtect;
        char* mem;
	int size;

	for (each_region) {
            mem = region->FirstTile();
	    PageMapDelRegion(mem, region->numbytes);
            size = region->numbytes;
            while (size > 0) {
                VirtualQuery(mem, &info, sizeof(info));
	        VirtualProtect(mem, info.RegionSize,
			_writeable ? PAGE_READWRITE : PAGE_READONLY,
			&OldProtect);
                mem += info.RegionSize;
                size -= info.RegionSize;
            }
        }
}



