/************************************************************\
** NAME.CPP:						    **
**          This module manages directories and variable    **
** scope and everything in-between: creating named objects, **
** resolving strings to names and so on.		    **
\************************************************************/

#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "barbados.h"
#include "namedecl.h"
#include "make_container.h"
#include "memory.h"
#include "source.h"
#include "array.h"
#include "make.h"
#include "name.h"


interface Directory* curdir, *root_dir=NULL;
interface Directory* common_dir;
interface Conim *curconim;		// = Ptr_to_heap(curdir)
interface Make *old_make;        // The make-entity for an overwritten object.
interface bool MakeCanBeClosed;

static void NameDestroy(Namedobj* obj);
static container_id COMMON_CID;
static Namedobj* scope_list;          // For scoping

Namedobj* debug_obj;







/*--------------- Directory Functions: ---------------*/

/*class Directory {
public:
	Namedobj **Hash;
	uint Hash_len, NumEntries;
	void Resize(uint newsize);
	void InsertSimple(Namedobj* obj);
	bool DeleteSimple(Namedobj* obj);
	uint CalcHash(str name);

public:
	Directory *parent;
	container_id parent_cid;

public:
	void Insert(Namedobj* obj);
	bool Delete(Namedobj* obj);
	Namedobj* Find(str name);
} *Directory*;*/


void Directory::init(Directory *dir)
/* Initialise a new directory: a child of 'dir'. */
{
	parent = dir;
	parent_cid = dir->parent_cid;
	Hash_len = 0;
	NumEntries = 0;
	Hash = NULL;
}


uint Directory::CalcHash(str name)
{	uint h=0;
	str s;

	for (s=name; *s; s++)
	    h ^= *s + (h << 3);
	h %= Hash_len;
	return h;
}


void Directory::InsertSimple(Namedobj* obj)
/* Insert this object without checking for a need to expand. */
{	uint h;

	h = CalcHash(obj->name);
	obj->next = Hash[h];
	Hash[h] = obj;
}


bool Directory::DeleteSimple(Namedobj* obj)
{	Namedobj* *objp;

	objp = &Hash[CalcHash(obj->name)];
	while (*objp != NULL) {
	    if (*objp == obj) {
		*objp = obj->next;
		return yes;
	    }
	    objp = &(*objp)->next;
	}
	return no;
}


void Directory::Resize(uint newsize)
{	Namedobj* *OldHash, *obj;
	uchar typebuf[20];
	int i, Old_len;
	Heap *heap;

	/* Move the old array to OldHash: */
	OldHash = Hash;
	Old_len = Hash_len;

	/* A minimum size of 7: */
	if (newsize < 7)
	    newsize = 7;

	/* Create the new hash array: */
	heap = Heap::Ptr_to_heap(this);
	typebuf[0] = tp_array;
	*(int*)(typebuf + 1) = newsize;
	typebuf[5] = tp_pointer;
	typebuf[6] = tp_class;
	*(Classdef**)(typebuf + 7) = NamedobjStruct;
	Hash = (Namedobj**)heap->New(typebuf);
	memset(Hash, 0, newsize * sizeof(str));
	Hash_len = newsize;

	/* Move all the objects across: */
	for (i=Old_len; --i >= 0; ) {
	    while (OldHash[i]) {
		obj = OldHash[i];
		OldHash[i] = obj->next;
		InsertSimple(obj);
	    }
	}

	/* Free the old array: */
	if (OldHash)
	    free(heap, OldHash);
}


void Directory::Insert(Namedobj* obj)
{
	if (++NumEntries >= Hash_len*2)
	    Resize(Hash_len * 2 + 1);
	InsertSimple(obj);
}


bool Directory::Delete(Namedobj* obj)
{
	if (not DeleteSimple(obj))
	    return no;
	if (--NumEntries < Hash_len*2) {
	    if (NumEntries == 0) {
		free(NULL, Hash);
		Hash = 0;
		Hash_len = 0;
	    }
	    else if (Hash_len > 7)
		Resize((Hash_len-1) / 2);
	}
	return yes;
}


Namedobj* Directory::Find(str name)
{	Namedobj* obj;

	if (Hash_len == 0)
	    return NULL;
	for (obj=Hash[CalcHash(name)]; obj; obj=obj->next) {
	    if (streq(obj->name, name))
		return obj;
	}
	return NULL;
}


Namedobj** Directory::SetFind(str name, Namedobj* *A)
/* Find ALL objects of this name in this directory,	*/
/* and append them to the array A (default value=NULL).	*/
/* Return 'A'. */
{	Namedobj* obj;

	if (Hash_len == 0)
	    return A;
	for (obj=Hash[CalcHash(name)]; obj; obj=obj->next) {
	    if (streq(obj->name, name))
		Array_Add(A, obj);
	}
	return A;
}


Namedobj* Directory::FindDirObj(str name, bool include_dirrefs, bool include_containers)
/* Find the first directory/container object of this name. */
{	Namedobj* obj;

	if (Hash_len == 0)
	    return NULL;
	for (obj=Hash[CalcHash(name)]; obj; obj=obj->next) {
	    if (streq(obj->name, name) and IsDirectory(obj, include_dirrefs, include_containers))
		return obj;
	}
	return NULL;
}


Directory* Directory::FindDirectory(str name)
/* Find the first directory/container object of this name. */
{	Namedobj* obj;

	if (Hash_len == 0)
	    return NULL;
	for (obj=Hash[CalcHash(name)]; obj; obj=obj->next) {
	    if (streq(obj->name, name) and IsDirectory(obj, no, no) and obj->storage == static_storage)
		return (Directory*)((StaticNamedobj*)obj)->location;
	}
	return NULL;
}


Namedobj** Directory::FindExistingObjp(str name,
		Type type, bool AllowOverload, uint *overload_versionp)
/* Search this directory for an object with the same name. */
/* If we find one, then either we need to overwrite it or  */
/* overload the name.  If we need to overwrite it, then    */
/* return a pointer to the pointer we need to update. Else */
/* return a pointer to a NULL. */
{	uint v,versions=(uint)-1;
	Namedobj* obj, **objp;
	Type footprint;

	if (Hash_len == 0)
	    return NULL;
	if (*type == tp_function)
	    footprint = type;
	else footprint = NULL;
	*overload_versionp = 0;
	for (objp=&Hash[CalcHash(name)]; (obj=*objp) != NULL; objp=&(obj->next)) {
	    if (streq(obj->name, name)) {
		if (not AllowOverload) {
		    *overload_versionp = obj->overload_version;
		    return objp;
		}
		if (EqualFootprint(obj->type, footprint)) {
		    *overload_versionp = obj->overload_version;
		    return objp;
		}
		versions &= ~(1 << obj->overload_version);
                // Currently this algorithm limits us to 32 versions.
	    }
	}
	if (versions != -1) {
	    for (v=0; v < 31; v++) {
		if (versions & (1<<v)) {
		    *overload_versionp = v;
		    break;
		}
	    }
	}
	return NULL;
}


Classdef* NameClassdefFromObject(Namedobj* obj)
/* This object (hopefully) contains a typedef for some class. */
/* Return the class. */
{	Classdef* classdef;
	Type type;

	if (obj->storage != typedef_storage)
	    return NULL;
	type = obj->type;
	if (*type != tp_class)
	    return NULL;
	type++;
	GetPtr(classdef, type);
	return classdef;
}


Namedobj* Directory::ObjFromUname(str name)
/* Map this str to the unique object which it represents.   */
/* For example:                                             */
/*      f()                                                 */
/*      Complex::Print()                                    */
/*      operator+'2()                                       */
/*      A::subA::subsubA::foo()                             */
{       Classdef* classdef;
	int overload_version;
	Namedobj* obj;
	char base[512];
	str s,d;


	/* Get the base word: */
	d = base;
	for (s=name; *s; ) {
	    if (*s == '\'' or (*s == ':' and s[1] == ':'))
		break;
	    else *d++ = *s++;
	}
	*d = '\0';
	obj = Find(base);
	if (obj == NULL)
	    return NULL;

        /* Is it a member? */
	while (*s == ':' and s[1] == ':') {
            classdef = TypeToClassdef(obj->type);
            if (classdef == NULL)
                return NULL;
            s += 2;
            d = base;
            for ( ; *s; ) {
                if (*s == '.' or (*s == ':' and s[1] == ':'))
                    break;
                else *d++ = *s++;
            }
            *d = '\0';

            for (obj=classdef->member; obj; obj=obj->next) {
                if (streq(obj->name, base))
                    goto FOUND;
            }
            return NULL;
            FOUND:;
        }

	/* Is it an overloaded name? */
	if (*s == '\0') {
	    until (obj->overload_version == 0)
		obj = obj->next;
	    return obj;
	}

	/* Search linearly: */
	assert(*s == '.');
	overload_version = atoi(++s);
	for ( ; obj; obj=obj->next) {
	    if (obj->overload_version == overload_version
			and streq(obj->name, base))
		return obj;
	}


	/* No luck? */
	return NULL;
}





/*--------------- Scope functions ---------------*/

interface void NameInit(void)
{
	scope_list = NULL;
	MakeCanBeClosed = yes;
}


interface scopemarker_type NameMarkScope()
/* Get a pointer which marks the current position on the stack of */
/* local variables. */
{
	return scope_list;
}


interface void NamePopScope(scopemarker_type scope)
/* Delete all local variables back to the specified position */
/* on the stack of local variables. */
{       Classdef* classdef;
	Namedobj* obj;

	while (scope_list != scope) {
	    obj = scope_list;
	    assert(obj != NULL);

	    /* Now remove it from the `scope' linked list: */
	    scope_list = obj->next;

	    /* Is it an original typedef_obj?  If so, free the classdef. */
	    if (NameIsOriginalClass(obj)) {
		classdef = NameClassdefFromObject(obj);
		default_heap->free(classdef);
	    }

	    /* Free the obj: */
            if (obj->storage != auto_storage and obj->storage != parameter_storage
                and obj->storage != local_static)
	        free(NULL, obj);
	}
}


interface Namedobj* NameGrabList(scopemarker_type scope)
/* Take all objects declared since 'scope' and return it as */
/* a linked list. */
{	Namedobj* obj, *temp, *temp2;

	temp = NULL;
	for (obj=scope_list; obj != scope; obj=temp2) {
	    temp2 = obj->next;
	    obj->next = temp;
	    temp = obj;
	}
	scope_list = scope;
	return temp;
}


interface Namedobj* NamePartialMembers()
/* Returns a pointer to the linked list of partially parsed members */
/* of a class/struct/union. */
{
	return scope_list;
}





/*------------------- Resolving names: ---------------------*/

interface Namedobj* NameLocalResolve(str name)
/* Resolve this name to a single object in the local-variable scope. */
{	Namedobj* obj;

	for (obj=scope_list; obj; obj=obj->next) {
	    if (streq(obj->name, name) and not IsConstructor(obj))
		return obj;
	}
        return NULL;
}


static Namedobj* DirToObj(Directory* dir)
/* Convert a directory to the Namedobj in its parent directory. */
{	Directory* parent;
	Namedobj *obj;
	uint h;

	parent = (Directory*)dir->parent;
	if (parent == NULL)
	    return NULL;
	for (each_dir_obj(parent))
	    if (((StaticNamedobj*)obj)->location == dir)
		return obj;
	return NULL;
}


interface Namedobj* NamePathToObj(str path)
/* Convert this path to a (Namedobj,directory).	 This algorithm is quite */
/* similar to the compiler's path-name resolution algorithm, because it  */
/* understands '..' and absolute vs relative path-names and the '::'     */
/* scope-resolution operator, however it also deals with 'unames', ie.   */
/* Unique Names, whereby overloaded functions are distinguished with a ' */
/* symbol and version number. */
{       container_id new_cid;
        void* location;
	Directory* dir;
        Namedobj* obj;
        char buf[512];
	str d,s;

	/* If it's wrapped in quotes, then ignore the quotes: */
	if (*path == '"')
	    path++;

	/* Get the starting point: */
	if (*path == '/') {
            new_cid = ROOT_CID;
	    s = path;
            goto HAVE_CID;
	}
	else if (*path == '.' and path[1] == '/') {
	    dir = curdir;
	    s = path + 2;
	}
	else if (*path == '.' and path[1] == '\0') {
	    return DirToObj(curdir);
	}
	else {
	    s = path;
	    for (d=buf; *s and *s != '/'; )
		*d++ = *s++;
	    if (d[-1] == '"')
		d--;
	    *d = '\0';
	    if (streq(buf, "..")) {
		dir = curdir->parent;
		if (dir == NULL and curdir->parent_cid) {
                    new_cid = curdir->parent_cid;
                    goto HAVE_CID;
                }
                else if (dir == NULL) {
		    NO_PARENT:
		    ErrorRun("That directory has no parent.");
		    return NULL;
		}
		if (*s == '\0')
		    return DirToObj(dir);
		else s++;
	    }
	    else {
                extern Namedobj* ResolveGlobal(str buf);
		obj = curdir->ObjFromUname(buf);
		if (obj == NULL)
		    obj = ResolveGlobal(buf);
		if (obj == NULL) {
		    NO_OBJECT:
		    ErrorRun("No such %s:  %s", *s ? "directory" : "object", buf);
		    return NULL;
		}
		if (*s == '\0')
		    return obj;
                if (obj->storage != static_storage)
                    goto NO_OBJECT;
		if (not TypeEqual(obj->type, direc_typstr)) {
		    NOT_DIR:
		    ErrorRun("%s  is not a directory", buf);
		    return NULL;
		}
		dir = (Directory*)((StaticNamedobj*)obj)->location;
		s++;
	    }
	}

	/* Get each directory in the chain: */
	do {
	    for (d=buf; *s and *s != '/'; )
		*d++ = *s++;
	    if (d[-1] == '"')
		d--;
	    *d = '\0';
	    if (streq(buf, "..")) {
		dir = dir->parent;
		if (dir == NULL and dir->parent_cid) {
                    new_cid = dir->parent_cid;
                    goto HAVE_CID;
                }
                else if (dir == NULL)
		    goto NO_PARENT;
		if (*s == '\0')
		    return DirToObj(dir);
	    }
	    else {
		obj = dir->ObjFromUname(buf);
		if (obj == NULL)
		    goto NO_OBJECT;
                if (obj->storage != static_storage and obj->storage != const_storage)
		    goto NO_OBJECT;
		if (*s == '\0')
		    return obj;
                location = ((StaticNamedobj*)obj)->location;
		if (TypeEqual(obj->type, direc_typstr))
                    dir = (Directory*)location;
	    	else if (TypeEqual(obj->type, dirref_typstr)) {
		    if (obj->storage != static_storage)
		    	goto STORAGE_ERR;
                    dir = *(Directory**)location;
                }
                else if (TypeEqual(obj->type, container_typstr)) {
                    if (obj->storage == static_storage)
                        new_cid = *(container_id*)location;
                    else if (obj->storage == const_storage)
                        new_cid = (container_id)location;
                    else {
                    	STORAGE_ERR:
                    	ErrorRun("Strange storage class");
                        return NULL;
                    }
                    HAVE_CID:
                    dir = OpenContainer(new_cid, READONLY);
                    curconim->PublicLink(Conim::FindConim(new_cid));
                    CloseContainer(new_cid);
		}
                else goto NOT_DIR;
	    }
	    s++;
	} forever;
}





/*------------- Declaring (i.e. creating) objects: ------------*/

interface uint SizeofObj(str name, Type type, storage_enum storage)
/* What size would we need for this named object? */
{       int basicsize;

        if (storage == member_storage or storage == inherit_storage)
            basicsize = sizeof(FieldNamedobj);
        else if (*type == tp_function)
            basicsize = sizeof(FunctionNamedobj);
        else if (storage == typedef_storage)
            basicsize = sizeof(Namedobj);
        else if (storage == const_storage)
            basicsize = sizeof(IntNamedobj);
        else if (storage == static_storage)
            basicsize = sizeof(StaticNamedobj);
        else assert(false);
        return basicsize + LengthOfTypeString(type) + strlen(name)+1;
}


static Namedobj* CreateObj(Directory* dir, int basicsize, str name, Type type,
                Type nob_type, storage_enum storage,
		Namedobj* realloc_me)
/* Create a Namedobj.  Create it in dir's heap. Create it by reallocking */
/* 'realloc_me' if specified.  The 'name', 'type' and 'storage' fields   */
/* are initialised, but the remainder of construction must be done by    */
/* the caller. */
{       uint tsize, size;
	Namedobj* obj;
	Heap* heap;

        assert (storage != parameter_storage and storage != auto_storage
                and storage != keyword_storage);
	tsize = LengthOfTypeString(type);
	heap = dir ? Ptr_to_conim(dir) : anon_heap;
        size = basicsize + tsize + strlen(name)+1;
        while (size&3)
            size++;
	obj = (Namedobj*)heap->realloc(realloc_me, size, nob_type);
	obj->storage = storage;
        obj->type = (Type)obj + basicsize;
	memcpy(obj->type, type, tsize);
	obj->name = strcpy((str)obj->type + tsize, name);
	return obj;
}


static Namedobj* CreateStaticObj(Directory* dir,
		str name, Type type, Namedobj* realloc_me)
/* Create a 'static_storage' Namedobj with the data part embedded in */
/* the same heap tile. */
{       int embed_size, vftable_size;
        StaticNamedobj* obj;
	Classdef* classdef;
        str s;

        embed_size = TypeSize(type);	// Currently, the type size
        // does _not_ include the virtual function pointer.
        if (*type == tp_class) {
            classdef = *(Classdef**)(type+1);
            vftable_size = classdef->VirtualFns ? sizeof(str) : 0;
            /* Leave space for a virtual-function table pointer. */
        }
        else {
            classdef = NULL;
            vftable_size = 0;
        }
        obj = (StaticNamedobj*)CreateObj(
                        dir, sizeof(StaticNamedobj) + vftable_size + embed_size,
                        name, type,
                        staticnamedobj_typstr, static_storage, realloc_me);
        s = (str)(obj + 1);
        if (vftable_size) {
            *(Classdef**)s = classdef;	    // Insert the virtual-function table pointer.
            s += sizeof(classdef);
        }
        obj->location = s;
        if (realloc_me == NULL)
            memset(s, 0, embed_size);
        return obj;
}


interface void DestroyObj(Namedobj* obj)
/* We are about to overwrite this variable.  Do anything */
/* we need to do to destroy the value. */
{
	if (obj->storage == static_storage)
	    AccessDeleteInterval(((StaticNamedobj*)obj)->location, TypeSize(obj->type));
	else if (obj->storage == macro_storage)
	    curconim->free(((MacroNamedobj*)obj)->macro);
	else if (obj->storage == straight_fn or obj->storage == member_fn)
	    curconim->free(((FunctionNamedobj*)obj)->u.fn);
}


static void ConstructObj(Namedobj* obj, Directory* dir)
/* When we create a new static_storage obj, call the constructor. */
{       Type type;

	if (obj->storage != static_storage)
	    return;

	type = obj->type;
	if (TypeEqual(type, direc_typstr)) {
	    Directory* newdir = (Directory*)((StaticNamedobj*)obj)->location;
	    newdir->init(dir);
	}
}



interface Namedobj* NameDeclare(Directory* dir,
	    str name, Type type, storage_enum storage,
	    int options)
/* Create a new named-obj.  If 'dir' then create it in this	*/
/* directory, otherwise presumably we're creating local	objects	*/
/* or member objects.  This function will never fail (so the    */
/* caller must perform all error-checking). */
{       LocalStaticNamedobj *objLS;
	uint overload_version=0;
        Namedobj* obj, **objp;
	Make *old_make;
	bool NeedsMake;
        int offset;


	/* Tidy up the parameters: */
	if (*type == tp_function) {
	    if (storage == static_storage)
		storage = straight_fn;
	    else if (storage == member_storage)
		storage = member_fn;
	}


	/*** Do slightly different stuff for each storage type: ***/
	old_make = NULL;
	switch (storage) {
					    /**** Persistent Entities: ****/
	    case static_storage:                /* Static (persistent) vars */
		assert(dir);
		objp = dir->FindExistingObjp(name, type, no, &overload_version);
		if (objp) {
		    obj = *objp;
		    if (obj->storage == storage and TypeEqual(obj->type, type)) {
			if (obj->make) {
			    MakeChanged(obj->make, no);
			    return obj;	    // Nothing has changed,
			    // so we just return the same obj.
			}
			else if (PredefinitionPhase or (options & NAME_CREATEMAKE) == 0)
			    return obj;	    // We didn't need the make node anyway.
		    }
		    DestroyObj(obj);
		    old_make = obj->make;
		}
                else obj = NULL;
                obj = CreateStaticObj(dir, name, type, obj);
                if (objp)
                    *objp = obj;
		ConstructObj(obj, dir);
		NeedsMake = yes;
		break;

	    case typedef_storage:               /* Typedefs */
		NeedsMake = (dir != NULL);
		if (dir)
		    objp = dir->FindExistingObjp(name, type, no, &overload_version);
		else objp = NULL;
		if (objp) {
		    obj = *objp;
		    if (obj->storage == storage and TypeEqual(obj->type, type)) {
			if (obj->make)
			    MakeChanged(obj->make, no);
			return obj;	    // Nothing has changed,
			// so we just return the same obj.
		    }
		    old_make = obj->make;
		    DestroyObj(obj);
		    obj = *objp = CreateObj(dir, sizeof(Namedobj), name, type,
                                namedobj_typstr, storage, obj);
		}
		else {
		    obj = CreateObj(dir, sizeof(Namedobj), name, type,
                                namedobj_typstr, storage, NULL);
		}
		break;

	    case const_storage:                 /* Enum constants */
		NeedsMake = (dir != NULL);
		if (dir)
		    objp = dir->FindExistingObjp(name, type, no, &overload_version);
		else objp = NULL;
		if (objp) {
		    obj = *objp;
		    /*if (obj->storage == storage and TypeEqual(obj->type, type)
			and obj->u.constval == enum_value) {
			assert(obj->make or PredefinitionPhase);
			MakeChanged(obj->make, no);
			return obj;	    // Nothing has changed,
			// so we just return the same obj.
		    }*/
		    /* We need a certain order of the linked list, so */
		    /* we need to delete and redeclare enum constants */
		    /* if they are being recompiled. */
		    old_make = obj->make;
		    DestroyObj(obj);
		    obj = *objp = CreateObj(dir, sizeof(IntNamedobj), name, type,
                                namedobj_typstr, storage, obj);
		}
		else {
		    obj = CreateObj(dir, sizeof(IntNamedobj), name, type,
                                namedobj_typstr, storage, NULL);
		}
		((IntNamedobj*)obj)->constval = enum_value;
		break;

	    case macro_storage:                 /* Preprocessor macros */
		assert(dir);
		objp = dir->FindExistingObjp(name, type, no, &overload_version);
		if (objp) {
		    obj = *objp;
		    old_make = obj->make;
		    DestroyObj(obj);
		    obj = *objp = CreateObj(dir, sizeof(MacroNamedobj), name, type,
                                namedobj_typstr, storage, *objp);
		}
		else {
		    obj = CreateObj(dir, sizeof(MacroNamedobj), name, type,
                                macronamedobj_typstr, storage, NULL);
		}
		((MacroNamedobj*)obj)->macro = NULL;
		NeedsMake = yes;
		break;

	    case straight_fn:                   /* Static functions */
		assert(dir);
		objp = dir->FindExistingObjp(name, type,
                        (options & NAME_OVERLOAD) != 0, &overload_version);
		if (objp) {
		    obj = *objp;
		    if (obj->storage == storage and TypeEqual(obj->type, type)) {
                        if (options & NAME_CREATEMAKE) {
                            if (obj->make == NULL)
                                obj->make = MakeEntityNew(obj);
                            MakeChanged(obj->make, no);
                        }
                        else obj->make = NULL;
			return obj;	    // Nothing has changed,
			// so we just return the same obj.
		    }
		    old_make = obj->make;
		    DestroyObj(obj);
		    obj = *objp = CreateObj(dir, sizeof(FunctionNamedobj), name, type,
                                functionnamedobj_typstr, storage, obj);
		}
		else {
		    obj = CreateObj(dir, sizeof(FunctionNamedobj), name, type,
                                functionnamedobj_typstr, storage, NULL);
		}
		((FunctionNamedobj*)obj)->u.fn = NULL;
		NeedsMake = yes;
		break;

	    case member_fn:			/* Member functions */
		obj = CreateObj(curdir, sizeof(FunctionNamedobj), name, type,
                                functionnamedobj_typstr, storage, NULL);
		objp = NULL;
		((FunctionNamedobj*)obj)->u.fn = NULL;
		NeedsMake = (options & NAME_CREATEMAKE);
		break;

	    case oneinstr_fn:                   /* Built-in operators */
		obj = CreateObj(dir, sizeof(FunctionNamedobj), name, type,
                                functionnamedobj_typstr, storage, NULL);
		objp = NULL;
		NeedsMake = no;
		break;

	    case virtual_fn:                    /* Virtual Functions */
		obj = CreateObj(dir, sizeof(FunctionNamedobj), name, type,
                                functionnamedobj_typstr, storage, NULL);
		objp = NULL;
		NeedsMake = (options & NAME_CREATEMAKE);
		break;

	    case auto_storage:                  /* Auto variables */
		auto_offset -= TypeSizeWord(type);
		obj = CreateAutoObj(name, type, storage, auto_offset);
		objp = NULL;
		// The stack grows downward, and so these values are
                // represented negatively.
                if (*type == tp_class) {
                    Classdef* classdef = *(Classdef**)(type+1);
                    if (classdef->VirtualFns)
                        auto_offset -= sizeof(void*);
                        // Leave room for the virtual function table pointers.
                }
		max_auto_offset = min(max_auto_offset, auto_offset);
		NeedsMake = no;
		break;

	    case parameter_storage:             /* Parameters */
		obj = CreateAutoObj(name, type, storage, parameter_location);
		objp = NULL;
		NeedsMake = no;
		break;

	    case local_static:                  /* Local statics */
		obj = objLS = CreateLocalStaticObj(name, type);
		objp = NULL;
		objLS->location = (void*)AllocateLocalStatic(TypeSizeWord(type), type);
		NeedsMake = no;
		break;

	    case inherit_storage:
	    case member_storage:                /* Members */
		obj = CreateObj(curdir, sizeof(FieldNamedobj), name, type,
                                namedobj_typstr, storage, NULL);
		objp = NULL;
		if (*type == tp_char or *type == tp_short or BitField)
		    ;
		else WordAlign(member_offset);
		assert(InsideAggregate or declarator_class);
		if (InsideAggregate == in_union) {
                    offset = 0;
		    member_offset = max(member_offset, TypeSizeWord(type));
		}
		else {
		    if (BitField) {
			assert(member_bit_offset > -8 and member_bit_offset <= 0);
			if (BitField == -1) {
			    /* This was declared as a width of 0.  It means: "byte-align the next field". */
			    member_bit_offset = 0;
			    offset = EncodeBitField(member_offset, 0, 0);
			}
			else {
			    if (member_bit_offset + BitField >= 32)
				member_bit_offset = 0;
			    if (member_bit_offset == 0)
				offset = EncodeBitField(member_offset, 0, BitField);
			    else offset = EncodeBitField(member_offset-1, 8+member_bit_offset, BitField);
			    member_bit_offset += BitField;
			    if (member_bit_offset > 0) {
				member_offset += 1 + (member_bit_offset >> 3);
				member_bit_offset -= 8 * (1 + (member_bit_offset >> 3));
			    }
			}
		    }
		    else {
			member_bit_offset = 0;
                        if (*type == tp_class and (*(Classdef**)(type+1))->VirtualFns)
                            member_offset += sizeof(void*);
			offset = member_offset;
			member_offset += TypeSize(type);
		    }
		}
                ((FieldNamedobj*)obj)->offset = offset;
		NeedsMake = no;
		break;

	    default:
		assert(false);
		break;
	}


	/*** Fill in the remaining fields: ***/
	obj->overload_version = overload_version;       /* Overload Version */
	obj->visibility = visibility;			/* Visibility */


	/*** Put it into the directory/class/scope: ***/
	if (dir) {					/* Scoping */
	    if (objp)
		;	// It must already be in the linked list.
	    else dir->Insert(obj);
	    obj->owner = dir;
	}
	else {
	    assert(objp == NULL);
	    obj->next = scope_list;
	    scope_list = obj;
	    obj->owner = NULL;
	}


	/*** Make: ***/
	if (NeedsMake and (options & NAME_CREATEMAKE) != 0 and not PredefinitionPhase) {
	    if (old_make) {				/* Make */
		// We already have a make-entity, so we must put it
		// into the new object.
		obj->make = old_make;
		MakeChanged(obj->make, yes);
		if (MakeCanBeClosed)
		    MakeEntityUpdate(obj);
	    }
	    else {
		obj->make = MakeEntityNew(obj);
		MakeChanged(obj->make, yes);
	    }
	}
	else {
	    assert(old_make == NULL);	// Otherwise we'd have to delete it.
	    obj->make = NULL;
	}


	/*** Return it: ***/
	return obj;
}







/*-------------------- Deleting objects: ---------------------*/

interface bool NameDeleteClass(Namedobj* typedef_obj)
/* Delete this class, including the class definition and */
/* the member functions. */
{	Classdef* classdef;
	Namedobj* obj;

	classdef = TypeToClassdef(typedef_obj->type);
	if (classdef == NULL)
	    return no;
	assert(classdef->typedef_obj == typedef_obj);

	for (obj=classdef->member; obj; obj=obj->next) {
	    if (obj->type[0] == tp_function) {
		if (obj->make)
		    MakeDelete(obj->make);
                default_heap->free(((FunctionNamedobj*)obj)->u.fn);
                // free() ignores NULL values being passed.
	    }
	}
	default_heap->free(classdef);
	NameDelete(curdir, typedef_obj);
	return yes;
}


static bool NameCanDelete(Namedobj* obj)
/* Can we delete this object?   The answer is no if */
/* it's a directory that's not empty.               */
{       Directory* dir;

	if (IsDirectory(obj, no, no)) {
	    dir = (Directory*)((StaticNamedobj*)obj)->location;
	    if (not dir->Empty())
		return no;
	    if (dir == curdir)
		return no;
	}

	return yes;
}


interface bool NameDelete(Directory* dir, Namedobj* obj)
/* Delete this object from this directory. */
{       Conim* conim;

	/* Do we have write-permission on this container? */
	conim = Ptr_to_conim(dir);
	if (not conim->IsWriteable())
	    return no;

	/* Is it deletable? */
	if (not NameCanDelete(obj)) {
	    ErrorRun("This object cannot be deleted.");
	    return no;
	}

	/* Delete it as a directory entry: */
	dir->Delete(obj);

	/* Delete its make object: */
	if (obj->make != NULL and obj->make != old_make)
	    MakeDelete(obj->make);

	/* Free any data associated with it: */
	DestroyObj(obj);

	/* Remove any breakpoints it had: */
	DebugDeleteObj(obj);

	/* Free the object itself: */
	conim->free(obj);
	return yes;
}






/*------------------- Miscellaneous Name functions: ----------------------*/

interface bool NameIsOriginalClass(Namedobj* obj)
/* Is this object the original typedef-object */
/* for a class? */
{	Classdef* classdef;

	if (obj->storage != typedef_storage)
	    return no;
	classdef = TypeToClassdef(obj->type);
	if (classdef == NULL)
	    return no;
	return (classdef->typedef_obj == obj);
}


interface bool NameIsOriginalEnum(Namedobj* obj)
/* Is this object the original typedef-object */
/* for a class? */
{	Namedobj* enum_obj;
	Type type;

	type = obj->type;
	if (*type != tp_enumerated)
	    return no;
	if (obj->storage != typedef_storage)
	    return no;
	GetPtrE(enum_obj, type);
	return (obj == enum_obj);
}


interface str NameUnameFromObj(Namedobj* obj, char dest[])
/* If this object is not overloaded, return its name. */
/* If it is, return a unique str consisting of the */
/* overloaded name and version number. */
{       Classdef* classdef;
        Directory* dir;
        str d;

        NameOwner(obj, &dir, &classdef);
        if (classdef) {
            NameUnameFromObj(classdef->typedef_obj, dest);
            d = dest + strlen(dest);
            strcpy(d, "::");
            d += 2;
        }
        else d = dest;
        strcpy(d, obj->name);
        d += strlen(d);
	if (obj->overload_version) {
	    *d++ = '.';
	    itoa(obj->overload_version, d, 10);
	}
	return dest;
}


interface bool NameBodyDefined(Namedobj* obj)
/* Has the body of this function or class definition been defined? */
/* Used to interface MAKE with the compiler. */
{
	if (obj->type[0] != tp_function)
	    return yes;
	else return (((FunctionNamedobj*)obj)->u.fn != NULL);
}


interface void NameOwner(Namedobj* obj, Directory* *dirp, Classdef* *classdefp)
/* Find the owner of this object.  It will be either a dir or a classdef. */
/* Set the relevant one and set the other to NULL. */
{       Classdef* classdef;

        classdef = (Classdef*)obj->owner;
        if (classdef->ValidClassdef()) {
             *classdefp = classdef;
             *dirp = NULL;
        }
        else {
            *dirp = (Directory*)classdef;
            *classdefp = NULL;
        }
}






/*--------------- Name-Space and Curdir Functions --------------*/

interface void Chdir(Directory* new_dir)
/* Change the current directory to the linked list rooted at 'new-root'. */
{
	/* It's actually quite important that we can easily access 	*/
        /* either the current directory, or the open container which 	*/
        /* contains the current directory. So we have these 2 variables,*/
        /* which are kept lock-step with each other. */
	curdir = new_dir;
	default_heap = curconim = Ptr_to_conim(new_dir);
}


interface bool ChdirContainer(container_id cid)
/* Change to this directory in another container. */
{       Directory* new_dir;
	Conim *new_conim;

	/* Are we going back to the container root? */
	if (cid == curconim->cid) {
	    Chdir(curconim->directory());
	    return yes;
	}

	/* Do we have a valid cid? */
	if (cid == 0)
	    return no;

	/* Open the new container: */
	new_dir = OpenContainer(cid, READWRITE);
	if (new_dir == NULL)
	    return no;
	new_conim = Ptr_to_conim(new_dir);
	default_heap = new_conim;

	/* Link in SRC if the container is not up-to-date. */
	ContainerCheckSource(new_conim);

        // Close the container left behind us.
        curconim->Assert();
	CloseContainer(curconim->cid);

	/* Do the Chdir(): */
	Chdir(new_dir);
	return yes;
}


interface void InitProvisionalTypstrs()
/* If we haven't compiled the directory class yet, initialise */
/* direc_typstr with a dummy classdef that passes as the     */
/* directory class.  Ditto for other key class definitions.   */
{	static Classdef dummy1, dummy2;

	if (*(Classdef**)(direc_typstr+1))
	    return;	/* They're already initialised. */

	/* Do the directory class: */
	dummy1.size = sizeof(Directory);
	dummy1.RoundupSize = Heap::RoundUp(dummy1.size) + 4;
	dummy1.signature = HELLO_BABE;
	dummy1.def_len = 0;
	dummy1.typedef_obj = NULL;
	dummy1.member = NULL;
	*(Classdef**)(direc_typstr+1) = &dummy1;

	/* Do the named_obj class: */
	dummy2 = dummy1;
	dummy2.size = sizeof(Namedobj);
	dummy2.RoundupSize = Heap::RoundUp(dummy2.size) + 4;
	*(Classdef**)(namedobj_typstr+1) = &dummy2;
}


static Namedobj* FindMemberFnInBase(Classdef* classdef, str name,
		Type footprint)
/* A recursive helper function for the below function. */
{	Namedobj *obj;

        for (obj=classdef->member; obj; obj=obj->next) {
            if (streq(obj->name, name) and EqualFootprint(obj->type, footprint))
            	return obj;
        }
        return NULL;
}


interface FunctionNamedobj* FindMemberFn(Classdef* classdef, str name, Type footprint)
/* Look for a function matching this (name,type) description in */
/* a base class of the given type. */
{	Namedobj *obj, *fobj;

	if (classdef == NULL)
            return NULL;
        for (obj=scope_list; obj; obj=obj->next) {
            if (obj->storage == inherit_storage) {
		fobj = FindMemberFnInBase(TypeToClassdef(obj->type), name, footprint);
		if (fobj)
                    return (FunctionNamedobj*)fobj;
            }
        }
        return NULL;
}


#if 1
#include <stdlib.h>
#include <direct.h>
#endif

interface void CurdirInit(void)
/* Clear 'scope_list' so that everything predefined stays pre-defined, */
/* and everything created from now onwards goes into some directory.   */
{       StaticNamedobj* obj;
	container_id cid;

	/* Clear the current directory. */
	scope_list = NULL;

	/* Initialise 'direc_typstr' with something that passes as */
	/* a Classdef*. */
	InitProvisionalTypstrs();

	/* Open the root container: */
	cid = ROOT_CID;
	root_dir = OpenContainer(cid, READWRITE);
	if (b_errno) {
	    if (b_errno == E_CORRUPT)
		DeleteContainer(cid);
	    root_dir = CreateContainer(0, &cid);
	    assert(cid == ROOT_CID);
	}
	curconim = FindConim(ROOT_CID);
	curdir = curconim->directory();
	default_heap = curconim;

	/* Open the 'bin' directory if there is one: */
	obj = (StaticNamedobj*)root_dir->Find("Common");
	if (obj and TypeEqual(obj->type, container_typstr) and obj->storage == static_storage) {
	    COMMON_CID = *(container_id*)obj->location;
	    common_dir = OpenContainer(COMMON_CID, READONLY);
	}
	else {
	    COMMON_CID = 0;
	    common_dir = CreateContainer(ROOT_CID, &COMMON_CID);
	    bool oldpd = PredefinitionPhase;
	    PredefinitionPhase = yes;
	    obj = (StaticNamedobj*)NameDeclare(root_dir, "Common",
                        container_typstr, static_storage, 0);
	    PredefinitionPhase = oldpd;
	    *(container_id*)obj->location = COMMON_CID;
	    common_dir->parent = NULL;
	    common_dir->parent_cid = ROOT_CID;
	}

	/* Set up the root directory: */
	Chdir(root_dir);
}


interface void CurdirClose(void)
/* Close the current directory, presumably because we're */
/* closing down the system. */
{
	CloseContainer(COMMON_CID);   /* The '/COMMON' container. */
	ContainerMakeAllObjs(curconim);
	CloseContainer(curconim->cid);
}


interface void CurdirInterfaceChange(bool just_address)
/* Register the fact that something in the current directory */
/* has changed its interface.  (Other containers need to know	     */
/* when containers they depend upon have changed).		     */
/* If 'just_address' is true, then it means only the address */
/* of an object has changed - i.e. the change will only      */
/* affect containers that are currently open.			     */
{
	if (curdir->parent != NULL or PredefinitionPhase)
	    return;
	/* Only update the container if we're in */
	/* the container's root-directory.       */
	assert(curdir == curconim->directory());
	ContainerInterfaceChange(curconim, just_address);
}





/*------------------ Printing directory listings: -----------------*/

static int GetObjKind(Namedobj* obj)
/* Classifies this object into a directory listing category. */
/* 0 = some kind of directory   */
/* 1 = function                 */
/* 2 = variable                 */
/* 3 = other                    */
{
	if (obj->storage == straight_fn)
	    return 1;
	if (obj->storage == static_storage) {
	    if (IsDirectory(obj, yes, yes))
		return 0;
	    else return 2;
	}
	if (obj->storage == typedef_storage or obj->storage == const_storage)
	    return 2;
	return 3;
}


static int compar_obj(Namedobj* *ap, Namedobj* *bp)
{       Namedobj* a=*ap, *b=*bp;
	int kinda, kindb;

	kinda = GetObjKind(a);
	kindb = GetObjKind(b);
	if (kinda != kindb)
	    return kinda - kindb;
	return stricmp(a->name, b->name);
}


interface void DirectoryPrint(Directory* dir)
/* Print the contents of this directory. */
{	Namedobj* obj, **list;
	char buf[200], buf2[512];
	int i,n,kind,r,c,height;
	str s,d;
	uint h;

	/* Put all the objects into the list: */
	n = 0;
	for (each_dir_obj(dir))
	    n++;
	list = (Namedobj**)malloc(anon_heap, n * sizeof(Namedobj*));
	n = 0;
	for (each_dir_obj(dir))
	    list[n++] = obj;

	/* Is it an empty directory? */
	if (n == 0) {
	    CoutPuts("<empty>");
	    free(anon_heap, list);
	    return;
	}

	/* Sort the list: */
	qsort(list, n, sizeof(list[0]), (cmp_fn)compar_obj);

	/* Print them: */
	height = (n + 3) / 4;
	for (r=0; r < height; r++) {
	    for (c=0; c < 4; c++) {
		i = c * height + r;
		if (i >= n)
		    break;
		obj = list[i];
		kind = GetObjKind(obj);
		d = buf;
		s = NameUnameFromObj(obj, buf2);
		if (kind == 3) {
		    *d++ = '(';
		    strcpy(d, s);
		    while (*d) d++;
		    *d++ = ')';
		}
		else {
		    strcpy(buf, s);
		    while (*d) d++;
		    if (kind == 0)
			*d++ = '/';
		    else if (kind == 1)
			*d++ = '(', *d++ = ')';
		    /*else if (kind == 2)
			;*/
		}
		while (d < buf + 20)
		    *d++ = ' ';
		d = buf + 20;
		if (c == 3)
		    d -= 2;
		*d = '\0';
		if (d[-1] != ' ')
		    d[-1] = '|';
		CoutPuts(buf);
	    }
	    CoutPuts("\n");
	}
	free(anon_heap, list);
}


interface void DirectoryPrintC(container_id cid)
/* Print the directory at the root of this container. */
{       Directory* dir;

	dir = OpenContainer(cid, READONLY);
	if (dir == NULL) {
	    char buf[512];
	    sprintf(buf, "B%d : %s\n", cid, BerrnoToString());
	    CoutPuts(buf);
	}
	else {
	    DirectoryPrint(dir);
	    CloseContainer(cid);
	}
}

