/* $Id: editor.h 881 2005-11-08 00:20:00Z jla $ */


#ifndef EDITOR_H
#define EDITOR_H

#ifndef TFC_H
#include "tfc.h"
#endif



typedef enum { pa_unknown, pa_paint, pa_measure, pa_cn2rect, pa_xy2cn,
                pa_getobj } pa_enum;

typedef struct Paragrapher_node {
        /* This represents the many input and output parameters to the        */
        /* Paragrapher_fn.  The Paragrapher function does everything        */
        /* the editor needs done regarding paragraphs:                        */
        /* pa_paint:            Paint the paragraph.                        */
        /* pa_measure:            Measure the dimensions of the paragraph.        */
        /* pa_cn2rect:            Map a character position to a rectangle.        */
        /* pa_xy2cn:            Map an x,y position to a character position.*/
        /* pa_getobj:            Return a pointer to the 'object' at (x,y).  */
        pa_enum pa;                    // What operation?
        int x1, x2, y1, y2;            // The rectangle to draw in, in ScrollWin coordinates
        int ha, hb;                    // The character positions where highlighting occurs
        int habuf, hbbuf;              // The buffer position corresponding to highligted characters positions
        int cn;                            // The character position of the cursor
        int cnbuf;                      // The position of the cursor in buffer
        int mouse_x, mouse_y;            // The Windows coordinates of the mouse
        bool IsCurrent;            // This is the current line
} *Paragrapher_type;


class Line : public Row {
        /* This represents one paragraph (or line, for program source-code). */
public:
        short state;                // This is the state _coming_in_.
        char type;                // What type of line is it?
        str buf;                // (The text itself).


        /*** The application-writer provides: ***/
        virtual void Paragrapher(Paragrapher_type P);
        /* All Editor Lines must define the 'paragrapher' function. */


        /*** TFC provides: ***/
        void Paint(int x1, int y1, int x2, int y2);
        void CursorPaint();

        int MapFromXY(int x, int y, int* nbuf = NULL);
        void MapToXY(int n, int *xp, int *yp);
        /* Convert this (ln,x,y) to a character position and back again. */
        /* (x and y in ScrollWin coordinates). */

        /*void Paint() { Litewin::Paint(); }
        /* Repaint the whole row. */

        void Measure();
        /* Use the 'paragrapher' to set up the 'width' and 'height' */
        /* parameters for this line. */

        void Measure(int *widthp, int *heightp) { Measure(); *widthp = width; *heightp = height; }
        /* Use the 'paragrapher' to set up the 'width' and 'height' */
        /* parameters for this line. */

        void SetCursor(int n);
        /* Create a cursor in this line at this position. */

        bool Mousestroke(int op, int x, int y);
        /* This row has been clicked-on. */

        virtual ~Line();
        /* The destructor. */

        friend class EditorWin;
};


class LineN_struct {
public:
        Line *ln;
        int n;       // number of letters (can be less than n for unicode)
        int nbuf;    // length of buffer
        bool operator==(LineN_struct B) { return B.ln == ln and B.n == n; }
};



typedef struct undo_node {
        Line *insert_this;            // A deleted line, owned by the undoer.
        Line *delete_this;            // A line in the editor that needs to be deleted.
        LineN_struct c;                    // The current line and position prior to the change
        struct undo_node *next;
} *undo_type;

#define EA_MAX_LINELEN   1024*5
#define multiundo        ((Line*)0x1)
#define MAX_UNDOS        10


class EditorWin : public RowWin {

protected:
        /*** Undo/Redo: ***/
        undo_type UndoRoot, RedoRoot;        // The chains of undo/redo info
        int NumUndos;                        // The number of undo's in our chain
        int TransLevel;                        // The number of nested transactions
        undo_type OldRoot;                // Temporary variable
        str IndentString;                   // The string to use when indenting (eg. "\t" or "   ")

		int nMaxLineLength;					// (awa) Allows the user to specify a variable maximum

        Line *InsertLineNoUndo(Line *prev, str buf, char type=0);
        Line *ReplaceLineNoUndo(Line *ln, str buf);
        void DeleteLineNoUndo(Line *ln);
        void CompressUndo(undo_type undo);
        void FreeUndoChain(undo_type undo);
        void CullUndos(void);
        void AddUndo(Line *insert_this, Line *delete_this, LineN_struct c);
        void PerformOperation(undo_type undo);
        void ReversePerformOperation(undo_type redo);
        void SetPosDirectly(Line* ln, int n) { c.ln = ln, c.n = n; }



        /*** Graphical editing: ***/
        LineN_struct c;                        // The current position
        LineN_struct temp;                // The position immediately before a move operation.
        LineN_struct temp2;                // A second temporary registered pointer.
        LineN_struct ReplaceStart;        // Where search'n'replace began from
        LineN_struct Marker;                // The start of selected text (first position shifted)
        LineN_struct Marker1;                // These two markers store c and Marker, with Marker1
        LineN_struct Marker2;                // storing the higher, lefter one and Marker2 storing the lower, righter one.
        struct {
            int x1,y1,x2,y2;
        } Cursor;


        void PointerHasChanged(Row *Old, Row *New);

        int AverageCharWidth(void);

        virtual void CursorPaint();
        void CursorPaint(Line *Row, int x1, int y1, int x2, int y2);
        /* These coordinates come from RowWin, so they're in Windows coordinates */
        /* suitable for the drawing commands. */



        /*** The line adt (editadt.cpp): ***/
        Line* NewLine(int n);
        Line* NewLine(str s);
        /* Construct a line. They both call the virtual NewLine() */
        /* which the application-writer provides. */

        void InsertLineAfter(Line *prev, str buf, LineN_struct c);
        /* Insert a line with this string after 'prev'. If 'prev==NULL', */
        /* insert it at the top.  Don't create any undo info. */

        void DeleteLine(Line *ln, LineN_struct c);
        /* Insert a line with this string after 'ln'. */

        void JoinLines(Line *ln, LineN_struct c);
        /* Join this line with its following line. */

        void BreakLine(Line *ln, int n, bool indent, LineN_struct c);
        /* Break this line at this point. */

        Line* ReplaceLine(Line *ln, str buf, LineN_struct c);
        /* Substitute this string with this line. */

        void DeleteAllLines();
        /* Clear out the whole file. */

        void Undo();
        /* Undo one operation. */

        void Redo();
        /* Redo one operation. */

        void StartTransaction(void);
        /* Set up a multi-operation transaction. */

        void FinishTransaction(void);
        /* Close the transaction.  (Can be nested). */

public:
        int TabSize;
        int insert_mode;
        int PaginatedWidth;
        int fontheight;
        char fontname[50];              // you can choose various font whithin LeToFontAndColours() formating
        TfcFont OverrideFont;           // If this != NULL, use it instead of LeToFontAndColours().
        Line *parse_line;
        char filename[512];
        bool NeedsSave;
        bool F3meansAgain;
        bool EditingOneLine;            // Yes for 1-line 'gets' edits
        bool FixedWidth;                // By default no, ie. use proportional font.
        int desired_x;
        int Foreground;                 // the normal foreground colour
        int LeftIndent;                 // default text indent from left  size
        int HighlightColour;            // default highlighting colour
        int CursorColour;               // default cursor colour


        /*** The application-writer provides: ***/
        virtual bool MoveUp(Line* ln) { return yes; }
        virtual bool MoveDown(Line* ln) { return yes; }
        virtual int DefineState(Line *ln) { return 0; };
        virtual void WindowClose();
        virtual void Save() { SaveFile(); }


        /*** Editadt.cpp provides: ***/
        EditorWin(char* Caption, int screenwidth, int screenheight,
                            int Background=DARK(BLUE), int Foreground=NOCOLOUR, 
							ScrollWin *OwnerSw=NULL);
        /* Initialise a blank editor.  The foreground colour is initialised */
        /* by default if you don't specify it. */



        /*** Editing commands: ***/
        virtual bool Keystroke(int key);


        /*** Moving: ***/
        bool MoveRight(void);
        /* Move right one character. Move to the next line if you're on the end. */
        /* Returns 'yes' on success. */

        bool MoveLeft(void);
        /* Move left one character. Move to the previous line if you're at the beginning. */
        /* Returns 'yes' on success. */

        bool MoveTo(int x, int y);
        /* Move to this location (x and y in ScrollWin coordinates). */

        bool MoveTo(Line *ln, int n);
        bool MoveTo(LineN_struct lnn) { return MoveTo(lnn.ln, lnn.n); }
        /* Move to this location (ln,n). */

        virtual bool IsWordStart(int n, bool for_ctrl_t);
        /* Are we at the start of a word at this position? */

        int IsBracket(int ch);
        /* 1=open bracket, -1=close bracket, 0=not a bracket. */

        void FindMatchingBracket(void);
        /* Move to the bracket matching this one. */



        /*** Selecting text and Cut/Copy/Paste/Gobble: ***/
        void StartSelection(void);
        /* Make the current position the start of a selection. */

        void UpdateSelection(LineN_struct);
        /* The selection has been extended to the current position. */

        void ClearSelection(void);
        /* The selection has been cleared. */

        bool HasSelection(void);
        /* Do we have anything selected? */

        bool IntersectSelection(int coln, int liney);
        /* Is this x and y coordinate inside the selected area? */

        void Copy(void);
        void CopyAndClear();
        /* Copy the selected text. */

        void Gobble(void);
        /* Gobble the selected text. */

        void Cut(void);
        /* Cut the selected text. */

        void Paste(void);
        /* Paste the clipboard into the editor. */

        void UndoOp() { Undo(); SetCursor(); }
        /* Undo one operation.   */

        void RedoOp() { Redo(); SetCursor(); }
        /* Redo one operation.  */



        /*** Applications interacting with the text: ***/
        void InsertChar(Line* ln, int n, char ch);
        void InsertWord(Line* ln, int n, str word);
        void InsertChar(LineN_struct pos, char ch) { InsertChar(pos.ln, pos.n, ch); }
        void ReplaceLineWithUndo(Line *ln, str buf) { ReplaceLine(ln, buf, c); }
        Line* InsertBigString(Line *AfterThis, str bigbuf);

        Line* GetPosition(int *np=NULL) { if (np) *np = c.n; return c.ln; }
        /* Get the current position. */

		void SetMaximumInputLength(int nMaximumLineLength);


        /*** Misc features: ***/
        void UseSpaceIndenting(int numspaces);
        void UseTabIndenting();
        bool UsingTabIndenting();        
        int UsingSpaceIndenting();
        
        void Indent(bool right);
        /* Indent (or unindent) the selected text. */

        void Resized(void);
        /* Re-measure lines in response to a change in the window dimensions. */

        void SetCursor() { c.ln->SetCursor(c.n); }
        /* Repaint the cursor. */

        int GetKey() { return ScrollWin::GetKey(); }
        int GetKey(Line *ln, int n) { ln->SetCursor(n); return ScrollWin::GetKey(); }
        /* Get a keystroke with the cursor being at a particular location. */

        void GuiFindOrReplace(bool find_again, bool replace, bool backwards);
        void GuiFind();
        void GuiReplace();
        void FindDlgDied();
        void FindNextFunction();
        /* Find a substring (dialog box version) */
        void FindOrReplace(char *findstring, char *replacewith,
                    bool case_sensitive, bool whole_word, bool prompt, bool backwards);
            /* Go to the next occurrence of this string and perhaps replace it. */

        void Clear();
        /* Clear all lines and start with a fresh screen. */

        void ClearAllUndos(void);
        /* Chop off the undo chain. */

        bool LoadFile(char *filename, int maxSize=0);
        /* Read this file into the editor. Returns 'yes' on success. */

        bool SaveFile();
        /* Save the contents of the editor to disk. Returns 'yes' on success. */

        void LoadString(char *text);
        /* Read this null-terminated string into the editor. */

        void AppendString(char* text);
        /* Read this null-terminated string into the editor - append it. */

        str SaveString();
        /* Package the contents of the editor into a single null-terminated */
        /* string allocated with strdup, and give it to the caller. */

        void Print(str jobname=NULL);
        /* Print the document. */


        /*** The main function: ***/
        void Edit();
        /* Edit this file with this editor. */

        str EditOneLine();
        /* Like above, except that we only edit a single line. */

        Line* GetCurrentLine();
        /* (awa) Added a some-what sensible missing abilitly to get
        the current line */

        int GetCurrentColumn();
        /* (awa) Ditto for column */

        friend class Line;


        /*** The application-writer provides: ***/
protected:
        virtual Line* NewLine() { return new Line; }
};

// functions for UTF-8
int charlen_bwd(str charBuf, int idx);
int charlen(str charBuf);
int utf8IndexToChar(char* buf, int charIndex);


/*--------------- An application of the editor: a teletype interface. -----------------*/

class Tty : public EditorWin {
        bool FlushEachLine;
        bool AlsoDisplay;
        char buffer[300];
        FILE* LogFile;

        Line* NewLine();
        int DefineState(Line *ln) { return 0; }
        void Flush();

public:
        int prompt_n;
        int MaxLines;
        str gets_buf;
        int sizeof_gets_buf;

        Tty(str caption, int width, int height, int Background=WHITE, unsigned int MaxLines=250,
						ScrollWin *OwnerSw=NULL);
        void WindowClose() { delete this; }
        virtual void OnEnter(str text) { }
        int printf(const char* fmt, ...);
        str puts(const char* buf);
        str gets(str buf, int sizeofbuf=512);
        bool Keystroke(int key);
        void RedirectToFile(kstr filename, bool FlushEachLine=yes,
                                bool AppendMode=yes, bool AlsoDisplay=no);
        void Finish(bool WaitForKeystroke);          // Close the file if we've redirected,
        // wait for a keystroke if we haven't.
        void Show() { if (AlsoDisplay or LogFile == NULL) EditorWin::Show(); }

        friend class TtyLine;
};



str _cdecl tfc_puts(const char* buf);
char* _cdecl tfc_gets(char buf[]);
int _cdecl tfc_printf(const char *fmt, ...);
        /* C-style versions of the functions. */

int HelpDefineState(Line *prev);
        /* For the default 'Help' Line-type, this is the define-state fn. */

void tfc_LogToFile(str filename, bool FlushEachLine,
                bool Append, bool AlsoDisplay);

void tfc_initTty(str title, int width, int height, int Background);
void tfc_printHardcopy(str title);
void tfc_clearTty();

#endif

